module no2info 
!$$$ module documentation block
!           .      .    .                                       .
! module:   no2info
!   prgmmr: treadon     org: np23                date: 2004-04-10
!
! abstract:  This module contains variables and routines related
!            to the assimilation of no2 observations (presently,
!            satellite based no2 observations)
!
! program history log:
!   2004-04-10  treadon - original code
!   2004-05-13  kleist  - original documentation
!   2004-06-16  treadon - update documentation
!   2005-09-28  derber  - Modify for new no2info file, add var qc parameters
!   2006-02-03  derber  - modify for new obs control and obs count
!   2007-06-29  Zhou    - change total number of no2 enteries (jpch_no2) from
!                         53 (version 6 SBUV/2) to 67 (version 8 SBUV/2)
!
! Subroutines Included:
!   sub init_no2       - set no2 related variables to defaults
!   sub no2info_read   - read in no2 info
!
! Functions Included:
!
! Variable Definitions:
!   def diag_no2     - logical to turn off or on the diagnostic no2 file (true=on)
!   def jpch_no2        - number of (levels+1) * number of satellites
!   def mype_no2        - task id for writing out radiance diagnostics
!   def pob_no2         - pressure level of observation (hPa)
!   def gross_no2       - gross error limit
!   def error_no2       - observation error
!   def nusis_no2       - sensor/intrument/satellite id (14=NOAA-14, 15=NOAA-15, 16=NOAA-16, etc)
!   def nulev          - integer level of no2 observation
!   def iuse_no2        - integer flag to control usage of no2 data (-1=don't use, 1=use)
!
! attributes:
!   language: f90
!   machine:  ibm RS/6000 SP
!
!$$$ end documentation block

  use kinds, only: r_kind,i_kind
  implicit none

! set default to private
  private
! set subroutines to public
  public :: init_no2
  public :: no2info_read
! set passed variables to pubic
  public :: jpch_no2,diag_no2,nusis_no2,iuse_no2,b_no2,pg_no2,gross_no2
  public :: error_no2,pob_no2,mype_no2,nulev
  public :: ihave_no2

  logical diag_no2
  integer(i_kind) mype_no2,jpch_no2
  real(r_kind),allocatable,dimension(:)::pob_no2,gross_no2,error_no2,pg_no2,b_no2
  integer(i_kind),allocatable,dimension(:):: nulev,iuse_no2
  character(len=20),allocatable,dimension(:):: nusis_no2

  logical :: ihave_no2

contains
  
  subroutine init_no2
!$$$  subprogram documentation block
!                .      .    .                                       .
! subprogram:    init_no2     initialize parameters for no2 data
!     prgmmr:    treadon     org: np23                date: 2004-04-10
!
! abstract:  This routine sets default values for variables used in 
!            the no2 processing routines
!
! program history log:
!   2004-04-10  treadon
!   2004-06-16  treadon, documentation
!   2005-07-28  treadon - increase jpch_no2 from 52 to 53 (add omi data)
!   2010-05-29  todling - check existence in state vector (viz., guess)
!
!   input argument list:
!
!   output argument list:
!
! attributes:
!   language: f90
!   machine:  ibm rs/6000 sp
!
!$$$
    use mpimod, only: npe              ! contains the number of mpi tasks, variable "npe"
    use state_vectors, only: svars3d
    use mpeu_util, only: getindex
    use mpimod, only : mype ! ajl
    implicit none

    jpch_no2 = 0                        ! number of enteries read from no2info
    diag_no2 = .true.                ! default is to generate no2 diagnostic file
    mype_no2     = max(0,npe-6)         ! mpi task to write no2 summary report
    ihave_no2=(getindex(svars3d,'no2')>0)! .t. when no2 present in state-vector
!    write(490+mype,*)'init_no2 ihave_no2',ihave_no2,'mype_no2',mype_no2
!    call flush(490+mype)
    

  end subroutine init_no2
  

  subroutine no2info_read
!$$$  subprogram documentation block
!                .      .    .                                       .
! subprogram:    no2info_read      read no2 information file
!     prgmmr:    treadon     org: np23                date: 2004-04-10
!
! abstract:  This routine reads the no2 information file, global_no2info.txt
!
! program history log:
!   2004-04-10  treadon
!   2004-06-16  treadon, documentation
!   2005-10-11  treadon - change no2info read to free format
!   2008-04-29  safford - rm redundant use
!   2010-05-29  todling - update interface
!
!   input argument list:
!
!   output argument list:
!
! attributes:
!   language: f90
!   machine:  ibm rs/6000 sp
!
!$$$
    use mpimod, only: mype
    use obsmod, only: iout_no2
    implicit none

    character(len=1):: cflg
    character(len=120) crecord
    integer(i_kind) lunin,j,k,istat,nlines
    data lunin / 47 /


!   Determine number of entries in no2 information file
    open(lunin,file='no2info',form='formatted')
    j=0
    nlines=0
    read1:  do 
       read(lunin,100,iostat=istat,end=123) cflg,crecord
       if (istat /= 0) exit
       nlines=nlines+1
       if (cflg == '!') cycle
       j=j+1
    end do read1
123 continue
    if (istat>0) then
       write(6,*)'NO2INFO_READ:  ***ERROR*** error reading no2info, istat=',istat
       close(lunin)
       write(6,*)'NO2INFO_READ:  stop program execution'
       call stop2(79)
    endif
    jpch_no2 = j
    write(6,*)'ajl jpch_no2 =',jpch_no2,'mype_no2',mype_no2,'iout_no2',iout_no2
    call flush(6)


!   Allocate arrays to hold no2 information
    allocate(nusis_no2(jpch_no2),nulev(jpch_no2),iuse_no2(jpch_no2), &
         pob_no2(jpch_no2),gross_no2(jpch_no2),error_no2(jpch_no2), &
         pg_no2(jpch_no2),b_no2(jpch_no2))


!   All mpi tasks open and read no2 information file.
!   Task mype_no2 writes information to no2 runtime file
  
    if (mype==mype_no2) then
       open(iout_no2)
       write(iout_no2,110) jpch_no2
110    format('NO2INFO_READ:  jpch_no2=',1x,i6)
    endif
    rewind(lunin)
    j=0
    do k=1,nlines
       read(lunin,100) cflg,crecord
       if (cflg == '!') cycle
       j=j+1
       read(crecord,*) nusis_no2(j),&
            nulev(j),iuse_no2(j),pob_no2(j),gross_no2(j),error_no2(j), &
            b_no2(j),pg_no2(j)
       if (mype==mype_no2) write(iout_no2,130) j,nusis_no2(j),nulev(j),&
               iuse_no2(j),pob_no2(j),gross_no2(j),error_no2(j),b_no2(j), &
               pg_no2(j)
    end do
    close(lunin)
    if (mype==mype_no2) close(iout_no2)

100 format(a1,a120)
130 format(i3,1x,a20,' lev = ',i4,' use = ',i2,' pob = ',f9.3,&
         ' gross = ',f7.3,' error = ',f7.3,' b_no2 = ',f7.3,' pg_no2 = ',f7.3)


!   Successful read, return to calling routine
    return
  end subroutine no2info_read
  
end module no2info
