import { defineStore } from 'pinia';
import * as mortalityApi from '../../api/mortality-rate';
import * as morbidityApi from '../../api/morbidity-rate';
import { useDesignValuesStore } from '../design-values';
import { initYear, yearOptions } from '../../libs/year-options';
import { ozoneCache as cache } from '../../libs/geoid-cache';
import { emitter, eventTypes } from '../../libs/event-hub';

export const useOzoneStore = defineStore('ozone', {
    state: () => {
        return {
            year: initYear,
            yearOptions: yearOptions,
            eightHour: {
                isChecked: true,
                currentValue: 70,
                maxValue: 100,
            },
            mortalityRate: {
                isChecked: false,
                radio: 0,
                currentValue: 10,
                maxValue: 15,
                currentPercentile: 90,
            },
            morbidityRate: {
                endpoint: 1001,
                endpointOptions: [
                    { value: 1001, label: 'ER Visits Respiratory' },
                    { value: 1002, label: 'HA All Respiratory' },
                    { value: 1004, label: 'School Loss Days All Cause' },
                ],
                isChecked: false,
                radio: 0,
                currentValue: 20,
                currentPercentile: 90,
            },
        };
    },
    getters: {
        morbidityRateMaxValue() {
            switch (this.morbidityRate.endpoint) {
                case 1001:
                    return 50;
                case 1002:
                    return 50;
                case 1004:
                    return 20000;
            }
        },
    },
    actions: {
        async handleOzoneEightHourData() {
            if (this.eightHour.isChecked) {
                const designValuesStore = useDesignValuesStore();
                cache.eightHour = await designValuesStore.getOzone8HourData(this.year, this.eightHour.currentValue);
            } else {
                cache.eightHour = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },
        async handleOzoneMortalityRateData() {
            if (this.mortalityRate.isChecked) {
                if (this.mortalityRate.radio === 0) {
                    cache.mortalityRate.values = await mortalityApi.GetOzoneGeoId11ByMinValue(this.year, 1007, this.mortalityRate.currentValue);
                } else {
                    cache.mortalityRate.percentiles = await mortalityApi.getOzoneGeoId11ByMinPercentile(this.year, 1007, this.mortalityRate.currentPercentile);
                }
            } else {
                cache.mortalityRate.values = [];
                cache.mortalityRate.percentiles = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },
        async handleOzoneMorbidityRateData() {
            if (this.morbidityRate.isChecked) {
                if (this.morbidityRate.radio === 0) {
                    cache.morbidityRate.values = await morbidityApi.GetOzoneGeoId11ByMinValue(this.year, this.morbidityRate.endpoint, this.morbidityRate.currentValue);
                } else {
                    cache.morbidityRate.percentiles = await morbidityApi.getOzoneGeoId11ByMinPercentile(
                        this.year,
                        this.morbidityRate.endpoint,
                        this.morbidityRate.currentPercentile
                    );
                }
            } else {
                cache.morbidityRate.values = [];
                cache.morbidityRate.percentiles = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },
    },
});
