import { defineStore } from 'pinia';
import * as mortalityApi from '../../api/mortality-rate';
import * as morbidityApi from '../../api/morbidity-rate';
import { useDesignValuesStore } from '../design-values';
import { initYear, yearOptions } from '../../libs/year-options';
import { pm25Cache as cache } from '../../libs/geoid-cache';
import { emitter, eventTypes } from '../../libs/event-hub';

export const usePM25Store = defineStore('pm25', {
    state: () => {
        return {
            year: initYear,
            yearOptions: yearOptions,
            annual: {
                isChecked: true,
                currentValue: 12,
                maxValue: 20,
            },
            daily: {
                isChecked: true,
                currentValue: 35,
                maxValue: 70,
            },
            mortalityRate: {
                isChecked: false,
                radio: 0,
                currentValue: 10,
                maxValue: 15,
                currentPercentile: 90,
            },
            morbidityRate: {
                endpoint: 1000,
                endpointOptions: [
                    { value: 1000, label: 'ER Visits Respiratory' },
                    { value: 1003, label: 'HA All Respiratory' },
                    { value: 1005, label: 'Work Loss Days' },
                ],
                isChecked: false,
                radio: 0,
                currentValue: 20,
                currentPercentile: 90,
            },
        };
    },
    getters: {
        morbidityRateMaxValue() {
            switch (this.morbidityRate.endpoint) {
                case 1000:
                    return 50;
                case 1003:
                    return 50;
                case 1005:
                    return 1500;
            }
        },
    },
    actions: {
        async handleYearChanged() {
            await this.handlePM25AnnualData();
            await this.handlePM25DailyData();
        },
        async handlePM25AnnualData() {
            if (this.annual.isChecked) {
                const designValuesStore = useDesignValuesStore();
                cache.annual = await designValuesStore.getPM25AnnualData(this.year, this.annual.currentValue);
            } else {
                cache.annual = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },

        async handlePM25DailyData() {
            if (this.daily.isChecked) {
                const designValuesStore = useDesignValuesStore();
                cache.daily = await designValuesStore.getPM25DailyData(this.year, this.daily.currentValue);
            } else {
                cache.daily = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },
        async handlePM25MortalityRateData() {
            if (this.mortalityRate.isChecked) {
                if (this.mortalityRate.radio === 0) {
                    cache.mortalityRate.values = await mortalityApi.GetPM25GeoId11ByMinValue(this.year, 1006, this.mortalityRate.currentValue);
                } else {
                    cache.mortalityRate.percentiles = await mortalityApi.getPM25GeoId11ByMinPercentile(this.year, 1006, this.mortalityRate.currentPercentile);
                }
            } else {
                cache.mortalityRate.values = [];
                cache.mortalityRate.percentiles = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },
        async handlePM25MorbidityRateData() {
            if (this.morbidityRate.isChecked) {
                if (this.morbidityRate.radio === 0) {
                    cache.morbidityRate.values = await morbidityApi.GetPM25GeoId11ByMinValue(this.year, this.morbidityRate.endpoint, this.morbidityRate.currentValue);
                } else {
                    cache.morbidityRate.percentiles = await morbidityApi.getPM25GeoId11ByMinPercentile(
                        this.year,
                        this.morbidityRate.endpoint,
                        this.morbidityRate.currentPercentile
                    );
                }
            } else {
                cache.morbidityRate.values = [];
                cache.morbidityRate.percentiles = [];
            }

            emitter.emit(eventTypes.OnRefreshDataAndSetStyle);
        },
    },
});
