import { defineStore } from 'pinia';
import * as api from '../api/design-values';

const cache = {};
const getByYear = (year) => {
    let result = cache[year];
    if (result === undefined) {
        result = {
            pm25: {
                annual: [],
                daily: [],
            },
            ozone: {
                eightHour: [],
            },
        };
        cache[year] = result;
    }

    return result;
}

const filter = (array, min) => {
    const result = [];
    for (let i = 0, len = array.length; i < len; i++) {
        const item = array[i];
        if (item.value >= min) {
            result.push(item.geoId5);
        }
    }

    return result;
}

export const useDesignValuesStore = defineStore('design-values', {
    state: () => cache,
    actions: {
        async getPM25AnnualData(year, min) {
            const pm25 = getByYear(year).pm25;

            if (pm25.annual.length === 0) {
                pm25.annual = await api.getPM25Annual(year);
            }

            return filter(pm25.annual, min);
        },
        async getPM25DailyData(year, min) {
            const pm25 = getByYear(year).pm25;

            if (pm25.daily.length === 0) {
                pm25.daily = await api.getPM25Daily(year);
            }

            return filter(pm25.daily, min);
        },
        async getOzone8HourData(year, min) {
            const ozone = getByYear(year).ozone;

            if (ozone.eightHour.length === 0) {
                ozone.eightHour = await api.getOzone8Hour(year);
            }

            return filter(ozone.eightHour, min);
        },
    },
});
