#' @title Create boxplots
#'
#' @description
#' Create the boxplots for number of hits across the food chemicals before and
#' after cytotox filtering
#'
#' @import data.table
#' @export


bxplt <- function(fname = NULL) {
  
  op <- par(no.readonly = TRUE)
  on.exit(par(op))
  
  data(TM, CD, envir = environment())
    
  hc_chem <- apply(TM$hitc[CD[nhit > 2, code], ], 1, sum, na.rm = TRUE)
  zh_chem <- apply(TM$filt_hit[CD[nhit > 2, code], ], 1, sum, na.rm = TRUE)
  
  ns <- c("Direct Food\nAdditives",
          "Food Contact\nSubstances",
          "Pesticides")
  n0 <- c(length(which(hc_chem[CD[i == 1, code]] > 0)),
          length(which(zh_chem[CD[i == 1, code]] > 0)),
          length(which(hc_chem[CD[i == 2, code]] > 0)),
          length(which(zh_chem[CD[i == 2, code]] > 0)),
          length(which(hc_chem[CD[i == 3, code]] > 0)),
          length(which(zh_chem[CD[i == 3, code]] > 0)))
  bxloc <- c(1:2, 4:5, 7:8)
  bxwd <- 0.8
  
  if (!is.null(fname)) png(fname, 6, 4.5, "in", res = 600)
  par(oma = c(7, 0, 0, 0), mar = c(0, 5, 2, 1))
  bres <- boxplot(x = list(hc_chem[hc_chem > 0][CD[i == 1, code]],
                           zh_chem[zh_chem > 0][CD[i == 1, code]],
                           hc_chem[hc_chem > 0][CD[i == 2, code]],
                           zh_chem[zh_chem > 0][CD[i == 2, code]],
                           hc_chem[hc_chem > 0][CD[i == 3, code]],
                           zh_chem[zh_chem > 0][CD[i == 3, code]]),
                  ylab = "# Active Endpoints",
                  frame = FALSE, 
                  notch = TRUE, 
                  xaxt = "n", 
                  at = bxloc,
                  log = "y", 
                  border = rep(c("gray50", "gray50"), 3),
                  boxwex = bxwd)
  pdim <- par()$usr
  par(oma = c(7, 0, 0, 0), mar = c(0, 5, 2, 1), 
      new = TRUE, xaxs = "i", yaxs = "i")
  plot.new()
  plot.window(xlim = pdim[1:2], ylim = pdim[3:4])
  shade <- function(n) {
    x0 <- bxloc[n]
    wd <- bxwd/2
    xpts <- c(rep(x0 - wd, 2), x0 - wd/2, rep(x0 - wd, 2),
              rep(x0 + wd, 2), x0 + wd/2, rep(x0 + wd, 2))
    res <- c(bres$stats[2:4, n], bres$conf[ , n])
    ordr <- c(1, 4, 2, 5, 3)
    ypts <- log10(c(res[ordr], res[rev(ordr)]))
    polygon(x = xpts, y = ypts, angle = 45, 
            density = 30, border = NA, col = "gray50")
  }
  shade(2); shade(4); shade(6)
  text(line2user(0, 2), line2user(0.5, 1),
       adj = c(1, 1), labels = "# Chemicals", xpd = NA)
  text(c(1:2, 4:5, 7:8), line2user(0.5, 1), adj = c(0.5, 1), xpd = NA, n0)
  text(c(1, 4, 7) + 0.5, line2user(2, 1), adj = c(0.5, 1), xpd = NA, ns)
  legend(y = line2user(5.5, 1), x = mean(par()$usr[1:2]),
         legend = c("Unfiltered Results", "Filtered Results"),
         bty = "n", border = c("gray50", "gray50"),
         horiz = TRUE, xpd = NA, xjust = 0.5, yjust = 0.5, angle = c(0, 45),
         density = c(0, 30), fill = c("gray50", "gray50"))
  
  if (!is.null(fname)) graphics.off()
  
}