#' @name honey.plot
#' @title Make honeycomb plot
#' @description Make a honeycomb heatmap
#'
#' @export

honey.plot <- function(xdim, ydim, vals, pal = NULL, border = NULL,
                       lwd = NULL, breaks = NULL, vrng = NULL, vlim = NULL,
                       key = TRUE, key.width = 0.5, key.lab = NULL,
                       key.cex = 1, key.border = NULL, hlab = FALSE,
                       hlab.col = "white", main = NULL, main.cex = 1,
                       hl = NULL, hl.col = "gray80", mars = NULL) {

  if (length(vals) != xdim*ydim) {
    stop("Length of vals does not match the number of bins")
  }

  op <- par()[c("mar")]
  on.exit(par(op))

  if (!is.null(vlim)) {
    vals[vals < vlim[1]] <- vlim[1]
    vals[vals > vlim[2]] <- vlim[2]
  }

  if (is.null(mars)) {
    mars <- rep(0, 4)
    if (key) mars[2] <- 3.5
    if (!is.null(main)) mars[3] <- 0.5
  }

  par(mar = mars)
  plot.new()
  plot.window(xlim = c(sqrt(3)/2 - 1/2,
                       sqrt(3)/2 - 1/2 + xdim*3/4 + 9/4 - sqrt(3)),
              ylim = c(0, (ydim - 1)*sqrt(3)/2 + 2 - sqrt(3)/4),
              asp = 1)
  title(main = main, line = -0.5)

  hex.xpts <- function(x) {
    c(x - 1/4, x - 1/2, x - 1/4, x + 1/4, x + 1/2, x + 1/4)
  }
  hex.ypts <- function(y) {
    c(y - sqrt(3)/4, y, y + sqrt(3)/4, y + sqrt(3)/4, y, y - sqrt(3)/4)
  }
  hgon <- function(x, y, ...) polygon(x = hex.xpts(x), y = hex.ypts(y), ...)

  hpts <- data.table(expand.grid(x = 1:xdim, y = 1:ydim))
  hpts[ , nx := 1 + 3/4*(x - 1)]
  hpts[ , ny := 1 + sqrt(3)/2*(y - 1) - sqrt(3)/4*((x - 1)%%2)]

  if (is.null(vrng)) vrng <- range(vals, na.rm = TRUE)
  if (is.null(pal)) pal <- heat.colors
  if (is.null(breaks)) breaks <- min(20, length(unique(vals)))
  cols <- pal(breaks)
  bins <- cols[as.numeric(cut(c(vals, vrng), breaks = breaks))]
  hpts[ , hc := bins[1:length(vals)]]

  if (!is.null(hl)) hpts[hl, hc := hl.col]

  for (i in 1:nrow(hpts)) {
    with(hpts[i], hgon(nx, ny, col = hc, border = border, lwd = lwd))
  }

  if (hlab) text(hpts$nx, hpts$ny, 1:(xdim*ydim), col = hlab.col)

  if (key) {
    lx <- c(0 - key.width, 0)
    ly <- range(hpts$ny)
    ly[1] <- ly[1] - sqrt(3)/4; ly[2] <- ly[2] + sqrt(3)/4
    rect(lx[1],
         seq(ly[1], ly[2], length.out = breaks + 1)[-(breaks + 1)],
         lx[2],
         seq(ly[1], ly[2], length.out = breaks + 1)[-1],
         col = pal(breaks),
         border = key.border)

    lv <- pretty(vrng)
    lv <- lv[lv <= max(vrng) & lv >= min(vrng)]
    yv <- ly[1] + diff(ly)*(lv - vrng[1])/diff(vrng)
    text(lx[2] - 1.3*diff(lx), yv, formatC(lv),
         xpd = NA, adj = 1, cex = key.cex)
    if (!is.null(key.lab)) {
      mtext(text = key.lab, side = 2, cex = key.cex, line = 2.5)
    }
  }

  invisible(hpts)

}

#' @name honey.box
#' @title Add box to honeycomb
#' @description Add a box to honeycomb heatmap pointing to bin
#'
#' @export

honey.box <- function(xdim, ydim, bin, p = 0.97, back.lines = TRUE,
                      use.oma = TRUE, ...) {

  hpts <- data.table(expand.grid(x = 1:xdim, y = 1:ydim))
  hpts[ , nx := 1 + 3/4*(x - 1)]
  hpts[ , ny := 1 + sqrt(3)/2*(y - 1) - sqrt(3)/4*((x - 1)%%2)]

  bl <- hpts[ , c(max(nx) + 1, min(ny) - sqrt(3)/4)]
  u <- if (use.oma) "ndc" else "nfc"
  tr <- hpts[ , c(grconvertX(p, u, "user"), max(ny) + sqrt(3)/4)]
  b <- hpts[bin, c(nx, ny)]

  lines(x = c(b[1], bl[1]), y = c(b[2], tr[2]), xpd = NA, ...)
  lines(x = c(b[1], bl[1]), y = c(b[2], bl[2]), xpd = NA, ...)

  if (back.lines) {
    lines(x = c(b[1], bl[1]),
          y = c(b[2], b[2] + (bl[1] - b[1])*(tr[2] - b[2])/(tr[1] - b[1])),
          xpd = NA, ...)
    lines(x = c(b[1], bl[1]),
          y = c(b[2], b[2] + (bl[1] - b[1])*(bl[2] - b[2])/(tr[1] - b[1])),
          xpd = NA, ...)
  }

  lines(x = c(bl[1], bl[1], tr[1], tr[1], bl[1]),
        y = c(bl[2], tr[2], tr[2], bl[2], bl[2]),
        xpd = NA, ...)

  invisible(list(bl = bl, tr = tr))

}

#' @name cbrh
#' @title Cool Blue Hot Red
#' @description Color palette ("cool blue hot red")
#'
#' @export

cbrh <- function(n, alpha = 1) rainbow(n, end=4/6, alpha=alpha)[n:1]

#' @name add.mol
#' @title Add a molecule structure to a plot
#' @description Add a molecule structure to a plot
#'
#' @import rcdk
#' @export

add.mol <- function(mol, xleft, ybottom, xright, ytop,
                    w = 500, h = 500, rot = 0) {
  tmp <- view.image.2d(mol, w, h)
  rasterImage(tmp, xleft, ybottom, xright, ytop, xpd = NA, angle = rot)
}

#' @name add.molgrid
#' @title Add a grid of molecule structures to a plot
#' @description Add a grid of molecule structures to a plot
#'
#' @import rcdk
#' @export


add.molgrid <- function(mols, labels, coln, bl, tr, space = 0.02, 
                        lab.font, lab.col) {

  n <- length(mols)
  rown <- ceiling(n/coln)

  s <- max((tr[1] - bl[1])*space, (tr[2] - bl[2])*space)
  w <- (tr[1] - bl[1] - s*(coln + 1))/coln
  h <- (tr[2] - bl[2] - s*(rown + 1))/rown

  g <- as.data.table(expand.grid(row = 1:rown, col = 1:coln))[order(-row)]
  g[ , x0 := bl[1] + col*s + (col - 1)*w]
  g[ , y0 := bl[2] + row*s + (row - 1)*h]
  g <- g[1:n]
  g[ , f := lab.font]
  g[ , c := lab.col]

  for (i in 1:nrow(g)) {
    with(g[i], add.mol(mols[[i]], x0, y0, x0 + w, y0 + h))
    with(g[i], 
         text(x = x0 + w/2, y = y0 + h, labels[i], 
              xpd = NA, font = f, col = c))
  }

}

#' @name honey.plot
#' @title Make honeycomb plot
#' @description Make a honeycomb heatmap
#'
#' @export

honey.layout <- function(xdim, ydim) {
  
  op <- par()[c("mar")]
  on.exit(par(op))
  
  par(mar = rep(0, 4))
  plot.new()
  plot.window(xlim = c(sqrt(3)/2 - 1/2,
                       sqrt(3)/2 - 1/2 + xdim*3/4 + 9/4 - sqrt(3)),
              ylim = c(0, (ydim - 1)*sqrt(3)/2 + 2 - sqrt(3)/4),
              asp = 1)
  
  hex.xpts <- function(x) {
    c(x - 1/4, x - 1/2, x - 1/4, x + 1/4, x + 1/2, x + 1/4)
  }
  hex.ypts <- function(y) {
    c(y - sqrt(3)/4, y, y + sqrt(3)/4, y + sqrt(3)/4, y, y - sqrt(3)/4)
  }
  hgon <- function(x, y, ...) polygon(x = hex.xpts(x), y = hex.ypts(y), ...)
  
  hpts <- data.table(expand.grid(x = 1:xdim, y = 1:ydim))
  hpts[ , nx := 1 + 3/4*(x - 1)]
  hpts[ , ny := 1 + sqrt(3)/2*(y - 1) - sqrt(3)/4*((x - 1)%%2)]
  
  for (i in 1:nrow(hpts)) {
    with(hpts[i], hgon(nx, ny))
  }
  
  text(hpts$nx, hpts$ny, 1:(xdim*ydim))
  
}

