#' @title Create ToxCast ternary diagram
#'
#' @description
#' Create the ternary digram showing log2 sum potency for the bins that had
#' ToxCast data
#'
#' @import data.table
#' @import tcpl
#' @import vcd
#' @export

ternplot <- function(fname = NULL) {
  
  data(SR)
  
  if (!is.null(fname)) {
    png(fname, pointsize = 12, height = 6, width = 6, units = "in", res = 600)
  }
  
  ## Prop in Category 1
  v1 <- SR[!is.na(casn) & toxc, list(p = lw(cat1)/.N), by = bin][J(1:480), p]
  
  ## Prop in Category 2
  v2 <- SR[!is.na(casn) & toxc, list(p = lw(cat2)/.N), by = bin][J(1:480), p]
  
  ## Prop in Category 3
  v3 <- SR[!is.na(casn) & toxc, list(p = lw(cat3)/.N), by = bin][J(1:480), p]
  
  ## Mean log2 sum potency 
  v5 <- SR[which(toxc), list(h = mean(log2(sump))), by = bin][J(1:480), h]
  v5[is.infinite(v5)] <- 0
  
  mat <- na.omit(do.call(cbind, list(v1, v2, v3)))
  colnames(mat) <- c("cat1", "cat2", "cat3")
  set.seed(9876)
  xpos <- mat[ , 2] + mat[ , 3]/2
  ypos <- mat[ , 3]*sqrt(3)/2
  xpos <- xpos + rnorm(n = length(xpos), mean = 0, sd = 0.075)
  ypos <- ypos + rnorm(n = length(ypos), mean = 0, sd = 0.075)
  
  pal <- cbrh
  vrng <- range(na.omit(v5))
  breaks <- 20
  cols <- pal(breaks, alpha = 0.65)
  bins <- cols[as.numeric(cut(c(na.omit(v5), vrng), breaks = breaks))]
  
  cntr <- 1/3*sqrt(3)*0.5
  add <- 0.5
  par(pty = "s", mar = rep(0, 4))
  plot.new()
  plot.window(xlim = c(0.5 - 1 - add, 0.5 + 1 + add), 
              ylim = c(cntr - 1 - add - 0.5, cntr + 1 + add))
  points(ypos ~ xpos, pch = 19, col = bins, cex = 1.2)
  arrows(x0 = 0.5, x1 = 0.5, y0 = cntr, y1 = cntr + 0.5*sin(pi/3) + add, 
         col = "darkgrey", lwd = 3, length = 0.2, angle = 45)
  arrows(x0 = 0.5, x1 = 0 - cos(pi/6)*add, y0 = cntr, y1 = 0 - sin(pi/6)*add, 
         col = "darkgrey", lwd = 3, length = 0.2, angle = 45)
  arrows(x0 = 0.5, x1 = 1 + cos(pi/6)*add, y0 = cntr, y1 = 0 - sin(pi/6)*add, 
         col = "darkgrey", lwd = 3, length = 0.2, angle = 45)
  text(x = 0 - cos(pi/6)*add, y = 0 - sin(pi/6)*add, adj = c(1.1, 1.1),
       labels = "Direct Food\nAdditives")
  text(x = 1 + cos(pi/6)*add, y = 0 - sin(pi/6)*add, adj = c(-0.1, 1.1),
       labels = "Food Contact\nSubstances")
  text(x = 0.5, y = cntr + 0.5*sin(pi/3) + add, adj = c(0.5, -1.1),
       labels = "Pesticides")
  
  ly <- c(par()$usr[3] + 1, par()$usr[3] + 1.075)
  lx <- c(par()$usr[1]*0.6, par()$usr[2]*0.6)
  xadj <- diff(lx)/breaks*2
  lx <- lx + xadj
  rect(seq(lx[1], lx[2], length.out = breaks + 1)[-(breaks + 1)],
       ly[1],
       seq(lx[1], lx[2], length.out = breaks + 1)[-1],
       ly[2],
       col = pal(breaks))
  
  lv <- pretty(vrng)
  lv <- lv[lv <= max(vrng) & lv >= min(vrng)]
  xv <- lx[1] + diff(lx)*(lv - vrng[1])/diff(vrng) + xadj/4
  text(xv, ly[2] - 2*diff(ly), formatC(lv),
       xpd = NA, adj = 1, cex = 1)
  text(x = 0.5, y = par()$usr[3] + 0.75, labels = "Mean ToxCast Bioactivity")
  
  if (!is.null(fname)) graphics.off()
  
}