#--------------------------------------------------------------------------------------
#
# run all of the diagnostics functions
#
#--------------------------------------------------------------------------------------
run.diagnostics <- function(do.prep=T) {
	print.current.function()
	if(do.prep) cell.stress.ranges()
	cell.stress.ranges.hm(to.file=T,cex.col=0.1,cex.row=0.8)
	cytotox.hm(T)
	cytotox.boxplot(T)
	cytotox.count.boxplot(T)
	source.Zdist.2(to.file=T,zmode="norm")
	assay.Zdist(to.file=T)

	assay.TWZ(to.file=T)
	assay.Thist(to.file=T)
	assay.TAC50(to.file=T)

	source.Thist(to.file=T)
	bioprocess.Thist(to.file=T)

	chemical.hitdist(to.file=T,target.gene=NA)
	chemical.scatterplot_legend(to.file=T)
	chemical.scatterplot(to.file=T)

	bioprocess.promiscuity(to.file=T,to.file2=T,to.file3=T)
	bioprocess.quadrant_boxplot(to.file=T,tcut=50)
	bioprocess.quadrant_trellis_Z(do.prep=T,to.file=T)
	#structure_category_hm(to.file=T)
	#NR.hm(to.file=T)
	negative.chems()
	#bruns_watson(to.file=T)
  chemical.hit.dist.summary(do.prep=T,to.file=T)
	target.analysis(T)
	cellfree.vs.cellbased(T)
}
#--------------------------------------------------------------------------------------
#
# Get the cytotoxicity ranges
#
#--------------------------------------------------------------------------------------
cell.stress.ranges <- function() {
	print.current.function()

    nchem <- dim(CHEMS)[1]

    result <- as.data.frame(cbind(CHEMS[,"CODE"],CHEMS[,"Name"]))
	result <- result[1:nchem,]
    names(result)[1] <- "CODE"
    names(result)[2] <- "Name"

	bio.list <- c("cytotoxicity BLA","cytotoxicity SRB","proliferation decrease","apoptosis up","ER stress",
	"microtubule up","heat shock","hypoxia","inflammation up","mitochondrial disruption up","oxidative stress up",
	"CCL2","cell cycle up","cellular adhesion up","chemokine up","coagulation up","CYP450","enzyme blocking",
	"estrogen receptor","extracellular matrix up","GPCR","growth factor up","ion channel","oncogene",
	"ppar signaling","PTPN","transporter",
	"AHR","AChE")
	name.list <- c("cytotoxBLA","cytotoxSRB","prolifDown","apoptosis","ERstress","microtubule","heatShock","hypoxia",
	"inflammation","mitochondria","oxidativeStress",   
    "CCL2","cellCycleUp","cellularAdhesionUp","chemokineUp","coagulationUp","CYP450","enzymeBlocking","estrogenReceptor",
    "extracellularMatrixUp","GPCR","growthFactorUp","ionChannel","oncogene","PPAR","PTPN","transporter",
    "AHR","AChE")
    
    
    for(b in 1:length(bio.list)) {
		print(bio.list[b])
		flush.console()
		assay.set <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bio.list[b]),"assay"]
		n <- vector(length=nchem,mode="numeric")
		minval <- vector(length=nchem,mode="numeric")
		maxval <- vector(length=nchem,mode="numeric")
		for(i in 1:nchem) {
			tested <- MAT.tested[i,assay.set]
			hit <- MAT.hitcall[i,assay.set]
			ac50 <- MAT.AC50[i,assay.set]
			hit <- hit[tested==1]
			hit[is.na(hit)] <- 0
			hit[hit<0] <- 0
			ac50 <- ac50[tested==1]
			ac50 <- ac50[hit==1]
			nhit <- sum(hit)
			min.i <- 1000
			max.i <- 1000
			if(nhit>0) {
				min.i <- min(ac50)
				max.i <- max(ac50)
			}
			n[i] <- nhit
			minval[i] <- min.i
			maxval[i] <- max.i
		}
		counter <- dim(result)[2]
		result <- cbind(result,n,minval,maxval)
		names(result)[counter+1] <- paste(name.list[b],".nhit.uM",sep="")
		names(result)[counter+2] <- paste(name.list[b],".min.uM",sep="")
		names(result)[counter+3] <- paste(name.list[b],".max.uM",sep="")
	}
	assay.set <- ASSAY.INFO[,"assay"]
	n <- vector(length=nchem,mode="numeric")
	minval <- vector(length=nchem,mode="numeric")
	maxval <- vector(length=nchem,mode="numeric")

	for(i in 1:nchem) {
		flush.console()
		tested <- MAT.tested[i,assay.set]
		hit <- MAT.hitcall[i,assay.set]
		ac50 <- MAT.AC50[i,assay.set]
		hit <- hit[tested==1]
		hit[is.na(hit)] <- 0
		hit[hit<0] <- 0
		ac50 <- ac50[tested==1]
		ac50 <- ac50[hit==1]
		nhit <- sum(hit)
		min.i <- 1000
		max.i <- 1000
		if(nhit>0) {
			min.i <- min(ac50)
			max.i <- max(ac50)
		}
		n[i] <- nhit
		minval[i] <- min.i
		maxval[i] <- max.i
	}

	counter <- dim(result)[2]
	result <- cbind(result,n,minval,maxval)
	names(result)[counter+1] <- "All.nhit.uM"
	names(result)[counter+2] <- "All.min.uM"
	names(result)[counter+3] <- "All.max.uM"

    outfile <- paste("../input/cell_stress_ranges_",VARMATDATE,".xlsx",sep="")
    write.xlsx(result,file=outfile)
    rownames(result) <- result[,"CODE"]
    CELL.STRESS.RANGES <<- result
}
#--------------------------------------------------------------------------------------
#
# Plot a histogram of the cytotoxicity hit numbers
#
#--------------------------------------------------------------------------------------
cytotox.count.boxplot <- function(to.file=F) {
  print.current.function()
  if(to.file) {
    file <- paste("../plots/cytotox_count_boxplot.pdf",sep="")
    pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
  }
  ctemp <- CHEMS[,c("CODE","Phase_I","Phase_II")]
  x <- ctemp[,"Phase_I"]
  x[is.na(x)] <- 0
  ctemp[,"Phase_I"] <- x
  x <- ctemp[,"Phase_II"]
  x[is.na(x)] <- 0
  ctemp[,"Phase_II"] <- x
  mask <- ctemp[,"Phase_I"]+ctemp[,"Phase_II"]
  code.list <- ctemp[mask>0,"CODE"]
  #code.list <- code.list[is.element(code.list,names(MAT.AC50))]
  nchem <- length(code.list)
  bio.list <- c("cytotoxicity BLA","cytotoxicity SRB","proliferation decrease")
  counts <- NULL  
  groups <- NULL
  assay.set.in <- NULL
  for(b in 1:length(bio.list)) assay.set.in <- c(assay.set.in,ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bio.list[b]),"assay"])
  tested <- MAT.tested[,assay.set.in]
  hit <- MAT.hitcall[,assay.set.in]
  ac50 <- MAT.AC50[,assay.set.in]
  hit[is.na(tested)] <- 0
  hit[tested==0] <- 0
  hit[is.na(hit)] <- 0
  hit[hit<0] <- 0
  nhit.in <- rowSums(hit)
  
  assay.set.out <- ASSAY.INFO[!is.element(ASSAY.INFO[,"biological_process"],bio.list[b]),"assay"]
  tested <- MAT.tested[,assay.set.out]
  hit <- MAT.hitcall[,assay.set.out]
  ac50 <- MAT.AC50[,assay.set.out]
  hit[is.na(tested)] <- 0
  hit[tested==0] <- 0
  hit[is.na(hit)] <- 0
  hit[hit<0] <- 0
  nhit.out <- rowSums(hit)
  
  
  boxplot(nhit.out~nhit.in,main="Counts of Hits vs. #Cytotoxicity Hits",cex.lab=1.2,cex.axis=1.2,xlab="Cytotox Hits",ylab="Non-cytotox Hits")
  if(to.file) dev.off()
  else browser()
}
#--------------------------------------------------------------------------------------
#
# do the heatmap of the cell stress ranges
#
# QC=OK
#--------------------------------------------------------------------------------------
cell.stress.ranges.hm <- function(to.file=T,cex.col=0.1,cex.row=0.8) {
	print.current.function()
  if(to.file) {
    file <- paste("../plots/cell_stress_ranges_hm.pdf",sep="")
    pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
  }

  col.list <- c("cytotoxSRB.min.uM","prolifDown.min.uM","apoptosis.min.uM","ERstress.min.uM","microtubule.min.uM","heatShock.min.uM","hypoxia.min.uM","inflammation.min.uM","mitochondria.min.uM","oxidativeStress.min.uM","All.min.uM")
  #print(col.list[!is.element(col.list,names(CELL.STRESS.RANGES))]  )
  #browser()
  temp <- CELL.STRESS.RANGES[,col.list]
  temp <- temp[1:100,]
  mat <- t(as.matrix(temp))
  mat <- -log10(mat/1000)
  mat[mat>6] <- 6
  result <- heatmap(mat,margins=c(10,10),scale="none",main="Cell Stress Ranges (min(AC50/log))",
                    xlab="",ylab="",cexCol=cex.col,cexRow=cex.row,col=brewer.pal(9,"Reds"),
                    hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F)
  if(to.file) dev.off()
  else browser()
}
#--------------------------------------------------------------------------------------
#
# do the heatmap of the cytotox assays
#
# QC=OK
#--------------------------------------------------------------------------------------
cytotox.hm <- function(to.file=T,cex.col=0.1,cex.row=0.8) {
	print.current.function()
    class.list <- c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease")
    cytotox.assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],class.list),"assay"]
    CYTOTOX.ASSAYS <<- cytotox.assays

    if(to.file) {
        file <- paste("../plots/cytotox_heatmap.pdf",sep="")
        pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    assay.list <- CYTOTOX.ASSAYS
    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))
    code.list <- code.list[!is.na(code.list)]

    rowcolors <- assay.list
    rowcolors[] <- "white"
    for(i in 1:length(assay.list)) {
        assay <- assay.list[i]
        asource <- ASSAY.INFO[assay,"biological_process"]
        if(asource=="cytotoxicity BLA") rowcolors[i] <- "red"
        if(asource=="cytotoxicity SRB") rowcolors[i] <- "blue"
        if(asource=="proliferation decrease") rowcolors[i] <- "white"
    }
    mat <- t(as.matrix(MAT.AC50[code.list,assay.list]))
    mat <- -log10(mat/1000000)
    mat[is.na(mat)] <- 0
    result <- heatmap(mat,margins=c(5,15),scale="none",main=paste(dim(mat)[2]," chemicals, ",dim(mat)[1]," assays"),
                      xlab="Chemicals",ylab="Assays",cexCol=cex.col,cexRow=cex.row,col=brewer.pal(9,"Reds"),
                      hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
                      RowSideColors=rowcolors)

    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# Assay source summary table
# zmode=original or norm
# QC=OK
#--------------------------------------------------------------------------------------
source.Zdist <- function(to.file=F,zmode="norm") {
	print.current.function()
    if(to.file) {
        fname <- paste("../plots/source_Zdist_",zmode,".pdf",sep="")
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)

    if(zmode=="original") z <- MAT.Z
    else if(zmode=="norm") z <- MAT.Z.NORM
    z[MAT.hitcall==0] <- NA
    z[MAT.tested==0] <- NA
    z[MAT.Z==0] <- NA

    ctemp <- MAT.tested
    chem.mask <- rowSums(ctemp)
    chem.mask[chem.mask<200] <- 0
    chem.mask[chem.mask>0] <- 1

    resmat <- as.data.frame(matrix(nrow=nsource,ncol=3))
    names(resmat) <- c("source_group","Center1","Center2")
    for(i in 1:nsource) {
        source <- source.list[i]
        assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
        assay.list <- assay.list[is.element(assay.list,colnames(MAT.AC50))]
        cat(source," : ",length(assay.list),"\n")
        flush.console()
        if(length(assay.list)>1) {
            ztemp <- z[,assay.list]
            atemp <- MAT.AC50[,assay.list]
            if(length(assay.list)==1) {
                atemp <- atemp[chem.mask==1]
                ztemp <- ztemp[chem.mask==1]
            }
            else {
                atemp <- atemp[chem.mask==1,]
                ztemp <- ztemp[chem.mask==1,]
                print("fix conversion")
                atemp <- as.numeric(as.matrix(atemp))
                ztemp <- as.numeric(as.matrix(ztemp))
            }
            xmax <- 15
            xmin <- -5
            ztemp <- ztemp[!is.na(ztemp)]
            ztemp <- ztemp[ztemp> xmin]
            ztemp <- ztemp[ztemp< xmax]
            if(length(ztemp)>10) {
				breaks <- seq(-6,xmax,by=0.2)
				x <- hist(ztemp,xlim=c(xmin,xmax),main=paste("Z distribution for",source),cex.lab=1.2,cex.axis=1.2,ylab="Chemical/Assay Hits",xlab="Z-score",breaks=breaks,freq=T)

				center1 <- 0
				center2 <- 0
				mymix <- cbind(x$breaks[1:length(x$counts)],x$counts)
				fit <- mix(mymix,mixparam(mu=c(0,xmax),sigma=c(2,2)),"norm")
				center1 <- fit[[1]][1,2]
				center2 <- fit[[1]][2,2]
				ymax <- max(x$counts)
				print(fit)
				if(zmode=="original") lines(c(center1,center1),c(0,ymax/5),lwd=3,col="red")
				if(zmode=="norm") lines(c(0,0),c(0,ymax),lwd=1,col="red")
				lines(c(3,3),c(0,ymax),lwd=2,col="red")
				lines(c(0,0),c(0,ymax),lwd=2,col="red")
											#lines(c(center2,center2),c(0,ymax/5),lwd=3,col="red")
				if(zmode=="original") text(7,ymax*0.9,paste("Peak 1:",format(center1,digits=2)),pos=4)
											#text(-6,ymax*0.7,paste("C2:",format(center2,digits=2)),pos=4)
				resmat[i,1] <- source
				resmat[i,2] <- center1
				resmat[i,3] <- center2
				eps <- 0.1
				xmin <- 1e-4
				xmax <- 1e4
				atemp <- atemp[!is.na(atemp)]
				atemp <- atemp[atemp<xmax]
				atemp <- atemp[atemp>xmin]

				breaks <- seq(0,xmax,by=0.2)
				breaks.min <- 1e-4
				nbreaks <- 100
				if(min(atemp)<breaks.min) {
					breaks.min <- breaks.min/100
				}
				if(min(atemp)<breaks.min) {
					breaks.min <- breaks.min/100
				}
				breaksA <- breaks.min
				for(i in 1:100) breaksA <- c(breaksA,1.2*breaksA[length(breaksA)])
				while(max(breaksA)<max(atemp)) {
					breaksA <- c(breaksA,1.2*breaksA[length(breaksA)])
					cat("Added another point to breaksA",max(breaksA),"\n")
				}
				xA <- hist(atemp,breaks=breaksA,plot=F)
				ymax <- 1.5*max(xA$counts)
				hist.log(breaksA,xA$counts,ylim=c(0,ymax),xlab="AC50 (uM)",ylab="Hits",main=source,1000000,1000000,1000000)
				if(!to.file) browser()
			}
        }
    }
    if(to.file) graphics.off()
    else browser()
    outfile <- paste("../output/source_Zshifts_",zmode,".txt",sep="")
    write.table(resmat,file=outfile, row.names=F, append=FALSE, quote=F, sep = "\t")
}
#--------------------------------------------------------------------------------------
#
# Assay source summary table
# zmode=original or norm
# QC=OK
#--------------------------------------------------------------------------------------
source.Zdist.2 <- function(to.file=F) {
  print.current.function()
  if(to.file) {
    fname <- paste("../plots/source_Zdist_2.pdf",sep="")
    pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
  }
  par(mfrow=c(4,2),mar=c(4,4,2,2))
  
  source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
  nsource <- length(source.list)
  
  z <- MAT.Z.NORM
  z[MAT.hitcall==0] <- NA
  z[MAT.tested==0] <- NA
  z[MAT.Z==0] <- NA
  
  ctemp <- CHEMS[,c("CODE","Phase_I","Phase_II")]
  mask <- ctemp[,"Phase_I"]+ctemp[,"Phase_II"]
  code.list <- ctemp[mask>0,"CODE"]
  
  code.cytotox.no <- CYTOTOX[CYTOTOX[,"nhit"]<2,"CODE"]
  code.cytotox.yes <- CYTOTOX[CYTOTOX[,"nhit"]>=2,"CODE"]
  
  code.cytotox.no <- code.cytotox.no[is.element(code.cytotox.no,code.list)]
  code.cytotox.yes <- code.cytotox.yes[is.element(code.cytotox.yes,code.list)]
  cat("chems no cytotox:",length(code.cytotox.no),"\n")
  cat("chems yes cytotox:",length(code.cytotox.yes),"\n")
  
  xmax <- 15
  xmin <- -5
  for(i in 1:nsource) {
    source <- source.list[i]
    assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
    assay.list <- assay.list[is.element(assay.list,colnames(MAT.AC50))]
    cat(source," : ",length(assay.list),"\n")
    flush.console()
    if(length(assay.list)>1) {
      ztemp <- z[code.cytotox.yes,assay.list]
      ztemp <- as.numeric(as.matrix(ztemp))
      ztemp <- ztemp[!is.na(ztemp)]
      ztemp <- ztemp[ztemp> xmin]
      ztemp <- ztemp[ztemp< xmax]
      breaks <- seq(-6,xmax,by=0.2)
      x <- hist(ztemp,xlim=c(xmin,xmax),main=paste("Z distribution for",source),cex.lab=1.2,cex.axis=1.2,ylab="Chemical/Assay Hits",xlab="Z-score",breaks=breaks,freq=T)
      
      center1 <- 0
      center2 <- 0
      mymix <- cbind(x$breaks[1:length(x$counts)],x$counts)
      fit <- mix(mymix,mixparam(mu=c(0,xmax),sigma=c(2,2)),"norm")
      center1 <- fit[[1]][1,2]
      center2 <- fit[[1]][2,2]
      ymax <- max(x$counts)
      lines(c(3,3),c(0,ymax),lwd=2,col="red")
      lines(c(0,0),c(0,ymax),lwd=2,col="red")
      
      ztemp <- z[code.cytotox.no,assay.list]
      ztemp <- as.numeric(as.matrix(ztemp))
      ztemp <- ztemp[!is.na(ztemp)]
      ztemp <- ztemp[ztemp> xmin]
      ztemp <- ztemp[ztemp< xmax]
      breaks <- seq(-6,xmax,by=0.2)
      x <- hist(ztemp,xlim=c(xmin,xmax),ylim=c(0,ymax),main=paste("Z distribution for",source),cex.lab=1.2,cex.axis=1.2,ylab="Chemical/Assay Hits",xlab="Z-score",breaks=breaks,freq=T)
      lines(c(3,3),c(0,ymax),lwd=2,col="red")
      lines(c(0,0),c(0,ymax),lwd=2,col="red")
      
      if(!to.file) browser()
    }
  }
  if(to.file) graphics.off()
  else browser()
}
#--------------------------------------------------------------------------------------
#
# Assay summary table
# mode=ToxCast or E1K
# QC=OK
#--------------------------------------------------------------------------------------
assay.Zdist <- function(to.file=F) {
	print.current.function()
   	if(to.file) {
        fname <- "../plots/assay_Zdist.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    z <- MAT.Z.NORM
    z[MAT.hitcall==0] <- NA
    z[MAT.Z==0] <- NA

    nassay <- length(ASSAY.LIST)
    ztemp <- as.vector(z)
    zdist <- ztemp[!is.na(ztemp)]
    ctemp <- MAT.hitcall
    ctemp[!is.na(MAT.hitcall)] <- 1
    ctemp[is.na(MAT.hitcall)] <- 0
    chem.mask <- rowSums(ctemp)
    chem.mask[chem.mask<200] <- 0
    chem.mask[chem.mask>0] <- 1

    for(i in 1:nassay) {
        assay <- ASSAY.LIST[i]

        source <- ASSAY.INFO[is.element(ASSAY.INFO[,"assay"],assay),"source_group"][1]
        print(assay)
        flush.console()
        if(is.na(source)) {
            cat("No source: ",assay,"\n")
            browser()
        }
        atemp <- MAT.AC50[,assay]
        atemp.disc <- MAT.hitcall[,assay]
        ztemp <- z[,assay]
        atemp <- atemp[chem.mask==1]
        atemp.disc <- atemp.disc[chem.mask==1]
        ztemp <- ztemp[chem.mask==1]

        test.mask <- atemp.disc
        test.mask[!is.na(test.mask)] <- 1
        test.mask[is.na(test.mask)] <- 0
        ntest <- sum(test.mask)

        hit.mask <- atemp.disc[test.mask==1]
        nhit <- sum(hit.mask)

        za <- z[,assay]
        mask <- za
        mask[] <- 1
        mask[is.na(za)] <- 0
        mask[za<3] <- 0

        if(nhit>0) {
            xmax <- 6
            xmin <- -6
            ztemp <- ztemp[!is.na(ztemp)]
            ztemp <- ztemp[ztemp> -5]
            ztemp <- ztemp[ztemp< 8]
            if(length(ztemp)>0) {
                if(max(ztemp)>6) xmax <- 8
                breaks <- seq(-6,xmax,by=0.2)
                x <- hist(ztemp,xlim=c(xmin,xmax),main=assay,cex.lab=1.2,cex.axis=1.2,ylab="Chemicals",xlab="Z-score",breaks=breaks,freq=T)
                ymax <- max(x$counts)
                eps <- 0.1
                text(-6,ymax*(1-1*eps),paste("ntry=",ntest,sep=""),pos=4)
                text(-6,ymax*(1-2*eps),paste("nhit=",nhit,sep=""),pos=4)
                lines(c(0,0),c(0,ymax/4),lwd=2,col="red")
                if(!to.file) browser()
            }
        }
    }
    if(to.file) graphics.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot a histogram on a log scale Calculate at the hit distribution by chemical
#
# QC=OK
#--------------------------------------------------------------------------------------
hist.log <- function(x,y,ylim,xlab,ylab,main,cytotox.median, cytotox.min, cytotox.max) {
	print.current.function()
	plot(x[1:length(y)],y,type="n",col="gray40",lwd=2.5,log="x",xlab=xlab,ylim=ylim,ylab=ylab,main=main,cex.lab=1.2,cex.axis=1.2)
	if(cytotox.min<100) {
            rect(cytotox.min,ylim[2],max(x),0,col="gray80")
            lines(c(cytotox.median,cytotox.median),ylim,col="red",lwd=3)
	}
	for(i in 1:length(y)) {
            rect(x[i],y[i],x[i+1],0)
	}
}
#--------------------------------------------------------------------------------------
#
# plot T, W by Z
#
#--------------------------------------------------------------------------------------
assay.TWZ <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/assay_TWZ.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		nassay <- length(assay.list)
		for(j in 1:nassay) {
			assay <- assay.list[j]
			W <- MAT.W[,assay]
			T <- MAT.T.SCALED[,assay]
			Z <- MAT.Z.NORM[,assay]
			A <- MAT.AC50[,assay]
			H <- MAT.hitcall[,assay]
			tested <- MAT.tested[,assay]

			W <- W[tested==1]
			T <- T[tested==1]
			Z <- Z[tested==1]
			A <- A[tested==1]
			H <- H[tested==1]

			W <- W[H==1]
			T <- T[H==1]
			Z <- Z[H==1]
			A <- A[H==1]
			H <- H[H==1]

			W <- W[A<50]
			T <- T[A<50]
			Z <- Z[A<50]
			H <- H[A<50]
			A <- A[A<50]

			W <- W[!is.na(T)]
			Z <- Z[!is.na(T)]
			H <- H[!is.na(T)]
			A <- A[!is.na(T)]
			T <- T[!is.na(T)]

			if(length(T)>=5) {
				tmax <- max(T)

				x <- quantile(T,probs=seq(0,1,0.1))
				tcut <- as.numeric(x[10])

				if(tmax<50) {
					t.scaled <- (T-1)/tcut
					zt <- Z
					zt[] <- 0
					zt[Z>3] <- 1
					zt[t.scaled>0.2] <- 1
					T2 <- T[zt==0]
					Z2 <- Z[zt==0]
				}
				else {
					zt <- Z
					zt[] <- 0
					zt[Z>3] <- 1
					zt[T>50] <- 1
					T2 <- T[zt==0]
					Z2 <- Z[zt==0]
					tmax <- 150
				}

				plot(T~Z,xlim=c(-5,15),ylim=c(0,tmax),xlab="Z",ylab="T",cex.lab=1.2,cex.axis=1.2,main=assay)
				points(T2~Z2,pch=21,fg="red",col="red")
				ntot <- length(Z)
				nscl <- length(Z2)
				frac <- nscl/ntot
				text(x=7,y=tmax*0.9,paste("Points:",ntot),pos=4)
				text(x=7,y=tmax*0.8,paste("Lo TZ :",nscl," : ",format(frac,digits=2)),pos=4)

				plot(W~Z,xlim=c(-5,15),ylim=c(0,8),xlab="Z",ylab="W",cex.lab=1.2,cex.axis=1.2,main=assay)
				if(!to.file) browser()
			}
		}
	}
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot T histograms
#
#--------------------------------------------------------------------------------------
assay.Thist <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/assay_Thist.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		nassay <- length(assay.list)
		for(j in 1:nassay) {
			assay <- assay.list[j]
			W <- MAT.W[,assay]
			T <- MAT.T.SCALED[,assay]
			Z <- MAT.Z.NORM[,assay]
			A <- MAT.AC50[,assay]
			H <- MAT.hitcall[,assay]
			tested <- MAT.tested[,assay]

			W <- W[tested==1]
			T <- T[tested==1]
			Z <- Z[tested==1]
			A <- A[tested==1]
			H <- H[tested==1]

			W <- W[H==1]
			T <- T[H==1]
			Z <- Z[H==1]
			A <- A[H==1]
			H <- H[H==1]

			W <- W[A<50]
			T <- T[A<50]
			Z <- Z[A<50]
			H <- H[A<50]
			A <- A[A<50]

			W <- W[!is.na(T)]
			Z <- Z[!is.na(T)]
			H <- H[!is.na(T)]
			A <- A[!is.na(T)]
			T <- T[!is.na(T)]

			if(length(T)>=5) {
				hist(T,main=assay)

				if(!to.file) browser()
			}
		}
	}
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot T histograms
#
#--------------------------------------------------------------------------------------
source.Thist <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/source_Thist.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		nassay <- length(assay.list)
		T.source <- NULL
		for(j in 1:nassay) {
			assay <- assay.list[j]
			W <- MAT.W[,assay]
			T <- MAT.T.SCALED[,assay]
			Z <- MAT.Z.NORM[,assay]
			A <- MAT.AC50[,assay]
			H <- MAT.hitcall[,assay]
			tested <- MAT.tested[,assay]

			W <- W[tested==1]
			T <- T[tested==1]
			Z <- Z[tested==1]
			A <- A[tested==1]
			H <- H[tested==1]

			W <- W[H==1]
			T <- T[H==1]
			Z <- Z[H==1]
			A <- A[H==1]
			H <- H[H==1]

			W <- W[A<50]
			T <- T[A<50]
			Z <- Z[A<50]
			H <- H[A<50]
			A <- A[A<50]

			W <- W[!is.na(T)]
			Z <- Z[!is.na(T)]
			H <- H[!is.na(T)]
			A <- A[!is.na(T)]
			T <- T[!is.na(T)]

			if(length(T)>=5) {
				T.source <- c(T.source,T)
			}
		}
		if(length(T.source)>=5) {
			T.source[T.source>150] <- 150
			hist(T.source,main=source)
			if(!to.file) browser()
		}
	}
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot T histograms
#
#--------------------------------------------------------------------------------------
bioprocess.Thist <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/bioprocess_Thist.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"biological_process"]))
    nsource <- length(source.list)

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],source),"assay"]
		nassay <- length(assay.list)
		T.source <- NULL
		for(j in 1:nassay) {
			assay <- assay.list[j]
			W <- MAT.W[,assay]
			T <- MAT.T.SCALED[,assay]
			Z <- MAT.Z.NORM[,assay]
			A <- MAT.AC50[,assay]
			H <- MAT.hitcall[,assay]
			tested <- MAT.tested[,assay]

			W <- W[tested==1]
			T <- T[tested==1]
			Z <- Z[tested==1]
			A <- A[tested==1]
			H <- H[tested==1]

			W <- W[H==1]
			T <- T[H==1]
			Z <- Z[H==1]
			A <- A[H==1]
			H <- H[H==1]

			W <- W[A<50]
			T <- T[A<50]
			Z <- Z[A<50]
			H <- H[A<50]
			A <- A[A<50]

			W <- W[!is.na(T)]
			Z <- Z[!is.na(T)]
			H <- H[!is.na(T)]
			A <- A[!is.na(T)]
			T <- T[!is.na(T)]

			if(length(T)>=5) {
				T.source <- c(T.source,T)
			}
		}
		if(length(T.source)>=5) {
			T.source[T.source>150] <- 150
			hist(T.source,main=source)
			if(!to.file) browser()
		}
	}
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot T vs. AC50
#
#--------------------------------------------------------------------------------------
assay.TAC50 <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/assay_TAC50.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(4,2),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		nassay <- length(assay.list)
		for(j in 1:nassay) {
			assay <- assay.list[j]
			W <- MAT.W[,assay]
			T <- MAT.T.SCALED[,assay]
			Z <- MAT.Z.NORM[,assay]
			A <- MAT.AC50[,assay]
			H <- MAT.hitcall[,assay]
			tested <- MAT.tested[,assay]

			W <- W[tested==1]
			T <- T[tested==1]
			Z <- Z[tested==1]
			A <- A[tested==1]
			H <- H[tested==1]

			W <- W[H==1]
			T <- T[H==1]
			Z <- Z[H==1]
			A <- A[H==1]
			H <- H[H==1]

			W <- W[!is.na(T)]
			Z <- Z[!is.na(T)]
			H <- H[!is.na(T)]
			A <- A[!is.na(T)]
			T <- T[!is.na(T)]

			if(length(T)>=5) {
				tmax <- 150
				if(max(T)<50) tmax <- max(T)

				plot(T~A,xlim=c(1E-4,100),ylim=c(0,tmax),log="x",xlab="AC50 (uM)",ylab="T",cex.lab=1.2,cex.axis=1.2,main=assay)
				if(!to.file) browser()
			}
		}
	}
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot the number of low z hit in cell-free vs. cell-based
#
#--------------------------------------------------------------------------------------
cellfree.vs.cellbased <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/cellfree_vs_cellbased.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(1,1),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)


	fname <- paste("../physchem/toxcast_physchem_cellstress_DFT.xlsx",sep="")
	pchem <- read.xlsx(fname)
	pcode <- pchem[,"CODE"]	
	rownames(pchem) <- pchem[,"CODE"]
	pchem.names <- names(pchem)[29:dim(pchem)[2]]
	npchem <- length(pchem.names)
	code.list <- CODE.LIST[is.element(CODE.LIST,pcode)]

	pchem.list <- c("SASA","volume","mol.MW","QPlogPo.w","Marvin.logD.pH7.3","DFT.EA","DFT.softness","DFT.hardness","DFT.polarizability")
	pchem.color <- pchem.list
	pchem.color[] <- "purple"
	pchem1 <- pchem[code.list,pchem.list]
	np <- length(pchem.list)
	for(i in 1:np) {
		varname <- pchem.list[i]
		temp <- pchem1[,varname]
		if(varname=="DFT.hardness") temp <- -temp
		temp1 <- temp
		temp1[] <- 0
		cutoff <- as.numeric(quantile(temp, probs=seq(0,1,0.1),na.rm=T)[10])
		temp1[temp>cutoff] <- 1
		pchem1[,pchem.list[i]] <- temp1
	}

	Z <- MAT.Z.NORM
	H <- MAT.hitcall
	tested <- MAT.tested
	rs <- rowSums(tested)
	Z <- Z[rs>500,]
	H <- H[rs>500,]
	tested <- tested[rs>500,]
	
	Z[tested==0] <- NA
	Z[H==0] <- NA

	Z[is.na(Z)] <- 1000000
	Z[Z>3] <- 1000000
	Z[Z<=3] <- 1
	Z[Z>10000] <- 0
	rs <- rowSums(Z)
	cs <- colSums(Z)
	Z <- Z[rs>0,cs>0]
	
	assay.list.cellbased <- NULL
	assay.list.cellfree <- NULL

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		if(is.element(source,c("NVS_ADME","NVS_ADME_Activator","NVS_ENZ","NVS_ENZ_Activator","NVS_GPCR","NVS_IC","NVS_MP","NVS_NR","NVS_TR"))) {
			assay.list.cellfree <- c(assay.list.cellfree,assay.list)		
		}
		else {
			assay.list.cellbased <- c(assay.list.cellbased,assay.list)
		}
	}
	assay.list.cellfree <- assay.list.cellfree[is.element(assay.list.cellfree,names(Z))]
	assay.list.cellbased <- assay.list.cellbased[is.element(assay.list.cellbased,names(Z))]
	Z.cellfree <- Z[,assay.list.cellfree]
	Z.cellbased <- Z[,assay.list.cellbased]
	n.cellbased <- rowSums(Z.cellbased)
	n.cellfree <- rowSums(Z.cellfree)
	
	nx <- dim(Z.cellfree)[1]
	ny <- dim(Z.cellbased)[1]
	f.cellbased <- as.double(n.cellbased) / nx
	f.cellfree <- as.double(n.cellfree) / ny
	
	plot(f.cellbased~f.cellfree,xlab="Fraction Hits Z<3 Cell-free",ylab="Fraction Hits Z<3 Cell-based",
			xlim=c(0,0.3),ylim=c(0,0.3),cex.lab=1.1,cex.axis=1.1)
	lines(c(0,160),c(0,160))
	text(-5,180,paste("Cell-free assays: ",dim(Z.cellfree)[2]),pos=4)
	text(-5,174,paste("Cell-based assays: ",dim(Z.cellbased)[2]),pos=4)

	x <- 100/nx
	x <- 0.22
	y <- 170/nx
	y <- 0.05
	dy <- 5/nx
	points(x,y,bg="red",pch=21,cex=1.5); text(x,y,"Surfactant",pos=4);y <- y-dy
	points(x,y,bg="blue",pch=21,cex=1.5); text(x,y,"Microbicide",pos=4);y <- y-dy
	points(x,y,bg="white",pch=21,cex=1.5); text(x,y,"Pharmaceutical",pos=4);y <- y-dy
	points(x,y,bg="cyan",pch=21,cex=1.5); text(x,y,"Pesticide",pos=4);y <- y-dy
	points(x,y,bg="yellow",pch=21,cex=1.5); text(x,y,"Herbicide",pos=4);y <- y-dy
	points(x,y,bg="gray",pch=21,cex=1.5); text(x,y,"Other",pos=4);y <- y-dy

	for(i in 1:length(n.cellbased)) {
		code <- rownames(Z.cellfree)[i]
		cname <- CHEMS[code,"Name"]
		supercat <- CHEMS[code,"use_super_category"]
		x <- f.cellfree[i]
		y <- f.cellbased[i]
		pch <- 21
		val1 <- sum(pchem1[code,c("SASA","volume","mol.MW")])
		val2 <- sum(pchem1[code,c("QPlogPo.w","Marvin.logD.pH7.3")])
		val3 <- sum(pchem1[code,c("DFT.EA","DFT.softness","DFT.hardness","DFT.polarizability")])
		if(supercat=="Surfactant") points(x,y,bg="red",pch=pch,cex=1.5)
		if(supercat=="Microbicide") points(x,y,bg="blue",pch=pch,cex=1.5)
		if(supercat=="Pharmaceutical") points(x,y,bg="white",pch=pch,cex=1.5)
		if(supercat=="Pesticide") points(x,y,bg="cyan",pch=pch,cex=1.5)
		if(supercat=="Herbicide") points(x,y,bg="yellow",pch=pch,cex=1.5)
		if(supercat=="Other") points(x,y,bg="gray",pch=pch,cex=1.5)
		doit <- F
		if(x>0.14) doit <- T
		if(x>y && y>0.05) doit <- T
		if(y<0.05 && x>(y*1.5) && supercat=="Pharmaceutical") doit <- T
		if(doit) text(x,y,cname,pos=4,cex=0.8)	
	}
	
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot the number of low z hit in cell-free vs. cell-based
#
#--------------------------------------------------------------------------------------
cellfree.vs.cellbased.target <- function(to.file=F,do.prep=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/cellfree_vs_cellbased_target.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(1,1),mar=c(4,4,2,2))

	if(do.prep) {
		fname <- "../physchem/toxcast_physchem_cellstress_DFT.xlsx"
		physchem <- read.xlsx(fname)
		rownames(physchem) <- physchem[,"CODE"]
		PHYSCHEM <<- physchem
	}
    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)
	
	Z <- MAT.Z.NORM
	H <- MAT.hitcall
	tested <- MAT.tested
	rs <- rowSums(tested)
	Z <- Z[rs>500,]
	H <- H[rs>500,]
	tested <- tested[rs>500,]
	
	Z[tested==0] <- NA
	Z[H==0] <- NA

	Z[is.na(Z)] <- 1000000
	Z[Z>3] <- 1000000
	Z[Z<=3] <- 1
	Z[Z>10000] <- 0
	rs <- rowSums(Z)
	cs <- colSums(Z)
	Z <- Z[rs>0,cs>0]
	
	assay.list.cellbased <- NULL
	assay.list.cellfree <- NULL

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		if(is.element(source,c("NVS_ADME","NVS_ADME_Activator","NVS_ENZ","NVS_ENZ_Activator","NVS_GPCR","NVS_IC","NVS_MP","NVS_NR","NVS_TR"))) {
			assay.list.cellfree <- c(assay.list.cellfree,assay.list)		
		}
		else {
			assay.list.cellbased <- c(assay.list.cellbased,assay.list)
		}
	}
	assay.list.cellfree <- assay.list.cellfree[is.element(assay.list.cellfree,names(Z))]
	assay.list.cellbased <- assay.list.cellbased[is.element(assay.list.cellbased,names(Z))]
	Z.cellfree <- Z[,assay.list.cellfree]
	Z.cellbased <- Z[,assay.list.cellbased]
	n.cellbased <- rowSums(Z.cellbased)
	n.cellfree <- rowSums(Z.cellfree)
	
	plot(n.cellbased~n.cellfree,xlab="Hits Z<3 Cell-free",ylab="Hits Z<3 Cell-based",
			xlim=c(0,180),ylim=c(0,180),cex.lab=1.1,cex.axis=1.1)
	lines(c(0,160),c(0,160))
	text(-5,180,paste("Cell-free assays: ",dim(Z.cellfree)[2]),pos=4)
	text(-5,174,paste("Cell-based assays: ",dim(Z.cellbased)[2]),pos=4)

	x <- 100
	y <- 180
	dy <- 5
	points(x,y,bg="red",pch=21,cex=1.5); text(x,y,"Surfactant",pos=4);y <- y-dy
	points(x,y,bg="blue",pch=21,cex=1.5); text(x,y,"Microbicide",pos=4);y <- y-dy
	points(x,y,bg="white",pch=21,cex=1.5); text(x,y,"Pharmaceutical",pos=4);y <- y-dy
	points(x,y,bg="cyan",pch=21,cex=1.5); text(x,y,"Pesticide",pos=4);y <- y-dy
	points(x,y,bg="yellow",pch=21,cex=1.5); text(x,y,"Herbicide",pos=4);y <- y-dy
	points(x,y,bg="gray",pch=21,cex=1.5); text(x,y,"Other",pos=4);y <- y-dy
	
	col.list <- c("gray","red","black")
	process.list <- c("oxidative stress up","mitochondrial disruption up","microtubule up")
	nprocess <- 5
	for(i in 1:length(n.cellbased)) {
		code <- rownames(Z.cellfree)[i]
		cname <- CHEMS[code,"Name"]
		x <- n.cellfree[i]
		y <- n.cellbased[i]
		for(j in 1:nprocess) {
			bp <- process.list[j]
			assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
			assay.list <- assay.list[is.element(assay.list,names(Z.cellbased))]
			temp <- Z.cellbased[code,assay.list]
			if(length(temp)>1) if(sum(temp)>1) points(x,y,bg=col.list[j],pch=21,cex=1.5)
		}
		logp <- PHYSCHEM[code,"QPlogPo.w"]
		if(!is.na(logp)) if(logp>6) points(x,y,bg="orange",pch=25,cex=2)
		if(!is.na(logp)) if(logp<1) points(x,y,bg="orange",pch=24,cex=2)
		ei <- PHYSCHEM[code,"DFT.electrophilicity"]
		if(!is.na(ei)) if(ei>10) points(x,y,bg="pink",pch=25,cex=2)
		if(!is.na(ei)) if(ei<1) points(x,y,bg="pink",pch=24,cex=2)
	}
	
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot the number of low z hit in cell-free vs. cell-based
#
#--------------------------------------------------------------------------------------
cellfree.vs.cellbased.hm <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/cellfree_vs_cellbased_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(1,1),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)
	
	Z <- MAT.Z.NORM
	H <- MAT.hitcall
	tested <- MAT.tested
	rs <- rowSums(tested)
	Z <- Z[rs>500,]
	H <- H[rs>500,]
	tested <- tested[rs>500,]
	
	Z[tested==0] <- NA
	Z[H==0] <- NA

	Z[is.na(Z)] <- 1000000
	Z[Z>3] <- 1000000
	Z[Z<=3] <- 1
	Z[Z>10000] <- 0
	rs <- rowSums(Z)
	cs <- colSums(Z)
	Z <- Z[rs>0,cs>0]
	
	assay.list.cellbased <- NULL
	assay.list.cellfree <- NULL

	for(i in 1:nsource) {
		source <- source.list[i]
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],source),"assay"]
		if(is.element(source,c("NVS_ADME","NVS_ADME_Activator","NVS_ENZ","NVS_ENZ_Activator","NVS_GPCR","NVS_IC","NVS_MP","NVS_NR","NVS_TR"))) {
			assay.list.cellfree <- c(assay.list.cellfree,assay.list)		
		}
		else {
			assay.list.cellbased <- c(assay.list.cellbased,assay.list)
		}
	}
	assay.list.cellfree <- assay.list.cellfree[is.element(assay.list.cellfree,names(Z))]
	assay.list.cellbased <- assay.list.cellbased[is.element(assay.list.cellbased,names(Z))]
	Z.cellfree <- Z[,assay.list.cellfree]
	Z.cellbased <- Z[,assay.list.cellbased]
	n.cellbased <- rowSums(Z.cellbased)
	n.cellfree <- rowSums(Z.cellfree)
	
	cs <- colSums(Z.cellfree)
	Z.cellfree <- Z.cellfree[,cs>49]
	rs <- rowSums(Z.cellfree)
	Z.cellfree <- Z.cellfree[rs>9,]
	name.list <- names(Z.cellfree)
	col.list <- name.list
	col.list[] <- "white"
	for(i in 1:length(name.list)) {
		nm <- name.list[i]
		if(sum(grep("GPCR",nm))>0) col.list[i] <- "red"
		if(sum(grep("_NR_",nm))>0) col.list[i] <- "black"
		if(sum(grep("ADME",nm))>0) col.list[i] <- "cyan"
	}
	name.list <- rownames(Z.cellfree)
	for(i in 1:length(name.list)) {
		code <- name.list[i]
		name.list[i] <- CHEMS[code,"Name"]
		if(sum(grep("Milbemectin",name.list[i]))>0) name.list[i] <- "Milbemectin"		
	}

	chem.list <- rownames(Z.cellfree)
	col2.list <- chem.list
	col2.list[] <- "white"
	for(i in 1:length(name.list)) {
		code <- chem.list[i]
		supercat <- CHEMS[code,"use_super_category"]
		if(supercat=="Surfactant") col2.list[i] <- "red"
		if(supercat=="Microbicide") col2.list[i] <- "blue"
		if(supercat=="Pharmaceutical") col2.list[i] <- "white"
		if(supercat=="Pesticide") col2.list[i] <- "cyan"
		if(supercat=="Herbicide") col2.list[i] <- "yellow"
		if(supercat=="Other") col2.list[i] <- "gray"
	}
	
	result <- heatmap(t(Z.cellfree),margins=c(10,10),scale="none",main="Cell Free",
					  xlab="",ylab="",cexCol=0.8,cexRow=0.6,col=brewer.pal(9,"Reds"),
					  hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
					  RowSideColors=col.list,ColSideColors=col2.list,labCol=name.list)

    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot the number of low z hit in cell-free vs. cell-based
#
#--------------------------------------------------------------------------------------
electrophil.hm <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/electrophil_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(1,1),mar=c(4,4,2,2))

    source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
    nsource <- length(source.list)
	
	Z <- MAT.Z.NORM
	H <- MAT.hitcall
	tested <- MAT.tested
	rs <- rowSums(tested)
	Z <- Z[rs>500,]
	H <- H[rs>500,]
	tested <- tested[rs>500,]
	
	Z[tested==0] <- NA
	Z[H==0] <- NA

	Z[is.na(Z)] <- 1000000
	Z[Z>3] <- 1000000
	Z[Z<=3] <- 1
	Z[Z>10000] <- 0
	rs <- rowSums(Z)
	cs <- colSums(Z)
	Z <- Z[rs>0,cs>0]
	
	code.list <- rownames(Z)
	ei <- PHYSCHEM[code.list,"DFT.electrophilicity"]
	emed <- median(ei,na.rm=T)
	code.list <- code.list[!is.na(ei)]
	ei <- ei[!is.na(ei)]
	index <- sort(ei,index.return=T)$ix
	n <- length(index)
	codes.use <- c(code.list[index[1:10]],code.list[index[(n-10):n]])
	zmat <- Z[codes.use,]
	rs <- rowSums(zmat)
	cs <- colSums(zmat)
	zmat <- zmat[rs>0,cs>0]
	code.list <- rownames(zmat)
	nchem <- length(code.list)
	name.list <- code.list
	col.list <- code.list
	for(i in 1:nchem) {
		name.list[i] <- CHEMS[code.list[i],"Name"]
		eii <- PHYSCHEM[code.list[i],"DFT.electrophilicity"]
		if(eii<emed) col.list[i] <- "blue"
		else  col.list[i] <- "red"
	}
	result <- heatmap(t(zmat),margins=c(10,10),scale="none",main="Electrophilicity",
					  xlab="",ylab="",cexCol=0.8,cexRow=0.6,col=brewer.pal(9,"Reds"),
					  hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
					  labCol=name.list,ColSideColors=col.list)
	

    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# Calculate at the hit distribution by chemical
#
# QC=OK
#--------------------------------------------------------------------------------------
chemical.hitdist <- function(to.file=F,target.gene=NA) {
	print.current.function()
    options(warn=1)
    warning(immediate.=T,call.=T)
    flush.console()
    alist <- ASSAY.LIST
    ztemp <- MAT.logAC50
    ztemp[] <- NA
    target.gene <- as.character(target.gene)
    nassay <- length(alist)
    nchem <- dim(CHEMS)[1]
    file <- "../output/chemical_hitdist.txt"
    s <- "CODE\tCASRN\tName\tIntendedTarget\tStructureCategory\tUseCategory\tUseSuperCategory\tAssaysTested\tHits\tHitRatio\tSelectiveHits\tSelectiveHitRatio\tCytotoxAssaysTested\tCytotoxHits\tMinAC50\tCytotoxMedian\tCytotoxMin\n"
    sall <- s
    if(is.na(target.gene)) cat(file=file,s,append=F)
    if(to.file) {
        fname <- "../plots/by_chemical_hit_dist.pdf"
        if(!is.na(target.gene)) fname <- paste("../plots/chemical_hit_dist_",target.gene,".pdf",sep="")
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,2),mar=c(4,4,2,2))

    target.assay.list <- NA
    if(!is.na(target.gene)) target.assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"intended_target"],target.gene),"assay"]
    cytotox.assay.set <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease")),"assay"]
    CYTOTOX.ASSAYS <<- cytotox.assay.set

    mask <- vector(mode="integer",length=dim(MAT.logAC50)[2])
    mask[] <- 1
    mask[is.element(names(MAT.logAC50),cytotox.assay.set)] <- 0
    istart <- 1

    for(i in istart:nchem) {
    	counter <- i
    	code <- CHEMS[i,"CODE"]
    	casrn <- CHEMS[i,"CASRN"]
    	cname <- CHEMS[i,"Name"]
    	target <- CHEMS[i,"target_gene"]
    	ccat <- CHEMS[i,"structure_category"]
    	ucat <- CHEMS[i,"use_category"]
    	uscat <- CHEMS[i,"use_super_category"]

    	cytotox.median <- as.numeric(as.character(CYTOTOX[code,"cytotox_median_um"]))
    	cytotox.mad <- as.numeric(as.character(CYTOTOX[code,"global_mad"]))
    	cytotox.min <- as.numeric(as.character(CYTOTOX[code,"cytotox_lower_bound_um"]))
        if(is.na(cytotox.median)) {
        	cytotox.median <- 1000
        	cytotox.min <- 100
        }
    	if(!is.na(target.gene)) {
            target.temp <- MAT.logAC50[code,target.assay.list]
            target.temp[is.na(target.temp)] <- 0
            if(sum(target.temp)==0) doit <- F
    	}
		tempA <- MAT.AC50[code,mask==1]
		hit.assays <- names(tempA)[!is.na(tempA)]
		tempA <- tempA[!is.na(tempA)]
		ntry <- length(tempA)
		hit.assays <- hit.assays[tempA<1000000]
		hit.genes <- sort(uniquify(ASSAY.INFO[is.element(ASSAY.INFO[,"assay"],hit.assays),"intended_target"]))
		nhit.gene <- length(hit.genes)
		tempA <- tempA[tempA<1000000]
		nhit <- length(tempA)
		hit.ratio <- nhit/ntry

		temp.selective <- tempA[tempA<cytotox.min]
		nhit.selective <- length(temp.selective)
		hit.ratio.selective <- nhit.selective / ntry
		hit.gene.ratio <- nhit.gene/ntry
		tempA <- sort(tempA)
		minAC50 <- min(tempA)
		cytotox.ntry <- NA
		cytotox.nhit <- NA
		cytotox.ratio <- NA

		cytotox.tempA <- MAT.AC50[code,cytotox.assay.set]
		cytotox.tempA <- cytotox.tempA[!is.na(cytotox.tempA)]
		cytotox.ntry <- length(cytotox.tempA)


		if(nhit>5) {
			tempA.log <- log10(tempA)-6
			xmin <- 0
			xmax <- 10
			xminA <- 1e-4
			xmaxA <- 1000
			if(minAC50<1E-4) {
				xmax <- 14
				xminA <- 1E-6
			}
			breaksA.min <- 1e-6
			nbreaks <- 80
			if(minAC50<breaksA.min) {
				breaksA.min <- breaksA.min/100
			}
			breaksA <- breaksA.min
			for(i in 1:85) breaksA <- c(breaksA,1.2*breaksA[length(breaksA)])
			while(max(breaksA)<=max(tempA)) {
				breaksA <- c(breaksA,1.2*breaksA[length(breaksA)])
				#cat("Added another point to breaksA",max(breaksA),"\n")
			}
			print(code)
			xA <- hist(tempA,breaks=breaksA,plot=F)
			ymax <- 1.5*max(xA$counts)
			if(ymax<5) ymax <- 5
			hist.log(breaksA,xA$counts,ylim=c(0,ymax),xlab="AC50 (uM)",ylab="Hits",main=paste(casrn,":",cname),cytotox.median, cytotox.min, cytotox.max)
			eps <- 0.08
			xpmin <- breaksA.min
			text(xpmin,ymax*(1-1*eps),paste("ntry=",ntry,sep=""),pos=4)
			text(xpmin,ymax*(1-2*eps),paste("nhit=",nhit,sep=""),pos=4)
			text(xpmin,ymax*(1-3*eps),paste("nhit (Z>3)=",nhit.selective,sep=""),pos=4)
			text(xpmin,ymax*(1-4*eps),paste("cytotox median=",format(cytotox.median,digits=2),sep=""),pos=4)
			text(xpmin,ymax*(1-5*eps),paste("cytotox min=",format(cytotox.min,digits=2),sep=""),pos=4)
			tempA.cytotox <- MAT.AC50[code,cytotox.assay.set]
			tempA.cytotox <- tempA.cytotox[!is.na(tempA.cytotox)]
			tempA.cytotox <- tempA.cytotox[tempA.cytotox<1000000]
			cytotox.nhit <- length(tempA.cytotox)
			text(xpmin,ymax*(1-6*eps),paste("cytotox try=",cytotox.ntry,sep=""),pos=4)
			text(xpmin,ymax*(1-7*eps),paste("cytotox hit=",cytotox.nhit,sep=""),pos=4)
			if(length(tempA.cytotox)>0) {
				for(j in 1:length(tempA.cytotox)) {
					ac50 <- tempA.cytotox[j]
					points(ac50,ymax*0.6,pch="*",col="red",cex=2)
				}
			}

			s <- paste(code,"\t",casrn,"\t",cname,"\t",target,"\t",ccat,"\t",ucat,"\t",uscat,"\t",ntry,"\t",nhit,"\t",format(hit.ratio,digits=2),"\t",nhit.selective,"\t",format(hit.ratio.selective,digits=2),"\t",cytotox.ntry,"\t",cytotox.nhit,"\t",format(minAC50,digits=3),"\t",format(cytotox.median,digits=2),"\t",format(cytotox.min,digits=2),"\n",sep="")
			cat(counter,":",s)
			if(is.na(target.gene))  cat(file=file,s,append=T)
			flush.console()

			if(!is.na(target.gene)) {
				agset <- ASSAY.INFO[is.element(ASSAY.INFO[,"intended_target"],target.gene),"assay"]
				for(g in 1:length(agset)) {
					assay.g <- agset[g]
					ac50 <- MAT.AC50[code,assay.g]
					asource <- ASSAY.INFO[is.element(ASSAY.INFO[,"assay"],agset[g]),"source_group"]
					pch <- 24
					color <- "white"
					if(asource=="NVS_NR") color <- "black"
					if(asource=="OT") color <- "green"
					if(asource=="Tox21_BLA_Agonist") color <- "gray"
					if(asource=="Tox21_BLA_Antagonist") color <- "gray"
					if(asource=="Tox21_LUC_Agonist") color <- "gray"
					if(asource=="Tox21_LUC_Antagonist") color <- "gray"
					if(asource=="ATG_CIS") color <- "violet"
					if(asource=="ATG_TRANS") color <- "violet"
					if(asource=="ACEA") color <- "red"
					if(asource=="BSK_up") color <- "orange"
					if(asource=="BSK_down") color <- "orange"
					if(asource=="Apr_up") color <- "yellow"
					if(asource=="Apr_dn") color <- "yellow"
					yval <- ymax*0.5*(1+0.2*rnorm(1,0.1))
					points(ac50,yval,pch=pch,bg=color,fg="black",cex=2)
				}
			}
			if(!to.file) browser()
		}
    }

    if(to.file) graphics.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot stressy stuff by chemical
#
#--------------------------------------------------------------------------------------
chemical.scatterplot <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/chemical_scatterplot.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,2),mar=c(4,5,4,2))

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))
    code.list <- CHEMS[,"CODE"]
    nchem <- length(code.list)

    structure.list <- sort(unique(CHEMS[,"structure_category"]))
    nstr <- length(structure.list)
    for(st in 1:nstr) {
        scat <- structure.list[st]
        scat.chems <- CHEMS[is.element(CHEMS[,"structure_category"],scat),"CODE"]
        chems.use <- code.list[is.element(code.list,scat.chems)]
        nchem <- length(chems.use)
        cat(scat,":",nchem,"\n")
        flush.console()
        if(nchem>0) {
            for(chem in 1:nchem) {
                code <- chems.use[chem]
                cname <- CHEMS[code,"Name"]
                casrn <- CHEMS[code,"CASRN"]
                gene <- CHEMS[code,"target_gene"]
                main <- paste(casrn,":",cname,"\n",scat)
                plot(1~1,type="n",log="x",xlim=c(0.001,100),ylim=c(0,150),xlab="AC50 (uM)",ylab="T",cex.lab=1.2,cex.axis=1.2,main=main)
                zcut <- CYTOTOX[code,"cytotox_lower_bound_um"]

                rect(zcut,0,1000,150,col="gray80")

                lines(c(zcut,zcut),c(0,150))
                lines(c(1e-8,1e8),c(50,50))
                if(!is.na(gene)  && str_length(gene)>1) text(x=1e-3,y=140,gene,pos=4)
                source.list <- sort(uniquify(ASSAY.INFO[,"biological_process"]))
                nsource <- length(source.list)

                for(i in 1:nsource) {
                    source <- source.list[i]

                    pch <- 1
                    col <- "black"
                    cex <- 1
                    if(source=="cytotoxicity BLA") {
                        pch <- 21
                        col <- "red"
                        cex <- 2
                    }
                    else if(source=="cytotoxicity SRB") {
                        pch <- 25
                        col <- "red"
                        cex <- 2
                    }
                    else if(source=="proliferation decrease") {
                        pch <- 25
                        col <- "pink"
                        cex <- 2
                    }
                    else if(is.element(source,c("apoptosis up","ER stress","heat shock" ,"inflammation up","microtubule up" ,"chemokine up","hypoxia" ,"mitochondrial disruption up","oxidative stress up"))) {
                        pch <- 24
                        col <- "orange"
                        cex <- 2
                    }
                    else if(is.element(source,c("cell cycle up","cellular adhesion up","coagulation up","extracellular matrix up","growth factor up"))) {
                        pch <- 24
                        col <- "gray"
                        cex <- 2
                    }
                    else if(source=="nuclear receptor ACEA") {
                        pch <- 21
                        cex <- 2
                        col <- "blue"
                    }
                    else if(source=="nuclear receptor ATG") {
                        pch <- 22
                        cex <- 2
                        col <- "blue"
                    }
                    else if(source=="nuclear receptor NVS") {
                        pch <- 23
                        cex <- 2
                        col <- "blue"
                    }
                    else if(source=="nuclear receptor OT") {
                        pch <- 21
                        cex <- 2
                        col <- "blue"
                    }
                    else if(source=="nuclear receptor Tox21") {
                        pch <- 24
                        cex <- 2
                        col <- "blue"
                    }
                    else if(source=="nuclear receptor Tox21 ant") {
                        pch <- 25
                        cex <- 2
                        col <- "blue"
                    }
                    else if(source=="GPCR") {
                        pch <- 23
                        cex <- 2
                        col <- "green"
                    }
                    else if(source=="CYP450") {
                        pch <- 21
                        cex <- 2
                        col <- "brown"
                    }
                    else if(source=="transporter") {
                        pch <- 21
                        cex <- 2
                        col <- "yellow"
                    }
		#"cell cycle down"
		#"apoptosis down"
		#"chemokine down"
		#"inflammation down"
		#"growth factor down"
		#"coagulation down"
		#"cellular adhesion down"
		#"extracellular matrix down"
		#"microtubule down"
		#"oxidative stress down"
		#"mitochondrial disruption down"

		#"enzyme anti-blocking"
		#"enzyme blocking"
		#"ion channel"
		#"oncogene"
		#"proliferation upregulation"
		#"receptor binding"
		#"transcription factor"
                    assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],source),"assay"]
                    ac50.list <- MAT.AC50[code,assay.list]
                    top.list <- MAT.T.SCALED[code,assay.list]
                    hit.list <- MAT.hitcall[code,assay.list]
                    mask <- hit.list
                    mask[is.na(mask)] <- 0
                    mask[mask<0] <- 0
                    mask[is.na(ac50.list)] <- 0
                    mask[is.na(top.list)] <- 0

                    ac50.list <- ac50.list[mask==1]
                    top.list <- top.list[mask==1]
                    top.list[top.list>140] <- 140
                    points(top.list~ac50.list,pch=pch,bg=col,cex=cex)
                }
                if(!to.file) if(chem%%1 == 0) browser()
            }
        }
    }
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot stressy stuff by chemical
#
#--------------------------------------------------------------------------------------
chemical.scatterplot.all <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/chemical_scatterplot_all.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,2),mar=c(4,5,4,2))

    code.list <- CHEMS[,"CODE"]
    nchem <- length(code.list)

	for(chem in 1:nchem) {
		code <- code.list[chem]
		cname <- CHEMS[code,"Name"]
		casrn <- CHEMS[code,"CASRN"]
		gene <- CHEMS[code,"target_gene"]
		main <- paste(casrn,":",cname)
		plot(1~1,type="n",log="x",xlim=c(0.001,100),ylim=c(0,150),xlab="AC50 (uM)",ylab="T",cex.lab=1.2,cex.axis=1.2,main=main)
		zcut <- CYTOTOX[code,"cytotox_lower_bound_um"]
		print(cname)
		flush.console()
		rect(zcut,0,1000,150,col="gray80")

		lines(c(zcut,zcut),c(0,150))
		lines(c(1e-8,1e8),c(50,50))
		if(!is.na(gene)  && str_length(gene)>1) text(x=1e-3,y=140,gene,pos=4)
		source.list <- sort(uniquify(ASSAY.INFO[,"biological_process"]))
		nsource <- length(source.list)

		for(i in 1:nsource) {
			source <- source.list[i]

			pch <- 1
			col <- "black"
			cex <- 1
			if(source=="cytotoxicity BLA") {
				pch <- 21
				col <- "red"
				cex <- 2
			}
			else if(source=="cytotoxicity SRB") {
				pch <- 25
				col <- "red"
				cex <- 2
			}
			else if(source=="proliferation decrease") {
				pch <- 25
				col <- "pink"
				cex <- 2
			}
			else if(is.element(source,c("apoptosis up","ER stress","heat shock" ,"inflammation up","microtubule up" ,"chemokine up","hypoxia" ,"mitochondrial disruption up","oxidative stress up"))) {
				pch <- 24
				col <- "orange"
				cex <- 2
			}
			else if(is.element(source,c("cell cycle up","cellular adhesion up","coagulation up","extracellular matrix up","growth factor up"))) {
				pch <- 24
				col <- "gray"
				cex <- 2
			}
			else if(source=="nuclear receptor ACEA") {
				pch <- 21
				cex <- 2
				col <- "blue"
			}
			else if(source=="nuclear receptor ATG") {
				pch <- 22
				cex <- 2
				col <- "blue"
			}
			else if(source=="nuclear receptor NVS") {
				pch <- 23
				cex <- 2
				col <- "blue"
			}
			else if(source=="nuclear receptor OT") {
				pch <- 21
				cex <- 2
				col <- "blue"
			}
			else if(source=="nuclear receptor Tox21") {
				pch <- 24
				cex <- 2
				col <- "blue"
			}
			else if(source=="nuclear receptor Tox21 ant") {
				pch <- 25
				cex <- 2
				col <- "blue"
			}
			else if(source=="GPCR") {
				pch <- 23
				cex <- 2
				col <- "green"
			}
			else if(source=="CYP450") {
				pch <- 21
				cex <- 2
				col <- "brown"
			}
			else if(source=="transporter") {
				pch <- 21
				cex <- 2
				col <- "yellow"
			}
			assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],source),"assay"]
			ac50.list <- MAT.AC50[code,assay.list]
			top.list <- MAT.T.SCALED[code,assay.list]
			hit.list <- MAT.hitcall[code,assay.list]
			mask <- hit.list
			mask[is.na(mask)] <- 0
			mask[mask<0] <- 0
			mask[is.na(ac50.list)] <- 0
			mask[is.na(top.list)] <- 0

			ac50.list <- ac50.list[mask==1]
			top.list <- top.list[mask==1]
			top.list[top.list>140] <- 140
			points(top.list~ac50.list,pch=pch,bg=col,cex=cex)
		}
		if(!to.file) browser()
	}
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# Calculate the least squares solution for one chemical
#
#--------------------------------------------------------------------------------------
chemical.scatterplot.legend <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/chemical_scatterplot_legend.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }

    ytop <- 2.8
    plot(1~1,type="n",col.axis="white",tcl=0.01,cex.axis=0.1,cex.lab=0.1,xlim=c(0,5),ylim=c(0,ytop),xlab="",ylab="",lwd=3,main="")
    text(x=1,y=ytop,labels="Chemical Scatterplot Legend",pos=4,cex=1.2)
    dy <- 0.18
    ytop <- ytop - 0.05

    namelist <- NULL
    pchlist <- NULL
    collist <- NULL
    cexlist <- NULL

    namelist <- c(namelist,"cytotoxicity BLA")
    pchlist <- c(pchlist,21)
    collist <- c(collist,"red")
    cexlist <- c(cexlist,2)

    namelist <- c(namelist,"cytotoxicity SRB")
    pchlist <- c(pchlist,25)
    collist <- c(collist,"red")
    cexlist <- c(cexlist,2)

    namelist <- c(namelist,"proliferation decrease")
    pchlist <- c(pchlist,25)
    collist <- c(collist,"pink")
    cexlist <- c(cexlist,2)

    namelist <- c(namelist,"Stress assay upregulation")
    pchlist <- c(pchlist,24)
    collist <- c(collist,"orange")
    cexlist <- c(cexlist,2)

    namelist <- c(namelist,"Other cell process upregulation")
    pchlist <- c(pchlist,24)
    collist <- c(collist,"gray")
    cexlist <- c(cexlist,2)

#    namelist <- c(namelist,"nuclear receptor ACEA")
#    pchlist <- c(pchlist,21)
#    cexlist <- c(cexlist,2)
#    collist <- c(collist,"blue")

    namelist <- c(namelist,"nuclear receptor ATG")
    pchlist <- c(pchlist,22)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"blue")

    namelist <- c(namelist,"nuclear receptor NVS")
    pchlist <- c(pchlist,23)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"blue")

    namelist <- c(namelist,"nuclear receptor OT")
    pchlist <- c(pchlist,21)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"blue")

    namelist <- c(namelist,"nuclear receptor Tox21")
    pchlist <- c(pchlist,24)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"blue")

    namelist <- c(namelist,"nuclear receptor Tox21 ant")
    pchlist <- c(pchlist,25)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"blue")

    namelist <- c(namelist,"GPCR")
    pchlist <- c(pchlist,23)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"green")

    namelist <- c(namelist,"CYP450")
    pchlist <- c(pchlist,21)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"brown")

    namelist <- c(namelist,"transporter")
    pchlist <- c(pchlist,21)
    cexlist <- c(cexlist,2)
    collist <- c(collist,"yellow")

    ntypes <- length(namelist)
    for(i in 1:ntypes) {
        points(x=1,y=c(ytop-dy*i),bg=collist[i],cex=cexlist[i],pch=pchlist[i])
        text(x=1.2,y=ytop-dy*i,labels=namelist[i],pos=4,cex=0.9)
    }

    if(to.file) dev.off()
}
#--------------------------------------------------------------------------------------
#
# histograms of fraction of hits in an assay class
#
#--------------------------------------------------------------------------------------
bioprocess.promiscuity <- function(to.file=T,to.file2=F,to.file3=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/bioprocess_assay_class_hist.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,2),mar=c(4,5,4,2))

	code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
	code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
	code.list <- sort(unique(code.list))
	nchem <- length(code.list)

	source.list <- sort(uniquify(ASSAY.INFO[,"biological_process"]))
	nsource <- length(source.list)
	result1 <- matrix(nrow=nchem,ncol=nsource)
	rownames(result1) <- code.list
	colnames(result1) <- source.list
	result1[] <- 0
	for(i in 1:nsource) {
		sourcei <- source.list[i]
		print(sourcei)
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],sourcei),"assay"]
		if(length(assay.list)>1) {
			temp <- MAT.hitcall[code.list,assay.list]
			tested <- MAT.tested[code.list,assay.list]
			temp[temp<0] <- 0
			temp[is.na(temp)] <- 0
			rs <- rowSums(temp)
			rs.denom <- rowSums(tested)
			rs[rs.denom==0] <- 0
			rs.denom[rs.denom==0] <- 1

			result <- as.numeric(rs/rs.denom)
			result1[,i] <- result

			if(length(result)>=5) {
				breaks <- seq(0,1,by=0.1)
				hist(result,breaks=breaks,main=sourcei)
				if(!to.file) browser()
			}
		}
	}

	source.list <- sort(uniquify(ASSAY.INFO[,"source_group"]))
	nsource <- length(source.list)
	result2 <- matrix(nrow=nchem,ncol=nsource)
	result2[] <- 0
	rownames(result2) <- code.list
	colnames(result2) <- source.list
	for(i in 1:nsource) {
		sourcei <- source.list[i]
		print(sourcei)
		assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"source_group"],sourcei),"assay"]
		if(length(assay.list)>1) {
			temp <- MAT.hitcall[code.list,assay.list]
			tested <- MAT.tested[code.list,assay.list]
			temp[temp<0] <- 0
			temp[is.na(temp)] <- 0
			rs <- rowSums(temp)
			rs.denom <- rowSums(tested)
			rs[rs.denom==0] <- 0
			rs.denom[rs.denom==0] <- 1

			result <- as.numeric(rs/rs.denom)
			result2[,i] <- result

			if(length(result)>=5) {
				breaks <- seq(0,1,by=0.1)
				hist(result,breaks=breaks,main=sourcei)
				#if(!to.file) browser()
			}
		}
	}
    if(to.file) dev.off()
	else browser()

    mat <- result1
    mat[mat<=0.5] <- 0
    cnames <- CHEMS[code.list,]
    outdata <- cbind(cnames,mat)
    outfile <- "../plots/bioprocess_promiscuity.csv"
    write.csv(outdata,file=outfile, row.names=F)

    cs <- colSums(mat)
    mat <- mat[,cs>1]
    rs <- rowSums(mat)
    mat <- mat[rs>0,]
    mat[mat>0] <- 1

    rowcolors <- colnames(mat)
    for(i in 1:length(rowcolors)) {
    	type <- drug.target(rowcolors[i])
    	if(type==1) rowcolors[i] <- "red"
    	else rowcolors[i] <- "white"
    }
    colcolors <- rownames(mat)
    for(i in 1:length(colcolors)) {
    	code <- colcolors[i]
    	cclass <- CHEMS[code,"use_super_category"]
    	if(str_detect(cclass,ignore.case("Pharmaceutical"))) colcolors[i] <- "red"
    	else if(str_detect(cclass,ignore.case("Pesticide"))) colcolors[i] <- "black"
    	else if(str_detect(cclass,ignore.case("Food/Flavor/Fragrance"))) colcolors[i] <- "green"
    	else colcolors[i] <- "white"
    }

	if(to.file2) {
		fname <- "../plots/bioprocess_promiscuity_hm.pdf"
		pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
	}
	result <- heatmap(t(mat),margins=c(10,10),scale="none",main="Biological Process Promiscuity",
					  xlab="",ylab="",cexCol=0.1,cexRow=0.8,col=brewer.pal(9,"Reds"),
					  hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
					  RowSideColors=rowcolors,ColSideColors=colcolors)

    if(to.file2) dev.off()
    else browser()

    # focused heatmap

    mat <- result1
    mat[mat<=0.5] <- 0

	class.list <- c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease","oxidative stress up","ER stress","apoptosis up","microtubule up")
	mat <- mat[,class.list]
    cs <- colSums(mat)
    mat <- mat[,cs>1]
    rs <- rowSums(mat)
    mat <- mat[rs>0,]
    mat[mat>0] <- 1

    rowcolors <- colnames(mat)
    for(i in 1:length(rowcolors)) {
    	type <- drug.target(rowcolors[i])
    	if(type==1) rowcolors[i] <- "red"
    	else rowcolors[i] <- "white"
    }
    colcolors <- rownames(mat)
    for(i in 1:length(colcolors)) {
    	code <- colcolors[i]
    	cclass <- CHEMS[code,"use_super_category"]
	    if(str_detect(cclass,ignore.case("Pharmaceutical"))) colcolors[i] <- "red"
		else if(str_detect(cclass,ignore.case("Pesticide"))) colcolors[i] <- "black"
		else if(str_detect(cclass,ignore.case("Food/Flavor/Fragrance"))) colcolors[i] <- "green"
		else colcolors[i] <- "white"
    }

	if(to.file3) {
		fname <- "../plots/cytotox_banding_hm.pdf"
		pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
	}
	result <- heatmap(t(mat),margins=c(10,10),scale="none",main="Cytotoxicity Banding",
					  xlab="",ylab="",cexCol=0.1,cexRow=0.8,col=brewer.pal(9,"Reds"),
					  hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
					  RowSideColors=rowcolors,ColSideColors=colcolors)

    if(to.file3) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# drug target classification
#
#--------------------------------------------------------------------------------------
drug.target <- function(bp) {
	print.current.function()
	if(is.element(bp,c("enzyme blocking",
	"GPCR",
	"ion channel",
	"nuclear receptor ACEA",
	"nuclear receptor ATG",
	"nuclear receptor NVS",
	"nuclear receptor OT",
	"nuclear receptor Tox21",
	"nuclear receptor Tox21 ant",
	"receptor binding",
	"CYP450",
	"transporter"))) return(1)
	else return(0)
}
#--------------------------------------------------------------------------------------
#
# heatmap of thequadrants (T / Z) vs biological process specificity
#
#--------------------------------------------------------------------------------------
bioprocess.quadrant.hm <- function(do.prep=T,to.file=F,tcut=25) {
	print.current.function()

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))
    nchem <- length(code.list)

    top <- MAT.T.SCALED[code.list,]
    z <-MAT.Z.NORM[code.list,]
    tested <- MAT.tested[code.list,]
    hit <- MAT.hitcall[code.list,]

    bio.list <-
 c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease",
 	"apoptosis up","cell cycle up","chemokine up","cellular adhesion up","coagulation up",
 	"inflammation up","extracellular matrix up","growth factor up","microtubule up","mitochondrial disruption up",
	"oxidative stress up",
	"ER stress","heat shock","hypoxia","oncogene",
 	"GPCR",
 	"nuclear receptor ACEA","nuclear receptor ATG","nuclear receptor NVS","nuclear receptor OT","nuclear receptor Tox21","nuclear receptor Tox21 ant",
 	"CYP450",
 	"ion channel",
 	"transporter",
 	"enzyme blocking",
 	"transcription factor")

    nbio <- length(bio.list)

    if(do.prep) {
        result <- matrix(nrow=nchem,ncol=4*nbio)
        rownames(result) <- code.list
        res.names <- vector(length=4*nbio,mode="character")
        for(i in 1:nbio) {
            bio <- bio.list[i]
            res.names[i] <- paste(bio,"_HiZ_HiT",sep="")
            res.names[i+nbio] <- paste(bio,"_HiZ_LoT",sep="")
            res.names[i+2*nbio] <- paste(bio,"_LoZ_HiT",sep="")
            res.names[i+3*nbio] <- paste(bio,"_LoZ_LoT",sep="")
        }
        colnames(result) <- res.names
        result[] <- 0

        top <- MAT.T[code.list,]
        z <-MAT.Z.NORM[code.list,]
        tested <- MAT.tested[code.list,]
        hit <- MAT.hitcall[code.list,]

        z[z==0] <- NA
        z[tested<=0] <- NA
        hit[tested<=0] <- NA
        top[tested<=0] <- NA

        z[hit<=0] <- NA
        top[hit<=0] <- NA

		# Q1 T>tcut,Z>3
        z1 <- z
        top1 <- top
        z1[z1<3] <- 0
        z1[is.na(z1)] <- 0
        z1[z1>0] <- 1
        top1[top1<tcut] <- 0
        top1[is.na(top1)] <- 0
        top1[top1>0] <- 1
        zt <- z1*top1
        for(i in 1:nbio) {
            bioproc <- bio.list[i]
            assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bioproc),"assay"]
            if(length(assay.list)>1) result[,i] <- rowSums(zt[,assay.list])/length(assay.list)
            else result[,i] <- zt[,assay.list]
        }

		# Q2 T<tcut,Z>3
        z1 <- z
        top1 <- top
        z1[z1<3] <- 0
        z1[is.na(z1)] <- 0
        z1[z1>0] <- 1
        top1[top1>tcut] <- 0
        top1[is.na(top1)] <- 0
        top1[top1>0] <- 1
        zt <- z1*top1
        for(i in 1:nbio) {
            bioproc <- bio.list[i]
            assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bioproc),"assay"]
            if(length(assay.list)>1) result[,(i+nbio)] <- rowSums(zt[,assay.list])/length(assay.list)
            else result[,(i+nbio)] <- zt[,assay.list]
        }

		# Q3 T>tcut,Z<3
        z1 <- z
        top1 <- top
        z1[z1>3] <- -100000
        z1[is.na(z1)] <- -100000
        z1[z1> -1000] <- 1
        z1[z1<0] <- 0
        top1[top1<tcut] <- 0
        top1[is.na(top1)] <- 0
        top1[top1>0] <- 1
        zt <- z1*top1
        for(i in 1:nbio) {
            bioproc <- bio.list[i]
            assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bioproc),"assay"]
            if(length(assay.list)>1) result[,(i+(2*nbio))] <- rowSums(zt[,assay.list])/length(assay.list)
            else result[,(i+2*nbio)] <- zt[,assay.list]
        }

		# Q4 T<tcut,Z<3
        z1 <- z
        top1 <- top
        z1[z1>3] <- -1000000
        z1[is.na(z1)] <- -1000000
        z1[z1> -1000] <- 1
        z1[z1<0] <- 0
        top1[top1>tcut] <- 0
        top1[is.na(top1)] <- 0
        top1[top1>0] <- 1
        zt <- z1*top1
        for(i in 1:nbio) {
            bioproc <- bio.list[i]
            assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bioproc),"assay"]
            if(length(assay.list)>1) result[,(i+(3*nbio))] <- rowSums(zt[,assay.list])/length(assay.list)
            else result[,(i+3*nbio)] <- zt[,assay.list]
        }

        RESULT <<- result
    }

######################################
    dtemp <- RESULT
    cl <- hclust(d=dist(dtemp),method="ward.D")
    hcut <- 1
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
        clout[i,1] <- CHEMS[i,"CODE"]
        clout[i,2] <- CHEMS[i,"Name"]
        clout[i,3] <- CHEMS[i,"use_category"]
        clout[i,4] <- CHEMS[i,"use_super_category"]
        clout[i,5] <- CHEMS[i,"structure_category"]
        clout[i,6] <- CHEMS[i,"structure_super_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","structure_super_category","Level_1")

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()
	nlevel <- 50
    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
   	outfile <- "../plots/bioprocess_quadrants_clusters.csv"
    write.csv(clout,file=outfile, row.names=F)


######################################
    dtemp <-  RESULT[,1:nbio]
    cl <- hclust(d=dist(dtemp),method="ward.D")
    hcut <- 1
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
        clout[i,1] <- CHEMS[i,"CODE"]
        clout[i,2] <- CHEMS[i,"Name"]
        clout[i,3] <- CHEMS[i,"use_category"]
        clout[i,4] <- CHEMS[i,"use_super_category"]
        clout[i,5] <- CHEMS[i,"structure_category"]
        clout[i,6] <- CHEMS[i,"structure_super_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","structure_super_category","Level_1")

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()
	nlevel <- 50
    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
   	outfile <- "../plots/bioprocess_quadrants_clusters_Q1.csv"
    write.csv(clout,file=outfile, row.names=F)

######################################
    dtemp <-  RESULT[,((nbio+1):(2*nbio))]
    cl <- hclust(d=dist(dtemp),method="ward.D")
    hcut <- 1
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
        clout[i,1] <- CHEMS[i,"CODE"]
        clout[i,2] <- CHEMS[i,"Name"]
        clout[i,3] <- CHEMS[i,"use_category"]
        clout[i,4] <- CHEMS[i,"use_super_category"]
        clout[i,5] <- CHEMS[i,"structure_category"]
        clout[i,6] <- CHEMS[i,"structure_super_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","structure_super_category","Level_1")

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()
	nlevel <- 50
    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
   	outfile <- "../plots/bioprocess_quadrants_clusters_Q2.csv"
    write.csv(clout,file=outfile, row.names=F)

######################################
    dtemp <-  RESULT[,((2*nbio+1):(3*nbio))]
    cl <- hclust(d=dist(dtemp),method="ward.D")
    hcut <- 1
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
        clout[i,1] <- CHEMS[i,"CODE"]
        clout[i,2] <- CHEMS[i,"Name"]
        clout[i,3] <- CHEMS[i,"use_category"]
        clout[i,4] <- CHEMS[i,"use_super_category"]
        clout[i,5] <- CHEMS[i,"structure_category"]
        clout[i,6] <- CHEMS[i,"structure_super_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","structure_super_category","Level_1")

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()
	nlevel <- 50
    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
   	outfile <- "../plots/bioprocess_quadrants_clusters_Q3.csv"
    write.csv(clout,file=outfile, row.names=F)

######################################
    dtemp <-  RESULT[,((3*nbio+1):(4*nbio))]
    cl <- hclust(d=dist(dtemp),method="ward.D")
    hcut <- 1
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
        clout[i,1] <- CHEMS[i,"CODE"]
        clout[i,2] <- CHEMS[i,"Name"]
        clout[i,3] <- CHEMS[i,"use_category"]
        clout[i,4] <- CHEMS[i,"use_super_category"]
        clout[i,5] <- CHEMS[i,"structure_category"]
        clout[i,6] <- CHEMS[i,"structure_super_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","structure_super_category","Level_1")

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()
	nlevel <- 50
    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
    outfile <- "../plots/bioprocess_quadrants_clusters_Q4.csv"
    write.csv(clout,file=outfile, row.names=F)

    rowcolors <- colnames(RESULT)
    rowcolors[] <- "white"

    rowcolors <- c("red","red","red","red",
                   "orange","orange","orange","orange","orange",
                   "orange","orange","orange","orange","orange",
                   "orange",
                   "orange","orange","orange","orange",
                   "green",
                   "blue","blue","blue","blue","blue","blue",
                   "brown",
                   "yellow",
                   "pink",
                   "pink",
                   "pink",
                   "gray","gray","gray","gray","gray",
                   "gray","gray","gray","gray","gray",
                   "gray",
                   "gray","gray")


    rowcolors <- c("red","red","red","red",
                   "orange","orange","orange","orange","orange",
                   "orange","orange","orange","orange","orange",
                   "orange",
                   "orange","orange","orange","orange",
                   "green",
                   "blue","blue","blue","blue","blue","blue",
                   "brown",
                   "yellow",
                   "pink",
                   "pink",
                   "pink")

    for(i in 1:nbio) {
        rowcolors[(i+nbio)] <- rowcolors[i]
        rowcolors[(i+2*nbio)] <- rowcolors[i]
        rowcolors[(i+3*nbio)] <- rowcolors[i]
    }
    rowcolors[1:3] <- "red"
    rowcolors[(nbio+1):(nbio+3)] <- "red"
    rowcolors[(2*nbio+1):(2*nbio+3)] <- "red"
    rowcolors[(3*nbio+1):(3*nbio+3)] <- "red"


    colcolors <- rownames(RESULT)
    for(i in 1:length(colcolors)) {
        code <- colcolors[i]
        cclass <- CHEMS[code,"use_super_category"]
        if(str_detect(cclass,ignore.case("Pharmaceutical"))) colcolors[i] <- "red"
        else if(str_detect(cclass,ignore.case("Pesticide"))) colcolors[i] <- "black"
        else if(str_detect(cclass,ignore.case("Food/Flavor/Fragrance"))) colcolors[i] <- "green"
        else if(str_detect(cclass,ignore.case("Solvent"))) colcolors[i] <- "gray"
        else if(str_detect(cclass,ignore.case("Surfactant"))) colcolors[i] <- "blue"
        else colcolors[i] <- "white"
    }

    if(to.file) {
        fname <- "../plots/bioprocess_quadrants_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result2 <- RESULT
    rhm <- heatmap(t(as.matrix(result2)),margins=c(10,10),scale="none",main="Biological Process Quadrants",
                   xlab="",ylab="",cexCol=0.1,cexRow=0.4,col=brewer.pal(9,"Reds"),
                   hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,Rowv=NA,
                   RowSideColors=rowcolors,ColSideColors=colcolors)
    if(to.file) dev.off()
    else browser()


    if(to.file) {
        fname <- "../plots/bioprocess_quadrants_Q1_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result2 <- RESULT[,1:nbio]
    rhm <- heatmap(t(as.matrix(result2)),margins=c(10,10),scale="none",main="Biological Process Hi-Z, Hi-T",
                   xlab="",ylab="",cexCol=0.1,cexRow=1,col=brewer.pal(9,"Reds"),
                   hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,Rowv=NA,
                   RowSideColors=rowcolors[1:nbio],ColSideColors=colcolors,labRow=bio.list)
    if(to.file) dev.off()
    else browser()

    if(to.file) {
        fname <- "../plots/bioprocess_quadrants_Q2_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result2 <- RESULT[,((nbio+1):(2*nbio))]
    rhm <- heatmap(t(as.matrix(result2)),margins=c(10,10),scale="none",main="Biological Process Hi-Z, Lo-T",
                   xlab="",ylab="",cexCol=0.1,cexRow=1,col=brewer.pal(9,"Reds"),
                   hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,Rowv=NA,
                   RowSideColors=rowcolors[1:nbio],ColSideColors=colcolors,labRow=bio.list)
    if(to.file) dev.off()
    else browser()

    if(to.file) {
        fname <- "../plots/bioprocess_quadrants_Q3_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result2 <- RESULT[,((2*nbio+1):(3*nbio))]
    rhm <- heatmap(t(as.matrix(result2)),margins=c(10,10),scale="none",main="Biological Process Lo-Z, Hi-T",
                   xlab="",ylab="",cexCol=0.1,cexRow=1,col=brewer.pal(9,"Reds"),
                   hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,Rowv=NA,
                   RowSideColors=rowcolors[1:nbio],ColSideColors=colcolors,labRow=bio.list)
    if(to.file) dev.off()
    else browser()

    if(to.file) {
        fname <- "../plots/bioprocess_quadrants_Q4_hm.pdf"
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result2 <- RESULT[,((3*nbio+1):(4*nbio))]
    rhm <- heatmap(t(as.matrix(result2)),margins=c(10,10),scale="none",main="Biological Process Lo-Z, Lo-T",
                   xlab="",ylab="",cexCol=0.1,cexRow=1,col=brewer.pal(9,"Reds"),
                   hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,Rowv=NA,
                   RowSideColors=rowcolors[1:nbio],ColSideColors=colcolors,labRow=bio.list)
    if(to.file) dev.off()
    else browser()

}
#--------------------------------------------------------------------------------------
#
# heatmap of thequadrants (T / Z) vs biological process specificity
#
#--------------------------------------------------------------------------------------
bioprocess.quadrant.useclass.hm <- function(to.file=F,tcut=50,cclass="Pharmaceutical") {
	print.current.function()

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))

    code.list.class <- CHEMS[is.element(CHEMS[,"use_super_category"],cclass),"CODE"]
    code.list <- code.list[is.element(code.list,code.list.class)]
    name.list <- CHEMS[code.list,"Name"]
    nchem <- length(code.list)


    bio.list <-
        c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease",
          "apoptosis up","cell cycle up","chemokine up","cellular adhesion up","coagulation up",
          "inflammation up","extracellular matrix up","growth factor up","microtubule up","mitochondrial disruption up",
          "oxidative stress up",
          "ER stress","heat shock","hypoxia","oncogene",
          "GPCR",
          "nuclear receptor ACEA","nuclear receptor ATG","nuclear receptor NVS","nuclear receptor OT","nuclear receptor Tox21","nuclear receptor Tox21 ant",
          "CYP450",
          "ion channel",
          "transporter",
          "enzyme blocking",
          "transcription factor")

    nbio <- length(bio.list)

    assay.list <- NULL
    rowcolors <- NULL
    for(i in 1:nbio) {
        biop <- bio.list[i]
        temp <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],biop),"assay"]
        assay.list <- c(assay.list,temp)
        rtemp <- temp
        if(i>=1 && i<=3) rtemp[] <- "red"
        if(i>=4 && i<=18) rtemp[] <- "orange"
        if(i>=19 && i<=19) rtemp[] <- "green"
        if(i>=20 && i<=25) rtemp[] <- "blue"
        if(i>=26 && i<=26) rtemp[] <- "brown"
        if(i>=27 && i<=27) rtemp[] <- "yellow"
        if(i>=28 && i<=30) rtemp[] <- "pink"
        rowcolors <- c(rowcolors,rtemp)
    }

    tested <- MAT.tested[code.list,assay.list]
    top <- MAT.T.SCALED[code.list,assay.list]
    z <- MAT.Z.NORM[code.list,assay.list]
    hit <- MAT.hitcall[code.list,assay.list]
    for(q in 1:4) {
        cat("Start quadrant: ",q,"\n")

        csv.file <- paste("../plots/bioprocess_Q",q,"_",cclass,".csv",sep="")
        pdf.file <- paste("../plots/bioprocess_Q",q,"_",cclass,"_hm.pdf",sep="")

        Q <- z
        Q[is.na(Q)] <- 0
        Q[tested<=0] <- 0
        if(q==1) {
            main <- paste("Hi-Z, Hi-T ",cclass)
            Q[z<3] <- 0
            Q[hit<=0] <- 0
            Q[top<tcut] <- 0
            Q[Q>0] <- 1
        }
        if(q==2) {
            main <- paste("Hi-Z, Lo-T ",cclass)
            Q[z<3] <- 0
            Q[hit<=0] <- 0
            Q[top>tcut] <- 0
            Q[Q>0] <- 1
        }
        if(q==3) {
            main <- paste("Lo-Z, Hi-T ",cclass)
            Q[hit<=0] <- 0
            Q[z<0] <- 1
            Q[z>3] <- 0
            Q[top<tcut] <- 0
            Q[Q>0] <- 1
        }
        if(q==4) {
            main <- paste("Lo-Z, Lo-T ",cclass)
            Q[hit<=0] <- 0
            Q[z<0] <- 1
            Q[z>3] <- 0
            Q[top>tcut] <- 0
            Q[Q>0] <- 1
        }
######################################
        dtemp <- Q
        cl <- hclust(d=dist(dtemp),method="ward.D2")
        hcut <- 1
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clcut,clcut)
        clout <- cbind(clout,clcut)
        clout <- cbind(clout,clcut)
        clout <- cbind(clout,clcut)
        clout <- cbind(clout,clcut)
        clout <- cbind(clout,clcut)
        clout<- as.data.frame(clout)
        for(i in 1:length(clcut)) {
            clout[i,1] <- CHEMS[i,"CODE"]
            clout[i,2] <- CHEMS[i,"Name"]
            clout[i,3] <- CHEMS[i,"use_category"]
            clout[i,4] <- CHEMS[i,"use_super_category"]
            clout[i,5] <- CHEMS[i,"structure_category"]
            clout[i,6] <- CHEMS[i,"structure_super_category"]
        }
        names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","structure_super_category","Level_1")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
        nlevel <- 20
        for(hcut in 2:nlevel) {
            clcut <- cutree(cl,h=hcut)
            clout <- cbind(clout,clcut)
            names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
            cat("Finished preping clusters for hcut: ",hcut,"\n")
            flush.console()
        }
        write.csv(clout,file=csv.file, row.names=F)

        colcolors <- rownames(Q)
        for(i in 1:length(colcolors)) {
            code <- colcolors[i]
            cclass <- CHEMS[code,"use_super_category"]
            if(str_detect(cclass,ignore.case("Pharmaceutical"))) colcolors[i] <- "red"
            else if(str_detect(cclass,ignore.case("Pesticide"))) colcolors[i] <- "black"
            else if(str_detect(cclass,ignore.case("FoodFlavorFragrance"))) colcolors[i] <- "green"
            else if(str_detect(cclass,ignore.case("Solvent"))) colcolors[i] <- "gray"
            else if(str_detect(cclass,ignore.case("Surfactant"))) colcolors[i] <- "blue"
            else colcolors[i] <- "white"
        }

        if(to.file)  pdf(file=pdf.file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)

        rhm <- heatmap(as.matrix(Q),margins=c(10,10),scale="none",main=main,
                       xlab="",ylab="",cexCol=0.1,cexRow=0.1,col=brewer.pal(9,"Reds"),
                       hclustfun=function(x) hclust(d=dist(x),method="ward.D2"),keep.dendro=T,verbose=F,
                       RowSideColors=colcolors,ColSideColors=rowcolors,labRow=name.list)
        if(to.file) dev.off()
        else browser()
    }
}
#--------------------------------------------------------------------------------------
#
# heatmap of thequadrants (T / Z) vs biological process specificity
#
#--------------------------------------------------------------------------------------
bioprocess.quadrant.boxplot <- function(to.file=F,tcut=50) {
	print.current.function()

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))

    name.list <- CHEMS[code.list,"Name"]
    nchem <- length(code.list)


    bio.list <-
        c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease",
          "apoptosis up","cell cycle up","chemokine up","cellular adhesion up","coagulation up",
          "inflammation up","extracellular matrix up","growth factor up","microtubule up","mitochondrial disruption up",
          "oxidative stress up",
          "ER stress","oncogene",
          "GPCR",
          "nuclear receptor ATG","nuclear receptor NVS","nuclear receptor OT","nuclear receptor Tox21","nuclear receptor Tox21 ant",
          "CYP450",
          "ion channel",
          "transporter",
          "enzyme blocking",
          "transcription factor")

    class.list <- c("FoodFlavorFragrance","Solvent","Surfactant","Other","Microbicide","Herbicide","Pesticide","Pharmaceutical")
    nclass <- length(class.list)
    nbio <- length(bio.list)

    pdf.file <- "../plots/bioprocess_quadrants_boxplot.pdf"
    if(to.file) pdf(file=pdf.file,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    par(mfrow=c(4,1),mar=c(4,4,2,2))

    tcols <- c("Quadrant","Tclass","Zclass","BioProocess",class.list)
    nrow <- 4*length(bio.list)
    rtable <- as.data.frame(matrix(nrow=,ncol=length(tcols)))
    names(rtable) <- tcols

    tested <- MAT.tested[code.list,]
    top <- MAT.T.SCALED[code.list,]
    z <- MAT.Z.NORM[code.list,]
    hit <- MAT.hitcall[code.list,]
    itable <- 0
    for(q in 1:4) {
        cat("Start quadrant: ",q,"\n")

        Q <- z
        Q[is.na(Q)] <- 0
        Q[tested<=0] <- 0
        if(q==1) {
            main <- paste("Hi-Z, Hi-T ")
            Q[z<3] <- 0
            Q[hit<=0] <- 0
            Q[top<tcut] <- 0
            Q[Q>0] <- 1
            tclass <- "Hi"
            zclass <- "Hi"
        }
        if(q==2) {
            main <- paste("Hi-Z, Lo-T ")
            Q[z<3] <- 0
            Q[hit<=0] <- 0
            Q[top>tcut] <- 0
            Q[Q>0] <- 1
            tclass <- "Hi"
            zclass <- "Lo"
        }
        if(q==3) {
            main <- paste("Lo-Z, Hi-T ")
            Q[hit<=0] <- 0
            Q[z<0] <- 1
            Q[z>3] <- 0
            Q[top<tcut] <- 0
            Q[Q>0] <- 1
            tclass <- "Lo"
            zclass <- "Hi"
        }
        if(q==4) {
            main <- paste("Lo-Z, Lo-T ")
            Q[hit<=0] <- 0
            Q[z<0] <- 1
            Q[z>3] <- 0
            Q[top>tcut] <- 0
            Q[Q>0] <- 1
            tclass <- "Lo"
            zclass <- "Lo"
        }

        for(i in 1:nbio) {
            bpnames <- NULL
            group <- 0
            groups <- NULL
            fraction.hit <- NULL
            biop <- bio.list[i]
            assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],biop),"assay"]

            itable <- itable+1
            rtable[itable,1] <- paste("Q",1,sep="")
            rtable[itable,2] <- tclass
            rtable[itable,3] <- zclass
            rtable[itable,4] <- biop


            if(length(assay.list)>2) {
                for(j in 1:nclass) {
                    cclass <- class.list[j]
                    cclass.plot <- cclass
                    if(cclass=="FoodFlavorFragrance") cclass.plot <- "Food/Fragrance"
                    code.list.class <- CHEMS[is.element(CHEMS[,"use_super_category"],cclass),"CODE"]
                    code.list.class <- code.list.class[is.element(code.list.class,code.list)]
                    temp <- Q[code.list.class,assay.list]
                    #cclass.plot <- paste(cclass.plot," (",length(code.list.class),")",sep="")
                    cat(biop,":",cclass.plot,"\n")
                    if(!is.null(dim(temp))) {
                        rs <- rowSums(temp)
                        rs <- as.numeric(rs/dim(temp)[2])
                        group <- group+1
                        gtemp <- rs
                        gtemp[] <- group
                        fraction.hit <- c(fraction.hit,rs)
                        groups <- c(groups,gtemp)
                        bpnames <- c(bpnames,cclass.plot)
                    }
                }
                ymax <- 1
                main <- paste(biop," : Q",q,sep="")
                boxplot(fraction.hit~groups,xlab="Chemical class",ylab="Hit Fraction",names=bpnames,cex.axis=0.8,cex.lab=0.8,ylim=c(0,1),main=main)
                for(g in 2:nclass) {
                    cclass <- class.list[g]
                    pclass <- 0
                    x <- fraction.hit[groups==1]
                    y <- fraction.hit[groups==g]
                    p.lt <- wilcox.test(y,x,alternative="less")$p.value
                    p.gt <- wilcox.test(y,x,alternative="greater")$p.value

                    if(is.nan(p.lt)) p.lt <- 1
                    if(is.nan(p.gt)) p.gt <- 1
                    if(p.lt<p.gt) {
                        p <- p.lt
                        pstring <- format(p,digits=2)
                        pstring <- paste(pstring,"(<)")
                        if(p>0.01) pstring <- "N.S."
                        text(g,ymax,pstring,pos=1,cex=1.)
                        if(p<0.01) pclass <- log10(p)
                    }
                    else {
                        p <- p.gt
                        pstring <- format(p,digits=2)
                        pstring <- paste(pstring,"(>)")
                        if(p>0.01) pstring <- "N.S."
                        text(g,ymax,pstring,pos=1,cex=1.)
                        if(p<0.01) pclass <- -log10(p)
                    }
                    rtable[itable,cclass] <- pclass
                }
                if(!to.file) browser()
            }
        }
    }
    csv.file <- "../plots/bioprocess_ptable.csv"
    write.csv(rtable,file=csv.file, row.names=F)
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# trellis plot on on Z
#
#--------------------------------------------------------------------------------------
bioprocess.quadrant.trellis.Z <- function(do.prep=T,to.file=F) {
	print.current.function()

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))

    name.list <- CHEMS[code.list,"Name"]
    nchem <- length(code.list)

    tested <- MAT.tested[code.list,]
    z <- MAT.Z.NORM[code.list,]
    hit <- MAT.hitcall[code.list,]

    bioproc.list <- sort(unique(ASSAY.INFO[,"biological_process"]))
    useclass.list <- c("FoodFlavorFragrance","Solvent","Surfactant","Other","Microbicide","Herbicide","Pesticide","Pharmaceutical")
    useclass.plot.names <- useclass.list
    for(i in 1:length(useclass.list)) useclass.plot.names[i] <- use.sub(useclass.list[i])
    useclass.plot.names <- sort(useclass.plot.names)

    nuseclass <- length(useclass.list)
    nbioproc <- length(bioproc.list)
    nz <- 2

    rtable <- NULL

    if(do.prep) {
        for(iz in 1:2) {
            Q <- z
            Q[Q<=0] <- 0.001
            Q[tested<=0] <- 0
            Q[hit<=0] <- 0
            Q[is.na(Q)] <- 0
            if(iz==1) {
                Q[z>3] <- 0; Q[Q>0] <- 1
                zclass <- "Low-Z"
            }
            if(iz==2) {
                Q[z<3] <- 0; Q[Q>0] <- 1
                zclass <- "High-Z"
            }
            assay.list <- ASSAY.LIST
            for(iu in 1:nuseclass) {
                useclass <- useclass.list[iu]
                code.list.useclass <- CHEMS[is.element(CHEMS[,"use_super_category"],useclass),"CODE"]
                code.list.useclass <- code.list.useclass[is.element(code.list.useclass,code.list)]
                temp <- Q[code.list.useclass,assay.list]

                if(!is.null(dim(temp))) {
                    rs <- rowSums(temp)
                    rs <- as.numeric(rs/dim(temp)[2])
                    block <- as.data.frame(matrix(ncol=8,nrow=length(code.list.useclass)))
                    name.list <- CHEMS[code.list.useclass,"Name"]
                    structure.list <- CHEMS[code.list.useclass,"structure_category"]
                    use.list <- CHEMS[code.list.useclass,"use_category"]
                    block[,1] <- code.list.useclass
                    block[,2] <- name.list
                    block[,3] <- structure.list
                    block[,4] <- use.list
                    block[,5] <- use.sub(useclass)
                    block[,6] <- "All Assays"
                    block[,7] <- zclass
                    block[,8] <- rs

                    names(block) <- c("CODE","Name","structure_category","use_category","use_super_category","BioProcess","Z.class","fraction.active")
                    rtable <- rbind(rtable,block)
                }
            }
            for(ibp in 1:nbioproc) {
                bioproc <- bioproc.list[ibp]
                assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bioproc),"assay"]
                if(length(assay.list)>2) {
                    for(iu in 1:nuseclass) {
                        useclass <- useclass.list[iu]
                        code.list.useclass <- CHEMS[is.element(CHEMS[,"use_super_category"],useclass),"CODE"]
                        code.list.useclass <- code.list.useclass[is.element(code.list.useclass,code.list)]
                        temp <- Q[code.list.useclass,assay.list]

                        if(!is.null(dim(temp))) {
                            rs <- rowSums(temp)
                            rs <- as.numeric(rs/dim(temp)[2])
                            block <- as.data.frame(matrix(ncol=8,nrow=length(code.list.useclass)))

                            name.list <- CHEMS[code.list.useclass,"Name"]
                            structure.list <- CHEMS[code.list.useclass,"structure_category"]
                            use.list <- CHEMS[code.list.useclass,"use_category"]
                            block[,1] <- code.list.useclass
                            block[,2] <- name.list
                            block[,3] <- structure.list
                            block[,4] <- use.list

                            block[,5] <- use.sub(useclass)
                            block[,6] <- bioproc
                            block[,7] <- zclass
                            block[,8] <- rs

                            names(block) <- c("CODE","Name","structure_category","use_category","use_super_category","BioProcess","Z.class","fraction.active")
                            rtable <- rbind(rtable,block)
                        }
                    }
                }
            }
        }
        RTABLE <<- rtable
        csv.file <- "../plots/bioprocess_use_z.csv"
        write.csv(RTABLE,file=csv.file, row.names=F)
    }

    cex <- 0.9
    pdf.file <- "../plots/bioprocess_quadrant_trellis_Z.pdf"
    if(to.file) pdf(file=pdf.file,width=7.5,height=7.5,pointsize=12,bg="white",paper="letter",pagecentre=T)
    par(mfrow=c(2,2),mar=c(6,4,1,1))
    for(bp in 0:nbioproc) {
    	if(bp==0) {
            bioproc <- "All Assays"
            ymax <- 0.4
    	}
    	else {
            bioproc <- bioproc.list[bp]
            ymax <- 1.2
        }
        print(bioproc)
        temp <- RTABLE[is.element(RTABLE[,"BioProcess"],bioproc),]
        if(dim(temp)[1]>5) {
            tempxx <- temp[is.element(temp[,"Z.class"],"High-Z"),]
            fraction.hit <- tempxx[,"fraction.active"]
            groups <- tempxx[,"use_super_category"]
            main <- paste(bioproc,"High-Z")
            boxplot(fraction.hit~groups,xlab="",ylab="Fraction Assays Active / Chemical",cex.axis=cex,cex.lab=cex,ylim=c(0,ymax),main=main,las=2)

            for(g in 1:nuseclass) {
                gname <- useclass.plot.names[g]
                x <- fraction.hit[is.element(groups,"A.Food")]
                y <- fraction.hit[is.element(groups,gname)]
                p.lt <- wilcox.test(y,x,alternative="less")$p.value
                p.gt <- wilcox.test(y,x,alternative="greater")$p.value
				nchem <- length(y)
                pstring <- "NS"; col="black"
                if(is.nan(p.lt)) p.lt <- 1
                if(is.nan(p.gt)) p.gt <- 1
                if(min(p.lt,p.gt)<0.05) {
                    if(p.lt<p.gt) {pstring <- format(p.lt,digits=2); col <- "blue" }
                    else {pstring <- format(p.gt,digits=2); col <- "red" }
                }
                text(g,ymax,nchem,pos=1,cex=cex,col="black")
                text(g,ymax*(0.9-0.05*(g%%2)),pstring,pos=1,cex=cex,col=col)
            }

            temp <- RTABLE[is.element(RTABLE[,"BioProcess"],bioproc),]
            tempxx <- temp[is.element(temp[,"Z.class"],"Low-Z"),]
            fraction.hit <- tempxx[,"fraction.active"]
            groups <- tempxx[,"use_super_category"]
            main <- paste(bioproc,"Low-Z")
            boxplot(fraction.hit~groups,xlab="",ylab="Fraction Assays Active / Chemical",cex.axis=cex,cex.lab=cex,ylim=c(0,ymax),main=main,las=2)

            for(g in 1:nuseclass) {
                gname <- useclass.plot.names[g]
                x <- fraction.hit[is.element(groups,"A.Food")]
                y <- fraction.hit[is.element(groups,gname)]
                p.lt <- wilcox.test(y,x,alternative="less")$p.value
                p.gt <- wilcox.test(y,x,alternative="greater")$p.value
				nchem <- length(y)

                pstring <- "NS"; col="black"
                if(is.nan(p.lt)) p.lt <- 1
                if(is.nan(p.gt)) p.gt <- 1
                if(min(p.lt,p.gt)<0.05) {
                    if(p.lt<p.gt) {pstring <- format(p.lt,digits=2); col <- "blue" }
                    else {pstring <- format(p.gt,digits=2); col <- "red" }
                }
                text(g,ymax,nchem,pos=1,cex=cex,col="black")
                text(g,ymax*(0.9-0.05*(g%%2)),pstring,pos=1,cex=cex,col=col)
            }

            if(!to.file) browser()
        }

    }
    if(to.file) dev.off()
}
use.sub <- function(useclass) {
    if(useclass=="FoodFlavorFragrance") return("A.Food")
    else if(useclass=="Solvent") return("B.Solvent")
    else if(useclass=="Surfactant") return("D.Surfactant")
    else if(useclass=="Other") return("C.Other")
    else if(useclass=="Pesticide") return("G.Pesticide")
    else if(useclass=="Microbicide") return("F.Microbicide")
    else if(useclass=="Herbicide") return("E.Herbicide")
    else if(useclass=="Pharmaceutical") return("H.Drug")
    else {
        print(useclass)
        return (useclass)
    }
}
#--------------------------------------------------------------------------------------
#
# heatmap of thequadrants (T / Z) vs biological process specificity
#
#--------------------------------------------------------------------------------------
bioprocess.quadrant.trellis <- function(do.prep=T,to.file=F,tcut=50) {
	print.current.function()

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))

    name.list <- CHEMS[code.list,"Name"]
    nchem <- length(code.list)

    tested <- MAT.tested[code.list,]
    top <- MAT.T.SCALED[code.list,]
    z <- MAT.Z.NORM[code.list,]
    hit <- MAT.hitcall[code.list,]

    bioproc.list <-
        c("cytotoxicity SRB","cytotoxicity BLA","proliferation decrease",
          "apoptosis up","cell cycle up","chemokine up","cellular adhesion up","coagulation up",
          "inflammation up","extracellular matrix up","growth factor up","microtubule up","mitochondrial disruption up",
          "oxidative stress up",
          "oncogene",
          "GPCR",
          "nuclear receptor ATG","nuclear receptor NVS","nuclear receptor OT","nuclear receptor Tox21","nuclear receptor Tox21 ant",
          "CYP450",
          "ion channel",
          "transporter",
          "enzyme blocking",
          "transcription factor")
    bioproc.list <- sort(unique(ASSAY.INFO[,"biological_process"]))

    useclass.list <- c("FoodFlavorFragrance","Solvent","Surfactant","Other","Microbicide","Herbicide","Pesticide","Pharmaceutical")
    useclass.plot.names <- useclass.list
    for(i in 1:length(useclass.list)) useclass.plot.names[i] <- use.sub(useclass.list[i])
    useclass.plot.names <- sort(useclass.plot.names)

    nuseclass <- length(useclass.list)
    nbioproc <- length(bioproc.list)
    nz <- 2
    nt <- 2

    rtable <- NULL

    if(do.prep) {
        for(iz in 1:2) {
            for(it in 1:2) {
                Q <- z
                Q[is.na(Q)] <- 0
                Q[tested<=0] <- 0

                if(iz==1 && it==1) {
                    Q[hit<=0] <- 0; Q[z<0] <- 1; Q[z>3] <- 0; Q[top>tcut] <- 0; Q[Q>0] <- 1
                    zclass <- "Z-Lo"; tclass <- "T-Lo"
                }
                if(iz==2 && it==1) {
                    Q[z<3] <- 0; Q[hit<=0] <- 0; Q[top>tcut] <- 0; Q[Q>0] <- 1
                    zclass <- "Z-Hi"; tclass <- "T-Lo"
                }
                if(iz==1 && it==2) {
                    Q[hit<=0] <- 0; Q[z<0] <- 1; Q[z>3] <- 0; Q[top<tcut] <- 0; Q[Q>0] <- 1
                    zclass <- "Z-Lo"; tclass <- "T-Hi"
                }
                if(iz==2 && it==2) {
                    Q[z<3] <- 0; Q[hit<=0] <- 0; Q[top<tcut] <- 0; Q[Q>0] <- 1
                    zclass <- "Z-Hi"; tclass <- "T-Hi"
                }

				assay.list <- ASSAY.LIST
				for(iu in 1:nuseclass) {
					useclass <- useclass.list[iu]
					code.list.useclass <- CHEMS[is.element(CHEMS[,"use_super_category"],useclass),"CODE"]
					code.list.useclass <- code.list.useclass[is.element(code.list.useclass,code.list)]
					temp <- Q[code.list.useclass,assay.list]

					if(!is.null(dim(temp))) {
						rs <- rowSums(temp)
						rs <- as.numeric(rs/dim(temp)[2])
						block <- as.data.frame(matrix(ncol=9,nrow=length(code.list.useclass)))
						name.list <- CHEMS[code.list.useclass,"Name"]
						structure.list <- CHEMS[code.list.useclass,"structure_category"]
						use.list <- CHEMS[code.list.useclass,"use_category"]
						block[,1] <- code.list.useclass
						block[,2] <- name.list
						block[,3] <- structure.list
						block[,4] <- use.list

						block[,5] <- use.sub(useclass)
						block[,6] <- "All Assays"
						block[,7] <- zclass
						block[,8] <- tclass
						block[,9] <- rs

						names(block) <- c("CODE","Name","structure_category","use_category","use_super_category","BioProcess","Z.class","T.class","fraction.active")
						rtable <- rbind(rtable,block)
					}
				}
                for(ibp in 1:nbioproc) {
                    bioproc <- bioproc.list[ibp]
                    assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bioproc),"assay"]
                    if(length(assay.list)>2) {
                        for(iu in 1:nuseclass) {
                            useclass <- useclass.list[iu]
                            code.list.useclass <- CHEMS[is.element(CHEMS[,"use_super_category"],useclass),"CODE"]
                            code.list.useclass <- code.list.useclass[is.element(code.list.useclass,code.list)]
                            temp <- Q[code.list.useclass,assay.list]

                            if(!is.null(dim(temp))) {
                                rs <- rowSums(temp)
                                rs <- as.numeric(rs/dim(temp)[2])
                                block <- as.data.frame(matrix(ncol=9,nrow=length(code.list.useclass)))

								name.list <- CHEMS[code.list.useclass,"Name"]
								structure.list <- CHEMS[code.list.useclass,"structure_category"]
								use.list <- CHEMS[code.list.useclass,"use_category"]
								block[,1] <- code.list.useclass
								block[,2] <- name.list
								block[,3] <- structure.list
								block[,4] <- use.list

								block[,5] <- use.sub(useclass)
								block[,6] <- bioproc
								block[,7] <- zclass
								block[,8] <- tclass
								block[,9] <- rs

								names(block) <- c("CODE","Name","structure_category","use_category","use_super_category","BioProcess","Z.class","T.class","fraction.active")
                                rtable <- rbind(rtable,block)
                            }
                        }
                    }
                }
            }
        }
        RTABLE <<- rtable
    }
    csv.file <- "../plots/bioprocess_use_zt.csv"
    write.csv(RTABLE,file=csv.file, row.names=F)

    cex <- 0.9
    pdf.file <- "../plots/bioprocess_quadrant_trellis.pdf"
    if(to.file) pdf(file=pdf.file,width=7.5,height=7.5,pointsize=12,bg="white",paper="letter",pagecentre=T)
    par(mfrow=c(2,2),mar=c(6,3,1,1))
    for(bp in 0:nbioproc) {
    	if(bp==0) {
    		bioproc <- "All Assays"
    		ymax <- 0.3
    	}
    	else {
    		bioproc <- bioproc.list[bp]
    		ymax <- 1.2
		}
        print(bioproc)
        temp <- RTABLE[is.element(RTABLE[,"BioProcess"],bioproc),]
        tempxx <- temp[is.element(temp[,"Z.class"],"Z-Hi"),]
        tempxx <- tempxx[is.element(temp[,"T.class"],"T-Hi"),]
        fraction.hit <- tempxx[,"fraction.active"]
        groups <- tempxx[,"use_super_category"]
        main <- paste(bioproc,"Z-Hi T-Hi")
        boxplot(fraction.hit~groups,xlab="",ylab="Fraction Assays Active / Chemical",cex.axis=cex,cex.lab=cex,ylim=c(0,ymax),main=main,las=2)

        for(g in 1:nuseclass) {
            gname <- useclass.plot.names[g]
            x <- fraction.hit[is.element(groups,"A.Food")]
            y <- fraction.hit[is.element(groups,gname)]
            p.lt <- wilcox.test(y,x,alternative="less")$p.value
            p.gt <- wilcox.test(y,x,alternative="greater")$p.value

            pstring <- "NS"; col="black"
            if(is.nan(p.lt)) p.lt <- 1
            if(is.nan(p.gt)) p.gt <- 1
            if(min(p.lt,p.gt)<0.05) {
#                if(p.lt<p.gt) {pstring <- paste(format(p.lt,digits=2),"(<)",sep=""); col <- "blue" }
#                else {pstring <- paste(format(p.gt,digits=2),"(>)",sep=""); col <- "red" }
                if(p.lt<p.gt) {pstring <- format(p.lt,digits=2); col <- "blue" }
                else {pstring <- format(p.gt,digits=2); col <- "red" }
            }
            text(g,ymax*(1-0.05*(g%%2)),pstring,pos=1,cex=cex,col=col)
        }

        temp <- RTABLE[is.element(RTABLE[,"BioProcess"],bioproc),]
        tempxx <- temp[is.element(temp[,"Z.class"],"Z-Lo"),]
        tempxx <- tempxx[is.element(temp[,"T.class"],"T-Hi"),]
        fraction.hit <- tempxx[,"fraction.active"]
        groups <- tempxx[,"use_super_category"]
        main <- paste(bioproc,"Z-Lo T-Hi")
        boxplot(fraction.hit~groups,xlab="",ylab="Fraction Assays Active / Chemical",cex.axis=cex,cex.lab=cex,ylim=c(0,ymax),main=main,las=2)

        for(g in 1:nuseclass) {
            gname <- useclass.plot.names[g]
            x <- fraction.hit[is.element(groups,"A.Food")]
            y <- fraction.hit[is.element(groups,gname)]
            p.lt <- wilcox.test(y,x,alternative="less")$p.value
            p.gt <- wilcox.test(y,x,alternative="greater")$p.value

            pstring <- "NS"; col="black"
            if(is.nan(p.lt)) p.lt <- 1
            if(is.nan(p.gt)) p.gt <- 1
            if(min(p.lt,p.gt)<0.05) {
                if(p.lt<p.gt) {pstring <- format(p.lt,digits=2); col <- "blue" }
                else {pstring <- format(p.gt,digits=2); col <- "red" }
            }
            text(g,ymax*(1-0.05*(g%%2)),pstring,pos=1,cex=cex,col=col)
        }

        temp <- RTABLE[is.element(RTABLE[,"BioProcess"],bioproc),]
        tempxx <- temp[is.element(temp[,"Z.class"],"Z-Hi"),]
        tempxx <- tempxx[is.element(temp[,"T.class"],"T-Lo"),]
        fraction.hit <- tempxx[,"fraction.active"]
        groups <- tempxx[,"use_super_category"]
        main <- paste(bioproc,"Z-Hi T-Lo")
        boxplot(fraction.hit~groups,xlab="",ylab="Fraction Assays Active / Chemical",cex.axis=cex,cex.lab=cex,ylim=c(0,ymax),main=main,las=2)
        for(g in 1:nuseclass) {
            gname <- useclass.plot.names[g]
            x <- fraction.hit[is.element(groups,"A.Food")]
            y <- fraction.hit[is.element(groups,gname)]
            p.lt <- wilcox.test(y,x,alternative="less")$p.value
            p.gt <- wilcox.test(y,x,alternative="greater")$p.value

            pstring <- "NS"; col="black"
            if(is.nan(p.lt)) p.lt <- 1
            if(is.nan(p.gt)) p.gt <- 1
            if(min(p.lt,p.gt)<0.05) {
                if(p.lt<p.gt) {pstring <- format(p.lt,digits=2); col <- "blue" }
                else {pstring <- format(p.gt,digits=2); col <- "red" }
            }
            text(g,ymax*(1-0.05*(g%%2)),pstring,pos=1,cex=cex,col=col)
        }


        temp <- RTABLE[is.element(RTABLE[,"BioProcess"],bioproc),]
        tempxx <- temp[is.element(temp[,"Z.class"],"Z-Lo"),]
        tempxx <- tempxx[is.element(temp[,"T.class"],"T-Lo"),]
        fraction.hit <- tempxx[,"fraction.active"]
        groups <- tempxx[,"use_super_category"]
        main <- paste(bioproc,"Z-Lo T-Lo")
        boxplot(fraction.hit~groups,xlab="",ylab="",cex.axis=cex,cex.lab=cex,ylim=c(0,ymax),main=main,las=2)

        for(g in 1:nuseclass) {
            gname <- useclass.plot.names[g]
            x <- fraction.hit[is.element(groups,"A.Food")]
            y <- fraction.hit[is.element(groups,gname)]
            p.lt <- wilcox.test(y,x,alternative="less")$p.value
            p.gt <- wilcox.test(y,x,alternative="greater")$p.value

            pstring <- "NS"; col="black"
            if(is.nan(p.lt)) p.lt <- 1
            if(is.nan(p.gt)) p.gt <- 1
            if(min(p.lt,p.gt)<0.05) {
                if(p.lt<p.gt) {pstring <- format(p.lt,digits=2); col <- "blue" }
                else {pstring <- format(p.gt,digits=2); col <- "red" }
            }
            text(g,ymax*(1-0.05*(g%%2)),pstring,pos=1,cex=cex,col=col)
        }

        if(!to.file) browser()
    }

    if(to.file) dev.off()
    else browser()
}
use.sub <- function(useclass) {
    if(useclass=="FoodFlavorFragrance") return("A.Food")
    else if(useclass=="Solvent") return("B.Solvent")
    else if(useclass=="Surfactant") return("D.Surfactant")
    else if(useclass=="Other") return("C.Other")
    else if(useclass=="Pesticide") return("G.Pesticide")
    else if(useclass=="Microbicide") return("F.Microbicide")
    else if(useclass=="Herbicide") return("E.Herbicide")
    else if(useclass=="Pharmaceutical") return("H.Drug")
    else {
        print(useclass)
        return (useclass)
    }
}
#--------------------------------------------------------------------------------------
#
# heatmap of NR assays
#
#--------------------------------------------------------------------------------------
NR.hm <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/NR_heatmap.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,2),mar=c(4,5,4,2))

	code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
	code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
	code.list <- sort(unique(code.list))
	nchem <- length(code.list)

 	bp.list <- c("nuclear receptor ACEA","nuclear receptor ATG","nuclear receptor NVS","nuclear receptor OT","nuclear receptor Tox21")
 	assay.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp.list),"assay"]
 	gene.list <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp.list),"intended_target"]

 	zmat <- MAT.Z.NORM[code.list,assay.list]
 	hitmat <- MAT.hitcall[code.list,assay.list]
 	testmat <- MAT.tested[code.list,assay.list]
 	zmat[is.na(zmat)] <- 0
 	zmat[is.na(hitmat)] <- 0
 	zmat[is.na(testmat)] <- 0
 	zmat[zmat<3] <- 0
 	zmat[hitmat<=0] <- 0
 	zmat[testmat<=0] <- 0

 	rowcolors <- gene.list
 	rowcolors[] <- "white"
 	rowcolors[gene.list=="AR"] <- "red"
	rowcolors[gene.list=="Ar"] <- "red"

	rowcolors[gene.list=="ESR1"] <- "blue"
	rowcolors[gene.list=="Esr1"] <- "blue"
	rowcolors[gene.list=="ESR1 ESR2"] <- "blue"
	rowcolors[gene.list=="ESR2"] <- "blue"

	rowcolors[gene.list=="ESRRA"] <- "cyan"
	rowcolors[gene.list=="ESRRG"] <- "cyan"

	rowcolors[gene.list=="PPARA"] <- "green"
	rowcolors[gene.list=="PPARD"] <- "green"
	rowcolors[gene.list=="PPARG"] <- "green"
	rowcolors[gene.list=="PPARA PPARG PPARD"] <- "green"

	rowcolors[gene.list=="RARA"] <- "pink"
	rowcolors[gene.list=="RARB"] <- "pink"
	rowcolors[gene.list=="RARG"] <- "pink"
	rowcolors[gene.list=="RARA RARB RARG"] <- "pink"


	rowcolors[gene.list=="RORA RORB RORC"] <- "yellow"
	rowcolors[gene.list=="RORB"] <- "yellow"
	rowcolors[gene.list=="RORC"] <- "yellow"

	rowcolors[gene.list=="RXRA"] <- "violet"
	rowcolors[gene.list=="RXRB"] <- "violet"

	rowcolors[gene.list=="THRA"] <- "black"
	rowcolors[gene.list=="THRA THRB"] <- "black"

	rowcolors[gene.list=="PGR"] <- "gray"

	rowcolors[gene.list=="NR1I2"] <- "khaki"
	rowcolors[gene.list=="NR1I3"] <- "khaki"

	rowcolors[gene.list=="NR1H3 NR1H2"] <- "cyan"
	rowcolors[gene.list=="NR1H4"] <- "cyan"
	rowcolors[gene.list=="NR1H3"] <- "cyan"
	rowcolors[gene.list=="NR1H2"] <- "cyan"

	rowcolors[gene.list=="NR3C1"] <- "cyan"
	rowcolors[gene.list=="Nr3c2"] <- "cyan"

	rowcolors[gene.list=="NR4A2"] <- "cyan"


	result <- heatmap(t(zmat),margins=c(10,10),scale="none",main="Nuclear Receptor Activity",
					  xlab="",ylab="",cexCol=0.1,cexRow=0.5,col=brewer.pal(9,"Reds"),
					  hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,labRow=gene.list,
					  RowSideColors=rowcolors)

    if(to.file) dev.off()
    else browser()

}
#--------------------------------------------------------------------------------------
#
# heatmap of tested matrix
#
#--------------------------------------------------------------------------------------
tested.hm <- function(to.file=F) {
	print.current.function()
    if(to.file) {
        fname <- "../plots/tested_heatmap.pdf"
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }

	code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
	code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
	#code.list <- sort(unique(code.list))
	mat <- MAT.tested#[code.list,]

	result <- heatmap(t(mat),margins=c(10,10),scale="none",main="Tested matrix",
					  xlab="",ylab="",cexCol=0.1,cexRow=0.1,col=brewer.pal(9,"Reds"),
					  hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F)

    if(to.file) dev.off()
    else browser()

}
#--------------------------------------------------------------------------------------
#
# do the heatmap of hits for a category
#
# QC=OK
#--------------------------------------------------------------------------------------
structure.category.hm <- function(to.file=F) {
	print.current.function()
    flush.console()

    if(to.file) {
        file <- paste("../plots/structure_category_hm.pdf",sep="")
        pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }

    cat.list <- sort(uniquify(CHEMS[,"structure_category"]))
    cat.list <- cat.list[2:length(cat.list)]
    ncat <- length(cat.list)
    for(i in 1:ncat) {
    	category <- cat.list[i]
    	code.list <- CHEMS[is.element(CHEMS[,"structure_category"],category),"CODE"]
    	name.list <- CHEMS[code.list,"Name"]
    	nchem <- length(code.list)
    	if(nchem>=5) {
            atemp <- 6-MAT.logAC50[code.list,]
            ztemp <- MAT.Z.NORM[code.list,]
            ztemp[is.na(ztemp)] <- 0
            ztemp[ztemp<3] <- 0
            ztemp[ztemp>0] <- 1
            atemp[is.na(atemp)] <- 0
            #btemp <- atemp*ztemp
            btemp <- atemp
            cs <- colSums(btemp)
            btemp <- btemp[,cs>0]
            print(category)
            result <- heatmap(as.matrix(btemp),margins=c(10,10),scale="none",labRow=name.list,main=category,
                              xlab="",ylab="",cexCol=0.6,cexRow=0.8,col=brewer.pal(9,"Reds"),
                              hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F)

            if(!to.file) browser()
    	}
    }
    cat.list <- sort(uniquify(CHEMS[,"use_super_category"]))
    cat.list <- cat.list[2:length(cat.list)]
    ncat <- length(cat.list)
    for(i in 1:ncat) {
    	category <- cat.list[i]
    	code.list <- CHEMS[is.element(CHEMS[,"use_super_category"],category),"CODE"]
    	name.list <- CHEMS[code.list,"Name"]
    	nchem <- length(code.list)
    	if(nchem>=5) {
            atemp <- 6-MAT.logAC50[code.list,]
            ztemp <- MAT.Z.NORM[code.list,]
            ztemp[is.na(ztemp)] <- 0
            ztemp[ztemp<3] <- 0
            ztemp[ztemp>0] <- 1
            atemp[is.na(atemp)] <- 0
            #btemp <- atemp*ztemp
            btemp <- atemp
            cs <- colSums(btemp)
            #btemp <- btemp[,cs>0]
            print(category)
            result <- heatmap(as.matrix(btemp),margins=c(10,10),scale="none",labRow=name.list,main=category,
                              xlab="",ylab="",cexCol=0.2,cexRow=0.5,col=brewer.pal(9,"Reds"),
                              hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F)

            if(!to.file) browser()
    	}
    }
    if(to.file) graphics.off()
    else browser()
}

#--------------------------------------------------------------------------------------
#
# print out the negative chemicals
#
# QC=OK
#--------------------------------------------------------------------------------------
negative.chems <- function() {
	print.current.function()

	nchem <- dim(MAT.hitcall)[1]
	code.list <- c()
	for(i in 1:nchem) {
		temp <- MAT.hitcall[i,]
		temp <- temp[!is.na(temp)]
		ss <- sum(temp)
		if(ss==0) code.list <- c(code.list,rownames(MAT.hitcall)[i])
	}
	temp <- CHEMS[code.list,]
    outfile <- paste("../output/negative_chemicals.txt",sep="")
    write.table(temp,file=outfile, row.names=F, append=FALSE, quote=F, sep = "\t")
}
#--------------------------------------------------------------------------------------
#
# plot the range of Bruns Watson calls vs promiscuity
#
# QC=OK
#--------------------------------------------------------------------------------------
bruns.watson <- function(to.file=F) {
	print.current.function()
    file <- "../BrunsWatson/BrunsWatsonOutput.txt"
    bdata <- read.table(file,header=T,sep="\t",stringsAsFactors=F,quote="\"",comment="")
    print(dim(bdata))

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))
    nchem <- length(code.list)

    #nchem <- 100
    hf.hi <- vector(length=nchem,mode="numeric")
    hf.lo <- vector(length=nchem,mode="numeric")
    bw.class <- vector(length=nchem,mode="numeric")
    hf.hi[] <- NA
    hf.lo[] <- NA
    bw.class[] <- NA
    for(i in 1:nchem) {
        code <- code.list[i]
        temp <- MAT.Z.NORM[code,]
        temp <- temp[!is.na(temp)]

        temp.lo <- temp[temp<3]
        temp.hi <- temp[temp>5]
        n.lo <- length(temp.lo)
        n.hi <- length(temp.hi)
        tested <- MAT.tested[code,]
        tested[is.na(tested)] <- 0
        tested[tested<0] <- 0
        denom <- sum(tested)

        hf.hi[i] <- n.hi/denom
        hf.lo[i] <- n.lo/denom
        if(is.element(code,bdata[,"Title"])) {
            res <- bdata[is.element(bdata[,"Title"],code),"Result"]
            if(res=="clean") bw.class[i] <- 0
            if(res=="dirty") bw.class[i] <- 1
            if(res=="fail") bw.class[i] <- 2
        }
    }
    hf.hi <- hf.hi[!is.na(bw.class)]
    hf.lo <- hf.lo[!is.na(bw.class)]
    bw.class <- bw.class[!is.na(bw.class)]

    p.20.lo <- wilcox.test(hf.lo[bw.class==2],hf.lo[bw.class==0],alternative="greater")$p.value
    p.21.lo <- wilcox.test(hf.lo[bw.class==2],hf.lo[bw.class==1],alternative="greater")$p.value
    p.10.lo <- wilcox.test(hf.lo[bw.class==1],hf.lo[bw.class==0],alternative="greater")$p.value

    p.20.hi <- wilcox.test(hf.hi[bw.class==2],hf.hi[bw.class==0],alternative="greater")$p.value
    p.21.hi <- wilcox.test(hf.hi[bw.class==2],hf.hi[bw.class==1],alternative="greater")$p.value
    p.10.hi <- wilcox.test(hf.hi[bw.class==1],hf.hi[bw.class==0],alternative="greater")$p.value

    if(to.file) {
        fname <- paste("../plots/bruns_watson.pdf",sep="")
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(2,1),mar=c(4,4,2,2))
    ymax <- 0.5
    ytop <- 0.45
    delta <- 0.05

    boxplot(hf.lo~bw.class,xlab="Bruns-Watson Class",ylab="Hit Fraction Lo-Z",names=c("clean","dirty","fail"),cex.axis=1.2,cex.lab=1.2,ylim=c(0,ymax),main="Actual vs. Predicted Promiscuity, Gene Hits")
    text(1.2,ytop,paste("p(fail:clean):",format(p.20.lo,digits=3)),pos=4)
    text(2.2,ytop,paste("p(fail:dirty):",format(p.21.lo,digits=3)),pos=4)
    text(1.2,ytop-delta,paste("p(dirty:clean):",format(p.10.lo,digits=3)),pos=4)

    boxplot(hf.hi~bw.class,xlab="Bruns-Watson Class",ylab="Hit Fraction Hi-Z",names=c("clean","dirty","fail"),cex.axis=1.2,cex.lab=1.2,ylim=c(0,ymax),main="Actual vs. Predicted Promiscuity, Gene Hits")
    text(1.2,ytop,paste("p(fail:clean):",format(p.20.hi,digits=3)),pos=4)
    text(2.2,ytop,paste("p(fail:dirty):",format(p.21.hi,digits=3)),pos=4)
    text(1.2,ytop-delta,paste("p(dirty:clean):",format(p.10.hi,digits=3)),pos=4)

    if(to.file) graphics.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# plot histograms of hits by chemical and assay
#
# QC=OK
#--------------------------------------------------------------------------------------
chemical.hit.dist.summary <- function(do.prep=T,to.file=F) {
	print.current.function()
    if(do.prep) {
        tested <- MAT.tested
        tested[tested<0] <- 0
        tested[is.na(tested)] <- 0

        hit <- MAT.hitcall
        hit[is.na(hit)] <- 0
        hit[hit>0] <- 1
        hit[hit<0] <- 0
		hit[tested==0] <- 0

        bot <- rowSums(tested)
        bot[bot==0] <- 1
        top.hit <- rowSums(hit)
        fraction.hit.by.chem <<- top.hit/bot

        bot <- colSums(tested)
        bot[bot==0] <- 1
        top.hit <- colSums(hit)
        fraction.hit.by.assay <<- top.hit/bot
    }
    if(to.file) {
        fname <- paste("../plots/chemical_hit_dist_summary.pdf",sep="")
        pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(2,1),mar=c(4,4,2,2))
    ylim <- 1
    breaks <- seq(0,ylim,by=0.02)
    hist(fraction.hit.by.chem,xlab="Fraction Hits",ylab="Chemicals",cex.axis=1.2,cex.lab=1.2,main="By Chemical",freq=T,breaks=breaks,ylim=c(0,1000))
    breaks <- seq(0,ylim,by=0.02)
    hist(fraction.hit.by.assay,xlab="Fraction Hits",ylab="Assays",cex.axis=1.2,cex.lab=1.2,main="By Assay",freq=T,breaks=breaks,ylim=c(0,500))

    if(to.file) graphics.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# boxplots as a function of cytotox hits
#
#--------------------------------------------------------------------------------------
cytotox.boxplot <- function(to.file=F) {
	print.current.function()

    if(to.file) {
        fname <- paste("../plots/cytotox_boxplot.pdf",sep="")
        pdf(file=fname,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(1,1),mar=c(4,4,2,2))
    ymax <- 0.3

    code.list <- CHEMS[is.element(CHEMS[,"Phase_I"],1),"CODE"]
    code.list <- c(code.list,CHEMS[is.element(CHEMS[,"Phase_II"],1),"CODE"])
    code.list <- sort(unique(code.list))
    code.list <- code.list[!is.na(code.list)]
    cyto.list <- CYTOTOX.ASSAYS
    non.cyto.list <- ASSAY.LIST[!is.element(ASSAY.LIST,CYTOTOX.ASSAYS)]
    hitmat <- MAT.hitcall
    hitmat[is.na(hitmat)] <- 0
    hitmat[hitmat<0] <- 0
    hitmat[is.na(MAT.tested)] <- 0
    hitmat[MAT.tested==0] <- 0
    hitmat[hitmat>0] <- 1
    hitmat.cyto <- hitmat[code.list,cyto.list]
    hitmat.non.cyto <- hitmat[code.list,non.cyto.list]
    groups <- rowSums(hitmat.cyto)
    values <- rowSums(hitmat.non.cyto) / length(non.cyto.list)

	result <- lm(values~groups)
	intercept <- result[[1]][1]
	slope <- result[[1]][2]
    boxplot(values~groups,xlab="Number of Cytotoxity Hits",ylab="Fraction of Hits",cex.axis=1.2,cex.lab=1.2,ylim=c(0,ymax),main="")
	lines(c(0,40),c(intercept,intercept+40*slope))
	rms <- summary(result)[[8]]
	text(x=2,y=0.25,paste("RMS =",format(rms,digits=2)),pos=4,cex=1.5)
    if(to.file) graphics.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# output for RTK Phase II paper
#
# ExpoDat_IVIVE_Css
# Ph2_IVIVE_Css
# PhaseIIRTK-CASQ
#--------------------------------------------------------------------------------------
RTKP2 <- function(prefix="") {
	print.current.function()
    
    #infile <- paste("../RTKP2/",prefix,".csv",sep="")
	#temp <- read.csv(infile,header=T,stringsAsFactors=F,quote="\"",comment.char="")
	#code.list <- temp[,"CODE"]
	#cname.list <- temp[,3]
	#casrn.list <- temp[,2]
	
	code.list <- CODE.LIST
	outfile <- paste("../RTKP2/",prefix,"_potentHitsNoFlags_allchems.txt",sep="")
	s <- "CODE\tCASRN\tName\tAssay\tTarget\tProcess\tAC50\n"
    cat(file=outfile,s,append=F)
	for(i in 1:length(code.list)) {
		code <- code.list[i]
		flagged.assays <- ALL.FLAGS[is.element(ALL.FLAGS[,"code"],code),"aenm"]
		if(!is.element(code,CODE.LIST)) {
			cat("Not in Current ToxCast data: ",code.list[i],":",casrn.list[i],":",cname.list[i],"[",prefix,"]\n")
		}
		else {
			casrn <- CHEMS[code,"CASRN"]
			cname <- CHEMS[code,"Name"]
			ac50 <- MAT.AC50[code,]
			hit <- MAT.hitcall[code,]
			tested <- MAT.tested[code,]
			hit[is.na(hit)] <- 0
			hit[hit<0] <- 0

			#flagged.assays <- ALL.FLAGS[is.element(ALL.FLAGS[,"code"],code),"aenm"]
			flags <- tested
			flags[] <- 1
			flags[is.element(colnames(MAT.AC50),flagged.assays)] <- 0

			mask <- hit * tested * flags
			ac50 <- ac50[as.numeric(mask)==1]
			if(length(ac50)==0) {
				cat("No unflagged hits for: ",code,":",casrn,":",cname,"[",prefix,"]\n")
			}
			else {
				anames <- names(ac50)
				values <- as.numeric(ac50)
				ix <- sort(values,index.return=T)$ix
				#if(length(ix)>10) ix <- ix[1:10]
				for(j in 1:length(ix)) {
					assay <- anames[ix[j]]
					target <- ASSAY.INFO[assay,"intended_target"]
					bioprocess <- ASSAY.INFO[assay,"biological_process"]
					s <- paste(code,"\t",casrn,"\t",cname,"\t",assay,"\t",target,"\t",bioprocess,"\t",format(values[ix[j]],digits=3),"\n",sep="")
					cat(file=outfile,s,append=T)
				}
			}
		}
	}
}		

#--------------------------------------------------------------------------------------
#
# chemical target analysis
#
#--------------------------------------------------------------------------------------
target.analysis <- function(to.file=F) {
	print.current.function()

	assay.temp <- ASSAY.INFO[,c("assay","intended_target")]
	assay.temp[,"intended_target"] <- toupper(assay.temp[,"intended_target"]) 
	fname <- "../input/chemical_targets.xlsx"
	tchems <- read.xlsx(fname)
	
	assay.temp <- assay.temp[is.element(assay.temp[,"intended_target"],tchems[,"target_gene"]),]
	n <- dim(assay.temp)[1]
	mask <- vector(length=n,mode="integer")
	mask[] <- 1
	for(i in 1:n) {
		doit <- 1
		if(length(grep("antagonist",assay.temp[i,"assay"],ignore.case=T))==1) doit <- 0
		if(length(grep("activator",assay.temp[i,"assay"],ignore.case=T))==1) doit <- 0
		mask[i] <- doit
	}
	assay.temp <- assay.temp[mask==1,]
	hit.list <- NULL
	z.list <- NULL
	gene.list <- sort(unique(assay.temp[,"intended_target"]))
	tchems <- tchems[is.element(tchems[,"target_gene"],gene.list),]
	n <- dim(tchems)[1]
	
	for(i in 1:n) {
		gene <- tchems[i,"target_gene"]
		code <- tchems[i,"CODE"]
		assays <- assay.temp[is.element(assay.temp[,"intended_target"],gene),"assay"]
		for(j in 1:length(assays)) {
			ztemp <- MAT.Z.NORM[code,assays[j]]
			#cat(code,":",gene,":",assays[j],":",ztemp,"\n")
			if(is.na(ztemp)) {
				hit.list <- c(hit.list,0)
			}
			else {
				hit.list <- c(hit.list,1)
				z.list <- c(z.list,ztemp)
			}
		}
	}	
		
	code.list.in <- sort(unique(tchems[,"CODE"]))
	
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
	cat("code.list: ",length(code.list),"\n")
	code.list.out <- code.list[!is.element(code.list,code.list.in)]
	assay.list <- sort(unique(assay.temp[,"assay"]))
	z.list.out <- NULL
	hit.list.out <- NULL
	for(i in 1:length(code.list.out)) {
		for(j in 1:length(assay.list)) {
			code <- code.list.out[i]
			assay <- assay.list[j]			
			ztemp <- MAT.Z.NORM[code,assay]
			if(is.na(ztemp)) {
				hit.list.out <- c(hit.list.out,0)
			}
			else {
				hit.list.out <- c(hit.list.out,1)
				z.list.out <- c(z.list.out,ztemp)
			}
		}
	}
	cat("Target hit mean:",mean(hit.list),"\n")
	cat("Other hit mean: ",mean(hit.list.out),"\n")
	
	ret <- t.test(z.list,z.list.out,alternative="greater")
	p.value <- ret$p.value
	group.1 <- z.list
	group.1[] <- 1
	group.2 <- z.list.out
	group.2[] <- 2
	groups <- c(group.1,group.2)
	vals <- c(z.list,z.list.out)
	
    if(to.file) {
        file <- paste("../plots/target_analysis.pdf",sep="")
        pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(1,1),mar=c(5,5,4,2))

   	boxplot(vals~groups,xlab="",ylab="Z-score",names=c("Target Class","Others"),cex.axis=1.2,cex.lab=1.2,ylim=c(-10,40),main="")
    text(1,37,format(median(z.list),digits=2))
    text(2,37,format(median(z.list.out),digits=2))
    text(0.5,37,"median:",pos=4)
    text(1,39,paste("p-value: ",format(p.value,digits=2)))
    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# bioclass analysis
#
#--------------------------------------------------------------------------------------
bioclass.analysis <- function(to.file=F) {
	print.current.function()

	bp.list.1 <- c("apoptosis up","ER stress","heat shock","microtubule up","mitochondrial disruption up","oxidative stress up")                                       
	bp.list.1.names <- c("Apoptosis Induction","ER Stress","Heat Shock","Microtubule Disruption","Mitochondrial Disruption","Oxidative Stress")   
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]

	code.list.ctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]<1000,"CODE"]
	code.list.noctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]>=1000,"CODE"]
	code.list.ctx <- code.list.ctx[is.element(code.list.ctx,code.list)]
	code.list.noctx <- code.list.noctx[is.element(code.list.noctx,code.list)]
	
	groups.1 <- NULL
	zvals.1 <- NULL
	n1 <- length(bp.list.1)
	assay.list <- NULL
	for(i in 1:n1) {
		bp <- bp.list.1[i]
		assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
		assay.list <- c(assay.list,assays)
		z <- as.numeric(as.matrix(MAT.Z.NORM[code.list.ctx,assays]))
		z <- z[!is.na(z)]
		groups <- z
		groups[] <- bp
		groups.1 <- c(groups.1,groups)
		zvals.1 <- c(zvals.1,z)
	}
		
	
  if(to.file) {
      file <- paste("../plots/bioclass_analysis.pdf",sep="")
      pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
  }
  par(mfrow=c(2,1),mar=c(5,5,2,2))

	za <- MAT.Z.NORM[code.list.ctx,assay.list]
	za[is.na(za)] <- 1000000
	za[za>3] <- 1000000
	za[za<=3] <- 1
	za[za>1] <- 0
	rs.ctx.le3 <- rowSums(za)
	z.nocellstress <- za[rs.ctx.le3==0,]
	code.list.nocellstress <- rownames(z.nocellstress)
	
	za <- MAT.Z.NORM[code.list.ctx,assay.list]
	za[is.na(za)] <- -1000000
	za[za<=3] <- 0
	za[za>0] <- 1
	rs.ctx.gt3 <- rowSums(za)

	za <- MAT.Z.NORM[code.list.noctx,assay.list]
	za[is.na(za)] <- 1000000
	za[za>3] <- 1000000
	za[za<=3] <- 1
	za[za>1] <- 0
	rs.noctx.le3 <- rowSums(za)
	za <- MAT.Z.NORM[code.list.noctx,assay.list]
	za[is.na(za)] <- -1000000
	za[za<=3] <- 0
	za[za>0] <- 1
	rs.noctx.gt3 <- rowSums(za)
	
	ctx.le3.mean <- mean(rs.ctx.le3)
	ctx.gt3.mean <- mean(rs.ctx.gt3)
	noctx.le3.mean <- mean(rs.noctx.le3)
	noctx.gt3.mean <- mean(rs.noctx.gt3)
	ctx.le3.sd <- sd(rs.ctx.le3)
	ctx.gt3.sd <- sd(rs.ctx.gt3)
	noctx.le3.sd <- sd(rs.noctx.le3)
	noctx.gt3.sd <- sd(rs.noctx.gt3)
	
	cat("Cytotox <3: ",format(ctx.le3.mean,digits=2)," : ",format(ctx.le3.sd,digits=2),"\n")
	cat("Cytotox >3: ",format(ctx.gt3.mean,digits=2)," : ",format(ctx.gt3.sd,digits=2),"\n")
	cat("No Cytotox <3: ",format(noctx.le3.mean,digits=2)," : ",format(noctx.le3.sd,digits=2),"\n")
	cat("No Cytotox >3: ",format(noctx.gt3.mean,digits=2)," : ",format(noctx.gt3.sd,digits=2),"\n")
	
	zvals.1[zvals.1>15] <- 15
 	boxplot(zvals.1~groups.1,ylab="",xlab="Z-score",names=NA,cex.axis=1.2,cex.lab=1.2,ylim=c(-10,35),main="",horizontal=T)
 	lines(c(0,0),c(0,20))
 	lines(c(3,3),c(0,20))
 	for(i in 1:length(bp.list.1)) text(18,i,bp.list.1.names[i],pos=4,cex=0.8)


	za <- MAT.Z.NORM[code.list.ctx,assay.list]
	za[is.na(za)] <- 1000000
	za[za<1000000] <- 1
	za[za>1] <- 0
	rs <- rowSums(za)
	code.list.zero <- rownames(za)[rs==0]
	breaks <- c(0:30)-1
	main <- paste("Chemicals:",length(rs))
	main <- ""
#	temp <- rs.ctx.le3+1
#	hist(temp,xlab="Cell Stress Hits Per Chemical Z<3",ylab="Chemicals",breaks=breaks,main=main,ylim=c(0,60))

	temp <- rs+1
	hist(temp,xlab="Cell Stress Hits Per Chemical Z<3",ylab="Chemicals",breaks=breaks,main=main,ylim=c(0,60))

	text(10,50,paste("Cytotoxic       Z<3:  ",format(ctx.le3.mean,digits=2)," (",format(ctx.le3.sd,digits=2),")",sep=""),pos=4)
	text(10,45,paste("Cytotoxic       Z>3:  ",format(ctx.gt3.mean,digits=2)," (",format(ctx.gt3.sd,digits=2),")",sep=""),pos=4)
	text(10,40,paste("Non-cytotoxic Z>3: ",format(noctx.gt3.mean,digits=2)," (",format(noctx.gt3.sd,digits=2),")",sep=""),pos=4)
	rs0 <- rs.ctx.le3[rs.ctx.le3==0]
	ratio <- 100*(1-length(rs0)/length(rs.ctx.le3))
	text(0,60,paste(format(ratio,digits=2),"% of chemicals have cell stress hits",sep=""),pos=4)
	
	n <- length(code.list.zero)
	for(i in 1:n) {
		code <- code.list.zero[i]
		cname <- CHEMS[code,"Name"]
		temp <- MAT.Z.NORM[code,]
		cat("\n",code,":",cname,"\n")
		for(j in 1:dim(temp)[2]) {
			if(!is.na(temp[1,j])) cat("   ",names(temp)[j],":",temp[1,j],"\n")
		}
		
		#browser()
	}


  if(to.file) dev.off()
  else browser()
}
#--------------------------------------------------------------------------------------
#
# bioclass analysis
#
#--------------------------------------------------------------------------------------
bioclass.hm <- function(to.file=F) {
	print.current.function()

	bp.list <- c("proliferation decrease","cytotoxicity SRB","cytotoxicity BLA","apoptosis up","oxidative stress up","ER stress","heat shock","mitochondrial disruption up","microtubule up")                                       
	bp.list.names <- c("Proliferation decrease","Cytotoxicty BLA","Cytotoxicity SRB","Apoptosis Induction","Oxidative Stress","ER Stress","Heat Shock","Mitochondrial Disruption","Microtubule Disruption")   
	bp.colors <- c("orange","red","violet","blue","gray","yellow","green","black","cyan")
	
	assay.list <- NULL
	assay.bp <- NULL
	assay.color <- NULL
	nbp <- length(bp.list)
	for(i in 1:nbp) {
		bp <- bp.list[i]
		bpcol <- bp.colors[i]
		assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
		assay.list <- c(assay.list,assays)
		temp <- assays
		temp[] <- bp
		assay.bp <- c(assay.bp,temp)
		temp <- assays
		temp[] <- bpcol
		assay.color <- c(assay.color,temp)
	}
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
	code.list.ctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]<1000,"CODE"]
	code.list.noctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]>=1000,"CODE"]
	code.list.ctx <- code.list.ctx[is.element(code.list.ctx,code.list)]
	code.list.noctx <- code.list.noctx[is.element(code.list.noctx,code.list)]
	
	z1 <- MAT.Z.NORM[code.list.ctx,assay.list]
	z1 <- MAT.Z.NORM[code.list,assay.list]
	z <- z1
	z[is.na(z)] <- -1000000
	z[z<=3] <- 1
	z[z>3] <- 2
	z[is.na(z1)] <- 0
	
    if(to.file) {
        file <- paste("../plots/bioclass_hm.pdf",sep="")
        pdf(file=file,width=8,height=8,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result <- heatmap(t(as.matrix(z)),margins=c(5,10),scale="none",main="Cytox and Cell Stress Z classes",
                      xlab="",ylab="",cexCol=0.005,cexRow=0.5,col=brewer.pal(9,"Reds"),
                      hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
                      RowSideColors=assay.color,Rowv=NA)

	dmat <- as.matrix(z)
    cl <- hclust(d=dist(dmat),method="ward.D")
    hcut <- 1
    nlevel <- 50
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
    	code <- rownames(clout)[i]
        clout[i,1] <- code
        clout[i,2] <- CHEMS[code,"Name"]
        clout[i,3] <- CHEMS[code,"use_category"]
        clout[i,4] <- CHEMS[code,"use_super_category"]
        clout[i,5] <- CHEMS[code,"structure_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","Level_1")

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()

    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
    
    clout <- cbind(clout,z)

	fname <- paste("../output/cell_stress_cytotox_clusters.xlsx",sep="")
	write.xlsx(clout,file=fname, row.names=F)

    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# bioclass physchem
#
#--------------------------------------------------------------------------------------
bioclass.physchem <- function(to.file=F) {
	print.current.function()

	fname <- paste("../physchem/toxcast_physchem_cellstress_DFT.xlsx",sep="")
	pchem <- read.xlsx(fname)
	rownames(pchem) <- pchem[,"CODE"]
	pchem.names <- names(pchem)[29:dim(pchem)[2]]
	npchem <- length(pchem.names)

	fname <- paste("../output/cell_stress_cytotox_clusters.xlsx",sep="")
	class.mat <- read.xlsx(fname)
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
	code.list <- code.list[is.element(code.list,rownames(pchem))]
	pchem <- pchem[code.list,]
	
	class.list <- code.list
	class.list[] <- "NoCTX"
	let.list <- c("A","B","C","D","E","F")
	for(i in 1:6) {
		code.let.list <- class.mat[is.element(class.mat[,"ClassLetter"],let.list[i]),"CODE"]
		class.list[is.element(code.list,code.let.list)] <- let.list[i]
	}

    if(to.file) {
        file <- paste("../plots/bioclass_physchem.pdf",sep="")
        pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,1),mar=c(3,5,2,2))

	for(i in 1:npchem) {
		varname <- pchem.names[i]
		vals <- pchem[,varname]
		qs <- quantile(vals, probs=seq(0,1,0.01),na.rm=T)[100]
		vtemp <- vals[vals<qs]
		ctemp <- class.list[vals<qs]
		cat(varname,":",length(vals),":",length(vtemp),"\n")
		bp <- boxplot(vtemp~ctemp,ylab="",xlab="",cex.axis=1.,cex.lab=1.,main=varname,horizontal=F)
		ymax <- as.numeric(quantile(vtemp, probs=seq(0,1,0.1),na.rm=T)[11])
		for(j in 1:6) {
			x <- vtemp[ctemp=="NoCTX"]
			y <- vtemp[ctemp==let.list[j]]
			x <- x[!is.na(x)]
			y <- y[!is.na(y)]
			
			ret <- t.test(x,y)
			p.value <- ret$p.value

			color <- "blue"
			if(mean(y)<mean(x)) color <- "red"
			if(p.value<0.05) text(j,ymax,format(p.value,digits=2),col=color)
		}
			
   		if(!to.file) browser()	
   	}

    if(to.file) dev.off()
    else browser()
	
}
#--------------------------------------------------------------------------------------
#
# physchem.analysis
#
#--------------------------------------------------------------------------------------
physchem.analysis.1 <- function(to.file=F) {
	print.current.function()

	fname <- paste("../physchem/toxcast_physchem_cellstress_DFT.xlsx",sep="")
	pchem <- read.xlsx(fname)
	rownames(pchem) <- pchem[,"CODE"]
	pchem.names <- names(pchem)[29:dim(pchem)[2]]
	npchem <- length(pchem.names)

	fname <- paste("../output/cell_stress_cytotox_clusters.xlsx",sep="")
	class.mat <- read.xlsx(fname)
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
	code.list <- code.list[is.element(code.list,rownames(pchem))]
	pchem <- pchem[code.list,]
	
	class.list <- code.list
	class.list[] <- "Non-Cytotoxic"
	let.list <- c("A","B","C","D","E","F")
	for(i in 1:6) {
		code.let.list <- class.mat[is.element(class.mat[,"ClassLetter"],let.list[i]),"CODE"]
		class.list[is.element(code.list,code.let.list)] <- "Cytotoxic"#let.list[i]
	}

    if(to.file) {
        file <- paste("../plots/physchem.analysis_1.pdf",sep="")
        pdf(file=file,width=7,height=7,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    par(mfrow=c(3,2),mar=c(3,5,2,2))

	for(i in 1:npchem) {
		varname <- pchem.names[i]
		vals <- pchem[,varname]

		qs <- quantile(vals, probs=seq(0,1,0.01),na.rm=T)[100]
		vtemp <- vals[vals<qs]
		ctemp <- class.list[vals<qs]
		x <- vtemp[ctemp=="Non-Cytotoxic"]
		y <- vtemp[ctemp=="Cytotoxic"]
		x <- x[!is.na(x)]
		y <- y[!is.na(y)]
			
		ret <- t.test(x,y)
		p.value <- ret$p.value

		main <- paste(varname,":",format(p.value,digits=2))
		cat(varname,":",length(vals),":",length(vtemp),"\n")
		bp <- boxplot(vtemp~ctemp,ylab="",xlab="",cex.axis=1.,cex.lab=1.,main=main,horizontal=F)
		ymax <- as.numeric(quantile(vtemp, probs=seq(0,1,0.1),na.rm=T)[11])

   		if(!to.file) browser()	
   	}

    if(to.file) dev.off()
    else browser()
	
}
#--------------------------------------------------------------------------------------
#
# bioclass physchem analysis
#
#--------------------------------------------------------------------------------------
bioclass.physchem.hm <- function(to.file=F) {
	print.current.function()

	bp.list <- c("proliferation decrease","cytotoxicity SRB","cytotoxicity BLA","apoptosis up","oxidative stress up","ER stress","heat shock","mitochondrial disruption up","microtubule up")                                       
	bp.list.names <- c("Proliferation decrease","Cytotoxicty BLA","Cytotoxicity SRB","Apoptosis Induction","Oxidative Stress","ER Stress","Heat Shock","Mitochondrial Disruption","Microtubule Disruption")   
	bp.colors <- c("orange","red","violet","blue","gray","yellow","green","black","cyan","white")

assay.list <- c(
"BSK_hDFCGF_Proliferation_down",
"BSK_SAg_Proliferation_down",
"BSK_3C_Proliferation_down",
"BSK_CASM3C_Proliferation_down",
"APR_HepG2_CellLoss_72h_dn",
"APR_HepG2_CellLoss_24h_dn",
"ACEA_T47D_80hr_Negative",
"BSK_3C_Vis_down")

assay.list <- c(assay.list,
"BSK_3C_SRB_down",
"BSK_SAg_SRB_down",
"BSK_LPS_SRB_down",
"BSK_SAg_PBMCCytotoxicity_down",
"BSK_hDFCGF_SRB_down",
"BSK_4H_SRB_down",
"BSK_BE3C_SRB_down",
"BSK_CASM3C_SRB_down",
"BSK_KF3CT_SRB_down"
)

assay.list <- c(assay.list,
"Tox21_AR_BLA_Antagonist_viability",
"Tox21_PPARd_BLA_Agonist_viability",
"Tox21_VDR_BLA_Agonist_viability",
"Tox21_NFkB_BLA_agonist_viability",
"Tox21_p53_BLA_p1_viability",
"Tox21_p53_BLA_p2_viability",
"Tox21_p53_BLA_p3_viability",
"Tox21_p53_BLA_p4_viability",
"Tox21_p53_BLA_p5_viability",
"Tox21_MitochondrialToxicity_viability",
"Tox21_ESRE_BLA_viability",
"Tox21_FXR_BLA_agonist_viability",
"Tox21_FXR_BLA_antagonist_viability",
"Tox21_GR_BLA_Antagonist_viability",
"Tox21_HSE_BLA_agonist_viability",
"Tox21_VDR_BLA_antagonist_viability",
"Tox21_PPARg_BLA_antagonist_viability",
"Tox21_ERa_BLA_Antagonist_viability"
)

assay.list <- c(assay.list,
"APR_HepG2_p53Act_24h_up",
"APR_HepG2_p53Act_72h_up",
"Tox21_p53_BLA_p1_ratio",
"Tox21_p53_BLA_p2_ratio",
"Tox21_p53_BLA_p3_ratio",
"Tox21_p53_BLA_p4_ratio",
"Tox21_p53_BLA_p5_ratio"
)

assay.list <- c(assay.list,
"APR_HepG2_StressKinase_24h_up",
"APR_HepG2_StressKinase_72h_up",
"APR_HepG2_OxidativeStress_24h_up",
"APR_HepG2_OxidativeStress_72h_up",
"ATG_MRE_CIS_up",
"ATG_NRF2_ARE_CIS_up"
)

assay.list <- c(assay.list,
"ATG_Xbp1_CIS_up",
"ATG_SREBP_CIS_up",
"Tox21_ESRE_BLA_ratio"
)

assay.list <- c(assay.list,
"ATG_HSE_CIS_up",
"Tox21_HSE_BLA_agonist_ratio"
)

assay.list <- c(assay.list,
"Tox21_MitochondrialToxicity_ratio",
"NVS_MP_hPBR",
"NVS_MP_rPBR",
"NVS_ENZ_rMAOAC",
"NVS_ENZ_rMAOAP",
"NVS_ENZ_rMAOBC",
"NVS_ENZ_rMAOBP",
"APR_HepG2_MitoMass_24h_up",
"APR_HepG2_MitoMass_72h_up"
)

assay.list <- c(assay.list,
"APR_HepG2_MicrotubuleCSK_24h_up",
"APR_HepG2_MicrotubuleCSK_72h_up"
)

#	assay.list <- NULL
	assay.bp <- NULL
	assay.color <- NULL
	nbp <- length(bp.list)
	for(i in 1:nbp) {
		bp <- bp.list[i]
		bpcol <- bp.colors[i]
		assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
#		assay.list <- c(assay.list,assays)
		temp <- assays
		temp[] <- bp
		assay.bp <- c(assay.bp,temp)
		temp <- assays
		temp[] <- bpcol
		assay.color <- c(assay.color,temp)
		cat(bp,":",length(temp),"\n")
	}
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
#	browser()
	fname <- paste("../physchem/toxcast_physchem_cellstress_DFT.xlsx",sep="")
	pchem <- read.xlsx(fname)
	pcode <- pchem[,"CODE"]	
	rownames(pchem) <- pchem[,"CODE"]
	pchem.names <- names(pchem)[29:dim(pchem)[2]]
	npchem <- length(pchem.names)
	code.list <- code.list[is.element(code.list,pcode)]
	#code.list.ctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]<1000,"CODE"]
	#code.list.noctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]>=1000,"CODE"]
	#code.list.ctx <- code.list.ctx[is.element(code.list.ctx,code.list)]
	#code.list.noctx <- code.list.noctx[is.element(code.list.noctx,code.list)]
	
	z1 <- MAT.Z.NORM[code.list,assay.list]
	z <- z1
	z[is.na(z)] <- -1000000
	z[z<=3] <- 1
	z[z>3] <- 2
	z[is.na(z1)] <- 0
	
	pchem.list <- c("SASA","volume","mol.MW","QPlogPo.w","Marvin.logD.pH7.3","DFT.EA","DFT.softness","DFT.hardness","DFT.polarizability")
	pchem.color <- pchem.list
	pchem.color[] <- "purple"
	pchem1 <- pchem[code.list,pchem.list]
	np <- length(pchem.list)
	for(i in 1:np) {
		varname <- pchem.list[i]
		temp <- pchem1[,varname]
		if(varname=="DFT.hardness") temp <- -temp
		temp1 <- temp
		temp1[] <- 0
		cutoff <- as.numeric(quantile(temp, probs=seq(0,1,0.1),na.rm=T)[10])
		temp1[temp>cutoff] <- 1
		pchem1[,pchem.list[i]] <- temp1
	}
	
	
	#assay.color <- c(assay.color,pchem.color)
	#z <- cbind(z,pchem1)
	
	
#	rs <- rowSums(z)
#	index.1 <- sort(rs,index.return=T)$ix
	
	atemp <- names(MAT.Z.NORM)
	atemp <- atemp[!is.element(atemp,assay.list)]
	z1 <- MAT.Z.NORM[code.list,atemp]
#	z1 <- z1[index.1,]
#	z <- z[index.1,]
	z2 <- z1
	z2[is.na(z2)] <- -1000000
	z2[z2<=3] <- 1
	z2[z2>3] <- 1
	z2[is.na(z1)] <- 0
	rs <- rowSums(z2)
#	index <- sort(rs,index.return=T)$ix
#	z <- z[index,]
#	rs <- rs[index]
	
	colCol <- vector(length=length(rs),mode="character")
	colCol[] <- "white"
	for(i in 1:length(rs)) {
		if(rs[i]<10) colCol[i] <- "blue"
		else if(rs[i]<50) colCol[i] <- "green"
		else if(rs[i]<100) colCol[i] <- "yellow"
		else if(rs[i]<150) colCol[i] <- "orange"
		else  colCol[i] <- "red"
	}
	
    if(to.file) {
        file <- paste("../plots/bioclass_physchem_hm.pdf",sep="")
        pdf(file=file,width=8,height=8,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    #z <- z[1:50,]
    result <- heatmap(t(as.matrix(z)),margins=c(5,10),scale="none",main="",
                      xlab="",ylab="",cexCol=0.005,cexRow=0.5,col=brewer.pal(9,"Reds"),
                      hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
                      RowSideColors=assay.color,ColSideColors=colCol,Rowv=NA)

	ord <- rev(result$colInd)
	
	dmat <- as.matrix(z)
    cl <- hclust(d=dist(dmat),method="ward.D")
    hcut <- 1
    nlevel <- 50
    clcut <- cutree(cl,h=hcut)
    clout <- cbind(clcut,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout <- cbind(clout,clcut)
    clout<- as.data.frame(clout)
    for(i in 1:length(clcut)) {
    	code <- rownames(clout)[i]
        clout[i,1] <- code
        clout[i,2] <- CHEMS[code,"Name"]
        clout[i,3] <- CHEMS[code,"use_category"]
        clout[i,4] <- CHEMS[code,"use_super_category"]
        clout[i,5] <- CHEMS[code,"structure_category"]
    }
    names(clout) <- c("CODE","Name","use_category","use_super_category","structure_category","sortOrder","Level_1")
    for(i in 1:length(clcut)) {
    	code <- rownames(z)[ord[i]]
        clout[is.element(clout[,"CODE"],code),"sortOrder"] <- i
    }
	#browser()

    cat("Finished preping clusters for hcut: ",hcut,"\n")
    flush.console()

    for(hcut in 2:nlevel) {
        clcut <- cutree(cl,h=hcut)
        clout <- cbind(clout,clcut)
        names(clout)[dim(clout)[2]] <- paste("Level_",hcut,sep="")
        cat("Finished preping clusters for hcut: ",hcut,"\n")
        flush.console()
    }
    
    clout <- cbind(clout,z)

	fname <- paste("../output/cell_stress_cytotox_physchem_clusters.xlsx",sep="")
	write.xlsx(clout,file=fname, row.names=F)

    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# bioclass physchem analysis
#
#--------------------------------------------------------------------------------------
bioclass.physchem.order.hm <- function(to.file=F) {
	print.current.function()

	bp.list <- c("proliferation decrease","cytotoxicity SRB","cytotoxicity BLA","apoptosis up","oxidative stress up","ER stress","heat shock","mitochondrial disruption up","microtubule up")                                       
	bp.list.names <- c("Proliferation decrease","Cytotoxicty BLA","Cytotoxicity SRB","Apoptosis Induction","Oxidative Stress","ER Stress","Heat Shock","Mitochondrial Disruption","Microtubule Disruption")   
	bp.colors <- c("orange","red","violet","blue","gray","yellow","green","black","cyan","white")

assay.list <- c(
"APR_HepG2_CellLoss_72h_dn",
"BSK_hDFCGF_Proliferation_down",
"BSK_SAg_Proliferation_down",
"BSK_3C_Proliferation_down",
"BSK_CASM3C_Proliferation_down",
"APR_HepG2_CellLoss_24h_dn",
"ACEA_T47D_80hr_Negative",
"BSK_3C_Vis_down")

assay.list <- c(assay.list,
"BSK_3C_SRB_down",
"BSK_SAg_SRB_down",
"BSK_LPS_SRB_down",
"BSK_SAg_PBMCCytotoxicity_down",
"BSK_hDFCGF_SRB_down",
"BSK_4H_SRB_down",
"BSK_BE3C_SRB_down",
"BSK_CASM3C_SRB_down",
"BSK_KF3CT_SRB_down"
)

assay.list <- c(assay.list,
"Tox21_MitochondrialToxicity_viability",
"Tox21_AR_BLA_Antagonist_viability",
"Tox21_PPARd_BLA_Agonist_viability",
"Tox21_VDR_BLA_Agonist_viability",
"Tox21_NFkB_BLA_agonist_viability",
"Tox21_p53_BLA_p1_viability",
"Tox21_p53_BLA_p2_viability",
"Tox21_p53_BLA_p3_viability",
"Tox21_p53_BLA_p4_viability",
"Tox21_p53_BLA_p5_viability",
"Tox21_ESRE_BLA_viability",
"Tox21_FXR_BLA_agonist_viability",
"Tox21_FXR_BLA_antagonist_viability",
"Tox21_GR_BLA_Antagonist_viability",
"Tox21_HSE_BLA_agonist_viability",
"Tox21_VDR_BLA_antagonist_viability",
"Tox21_PPARg_BLA_antagonist_viability",
"Tox21_ERa_BLA_Antagonist_viability"
)

assay.list <- c(assay.list,
"APR_HepG2_p53Act_72h_up",
"APR_HepG2_p53Act_24h_up",
"Tox21_p53_BLA_p1_ratio",
"Tox21_p53_BLA_p2_ratio",
"Tox21_p53_BLA_p3_ratio",
"Tox21_p53_BLA_p4_ratio",
"Tox21_p53_BLA_p5_ratio"
)

assay.list <- c(assay.list,
"ATG_NRF2_ARE_CIS_up",
"ATG_MRE_CIS_up",
"APR_HepG2_OxidativeStress_72h_up",
"APR_HepG2_OxidativeStress_24h_up",
"APR_HepG2_StressKinase_24h_up",
"APR_HepG2_StressKinase_72h_up"
)

assay.list <- c(assay.list,
"ATG_SREBP_CIS_up",
"ATG_Xbp1_CIS_up",
"Tox21_ESRE_BLA_ratio"
)

assay.list <- c(assay.list,
"ATG_HSE_CIS_up",
"Tox21_HSE_BLA_agonist_ratio"
)

assay.list <- c(assay.list,
"Tox21_MitochondrialToxicity_ratio",
"NVS_MP_hPBR",
"NVS_MP_rPBR",
"NVS_ENZ_rMAOAC",
"NVS_ENZ_rMAOAP",
"NVS_ENZ_rMAOBC",
"NVS_ENZ_rMAOBP",
"APR_HepG2_MitoMass_24h_up",
"APR_HepG2_MitoMass_72h_up"
)

assay.list <- c(assay.list,
"APR_HepG2_MicrotubuleCSK_72h_up",
"APR_HepG2_MicrotubuleCSK_24h_up"
)

#	assay.list <- NULL
	assay.bp <- NULL
	assay.color <- NULL
	nbp <- length(bp.list)
	for(i in 1:nbp) {
		bp <- bp.list[i]
		bpcol <- bp.colors[i]
		assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
#		assay.list <- c(assay.list,assays)
		temp <- assays
		temp[] <- bp
		assay.bp <- c(assay.bp,temp)
		temp <- assays
		temp[] <- bpcol
		assay.color <- c(assay.color,temp)
		cat(bp,":",length(temp),"\n")
	}
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
#	browser()
	fname <- paste("../physchem/toxcast_physchem_cellstress_DFT.xlsx",sep="")
	pchem <- read.xlsx(fname)
	pcode <- pchem[,"CODE"]	
	rownames(pchem) <- pchem[,"CODE"]
	pchem.names <- names(pchem)[29:dim(pchem)[2]]
	npchem <- length(pchem.names)
	code.list <- code.list[is.element(code.list,pcode)]
	#code.list.ctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]<1000,"CODE"]
	#code.list.noctx <- CYTOTOX[CYTOTOX[,"cytotox_median_um"]>=1000,"CODE"]
	#code.list.ctx <- code.list.ctx[is.element(code.list.ctx,code.list)]
	#code.list.noctx <- code.list.noctx[is.element(code.list.noctx,code.list)]
	
	z1 <- MAT.Z.NORM[code.list,assay.list]
	z <- z1
	z[is.na(z)] <- -1000000
	z[z<=3] <- 1
	z[z>3] <- 2
	z[is.na(z1)] <- 0
	
	pchem.list <- c("SASA","volume","mol.MW","QPlogPo.w","Marvin.logD.pH7.3","DFT.EA","DFT.softness","DFT.hardness","DFT.polarizability")
	pchem.color <- pchem.list
	pchem.color[] <- "purple"
	pchem1 <- pchem[code.list,pchem.list]
	np <- length(pchem.list)
	for(i in 1:np) {
		varname <- pchem.list[i]
		temp <- pchem1[,varname]
		if(varname=="DFT.hardness") temp <- -temp
		temp1 <- temp
		temp1[] <- 0
		cutoff <- as.numeric(quantile(temp, probs=seq(0,1,0.1),na.rm=T)[10])
		temp1[temp>cutoff] <- 1
		pchem1[,pchem.list[i]] <- temp1
	}
	
	
	#assay.color <- c(assay.color,pchem.color)
	#z <- cbind(z,pchem1)
	rs <- rowSums(z)
	index.1 <- sort(rs,index.return=T)$ix
	
	atemp <- names(MAT.Z.NORM)
	atemp <- atemp[!is.element(atemp,assay.list)]
	z1 <- MAT.Z.NORM[code.list,atemp]
	z1 <- z1[index.1,]
	z <- z[index.1,]
	z2 <- z1
	z2[is.na(z2)] <- -1000000
	z2[z2<=3] <- 1
	z2[z2>3] <- 1
	z2[is.na(z1)] <- 0
	rs <- rowSums(z2)
	index <- sort(rs,index.return=T)$ix
	z <- z[index,]
	rs <- rs[index]
	
	colCol <- vector(length=length(rs),mode="character")
	colCol[] <- "white"
	for(i in 1:length(rs)) {
		if(rs[i]<10) colCol[i] <- "blue"
		else if(rs[i]<50) colCol[i] <- "green"
		else if(rs[i]<100) colCol[i] <- "yellow"
		else if(rs[i]<150) colCol[i] <- "orange"
		else  colCol[i] <- "red"
	}
    if(to.file) {
        file <- paste("../plots/bioclass_physchem_ordered_hm.pdf",sep="")
        pdf(file=file,width=8,height=8,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
    result <- heatmap(t(as.matrix(z)),margins=c(5,10),scale="none",main="",
                      xlab="",ylab="",cexCol=0.005,cexRow=0.5,col=brewer.pal(9,"Reds"),
                      hclustfun=function(x) hclust(d=dist(x),method="ward.D"),keep.dendro=T,verbose=F,
                      RowSideColors=assay.color,ColSideColors=colCol,Rowv=NA,Colv=NA)


    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# stress.potency
#
#--------------------------------------------------------------------------------------
stress.potency <- function(to.file=F) {
	print.current.function()

	bp.list <- c("apoptosis up","oxidative stress up","ER stress","heat shock","mitochondrial disruption up","microtubule up")

	assay.list <- NULL
	nbp <- length(bp.list)
	for(i in 1:nbp) {
		bp <- bp.list[i]
		assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
		assay.list <- c(assay.list,assays)
	}
	
	rs <- rowSums(MAT.tested)
	code.list <- CODE.LIST[rs>500]
	hitmat <- MAT.hitcall[code.list,]
	ac50mat <- MAT.AC50[code.list,]
	hitmat[is.na(hitmat)] <- 0
	ac50mat[is.na(ac50mat)] <- 1000000
	ac50mat[hitmat==0] <- 1000000
	ac50mat <- -log10(ac50mat)+6
	potency <- apply(ac50mat[,assay.list],FUN=sum,MARGIN=1)
	
    if(to.file) {
        file <- paste("../plots/stress.potency.pdf",sep="")
        pdf(file=file,width=8,height=8,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
	par(mfrow=c(3,2),mar=c(5,5,2,2))

	bp.list <- sort(unique(ASSAY.INFO[,"biological_process"]))
	nbp <- length(bp.list)
	for(i in 1:nbp) {
		bp <- bp.list[i]
		assays <- ASSAY.INFO[is.element(ASSAY.INFO[,"biological_process"],bp),"assay"]
		print(bp)
		if(length(assays)>1) {
			hits <- hitmat[,assays]
			rs <- rowSums(hits)
			ret <- lm(rs~potency)
			intercept <- ret[[1]][1]
			slope <- ret[[1]][2]
			x0 <- 0
			x1 <- 100
			y0 <- (x0-intercept)*slope
			y1 <- (x1-intercept)*slope
			r2 <- summary(ret)[[8]]
			main <- paste(bp," : R2=",format(r2,digits=2),sep="")

			plot(rs~potency,xlab="Maximum Potency",ylab="Hits",main=main,cex.lab=1.5,cex.axis=1.2)
			lines(c(x0,x1),c(y0,y1))
			if(!to.file) browser()
		}
	}

    if(to.file) dev.off()
    else browser()
}
#--------------------------------------------------------------------------------------
#
# AR cytotox values
#
#--------------------------------------------------------------------------------------
mda.kb2 <- function(to.file=F) {
	print.current.function()

	fname <- "../MDA_KB2/AR_superMatrix_MDA_KB2.xlsx"
	ar.data <- read.xlsx(fname)
	ar.data <- ar.data[!is.na(ar.data[,"DSSTox_GSID"]),]
	code.list <- ar.data[,"CODE"]
	rownames(ar.data) <- ar.data[,"CODE"]
	assay.list <- c("NVS_NR_hAR","NVS_NR_cAR","OT_AR_ARSRC1_0480","OT_AR_ARSRC1_0960","ATG_AR_TRANS_up","Tox21_AR_BLA_Agonist_ratio","Tox21_AR_LUC_MDAKB2_Agonist","Tox21_AR_BLA_Antagonist_ratio","Tox21_AR_LUC_MDAKB2_Antagonist")
	
	assay.list.ctx <- c(
	"APR_HepG2_CellLoss_72h_dn",
	"BSK_hDFCGF_Proliferation_down",
	"BSK_SAg_Proliferation_down",
	"BSK_3C_Proliferation_down",
	"BSK_CASM3C_Proliferation_down",
	"APR_HepG2_CellLoss_24h_dn",
	"ACEA_T47D_80hr_Negative",
	"BSK_3C_Vis_down")

	assay.list.ctx <- c(assay.list.ctx,
	"BSK_3C_SRB_down",
	"BSK_SAg_SRB_down",
	"BSK_LPS_SRB_down",
	"BSK_SAg_PBMCCytotoxicity_down",
	"BSK_hDFCGF_SRB_down",
	"BSK_4H_SRB_down",
	"BSK_BE3C_SRB_down",
	"BSK_CASM3C_SRB_down",
	"BSK_KF3CT_SRB_down"
	)

	assay.list.ctx <- c(assay.list.ctx,
	"Tox21_MitochondrialToxicity_viability",
	"Tox21_AR_BLA_Antagonist_viability",
	"Tox21_PPARd_BLA_Agonist_viability",
	"Tox21_VDR_BLA_Agonist_viability",
	"Tox21_NFkB_BLA_agonist_viability",
	"Tox21_p53_BLA_p1_viability",
	"Tox21_p53_BLA_p2_viability",
	"Tox21_p53_BLA_p3_viability",
	"Tox21_p53_BLA_p4_viability",
	"Tox21_p53_BLA_p5_viability",
	"Tox21_ESRE_BLA_viability",
	"Tox21_FXR_BLA_agonist_viability",
	"Tox21_FXR_BLA_antagonist_viability",
	"Tox21_GR_BLA_Antagonist_viability",
	"Tox21_HSE_BLA_agonist_viability",
	"Tox21_VDR_BLA_antagonist_viability",
	"Tox21_PPARg_BLA_antagonist_viability",
	"Tox21_ERa_BLA_Antagonist_viability"
	)

	ac50.ar <- MAT.AC50[code.list,assay.list]
	top.ar <- MAT.T[code.list,assay.list]
	w.ar <- MAT.W[code.list,assay.list]

	ac50.ctx <- MAT.AC50[code.list,assay.list.ctx]
	top.ctx <- MAT.T[code.list,assay.list.ctx]
	w.ctx <- MAT.W[code.list,assay.list.ctx]
	
	all.ctx <- cbind(CHEMS[code.list,],ac50.ctx)
	fname <- "../MDA_KB2/cytotox_AC50.xlsx"
	write.xlsx(all.ctx,fname,rownames=F)
    if(to.file) {
        file <- paste("../MDA_KB2/ar_summary.pdf",sep="")
        pdf(file=file,width=8,height=8,pointsize=12,bg="white",paper="letter",pagecentre=T)
    }
	par(mfrow=c(1,1),mar=c(5,5,2,2))

	nchem <- length(code.list)
	nassay <- length(assay.list)
	nassay.ctx <- length(assay.list.ctx)
	pch.list <- c(8,9,21,23,23,24,24,25,25)
	col.list <- c("red","orange","black","purple","white","gray","green","yellow","blue")
	xmin <- 1e-6
	deltay <- 5
	ymax <- 200
	for(i in 1:nchem) {
		code <- code.list[i]
		casrn <- ar.data[code,"CASRN"]
		cname <- ar.data[code,"Name"]
		plot(1~1,type="n",xlab="AC50 uM",ylab="T",main=paste(casrn,":",cname),log="x",xlim=c(xmin,1e6),ylim=c(0,200),cex.lab=1.5,cex.axis=1.2)
		rect(100,0,1000000,ymax,col="lightgray",border="black")
		for(j in 1:nassay) {
			assay <- assay.list[j]
			ac50 <- ac50.ar[code,assay]
			top <- top.ar[code,assay]
			points(top~ac50,bg=col.list[j],pch=pch.list[j],cex=1)
			y <- 210-deltay*j
			points(y~xmin,bg=col.list[j],pch=pch.list[j],cex=1)
			text(xmin,y,assay,pos=4,cex=0.8)
		}
		y <- y-deltay
		points(y~xmin,col="orange",cex=2,pch="*")
		text(xmin,y,"proliferation decrease",pos=4,cex=0.8)
		y <- y-deltay
		points(y~xmin,col="pink",cex=2,pch="*")
		text(xmin,y,"cytotox SRB",pos=4,cex=0.8)
		y <- y-deltay
		points(y~xmin,col="red",cex=2,pch="*")
		text(xmin,y,"cytotox BLA",pos=4,cex=0.8)
		
		for(j in 1:nassay.ctx) {
			assay <- assay.list.ctx[j]
			ac50 <- ac50.ctx[code,assay]
			top <- top.ctx[code,assay]
			top <- 5+j
			color <- "orange"
			if(j>8) color <- "pink"
			if(j>17) color <- "red"
			points(top~ac50,col=color,cex=2,pch="*")
		}
		
		x <- ar.data[code,"MDA_KB2_agonist_LOEC"]*1000000
		if(!is.na(x)) lines(c(x,x),c(0,100),col="green",lwd=3)
		x <- ar.data[code,"MDA_KB2_antagonist_LOEC"]*1000000
		if(!is.na(x)) lines(c(x,x),c(0,100),col="blue",lwd=3)

		x <- ar.data[code,"MDA_KB2_agonist_cytotox_LOEC"]*1000000
		if(!is.na(x)) lines(c(x,x),c(0,100),col="green",lwd=3,lty="dashed")
		x <- ar.data[code,"MDA_KB2_antagonist_cytotox_LOEC"]*1000000
		if(!is.na(x)) lines(c(x,x),c(0,100),col="blue",lwd=3,lty="dashed")

		y <- y-deltay
		lines(c(xmin,xmin*5),c(y,y),col="green",lwd=3)
		text(xmin*5,y,"MDA KB2 agonist",pos=4,cex=0.8)
		y <- y-deltay
		lines(c(xmin,xmin*5),c(y,y),col="green",lwd=3,lty="dashed")
		text(xmin*5,y,"MDA KB2 agonist cytotox",pos=4,cex=0.8)

		y <- y-deltay
		lines(c(xmin,xmin*5),c(y,y),col="blue",lwd=3)
		text(xmin*5,y,"MDA KB2 antagonist",pos=4,cex=0.8)
		y <- y-deltay
		lines(c(xmin,xmin*5),c(y,y),col="blue",lwd=3,lty="dashed")
		text(xmin*5,y,"MDA KB2 antagonist cytotox",pos=4,cex=0.8)


		if(!to.file) browser()
	}
	if(to.file) dev.off()
}
#--------------------------------------------------------------------------------------
#
# pfaa data
#
#--------------------------------------------------------------------------------------
pfaa <- function(to.file=F) {
	print.current.function()

	casrn.list <- c("375-85-9","375-95-1","335-76-2","2058-94-8","307-55-1","335-67-1","307-24-4")
	code.list <- c("C375859","C375951","C335762","C2058948","C307551","C335671","C307244")
	code.list <- code.list[is.element(code.list,CODE.LIST)]
	
	atemp <- MAT.AC50[code.list,]
	ztemp <- MAT.Z.NORM[code.list,]
	
	all.data <- cbind(CHEMS[code.list,],atemp,ztemp)
	
	fname <- "../output/pfaa.xlsx"
	write.xlsx(all.data,fname,rownames=F)
}
#--------------------------------------------------------------------------------------
#
# drug.Z - do some analyses for Dan V.
#
#--------------------------------------------------------------------------------------
drug.Z <- function() {
	print.current.function()

	casrn.list <- c("10161-33-8","10161-33-8","103-90-2","114084-78-5","130929-57-6","15307-86-5","15687-27-1","169590-42-5","23241-92-8","25812-30-0","38677-85-9","39413-69-5","50-02-2","52-01-7","525-66-6","54910-89-3","57-27-2","57-63-6","58-73-1","60-54-8","60207-90-1","65277-42-1","66357-35-5","68291-97-4","69-53-4","723-46-6","79617-96-2")
	code.list <- c("C10161338","C10161338","C103902","C114084785","C130929576","C15307865","C15687271","C169590425","C23241928","C25812300","C38677859","C39413695","C50022","C52017","C525666","C54910893","C57272","C57636","C58731","C60548","C60207901","C65277421","C66357355","C68291974","C69534","C723466","C79617962","C147240","C15307796","C169590425","C2392394","C25812300","C298464","C52017","C57636","C60207901","C60548","C65277421","C66357355")
	code.list <- sort(unique(code.list))
	code.list <- code.list[is.element(code.list,CODE.LIST)]
	atemp <- MAT.AC50[code.list,]
	ztemp <- MAT.Z.NORM[code.list,]
	
	all.data <- cbind(CHEMS[code.list,],atemp,ztemp)
	
	fname <- "../output/drugZ.xlsx"
	write.xlsx(all.data,fname,rownames=F)
}


