# Required packages
library(data.table)
# library(microbenchmark)
library(reshape2)
library(parallel)
library(RColorBrewer)
library(RMySQL)
library(numDeriv) ## needed to get prediction intervals
library(MASS) ## needed to get prediction intervals

## Utility functions for ToxCast Data Pipeline

#-------------------------------------------------------------------------------
# lw: Length of which is true
#-------------------------------------------------------------------------------

lw <- function(x) length(which(x))

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# lu: Length of unique
#-------------------------------------------------------------------------------

lu <- function(x) length(unique(x))

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# tmquery: Query toxminer databases
#-------------------------------------------------------------------------------

tmquery <- function(query, db) {
  
  ### This function takes a sql query string and a toxminer database name and 
  ### returns a data.table containing the queried data. 
  ###
  ### Arguments: 
  ###   query: a character vector of length one containing a sql query
  ###   db:    a character vector of length one containing the name of the 
  ###          toxminer database to be queried
  ### 
  ### Value:
  ###   A data.table containing the data from the queried database
  
  #Check for valid inputs
  if (length(query) != 1 | class(query) != "character") {
    stop("The input 'query' must be a character of length one.")
  }
  if (length(db) != 1 | class(db) != "character") {
    stop("The input 'db' must be a character of length one.")
  }
  
  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer",
                     password = "pass",
                     dbname = db,
                     host = "134.67.216.45")
  temp <- dbSendQuery(dbcon, query)
  result <- fetch(res = temp, n = -1)
  
  dbDisconnect(dbcon)
  
  result <- as.data.table(result)
  result
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAcidInfo: Load assay source map from assay annotation db
#-------------------------------------------------------------------------------

loadAcidInfo <- function(acsns) {
  
  ### This function queries the invitrodb database and returns the assay 
  ### component source names mapped to the assay component ids.
  ###
  ### Arguments:
  ###   acsns: a character vector containing the acsns to query on
  ###
  ### Value:
  ###   A data.table containing the acsn mapped to acid.
  
  qformat <- 
    "
    SELECT
      source_assay_component_name AS acsn,
      assay_component_id AS acid
    FROM 
      assay_component_map
    WHERE
      source_assay_component_name IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", acsns, "\"", collapse = ","))
  
  acid_info <- tmquery(query = qstring, db = "invitrodb")
  if (ncol(acid_info) == 0) {
    warning("The given acsn(s) do not have any acids in invitrodb.")
    return(acid_info)
  }
  acid_info[ , acid := as.character(acid)]
    
  acid_info
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAsidAcid: Load acids for a given assay source from assay annotation db
#-------------------------------------------------------------------------------

loadAsidAcid <- function(asids) {
  
  ### This function queries the invitrodb database and returns the assay 
  ### component ids for the given assay source ids.
  ###
  ### Arguments:
  ###   asids: a character vector containing the asids to query on
  ###
  ### Value:
  ###   A character vector containing the acids for the given asids.
  
  qformat <- 
    "
    SELECT
      assay_component_id AS acid,
      assay_source_id AS asid
    FROM 
      assay AS a,
      assay_component AS b
    WHERE
      a.assay_id = b.assay_id
      AND
      assay_source_id IN (%s)
    "
  
  qstring <- sprintf(qformat, paste0("\"", asids, "\"", collapse = ","))
  
  asid_acid <- tmquery(query = qstring, db = "invitrodb")
  if (ncol(asid_acid) == 0) {
    stop("The given asid(s) do not have any acids in invitrodb.")
  }
  
  asid_acid[ , as.character(acid)]
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAsidAeid: Load aeids for a given assay source from invitrodb
#-------------------------------------------------------------------------------

loadAsidAeid <- function(asids) {
  
  ### This function queries the invitrodb database and returns the assay 
  ### component ids for the given assay source ids.
  ###
  ### Arguments:
  ###   asids: a character vector containing the asids to query on
  ###
  ### Value:
  ###   A character vector containing the aeids for the given asids.
  
  qformat <- 
    "
    SELECT
      assay_component_id AS acid,
      assay_source_id AS asid
    FROM 
      assay AS a,
      assay_component AS b
    WHERE
      a.assay_id = b.assay_id
      AND
      assay_source_id IN (%s)
    "
  
  qstring <- sprintf(qformat, paste0("\"", asids, "\"", collapse = ","))
  
  asid_acid <- tmquery(query = qstring, db = "invitrodb")
  if (ncol(asid_acid) == 0) {
    stop("The given asid(s) do not have any acids in invitrodb.")
  }
  
  acs <- asid_acid[ , as.character(acid)]
  
  suppressWarnings(loadAeidInfo(acs)[ , as.character(aeid)])
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadCpidInfo: Load concentration information from inventorydb
#-------------------------------------------------------------------------------

loadCpidInfo <- function(vendor_id) {
  
  ### This function queries the inventorydb database and returns spid mapped
  ### to the chemical plate ids and ship date. 
  ###
  ### Arguments:
  ###   spids: a character vector containing the spids to query on
  ###
  ### Value:
  ###   A data.table containing the all spids mapped to cpid and ship_date.
  
  qformat <- 
    "
    SELECT 
      v_vendor_id AS vendor,
      s_shipped_date AS ship_date,
      p_plate_id AS cpid,
      sa_sample_id AS spid,
      pd_well AS well
    FROM 
      vendor 
        INNER JOIN ship ON ship.s_v_id = vendor.v_id  
        INNER JOIN plate ON plate.p_ship_id = ship.s_ship_id  
        INNER JOIN plate_detail ON plate_detail.pd_plate_id = plate.p_plate_id  
        INNER JOIN sample ON sample.sa_sample_id = plate_detail.pd_sample_id  
    WHERE  
      v_vendor_id = %s;
    "
  
  qstring <- sprintf(qformat, paste0("\"", vendor_id, "\""))
  
  cpid_info <- tmquery(query = qstring, db = "inventorydb")
  if (ncol(cpid_info) == 0) {
    warning("The given vendor_id us not listed in inventorydb.")
  }
  
  cpid_info
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl0Data: Load level 0 data from invitrodb
#-------------------------------------------------------------------------------

loadLvl0Data <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 0 data 
  ### for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 0 data for the given parameters.
  
  qformat <- 
    "
    SELECT
      l0id,
      spid,
      cpid,
      acid,
      apid,
      rowi,
      coli,
      wllt,
      wllq,
      conc,
      rval,
      srcf
    FROM 
      level0
    WHERE
    "
  
  if (!is.null(field)) {
    
    field <- paste0("level0.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
  
  dat[ , acid := as.character(acid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl1Data: Load level 1 data from invitrodb
#-------------------------------------------------------------------------------

loadLvl1Data <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 1 data 
  ### for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 1 data for the given parameters.
  
  qformat <- 
    "
    SELECT
      level1.l0id,
      l1id,
      spid,
      cpid,
      level1.acid,
      apid,
      rowi,
      coli,
      wllt,
      wllq,
      conc,
      rval,
      cndx,
      repi,
      srcf
    FROM 
      level0,
      level1
    WHERE
      level0.l0id = level1.l0id
  "
  
  if (!is.null(field)) {
    
    qformat <- paste(qformat, "AND")
    
    field <- paste0("level1.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
    
  dat[ , acid := as.character(acid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl2Data: Load level 2 data from invitrodb
#-------------------------------------------------------------------------------

loadLvl2Data <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 2 data 
  ### for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 2 data for the given parameters.
  
  qformat <- 
    "
    SELECT
      level2.l0id,
      level2.l1id,
      l2id,
      spid,
      level2.acid,
      apid,
      wllt,
      conc,
      cval,
      cndx,
      repi
    FROM 
      level0,
      level1,
      level2
    WHERE
      level0.l0id = level1.l0id
      AND
      level1.l0id = level2.l0id
    "
  
  if (!is.null(field)) {
    
    qformat <- paste(qformat, "AND")
    
    field <- paste0("level2.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
  
  dat[ , acid := as.character(acid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl3Data: Load level 3 data from invitrodb
#-------------------------------------------------------------------------------

loadLvl3Data <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 3 data 
  ### for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 3 data for the given parameters.
  ###
  ### Note:
  ###   This function only loads well types of t, c, o. 
  
  qformat <- 
    "
    SELECT
      level3.l0id,
      level3.l1id,
      level3.l2id,
      l3id,
      spid,
      aeid,
      logc,
      resp,
      cndx,
      wllt,
      repi
    FROM 
      level0,
      level1,
      level3
    WHERE
      level0.l0id = level1.l0id
      AND
      level1.l0id = level3.l0id
      AND
      wllt IN (\"t\",\"c\",\"o\")
  "
  
  if (!is.null(field)) {
    
    qformat <- paste(qformat, "AND")
    
    field <- paste0("level3.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
  
  dat[ , aeid := as.character(aeid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl4Data: Load level 4 data from invitrodb
#-------------------------------------------------------------------------------

loadLvl4Data <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 4 data 
  ### for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 4 data for the given parameters.
  
  qformat <- 
    "
    SELECT
      l4id,
      aeid,
      agby,
      bmad,
      resp_max,
      resp_min,
      max_mean,
      max_mean_conc,
      max_med,
      max_med_conc,
      logc_max,
      logc_min,
      cnst,
      hill,
      hcov,
      gnls,
      gcov,
      cnst_er,
      cnst_aic,
      hill_tp,
      hill_tp_sd,
      hill_ga,
      hill_ga_sd,
      hill_gw,
      hill_gw_sd,
      hill_er,
      hill_er_sd,
      hill_aic,
      gnls_tp,
      gnls_tp_sd,
      gnls_ga,
      gnls_ga_sd,
      gnls_gw,
      gnls_gw_sd,
      gnls_la,
      gnls_la_sd,
      gnls_lw,
      gnls_lw_sd,
      gnls_er,
      gnls_er_sd,
      gnls_aic
    FROM 
      level4
    WHERE
    "
  
  if (!is.null(field)) {
    
    field <- paste0("level4.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
  
  dat[ , aeid := as.character(aeid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl4Agg: Load level 4 aggregate data from invitrodb
#-------------------------------------------------------------------------------

loadLvl4Agg <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 4
  ### aggregate data, including spid, logc, and resp for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 4 aggregate data for the given 
  ###   parameters.
  
  qformat <- 
    "
    SELECT
      agg_level4.aeid,
      agg_level4.l4id,
      agg_level4.l3id,
      level0.spid,
      logc,
      resp
    FROM 
      level0,
      level3,
      agg_level4
    WHERE
      level3.l3id = agg_level4.l3id
      AND
      level0.l0id = agg_level4.l0id
    "
  
  if (!is.null(field)) {
    
    qformat <- paste(qformat, "AND")
    
    field <- paste0("agg_level4.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
  
  dat[ , aeid := as.character(aeid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadLvl5Data: Load level 5 data from invitrodb
#-------------------------------------------------------------------------------

loadLvl5Data <- function(field = NULL, val = NULL) {
  
  ### This function queries the invitrodb database and returns level 5 data 
  ### for the given parameters.
  ###
  ### Arguments:
  ###   field: character vector containing the fields(s) to query on
  ###   val:   list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field.
  ### Value:
  ###   A data.table containing the level 5 data for the given parameters.
  
  qformat <- 
    "
    SELECT
      l5id,
      level5.l4id,
      level5.aeid,
      agby,
      bmad,
      resp_max,
      resp_min,
      max_mean,
      max_mean_conc,
      max_med,
      max_med_conc,
      logc_max,
      logc_min,
      cnst,
      hill,
      hcov,
      gnls,
      gcov,
      cnst_er,
      cnst_aic,
      hill_tp,
      hill_tp_sd,
      hill_ga,
      hill_ga_sd,
      hill_gw,
      hill_gw_sd,
      hill_er,
      hill_er_sd,
      hill_aic,
      gnls_tp,
      gnls_tp_sd,
      gnls_ga,
      gnls_ga_sd,
      gnls_gw,
      gnls_gw_sd,
      gnls_la,
      gnls_la_sd,
      gnls_lw,
      gnls_lw_sd,
      gnls_er,
      gnls_er_sd,
      gnls_aic,
      hitc,
      modl,
      fitc
    FROM 
      level4,
      level5
    WHERE
      level4.l4id = level5.l4id
  "
  
  if (!is.null(field)) {
    
    qformat <- paste(qformat, "AND")
    
    field <- paste0("level5.", field)
    qformat <- paste0(qformat, 
                      "  ", 
                      paste(field, "IN (%s)", collapse = " AND "))
    qformat <- paste0(qformat, ";")
    
    if (!is.list(val)) val <- list(val)
    val <- lapply(val, function(x) paste0("\"", x, "\"", collapse = ","))
    
    qstring <- do.call(sprintf, args = c(qformat, val))
    
  } else {
    
    qstring <- qformat
    
  }
  
  
  dat <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(dat) == 0) return(dat)
  
  dat[ , aeid := as.character(aeid)]
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadChidInfo: Load chemical information from inventorydb
#-------------------------------------------------------------------------------

loadChidInfo <- function(spids) {
  
  ### This function queries the inventorydb database and returns spid mapped
  ### to chid.
  ###
  ### Arguments:
  ###   spids: a character vector containing the spids to query on
  ###
  ### Value:
  ###   A data.table containing the all spids mapped to chid
  
  qformat <- 
    "
    SELECT 
      sa_sample_id AS spid,  
      c_gsid_id AS chid  
    FROM 
      casrn,
      sample
    WHERE
      sample.sa_gsid = casrn.c_gsid_id
      AND
      sample.sa_sample_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", spids, "\"", collapse = ","))
  
  chid_info <- tmquery(query = qstring, db = "inventorydb")
  
  if (ncol(chid_info) == 0) {
    warning("The given spid(s) are not listed in the inventorydb.")
    return(chid_info)
  }
  
  chid_info[ , chid := as.character(chid)]
  
  chid_info

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadChemName: Load chemical names from inventorydb
#-------------------------------------------------------------------------------

loadChemName <- function(chids) {
  
  ### This function queries the inventorydb database and returns chid mapped
  ### to name, and casrn. 
  ###
  ### Arguments:
  ###   chids: a character vector containing the chids to query on
  ###
  ### Value:
  ###   A data.table containing the all chids mapped to name and casrn
  
  qformat <- 
    "
    SELECT 
      c_gsid_id AS chid,
      c_casrn_id AS casn,
      IF(c_shortname IS NULL, c_chemicalname, c_shortname) AS chnm
    FROM 
      casrn
    WHERE
      c_gsid_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", chids, "\"", collapse = ","))
  
  chem_name <- tmquery(query = qstring, db = "inventorydb")
  
  if (ncol(chem_name) == 0) {
    warning("The given chid(s) are not listed in the inventorydb.")
    return(chem_name)
  }
  
  chem_name[ , chid := as.character(chid)]
  setkey(chem_name, chid)
  
  chem_name
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadChemInfo: Load chemical information from inventorydb
#-------------------------------------------------------------------------------

loadChemInfo <- function(field = NULL, vals = NULL, exact = FALSE) {
  
  ### This function queries the inventorydb database and returns chemical
  ### information for the given field and values. If field is NULL the function
  ### returns chemical information for all chemicals in inventorydb.
  ###
  ### Arguments:
  ###   field: a character vector of length 1 containing the field to query on
  ###   vals:  a vector containging the values to query on
  ###
  ### Value:
  ###   A data.table containing chemical information for the given field and 
  ###   values.
  
  qformat <- 
    "
    SELECT 
      sa_sample_id AS spid,  
      c_gsid_id AS chid,  
      c_casrn_id AS casn,
      IF(c_shortname IS NULL, c_chemicalname, c_shortname) AS chnm
    FROM 
      casrn,
      sample
    WHERE
      sample.sa_gsid = casrn.c_gsid_id
    "
  
  if (!is.null(field)) {
    
    field <- switch(field,
                    spid = "sa_sample_id",
                    chid = "c_gsid_id",
                    casn = "c_casrn_id",
                    "chnm")  
    
    qformat <- paste(qformat, "AND")
    
    if (field == "chnm") {
      if (exact) {
        qformat <- paste(qformat, 
                         "(c_shortname IN (%s) OR c_chemicalname IN (%s));")
        vals <- paste0("\"", vals, "\"", collapse = ",")
        qstring <- sprintf(qformat, vals, vals)
      } else {
        qformat <- paste(qformat, 
                         "(c_shortname RLIKE %s OR c_chemicalname RLIKE %s);")
        vals <- paste0("\"", paste(vals, collapse = "|"), "\"")
        qstring <- sprintf(qformat, vals, vals)
      }
    } else {
      qformat <- paste(qformat, field, "IN (%s)")
      qstring <- sprintf(qformat, paste0("\"", vals, "\"", collapse = ","))
    }
    
  } else {
    
    qstring <- qformat
    
  }
    
  smpl_info <- tmquery(qstring, "inventorydb")
  
  if (ncol(smpl_info) == 0) {
    warning("The given values are not listed in the inventorydb.")
    return(smpl_info)
  }
  
  smpl_info[ , chid := as.character(chid)]
  unique(smpl_info)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadApidSize: Load apid size map from invitrodb
#-------------------------------------------------------------------------------

loadApidSize <- function(acids) {
  
  ### This function queries the invitrodb database and returns the apid mapped 
  ### to microtitter plate size used in the assay.
  ###
  ### Arguments:
  ###   acids: a character vector containing the acids to query on
  ###
  ### Value:
  ###   A data.table containing the all apids mapped to plate size.
  
  qformat <- 
    "
    SELECT
      assay_component_id AS acid,
      assay_footprint AS nwll
    FROM 
      assay,
      assay_component
    WHERE
      assay.assay_id = assay_component.assay_id
      AND
      assay_component.assay_component_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", acids, "\"", collapse = ","))
  
  apid_size <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(apid_size) == 0) {
    warning("The given acid(s) are not listed in the invitrodb.")
    return(apid_size)
  }
  
  apid_size[ , acid := as.character(acid)]
  apid_size[ , nwll := as.numeric(gsub("[^0-9]", "", nwll))]
  
  plate_dim <- data.table(nwll = c(6, 12, 24, 48, 96, 384, 1536),
                          coln = c(3,  4,  6,  8, 12,  24,   48),
                          rown = c(2,  3,  4,  6,  8,  16,   32))
  
  setkey(plate_dim, nwll)
  setkey(apid_size, nwll)
  apid_size <- plate_dim[apid_size]
  setcolorder(apid_size, c("acid", "nwll", "rown", "coln"))
  
  apid_size
  
}

#-------------------------------------------------------------------------------


#-------------------------------------------------------------------------------
# loadL2Mthds: Load assay-specific level 2 correction methods
#-------------------------------------------------------------------------------

loadL2Mthds <- function(acids) {

  ### This function queries the invitrodb database and returns the acids mapped 
  ### to the level 2 correction methods.
  ###
  ### Arguments:
  ###   acids: vector containing the acids to query
  ###
  ### Value:
  ###   A data.table containing the acids mapped to the correction methods and 
  ###   orders.
  
  qformat <- 
    "
    SELECT 
      a.ac_id AS acid,
      b.l2_method AS mthd,
      a.execute_order AS ordr
    FROM 
      l2_ac AS a,
      l2_methods AS b
    WHERE
      a.l2_method_id = b.l2_method_id
      AND
      a.ac_id IN (%s)
    ORDER BY 
      a.ac_id,
      a.execute_order;
    "
  
  qstring <- sprintf(qformat, paste(acids, collapse = ","))
  
  l2_mthds <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(l2_mthds) == 0) {
    warning("The given acid(s) do not have methods in the invitrodb ",
            "l2_ac table.")
    return(l2_mthds)
  }
  
  l2_mthds[ , acid := as.character(acid)]
  
  l2_mthds

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAcidName: Load assay component names from invitrodb
#-------------------------------------------------------------------------------

loadAcidName <- function(acids) {
  
  ### This function queries the invitrodb database and returns the acid mapped 
  ### to assay component name.
  ###
  ### Value:
  ###   A data.table containing the all acids mapped to assay component name.
  
  qformat <- 
    "
    SELECT
      assay_component_id AS acid,
      assay_component_name AS acnm
    FROM 
      assay_component
    WHERE
      assay_component_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", acids, "\"", collapse = ","))
  
  acid_name <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(acid_name) == 0) {
    warning("The given acid(s) are not listed in the invitrodb ",
            "assay_component table.")
    return(acid_name)
  }
  
  acid_name[ , acid := as.character(acid)]
  setkey(acid_name, acid)
    
  acid_name
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAeidInfo: Load assay endpoint map from invitrodb
#-------------------------------------------------------------------------------

loadAeidInfo <- function(acids) {
  
  ### This function queries the invitrodb database and returns the acid mapped 
  ### to aeid.
  ###
  ### Value:
  ###   A data.table containing the all acids mapped to aeid.
  
  qformat <- 
    "
    SELECT
      assay_component_id AS acid,
      assay_component_endpoint_id AS aeid
    FROM 
      assay_component_endpoint
    WHERE
      assay_component_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste(acids, collapse = ","))
  
  aeid_info <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(aeid_info) == 0) {
    warning("The given acid(s) are not listed in the invitrodb ",
            "assay_component_endpoint table.")
    return(aeid_info)
  }
  
  aeid_info[ , acid := as.character(acid)]
  aeid_info[ , aeid := as.character(aeid)]
  aeid_info
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadL3Mthds: Load assay-specific level 3 normalization methods
#-------------------------------------------------------------------------------

loadL3Mthds <- function(aeids) {

  ### This function the queries invitrodb database and returns the aeid mapped 
  ### to the level 3 normalization methods.
  ###
  ### Arguments:
  ###   aeids: a character vector containing the aeids to query
  ###
  ### Value:
  ###   A data.table containing the all aeids mapped to the normalizatoin
  ###   methods and orders.
  
  qformat <- 
    "
    SELECT 
      a.ace_id AS aeid,
      b.l3_method AS mthd,
      a.execute_order AS ordr,
      b.l3_method_id AS mthd_id
    FROM 
      l3_ace AS a,
      l3_methods AS b
    WHERE
      a.l3_method_id = b.l3_method_id
      AND
      a.ace_id IN (%s)
    ORDER BY 
      a.ace_id,
      a.execute_order;
    "
  qstring <- sprintf(qformat, paste(aeids, collapse = ","))
  
  l3_mthds <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(l3_mthds) == 0) {
    warning("The given aeid(s) do not have methods in the invitrodb ",
            "l3_ace table.")
    return(l3_mthds)
  }
  
  l3_mthds[ , aeid := as.character(aeid)]
  
  l3_mthds

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAeidName: Load assay endpoint names from invitrodb
#-------------------------------------------------------------------------------

loadAeidName <- function(aeids) {
  
  ### This function queries the invitrodb database and returns the aeid mapped 
  ### to assay endpoint name.
  ###
  ### Value:
  ###   A data.table containing the all aeids mapped to assay endpoint name.
  
  qformat <- 
    "
    SELECT
      assay_component_endpoint_id AS aeid,
      assay_component_endpoint_name AS aenm
    FROM 
      assay_component_endpoint
    WHERE
      assay_component_endpoint_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", aeids, "\"", collapse = ","))
  
  aeid_name <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(aeid_name) == 0) {
    warning("The given aeid(s) are not listed in the invitrodb ",
            "assay_component_endpoint table.")
    return(aeid_name)
  }
  
  aeid_name[ , aeid := as.character(aeid)]
  setkey(aeid_name, aeid)
    
  aeid_name
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadSgnlInfo: Load signal direction information from invitrodb
#-------------------------------------------------------------------------------

loadSgnlInfo <- function(acids) {
  
  ### This function queries the invitrodb database and returns aeid mapped to 
  ### the positive control raw signal direction.
  ###
  ### Arguments:
  ###   aeids: a character vector containing the aeids to query on
  ###
  ### Value:
  ###   A data.table containing the all acsns mapped to acid.
  
  qformat <- 
    "
    SELECT
      b.assay_component_id as acid,
      b.assay_component_endpoint_id AS aeid,
      b.assay_component_endpoint_name AS aenm,
      a.signal_direction_type,
      b.signal_direction,
      b.analysis_direction,
      b.key_positive_control
    FROM 
      assay_component AS a,
      assay_component_endpoint AS b
    WHERE
      a.assay_component_id = b.assay_component_id
      AND
      b.assay_component_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", acids, "\"", collapse = ","))
  
  sgnl_info <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(sgnl_info) == 0) {
    warning("The given acid(s) are not listed in invitrodb.")
    return(sgnl_info)
  }
  
  sgnl_info[ , acid := as.character(acid)]
  sgnl_info[ , aeid := as.character(aeid)]
  setkey(sgnl_info, acid)
  
  sgnl_info
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadCorrFuncs: Load list of correction methods (to be used at level 2)
#-------------------------------------------------------------------------------
 
loadCorrFuncs <- function() {
  
  ### This function returns a list of functions that take in a character vectors 
  ### containing acids and generate expressions with data.table calls to 
  ### execute a correction method on the data for the given acids. 
  
  list(
    
    ### List of functions to return correction expressions. 
    ###
    ### Arguments:
    ###   acids: a character vector containing the acids for processing
    ###
    ### Value:
    ###   An expression object with the data.table call to execute the method
    ###   on the given acids.
  
    log2 = function(acids) {
      
      ## This method takes the log base 2 of the data
     
      e1 <- bquote(dat[.(acids), cval := log2(cval)])
      list(e1)
      
    },
    
    log10 = function(acids) {
      
      ## This method takes the log base 10 of the data
          
      e1 <- bquote(dat[.(acids), cval := log10(cval)])
      list(e1)
    
    },
    
    rmneg = function(acids) {
      
      ## This method makes any values less than 0, NA
      
      e1 <- bquote(dat[acid %in% .(acids) & cval < 0, 
                       c('cval', 'wllq') := list(NA_real_, 0)])
      list(e1)
      
    },
    
    rmzero = function(acids) {
      
      ## This method makes any values equal to 0, NA
      
      e1 <- bquote(dat[acid %in% .(acids) & cval == 0, 
                       c('cval', 'wllq') := list(NA_real_, 0)])
      list(e1)
      
    },
    
    mult25 = function(acids) {
      
      ## This method multiplies all the values by 25
      
      e1 <- bquote(dat[acid %in% .(acids), cval := cval * 25])
      list(e1)
      
    },
    
    mult100 = function(acids) {
      
      ## This method multiplies all the values by 100
      
      e1 <- bquote(dat[acid %in% .(acids), cval := cval * 100])
      list(e1)
      
    }
    
  )
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# blineShift: Shift baseline to 0
#-------------------------------------------------------------------------------

blineShift <- function(resp, logc, wndw) {
  
  ### This function does base-line correction.
  ###
  ### Arguments:
  ###   resp: the response values
  ###   logc: the log concentration
  ###   wndw: the window in which to shift baseline
  ###
  ### Value:
  ###   a numeric vector containing the shifted resp values
  
  wndw <- unique(wndw)[1]
  
  ordr <- order(logc)
  resp <- resp[ordr]
  logc <- logc[ordr]
    
  uconc <- unique(logc)
  nconc <- length(uconc)
  
  if (nconc < 4) return(resp)
    
  low <- 1:max(ceiling(nconc/4), 2)
  rsub <- resp[which(logc %in% uconc[low])]
  csub <- logc[which(logc %in% uconc[low])]
  low_med <- median(rsub)
  m <- lm(rsub ~ csub)$coefficients["csub"]
  if (is.na(m)) m <- 0
  test1 <- abs(low_med) <= wndw
  test2 <- abs(m) <= wndw/low[length(low)]
  if (test1 & test2) resp <- resp - low_med
  
  resp[order(ordr)]
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadNormFuncs: Load list of normalization methods (to be used at level 3)
#-------------------------------------------------------------------------------
 
loadNormFuncs <- function() {
  
  ### This function returns a list of functions that take in a character vectors 
  ### containing aeids and generate expressions with data.table calls to 
  ### execute a normalization method on the data for the given aeids. 
  
  list(
    
    ### List of functions to return normalization expressions.
    ###
    ### Arguments:
    ###   aeids: a character vector containing the acids for processing
    ###
    ### Value:
    ###   An expression object with the data.table call to execute the method
    ###   on the given aeids.
    
    bval.apid.nwlls.med = function(aeids) {
      
      ## Take the median of all the well type "n" values, by apid
      
      e1 <- bquote(dat[.(aeids), 
                       bval := median(cval[wllt == "n"], na.rm = TRUE), 
                       by = list(aeid, apid)])
      list(e1)
      
    },    
    
    bval.apid.1owconc.med = function(aeids) {
      
      ## Take the median cval of the first two concentrations, by apid
      
      e1 <- bquote(dat[.(aeids), 
                       bval := median(cval[cndx %in% 1:2 & wllt == "t"],
                                      na.rm = TRUE),
                       by = list(aeid, apid)])
      list(e1)
      
    },
    
    bval.apid.twlls.med = function(aeids) {
      
      ## Take the median cval of the t wells, by apid
      
      e1 <- bquote(dat[.(aeids), 
                       bval := median(cval[wllt == "t"], na.rm = TRUE),
                       by = list(aeid, apid)])
      list(e1)
      
    },
    
    bval.apid.nwllslowconc.med = function(aeids) {
      
      ## Take the median cval of the n wells and the first two concentrations, 
      ## by apid
      
      e1 <- bquote(dat[.(aeids), 
                       bval := median(cval[(cndx %in% 1:2 & wllt == "t") | 
                                             wllt == "n"],
                                      na.rm = TRUE),
                       by = list(aeid, apid)])
      list(e1)
      
    },
    
    bval.spid.lowconc.med = function(aeids) {
      
      ## Take the median cval of the first three concentrations, by spid
      
      e1 <- bquote(dat[.(aeids), 
                       bval := median(cval[cndx %in% 1:3 & wllt == "t"],
                                      na.rm = TRUE),
                       by = list(aeid, cpid, spid)])
      list(e1)
      
    },
    
    pval.apid.pwlls.med = function(aeids) {
      
      ## Take the median of all the well type "p" values, by apid
      
      e1 <- bquote(dat[.(aeids), 
                       pval := median(cval[wllt == "p"], na.rm = TRUE), 
                       by = list(aeid, apid)])
      list(e1)
      
    },
    
    pval.apid.mwlls.med = function(aeids) {
      
      ## Take the median of all the well type "m" values, by apid
      
      e1 <- bquote(dat[.(aeids), 
                       pval := median(cval[wllt == "m"], na.rm = TRUE), 
                       by = list(aeid, apid)])
      list(e1)
      
    },
    
    pval.apid.medpcbyconc.max = function(aeids) {
      
      ## Calculate the median positive control values by concentration, then
      ## take the maximum by apid. 
      
      e1 <- bquote(dat[.(aeids),
                       temp := median(cval[wllt %in% c("c", "p")], 
                                      na.rm = TRUE),
                       by = list(aeid, apid, wllt, conc)])
      e2 <- bquote(dat[.(aeids),
                       pval := max(temp, na.rm = TRUE),
                       by = list(aeid, apid)])
      e3 <- bquote(dat[ , temp := NULL])
      list(e1, e2, e3)
      
    },
    
    pval.apid.medpcbyconc.min = function(aeids) {
      
      ## Calculate the median positive control values by concentration, then
      ## take the minimum by apid. 
      
      e1 <- bquote(dat[.(aeids),
                       temp := median(cval[wllt %in% c("c", "p")], 
                                      na.rm = TRUE),
                       by = list(aeid, apid, wllt, conc)])
      e2 <- bquote(dat[.(aeids),
                       pval := min(temp, na.rm = TRUE),
                       by = list(aeid, apid)])
      e3 <- bquote(dat[ , temp := NULL])
      list(e1, e2, e3)
      
    },
    
    pval.apid.medncbyconc.min = function(aeids) {
      
      ## Calculate the median negative control values by concentration, then
      ## take the minimum by apid. 
      
      e1 <- bquote(dat[.(aeids),
                       temp := median(cval[wllt %in% c("m","o")], 
                                      na.rm = TRUE),
                       by = list(aeid, apid, wllt, conc)])
      e2 <- bquote(dat[.(aeids),
                       pval := min(temp, na.rm = TRUE),
                       by = list(aeid, apid)])
      e3 <- bquote(dat[ , temp := NULL])
      list(e1, e2, e3)
      
    },
    
    resp.pc = function(aeids) {
      
      ## Calculate the response as the percetage of postive control resp
      
      e1 <- bquote(dat[.(aeids),
                       resp := (cval - bval)/(pval - bval)*100])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    resp.fc = function(aeids) {
      
      ## Calculate the response as a fold change over baseline
      
      e1 <- bquote(dat[.(aeids), resp := cval/bval])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    resp.log2 = function(aeids) {
      
      ## Take log2 of response
      
      e1 <- bquote(dat[.(aeids), resp := log2(resp)])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    resp.mult25 = function(aeids) {
      
      ## Multiply the response by 25
      
      e1 <- bquote(dat[.(aeids), resp := resp * 25])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    resp.multneg1 = function(aeids) {
      
      ## Multiply the response by -1
      
      e1 <- bquote(dat[.(aeids), resp := resp * -1])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    resp.shiftneg = function(aeids) {
      
      ## Shift response values falling below the baseline to 0.
      
      e1 <- bquote(dat[.(aeids), 
                       bmad := mad(resp[cndx %in% 1:2 & wllt == "t"], 
                                   na.rm = TRUE),
                       by = aeid])
      e2 <- bquote(dat[aeid %in% .(aeids) & resp < -3 * bmad, resp := 0])
      e3 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      e4 <- bquote(dat[.(aeids), bmad := NULL])
      list(e1, e2, e3, e4)
      
    },
    
    resp.blineshift.3bmad.repi = function(aeids) {
      
      ## Do baseline correction by repi, with a window of 3*bmad.
      
      e1 <- bquote(dat[.(aeids), 
                       wndw := mad(resp[cndx %in% 1:2 & wllt == "t"], 
                                   na.rm = TRUE) * 3,
                       by = aeid])
      e2 <- bquote(dat[.(aeids), 
                       resp := blineShift(resp, logc, wndw), 
                       by = list(aeid, spid, repi)])
      e3 <- bquote(dat[.(aeids), wndw := NULL])
      e4 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2, e3, e4)
      
    },
    
    resp.blineshift.50.repi = function(aeids) {
      
      ## Do baseline correction by repi, with a window of 50%.
      
      e1 <- bquote(dat[.(aeids), 
                       resp := blineShift(resp, logc, wndw = 50), 
                       by = list(aeid, spid, repi)])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    resp.blineshift.3bmad.spid = function(aeids) {
      
      ## Do baseline correction by spid, with a window of 3*bmad.
      
      e1 <- bquote(dat[.(aeids), 
                       wndw := mad(resp[cndx %in% 1:2 & wllt == "t"], 
                                   na.rm = TRUE) * 3,
                       by = aeid])
      e2 <- bquote(dat[.(aeids), 
                       resp := blineShift(resp, logc, wndw), 
                       by = list(aeid, spid)])
      e3 <- bquote(dat[.(aeids), wndw := NULL])
      e4 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2, e3, e4)
      
    },
    
    resp.blineshift.50.spid = function(aeids) {
      
      ## Do baseline correction by spid, with a window of 50%.
      
      e1 <- bquote(dat[.(aeids), 
                       resp := blineShift(resp, logc, wndw = 50), 
                       by = list(aeid, spid)])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    },
    
    none = function(aeids) {
      
      ## Add default resp column
      
      e1 <- bquote(dat[.(aeids), resp := cval])
      e2 <- bquote(dat[.(aeids), resp := round(resp, 3)])
      list(e1, e2)
      
    }
    
  )
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# objFunHill: Generate a hill model objective function to optimize
#-------------------------------------------------------------------------------

objFunHill <- function(p, lconc, resp) {
  
  ### This function takes creates an objective function to be optimized using
  ### the starting hill parameters, log concentration, and response. 
  ###
  ### Arguments: 
  ###   p:     a numeric vector of length 4 containg the starting values for 
  ###          the hill model, in order: top, log AC50, hill 
  ###          coefficient, and log error term
  ###   lconc: a numeric vector containing the log concentration values to  
  ###          produce the objective function
  ###   lresp: a numeric vector containing the response values to produce the 
  ###          objective function
  ### 
  ### Value:
  ###   An objective function for the hill model and the given conc-resp data
  
  mu <- p[1]/(1 + 10^((p[2] - lconc)*p[3]))
  sum(dt((resp - mu)/10^p[4], df = 1, log = TRUE) - p[4])
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# objFunGnLs: Generate a gain-loss model objective function to optimize
#-------------------------------------------------------------------------------

objFunGnLs <- function(p, lconc, resp) {
  
  ### This function takes creates an objective function to be optimized using
  ### the starting gain-loss parameters, log concentration, and response. 
  ###
  ### Arguments: 
  ###   p:     a numeric vector of length 5 containg the starting values for 
  ###          the gain-loss model, in order: top, gain log AC50, gain hill 
  ###          coefficient, loss log AC50, loss hill coefficient and log error 
  ###          term
  ###   lconc: a numeric vector containing the log concentration values to  
  ###          produce the objective function
  ###   lresp: a numeric vector containing the response values to produce the 
  ###          objective function
  ### 
  ### Value:
  ###   An objective function for the gain-loss model and the given conc-resp 
  ###   data
    
  gn <- 1/(1 + 10^((p[2] - lconc)*p[3]))
  ls <- 1/(1 + 10^((lconc - p[4])*p[5]))
  mu <- p[1]*gn*ls
  sum(dt((resp - mu)/10^p[6], df = 1, log = TRUE) - p[6])
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# objFunCnst: Generate a constant model objective function to optimize
#-------------------------------------------------------------------------------

objFunCnst <- function(p, resp) {
  
  ### This function takes creates an objective function to be optimized using
  ### the starting constant model parameter, and response. 
  ###
  ### Arguments: 
  ###   p:     a numeric vector of length 1 containg the starting values for 
  ###          the constant model, in order: log error term
  ###   lresp: a numeric vector containing the response values to produce the 
  ###          objective function
  ### 
  ### Value:
  ###   An objective function for the constant model and the given resp data
  
  mu <- 0
  sum(dt((resp - mu)/10^p[1], df = 1, log = TRUE) - p[1])

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# doFit: Fit data to a Hill model and a flat line1
#-------------------------------------------------------------------------------

doFit <- function(logc, resp, bmad, ...) {
  
  ### This function takes fits the given data to a constant model, and when 
  ### evidence for activity also a Hill model and gain-loss model. 
  ###
  ### Arguments: 
  ###   logc: a numeric vector containing the log concentration values to  
  ###         be fit 
  ###   resp: a numeric vector containing the response values to be fit
  ###   bmad: the baseline median absolute deviation for the assay, defined
  ###         as the median absolute deviation of the response values for the 
  ###         first two concentrations of every test chemical tested across the
  ###         assay
  ### 
  ### Value:
  ###   A tcpl.fit object containing the model information for the given data.
  ###   cnst - list containing the constant model output
  ###   hill - list containing the hill model output
  ###   gnls - list containing the gain-loss model output
  ###   msgs - character with concatenated messages from the modeling
  ###   logc - the input log concentration data  
  ###   resp - the input response data
  ###   bmad - the input baseline median absolute deviation
  ###   rmns - the response means, by concentration
  ###   rmds - the response medians, by concentration
  ###   mmed - the maximum median response
  ###   cmmd - the concentration of the maximum median response
  ###   mmen - the maximum mean reponse
  ###   cmmn - the concentration of the maximum mean response
  ###
  ### Details:
  ###   
  
  fenv <- environment()
  
  bmad <- min(bmad)
  rmns <- tapply(resp, logc, mean)
  rmds <- tapply(resp, logc, median)
  mmed <- max(rmds)
  mmed_conc <- as.numeric(names(which.max(rmds)))
    
  hprs <- paste0("hill_", c("tp", "ga", "gw", "er"))
  hsds <- paste0("hill_", c("tp", "ga", "gw", "er"), "_sd")
  gprs <- paste0("gnls_", c("tp", "ga", "gw", "la", "lw", "er"))
  gsds <- paste0("gnls_", c("tp", "ga", "gw", "la", "lw", "er"), "_sd")
  
  resp_max <- max(resp)
  resp_min <- min(resp)
  logc_med <- median(logc)
  logc_min <- min(logc)
  logc_max <- max(logc)
      
  ## Do not fit anything with less than four concentrations of data.
  if (length(rmds) >= 4) {
    
    er_est <- max(log10(mad(resp)), 0)
    
    ###----------------------- Fit the Constant Model -----------------------###
    cfit <- optim(er_est, 
                  objFunCnst,
                  method = "Brent",
                  lower = er_est - 2,
                  upper = er_est + 2,
                  control = list(fnscale = -1,
                                 reltol = 1e-4,
                                 maxit = 500),
                  resp = resp)
    
    if (!is(cfit, "try-error")) {
      
      cnst <- 1L
      cnst_er <- cfit$par
      caic <- 2 - 2*cfit$value # 2*length(cfit$par) - 2*cfit$value
      
    } else {
      
      cnst <- 0L
      cnst_er <- NA_real_
      caic <- NA_integer_
      
    }
    
    if (lw(resp >= 3*bmad) > 0) {
      
      ###------------------------ Fit the Hill Model ------------------------###
      ## Starting parameters for the Hill Model
      # cind <- (ceiling(length(meds)/2) + 1):length(meds)
      h <- c(mmed, # top
             mmed_conc - 1, # logAC50
             1.2, # hill coefficient
             er_est) # logSigma
      
      if (h[1] == 0) h[1] <- 0.1
      
      ## Generate the bound matrices to constrain the model.  
      #                tp   ac   w    er
      hUi <- matrix(c( 1,   0,   0,   0,
                      -1,   0,   0,   0,
                       0,   1,   0,   0,
                       0,  -1,   0,   0,
                       0,   0,   1,   0,
                       0,   0,  -1,   0),
                    byrow = TRUE, nrow = 6, ncol = 4)
      
      hbnds <- c(0, -1.2*resp_max, # tp bounds
                 logc_min - 2, -(logc_max + 2), # ac bounds 
                 0.3, -8) # w bounds
      
      hCi <- matrix(hbnds, nrow = 6, ncol = 1)
      
      ## Optimize the hill model
      hfit <- try(constrOptim(h, 
                              objFunHill,
                              ui = hUi,
                              ci = hCi,
                              mu = 1e-6,
                              method = "Nelder-Mead",
                              control = list(fnscale = -1,
                                             reltol = 1e-10,
                                             maxit = 6000),
                              lconc = logc,
                              resp = resp),
                  silent = TRUE)
      
      ## Generate some summary statistics
      if (!is(hfit, "try-error")) { # Hill model fit the data
        
        hill <- 1L
        haic <- 8 - 2*hfit$value # 2*length(hfit$par) - 2*hfit$value
        mapply(assign,
               c(hprs),
               hfit$par,
               MoreArgs = list(envir = fenv))       
        
        hfit$cov <- try(solve(-hessian(objFunHill,
                                       hfit$par,
                                       lconc = logc,
                                       resp = resp)),
                        silent = TRUE)
        
        if (!is(hfit$cov, "try-error")) { # Could invert hill Hessian
          
          hcov <- 1L
          suppressWarnings(mapply(assign,
                                  hsds,
                                  sqrt(diag(hfit$cov)),
                                  MoreArgs = list(envir = fenv)))
          
                    
        } else { # Could not invert hill Hessian
          
          hcov <- 0L
          mapply(assign,
                 c(hsds),
                 NA_real_,
                 MoreArgs = list(envir = fenv))
            
        } 
        
      } else { # Hill model did not fit the data
        
        hill <- 0L
        haic <- NA_real_
        hcov <- NA_integer_
        
        mapply(assign,
               c(hprs, hsds),
               NA_real_,
               MoreArgs = list(envir = fenv))
        
      }
      
      ###--------------------- Fit the Gain-Loss Model ----------------------###
      ## Starting parameters for the Gain-Loss Model
      # cind <- (ceiling(length(meds)/2) + 1):length(meds)
      g <- c(mmed, # top
             mmed_conc - 1, # gain logAC50
             1.2, # gain hill coefficient,
             mmed_conc + 0.1, # loss logAC50,
             5, # loss hill coefficient,
             er_est) # logSigma
      
      if (g[1] == 0) g[1] <- 0.1
      
      ## Generate the bound matrices to constrain the model.  
      #                tp   ga   gw   la   lw   er
      gUi <- matrix(c( 1,   0,   0,   0,   0,   0,
                      -1,   0,   0,   0,   0,   0,
                       0,   1,   0,   0,   0,   0,
                       0,  -1,   0,   0,   0,   0,
                       0,   0,   1,   0,   0,   0,
                       0,   0,  -1 ,  0,   0,   0,
                       0,   0,   0,   1,   0,   0,
                       0,   0,   0,  -1,   0,   0,
                       0,   0,   0,   0,   1,   0,
                       0,   0,   0,   0,  -1,   0,
                       0,  -1,   0,   1,   0,   0),
                    byrow = TRUE, nrow = 11, ncol = 6)
      
      gbnds <- c(0, -1.2*resp_max, # tp bounds
                 logc_min - 2, -(logc_max), # ga bounds
                 0.3, -8, # gw bounds
                 logc_min - 2, -(logc_max + 2), # la bounds
                 0.3, -18, # lw bounds
                 0.25) # ga < la
      
      # if (mmed_conc > logc_min) g[7] <- mmed_conc - 0.25
      
      gCi <- matrix(gbnds, nrow = 11, ncol = 1)
      
      ## Optimize the hill model
      gfit <- try(constrOptim(g, 
                              objFunGnLs,
                              ui = gUi,
                              ci = gCi,
                              mu = 1e-6,
                              method = "Nelder-Mead",
                              control = list(fnscale = -1,
                                             reltol = 1e-10,
                                             maxit = 6000),
                              lconc = logc,
                              resp = resp),
                  silent = TRUE)
      
      ## Generate some summary statistics
      if (!is(gfit, "try-error")) { # Gain-loss fit the data
        
        gnls <- 1L
        gaic <- 12 - 2*gfit$value # 2*length(gfit$par) - 2*gfit$value
        mapply(assign,
               c(gprs),
               gfit$par,
               MoreArgs = list(envir = fenv))
        
        gfit$cov <- try(solve(-hessian(objFunGnLs,
                                       gfit$par,
                                       lconc = logc,
                                       resp = resp)),
                        silent = TRUE)
        
        if (!is(gfit$cov, "try-error")) { # Could invert gnls Hessian
          
          gcov <- 1L
          suppressWarnings(mapply(assign,
                                  c(gsds),
                                  sqrt(diag(gfit$cov)),
                                  MoreArgs = list(envir = fenv)))
          
        } else { # Could not invert gnls Hessian
          
          gcov <- 0L
          mapply(assign,
                 c(gsds),
                 NA_real_,
                 MoreArgs = list(envir = fenv))
          
        } 
        
      } else { # Gain-loss did not fit the data
        
        gnls <- 0L
        gaic <- NA_real_
        gcov <- NA_integer_
        
        mapply(assign,
               c(gprs, gsds),
               NA_real_,
               MoreArgs = list(envir = fenv))
        
      }
      
    } else { # None of the response values fell outside 3*bmad
      
      hill <- NA_integer_
      haic <- NA_real_
      hcov <- NA_integer_
      gnls <- NA_integer_
      gaic <- NA_real_
      gcov <- NA_integer_
      
      mapply(assign,
             c(hprs, hsds, gprs, gsds),
             NA_real_,
             MoreArgs = list(envir = fenv))
      
    } 
    
  } else { # Data has response data for less than four concentrations. 
    
    cnst <- NA_integer_
    cnst_er <- NA_real_
    caic <- NA_real_
    
    hill <- NA_integer_
    haic <- NA_real_
    hcov <- NA_integer_
    gnls <- NA_integer_
    gaic <- NA_real_
    gcov <- NA_integer_
    
    mapply(assign,
           c(hprs, hsds, gprs, gsds),
           NA_real_,
           MoreArgs = list(envir = fenv))
        
  }
  
  out <- list(resp_max      = resp_max,
              resp_min      = resp_min,
              max_mean      = max(rmns),
              max_mean_conc = as.numeric(names(which.max(rmns))),
              max_med       = mmed,
              max_med_conc  = mmed_conc,
              logc_max      = logc_max,
              logc_min      = logc_min,
              cnst          = cnst,
              hill          = hill,
              hcov          = hcov,
              gnls          = gnls,
              gcov          = gcov,
              cnst_er       = cnst_er,
              cnst_aic      = caic,
              hill_tp       = hill_tp,
              hill_tp_sd    = hill_tp_sd,
              hill_ga       = hill_ga,
              hill_ga_sd    = hill_ga_sd,
              hill_gw       = hill_gw,
              hill_gw_sd    = hill_gw_sd,
              hill_er       = hill_er,
              hill_er_sd    = hill_er_sd,
              hill_aic      = haic,
              gnls_tp       = gnls_tp,
              gnls_tp_sd    = gnls_tp_sd,
              gnls_ga       = gnls_ga,
              gnls_ga_sd    = gnls_ga_sd,
              gnls_gw       = gnls_gw,
              gnls_gw_sd    = gnls_gw_sd,
              gnls_la       = gnls_la,
              gnls_la_sd    = gnls_la_sd,
              gnls_lw       = gnls_lw,
              gnls_lw_sd    = gnls_lw_sd,
              gnls_er       = gnls_er,
              gnls_er_sd    = gnls_er_sd,
              gnls_aic      = gaic,
              ...)
  
  out
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# inspectFit: Inspect the output from a fit
#-------------------------------------------------------------------------------

inspectFit <- function(l4id, do.plot = TRUE) {
  
  ### This function gives a detailed output from a fit
  ###
  ### Arguments:
  ###   l4id: l4id of the fit to inspect
  ###
  ### Value:
  ###   List containing data.tables with information about the fit
  
  dat <- loadLvl4Data("l4id", l4id)
  agg <- loadLvl4Agg("l4id", l4id)[order(logc)]
  
  if (do.plot) plotTcplFits(dat = dat, agg = agg)
  
  hill <- with(dat, hill_tp/(1 + 10^((hill_ga - agg$logc) * hill_gw)))
  hobj <- dt((agg$resp - hill)/10^dat$hill_er, df = 1, log = T) - dat$hill_er
  
  cobj <- with(dat, dt(agg$resp/10^cnst_er, df = 1, log = TRUE) - cnst_er)
  
  gn <- with(dat, 1/(1 + 10^((gnls_ga - agg$logc) * gnls_gw)))
  ls <- with(dat, 1/(1 + 10^((agg$logc - gnls_la) * gnls_lw)))
  gnls <- with(dat, gnls_tp * gn * ls)
  gobj <- dt((agg$resp - gnls)/10^dat$gnls_er, df = 1, log = T) - dat$gnls_er
  
  sdat <- agg[ , list(logc, resp, cobj = cobj, hill = hill, hobj = hobj, 
                      gnls = gnls, gobj = gobj)]
  
  list(fit = list(info      = dat[ , list(l4id, aeid, agby, bmad)],
                  raw_dat   = dat[ , .SD, .SDcols = grep("max|min", 
                                                         names(dat))],
                  cnst      = dat[ , list(cnst, cnst_er)],
                  hill_pars = dat[ , list(hill_tp, hill_ga, hill_gw, hill_er)],
                  hill_sds  = dat[ , list(hill_tp_sd, hill_ga_sd, hill_gw_sd, 
                                          hill_er_sd)],
                  gnls_pars = dat[ , list(gnls_tp, gnls_ga, gnls_gw, gnls_la,
                                          gnls_lw, gnls_er)],
                  gnls_sds  = dat[ , list(gnls_tp_sd, gnls_ga_sd, gnls_gw_sd, 
                                          gnls_la_sd, gnls_lw_sd, gnls_er_sd)],
                  aics      = dat[ , list(cnst_aic, hill_aic, gnls_aic)]), 
       output = sdat)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# ACXX: Calculate the ACXX for the given activity level
#-------------------------------------------------------------------------------

ACXX <- function(XX, t, a, w, b = 0) {
  
  ### This funciton computes the log concentration for a specific modeled
  ### activity, given by XX. For example, an XX value of 25 would return the 
  ### logAC25.
  ###
  ### Arguments:
  ###   XX: the activity level
  ###   t:  the top value from the Hill model
  ###   a:  the logAC50 value from the Hill model
  ###   w:  the Hill coefficient from the Hill model
  ###   b:  the bottom value from the Hill model
  ###
  ### Value:
  ###   A double of length 1, the log concentration for the specified activity
  
  y <- t * XX/100
  
  a - log10((t - b)/(y - b) - 1)/w

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadL5Mthds: Load assay-specific level 5 activity cutoffs
#-------------------------------------------------------------------------------

loadL5Mthds <- function(aeids) {
  
  ### This function the queries invitrodb database and returns the aeid mapped 
  ### to the level 5 activity cutoff values.
  ###
  ### Arguments:
  ###   aeids: a character vector containing the aeids to query
  ###
  ### Value:
  ###   A data.table containing the all aeids mapped to the activity cutoffs
  
  qformat <- 
    "
    SELECT 
      aeid,
      coff
    FROM 
      l5_cutoffs
    WHERE
      aeid IN (%s);
    "
  qstring <- sprintf(qformat, paste(aeids, collapse = ","))
  
  l5_coffs <- tmquery(query = qstring, db = "invitrodb")
  
  if (ncol(l5_coffs) == 0) {
    warning("The given aeid(s) do not have methods in the invitrodb ",
            "l5_coffs table.")
    return(l5_coffs)
  }
  
  l5_coffs[ , aeid := as.character(aeid)]
  
  l5_coffs
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# fitSubset: fit a subset of data 
#-------------------------------------------------------------------------------

fitSubset <- function(dat, bmad = NULL, agby = "spid") {

  if (is.null(bmad)) bmad <- 0.1
  if (length(bmad) > 1) {
    warning("Only the first value of bmad will be used.")
    bmad <- bmad[1]
  }
  
  chid_info <- loadChidInfo(dat[ , unique(spid)])
  setkey(chid_info, spid) 
  setkey(dat, spid)
  dat <- chid_info[dat]
  rm(chid_info)
  dat[is.na(chid), chid := spid]
  
  if (agby == "spid") {
    dat[ , l4id := paste0("DUMMY", .GRP), by = list(aeid, chid, spid)]
    fits <- dat[ , doFit(logc, resp, bmad, l4id = unique(l4id), agby = agby),
                by = list(aeid, chid, spid)]
  } else {
    dat[ , l4id := paste0("DUMMY", .GRP), by = list(aeid, chid)]
    fits <- dat[ , doFit(logc, resp, bmad, l4id = unique(l4id), agby = agby,
                         spid = paste(unique(spid), collapse = ";")), 
                by = list(aeid, chid)]
  }
  fits[ , bmad := bmad]
  
  list(dat = fits, agg = dat[ , list(aeid, l4id, spid, logc, resp)])
    
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# drawCircles: draw circles on the current plot device
#-------------------------------------------------------------------------------

drawCircles <- function(x, y, r, border = "black", col = NA, 
                        lwd = 1, lty = "solid", n = 100) {
  
  if (length(x) < length(y)) {
    x <- rep(x, length.out = length(y))
  }
  
  if (length(y) < length(x)) {
    y <- rep(y, length.out = length(x))
  }
  
  inc <- 2*pi/n
  angles <- angles <- seq(0, 2 * pi - inc, by = inc)
  
  if (length(col) < length(x)) {
    col <- rep(col, length.out = length(x))
  }
  
  if (length(r) < length(x)) {
    r <- rep(r, length.out = length(x))
  }
  
  if (length(lwd) < length(x)) {
    lwd <- rep(lwd, length.out = length(x))
  }
  
  if (length(lty) < length(x)) {
    lty <- rep(lty, length.out = length(x))
  }
  
  if (length(border) < length(x)) {
    border <- rep(border, length.out = length(x))
  }
  
  invisible(
    lapply(1:length(x),
           function(i) {
             xv <- cos(angles) * r[i] + x[i]
             yv <- sin(angles) * r[i] + y[i]
             polygon(xv, yv, 
                     border = border[i], 
                     col = col[i], 
                     lty = lty[i], 
                     lwd = lwd[i])
           })
  )
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotPlateHeat: create a plot showing the assay plate data
#-------------------------------------------------------------------------------

plotPlateHeat <- function(vals, rowi, coli, wllt, wllq, rown, coln, 
                          id, assay, arng) {
  
  opar <- par()[c("pty", "mar")]
  on.exit(par(opar))
  par(pty = "m")
  
  myPal <- colorRampPalette(rev(brewer.pal(11, "Spectral")), space = "Lab")
  
  outside <- vals < arng[1] | vals > arng[2]
  vals[vals < arng[1]] <- arng[1]
  vals[vals > arng[2]] <- arng[2]
  badwlls <- cbind(rowi[!wllq], coli[!wllq])
    
  layout(matrix(c(1, 1, 2, 3), ncol = 2, byrow = TRUE),
         widths = c(9, 1),
         heights = c(1, 9))
  
  par(mar = c(0, 0, 0, 0))
  plot.new()
  plot.window(xlim = 0:1,
              ylim = 0:1)
  text(0.5, 0.5, paste(assay, id, sep = ": "), font = 2, cex = 1.5)
  
  plot.new()
  par(mar = c(2, 3, 2, 2) + 0.1)
  plot.window(xlim = c(0, coln) + 0.5, 
              ylim = rev(c(0, rown) + 0.5),
              xaxs = "i", 
              yaxs = "i")
  box(which = "plot")
  abline(v = 1:(coln - 1) + 0.5)
  abline(h = 1:(rown - 1) + 0.5)
  axis(side = 3, 
       at = 1:coln, 
       tick = FALSE, 
       labels = sprintf("%02d", 1:coln), 
       cex = 0.75)
  axis(side = 2, 
       at = 1:rown, 
       tick = FALSE, 
       labels = sprintf("%02d", 1:rown), 
       cex = 0.75,
       las = 2)
  allwells <- expand.grid(1:coln, 1:rown)
  drawCircles(x = allwells[ , 1], 
              y = allwells[ , 2], 
              r = 0.4, 
              border = "gray30")
  wcol <- myPal(500)[as.numeric(cut(c(vals, arng), breaks = 500))]
  wcol <- wcol[1:(length(vals))]
  drawCircles(x = coli, 
              y = rowi, 
              r = 0.4, 
              border = "gray30",
              col = wcol,
              lwd = 1 + 3*outside)
  invisible(apply(badwlls, 
                  1, 
                  function(x) {
                    lines(x = c(x[2] - 0.5, x[2] + 0.5),
                          y = c(x[1] - 0.5, x[1] + 0.5))
                    lines(x = c(x[2] - 0.5, x[2] + 0.5),
                          y = c(x[1] + 0.5, x[1] - 0.5))
                  }))  
  drawCircles(x = coli, 
              y = rowi, 
              r = 0.2, 
              border = "gray30",
              col = "white")
  text(x = coli,
       y = rowi,
       label = wllt,
       font = 2,
       col = "black",
       cex = 0.75)
    
  plot.new()
  par(mar = c(2, 0, 2, 6) + 0.1)
  plot.window(xlim = c(0, 1), 
              ylim = arng, 
              xaxs = "i", 
              yaxs = "i")
  rect(0, 
       seq(arng[1], arng[2], length.out = 101)[-101],
       1,
       seq(arng[1], arng[2], length.out = 101)[-1],
       col = myPal(100),
       border = myPal(100))
  apts <- quantile(arng, probs = seq(0, 1, length.out = 10))
  axis(side = 4,
       at = apts, 
       las = 2, 
       cex = 0.5,
       labels = sprintf("%5.1f", apts)) 
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# inspectPlate: look at the plate map for a given apid
#-------------------------------------------------------------------------------

inspectPlate <- function(dat, ap, lvl, assay = NULL) {
  
  ### This function plots a plate heatmap for the given data, assay plate, and
  ### assay
  ###
  ### Arguments:
  ###   dat:   a tcpl1, tcpl2, or tcpl3 object containing the plate data to 
  ###          plot
  ###   ap:    the apid to plot
  ###   assay: the desired assay, only necessary to specify when an apid has 
  ###          more than one acid or aeid with associated data
  
  if (length(ap) != 1) stop("ap must be of length 1.")
  
  if (lvl == 1) {
    
    sub <- dat[apid == ap]
    
    if (is.null(assay)) {
      
      ac <- sub[ , unique(acid)]
      if (length(ac) != 1) stop("multiple acids for the given plate, ",
                                "must specifiy assay.")
      
    } else {
      
      ac <- assay
      if (length(ac) != 1) stop("assay must be of length 1.")
    
    }
    
    nwlt <- sub[ , paste0(wllt, cndx)]
    
    ap_size <- loadApidSize(ac)
    ap_size[ , rown := max(rown, sub[ , max(rowi)])]
    ap_size[ , coln := max(coln, sub[ , max(coli)])]
    
    ac_name <- loadAcidName(ac)
    ac_name <- ac_name[ , cat_name := paste0(acid, " (", acnm, ")")]
        
    arng <- dat[acid == ac, 
                quantile(rval, c(0.001, 0.999))]
    with(data = sub, plotPlateHeat(vals = rval,
                                   rowi = rowi, 
                                   coli = coli, 
                                   wllt = nwlt,
                                   wllq = wllq,
                                   rown = ap_size[ , rown], 
                                   coln = ap_size[, coln], 
                                   id = ap, 
                                   assay = ac_name[ , cat_name], 
                                   arng = arng))
  
  }
  
  if (lvl == 2) {
    
    sub <- dat[apid == ap]
    
    if (is.null(assay)) {
      
      ac <- sub[ , unique(acid)]
      if (length(ac) != 1) stop("multiple acids for the given plate, ",
                                "must specifiy assay.")
      
    } else {
      
      ac <- assay
      if (length(ac) != 1) stop("assay must be of length 1.")
      
    }
    
    nwlt <- sub[ , paste0(wllt, cndx)]
    
    ap_size <- loadApidSize(ac)
    ac_name <- loadAcidName(ac)
    ac_name <- ac_name[ , cat_name := paste0(acid, " (", acnm, ")")]
    
    arng <- dat[acid == ac, 
                quantile(cval, c(0.001, 0.999))]
    with(data = sub, plotPlateHeat(vals = cval,
                                   rowi = rowi, 
                                   coli = coli, 
                                   wllt = nwlt,
                                   wllq = wllq, 
                                   rown = ap_size[ , rown], 
                                   coln = ap_size[, coln], 
                                   id = ap, 
                                   assay = ac_name[ , cat_name], 
                                   arng = arng))
    
  }
  
  if (lvl == 3) {
    
    sub <- dat[apid == ap]
    
    if (is.null(assay)) {
      
      ae <- sub[ , unique(aeid)]
      if (length(ae) != 1) stop("multiple aeids for the given plate, ",
                                "must specifiy assay.")
      
    } else {
      
      ae <- assay
      if (length(ae) != 1) stop("assay must be of length 1.")
      
    }
    
    ac <- sub[aeid == ae , unique(acid)]
    nwlt <- sub[ , paste0(wllt, cndx)]
    
    ap_size <- loadApidSize(ac)
    ae_name <- loadAeidName(ae)
    ae_name <- ae_name[ , cat_name := paste0(aeid, " (", aenm, ")")]
    
    arng <- dat[aeid == ae, 
                quantile(resp, c(0.001, 0.999))]
    with(data = sub, plotPlateHeat(vals = resp,
                                   rowi = rowi, 
                                   coli = coli, 
                                   wllt = nwlt, 
                                   wllq = wllq,
                                   rown = ap_size[ , rown], 
                                   coln = ap_size[, coln], 
                                   id = ap, 
                                   assay = ae_name[ , cat_name], 
                                   arng = arng))
    
  }
  
  if (!lvl %in% 1:3) {
    
    stop("lvl must be 1, 2, or 3.")
    
  }
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotFit: Create plot of the dose-response with associated models
#-------------------------------------------------------------------------------

plotFit <- function(resp, logc, pars) {
  
  ###--------------------------- Draw Left Panel ----------------------------###
  
  layout(mat = matrix(1:2, nrow = 1), widths = c(4, 5.5), heights = 3.5)
  on.exit(layout(1))
  
  opar <- par()[c("pty", "mar", "family")]
  on.exit(par(opar), add = TRUE)
  par(pty = "s",
      mar = c(4, 4.5, 2, 2) + 0.1,
      family = "mono")
  
  p <- list(
    ylim = c(min(-50, min(resp)*1.2), max(150, max(resp)*1.2)),
    cex.lab = 1.2,
    cex.axis = 1.2,
    font.lab = 2,
    col = "black",
    cex = 2,
    xlab = expression(bold(paste("Concentration (",mu,"M)"))),
    ylab = paste0("Activity"),
    main = "",
    bty = "n",
    xaxt = "n",
    yaxt = "n",
    type = "n"
  )
  
  do.call(what = plot, args = c(resp ~ logc, p), quote = TRUE)
  
  rect(xleft = par()$usr[1],
       xright = par()$usr[2], 
       ybottom = -3 * pars$bmad, 
       ytop = 3 * pars$bmad,
       border = NA, 
       col = "gray70",
       density = 15, 
       angle = 45)
  
  maic <- with(pars, min(cnst_aic, hill_aic, gnls_aic, 1e6, na.rm = TRUE))
  
  if (!is.na(pars$cnst) & pars$cnst) {
    
    abline(h = 0,
           lwd = 4, # ifelse(fit$cnst$aic == maic, 4, 2.5),
           col = "darkorange",
           lty = ifelse(pars$cnst_aic == maic, "solid", "dashed"))
    
  }
  
  if (!is.na(pars$hill) & pars$hill) {
    
    hill.eq <- function(x) with(pars, hill_tp/(1 + 10^((hill_ga - x)*hill_gw)))
    curve(hill.eq, 
          from = pars$logc_min, 
          to = pars$logc_max,
          add = T, 
          n = 1e4, 
          lwd = 4, # ifelse(fit$hill$aic == maic, 4, 2.5), 
          col = "tomato3",
          lty = ifelse(pars$hill_aic == maic, "solid", "dashed"))  
    abline(v = pars$hill_ga,
           lwd = 2.5,
           lty = ifelse(pars$hill_aic == maic, "solid", "dashed"),
           col = "tomato3")
    
  }
  
  if (!is.na(pars$gnls) & pars$gnls) {
    
    gnls.eq <- function(x) {
      with(pars, {
        h1 <- (1/(1 + 10^((gnls_ga - x)*gnls_gw)))
        h2 <- (1/(1 + 10^((x - gnls_la)*gnls_lw)))
        gnls_tp*h1*h2
      })
    } 
    curve(gnls.eq, 
          from = pars$logc_min, 
          to = pars$logc_max,
          add = T, 
          n = 1e4, 
          lwd = 4, # ifelse(gnls$hill$aic == maic, 4, 2.5), 
          col = "dodgerblue2",
          lty = ifelse(pars$gnls_aic == maic, "solid", "dashed"))  
    abline(v = pars$gnls_ga,
           lwd = 2.5,
           lty = ifelse(pars$gnls_aic == maic, "solid", "dashed"),
           col = "dodgerblue2")
    
  }
  
  axis(side = 1, 
       at = axTicks(side = 1),
       labels = signif(10^axTicks(side = 1), digits = 1),
       font = 1, 
       lwd = 2, 
       cex.axis = 1.2, 
       col = "gray35")
  axis(side = 2, 
       at = axTicks(side = 2), 
       labels = axTicks(side = 2),
       font = 1, 
       lwd = 2, 
       cex.axis = 1.2, 
       col = "gray35")
  
  # points(x = pars$emax_conc,
  #        y = pars$emax,
  #        pch = 22,
  #        cex = 2,
  #        col = "gray35",
  #        lwd = 1,
  #        bg = "yellow2")
  
  points(resp ~ logc, cex = 1.5, lwd = 2.5, col = "gray30")
  
  ###--------------------- Prepare Text for Right Panel ---------------------###
  
  spaces <- function(x) paste(rep(" ", x), collapse = "")
  
  itxt <- with(pars, {
    paste0("ASSAY:   ", aenm, "\n\n",
           "NAME:    ", chnm, "\n",
           "CHID:    ", chid, spaces(8 - nchar(chid)),
           "CASRN: ", casn, spaces(16 - nchar(casn)),
           "AGBY: ", agby, "\n",
           "SPID(S): ", spid, "\n",
           "L4ID:    ", l4id, "\n\n"
    )
  })
  
  if (!is.na(pars$hill) & pars$hill) {
    
    if (pars$hcov) {
      hsds <- with(pars, signif(c(hill_tp_sd, hill_ga_sd, hill_gw_sd), 3))
      hsds[is.na(hsds)] <- NaN
    } else {
      hsds <- rep(NA, 5)
    }
    
    hprs <- with(pars, signif(c(hill_tp, hill_ga, hill_gw), 3))
    
    htxt1 <- paste("HILL MODEL (in red):\n      tp", 
                   "ga",
                   "gw\n",
                   sep = spaces(7))
    
    htxt2 <- paste0(c("val:  ", "sd:   "),
                    c(paste(sapply(hprs, 
                                   function(x) {
                                     paste0(x, spaces(9 - nchar(x)))
                                   }),
                            collapse = ""),
                      paste(sapply(hsds, 
                                   function(x) {
                                     paste0(x, spaces(9 - nchar(x)))
                                   }),
                            collapse = "")),
                    collapse = "\n")
    
    htxt <- paste0(htxt1, htxt2, "\n\n")
    
  } else {
    
    if (is.na(pars$hill)) {
      htxt <- "HILL MODEL: Not applicable.\n\n"
    } else {
      htxt <- "HILL MODEL: Failed to converge.\n\n"
    } 
    
  }
  
  if (!is.na(pars$gnls) & pars$gnls) {
    
    if (pars$gcov) {
      gsds <- with(pars, 
                   signif(c(gnls_tp_sd, 
                            gnls_ga_sd, 
                            gnls_gw_sd, 
                            gnls_la_sd, 
                            gnls_lw_sd),
                          3)
      )
      gsds[is.na(gsds)] <- NaN
    } else {
      gsds <- rep(NA, 5)
    }
    
    gprs <- with(pars, 
                 signif(c(gnls_tp, gnls_ga, gnls_gw, gnls_la, gnls_lw), 3))
    
    gtxt1 <- paste("GAIN-LOSS MODEL (in blue):\n      tp",
                   "ga", 
                   "gw",
                   "la",
                   "lw\n",
                   sep = spaces(7))
    
    gtxt2 <- paste0(c("val:  ", "sd:   "),
                    c(paste(sapply(gprs, 
                                   function(x) {
                                     paste0(x, spaces(9 - nchar(x)))
                                   }),
                            collapse = ""),
                      paste(sapply(gsds, 
                                   function(x) {
                                     paste0(x, spaces(9 - nchar(x)))
                                   }),
                            collapse = "")),
                    collapse = "\n")
    
    gtxt <- paste0(gtxt1, gtxt2, "\n\n")
    
  } else {
    
    if (is.na(pars$hill)) {
      gtxt <- "GAIN-LOSS MODEL: Not applicable.\n\n"
    } else {
      gtxt <- "GAIN-LOSS MODEL: Failed to converge.\n\n"
    } 
    
  }
  
  aics <- with(pars, round(c(cnst_aic, hill_aic, gnls_aic), 2))
  models <- c("CNST", "HILL", "GNLS")
  
  atxt <- paste0(spaces(5), 
                 models[1], 
                 spaces(8), 
                 models[2], 
                 spaces(8), 
                 models[3],
                 "\n",
                 paste0("AIC: ", 
                        aics[1], 
                        spaces(12 - nchar(aics[1])),
                        aics[2],
                        spaces(12 - nchar(aics[2])),
                        aics[3]),
                 "\n\n")
  
  pars$max_mean <- signif(pars$max_mean, 3)
  
  ntxt <- paste0("EMAX:  ", pars$max_mean, 
                 spaces(12 - nchar(pars$max_mean)),
                 "BMAD:  ", signif(pars$bmad, 3),
                 "\n\n")
  
  if (!is.null(pars$hitc)) {
    
    ctxt <- paste0("COFF: ", pars$coff, spaces(4 - nchar(pars$coff)),
                   "HIT-CALL: ", pars$hitc, spaces(4 - nchar(pars$hitc)), 
                   "FITC: ", pars$fitc)
    
  } else {
    
    ctxt <- NULL
    
  }
  
  plot_txt1 <- paste0(itxt, htxt, gtxt, atxt, ntxt, ctxt)
  
  if (maic != 1e6) {
    nlines <- sum(7,
                  length(gregexpr("\n", htxt)[[1]]),
                  length(gregexpr("\n", gtxt)[[1]]))
    winner <- which(aics == round(maic, 2))
    if (length(winner) > 1) winner <- winner[1]
    
    plot_txt2 <- paste0(paste(rep("\n", nlines), collapse = ""),
                        spaces(5 + 12*(winner - 1)),
                        models[winner],
                        "\n",
                        spaces(5 + 12*(winner - 1)),
                        aics[winner])
    
  } else {
    
    plot_txt2 <- NULL
    
  }
  
  
  ###--------------------------- Draw Right Panel ---------------------------###
  
  par(pty = "m", 
      family = "mono",
      mar = rep(2,4) + 0.1)
  
  plot(0, 
       type = "n", 
       bty = "n", 
       xaxt = "n", 
       yaxt = "n", 
       ylab = "", 
       xlab = "", 
       xlim = c(0, 16), 
       ylim = c(0, 16))
  
  suppressWarnings(
    text(y = 15, 
         x = 1,
         labels = plot_txt1, 
         adj = c(0, 1),
         font = 2,
         cex = 1)
  )
  
  suppressWarnings(
    text(y = 15, 
         x = 1,
         labels = plot_txt2, 
         adj = c(0, 1),
         font = 2,
         cex = 1,
         col = "red")
  )
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadFitc: Load the fit categories from invitrodb
#-------------------------------------------------------------------------------

loadFitc <- function(aeid) {
  
  qformat <- 
    "
    SELECT
      fitc,
      aeid
    FROM 
      level5
    WHERE
      aeid IN (%s)
    "
  
  qstring <- sprintf(qformat, paste0("\"", aeid, "\"", collapse = ","))
  
  fitcs <- tmquery(qstring, "invitrodb")
  
  if (ncol(fitcs) == 0) {
    warning("The given aeid(s) are not listed in the invitrodb level5 table.")
    return(fitcs)
  }
  
  fitcs[ , aeid := as.character(aeid)]
  setkey(fitcs, aeid)
  
  if (length(aeid) > 1) {
    
    fail <- sapply(aeid, function(x) nrow(fitcs[aeid == x]) == 0)
    
    if (any(fail)) {
      
      warning("No fit categories loaded for: ", 
              paste(aeid[fail], collapse = ", "),
              ".")
      
    }
    
  }
    
  fitcs
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadAsidName: Load assay source name from invitrodb
#-------------------------------------------------------------------------------

loadAsidName <- function(asid) {
  
  qformat <- 
    "
    SELECT
      assay_source_id as asid,
      assay_source_name AS asnm,
      assay_source_long_name AS asnm_long
    FROM
      assay_source
    WHERE
      assay_source_id IN (%s);
    "
  
  qstring <- sprintf(qformat, paste0("\"", asid, "\"", collapse = ","))
  
  asnm <- tmquery(qstring, "invitrodb")
  
  if (ncol(asnm) == 0) {
    warning("The given asid(s) are not listed in the invitrodb ",
            "assay_source table.")
    return(asnm)
  }
  
  asnm[ , asid := as.character(asid)]
  setkey(asnm, asid)
  
  if (length(asid) > 1) {
    
    fail <- sapply(asid, function(x) nrow(asnm[asid == x]) == 0)
    
    if (any(fail)) {
      
      warning("No assay source names categories loaded for: ", 
              paste(asid[fail], collapse = ", "),
              ".")
      
    }
    
  }
  
  asnm
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotTcplFits: Plot fits based on level4/5 and agg_level4
#-------------------------------------------------------------------------------

plotTcplFits <- function(dat, agg, browse = FALSE) {
  
  ## Load and map chemical information
  chid_info <- suppressWarnings(loadChidInfo(agg[ , unique(spid)]))
  if (ncol(chid_info) != 0) {
    setkey(chid_info, spid)
    setkey(agg, spid)
    agg <- chid_info[agg]
    chem_name <- loadChemName(agg[ , unique(chid)])
    setkey(chem_name, chid)
    setkey(agg, chid)
    agg <- chem_name[agg]
  } else {
    agg[ , c("chid", "casn") := NA_character_] 
  }
  agg[is.na(chid), chnm := paste(spid, "(spid not in DB)")]
  
  ## Load assay name
  aenms <- loadAeidName(dat$aeid)
  setkey(dat, aeid)
  setkey(aenms, aeid)
  dat <- aenms[dat]
  dat[ , aenm := paste0(aeid, " (", aenm, ")")]
  
  ## Load coff values, if applicable
  if ("hitc" %in% names(dat)) {
    coffs <- loadL5Mthds(dat[ , unique(aeid)])
    setkey(coffs, aeid)
    setkey(dat, aeid)
    dat <- coffs[dat]
  }
    
  agg[ , spid := paste(unique(spid), collapse = ";"), by = l4id]
  
  setkey(dat, l4id)
  setkey(agg, l4id)
  
  chem_info_cols <- c("chid", "casn", "chnm", "spid")
  dat <- dat[agg[ , unique(.SD), .SDcols = chem_info_cols, key = "l4id"]]
  agg <- agg[ , list(l4id, resp, logc)]
  setkey(agg, l4id)
  
  l4ids <- dat[order(chid), unique(l4id)]
  
  for (i in l4ids) {
    
    resp <- agg[J(i), resp]
    logc <- agg[J(i), logc]
    pars <- dat[J(i)]
    plotFit(resp, logc, pars)
    if (browse) browser()
    
  }
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotL4Aeid: Create a .pdf with level 4 plots
#-------------------------------------------------------------------------------

plotL4Aeid <- function(ae, pdir) {
  
  dat <- loadLvl4Data("aeid", ae)
  agg <- loadLvl4Agg("aeid", ae)
  graphics.off()
  pdf(file = file.path(pdir,
                       paste("L4DR",
                             paste0("AEID", ae),
                             loadAeidName(ae)[ , aenm],
                             format(Sys.Date(), "%y%m%d.pdf"),
                             sep = "_")),
      height = 6, 
      width = 10, 
      pointsize = 10)
  plotTcplFits(dat, agg)
  graphics.off()
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotL5Aeid: Create a .pdf with level 5 plots
#-------------------------------------------------------------------------------

plotL5Aeid <- function(ae, pdir) {
  
  dat <- loadLvl5Data("aeid", ae)
  agg <- loadLvl4Agg("aeid", ae)
  pdf(file = file.path(pdir,
                       paste("L5DR",
                             paste0("AEID", ae),
                             loadAeidName(ae)[ , aenm],
                             format(Sys.Date(), "%y%m%d.pdf"),
                             sep = "_")),
      height = 6, 
      width = 10, 
      pointsize = 10)
  plotTcplFits(dat, agg)
  graphics.off()
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# Functions to calculate sample sizes for plotting by fit category
#-------------------------------------------------------------------------------

of <- function(m, Z = 0, N = 10000, P = 0.01, T2E = 0.2, pwr = FALSE) {
  if (pwr) T2E <- 0.0
  a <- round(N * P)
  phyper(Z, a, N - a, m, lower.tail = !pwr) - T2E
}

getSS <- function(Z = 0, N = 10000, P = 0.01, T2E = 0.2, ub = N) {
  out <- try(uniroot(of, interval = c(1, ub), Z = Z, P = P, N = N, T2E = T2E))
  if (!is(out, "try-error")) round(out$root) else NA
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotL5Fitc: Create a .pdf with level 5 plots for a specific fit category
#-------------------------------------------------------------------------------

plotL5Fitc <- function(asid, fitc, n = 150, pdir = getwd()) {
  
  aes <- loadAsidAeid(asid)
  dat <- invisible(loadLvl5Data(c("aeid", "fitc"), list(aes, fitc)))
  
  if (nrow(dat) > n) {
    ss <- getSS(N = nrow(dat), T2E = 0.05)
    dat <- dat[sample(1:nrow(dat), ss)]
  }
  
  agg <- invisible(loadLvl4Agg("l4id", dat[ , l4id]))
  coffs <- loadL5Mthds(aes)
  setkey(coffs, aeid)
  setkey(dat, aeid)
  dat <- coffs[dat]
  graphics.off()
  pdf(file = file.path(pdir,
                       paste("L5DR",
                             paste0("ASID", asid),
                             loadAsidName(asid)[ , asnm],
                             paste0("FITC", fitc),
                             format(Sys.Date(), "%y%m%d.pdf"),
                             sep = "_")),
      height = 6, 
      width = 10, 
      pointsize = 10)
  plotTcplFits(dat, agg)
  graphics.off()  
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotVendFitc: Create level 5 plots for all the fit categories, by vendor
#-------------------------------------------------------------------------------

plotVendFitc <- function(asids, n = 150, pdir = getwd()) {
  
  fitcs <- mclapply(asids, 
                    mc.cores = min(length(asids), detectCores() - 1),
                    function(x) {
                      aes <- loadAsidAeid(x)
                      data.table(asid = x, 
                                 fitc = loadFitc(aes)[ , unique(fitc)])
                    })
  fitcs <- rbindlist(fitcs)
  invisible(mcmapply(plotL5Fitc,
                     mc.preschedule = FALSE,
                     mc.cores = min(nrow(fitcs), detectCores() - 1),
                     asid = fitcs[ , asid],
                     fitc = fitcs[ , fitc],
                     MoreArgs = list(n = n, pdir = pdir)))
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotFitcTree: Plot the fitc tree diagram
#-------------------------------------------------------------------------------

plotFitcTree <- function(fitc, main = "") {
  
  vals <- data.table(fitc = fitc)[ , .N, by = fitc]
  
  mypal <- rev(c("#A50026", "#FF7F00", "#FFFF33", "#33A02C",
                 "#1F78B4", "#762A83"))
  
  clrs <- data.table(edge = 1:8,
                     r = col2rgb(colorRampPalette(mypal)(8))["red", ]/255,
                     g = col2rgb(colorRampPalette(mypal)(8))["green", ]/255,
                     b = col2rgb(colorRampPalette(mypal)(8))["blue", ]/255)
  clrs[ , col := rgb(r, g, b, 0.4)]
  
  vmax <- unname(quantile(vals[ , N], 0.9))
  vmin <- min(vals[ , N])
  
  b <- unlist(lapply(0:7, function(x) { vmax/((vmax/vmin)^(1/8))^x }))
  b <- round(rev(b[1:7]), 0)
  
  vals[N <= b[1],            col := clrs[edge == 1, col]]
  vals[N <= b[2] & N > b[1], col := clrs[edge == 2, col]]
  vals[N <= b[3] & N > b[2], col := clrs[edge == 3, col]]
  vals[N <= b[4] & N > b[3], col := clrs[edge == 4, col]]
  vals[N <= b[5] & N > b[4], col := clrs[edge == 5, col]]
  vals[N <= b[6] & N > b[5], col := clrs[edge == 6, col]]
  vals[N <= b[7] & N > b[6], col := clrs[edge == 7, col]]
  vals[N >  b[7],            col := clrs[edge == 8, col]]
  
  tree <- tmquery("SELECT * FROM l5_fit_categories;", "invitrodb")
  tree[ , xloc := xloc - xloc[1]]
  tree[ , yloc := yloc - yloc[1]]
  
  # pdf("test.pdf", width = 10, height = 7.5, pointsize = 9)
  
  opar <- par()[c("pty")]
  on.exit(par(opar))
  par(pty = "s")
  
  p <- list(type = "n",
            bty = "n",
            xlim = c(-1000, 1000),
            ylim = c(-1000, 1000),
            ylab = "",
            xlab = "",
            xaxt = "n",
            yaxt = "n",
            main = main
  )
  
  do.call(plot, c(tree$yloc ~ tree$xloc, p), quote = TRUE)
  
  for (i in 2:55) {
    lines(x = c(tree[i, xloc], tree[tree[i, parent_fitc], xloc]),
          y = c(tree[i, yloc], tree[tree[i, parent_fitc], yloc]))
  }
  
  rect(xleft = tree[ , xloc - 120], 
       xright = tree[ , xloc + 120], 
       ybottom = tree[ , yloc - 15], 
       ytop = tree[ , yloc + 15],
       border = NA, 
       col = "white")
  
  drawCircles(x = tree[vals[ , fitc], xloc], 
              y = tree[vals[ , fitc], yloc], 
              r = vals[ , (log(N, 2) + 2)*15], 
              border = NA, 
              col = vals[ , col])
  
  text(x = tree[ , xloc], 
       y = tree[ , yloc], 
       labels = tree[ , name], 
       cex = 0.45)
  
  legend(x = "bottom",
         ncol = 8, 
         bty = "n",
         pch = 19,
         cex = 0.8,
         col = clrs[ , col],
         legend = c(paste0("1-", b[1]), 
                    paste0(b[1] + 1, "-", b[2]), 
                    paste0(b[2] + 1, "-", b[3]), 
                    paste0(b[3] + 1, "-", b[4]), 
                    paste0(b[4] + 1, "-", b[5]), 
                    paste0(b[5] + 1, "-", b[6]), 
                    paste0(b[6] + 1, "-", b[7]), 
                    paste0(b[7] + 1, "+")))
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotVendTree: Create .pdf containing the fit category trees, by vendor
#-------------------------------------------------------------------------------

plotVendTree <- function(asids, pdir = getwd()) {
  
  invisible(mclapply(asids, mc.cores = min(length(asids), detectCores() - 1),
                     function(x) {
                       aes <- loadAsidAeid(x)
                       asnm <- loadAsidName(x)
                       graphics.off()
                       pdf(file.path(pdir,
                                     paste("FCTREE",
                                           paste0("ASID", x),
                                           asnm[ , asnm],
                                           format(Sys.Date(), "%y%m%d.pdf"),
                                           sep = "_")), 
                           width = 10, 
                           height = 7.5, 
                           pointsize = 9)
                       plotFitcTree(loadFitc(aes)[ , fitc],
                                    main = asnm[ , asnm_long])
                       graphics.off()
                     })) 
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# plotL4ID: Plot dose-response by l4id
#-------------------------------------------------------------------------------

plotL4ID <- function(l4id, lvl = 4L) {
  
  if (length(lvl) > 1 | !lvl %in% 4:5) stop("invalid lvl input.")
  
  if (lvl == 4L) dat <- loadLvl4Data("l4id", l4id)
  if (lvl == 5L) dat <- loadLvl5Data("l4id", l4id)
  agg <- loadLvl4Agg("l4id", l4id)
  
  plotTcplFits(dat = dat, agg = agg)

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# LoadL0IDSpid: Load all the level ids associated with an spid and acid
#-------------------------------------------------------------------------------

loadL0IDSpid <- function(spid, acid) {
  
  qformat <- "SELECT l0id FROM level0 WHERE spid IN (%s) AND acid IN (%s);"
  qstring <- sprintf(qformat, 
                     paste0("\"", spid, "\"", collapse = ","),
                     paste0("\"", acid, "\"", collapse = ","))
  
  l0ids <- tmquery(qstring, "invitrodb")[ , l0id]
  l0ids
  
}

#-------------------------------------------------------------------------------

### The following functions work to modify the toxminer databases. 
### PLEASE PROCEDE WITH CAUTION.

#-------------------------------------------------------------------------------
# tmupdate: Update Toxminer Databases
#-------------------------------------------------------------------------------

tmupdate <- function(query, db) {
  
  ### This function takes a sql query string and a toxminer database name and 
  ### updates the database. 
  ###
  ### Arguments: 
  ###   query: a character vector of length one containing a sql query
  ###   db:    a character vector of length one containing the name of the 
  ###          toxminer database to be queried
  
  #Check for valid inputs
  if (length(query) != 1 | class(query) != "character") {
    stop("The input 'query' must be a character of length one.")
  }
  if (length(db) != 1 | class(db) != "character") {
    stop("The input 'db' must be a character of length one.")
  }
  
  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer_su",
                     password = "pass",
                     dbname = db,
                     host = "134.67.216.45")
  
  temp <- dbSendQuery(dbcon, query)
  dbDisconnect(dbcon)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# deleteRows: Delete rows from invitro db
#-------------------------------------------------------------------------------

deleteRows <- function(tbl, field, vals) {
  
  ### This function deletes rows from the invitro database 
  ### for the given table. 
  ###
  ### Arguments:
  ###   tbl:   character of length 1, the table to delete rows from
  ###   field: character, the field(s) to select the rows to delete
  ###   vals:  list containing the desired values from the field(s) specified
  ###          by field. Must be in the same order as field. 
  
  # Check for valid inputs
  if (length(tbl) != 1 | class(tbl) != "character") {
    stop("The input 'tbl' must be a character of length one.")
  }
    
  qformat <- paste("DELETE FROM", tbl, "WHERE")
  
  qformat <- paste0(qformat, "  ", paste(field, "IN (%s)", collapse = " AND "))
  qformat <- paste0(qformat, ";")
  
  if (!is.list(vals)) vals <- list(vals)
  vals <- lapply(vals, function(x) paste0("\"", x, "\"", collapse = ","))
  
  qstring <- do.call(sprintf, args = c(qformat, vals))
  
  tmupdate(query = qstring, db = "invitrodb")
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# deleteCascade: Do a cascading delete in invitro db
#-------------------------------------------------------------------------------

deleteCascade <- function(lvl, ids) {
  
  ### This function deletes out all the data for the given ids starting at the 
  ### level given. The delete will cascade through all subsequent tables.
  ### 
  ### Arguments:
  ###   lvl: an integer of length one, the first level to delete from
  ###   ids: the ids to delete, can give acid or aeid values, depending on the
  ###        starting level
  ###
  ### Note:
  ###   If lvl is less than 4, ids are interpreted as acids and if lvl is
  ###   greater than 4 ids are interpreted as aeids.
  
  stime <- Sys.time()
  
  if (length(lvl) > 1 | !is.integer(lvl)) {
    stop("Invalid lvl input - must be an integer of length 1.")
  }
  
  if (lvl == 0L) deleteRows(tbl = "level0", field = "acid", vals = ids)
  if (lvl <= 1L) deleteRows(tbl = "level1", field = "acid", vals = ids)
  if (lvl <= 2L) deleteRows(tbl = "level2", field = "acid", vals = ids)
  if (lvl <= 3L) deleteRows(tbl = "level3", field = "acid", vals = ids)
  if (lvl <  4L) {
    ids <- suppressWarnings(try(loadAeidInfo(acids = ids)[ , aeid], 
                                silent = TRUE))
  }
  if (is(ids, "try-error")) return(TRUE)
  if (lvl <= 4L) {
    deleteRows(tbl = "level4", field = "aeid", vals = ids)
    deleteRows(tbl = "agg_level4", field = "aeid", vals = ids)
  }
  if (lvl <= 5L) deleteRows(tbl = "level5", field = "aeid", vals = ids)
  
  ttime <- round(difftime(Sys.time(), stime, units = "sec"), 2)
  ttime <- paste(unclass(ttime), units(ttime))
  cat("Completed delete cascade for ", length(ids), " ids (", ttime, 
      ")\n", sep = "")
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertInvtDat: Insert invitro data in invitro tb
#-------------------------------------------------------------------------------

insertInvtDat <- function(dat, lvl) {
  
  ### This function inserts rows in the invitro database, for the table given
  ### by lvl. 
  ###
  ### Arguments:
  ###   dat: a data.table containing the data to be added to the given table
  ###   lvl: an integer of length 1, the table level to insert data into
  ###
  ### Note:
  ###   This function appends data onto the existing table. It also deletes all
  ###   the data for any acids or aeids dat contains from the given and all 
  ###   downstream tables.
  
  if (length(lvl) > 1 | !is.integer(lvl)) {
    stop("Invalid lvl input - must be an integer of length 1.")
  }
  
  if (!lvl %in% 0L:6L) {
    stop("This function is only for adding in vitro data, which is only loaded",
         "\nin the 'level#' tables - excluding level4.")    
  }
  
  if (lvl == 0L) {
    chid_info <- loadChidInfo(dat[ , unique(spid)])
    setkey(chid_info, spid)
    if (dat[wllt == "t" , lw(!spid %in% chid_info$spid)] > 0) {
      cat("The following test compounds did not map to the inventorydb:\n")
      print(dat[wllt == "t" & !spid %in% chid_info$spid, unique(spid)])
      stop("Must correct the test compound mapping before loading the data.")
    }    
  }
  
  if (lvl > 3) fkey <- "aeid" else fkey <- "acid"
  
  deleteCascade(lvl = lvl, ids = dat[ , unique(get(fkey))])
  
  dat[ , modified_by := paste(Sys.info()[c("login","user","effective_user")],
                              collapse = ".")]
  
  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer_su",
                     password = "pass",
                     dbname = "invitrodb",
                     host = "134.67.216.45")
  
  dbWriteTable(conn = dbcon, 
               name = paste0("level", lvl), 
               value = dat, 
               row.names = FALSE, 
               append = TRUE)
  
  dbDisconnect(dbcon)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertLvl4Dat: Insert data in invitro level4
#-------------------------------------------------------------------------------

insertLvl4Dat <- function(dat) {
  
  ### This function inserts rows in the invitro database level4 and agg_level4
  ### tables. 
  ###
  ### Arguments:
  ###   dat: a data.table containing the level 4 data to be added to level4 and
  ###        agg_level4
  
  
  level4_cols <- fitpars <- c("aeid",
                              "agby",
                              "bmad",
                              "resp_max",
                              "resp_min",
                              "max_mean",
                              "max_mean_conc",
                              "max_med",
                              "max_med_conc",
                              "logc_max",
                              "logc_min",
                              "cnst",
                              "hill",
                              "hcov",
                              "gnls",
                              "gcov",
                              "cnst_er",
                              "cnst_aic",
                              "hill_tp",
                              "hill_tp_sd",
                              "hill_ga",
                              "hill_ga_sd",
                              "hill_gw",
                              "hill_gw_sd",
                              "hill_er",
                              "hill_er_sd",
                              "hill_aic",
                              "gnls_tp",
                              "gnls_tp_sd",
                              "gnls_ga",
                              "gnls_ga_sd",
                              "gnls_gw",
                              "gnls_gw_sd",
                              "gnls_la",
                              "gnls_la_sd",
                              "gnls_lw",
                              "gnls_lw_sd",
                              "gnls_er",
                              "gnls_er_sd",
                              "gnls_aic",
                              "tmpi")
  
  agg_cols <- c(paste0("l", 0:4, "id"), "aeid")
  
  insertInvtDat(dat = dat[ , unique(.SD) , .SDcols = level4_cols], 4L)
  
  qformat <- "SELECT l4id, tmpi FROM level4 WHERE aeid = %s;"
  qstring <- sprintf(qformat, dat[ , unique(aeid)])
  
  l4id_map <- tmquery(query = qstring, db = "invitrodb")
  setkey(l4id_map, tmpi)
  setkey(dat, tmpi)
  
  dat <- l4id_map[dat]
  
  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer_su",
                     password = "pass",
                     dbname = "invitrodb",
                     host = "134.67.216.45")
  
  dbWriteTable(conn = dbcon, 
               name = "agg_level4", 
               value = dat[ , .SD, .SDcols = agg_cols], 
               row.names = FALSE, 
               append = TRUE)
  
  dbDisconnect(dbcon)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertL2AC: Insert rows in invitro l2_ac
#-------------------------------------------------------------------------------

insertL2AC <- function(acids, mthd, ordr) {
  
  ### This function inserts rows in the invitro database l2_ac table. 
  ###
  ### Arguments:
  ###   acids: a character vector containing the acids to insert
  ###   mthd:  an integer of length 1 with the new l2_method_id value 
  ###   ordr:  an integer of length 1 with the new execute_order value
  ###
  ### Notes:
  ###   Running this function will do a cascading delete starting at level 2
  ###   for the given acids. 
  
  if (length(mthd) != 1 | !is.integer(mthd)) {
    stop("mthd must be integer of length 1.")
  }
  if (length(ordr) != 1 | !is.integer(ordr)) {
    stop("mthd must be integer of length 1.")
  }
  
  qformat <- 
    "
    INSERT INTO
      l2_ac 
    VALUES
      %s;
    "
  
  qstring <- sprintf(qformat, paste(paste0("(", 
                                           mthd, 
                                           ", ", 
                                           acids, 
                                           ", ", 
                                           ordr, 
                                           ")"),
                                    collapse = ",\n\t\t\t\t"))
  
  tmupdate(query = qstring, db = "invitrodb")
  
  deleteCascade(lvl = 2L, ids = acids)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertL3AE: Insert rows in invitro l3_ace 
#-------------------------------------------------------------------------------

insertL3AE <- function(aeids, mthd, ordr) {
  
  ### This function inserts rows in the invitro database l3_ace table. 
  ###
  ### Arguments:
  ###   acids: a character vector containing the aeids to insert
  ###   mthd:  an integer of length 1 with the new l3_method_id value 
  ###   ordr:  an integer of length 1 with the new execute_order value
  ###
  ### Notes:
  ###   Running this function will do a cascading delete starting at level 3
  ###   for the given aeids. 
  
  qformat <- 
    "
    INSERT INTO
      l3_ace
    VALUES
      %s;
    "
  
  qstring <- sprintf(qformat, paste(paste0("(", 
                                           mthd, 
                                           ", ", 
                                           aeids, 
                                           ", ", 
                                           ordr, 
                                           ")"),
                                    collapse = ",\n\t\t\t\t"))
  
  tmupdate(query = qstring, db = "invitrodb")
  
  deleteRows(tbl = "level3", field = "aeid", vals = aeids)
  deleteCascade(lvl = 4L, ids = aeids)
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertL2Mthd: Add method in invitro l2_methods
#-------------------------------------------------------------------------------

insertL2Mthd <- function(mthd, desc) {
  
  ### This function adds a method to the the invitro database l2_methods table. 
  ###
  ### Arguments:
  ###   mthd: the method name, assigned to l2_method
  ###   vals: the method description, assigned to desc 
  
  if (length(mthd) != length(desc)) {
    stop("length of mthd must equal length of desc.")
  }
  
  qformat <- 
    "
    INSERT INTO
      `l2_methods` (`l2_method_id`, `l2_method`, `desc`)
    VALUES
      %s;
    "
  
  qstring <- sprintf(qformat, 
                     paste0("(NULL, \'", mthd, "\', \'", desc, "\');"))
  
  tmupdate(query = qstring, db = "invitrodb")
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertL3Mthd: Add method in invitro l3_methods
#-------------------------------------------------------------------------------

insertL3Mthd <- function(mthd, desc) {
  
  ### This function adds a method to the the invitro database l3_methods table. 
  ###
  ### Arguments:
  ###   mthd: the method name, assigned to l3_method
  ###   vals: the method description, assigned to desc 
  
  if (length(mthd) != length(desc)) {
    stop("length of mthd must equal length of desc.")
  }
  
  qformat <- 
    "
    INSERT INTO
      `l3_methods` (`l3_method_id`, `l3_method`, `desc`)
    VALUES
      %s;
    "
  qstring <- sprintf(qformat, 
                     paste0("(NULL, \'", mthd, "\', \'", desc, "\');"))
  
  tmupdate(query = qstring, db = "invitrodb")
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# insertL5Coff: Insert rows in invitro l5_cutoffs 
#-------------------------------------------------------------------------------

insertL5Coff <- function(dat) {
  
  ### This function inserts rows in the invitro database l5_cutoffs table. 
  ###
  ### Arguments:
  ###   dat: a data.table containing the aeid and cutoff value (coff), or NA
  ###
  ### Notes:
  ###   Updating this table will remove any previous values for any aeids given
  ###   in dat, and do a cascading delete starting at level 5.
  
  if (!"data.table" %in% class(dat) | !all(c("aeid", "coff") %in% names(dat))) {
    stop("dat must be a data.table containing the fields aeid and coff. No ",
         "updates\nmade to l5_cutoffs.")
  }
  
  dat <- dat[ , unique(.SD), .SDcols = c("aeid", "coff")]
  
  if (any(duplicated(dat[ , aeid]))) {
    stop("dat has multiple cutoffs for at least 1 aeid. Must provide only one",
         " cutoff\nvalue per aeid. No updates made to l5_cutoffs.")
  }
  
  # qformat <- 
  #   "
  #   SELECT DISTINCT
  #     aeid,
  #     bmad
  #   FROM 
  #    level4
  #   WHERE 
  #     aeid IN (%s);
  #   "
  # qstring <- sprintf(qformat, 
  #                    paste0("\"", dat[ , aeid], "\"", collapse = ","))
  #   
  # bmads <- tmquery(query = qstring, db = "invitrodb")
  # bmads[ , bmad3 := 3 * bmad]
  
  dat[ , modified_by := paste(Sys.info()[c("login","user","effective_user")],
                              collapse = ".")]
  
  deleteRows(tbl = "l5_cutoffs", field = "aeid", vals = dat[ , aeid])
  
  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer_su",
                     password = "pass",
                     dbname = "invitrodb",
                     host = "134.67.216.45")
  
  dbWriteTable(conn = dbcon, 
               name = "l5_cutoffs", 
               value = dat, 
               row.names = FALSE, 
               append = TRUE)
  
  dbDisconnect(dbcon)
  
  deleteCascade(lvl = 5L, ids = dat[ , aeid])
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# appendToTbl: Append rows to a table on titan.epa.gov MySQL Server
#-------------------------------------------------------------------------------

appendToTbl <- function(dat, tbl, db = "invitrodb") {
  
  ### This function appends rows to the given database table.. 
  ###
  ### Arguments:
  ###   dat: a data.table containing the rows to be appended, with the
  ###        appropriate columns names
  ###   tbl: a character of length 1, the table to append to
  ###   db:  a character of length 1, the database
  ###
  ### Notes:
  ###   Please be cautious when using this function.

  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer_su",
                     password = "pass",
                     dbname = db,
                     host = "134.67.216.45")
  
  dbWriteTable(conn = dbcon, 
               name = tbl, 
               value = dat, 
               row.names = FALSE, 
               append = TRUE)
  
  dbDisconnect(dbcon)

}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# loadTbl: Load a database table from titan.epa.gov MySQL Server
#-------------------------------------------------------------------------------

loadTbl <- function(tbl, db = "invitrodb") {
  
  ### This function loads the given database table. 
  ###
  ### Arguments:
  ###   tbl: a character of length 1, the table to load
  ###   db:  a character of length 1, the database
  ###
  ### Value:
  ###   A data.table containing the data from the given table
  
  dbcon <- dbConnect(drv = MySQL(), 
                     user = "toxminer",
                     password = "pass",
                     dbname = db,
                     host = "134.67.216.45")
  
  dat <- as.data.table(dbReadTable(conn = dbcon, name = tbl))
  
  dbDisconnect(dbcon)
  
  dat
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# exportAssayFit: Write the assay_fit table in toxminer  
#-------------------------------------------------------------------------------

exportAssayFit <- function(ae) {
  
  if (length(ae) > 1) stop("ae must be of length 1.")
  
  stime <- Sys.time()
  
  dat <- loadLvl5Data("aeid", ae)
  if (ncol(dat) == 0) return(FALSE)
  setkey(dat, l4id)
  
  qformat <- 
    "
    SELECT
      l4id,
      spid
    FROM 
      agg_level4,
      level0
    WHERE
      agg_level4.l0id = level0.l0id
      AND
      aeid IN (%s);
    "
  
  sp_dat <- tmquery(sprintf(qformat, ae), "invitrodb")  
  setkey(sp_dat, spid)
  
  chem <- loadChemInfo(field = "spid", sp_dat[ , unique(spid)])
  setkey(chem, spid)
  
  sp_dat <- chem[sp_dat]  
  sp_dat <- sp_dat[ , 
                   list(spid = paste0(unique(spid), collapse = ";")), 
                   by = list(l4id, chid, casn, chnm)]
  setkey(sp_dat, l4id)
  
  dat <- sp_dat[dat]
  dat[is.na(chid), chid := spid]
  dat[ , resp_cutoff := loadL5Mthds(ae)[ , coff]]
  pars <- c("AC50", "T", "W", "AC50_loss")
  dat[ , pars := NA_real_, with = FALSE]
  dat[modl == "hill", 
      pars := list(hill_ga, hill_tp, hill_gw, NA_real_), 
      with = FALSE]
  dat[modl == "gnls", 
      pars := list(gnls_ga, gnls_tp, gnls_gw, gnls_la),
      with = FALSE]
  dat[ , AC50 := 10^AC50]
  dat[is.na(AC50), AC50 := 1e6]
  dat[fitc %in% c(36, 40, 45, 49), AC50_mod := "<="]
  dat[fitc %in% c(37, 41, 46, 50), AC50_mod := "=="]
  dat[fitc %in% c(38, 42, 47, 51), AC50_mod := "=="]
  
  setkeyv(dat, c("chid", "AC50"))
  min_AC50 <- dat[ , list(ind = .I[1]), by = chid]
  dat <- dat[min_AC50$ind]
  dat[ , source_name_aid := loadAeidName(ae)[ , aenm]]
  
  dat <- dat[!is.na(casn), list(l4id,
                                assay_id = aeid,
                                source_name_aid,
                                dsstox_gsid = chid,
                                casrn = casn,
                                sample_id_string = spid,
                                model = modl,
                                AC50,
                                T,
                                W,
                                AC50_loss,
                                Emax = max_mean,
                                min_conc = logc_min,
                                max_conc = logc_max,
                                resp_cutoff,
                                bmad,
                                level5_hitcall = hitc)]
  
  appendToTbl(dat = dat, tbl = "assay_fit", db = "toxminer_v21")
  
  ttime <- round(difftime(Sys.time(), stime, units = "sec"), 2)
  ttime <- paste(unclass(ttime), units(ttime))
  
  cat("Processed ", ae, " (", ttime, ")\n", sep = "")
  
  TRUE
  
}

#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# createVarMat: Create a variable matrix from toxminer assay_fit  
#-------------------------------------------------------------------------------

createVarMat <- function(var, pdir = NULL) {
  
  qformat <- 
    "
    SELECT
      casrn,
      source_name_aid,
      %s
    FROM
      assay_fit;
    "
  
  dat <- tmquery(sprintf(qformat, var), "toxminer_v21")
  dat <- dat[!is.na(casrn)]
  dat[ , casrn := paste0("C", gsub("-", "", casrn))]
  dat[AC50 < 1e6, AC50 := 10^AC50]
  
  mat <- dcast.data.table(dat, casrn ~ source_name_aid, value.var = "AC50")
  setnames(mat, "casrn", "CODE")
  
  if (!is.null(pdir)) {
    
    fdate <- format(Sys.Date(), "%y%m%d.csv")
    write.csv(mat,
              file.path(pdir,
                        paste("AllResults", var, "Matrix", fdate, sep = "_")),
              row.names = FALSE)
    
  }
  
  mat
    
}

























