CREATE OR REPLACE PACKAGE Sim2_Import_Result_Pkg IS
/*
SIM200

Modified 7/17/02 by SF - fixed activity checking on first activity.
Modified 7/15/02 by SF - initilized trawl type so it is always zero unless new activity.
Modified 3/25/02 by SF - checked all the things that don't apply to HA and delt with them appropriately
Modified 3/20/02 by SF - more HA stuff
Modified 2/25/02 by SF to add checks for Habitat Assessment business rules.
*/

  -- Author  : KHENAS
  -- Created : 11/15/00 11:35:05 AM
  -- Purpose : import samples and results data into SIM tables from a text file

  /* KMS

  Package that imports activity and result records into SIM2 from a text file.
  Validates items within the text file to ensure that they are STORET friendly -
  otherwise an error message is issued and the data will be prevented from being
  exported to STORET without further manipulation (either through the SIM2 interface
  or by reloading).

  Before the package is run, the form (IMPF2_D) reads the text file into the
  SIM_IMP_LINES table so that each line is now stored as a separate record.  In
  addition, the import log for the import is created in the form.

  The goal in coding this package was to make it as modularized as possible.  Thus,
  most of the work is done in separate functions and procedures.  However, more than
  half of the package consists of the main import procedure.  This procedure gets basic,
  core information needed to load the data into SIM2 and to run the rest of the
  functions and procedures in the package.  Such information includes the is_number for
  the organization that is running the import, the delimiter used in the text file, the
  number of columns included in the import, the defaults for these columns, and the actual
  positions of the columns within the text file.  The actual positions of the columns are
  stored in table indexed by the default position of that columns.  Thus, for results,
  this table contains 57 entries which are either null (in the case of columns that have
  not been included in the import) or the actual position of the column in the line.

  Once this core information has been retrieved, the procedure begins a loop through the
  lines that have been imported into SIM_IMP_LINES.  Line specific variables are then
  adjusted (i.e. incremented in the case of line number), and the line checked to insure
  that it contains the correct number of items (columns).  The line is then parsed into
  separate items to be loaded, these items are stored in v_line_table, which is indexed
  by the actual position of the item within the text line.

  The procedure then loops through the columns (items) in the line.  It sets up variables
  that will be used later to determine what the item is and what should be done with it.
  This process is performed in a large if..then statement.  All blocks of this code are
  only entered if the item name is correct for that particular block.  While it is easy to
  forget that this part of the code is an if..then statement and to think instead that all
  lines of code are accessed in the order they appear in the procedure, it is important to
  remember that this code is processed is according to the processing order worked out in
  the sim_imp_cols table.

  Each block of code in the if..then statement deals with a particular item/column.  Within
  these blocks, the data that was brought in from the text file is prepared for loading into
  SIM2 and check against the requirements of STORET.  In many cases, this involves validating
  the item against the SIM2 allowable values table (created for items whose values have been
  limited by the STORET interface rather than a STORET table) or validated against values that
  are stored in STORET tables.  In both of these validation procedures, the translation table
  is checked before the items are validated.  In addition to these validations, items are
  validated against the data that has already been loaded from the line being imported.  This
  includes checks to insure that the activity being imported occured between the trip start
  and end dates as well as similar integrity concerns.

  Following the if..then statement, any other integrity checks/validations that have not yet
  been taken care of are performed.  These includes making sure that results are valid for the
  characteristic chosen for it.

  The activities and results are created by assigning the values that were obtained earlier in
  the program to records containing items for every field in the SIM tables for activities and
  results.  These records are then passed to functions that insert the data into the tables.

  Once the procedure has looped through all of the lines, it counts the number that were approved
  for export to STORET and updates all tables that are related to this.

  KMS */


   TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;

   TYPE t_rci_pk_type IS RECORD (
     is_number         NUMBER(8),
     org_id            VARCHAR2(8),
     indicator         NUMBER(1)
     );

   TYPE t_stvst_id_rec IS RECORD (
     trip_seq            NUMBER(10),
     station_is_nbr      NUMBER(8),
     visit_id            NUMBER(3)
     );
     
   TYPE t_stvst_id_type IS TABLE OF t_stvst_id_rec INDEX BY BINARY_INTEGER;

   TYPE t_stvst_id_key_type IS TABLE OF NUMBER(19) INDEX BY BINARY_INTEGER;

  -- Public function and procedure declarations

   FUNCTION get_date(p_column_name IN VARCHAR2,p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,
                     p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER, p_sid_seq IN NUMBER) RETURN DATE;

   FUNCTION get_time(p_column_name IN VARCHAR2,p_raw_time IN VARCHAR2,p_col_seq IN NUMBER,
                     p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER) RETURN DATE;

   PROCEDURE import_results (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                          p_imp_cfg_seq IN NUMBER,p_test_num OUT NUMBER);


   FUNCTION  validate_class_descriptor(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_char_is_nbr IN NUMBER,
          p_item_name IN VARCHAR2,p_col_seq IN NUMBER,p_fa_seq IN NUMBER, p_line_cnt IN NUMBER,
          p_import_log_seq IN NUMBER) RETURN VARCHAR2;

   FUNCTION create_station (p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER;

   FUNCTION create_well(p_well_rec IN SIM_STATION_WELLS%ROWTYPE) RETURN NUMBER;

   FUNCTION create_station_location (p_station_location_rec IN sim_station_locations%ROWTYPE) RETURN NUMBER;

END Sim2_Import_Result_Pkg;
/
CREATE OR REPLACE PACKAGE BODY Sim2_Import_Result_Pkg IS


   TYPE t_acode_defs_rec IS RECORD (
     char_pk             Sim_import_result_extras.t_storet_pk_type,
     anlpr_pk            Sim_import_result_extras.t_storet_pk_type,
     labprep_pk          Sim_import_result_extras.t_storet_pk_type,
     uom_pk              Sim_import_result_extras.t_storet_pk_type,
     lab_pk              Sim_import_result_extras.t_storet_pk_type,
     fraction_is         NUMBER(12),
     voltinism_is        NUMBER(12),
     habit_is            NUMBER(12),
     value_type          VARCHAR2(10),
     stat_type           VARCHAR2(18),
     dur_basis           VARCHAR2(10),
     temp_basis          VARCHAR2(8),
     weight_basis        VARCHAR2(12),
     pollution_tol       VARCHAR2(4),
     funct_feed_grp      VARCHAR2(6),
     trophic_level       VARCHAR2(4),
     species             VARCHAR2(8),
     particle_size_basis VARCHAR2(40)  -- jah 02-20-04  Added Particle Size Basis
     );


   v_line_cnt           NUMBER(10) := 0;
   v_activity_count     t_activity_count;
   v_acode_defs_rec     t_acode_defs_rec;
   v_acode_init_rec     t_acode_defs_rec;
   v_org_is_nbr         NUMBER(12);
   v_import_log_seq     NUMBER(10);
   v_trip_qc            NUMBER(1) := 0;
   v_cur_trip_is_nbr    NUMBER(12) := NULL;
   gv_last_visit_id     NUMBER(5) := 0;
   v_char_result_type   VARCHAR2(4) := NULL;
   v_sid_seq            NUMBER(10) := NULL;


/******************************************************************************************************/
FUNCTION get_date(p_column_name IN VARCHAR2,p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,
                  p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER, p_sid_seq IN NUMBER) RETURN DATE IS

  v_start_date                  DATE;
  v_fmt                         VARCHAR2(80);

  e_no_cfg_found                EXCEPTION;

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;

BEGIN
  /* kms - function that uses the value passed in from the text file to get a date based
     on the date format selected in the import configuration. */

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      /* no format specified, use default */
      IF p_sid_seq = 10007 THEN
        v_fmt := 'RRRR-MM-DD';
      ELSE
        v_fmt := 'MM-DD-RRRR';
      END IF;
    END IF;
  CLOSE c_get_date_fmt;

  IF v_fmt IS NOT NULL THEN
    BEGIN
      v_start_date := TRUNC(TO_DATE(p_raw_date,v_fmt));
    EXCEPTION
      WHEN OTHERS THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_column_name||', data does not match the format chosen in the import configuration.', p_fa_seq);
        v_start_date := NULL;
    END;
  END IF;

  RETURN v_start_date;

EXCEPTION

  WHEN e_no_cfg_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Import configuration for start, end, or analysis date is incomplete or does not exist.', p_fa_seq);
    RETURN NULL;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get date, column '||p_column_name||' '||p_raw_date,p_fa_seq);
    RETURN NULL;

END get_date;

/******************************************************************************************************/
FUNCTION get_time(
         p_column_name IN VARCHAR2,
         p_raw_time IN VARCHAR2,
         p_col_seq IN NUMBER,
         p_cfg_seq IN NUMBER,
         p_fa_seq IN NUMBER) 
         RETURN DATE IS

  v_start_time                  DATE;
  v_fmt                         VARCHAR2(80);
  v_raw_time                    VARCHAR2(12);

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;

BEGIN
  /* kms - function that converts the value in the text file to a time based on the the format
     selected in the import configuration */

  --SF 11/4/02 Check for lack of leading zeros
  IF SUBSTR(p_raw_time,2,1) = ':' THEN
    v_raw_time := '0'||p_raw_time;
  ELSE
    v_raw_time := p_raw_time;
  END IF;

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      /* no format specified, use default */
      v_fmt := 'HH24:MI:SS';
    END IF;
  CLOSE c_get_date_fmt;

  BEGIN
    --now support HHMM for STORET V2.  In STORET, this format is specified and means 'HH24MI' in Oracle speak.
    IF v_fmt = 'HHMM' THEN
      v_fmt := 'HH24MI';  
    END IF;
    v_start_time := TO_DATE('01/01/0001 '||SUBSTR(v_raw_time,1,8),'MM/DD/RRRR '||v_fmt);
  EXCEPTION
    WHEN OTHERS THEN
      v_start_time := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_column_name||', data does not match the format chosen in the import configuration.', p_fa_seq);
      RETURN v_start_time;
  END;

  RETURN v_start_time;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get time '||p_raw_time||', column '||p_column_name,p_fa_seq);
    RETURN NULL;

END get_time;
/******************************************************************************************************/
FUNCTION GET_VALUE_MSR(P_RESULT_TEXT    IN VARCHAR2,
                         P_RESULT_OK      OUT BOOLEAN,
                         P_FA_SEQ         IN NUMBER,
                         V_LINE_CNT       IN NUMBER,
                         P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2 IS

  v_result_num      NUMBER;  
  v_result          VARCHAR2(50);  

  BEGIN
    BEGIN
      v_result := p_result_text;
      p_result_ok := TRUE;
      /* Check if the result is a number and round to 5 places */
      v_result_num := ROUND(TO_NUMBER(v_result),5);
    
      IF v_result_num > 99999999.99999 THEN
        /* Check if number is too large for STORET */
        p_result_ok := FALSE;
        v_result := '* TO LARGE';
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                    'Line ' || V_LINE_CNT ||
                                    ': Result is too large for STORET: ' ||
                                    p_result_text,
                                    P_FA_SEQ);  
      ELSIF v_result_num <> p_result_text THEN
      /* if result tail is longer than 5 decimal places, warn user that it will be rounded */
        p_result_ok := TRUE; 
        v_result := v_result_num;    
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                       'Warning: Line ' || V_LINE_CNT ||
                                        ': Result, ' || p_result_text ||
                                        ' will be rounded to 5 places.',
                                        P_FA_SEQ);   
     END IF;                                                                                
                                            
    EXCEPTION
      WHEN OTHERS THEN
        /* If the result is not a number then fail*/
        p_result_ok := FALSE;
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                    'Line '|| V_LINE_CNT ||
                                    ': Result, ' || P_RESULT_TEXT ||
                                    ', is not a number.',
                                    P_FA_SEQ);  
         /*Returns the failed result, gives error and trims the result to fit into SIM*/                           
        v_result := SUBSTR(p_result_text,1,15);                             
    END;
                     
  RETURN(v_result);
  
  EXCEPTION
    WHEN OTHERS THEN
      p_result_ok := FALSE;    
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get value measure.',
                                  P_FA_SEQ);
      RETURN NULL;                                                                              
END GET_VALUE_MSR;                                            

/******************************************************************************************************/
PROCEDURE get_or_add_trip_rec(
          p_trip_rec        IN OUT SIM_TRIPS%ROWTYPE,
          p_org_id          IN VARCHAR2,
          p_fa_seq          IN NUMBER,
          p_trip_start_date IN DATE) IS

  v_trip_seq         NUMBER(10);
  v_trip_start_date  DATE;
  v_trip_end_date    DATE;
  v_success          NUMBER(1);

  e_no_start_date    EXCEPTION;

  -- Retrieve Trip from SIM
  CURSOR c_get_trip_seq(
    p_trip_id IN VARCHAR2, 
    p_org_id  IN VARCHAR2) IS
  SELECT STP_SEQ, STP_TSRTRIP_IS_NUMBER, STP_TSRTRIP_ORG_ID 
  FROM   SIM_TRIPS
  WHERE  STP_TRIP_ID = p_trip_id 
  AND    STP_TSMORGAN_ORG_ID = RPAD(p_org_id,8);

  -- Retrieve the trip from STORET
  CURSOR c_get_trip_key(
    p_trip_id IN VARCHAR2, 
    p_org_id IN VARCHAR2) IS
  SELECT TSRTRIP_IS_NUMBER, TSRTRIP_ORG_ID, START_DATE, START_TIME 
  FROM   TSRTRIP
  WHERE  ID_CODE = RPAD(p_trip_id,15) 
  AND    TSRTRIP_ORG_ID = RPAD(p_org_id,8);

  -- Retrieve the next Trip Sequence
  CURSOR c_get_next_trip_seq IS 
  SELECT SIM_TRIPS_SEQ.NEXTVAL FROM DUAL;

  /* This function looks for the Trip in the SIM table.
     If it is not there, then it looks in STORET.
     
     If the Trip ID is in STORET, this function returns 
     the IS_NUMBER and ORG_ID to the trip to the calling 
     routine.

     If the Trip ID is not in SIM, this function inserts 
     the trip into SIM_TRIPS. */

BEGIN

  -- Try to retrieve the Trip from SIM
  OPEN c_get_trip_seq(p_trip_rec.STP_TRIP_ID,p_org_id);
  FETCH c_get_trip_seq INTO 
    v_trip_seq, 
    p_trip_rec.STP_TSRTRIP_IS_NUMBER, 
    p_trip_rec.STP_TSRTRIP_ORG_ID;

  -- did we find the Trip in SIM?
  IF c_get_trip_seq%NOTFOUND THEN
    -- if no

    -- Indicate the Trip is NOT in SIM
    v_trip_seq := NULL;
    
    -- Put default values in the IS_NUMBER and ORG_ID in case Trip is not in STORET.
    p_trip_rec.STP_TSMORGAN_IS_NUMBER := v_org_is_nbr;  -- v_org_is_number is a global variable
    p_trip_rec.STP_TSMORGAN_ORG_ID := p_org_id;

    -- Try to retrieve the Trip from STORET
    OPEN c_get_trip_key(p_trip_rec.STP_TRIP_ID, p_org_id);
    FETCH c_get_trip_key INTO 
      p_trip_rec.STP_TSRTRIP_IS_NUMBER,
      p_trip_rec.STP_TSRTRIP_ORG_ID,
      v_trip_start_date,
      v_trip_end_date;

    -- did we find the Trip ID in STORET?
    IF c_get_trip_key%NOTFOUND THEN
      -- if no
      
      -- tell calling routine that SIM is setting the Trip start date
      p_trip_rec.STP_CREATED_BY_SIM := 1;

      -- does the Trip record already contain the Trip start date?
      IF p_trip_rec.STP_START_DATE IS NULL THEN
        -- if no
        
        -- did the calling routine pass the Trip start date?
        IF p_trip_start_date IS NOT NULL THEN
          -- if yes
          
          -- return the Trip start date back to the calling routine
          p_trip_rec.STP_START_DATE := p_trip_start_date;
          
        ELSE
          -- if no (i.e., the calling routine did not pass a Trip start date)
          
          RAISE e_no_start_date;
 
        END IF;

      END IF;

    ELSE
      -- if yes (i.e., we found the Trip ID in STORET)
      
      -- tell calling routine that SIM is not setting the Trip start date
      p_trip_rec.STP_CREATED_BY_SIM := 0;
      
      -- return the the Trip start and end dates to the calling routine
      p_trip_rec.STP_START_DATE := v_trip_start_date;
      p_trip_rec.STP_END_DATE := v_trip_end_date;

    END IF;
    CLOSE c_get_trip_key;

  END IF;
  CLOSE c_get_trip_seq;

  -- Is the Trip in SIM?
  IF v_trip_seq IS NULL THEN
    -- if no
    
    -- retrieve the next Trip sequence
    OPEN c_get_next_trip_seq;
    FETCH c_get_next_trip_seq INTO p_trip_rec.STP_SEQ;
    CLOSE c_get_next_trip_seq;

    -- Insert the Trip into SIM_TRIPS */
    v_success := sim_import_result_extras.create_sim_trip(
      p_trip_rec,  /* jah 3-5-2004 
                      Changed this parameter from IN to IN OUT 
                      so STP_SEQ passes back to calling routine. */
      p_fa_seq,
      v_import_log_seq
    );
    
  ELSE
    -- if yes (i.e., the Trip is in SIM)
    
    -- return the trip sequence to the calling routine    
    p_trip_rec.STP_SEQ := v_trip_seq;

  END IF;

EXCEPTION

  WHEN e_no_start_date THEN
    Sim2_Import_Pkg.WRITE_ERROR(
      v_import_log_seq , 
      'Line ' || v_line_cnt || ': ' ||
      'Could not find Trip ID in SIM or STORET.  ' ||
      'Could not create Trip ID because Trip Start Date is not in the import record.',
       p_fa_seq
    );
    p_trip_rec.STP_TSRTRIP_ORG_ID := NULL;
    p_trip_rec.STP_TSRTRIP_IS_NUMBER := NULL;
    p_trip_rec.STP_SEQ := NULL;
  
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(
      v_import_log_seq , 
      'Line ' || v_line_cnt || ': ' ||
      SQLERRM || ' - in get or add trip rec.',
      p_fa_seq
    );
    p_trip_rec.STP_TSRTRIP_ORG_ID := NULL;
    p_trip_rec.STP_TSRTRIP_IS_NUMBER := NULL;
    p_trip_rec.STP_SEQ := NULL;

END get_or_add_trip_rec;

/******************************************************************************************************/
PROCEDURE load_station_visits(
          p_trip_seq IN NUMBER, 
          p_trip_key IN Sim_import_result_extras.t_storet_pk_type, 
          p_org_id IN VARCHAR2,
          p_stvst_id_table IN OUT t_stvst_id_type, 
          p_fa_seq IN NUMBER) IS

  v_id_nbr             NUMBER(3);
  v_max_nbr            NUMBER(3) := 0;
  v_trip_sta_idx       NUMBER(19);
  v_visit_seq          NUMBER(10);
  v_visit_rec          SIM_STATION_VISITS%ROWTYPE;
  v_last_statn_is_nbr  NUMBER(12);
  v_success            NUMBER(1);

  -- Retrieve station visits from STORET
  CURSOR c_get_trip_station_visits(
    p_trip_is_nbr IN NUMBER, 
    p_trip_org_id IN VARCHAR2) IS
  SELECT ID_NUMBER vst_id, ARRIVAL_DATE, ARRIVAL_TIME, DEPARTURE_DATE, DEPARTURE_TIME,
    TSRSTVST_IS_NUMBER is_nbr, TSRSTVST_ORG_ID org_id, TSMSTATN_IS_NUMBER statn_is_nbr,
    TSMSTATN_ORG_ID statn_org_id, BLOB_TITLE 
  FROM   TSRSTVST
  WHERE  TSRTRIP_IS_NUMBER = p_trip_is_nbr 
  AND    TSRTRIP_ORG_ID    = RPAD(p_trip_org_id,8)
  ORDER BY TSMSTATN_IS_NUMBER;

  -- Retrieve station visits from SIM
  CURSOR c_check_visit(
    p_visit_is_nbr IN NUMBER, 
    p_visit_org_id IN VARCHAR2) IS
  SELECT SSV_SEQ
  FROM   SIM_STATION_VISITS 
  WHERE  SSV_TSRSTVST_IS_NUMBER = p_visit_is_nbr
  AND    SSV_TSRSTVST_ORG_ID    = RPAD(p_visit_org_id,8);

  -- Retrieve next Station Visit Sequence
  CURSOR c_get_next_visit_seq IS
  SELECT SIM_STATION_VISITS_SEQ.NEXTVAL FROM DUAL;

  /* kms (6/8/01) - get all the station visits that exist on this trip in STORET and load
     them into the SIM table if they are not already there.  Update visits that are already
     in SIM with what's in STORET. */

  /* kms - loop through all station visits on the current trip */

BEGIN

  -- loop thru the Station Visits in STORET
  FOR v_visit IN c_get_trip_station_visits(p_trip_key.is_number, p_trip_key.org_id) LOOP

    -- try to find the station visit in SIM
    OPEN c_check_visit(v_visit.is_nbr, v_visit.org_id);
    FETCH c_check_visit INTO v_visit_seq;
    
    -- is the station visit in SIM?
    IF c_check_visit%NOTFOUND THEN
      -- if no

      -- get sequence for new record
      OPEN c_get_next_visit_seq;
      FETCH c_get_next_visit_seq INTO v_visit_seq;
      CLOSE c_get_next_visit_seq;

      -- create record to insert station visit into sim
      v_visit_rec.SSV_SEQ                := v_visit_seq;
      v_visit_rec.SSV_STP_SEQ            := p_trip_seq;
      v_visit_rec.SSV_ID                 := RTRIM(v_visit.vst_id); -- jah 3-10-2004 Removed trailing spaces
      v_visit_rec.SSV_START_DATE         := v_visit.arrival_date;
      v_visit_rec.SSV_START_TIME         := v_visit.arrival_time;
      v_visit_rec.SSV_END_DATE           := v_visit.departure_date;
      v_visit_rec.SSV_END_TIME           := v_visit.departure_time;
      v_visit_rec.SSV_TSRSTVST_IS_NUMBER := v_visit.is_nbr;
      v_visit_rec.SSV_TSRSTVST_ORG_ID    := v_visit.org_id;
      v_visit_rec.SSV_TSMSTATN_IS_NUMBER := v_visit.statn_is_nbr;
      v_visit_rec.SSV_TSMSTATN_ORG_ID    := v_visit.statn_org_id;
      v_visit_rec.SSV_TSMORGAN_IS_NUMBER := v_org_is_nbr;
      v_visit_rec.SSV_TSMORGAN_ORG_ID    := p_org_id;
      v_visit_rec.SSV_CREATED_BY_SIM     := 0;
      v_visit_rec.SSV_IMPORT_SEQ         := v_import_log_seq;
      v_visit_rec.SSV_DOCUMENT           := v_visit.blob_title;

      -- insert station visit into sim
      v_success := sim_import_result_extras.create_sim_station_visit(
        v_visit_rec,
        v_import_log_seq
      );
      
    ELSE
      -- if yes (i.e., the station visit is in SIM)

      -- update the station visit in SIM with the values from STORET.
      UPDATE SIM_STATION_VISITS
      SET    SSV_ID = RTRIM(v_visit.vst_id), -- jah 3-10-2004 Removed trailing spaces
             SSV_START_DATE = v_visit.arrival_date,
             SSV_START_TIME = v_visit.arrival_time,
             SSV_END_DATE = v_visit.departure_date,
             SSV_END_TIME = v_visit.departure_time
      WHERE  SSV_SEQ = v_visit_seq;

    END IF;
    CLOSE c_check_visit;

  END LOOP;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (
      v_import_log_seq , 
      'Line ' || v_line_cnt || ': '
      || SQLERRM || 
      ' - in load_station_visits.',
      p_fa_seq
    );

END load_station_visits;

/******************************************************************************************************/
PROCEDURE get_next_stavst_id (p_stavst_id OUT VARCHAR2) IS

  /* This function returns the next available station visit number.
     Notice that the station visit number begins with 'UNA'.
     UNA is short for UNAssigned. This tells the migration code to
     determines the actual station visit number. */
  
BEGIN

  gv_last_visit_id := gv_last_visit_id + 1;
  p_stavst_id := 'UNA'||TO_CHAR(gv_last_visit_id);

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(
      v_import_log_seq, 
      'Line ' || v_line_cnt || ': '
      || SQLERRM ||
      ' - in get next visit id.',
      NULL
    );

END get_next_stavst_id;

/******************************************************************************************************/
FUNCTION check_existing_visit (
         p_instr_choice      IN VARCHAR2,
         p_act_start_date    IN DATE,
         p_act_start_time    IN DATE,
         p_trip_seq          IN NUMBER,
         p_station_key       IN Sim_import_result_extras.t_storet_pk_type, 
         p_visit_start_date  OUT DATE,
         p_visit_start_time  OUT DATE, 
         p_visit_end_date    OUT DATE,
         p_visit_end_time    OUT DATE, 
         p_visit_key         OUT Sim_import_result_extras.t_storet_pk_type,
         p_visit_seq         IN OUT NUMBER, 
         p_visit_id_tab      IN OUT t_stvst_id_type,
         p_visit_id          OUT VARCHAR2)
         RETURN NUMBER IS

  -- Retrieve station visit record by station, trip, and date
  CURSOR c_find_station_visit(
         cp_stat_is_nbr IN NUMBER, 
         cp_stat_org_id IN VARCHAR2,
         cp_trip_seq    IN NUMBER, 
         cp_date        IN DATE) IS
  SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID, SSV_ID
  FROM   SIM_STATION_VISITS
  WHERE  TRUNC(SSV_START_DATE,'DD') = TRUNC(cp_date,'DD')
  AND    SSV_TSMSTATN_IS_NUMBER  =  cp_stat_is_nbr
  AND    SSV_TSMSTATN_ORG_ID     = RPAD(cp_stat_org_id,8) 
  AND    SSV_STP_SEQ             = cp_trip_seq;

  -- Retrieve station visit record by station, trip, date, and time
  CURSOR c_find_station_visit_dt(
         p_stat_is_nbr IN NUMBER, 
         p_stat_org_id IN VARCHAR2,
         p_trip_seq    IN NUMBER, 
         p_date        IN DATE, 
         p_time        IN DATE) IS
  SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID, SSV_ID
  FROM   SIM_STATION_VISITS
  WHERE  NVL(TO_CHAR(SSV_START_TIME,'HH24:MI:SS'),'x') = NVL(TO_CHAR(p_time,'HH24:MI:SS'),'x')
  AND    TRUNC(SSV_START_DATE,'DD') = TRUNC(p_date,'DD')
  AND    SSV_TSMSTATN_IS_NUMBER =  p_stat_is_nbr
  AND    SSV_TSMSTATN_ORG_ID = RPAD(p_stat_org_id,8) 
  AND    SSV_STP_SEQ = p_trip_seq;

  /* Call this function when you need to find a station visit record in SIM. */

BEGIN

  -- Is the import instruction "Create one station visit per day"?
  IF p_instr_choice = 'RES-STVST-01' THEN
    -- if yes

    -- try to retrieve a station visit for this Station, Trip, and Date
    OPEN c_find_station_visit(
      p_station_key.is_number, 
      p_station_key.org_id,
      p_trip_seq, 
      p_act_start_date
    );
    FETCH c_find_station_visit INTO 
      p_visit_seq, 
      p_visit_key.is_number, 
      p_visit_key.org_id,
      p_visit_id;  -- jah 3-26-04
    
    -- Does a station visit exist for this Station, Trip, and Date?
    IF c_find_station_visit%NOTFOUND THEN
      -- if no
      
      -- tell subsequent code in this function to determine the next station visit number
      -- and also tell the calling routine the station visit record is not in SIM
      p_visit_seq := NULL;

      -- tell the calling routine the station visit record is not in STORET
      p_visit_key.is_number := NULL;
      p_visit_key.org_id := NULL;
      p_visit_id := NULL;  -- jah 3-26-04

    END IF;
    CLOSE c_find_station_visit;

    -- return station visit start and end parameters to the calling routine
    p_visit_start_date := p_act_start_date;
    p_visit_end_date := NULL;
    p_visit_start_time := NULL;
    p_visit_end_time := NULL;

  -- Is the instruction "Create one station visit per date and time"?  
  ELSIF p_instr_choice = 'RES-STVST-03' THEN
    -- if yes

    -- try to retrieve a station visit for this Station, Trip, Date, and Time
    OPEN c_find_station_visit_dt (
      p_station_key.is_number, 
      p_station_key.org_id,
      p_trip_seq, 
      p_act_start_date, 
      NVL(p_act_start_time,TO_DATE('00:00','HH24:MI')) -- jah 3-9-2004 Changed from '00:01' to '00:00'
    );
    FETCH c_find_station_visit_dt INTO 
      p_visit_seq, 
      p_visit_key.is_number, 
      p_visit_key.org_id,
      p_visit_id;  -- jah 3-26-04

    -- Does a station visit exist for this Station, Trip, Date, and Time?
    IF c_find_station_visit_dt%NOTFOUND THEN
      -- if no
      
      -- tell subsequent code in this function to determine the next station visit number
      -- and also tell the calling routine the station visit record is not in SIM
      p_visit_seq := NULL;
      
      -- tell the calling routine the station visit record is not in STORET
      p_visit_key.is_number := NULL;
      p_visit_key.org_id := NULL;
      p_visit_id := NULL;  -- jah 3-26-04

    END IF;
    CLOSE c_find_station_visit_dt;
    
    -- return station visit start and end parameters to the calling routine
    p_visit_start_date := p_act_start_date;
    p_visit_end_date := NULL;
    p_visit_end_time := NULL;
    p_visit_start_time := NVL(p_act_start_time,TO_DATE('00:00','HH24:MI')); -- jah 3-9-2004 Changed from '00:01' to '00:00'
    
    IF (p_act_start_time IS NULL) THEN
      Sim2_Import_Pkg.write_error(
        v_import_log_seq, 
        'Warning: Line ' || v_line_cnt || 
        ': Missing Activity Start Time is equivalent to Midnight.',
        NULL
      );
    END IF;

  END IF;

  -- Did we find a matching station visit record?
  IF p_visit_seq IS NULL THEN
    -- if no
    
    -- determine the next station visit number and return it to the calling routine
    get_next_stavst_id(p_visit_id);

  END IF;
  
  -- Tell calling routine NO exceptions
  RETURN NULL;

EXCEPTION

   WHEN OTHERS THEN
   
      Sim2_Import_Pkg.WRITE_ERROR(
        v_import_log_seq, 
        'Line ' || v_line_cnt || ': ' || SQLERRM ||
        ' - in check existing visit.',NULL
      );
      
      -- Tell calling routine OTHERS exception
      RETURN 1;

END check_existing_visit;

/***************************************************************************************/
FUNCTION add_project_to_activity(p_fa_seq IN NUMBER, p_project_is_nbr IN NUMBER,
                                   p_project_org IN VARCHAR2) RETURN NUMBER IS

  v_act_project_seq                  NUMBER(10);

  CURSOR c_get_act_project_seq IS
    SELECT SIM_ACTIVITY_PROJECTS_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms - function that associates projects with activities in SIM */

  /* get next sequence */
  OPEN c_get_act_project_seq;
  FETCH c_get_act_project_seq INTO v_act_project_seq;
    IF c_get_act_project_seq%NOTFOUND THEN
      v_act_project_seq := NULL;
    END IF;
  CLOSE c_get_act_project_seq;

  INSERT INTO SIM_ACTIVITY_PROJECTS (
    SAPJ_SEQ,
    SAPJ_FA_SEQ,
    SAPJ_TSMPROJ_IS_NUMBER,
    SAPJ_TSMPROJ_ORG_ID
    )
  VALUES (
    v_act_project_seq,               /* SAPJ_SEQ */
    p_fa_seq,                        /* SAPJ_FA_SEQ */
    p_project_is_nbr,                /* SAPJ_TSMPROJ_IS_NUMBER */
    p_project_org                    /* SAPJ_TSMPROJ_ORG_ID */
    );

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in add project to activity.',p_fa_seq);
    RETURN 0;

END add_project_to_activity;


/******************************************************************************************************/
PROCEDURE remove_lab_remark_dups(p_remark_codes_table IN OUT Sim2_Import_Pkg.col_table_type, p_no_remark_codes IN OUT NUMBER) IS
  -- DTW 10/21/03 - If duplicate lab remark codes are imported, remove them with a warning
  v_remark_codes_temp     Sim2_Import_Pkg.col_table_type;
  v_no_remark_codes_temp  NUMBER := 0;
  v_dup_found             BOOLEAN;
BEGIN
  FOR v_cntr IN 1..p_no_remark_codes LOOP
    v_dup_found := FALSE;
    FOR v_inner_cntr IN 1..(v_cntr-1) LOOP
      IF p_remark_codes_table(v_cntr) = p_remark_codes_table(v_inner_cntr) THEN
        v_dup_found := TRUE;
      END IF;
    END LOOP;
    IF v_dup_found = TRUE THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Duplicate Lab Remark Code Will be Ignored.',NULL);
    ELSE
      v_no_remark_codes_temp := v_no_remark_codes_temp + 1;
      v_remark_codes_temp(v_no_remark_codes_temp) := p_remark_codes_table(v_cntr);
    END IF;
  END LOOP;

  -- Return remark code table and count with duplicates removed
  p_remark_codes_table := v_remark_codes_temp;
  p_no_remark_codes := v_no_remark_codes_temp;

END remove_lab_remark_dups;


/******************************************************************************************************/
FUNCTION add_remark_to_results(p_rs_seq IN NUMBER, p_remark_is_nbr IN NUMBER,
                               p_remark_org_id IN VARCHAR2)
                               RETURN NUMBER IS
                               
                               
BEGIN

  INSERT INTO sim_result_lab_remarks
    (srlr_seq,srlr_rs_seq,srlr_tsrlbrmk_is_number,srlr_tsrlbrmk_org_id)
   VALUES
    (sim_srlr_seq.NEXTVAL,p_rs_seq,p_remark_is_nbr,p_remark_org_id);
    
  COMMIT;

  RETURN 1;
      
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in add_remark_to_results.',p_rs_seq);
    RETURN 0;
END add_remark_to_results;

/******************************************************************************************************/                               
FUNCTION add_personnel_to_activity(p_fa_seq IN NUMBER, p_persn_is_nbr IN NUMBER,
                                   p_persn_org IN VARCHAR2) RETURN NUMBER IS

  v_act_persn_seq                  NUMBER(10);

  CURSOR c_get_act_persn_seq IS
    SELECT SIM_ACTIVITY_PERSONNEL_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms - function that associates personnel with activities in SIM. */

  /* get next sequence */
  OPEN c_get_act_persn_seq;
  FETCH c_get_act_persn_seq INTO v_act_persn_seq;
  CLOSE c_get_act_persn_seq;

  INSERT INTO SIM_ACTIVITY_PERSONNEL (
    SAP_SEQ,
    SAP_FA_SEQ,
    SAP_TSMPERSN_IS_NUMBER,
    SAP_TSMPERSN_ORG_ID
    )
  VALUES (
    v_act_persn_seq,                 /* SAP_SEQ */
    p_fa_seq,                        /* SAP_FA_SEQ */
    p_persn_is_nbr,                  /* SAP_TSMPERSN_IS_NUMBER */
    p_persn_org                      /* SAP_TSMPERSN_ORG_ID */
    );

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in add personnel to activity.',p_fa_seq);
    RETURN 0;

END add_personnel_to_activity;

/******************************************************************************************************/                               
FUNCTION add_cporg_to_activity(p_fa_seq IN NUMBER, p_cporg_is_nbr IN NUMBER,
                                   p_cporg_org_id IN VARCHAR2) RETURN NUMBER IS

  v_sac_seq  NUMBER(10);

  CURSOR c_sim_activity_cporg_seq IS
    SELECT SIM_ACTIVITY_CPORG_SEQ.NEXTVAL FROM DUAL;

BEGIN

  -- get next sequence
  OPEN c_sim_activity_cporg_seq;
  FETCH c_sim_activity_cporg_seq INTO v_sac_seq;
  CLOSE c_sim_activity_cporg_seq;

  -- insert Cooperating Organization
  INSERT INTO SIM_ACTIVITY_CPORG
  (
    SAC_SEQ,
    SAC_FA_SEQ,
    SAC_TSMCPORG_IS_NUMBER,
    SAC_TSMCPORG_ORG_ID
  )
  VALUES
  (
    v_sac_seq,       /* SAC_SEQ */
    p_fa_seq,        /* SAC_FA_SEQ */
    p_cporg_is_nbr,  /* SAC_TSMCPORG_IS_NUMBER */
    p_cporg_org_id   /* SAC_TSMCPORG_ORG_ID */
  );
  COMMIT;

  -- Return success
  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    -- output error message
    Sim2_Import_Pkg.write_error(
      v_import_log_seq,
      'Line '||v_line_cnt||': '||SQLERRM||' - in add cooperating organization to activity.',
      p_fa_seq
    );

    -- return failure
    RETURN 0;

END add_cporg_to_activity;

/******************************************************************************************************/
FUNCTION get_char_result_type(p_char_is_nbr IN NUMBER) RETURN VARCHAR2 IS

v_char_result_type VARCHAR2(4);

  CURSOR c_get_char_type(p_char_is_nbr IN NUMBER) IS
    SELECT D_SCR_TYPE_CD FROM TSRCHAR
      WHERE TSRCHAR_IS_NUMBER = p_char_is_nbr
      AND TSRCHAR_ORG_ID = 'ZZZZZZZZ';

BEGIN
  /* kms - function that gets characteristic type in order to see if it has permitted values
     or if other checks need to be performed on the result. */

  OPEN c_get_char_type(p_char_is_nbr);
  FETCH c_get_char_type INTO v_char_result_type;
    IF c_get_char_type%NOTFOUND THEN
      v_char_result_type := NULL;
    END IF;
  CLOSE c_get_char_type;

RETURN v_char_result_type;

END get_char_result_type;

/******************************************************************************************************/
FUNCTION get_col_seq(p_col_name IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_col_seq NUMBER(10);

  CURSOR c_get_col_seq(cp_col_name IN VARCHAR2, cp_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_SEQ FROM SIM_IMP_COLS, SIM_IMP_CFG_DETS, SIM_IMP_CFG
    WHERE SIC_SEQ  = SICDT_SIC_SEQ
    AND   SICC_SEQ = SICDT_SICC_SEQ
    AND   SICC_SEQ = cp_imp_cfg_seq
    AND   SIC_NAME = cp_col_name;

BEGIN

  OPEN c_get_col_seq(p_col_name, p_imp_cfg_seq);
  FETCH c_get_col_seq INTO v_col_seq;
  CLOSE c_get_col_seq;

  RETURN v_col_seq;

  EXCEPTION WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - In get col seq.',p_fa_seq);
    RETURN NULL;

END get_col_seq;

/******************************************************************************************************/
FUNCTION get_char_defaults(
         p_analysis_seq IN NUMBER, 
         p_row_pk       IN Sim_import_result_extras.t_storet_pk_type,
         p_imp_cfg_seq  IN NUMBER,
         p_col_seq      IN NUMBER, 
         p_fa_seq       IN NUMBER)
         RETURN NUMBER
  IS

  v_dummy                  NUMBER(8) := NULL;
  v_uom_name               VARCHAR2(10);
  v_dur_basis              VARCHAR2(10);
  v_stat_type              VARCHAR2(18);
  v_value_type             VARCHAR2(10);
  v_temp_basis             VARCHAR2(8);
  v_col_seq                NUMBER(10);

  e_bad_uom                EXCEPTION;
  e_bad_def_map            EXCEPTION;
  e_bad_def_pk             EXCEPTION;

  CURSOR c_get_storet_row_defs(p_row_pk IN Sim_import_result_extras.t_storet_pk_type) IS
  SELECT DEF.TSMPRMVL_IS_NUMBER fraction_is,
         DEF.TSMPRMVL0IS_NUMBER voltinism_is,
         DEF.TSMPRMVL1IS_NUMBER habit_is,
         DEF.UOM_NAME uom_name,
         DEF.DUR_BASIS_TYPE_NM dur_basis,
         DEF.STATISTIC_TYPE_NM stat_type,
         DEF.VALUE_TYPE_NAME value_type,
         DEF.TEMP_BASIS_LVL_NM temp_basis,
         DEF.WT_BASIS_TYPE_NM wieght_basis,
         DEF.SPECIES_NUMBER species,
         DEF.TAXON_POLLUTION pollution_tol,
         DEF.FNCTIONAL_FEED_GRP funct_feed_group,
         DEF.TROPHIC_LEVEL trophic_level,
         DEF.PARTICLE_SIZE_BASIS particle_size_basis,  -- jah 02-20-04  Added Particle Size Basis
         DEF.TSRANLPR_IS_NUMBER anlpr_is_nbr,
         DEF.TSRANLPR_ORG_ID anlpr_org_id,
         DEF.TSRLSPP_IS_NUMBER lspp_is_number,
         DEF.TSRLSPP_ORG_ID lspp_org_id,
         CCGA.TSRCHAR_IS_NUMBER char_is_nbr,
         CCGA.TSRCHAR_ORG_ID char_org_id
  FROM   TSRCHDEF DEF, TSRCCGA CCGA
  WHERE  DEF.TSRCCGA_IS_NUMBER = CCGA.TSRCCGA_IS_NUMBER
  AND    DEF.TSRCCGA_ORG_ID = CCGA.TSRCCGA_ORG_ID
  AND    DEF.TSRCHDEF_IS_NUMBER = p_row_pk.is_number
  AND    DEF.TSRCHDEF_ORG_ID = RPAD(p_row_pk.org_id,8);

  CURSOR c_get_col_seq(p_col_name IN VARCHAR2) IS
  SELECT SIC_SEQ FROM SIM_IMP_COLS WHERE SIC_NAME = p_col_name;

BEGIN
  /* kms - function that gets characteristic defaults from STORET to be stored with results in SIM */

  IF p_row_pk.is_number IS NOT NULL THEN
    /* kms - used characteristic group and row */
    OPEN c_get_storet_row_defs(p_row_pk);
    FETCH c_get_storet_row_defs INTO 
      v_acode_defs_rec.fraction_is, 
      v_acode_defs_rec.voltinism_is,
      v_acode_defs_rec.habit_is, 
      v_uom_name, 
      v_acode_defs_rec.dur_basis,
      v_acode_defs_rec.stat_type, 
      v_acode_defs_rec.value_type,
      v_acode_defs_rec.temp_basis,
      v_acode_defs_rec.weight_basis, 
      v_acode_defs_rec.species,
      v_acode_defs_rec.pollution_tol, 
      v_acode_defs_rec.funct_feed_grp,
      v_acode_defs_rec.trophic_level,
      v_acode_defs_rec.particle_size_basis,  -- jah 02-20-04  Added Particle Size Basis
      v_acode_defs_rec.anlpr_pk.is_number, 
      v_acode_defs_rec.anlpr_pk.org_id,
      v_acode_defs_rec.labprep_pk.is_number, 
      v_acode_defs_rec.labprep_pk.org_id,
      v_acode_defs_rec.char_pk.is_number, 
      v_acode_defs_rec.char_pk.org_id;
      
    IF c_get_storet_row_defs%NOTFOUND THEN
      /* kms - row primary key could not be found in STORET, raise exception */
      CLOSE c_get_storet_row_defs;
      RAISE e_bad_def_pk;
    END IF;
    
    CLOSE c_get_storet_row_defs;

    v_acode_defs_rec.lab_pk.is_number := NULL;
    v_acode_defs_rec.lab_pk.org_id := NULL;
    
  END IF;

    /* look up pk, allowable value sequences */
    /* get uom pk */
  v_char_result_type := get_char_result_type(v_acode_defs_rec.char_pk.is_number);

  IF v_uom_name LIKE '    %' THEN
    v_uom_name := 'None      ';
  END IF;
  
  -- Get the Unit of Measure
  v_acode_defs_rec.uom_pk := sim_import_result_extras.get_uom_is_nbr(
    v_uom_name, 
    p_fa_seq,
    v_import_log_seq,
    v_line_cnt
  );

  -- jah 3-9-2004
  -- Is the unit of measure valid?  
  IF v_acode_defs_rec.uom_pk.IS_NUMBER IS NULL THEN
    -- if no
    RAISE e_bad_uom;
  END IF;

/*  SF 8/6/2002 Having changed all this for V2 we are no longer storing these sequences.
    * get duration basis sequence
    * get stat type sequence
    * get value type sequence
    * get temp basis sequence
*/

  -- Tell calling routine NO ERRORS occurred
  RETURN 1;

EXCEPTION

  WHEN e_bad_uom THEN

    /* jah 3-9-2004
       Function sim_import_result_extras.get_uom_is_nbr has already output the error message for this.
       This error happens when the EPA updates the Unit of Measure table (TSROUM) but not the
       characteristic default table (TSMCHDEF) */

    -- re-initialize v_acode_def_recs
    v_acode_defs_rec := v_acode_init_rec;

    -- Tell calling routine an error occurred
    RETURN NULL;  

  WHEN e_bad_def_map THEN

    Sim2_Import_Pkg.write_error(
      v_import_log_seq, 
      'Line '||v_line_cnt||': Unable to get char defaults from STORET due to faulty mapping.',
      p_fa_seq
    );

    -- re-initialize v_acode_def_recs
    v_acode_defs_rec := v_acode_init_rec;

    -- Tell calling routine an error occurred
    RETURN NULL;

  WHEN OTHERS THEN

    Sim2_Import_Pkg.write_error(
      v_import_log_seq, 
      'Line '||v_line_cnt||': '||SQLERRM||' - in get char defaults.',
      p_fa_seq
    );

    -- re-initialize v_acode_def_recs
    v_acode_defs_rec := v_acode_init_rec;

    -- Tell calling routine an error occurred
    RETURN NULL;

END get_char_defaults;

/*********************************************************************************/
FUNCTION check_allow_species(p_char_is_number IN NUMBER) RETURN NUMBER IS

  -- This function checks if the taxon provided can have an associated species based
  -- on the rank of the taxon.  The function returns a '0' if a species is NOT allowed
  -- and a 1 if it is.
  v_rank_code CHAR(8);

  CURSOR c_check_species IS
    SELECT TAXON_RANK_CODE
      FROM TSRCHAR
      WHERE TSRCHAR_IS_NUMBER = p_char_is_number;

  BEGIN
    OPEN c_check_species;
      FETCH c_check_species INTO v_rank_code;
      IF v_rank_code IN ('SPE     ','FOR     ','SSP     ','VAR     ') THEN
        RETURN 0;
      ELSE
        RETURN 1;
      END IF;
    CLOSE c_check_species;
  EXCEPTION
    WHEN OTHERS THEN
      RETURN 0;

END check_allow_species;

/*********************************************************************************/

FUNCTION  validate_class_descriptor(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_char_is_nbr IN NUMBER,
          p_item_name IN VARCHAR2,p_col_seq IN NUMBER,p_fa_seq IN NUMBER, p_line_cnt IN NUMBER,
          p_import_log_seq IN NUMBER) RETURN VARCHAR2 IS

-- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item            VARCHAR2(40);
  v_trans_exist               BOOLEAN := TRUE;
  v_storet_value              VARCHAR2(12);
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);
  v_storet_pk                 Sim_import_result_extras.t_storet_pk_type;  

  CURSOR c_check_char_pv(p_result IN VARCHAR2, p_char_is_nbr IN NUMBER) IS
    SELECT SHORT_NAME
    FROM TSRCPV
    WHERE SHORT_NAME = RPAD(p_result,12)
    AND TSRCHAR_IS_NUMBER = p_char_is_nbr;

   BEGIN
   
   
   
     v_storet_value := NULL;
     
     
     /* get the char group is number - check translations and STORET tables */
     V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.CHECK_STORET_TRANSLATION(P_ORIG_ITEM,
                                                                      P_IMP_CFG_SEQ,
                                                                      P_COL_SEQ,
                                                                      V_TRANS_EXIST,                                  
                                                                      P_IMPORT_LOG_SEQ,
                                                                      P_LINE_CNT,
                                                                      V_CONVERTED_ITEM,
                                                                      P_FA_SEQ);     

     v_storet_value := v_converted_item;   
   
   
     IF v_storet_pk.is_number IS NULL AND NOT v_trans_exist THEN
       OPEN c_check_char_pv(v_converted_item, p_char_is_nbr);
         FETCH c_check_char_pv INTO v_storet_value;
         IF c_check_char_pv%NOTFOUND THEN
           v_storet_value := NULL;
         END IF;
       CLOSE c_check_char_pv;

       IF v_storet_value IS NULL THEN
         v_instr_id  := 'RES-TR';
         /* check instructions to see if translation record should be created */
         v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq,v_import_log_seq,v_line_cnt);
         IF v_instr_code = 'RES-TR-01' THEN
         /* if record could not be matched to anything in translations or STORET tables,
           create new record */
           sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq,v_import_log_seq,v_line_cnt);
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_fa_seq);
         ELSE
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_fa_seq);
         END IF;
       ELSIF v_storet_value IS NULL AND v_trans_exist THEN
        /*  item has been translated to an invalid storet value */
         Sim2_Import_Pkg.write_error(v_import_log_seq,p_item_name||', '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_fa_seq);
       END IF;
     ELSIF v_storet_pk.is_number IS NULL AND v_trans_exist THEN
      /* item exists in translations table but is not mapped to anything, write error message.
         Status will be changed to 'N' later in the program. */
       v_storet_value := NULL;
     END IF;

     RETURN v_storet_value;

   EXCEPTION
     WHEN OTHERS THEN
       Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - checking class descriptor value.',p_fa_seq);
       RETURN NULL;
END validate_class_descriptor;

/******************************************************************************************************/
/******************************************************************************************************/

FUNCTION  validate_prmvl_item(p_orig_item IN CHAR, p_imp_cfg_seq IN NUMBER,
          p_item_name IN VARCHAR2,p_col_seq IN NUMBER,p_table_name IN VARCHAR,
          p_column_name IN VARCHAR2,p_fa_seq IN NUMBER, p_import_log_seq IN NUMBER,
          p_line_cnt IN NUMBER) RETURN VARCHAR2 IS

-- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item            VARCHAR2(40);
  v_trans_exist               BOOLEAN := TRUE;
  v_storet_value              VARCHAR2(40);
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);
  v_storet_pk                 Sim_import_result_extras.t_storet_pk_type;
  
  
  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT FIELD_VALUE
      FROM TSMPRMVL
      WHERE TABLE_NAME = RPAD(p_table_name,8)
      AND FIELD_NAME = p_column_name
      AND FIELD_VALUE = p_item;


BEGIN

  v_storet_value := NULL;
     
     
  /* get the char group is number - check translations and STORET tables */
  V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.CHECK_STORET_TRANSLATION(P_ORIG_ITEM,
                                          P_IMP_CFG_SEQ,
                                          P_COL_SEQ,
                                          V_TRANS_EXIST,                                  
                                          P_IMPORT_LOG_SEQ,
                                          P_LINE_CNT,
                                          V_CONVERTED_ITEM,
                                          P_FA_SEQ);     

  v_storet_value := v_converted_item;

  IF v_storet_pk.is_number IS NULL AND NOT v_trans_exist THEN 
    OPEN c_check_prmvl(v_converted_item);
      FETCH c_check_prmvl INTO v_storet_value;
      IF c_check_prmvl %NOTFOUND THEN
        v_storet_value := NULL;
      END IF;
    CLOSE c_check_prmvl;

    IF v_storet_value IS NULL THEN
      v_instr_id  := 'RES-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq,v_import_log_seq,v_line_cnt);
      IF v_instr_code = 'RES-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
          create new record */
        sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq,v_import_log_seq,v_line_cnt);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_fa_seq);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_fa_seq);
      END IF;
    END IF;
  ELSIF v_storet_pk.is_number IS NULL AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
      Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - checking STORET PRMVL.',p_fa_seq);
    RETURN NULL;
END validate_prmvl_item;

/******************************************************************************************************/
FUNCTION  get_prmvl_key(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
          p_item_name IN VARCHAR2,p_col_seq IN NUMBER,p_table_name IN VARCHAR,
          p_column_name IN VARCHAR2,p_fa_seq IN NUMBER, p_import_log_seq IN NUMBER,
          p_line_cnt IN NUMBER)
          RETURN sim_import_result_extras.t_storet_pk_type IS

-- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item            VARCHAR2(40);
  v_trans_exist               BOOLEAN := TRUE;
  --v_storet_value             VARCHAR2(40);
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);
  v_storet_key                sim_import_result_extras.t_storet_pk_type;

  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT TSMPRMVL_IS_NUMBER, TSMPRMVL_ORG_ID
      FROM TSMPRMVL
      WHERE TABLE_NAME = RPAD(p_table_name,8)
      AND FIELD_NAME = p_column_name
      AND FIELD_VALUE = p_item;


   BEGIN
    
     
     /* get the char group is number - check translations and STORET tables */
     V_STORET_KEY := SIM_IMPORT_RESULT_EXTRAS.CHECK_STORET_TRANSLATION(P_ORIG_ITEM,
                                             P_IMP_CFG_SEQ,
                                             P_COL_SEQ,
                                             V_TRANS_EXIST,                                  
                                             P_IMPORT_LOG_SEQ,
                                             P_LINE_CNT,
                                             V_CONVERTED_ITEM,
                                             P_FA_SEQ);     

     IF v_storet_key.is_number IS NULL AND NOT v_trans_exist THEN 
       OPEN c_check_prmvl(v_converted_item);
           FETCH c_check_prmvl INTO v_storet_key.is_number, v_storet_key.org_id;
           IF c_check_prmvl %NOTFOUND THEN
             v_storet_key.is_number := NULL;
             v_storet_key.org_id := NULL;
           END IF;
         CLOSE c_check_prmvl;

       IF v_storet_key.is_number IS NULL AND NOT v_trans_exist THEN
         v_instr_id  := 'RES-TR';
         /* check instructions to see if translation record should be created */
         v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq,v_import_log_seq,v_line_cnt);
         IF v_instr_code = 'RES-TR-01' THEN
         /* if record could not be matched to anything in translations or STORET tables,
           create new record */
           sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq,v_import_log_seq,v_line_cnt);
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_fa_seq);
         ELSE
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_fa_seq);
         END IF;
       ELSIF v_storet_key.is_number IS NULL AND v_trans_exist THEN
        /*  item has been translated to an invalid storet value */
         Sim2_Import_Pkg.write_error(v_import_log_seq,p_item_name||', '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_fa_seq);
       END IF;
     ELSIF v_storet_key.is_number IS NULL AND v_trans_exist THEN
      /* item exists in translations table but is not mapped to anything, write error message.
         Status will be changed to 'N' later in the program. */
       v_storet_key.is_number := NULL;
       v_storet_key.org_id := NULL;
     END IF;

     RETURN v_storet_key;

   EXCEPTION
     WHEN OTHERS THEN
       Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - in get STORET PRMVL key.',p_fa_seq);
       RETURN NULL;
END get_prmvl_key;
/******************************************************************************************************/
FUNCTION  validate_sample_matrix(p_orig_item IN CHAR, p_imp_cfg_seq IN NUMBER,
          p_fa_seq IN NUMBER, p_import_log_seq IN NUMBER,p_col_seq IN NUMBER,
          p_line_cnt IN NUMBER) RETURN sim_import_result_extras.t_storet_pk_type IS

-- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item            VARCHAR2(40);
  v_trans_exist               BOOLEAN := TRUE;
  v_matrx_key                 Sim_import_result_extras.t_storet_pk_type;
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);

  CURSOR c_check_matrx(p_item IN VARCHAR2) IS
    SELECT TSRMATRX_IS_NUMBER, TSRMATRX_ORG_ID
      FROM TSRMATRX
      WHERE NAME = p_item;


   BEGIN

   
   
     v_matrx_key.is_number := NULL;
     
     
     /* get the char group is number - check translations and STORET tables */
     v_matrx_key := SIM_IMPORT_RESULT_EXTRAS.CHECK_STORET_TRANSLATION(P_ORIG_ITEM,
                                          P_IMP_CFG_SEQ,
                                          P_COL_SEQ,
                                          V_TRANS_EXIST,                                  
                                          P_IMPORT_LOG_SEQ,
                                          P_LINE_CNT,
                                          V_CONVERTED_ITEM,
                                          P_FA_SEQ);     

     IF v_matrx_key.is_number IS NULL AND NOT v_trans_exist THEN   
       OPEN c_check_matrx(v_converted_item);
           FETCH c_check_matrx INTO v_matrx_key.is_number, v_matrx_key.org_id ;
           IF c_check_matrx %NOTFOUND THEN
             v_matrx_key.is_number := NULL;
           END IF;
       CLOSE c_check_matrx;

       IF v_matrx_key.is_number IS NULL THEN
         v_instr_id  := 'RES-TR';
         /* check instructions to see if translation record should be created */
         v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq,v_import_log_seq,v_line_cnt);
         IF v_instr_code = 'RES-TR-01' THEN
         /* if record could not be matched to anything in translations or STORET tables,
           create new record */
           sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq,v_import_log_seq,v_line_cnt);
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Matrix, '||v_converted_item||', is not a valid value - translation record has been created.',p_fa_seq);
         ELSE
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Matrix, '||v_converted_item||', is not a valid value.',p_fa_seq);
         END IF;
       ELSIF v_matrx_key.is_number IS NULL AND v_trans_exist THEN
        /*  item has been translated to an invalid storet value */
         Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Sample Matrix, '||p_orig_item||', has been mapped to the invalid storet value: '||v_converted_item,p_fa_seq);
       END IF;
     ELSIF v_matrx_key.is_number IS NULL AND v_trans_exist THEN
      /* item exists in translations table but is not mapped to anything, write error message.
         Status will be changed to 'N' later in the program. */
       v_matrx_key.is_number := NULL;
       v_matrx_key.org_id := NULL;
     END IF;

     RETURN v_matrx_key;
    
     
   EXCEPTION
     WHEN OTHERS THEN
       Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - checking STORET TSRMATRX.',p_fa_seq);
       RETURN NULL;
END validate_sample_matrix;

/******************************************************************************************************/
FUNCTION  check_existing_MTPC(p_brg_type IN VARCHAR2, p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_return  NUMBER(1);
  v_seq     NUMBER(10);

  CURSOR c_check_MTPC IS
    SELECT BRG_SEQ FROM SIM_BIO_RESULTS_GRP
    WHERE BRG_FA_SEQ = p_fa_seq
    AND BRG_TYPE = 'Multi-Taxon Population Census';

  BEGIN
    OPEN c_check_MTPC;
      FETCH c_check_MTPC INTO v_seq;
      IF c_check_MTPC%FOUND THEN
        v_return := 2;
      ELSE
        v_return := 0;
      END IF;
    CLOSE c_check_MTPC;
    RETURN v_return;

  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - in check existing MTPC.',p_fa_seq);
      RETURN 1;
  END check_existing_mtpc;

/*********************************************************************************/
FUNCTION check_if_station_exists(p_station_cd IN VARCHAR2,p_org_id IN VARCHAR2,
                                 p_sta_seq OUT NUMBER) RETURN NUMBER IS

/*  SF-12/11/02
In this function we check to see if the station existis in SIM, and if all the wells and locations
that exist in storet also are reflected in SIM.
*/
  v_storet_station_cd     VARCHAR2(15) := p_station_cd;
  v_sim_station_cd        VARCHAR2(15) := p_station_cd;
  v_return                NUMBER(1)    := 0;
  v_seq                   NUMBER(10);
  v_is_nbr                NUMBER(10);
  v_success               NUMBER(1);
  v_sta_rec               SIM_STATIONS%ROWTYPE;
  v_stor_sta_rec          TSMSTATN%ROWTYPE;
  v_tsmstatn_is_nbr       NUMBER(12);
  v_type                  NUMBER(8);
  v_well_rec              SIM_STATION_WELLS%ROWTYPE;
  v_storet_wells          TSMWELL%ROWTYPE;
  v_loc_rec               SIM_STATION_LOCATIONS%ROWTYPE;
  v_storet_loc_rec        TSMALP%ROWTYPE;
  v_well_seq              NUMBER(10);
  v_loc_seq               NUMBER(10);
  v_well_is_number        NUMBER(12);

  e_bad_well              EXCEPTION;
  e_bad_station           EXCEPTION;

  CURSOR c_compare_station_id IS
    SELECT IDENTIFICATION_CD, tsmstatn_is_number, tsmvstc_is_number
    FROM TSMSTATN
    WHERE  IDENTIFICATION_CD = RPAD(p_station_cd,15)
      AND TSMSTATN_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_compare_sim_stat_id(p_station_cd IN VARCHAR2,p_station_org_id IN VARCHAR2,p_tsmstatn_is_nbr IN NUMBER) IS
    SELECT STA_SEQ, STA_ID
    FROM SIM_STATIONS
    WHERE  STA_ID = p_station_cd
      AND STA_IS_NUMBER = p_tsmstatn_is_nbr
      AND STA_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_compare_sim_well(p_station_seq IN NUMBER, p_well_is_nbr IN NUMBER, p_well_org_id IN VARCHAR2) IS
    SELECT SSW_IS_NUMBER
    FROM SIM_STATION_WELLS
    WHERE SSW_STA_SEQ = p_station_seq
      AND SSW_IS_NUMBER = p_well_is_nbr
      AND SSW_ORG_ID = RPAD(p_well_org_id,8);

  CURSOR c_compare_sim_loc(p_station_seq IN NUMBER,p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2) IS
    SELECT STL_IS_NUMBER
    FROM SIM_STATION_LOCATIONS
    WHERE STL_STA_SEQ = p_station_seq
      AND STL_IS_NUMBER = p_tsmalp_is_number
      AND STL_ORG_ID = RPAD(p_tsmalp_org_id,8);

  CURSOR c_get_station_seq IS
    SELECT SIM_STATIONS_SEQ.NEXTVAL
    FROM DUAL;

  CURSOR c_get_well_seq IS
    SELECT SIM_STATION_WELLS_SEQ.NEXTVAL
    FROM DUAL;

  CURSOR c_get_location_seq IS
    SELECT SIM_STATION_LOCATIONS_SEQ.NEXTVAL
    FROM DUAL;

  CURSOR c_storet_station IS
    SELECT *
    FROM TSMSTATN
    WHERE identification_cd = RPAD(p_station_cd,15)
      AND tsmstatn_org_id = RPAD(p_org_id,8);

  CURSOR c_storet_wells(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT *
    FROM TSMWELL
    WHERE TSMSTATN_IS_NUMBER = v_tsmstatn_is_nbr
      AND TSMSTATN_ORG_ID = RPAD(p_tsmstatn_org_id,8);

  CURSOR c_storet_loc(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT *
    FROM TSMALP
    WHERE TSMSTATN0IS_NUMBER = p_tsmstatn_is_number
      AND TSMSTATN0ORG_ID = RPAD(p_tsmstatn_org_id,8);

      BEGIN
  v_return := 1;

  /* Check STORET */
  OPEN c_compare_station_id;
    FETCH c_compare_station_id INTO v_storet_station_cd, v_tsmstatn_is_nbr,v_type;

    IF c_compare_station_id%NOTFOUND THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line: '||v_line_cnt||' - Station, '||p_station_cd||', does not exist in STORET. Cannot proceed with Import' ,NULL);
      v_return := 0;

    ELSE       --Check to see if it's in SIM, too
      BEGIN
        OPEN c_compare_sim_stat_id(p_station_cd,p_org_id,v_tsmstatn_is_nbr);
          FETCH c_compare_sim_stat_id INTO v_seq, v_sim_station_cd;
          IF c_compare_sim_stat_id%NOTFOUND THEN
            OPEN c_get_station_seq;
              FETCH c_get_station_seq INTO v_seq;
            CLOSE c_get_station_seq;
            OPEN c_storet_station;
              FETCH c_storet_station INTO v_stor_sta_rec;
            CLOSE c_storet_station;
            v_sta_rec.STA_SEQ                := v_seq;
            v_sta_rec.STA_IS_NUMBER          := v_stor_sta_rec.tsmstatn_is_number;
            v_sta_rec.STA_ORG_ID             := RTRIM(v_stor_sta_rec.tsmstatn_org_id);
            v_sta_rec.STA_ID                 := RTRIM(v_stor_sta_rec.identification_cd);
            v_sta_rec.STA_NAME               := v_stor_sta_rec.name;
            v_sta_rec.STA_ESTAB_DATE         := v_stor_sta_rec.establishment_date;
            v_sta_rec.STA_DESC               := v_stor_sta_rec.description_text;
            v_sta_rec.STA_TSMORGAN_ORG_ID    := p_org_id;
            v_sta_rec.STA_TSMORGAN_IS_NUMBER := v_stor_sta_rec.tsmorgan_is_number;
            v_sta_rec.STA_TSMVSTC_IS_NUMBER  := v_stor_sta_rec.tsmvstc_is_number;
            v_sta_rec.STA_TSMVSTC_ORG_ID     := v_stor_sta_rec.tsmvstc_org_id;
            v_sta_rec.STA_STATUS             := 'U';
            v_sta_rec.STA_EXPORT_SEQ         := NULL;
            v_sta_rec.STA_IMPORT_SEQ         := v_import_log_seq;
            v_sta_rec.STA_EPA_KEY_ID         := v_stor_sta_rec.epa_key_identifier;
            v_sta_rec.STA_ZID_REL            := v_stor_sta_rec.zid_relation_code;
            v_sta_rec.STA_INFLUENCE_AREA     := v_stor_sta_rec.influence_area;
            v_sta_rec.STA_TRAVEL_DIR         := v_stor_sta_rec.travel_dir_txt;
            v_sta_rec.STA_WATER_DEPTH        := v_stor_sta_rec.water_depth;
            v_sta_rec.STA_WATER_DEPTH_UNIT   := v_stor_sta_rec.water_depth_unit;
            v_sta_rec.STA_ECOREGION          := v_stor_sta_rec.ecoregion_name;
            v_sta_rec.STA_DOCUMENT           := v_stor_sta_rec.blob_title;
            v_success := create_station(v_sta_rec);
            IF v_success = 1 THEN -- go ahead and get the wells
              FOR v_storet_wells IN c_storet_wells(v_tsmstatn_is_nbr, p_org_id) LOOP
                OPEN c_get_well_seq;
                  FETCH c_get_well_seq INTO v_well_seq;
                CLOSE c_get_well_seq;

                v_well_rec.SSW_SEQ                      := v_well_seq;
                v_well_rec.SSW_STA_SEQ                  := v_seq;
                v_well_rec.SSW_WELL_STATUS              := v_storet_wells.STATUS_CODE;
                v_well_rec.SSW_IS_NUMBER                := v_storet_wells.TSMWELL_IS_NUMBER;
                v_well_rec.SSW_ORG_ID                   := v_storet_wells.TSMWELL_ORG_ID;
                v_well_rec.SSW_NAME                     := v_storet_wells.NAME;
                v_well_rec.SSW_WELL_NO                  := v_storet_wells.ID_NUMBER;
                v_well_rec.SSW_WELL_USE                 := v_storet_wells.USE_CODE;
                v_well_rec.SSW_DEVELOP_METHOD           := v_storet_wells.DEVELOPMNT_MTHD_CD;
                v_well_rec.SSW_WELL_CONST_METH          := v_storet_wells.CONSTRUCTN_MTHD_CD;
                v_well_rec.SSW_WATER_USE               := v_storet_wells.WTR_PRIMRY_USE_CD;
                v_well_rec.SSW_GRADIENT_TYPE            := v_storet_wells.GRADIENT_TYPE_CODE;
                v_well_rec.SSW_DEPTH_COMPLETION         := v_storet_wells.DEPTH_CMPLTN_MSR;
                v_well_rec.SSW_DEPTH_COMPLETION_UNIT    := v_storet_wells.DPTH_CMPLTN_UNT_CD;
                v_well_rec.SSW_DEPTH_HOLE               := v_storet_wells.DEPTH_OF_HOLE_MSR;
                v_well_rec.SSW_DEPTH_HOLE_UNIT          := v_storet_wells.DEPTH_HOLE_UNT_CD;
                v_well_rec.SSW_WELL_HEIGHT              := v_storet_wells.WELL_HEAD_HT_MSR;
                v_well_rec.SSW_WELL_HEIGHT_UNIT         := v_storet_wells.WL_HD_HT_UNIT_CD;
                v_well_rec.SSW_CASING_HEIGHT            := v_storet_wells.CASING_HGT_MSR;
                v_well_rec.SSW_CASING_HEIGHT_UNIT       := v_storet_wells.CASING_HGT_UNIT_CD;
                v_well_rec.SSW_DISINFECTED              := v_storet_wells.DISINFECTED_IND_CD;
                v_well_rec.SSW_CONST_START_DATE         := v_storet_wells.CONSTRUCTN_STRT_DT;
                v_well_rec.SSW_CONST_END_DATE           := v_storet_wells.CONSTRUCTN_END_DT;
                v_well_rec.SSW_INIT_PUMP_DUR            := v_storet_wells.INIT_PUMP_DUR;
                v_well_rec.SSW_INIT_PUMP_DUR_UNIT       := v_storet_wells.INT_PMP_DUR_UNT_CD;
                v_well_rec.SSW_INIT_PUMP_RATE           := v_storet_wells.INIT_PUMP_RATE;
                v_well_rec.SSW_INIT_PUMP_RATE_UNIT      := v_storet_wells.INT_PMP_RAT_UNT_CD;
                v_well_rec.SSW_NATURAL_FLOW             := v_storet_wells.NATURL_FLOW_IND_CD;
                v_well_rec.SSW_INIT_BOREHOLE_DIAM       := v_storet_wells.INIT_BOREHOLE_DIAM;
                v_well_rec.SSW_INIT_BOREHOLE_DIAM_UNIT  := v_storet_wells.INIT_BHOLE_DIAM_UN;
                v_well_rec.SSW_TSMSTATN_IS_NUMBER       := v_storet_wells.TSMSTATN_IS_NUMBER;
                v_well_rec.SSW_TSMSTATN_ORG_ID          := v_storet_wells.TSMSTATN_ORG_ID;
                v_well_rec.SSW_STATUS                   := 'U';
                v_well_rec.SSW_IMPORT_SEQ               := NULL;
                v_well_rec.SSW_EXPORT_SEQ               := NULL;
                v_success := create_well(v_well_rec);
                IF v_success != 1 THEN
                  RAISE e_bad_well;
                END IF;
              END LOOP;

              --now locations
              FOR v_storet_loc IN c_storet_loc(v_tsmstatn_is_nbr, p_org_id) LOOP
                OPEN c_get_location_seq;
                  FETCH c_get_location_seq INTO v_loc_seq;
                CLOSE c_get_location_seq;
                v_loc_rec.STL_SEQ                := v_loc_seq;
                v_loc_rec.STL_STA_SEQ            := v_seq;
                v_loc_rec.STL_IS_NUMBER          := v_storet_loc.TSMALP_IS_NUMBER;
                v_loc_rec.STL_ORG_ID             := v_storet_loc.TSMALP_ORG_ID;
                v_loc_rec.STL_POINT_NAME         := v_storet_loc.POINT_NAME;
                v_loc_rec.STL_TYPE_CODE          := v_storet_loc.TYPE_CODE;
                v_loc_rec.STL_SEQUENCE_NUMBER    := v_storet_loc.SEQUENCE_NUMBER;
                v_loc_rec.STL_ELEV_DATUM         := NULL;
                v_loc_rec.STL_ELEV_METHOD        := NULL;
                v_loc_rec.STL_ELEV_MSR_DATE      := v_storet_loc.elvtn_msr_dt;
                v_loc_rec.STL_ELEVATION_MSR      := v_storet_loc.elevation_msr;
                v_loc_rec.STL_ELEVATION_UNIT     := NULL;
                v_loc_rec.STL_GEOPOS_DATUM       := NULL;
                v_loc_rec.STL_GEOPOS_METHOD      := NULL;
                v_loc_rec.STL_GPS_SCALE          := v_storet_loc.geopstng_scale_txt;
                v_loc_rec.STL_LAT_DIR            := v_storet_loc.lat_direction;
                v_loc_rec.STL_LAT_DEGREES        := v_storet_loc.lat_degree_msr;
                v_loc_rec.STL_LAT_MINUTES        := v_storet_loc.lat_minute_msr;
                v_loc_rec.STL_LAT_SECONDS        := v_storet_loc.lat_second_msr;
                v_loc_rec.STL_LONG_DIR           := v_storet_loc.long_direction;
                v_loc_rec.STL_LONG_DEGREES       := v_storet_loc.long_degree_msr;
                v_loc_rec.STL_LONG_MINUTES       := v_storet_loc.long_minute_msr;
                v_loc_rec.STL_LONG_SECONDS       := v_storet_loc.long_second_msr;
                v_loc_rec.STL_LAT_LONG_MSR_DATE  := v_storet_loc.lat_long_msr_dt;
                v_loc_rec.STL_LATLONG_ACCURCY    := NULL;
                v_loc_rec.STL_LATLONG_ACCURCY_UNIT  := NULL;
                v_loc_rec.STL_TSMRRR_IS_NUMBER      := NULL;
                v_loc_rec.STL_TSMRRR_ORG_ID         := NULL;
                v_loc_rec.STL_TSMFHU_IS_NUMBER      := v_storet_loc.tsmfhu_is_number;
                v_loc_rec.STL_TSMFHU_ORG_ID         := v_storet_loc.tsmfhu_org_id;
                v_loc_rec.STL_TSMSTATN0IS_NUMBER    := v_storet_loc.tsmstatn0is_number;
                v_loc_rec.STL_TSMSTATN0ORG_ID       := v_storet_loc.tsmstatn0org_id;
                v_loc_rec.STL_TSMWELL_IS_NUMBER     := v_storet_loc.tsmwell_is_number;
                v_loc_rec.STL_TSMWELL_ORG_ID        := v_storet_loc.tsmwell_org_id;
                v_loc_rec.STL_TSMGEOPA_IS_NUMBER    := v_storet_loc.tsmgeopa_is_number;
                v_loc_rec.STL_TSMGEOPA_ORG_ID       := v_storet_loc.tsmgeopa_org_id;
                v_loc_rec.STL_LAT_DEC_MINUTES       := NULL;
                v_loc_rec.STL_LONG_DEC_MINUTES      := NULL;
                v_loc_rec.STL_TSMPIPE_IS_NUMBER     := NULL;
                v_loc_rec.STL_TSMPIPE_ORG_ID        := NULL;
                v_loc_rec.STL_TSMSTATN1ORG_ID       := NULL;
                v_loc_rec.STL_TSMSTATN1IS_NUMBER    := NULL;
                v_loc_rec.STL_TSMGEOPA0IS_NUMBER    := NULL;
                v_loc_rec.STL_TSMGEOPA0ORG_ID       := NULL;
                v_loc_rec.STL_TSMNAL_CD             := v_storet_loc.tsmnal_cd;
                v_loc_rec.STL_TSMNAL_STATE          := v_storet_loc.tsmnal_state;
                v_loc_rec.STL_NRCS                  := v_storet_loc.nrcs_wtrsd_id_num;
                v_loc_rec.STL_STATUS                := 'U';
                v_loc_rec.STL_IMPORT_SEQ            := NULL;
                v_loc_rec.STL_EXPORT_SEQ            := NULL;

                v_success := create_station_location(v_loc_rec);
              END LOOP;
            ELSE
              v_return := 0;
              RAISE e_bad_station;
            END IF;
          ELSE --the station is there, check its wells and locations
            FOR v_storet_wells IN c_storet_wells(v_tsmstatn_is_nbr,p_org_id) LOOP
              OPEN c_compare_sim_well(v_seq, v_storet_wells.TSMWELL_IS_NUMBER,v_storet_wells.TSMWELL_ORG_ID);
                FETCH c_compare_sim_well INTO v_is_nbr;
                IF c_compare_sim_well%NOTFOUND THEN
                  OPEN c_get_well_seq;
                    FETCH c_get_well_seq INTO v_well_seq;
                  CLOSE c_get_well_seq;
                  v_well_rec.SSW_SEQ                      := v_well_seq;
                  v_well_rec.SSW_STA_SEQ                  := v_seq;
                  v_well_rec.SSW_WELL_STATUS              := v_storet_wells.STATUS_CODE;
                  v_well_rec.SSW_IS_NUMBER                := v_storet_wells.TSMWELL_IS_NUMBER;
                  v_well_rec.SSW_ORG_ID                   := v_storet_wells.TSMWELL_ORG_ID;
                  v_well_rec.SSW_NAME                     := v_storet_wells.NAME;
                  v_well_rec.SSW_WELL_NO                  := v_storet_wells.ID_NUMBER;
                  v_well_rec.SSW_WELL_USE                 := v_storet_wells.USE_CODE;
                  v_well_rec.SSW_DEVELOP_METHOD           := v_storet_wells.DEVELOPMNT_MTHD_CD;
                  v_well_rec.SSW_WELL_CONST_METH          := v_storet_wells.CONSTRUCTN_MTHD_CD;
                  v_well_rec.SSW_WATER_USE               := v_storet_wells.WTR_PRIMRY_USE_CD;
                  v_well_rec.SSW_GRADIENT_TYPE            := v_storet_wells.GRADIENT_TYPE_CODE;
                  v_well_rec.SSW_DEPTH_COMPLETION         := v_storet_wells.DEPTH_CMPLTN_MSR;
                  v_well_rec.SSW_DEPTH_COMPLETION_UNIT    := v_storet_wells.DPTH_CMPLTN_UNT_CD;
                  v_well_rec.SSW_DEPTH_HOLE               := v_storet_wells.DEPTH_OF_HOLE_MSR;
                  v_well_rec.SSW_DEPTH_HOLE_UNIT          := v_storet_wells.DEPTH_HOLE_UNT_CD;
                  v_well_rec.SSW_WELL_HEIGHT              := v_storet_wells.WELL_HEAD_HT_MSR;
                  v_well_rec.SSW_WELL_HEIGHT_UNIT         := v_storet_wells.WL_HD_HT_UNIT_CD;
                  v_well_rec.SSW_CASING_HEIGHT            := v_storet_wells.CASING_HGT_MSR;
                  v_well_rec.SSW_CASING_HEIGHT_UNIT       := v_storet_wells.CASING_HGT_UNIT_CD;
                  v_well_rec.SSW_DISINFECTED              := v_storet_wells.DISINFECTED_IND_CD;
                  v_well_rec.SSW_CONST_START_DATE         := v_storet_wells.CONSTRUCTN_STRT_DT;
                  v_well_rec.SSW_CONST_END_DATE           := v_storet_wells.CONSTRUCTN_END_DT;
                  v_well_rec.SSW_INIT_PUMP_DUR            := v_storet_wells.INIT_PUMP_DUR;
                  v_well_rec.SSW_INIT_PUMP_DUR_UNIT       := v_storet_wells.INT_PMP_DUR_UNT_CD;
                  v_well_rec.SSW_INIT_PUMP_RATE           := v_storet_wells.INIT_PUMP_RATE;
                  v_well_rec.SSW_INIT_PUMP_RATE_UNIT      := v_storet_wells.INT_PMP_RAT_UNT_CD;
                  v_well_rec.SSW_NATURAL_FLOW             := v_storet_wells.NATURL_FLOW_IND_CD;
                  v_well_rec.SSW_INIT_BOREHOLE_DIAM       := v_storet_wells.INIT_BOREHOLE_DIAM;
                  v_well_rec.SSW_INIT_BOREHOLE_DIAM_UNIT  := v_storet_wells.INIT_BHOLE_DIAM_UN;
                  v_well_rec.SSW_TSMSTATN_IS_NUMBER       := v_storet_wells.TSMSTATN_IS_NUMBER;
                  v_well_rec.SSW_TSMSTATN_ORG_ID          := v_storet_wells.TSMSTATN_ORG_ID;
                  v_well_rec.SSW_STATUS                   := 'U';
                  v_well_rec.SSW_IMPORT_SEQ               := NULL;
                  v_well_rec.SSW_EXPORT_SEQ               := NULL;

                  v_success := create_well(v_well_rec);
                  IF v_success != 1 THEN
                    RAISE e_bad_well;
                  END IF;
                END IF;
              CLOSE c_compare_sim_well;
            END LOOP;
            FOR v_storet_loc_rec IN c_storet_loc(v_tsmstatn_is_nbr,p_org_id) LOOP
              OPEN c_compare_sim_loc(v_seq, v_storet_loc_rec.TSMALP_IS_NUMBER,v_storet_loc_rec.TSMALP_ORG_ID);
                FETCH c_compare_sim_loc INTO v_is_nbr;
                IF c_compare_sim_loc%NOTFOUND THEN

                  OPEN c_get_location_seq;
                    FETCH c_get_location_seq INTO v_loc_seq;
                  CLOSE c_get_location_seq;
                  v_loc_rec.STL_SEQ                := v_loc_seq;
                  v_loc_rec.STL_STA_SEQ            := v_seq;
                  v_loc_rec.STL_IS_NUMBER          := v_storet_loc_rec.TSMALP_IS_NUMBER;
                  v_loc_rec.STL_ORG_ID             := v_storet_loc_rec.TSMALP_ORG_ID;
                  v_loc_rec.STL_POINT_NAME         := v_storet_loc_rec.POINT_NAME;
                  v_loc_rec.STL_TYPE_CODE          := v_storet_loc_rec.TYPE_CODE;
                  v_loc_rec.STL_SEQUENCE_NUMBER    := v_storet_loc_rec.SEQUENCE_NUMBER;
                  v_loc_rec.STL_ELEV_DATUM         := NULL;
                  v_loc_rec.STL_ELEV_METHOD        := NULL;
                  v_loc_rec.STL_ELEV_MSR_DATE      := v_storet_loc_rec.elvtn_msr_dt;
                  v_loc_rec.STL_ELEVATION_MSR      := v_storet_loc_rec.elevation_msr;
                  v_loc_rec.STL_ELEVATION_UNIT     := NULL;
                  v_loc_rec.STL_GEOPOS_DATUM       := NULL;
                  v_loc_rec.STL_GEOPOS_METHOD      := NULL;
                  v_loc_rec.STL_GPS_SCALE          := v_storet_loc_rec.geopstng_scale_txt;
                  v_loc_rec.STL_LAT_DIR            := v_storet_loc_rec.lat_direction;
                  v_loc_rec.STL_LAT_DEGREES        := v_storet_loc_rec.lat_degree_msr;
                  v_loc_rec.STL_LAT_MINUTES        := v_storet_loc_rec.lat_minute_msr;
                  v_loc_rec.STL_LAT_SECONDS        := v_storet_loc_rec.lat_second_msr;
                  v_loc_rec.STL_LONG_DIR           := v_storet_loc_rec.long_direction;
                  v_loc_rec.STL_LONG_DEGREES       := v_storet_loc_rec.long_degree_msr;
                  v_loc_rec.STL_LONG_MINUTES       := v_storet_loc_rec.long_minute_msr;
                  v_loc_rec.STL_LONG_SECONDS       := v_storet_loc_rec.long_second_msr;
                  v_loc_rec.STL_LAT_LONG_MSR_DATE  := v_storet_loc_rec.lat_long_msr_dt;
                  v_loc_rec.STL_LATLONG_ACCURCY    := NULL;
                  v_loc_rec.STL_LATLONG_ACCURCY_UNIT  := NULL;
                  v_loc_rec.STL_TSMRRR_IS_NUMBER      := NULL;
                  v_loc_rec.STL_TSMRRR_ORG_ID         := NULL;
                  v_loc_rec.STL_TSMFHU_IS_NUMBER      := v_storet_loc_rec.tsmfhu_is_number;
                  v_loc_rec.STL_TSMFHU_ORG_ID         := v_storet_loc_rec.tsmfhu_org_id;
                  v_loc_rec.STL_TSMSTATN0IS_NUMBER    := v_storet_loc_rec.tsmstatn0is_number;
                  v_loc_rec.STL_TSMSTATN0ORG_ID       := v_storet_loc_rec.tsmstatn0org_id;
                  v_loc_rec.STL_TSMWELL_IS_NUMBER     := v_storet_loc_rec.tsmwell_is_number;
                  v_loc_rec.STL_TSMWELL_ORG_ID        := v_storet_loc_rec.tsmwell_org_id;
                  v_loc_rec.STL_TSMGEOPA_IS_NUMBER    := v_storet_loc_rec.tsmgeopa_is_number;
                  v_loc_rec.STL_TSMGEOPA_ORG_ID       := v_storet_loc_rec.tsmgeopa_org_id;
                  v_loc_rec.STL_LAT_DEC_MINUTES       := NULL;
                  v_loc_rec.STL_LONG_DEC_MINUTES      := NULL;
                  v_loc_rec.STL_TSMPIPE_IS_NUMBER     := NULL;
                  v_loc_rec.STL_TSMPIPE_ORG_ID        := NULL;
                  v_loc_rec.STL_TSMSTATN1ORG_ID       := NULL;
                  v_loc_rec.STL_TSMSTATN1IS_NUMBER    := NULL;
                  v_loc_rec.STL_TSMGEOPA0IS_NUMBER    := NULL;
                  v_loc_rec.STL_TSMGEOPA0ORG_ID       := NULL;
                  v_loc_rec.STL_TSMNAL_CD             := v_storet_loc_rec.tsmnal_cd;
                  v_loc_rec.STL_TSMNAL_STATE          := v_storet_loc_rec.tsmnal_state;
                  v_loc_rec.STL_NRCS                  := v_storet_loc_rec.nrcs_wtrsd_id_num;
                  v_loc_rec.STL_STATUS                := 'U';
                  v_loc_rec.STL_IMPORT_SEQ            := NULL;
                  v_loc_rec.STL_EXPORT_SEQ            := NULL;

                  v_success := create_station_location(v_loc_rec);

                END IF;
              CLOSE c_compare_sim_loc;
            END LOOP;
          END IF;
        CLOSE c_compare_sim_stat_id;
      EXCEPTION
        WHEN OTHERS THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' in c_compare_station_id in check_if_station_exists. Line: '||v_line_cnt,NULL);
          v_return := 0;
          CLOSE c_compare_sim_stat_id;
      END;
    END IF;

  CLOSE c_compare_station_id;

  RETURN v_return;

EXCEPTION
  WHEN e_bad_well THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' grabbing STORET wells in check_if_station_exists. Line: '||v_line_cnt,NULL);
    RETURN 0;
  WHEN e_bad_station THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' grabbing STORET station in check_if_station_exists. Line: '||v_line_cnt,NULL);
    RETURN 0;
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check_if_station_exists. Line: '||v_line_cnt,NULL);
    RETURN 0;

END CHECK_IF_STATION_EXISTS;

/******************************************************************************************************
  The create_station_location function inserts known good data into the SIM table SIM_STATIONS
  After the station_location has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_station_location (p_station_location_rec IN sim_station_locations%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create station using values in station record */

  INSERT INTO SIM_STATION_LOCATIONS (
                                     STL_SEQ
                                     ,STL_STA_SEQ
                                     ,STL_SSW_SEQ
                                     ,STL_IS_NUMBER
                                     ,STL_ORG_ID
                                     ,STL_POINT_NAME
                                     ,STL_TYPE_CODE
                                     ,STL_SEQUENCE_NUMBER
                                     ,STL_ELEV_DATUM
                                     ,STL_ELEV_METHOD
                                     ,STL_ELEV_MSR_DATE
                                     ,STL_ELEVATION_MSR
                                     ,STL_ELEVATION_UNIT
                                     ,STL_GEOPOS_DATUM
                                     ,STL_GEOPOS_METHOD
                                     ,STL_GPS_SCALE
                                     ,STL_LAT_DIR
                                     ,STL_LAT_DEGREES
                                     ,STL_LAT_MINUTES
                                     ,STL_LAT_SECONDS
                                     ,STL_LONG_DIR
                                     ,STL_LONG_DEGREES
                                     ,STL_LONG_MINUTES
                                     ,STL_LONG_SECONDS
                                     ,STL_LAT_LONG_MSR_DATE
                                     ,STL_LATLONG_ACCURCY
                                     ,STL_LATLONG_ACCURCY_UNIT
                                     ,STL_TSMRRR_IS_NUMBER
                                     ,STL_TSMRRR_ORG_ID
                                     ,STL_TSMFHU_IS_NUMBER
                                     ,STL_TSMFHU_ORG_ID
                                     ,STL_TSMSTATN0IS_NUMBER
                                     ,STL_TSMSTATN0ORG_ID
                                     ,STL_TSMWELL_IS_NUMBER
                                     ,STL_TSMWELL_ORG_ID
                                     ,STL_TSMGEOPA_IS_NUMBER
                                     ,STL_TSMGEOPA_ORG_ID
                                     ,STL_LAT_DEC_MINUTES
                                     ,STL_LONG_DEC_MINUTES
                                     ,STL_TSMPIPE_IS_NUMBER
                                     ,STL_TSMPIPE_ORG_ID
                                     ,STL_TSMSTATN1ORG_ID
                                     ,STL_TSMSTATN1IS_NUMBER
                                     ,STL_TSMGEOPA0IS_NUMBER
                                     ,STL_TSMGEOPA0ORG_ID
                                     ,STL_TSMNAL_CD
                                     ,STL_TSMNAL_STATE
                                     ,STL_NRCS
                                     ,STL_STATUS
                                     ,STL_IMPORT_SEQ
                                     ,STL_EXPORT_SEQ
                                     ,STL_LAT_DEC_DEGREES
                                     ,STL_LONG_DEC_DEGREES
                                     )
  VALUES (
         p_station_location_rec.STL_SEQ,
         p_station_location_rec.STL_STA_SEQ,
         p_station_location_rec.STL_SSW_SEQ,
         p_station_location_rec.STL_IS_NUMBER,
         p_station_location_rec.STL_ORG_ID,
         p_station_location_rec.STL_POINT_NAME,
         p_station_location_rec.STL_TYPE_CODE,
         p_station_location_rec.STL_SEQUENCE_NUMBER,
         p_station_location_rec.STL_ELEV_DATUM,
         p_station_location_rec.STL_ELEV_METHOD,
         p_station_location_rec.STL_ELEV_MSR_DATE,
         p_station_location_rec.STL_ELEVATION_MSR,
         p_station_location_rec.STL_ELEVATION_UNIT,
         p_station_location_rec.STL_GEOPOS_DATUM
         ,p_station_location_rec.STL_GEOPOS_METHOD
         ,p_station_location_rec.STL_GPS_SCALE
         ,p_station_location_rec.STL_LAT_DIR
         ,p_station_location_rec.STL_LAT_DEGREES
         ,p_station_location_rec.STL_LAT_MINUTES
         ,p_station_location_rec.STL_LAT_SECONDS
         ,p_station_location_rec.STL_LONG_DIR
         ,p_station_location_rec.STL_LONG_DEGREES
         ,p_station_location_rec.STL_LONG_MINUTES
         ,p_station_location_rec.STL_LONG_SECONDS
         ,p_station_location_rec.STL_LAT_LONG_MSR_DATE
         ,p_station_location_rec.STL_LATLONG_ACCURCY
         ,p_station_location_rec.STL_LATLONG_ACCURCY_UNIT
         ,p_station_location_rec.STL_TSMRRR_IS_NUMBER
         ,p_station_location_rec.STL_TSMRRR_ORG_ID
         ,p_station_location_rec.STL_TSMFHU_IS_NUMBER
         ,p_station_location_rec.STL_TSMFHU_ORG_ID
         ,p_station_location_rec.STL_TSMSTATN0IS_NUMBER
         ,p_station_location_rec.STL_TSMSTATN0ORG_ID
         ,p_station_location_rec.STL_TSMWELL_IS_NUMBER
         ,p_station_location_rec.STL_TSMWELL_ORG_ID
         ,p_station_location_rec.STL_TSMGEOPA_IS_NUMBER
         ,p_station_location_rec.STL_TSMGEOPA_ORG_ID
         ,p_station_location_rec.STL_LAT_DEC_MINUTES
         ,p_station_location_rec.STL_LONG_DEC_MINUTES
         ,p_station_location_rec.STL_TSMPIPE_IS_NUMBER
         ,p_station_location_rec.STL_TSMPIPE_ORG_ID
         ,p_station_location_rec.STL_TSMSTATN1ORG_ID
         ,p_station_location_rec.STL_TSMSTATN1IS_NUMBER
         ,p_station_location_rec.STL_TSMGEOPA0IS_NUMBER
         ,p_station_location_rec.STL_TSMGEOPA0ORG_ID
         ,p_station_location_rec.STL_TSMNAL_CD
         ,p_station_location_rec.STL_TSMNAL_STATE
         ,p_station_location_rec.STL_NRCS
         ,p_station_location_rec.STL_STATUS
         ,p_station_location_rec.STL_IMPORT_SEQ
         ,p_station_location_rec.STL_EXPORT_SEQ
         ,p_station_location_rec.STL_LAT_DEC_DEGREES
         ,p_station_location_rec.STL_LONG_DEC_DEGREES
         );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station locations.',NULL);
    RETURN 0;
END create_station_location;

/******************************************************************************************************
  The create_well function inserts known good data into the SIM table SIM_STATION_WELLS
  After the well has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_well(p_well_rec IN SIM_STATION_WELLS%ROWTYPE) RETURN NUMBER IS

  BEGIN
    /* create well using values in well record */
    INSERT INTO SIM_STATION_WELLS  (SSW_SEQ,
                                    SSW_STA_SEQ,
                                    SSW_WELL_STATUS,
                                    SSW_IS_NUMBER,
                                    SSW_ORG_ID,
                                    SSW_NAME,
                                    SSW_WELL_NO,
                                    SSW_WELL_USE,
                                    SSW_DEVELOP_METHOD,
                                    SSW_WELL_CONST_METH,
                                    SSW_WATER_USE,
                                    SSW_GRADIENT_TYPE,
                                    SSW_DEPTH_COMPLETION,
                                    SSW_DEPTH_COMPLETION_UNIT,
                                    SSW_DEPTH_HOLE,
                                    SSW_DEPTH_HOLE_UNIT ,
                                    SSW_WELL_HEIGHT,
                                    SSW_WELL_HEIGHT_UNIT,
                                    SSW_CASING_HEIGHT,
                                    SSW_CASING_HEIGHT_UNIT,
                                    SSW_DISINFECTED,
                                    SSW_CONST_START_DATE,
                                    SSW_CONST_END_DATE,
                                    SSW_INIT_PUMP_DUR,
                                    SSW_INIT_PUMP_DUR_UNIT,
                                    SSW_INIT_PUMP_RATE,
                                    SSW_INIT_PUMP_RATE_UNIT,
                                    SSW_NATURAL_FLOW,
                                    SSW_INIT_BOREHOLE_DIAM,
                                    SSW_INIT_BOREHOLE_DIAM_UNIT,
                                    SSW_TSMSTATN_IS_NUMBER,
                                    SSW_TSMSTATN_ORG_ID,
                                    SSW_STATUS,
                                    SSW_IMPORT_SEQ,
                                    SSW_EXPORT_SEQ
                                   )
                    VALUES (
                                    p_well_rec.SSW_SEQ,
                                    p_well_rec.SSW_STA_SEQ,
                                    p_well_rec.SSW_WELL_STATUS,
                                    p_well_rec.SSW_IS_NUMBER ,
                                    p_well_rec.SSW_ORG_ID,
                                    p_well_rec.SSW_NAME,
                                    p_well_rec.SSW_WELL_NO,
                                    p_well_rec.SSW_WELL_USE,
                                    p_well_rec.SSW_DEVELOP_METHOD,
                                    p_well_rec.SSW_WELL_CONST_METH,
                                    p_well_rec.SSW_WATER_USE,
                                    p_well_rec.SSW_GRADIENT_TYPE,
                                    p_well_rec.SSW_DEPTH_COMPLETION,
                                    p_well_rec.SSW_DEPTH_COMPLETION_UNIT,
                                    p_well_rec.SSW_DEPTH_HOLE,
                                    p_well_rec.SSW_DEPTH_HOLE_UNIT,
                                    p_well_rec.SSW_WELL_HEIGHT,
                                    p_well_rec.SSW_WELL_HEIGHT_UNIT,
                                    p_well_rec.SSW_CASING_HEIGHT,
                                    p_well_rec.SSW_CASING_HEIGHT_UNIT,
                                    p_well_rec.SSW_DISINFECTED,
                                    p_well_rec.SSW_CONST_START_DATE,
                                    p_well_rec.SSW_CONST_END_DATE,
                                    p_well_rec.SSW_INIT_PUMP_DUR,
                                    p_well_rec.SSW_INIT_PUMP_DUR_UNIT,
                                    p_well_rec.SSW_INIT_PUMP_RATE,
                                    p_well_rec.SSW_INIT_PUMP_RATE_UNIT,
                                    p_well_rec.SSW_NATURAL_FLOW,
                                    p_well_rec.SSW_INIT_BOREHOLE_DIAM,
                                    p_well_rec.SSW_INIT_BOREHOLE_DIAM_UNIT,
                                    p_well_rec.SSW_TSMSTATN_IS_NUMBER,
                                    p_well_rec.SSW_TSMSTATN_ORG_ID,
                                    p_well_rec.SSW_STATUS,
                                    p_well_rec.SSW_IMPORT_SEQ,
                                    p_well_rec.SSW_EXPORT_SEQ

                            );
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting well.',NULL);
      RETURN 0;
END create_well;
FUNCTION create_station (p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER IS
BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATIONS (
                            STA_SEQ,
                            STA_ID,
                            STA_NAME,
                            STA_IS_NUMBER,
                            STA_ORG_ID,
                            STA_DESC,
                            STA_TSMORGAN_ORG_ID,
                            STA_TSMORGAN_IS_NUMBER,
                            STA_STATUS,
                            STA_ESTAB_DATE,
                            STA_EXPORT_SEQ,
                            STA_IMPORT_SEQ,
                            STA_TSMVSTC_IS_NUMBER,
                            STA_TSMVSTC_ORG_ID,
                            STA_EPA_KEY_ID,
                            STA_ZID_REL,
                            STA_INFLUENCE_AREA,
                            STA_TRAVEL_DIR,
                            STA_WATER_DEPTH,
                            STA_WATER_DEPTH_UNIT,
                            STA_ECOREGION
                           )
  VALUES (
    p_station_rec.STA_SEQ,                               /* STA_SEQ            */
    p_station_rec.STA_ID,                                /* STA_ID             */
    p_station_rec.STA_NAME,                              /* STA_NAME           */
    p_station_rec.STA_IS_NUMBER,
    p_station_rec.STA_ORG_ID,                             /* STATION ORG ID     */
    p_station_rec.STA_DESC,                              /* STA_DESC           */
    p_station_rec.STA_TSMORGAN_ORG_ID,                   /* TSMORGAN_ORG_ID    */
    p_station_rec.STA_TSMORGAN_IS_NUMBER,                /* TSMORGAN_IS_NUMBER */
    p_station_rec.STA_STATUS,                            /* STA_STATUS         */
    p_station_rec.STA_ESTAB_DATE,                        /* STA_ESTAB_DATE     */
    p_station_rec.STA_EXPORT_SEQ,                        /* STA_EXPORT_SEQ     */
    p_station_rec.STA_IMPORT_SEQ,                        /* STA_IMPORT_SEQ     */
    p_station_rec.STA_TSMVSTC_IS_NUMBER,                 /* STA_TSMVSTC_IS_NUMBER     */
    p_station_rec.STA_TSMVSTC_ORG_ID,                     /* STA_TSMVSTC_ORG_ID */
    p_station_rec.STA_EPA_KEY_ID,
    p_station_rec.STA_ZID_REL,
    p_station_rec.STA_INFLUENCE_AREA,
    p_station_rec.STA_TRAVEL_DIR,
    p_station_rec.STA_WATER_DEPTH,
    p_station_rec.STA_WATER_DEPTH_UNIT,
    p_station_rec.STA_ECOREGION
    );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station.',NULL);
    RETURN 0;
END create_station;
/******************************************************************************************************/

/******************************************************************************************************/

PROCEDURE import_results (
          p_org_id         IN VARCHAR2, 
          p_import_log_seq IN NUMBER,
          p_imp_cfg_seq    IN NUMBER,
          p_test_num       OUT NUMBER)
  IS

  v_group               VARCHAR2(10);
  v_HA_group_id         VARCHAR2(10);

  v_status              VARCHAR2(1);
  v_approved_by         VARCHAR2(20);
  v_approved_on         DATE;

  v_groups              BOOLEAN := FALSE;
  v_char_names          BOOLEAN := FALSE;

  v_InLine              LONG;
  v_bad_parse           BOOLEAN;
  
  v_trans_exist         BOOLEAN;
  
  v_error_code          NUMBER(3);
 
  -- TABLES
  v_line_table             Sim2_Import_Pkg.col_table_type;
  v_personnel_table        Sim2_Import_Pkg.col_table_type;
  v_cporg_table            Sim2_Import_Pkg.col_table_type;
  v_parent_id_table        Sim2_Import_Pkg.col_table_type;
  v_parent_fa_seq_table    Sim2_Import_Pkg.col_table_type;
  v_parent_is_number_table Sim2_Import_Pkg.col_table_type;
  
  v_parent_org_id_table    Sim2_Import_Pkg.col_table_type;
  
  v_remark_codes_table     Sim2_Import_Pkg.col_table_type;
  v_remark_is_table        Sim_import_result_extras.col_table_num_type;
  v_remark_org_table       Sim_import_result_extras.col_table_type;  
  v_persn_is_table         Sim_import_result_extras.col_table_num_type;
  v_persn_org_table        Sim_import_result_extras.col_table_type;
  v_cporg_is_table         Sim_import_result_extras.col_table_num_type;
  v_cporg_org_table        Sim_import_result_extras.col_table_type;
  v_project_table          Sim2_Import_Pkg.col_table_type;
  v_proj_is_table          Sim_import_result_extras.col_table_num_type;
  v_proj_org_table         Sim_import_result_extras.col_table_type;
  v_default_table          Sim_import_result_extras.col_table_type;
  v_pos_table              Sim_import_result_extras.col_table_num_type;
  v_trip_table             Sim_import_result_extras.col_table_type;
  v_stvst_id_table         Sim2_Import_Result_Pkg.t_stvst_id_type;
  
  v_delimiter           VARCHAR2(1);
  v_num_cols            NUMBER(3);
  v_col_seq             NUMBER(10);
  v_dflt_pos            NUMBER(3);

  v_item_name           VARCHAR2(55);
  v_item                VARCHAR2(4000);
  v_converted_item      VARCHAR2(4000);
  v_orig_item           VARCHAR(256);
  v_instr_code          VARCHAR2(15);
  v_comp                NUMBER(1);
  v_gear_type_ok        NUMBER(1);
  v_success             NUMBER(1);
  v_dummy_seq           NUMBER(10);
  v_dummy_char          VARCHAR2(60);
  v_detection           VARCHAR2(40);
  v_detect_check        NUMBER(1);

  v_storet_key          Sim_import_result_extras.t_storet_pk_type;
  v_rci_key             t_rci_pk_type;
  v_lat_key             Sim_import_result_checks.t_lat_pk_type;
  v_lon_key             Sim_import_result_checks.t_lon_pk_type;
  v_temp_key            Sim_import_result_extras.t_storet_pk_type;
  v_dummy_key           Sim_import_result_extras.t_storet_pk_type;

  v_fa_rec              SIM_FIELD_ACTIVITIES%ROWTYPE;
  v_fap_rec             SIM_FIELD_ACTIVITY_PARENT%ROWTYPE;
  v_result_rec          SIM_RESULTS%ROWTYPE;
  v_trip_rec            SIM_TRIPS%ROWTYPE;
  v_visit_rec           SIM_STATION_VISITS%ROWTYPE;
  v_sample_rec          SIM_SAMPLE%ROWTYPE;
  v_tsrsdp_rec          TSRSDP%ROWTYPE;
  v_brg_rec             SIM_BIO_RESULTS_GRP%ROWTYPE;
  v_brgi_rec            SIM_BIO_RESULTS_GRP_INDIVIDUAL%ROWTYPE;
  v_tod_rec             SIM_TRAWL_OPS_DET%ROWTYPE;
  v_rci_rec             SIM_RESULT_CLASS_INDICATOR%ROWTYPE;
  v_aal_rec             SIM_ACTUAL_ACTIVITY_LOC%ROWTYPE;
  v_stl_rec             SIM_STATION_LOCATIONS%ROWTYPE;
  v_stn_rec             SIM_TRAP_NET%ROWTYPE;

  v_fap_seq                 NUMBER(10);
  p_sta_seq                 NUMBER(10);
  v_result_seq_num          NUMBER(5);
  v_assnd_seq_num           NUMBER(5);
  v_sequence_number         NUMBER(4);
  v_well_pipe_num           NUMBER(3);
  v_point_type              VARCHAR2(16);
  v_result_seq              NUMBER(10);
  v_stn_seq                 NUMBER(10);
  v_stl_seq                 NUMBER(10);
  v_activity_cat            VARCHAR2(30);
  v_rel_depth               VARCHAR2(15);
  v_field_set_name          VARCHAR2(30);
  v_field_set_nbr           VARCHAR2(10);
  v_lowq_limit              VARCHAR2(12);
  v_upq_limit               VARCHAR2(12);
  v_detect_limit            VARCHAR2(12);
  v_det_comment             VARCHAR2(254);
  v_det_uom_is_nbr          NUMBER(12);
  v_det_uom_org_id          VARCHAR2(8);
  v_result_cmt              VARCHAR2(4000) := NULL;
  v_result_status           CHAR(1);
  v_batch_id                VARCHAR2(12);
  v_lab_cert                VARCHAR2(1);
  v_stvst_cmt               VARCHAR2(4000);
  v_up_depth                VARCHAR2(8);
  v_low_depth               VARCHAR2(8);
  v_chain_of_custody        VARCHAR2(30);
  v_class_count             VARCHAR2(12);
  v_thermo                  VARCHAR2(5);
  v_halo                    VARCHAR2(5);
  v_pycno                   VARCHAR2(5);
  v_depth_ref_point         VARCHAR2(30);
  v_depth_zone              VARCHAR2(15);
  v_depth_ref_unit          VARCHAR2(3);
  v_QC_indicator            VARCHAR2(1);
  v_cell_type               VARCHAR2(11);
  v_cell_shape              VARCHAR2(18);
  v_bias                    VARCHAR2(12);
  v_cl_corrected            VARCHAR2(1);
  v_add_loc_info            VARCHAR2(254);
  v_tsrmatrx_is_number      NUMBER(12);
  v_tsrmatrx_org_id         VARCHAR2(8);

  v_last_activity_id    VARCHAR2(12) := '$$$$$$$$$$$$';
  v_activity_id         VARCHAR2(12);
  v_new_activity        NUMBER(1);
  v_activity_seq        NUMBER(10);
  v_replicate_nbr       VARCHAR2(3);
  v_last_rep_nbr        VARCHAR2(3);
  v_last_station_id     VARCHAR2(15);
  v_station_id          VARCHAR2(15);
  v_last_trip_seq       NUMBER(10);
  v_last_visit_seq      NUMBER(10);
  v_station_visit       VARCHAR2(8);
  v_act_is_nbr          NUMBER(12);
  v_act_org_id          VARCHAR2(8);

  v_error_cnt           NUMBER(10) := 0;
  v_error_cnt_new       NUMBER(10) := 0;

  v_complete_cnt        NUMBER(10) := 0;
  v_skipped_results_cnt NUMBER(10) := 0;
  v_result_cnt          NUMBER(10) := 0;
  v_import_desc         VARCHAR2(200);

  v_medium              VARCHAR2(20);
  v_last_medium         VARCHAR2(20);
  v_comment             VARCHAR2(1999);
  v_tsmpersn_is_nbr     NUMBER(12);
  v_tsmpersn_org_id     VARCHAR2(8);
  v_tsmcporg_is_nbr     NUMBER(12);
  v_tsmcporg_org_id     VARCHAR2(8);
  v_tsrlbrmk_is_nbr     NUMBER(12);
  v_tsrlbrmk_org_id     VARCHAR2(8);  
  v_type                VARCHAR2(20);
  v_last_type           VARCHAR2(20);

  v_intent              VARCHAR2(20);
  v_taxon_is_nbr        NUMBER(12);
  v_taxon_org_id        VARCHAR2(8);
  v_char_species_number VARCHAR2(8);
  v_allow_species        NUMBER(1);
  v_taxon_species_number VARCHAR2(8);
  v_result_species      VARCHAR2(8);
  v_BRG_species         VARCHAR2(8);
  v_act_species         VARCHAR2(8);
  v_community_is_nbr    NUMBER(12);
  v_community_org_id    VARCHAR2(8);
  v_biopart_is_nbr      NUMBER(12);
  v_biopart_org_id      VARCHAR2(8);
  v_data_file_name_loc  VARCHAR2(60);
  v_gear_deployment     VARCHAR2(1999);
  v_transport_storage   VARCHAR2(1999);
  v_sample_seq          VARCHAR2(10);
  v_ttl_smpl_vol        NUMBER(7,2);
  v_smpl_vol_unit_seq   NUMBER(10);
  v_smpl_wt             NUMBER(7,2);
  v_total_wt_uom        VARCHAR2(3);
  v_parent_ind_code     VARCHAR2(1);

  v_check_trawl_gear    NUMBER(1);
  v_tsmalp_is_nbr       NUMBER(12);
  v_tsmalp_org_id       VARCHAR2(8);
  v_loc_error           NUMBER(1);
  v_fished_duration     NUMBER(4,2);
  v_fished_dur_uom      VARCHAR2(10);
  v_distance_fished     NUMBER(4,2);
  v_dist_fished_uom     VARCHAR2(3);
  v_boat_speed          NUMBER(3,1);
  v_boat_speed_uom      VARCHAR2(10);
  v_curr_group          VARCHAR2(8);
  v_last_group          VARCHAR2(8) := '$$$$$$$$';
  v_group_id            VARCHAR2(8);
  v_new_group           NUMBER(1) := 1;
  v_group_cnt           NUMBER(1) := 0;
  v_brg_id              VARCHAR2(8);
  v_brg_seq             NUMBER(10);
  v_brg_text            VARCHAR2(1999);

  v_brgi_seq            NUMBER(10);
  v_brgi_brg_seq        NUMBER(10);
  v_brg_type            VARCHAR2(32);
  v_value_type_name     VARCHAR2(10);
  v_ttl_nbr_in_grp      NUMBER(8);
  v_ttl_nbr_indiv       NUMBER(5);
  v_indicator           NUMBER(1);
  v_brg_freq_analysis   CHAR(1);
  v_sexname_is_nbr      NUMBER(12);
  v_sexname_org_id      VARCHAR2(8);
  v_lifestage_is_nbr    NUMBER(12);
  v_lifestage_org_id    VARCHAR2(8);
  
  -- jah 2-20-04
  -- SIM_RESULT_CLASS_INDICATOR variables
  v_primary_class         VARCHAR2(9);
  v_secondary_class       VARCHAR2(9);
  v_primary_class_value   VARCHAR2(12);
  v_secondary_class_value VARCHAR2(12);
  v_particle_size_basis   VARCHAR2(40);  -- jah 02-20-04  Added Particle Size Basis
  v_lower_bnd_amt         NUMBER(7,2);
  v_upper_bnd_amt         NUMBER(7,2);
  v_individual_nbr        NUMBER(4);

--  v_pathology           VARCHAR2(1999);
--  v_gen_obs             VARCHAR2(1999);

  v_depth_ref_seq       NUMBER(10);
  v_depth_ref           VARCHAR2(30);
  v_depth_uom           VARCHAR2(3);
  v_depth_range_uom     VARCHAR2(3);
  v_tsmstatn_is_nbr     NUMBER(12);
  v_tsmstatn_org_id     VARCHAR2(8);
  v_visit_start_date    DATE;
  v_visit_start_time    DATE;
  v_visit_end_date      DATE;
  v_visit_end_time      DATE;
  v_visit_seq           NUMBER(10);
  v_tsrstvst_is_nbr     NUMBER(12);
  v_tsrstvst_org_id     VARCHAR2(8);
  v_tsrlab_is_nbr       NUMBER(12);
  v_tsrlab_org_id       VARCHAR2(8);
  v_trip_id             VARCHAR2(15);
  v_trip_name           VARCHAR2(60);
  v_trip_seq            NUMBER(10);
  v_trip_is_nbr         NUMBER(12);
  v_trip_org_id         VARCHAR2(8);
  v_trip_start_date     DATE;
  v_trip_stop_date      DATE;
  v_proj_is_nbr         NUMBER(12);
  v_proj_org_id         VARCHAR2(8);
  v_gear_is_nbr         NUMBER(12);
  v_gear_org_id         VARCHAR2(8);
  v_gearcfg_is_nbr      NUMBER(12);
  v_gearcfg_org_id      VARCHAR2(8);
  v_proc_is_nbr         NUMBER(12);
  v_proc_org_id         VARCHAR2(8);
  v_proc_name           VARCHAR(30);
  v_sample_pres_is_nbr  NUMBER(12);
  v_sample_pres_org_id  VARCHAR2(8);
  v_depth               VARCHAR2(8) := NULL;
  v_dummy_num           NUMBER(8);

  v_start_date          DATE;
  v_start_date_val      VARCHAR2(15);
  v_start_time          DATE;
  v_start_time_zone     VARCHAR2(3);
  v_stop_time_zone      VARCHAR2(3);
  v_end_date            DATE;
  v_end_time            DATE;

  v_trap_net_duration   NUMBER(4,2);
  v_trap_net_dur_unit   VARCHAR2(10);
  v_relative_current    NUMBER(3);
  v_relative_wind       NUMBER(3);
  v_orient_to_cur       VARCHAR2(15);
  v_trap_net_comment    VARCHAR2(254);

  v_func_feed_grp       VARCHAR2(6);
  v_trophic_level       VARCHAR2(4);
  v_taxon_pollution     VARCHAR2(4);

  v_analysis_date       DATE;
  v_analysis_time       DATE;
  v_analysis_time_zone  VARCHAR2(3);
  v_analysis_code       VARCHAR2(15);
  v_result              VARCHAR2(256); -- DCL Changed from 20 (09/05/01)
  v_result_save         VARCHAR2(256); -- char may not be large enough

  v_dummy               BOOLEAN;
  v_units_ok            BOOLEAN;
  v_result_ok           BOOLEAN;
  v_result_msr          NUMBER(1);
  v_tsrchgrp_is_nbr     NUMBER(12);
  v_tsrchgrp_org_id     VARCHAR2(8);
  v_tsrhcsc_is_nbr      NUMBER(12);
  v_tsrhcsc_org_id      VARCHAR2(8);
--  v_row_id              VARCHAR2(80); -- dcl this was originally 15 changed to 80 to accomodate imports with funny row id's
  v_tsrchdef_is_nbr     NUMBER(12);
  v_tsrchdef_org_id     VARCHAR2(8);
  v_char_is_nbr         NUMBER(12);
  v_char_org_id         VARCHAR2(8);
  v_uom_is_nbr          NUMBER(12);
  v_uom_org_id          VARCHAR2(8);
  v_fraction            VARCHAR2(40);
  v_fraction_is_nbr     NUMBER(12);
  v_fraction_org_id     VARCHAR2(8);
  v_value_type          VARCHAR2(10);
  v_stat_type           VARCHAR2(18);
  v_anlpr_is_nbr        NUMBER(12);
  v_anlpr_org_id        VARCHAR2(8);
  
  v_lab_procedure       VARCHAR2(15);  -- jah 3-2-2004 Changed field size from 15 to 40 and back
  v_lab_source          VARCHAR2(12);  -- jah 3-2-2004 Changed field size from 12 to 40 and back
  
  v_labprep_is_nbr      NUMBER(12);
  v_labprep_org_id      VARCHAR2(8);
  v_labprep_id          VARCHAR2(15);
  v_labprep_source      VARCHAR2(12);
  v_duration_basis      VARCHAR2(10);
  v_temp_basis          VARCHAR2(8);
  v_weight_basis        VARCHAR2(12);
  v_precision           VARCHAR2(12);
  v_tsrchdef_is_number  NUMBER(12);
  v_conf_level            VARCHAR2(8);
  v_tsrchgrp_is_number  NUMBER(12);
  v_habit_is_number     NUMBER(12);
  v_volt_is_number      NUMBER(12);
  v_habit_org_id        VARCHAR2(8);
  v_volt_org_id         VARCHAR2(8);
  v_bio_condition_is_nbr   NUMBER(12);
  v_bio_condition_org_id   VARCHAR2(8);
  v_sec_bio_cond_is_nbr    NUMBER(12);
  v_sec_bio_cond_org_id    VARCHAR2(8);
  v_upper_range            NUMBER := 0;
  v_lower_range            NUMBER := 0;
 

  v_act_cnt             NUMBER(10) := 0;
  v_result_value_msr    NUMBER(1);
  v_first_lab_id        VARCHAR2(8);
  v_no_person           NUMBER(2);
  v_no_cporgs           NUMBER(2);
  v_no_parent_ids       NUMBER(2);
  v_no_remark_codes     NUMBER(2);
  v_no_projects         NUMBER(2);

  v_create_act_id_cd    VARCHAR2(1);
  v_habitat             NUMBER(1);
  v_HA_group            NUMBER(1);
  v_HA_char_name        VARCHAR2(60);

  v_pre_check           NUMBER(1);
  v_pre_check_status    VARCHAR2(1);
  v_dup_brgi_success    NUMBER(1);
  v_data_line_number    NUMBER(8);
  v_data_line_name      VARCHAR2(25);

  v_visit_document      VARCHAR2(1999);
  v_activity_document   VARCHAR2(1999);
  v_result_document     VARCHAR2(1999);

  e_fileread            EXCEPTION;
  e_bad_org             EXCEPTION;
  e_no_cfg              EXCEPTION;
  e_e_bad_delimiter     EXCEPTION;
  e_bad_delimiter       EXCEPTION;
  e_NO_DATA_FOUND       EXCEPTION;
  e_project_is          EXCEPTION;
  e_no_sid_seq          EXCEPTION;
  e_bad_station         EXCEPTION;


  CURSOR c_activity_seq IS SELECT SIM_FIELD_ACTIVITIES_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_result_seq IS SELECT SIM_RESULTS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_visit_seq IS SELECT SIM_STATION_VISITS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_trap_net_seq IS SELECT SIM_TRAP_NET_SEQ.NEXTVAL FROM DUAL;

  /* cursors for ocsim */
  CURSOR c_get_next_sample_seq IS SELECT SIM_SAMPLE_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_tsrsdp_values(p_is_nbr IN NUMBER,p_org_id IN VARCHAR2) IS
    SELECT *
      FROM TSRSDP
     WHERE TSRSDP_IS_NUMBER = p_is_nbr
       AND TSRSDP_ORG_ID= RPAD(p_org_id,8);

  CURSOR c_get_next_brg_seq IS SELECT SIM_BRG_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_find_brg_seq(p_curr_group IN VARCHAR2, p_activity_seq IN NUMBER) IS
    SELECT  BRG_SEQ
      FROM  SIM_BIO_RESULTS_GRP
      WHERE BRG_ID_CODE = p_curr_group
        AND BRG_FA_SEQ = P_activity_seq;

  CURSOR c_get_next_brgi_seq IS SELECT SIM_BRGI_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_tod_seq IS SELECT SIM_TRAWL_OPS_DET_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_rci_seq  IS SELECT SIM_RESULT_CLASS_INDICATOR_SEQ.NEXTVAL FROM DUAL;

--  CURSOR c_get_next_gen_txt_seq IS SELECT SIM_GEN_TEXT_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_aal_seq IS SELECT SIM_ACTIVITY_LOC_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_stl_seq IS SELECT SIM_STATION_LOCATIONS_SEQ.NEXTVAL FROM DUAL;--see if exits in database

  CURSOR c_get_import_type(p_imp_cfg_seq IN NUMBER) IS
    SELECT SICC_SID_SEQ
      FROM SIM_IMP_CFG
      WHERE SICC_SEQ = p_imp_cfg_seq;

  /* end new cursors for ocsim */

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT  SIPL_SEQ, SIPL_LINE_TEXT
      FROM  SIM_IMP_LINES
      WHERE SIPL_SIL_SEQ = p_import_seq
      AND   SIPL_LINE_TEXT IS NOT NULL   -- bka  added to eliminate null lines from the import.
      ORDER BY SIPL_LINE_NO;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_PROC_ORDER, SIC_DFLT_POS, SIC_NAME, SIC_SEQ
      FROM SIM_IMP_COLS, SIM_IMP_DEFS, SIM_IMP_CFG
      WHERE SICC_SID_SEQ = SID_SEQ AND SIC_SID_SEQ = SID_SEQ AND SICC_SEQ = p_imp_cfg_seq
      ORDER BY SIC_PROC_ORDER;

  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(RS_SEQ) FROM SIM_FIELD_ACTIVITIES, SIM_RESULTS
      WHERE RS_FA_SEQ = FA_SEQ AND FA_IMPORT_SEQ = p_import_log_seq
      AND FA_STATUS = 'A';

  CURSOR c_trip_dates (p_trip_seq IN NUMBER) IS
      SELECT STP_START_DATE FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

  CURSOR c_create_act_id_cd IS
    SELECT SC_VALUE FROM SIM_CONFIG WHERE SC_ITEM = 'CREATE_ACTIVITY_ID';

  CURSOR c_get_location_info IS
    SELECT STL_SEQ FROM SIM_STATION_LOCATIONS
    WHERE STL_TSMSTATN0IS_NUMBER = v_tsmstatn_is_nbr
    AND STL_TSMSTATN0ORG_ID = RPAD(v_tsmstatn_org_id,8)
    AND STL_IS_NUMBER = v_tsmalp_is_nbr
    AND STL_ORG_ID = RPAD(v_tsmalp_org_id,8);

  CURSOR c_matching_config(p_proc_name IN VARCHAR2) IS
    SELECT TSRGRCFG_IS_NUMBER, TSRGRCFG_ORG_ID, TSRFLDGR_IS_NUMBER, TSRFLDGR_ORG_ID
    FROM TSRGRCFG
    WHERE ID_CODE = RPAD(p_proc_name,10)
    AND TSRGRCFG_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_get_pre_check IS
    SELECT sc_value FROM SIM_CONFIG WHERE sc_item = 'DO_PRE_CHECK';

  CURSOR c_find_pre_check IS
    SELECT SMD_SEQ, SIL_SEQ FROM SIM_IMPORT_DETAILS, SIM_IMPORT_LOG
      WHERE SMD_SIL_SEQ = SIL_SEQ AND SIL_TSMORGAN_ORG_ID = RPAD(p_org_id,8) AND NVL(SIL_PRE_CHECK_FLAG,0) = 1;

  CURSOR c_find_pre_check_logs IS
    SELECT
     SMD_SEQ, SIL_SEQ FROM SIM_IMPORT_DETAILS, SIM_IMPORT_LOG
      WHERE SMD_SIL_SEQ = SIL_SEQ AND SIL_TSMORGAN_ORG_ID = RPAD(p_org_id,8) AND NVL(SIL_PRE_CHECK_FLAG,0) = 1
      AND SIL_SEQ != p_import_log_seq;

  CURSOR c_act_cnt(p_seq NUMBER) IS
    SELECT COUNT(fa_seq)
     FROM sim_field_activities
     WHERE fa_import_seq = p_seq
     AND fa_status = 'A';

  CURSOR c_char_group(p_act_seq IN NUMBER) IS
    SELECT fa_char_group
      FROM sim_field_activities
      WHERE fa_seq = p_act_seq;

  CURSOR c_tsrfdact (
    p_is_number  IN NUMBER,
    p_org_id     IN CHAR) IS
  SELECT TSRFDACT0IS_NUMBER, TSRFDACT0ORG_ID, TSRSTVST_IS_NUMBER, TSRSTVST_ORG_ID
  FROM   TSRFDACT
  WHERE  TSRFDACT_IS_NUMBER = p_is_number
  AND    TSRFDACT_ORG_ID    = p_org_id;
  
  CURSOR c_station_visit (p_ssv_seq IN NUMBER) IS
  SELECT SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID
  FROM   SIM_STATION_VISITS
  WHERE  SSV_SEQ = p_ssv_seq;
  
  CURSOR c_activity_parent (
    p_parent_seq IN NUMBER,
    p_is_number  IN NUMBER,
    p_org_id     IN CHAR,
    p_category   IN VARCHAR2) IS
  SELECT FAP_SEQ
  FROM   SIM_FIELD_ACTIVITY_PARENT, SIM_FIELD_ACTIVITIES
  WHERE  FAP_FA_SEQ        = FA_SEQ
  AND    (FAP_PARENT_FA_SEQ = p_parent_seq)
         OR
         (FAP_PARENT_IS_NUMBER = p_is_number AND FAP_PARENT_ORG_ID = RPAD(p_org_id,8))
  AND    FA_CATEGORY       = p_category
  AND    FA_STATUS         ='A';

  -- jah 3-2-2004
  -- Cursor to determine is a Gear Configuration ID is valid for a given Gear ID
  CURSOR c_get_gear_configuration(
         cp_gear_is_number    NUMBER,
         cp_gear_org_id       VARCHAR2,
         cp_gearcfg_is_number NUMBER,
         cp_gearcfg_org_id    VARCHAR2) IS
  SELECT 1
  FROM   TSRGRCFG
  WHERE  TSRGRCFG_IS_NUMBER = cp_gearcfg_is_number
  AND    TSRGRCFG_ORG_ID    = RPAD(cp_gearcfg_org_id, 8) 
  AND    TSRFLDGR_IS_NUMBER = cp_gear_is_number 
  AND    TSRFLDGR_ORG_ID    = RPAD(cp_gear_org_id, 8);

  r_tsrfdact                c_tsrfdact%ROWTYPE;
  r_station_visit           c_station_visit%ROWTYPE;
  r_activity_parent         c_activity_parent%ROWTYPE;
  r_get_gear_configuration  c_get_gear_configuration%ROWTYPE;

BEGIN
  /* For now we'll delete all pre-check import log info right now. */

  BEGIN
    FOR v_find_pre_check_logs IN c_find_pre_check_logs LOOP
      DELETE SIM_IMPORT_DETAILS WHERE SMD_SEQ = v_find_pre_check_logs.smd_seq;
      DELETE SIM_IMP_LINES WHERE SIPL_SIL_SEQ = v_find_pre_check_logs.sil_seq;
    END LOOP;
    DELETE SIM_IMPORT_LOG WHERE SIL_TSMORGAN_ORG_ID = RPAD(p_org_id,8) AND NVL(SIL_PRE_CHECK_FLAG,0) = 1;

  EXCEPTION WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq, SQLERRM||' in c_find_pre_check',NULL);
  END;

  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  /* Find out if we are pre-checking or not. */
  BEGIN
    OPEN c_get_pre_check;
      FETCH c_get_pre_check INTO v_dummy_char;
      IF c_get_pre_check%NOTFOUND THEN
        v_pre_check := 0;
      ELSE
        IF UPPER(v_dummy_char) = 'YES' THEN
          v_pre_Check := 1;
        ELSE
          v_pre_check := 0;
        END IF;
      END IF;
      v_dummy_char := NULL;
    CLOSE c_get_pre_check;
  EXCEPTION WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq, SQLERRM||' in c_get_pre_check',NULL);
  END;

<<Start_Checking>>
  /* Update the import log with Pre-check info */
  BEGIN
    UPDATE SIM_IMPORT_LOG
      SET SIL_PRE_CHECK_FLAG = v_pre_check
      WHERE SIL_SEQ = v_import_log_seq;
  EXCEPTION WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq, SQLERRM||' in update log',NULL);
  END;

  /* kms - initialize global variables for this time through the package */
  v_line_cnt := 0;
  v_acode_defs_rec :=  v_acode_init_rec;
  v_org_is_nbr := NULL;
  --v_import_log_seq := 0;
  v_trip_qc := 0;
  v_habitat := 0;
  v_cur_trip_is_nbr := NULL;
  v_pre_check_status := 'A';
  gv_last_visit_id := 0;
  v_new_activity := 1;
  v_activity_seq := NULL;
  v_last_activity_id := '$$$$$$$$$$$$';

  Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq, 'Start at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL);

    /* get tsmorgan_is_number */
  BEGIN
    v_org_is_nbr := Sim_import_result_extras.get_org_info(p_org_id,v_import_log_seq);
    IF v_org_is_nbr IS NULL THEN
      RAISE e_bad_org;
    END IF;

  EXCEPTION
    WHEN e_bad_org THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to find the specified org id in local version of STORET.',NULL);
      RAISE NO_DATA_FOUND;
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting org is number.',NULL);
      RAISE NO_DATA_FOUND;
  END;

    /*  get the delimiter */
  BEGIN
    v_delimiter := Sim_import_result_extras.get_delimiter(p_imp_cfg_seq,v_line_cnt,v_import_log_seq);
    IF v_delimiter IS NULL THEN
      RAISE e_no_cfg;
    END IF;

  EXCEPTION
    WHEN e_no_cfg THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Configuration invalid or does not exist - select new configuration or modify existing one.', NULL);
      RAISE NO_DATA_FOUND;
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting delimiter.',NULL);
      RAISE NO_DATA_FOUND;
  END;

    /* SF - 1/8/02 We'll be needing the Import Type */
  OPEN c_get_import_type(p_imp_cfg_seq);
    FETCH c_get_import_type INTO v_sid_seq;
  CLOSE c_get_import_type;
  IF v_sid_seq IS NULL THEN
    RAISE e_no_sid_seq;
  END IF;

    /* get actual positions and default information for items being imported */
  v_success := Sim_import_result_extras.get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table,v_num_cols,v_line_cnt,v_import_log_seq);
  IF v_success != 1 THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to find the import type in the import configurations - select or create a new configuration.',NULL);
  END IF;
        /* loop through the lines in the text table */
  FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP

    BEGIN
        /* read one line */
      v_inline := v_text_line.SIPL_LINE_TEXT;
        /** dcl 09/13/01 null values which might be carried through to the next line **/
      v_comment    := NULL;
      v_result_cmt := NULL;
      v_result_ok := TRUE;
      v_result     := NULL;
      v_last_type := v_type;
      v_type := NULL;
      v_line_cnt := v_line_cnt + 1;
      v_new_activity := 1; -- set to create new activity at beginning of loop
      v_result_seq_num := 0;
      v_status := 'A';
      v_check_trawl_gear := 0;
      v_data_line_number := NULL;
      v_data_line_name := NULL;     
      v_trip_rec.stp_start_date := NULL;
      v_trip_start_date := NULL; 
      v_visit_start_date := NULL;
      v_particle_size_basis := NULL;  -- jah 2-23-04 Added Particle Size Basis     

      -- These variables are initialized to NULL at the end of the loop.
      -- v_uom_is_nbr := NULL;
      -- v_uom_org_id := NULL;
      -- v_char_is_nbr := NULL;
      -- v_char_org_id := NULL;
      
      -- send line to parsing routine:
      BEGIN
        Sim2_Import_Pkg.PARSE_LINE_TABLE(
          v_inline, 
          v_line_table, 
          v_num_cols, 
          v_delimiter,
          v_bad_parse,
          v_import_log_seq
        );

        IF v_bad_parse THEN
          RAISE e_e_bad_delimiter;
        END IF;

      EXCEPTION
        WHEN e_e_bad_delimiter THEN
          RAISE e_bad_delimiter;

        WHEN OTHERS THEN
          Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' parsing one line.',NULL);
          RAISE e_fileread;
      END;

        /* loop through all result columns */
      FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP
        /* get dflt position, item name, and column sequence */
        v_dflt_pos := v_columns.SIC_DFLT_POS;
        v_item_name := v_columns.SIC_NAME;
        v_col_seq := v_columns.SIC_SEQ;
        
        /* if column is included on the import, continue */
        IF v_pos_table(v_dflt_pos) IS NOT NULL THEN

          /* get the actual item, or use default if null */
          IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
            IF LENGTH(v_line_table(v_pos_table(v_dflt_pos))) > 4000 THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': too many characters, truncating to 4000',NULL);
            END IF;
            v_item := SUBSTR(v_line_table(v_pos_table(v_dflt_pos)),1,4000);
          ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL AND v_default_table(v_dflt_pos) IS NOT NULL THEN
            IF LENGTH(v_line_table(v_pos_table(v_dflt_pos))) > 4000 THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': too many characters, truncating to 4000',NULL);
            END IF;
            v_item := SUBSTR(v_default_table(v_dflt_pos),1,4000);
          ELSE
            v_item := NULL;
          END IF;

          -- Does v_item contain only spaces? (or NULL)
          IF (TRIM(v_item) IS NULL) THEN
            -- if yes
            -- by definition V_ITEM is NULL!
            v_item := NULL;
          END IF;

          /* assign item to appropriate variables and take care of any other procedural issues that need to be
             taken care of for or with a particular item */
          IF v_item_name = 'Replicate Number' OR v_item_name = 'Activity Replicate Number' THEN
            IF v_item IS NOT NULL THEN
  						-- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF length(v_item) > 3 THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Replicate Number, '||v_item||', is invalid. It is longer than 3 characters.',v_activity_seq);
                  v_replicate_nbr := NULL;
                  v_status := 'N';
                ELSE
                  v_replicate_nbr := v_item;
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_replicate_nbr := NULL;
  						END IF;
            ELSE
              v_replicate_nbr := NULL;
	          END IF;	


          ----------------										
          -- Activity Type
          ----------------										
          ELSIF v_item_name = 'Activity Type' THEN
            /* Business Rules
            
               Each activity id must be unique on a station visit.  
               
               Activity type and category have been moved up in the processing order 
               so that quality control activities (aka "Trip qcs") do not causes error 
               messages. (Trip qcs do not have station ids or station visits. */

            -- Does the Activity Type field contain anything?
            IF v_item IS NOT NULL THEN
              -- if yes

              -- Remove trailing spaces
              v_item := RTRIM(v_item);

              -- Translate the Activity Type and then determine if the Activity Type is valid
              v_dummy_seq := sim_import_result_extras.validate_alval_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                NULL,
                v_import_log_seq,
                v_line_cnt
              );

              -- Is the Activity Type valid?
              IF v_dummy_seq IS NULL THEN
                -- if no
                -- prevent migration of this record
                v_status := 'N';
                v_type := NULL;

              ELSE

                -- if yes, the Activity Type is valid
                
                -- retrieve the item value from the SIM Allowable Value table.
                v_type := Sim_import_result_extras.get_item_value(
                  v_dummy_seq,     -- this parm contains the record number in the SIM Allowable Value table.
                  v_line_cnt,
                  v_import_log_seq
                );
                
                -- is this a record for a Qualify Control activity?
                IF v_type = 'Trip QC' THEN
                  -- if yes
                  -- tell subsequent processing this record is for a Quality Control activity
								  v_trip_qc := 1;
                ELSE 
                  -- if no
                  -- tell subsequent processing this record is NOT for a Quality Control activity
                  v_trip_qc := 0;  
								END IF;
                
              END IF;
              
            ELSE
              -- if no, the Activity Type field does not contain anything
              
              -- Prevent migration of this record              
              v_type := NULL;
              v_status := 'N';
              
              Sim2_Import_Pkg.write_error(
                v_import_log_seq, 
                'Line ' || v_line_cnt || ': ' || 
                'Activity Type is required, but is not in the import data file.',
                v_activity_seq
              );
            END IF;


          --------------------
          -- Activity Category
          --------------------
          ELSIF v_item_name = 'Activity Category' THEN
  
            IF v_item IS NOT NULL THEN
              IF v_trip_qc != 1 THEN 
                /* get category - check translations and STORET TSMPRMVL table */
                v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq,
                              v_item_name, v_col_seq, 'TSRFDACT','CATEGORY_TYPE_NAME',
                              v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_char IS NOT NULL THEN
                  v_activity_cat := SUBSTR(RTRIM(v_dummy_char),1,30);
                  IF v_type IS NOT NULL THEN
                    v_success := Sim_import_result_checks.check_type_category_match(v_type,v_activity_cat,NULL,v_line_cnt,v_import_log_seq, v_habitat);
                    IF v_success = 2 THEN
                      v_status := 'N';
                      Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Activity category is not valid for the chosen activity type.',NULL);
                    END IF;
                  ELSIF v_sid_seq = 10007 THEN
                    --for WEBSIM the type is inferred from the category
                    IF v_activity_cat = 'Routine Sample'
                      OR  v_activity_cat = 'Field Replicate/Duplicate'
                      OR  v_activity_cat = 'Depletion Replicate'
                      OR  v_activity_cat = 'Integrated Time Series'
                      OR  v_activity_cat = 'Integrated Flow Proportioned'
                      OR  v_activity_cat = 'Integrated Horizontal Profile'
                      OR  v_activity_cat = 'Integrated Vertical Profile'
                      OR  v_activity_cat = 'Composite w/o Parents'
                      OR  v_activity_cat = 'Field Spike'
                      OR  v_activity_cat = 'Field Equipment Rinsate Blank'
                      OR  v_activity_cat = 'Field Blank'
                      OR  v_activity_cat = 'Field Subsample'
                      OR  v_activity_cat = 'Field Split'
                      OR  v_activity_cat = 'Field Surrogate Spike'
                      OR  v_activity_cat = 'Field Ambient Conditions Blank' THEN
                        v_type := 'Sample';
                    ELSIF v_activity_cat = 'Routine Msr/Obs'
                      OR  v_activity_cat = 'Routine Habitat Assessment'
                      OR  v_activity_cat = 'Replicate Msr/Obs'
                      OR  v_activity_cat = 'Replicate Habitat Assessment'
                      OR  v_activity_cat = 'Field Calibration Check' THEN
                        v_type := 'Field Msr/Obs';
                    ELSE
                      v_type := NULL;
                      v_status := 'N';
                      Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Activity category is not valid.',NULL);
                    END IF;
                  ELSE
                    v_status := 'N';
                  END IF;
                  /* replicate check */
                  IF v_replicate_nbr IS NULL AND (v_activity_cat = 'Field Replicate/Duplicate'
                                              OR v_activity_cat = 'Depletion replicate'
                                              OR v_activity_cat = 'Replicate Msr/Obs'
                                              OR v_activity_cat = 'Replicate Habitat Assessment') THEN

                    /* if activity indicates a replicate and it has been left blank, write error message */
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity category indicates the activity is a replicate, but no replicate number has been provided.',NULL);

                  ELSIF v_replicate_nbr IS NOT NULL AND (v_activity_cat != 'Field Replicate/Duplicate'
                                                     AND v_activity_cat != 'Depletion Replicate'
                                                     AND v_activity_cat != 'Replicate Msr/Obs'
                                                     AND v_activity_cat != 'Replicate Habitat Assessment') THEN

                  /* if activity is not a replicate and a replicate number has been given, write error message */
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Activity is not a replicate, but a replicate number has been provided.',NULL);
                    v_replicate_nbr := NULL;
                  END IF;

                ELSE
                  v_status := 'N';
                  v_activity_cat := NULL;
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Activity Category ignored. It is not used when Activity Type = Trip QC.', NULL);
                v_activity_cat := NULL;
              END IF;
            ELSE
              IF NVL(v_type,'x') != 'Trip QC' THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Activity Category is required and has not been provided.', NULL);
                v_activity_cat := NULL;
                v_status := 'N';
              END IF;
            END IF;
            
          ---------------              
          -- Trip QC Type
          ---------------              
          ELSIF v_item_name = 'Trip QC Type' THEN
            IF v_type = 'Trip QC' THEN
              IF v_item IS NOT NULL THEN
              /* get trip QC type - check translations and STORET TSMPRMVL table */
                v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq,
                              v_item_name, v_col_seq, 'TSRFQS','TYPE_NAME',
                              v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_char IS NOT NULL THEN
                  v_activity_cat := RTRIM(v_dummy_char);
        
                  /* make sure valid for activity type */
                  v_success := Sim_import_result_checks.check_type_category_match(v_type,v_activity_cat,NULL,v_line_cnt,v_import_log_seq, v_habitat);
									IF v_success = 2 THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Trip QC Type is not valid for the chosen activity type.',NULL);
                  END IF;
                ELSE
                  v_status := 'N';
                END IF;
              ELSE
                -- Error for null trip QC Type handled after the loop.
                NULL;
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Trip QC Type ignored. It is only used when Activity Type is "Trip QC".', NULL);
              END IF;
            END IF;


          ----------
          -- Trip ID
          ----------
          ELSIF v_item_name = 'Trip ID' THEN
          
              -- jah 2-23-04 trim trailing spaces (but do NOT upper case until after translation!)
              v_item := RTRIM(v_item);
              
              /* check instructions to see what SIM should do */
              v_instr_code := sim_import_result_extras.get_instructions(
                p_imp_cfg_seq, 
                'RES-TRIP', 
                v_dummy_seq,
                v_import_log_seq,
                v_line_cnt
              );
  
              -- Is there a trip instruction?
              IF v_instr_code IS NOT NULL THEN
                -- if yes
  
                -- Is the instruction "Read existing trip ID from data file"?
                IF v_instr_code = 'RES-TRIP-04' THEN
                  -- if yes

                  -- does the data file contain a trip ID?
                  IF v_item IS NOT NULL THEN
                    -- if yes
                    
                    /* kms (6/4/01) 
                       set trip id equal to whatever is in the column (check for translation), 
                       will validate later. */

                    -- translate
                    v_trip_id := sim_import_result_extras.check_translations(
                      v_item, 
                      p_imp_cfg_seq, 
                      v_col_seq, 
                      v_dummy,
                      v_import_log_seq,
                      v_line_cnt,
                      v_activity_seq
                    );

                    -- did translate work properly?
										IF (v_trip_id IS NOT NULL) THEN
                      -- if yes
                  
                      -- jah 2-23-04 Convert to uppercase because STORET GUI converts Trip ID to uppercase
                      v_trip_id := UPPER(v_trip_id);
                  
                    ELSE
                      -- if no
                      
                      -- prevent migration of this record
                  	  v_status := 'N';
                      v_trip_id  := NULL;
                      v_trip_seq := NULL;
									
                  	END IF;

                  ELSE
                    -- if no (i.e., the data file does not contain a trip ID)

                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq, 
                      'Line ' || v_line_cnt || ': ' ||
                      'Trip ID is required and has not been provided. ' ||
                      'Provide the trip id or select an instruction to create trips.',
                      NULL
                    );
                    
                    -- prevent migration of this record
                    v_status   := 'N';
                    v_trip_id  := NULL;
                    v_trip_seq := NULL;
                    
                  END IF;
  
                ELSE
                  -- if no (i.e., the instruction is NOT "Read existing trip ID from data file")
                  
                  -- does the data file contain an activity start date?
                  IF v_start_date IS NOT NULL THEN
                    -- if yes
                
                    -- use the activity start date to get the trip ID
                    v_trip_id := Sim_import_result_extras.get_trip_id(
                      v_instr_code, 
                      v_start_date, 
                      v_trip_rec.stp_start_date,
                      v_trip_rec.stp_end_date, 
                      p_org_id, 
                      v_dummy_seq,
                      v_line_cnt,
                      v_import_log_seq
                    );
                
                  ELSE
                    -- if no (i.e., the data file does NOT contain an activity start date)
                    
                    -- prevent migration of this record
                    v_status   := 'N';
                    v_trip_seq := NULL;
                    v_trip_id  := NULL;
                    
                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq, 
                      'Line ' || v_line_cnt || ': ' ||
                      'Activity Start Date is required to create Trip ID.',
                      NULL
                    );

                  END IF;
                
                END IF;

                -- do we have the trip ID yet?  
                IF v_trip_id IS NOT NULL THEN
                  -- if yes
                  
                  /* kms (6/4/01)
                     Check SIM table to see if trip is there, 
                     if not, check STORET. Either way, add
                     the trip to SIM. */

                  v_trip_rec.stp_trip_id           := v_trip_id;
                  v_trip_rec.stp_trip_name         := v_trip_name;
                  v_trip_rec.stp_end_date          := v_trip_stop_date;
                  v_trip_rec.stp_tsrtrip_is_number := NULL;
                  v_trip_rec.stp_tsrtrip_org_id    := NULL;
                  v_trip_rec.stp_import_seq        := v_import_log_seq;

                  -- try to ensure the trip is in SIM
                  get_or_add_trip_rec(
                    v_trip_rec, 
                    p_org_id, 
                    v_dummy_seq, 
                    v_trip_start_date
                  );
  
                  -- is the Trip in SIM?
                  IF v_trip_rec.STP_SEQ IS NULL THEN
                    -- if no
                    
                    -- prevent migration of this record
                    v_status   := 'N';
                    v_trip_seq := NULL;
                    v_trip_id  := NULL;

                  ELSE
                    -- if yes (i.e., the Trip is in SIM)

                    v_trip_seq             := v_trip_rec.stp_seq;
                    v_storet_key.is_number := v_trip_rec.stp_tsrtrip_is_number;
                    v_storet_key.org_id    := v_trip_rec.stp_tsrtrip_org_id;
  
                    /* kms (6/4/01) - if this is the first time the trip has been processed by this import
                       and it already exists in STORET, load all station visits on this trip in STORET
                       into the SIM table. */

                    -- is the Trip in STORET
                    -- AND do we need to copy the station visits into SIM?
                    IF (v_storet_key.is_number IS NOT NULL)
                      AND (NOT v_trip_table.EXISTS(v_trip_seq)) THEN
                      -- if yes
                      
                      -- load the station visits into SIM
                      load_station_visits(
                        v_trip_seq, 
                        v_storet_key, 
                        p_org_id, 
                        v_stvst_id_table, 
                        v_dummy_seq
                      );
                      
                    END IF;
                    
                    -- indicate the Trip is in SIM, so we load everything again for subsequent records
                    v_trip_table(v_trip_seq) := v_trip_id;

                    -- do we need to see if the activity falls within the start and end dates?
                    IF v_trip_seq IS NOT NULL AND v_start_date IS NOT NULL THEN
                      -- if yes
                    
                      v_success := Sim_import_result_checks.check_trip_dates(
                        v_trip_seq,
                        v_start_date,
                        p_org_id,
                        v_dummy_seq,
                        v_line_cnt,
                        v_import_log_seq
                      );
  
                      -- does the activity falls within trip start and end dates?
                      IF v_success != 0 THEN
                        -- if yes
                        
                        v_trip_is_nbr := v_storet_key.is_number;
                        v_trip_org_id := v_storet_key.org_id;
                        
                      ELSE
                        -- if no
                        
                        -- prevent migration of this record
                        v_status := 'N';
                        v_trip_is_nbr := NULL;
                        v_trip_is_nbr := NULL;
                        v_trip_seq := NULL;
                        
                      END IF;
  
                    ELSIF v_trip_seq IS NOT NULL AND v_start_date IS NULL THEN
                      /* kms whenever the activity start date is null, set it to be the start date for the trip */
                      OPEN c_trip_dates(v_trip_seq);
                      FETCH c_trip_dates INTO v_start_date;
                      CLOSE c_trip_dates;
  
                      v_trip_is_nbr := v_storet_key.is_number;
                      v_trip_org_id := v_storet_key.org_id;
  
                      /* issue warning telling user the start date has been added */
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Activity Start Date has been set to '||v_start_date||', the trip start date.',NULL);
                    ELSE
                      /* no trip, invalidate activity and make sure error message
                        written to the error log */
                      v_trip_is_nbr := NULL;
                      v_trip_is_nbr := NULL;
                      v_trip_seq    := NULL;
                      v_status      := 'N';
  
                      IF v_start_date IS NULL AND v_start_date_val IS NULL THEN
                        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Start Date is required and has not been provided.',NULL);
                      END IF;
  
                    END IF;
  
                  END IF;
  
                END IF;
  
              ELSE
                /* if no instructions provided, write error message and set status to 'N' */
                v_trip_is_nbr := NULL;
                v_trip_org_id := NULL;
                v_trip_seq    := NULL;
                v_status      := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': no instructions provided for trip, update current import configuration or select a new one.',NULL);
              END IF;

              
          ------------------
          -- Trip Start Date
          ------------------
          ELSIF v_item_name = 'Trip Start Date' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_trip_start_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq,v_sid_seq);
                IF v_trip_start_date IS NULL THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trip Start Date format not recognized.',NULL);
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_trip_start_date := NULL;
  						END IF;
            ELSE
              v_trip_start_date := NULL;
            END IF;

              
          -----------------
          -- Trip Stop Date
          -----------------
          ELSIF v_item_name = 'Trip Stop Date' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_trip_stop_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq,v_sid_seq);
                IF v_trip_stop_date IS NULL THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trip Stop Date format not recognized.',NULL);
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
	              v_trip_stop_date := NULL;
  						END IF;
            ELSE
              v_trip_stop_date := NULL;
            END IF;

          ELSIF v_item_name = 'Trip Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 60 THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trip Name is longer that 60 characters.',NULL);
                  v_status := 'N';
                  v_trip_name := NULL;
                ELSE
                  v_trip_name := v_item;
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_trip_name := NULL;
  						END IF;
            ELSE
              v_trip_name := NULL;
            END IF;

          ELSIF v_item_name = 'Station ID' THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
           
              -- jah 2-23-04 trim trailing spaces (but do NOT upper case until after translation!)
              v_item := RTRIM(v_item);

              /* get the station is number - check translations and STORET tables */
              v_storet_key := sim_import_result_extras.validate_storet_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                v_dummy_key,
                p_org_id,
                v_line_cnt,
                v_dummy_seq,
                v_import_log_seq,
                v_station_id
              );

              -- did item translate okay?
              IF v_storet_key.is_number IS NOT NULL THEN
                -- if yes
                v_tsmstatn_is_nbr := v_storet_key.is_number;
                v_tsmstatn_org_id := v_storet_key.org_id;
                -- jah 2-23-04 Convert to uppercase because STORET GUI converts Station ID to uppercase
                v_station_id := UPPER(v_station_id);
              ELSE
                -- if no
                /* station could not be found in storet, prevent export. */
                v_tsmstatn_is_nbr := NULL;
                v_tsmstatn_org_id := NULL;
                v_station_id := NULL;
                v_status := 'N';
              END IF;
            ELSIF v_trip_qc != 1 THEN
              v_tsmstatn_org_id := NULL;
              v_tsmstatn_is_nbr := NULL;
              v_station_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Station ID is required and has not been provided.',NULL);
            ELSE
              v_tsmstatn_org_id := NULL;
              v_tsmstatn_is_nbr := NULL;
              v_station_id := NULL;
            END IF;


          -------------
          -- Point Type
          -------------
          ELSIF v_item_name = 'Point Type' AND v_new_activity = 1 THEN

            -- jah 2-23-04 trim trailing spaces (but do NOT upper case until after translation!)
            v_item := RTRIM(v_item);

            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
            
              IF v_activity_cat = 'Created from Sample' THEN
                v_point_type := NULL;
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq,
                  'Warning: Line '||v_line_cnt||
                  ': Point Type is not used on child (Created from) Samples and will be ignored.',
                  NULL
                );
              ELSE
                v_dummy_seq := sim_import_result_extras.validate_alval_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_activity_seq,
                  v_import_log_seq,
                  v_line_cnt
                );
                IF v_dummy_seq IS NULL THEN
                  v_status := 'N';
                  v_point_type := NULL;
                ELSE
                  v_point_type := sim_import_result_extras.get_allowable_value(
                    v_dummy_seq,
                    v_import_log_seq
                  );
                  -- jah 2-24-04 Convert to uppercase because STORET GUI converts Point Type to uppercase
                  v_point_type := UPPER(v_point_type);
                END IF;
              END IF;
            ELSE
              v_point_type := NULL;
            END IF;

          ELSIF v_item_name = 'Sequence Number' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);           
              IF v_item IS NOT NULL THEN
                IF v_activity_cat = 'Created from Sample' THEN
                  v_sequence_number := NULL;
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Sequence Number is not used on child (Created from) Samples and will be ignored.',NULL);
                ELSE
                  IF v_point_type IS NULL THEN
                    v_sequence_number := NULL;
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||' Point Type is required if Sequence Number is provided.' ,NULL);
                  ELSE
                    IF v_point_type != 'WELL HEAD' AND v_point_type != 'PIPE END' THEN
                      v_sequence_number := sim_import_result_extras.validate_location(v_item, v_point_type, 'SEQ',
                                            v_tsmstatn_is_nbr, v_tsmstatn_org_id, v_activity_seq, v_line_cnt,v_import_log_seq,
                                            v_tsmalp_is_nbr, v_tsmalp_org_id);
                      IF v_sequence_number = -1 THEN
                        v_status := 'N';
                      END IF;
                    ELSE
                      v_sequence_number := 0;
                    END IF;
                  END IF;
                END IF;
              ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'  
                v_sequence_number := NULL;                              
              END IF;
            ELSE
              v_sequence_number := NULL;
            END IF;

          ELSIF v_item_name = 'Well or Pipe ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_activity_cat = 'Created from Sample' THEN
                  v_sequence_number := NULL;
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Well or Pipe ID is not used on child (Created from) Samples and will be ignored.',NULL);
                ELSE
                  IF v_point_type = 'WELL HEAD' OR v_point_type = 'PIPE END' THEN
                    v_well_pipe_num := sim_import_result_extras.validate_location(v_item, v_point_type, 'WELL',
                                         v_tsmstatn_is_nbr, v_tsmstatn_org_id, v_activity_seq, v_line_cnt,v_import_log_seq,
                                            v_tsmalp_is_nbr, v_tsmalp_org_id);
                    IF v_well_pipe_num = -1 THEN
                       v_status := 'N';
                       v_well_pipe_num := NULL;
                    END IF;
                  ELSE
                    v_well_pipe_num := NULL;
                    Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Well or Pipe ID is not used unless point type is WELL HEAD or PIPE END and will be ignored.',NULL);
                  END IF;
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_well_pipe_num := NULL;
  						END IF;
						ELSE
              v_well_pipe_num := NULL;
            END IF;

          ELSIF v_item_name = 'Additional Location Information' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 254 THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer that 254 characters and will be truncated.',v_activity_seq);
                  v_add_loc_info := SUBSTR(v_item,1,254);
                ELSE
                  v_add_loc_info := v_item;
                END IF;
							ELSE
								v_status := 'N';-- v_item translated to null - set status to 'N'
                v_add_loc_info := NULL;
							END IF;
            ELSE
              v_add_loc_info := NULL;
            END IF;

          ELSIF v_item_name = 'Station Visit Arrival Date' THEN
            v_start_date_val := v_item;

            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              /* kms 6/2/01) - this date will be replaced later if the station visit already exists. */
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              
              IF v_item IS NOT NULL THEN
                v_visit_start_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,
                                               v_activity_seq,v_sid_seq);
                IF v_start_date IS NULL THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Station Visit Start Date format not recognized.',NULL);
                END IF;
							ELSE
								v_status := 'N';-- v_item translated to null - set status to 'N'
                v_visit_start_date := NULL;
							END IF;
            ELSE
              /* kms - visit start date has not been included, wait to see if it can be obtained from
                 the station visit (if it already exists) or the trip. */
              v_visit_start_date := NULL;
            END IF;


          -----------------------
          -- Station Visit Number
          -----------------------
          ELSIF v_item_name = 'Station Visit Number' THEN

            -- SLP 06/13/02 Remove trailing spaces
            v_item := RTRIM(v_item);

            /* Business Rule
               Activity records must specify a Station
               and the Station must already be in STORET. */
               
            -- does the import data record specify a Station and is the Station in STORET?
            -- AND is this a quality control record?
            IF (v_tsmstatn_is_nbr IS NOT NULL) AND (v_trip_qc != 1) THEN
              -- if YES (the station is specified and in STORET)
              -- and NO (this is not a quality control record)
              
              /* kms (5/22/01) - set temporary key to station key used in preparation
                 for insuring the station visit is valid for this activity. */
              v_temp_key.is_number := v_tsmstatn_is_nbr;
              v_temp_key.org_id := v_tsmstatn_org_id;

              -- initialize the STORET Station Visit key
              v_storet_key.is_number := NULL;
              v_storet_key.org_id := NULL;

              -- is this an "XML Results" record?
              IF v_sid_seq = 10007 THEN
                -- if yes

                -- Business Rule: 
                --   XML Results import data records must specify the Station Visit Number

                -- Force the import instruction to be "Read existing station visit from data file"
                v_instr_code := 'RES-STVST-04';

              ELSE
                -- if no (i.e., this is NOT an "XML Results" record)
                
                -- get the import instruction for the station visit number
                v_instr_code := sim_import_result_extras.get_instructions(
                  p_imp_cfg_seq,
                  'RES-STVST',
                  v_dummy_seq,
                  v_import_log_seq,
                  v_line_cnt
                );

              END IF;

              -- do we have a Station Visits instruction?
              IF v_instr_code IS NOT NULL THEN
                -- if yes
                
                -- Is the instruction "Read existing station visit from data file"?
                IF v_instr_code = 'RES-STVST-04' THEN
                  -- if yes
                  
                  -- does the import data record contain the station visit number?
                  IF (v_item IS NOT NULL) THEN
                    -- if yes

                    -- translate
                    v_station_visit := sim_import_result_extras.check_translations(
                      v_item, 
                      p_imp_cfg_seq, 
                      v_col_seq, 
                      v_dummy,
                      v_import_log_seq,
                      v_line_cnt,
                      v_activity_seq
                    );

                    -- did the translate work properly?
                    IF (v_station_visit IS NULL) THEN
                      -- if no
                      
                      -- sim_import_result_extras.check_translations is responsible for the error message.

                      -- prevent migration of this record
											v_status := 'N';
                      
                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_station_visit := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                      
										ELSE
                      -- if yes (i.e., the translate worked properly)

                      -- try to retrieve the key to the station visit in SIM_STATION_VISITS table
                      v_visit_seq := Sim_import_result_extras.get_visit_seq(
                        v_storet_key, 
                        v_temp_key, 
                        v_visit_start_date,
                        v_station_visit,
                        v_trip_seq, 
                        v_dummy_seq,
                        v_line_cnt,
                        v_import_log_seq
                      );
  
                      -- Do we need to create a Station Visit record in SIM_STATION_VISITS
                      -- AND is the visit start date NOT in the import data file?
                      IF (v_visit_seq IS NULL) AND (v_visit_start_date IS NULL) THEN
                        -- if yes
                        
                        -- try to use the SIM_TRIPS start date for the Station Visit start date
                        v_visit_start_date := Sim_import_result_extras.get_visit_start_date(
                          v_storet_key,
                          v_trip_seq,
                          v_dummy_seq,
                          v_line_cnt,
                          v_import_log_seq
                        );

                      END IF;
  
                      -- Do we need to create a Station Visit record in SIM_STATION_VISITS
                      -- AND do we still NOT have the visit start date?
                      IF v_visit_seq IS NULL AND v_visit_start_date IS NULL THEN
                        -- if so
                        
                        -- prevent migration of this record
                        v_status := 'N';
                        
                        -- set associated variables to NULL
                        v_visit_seq := NULL;
                        v_station_visit := NULL;
                        v_visit_start_date := NULL;
                        v_storet_key.is_number := NULL;
                        v_storet_key.org_id := NULL;
                        v_tsrstvst_is_nbr := NULL;
                        v_tsrstvst_org_id := NULL;
                        v_temp_key.is_number := NULL;
                        v_temp_key.org_id := NULL;

                        Sim2_Import_Pkg.write_error(
                          v_import_log_seq, 
                          'Line ' || v_line_cnt || ': ' ||
                          'Station Visit Start Date is required when the station visit does not already exist in STORET.  ' ||
                          'Unable to get start date from trip.',
                          NULL
                        );
                      END IF;
                      
                    END IF;
                    
                  ELSE
                    -- if no (i.e., the import data record does NOT contain the station visit number)

                    -- prevent migration of this record
                    v_status := 'N';

                    -- set associated variables to NULL
                    v_visit_seq := NULL;
                    v_station_visit := NULL;
                    v_visit_start_date := NULL;
                    v_storet_key.is_number := NULL;
                    v_storet_key.org_id := NULL;
                    v_tsrstvst_is_nbr := NULL;
                    v_tsrstvst_org_id := NULL;
                    v_temp_key.is_number := NULL;
                    v_temp_key.org_id := NULL;
                    
                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq,
                      'Line '||v_line_cnt||': ' ||
                      'Station Visit is required when instruction to read visits from the import file is chosen.',
                      NULL
                    );
                    
                  END IF;
                  
                ELSE
                  -- if no (i.e., the instruction is NOT "Read existing station visit from data file")

                  -- figure out the station visit number and other data
                  v_error_code := check_existing_visit(
                    v_instr_code,       -- IN
                    v_start_date,       -- IN          activity start date
                    v_start_time,       -- IN          activity start time
                    v_trip_seq,         -- IN
                    v_temp_key,         -- IN
                    v_visit_start_date, -- OUT
                    v_visit_start_time, -- OUT         if v_start_time is NULL, return value is 00:00 (midnight)
                    v_visit_end_date,   -- OUT
                    v_visit_end_time,   -- OUT
                    v_storet_key,       -- OUT
                    v_visit_seq,        -- IN OUT
                    v_stvst_id_table,   -- IN OUT
                    v_station_visit     -- OUT
                  );
                  
                  -- did check_existing_visit work properly?
                  IF (v_error_code is NOT NULL) THEN
                    -- if no

                    -- Prevent migration of this record
                    v_status := 'N';

                    -- set associated variables to NULL
                    v_visit_seq := NULL;
                    v_station_visit := NULL;
                    v_visit_start_date := NULL;
                    v_storet_key.is_number := NULL;
                    v_storet_key.org_id := NULL;
                    v_tsrstvst_is_nbr := NULL;
                    v_tsrstvst_org_id := NULL;
                    v_temp_key.is_number := NULL;
                    v_temp_key.org_id := NULL;

                  END IF;

                END IF;

                -- Do we know the station visit ID?
                IF (v_station_visit IS NOT NULL) THEN
                  -- if yes

                  -- set the STORET station visit key for subsequent processing
                  v_tsrstvst_is_nbr := v_storet_key.is_number;
                  v_tsrstvst_org_id := v_storet_key.org_id;
                  
                  -- re-initialize the temp key
                  v_temp_key.is_number := NULL;
                  v_temp_key.org_id := NULL;
                  
                END IF;

                -- Do we need to create a station visit record?
                IF (v_visit_seq IS NULL) AND (v_station_visit IS NOT NULL) THEN
                  -- if yes
              
                  /* To create a station visit you need the following.
                     o  The Trip on which the station visit occurs
                     o  The Station Visit Start Time (returned by FUNCTION check_existing_visit)
                     o  Activity Start Time Zone
                        (Business Rule:)
                        but ONLY if the import instruction is 
                        'Create one station visit per date and time'
                        and the station visit start time is not midnight. */

                  -- Is the Trip missing?
   								IF (v_trip_seq IS NULL) THEN
                    -- if yes

                    -- Prevent migration of this record
                    v_status := 'N';

                    -- set associated variables to NULL
                    v_visit_seq := NULL;
                    v_station_visit := NULL;
                    v_visit_start_date := NULL;
                    v_storet_key.is_number := NULL;
                    v_storet_key.org_id := NULL;
                    v_tsrstvst_is_nbr := NULL;
                    v_tsrstvst_org_id := NULL;
                    v_temp_key.is_number := NULL;
                    v_temp_key.org_id := NULL;

                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq,
                      'Line ' || v_line_cnt || ': ' ||
                      'Could not create the Station Visit because the Trip ID is missing.',
                      NULL
                    );

                  -- Do we need the Activity Start Time Zone?
                  -- AND is the Visit Start Time something other than midnight?
                  -- AND is the Activity Start Time Zone missing?
                  ELSIF (v_instr_code = 'RES-STVST-03')
                    AND (v_visit_start_time <> TO_DATE('00:00','HH24:MI'))
                    AND (v_start_time_zone IS NULL) THEN
                    -- if yes
                    
                    -- Prevent migration of this record
                    v_status := 'N';

                    -- set associated variables to NULL
                    v_visit_seq := NULL;
                    v_station_visit := NULL;
                    v_visit_start_date := NULL;
                    v_storet_key.is_number := NULL;
                    v_storet_key.org_id := NULL;
                    v_tsrstvst_is_nbr := NULL;
                    v_tsrstvst_org_id := NULL;
                    v_temp_key.is_number := NULL;
                    v_temp_key.org_id := NULL;

                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq,
                      'Line ' || v_line_cnt || ': ' ||
                      'Could not create the Station Visit because the Activity Start Time Zone is missing.',
                      NULL
                    );
                    
                  ELSE  
                    -- We have what we need to create a station visit record
                        
                    -- get the get next station visit key
                    OPEN c_get_next_visit_seq;
                    FETCH c_get_next_visit_seq INTO v_visit_rec.SSV_SEQ;
                    CLOSE c_get_next_visit_seq;
  
                    -- populate the columns in the Station Visit record
                    v_visit_rec.SSV_STP_SEQ := v_trip_seq;
                    v_visit_rec.SSV_ID := v_station_visit;
                    v_visit_rec.SSV_START_DATE := v_visit_start_date;
                    v_visit_rec.SSV_START_TIME := v_visit_start_time;
                    v_visit_rec.SSV_ARRIVAL_TIME_ZONE := v_start_time_zone; -- jah 3-9-2004
                    v_visit_rec.SSV_END_DATE := v_visit_end_date;
                    v_visit_rec.SSV_END_TIME := v_visit_end_time;
                    v_visit_rec.SSV_TSMSTATN_IS_NUMBER := v_tsmstatn_is_nbr;
                    v_visit_rec.SSV_TSMSTATN_ORG_ID := v_tsmstatn_org_id;
                    v_visit_rec.SSV_TSRSTVST_IS_NUMBER := v_storet_key.is_number;
                    v_visit_rec.SSV_TSRSTVST_ORG_ID := v_storet_key.org_id;
                    v_visit_rec.SSV_TSMORGAN_IS_NUMBER := v_org_is_nbr;
                    v_visit_rec.SSV_TSMORGAN_ORG_ID := p_org_id;
                    v_visit_rec.SSV_CREATED_BY_SIM := 1;
                    v_visit_rec.SSV_IMPORT_SEQ  := v_import_log_seq;
                    v_visit_rec.SSV_DOCUMENT := v_visit_document;
  
                    -- Try to insert the Station Visit into SIM_STATION_VISITS
                    v_success := sim_import_result_extras.create_sim_station_visit(
                      v_visit_rec,
                      p_import_log_seq
                    );
                      
                    -- Were we able to insert the Station Visit into SIM_STATION_VISITS?
                    IF (v_success IS NOT NULL) THEN
                      -- if yes
  
                      -- set the SIM_STATION_VISITS key for subsequent processing
                      v_visit_seq := v_visit_rec.SSV_SEQ;
                    ELSE
                      -- if no (i.e., we were NOT able to insert the Station Visit into SIM_STATION_VISITS)
                        
                      -- Prevent migration of this record
                      v_status := 'N';

                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_station_visit := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                      
                    END IF;

                  END IF;

                END IF;
                
              ELSE
                -- if no (i.e., we do NOT have a Station Visits instruction)
                
                -- Prevent migration of this record
                v_status := 'N';

                -- set associated variables to NULL
                v_visit_seq := NULL;
                v_station_visit := NULL;
                v_visit_start_date := NULL;
                v_storet_key.is_number := NULL;
                v_storet_key.org_id := NULL;
                v_tsrstvst_is_nbr := NULL;
                v_tsrstvst_org_id := NULL;
                v_temp_key.is_number := NULL;
                v_temp_key.org_id := NULL;
                
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq,
                  'Line ' || v_line_cnt || ': ' ||
                  'No instructions provided for station visit, update import configuration or select a new one.',
                  NULL
                );
              
              END IF;

              -- Do we know the Station Visit Number?
              IF (v_station_visit IS NOT NULL) THEN
                -- if yes
                
  							/* kms (6/6/01)- make sure station visit start date comes before or is the same as
                   activity start date */
                   
                -- Do we know the key of the Station Visit record in SIM?
                IF v_visit_seq IS NOT NULL THEN
                  -- if yes
                  
                  -- Double check the station visit dates
                  v_success := Sim_import_result_checks.check_visit_dates(
                    v_visit_seq, 
                    v_start_date,
                    p_org_id, 
                    v_dummy_seq,
                    v_line_cnt,
                    v_import_log_seq
                  );
  
                  -- Is there a problem with the station visit dates?
                  IF v_success != 1 THEN
                    -- if yes
                    
                    -- Prevent migration of this import data record
                    v_status:= 'N';

                    -- set associated variables to NULL
                    v_visit_seq := NULL;
                    v_visit_start_date := NULL;
                    v_storet_key.is_number := NULL;
                    v_storet_key.org_id := NULL;
                    v_tsrstvst_is_nbr := NULL;
                    v_tsrstvst_org_id := NULL;
                    v_temp_key.is_number := NULL;
                    v_temp_key.org_id := NULL;
                    
                  END IF;
                  
                END IF;
                
              END IF;  
            
            -- is this a quality control record?
            -- (we already know the Station is not in the import record or is NOT in STORET)
            ELSIF v_trip_qc != 1 THEN
              -- if no
            
              -- Prevent migration of this record
              v_status := 'N';

              -- set associated variables to NULL
              v_visit_seq := NULL;
              v_station_visit := NULL;
              v_visit_start_date := NULL;
              v_storet_key.is_number := NULL;
              v_storet_key.org_id := NULL;
              v_tsrstvst_is_nbr := NULL;
              v_tsrstvst_org_id := NULL;
              v_temp_key.is_number := NULL;
              v_temp_key.org_id := NULL;

              Sim2_Import_Pkg.write_error(
                v_import_log_seq,
                'Line ' || v_line_cnt || ': ' ||
                'Unable to load Station Visit on this activity since Station was either not provided or is invalid.',
                NULL
              );
              
            ELSE
              -- otherwise (i.e., this is a quality control record)
              
              -- Note: 
              -- DO NOT put any code here to prevent migration of this record, 
              -- it is a quality control record!
            
              -- set associated variables to NULL
              v_visit_seq := NULL;
              v_station_visit := NULL;
              v_visit_start_date := NULL;
              v_storet_key.is_number := NULL;
              v_storet_key.org_id := NULL;
              v_tsrstvst_is_nbr := NULL;
              v_tsrstvst_org_id := NULL;
              v_temp_key.is_number := NULL;
              v_temp_key.org_id := NULL;

            END IF;


          --------------						
          -- Activity ID
          --------------						
          ELSIF v_item_name = 'Activity ID' THEN

            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF length(v_item) > 12 THEN
                  v_activity_id := UPPER(RPAD(SUBSTR(v_item,1,12),12)); -- truncate to allowed length
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line '||v_line_cnt||': Activity ID is longer than 12 characters.',
                    v_activity_seq
                  );
                  v_status := 'N';
                ELSE
                  v_activity_id := UPPER(v_item); -- dcl id's need to always be uppers
                END IF;
							ELSE
								v_status := 'N';-- v_item translated to null - set status to 'N'
                v_activity_id := '????????';
							END IF;
            ELSE
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity ID is required and has not been provided.',v_activity_seq);
              v_activity_id := '????????';
              v_status := 'N';
            END IF;

            IF v_last_activity_id IS NULL OR  v_activity_id != v_last_activity_id
              OR NVL(v_replicate_nbr,'`') != NVL(v_last_rep_nbr,'`')
              OR NVL(v_visit_seq,'-1') != NVL(v_last_visit_seq,'-1')
              OR NVL(v_trip_seq,'-1') != NVL(v_last_trip_seq,'-1') THEN

              /* check for activity in sim_field_activities and decide if it needs to be made */
              v_activity_seq := Sim_import_result_checks.check_activity_id(v_activity_id,
                                                    v_replicate_nbr, v_tsmstatn_is_nbr,
                                                    v_trip_seq, v_visit_seq, RPAD(p_org_id,8),
                                                    v_type, v_medium, v_status, v_line_cnt,
                                                    v_import_log_seq,v_assnd_seq_num);

              IF v_activity_seq IS NULL THEN
                /* not first line, but new activity id */
                v_last_activity_id := v_activity_id;
                v_last_rep_nbr := v_replicate_nbr;
                v_last_visit_seq := v_visit_seq;
                v_last_trip_seq := v_trip_seq;
                v_new_activity := 1;
                v_result_seq_num := 0;
              ELSE
                /* activity already exists, do not create */
                v_new_activity := 0;
                v_result_seq_num := v_assnd_seq_num;
                v_last_activity_id := v_activity_id;
                v_last_rep_nbr := v_replicate_nbr;
                v_last_visit_seq := v_visit_seq;
                v_last_trip_seq := v_trip_seq;
              END IF;

            ELSE

              /* Check the medium to make sure they are the same */
              IF v_habitat <> 1 THEN -- If Habitat Assessment then Medium is not required
                IF v_last_medium != v_medium  THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A new medium for this activity has been found.  Only one medium per activity is allowed.',v_activity_seq);
                END IF;
              END IF;
              
              /* Check the activity type to make sure they are the same */              
              IF v_last_type != v_type THEN
                v_status := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A new activity type for this activity has been found.  Only one activity type per activity is allowed.',v_activity_seq);
              END IF;

                -- another line, same activity
              v_new_activity := 0;
              v_last_activity_id := v_activity_id;
              v_last_rep_nbr := v_replicate_nbr;
              v_last_visit_seq := v_visit_seq;
              v_last_trip_seq := v_trip_seq;
            END IF;

            IF v_new_activity = 1 THEN
                -- Get sequence for field activity record (save so can use for related results)
              OPEN c_activity_seq;
                FETCH c_activity_seq INTO v_activity_seq;
                IF c_activity_seq%NOTFOUND THEN
                  v_activity_seq := NULL;
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to get new activity sequence.',v_activity_seq);
                END IF;
              CLOSE c_activity_seq;

            END IF;

              /* kms - update error messages generated for this activity before activity sequence was obtained.
                 Use line nbr to find the error messages that need to be updated */
            UPDATE SIM_IMPORT_DETAILS
                SET smd_import_seq = v_activity_seq
                WHERE smd_desc LIKE '%Line '||v_line_cnt||':%'
                AND smd_sil_seq = v_import_log_seq;

             /*SF - 12/12/02 Check for stations, wells, and locations in SIM. Doing this now because
             we only want to do it for each NEW activity */
            IF v_tsmstatn_is_nbr IS NOT NULL AND v_new_activity = 1 AND v_trip_qc != 1 THEN
              v_success := check_if_station_exists(v_station_id,p_org_id,p_sta_seq);
              IF v_success = 0 THEN
                RAISE e_bad_station;
                v_status := 'N';
              END IF;
            END IF;

          ELSIF (v_item_name = 'Visit Comments' OR v_item_name = 'Station Visit Conditions and Other Comments') AND v_new_activity = 1 THEN
            IF v_trip_qc != 1 AND v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_stvst_cmt := v_item;
							ELSE
								v_status := 'N';-- v_item translated to null - set status to 'N'
                v_stvst_cmt := NULL;
							END IF;
            ELSE
              v_stvst_cmt := NULL;
            END IF;

          ELSIF v_item_name = 'Visit Document/Graphic' THEN
             IF v_item IS NOT NULL THEN
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF sim2_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                  v_visit_document := v_item;
                ELSE
                  v_status := 'N';
                  v_visit_document := NULL;
                END IF;
             	ELSE
						  	v_status := 'N';-- v_item translated to null - set status to 'N'
                v_visit_document := NULL;
							END IF;
            ELSE
              v_visit_document := NULL;
            END IF;

          ELSIF v_item_name = 'Medium' OR v_item_name = 'Activity Medium'  THEN  --need medium on every result to check char group id
            v_last_medium := v_medium;
            IF v_item IS NOT NULL THEN
              /* get medium - check translations and allowable values tables */
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_medium := Sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_status := 'N';
                v_medium := NULL;
              END IF;
        /* SF (2/4/02) Make sure Bio Medium is not on a Field Measurement */
              IF v_medium = 'Biological' and v_type = 'Field Msr/Obs' THEN
                v_status := 'N';
                sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field Measurements cannot be made in a Biological Medium',v_activity_seq);
              END IF;
            ELSE
              v_medium := NULL;
--              v_status := 'N';
--              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Medium is required and has not been provided.',v_activity_seq);
            END IF;

          ELSIF v_item_name = 'Activity Start Date' AND v_new_activity = 1 THEN

              BEGIN
              /* if there is no start date, set status to 'N' so it will not be exported */
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
                IF v_item IS NOT NULL THEN
                  /* get start date using formatting options */
                  v_start_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq,v_sid_seq);
  
                  IF v_start_date IS NULL THEN
                    v_status := 'N';
                    v_start_date_val := v_item;
                  END IF;
								ELSE
									v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_start_date := NULL;
								END IF;
              ELSE
                v_start_date := NULL;
              END IF;

            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_start_date := NULL;
                v_start_time := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting start date.',v_activity_seq);
            END;


          ----------------------  
          -- Activity Start Time
          ----------------------  
          ELSIF v_item_name = 'Activity Start Time' AND v_new_activity = 1 THEN

            BEGIN
              /* if a start time is supplied, format it for STORET */
              IF v_item IS NOT NULL /* AND v_trip_qc != 1 */ THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_result_extras.check_translations(
                  v_item, 
                  p_imp_cfg_seq, 
                  v_col_seq, 
                  v_dummy,
                  v_import_log_seq,
                  v_line_cnt,
                  v_activity_seq
                );
                IF v_item IS NOT NULL THEN
                  /* get start time using formatting options */
                  v_start_time := get_time(
                    v_item_name,
                    v_item,
                    v_col_seq,
                    p_imp_cfg_seq,
                    v_activity_seq
                  );
                  IF v_start_time IS NULL THEN
                    v_status := 'N';
                  END IF;
								ELSE
									v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_start_time := NULL;
								END IF;
              ELSE
                v_start_time := NULL;
              END IF;

            EXCEPTION
              WHEN OTHERS THEN
                v_start_time :=  NULL;
                v_start_date := NULL;
                
                -- Prevent migration of this record
                v_status := 'N';
                
                v_import_desc := 'Line ' || v_line_cnt || ': ' ||
                  'Error in date/time "' || v_start_date || '/' || v_item || '".';

                Sim2_Import_Pkg.WRITE_ERROR(
                  v_import_log_seq, 
                  v_import_desc,
                  v_activity_seq
                );
            END;

            
          ---------------------------
          -- Activity Start Time Zone
          ---------------------------
          ELSIF v_item_name = 'Activity Start Time Zone' AND v_new_activity = 1 THEN
          
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              /* get allowable value sequence for time zone */
              v_dummy_seq := sim_import_result_extras.validate_alval_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                v_activity_seq,
                v_import_log_seq,
                v_line_cnt
              );
              IF v_dummy_seq IS NOT NULL THEN
                v_start_time_zone := sim_import_result_extras.get_item_value(
                  v_dummy_seq,
                  v_line_cnt,
                  v_import_log_seq
                );
              ELSE
                v_status := 'N';
                v_start_time_zone := NULL;
              END IF;
            ELSE
              v_start_time_zone := NULL;
            END IF;

          ELSIF v_item_name = 'Activity End Date' AND v_new_activity = 1 THEN
            BEGIN
              IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
                IF v_item IS NOT NULL THEN
                  v_end_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq,v_sid_seq);
  
                  IF v_end_date IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
								  v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_end_date := NULL;
								END IF;
              ELSE
                v_end_date := NULL;
              END IF;

            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_end_date := NULL;
                v_end_time := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting stop date.',v_activity_seq);
            END;

          ELSIF v_item_name = 'Activity End Time' AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
                IF v_item IS NOT NULL THEN
                  v_end_time := get_time(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);
  
                  IF v_end_time IS NULL THEN
                    v_status := 'N';
                  END IF;
								ELSE
								  v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_end_time := NULL;
								END IF;
              ELSE
                v_end_time := NULL;
              END IF;

          ELSIF v_item_name = 'Activity End Time Zone' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              /* get allowable value sequence for time zone */
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,
                                                 v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_stop_time_zone := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_status := 'N';
                v_stop_time_zone := NULL;
              END IF;
            ELSE
              v_stop_time_zone := NULL;
            END IF;

          ELSIF v_item_name = 'Activity Comments' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                /* comments */
                v_comment := RTRIM(v_item);
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_comment := NULL;
							END IF;
            ELSE
              v_comment := NULL;
            END IF;

          ELSIF v_item_name = 'Depth to Activity' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                BEGIN --check to make sure it's a valid number
                  IF v_trip_qc != 1 AND v_habitat != 1 THEN
                    v_dummy_num := TO_NUMBER(v_item);
                    v_depth := v_item;
                  ELSE
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
                    v_depth := NULL;
                  END IF;
                EXCEPTION
                  WHEN OTHERS THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' is not a valid number of 8 or fewer characters.',v_activity_seq);
                    v_status := 'N';
                END;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_depth := NULL;
							END IF;
            ELSE
              v_depth := NULL;
            END IF;

          ELSIF v_item_name = 'Depth to Activity Units' AND v_new_activity = 1 THEN
            v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1 THEN
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_depth_uom := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_depth_uom := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_depth_uom := NULL;
            END IF;

          ELSIF (v_item_name = 'Relative Depth' OR v_item_name = 'Depth to Activity Relative Depth Indicator') AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_rel_depth := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_rel_depth := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
              END IF;
              v_rel_depth := NULL;
            END IF;

          ELSIF v_item_name = 'Upper Depth' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                BEGIN --check to make sure it's a valid number
                  IF (v_depth IS NOT NULL OR v_rel_depth IS NOT NULL OR v_depth_uom IS NOT NULL) AND v_item IS NOT NULL THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.',v_activity_seq);
                    v_up_depth := NULL;
                  ELSE
                    IF v_trip_qc != 1 AND v_habitat != 1 THEN
                      v_dummy_num := TO_NUMBER(v_item);
                      v_up_depth := v_item;
                    ELSE
                      IF v_item IS NOT NULL THEN
                        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
                        v_up_depth := NULL;
                      END IF;
                    END IF;
                  END IF;
                EXCEPTION
                  WHEN OTHERS THEN
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' is not a valid number of 8 or fewer characters.',v_activity_seq);
                   v_status := 'N';
                END;
  						ELSE
  						  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_up_depth := NULL;
              END IF;
						ELSE
              v_up_depth := NULL;
            END IF;

          ELSIF v_item_name = 'Lower Depth' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                BEGIN --check to make sure it's a valid number
                  IF (v_depth IS NOT NULL OR v_rel_depth IS NOT NULL OR v_depth_uom IS NOT NULL) AND v_item IS NOT NULL THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.',v_activity_seq);
                    v_low_depth := NULL;
                  ELSE
                    IF v_trip_qc != 1 AND v_habitat != 1 THEN
                      v_dummy_num := TO_NUMBER(v_item);
                      v_low_depth := v_item;
                    ELSE
                      IF v_item IS NOT NULL THEN
                        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
                      END IF;
                      v_low_depth := NULL;
                    END IF;
                  END IF;
                EXCEPTION
                  WHEN OTHERS THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' is not a valid number of 8 or fewer characters.',v_activity_seq);
                    v_status := 'N';
                END;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_low_depth := NULL;
  						END IF;
						ELSE
              v_low_depth := NULL;
						END IF;

          ELSIF v_item_name = 'Upper/Lower Depth Units' AND v_new_activity = 1 THEN
            v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
            IF (v_depth IS NOT NULL OR v_rel_depth IS NOT NULL OR v_depth_uom IS NOT NULL) AND v_item IS NOT NULL THEN
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.',v_activity_seq);
              v_depth_range_uom := NULL;
            ELSE
              IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1 THEN
                /* get the depth range units allowable value - check translations table */
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_depth_range_uom := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                ELSE
                  v_depth_range_uom := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
                END IF;
                v_depth_range_uom := NULL;
              END IF;
            END IF;

          ELSIF (v_item_name = 'Depth Measured From' OR v_item_name = 'Depth to Activity Measured From Reference Point') AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_trip_qc != 1 AND v_habitat != 1 THEN
                  IF length(v_item) > 30 THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Depth Measured From is longer than 30 characters and will be truncated',v_activity_seq);
                    v_item := instr(v_item,1,30);
                  END IF;
                  v_depth_ref := UPPER(v_item);
                ELSE
                  IF v_item IS NOT NULL THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
                  END IF;
                  v_depth_ref := NULL;
                END IF;
  						ELSE
  						  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_depth_ref := NULL;
  						END IF;
						ELSE
              v_depth_ref := NULL;
						END IF;

          ELSIF v_item_name = 'Depth Zone Type' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1 THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_depth_zone := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_depth_zone := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
                v_depth_zone := NULL;
              ELSE
                v_depth_zone := NULL;
              END IF;
            END IF;

          ELSIF v_item_name = 'Thermocline' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1 THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_thermo := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_thermo := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
              END IF;
              v_thermo := NULL;
            END IF;

          ELSIF v_item_name = 'Halocline' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1  AND v_habitat != 1THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_halo := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_halo := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
              END IF;
              v_halo := NULL;
            END IF;

          ELSIF v_item_name = 'Pycnocline' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1 THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_pycno := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_pycno := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
              END IF;
              v_pycno := NULL;
            END IF;

          ELSIF v_item_name = 'Personnel' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_no_person := 0;

              /* can have multiple personnel separated by '\' - get number of personnel to expect */
              /* kms - translate function removes all occurences of the delimiter.  By subtracting
                 the length of what it returns from the length of the original, we find the number
                 of delimiters that were in the original.  We then add 1 since we are counting the
                 number of personnel, not the number of delimiters. */
              v_no_person := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;

              /* separate item into personnel using parse procedure if more than 1 person */
              IF v_no_person > 1 THEN
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(v_item,v_personnel_table,v_no_person,'\',v_bad_parse,v_import_log_seq);
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
              ELSE
                v_personnel_table(1) := v_item;
              END IF;

              /* loop through personnel table and add each person in column to the activity */
              FOR v_number IN 1..v_no_person LOOP

                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_personnel_table(v_number),p_imp_cfg_seq,v_col_seq,
                                                     v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);

                IF v_storet_key.is_number IS NULL THEN
                  v_persn_is_table(v_number) := NULL;
                  v_persn_org_table(v_number) := NULL;
                  v_status := 'N';
                ELSE
                  v_persn_is_table(v_number) := v_storet_key.is_number;
                  v_persn_org_table(v_number) := v_storet_key.org_id;
                END IF;

              END LOOP;

            ELSE
              v_no_person := 0;
              v_tsmpersn_is_nbr := NULL;
              v_tsmpersn_org_id := NULL;
            END IF;

          ------------------------------
          -- Cooperating Organization --
          ------------------------------
          ELSIF v_item_name = 'Cooperating Organization' AND v_new_activity = 1 THEN
          
            -- does the cooperating organization field have something in it?
            IF v_item IS NOT NULL THEN
              -- if yes
              
              /*
                Note: The Cooperating Organization field can have multiple values
                      seperated by \ (i.e., backward slashes). Count the number of 
                      backward slashes in the item and add one to determine how 
                      many cooperating organizations the field contains.
              */
              
              -- Count the backward slashes and add one
              v_no_cporgs := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;
              
              -- Does this field contain more than one cooperating organization?
              IF v_no_cporgs > 1 THEN
                -- if yes
                -- Put the cooperating organizations into an array (v_cporg_table)
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(
                    v_item,
                    v_cporg_table,
                    v_no_cporgs,
                    '\',
                    v_bad_parse,
                    v_import_log_seq
                  );
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
              ELSE
                -- if no (it contains only one cooperating organization)
                -- Put the cooperating organization into an array (v_cporg_table)
                v_cporg_table(1) := v_item;
              END IF;

              -- loop through cooperating organizations array
              FOR v_number IN 1..v_no_cporgs LOOP

                -- get the associated STORET key
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_cporg_table(v_number),
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq,
                  v_dummy_char
                );

                -- did we get the associated STORET key?
                IF v_storet_key.is_number IS NOT NULL THEN
                  -- if yes
                  -- Add the cooperating organization to the activity
                  v_cporg_is_table(v_number) := v_storet_key.is_number;
                  v_cporg_org_table(v_number) := v_storet_key.org_id;
                ELSE
                  -- if no
                  -- Do not add the cooperating organization to the activity
                  v_cporg_is_table(v_number) := NULL;
                  v_cporg_org_table(v_number) := NULL;
                  v_status := 'N';
                END IF;

              END LOOP;

            ELSE
              -- if no (the cooperating organization field is NULL)
              -- set everything to zero or NULL, as appropriate
              v_no_cporgs := 0;
              v_tsmcporg_is_nbr := NULL;
              v_tsmcporg_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Activity Document/Graphic' THEN
            IF v_item IS NOT NULL THEN 
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN 
                IF sim2_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                  v_activity_document := v_item;
                ELSE
                  v_status := 'N';              --Invalid document.  Set to 'N'.
                  v_activity_document := NULL;  
                END IF;    
              ELSE
                v_status := 'N';                --Translates to NULL.  Set to 'N'.
                v_activity_document := NULL;
              END IF;
            ELSE
              v_activity_document := NULL;
            END IF;
            
          ---------------------------------
          -- Sample Collection Procedure ID
          ---------------------------------

          ELSIF v_item_name = 'Sample Collection Procedure ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := RTRIM(v_item);

              IF v_trip_qc != 1 AND v_habitat != 1 AND v_type != 'Field Msr/Obs' THEN
                /* get the procedure is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );

                IF v_storet_key.is_number IS NOT NULL THEN
                  v_proc_is_nbr := v_storet_key.is_number;
                  v_proc_org_id := v_storet_key.org_id;
                  v_proc_name := v_dummy_char;
                   -- check if trawl gear is used and get tsmalp_is_nbr
                   -- 1 = trawl, 2 = trap/net, 0 = all others
                  v_check_trawl_gear := Sim_import_result_checks.check_trawl_gear(
                    v_proc_is_nbr, 
                    v_proc_org_id,
                    v_activity_seq,
                    v_line_cnt,
                    v_import_log_seq
                  );
                ELSE
                  v_proc_org_id := NULL;
                  v_proc_is_nbr := NULL;
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line '||v_line_cnt||':  Sample Collection Procedure ID is required.',
                    v_activity_seq
                  );
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line '||v_line_cnt||
                  ' Sample Collection Procedure ID is not used for QC, Field Measurements, or Habitat Assessment ',
                  v_activity_seq
                );
                v_proc_is_nbr := NULL;
                v_proc_org_id := NULL;
                v_gearcfg_is_nbr := NULL;
                v_gearcfg_org_id := NULL;
                v_gear_is_nbr := NULL;
                v_gear_org_id := NULL;
              END IF;
            ELSE
              v_proc_is_nbr := NULL;
              v_proc_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Gear ID' AND v_new_activity = 1 THEN
            -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
            v_item := RTRIM(v_item);
            
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1  AND v_type != 'Field Msr/Obs' THEN

              /* get the gear is number - check translations and STORET tables */
              v_storet_key := sim_import_result_extras.validate_storet_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                v_dummy_key,
                p_org_id,
                v_line_cnt,
                v_activity_seq,
                v_import_log_seq, 
                v_dummy_char
              );

              IF v_storet_key.is_number IS NOT NULL THEN
                v_gear_is_nbr := v_storet_key.is_number;
                v_gear_org_id := v_storet_key.org_id;
              ELSE
                v_gear_org_id := NULL;
                v_gear_is_nbr := NULL;
                v_status := 'N';
              END IF;

            ELSE
              /* if gear id is missing, write error to error log */
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line '||v_line_cnt||': '||v_item_name||
                  ' ignored - it is not used for QC, Field Measurements, or Habitat Assessment.',
                  v_activity_seq
                );
              END IF;
              v_gear_org_id := NULL;
              v_gear_is_nbr := NULL;
            END IF;

          ELSIF v_item_name = 'Gear Configuration ID' AND v_new_activity = 1 THEN
            -- jah 2-24-04 Remove trailing space but do NOT convert to uppercase
            v_item := RTRIM(v_item);
            
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1  AND v_type != 'Field Msr/Obs' THEN
              /* kms - assign gear to temporary key that will limit the valid gear configurations
                 to those that are of this gear type */
              v_temp_key.is_number := v_gear_is_nbr;
              v_temp_key.org_id := v_gear_org_id;
              /* get the gear configuration is number - check translations and STORET tables */
              v_storet_key := sim_import_result_extras.validate_storet_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                v_temp_key,
                p_org_id,
                v_line_cnt,
                v_activity_seq,
                v_import_log_seq, 
                v_dummy_char
              );
              
              IF v_storet_key.is_number IS NOT NULL THEN
                v_gearcfg_is_nbr := v_storet_key.is_number;
                v_gearcfg_org_id := v_storet_key.org_id;
              ELSE
                v_gearcfg_is_nbr := NULL;
                v_gearcfg_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
            
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq,
                  'Warning: Line '||v_line_cnt||': '||v_item_name||
                  ' is not used for QC, Field Measurements, or Habitat Assessment',
                  v_activity_seq
                );
              END IF;
              v_gearcfg_is_nbr := NULL;
              v_gearcfg_org_id := NULL;
            END IF;

            /* kms - null out temporary key */
            v_temp_key.is_number := NULL;
            v_temp_key.org_id := NULL;

          ELSIF (v_item_name = 'Gear Deployment Comments' OR v_item_name = 'Sample Collection Procedure Gear Deployment Comments')
                 AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_trip_qc != 1 AND v_habitat != 1  AND v_type != 'Field Msr/Obs' THEN
                  IF LENGTH(v_item) > 1999 THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 1999 characters and will be truncated.',v_activity_seq);
                    v_item := substr(v_item,1,1999);
                  END IF;
                  v_gear_deployment := v_item;
                ELSE
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' is not used for QC, Field Measurements, or Habitat Assessment',v_activity_seq);
                  v_gear_deployment := NULL;
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_gear_deployment := NULL;
							END IF;
            ELSE
              v_gear_deployment := NULL;
            END IF;

          ELSIF v_item_name = 'Sample Preservation Transport and Storage ID' OR v_item_name = 'Sample Transport and Storage ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := RTRIM(v_item);
              
              IF v_trip_qc != 1 AND v_habitat != 1  AND v_type != 'Field Msr/Obs' THEN

                /* get the sample preservation proc is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );

                IF v_storet_key.is_number IS NOT NULL THEN
                  v_sample_pres_is_nbr := v_storet_key.is_number;
                  v_sample_pres_org_id := v_storet_key.org_id;
                ELSE
                  v_sample_pres_is_nbr := NULL;
                  v_sample_pres_org_id := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq,
                  'Warning: Line '||v_line_cnt||': '||v_item_name||
                  ' is not used for QC, Field Measurements, or Habitat Assessment',
                  v_activity_seq
                );
                v_sample_pres_is_nbr := NULL;
                v_sample_pres_org_id := NULL;
              END IF;
            ELSE
              v_sample_pres_is_nbr := NULL;
              v_sample_pres_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Sample Transport and Storage Comments' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_trip_qc != 1 AND v_habitat != 1  AND v_type != 'Field Msr/Obs' THEN
                  IF LENGTH(v_item) > 1999 THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 1999 characters and will be truncated.',v_activity_seq);
                    v_item := substr(v_item,1,1999);
                  END IF;
                  v_transport_storage := v_item;
                ELSE
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' is not used for QC, Field Measurements, or Habitat Assessment',v_activity_seq);
                  v_transport_storage := NULL;
                END IF;
							ELSE
								v_status := 'N';-- v_item translated to null - set status to 'N'
                v_transport_storage := NULL;
							END IF;
            ELSE
              v_transport_storage := NULL;
            END IF;

          ELSIF (v_item_name = 'QC Indicator' OR v_item_name = 'Activity QC Indicator') AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_QC_indicator := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_QC_indicator := 'N';
                v_status := 'N';
              END IF;
            ELSE
              v_QC_indicator := NULL;
            END IF;

          ELSIF (v_item_name = 'Sample Matrix' OR v_item_name = 'Activity Sample Matrix') AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_storet_key := validate_sample_matrix(v_item, p_imp_cfg_seq, v_activity_seq,
                                  v_import_log_seq,v_col_seq,v_line_cnt);
              IF v_storet_key.is_number IS NULL THEN
                v_tsrmatrx_is_number := NULL;
                v_tsrmatrx_org_id := NULL;
                v_status := 'N';
              ELSE
                v_tsrmatrx_is_number := v_storet_key.is_number;
                v_tsrmatrx_org_id := v_storet_key.org_id;
              END IF;
            ELSE
              v_tsrmatrx_is_number := NULL;
              v_tsrmatrx_org_id := NULL;
            END IF;


          ELSIF v_item_name = 'Chain of Custody ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 30 characters and will be truncated.',v_activity_seq);
                  v_item := substr(v_item,1,30);
                END IF;
                v_chain_of_custody := v_item;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_chain_of_custody := NULL;
							END IF;
            ELSE
              v_chain_of_custody := NULL;
            END IF;

          ELSIF v_item_name = 'Field Set ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := UPPER(RTRIM(v_item)); -- 06/13/03 SLP - Set to uppercase and remove trailing spaces.
                IF length(v_item) > 10 THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Field Set ID, '||v_item||', is longer than 10 characters and will be truncated',v_activity_seq);
                  v_item := instr(v_item,1,10);
                END IF;
                IF v_trip_qc != 1 THEN
                  v_field_set_nbr := v_item;
                ELSE
                  v_field_set_nbr := NULL;
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_field_set_nbr := NULL;
  						END IF;
						ELSE
              v_field_set_nbr := NULL;
						END IF;

          ELSIF v_item_name = 'Field Set Name' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_trip_qc != 1 THEN
                  v_field_set_name := v_item;
                ELSE
                  v_field_set_name := NULL;
                END IF;
  						ELSE
  							v_status := 'N';-- v_item translated to null - set status to 'N'
                v_field_set_name := NULL;
  						END IF;
						ELSE
              v_field_set_name := NULL;
						END IF;

          ELSIF v_item_name = 'Characteristic Detection Condition' OR v_item_name = 'Detection Condition' THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 AND v_habitat != 1 THEN

              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_detection := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_detection := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Trip QC or Habitat Assessment.',v_activity_seq);
              END IF;
              v_thermo := NULL;
            END IF;

          ----------------------
          -- Characteristic Name
          ----------------------
          ELSIF v_item_name = 'Characteristic Name' THEN
            --If they have already imported a line with a Char Group the whole thing bombs. Return to Form...
            v_char_names := TRUE;
            IF v_item IS NOT NULL THEN
            
              IF v_activity_cat = 'Portable Data Logger' THEN
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );

                IF v_storet_key.is_number IS NOT NULL AND v_storet_key.org_id IS NOT NULL THEN
                  v_success := sim_import_result_checks.check_data_logger_item(
                    v_dummy_char,
                    v_storet_key.is_number,
                    v_line_cnt,
                    v_storet_key.org_id,
                    v_activity_seq,
                    v_import_log_seq
                  );
                  IF v_success != 1 THEN
                    v_status := 'N';
                  END IF;                       
                ELSE
                  v_status := 'N';-- v_item translated to null - set status to 'N'
                END IF;
      
              -- if bio result type is Multi Taxon PC then char has to be of type TAXA. This is checked
              -- when validating Subject Taxon, so validate char as if it were Subject Taxon
              ELSIF v_brg_type = 'Multi-Taxon Population Census' THEN
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  'Subject Taxon',
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );
                IF v_storet_key.is_number IS NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Characteristic must be of TAXA type for Multi-Taxon Population Census.',v_activity_seq);
                  v_status := 'N';
                END IF;
                
              ELSIF v_brg_type = 'Single Taxon Frequency Classes' THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Characteristic is not used for Single Taxon Frequency Classes. Use Class Descriptors instead.',v_activity_seq);
                v_storet_key.is_number := NULL;
                v_storet_key.org_id := NULL;
                
              ELSE
              
                /* get the characteristic is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );
                
              END IF;

              IF v_storet_key.is_number IS NOT NULL THEN
                v_char_is_nbr := v_storet_key.is_number;
                v_char_org_id := v_storet_key.org_id;

                /** Assign Result type value to global Package variable **/
                v_char_result_type := get_char_result_type(v_char_is_nbr);
                /* Further check of char type */
                IF v_brg_type = 'Single Taxon Individuals' THEN
                  v_success := Sim_import_result_checks.check_individual_type(
                    v_char_is_nbr,
                    p_org_id, 
                    v_activity_seq,
                    v_line_cnt, 
                    p_import_log_seq
                  );
                  IF v_success != 1 THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic is of wrong type for Single Taxon Individuals.',v_activity_seq);
                  END IF;
                END IF;
                
              ELSE
                
                v_status := 'N';-- v_item translated to null - set status to 'N'
                
              END IF;

            END IF;


          -----------------------------------
          -- Result Value Units
          -- OR
          -- Characteristic Result Value Unit
          -----------------------------------
          ELSIF v_item_name = 'Result Value Units' OR v_item_name = 'Characteristic Result Value Unit' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line '||v_line_cnt||
                  ': Result Value Units ignored. Not used for Single Taxon Frequency Classes.',
                  v_activity_seq
                );
              ELSE
              /* get the unit is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);

                IF v_storet_key.is_number IS NULL THEN
                  /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                  v_uom_is_nbr := NULL;
                  v_uom_org_id := NULL;
                  v_status := 'N';
                ELSE
                  v_uom_is_nbr := v_storet_key.is_number;
                  v_uom_org_id := v_storet_key.org_id;
                  -- In STORET V2 any char can have any UOM, so we no longer need to check the type...
                END IF;
              END IF;
            ELSE
              IF NVL(v_brg_type,'x') != 'Single Taxon Frequency Classes' THEN
                v_uom_is_nbr := NULL;
                v_uom_org_id := NULL;
              END IF;
            END IF;

          ELSIF v_item_name = 'Sample Fraction' OR v_item_name = 'Characteristic Sample Fraction' THEN
            /* kms - validate fraction only if loaded from txt, not the analysis code. */
            IF v_item IS NOT NULL AND v_habitat != 1 THEN
              v_fraction := sim_import_result_extras.validate_fraction(v_item,p_imp_cfg_seq,v_col_seq,v_char_is_nbr,v_activity_seq,v_import_log_seq,v_line_cnt,v_storet_key);

              IF v_fraction IS NULL THEN
                v_status := 'N';
                v_fraction_is_nbr := NULL;
                v_fraction_org_id := NULL;
              ELSIF v_fraction = 'NR' THEN  --not required
                v_fraction_is_nbr := NULL;
                v_fraction_org_id := NULL;
                v_fraction := NULL;
              ELSE
                v_fraction_is_nbr := v_storet_key.is_number;
                v_fraction_org_id := v_storet_key.org_id;
              END IF;

            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_fraction := NULL;
              v_fraction_is_nbr := NULL;
              v_fraction_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Value Type' OR v_item_name = 'Characteristic Value Type' THEN
            IF v_item IS NOT NULL AND v_habitat != 1 THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NULL THEN
                v_status := 'N';
                v_value_type := NULL;
              ELSE
                v_value_type := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              END IF;

            ELSIF v_habitat = 1 THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              v_value_type := NULL;
            ELSE
--              v_value_type := NULL;
              v_value_type := 'Actual';
--              Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Null value for value type has resulted in the assignment of a default value of Actual.',v_activity_seq);
            END IF;

          ELSIF v_item_name = 'Statistic Type' OR v_item_name = 'Characteristic Result Statistic Type' THEN
            IF v_item IS NOT NULL AND v_habitat != 1  THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_stat_type := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_stat_type := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_stat_type := NULL;
            END IF;

          ELSIF v_item_name = 'Weight Basis' OR v_item_name = 'Characteristic Result Weight Basis' THEN
            IF v_item IS NOT NULL AND v_habitat != 1  THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_weight_basis := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_weight_basis := NULL;
                v_status := 'N';
              END IF;

            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_weight_basis := NULL;
            END IF;

          ELSIF v_item_name = 'Duration Basis' OR v_item_name = 'Characteristic Result Duration Basis'THEN
            IF v_item IS NOT NULL AND v_habitat != 1  THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_duration_basis := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_duration_basis := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_duration_basis := NULL;
            END IF;

          ELSIF v_item_name = 'Temperature Basis' OR v_item_name = 'Characteristic Result Temperature Basis' THEN
            IF v_item IS NOT NULL AND v_habitat != 1  THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_temp_basis := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_temp_basis := NULL;
                v_status := 'N';
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_temp_basis := NULL;
            END IF;

          ELSIF v_item_name = 'Result Value' OR v_item_name = 'Characteristic Result Value' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              /*SF - 10/12/01 If it's a PV type char we'll check the translation later.
                If we do it now, and it's empty we get a null value in v_result. */
              IF v_char_result_type IN ('PV  ','HA  ')  THEN
                v_result := v_item;
              ELSE
                v_result := sim_import_result_extras.check_translations(
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy,
                  v_import_log_seq,
                  v_line_cnt,
                  v_activity_seq
                );
								IF v_result IS NULL THEN
								  v_status := 'N'; --v_result translated to null - set status to 'N'
								END IF;
              END IF;

              IF v_char_result_type NOT IN ('PV  ','HA  ','TEXT') 
               AND v_result NOT IN ('*Non-detect','*Present <QL','*Present >QL','*Not Reported','*Present') THEN  
                 /* pwh (2/18/04) - check result text to make sure it is a number, write error if the value
                 is too large, too precise, or not a number */               
                v_result := GET_VALUE_MSR(
                  v_result,
                  v_result_ok,
                  v_activity_seq,
                  v_line_cnt,
                  v_import_log_seq
                );
               
                IF v_result_ok = FALSE THEN
                  v_status := 'N';
                END IF;
                
              END IF;              
              
              IF v_brg_type != 'Single Taxon Frequency Classes' THEN
                v_result_seq_num := v_result_seq_num + 1;
              END IF;
            ELSE
               IF NVL(v_brg_type, '1') != 'Single Taxon Frequency Classes' THEN
                 /* Single Taxon Frequency Classes do not have result values. */
                 IF NVL(v_detection, 'Detected and Quantified') = 'Detected and Quantified' THEN
                  /* If detection condition is not passed in or is set to Detected and Quantified, a result value is required. */
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Warning: Line '||v_line_cnt||
                    ': Result Value has not been provided. The activity will be loaded without this result.',
                    v_activity_seq
                  );
                  /* slp - 05/21/03 Moved skipped cnt inside if statement and added NVL to Detection IF statement. */
                  /* kms - there is no result reported, increment counter */
                  v_skipped_results_cnt := v_skipped_results_cnt + 1;
                 END IF;
               END IF;
            END IF;

          ELSIF v_item_name = 'Laboratory ID' THEN
            IF v_item IS NOT NULL AND v_habitat != 1 THEN
              -- jah 2-24-04 Remove trailing space but do NOT convert to uppercase
              v_item := RTRIM(v_item);
              
              /* get the lab is number - check translations and STORET tables */
              v_storet_key := sim_import_result_extras.validate_storet_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                v_dummy_key,
                p_org_id,
                v_line_cnt,
                v_activity_seq,
                v_import_log_seq, 
                v_dummy_char
              );

              IF v_storet_key.is_number IS NOT NULL THEN
                v_tsrlab_is_nbr := v_storet_key.is_number;
                v_tsrlab_org_id := v_storet_key.org_id;
              ELSE
                v_tsrlab_is_nbr := NULL;
                v_tsrlab_org_id := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_tsrlab_is_nbr := NULL;
              v_tsrlab_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Analysis Date' OR v_item_name = 'Laboratory Analysis Date' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL AND v_habitat != 1 THEN
                v_analysis_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq,v_sid_seq);
                IF v_analysis_date IS NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Analysis Date format not recognized.',v_activity_seq);
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
    						ELSE
    						  v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_analysis_date := NULL;
                END IF;
              END IF;
						ELSE
              v_analysis_date := NULL;
						END IF;

          ELSIF v_item_name = 'Analysis Time' OR v_item_name = 'Laboratory Analysis Time' THEN
            IF v_analysis_date IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              IF v_item IS NOT NULL THEN
                v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
                IF v_item IS NOT NULL THEN
                    IF v_habitat != 1 THEN
                      v_analysis_time := get_time(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);
                      IF v_analysis_time IS NULL THEN
                        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Analysis Time format not recognized.',v_activity_seq);
                        v_status := 'N';
                      END IF;
                    ELSE 
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
                      v_analysis_time := NULL;
                    END IF;
                ELSE
								  v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_analysis_time := NULL;
                END IF;
    					ELSE
                v_analysis_time := NULL;
							END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not allowed unless Analysis Date is provided.',v_activity_seq);
              END IF;
                v_analysis_time := NULL;
            END IF;


          ELSIF v_item_name = 'Analysis Time Zone' OR v_item_name = 'Laboratory Analysis Time Zone' THEN
            IF v_analysis_time IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_item IS NOT NULL AND v_habitat != 1 THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NULL THEN
                  v_status := 'N';
                ELSE
                  v_analysis_time_zone := sim_import_result_extras.get_allowable_value(v_dummy_seq,v_import_log_seq);
                END IF;
              ELSIF v_habitat = 1 THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
                v_analysis_time_zone := NULL;
              ELSE
                v_analysis_time_zone := NULL;
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not valid unless Analysis Time is provided.',v_activity_seq);
              END IF;
              v_analysis_time_zone := NULL;
            END IF;

          ELSIF v_item_name = 'Lab Remark Codes' THEN
            IF v_item IS NOT NULL THEN
              v_no_remark_codes := 0;

              /* can have multiple Remark Codes separated by '\' - get number of Remark Codes to expect */
              /* kms - translate function removes all occurences of the delimiter.  By subtracting
                 the length of what it returns from the length of the original, we find the number
                 of delimiters that were in the original.  We then add 1 since we are counting the
                 number of remark Codes, not the number of delimiters. */
              v_no_remark_codes := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;

              /* separate item into Remark Codes using parse procedure if more than 1 Code */
              IF v_no_remark_codes > 1 THEN
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(v_item,v_remark_codes_table,v_no_remark_codes,'\',v_bad_parse,v_import_log_seq);
                  /*  Now call procedure to remove any duplicate codes */
                  remove_lab_remark_dups(v_remark_codes_table, v_no_remark_codes);
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
              ELSE
                v_remark_codes_table(1) := v_item;
              END IF;
              
              /* loop through Remark Codes table and add each remark in column to the activity */
              FOR v_number IN 1..v_no_remark_codes LOOP
                              
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_remark_codes_table(v_number),p_imp_cfg_seq,v_col_seq,
                                                       v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
              
                IF v_storet_key.is_number IS NULL THEN
                  v_remark_is_table(v_number) := NULL;
                  v_remark_org_table(v_number) := NULL;
                  v_status := 'N';
                ELSE
                  v_remark_is_table(v_number) := v_storet_key.is_number;
                  v_remark_org_table(v_number) := v_storet_key.org_id;
                END IF;
                               
              END LOOP;

            ELSE
              v_no_remark_codes := 0;
              v_tsrlbrmk_is_nbr := NULL;
              v_tsrlbrmk_org_id := NULL;              
            END IF;
            
          ---------------------
          -- Quantification Low
          ---------------------
          ELSIF v_item_name = 'Quantification Low' THEN

            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_habitat != 1 THEN
                  IF LENGTH(v_item) > 12 THEN
                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq, 
                      'Line '||v_line_cnt||': '||
                      v_item_name||' is too long. It can be a maximum lenght of 12 digits.',
                      v_activity_seq
                    );
                    v_lowq_limit := NULL;
                    v_status := 'N';
                  ELSE
                    v_lowq_limit := v_item;
                  END IF;
                ELSE
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line '||v_line_cnt||': '||
                    v_item_name||' has been provided and is not allowed for Habitat Assessment.',
                    v_activity_seq
                  );
                  v_lowq_limit := NULL;
                  v_status := 'N';                
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_lowq_limit := NULL;
							END IF;
            ELSE
              v_lowq_limit := NULL;
            END IF;

          ----------------------
          -- Quantification High
          ----------------------
          ELSIF v_item_name = 'Quantification High' THEN

            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_habitat != 1 THEN
                  IF LENGTH(v_item) > 12 THEN
                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq, 
                      'Line '||v_line_cnt||': '||
                      v_item_name||' is too long. It can be a maximum lenght of 12 digits.',
                      v_activity_seq
                    );
                    v_upq_limit := NULL;
                    v_status := 'N';                  
                  ELSE
                    v_upq_limit := v_item;
                  END IF;
                ELSE
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line '||v_line_cnt||': '||
                    v_item_name||' has been provided and is not allowed for Habitat Assessment.',
                    v_activity_seq
                  );
                  v_status := 'N';                
                  v_upq_limit := NULL;
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_upq_limit := NULL;
							END IF;
            ELSE
              v_upq_limit := NULL;
            END IF;

          ------------------------------------
          -- Detection Limit
          -- Laboratory Result Detection Limit
          ------------------------------------
          ELSIF v_item_name = 'Detection Limit' OR v_item_name = 'Laboratory Result Detection Limit' THEN

           IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_habitat != 1 THEN
                  IF LENGTH(v_item) > 12 THEN
                    Sim2_Import_Pkg.write_error(
                      v_import_log_seq, 
                      'Line '||v_line_cnt||': '||
                      v_item_name||' is too long. It can be a maximum lenght of 12 digits.',
                      v_activity_seq
                    );
                    v_detect_limit := NULL;
                    v_status := 'N';                  
                  ELSE
                    v_detect_limit := v_item;
                  END IF;
                ELSE
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line '||v_line_cnt||': '||
                    v_item_name||' has been provided and is not allowed for Habitat Assessment.',
                    v_activity_seq
                  );
                  v_detect_limit := NULL;
                  v_status := 'N';                
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_detect_limit := NULL;
							END IF;
            ELSE
              v_detect_limit := NULL;
            END IF;

          -----------------------------------------
          -- Detection Limit Unit
          -- Laboratory Result Detection Limit Unit
          -----------------------------------------
          ELSIF v_item_name = 'Detection Limit Unit' OR v_item_name = 'Laboratory Result Detection Limit Unit' THEN

            IF v_item IS NOT NULL AND (v_detect_limit IS NOT NULL OR v_upq_limit IS NOT NULL OR v_lowq_limit IS NOT NULL) AND v_habitat != 1 THEN
             v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
             /* get the detection unit is number - check translations and STORET tables */
             IF NVL(v_char_result_type,'x') <> 'TEXT' THEN -- IF added by DCL to avoid unit of measure error if result is a text type
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
             END IF;

              IF v_storet_key.is_number IS NULL THEN
                /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                v_det_uom_is_nbr := NULL;
                v_det_uom_org_id := NULL;
                v_status := 'N';

              ELSE
                v_det_uom_is_nbr := v_storet_key.is_number;
                v_det_uom_org_id := v_storet_key.org_id;
                /* kms - need to make sure unit types match */
                --v_success := Sim_import_result_checks.check_uom_type(v_det_uom_is_nbr,v_char_is_nbr,v_line_cnt,v_import_log_seq);

                --IF v_success = 2 THEN
                  --v_status := 'N';
                  --Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure type does not match the unit of measure type for the indicated characteristic.',v_activity_seq);
                --END IF;
              END IF;
            ELSIF v_item IS NOT NULL THEN
              IF v_detect_limit IS NULL AND v_upq_limit IS NULL AND v_lowq_limit IS NULL THEN
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Detection limit unit of measure has been provided and is not used. It has been removed.',v_activity_seq);
                v_det_uom_is_nbr := NULL;
                v_det_uom_org_id := NULL;
              END IF;
            ELSE
              v_det_uom_is_nbr := NULL;
              v_det_uom_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Detection Limit Comment' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_item IS NOT NULL AND v_habitat != 1 THEN
                  v_det_comment := SUBSTR(v_item,1,254);
                ELSE
                  IF v_item IS NOT NULL THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
                  END IF;
                  v_det_comment := NULL;
                END IF;
  						ELSE
  						  v_status := 'N';-- v_item translated to null - set status to 'N'
  						END IF;
						ELSE
						  v_det_comment := NULL;
						END IF;

          ELSIF v_item_name = 'Precision' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_item IS NOT NULL AND v_habitat != 1 THEN
                  v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_precision := v_item;
                ELSE
                  IF v_item IS NOT NULL THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
                  END IF;
                  v_precision := NULL;
                END IF;
  						ELSE
  						  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_precision := NULL;
  						END IF;
						ELSE
              v_precision := NULL;
						END IF;

          ELSIF v_item_name = 'Confidence Level' THEN
            IF v_item IS NOT NULL AND v_habitat != 1 THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_conf_level := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_conf_level := NULL;
                v_status := 'N';
              END IF;
              IF v_precision IS NULL THEN
                v_status := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Precision is required when Confidence level is given.',v_activity_seq);
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_conf_level := NULL;
            END IF;

          ELSIF v_item_name = 'Bias' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_trip_qc != 1 AND v_habitat != 1 THEN
                  IF length(v_item) > 12 THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 12 characters.',v_activity_seq);
                    v_status := 'N';
                  ELSE
                    v_bias := v_item;
                  END IF;
                ELSE
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used for Trip QC or Habitat Assessment.',v_activity_seq);
                  v_bias := NULL;
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_bias := NULL;
							END IF;
            ELSE
              v_bias := NULL;
            END IF;

          ELSIF v_item_name = 'CL Corrected for Bias'THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_trip_qc != 1 AND v_habitat != 1 THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_cl_corrected := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                ELSE
                  v_cl_corrected := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. Not used for Habitat assessment or Trip QC.',v_activity_seq);
                v_cl_corrected := NULL;
              END IF;
            ELSE
              v_cl_corrected := NULL;
            END IF;

          ELSIF v_item_name = 'Result Comment' OR v_item_name = 'Characteristic Result Comment' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF NVL(v_char_result_type,'x') = 'TEXT' AND v_item IS NOT NULL THEN
                /** dcl 09/13/01 if a result comment exists for a text type result **/
                 Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Result Comments are not used with a Text-type result - comment ignored',v_activity_seq);
                 v_result_cmt := NULL;
              ELSIF v_item IS NOT NULL THEN
                v_result_cmt := v_item;
                IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                  v_result_cmt := NULL;
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed on Single Taxon Frequency Classes.',v_activity_seq);
                ELSIF v_habitat = 1 THEN
                  v_result_cmt := NULL;
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
                END IF;
							ELSIF v_item IS NULL THEN
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_result_cmt := NULL;
							END IF;
            ELSE
              v_result_cmt := NULL;
            END IF;

          ELSIF v_item_name = 'Result Document/Graphic' THEN
            IF v_item IS NOT NULL THEN
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF sim2_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                  v_result_document := v_item;
                ELSE
                  v_status := 'N';       --Invalid document.  Set to 'N'.
                  v_result_document := NULL;
                END IF;    
              ELSE
                v_status := 'N';          --Translated to NULL.  Set to 'N'.
                v_result_document := NULL;
              END IF;
            ELSE
              v_result_document := NULL;
            END IF;

          /* 06/09/03 slp - If the any result other than 'Single Taxon', validate the result status.
                          - All status for the 'Single Taxon' group must be the same in STORET.  Therefore, if the record
                            - is the first of the 'Single Taxon' group, validate the result status.
                            - If the record is within the 'Single Taxon' group, set the status to NULL. */
          ELSIF v_item_name = 'Result Status'  THEN
            IF v_new_group = 1 OR v_brg_type != 'Single Taxon Frequency Classes' THEN
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := RTRIM(v_item);
              
              IF v_item IS NOT NULL AND v_habitat != 1 THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_activity_seq,
                  v_import_log_seq,
                  v_line_cnt
                );
                IF v_dummy_seq IS NOT NULL THEN
                  v_result_status := sim_import_result_extras.get_item_value(
                    v_dummy_seq,
                    v_line_cnt,
                    v_import_log_seq
                  );
                  -- jah 2-24-04 Convert to uppercase because STORET GUI converts Result Status to uppercase
                  v_result_status := UPPER(v_result_status);
                ELSE
                  v_result_status := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_result_status := NULL;
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line '||v_line_cnt||
                  ': Result Status has been left blank and will be set to ''Final.''',
                  v_activity_seq
                );
              END IF;
            ELSE
              v_result_status := NULL;
            END IF;

          -- This code is for the Portable Data Logger Data Lines
          ELSIF v_item_name = 'PDL Line Number' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_activity_cat != 'Portable Data Logger' THEN      
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity Category must be ''Portable Data Logger'' if there is a Data Line Number',v_activity_seq);                     
                  v_status := 'N';
                  v_data_line_number := NULL;
                ELSE
                  v_data_line_number := v_item;
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_data_line_number := NULL;
							END IF;
            ELSE
              v_data_line_number := NULL;
						END IF;         
            
          -- This code is for the Portable Data Logger Data Line Name
          ELSIF v_item_name = 'PDL Line Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_activity_cat != 'Portable Data Logger' THEN      
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity Category must be ''Portable Data Logger'' if there is a Data Line Name',v_activity_seq);                     
                  v_data_line_name := NULL;
                  v_status := 'N';
                ELSIF v_data_line_number IS NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': There must be a Data Line Number if there is a Data Line Name',v_activity_seq);               
                  v_data_line_name := NULL;
                  v_status := 'N';                
                ELSE 
                  v_data_line_name := v_item;
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_data_line_name := NULL;
							END IF;
            ELSE
              v_data_line_name := NULL;
            END IF; 
                 
          ELSIF v_item_name = 'Laboratory Batch ID' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL AND v_habitat != 1 THEN
                v_item := UPPER(RTRIM(v_item)); -- 06/13/03 SLP - Set to uppercase and removed trailing spaces.
                IF LENGTH(v_item) > 12 THEN
                  v_item := substr(v_item,1,12);
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer that 12 characters and will be truncated.',v_activity_seq);
                END IF;
                v_batch_id := v_item;
              ELSE
                IF v_item IS NOT NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
                  v_batch_id := NULL;
                ELSE  
  	    				  v_status := 'N';-- v_item translated to null - set status to 'N'
                  v_batch_id := NULL;
                END IF;
              END IF;
  					ELSE
              v_batch_id := NULL;
  					END IF;

          ELSIF v_item_name = 'Laboratory Certified' OR v_item_name = 'Laboratory Certification Indicator' THEN
            IF v_item IS NOT NULL  AND v_habitat != 1 THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq, v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NULL THEN
                v_status := 'N';
                v_lab_cert := NULL;
              ELSE
                v_lab_cert := Sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_lab_cert := NULL;
            END IF;

          ELSIF v_item_name = 'Project ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- jah 2-24-04 Remove trailing spaces but do not convert to uppercase
              v_item := RTRIM(v_item);
              v_no_projects := 0;

              /* can have multiple projects separated by '\' - get number of projects to expect */
              v_no_projects := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;

              /* separate item into projects using parse procedure if more than 1 project */
              IF v_no_projects > 1 THEN
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(
                    v_item,
                    v_project_table,
                    v_no_projects,
                    '\',
                    v_bad_parse,
                    v_import_log_seq
                  );
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
              ELSE
                v_project_table(1) := v_item;
              END IF;

              /* loop through projects table and add each project in column to the activity */
              FOR v_number IN 1..v_no_projects LOOP
                /* Check that Project ID is not longer than 8 characters -- sfp 061101 */
                /* kms (6/21/01) - changed to check length of item in project table rather than
                   of v_item. */
                IF LENGTH(v_project_table(v_number)) > 8 THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line '||v_line_cnt||': Project ID ('||v_project_table(v_number)||
                    ') must not contain more than 8 characters.',
                    v_activity_seq
                  );
                END IF;

                /* get the project is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_project_table(v_number),
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );

                IF v_storet_key.is_number IS NOT NULL THEN
                  v_proj_is_table(v_number) := v_storet_key.is_number;
                  v_proj_org_table(v_number) := v_storet_key.org_id;
                ELSE
                  v_status := 'N';
                END IF;

              END LOOP;

            ELSE
              /* if no project id, write error to error log and set status to 'N' */
              v_proj_is_nbr := NULL;
              v_proj_org_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(
                v_import_log_seq, 
                'Line '||v_line_cnt||
                ': Project ID is required and has not been provided.',
                v_activity_seq
              );
            END IF;

          ELSIF v_item_name = 'Characteristic Group ID' THEN
            v_groups := TRUE;

            IF v_item IS NOT NULL THEN
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := RTRIM(v_item);

              /* get the char group is number - check translations and STORET tables */
              v_storet_key := sim_import_result_extras.validate_storet_item(
                v_item_name,
                v_item,
                p_imp_cfg_seq,
                v_col_seq,
                v_dummy_key,
                p_org_id,
                v_line_cnt,
                v_activity_seq,
                v_import_log_seq, 
                v_dummy_char
              );

              IF v_storet_key.is_number IS NOT NULL THEN
                -- jah 2-24-04 Convert to uppercase because STORET GUI converts Characteristic Group ID to uppercase
                -- (Note: Always convert to uppercase AFTER translation!)
                v_item := UPPER(v_item);

                /* kms - make sure group is valid for this activity.  Check type and medium. */
                v_success := Sim_import_result_checks.check_group_type_med(
                  v_storet_key,
                  v_type,
                  v_medium, 
                  v_intent,
                  v_brg_type,
                  v_community_is_nbr, 
                  v_community_org_id,
                  v_activity_seq,
                  v_line_cnt,
                  v_import_log_seq, 
                  v_HA_group
                );

                IF v_success != 1 THEN
                  /* kms - group is not valid for this activity or an error occured when checking */
                  v_tsrchgrp_is_nbr := NULL;
                  v_tsrchgrp_org_id := NULL;
                  v_status := 'N';
                ELSE
                  /* kms - group is valid for this activity */
                  v_tsrchgrp_is_nbr := v_storet_key.is_number;
                  v_tsrchgrp_org_id := v_storet_key.org_id;
                  /* Check to make sure Category is Routine Habitat Assessment */
                  v_group := NULL;
                  IF v_ha_group IN (1,2) THEN
                    v_HA_group_id := v_dummy_char;
                    IF v_activity_cat != 'Routine Habitat Assessment' AND v_activity_cat != 'Replicate Habitat Assessment' THEN
                      v_status := 'N';
                      Sim2_Import_Pkg.write_error(
                        v_import_log_seq,
                        'Line '||v_line_cnt||
                        ': Habitat Assessment Characteristics must have Category of either Routine or Replicate Habitat Assessment.',
                        v_activity_seq
                      );
                    END IF;
                  ELSE
                    v_HA_group_id := NULL;
                  END IF;
                END IF;
          
                /* Make sure two different groups are not used on an HA activity */
                IF v_new_activity != 1 THEN
                  OPEN c_char_group(v_activity_seq);
                    FETCH c_char_group INTO v_group;
                    IF NVL(v_group,'x') != NVL(v_HA_group_id,'x') THEN
                      Sim2_Import_Pkg.write_error(
                        v_import_log_seq,
                        'Line '||v_line_cnt||
                        ': Habitat Assessment Activities can only use one Characteristic Group.',
                        v_activity_seq
                      );
                      v_status := 'N';
                    END IF;
                  CLOSE c_char_group;
                ELSIF v_ha_group IN(1,2) THEN
                  v_group := v_dummy_char;
                END IF;              

              ELSE
                v_tsrchgrp_is_nbr := NULL;
                v_tsrchgrp_org_id := NULL;
                v_status := 'N'; -- v_item translated to a null - set status to 'N'
              END IF;

            ELSE
              --v_status := 'N';
              v_tsrchgrp_is_nbr := NULL;
              v_tsrchgrp_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Characteristic Row ID' THEN
       
            IF v_item IS NOT NULL AND v_tsrchgrp_is_nbr IS NOT NULL THEN
              /* sef - get char defaults from different places depending ow whether they are Habitat Assessment */
              /* kms - set temp key to group pk values in order to send to validation routine */
              v_temp_key.is_number := v_tsrchgrp_is_nbr;
              v_temp_key.org_id := v_tsrchgrp_org_id;
              
              IF v_HA_group = 1 THEN
                /* get the HA row is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  'Habitat Assessment Row ID',
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_temp_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_dummy_char
                );

                IF v_storet_key.is_number IS NOT NULL THEN
                  v_tsrhcsc_is_nbr := v_storet_key.is_number;
                  v_tsrhcsc_org_id := v_storet_key.org_id;

                  v_HA_char_name := sim_import_result_extras.get_HA_default(
                    v_storet_key, 
                    v_activity_seq,
                    v_import_log_seq,
                    v_line_cnt
                  );
                  IF v_HA_char_name IS NULL THEN
                    v_status := 'N';
                  ELSE
                    v_char_result_type := 'HA  ';
                  END IF;
                ELSE
                  v_tsrhcsc_is_nbr := NULL;
                  v_tsrhcsc_org_id := NULL;
                  v_status := 'N';
                END IF;

                /* assign defaults to result variables */
                v_tsrchdef_is_nbr := NULL;
                v_tsrchdef_org_id := NULL;
                v_char_is_nbr := NULL;
                v_char_org_id := NULL;
                v_uom_is_nbr := NULL;
                v_uom_org_id := NULL;
                v_anlpr_is_nbr := NULL;
                v_anlpr_org_id := NULL;
                v_labprep_is_nbr := NULL;
                v_labprep_org_id := NULL;
                v_fraction := NULL;
                v_fraction_is_nbr := NULL;
                v_fraction_org_id := NULL;
                v_volt_is_number := NULL;
                v_volt_org_id := NULL;
                v_habit_is_number := NULL;
                v_habit_org_id := NULL;
                v_value_type := NULL;
                v_stat_type := NULL;
                v_duration_basis := NULL;
                v_temp_basis := NULL;
                v_weight_basis := NULL;
                v_char_species_number := NULL;
                v_func_feed_grp := NULL;
                v_trophic_level := NULL;
                v_taxon_pollution  := NULL;
                v_particle_size_basis := NULL;  -- jah 2-23-04 Added Particle Size Basis

              ElSE

                /* get the char row is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_temp_key,
                  p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char
                );

                IF v_storet_key.is_number IS NOT NULL THEN
                  v_tsrchdef_is_nbr := v_storet_key.is_number;
                  v_tsrchdef_org_id := v_storet_key.org_id;
                  
                  /* populate result char fields based on what's in the default table */
                  v_dummy_seq := NULL;
                  v_success := get_char_defaults(
                    v_dummy_seq, 
                    v_storet_key, 
                    p_imp_cfg_seq, 
                    v_col_seq, 
                    v_activity_seq
                  );
                      
                  IF v_success IS NULL THEN
                    v_status := 'N';
                    -- ELSE
                    --   v_char_result_type := 'HA  ';
                  END IF;
                ELSE
                  v_tsrchdef_is_nbr := NULL;
                  v_tsrchdef_org_id := NULL;
                  v_status := 'N';
                END IF;                  
                  
                /* assign defaults to result variables */
                v_char_is_nbr         := v_acode_defs_rec.char_pk.is_number;
                v_char_org_id         := v_acode_defs_rec.char_pk.org_id;
                v_uom_is_nbr          := v_acode_defs_rec.uom_pk.is_number;
                v_uom_org_id          := v_acode_defs_rec.uom_pk.org_id;
                v_anlpr_is_nbr        := v_acode_defs_rec.anlpr_pk.is_number;
                v_anlpr_org_id        := v_acode_defs_rec.anlpr_pk.org_id;
                v_labprep_is_nbr      := v_acode_defs_rec.labprep_pk.is_number;
                v_labprep_org_id      := v_acode_defs_rec.labprep_pk.org_id;
                v_fraction_is_nbr     := v_acode_defs_rec.fraction_is;
                v_fraction_org_id     := 'ZZZZZZZZ';
                v_volt_is_number      := v_acode_defs_rec.voltinism_is;
                v_volt_org_id         := 'ZZZZZZZZ';
                v_habit_is_number     := v_acode_defs_rec.habit_is;
                v_habit_org_id        := 'ZZZZZZZZ';
                v_value_type          := v_acode_defs_rec.value_type;
                v_stat_type           := v_acode_defs_rec.stat_type;
                v_duration_basis      := v_acode_defs_rec.dur_basis;
                v_temp_basis          := v_acode_defs_rec.temp_basis;
                v_weight_basis        := v_acode_defs_rec.weight_basis;
                v_char_species_number := v_acode_defs_rec.species;
                v_func_feed_grp       := v_acode_defs_rec.funct_feed_grp;
                v_trophic_level       := v_acode_defs_rec.trophic_level;
                
                -- jah 2-20-04 Added Particle Size Basis
                -- Does the acode field for particle size basis contain anything?
                IF (NVL(v_acode_defs_rec.particle_size_basis,' ') > ' ') THEN
                  -- if yes
                  v_particle_size_basis := v_acode_defs_rec.particle_size_basis;
                ELSE
                  -- if no
                  -- ensure v_particle_size_basis contains NULL (and not ' ')
                  v_particle_size_basis := NULL;
                END IF;

                v_taxon_pollution     := v_acode_defs_rec.pollution_tol;
                v_tsrhcsc_is_nbr      := NULL;
                v_tsrhcsc_org_id      := NULL;

                /* Old SIM v1.2.2 checks have been removed in v2  */
                -- For V2 we can assume that sample fraction is valid in a char group
                -- Check UOM type on Char group - this never happens any more. In v2 any UOM is okay.

              END IF;

              IF v_brg_type = 'Multi-Taxon Population Census' AND v_char_result_type != 'TAXA' THEN
                v_status := 'N';
                v_char_is_nbr := NULL;
                v_char_org_id := NULL;
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Characteristic must be of type TAXA for Multi-Taxon Population Census.',v_activity_seq);
              END IF;

            ELSIF v_item IS NOT NULL AND v_tsrchgrp_is_nbr IS NULL THEN
              /* kms - importing characteristic row, but there's no characteristic group
                 to validate it with.  Invalidate the activity and write error message. */
              v_tsrchdef_is_nbr := NULL;
              v_tsrchdef_org_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(
                v_import_log_seq,
                'Line '||v_line_cnt||
                ': Characteristic Row ID has been included but cannot be validated due to missing or invalid Characteristic Group ID.',
                v_activity_seq
              );
            ELSIF v_item IS NULL AND v_tsrchgrp_is_nbr IS NOT NULL THEN
              /* kms - characteristic group has been imported, but no row.  Invalidate activity
               and write error message. */
              v_tsrchdef_is_nbr := NULL;
              v_tsrchdef_org_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(
                v_import_log_seq,
                'Line '||v_line_cnt||
                ': Characteristic Row ID is required when Characteristic Group ID has been given.',
                v_activity_seq
              );
            ELSE
              v_tsrchdef_is_nbr := NULL;
              v_tsrchdef_org_id := NULL;
            END IF;


          /* The following import fields are relevant only to biological results
          only when v_medium = 'Biological'  */

          ELSIF v_item_name = 'Intent' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_medium = 'Biological' THEN --Intent is required where medium = 'Biological'
                /* get intent - check translations and allowable values tables */
                IF v_dummy_seq IS NULL THEN
                   v_status := 'N';
                   sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Intent was not found.',v_activity_seq);
                ELSE
                  v_intent := Sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                END IF;
              ELSE
                v_intent := NULL;
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Intent ignored. It is only applicable when Medium is Biological.',v_activity_seq);
              END IF;
            ELSE
              v_intent := NULL;
            END IF;

          ELSIF v_item_name = 'Subject Taxon'  THEN  -- Subject Taxon is required where intent = Individual or Tissue
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
              /*get characteristic is number - check translations and STORET tables */
              v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,
                              v_col_seq,v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
              IF v_storet_key.is_number IS NOT NULL THEN
                 v_taxon_is_nbr := v_storet_key.is_number;
                 v_taxon_org_id := v_storet_key.org_id;
              ELSE
                v_taxon_is_nbr := NULL;
                v_taxon_org_id := NULL;
                v_status := 'N';
                -- sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': The Subject Taxon was not found in STORET',v_activity_seq);
              END IF;
            ELSE
              v_taxon_is_nbr := NULL;
              v_taxon_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Subject Taxon Species Number' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_taxon_species_number := sim_import_result_extras.get_allowable_value(v_dummy_seq,v_import_log_seq);
              ELSE
                v_taxon_species_number := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_taxon_species_number := NULL;
            END IF;

          ELSIF v_item_name = 'Characteristic Species Number' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_char_species_number := sim_import_result_extras.get_allowable_value(v_dummy_seq,v_import_log_seq);
              ELSE
                v_char_species_number := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_char_species_number := NULL;
            END IF;


          ----------------------
          -- Parent Sample ID --
          ----------------------
          ELSIF v_item_name = 'Parent Sample ID' THEN

            -- do we need to process this field?
            IF  v_new_activity = 1 
              AND v_type != 'Field Msr/Obs'
              AND v_item IS NOT NULL THEN
              --  if yes

              -- does this activity need a Parent Sample ID?
              IF (v_activity_cat = 'Created from Sample')
                OR (v_activity_cat = 'Composite-with Parents') THEN
                -- if yes
  
                /*
                Rules:
                o  Process Parent Sample ID after validating Activity Category
                o  SIM no longer validates Intent, Visit, and Subject Taxon for Parent Sample ID
                o  Parent must exist in STORET (TSRFDACT) or SIM (SIM_FIELD_ACTIVITIES)
                o  If Parent record does not exist in STORET then Parent record must 
                   occur before child in the import file
                o  Ignore data in child records for Actual Activity Location, Tow/Net Operation, 
                   and Depth/Stratification
  
                Notes:
                o   The Parent Sample ID field can have multiple values
                    seperated by backward slashes (\).
                */
    
                -- Count the backward slashes and add one to determine how many Parent Sample IDs
                v_no_parent_ids := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;
    
                -- load the Parent Sample IDs into a table
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(
                    v_item,
                    v_parent_id_table,  -- table of Parent Sample IDs
                    v_no_parent_ids,
                    '\',
                    v_bad_parse,
                    v_import_log_seq
                  );
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
    
                -- loop through Parent Sample IDs table
                FOR v_number IN 1..v_no_parent_ids LOOP
    
                  -- remove trailing spaces from Parent Sample ID
                  v_parent_id_table(v_number) := RTRIM(v_parent_id_table(v_number));

                  -- Check for a translation of the Parent Sample ID in STORET
                  v_storet_key := sim_import_result_extras.CHECK_STORET_TRANSLATION(
                    v_parent_id_table(v_number),
                    p_imp_cfg_seq,
                    v_col_seq,
                    v_trans_exist,
                    v_import_log_seq,
                    v_line_cnt,
                    v_converted_item,
                    v_activity_seq                    
                  );
                  
                  -- did we find a translation?
                  IF v_storet_key.is_number IS NOT NULL THEN
                    -- if yes
                    
                    -- save the parent information in tables for later processing
                    v_parent_id_table(v_number)        := v_converted_item;
                    v_parent_fa_seq_table(v_number)    := NULL;
                    v_parent_is_number_table(v_number) := v_storet_key.is_number;
                    v_parent_org_id_table(v_number)    := v_storet_key.org_id;
                  
                  -- did it translate?
                  ELSIF v_trans_exist = FALSE THEN
                    -- if no (it did not translate)

                    -- jah 3-9-2004                  
                    -- Convert to upper case (but only after translation!)
                    v_parent_id_table(v_number) := UPPER(v_parent_id_table(v_number));
                    
                    -- save the parent information in tables for later processing
                    v_error_code := Sim_import_result_checks.check_parent(
                      v_parent_id_table(v_number),         -- IN: Parent Sample ID
                      v_visit_seq,                         -- IN: Station Visit
                      v_line_cnt,                          -- IN:
                      v_import_log_seq,                    -- IN:
                      v_parent_fa_seq_table(v_number),     -- OUT: from SIM_FIELD_ACTIVITIES
                      v_parent_is_number_table(v_number),  -- OUT: from TSRFDACT
                      v_parent_org_id_table(v_number)      -- OUT: from TSRFDACT
                    );
    
                    -- did we get an error in the check_parent function?
                    IF (v_error_code IS NOT NULL) THEN
                      -- if yes
                       
                      -- set status flag to reject this import record
                      v_status := 'N';
      
                      -- note: Function CHECK_PARENT has already displayed the error message

                    END IF;
                  ELSE
                    -- otherwise, it translated to NULL
                       
                    -- Prent migration of this record
                    v_status := 'N';
      
                    -- note: Function CHECK_STORET_TRANSLATION has already displayed the error message
                  END IF;
                    
                END LOOP;

              ELSE
                -- if no (this activity does not need a Parent Sample ID)
  
                -- output warning message
                sim2_import_pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line ' || v_line_cnt|| 
                  ': Parent Sample ID ignored -- Not applicable to "' || v_activity_cat || '".',
                  v_activity_seq
                );

              END IF;

            ELSE
              -- if no (we do not need to process Parent Sample ID for this activity)

              -- indicate the Parent Sample ID does not exist for this activity
              v_no_parent_ids := 0;

            END IF;


          ---------------  
          -- Community --
          ---------------  
          ELSIF v_item_name = 'Community' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
              /* get community name - check translations and STORET tables*/
              v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
              IF v_intent = 'Taxon Abundance' THEN
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_community_is_nbr := v_storet_key.is_number;
                  v_community_org_id := v_storet_key.org_id;
                ELSE
                  v_community_is_nbr := NULL;
                  v_community_org_id := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_community_is_nbr := NULL;
                v_community_org_id := NULL;
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Community ignored. It is not used unless intent is Taxon Abundance.',v_activity_seq);
              END IF;
            ELSE
              v_community_is_nbr := NULL;
              v_community_org_id := NULL;
            END IF;
           
          ELSIF v_item_name = 'Bio Part' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
               v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
               /*get bio part name - check translations and STORET tables*/
               v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
               IF v_intent = 'Tissue' THEN
                 IF v_storet_key.is_number IS NOT NULL THEN
                    v_biopart_is_nbr := v_storet_key.is_number;
                    v_biopart_org_id := v_storet_key.org_id;
                 ELSE
                    v_biopart_is_nbr := NULL;
                    v_biopart_org_id := NULL;
                    v_status := 'N';
                 END IF;
               ELSE
                 v_biopart_is_nbr := NULL;
                 v_biopart_org_id := NULL;
                 sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Bio Part is not applicable and has been ignored',v_activity_seq);
               END IF;
            ELSE
              v_biopart_is_nbr := NULL;
              v_biopart_org_id := NULL;
            END IF;

          -- Following 2 items are distined for TSRSMPLE via SIM_SAMPLE
          -- Total Sample Weight
          ELSIF v_item_name = 'Total Sample Weight' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                BEGIN
                  v_smpl_wt := to_number(v_item,'99999.99');
                EXCEPTION
                  WHEN OTHERS THEN
                    sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Total Sample Weight is not a valid number.',v_activity_seq);
                    v_smpl_wt := NULL;
                END;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_smpl_wt := NULL;
							END IF;
            ELSE
              v_smpl_wt := NULL;
            END IF;

          -- Total Sample Weight Units
          ELSIF v_item_name = 'Total Sample Weight Units' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_total_wt_uom := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_total_wt_uom := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_total_wt_uom := NULL;
            END IF;

          -- Following 9 items are destined for TSRTOD via SIM_TRAWL_OPS_DET
          
          
          ------------------
          -- Fished Duration
          ------------------

          ELSIF v_item_name = 'Fished Duration' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_check_trawl_gear = 1 THEN
                  BEGIN
                    v_fished_duration := to_number(v_item,'99.99');
                  EXCEPTION
                    WHEN OTHERS THEN
                      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' is not a valid number.',v_activity_seq);
                      v_status := 'N';
                      v_fished_duration := NULL;
                  END;
                ELSE
                  v_fished_duration := NULL;
                  sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It applies only when Procedure has a Horizontal Tow Gear Type.',v_activity_seq);
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_fished_duration := NULL;
							END IF;
            ELSE
              v_fished_duration := NULL;
            END IF;

          ------------------------
          -- Fished Duration Units
          ------------------------
          
          ELSIF v_item_name = 'Fished Duration Units' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_fished_dur_uom := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_fished_dur_uom := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_fished_dur_uom := NULL;
            END IF;

          ELSIF v_item_name = 'Distance Fished' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_check_trawl_gear = 1 THEN
                  BEGIN
                    v_distance_fished := to_number(v_item,'99.99');
                  EXCEPTION
                    WHEN OTHERS THEN
                      sim2_import_pkg.write_error(
                        v_import_log_seq, 
                        'Line '||v_line_cnt||': '||v_item_name||' is not a valid number.',
                        v_activity_seq
                      );
                      v_status := 'N';
                      v_distance_fished := NULL;
                  END;
                ELSE
                  v_distance_fished := NULL;
                  sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It applies only when Procedure has a Horizontal Tow Gear Type.',v_activity_seq);
                END IF;
							ELSE
							  v_status := 'N';-- v_item translated to null - set status to 'N'
                v_distance_fished := NULL;
							END IF;
            ELSE
              v_distance_fished := NULL;
            END IF;

          ELSIF v_item_name = 'Distance Fished Units' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_dummy_seq IS NOT NULL THEN
                v_dist_fished_uom := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
              ELSE
                v_dist_fished_uom := NULL;
                v_status := 'N';
              END IF;
            ELSE
              v_dist_fished_uom := NULL;
            END IF;

          --Next 7 items destined for TSRBRG via SIM_BIO_RESULTS_GRP
          --Activity ID and Intent must have already been processed before this point
          --An activity may have multiple Bio Results Groups, but only one MTPC
          --A routine is needed to cause processing on only the first occurence of a new group
          --The data elements must occur together in the text file in the specified order
          --The SIM_BIO_RESULTS_GRP record will be created after the column loop and after the
          -- activity is created

          ELSIF v_item_name = 'Bio Results Group ID' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_curr_group := UPPER(RTRIM(v_item));  -- 06/13/03 SLP - will use v_curr_group later to get BRGI_BRG_SEQ for BRGI_INDIVIDUAL
                OPEN c_find_brg_seq(v_curr_group, v_activity_seq);
                  FETCH c_find_brg_seq INTO v_dummy_seq;
                  IF c_find_brg_seq%NOTFOUND THEN  -- it is a new group
                    v_new_group := 1;                 -- set new group flag
                  ELSE  -- it is not a new group
                    v_new_group := 0;                 -- set not new group flag
                    v_brg_seq := v_dummy_seq;
                  END IF;
                CLOSE c_find_brg_seq;
                IF v_new_group = 1 THEN
                  IF v_intent = 'Taxon Abundance' THEN
                    IF v_curr_group IS NOT NULL THEN
                      v_brg_id := v_curr_group;
                    ELSE
                      v_status := 'N';
                      v_brg_id := NULL;
                      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Bio Results Group ID must be given when Intent is Taxon Abundance',v_activity_seq);
                    END IF;
                  ELSE
                    v_brg_id := NULL;
                    sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Bio Results Group ID ignored. Only applies when Intent is Taxon Abundance',v_activity_seq);
                  END IF;
                END IF;
              ELSE
                v_new_group := 1;
                v_brg_id := NULL;
                v_status := 'N';-- v_item translated to null - set status to 'N'
              END IF;
            ELSE  --v_item is NULL
              v_new_group := 1;
              v_brg_id := NULL;
            END IF;

          ELSIF v_item_name = 'Bio Results Group Description' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_intent = 'Taxon Abundance' THEN
                  IF LENGTH(v_item) > 1999 THEN
                    sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 1999 characters and will be truncated.',v_activity_seq);
                    v_item := SUBSTR(v_item,1,1999);
                  END IF;
                  v_brg_text := v_item;
                ELSE
                  sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used when Intent is Taxon Abundance.',v_activity_seq);
                  v_brg_text := NULL;
                END IF;
              ELSE
                v_brg_text := NULL;
                v_status := 'N';-- v_item translated to null - set status to 'N'
              END IF;
            ELSE  --v_item is NULL
              v_brg_text := NULL;
            END IF;

          ELSIF v_item_name = 'Bio Results Type' AND v_new_group = 1 THEN
            IF v_intent = 'Taxon Abundance' THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_item IS NOT NULL THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_brg_type := Sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                  --check for MTPC on this activity
                  IF v_brg_type = 'Multi-Taxon Population Census' THEN
                    v_success := check_existing_MTPC(v_brg_type, v_activity_seq);
                    IF v_success = 2 THEN
                      v_status := 'N';
                      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Multi-Taxon Population Census already exisit on this activity.',v_activity_seq);
                    END IF;
                  END IF;
                ELSE
                  v_status := 'N';
                  v_brg_type := NULL;
--                  sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Invalid Bio Results Type',NULL);
                END IF;
              ELSE
                v_status := 'N';
              END IF;
            ELSE
              sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Bio Results Type ignored. It is only used if Intent is Taxon Abundance',v_activity_seq);
              v_brg_type := NULL;
            END IF;

         ELSIF v_item_name = 'Frequency Analysis Type' AND v_new_group = 1 THEN --B or P
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
              IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                IF v_dummy_seq IS NOT NULL THEN
                   v_brg_freq_analysis := Sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                ELSE
                  v_brg_freq_analysis := NULL;
                  v_status := 'N';
--                  sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': The Bio Results Type Indicator was not found in STORET',NULL);
                END IF;
              ELSE
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||' Frequency Analysis Type is not used for '||v_brg_type||' and will be ignored',v_activity_seq);
                v_brg_freq_analysis := NULL;
              END IF;
            ELSE
              v_brg_freq_analysis := NULL;
            END IF;

          ELSIF v_item_name = 'Sex' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              --IF v_brg_freq_analysis = 'P' THEN
              IF v_BRG_type = 'Single Taxon Frequency Classes' THEN
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_sexname_is_nbr := v_storet_key.is_number;
                  v_sexname_org_id := v_storet_key.org_id;
                ELSE
                  v_status := 'N';
                  v_sexname_is_nbr := NULL;
                  v_sexname_org_id := NULL;
                END IF;
              ELSE
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Sex ignored. It is only used for Single Taxon Frequency Classes',v_activity_seq);
                v_sexname_is_nbr := NULL;
                v_sexname_org_id := NULL;
              END IF;
            ELSE
              v_sexname_is_nbr := NULL;
              v_sexname_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Life Stage' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
--              IF v_brg_freq_analysis = 'P' THEN
              IF v_BRG_type = 'Single Taxon Frequency Classes' THEN
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_lifestage_is_nbr := v_storet_key.is_number;
                  v_lifestage_org_id := v_storet_key.org_id;
                ELSE
                  v_status := 'N';
                  v_lifestage_is_nbr := NULL;
                  v_lifestage_org_id := NULL;
                END IF;
              ELSE
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Life Stage ignored. It is only used for Single Taxon Frequency Classes.',v_activity_seq);
                v_lifestage_is_nbr := NULL;
                v_lifestage_org_id := NULL;
              END IF;
            ELSE
              v_lifestage_is_nbr := NULL;
              v_lifestage_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Common Class Descriptor' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_freq_analysis = 'P' THEN
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_char_is_nbr := v_storet_key.is_number;
                  v_char_org_id := v_storet_key.org_id;
                ELSE
                  v_char_is_nbr := NULL;
                  v_char_org_id := NULL;
                  v_status := NULL;
                  sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Common Class Descriptor is required if Frequency Analysis Type is "Physical."',v_activity_seq);
                END IF;
              ELSE
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Common Class Descriptor should not be given unless Frequency Analysis Type is "Physical."',v_activity_seq);
              END IF;
            END IF;

          ELSIF v_item_name = 'Common Class Descriptor Units' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_freq_analysis = 'P' THEN
                v_storet_key := sim_import_result_extras.validate_storet_item('Result Value Units',v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);

                IF v_storet_key.is_number IS NULL THEN
                /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                  v_uom_is_nbr := NULL;
                  v_uom_org_id := NULL;
                  v_status := 'N';
                ELSE
                  v_uom_is_nbr := v_storet_key.is_number;
                  v_uom_org_id := v_storet_key.org_id;
                END IF;
              ELSE
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Common Class Descriptor Units should not be given unless Frequency Analysis Type is "Physical."',v_activity_seq);
              END IF;
              --v_uom_is_nbr := NULL;
              --v_uom_org_id := NULL;
            END IF;


          ELSIF v_item_name = 'Bio Group Count Type' AND v_new_group = 1 THEN  -- This column was originally named 'Value Type Name'
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_type = 'Single Taxon Frequency Classes' OR v_brg_type = 'Single Taxon Group Summary' THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_value_type_name := sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                ELSE
                  v_value_type_name := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_value_type_name := NULL;
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Bio Group Count Type is not used for '||v_brg_type||' and will be ignored.',v_activity_seq);
              END IF;
            ELSE
              v_value_type_name := NULL;
            END IF;

          -- Total Number in Group
          ELSIF v_item_name = 'Total Number in Group' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_brg_type IN ('Single Taxon Group Summary') THEN
                  BEGIN
                    IF v_item IS NOT NULL THEN
                      v_ttl_nbr_in_grp := to_number(v_item);
                      IF v_ttl_nbr_in_grp > 99999999 THEN
                        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Total Number in Group must be smaller than 100000000.',v_activity_seq);
                        v_status := 'N';
                        v_ttl_nbr_in_grp := 999;
                      END IF;
                      IF v_ttl_nbr_in_grp <= 0 THEN
                        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Total Number in Group must be greater than zero.',v_activity_seq);
                        v_status := 'N';
                      END IF;
                    ELSE
                      v_ttl_nbr_in_grp := NULL;
                      v_status := 'N';
                      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Total Number in Group is required for Single Taxon Group Summary.',v_activity_seq);
                    END IF;
                  EXCEPTION
                    WHEN OTHERS THEN
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' is not a valid number of 8 or fewer digits.',v_activity_seq);
                      v_status := 'N';
                  END;
                ELSE
                  IF v_item IS NOT NULL THEN
                    sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Total Number in Group ignored. Only used for Single Taxon Group Summary.',v_activity_seq);
                  END IF;
                  v_ttl_nbr_in_grp := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_ttl_nbr_in_grp := NULL;
              END IF;
            ELSE  --v_item is NULL
              v_ttl_nbr_in_grp := NULL;
            END IF;

          ELSIF v_item_name = 'Number of Individuals in Group' AND v_new_group = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_brg_type IN ('Single Taxon Individuals') THEN
                  IF v_item IS NOT NULL THEN
                    v_ttl_nbr_indiv := v_item;
                    IF v_ttl_nbr_indiv > 999 THEN
                      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Number of Individuals in Group must be smaller than 1000.',v_activity_seq);
                      v_status := 'N';
                      v_ttl_nbr_indiv := 999;
                    END IF;
                    IF v_ttl_nbr_indiv <= 0 THEN
                      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Number of Individuals in Group must be greater than zero.',v_activity_seq);
                      v_status := 'N';
                    END IF;
                  ELSE
                    v_ttl_nbr_indiv := NULL;
                    v_status := 'N';
                    sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Number of Individuals in Group is required for Single Taxon Individuals.',v_activity_seq);
                  END IF;
                ELSE
                  IF v_item IS NOT NULL THEN
                    sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Total Number in Group ignored. Only used for Single Taxon Individuals',v_activity_seq);
                  END IF;
                  v_ttl_nbr_indiv := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_ttl_nbr_indiv := NULL;
              END IF;
            ELSE  --v_item is NULL
              v_ttl_nbr_indiv := NULL;
            END IF;

          ELSIF v_item_name = 'Lower Class Bound' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_brg_freq_analysis = 'P' THEN
                IF  v_item IS NOT NULL THEN
                  v_lower_bnd_amt := v_item;
                ELSE
                  v_status := 'N';-- v_item translated to null - set status to 'N'
                  sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Lower and Upper Bounds are required for physical classes',v_activity_seq);
                END IF;
              ELSE
                IF  v_item IS NOT NULL THEN
                  sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Lower Bound is not used except for Physical Class Indicators',v_activity_seq);
                END IF;
                v_lower_bnd_amt := NULL;
              END IF;
            ELSE
              v_lower_bnd_amt := NULL;
            END IF;

          -- Upper Boundary of class

          ELSIF v_item_name = 'Upper Class Bound' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_brg_freq_analysis = 'P' THEN
                IF v_item IS NOT NULL THEN
                  v_upper_bnd_amt := v_item;
                ELSE
                  v_status := 'N';-- v_item translated to null - set status to 'N'
                  sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Lower and Upper Bounds are required for physical classes',v_activity_seq);
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                  sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Upper Bounds are not used except for Physical Class Indicators',v_activity_seq);
                END IF;
                v_upper_bnd_amt := NULL;
              END IF;
            ELSE
              v_upper_bnd_amt := NULL;
            END IF;


/*          ELSIF v_item_name = 'Upper/Lower Bound Units'  THEN
            IF v_item IS NOT NULL THEN
              v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_dummy_char);
              IF v_storet_key.is_number IS NOT NULL THEN NULL;
                v_bound_unit_is_nbr := v_storet_key.is_number;
                v_bound_unit_org_id := v_storet_key.org_id;
              ELSE
                v_bound_unit_is_nbr := NULL;
                v_bound_unit_org_id := NULL;
                v_status := 'N';
                sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Upper/Lower Bound Unit was not found in STORET.',v_activity_seq);
              END IF;
            ELSE
              v_bound_unit_is_nbr := NULL;
              v_bound_unit_org_id := NULL;
            END IF;
*/

          ELSIF v_item_name = 'Individual Number'  THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_brg_type = 'Single Taxon Individuals' THEN               
                 --Checking for duplicate brg_is_number||v_individual_id will be done during the create/insert record phase
                 v_individual_nbr := v_item;            
                ELSE
                 v_individual_nbr := NULL;
                 sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Individual Number ignored. It is only valid for Single Taxon Individuals groups.',v_activity_seq);
                END IF; 
              ELSE
                -- Cannot translate to a NULL.              
                v_status := 'N';
                v_individual_nbr := NULL; 
              END IF;
            ELSE
              IF v_brg_type = 'Single Taxon Individuals' THEN
                v_status := 'N';
                v_individual_nbr := NULL; 
                sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Individual Number is required for Single Taxon Individuals group',v_activity_seq);    
              ELSE  
                v_individual_nbr := NULL; 
              END IF;  
            END IF;

          --Trap/Net details
          
          --------------------------------
          -- Trap or Net Sampling Duration
          --------------------------------
          
          ELSIF v_item_name = 'Trap or Net Sampling Duration' AND v_new_activity = 1 THEN
          
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              
              IF v_item IS NOT NULL THEN
              
                -- is the trawl gear for trap/net?  (See Sample Procedure ID)
                -- 1 = trawl, 2 = trap/net, 0 = all others
                IF v_check_trawl_gear = 2 THEN
                  -- if yes
                  v_success := Sim_import_result_extras.Validate_number(
                    v_item,
                    v_item_name,
                    '99.99', 
                    v_import_log_seq,
                    v_line_cnt
                  );
                  
                  IF v_success = 1 THEN
                    v_trap_net_duration := TO_NUMBER(v_item);
                  ELSE
                    v_status := 'N';
                    v_trap_net_duration := NULL;
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.',v_activity_seq);
                  v_trap_net_duration := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_trap_net_duration := NULL;
              END IF;
              
            ELSE
              v_trap_net_duration := NULL;
            END IF;

          -----------------------------
          -- Trap or Net Duration Units
          -----------------------------
          
          ELSIF v_item_name = 'Trap or Net Duration Units' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_check_trawl_gear = 2 THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_trap_net_dur_unit := Sim_import_result_extras.get_allowable_value(v_dummy_seq, v_import_log_seq);
                ELSE
                  v_status := 'N';
                  v_trap_net_dur_unit := NULL;
                END IF;
              ELSE
                Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.',v_activity_seq);
                v_trap_net_dur_unit := NULL;
              END IF;
              
            ELSE
              v_trap_net_dur_unit := NULL;
            END IF;

          ELSIF v_item_name = 'Relative Current Direction' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_check_trawl_gear = 2 OR v_check_trawl_gear = 1 THEN
                  v_success := Sim_import_result_extras.Validate_number(v_item,v_item_name,
                                               '999', v_import_log_seq,v_line_cnt);
                  IF v_success = 1 THEN
                    v_relative_current := TO_NUMBER(v_item);
                    IF v_relative_current < 0 OR v_relative_current > 359 THEN
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' must be an integer between 0 and 360.',v_activity_seq);
                    END IF;
                  ELSE
                    v_relative_current := NULL;
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow or Net/Horizontal Tow.',v_activity_seq);
                  v_relative_current := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_relative_current := NULL;
              END IF;
            ELSE
              v_relative_current := NULL;
            END IF;

          ELSIF v_item_name = 'Relative Wind Direction' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_check_trawl_gear = 1 OR v_check_trawl_gear = 2 THEN
                  v_success := Sim_import_result_extras.Validate_number(v_item,v_item_name,
                                               '999', v_import_log_seq,v_line_cnt);
                  IF v_success = 1 THEN
                    v_relative_wind := TO_NUMBER(v_item);
                    IF v_relative_wind < 0 OR v_relative_wind > 359 THEN
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_item_name||' must be an integer between 0 and 360.',v_activity_seq);
                    END IF;
                  ELSE
                    v_relative_wind := NULL;
                    v_success := 'N';
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow or Net/Horizontal Tow.',v_activity_seq);
                  v_relative_wind := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_relative_wind := NULL;
              END IF;
            ELSE
              v_relative_wind := NULL;
            END IF;

          -------------------------  
          -- Orientation to Current
          -------------------------
          ELSIF v_item_name = 'Orientation to Current' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_check_trawl_gear = 2 THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq, v_col_seq,v_activity_seq,v_import_log_seq,v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_orient_to_cur := Sim_import_result_extras.get_item_value(v_dummy_seq,v_line_cnt,v_import_log_seq);
                ELSE
                  v_status := 'N';
                  v_orient_to_cur := NULL;
                END IF;
              ELSE
                Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.',v_activity_seq);
                v_orient_to_cur := NULL;
              END IF;
            ELSE
              v_orient_to_cur := NULL;
            END IF;

          -----------------------
          -- Trap or Net Comments
          -----------------------
          
          ELSIF v_item_name = 'Trap or Net Comments' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_check_trawl_gear = 2 THEN
                  IF LENGTH(v_item) > 254 THEN
                    Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 254 characters and will be truncated.',v_activity_seq);
                    v_trap_net_comment := SUBSTR(v_item,1,254);
                  ELSE
                    v_trap_net_comment := v_item;
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.',v_activity_seq);
                  v_trap_net_comment := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_trap_net_comment := NULL;
              END IF;
            ELSE
              v_trap_net_comment := NULL;
            END IF;

          ELSIF v_item_name = 'Functional Feeding Group'  THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                  IF LENGTH(v_item) > 6 THEN
                    Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 6 characters and will be truncated.',v_activity_seq);
                    v_func_feed_grp := SUBSTR(v_item,1,6);
                  ELSE
                    v_func_feed_grp := v_item;
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Group Type is Multi-Taxon Population Ceusus.',v_activity_seq);
                  v_func_feed_grp := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_func_feed_grp := NULL;
              END IF;
            ELSE
              v_func_feed_grp := NULL;
            END IF;

          ELSIF v_item_name = 'Taxon Pollution Tolerance'  THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                  IF LENGTH(v_item) > 4 THEN
                    Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 4 characters and will be truncated.',v_activity_seq);
                    v_taxon_pollution := SUBSTR(v_item,1,4);
                  ELSE
                    v_taxon_pollution := v_item;
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Group Type is Multi-Taxon Population Ceusus.',v_activity_seq);
                  v_taxon_pollution := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_taxon_pollution := NULL;
              END IF;
            ELSE
              v_taxon_pollution := NULL;
            END IF;

          ELSIF v_item_name = 'Trophic Level'  THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                  IF LENGTH(v_item) > 4 THEN
                    Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 4 characters and will be truncated.',v_activity_seq);
                    v_trophic_level := SUBSTR(v_item,1,4);
                  ELSE
                    v_trophic_level := v_item;
                  END IF;
                ELSE
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not used unless Group Type is Multi-Taxon Population Ceusus.',v_activity_seq);
                  v_trophic_level := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_trophic_level := NULL;
              END IF;
            ELSE
              v_trophic_level := NULL;
            END IF;

          ELSIF v_item_name = 'Cell Form' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
              IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                IF v_community_is_nbr > 207 AND v_community_is_nbr < 211 THEN
                  v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq,
                              v_item_name, v_col_seq, 'TSRCLDES','CELL_TYPE_NM',
                              v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_char IS NOT NULL THEN
                    v_cell_type := rtrim(v_dummy_char);
                  ELSE
                    v_cell_type := '<Spaces>';
                    v_status := 'N';-- v_item invalid or translated to null - set status to 'N'                    
                  END IF;
                ELSE  --Wrong community type
                  v_cell_type := NULL;
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used with this community.',v_activity_seq);
                END IF;
              ELSE
                v_cell_type := NULL;
                Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used with this group type or intent.',v_activity_seq);
              END IF;
            ELSE
              IF v_brg_type IN ('Multi-Taxon Population Census') AND v_community_is_nbr > 207 AND v_community_is_nbr < 211 THEN
                v_cell_type := '<Spaces>';
              ELSE
                v_cell_type := NULL;
              END IF;
            END IF;

          ELSIF v_item_name = 'Cell Shape' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061603 slp - Removed trailing spaces.
              IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                IF v_community_is_nbr > 207 AND v_community_is_nbr < 211 THEN
                  v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq,
                              v_item_name, v_col_seq, 'TSRCLDES','CELL_SHAPE_TYPE_NM',
                              v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_char IS NOT NULL THEN
                    v_cell_shape := rtrim(v_dummy_char);
                  ELSE
                    v_cell_shape := '<Spaces>';
                    v_status := 'N';-- v_item invalid or translated to null - set status to 'N'                      
                  END IF;
                ELSE  --Wrong community type
                  v_cell_shape := NULL;
                  Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used with this community.',v_activity_seq);
                END IF;
              ELSE
                v_cell_shape := NULL;
                Sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used with this group type or intent.',v_activity_seq);
              END IF;
            ELSE
              IF v_brg_type IN ('Multi-Taxon Population Census') AND v_community_is_nbr > 207 AND v_community_is_nbr < 211 THEN
                v_cell_shape := '<Spaces>';
              ELSE
                v_cell_shape := NULL;
              END IF;
            END IF;


          ELSIF v_item_name = 'Habit' THEN
            IF v_item IS NOT NULL THEN
              v_storet_key := get_prmvl_key(v_item, p_imp_cfg_seq,v_item_name,
                              v_col_seq, 'TSRRSULT','HABIT',v_activity_seq, v_import_log_seq, v_line_cnt);
              IF v_storet_key.is_number IS NULL THEN
                v_status := 'N';
                v_habit_is_number := NULL;
                v_habit_org_id := NULL;
              ELSE
                v_habit_is_number := v_storet_key.is_number;
                v_habit_org_id := v_storet_key.org_id;
              END IF;
            ELSE
              v_habit_is_number := NULL;
              v_habit_org_id := NULL;
            END IF;


          ELSIF v_item_name = 'Voltinism' THEN
            IF v_item IS NOT NULL THEN
              v_storet_key := get_prmvl_key(v_item, p_imp_cfg_seq,v_item_name,
                              v_col_seq, 'TSRRSULT','VOLTINISM',v_activity_seq, v_import_log_seq, v_line_cnt);
              IF v_storet_key.is_number IS NULL THEN
                v_status := 'N';
                v_volt_is_number := NULL;
                v_volt_org_id := NULL;
              ELSE
                v_volt_is_number := v_storet_key.is_number;
                v_volt_org_id := v_storet_key.org_id;
              END IF;
            ELSE
              v_volt_is_number := NULL;
              v_volt_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Primary Class Descriptor' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_freq_analysis = 'B' THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_Cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_primary_class := sim_import_result_extras.get_allowable_value(v_dummy_seq,p_import_log_seq);
                  IF v_primary_class = 'SEX' THEN
                    v_bio_condition_is_nbr := 441;
                    v_bio_condition_org_id := 'ZZZZZZZZ';
                  ELSIF v_primary_class = 'LIFESTAGE' THEN
                    v_bio_condition_is_nbr := 559;
                    v_bio_condition_org_id := 'ZZZZZZZZ';
                  END IF;
                ELSE
                  v_status := 'N';
                  v_bio_condition_is_nbr := NULL;
                  v_bio_condition_org_id := NULL;
                  v_primary_class := NULL;
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.',v_activity_seq);
                v_bio_condition_is_nbr := NULL;
                v_bio_condition_org_id := NULL;
                v_primary_class := NULL;
              END IF;
            ELSE
              v_bio_condition_is_nbr := NULL;
              v_bio_condition_org_id := NULL;
              v_primary_class := NULL;
            END IF;


          ELSIF v_item_name = 'Secondary Class Descriptor' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_freq_analysis = 'B' THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq,v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_secondary_class := sim_import_result_extras.get_allowable_value(v_dummy_seq,p_import_log_seq);
                  IF v_secondary_class = 'SEX' THEN
                    v_sec_bio_cond_is_nbr := 441;
                    v_sec_bio_cond_org_id := 'ZZZZZZZZ';
                  ELSIF v_secondary_class = 'LIFESTAGE' THEN
                    v_sec_bio_cond_is_nbr := 559;
                    v_sec_bio_cond_org_id := 'ZZZZZZZZ';
                  END IF;
                ELSE
                  v_status := 'N';
                  v_sec_bio_cond_is_nbr := NULL;
                  v_sec_bio_cond_org_id := NULL;
                  v_secondary_class := NULL;
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.',v_activity_seq);
                v_sec_bio_cond_is_nbr := NULL;
                v_sec_bio_cond_org_id := NULL;
                v_secondary_class := NULL;
              END IF;
            ELSE
              v_sec_bio_cond_is_nbr := NULL;
              v_sec_bio_cond_org_id := NULL;
              v_secondary_class := NULL;
            END IF;

          ELSIF v_item_name = 'Primary Class Value' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_freq_analysis = 'B' THEN
                v_primary_class_value := validate_class_descriptor(v_item, p_imp_cfg_seq,v_bio_condition_is_nbr,
                                v_item_name, v_col_seq, v_activity_seq, v_line_cnt, v_import_log_seq);
                IF v_primary_class_value IS NULL THEN
                  v_status := 'N';
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.',v_activity_seq);
                v_primary_class_value := NULL;
              END IF;
            ELSE
              v_primary_class_value := NULL;
            END IF;

          ELSIF v_item_name = 'Secondary Class Value' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_brg_freq_analysis = 'B' THEN
                v_secondary_class_value := validate_class_descriptor(v_item, p_imp_cfg_seq,v_sec_bio_cond_is_nbr,
                                v_item_name, v_col_seq, v_activity_seq, v_line_cnt, v_import_log_seq);
                IF v_secondary_class_value IS NULL THEN
                  v_status := 'N';
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.',v_activity_seq);
                v_secondary_class_value := NULL;
              END IF;
            ELSE
              v_secondary_class_value := NULL;
            END IF;
            
          ----------------------
          -- jah 2-20-2004
          -- Particle Size Basis
          ----------------------
          ELSIF (v_item_name = 'Particle Size Basis') THEN
          
            -- Does the field contain anything?
            IF (v_item IS NOT NULL) THEN
              -- if yes
              
              -- translate the field
              v_item := sim_import_result_extras.check_translations(
                v_item, 
                p_imp_cfg_seq, 
                v_col_seq, 
                v_dummy,
                v_import_log_seq,
                v_line_cnt,
                v_activity_seq
              );
              
              -- did the field translate to okay?
              IF (v_item IS NOT NULL) THEN
                -- if yes
                
                -- prepare to INSERT Particle Size Basis INTO SIM_RESULT_CLASS_INDICATOR
                v_particle_size_basis := v_item;
                
              ELSE
                -- v_item translated to null
                
                -- prevent migration
                v_status := 'N';
                v_particle_size_basis := NULL;
              END IF;

            END IF;

          ELSIF v_item_name = 'Frequency Class Count' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := sim_import_result_extras.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy,v_import_log_seq,v_line_cnt,v_activity_seq);
              IF v_item IS NOT NULL THEN
                IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                  v_dummy_seq := sim_import_result_extras.validate_number(v_item,v_item_name,'999999999',v_import_log_seq,
                                                                          v_line_cnt);
                  IF v_dummy_seq = 1 THEN
                    v_class_count := v_item;
                    v_result_seq_num := v_result_seq_num + 1;
                  ELSE
                    v_class_count := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used for Single Taxon Frequency Classes.',v_activity_seq);
                  v_class_count := NULL;
                END IF;
              ELSE
                v_status := 'N';-- v_item translated to null - set status to 'N'
                v_class_count := NULL;
              END IF;
            ELSE
              v_class_count := NULL;
            END IF;


          ELSIF v_item_name = 'Field/Lab Procedure' OR v_item_name = 'Characteristic Field/Lab Analytical Procedure' THEN
            IF v_item IS NOT NULL AND v_habitat != 1  THEN
              IF v_item IS NOT NULL THEN
                
                -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                v_item := RTRIM(v_item);

                /* get the analysis procedure is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_lab_procedure
                );

                /* 06/09/03 SLP - If a storet key is returned, then the Field/Lab Proc ID provided is valid.  The key however may not
                                  represent the correct is number and org id as Field/Lab Proc Source was not used in this check.  The
                                  v_anlpr_is_nbr and v_anlpr_org_id are not set until the combination of ID and source are
                                  validated in the sim_import_result_extras.get_field_lab_procedure.  */

                v_anlpr_is_nbr := NULL;
                v_anlpr_org_id := NULL;

                IF v_storet_key.is_number IS NULL THEN
                  v_lab_procedure := NULL;
                  v_status := 'N';
                END IF;
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line '||v_line_cnt||': '||v_item_name||
                  ' has been provided and is not allowed for Habitat Assessment.',
                  v_activity_seq
                );
              END IF;
              v_anlpr_is_nbr := NULL;
              v_anlpr_org_id := NULL;
              v_lab_procedure := NULL;
            END IF;


          ELSIF v_item_name = 'Field/Lab Procedure Source' OR v_item_name = 'Characteristic Field/Lab Analytical Procedure Source' THEN
            
            IF v_item IS NOT NULL AND v_habitat != 1 THEN
            
              IF v_item IS NOT NULL THEN
            
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_storet_key := sim_import_result_extras.validate_storet_item(
                  v_item_name,
                  v_item,
                  p_imp_cfg_seq,
                  v_col_seq,
                  v_dummy_key,
                  p_org_id,
                  v_line_cnt,
                  v_activity_seq,
                  v_import_log_seq, 
                  v_lab_source
                );

                /* 06/09/03 SLP - If a storet key is returned, then the Field/Lab Proc Source provided is valid.  The key however may not
                                  represent the correct is number and org id as Field/Lab Proc ID was not used in this check.  The
                                  v_anlpr_is_nbr and v_anlpr_org_id are not set until the combination of ID and source are
                                  validated in the sim_import_result_extras.get_field_lab_procedure.  */

                v_anlpr_is_nbr := NULL;
                v_anlpr_org_id := NULL;

                IF v_storet_key.is_number IS NULL THEN
                  v_lab_source := NULL;
                  v_status := 'N';
                END IF;
              END IF;  
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Warning: Line '||v_line_cnt||': '||v_item_name||
                  ' has been provided and is not allowed for Habitat Assessment.',
                  v_activity_seq
                );
              END IF;
              v_anlpr_is_nbr := NULL;
              v_anlpr_org_id := NULL;
              v_lab_source := NULL;
            END IF;


          ELSIF v_item_name = 'Lab Sample Prep Procedure' THEN
            IF v_item IS NOT NULL AND v_habitat != 1  THEN
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                /* get the lab preparation procedure is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                     p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_labprep_id);

                /* 06/09/03 SLP - If a storet key is returned, then the LSPP ID provided is valid.  The key however may not
                                  represent the correct is number and org id as LSPP Source was not used in this check.  The
                                  v_labprep_is_nbr and v_labprep_org_id are not set until the combination of ID and source are
                                  validated in the sim_import_result_extras.get_lab_sample_prep_procedure.  */

                v_labprep_is_nbr := NULL;
                v_labprep_org_id := NULL;

                IF v_storet_key.is_number IS NULL THEN
                  v_labprep_id := NULL;
                  v_status := 'N';
                END IF;
              END IF;                
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_labprep_is_nbr := NULL;
              v_labprep_org_id := NULL;
              v_labprep_id := NULL;
            END IF;


          ELSIF v_item_name = 'Lab Sample Prep Procedure Source' THEN
            IF v_item IS NOT NULL AND v_habitat != 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                     p_org_id,v_line_cnt,v_activity_seq,v_import_log_seq, v_labprep_source);

                /* 06/09/03 SLP - If a storet key is returned, then the LSPP source provided is valid.  The key however may not
                                  represent the correct is number and org id as LSPP ID was not used in this check.  The
                                  v_labprep_is_nbr and v_labprep_org_id are not set until the combination of ID and source are
                                  validated in the sim_import_result_extras.get_lab_sample_prep_procedure.  */

                v_labprep_is_nbr := NULL;
                v_labprep_org_id := NULL;

                IF v_storet_key.is_number IS NULL THEN
                   v_labprep_source := NULL;
                   v_status := 'N';
                END IF;
              END IF;
            ELSE
              IF v_item IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': '||v_item_name||' has been provided and is not allowed for Habitat Assessment.',v_activity_seq);
              END IF;
              v_labprep_is_nbr := NULL;
              v_labprep_org_id := NULL;
              v_labprep_source := NULL;
            END IF;


          END IF;
        END IF;

      END LOOP;
      
      /******* End of column loop ********/



      ---------------------------------------------------------------------------------
      --
      --
      --  E N D   O F   C O L U M N   L O O P
      --
      --
      ---------------------------------------------------------------------------------



      ---------------------------------------------------------------------------------
      --
      --
      -- The following checks happen after one entire line has been read and validated.
      -- 
      -- 
      ---------------------------------------------------------------------------------
      
      

      /* bka (8-20-01) check to see if value type was included in row,
         if not, then add it with a default value of 'Actual' */
      IF v_value_type IS NULL AND v_habitat != 1 THEN
        IF v_brg_type <> 'Single Taxon Group Summary' THEN
          v_value_type := 'Actual';
          Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Value type is required and has not been included. A default value type of "Actual" has been assigned.',v_activity_seq);
        END IF;
      END IF;

      /* see if activity already exists in STORET */
      IF v_new_activity = 1 THEN
        
        /* dtw 10-21-03 - If Trip QC Sample. check to see if a valid Trip QC Type was included. */
		  	IF v_trip_qc = 1 AND v_activity_cat IS NULL THEN
			    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||':Trip QC Type is required when Activity Type = Trip QC.', NULL);
				  v_status := 'N';
        END IF;

        /* DTW 6/18/03 - Added logic to only check one time for each activity. */
        v_storet_key := Sim_import_result_checks.check_storet_act_id(v_activity_id, v_replicate_nbr,
                                                                     v_tsrstvst_is_nbr, p_org_id, v_activity_seq,
                                                                     v_line_cnt,v_import_log_seq);

        IF v_storet_key.is_number IS NOT NULL THEN
          -- activity already exists in STORET, create in sim but do not allow it to be exported
          v_act_is_nbr := v_storet_key.is_number;
          v_act_org_id := v_storet_key.org_id;
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_activity_id||' already exists in STORET, this activity will not be exported.',v_activity_seq);
        ELSE
          v_act_is_nbr := NULL;
          v_act_org_id := NULL;
        END IF;
      END IF;

      /* SF (10/7/02) Check Field/Lab Procedure */
      /* This routine validates the combination of field/lab procedure id and source. Source was added as an import column
         when we determined that ID is not unique within this table. */
      IF v_lab_procedure IS NOT NULL AND v_lab_source IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Field/Lab Procedure Source is required to identify Procedure.',v_activity_seq);
        v_status := 'N';
      ELSIF v_lab_procedure IS NULL AND v_lab_source IS NOT NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Field/Lab Procedure AND Source are both required to identify Procedure.',v_activity_seq);
        v_status := 'N';
      ELSIF v_lab_procedure IS NOT NULL AND v_lab_source IS NOT NULL THEN
        v_storet_key := sim_import_result_extras.get_field_lab_procedure(v_lab_procedure, v_lab_source,p_org_id,
        p_imp_cfg_seq,v_col_Seq, v_import_log_seq,v_line_cnt, v_activity_seq);
        IF v_storet_key.is_number IS NOT NULL THEN
          v_anlpr_is_nbr := v_storet_key.is_number;
          v_anlpr_org_id := v_storet_key.org_id;
        ELSE
          v_status := 'N';
        END IF;
      END IF;

      /* 06/09/03 SLP - Check Lab Sample Prep Procedure */
      /* This routine validates the combination of lab sample prep procedure ID and source. Both are required to
         guarantee uniqueness. Although ID is currently sufficient to uniquely identify the LSPP, we implemented
         this in the same fashion as field/lab procedure in order to protect against future data added to this table
         that violates this rule. (There is no unique constraint on lab sample procedure ID.) */
      IF v_labprep_id IS NOT NULL AND v_labprep_source IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Lab Sample Prep Procedure Source is required to identify Lab Sample Prep Procedure.',v_activity_seq);
        v_status := 'N';
      ELSIF v_labprep_id IS NULL AND v_labprep_source IS NOT NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Lab Sample Prep Procedure AND Source are both required to identify Lab Sample Prep Procedure.',v_activity_seq);
        v_status := 'N';
      ELSIF v_labprep_id IS NOT NULL AND v_labprep_source IS NOT NULL THEN
        v_storet_key := sim_import_result_extras.get_lab_sample_prep_procedure(v_labprep_id, v_labprep_source, p_org_id,
        p_imp_cfg_seq, v_col_Seq, v_import_log_seq, v_line_cnt, v_activity_seq);
        IF v_storet_key.is_number IS NOT NULL THEN
          v_labprep_is_nbr := v_storet_key.is_number;
          v_labprep_org_id := v_storet_key.org_id;
        ELSE
          v_status := 'N';
        END IF;
      END IF;

      /* Medium is not allowed for Habitat Assessments */
      IF v_medium IS NULL AND v_habitat != 1 THEN
        v_status := 'N';
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Medium is required and has not been provided',v_activity_seq);
      ELSIF v_medium IS NULL AND v_habitat = 1 THEN
        v_medium := ' ';
      ELSIF v_medium IS NOT NULL AND v_habitat = 1 THEN
        v_medium := ' ';
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Medium ignored. It is not used for Habitat Assesment.',v_activity_seq);
      END IF;

      -- This section is for Parent Sample ID
      IF NVL(v_no_parent_ids,0) != 0 THEN
  
        -- do we have too few Parent Sample IDs for 'Created from Sample'?
        IF (v_activity_cat = 'Created from Sample')
          AND (v_no_parent_ids < 1) THEN
          -- if yes
          -- set status flag to reject this import record
          v_status := 'N';
          -- output error message
          sim2_import_pkg.write_error(
            v_import_log_seq, 
            'Line ' || v_line_cnt || 
            ': A Parent Sample ID is required for "Created from Sample" activities.',
            v_activity_seq
          );
        END IF;
          
        -- do we have too many Parent Sample IDs for 'Created from Sample'?
        IF (v_activity_cat = 'Created from Sample')
          AND (v_no_parent_ids > 1) THEN
          -- if yes
          -- set status flag to reject this import record
          v_status := 'N';
          -- output error message
          sim2_import_pkg.write_error(
            v_import_log_seq, 
            'Line ' || v_line_cnt|| 
            ': Only one Parent Sample ID is allowed in "Created from Sample" activities.',
            v_activity_seq
          );
        END IF;
  
        -- do we have too few Parent Sample IDs for 'Composite-with Parents'?
        IF (v_activity_cat = 'Composite-with Parents')
          AND (v_no_parent_ids < 2) THEN
          -- if yes
          -- set status flag to reject this import record
          v_status := 'N';
          -- output error message
          sim2_import_pkg.write_error(
            v_import_log_seq, 
            'Line ' || v_line_cnt || 
            ': STORET requires at least two Parent Sample IDs for "Composite-with Parents"',
            v_activity_seq
          );
        END IF;
        
        -- Do we have any duplicate parent IDs in the table?
        FOR v_number_i IN 1..(v_no_parent_ids-1) LOOP
          FOR v_number_j IN (v_number_i+1)..v_no_parent_ids LOOP
          
            -- Do we have a match?
            IF (v_parent_id_table(v_number_i) = v_parent_id_table(v_number_j)) THEN
              -- if yes
              -- set status flag to reject this import record
              v_status := 'N';
              -- output error message
              sim2_import_pkg.write_error(
                v_import_log_seq, 
                'Line ' || v_line_cnt || 
                ': STORET does not allow duplicate Parent Sample IDs -- ' || 
                v_parent_id_table(v_number_i) || ' occurs more than once.',
                v_activity_seq
              );
            END IF;
          END LOOP;
        END LOOP;
  
        /*
          The following routine determines if the parent records in STORET have room 
          for "Composite-with Parent" records. Also, it determines if we have any
          station visit mis-matches.
        */
        
        -- Do we have any parent ids?
        IF (v_no_parent_ids > 0) THEN
          -- if yes
          -- retrieve the station visit information from SIM_STATION_VISIT
          OPEN  c_station_visit (v_visit_seq);
          FETCH c_station_visit INTO r_station_visit;
          CLOSE c_station_visit;
          
          FOR v_number IN 1..v_no_parent_ids LOOP

            -- Try to retrieve the parent record from STORET
            OPEN  c_tsrfdact (
                  v_parent_is_number_table(v_number),
                  v_parent_org_id_table(v_number));
            FETCH c_tsrfdact INTO r_tsrfdact;
            
            -- did we find the parent in STORET?
            IF (c_tsrfdact%FOUND) THEN
              -- if yes
              -- Does the parent record in STORET have room for a "Composite-with Parent"?
              IF (r_tsrfdact.TSRFDACT0IS_NUMBER IS NOT NULL) THEN
                -- if no
                -- set status flag to reject this import record
                v_status := 'N';
                -- output error message
                sim2_import_pkg.write_error(
                  v_import_log_seq, 
                  'Line ' || v_line_cnt || 
                  ': Parent Activity ' || v_parent_id_table(v_number) ||
                  ' in the STORET database already has a "Composite-with Parents" child record.',
                  v_activity_seq
                );
              END IF;

              -- do we have a station visit mis-match? (Re-check this in case
              -- the parent ID came from a translation.)
              IF (r_tsrfdact.tsrstvst_is_number != r_station_visit.ssv_tsrstvst_is_number)
                OR (r_tsrfdact.tsrstvst_org_id != r_station_visit.ssv_tsrstvst_org_id) THEN
                -- if yes
                -- set status flag to reject this import record
                v_status := 'N';
                -- output error message
                sim2_import_pkg.write_error(
                  v_import_log_seq, 
                  'Line ' || v_line_cnt || 
                  ': Parent Activity ' || v_parent_id_table(v_number) ||
                   ' has the wrong station visit.',
                   v_activity_seq
                );
              END IF;
            END IF;
            -- Close the tsrfdact cursor
            CLOSE c_tsrfdact;
            
            /*
              jah 2-19-04
              Do we already have a 'Composite...' child for this parent in SIM?
            */
            
            -- Is this a 'Composite...' child?
            IF (v_activity_cat = 'Composite-with Parents') THEN
              -- if yes
              
              -- Try to find a 'Composite ...' child for this parent already in SIM
              OPEN  c_activity_parent(
                v_parent_fa_seq_table(v_number),
                v_parent_is_number_table(v_number),
                v_parent_org_id_table(v_number),
                'Composite-with Parents'
              );
              FETCH c_activity_parent INTO r_activity_parent;
  
              -- Did we find a 'Composite...' child for this parent alreqady in SIM?
              IF c_activity_parent%FOUND THEN
                -- if yes
                -- set status flag to reject this import record
                v_status := 'N';
                -- output error message
                sim2_import_pkg.write_error(
                  v_import_log_seq, 
                  'Line ' || v_line_cnt || 
                  ': Parent Activity ' || v_parent_id_table(v_number) ||
                  ' already has a "Composite-with Parents" child in SIM.',
                  v_activity_seq
                );
              END IF;
              -- Close the c_activity_parent cursor
              CLOSE c_activity_parent;
            END IF;
            
          END LOOP;
        END IF;
      END IF;

      /* SF - 11/9/02 Default a missing QC Indicator to 'N' */
      IF v_Qc_Indicator IS NULL THEN
        v_qc_indicator := 'N';
      END IF;

      /* kms (6/28/01) - make sure field set id is included if field set name is there and
         vice versa. */
      IF v_field_set_nbr IS NOT NULL AND v_field_set_name IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field SET Name must be included when field set id is present.',v_activity_seq);
      ELSIF v_field_set_nbr IS NULL AND v_field_set_name IS NOT NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field set ID must be included when field set name is present.',v_activity_seq);
      END IF;
      

      -- jah 3-2-2004
      /* Business Rule
         The Gear Configuration ID must be valid for the Gear ID */
      
      -- do we need to check the gear configuration?
      IF (v_gearcfg_is_nbr IS NOT NULL) THEN
        -- if yes

        -- try to retrieve the gear configuration ID for the given gear ID
        OPEN c_get_gear_configuration(
          v_gear_is_nbr, 
          v_gear_org_id, 
          v_gearcfg_is_nbr, 
          v_gearcfg_org_id);
        FETCH c_get_gear_configuration INTO r_get_gear_configuration;
        
        -- is the Gear Configuration invalid?
        IF (c_get_gear_configuration%NOTFOUND) THEN
         -- if yes

         -- Prevent migration to STORET
          v_status := 'N';

          Sim2_Import_Pkg.write_error(
            v_import_log_seq, 
            'Line '||v_line_cnt||
            ': The selected Gear Configuration ID is not valid for the selected Gear ID.'
            ,v_activity_seq
          );
        END IF;
        CLOSE c_get_gear_configuration;

      END IF;
      

      /* make sure field gear types in the gear and collection procedure are the same */
      IF v_proc_is_nbr IS NOT NULL THEN
        /* If it's WEBSIM look for a matching Gear Config */
        IF v_sid_seq = 10007 AND v_type != 'Field Msr/Obs' THEN
          OPEN c_matching_config(v_proc_name);
            FETCH c_matching_config INTO v_gearcfg_is_nbr, v_gearcfg_org_id,v_gear_is_nbr, v_gear_org_id;
          CLOSE c_matching_config;
        END IF;

        /* see if gear is required, compare gear types and return 0 if they match */
        v_gear_type_ok := Sim_import_result_checks.check_gear_type(v_proc_is_nbr, v_gear_is_nbr, p_org_id, v_activity_seq,v_line_cnt,v_import_log_seq);
        IF v_gear_type_ok = 1 THEN
          /* gear types do not match, write error to log and change status to 'N' */
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Gear types for collection procedure and gear do not match.',v_activity_seq);
        ELSIF v_gear_type_ok = 2 THEN
          /* sample collection procedure has a gear type but gear has not been provided, invalidate activity
             and write error message */
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Gear is required when the collection procedure has a gear type.',v_activity_seq);
        END IF;
      ELSIF v_proc_is_nbr IS NULL AND v_new_activity = 1 AND v_type = 'Sample' THEN
        /* write error if sample collection procedure has not been included for a sample */
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Collection procedure is required and has not been included or is invalid.',v_activity_seq);
      END IF;

      /* make sure detection limits and quantification limits have a unit of measure */
      IF v_detect_limit IS NOT NULL AND v_det_uom_is_nbr IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when detection limit is given.',v_activity_seq);
        v_status := 'N';
      END IF;
      IF v_lowq_limit IS NOT NULL AND v_det_uom_is_nbr IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when lower quantification limit is given.',v_activity_seq);
        v_status := 'N';
      END IF;
      IF v_upq_limit IS NOT NULL AND v_det_uom_is_nbr IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when upper quantification limit is given.',v_activity_seq);
        v_status := 'N';
      END IF;

      /* 06/09/03 slp - Only set the result status to Final if the record is the first of the group.
                      - If the record is not the first for 'Single Taxon', then allow the status to be NULL. */
      /* SF - 1/8/02 Default a missing Result Status to "Final" */
      IF v_new_group = 1 OR v_brg_type != 'Single Taxon Frequency Classes' THEN
        IF v_result_status IS NULL THEN
          v_result_status := 'F';
        END IF;
      END IF;

      /* The following checks are for Biological Samples Only */
      /*******************************************************/
      IF v_medium = 'Biological' THEN

        /* SF (1/14/02) Check to see that intent is there for BIO */
        IF v_intent IS NULL AND v_new_activity = 1 THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Intent is required for Biological Results and has not been provided',v_activity_seq);
        END IF;

        /* SF (1/8/02) Check that community exists for Taxon Abundance */
        IF v_intent = 'Taxon Abundance' and v_community_is_nbr IS NULL THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Community is required for Taxon Abundance Intent',v_activity_seq);
        END IF;

        /* SF (1/8/02) Check that brg exists for Taxon Abundance */
        IF v_intent = 'Taxon Abundance' AND v_brg_id IS NULL AND v_new_group = 1 THEN
           sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Bio Results Group is required when Intent is Taxon Abundance.',v_activity_seq);
           v_status := 'N';
        END IF;

        /* SF (1/8/02) Check that br type exists for Taxon Abundance */
        IF v_intent = 'Taxon Abundance' AND v_brg_type IS NULL AND v_new_group = 1 THEN
           sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Bio Results Type is required when Intent is Taxon Abundance.',v_activity_seq);
           v_status := 'N';
        END IF;

        /* SF (1/8/02) Check that Subject Taxon exists for Tissue, Individual */
        IF v_intent IN ('Individual','Tissue') AND v_taxon_is_nbr IS NULL THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Subject Taxon is required and has not been provided',v_activity_seq);
        END IF;

        /* SF (1/8/02) Check that Bio Part exists for Tissue */
        IF v_intent = 'Tissue' and v_biopart_is_nbr IS NULL AND v_new_activity = 1 THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Bio Part is required for TISSUE',v_activity_seq);
        END IF;

        /* Subject Taxon Required for all BRG except MTPC*/
        IF v_brg_type LIKE 'Single%' AND v_taxon_is_nbr IS NULL THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Subject Taxon for Single Taxon group is required',v_activity_seq);
        END IF;

        /* SF (1/4/02) make sure there is a count type for STGS */
        IF v_value_type_name IS NULL THEN
          IF v_brg_type = 'Single Taxon Group Summary' THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Count Type is required for Single Taxon Results.',v_activity_seq);
            v_status := 'N';
          END IF;
        ELSE
          IF v_brg_type = 'Single Taxon Frequency Classes' THEN
            --it goes on the result instead of on the BRG
            v_value_type := v_value_type_name;
            v_value_type_name := NULL;
          END IF;
        END IF;

        /* SF (1/8/02) Check that total number exists for Individual, group summary */
        IF v_brg_type = 'Single Taxon Group Summary' AND v_ttl_nbr_in_grp IS NULL THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Total Number in Group is required and has not been provided',v_activity_seq);
        END IF;

        IF v_brg_type = 'Single Taxon Individuals' AND v_ttl_nbr_indiv IS NULL THEN
          v_status := 'N';
          sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Number of Individuals in Group is required and has not been provided',v_activity_seq);
        END IF;

        /* SF - 10/2/02 Check Frequency Class rules */
        IF v_brg_type = 'Single Taxon Frequency Classes' THEN

          /* SF (1/15/02) Check Bio Rslt Type Ind for STFC */
          IF v_brg_freq_analysis IS NULL THEN
            v_status := 'N';
            sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Frequency Analysis Type is required for Single Taxon Frequency Classes',v_activity_seq);
          END IF;

          IF v_brg_freq_analysis = 'B' THEN
            IF v_primary_class IS NULL THEN
              v_status := 'N';
              sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Primary Class Descriptor is required for BIOLOGICAL Frequency Analyses.',v_activity_seq);
              IF v_secondary_class IS NOT NULL THEN
                sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Secondary Class Descriptor ignored. It is only used when Primary Class Descriptor is also given.',v_activity_seq);
              END IF;
            ELSIF v_primary_class = v_secondary_class THEN
              -- DTW 6/16/03 - Validate that Primary and Secondary Class Descriptors are not the same
              v_status := 'N';
              sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_primary_class||' cannot be provided as both the Primary and Secondary Class Descriptors.',v_activity_seq);
            END IF;

            IF v_sexname_is_nbr IS NOT NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Sex ignored. It is not used for Biological frequency analyses.',v_activity_seq);
              v_sexname_is_nbr := NULL;
              v_sexname_org_id := NULL;
            END IF;

            IF v_lifestage_is_nbr IS NOT NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Life Stage ignored. It is not used for Biological frequency analyses.',v_activity_seq);
              v_lifestage_is_nbr := NULL;
              v_lifestage_org_id := NULL;
            END IF;

            IF v_value_type IS NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Bio Count Type is required for Single Taxon Frequency Classes.',v_activity_seq);
              v_status := 'N';
            END IF;
          END IF;
        END IF;

        /* Validate Species validity and requirements */
        IF v_intent = 'Taxon Abundance' THEN
          v_act_Species := NULL;
          IF v_brg_type LIKE 'Single%' THEN  -- Single Taxon Group Summary, Single Taxon Individual, or Single Taxon Frequency Class
            v_result_species := NULL;
            -- Taxa recorded at group level, only check first row in group.
            IF v_new_group = 1 THEN
              -- Species never required, but only allowed for certain taxa
              v_allow_species := check_allow_species(v_taxon_is_nbr);
              IF v_allow_species != 1 AND v_taxon_species_number IS NOT NULL THEN
                sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Species Number ignored. The Subject Taxon is already identified to the species level.',v_activity_seq);
                v_taxon_species_number := NULL;
              END IF;
              v_BRG_Species := v_taxon_species_number;
            END IF;
          ELSIF v_brg_type LIKE 'Multi%' THEN
            -- Taxa recorded at result level, check each row in group.  Taxa stored in Characteristic instead of subject taxon.
            v_allow_species := check_allow_species(v_char_is_nbr);
            IF v_allow_species != 1 AND v_char_species_number IS NOT NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Species Number ignored. The Taxa Characteristic is already identified to the species level.',v_activity_seq);
              v_char_species_number := NULL;
            END IF;
            v_result_species := v_char_species_number;
          END IF;
        ELSIF v_intent = 'Tissue' OR v_intent = 'Individual' THEN
          -- Subject Taxon Species recorded at the Activity Level
          -- Can also have Characteristic Taxa recorded at Result Level
          -- When Subject Taxon (for the Activity) is allowed, it is required.
          v_brg_species := NULL;
          IF v_new_activity = 1 THEN
            v_allow_species := check_allow_species(v_taxon_is_nbr);
            IF v_allow_species != 1 AND v_taxon_species_number IS NOT NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Species Number ignored. The Subject Taxon is already identified to the species level.',v_activity_seq);
              v_taxon_species_number := NULL;
            ELSIF v_allow_species = 1 AND v_taxon_species_number IS NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Species Number Required for Subject Taxon.',v_activity_seq);
              v_status := 'N';
            END IF;
            v_act_species := v_taxon_species_number;
          END IF;
          -- Now Check the Results level - Species is Never required at the results level.
          IF v_char_result_type = 'TAXA' THEN
            v_allow_species := check_allow_species(v_char_is_nbr);
            IF v_allow_species != 1 AND v_char_species_number IS NOT NULL THEN
              sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Species Number ignored. The Taxa Characteristic is already identified to the species level.',v_activity_seq);
              v_char_species_number := NULL;
            END IF;
            v_result_species := v_char_species_number;
          ELSE
            v_result_species := NULL;
          END IF;
        END IF;

        IF v_brg_type = 'Single Taxon Frequency Classes' THEN
          IF v_brg_freq_analysis = 'P' AND v_new_group = 1 THEN
            v_success := sim_import_result_checks.check_class_descriptor(v_char_is_nbr, v_import_log_seq);
            IF v_success = 0 THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic must be a Common Class Descriptor for Single Taxon Frequency Classes.',v_activity_seq);
              v_status := 'N';
            END IF;
          ELSIF v_brg_freq_analysis = 'B'  THEN
            IF v_char_is_nbr IS NOT NULL THEN
              --Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic not used for BIOLOGICAL Frequency Analyses. Use Primary and Secondary Class Descriptors instead.',v_activity_seq);
              v_char_is_nbr := NULL;
              v_char_org_id := NULL;
            END IF;
          END IF;
        ELSIF v_brg_type = 'Single Taxon Group Summary' OR v_brg_type = 'Single Taxon Individuals' AND v_new_group = 1 THEN
          v_success := sim_import_result_checks.check_class_descriptor(v_char_is_nbr, v_import_log_seq);
          IF v_success = 0 THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic must be of category BIOLOGICAL/PHYSICAL MEASUREMENT for Single Taxon Results Groups.',v_activity_seq);
            v_status := 'N';
          END IF;
        END IF;

      
        -- jah 5-13-04 Added Trap and trawl warnings
        -- Trap and trawl warnings

        -- New Activity?
        IF v_new_activity = 1 THEN
          -- if yes
  
          ------------------
          -- Distance Fished
          ------------------
          
          -- Does import contain Distance Fished
          -- but the required field Fished Duration missing?
          IF v_distance_fished IS NOT NULL
          AND v_fished_duration IS NULL 
          THEN
            -- if yes            
            sim2_import_pkg.write_error(
              v_import_log_seq, 
              'Warning: Line ' || v_line_cnt ||
              ': Distance Fished ignored. ' ||
              'It only applies when Fished Duration is given.',
              v_activity_seq
            );          
          END IF;
          
          --------------------------
          -- Relative Wind Direction
          -------------------------- 
                 
          -- Does import contain Relative Wind Direction
          -- but the required field, either Fished Duration
          -- or Trap or Net Sampling Duration, is missing?
          IF (v_relative_wind IS NOT NULL)
          AND (v_fished_duration IS NULL)
          AND (v_trap_net_duration IS NULL)
          THEN
            -- if yes            
            sim2_import_pkg.write_error(
              v_import_log_seq, 
              'Warning: Line ' || v_line_cnt ||
              ': Relative Wind Direction ignored. ' ||
              'It only applies when ' ||
              'either Fished Duration or Trap or Net Sampling Duration is given.',
              v_activity_seq
            );          
          END IF;
          
          -----------------------------
          -- Relative Current Direction
          -----------------------------
          
          -- Does import contain Relative Current Direction        
          -- but the required field, either Fished Duration
          -- or Trap or Net Sampling Duration, is missing?
          IF v_relative_current IS NOT NULL
          AND (v_fished_duration IS NULL)
          AND (v_trap_net_duration IS NULL)
          THEN
            -- if yes            
            sim2_import_pkg.write_error(
              v_import_log_seq, 
              'Warning: Line ' || v_line_cnt ||
              ': Relative Current Direction ignored. ' ||
              'It only applies when ' ||
              'either Fished Duration or Trap or Net Sampling Duration is given.',
              v_activity_seq
            );          
          END IF;
  
          -------------------------  
          -- Orientation to Current
          -------------------------
  
          -- Does import contain Orientation to Current
          -- but the required field, Trap or Net Sampling Duration, is missing?      
          IF (v_orient_to_cur IS NOT NULL)
          AND (v_trap_net_duration IS NULL)
          THEN
            -- if yes            
            sim2_import_pkg.write_error(
              v_import_log_seq, 
              'Warning: Line ' || v_line_cnt ||
              ': Relative Orientation to Current ignored. ' ||
              'It only applies when ' ||
              'Trap or Net Sampling Duration is given.',
              v_activity_seq
            );          
          END IF;
  
          -----------------------        
          -- Trap or Net Comments
          -----------------------
          IF (v_trap_net_comment IS NOT NULL)
          AND (v_trap_net_duration IS NULL)
          THEN
            -- if yes            
            sim2_import_pkg.write_error(
              v_import_log_seq, 
              'Warning: Line ' || v_line_cnt ||
              ': Trap or Net Comments ignored ' ||
              'It only applies when ' ||
              'Trap or Net Sampling Duration is given.',
              v_activity_seq
            );          
          END IF;
          
        END IF;
        
      END IF;  -- End of Biological Only Checks


      /* check all units of measure */
      IF v_smpl_wt IS NOT NULL AND v_total_wt_uom IS NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Total Sample Weight Units required when Total Sample Weight is given',v_activity_seq);
        v_status := 'N';
      END IF;

      IF v_smpl_wt IS NULL AND v_total_wt_uom IS NOT NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Total Sample Weight Units ignored. It applies only when Total Sample Weight is given.',v_activity_seq);
        v_total_wt_uom := NULL;
      END IF;

      IF v_depth IS NOT NULL AND v_depth_uom IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Depth Unit is required if depth has been provided.',v_activity_seq);
      END IF;

      IF v_depth IS NULL AND v_depth_uom IS NOT NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Activity Depth Units ignored. It applies only when Activity Depth is given.',v_activity_seq);
        v_depth_uom := NULL;
      END IF;

      IF (v_low_depth IS NOT NULL OR v_up_depth IS NOT NULL) AND v_depth_range_uom IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Upper/Lower Depth Unit is required if Upper/Lower Depth is provided.',v_activity_seq);
      END IF;

      IF (v_low_depth IS NULL AND v_up_depth IS NULL) AND v_depth_range_uom IS NOT NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Upper/Lower Depth Units ignored. It applies only when Upper/Lower Depth is given.',v_activity_seq);
        v_depth_range_uom := NULL;
      END IF;

      IF v_fished_duration IS NOT NULL AND v_fished_dur_uom IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Fished Duration Unit is required when Fished Duration is provided.',v_activity_seq);
      END IF;

      IF v_fished_duration IS NULL AND v_fished_dur_uom IS NOT NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Fished Duration Units ignored. It applies only when Fished Duration is given.',v_activity_seq);
        v_fished_dur_uom := NULL;
      END IF;

      IF v_distance_fished IS NOT NULL AND v_dist_fished_uom IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Distance Fished Unit is required when Distance Fished is provided.',v_activity_seq);
      END IF;

      IF v_distance_fished IS NULL AND v_dist_fished_uom IS NOT NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Distance Fished Units ignored. It applies only when Distance Fished is given.',v_activity_seq);
        v_dist_fished_uom := NULL;
      END IF;
      

      IF (v_lower_bnd_amt IS NOT NULL OR v_upper_bnd_amt IS NOT NULL) AND v_uom_is_nbr IS NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Common Class Unit is required when Upper and Lower Bounds are given.',v_activity_seq);
        v_status := 'N';
      END IF;

      IF (v_lower_bnd_amt IS NULL OR v_upper_bnd_amt IS NULL) AND v_brg_freq_analysis = 'P' THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Upper and Lower Bounds are required for Physical Frequency Analyses.',v_activity_seq);
        v_status := 'N';
      END IF;

      IF v_trap_net_duration IS NOT NULL AND v_trap_net_dur_unit IS NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trap/Net Sampling Duration Unit is required when Trap/Net Sampling Duration is given.',v_activity_seq);
        v_status := 'N';
      END IF;

      IF v_trap_net_duration IS NULL AND v_trap_net_dur_unit IS NOT NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Trap/Net Sampling Duration Units ignored. It applies only when Trap/Net Sampling Duration is given.',v_activity_seq);
        v_trap_net_dur_unit := NULL;
      END IF;

      /* check sequence number and point type */
      IF v_sequence_number IS NULL AND v_well_pipe_num IS NULL AND v_point_type IS NOT NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Sequence Number or Well/Pipe ID is required when Point Type is provided.',v_activity_seq);
        v_point_type := NULL;
        v_status := 'N';
      END IF;

      IF (v_sequence_number IS NOT NULL AND v_sequence_number != -1) AND v_point_type IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Point Type is required when Sequence Number is provided.',v_activity_seq);
        v_status := 'N';
        v_sequence_number := NULL;
      END IF;

      IF v_point_type = 'WELL HEAD' OR v_point_type = 'PIPE END' THEN
        IF v_well_pipe_num IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Well or Pipe ID is required when Point Type is WELL HEAD or PIPE END.',v_activity_seq);
          v_point_type := NULL;
          v_sequence_number := NULL;
          v_status := 'N';
        END IF;
      END IF;

      /* Make sure both upper and lower bounds are given */
      IF v_upper_bnd_amt IS NULL AND v_lower_bnd_amt IS NOT NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Upper Bound is required when Lower Bound is given.',v_activity_seq);
        v_status := 'N';
      END IF;

      IF v_upper_bnd_amt IS NOT NULL AND v_lower_bnd_amt IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Lower Bound is required when Upper Bound is given.',v_activity_seq);
        v_status := 'N';
      END IF;

      IF v_new_activity = 1 THEN
        /* Check times and dates on new activities */
        IF v_start_time IS NOT NULL THEN
          IF v_start_date IS NULL THEN
            Sim2_Import_Pkg.write_error(
              v_import_log_seq, 
              'Line '||v_line_cnt||': ' ||
              'Activity Start Date is required when Activity Start Time given.',
              v_activity_seq
            );
            v_status := 'N';
          END IF;
          IF v_start_time_zone IS NULL THEN
            Sim2_Import_Pkg.write_error(
              v_import_log_seq, 
              'Line '||v_line_cnt||': ' ||
              'Activity Start Time Zone is required when Activity Start Time is given.',
              v_activity_seq
            );
            v_status := 'N';
          END IF;
        END IF;
        IF v_end_time IS NOT NULL THEN
          IF v_end_date IS NULL THEN
            Sim2_Import_Pkg.write_error(
              v_import_log_seq, 
              'Line '||v_line_cnt||': ' ||
              'Activity Stop Date is required when Stop Time given.',
              v_activity_seq
            );
            v_status := 'N';
          END IF;
          IF v_stop_time_zone IS NULL THEN
            Sim2_Import_Pkg.write_error(
              v_import_log_seq, 
              'Line '||v_line_cnt||': ' ||
              'Activity Stop Time Zone is required when Activity Stop Time is given.',
              v_activity_seq
            );
            v_status := 'N';
          END IF;
        END IF;

        --does the visit start after the trip?
        IF v_visit_start_date < v_trip_start_date  THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Visit Start Date is before Trip Start Date.',v_activity_seq);
          v_status := 'N';
        END IF;

        --does the trip start before the visit ends?
        IF v_visit_end_date IS NOT NULL AND v_visit_end_date < v_trip_start_date THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trip starts after the Visit End Date.',v_activity_seq);
          v_status := 'N';
        END IF;

        --does the Activity start before the visit starts?
        IF v_start_date < v_visit_start_date THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field Activity starts before the Station Visit Start Date.',v_activity_seq);
          v_status := 'N';
        END IF;

        --does the Activity end before it starts?
        IF v_end_date IS NOT NULL THEN
          IF v_end_date < v_start_date THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity End date is before Activity Start Date.',v_activity_seq);
            v_status := 'N';
          END IF;

          --does the activity end after the visit ends?
          IF v_visit_end_Date IS NOT NULL THEN
            IF v_end_date > v_visit_end_date THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity End date is after the Station Visit End Date.',v_activity_seq);
              v_status := 'N';
            END IF;
          END IF;

          --does the Activity start after the Visit ends?
          IF v_start_date > v_visit_end_date THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity Start Date is after the Station Visit End Date.',v_activity_seq);
            v_status := 'N';
          END IF;
        END IF;
      END IF;

      /* */
      IF v_char_is_nbr IS NULL AND v_brg_type = 'Multi-Taxon Population Census' AND v_taxon_is_nbr IS NOT NULL THEN
        v_char_is_nbr := v_taxon_is_nbr;
        v_char_org_id := v_taxon_org_id;
        v_char_result_type := get_char_result_type(v_char_is_nbr);

      ELSIF v_result IS NOT NULL AND v_char_is_nbr IS NULL AND v_tsrhcsc_is_nbr IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Either a characteristic or characteristic group must be provided.',v_activity_seq);
      END IF;

      IF v_char_result_type = 'TAXA' AND v_medium != 'Biological' THEN
        v_status := 'N';
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': TAXA Type characteristic is not allowed on non-biological activity.',v_activity_seq);
      END IF;

      IF v_taxon_species_number IS NOT NULL AND v_taxon_is_nbr IS NULL THEN
        sim2_import_pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Subject Taxon Species Number ignored. It is not used unless Subject Taxon is given.',v_activity_seq);
        v_taxon_species_number := NULL;
      END IF;

      -- Check for Result Value and Units
      IF v_result IS NULL THEN
        IF v_detection = 'Detected not Quantified' THEN
          v_result := '*Present';
        ELSIF v_detection = 'Not Detected' THEN
          v_result := '*Non-detect';
        ELSIF v_detection = 'Present above Quantification' THEN
          v_result := '*Present >QL';
        ELSIF v_detection = 'Present below Quantification' THEN
          v_result := '*Present <QL';
        ELSIF v_detection = 'Not Reported' THEN
          v_result := '*Not Reported';
        END IF;
      END IF;

      IF v_brg_type = 'Single Taxon Frequency Classes' THEN
        IF v_class_count IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Frequency Class Count is required for Single Taxon Frequency Classes.',v_activity_seq);
          v_status := 'N';
        ELSE
          v_result := v_class_count;
        END IF;
        IF v_brg_freq_analysis = 'B' THEN
          /*  Hard-coded unit of measure for 'Count' for Biological Measures*/
          v_uom_is_nbr := 72;
          v_uom_org_id := 'ZZZZZZZZ';
        ELSIF v_brg_freq_analysis = 'P' THEN
          IF v_uom_is_nbr IS NULL THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Common Class Descriptor Units is required.',v_activity_seq);
            v_status := 'N';
          END IF;
        END IF;
      END IF;

      /* Check the characteristic type, if physical or permitted value type, perform
         further checks on result.
         If the char type is text, copy result value into the result comment part of the code. */

      IF v_result IS NOT NULL THEN
        v_result_save := v_result;
        v_detect_check := 1;
        v_result := sim_import_result_extras.update_result(
          v_result,
          v_char_is_nbr,
          v_units_ok,
          v_result_ok,
          p_imp_cfg_seq,
          v_activity_seq,
          v_char_result_type,
          v_import_log_seq,
          v_line_cnt,
          v_detection,
          v_detect_check
        );
        
        IF v_result_ok = FALSE THEN
          v_status := 'N';
        END IF;
        
        IF v_detect_check = 0 THEN
          Sim2_Import_Pkg.write_error(
            v_import_log_seq, 
            'Warning: Line '||v_line_cnt||
            ': Result Value does not match your Detection Condition and has beed modified.',
            v_activity_seq
          );
        END IF;

        IF v_char_result_type IN ('TEXT') THEN
          v_result_cmt := v_result_save;
          v_result_save := NULL; -- reset after each use.
        END IF;

      END IF;

      IF v_char_is_nbr IS NOT NULL THEN

        v_success := Sim_import_result_checks.check_chardet_comp(
          v_char_is_nbr, 
          v_char_org_id, 
          v_anlpr_is_nbr,
          v_uom_is_nbr,
          v_uom_org_id, 
          v_fraction_is_nbr, 
          v_value_type, 
          v_units_ok,
          v_brg_type, 
          v_activity_seq,
          v_line_cnt,
          v_import_log_seq,
          v_char_result_type
        );

        IF v_success = 2 THEN
          v_status := 'N';
        ELSIF v_success = 3 THEN
          v_uom_is_nbr := NULL;
          v_uom_org_id := NULL;
        ELSIF v_success = 4 THEN
          v_uom_is_nbr := 1;
          v_uom_org_id := 'ZZZZZZZZ';
        ELSIF v_success = 5 THEN
          v_value_type := 'Actual';
        END IF;

      END IF;

      ---------------------------------------------------------------------
      -- see if characteristic already exists on activity (in SIM)
      -- jah 2-25-04 
      --     It is okay to get duplicate results from Portable Data Loggers
      ---------------------------------------------------------------------
      
      -- do we have a result?
      IF (v_result IS NOT NULL) THEN
        -- if yes
        
        -- jah 2-25-04 
        -- is result from something other than a Portable Data Logger?
        IF (v_activity_cat != 'Portable Data Logger') THEN
          -- if yes 
  
          -- check for duplicates and issue warnings as necessary
          IF sim_import_result_checks.check_duplicate_result(
            v_char_is_nbr,
            v_activity_seq,
            v_brg_id,
            v_individual_nbr,
            v_brg_type,
            v_result,
            v_uom_is_nbr,
            v_char_species_number,
            v_lower_bnd_amt, 
            v_upper_bnd_amt,
            v_line_cnt,
            v_char_result_type,
            p_import_log_seq) = FALSE THEN
  
            -- do nothing; CHECK_DUPLICATE_RESULT issues the warnings if necessary
            NULL;
  
          END IF;
          
        END IF;
        
      END IF;

      IF NOT v_result_ok THEN
        v_status := 'N';
      ELSIF v_result = '0' THEN
        v_result_value_msr := 0;
      END IF;

      /* Check for dup brgi */
      IF  v_lower_bnd_amt IS NOT NULL OR v_primary_class IS NOT NULL THEN
        v_dup_brgi_success := Sim_import_result_checks.check_dup_class(v_activity_seq, v_result_rec.RS_BRG_SEQ, v_lower_bnd_amt, v_upper_bnd_amt,v_line_cnt,v_import_log_seq);
        IF v_dup_brgi_success = 0 THEN -- dupe class found
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A duplicate class exists for this group - duplicate result or invalid class bounds likely',v_activity_seq);
        END IF;
      END IF;

      /* make sure the TSMALP record exists */
      IF v_new_activity = 1 AND v_tsmstatn_is_nbr IS NOT NULL  THEN --AND v_sequence_number != -1 THEN
        IF v_point_type IS NULL  THEN --default the AAL to Point of Record and, if app, Trawl Start/Stop
          v_point_type := '*POINT OF RECORD';
          v_sequence_number := 0;
        END IF;
        --go get the tsmalp is number
        IF NVL(v_sequence_number,1) != -1 THEN
          IF v_point_type != 'WELL HEAD' AND v_point_type != 'PIPE END' THEN
          --would already have TSMALP info if WELL OR PIPE
            v_storet_key := sim_import_result_extras.get_tsmalp_is_nbr(v_point_type,v_sequence_number,v_tsmstatn_is_nbr,v_tsmstatn_org_id,v_line_cnt,v_import_log_seq);
            IF v_storet_key.is_number IS NOT NULL THEN
              v_tsmalp_is_nbr := v_storet_key.is_number;
              v_tsmalp_org_id := v_storet_key.org_id;
            ELSE
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A tsmalp location record was not found ',v_activity_seq);
            END IF;
          END IF;
        END IF;
      END IF;

      /* Check all the result stuff */
      IF v_result IS NOT NULL THEN
      
        --  We do not support Upper and Lower Range checking in SIM v2.0.1.  Default values for
        --  Physical Characteristics to NULL and all else to zero to match the STORET interface.
        IF v_char_result_type = 'PHYS' THEN
          v_upper_range := NULL;
          v_lower_range := NULL;
        ELSE 
          v_upper_range := 0;
          v_lower_range := 0;
        END IF;  
        
        IF v_char_result_type IN ('TEXT','PV  ')  THEN -- DCL added this to keep extraneous data from being imported if the result is text or pv
          IF v_DETECT_LIMIT IS NOT NULL THEN
            v_DETECT_LIMIT := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Detection Limit not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_LAB_CERT IS NOT NULL THEN
            v_LAB_CERT := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Lab Certification not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_ANALYSIS_DATE IS NOT NULL THEN
            v_ANALYSIS_DATE := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Analysis Date not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_ANALYSIS_TIME IS NOT NULL THEN
            v_ANALYSIS_TIME := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Analysis Time not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_det_uom_is_nbr IS NOT NULL THEN
            v_det_uom_is_nbr := NULL;
            v_det_uom_org_id := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Detection limit UOM not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_DET_COMMENT IS NOT NULL THEN
            v_DET_COMMENT := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Detect Comment not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_UPQ_LIMIT IS NOT NULL THEN
            v_UPQ_LIMIT := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Upper Quantificatin Limit not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_LOWQ_LIMIT IS NOT NULL THEN
            v_LOWQ_LIMIT := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Lower Quantification Limit not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_UOM_IS_NBR IS NOT NULL THEN
            v_UOM_IS_NBR := NULL;
            v_UOM_ORG_ID := NULL;
            --Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Unit of Measure not needed for text type result',v_activity_seq);
          END IF;
          IF v_FRACTION IS NOT NULL THEN
            v_FRACTION := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Fraction not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_value_type IS NOT NULL THEN
            v_value_type := NULL;
            -- Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Value Type not needed for text type result',v_activity_seq);
          END IF;
          IF v_stat_type IS NOT NULL THEN
            v_stat_type := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Statistic Type not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_ANLPR_IS_nbr IS NOT NULL THEN
            v_ANLPR_IS_nbr := NULL;
            v_ANLPR_ORG_ID := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Analysis Procedure not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_labprep_IS_nbr IS NOT NULL THEN
            v_labprep_IS_nbr := NULL;
            v_labprep_ORG_ID := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Lab Procedure not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_duration_basis IS NOT NULL THEN
            v_duration_basis := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Duration Basis not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_TEMP_BASIS IS NOT NULL THEN
            v_TEMP_BASIS := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Temperature Basis not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_PRECISION IS NOT NULL THEN
            v_PRECISION := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Precision not needed for choice list or text type results.',v_activity_seq);
          END IF;
          IF v_conf_level IS NOT NULL THEN
            v_conf_level := NULL;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Confidence Level not needed for choice list or text type results.',v_activity_seq);
          END IF;
        END IF;
      END IF;

      /* Make sure no replicate number for trip QC. */
      IF v_trip_qc = 1 AND v_replicate_nbr IS NOT NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Replicate Number ignored. It is not used for Trip QC.',v_activity_seq);
        v_replicate_nbr := NULL;
      END IF;

      /*Check for projects */
      IF v_no_projects IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Project is required and has not been provided.',v_activity_seq);
      END IF;

      
      ---------------------------------------------------------------------------------
      --
      --
      --  create activity
      --
      --
      ---------------------------------------------------------------------------------

      BEGIN
        IF v_new_activity = 1 THEN
          IF v_status = 'N' THEN
            v_approved_on := NULL;
            v_approved_by := NULL;
          ELSE
            v_approved_on := SYSDATE;
            v_approved_by := USER;
          END IF;
          IF v_activity_cat = 'Created from Sample' THEN -- get rid of unwanted activity items
            v_depth := NULL;
            v_depth_uom := NULL;
            v_depth_ref := NULL;
            v_depth_ref_seq := NULL;
            v_depth_ref_unit := NULL;
            v_depth_range_uom := NULL;
            v_depth_ref := NULL;
            v_depth_zone := NULL;
            v_up_depth := NULL;
            v_low_depth := NULL;
            v_thermo := NULL;
            v_halo := NULL;
            v_pycno := NULL;
          END IF;
          /* assign values to record in order to send all of these variables to function easily */
          v_fa_rec.FA_SEQ := v_activity_seq;
          v_fa_rec.FA_TYPE_NAME := v_type;
          v_fa_rec.FA_CATEGORY :=  v_activity_cat;
          v_fa_rec.FA_ACTIVITY_ID := v_activity_id;
          v_fa_rec.FA_START_DATE := v_start_date;
          v_fa_rec.FA_START_TIME := v_start_time;
          v_fa_rec.FA_START_TIME_ZONE := v_start_time_zone;
          v_fa_rec.FA_STOP_DATE := v_end_date;
          v_fa_rec.FA_STOP_TIME := v_end_time;
          v_fa_rec.FA_STOP_TIME_ZONE := v_stop_time_zone;
          v_fa_rec.FA_DEPTH := v_depth;
          v_fa_rec.FA_DEPTH_REF_POINT := v_depth_ref;
          v_fa_rec.FA_ACTIVITY_COMMENTS := v_comment;
          v_fa_rec.FA_TSMSTATN_IS_NUMBER := v_tsmstatn_is_nbr;
          v_fa_rec.FA_TSMSTATN_ORG_ID := v_tsmstatn_org_id;
          v_fa_rec.FA_TSRTRIP_IS_NUMBER := v_trip_is_nbr;
          v_fa_rec.FA_TSRTRIP_ORG_ID := v_trip_org_id;
          v_fa_rec.FA_TSRSTVST_IS_NUMBER := v_tsrstvst_is_nbr;
          v_fa_rec.FA_TSRSTVST_ORG_ID := v_tsrstvst_org_id;
          v_fa_rec.FA_TSRFLDPR_IS_NUMBER := v_proc_is_nbr;
          v_fa_rec.FA_TSRFLDPR_ORG_ID := v_proc_org_id;
          v_fa_rec.FA_TSRFLDGR_IS_NUMBER := v_gear_is_nbr;
          v_fa_rec.FA_TSRFLDGR_ORG_ID := v_gear_org_id;
          v_fa_rec.FA_IMPORT_SEQ := v_import_log_seq;
          v_fa_rec.FA_EXPORT_SEQ := NULL;
          v_fa_rec.FA_STATUS := v_status;
          v_fa_rec.FA_TSMORGAN_IS_NUMBER := v_org_is_nbr;
          v_fa_rec.FA_TSMORGAN_ORG_ID := p_org_id;
          v_fa_rec.FA_APPROVED_BY := v_approved_by;
          v_fa_rec.FA_APPROVED_ON := v_approved_on;
          v_fa_rec.FA_STVST_COMMENTS := v_stvst_cmt;
          v_fa_rec.FA_DEPTH_UOM := v_depth_uom;
          v_fa_rec.FA_MEDIUM := v_medium;
          v_fa_rec.FA_TSRLAB_IS_NUMBER := v_tsrlab_is_nbr;
          v_fa_rec.FA_TSRLAB_ORG_ID := v_tsrlab_org_id;
          v_fa_rec.FA_IMPORT_LINE_NO := v_line_cnt;
          v_fa_rec.FA_TSRFDACT_IS_NUMBER := v_act_is_nbr;
          v_fa_rec.FA_TSRFDACT_ORG_ID := v_act_org_id;
          v_fa_rec.FA_REPLICATE_NBR := v_replicate_nbr;
          v_fa_rec.FA_FDSET_ID := v_field_set_nbr;
          v_fa_rec.FA_FDSET_NAME := v_field_set_name;
          v_fa_rec.FA_UPDEPTH := v_up_depth;
          v_fa_rec.FA_LOWDEPTH := v_low_depth;
          v_fa_rec.FA_THERMO := SUBSTR(v_thermo,1,1);
          v_fa_rec.FA_HALO := SUBSTR(v_halo,1,1);
          v_fa_rec.FA_PYCNO := SUBSTR(v_pycno,1,1);
          v_fa_rec.FA_REL_DEPTH := v_rel_depth;
          v_fa_rec.FA_DEPTH_ZONE := v_depth_zone;
          v_fa_rec.FA_DEPTH_RANGE_UOM := v_depth_range_uom;
          v_fa_rec.FA_TSRGRCFG_IS_NUMBER := v_gearcfg_is_nbr;
          v_fa_rec.FA_TSRGRCFG_ORG_ID := v_gearcfg_org_id;
          v_fa_rec.FA_TSRSDP_IS_NUMBER := v_sample_pres_is_nbr;
          v_fa_rec.FA_TSRSDP_ORG_ID := v_sample_pres_org_id;
          v_fa_rec.FA_STP_SEQ := v_trip_seq;
          v_fa_rec.FA_SSV_SEQ := v_visit_seq;
          v_fa_rec.FA_INTENT := v_intent;
          v_fa_rec.FA_TSMPRMVL_COMMUNITY_IS_NBR := v_community_is_nbr;
          v_fa_rec.FA_TSMPRMVL_COMMUNITY_ORG_ID := v_community_org_id;
          v_fa_rec.FA_TSRBIOPT_IS_NUMBER := v_biopart_is_nbr;
          v_fa_rec.FA_TSRBIOPT_ORG_ID := v_biopart_org_id;

          IF   v_intent IN ('Individual','Tissue') THEN
            v_fa_rec.FA_TSRCHAR_IS_NUMBER := v_taxon_is_nbr;
            v_fa_rec.FA_TSRCHAR_ORG_ID := v_taxon_org_id;
          ELSE v_fa_rec.FA_TSRCHAR_IS_NUMBER := NULL;
            v_fa_rec.FA_TSRCHAR_ORG_ID := NULL;
          END IF;

          -- New Stuff for V2
          v_fa_rec.FA_TSRMATRX_IS_NUMBER := v_tsrmatrx_is_number;
          v_fa_rec.FA_TSRMATRX_ORG_ID := v_tsrmatrx_org_id;
          v_fa_rec.FA_TSROPPRD_ORG_ID := NULL;
          v_fa_rec.FA_TSROPPRD_IS_NUMBER := NULL;
          v_fa_rec.FA_QC_INDICATOR := v_QC_indicator;
          v_fa_rec.FA_CHAIN_OF_CUSTODY_ID := v_chain_of_custody;
          v_fa_rec.FA_SPECIES_NUMBER := v_act_species;
          v_fa_rec.FA_CHAR_GROUP := v_group;
          v_fa_rec.FA_DOCUMENT := v_activity_document;

          /* insert field activity */
          v_success := sim_import_result_extras.create_field_activity(v_fa_rec,v_import_log_seq);
          IF v_success != 1 THEN
            UPDATE SIM_FIELD_ACTIVITIES
              SET fa_status = 'N'
              WHERE fa_seq = v_activity_seq;
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create activity in sim_field_activities, activity will not be exported.',v_activity_seq);
          END IF;
        END IF;
      EXCEPTION
        WHEN OTHERS THEN
          Sim2_Import_Pkg.write_error(
            v_import_log_seq, 
            SQLERRM||' Line '||v_line_cnt||': in create activity.',
            v_activity_seq
          );
      END;
 
 
      BEGIN
      
        IF v_pre_check = 0 THEN
          
          IF v_new_activity = 1 THEN
          
            -- add projects to activity
            BEGIN
              IF v_no_projects IS NOT NULL THEN
                FOR v_number IN 1..v_no_projects LOOP
                  IF v_proj_is_table(v_number) IS NOT NULL THEN
                    v_proj_is_nbr := v_proj_is_table(v_number);
                    v_proj_org_id := v_proj_org_table(v_number);
                    v_success := add_project_to_activity(v_activity_seq, v_proj_is_nbr, v_proj_org_id);
                  END IF;
                END LOOP;
              ELSE
                v_status := 'N';
                UPDATE SIM_FIELD_ACTIVITIES
                  SET FA_STATUS = 'N'
                  WHERE FA_SEQ = v_activity_seq;
              --Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Project is required and has not been provided.',v_activity_seq);
              END IF;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN
                NULL;
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||SQLERRM||' - adding projects to activity.',v_activity_seq);
            END;
            
            -- add personnel to activity
            BEGIN
              IF v_no_person > 0 THEN
                FOR v_number IN 1..v_no_person LOOP
                  IF v_persn_is_table(v_number) IS NOT NULL THEN
                    v_tsmpersn_is_nbr := v_persn_is_table(v_number);
                    v_tsmpersn_org_id := v_persn_org_table(v_number);
                    v_success := add_personnel_to_activity(v_activity_seq, v_tsmpersn_is_nbr, v_tsmpersn_org_id);
                  END IF;
                END LOOP;
              END IF;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN
                NULL;
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||SQLERRM||' - adding personnel to activity.',v_activity_seq);
            END;
            
            -- add cooperating organizations to activity
            BEGIN
              -- are there any cooperating organizations?
              IF v_no_cporgs > 0 THEN
                -- if yes
                -- loop through the cooperating organizations array
                FOR v_number IN 1..v_no_cporgs LOOP
                  -- is there a cooperating organization in this element of the array?
                  IF v_cporg_is_table(v_number) IS NOT NULL THEN
                    -- if yes
                    -- set the STORET organization number and id
                    v_tsmcporg_is_nbr := v_cporg_is_table(v_number);
                    v_tsmcporg_org_id := v_cporg_org_table(v_number);
                    -- insert the cooperating organization record
                    v_success := add_cporg_to_activity(
                      v_activity_seq,
                      v_tsmcporg_is_nbr, 
                      v_tsmcporg_org_id
                    );
                  END IF;
                END LOOP;
              END IF;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN
                NULL;
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(
                  v_import_log_seq,
                  'Line '||v_line_cnt||': '||SQLERRM||' - adding cooperating organization to activity.',
                  v_activity_seq
                );
            END;
            
            BEGIN           

              -- Create the parent records for the activity
              FOR v_number IN 1..nvl(v_no_parent_ids,0) LOOP
              
                -- get the next sequence number
                SELECT SIM_field_activity_parent_seq.Nextval into v_fap_seq FROM dual;
                
                -- put parent information in a record
                v_fap_rec.FAP_SEQ :=              v_fap_seq;
                v_fap_rec.FAP_FA_SEQ :=           v_fa_rec.FA_SEQ;
                v_fap_rec.FAP_PARENT_FA_SEQ :=    v_parent_fa_seq_table(v_number);
                v_fap_rec.FAP_PARENT_IS_NUMBER := v_parent_is_number_table(v_number);
                v_fap_rec.FAP_PARENT_ORG_ID :=    RPAD(v_parent_org_id_table(v_number),8);
                
                -- insert the record into SIM_FIELD_ACTIVITY_PARENT
                v_error_code := sim_import_result_extras.create_field_activity_parent(
                  v_fap_rec,
                  v_import_log_seq);
        
                -- did an error occur while inserting trying to insert the record?
                IF (v_error_code IS NOT NULL) THEN
                  -- if yes
                  
                  -- set the status flag so the activity cannot not migrate
                  UPDATE SIM_FIELD_ACTIVITIES
                    SET fa_status = 'N'
                    WHERE fa_seq = v_activity_seq;
                    
                  -- output error message
                  Sim2_Import_Pkg.write_error(
                    v_import_log_seq, 
                    'Line ' || v_line_cnt || 
                    ': Failed to create parent in SIM_FIELD_ACTIVITY_PARENT -- ',
                    v_activity_seq);
        
                END IF;
        
              END LOOP;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN
                NULL;
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(
                  v_import_log_seq,
                  'Line '||v_line_cnt||': '||SQLERRM||' - adding Parent Sample IDs to activity.',
                  v_activity_seq
                );

            END;

          END IF;

          /*  THIS WILL CREATE SIM_ACTUAL_ACTIVITY_LOC RECORDS */
          IF v_new_activity = 1 AND v_tsmstatn_is_nbr IS NOT NULL  THEN --AND v_sequence_number != -1 THEN
            IF NVL(v_sequence_number,1) != -1 THEN
              OPEN c_get_location_info;
                FETCH c_get_location_info INTO v_stl_seq;
              CLOSE c_get_location_info;
              v_point_type := NULL; --reset it for the next row
              v_sequence_number := NULL;  --reset it for the next row

              -- create a single aal record */
              OPEN c_get_next_aal_seq;
                FETCH c_get_next_aal_seq INTO v_aal_rec.AAL_SEQ;
              CLOSE c_get_next_aal_seq;

              /* assign values to aal general record */
              v_aal_rec.AAL_SEQ := v_aal_rec.AAL_SEQ;
              v_aal_rec.AAL_FA_SEQ := v_activity_seq;
              v_aal_rec.AAL_STL_SEQ := v_stl_seq;
              v_aal_rec.AAL_TSMALP_IS_NUMBER := v_tsmalp_is_nbr;
              v_aal_rec.AAL_TSMALP_ORG_ID := v_tsmalp_org_id;
              v_aal_rec.AAL_TSRAAL_TYPE_NAME := 'General';
              v_aal_rec.AAL_BOTTOM_DEPTH_MSR := NULL;
              v_aal_rec.AAL_BOTTOM_DEPTH_UOM := NULL;
              v_aal_rec.AAL_ADDTNL_LOC_INFO := v_add_loc_info;

             /* insert a general aal record */
              v_success := sim_import_result_extras.create_aal(v_aal_rec,v_import_log_seq);
              IF v_success != 1 THEN  -- invalidate activity if create_aal fails
                UPDATE sim_field_activities
                  SET fa_status = 'N'
                  WHERE fa_seq = v_activity_seq;
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create "General" record in sim_actual_activity_loc, activity will not be exported.',v_activity_seq);
              END IF;
            --END IF;
            END IF;
          END IF;


          /* THIS WILL CREATE A RECORD FOR SIM_TRAWL_OPS_DET  */
          -- jah 5-13-04
          -- Modified the following code to create 
          -- either a SIM_TRAWL_OPS_DET record or 
          -- a SIM TRAP_NET record, but not both.
          
          -- Do we need to create a SIM_TRAWL_OPS_DET record or a SIM TRAP_NET record?
          IF v_activity_cat != 'Created from Sample' AND v_new_activity = 1 THEN
            -- if yes
            
            -- Do we need to create a SIM_TRAWL_OPS_DET record?
            IF (v_fished_duration IS NOT NULL)
            AND (v_check_trawl_gear = 1)
            THEN
              -- if yes

              -- get next sequence number
              OPEN c_get_next_tod_seq;
              FETCH c_get_next_tod_seq INTO v_tod_rec.TOD_SEQ;
              CLOSE c_get_next_tod_seq;

              -- assign values
              v_tod_rec.TOD_FA_SEQ := v_activity_seq;
              v_tod_rec.TOD_FISHED_DURATION := v_fished_duration;
              v_tod_rec.TOD_FISHED_DUR_UOM := v_fished_dur_uom;
              v_tod_rec.TOD_BOAT_SPEED_MSR := v_boat_speed;
              v_tod_rec.TOD_BOAT_SPEED_UOM := v_boat_speed_uom;
              v_tod_rec.TOD_FISHED_DISTANCE := v_distance_fished;
              v_tod_rec.TOD_FISHED_DISTANCE_UOM := v_dist_fished_uom;
              v_tod_rec.TOD_REL_CURRENT_DIR := v_relative_current;
              v_tod_rec.TOD_REL_WIND_DIR := v_relative_wind;

              -- Attempt to create a SIM_TRAWL_OPS_DET record
              v_success := sim_import_result_extras.create_tod(
                v_tod_rec,
                v_import_log_seq
              );

              -- Was attempt to create a SIM_TRAWL_OPS_DET record successful?
              IF v_success != 1 THEN
                -- if no
                UPDATE sim_field_activities
                SET fa_status = 'N'
                WHERE fa_seq = v_activity_seq;
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Line '||v_line_cnt||
                  ': Failed to create record in sim_trawl_ops_det, ' ||
                  'activity will not be exported.',
                  v_activity_seq
                );
              END IF;  
              
            END IF;
            
            -- Do we need to create a SIM TRAP_NET record
            IF (v_trap_net_duration IS NOT NULL)
            AND (v_check_trawl_gear = 2)
            THEN
              -- if yes

              -- get next sequence number
              OPEN c_get_trap_net_seq;
              FETCH c_get_trap_net_seq INTO v_stn_seq;
              CLOSE c_get_trap_net_seq;

              -- assign values
              v_stn_rec.STN_SEQ := v_stn_seq;
              v_stn_rec.STN_TSRFDACT_IS_NUMBER := NULL;
              v_stn_rec.STN_TSRFDACT_ORG_ID := NULL;
              v_stn_rec.STN_SAMPLING_DUR := v_trap_net_duration;
              v_stn_rec.STN_SAMPLING_DUR_UNIT := v_trap_net_dur_unit;
              v_stn_rec.STN_ORIENT_TO_CUR := v_orient_to_cur;
              v_stn_rec.STN_REL_CURRENT := v_relative_current;
              v_stn_rec.STN_REL_WIND := v_relative_wind;
              v_stn_rec.STN_COMMENT := v_trap_net_comment;
              v_stn_rec.STN_FA_SEQ := v_activity_seq;

              -- Attempt to create a SIM_TRAP_NET record
              v_success := sim_import_result_extras.create_trap_net(
                v_stn_rec, 
                v_import_log_seq
              );

              -- Was attempt to create a SIM_TRAP_NET record successful?
              IF v_success != 1 THEN
                -- if no
                UPDATE sim_field_activities
                SET fa_status = 'N'
                WHERE fa_seq = v_activity_seq;
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq, 
                  'Line ' || v_line_cnt ||
                  ': Failed to create record in sim_trap_net, ' ||
                  'activity will not be exported.',
                  v_activity_seq
                );
              END IF;

            END IF;

          END IF;


          /* THIS WILL CREATE A RECORD FOR SIM_SAMPLE*/
          IF  v_type = 'Sample' AND v_new_activity = 1 THEN
            IF v_sample_pres_is_nbr IS NOT NULL AND v_sample_pres_org_id IS NOT NULL THEN
              OPEN  c_get_tsrsdp_values(v_sample_pres_is_nbr, v_sample_pres_org_id);
                FETCH c_get_tsrsdp_values INTO v_tsrsdp_rec;
              CLOSE c_get_tsrsdp_values;
            END IF;
            OPEN c_get_next_sample_seq;
              FETCH c_get_next_sample_seq INTO v_sample_seq;
            CLOSE c_get_next_sample_seq;

            /* assign values to record and send to create_sample function */
            v_sample_rec.SS_SEQ := v_sample_seq;
            v_sample_rec.SS_FA_SEQ := v_activity_seq;
            v_sample_rec.SS_TOTAL_VOLUME_MSR := v_ttl_smpl_vol;
            v_sample_rec.SS_TOTAL_VOL_UOM := v_smpl_vol_unit_seq;
            v_sample_rec.SS_TOTAL_WEIGHT_MSR := v_smpl_wt;
            v_sample_rec.SS_TOTAL_WT_UOM := v_total_wt_uom;
            v_sample_rec.SS_DATA_FILE_NAME_LOC := v_data_file_name_loc;
            v_sample_rec.SS_CONTAINER_TYPE_NM := v_tsrsdp_rec.CONTAINER_TYPE_NM;
            v_sample_rec.SS_CONTAINER_COLOR := v_tsrsdp_rec.CONTAINER_COLOR;
            v_sample_rec.SS_CONTAINER_SIZE_MSR := v_tsrsdp_rec.CONTAINER_SIZE_MSR;
            v_sample_rec.SS_CONTAINER_SIZE_UN := v_tsrsdp_rec.CONTAINER_SIZE_UN;
            v_sample_rec.SS_TEMP_PRESERVN_TYPE := v_tsrsdp_rec.TEMP_PRESRV_TYPE;
            v_sample_rec.SS_PARENT_IND_CODE := v_parent_ind_code;
            v_sample_rec.SS_GEAR_DEPLOYMENT := v_gear_deployment;
            v_sample_rec.SS_TRANSPORT_STORAGE := v_transport_storage;
            v_sample_rec.SS_PRESRV_STRGE_PRCDR := v_tsrsdp_rec.PRESRV_STRGE_PRCDR;
            -- We do not support sampling duration in SIM v2.0.1.  Default values 
            --  to zero for time and null for unit to be consistent with the 
            --  STORET interface.
            v_sample_rec.SS_DURATION_TIME := 0;
            v_sample_rec.SS_DURATION_UNIT := NULL;

    /* insert sample record */
            v_success := sim_import_result_extras.create_sample(v_sample_rec,v_import_log_seq);

            IF v_success != 1 THEN  -- invalidate activity if create_sample fails
             UPDATE sim_field_activities
                SET fa_status = 'N'
                WHERE fa_seq = v_activity_seq;
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create record in sim_sample, activity will not be exported.',v_activity_seq);
            END IF;
          END IF;
        END IF;  --stuff we don't do for pre-check

        /* THIS WILL CREATE A RECORD FOR SIM_BIO_RESULTS_GRP */
        IF  v_brg_id IS NOT NULL AND v_new_group = 1 THEN
          OPEN c_get_next_brg_seq;
            FETCH c_get_next_brg_seq INTO v_brg_seq;
          CLOSE c_get_next_brg_seq;
          /* assign values to record and send to create_brg function */
          v_brg_rec.BRG_SEQ := v_brg_seq;
          v_brg_rec.BRG_FA_SEQ := v_activity_seq;
          v_brg_rec.BRG_ID_CODE := v_brg_id;
          v_brg_rec.BRG_TYPE := v_brg_type;
          v_brg_rec.BRG_FREQ_ANALYSIS := v_brg_freq_analysis;
          v_brg_rec.BRG_TSRCPV_SEX_IS_NUMBER := v_sexname_is_nbr;
          v_brg_rec.BRG_TSRCPV_SEX_ORG_ID := v_sexname_org_id;
          v_brg_rec.BRG_TSRCPV_LIFESTAGE_IS_NBR := v_lifestage_is_nbr;
          v_brg_rec.BRG_TSRCPV_LIFESTAGE_ORG_ID := v_lifestage_org_id;
          v_brg_rec.BRG_VALUE_TYPE_NAME := v_value_type_name;
          v_brg_rec.BRG_SPECIES_NUMBER := v_BRG_Species;
          v_brg_rec.BRG_DESCRIPTION_TEXT := v_BRG_text;
          IF v_brg_type IN ('Multi-Taxon Population Census','Single Taxon Frequency Classes') THEN
            v_brg_rec.BRG_SUMMARY_GRP_COUNT := NULL;
          ELSIF v_brg_type IN ('Single Taxon Individuals') THEN
            v_brg_rec.BRG_SUMMARY_GRP_COUNT := v_ttl_nbr_indiv;
          ELSE
            v_brg_rec.BRG_SUMMARY_GRP_COUNT := v_ttl_nbr_in_grp;
          END IF;
          IF  v_brg_type LIKE 'Single%' THEN
            v_brg_rec.BRG_TSRCHAR_IS_NUMBER := v_taxon_is_nbr;
            v_brg_rec.BRG_TSRCHAR_ORG_ID := v_taxon_org_id;
          ELSE v_brg_rec.BRG_TSRCHAR_IS_NUMBER := NULL;
            v_brg_rec.BRG_TSRCHAR_ORG_ID := NULL;
          END IF;
          v_brg_rec.BRG_IMPORT_SEQ := v_import_log_seq;

    /* insert brg record */
          v_success := sim_import_result_extras.create_brg(v_brg_rec,v_import_log_seq);
          IF v_success != 1 THEN  -- invalidate activity if create_brg fails
            UPDATE sim_field_activities
              SET fa_status = 'N'
              WHERE fa_seq = v_activity_seq;
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create record in sim_bio_results_grp, activity will not be exported.',v_activity_seq);
          END IF;
        END IF;

        /* THIS WILL CREATE A RECORD FOR SIM_BIO_RESULTS_GRP_INDIVIDUAL */

        IF  v_brg_id IS NOT NULL AND v_individual_nbr IS NOT NULL THEN
         /* get existing brg_seq (SF) don't check for duplicate individual numbers here, already did it in check_duplicate_results */
          OPEN c_find_brg_seq(v_brg_id, v_activity_seq);
            FETCH c_find_brg_seq INTO v_brgi_brg_seq;
          CLOSE c_find_brg_seq;

          v_success := Sim_import_result_checks.check_dup_brgi(v_activity_seq,v_brgi_brg_seq,v_individual_nbr,v_char_is_nbr,v_line_cnt,v_import_log_seq);

          IF v_success = 1 THEN  -- individual does not exist, get next brgi_seq and create record
            OPEN c_get_next_brgi_seq;
              FETCH c_get_next_brgi_seq INTO v_brgi_seq;
            CLOSE c_get_next_brgi_seq;
        /* assign values to record and send to create_brgi function */
            v_brgi_rec.BRGI_SEQ := v_brgi_seq;
            v_brgi_rec.BRGI_BRG_SEQ := v_brgi_brg_seq;
            v_brgi_rec.BRGI_INDIVIDUAL_NUMBER := v_individual_nbr;
            v_brgi_rec.BRGI_FA_SEQ := v_activity_seq;
            v_brgi_rec.BRGI_IMPORT_SEQ := v_import_log_seq;
          /* insert brgi record */

            v_success := sim_import_result_extras.create_brgi(v_brgi_rec,v_import_log_seq);

            IF v_success != 1 THEN  -- invalidate activity if create_brgi fails
              UPDATE sim_field_activities
                SET fa_status = 'N'
                WHERE fa_seq = v_activity_seq;
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create record in sim_bio_results_grp_individual, activity will not be exported.',v_activity_seq);
            END IF;
          END IF;
        END IF;

        /* create result */
        /* write data to record and send to create result function */
         /* get result sequence */
         
        IF v_result IS NOT NULL THEN
          OPEN c_get_result_seq;
            FETCH c_get_result_seq INTO v_result_seq;
          CLOSE c_get_result_seq;
          v_result_rec.RS_SEQ := v_result_seq;
          v_result_rec.RS_AC_SEQ := NULL;
          v_result_rec.RS_FA_SEQ := v_activity_seq;
          v_result_rec.RS_BRG_SEQ := v_brg_seq;
          v_result_rec.RS_BRGI_SEQ := v_brgi_seq;
          v_result_rec.RS_RESULT := v_result;
          v_result_rec.RS_D_ASSND_SEQ_NUM := v_result_seq_num;
          v_result_rec.RS_COMMENT := v_result_cmt;
          v_result_rec.RS_DETECT_LIMIT := v_detect_limit;
          v_result_rec.RS_IMPORT_LINE_NO := v_line_cnt;
          v_result_rec.RS_VALUE_STATUS := v_result_status;
          v_result_rec.RS_LAB_BATCH_ID := v_batch_id;
          v_result_rec.RS_LAB_CERT := v_lab_cert;
          v_result_rec.RS_ANALYSIS_DATE := v_analysis_date;
          v_result_rec.RS_ANALYSIS_TIME := v_analysis_time;
          v_result_rec.RS_ANALYSIS_TIME_ZONE := v_analysis_time_zone;
          v_result_rec.RS_DET_LIMIT_UOM_NBR := v_det_uom_is_nbr;
          v_result_rec.RS_DET_LIMIT_UOM_ORG := v_det_uom_org_id;
          v_result_rec.RS_DETECT_COMMENT := v_det_comment;
          v_result_rec.RS_UPPER_QUANT_LIMIT := v_upq_limit;
          v_result_rec.RS_LOWER_QUANT_LIMIT := v_lowq_limit;
          
          IF v_brg_freq_analysis = 'B' THEN
            v_result_rec.RS_TSRCHAR_IS_NUMBER := v_bio_condition_is_nbr;
            v_result_rec.RS_TSRCHAR_ORG_ID := v_bio_condition_org_id;
            v_result_rec.RS_TSRCHAR0IS_NUMBER := v_sec_bio_cond_is_nbr;
            v_result_rec.RS_TSRCHAR0ORG_ID := v_sec_bio_cond_org_id;
          ELSE
            v_result_rec.RS_TSRCHAR_IS_NUMBER := v_char_is_nbr;
            v_result_rec.RS_TSRCHAR_ORG_ID := v_char_org_id;
            v_result_rec.RS_TSRCHAR0IS_NUMBER := NULL;
            v_result_rec.RS_TSRCHAR0ORG_ID := NULL;
          END IF;
          
          IF v_brg_type = 'Single Taxon Frequency Classes' THEN
            IF v_brg_freq_analysis = 'B' THEN
              /* For STFC-Bio, the UOM of Count is stored in the TSRRCI table and nothing is stored in the TSRRSULT table. */
              v_result_rec.RS_TSRUOM_IS_NBR := NULL;
              v_result_rec.RS_TSRUOM_ORG_ID := NULL;
            ELSE 
              /* For STFC-Physical, the actual UOM is stored in the TSRRCI table and Count is stored in the TSRRSULT table. */
              v_result_rec.RS_TSRUOM_IS_NBR := 72;
              v_result_rec.RS_TSRUOM_ORG_ID := 'ZZZZZZZZ';            
            END IF;
          ELSE  
            /* For everything else, the actual UOM is stored in the TSRRSULT table. */
            v_result_rec.RS_TSRUOM_IS_NBR := v_uom_is_nbr;
            v_result_rec.RS_TSRUOM_ORG_ID := v_uom_org_id;
          END IF;
          
          v_result_rec.RS_FRACTION := v_fraction;
          v_result_rec.RS_VALUE_TYPE := v_value_type;
          v_result_rec.RS_STAT_TYPE := v_stat_type;
          v_result_rec.RS_TSRANLPR_IS_NUMBER := v_anlpr_is_nbr;
          v_result_rec.RS_TSRANLPR_ORG_ID := v_anlpr_org_id;
          v_result_rec.RS_TSRLSPP_IS_NUMBER := v_labprep_is_nbr;
          v_result_rec.RS_TSRLSPP_ORG_ID := v_labprep_org_id;
          v_result_rec.RS_DURATION_BASIS := v_duration_basis;
          v_result_rec.RS_TEMP_BASIS := v_temp_basis;
          v_result_rec.RS_WEIGHT_BASIS := v_weight_basis;
          v_result_rec.RS_PRECISION := v_precision;
          v_result_rec.RS_CONF_LEVEL := v_conf_level;
          v_result_rec.RS_TSRCHGRP_IS_NUMBER := v_tsrchgrp_is_nbr;
          v_result_rec.RS_TSRCHGRP_ORG_ID := v_tsrchgrp_org_id;
          v_result_rec.RS_TSRCHDEF_IS_NUMBER := v_tsrchdef_is_nbr;
          v_result_rec.RS_TSRCHDEF_ORG_ID := v_tsrchdef_org_id;
          v_result_rec.RS_TROPHIC_LEVEL := v_trophic_level;
          v_result_rec.RS_FNCTIONAL_FEED_GRP := v_func_feed_grp;
          v_result_rec.RS_TAXON_POLLUTION := v_taxon_pollution;
          v_result_rec.RS_TSRLAB_IS_NUMBER := v_tsrlab_is_nbr;
          v_result_rec.RS_TSRLAB_ORG_ID := v_tsrlab_org_id;
          v_result_rec.RS_TSRHCSC_IS_NUMBER := v_tsrhcsc_is_nbr;
          v_result_rec.RS_TSRHCSC_ORG_ID := v_tsrhcsc_org_id;

        --new v2 stuff:
          v_result_rec.RS_BIAS := v_bias;
          v_result_rec.RS_CELL_SHAPE_TYPE_NM := v_cell_shape;
          v_result_rec.RS_CELL_TYPE_NM := v_cell_type;
          v_result_rec.RS_CONF_LVL_CORR_BIAS := v_cl_corrected;
          v_result_rec.RS_CORRECTION_IND_CD := 'N';
          v_result_rec.RS_DILUTION_IND_CODE := 'N';
          v_result_rec.RS_FRACTION_TSMPRMVL_IS_NUMBER := v_fraction_is_nbr;
          v_result_rec.RS_FRACTION_TSMPRMVL_ORG_ID := v_fraction_org_id;
          v_result_rec.RS_HABIT_TSMPRMVL0IS_NUMBER := v_habit_is_number;
          v_result_rec.RS_HABIT_TSMPRMVL0ORG_ID := v_habit_org_id;
          v_result_rec.RS_VOLT_TSMPRMVL1IS_NUMBER := v_volt_is_number;
          v_result_rec.RS_VOLT_TSMPRMVL1ORG_ID := v_volt_org_id;
          
          v_result_rec.RS_LOWER_RANGE_VALUE := v_lower_range;
          v_result_rec.RS_UPPER_RANGE_VALUE := v_upper_range;
          v_result_rec.RS_RECOVERY_IND_CODE := 'N';
          v_result_rec.RS_REF_PT_FROM_NAME := NULL;
          v_result_rec.RS_REF_PT_TO_NAME := NULL;
          v_result_rec.RS_REPL_ANALYSIS_NUM := 0;
          v_result_rec.RS_SPECIES_NUMBER := v_result_species;
          v_result_rec.RS_TSRBRG_IS_NUMBER := NULL;
          v_result_rec.RS_TSRBRG_ORG_ID := NULL;
          v_result_rec.RS_TSRBRGI_IS_NUMBER := NULL;
          v_result_rec.RS_TSRBRGI_ORG_ID := NULL;
          v_result_rec.RS_TSRDLIN_IS_NUMBER := NULL;
          v_result_rec.RS_TSRDLIN_ORG_ID := NULL;
          v_result_rec.RS_IMPORT_SEQ := v_import_log_seq;
          v_result_rec.RS_DATA_LINE_NUMBER := v_data_line_number;
          v_result_rec.RS_DATA_LINE_NAME := v_data_line_name;          
          v_result_rec.RS_DOCUMENT := v_result_document;

          /* insert result */
          v_success := sim_import_result_extras.create_result(v_result_rec,p_import_log_seq);
 
            /* if insert function failed, write error and prevent activity from being exported */
          IF v_success != 1 THEN
            UPDATE SIM_FIELD_ACTIVITIES
              SET fa_status = 'N'
            WHERE fa_seq = v_activity_seq;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create result in sim_results, activity will not be exported.',v_activity_seq);
          ELSE
            BEGIN
              IF v_no_remark_codes > 0 THEN
                FOR v_number IN 1..v_no_remark_codes LOOP
                  IF v_remark_is_table(v_number) IS NOT NULL THEN
                    v_tsrlbrmk_is_nbr := v_remark_is_table(v_number);
                    v_tsrlbrmk_org_id := v_remark_org_table(v_number);
                    v_success := add_remark_to_results(v_result_rec.rs_seq, v_tsrlbrmk_is_nbr, v_tsrlbrmk_org_id);
                  END IF;
                END LOOP;
              END IF;
            EXCEPTION
              WHEN NO_DATA_FOUND THEN
                NULL;
              WHEN OTHERS THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||SQLERRM||' - adding lab remark codes to activity.',v_activity_seq);
            END;          
          END IF;
        END IF;

        /* THIS WILL CREATE A RECORD FOR SIM_RESULT_CLASS_INDICATOR */

        -- jah 2-20-04
        -- do we need to create a SIM_RESULT_CLASS_INDICATOR record?
        IF((v_lower_bnd_amt IS NOT NULL OR v_primary_class IS NOT NULL) AND (NVL(v_dup_brgi_success,0) != 0)) 
          OR (v_particle_size_basis IS NOT NULL) THEN
          -- if yes
        
          OPEN c_get_next_rci_seq;
          FETCH c_get_next_rci_seq INTO v_rci_rec.RCI_SEQ;
          CLOSE c_get_next_rci_seq;

          /* assign values to record and send to create_rci function */
          v_rci_rec.RCI_RS_SEQ := v_result_rec.RS_SEQ;  --gets RCI_RS_SEQ from v_result_rec
          v_rci_rec.RCI_FA_SEQ := v_activity_seq;
          v_rci_rec.RCI_IMPORT_SEQ := v_import_log_seq;
          
          -- jah 2-20-04
          -- Set RCI_PARTICLE_SIZE_BASIS to value (instead of NULL)
          IF (v_particle_size_basis IS NOT NULL) THEN
            v_rci_rec.RCI_PARTICLE_SIZE_BASIS := v_particle_size_basis;
            v_rci_rec.RCI_TSRUOM_IS_NUMBER := 1;
            v_rci_rec.RCI_TSRUOM_ORG_ID := 'ZZZZZZZZ';
            v_rci_rec.RCI_LOWER_BND_AMT := NULL;
            v_rci_rec.RCI_UPPER_BND_AMT := NULL;
            v_rci_rec.RCI_PRIMARY_CLASS_IND := NULL;
            v_rci_rec.RCI_SECOND_CLASS_IND := NULL;
            v_rci_rec.RCI_BRG_SEQ := NULL;
          ELSE
            v_rci_rec.RCI_PARTICLE_SIZE_BASIS := NULL;
            v_rci_rec.RCI_TSRUOM_IS_NUMBER := v_uom_is_nbr;
            v_rci_rec.RCI_TSRUOM_ORG_ID := v_uom_org_id;
            v_rci_rec.RCI_LOWER_BND_AMT := v_lower_bnd_amt;
            v_rci_rec.RCI_UPPER_BND_AMT := v_upper_bnd_amt;
            v_rci_rec.RCI_PRIMARY_CLASS_IND := v_primary_class_value;
            v_rci_rec.RCI_SECOND_CLASS_IND := v_secondary_class_value;
            v_rci_rec.RCI_BRG_SEQ := v_result_rec.RS_BRG_SEQ;
          END IF;

          /* insert rci record */
          v_success := sim_import_result_extras.create_rci(v_rci_rec,v_import_log_seq);

          IF v_success != 1 THEN  -- invalidate activity if create_rci fails
            UPDATE sim_field_activities
              SET fa_status = 'N'
              WHERE fa_seq = v_activity_seq;
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create record in sim_results_class_indicator, activity will not be exported.',v_activity_seq);
          END IF;
        END IF;

         --get ready for the next line
        IF NVL(v_brg_type,'x') != 'Single Taxon Frequency Classes' THEN
          v_uom_is_nbr := NULL;
          v_uom_org_id := NULL;
          v_char_is_nbr := NULL;
          v_char_org_id := NULL;
        END IF;
        
        /* Update status of the field activity */
        IF v_status = 'N' THEN
          UPDATE SIM_FIELD_ACTIVITIES
            SET fa_status = 'N'
            WHERE fa_seq = v_activity_seq AND fa_import_seq = p_import_log_seq;
          v_error_cnt_new := v_error_cnt_new + 1;
          v_pre_check_status := 'N';
        END IF;

        /* update import lines table with activity and result sequences */
        UPDATE SIM_IMP_LINES SET SIPL_FA_SEQ = v_activity_seq
          WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;
        UPDATE SIM_IMP_LINES SET SIPL_RS_SEQ = v_result_rec.RS_SEQ
          WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;

     
      EXCEPTION
      
        WHEN OTHERS THEN
          Sim2_Import_Pkg.write_error(
            v_import_log_seq, 
            'Line '||v_line_cnt||': '||
            SQLERRM||' - creating result. This line has been skipped.',
            v_activity_seq
          );
          
      END;


    EXCEPTION
      WHEN OTHERS THEN
        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'Line '||v_line_cnt||': '||
          SQLERRM||' - creating result. Line has been skipped.',
          v_activity_seq
        );
    END;

 
  END LOOP; /* end of text loop */
  IF v_pre_check = 1 THEN
  
    --delete field activities, visits, and trips made on this pass
    DELETE SIM_RESULT_CLASS_INDICATOR WHERE RCI_IMPORT_SEQ = v_import_log_seq;
    DELETE SIM_RESULTS WHERE RS_IMPORT_SEQ = v_import_log_seq;
    COMMIT;

    DELETE SIM_BIO_RESULTS_GRP_INDIVIDUAL WHERE BRGI_IMPORT_SEQ = v_import_log_seq;
    DELETE SIM_BIO_RESULTS_GRP WHERE BRG_IMPORT_SEQ = v_import_log_seq;
    COMMIT;

    DELETE SIM_FIELD_ACTIVITIES WHERE FA_IMPORT_SEQ = v_import_log_seq;
    COMMIT;
    DELETE SIM_STATION_VISITS WHERE SSV_IMPORT_SEQ = v_import_log_seq;
    DELETE SIM_TRIPS WHERE STP_IMPORT_SEQ = v_import_log_seq;
    COMMIT;

    IF v_pre_check_status != 'N' THEN --we're good to load the data!
      v_pre_check := 0;
      --delete the log records
      FOR v_smd_seq IN c_find_pre_check LOOP
        DELETE SIM_IMPORT_DETAILS WHERE SMD_SEQ = v_smd_seq.smd_seq;
      END LOOP;
      GOTO Start_Checking;
    ELSE
      --delete import lines table and we're done for this round
      DELETE SIM_IMP_LINES WHERE SIPL_SIL_SEQ = v_import_log_seq;
    END IF;
  END IF;

  BEGIN
    OPEN c_comp_count(v_import_log_seq);
      FETCH c_comp_count INTO v_complete_cnt;
    CLOSE c_comp_count;
    v_result_cnt := v_complete_cnt;
    v_complete_cnt := v_complete_cnt + v_skipped_results_cnt;
    v_error_cnt := v_line_cnt - v_complete_cnt;

    OPEN c_act_cnt(v_import_log_seq);
      FETCH c_act_cnt INTO v_act_cnt;
    CLOSE c_act_cnt;

     /* update counts in the import log */
    UPDATE SIM_IMPORT_LOG
      SET SIL_ROWS_IMPORTED = v_line_cnt,
        SIL_ROWS_ERRORS = v_error_cnt,
        SIL_ROWS_READY = v_result_cnt,
        SIL_LAB_ID = v_first_lab_id,
        SIL_ACTIVITIES_IMPORTED = v_act_cnt
      WHERE SIL_SEQ = v_import_log_seq;
      
      /* slp 05/21/03 - Modified SIL_ROWS_READY from complete count to result count because it appeared
        to the user that the skipped row was also ready for migration which it is not. */
    IF v_pre_check = 1 THEN --If pre-check is still 1 then we're not loading anything.
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,v_line_cnt||' lines processed, 0 activities, 0 results loaded.',NULL);
    ELSE
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,v_line_cnt||' lines processed '||v_act_cnt||' activities, '||v_result_cnt||' results loaded. ',NULL);
    END IF;

    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq ,'Finished at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL);

    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq ,'Error in post line processing: '||SQLERRM,NULL);
  END;


EXCEPTION

  WHEN e_no_cfg THEN
    RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');
  WHEN e_bad_delimiter THEN
    RAISE_APPLICATION_ERROR(-20197,'The import file contains the wrong number of columns on line: '||v_line_cnt);
  WHEN NO_DATA_FOUND THEN
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR('Line '||v_line_cnt||', Last Item Processed: '||v_item_name||', NO_DATA_FOUND : '||SQLERRM||' - IN Sim2_Import_Result_Pkg.import_results.',1,240),NULL);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_act_cnt)||' activities loaded',NULL);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL);

  WHEN e_no_sid_seq THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Import Type not found. Something is seriously wrong.', NULL);

  WHEN e_bad_station THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Station Problem. Something is seriously wrong.', NULL);

  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR('Line '||v_line_cnt||', Last Item Processed: '||v_item_name||', Fatal Error : '||SQLERRM||' - IN Sim2_Import_Result_Pkg.import_results.',1,240),v_activity_seq);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_act_cnt)||' activities loaded',null);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),null);

END;

/******************************************************************************************************/
END Sim2_Import_Result_Pkg;
/
