CREATE OR REPLACE PACKAGE SIM_IMPORT_LOCATIONS_PKG IS
  -- Author  : Susan F. Pete H.

  -- Created : 05/03/02
  -- Purpose : import well data into SIM tables from a text file
  TYPE COL_TABLE_TYPE IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;
  TYPE COL_TABLE_NUM_TYPE IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;
  TYPE T_ACTIVITY_COUNT IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;
  TYPE T_STORET_PK_TYPE IS RECORD(
    IS_NUMBER NUMBER(8),
    ORG_ID    VARCHAR2(8));
  TYPE T_STORET_TSMVSTC_TYPE IS RECORD(
    TSMVSTC_IS_NUMBER NUMBER(8),
    TSMVSTC_ORG_ID    VARCHAR2(8));
  TYPE T_STORET_TSMMAD_TYPE IS RECORD(
    TSMMAD_IS_NUMBER NUMBER(8),
    TSMMAD_ORG_ID    VARCHAR2(8));
  TYPE T_AQUIFER_TYPE IS RECORD(
    TSMGEOUN_CD  VARCHAR2(12),
    AQUIFER_NAME VARCHAR2(80));

  V_ACTIVITY_COUNT T_ACTIVITY_COUNT;
  V_ORG_IS_NBR     NUMBER(8) := 0;
  V_IMPORT_LOG_SEQ NUMBER(10);
  V_STATUS         VARCHAR2(1);
  V_LINE_CNT       NUMBER(10) := 0;

  -- Public function and procedure declarations

  PROCEDURE IMPORT_LOCATIONS(P_ORG_ID         IN VARCHAR2,
                             P_IMPORT_LOG_SEQ IN NUMBER,
                             P_IMP_CFG_SEQ    IN NUMBER,
                             P_TEST_COUNT     IN NUMBER);

  FUNCTION CREATE_STATION_LOCATION(P_STATION_LOCATION_REC IN SIM_STATION_LOCATIONS%ROWTYPE)
    RETURN NUMBER;

END SIM_IMPORT_LOCATIONS_PKG;
/
CREATE OR REPLACE PACKAGE BODY SIM_IMPORT_LOCATIONS_PKG IS

  FUNCTION CHECK_TRANSLATIONS(P_CODE        IN VARCHAR2,
                              P_IMP_CFG_SEQ IN NUMBER,
                              P_COL_SEQ     IN NUMBER,
                              P_TRANS_EXIST OUT BOOLEAN) RETURN VARCHAR2;

  PROCEDURE CREATE_TRANS_RECORD(P_CODE        IN VARCHAR2,
                                P_IMP_CFG_SEQ IN NUMBER,
                                P_COL_SEQ     IN NUMBER);

  FUNCTION CHECK_STORET_TRANSLATION(P_CODE IN VARCHAR2,
                                  P_IMP_CFG_SEQ IN NUMBER,
                                  P_COL_SEQ IN NUMBER,
                                  P_TRANS_EXIST OUT BOOLEAN,
                                  V_CONVERTED_ITEM OUT VARCHAR2,
                                  P_STA_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE;
                                  
/*********************************************************************************************************/
-- Function and procedure implementations

/******************************************************************************************************/
FUNCTION VALIDATE_NUMBER(P_ITEM      IN VARCHAR2,
                       P_ITEM_NAME IN VARCHAR2,
                       P_FORMAT    IN VARCHAR2,
                       P_SEQ       IN NUMBER) RETURN NUMBER IS
  /* 4/12/02 - SF
     This function is to test any value that is supposed to be a number. 
     If it's not a number a 0 is returned. */

  V_RETURN NUMBER(10);

BEGIN
  V_RETURN := TO_NUMBER(P_ITEM, P_FORMAT);
  -- If no exception raised, P_Item was a number
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                'Line: ' || V_LINE_CNT || ': ' ||
                                 P_ITEM_NAME || ', ' || P_ITEM ||
                                 ', is not in an appropriate numeric format of ' ||
                                 P_FORMAT || '.',
                                P_SEQ);
    RETURN 0;
END VALIDATE_NUMBER;

/******************************************************************************************************

/* The get_latitude and get_longitude functions are used by stations, wells, and locations imports to 
interpret the values sent in to the Latitude and Longitude fields respectively. Several assumptions are 
made and they are noted in the code. */

FUNCTION GET_LATITUDE(v_item_in IN VARCHAR2, 
                      v_column_seq IN NUMBER, 
                      p_imp_cfg_seq IN NUMBER,
                      v_sta_seq IN NUMBER, 
                      v_line_cnt IN NUMBER, 
                      p_lat_degrees OUT NUMBER,
                      p_lat_minutes OUT NUMBER, 
                      p_lat_seconds OUT NUMBER, 
                      p_lat_dir OUT VARCHAR2,
                      p_lat_dec_min OUT NUMBER, 
                      p_lat_dec_deg OUT NUMBER) RETURN NUMBER IS

  v_lat_format     VARCHAR2(15);
  v_latitude       VARCHAR2(20);
  WhatsBad         VARCHAR2(15);
  v_item           VARCHAR2(60);
  v_temp_min       NUMBER(10,7);

  e_bad_format     EXCEPTION;
  e_bad_number     EXCEPTION;
  
  CURSOR c_get_format IS
    SELECT SIFC_FMT_CHOICE
      FROM SIM_IMP_FMT_CHOICES C, SIM_IMP_CFG_DETS D
      WHERE D.SICDT_SIC_SEQ = v_column_seq
      AND D.SICDT_SICC_SEQ = p_imp_cfg_seq
      AND D.SICDT_SIFC_SEQ = C.SIFC_SEQ;
  
BEGIN
  v_item := v_item_in;
  OPEN c_get_format;
    FETCH c_get_format INTO v_lat_format;      
    IF c_get_format%NOTFOUND THEN --require format
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Line '||v_line_cnt||' you must choose a format for Latitude.',v_sta_seq);
    END IF;      
  CLOSE c_get_format;
     
  IF v_lat_format = 'DD-MM-SS.SSSS' THEN
    IF SUBSTR(v_item,3,1) != '-' OR SUBSTR(v_item,6,1) != '-'  THEN
      IF SUBSTR(v_item,2,1) = '-' AND SUBSTR(v_item,5,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item,1,2));
    p_lat_minutes := NVL(TO_NUMBER(SUBSTR(v_item,4,2)),0);
    p_lat_seconds := NVL(TO_NUMBER(SUBSTR(v_item,7,7)),0);
    v_temp_min := p_lat_minutes + (p_lat_seconds/60); --use this to keep extra precision
    p_lat_dec_min := TRUNC(v_temp_min,4);
    p_lat_dec_deg := TRUNC(p_lat_degrees + (v_temp_min/60),7);
    p_lat_dir := NULL;
    
  ELSIF v_lat_format = 'DD MM SS.SSSS' THEN
    IF SUBSTR(v_item,3,1) != ' ' OR SUBSTR(v_item,6,1) != ' '  THEN
      IF SUBSTR(v_item,2,1) = ' ' AND SUBSTR(v_item,5,1) = ' ' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item,1,2));
    p_lat_minutes := NVL(TO_NUMBER(SUBSTR(v_item,4,2)),0);
    p_lat_seconds := NVL(TO_NUMBER(SUBSTR(v_item,7,7)),0);
    v_temp_min := p_lat_minutes + (p_lat_seconds/60); --use this to keep extra precision
    p_lat_dec_min := TRUNC(v_temp_min,4);
    p_lat_dec_deg := TRUNC(p_lat_degrees + (v_temp_min/60),7);
    p_lat_dir := NULL;      
    
  ELSIF v_lat_format = 'DD-MM.MMMM' THEN
    IF SUBSTR(v_item,3,1) != '-' THEN
      IF SUBSTR(v_item,2,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item,1,2));
    p_lat_minutes := NVL(TO_NUMBER(SUBSTR(v_item,4,2)),0);
    p_lat_dec_min := NVL(TO_NUMBER(SUBSTR(v_item,4,7)),0);
    p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
    p_lat_dec_deg := TRUNC(p_lat_degrees + (p_lat_dec_min/60),7);
    p_lat_dir := NULL;
  
  ELSIF v_lat_format = 'DD.DDDDDDD' THEN
    IF SUBSTR(v_item,3,1) != '.' THEN
      IF SUBSTR(v_item,2,1) = '.' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_dec_deg := TRUNC(TO_NUMBER(v_item),7);
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item, 1,2));
    p_lat_dec_min := 60 * (p_lat_dec_deg - p_lat_degrees);
    p_lat_minutes := FLOOR(p_lat_dec_min);
    p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
    p_lat_dir := NULL;
  
  ELSIF V_LAT_FORMAT = '-DD.DDDDDD' THEN
    
    -- start by making sure there is a sign on the front
    IF SUBSTR(v_item,1,1) <> '+' AND SUBSTR(v_item,1,1) <> '-' THEN 
      -- assume - (Northern hemisphere)
      v_latitude := '+'||v_item;
    ELSE
      v_latitude := v_item;
    END IF;
    
    IF LENGTH(v_latitude) > 3 THEN
      IF SUBSTR(v_latitude,4,1) <> '.' THEN
        RAISE e_bad_format;
      END IF;
    END IF;
    
    IF SUBSTR(v_latitude,1,1) = '-' THEN
      p_lat_dir := 'S';
    ELSE
      p_lat_dir := 'N';
    END IF;
    
    p_lat_dec_deg := TRUNC(TO_NUMBER(v_latitude),7);
    p_lat_degrees := ABS(TO_NUMBER(SUBSTR(v_latitude,2,2)));
    p_lat_dec_min := 60 * (ABS(p_lat_dec_deg) - p_lat_degrees);
    p_lat_minutes := FLOOR(p_lat_dec_min);
    p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
  END IF;

  --truncate dec degrees
  IF p_lat_dir = 'S' THEN
    IF abs(p_lat_dec_deg) < 10 THEN
      p_lat_dec_deg := TRUNC(p_lat_dec_deg,7);
    ELSE
      p_lat_dec_deg := TRUNC(p_lat_dec_deg,6);
    END IF;
  ELSE
    p_lat_dec_deg := TRUNC(p_lat_dec_deg,7);
  END IF;

  --Check to see if any of the values are out of reasonable range.
  IF p_lat_degrees < 0 OR p_lat_degrees > 90 THEN
    WhatsBad := 'Lat Degrees';
    RAISE e_bad_number;
  ELSIF p_lat_minutes < 0 OR p_lat_minutes > 60 THEN
    WhatsBad := 'Lat Minutes';
    RAISE e_bad_number;
  ELSIF p_lat_seconds < 0 OR p_lat_seconds > 60 THEN
    WhatsBad := 'Lat Seconds';
    RAISE e_bad_number;
  END IF;

  RETURN 1;
  
EXCEPTION
  WHEN e_bad_format THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Latitude does not match chosen format.',v_sta_seq);
     RETURN 2;
  WHEN e_bad_number THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': '||WhatsBad||' is out of range.',v_sta_seq);
      RETURN 2;
  WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Latitude does not match chosen format.',v_sta_seq);
      RETURN 2;
END get_latitude;


/******************************************************************************************************/
FUNCTION GET_LONGITUDE(v_item_in IN VARCHAR2, 
                       v_column_seq IN NUMBER, 
                       p_imp_cfg_seq IN NUMBER,
                       v_sta_seq IN NUMBER, 
                       v_line_cnt IN NUMBER, 
                       p_long_degrees OUT NUMBER,
                       p_long_minutes OUT NUMBER, 
                       p_long_seconds OUT NUMBER, 
                       p_long_dec_min OUT NUMBER,
                       p_long_dec_deg OUT NUMBER, 
                       p_long_dir OUT VARCHAR2) RETURN NUMBER IS

  v_long_format VARCHAR2(15);
  v_longitude   VARCHAR2(20);
  WhatsBad      VARCHAR2(15);
  v_item        VARCHAR2(60);
  
  e_bad_format  EXCEPTION;
  e_bad_number  EXCEPTION;


  CURSOR c_get_format IS
    SELECT SIFC_FMT_CHOICE
      FROM SIM_IMP_FMT_CHOICES C, SIM_IMP_CFG_DETS D
      WHERE D.SICDT_SIC_SEQ = v_column_seq
      AND D.SICDT_SICC_SEQ = p_imp_cfg_seq
      AND D.SICDT_SIFC_SEQ = C.SIFC_SEQ;
  
BEGIN
  v_item := v_item_in;
  OPEN c_get_format;
    FETCH c_get_format INTO v_long_format;
    IF c_get_format%NOTFOUND THEN --require a format
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,' Line '||v_line_cnt||' you must choose a format for Longitude.',v_sta_seq);
      RETURN 2;
    END IF;
  CLOSE c_get_format;

  IF v_long_format = 'DDD-MM-SS.SSSS' THEN
    IF SUBSTR(v_item,4,1) != '-' OR SUBSTR(v_item,7,1) != '-' THEN
      IF SUBSTR(v_item,3,1) = '-' AND SUBSTR(v_item,6,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_degrees := TO_NUMBER(SUBSTR(v_item,1,3));
    p_long_minutes := NVL(TO_NUMBER(SUBSTR(v_item,5,2)),0);
    p_long_seconds := NVL(TO_NUMBER(SUBSTR(v_item,8,7)),0);
    p_long_dec_min := p_long_minutes + (p_long_seconds/60);
    p_long_dec_deg := TRUNC(p_long_degrees + (p_long_dec_min/60),7);
    p_long_dir := NULL;
    
  ELSIF v_long_format = 'DDD MM SS.SSSS' THEN
    IF SUBSTR(v_item,4,1) != ' ' OR SUBSTR(v_item,7,1) != ' ' THEN
      IF SUBSTR(v_item,3,1) = ' ' AND SUBSTR(v_item,6,1) = ' ' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_degrees := TO_NUMBER(SUBSTR(v_item,1,3));
    p_long_minutes := NVL(TO_NUMBER(SUBSTR(v_item,5,2)),0);
    p_long_seconds := NVL(TO_NUMBER(SUBSTR(v_item,8,7)),0);
    p_long_dec_min := p_long_minutes + (p_long_seconds/60);
    p_long_dec_deg := TRUNC(p_long_degrees + (p_long_dec_min/60),7);
    p_long_dir := NULL;     

  ELSIF v_long_format = 'DDD-MM.MMMM' THEN
    IF SUBSTR(v_item,4,1) != '-' THEN
      IF SUBSTR(v_item,3,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_degrees := TO_NUMBER(SUBSTR(v_item,1,3));
    p_long_minutes := NVL(TO_NUMBER(SUBSTR(v_item,5,2)),0);
    p_long_dec_min := NVL(TO_NUMBER(SUBSTR(v_item,5,7)),0);
    p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
    p_long_dec_deg := TRUNC(p_long_degrees + (p_long_dec_min/60),7);
    p_long_dir := NULL;

  ELSIF v_long_format = 'DDD.DDDDDD' THEN
    IF SUBSTR(v_item,4,1) != '.' THEN
      IF SUBSTR(v_item,3,1) = '.' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_dec_deg := TRUNC(TO_NUMBER(v_item),7);
    p_long_degrees := TO_NUMBER(SUBSTR(v_item, 1,3));
    p_long_dec_min := 60 * (p_long_dec_deg - p_long_degrees);
    p_long_minutes := FLOOR(p_long_dec_min);
    p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
    p_long_dir := NULL;

  ELSIF v_long_format = '-DDD.DDDDD' THEN
    
    --start by making sure there is a sign on the front
    IF SUBSTR(v_item,1,1) <> '+' AND SUBSTR(v_item,1,1) <> '-' THEN --assume + (Eastern hemisphere)
      v_longitude := '+'||v_item;
    ELSE
      v_longitude := v_item;
    END IF;
    
    -- Pad longitude with 0 for 2 digit longitudes
    IF LENGTH(v_longitude) > 4 THEN
      IF SUBSTR(v_longitude,5,1) <> '.' THEN
        IF SUBSTR(v_longitude,4,1) = '.' THEN
          v_longitude := SUBSTR(v_longitude,1,1)||'0'||SUBSTR(v_longitude,2,LENGTH(v_longitude)) ;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
    ELSE
      IF LENGTH(v_longitude) <> 4 THEN
        IF LENGTH(v_longitude) = 3 THEN
          v_longitude := SUBSTR(v_longitude,1,1)||'0'||SUBSTR(v_longitude,2,LENGTH(v_longitude)) ;
        ELSE 
          RAISE e_bad_format;
        END IF;
      END IF;  
    END IF;
    
    IF SUBSTR(v_longitude,1,1) = '+' THEN
      p_long_dir := 'E';
    ELSE
      p_long_dir := 'W';
    END IF;
    
    p_long_dec_deg := TRUNC(TO_NUMBER(v_longitude),7);
    p_long_degrees := ABS(TO_NUMBER(SUBSTR(v_longitude, 2,3)));
    p_long_dec_min := 60 * (ABS(p_long_dec_deg) - p_long_degrees);
    p_long_minutes := FLOOR(p_long_dec_min);
    p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
  END IF;

  --truncate dec_degrees
  IF p_long_dir = 'W' THEN
    IF abs(p_long_dec_deg) < 100 THEN
      p_long_dec_deg := TRUNC(p_long_dec_deg,6);
    ELSE
      p_long_dec_deg := TRUNC(p_long_dec_deg,5);
    END IF;
  ELSE
    IF p_long_dec_deg < 100 THEN
      p_long_dec_deg := TRUNC(p_long_dec_deg,7);
    ELSE
      p_long_dec_deg := TRUNC(p_long_dec_deg,6);
    END IF;
  END IF;

  --Check to see if any of the values are out of reasonable range.
  IF p_long_degrees < 0 OR p_long_degrees > 180 THEN
    WhatsBad := 'Long Degrees';
    RAISE e_bad_number;
  ELSIF p_long_minutes < 0 OR p_long_minutes > 60 THEN
    WhatsBad := 'Long Minutes';
    RAISE e_bad_number;
  ELSIF p_long_seconds < 0 OR p_long_seconds > 60 THEN
    WhatsBad := 'Long Seconds';
    RAISE e_bad_number;
  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_bad_format THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Longitude does not match chosen format.',v_sta_seq);
     RETURN 2;
  WHEN e_bad_number THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': '||WhatsBad||' is out of range.',v_sta_seq);
      RETURN 2;
  WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Longitude does not match chosen format.',v_sta_seq);
      RETURN 2;
END get_longitude;

  /******************************************************************************************************
    The create_station function inserts known good data into the SIM table SIM_STATIONS
    After the station has loaded sucessfully the function returns a 1
    If there was a problem the function returns a zero.
  /************************************************************************************************************/
  FUNCTION CREATE_STATION(P_STATION_REC IN SIM_STATIONS%ROWTYPE)
    RETURN NUMBER IS
  BEGIN
    /* create station using values in station record */
    INSERT INTO SIM_STATIONS
      (STA_SEQ,
       STA_ID,
       STA_NAME,
       STA_DESC,
       STA_IS_NUMBER,
       STA_ORG_ID,
       STA_TSMORGAN_ORG_ID,
       STA_TSMORGAN_IS_NUMBER,
       STA_STATUS,
       STA_ESTAB_DATE,
       STA_EXPORT_SEQ,
       STA_IMPORT_SEQ,
       STA_TSMVSTC_IS_NUMBER,
       STA_TSMVSTC_ORG_ID,
       STA_EPA_KEY_ID,
       STA_ZID_REL,
       STA_INFLUENCE_AREA,
       STA_TRAVEL_DIR,
       STA_WATER_DEPTH,
       STA_WATER_DEPTH_UNIT,
       STA_ECOREGION)
    VALUES
      (P_STATION_REC.STA_SEQ, /* STA_SEQ            */
       P_STATION_REC.STA_ID, /* STA_ID             */
       P_STATION_REC.STA_NAME, /* STA_NAME           */
       P_STATION_REC.STA_DESC, /* STA_DESC           */
       P_STATION_REC.STA_IS_NUMBER,
       P_STATION_REC.STA_ORG_ID, /* STATION ORG ID     */
       P_STATION_REC.STA_TSMORGAN_ORG_ID, /* TSMORGAN_ORG_ID    */
       P_STATION_REC.STA_TSMORGAN_IS_NUMBER, /* TSMORGAN_IS_NUMBER */
       P_STATION_REC.STA_STATUS, /* STA_STATUS         */
       P_STATION_REC.STA_ESTAB_DATE, /* STA_ESTAB_DATE     */
       P_STATION_REC.STA_EXPORT_SEQ, /* STA_EXPORT_SEQ     */
       P_STATION_REC.STA_IMPORT_SEQ, /* STA_IMPORT_SEQ     */
       P_STATION_REC.STA_TSMVSTC_IS_NUMBER, /* STA_TSMVSTC_IS_NUMBER     */
       P_STATION_REC.STA_TSMVSTC_ORG_ID, /* STA_TSMVSTC_ORG_ID */
       P_STATION_REC.STA_EPA_KEY_ID,
       P_STATION_REC.STA_ZID_REL,
       P_STATION_REC.STA_INFLUENCE_AREA,
       P_STATION_REC.STA_TRAVEL_DIR,
       P_STATION_REC.STA_WATER_DEPTH,
       P_STATION_REC.STA_WATER_DEPTH_UNIT,
       P_STATION_REC.STA_ECOREGION);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting station.',
                                  NULL);
      RETURN 0;
  END CREATE_STATION;

  /*************************************************************************************************
  * This function checks to see if a station already exists in STORET and SIM.                    *
  * If a station is found in STORET but not in SIM, a copy of the Station is created in SIM.      *
  * A copy of all locations is also added to SIM at this time.                                    *
  *************************************************************************************************/
  FUNCTION CHECK_IF_STATION_EXISTS(P_STATION_CD IN VARCHAR2,
                                   P_ORG_ID     IN VARCHAR2,
                                   P_STA_SEQ    OUT NUMBER,
                                   P_STA_IS_NBR OUT NUMBER) RETURN NUMBER IS
  
    V_STORET_STATION_CD VARCHAR2(15) := P_STATION_CD;
    V_SIM_STATION_CD    VARCHAR2(15) := P_STATION_CD;
    V_RETURN            NUMBER(1) := 0;
    V_SEQ               NUMBER(10);
    V_SUCCESS           NUMBER(1);
    V_STA_REC           SIM_STATIONS%ROWTYPE;
    V_STOR_STA_REC      TSMSTATN%ROWTYPE;
    V_TSMSTATN_IS_NBR   NUMBER(12);
    V_SIM_LOC_REC       SIM_STATION_LOCATIONS%ROWTYPE;
    V_STORET_LOC_REC    TSMALP%ROWTYPE;
    V_LOCATION_SEQ      NUMBER(10);
  
    E_BAD_LOC EXCEPTION;
  
    CURSOR C_COMPARE_STATION_ID(P_STATION_CD IN VARCHAR2, P_STATION_ORG_ID IN VARCHAR2) IS
      SELECT IDENTIFICATION_CD, TSMSTATN_IS_NUMBER
        FROM TSMSTATN
       WHERE IDENTIFICATION_CD = RPAD(P_STATION_CD, 15) AND
             TSMSTATN_ORG_ID = RPAD(P_STATION_ORG_ID, 8);
  
    CURSOR C_COMPARE_SIM_STAT_ID(P_STATION_CD IN VARCHAR2, P_STATION_ORG_ID IN VARCHAR2, P_TSMSTATN_IS_NBR IN NUMBER) IS
      SELECT STA_SEQ, STA_ID
        FROM SIM_STATIONS
       WHERE STA_ID = P_STATION_CD AND STA_IS_NUMBER = P_TSMSTATN_IS_NBR AND
             STA_ORG_ID = RPAD(P_STATION_ORG_ID, 8);
  
    CURSOR C_GET_STATION_SEQ IS
      SELECT SIM_STATIONS_SEQ.NEXTVAL FROM DUAL;
  
    CURSOR C_GET_LOCATION_SEQ IS
      SELECT SIM_STATION_LOCATIONS_SEQ.NEXTVAL FROM DUAL;
  
    CURSOR C_STORET_LOCATIONS(P_TSMSTATN_IS_NUMBER IN NUMBER, P_TSMSTATN_ORG_ID IN VARCHAR2) IS
      SELECT * --TSMALP_IS_NUMBER
        FROM TSMALP
       WHERE TSMSTATN0IS_NUMBER = P_TSMSTATN_IS_NUMBER AND
             TSMSTATN0ORG_ID = RPAD(P_TSMSTATN_ORG_ID, 8);
  
    CURSOR C_STORET_STATION IS
      SELECT *
        FROM TSMSTATN
       WHERE IDENTIFICATION_CD = RPAD(P_STATION_CD, 15) AND
             TSMSTATN_ORG_ID = RPAD(P_ORG_ID, 8);
  
  BEGIN
  
    /* Check STORET */
    OPEN C_COMPARE_STATION_ID(P_STATION_CD, P_ORG_ID);
    FETCH C_COMPARE_STATION_ID INTO V_STORET_STATION_CD, V_TSMSTATN_IS_NBR;
      
    IF C_COMPARE_STATION_ID%NOTFOUND THEN
      V_RETURN := 0;
    ELSE

      --Check to see if it's in SIM, too. If it's not then create the SIM station and location records      
      OPEN C_COMPARE_SIM_STAT_ID(P_STATION_CD, P_ORG_ID, V_TSMSTATN_IS_NBR);
      FETCH C_COMPARE_SIM_STAT_ID
        INTO V_SEQ, V_SIM_STATION_CD;

      IF C_COMPARE_SIM_STAT_ID%NOTFOUND THEN
        OPEN C_GET_STATION_SEQ;
        FETCH C_GET_STATION_SEQ
          INTO V_SEQ;
        CLOSE C_GET_STATION_SEQ;
      
        OPEN C_STORET_STATION;
        FETCH C_STORET_STATION
          INTO V_STOR_STA_REC;
        CLOSE C_STORET_STATION;
      
        V_STA_REC.STA_SEQ                := V_SEQ;
        V_STA_REC.STA_IS_NUMBER          := V_STOR_STA_REC.TSMSTATN_IS_NUMBER;
        V_STA_REC.STA_ORG_ID             := V_STOR_STA_REC.TSMSTATN_ORG_ID;
        V_STA_REC.STA_ID                 := RTRIM(V_STOR_STA_REC.IDENTIFICATION_CD);
        V_STA_REC.STA_NAME               := RTRIM(V_STOR_STA_REC.NAME);
        V_STA_REC.STA_ESTAB_DATE         := V_STOR_STA_REC.ESTABLISHMENT_DATE;
        V_STA_REC.STA_DESC               := V_STOR_STA_REC.DESCRIPTION_TEXT;
        V_STA_REC.STA_TSMORGAN_ORG_ID    := P_ORG_ID;
        V_STA_REC.STA_TSMORGAN_IS_NUMBER := V_STOR_STA_REC.TSMORGAN_IS_NUMBER;
        V_STA_REC.STA_TSMVSTC_IS_NUMBER  := V_STOR_STA_REC.TSMVSTC_IS_NUMBER;
        V_STA_REC.STA_TSMVSTC_ORG_ID     := V_STOR_STA_REC.TSMVSTC_ORG_ID;
        V_STA_REC.STA_STATUS             := 'U';
        V_STA_REC.STA_EXPORT_SEQ         := NULL;
        V_STA_REC.STA_IMPORT_SEQ         := V_IMPORT_LOG_SEQ;
        V_STA_REC.STA_EPA_KEY_ID         := V_STOR_STA_REC.EPA_KEY_IDENTIFIER;
        V_STA_REC.STA_ZID_REL            := V_STOR_STA_REC.ZID_RELATION_CODE;
        V_STA_REC.STA_INFLUENCE_AREA     := V_STOR_STA_REC.INFLUENCE_AREA;
        V_STA_REC.STA_TRAVEL_DIR         := V_STOR_STA_REC.TRAVEL_DIR_TXT;
        V_STA_REC.STA_WATER_DEPTH        := V_STOR_STA_REC.WATER_DEPTH;
        V_STA_REC.STA_WATER_DEPTH_UNIT   := V_STOR_STA_REC.WATER_DEPTH_UNIT;
        V_STA_REC.STA_ECOREGION          := V_STOR_STA_REC.ECOREGION_NAME;
      
        V_SUCCESS := CREATE_STATION(V_STA_REC);
        IF V_SUCCESS = 1 THEN
          V_RETURN := 1;
        END IF;
        IF V_RETURN = 1 THEN
          FOR V_STORET_LOC_REC IN C_STORET_LOCATIONS(V_TSMSTATN_IS_NBR,
                                                     P_ORG_ID) LOOP
            V_RETURN := 0;
            OPEN C_GET_LOCATION_SEQ;
            FETCH C_GET_LOCATION_SEQ
              INTO V_LOCATION_SEQ;
            CLOSE C_GET_LOCATION_SEQ;
          
            V_SIM_LOC_REC.STL_SEQ                  := V_LOCATION_SEQ;
            V_SIM_LOC_REC.STL_STA_SEQ              := V_SEQ;
            V_SIM_LOC_REC.STL_TYPE_CODE            := V_STORET_LOC_REC.TYPE_CODE;
            V_SIM_LOC_REC.STL_SEQUENCE_NUMBER      := V_STORET_LOC_REC.SEQUENCE_NUMBER;
            V_SIM_LOC_REC.STL_ELEV_DATUM           := NULL;
            V_SIM_LOC_REC.STL_ELEV_METHOD          := NULL;
            V_SIM_LOC_REC.STL_ELEV_MSR_DATE        := V_STORET_LOC_REC.ELVTN_MSR_DT;
            V_SIM_LOC_REC.STL_ELEVATION_MSR        := V_STORET_LOC_REC.ELEVATION_MSR;
            V_SIM_LOC_REC.STL_ELEVATION_UNIT       := NULL;
            V_SIM_LOC_REC.STL_GEOPOS_DATUM         := NULL;
            V_SIM_LOC_REC.STL_GEOPOS_METHOD        := NULL;
            V_SIM_LOC_REC.STL_GPS_SCALE            := V_STORET_LOC_REC.GEOPSTNG_SCALE_TXT;
            V_SIM_LOC_REC.STL_LAT_DIR              := V_STORET_LOC_REC.LAT_DIRECTION;
            V_SIM_LOC_REC.STL_LAT_DEGREES          := V_STORET_LOC_REC.LAT_DEGREE_MSR;
            V_SIM_LOC_REC.STL_LAT_MINUTES          := V_STORET_LOC_REC.LAT_MINUTE_MSR;
            V_SIM_LOC_REC.STL_LAT_SECONDS          := V_STORET_LOC_REC.LAT_SECOND_MSR;
            V_SIM_LOC_REC.STL_LONG_DIR             := V_STORET_LOC_REC.LONG_DIRECTION;
            V_SIM_LOC_REC.STL_LONG_DEGREES         := V_STORET_LOC_REC.LONG_DEGREE_MSR;
            V_SIM_LOC_REC.STL_LONG_MINUTES         := V_STORET_LOC_REC.LONG_MINUTE_MSR;
            V_SIM_LOC_REC.STL_LONG_SECONDS         := V_STORET_LOC_REC.LONG_SECOND_MSR;
            V_SIM_LOC_REC.STL_LAT_LONG_MSR_DATE    := V_STORET_LOC_REC.LAT_LONG_MSR_DT;
            V_SIM_LOC_REC.STL_LATLONG_ACCURCY      := NULL;
            V_SIM_LOC_REC.STL_LATLONG_ACCURCY_UNIT := NULL;
            V_SIM_LOC_REC.STL_TSMRRR_IS_NUMBER     := V_STORET_LOC_REC.TSMRRR_IS_NUMBER;
            V_SIM_LOC_REC.STL_TSMRRR_ORG_ID        := V_STORET_LOC_REC.TSMRRR_ORG_ID;
            V_SIM_LOC_REC.STL_TSMFHU_IS_NUMBER     := V_STORET_LOC_REC.TSMFHU_IS_NUMBER;
            V_SIM_LOC_REC.STL_TSMFHU_ORG_ID        := V_STORET_LOC_REC.TSMFHU_ORG_ID;
            V_SIM_LOC_REC.STL_TSMSTATN0IS_NUMBER   := V_STORET_LOC_REC.TSMSTATN0IS_NUMBER;
            V_SIM_LOC_REC.STL_TSMSTATN0ORG_ID      := V_STORET_LOC_REC.TSMSTATN0ORG_ID;
            V_SIM_LOC_REC.STL_TSMWELL_IS_NUMBER    := V_STORET_LOC_REC.TSMWELL_IS_NUMBER;
            V_SIM_LOC_REC.STL_TSMWELL_ORG_ID       := V_STORET_LOC_REC.TSMWELL_ORG_ID;
            V_SIM_LOC_REC.STL_TSMGEOPA_IS_NUMBER   := V_STORET_LOC_REC.TSMGEOPA_IS_NUMBER;
            V_SIM_LOC_REC.STL_TSMGEOPA_ORG_ID      := V_STORET_LOC_REC.TSMGEOPA_ORG_ID;
            V_SIM_LOC_REC.STL_LAT_DEC_MINUTES      := NULL;
            V_SIM_LOC_REC.STL_LONG_DEC_MINUTES     := NULL;
            V_SIM_LOC_REC.STL_TSMPIPE_IS_NUMBER    := NULL;
            V_SIM_LOC_REC.STL_TSMPIPE_ORG_ID       := NULL;
            V_SIM_LOC_REC.STL_TSMSTATN1ORG_ID      := NULL;
            V_SIM_LOC_REC.STL_TSMSTATN1IS_NUMBER   := NULL;
            V_SIM_LOC_REC.STL_TSMGEOPA0IS_NUMBER   := NULL;
            V_SIM_LOC_REC.STL_TSMGEOPA0ORG_ID      := NULL;
            V_SIM_LOC_REC.STL_TSMNAL_CD            := V_STORET_LOC_REC.TSMNAL_CD;
            V_SIM_LOC_REC.STL_TSMNAL_STATE         := V_STORET_LOC_REC.TSMNAL_STATE;
            V_SIM_LOC_REC.STL_NRCS                 := V_STORET_LOC_REC.NRCS_WTRSD_ID_NUM;
            V_SIM_LOC_REC.STL_STATUS               := 'U';
            V_SIM_LOC_REC.STL_IMPORT_SEQ           := V_IMPORT_LOG_SEQ;
            V_SIM_LOC_REC.STL_EXPORT_SEQ           := NULL;
          
            V_SUCCESS := CREATE_STATION_LOCATION(V_SIM_LOC_REC);
            IF V_SUCCESS = 1 THEN
              V_RETURN := 1;
            ELSE
              RAISE E_BAD_LOC;
            END IF;
          END LOOP;
        END IF;

      ELSE
        V_RETURN := 1;
      END IF;
      CLOSE C_COMPARE_SIM_STAT_ID;
    END IF;

    P_STA_SEQ    := V_SEQ;
    P_STA_IS_NBR := V_TSMSTATN_IS_NBR;
  
    CLOSE C_COMPARE_STATION_ID;
    /** Station is unique **/
    RETURN V_RETURN;
  
  EXCEPTION
  
    WHEN E_BAD_LOC THEN
  
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM ||
        ' grabbing STORET locatins in check_if_station_exists. Line: ' ||
        V_LINE_CNT,
        NULL
      );
      
      RETURN 0;
  
    WHEN OTHERS THEN
  
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM ||
        ' - in check_if_station_exists. Line: ' ||
        V_LINE_CNT,
        NULL
      );
      
      RETURN 0;
      
  END CHECK_IF_STATION_EXISTS;

  /*********************************************************************************************************/
  FUNCTION GET_ORG_INFO(P_IMP_CFG_SEQ     IN NUMBER,
                        P_TSMORGAN_ORG_ID IN VARCHAR2,
                        P_TSMORGAN_IS_NBR OUT NUMBER) RETURN NUMBER IS
    CURSOR C_GET_TSMORGAN_NBR(P_CURSOR_ORG_ID IN VARCHAR2) IS
      SELECT TSMORGAN_IS_NUMBER
        FROM TSMORGAN
       WHERE ORG_ID = RPAD(P_CURSOR_ORG_ID, 8);
  BEGIN
    OPEN C_GET_TSMORGAN_NBR(P_TSMORGAN_ORG_ID);
    FETCH C_GET_TSMORGAN_NBR
      INTO P_TSMORGAN_IS_NBR;
    IF C_GET_TSMORGAN_NBR%NOTFOUND THEN
      P_TSMORGAN_IS_NBR := NULL;
    END IF;
    CLOSE C_GET_TSMORGAN_NBR;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      RETURN 0;
  END GET_ORG_INFO;

  /******************************************************************************************************/
  FUNCTION GET_INSTRUCTIONS(P_IMP_CFG_SEQ      IN NUMBER,
                            P_INSTRUCTION_TYPE IN VARCHAR2) RETURN VARCHAR2 IS
    V_INSTR_CHOICE VARCHAR2(15);
    CURSOR C_GET_CFG_INSTR(P_CFG_SEQ IN NUMBER, P_INSTR_TYPE IN VARCHAR2) IS
      SELECT SIIC_CODE
        FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
       WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ AND
             SII_CODE = P_INSTR_TYPE AND SICI_SICC_SEQ = P_CFG_SEQ;
    CURSOR C_GET_INSTR_DEFAULT(P_INSTR_TYPE IN VARCHAR2) IS
      SELECT SIIC_CODE
        FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
       WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ AND
             SII_CODE = P_INSTR_TYPE AND SIIC_DFLT_FLAG = 1;
  BEGIN
    /* kms - get the instruction choice for this item */
    OPEN C_GET_CFG_INSTR(P_IMP_CFG_SEQ, P_INSTRUCTION_TYPE);
    FETCH C_GET_CFG_INSTR
      INTO V_INSTR_CHOICE;
    IF C_GET_CFG_INSTR%NOTFOUND THEN
      /* no choice could be found, get default choice */
      OPEN C_GET_INSTR_DEFAULT(P_INSTRUCTION_TYPE);
      FETCH C_GET_INSTR_DEFAULT
        INTO V_INSTR_CHOICE;
      CLOSE C_GET_INSTR_DEFAULT;
    END IF;
    CLOSE C_GET_CFG_INSTR;
    RETURN V_INSTR_CHOICE;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get_instructions.',
                                  NULL);
      RETURN NULL;
  END GET_INSTRUCTIONS;

  /******************************************************************************************************/
  FUNCTION GET_ALLOWABLE_VALUE(
           P_ALVAL_SEQ IN NUMBER, 
           P_SEQ       IN NUMBER)
           RETURN VARCHAR2 IS
           
    V_ITEM_NAME VARCHAR2(80);
  
    CURSOR C_GET_ALVAL_NAME(P_ALVAL_SEQ IN NUMBER) IS
    SELECT SIA_ITEM
    FROM   SIM_IMP_ALLOW_VALUES
    WHERE  SIA_SEQ = P_ALVAL_SEQ;
  
  BEGIN
    OPEN C_GET_ALVAL_NAME(P_ALVAL_SEQ);
    FETCH C_GET_ALVAL_NAME INTO V_ITEM_NAME;
    
    IF C_GET_ALVAL_NAME%NOTFOUND THEN
      V_ITEM_NAME := NULL;
    END IF;
    CLOSE C_GET_ALVAL_NAME;
    
    RETURN V_ITEM_NAME;
    
  EXCEPTION
    
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM || ' - getting allowable value from sequence.',
        P_SEQ
      );
      RETURN NULL;
      
  END GET_ALLOWABLE_VALUE;

  /******************************************************************************************************/
  FUNCTION VALIDATE_PRMVL_ITEM(P_ORIG_ITEM   IN CHAR,
                               P_IMP_CFG_SEQ IN NUMBER,
                               P_ITEM_NAME   IN VARCHAR2,
                               P_COL_SEQ     IN NUMBER,
                               P_TABLE_NAME  IN VARCHAR,
                               P_COLUMN_NAME IN VARCHAR2,
                               P_SEQ         IN NUMBER) RETURN VARCHAR2 IS
  
    -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.
  
    V_CONVERTED_ITEM VARCHAR2(40);
    V_TRANS_EXIST    BOOLEAN := TRUE;
    V_STORET_VALUE   VARCHAR2(40);
    V_INSTR_ID       VARCHAR2(8);
    V_INSTR_CODE     VARCHAR2(15);
  
    CURSOR C_CHECK_PRMVL(P_ITEM IN VARCHAR2) IS
      SELECT FIELD_VALUE
        FROM TSMPRMVL
       WHERE TABLE_NAME = RPAD(P_TABLE_NAME, 8) AND
             FIELD_NAME = RPAD(P_COLUMN_NAME, 20) AND
             UPPER(FIELD_VALUE) = RPAD(UPPER(P_ITEM), 40);
  
  BEGIN
    V_STORET_VALUE   := NULL;
    V_CONVERTED_ITEM := CHECK_TRANSLATIONS(P_ORIG_ITEM,
                                           P_IMP_CFG_SEQ,
                                           P_COL_SEQ,
                                           V_TRANS_EXIST);
  
    IF V_CONVERTED_ITEM IS NOT NULL THEN
      OPEN C_CHECK_PRMVL(V_CONVERTED_ITEM);
      FETCH C_CHECK_PRMVL
        INTO V_STORET_VALUE;
      IF C_CHECK_PRMVL %NOTFOUND THEN
        V_STORET_VALUE := NULL;
      END IF;
      CLOSE C_CHECK_PRMVL;
    
      IF V_STORET_VALUE IS NULL AND NOT V_TRANS_EXIST THEN
        V_INSTR_ID := 'LOC-TR';
        /* check instructions to see if translation record should be created */
        V_INSTR_CODE := GET_INSTRUCTIONS(P_IMP_CFG_SEQ, V_INSTR_ID);
        IF V_INSTR_CODE = 'LOC-TR-01' THEN
          /* if record could not be matched to anything in translations or STORET tables,
          create new record */
          CREATE_TRANS_RECORD(P_ORIG_ITEM, P_IMP_CFG_SEQ, P_COL_SEQ);
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT || ': ' ||
                                       P_ITEM_NAME || ', ' ||
                                       V_CONVERTED_ITEM ||
                                       ' is not a valid value - translation record has been created.',
                                      P_SEQ);
        ELSE
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT || ': ' ||
                                       P_ITEM_NAME || ', ' ||
                                       V_CONVERTED_ITEM ||
                                       ' is not a valid value.',
                                      P_SEQ);
        END IF;
      ELSIF V_STORET_VALUE IS NULL AND V_TRANS_EXIST THEN
        /*  item has been translated to an invalid storet value */
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                    P_ITEM_NAME || ', ' || P_ORIG_ITEM ||
                                    ' has been mapped to the invalid storet value: ' ||
                                    V_CONVERTED_ITEM,
                                    P_SEQ);
      END IF;
    ELSIF V_TRANS_EXIST THEN
      /* item exists in translations table but is not mapped to anything, write error message.
      Status will be changed to 'N' later in the program. */
      V_STORET_VALUE := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  P_ITEM_NAME || ', ' || P_ORIG_ITEM ||
                                   ' exists in translation table but has not been mapped to a STORET code. ',
                                  P_SEQ);
    END IF;
  
    RETURN V_STORET_VALUE;
  
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - checking STORET PRMVL.',
                                  P_SEQ);
      RETURN NULL;
  END VALIDATE_PRMVL_ITEM;

  /******************************************************************************************************
      The get_delimiter function checks the table sim_imp_config to find the value entered for the
      delimiter.
  /****************************************************************************************************/
  FUNCTION GET_DELIMITER(P_IMP_CFG_SEQ IN NUMBER, P_DELIMITER OUT VARCHAR2)
    RETURN NUMBER IS
  
    CURSOR C_GET_IMP_CFG(P_SICC_SEQ IN NUMBER) IS
      SELECT SICC_DELIMITER FROM SIM_IMP_CFG WHERE SICC_SEQ = P_SICC_SEQ;
  
  BEGIN
    OPEN C_GET_IMP_CFG(P_IMP_CFG_SEQ);
    FETCH C_GET_IMP_CFG
      INTO P_DELIMITER;
    IF C_GET_IMP_CFG%NOTFOUND THEN
      RETURN 0;
    END IF;
    CLOSE C_GET_IMP_CFG;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      RETURN 0;
  END GET_DELIMITER;

  /******************************************************************************************************
      The get_item_pos function checks the table sim_imp_config SIM_IMP_CFG_DETS SIM_IMP_COLS to find the value entered for the
      the position of the item in the list
  /****************************************************************************************************/
  FUNCTION GET_ITEM_POS(P_SICC_SEQ      IN NUMBER,
                        P_POS_TABLE     IN OUT COL_TABLE_NUM_TYPE,
                        P_DEFAULT_TABLE IN OUT COL_TABLE_TYPE,
                        P_COL_CNT       IN OUT NUMBER) RETURN NUMBER IS
    V_IMPORT_TYPE_SEQ NUMBER(10);
    CURSOR C_GET_IMPORT_TYPE(P_SICC_SEQ IN NUMBER) IS
      SELECT SICC_SID_SEQ FROM SIM_IMP_CFG WHERE SICC_SEQ = P_SICC_SEQ;
    CURSOR C_GET_DFLT_POS(P_TYPE_SEQ IN NUMBER) IS
      SELECT SIC_DFLT_POS FROM SIM_IMP_COLS WHERE SIC_SID_SEQ = P_TYPE_SEQ;
    CURSOR C_GET_COL_POS(P_SICC_SEQ IN NUMBER, P_DFLT_POS IN NUMBER) IS
      SELECT SICDT_POS, SICDT_DEFAULT
        FROM SIM_IMP_CFG_DETS, SIM_IMP_COLS
       WHERE SIC_SEQ = SICDT_SIC_SEQ AND SICDT_SICC_SEQ = P_SICC_SEQ AND
             SIC_DFLT_POS = P_DFLT_POS AND
             (SICDT_INCLUDED IS NOT NULL OR SICDT_GENERATE IS NOT NULL);
  BEGIN
    /* get the import type in order to limit the columns selected to the right type */
    OPEN C_GET_IMPORT_TYPE(P_SICC_SEQ);
    FETCH C_GET_IMPORT_TYPE
      INTO V_IMPORT_TYPE_SEQ;
    IF C_GET_IMPORT_TYPE%NOTFOUND THEN
      V_IMPORT_TYPE_SEQ := NULL;
      RAISE_APPLICATION_ERROR(-20102,
                              'Unable to get import type from import configuration.');
    END IF;
    CLOSE C_GET_IMPORT_TYPE;
    P_COL_CNT := 0;
    /* loop through import columns and assign actual position to table indexed by
    default position */
    FOR V_DFLT_POS IN C_GET_DFLT_POS(V_IMPORT_TYPE_SEQ) LOOP
      /* get actual position and item default from configuration table */
      OPEN C_GET_COL_POS(P_SICC_SEQ, V_DFLT_POS.SIC_DFLT_POS);
      FETCH C_GET_COL_POS
        INTO P_POS_TABLE(V_DFLT_POS.SIC_DFLT_POS), P_DEFAULT_TABLE(V_DFLT_POS.SIC_DFLT_POS);
      IF C_GET_COL_POS%NOTFOUND THEN
        /* if there is nothing in the configuration table for this column, position and default will be null */
        P_POS_TABLE(V_DFLT_POS.SIC_DFLT_POS) := NULL;
        P_DEFAULT_TABLE(V_DFLT_POS.SIC_DFLT_POS) := NULL;
      ELSE
        P_COL_CNT := P_COL_CNT + 1;
      END IF;
      CLOSE C_GET_COL_POS;
    END LOOP;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20103,
                              'Failed to get column positions, item names, and column sequences.');
  END GET_ITEM_POS;

  /******************************************************************************************************/
  FUNCTION GET_COLUMN_SEQ(P_CFG_SEQ IN NUMBER, P_CFG_POS IN NUMBER)
    RETURN NUMBER IS
    V_IMP_COL_SEQ NUMBER(10);
    E_NO_CFG_FOUND EXCEPTION;
    CURSOR C_GET_COL_SEQ(P_CFG_SEQ IN NUMBER, P_CFG_POS IN NUMBER) IS
      SELECT SICDT_SIC_SEQ
        FROM SIM_IMP_CFG, SIM_IMP_CFG_DETS
       WHERE SICDT_SICC_SEQ = SICC_SEQ AND SICC_SEQ = P_CFG_SEQ AND
             SICDT_POS = P_CFG_POS;
  BEGIN
    OPEN C_GET_COL_SEQ(P_CFG_SEQ, P_CFG_POS);
    FETCH C_GET_COL_SEQ
      INTO V_IMP_COL_SEQ;
    IF C_GET_COL_SEQ%NOTFOUND THEN
      V_IMP_COL_SEQ := NULL;
      RAISE E_NO_CFG_FOUND;
    END IF;
    CLOSE C_GET_COL_SEQ;
    RETURN V_IMP_COL_SEQ;
  EXCEPTION
    WHEN E_NO_CFG_FOUND THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                  ' - unable to find sequence in sim_imp_cfg_dets for pos: ' ||
                                  P_CFG_POS,
                                  NULL);
      RAISE_APPLICATION_ERROR(-20100,
                              'Import configuration is not complete');
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - getting column sequence.',
                                  NULL);
      RETURN NULL;
  END GET_COLUMN_SEQ;

  /******************************************************************************************************/
  FUNCTION CHECK_TRANSLATIONS(P_CODE        IN VARCHAR2,
                            P_IMP_CFG_SEQ IN NUMBER,
                            P_COL_SEQ     IN NUMBER,
                            P_TRANS_EXIST OUT BOOLEAN) RETURN VARCHAR2 IS
  V_STORET_VALUE VARCHAR2(120);
	v_location_seq NUMBER(10);

  CURSOR C_GET_LOCATION_SEQ IS
    SELECT SIM_STATION_LOCATIONS_SEQ.CURRVAL FROM DUAL;

  CURSOR C_GET_STORET_CODE(P_CODE IN VARCHAR2, P_IMP_CFG_SEQ IN NUMBER, P_COL_SEQ IN NUMBER) IS
    SELECT ST_STORET_CODE
      FROM SIM_TRANSLATIONS, SIM_IMP_CFG_DETS
     WHERE ST_USER_CODE = P_CODE AND ST_SICDT_SEQ = SICDT_SEQ AND
           SICDT_SICC_SEQ = P_IMP_CFG_SEQ AND SICDT_SIC_SEQ = P_COL_SEQ;

BEGIN
  /* check translations table to see if the code should be swapped with another value */
  OPEN C_GET_STORET_CODE(P_CODE, P_IMP_CFG_SEQ, P_COL_SEQ);
  FETCH C_GET_STORET_CODE
    INTO V_STORET_VALUE;
  IF C_GET_STORET_CODE%NOTFOUND THEN
    /* if not in translations table return original value - will check elsewhere for sequences */
    V_STORET_VALUE := P_CODE;
    P_TRANS_EXIST  := FALSE;
  ELSIF C_GET_STORET_CODE%FOUND THEN
    /* if the code is in the translations table, flag so it's not recreated */
      P_TRANS_EXIST := TRUE;
      IF v_storet_value IS NULL THEN
        /* if there is a code in the translations table but no value to translate to, write error message
           status will be changed to 'N' later in the program */
				OPEN c_get_location_seq;
				FETCH c_get_location_seq INTO v_location_seq;
				CLOSE c_get_location_seq;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_code||' exists in translations table but has not been translated to a value.',v_location_seq);
			END IF;
    END IF;
    CLOSE C_GET_STORET_CODE;
    RETURN V_STORET_VALUE;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - in check translations',
                                  NULL);
  END CHECK_TRANSLATIONS;

  /******************************************************************************************************/
FUNCTION CHECK_STORET_TRANSLATION(P_CODE IN VARCHAR2,
                                  P_IMP_CFG_SEQ IN NUMBER,
                                  P_COL_SEQ IN NUMBER,
                                  P_TRANS_EXIST OUT BOOLEAN,
                                  V_CONVERTED_ITEM OUT VARCHAR2,
                                  P_STA_SEQ IN NUMBER) 
  RETURN T_STORET_PK_TYPE IS  
  
  v_storet_pk      SIM_IMPORT_LOCATIONS_PKG.T_STORET_PK_TYPE;
  v_seq            NUMBER(10);
  v_table_name     VARCHAR2(80);
  v_column_name    VARCHAR2(80);
  v_dyna_cursor    VARCHAR2(200);
  v_comma_pos      NUMBER(10);
  
  TYPE SORT_CURSOR_TYPE IS REF CURSOR;
  sort_cursor            sort_cursor_type;
    
  
  CURSOR c_get_storet_key(p_c_code VARCHAR2,p_c_imp_cfg_seq NUMBER,
                          p_c_col_seq NUMBER) IS
    SELECT st_seq,st_is_number,st_org_id,NULL
      FROM sim_translations,sim_imp_cfg_dets
      WHERE st_user_code = p_code
        AND st_sicdt_seq = sicdt_seq
        AND sicdt_sicc_seq = p_imp_cfg_seq
        AND sicdt_sic_seq = p_col_seq;

  CURSOR c_imp_cols(p_col_seq NUMBER) IS
    SELECT sic_storet_tab,sic_storet_col
      FROM sim_imp_cols
      WHERE sic_seq = p_col_seq;  
      
  CURSOR c_comma(p_col_seq NUMBER) IS
    SELECT INSTR(sic_storet_col,',') col_pos
      FROM sim_imp_cols
      WHERE sic_seq = p_col_seq;      
      
  CURSOR c_inst_col(p_col_seq NUMBER) IS
    SELECT SUBSTR(SIC_STORET_COL,1,INSTR(SIC_STORET_COL,',')-1) col_name
      FROM sim_imp_cols
      WHERE sic_seq = p_col_seq;             
        
BEGIN

  OPEN c_get_storet_key(p_code,p_imp_cfg_seq,p_col_seq);
    FETCH c_get_storet_key INTO v_seq,v_storet_pk.is_number,v_storet_pk.org_id,v_converted_item;
  CLOSE c_get_storet_key;
  IF v_seq IS NULL THEN
    v_converted_item := p_code;
    P_TRANS_EXIST  := FALSE;
  ELSIF v_seq IS NOT NULL THEN
    /* if the code is in the translations table, flag so it's not recreated */
    P_TRANS_EXIST := TRUE;
    
    IF v_storet_pk.is_number IS NULL THEN
      /* if there is a code in the translations table but no value to translate to, write error message
         status will be changed to 'N' later in the program */
       Sim2_Import_Pkg.write_error(
         v_import_log_seq, 
         'Line '||v_line_cnt||': '||
         p_code||' exists in translations table but has not been translated to a value.',
         p_sta_seq
       );
    ELSE
      OPEN c_imp_cols(p_col_seq);
        FETCH c_imp_cols INTO v_table_name,v_column_name;
      CLOSE c_imp_cols;
      OPEN c_comma(p_col_seq);
        FETCH c_comma INTO v_comma_pos;
      CLOSE c_comma;
      IF v_comma_pos = 0 THEN
        v_dyna_cursor := 'SELECT RTRIM('||v_column_name||') '||
                         ' FROM '||v_table_name||
                         ' WHERE '||v_table_name||'_IS_NUMBER = '||v_storet_pk.is_number||
                         ' AND '||v_table_name||'_ORG_ID = '''||v_storet_pk.org_id||'''';
      ELSE
        OPEN c_inst_col(p_col_seq);
          FETCH c_inst_col INTO v_column_name;
        CLOSE c_inst_col;
        v_dyna_cursor := 'SELECT RTRIM('||v_column_name||') '||
                         ' FROM '||v_table_name||
                         ' WHERE '||v_table_name||'_IS_NUMBER = '||v_storet_pk.is_number||
                         ' AND '||v_table_name||'_ORG_ID = '''||v_storet_pk.org_id||'''';
      END IF; 

      OPEN sort_cursor FOR v_dyna_cursor;
      LOOP
        FETCH sort_cursor INTO v_converted_item;
      EXIT WHEN sort_cursor%NOTFOUND;
      END LOOP;      
      IF v_converted_item IS NOT NULL THEN
        P_TRANS_EXIST := TRUE;
      END IF;
		END IF;
  END IF;  
  RETURN v_storet_pk;
  
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,'Line ' || V_LINE_CNT || ': ' || SQLERRM || ' - in check storet translations',NULL);
      RETURN NULL;
END CHECK_STORET_TRANSLATION;
/*****************************************************************************************/  
  PROCEDURE CREATE_TRANS_RECORD(P_CODE        IN VARCHAR2,
                                P_IMP_CFG_SEQ IN NUMBER,
                                P_COL_SEQ     IN NUMBER) IS
  
    V_ST_SEQ    NUMBER(10);
    V_SICDT_SEQ NUMBER(10);
    E_IMP_CFG_DETS EXCEPTION;
    E_TRANS_ERROR EXCEPTION;
  
    CURSOR C_GET_CFG_DET_SEQ(P_IMP_CFG_SEQ IN NUMBER, P_COL_SEQ IN NUMBER) IS
      SELECT SICDT_SEQ
        FROM SIM_IMP_CFG_DETS
       WHERE SICDT_SICC_SEQ = P_IMP_CFG_SEQ AND SICDT_SIC_SEQ = P_COL_SEQ;
  
    CURSOR C_GET_ST_SEQ IS
      SELECT SIM_TRANSLATIONS_SEQ.NEXTVAL FROM DUAL;
  
  BEGIN
    OPEN C_GET_CFG_DET_SEQ(P_IMP_CFG_SEQ, P_COL_SEQ);
    FETCH C_GET_CFG_DET_SEQ
      INTO V_SICDT_SEQ;
    IF C_GET_CFG_DET_SEQ%NOTFOUND THEN
      V_SICDT_SEQ := NULL;
      RAISE E_IMP_CFG_DETS;
    END IF;
    CLOSE C_GET_CFG_DET_SEQ;
    /* get the sequence for the new record in translations table */
    OPEN C_GET_ST_SEQ;
    FETCH C_GET_ST_SEQ
      INTO V_ST_SEQ;
    IF C_GET_ST_SEQ%NOTFOUND THEN
      V_ST_SEQ := NULL;
      RAISE E_TRANS_ERROR;
    END IF;
    CLOSE C_GET_ST_SEQ;
    INSERT INTO SIM_TRANSLATIONS
      (ST_SEQ, ST_SICDT_SEQ, ST_USER_CODE, ST_STORET_CODE)
    VALUES
      (V_ST_SEQ, /* st_seq */
       V_SICDT_SEQ, /* st_sicdt_seq */
       P_CODE, /* st_user_code */
       NULL /* st_storet_code */);
  
  EXCEPTION
    WHEN E_IMP_CFG_DETS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Unable to find import configuration - stopped import',
                                  NULL);
      RAISE_APPLICATION_ERROR(-20101,
                              'Unable to find import configuration');
    WHEN E_TRANS_ERROR THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Unable to create new record in translations table - error getting sequence',
                                  NULL);
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                   ' - adding record to translations table.',
                                  NULL);
  END CREATE_TRANS_RECORD;

  /******************************************************************************************************/
  FUNCTION CHECK_ALLOWABLE_VALUES(
           P_ITEM        IN VARCHAR2,
           P_IMP_CFG_SEQ IN NUMBER,
           P_COL_SEQ     IN NUMBER) 
           RETURN NUMBER IS

    V_SIA_SEQ NUMBER(10);
      
    CURSOR C_CHECK_ALLOWABLE_VAL(
           P_ITEM        IN VARCHAR2, 
           P_IMP_CFG_SEQ IN NUMBER, 
           P_COL_SEQ     IN NUMBER) IS
    SELECT SIA_SEQ
    FROM   SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS
    WHERE  SIA_SIC_SEQ = SIC_SEQ AND SICDT_SICC_SEQ = P_IMP_CFG_SEQ 
    AND    SICDT_SIC_SEQ = SIC_SEQ AND SIC_SEQ = P_COL_SEQ 
    AND    SIA_ITEM = P_ITEM;
  
  BEGIN

    -- try to find the value (P_ITEM) in the allowable values table
    OPEN C_CHECK_ALLOWABLE_VAL(P_ITEM, P_IMP_CFG_SEQ, P_COL_SEQ);
    FETCH C_CHECK_ALLOWABLE_VAL INTO V_SIA_SEQ;
    
    -- did we find the value in the allowable values table?
    IF C_CHECK_ALLOWABLE_VAL%NOTFOUND THEN
      -- if no
      -- setup to return NULL (calling routine is responsible for error messages)
      V_SIA_SEQ := NULL;
    END IF;
    CLOSE C_CHECK_ALLOWABLE_VAL;

    -- return the primary key of the value in SIM_IMP_ALLOW_VALUES
    RETURN V_SIA_SEQ;
    
  EXCEPTION
    
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM || ' - in check allowable values',
        NULL
      );
      RETURN NULL;
      
  END CHECK_ALLOWABLE_VALUES;
  
  /******************************************************************************************************/
  FUNCTION VALIDATE_SEQUENCE_NUMBER(V_SEQ_NUM        IN NUMBER,
                                    V_STA_SEQ        IN NUMBER,
                                    V_TYPE           IN VARCHAR,
                                    V_IMPORT_LOG_SEQ IN NUMBER,
                                    V_SEQ            IN NUMBER) RETURN NUMBER IS
  
    V_STL_SEQ_NUM NUMBER(4);
  
    CURSOR C_GET_SEQ_NUM IS
      SELECT STL_SEQUENCE_NUMBER
        FROM SIM_STATION_LOCATIONS
       WHERE STL_STA_SEQ = V_STA_SEQ 
       AND STL_SEQUENCE_NUMBER = V_SEQ_NUM
       AND STL_TYPE_CODE = V_TYPE;
  
  BEGIN
    OPEN C_GET_SEQ_NUM;
    FETCH C_GET_SEQ_NUM
      INTO V_STL_SEQ_NUM;
      
      --01/30/04 slp - If a sequence number of 0 is returned,
      --  Sampling and Boundary location types cannot have a sequence number of 0. 
      IF V_SEQ_NUM = 0 THEN
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                 'Line ' || V_LINE_CNT ||
                                 ': A Sequence Number of ' || V_SEQ_NUM ||
                                 ' cannot be used for a '|| V_TYPE || ' location.',
                                  V_SEQ);
      END IF;                                  
      
      IF C_GET_SEQ_NUM%FOUND THEN     
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                  ': Sequence Number ' || V_SEQ_NUM ||
                                  ' is already used for a '|| V_TYPE || ' location on this station.',
                                  V_SEQ);
                                                              
        --if a sequence number is found for a specific type, return an error
        V_STL_SEQ_NUM := 0;

      ELSE
        V_STL_SEQ_NUM := V_SEQ_NUM;
      END IF;
    
    CLOSE C_GET_SEQ_NUM;
    
    RETURN V_STL_SEQ_NUM;
    
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in validate sequence number.',
                                  NULL);
      RETURN 0;
  END VALIDATE_SEQUENCE_NUMBER;

  /******************************************************************************************************
    function to look in the translations and allowable values tables for items that are
    hard coded values in STORET
  ********************************************************************************************************/
  FUNCTION VALIDATE_ALVAL_ITEM(P_ITEM_NAME   IN VARCHAR2,
                               P_ORIG_ITEM   IN VARCHAR2,
                               P_IMP_CFG_SEQ IN NUMBER,
                               P_COL_SEQ     IN NUMBER,
                               P_SEQ         IN NUMBER) RETURN NUMBER IS
    V_CONVERTED_ITEM VARCHAR2(80);
    V_ITEM_SIA_SEQ   NUMBER(10);
    V_TRANS_EXIST    BOOLEAN;
    V_INSTR_CODE     VARCHAR2(15);
    
  BEGIN
    -- this function validates item values against the hard-coded allowable values in STORET.
    
    -- if translation exists, return translated; otherwise return original value
    V_CONVERTED_ITEM := CHECK_TRANSLATIONS(
      P_ORIG_ITEM,
      P_IMP_CFG_SEQ,
      P_COL_SEQ,
      V_TRANS_EXIST  -- OUT: if true then value returned is a translation
    );

    -- did the call to CHECK_TRANSLATIONS work properly?
    IF V_CONVERTED_ITEM IS NOT NULL THEN
      -- if yes
 
      -- jah 2-25-04 Convert to uppercase to mimic STORET GUI
      -- Does the STORET GUI convert this item to uppercase?
      IF P_ITEM_NAME IN ('Point Type') THEN
        -- if yes
        V_CONVERTED_ITEM := UPPER(V_CONVERTED_ITEM);
      END IF;

      -- Get the primary key of the value in SIM_IMP_ALLOWABLE_VALUES table (NULL = invalid value)
      V_ITEM_SIA_SEQ := CHECK_ALLOWABLE_VALUES(
        V_CONVERTED_ITEM,
        P_IMP_CFG_SEQ,
        P_COL_SEQ
      );

      -- is the value for this item invalid AND did CHECK_TRANSLATIONS return the original value?
      IF V_ITEM_SIA_SEQ IS NULL AND NOT V_TRANS_EXIST THEN
        -- if yes

        -- check instructions to see if translation record should be created      
        V_INSTR_CODE := GET_INSTRUCTIONS(P_IMP_CFG_SEQ, 'LOC-TR');

        -- should translation record be created?
        IF V_INSTR_CODE = 'LOC-TR-01' THEN
          -- if yes
          CREATE_TRANS_RECORD(
            P_ORIG_ITEM, 
            P_IMP_CFG_SEQ, 
            P_COL_SEQ
          );

          -- Tell the end-user what happened
          SIM2_IMPORT_PKG.WRITE_ERROR(
            V_IMPORT_LOG_SEQ,'Line ' || V_LINE_CNT || ': ' || P_ITEM_NAME || ', ' || V_CONVERTED_ITEM || 
            ' is not an allowable value - translation record has been created.',
            P_SEQ
          );
          
        ELSE

          -- if no (a translation record should not be created)
          SIM2_IMPORT_PKG.WRITE_ERROR(
            V_IMPORT_LOG_SEQ,
            'Line ' || V_LINE_CNT || ': ' ||P_ITEM_NAME || ', ' || V_CONVERTED_ITEM ||
            ' is not an allowable value.',
            P_SEQ
          );

        END IF;

      -- is the value for this item invalid AND did CHECK_TRANSLATIONS return a translated value?
      ELSIF (V_ITEM_SIA_SEQ IS NULL) AND (V_TRANS_EXIST) THEN
        -- if yes
        -- tell end-user the original item value translated to an invalid STORET value
        SIM2_IMPORT_PKG.WRITE_ERROR(
          V_IMPORT_LOG_SEQ,
          'Line ' || V_LINE_CNT || ': ' ||P_ITEM_NAME || ', ' || P_ORIG_ITEM ||
          ' has been mapped to the invalid storet value: ' || V_CONVERTED_ITEM,
          P_SEQ
        );
      END IF;
      
    END IF;
  
    RETURN V_ITEM_SIA_SEQ;
    
  EXCEPTION
    
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
        ' - in validate allowable values.',
        P_SEQ
      );
      RETURN NULL;
      
  END VALIDATE_ALVAL_ITEM;

  /******************************************************************************************************/
  FUNCTION GET_DATE(P_RAW_DATE IN VARCHAR2,
                    P_COL_SEQ  IN NUMBER,
                    P_CFG_SEQ  IN NUMBER,
                    P_LINE_CNT IN NUMBER,
                    P_SEQ      IN NUMBER) RETURN DATE IS
    V_START_DATE DATE;
    V_FMT        VARCHAR2(80);
  
    CURSOR C_GET_DATE_FMT(P_COL_SEQ IN NUMBER, P_CFG_SEQ IN NUMBER) IS
      SELECT SIFC_FMT_CHOICE
        FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
       WHERE SICDT_SIC_SEQ = P_COL_SEQ AND SICDT_SICC_SEQ = P_CFG_SEQ AND
             SICDT_SIFC_SEQ = SIFC_SEQ;
  BEGIN
    /* get format sequence from the configuration table */
    OPEN C_GET_DATE_FMT(P_COL_SEQ, P_CFG_SEQ);
    FETCH C_GET_DATE_FMT
      INTO V_FMT;
    IF C_GET_DATE_FMT%NOTFOUND THEN
      V_FMT := NULL;
      /* no format specified, use default */
      V_START_DATE := TRUNC(TO_DATE(P_RAW_DATE, 'MM-DD-RRRR'));
    END IF;
    CLOSE C_GET_DATE_FMT;
    IF V_FMT IS NOT NULL THEN
      V_START_DATE := TRUNC(TO_DATE(P_RAW_DATE, V_FMT));
    END IF;
    RETURN V_START_DATE;
  
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  ' Line: ' || P_LINE_CNT || ' - date, ' ||
                                   P_RAW_DATE ||
                                   ', format does not match given import format ',
                                  P_SEQ);
      RETURN NULL;
  END GET_DATE;
  /******************************************************************************************************/
  FUNCTION GET_TIME(P_RAW_TIME IN VARCHAR2,
                    P_COL_SEQ  IN NUMBER,
                    P_CFG_SEQ  IN NUMBER) RETURN DATE IS
    V_START_TIME DATE;
    V_FMT        VARCHAR2(80);
    E_NO_CFG_FOUND EXCEPTION;
    CURSOR C_GET_DATE_FMT(P_COL_SEQ IN NUMBER, P_CFG_SEQ IN NUMBER) IS
      SELECT SIFC_FMT_CHOICE
        FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
       WHERE SICDT_SIC_SEQ = P_COL_SEQ AND SICDT_SICC_SEQ = P_CFG_SEQ AND
             SICDT_SIFC_SEQ = SIFC_SEQ;
  BEGIN
    /* get format sequence from the configuration table */
    OPEN C_GET_DATE_FMT(P_COL_SEQ, P_CFG_SEQ);
    FETCH C_GET_DATE_FMT
      INTO V_FMT;
    IF C_GET_DATE_FMT%NOTFOUND THEN
      V_FMT := NULL;
      /* no format specified, use default */
      V_START_TIME := TO_DATE(SUBSTR(P_RAW_TIME, 1, 8), 'HH24:MI:SS');
    END IF;
    CLOSE C_GET_DATE_FMT;
    IF V_FMT IS NOT NULL THEN
      V_START_TIME := TO_DATE(SUBSTR(P_RAW_TIME, 1, 8), V_FMT);
    END IF;
    RETURN V_START_TIME;
  EXCEPTION
    WHEN E_NO_CFG_FOUND THEN
      RAISE_APPLICATION_ERROR(-20100,
                              'Import configuration for start or end time is incomplete or does not exist.');
      RETURN NULL;
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - in get time.',
                                  NULL);
      RETURN NULL;
  END GET_TIME;
  /************************************************************************************************************
/************************************************************************************************************
  The get_tsmgeopa_is_nbr function finds the station IS number and ORG id from  the Storet
  table TSMGEOPA The values are returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #
  This function accounts for the 4 possible combinations of state and county ID's,
      1 It's a state postal code and a county name
  2 It's a state FIPS and a county name
      3 It's a state postal code and  county FIPS
  4 It's a  state FIPS and county FIPS
    /************************************************************************************************************/
  FUNCTION GET_TSMGEOPA_IS_NBR(P_STATE IN VARCHAR2, P_COUNTY IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  V_STORET_TSMGEOPA T_STORET_PK_TYPE;
  /*** four cursors for the four possible combinations of state and county id's *****/
    CURSOR C_FIND_GEOPA_1(P_STATE IN VARCHAR2, P_COUNTY IN VARCHAR2) IS
      SELECT TSMGEOPA_IS_NUMBER, TSMGEOPA_ORG_ID
        FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(P_STATE, 2) AND
             COUNTY_NAME = P_COUNTY;
    CURSOR C_FIND_GEOPA_2(P_STATE IN VARCHAR2, P_COUNTY IN VARCHAR2) IS
      SELECT TSMGEOPA_IS_NUMBER, TSMGEOPA_ORG_ID
        FROM TSMGEOPA
       WHERE STATE_FIPS_CODE = RPAD(P_STATE, 2) AND COUNTY_NAME = P_COUNTY;
    CURSOR C_FIND_GEOPA_3(P_STATE IN VARCHAR2, P_COUNTY IN VARCHAR2) IS
      SELECT TSMGEOPA_IS_NUMBER, TSMGEOPA_ORG_ID
        FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(P_STATE, 2) AND
             FIPS_COUNTY_CODE = RPAD(P_COUNTY, 3);
    CURSOR C_FIND_GEOPA_4(P_STATE IN VARCHAR2, P_COUNTY IN VARCHAR2) IS
      SELECT TSMGEOPA_IS_NUMBER, TSMGEOPA_ORG_ID
        FROM TSMGEOPA
       WHERE STATE_FIPS_CODE = RPAD(P_STATE, 2) AND
             FIPS_COUNTY_CODE = RPAD(P_COUNTY, 3);
  BEGIN
    BEGIN
      IF TRANSLATE(P_STATE, 'a1234567890.- ', 'a') IS NOT NULL AND
         TRANSLATE(P_COUNTY, 'a1234567890.- ', 'a') IS NOT NULL THEN
        -- it's a state postal and a county NAME
  OPEN C_FIND_GEOPA_1(P_STATE, P_COUNTY);
  FETCH C_FIND_GEOPA_1
    INTO V_STORET_TSMGEOPA;
  IF C_FIND_GEOPA_1%NOTFOUND THEN
    V_STORET_TSMGEOPA.IS_NUMBER := NULL;
  END IF;
  CLOSE C_FIND_GEOPA_1;
ELSIF TRANSLATE(P_STATE, 'a1234567890.- ', 'a') IS NULL AND
      TRANSLATE(P_COUNTY, 'a1234567890.- ', 'a') IS NOT NULL THEN
  -- It's a state fips and a county name
        OPEN C_FIND_GEOPA_2(P_STATE, P_COUNTY);
        FETCH C_FIND_GEOPA_2
          INTO V_STORET_TSMGEOPA;
        IF C_FIND_GEOPA_2%NOTFOUND THEN
          V_STORET_TSMGEOPA.IS_NUMBER := NULL;
        END IF;
        CLOSE C_FIND_GEOPA_2;
      ELSIF TRANSLATE(P_STATE, 'a1234567890.- ', 'a') IS NOT NULL AND
            TRANSLATE(P_COUNTY, 'a1234567890.- ', 'a') IS NULL THEN
        -- it's a state postal and  county fips
  OPEN C_FIND_GEOPA_3(P_STATE, P_COUNTY);
  FETCH C_FIND_GEOPA_3
    INTO V_STORET_TSMGEOPA;
  IF C_FIND_GEOPA_3%NOTFOUND THEN
    V_STORET_TSMGEOPA.IS_NUMBER := NULL;
  END IF;
  CLOSE C_FIND_GEOPA_3;
ELSIF TRANSLATE(P_STATE, 'a1234567890.- ', 'a') IS NULL AND
      TRANSLATE(P_COUNTY, 'a1234567890.- ', 'a') IS NULL THEN
  -- it's a  state fips and county FIPS
        OPEN C_FIND_GEOPA_4(P_STATE, P_COUNTY);
        FETCH C_FIND_GEOPA_4
          INTO V_STORET_TSMGEOPA;
        IF C_FIND_GEOPA_4%NOTFOUND THEN
          V_STORET_TSMGEOPA.IS_NUMBER := NULL;
        END IF;
        CLOSE C_FIND_GEOPA_4;
      ELSE
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                    'get TSMgeopa IS COMBO NOT FOUND: ' ||
                                    V_STORET_TSMGEOPA.IS_NUMBER ||
                                    'p_state -> ' || P_STATE ||
                                    ' p_county -> ' || P_COUNTY,
                                    NULL);
      END IF;
    EXCEPTION
      WHEN OTHERS THEN
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                    SQLERRM || ' - find geopa',
                                    NULL);
    END;
    RETURN V_STORET_TSMGEOPA;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - PROC GET_geopa_IS_NBR ',
                                  NULL);
      V_STORET_TSMGEOPA.IS_NUMBER := NULL;
      V_STORET_TSMGEOPA.ORG_ID    := NULL;
      RETURN V_STORET_TSMGEOPA;
  END GET_TSMGEOPA_IS_NBR;
  /******************************************************************************************************
    The create_station_location function inserts known good data into the SIM table SIM_STATIONS
    After the station_location has loaded sucessfully the function returns a 1
    If there was a problem the function returns a zero.
  /************************************************************************************************************/
  FUNCTION CREATE_STATION_LOCATION(P_STATION_LOCATION_REC IN SIM_STATION_LOCATIONS%ROWTYPE)
    RETURN NUMBER IS
  BEGIN

    COMMIT; -- was getting deadlock without commit
  
    /* create station using values in station record */
    INSERT INTO SIM_STATION_LOCATIONS
      (STL_SEQ,
       STL_STA_SEQ,
       STL_SSW_SEQ,
       STL_POINT_NAME,
       STL_TYPE_CODE,
       STL_SEQUENCE_NUMBER,
       STL_ELEV_DATUM,
       STL_ELEV_METHOD,
       STL_ELEV_MSR_DATE,
       STL_ELEVATION_MSR,
       STL_ELEVATION_UNIT,
       STL_GEOPOS_DATUM,
       STL_GEOPOS_METHOD,
       STL_GPS_SCALE,
       STL_LAT_DIR,
       STL_LAT_DEGREES,
       STL_LAT_MINUTES,
       STL_LAT_SECONDS,
       STL_LONG_DIR,
       STL_LONG_DEGREES,
       STL_LONG_MINUTES,
       STL_LONG_SECONDS,
       STL_LAT_LONG_MSR_DATE,
       STL_LATLONG_ACCURCY,
       STL_LATLONG_ACCURCY_UNIT,
       STL_TSMRRR_IS_NUMBER,
       STL_TSMRRR_ORG_ID,
       STL_TSMFHU_IS_NUMBER,
       STL_TSMFHU_ORG_ID,
       STL_TSMSTATN0IS_NUMBER,
       STL_TSMSTATN0ORG_ID,
       STL_TSMWELL_IS_NUMBER,
       STL_TSMWELL_ORG_ID,
       STL_TSMGEOPA_IS_NUMBER,
       STL_TSMGEOPA_ORG_ID,
       STL_LAT_DEC_DEGREES, -- task 314
       STL_LAT_DEC_MINUTES,
       STL_LONG_DEC_DEGREES, -- task 314           
       STL_LONG_DEC_MINUTES,
       STL_TSMPIPE_IS_NUMBER,
       STL_TSMPIPE_ORG_ID,
       STL_TSMSTATN1ORG_ID,
       STL_TSMSTATN1IS_NUMBER,
       STL_TSMGEOPA0IS_NUMBER,
       STL_TSMGEOPA0ORG_ID,
       STL_TSMNAL_CD,
       STL_TSMNAL_STATE,
       STL_NRCS,
       STL_STATUS,
       STL_IMPORT_SEQ,
       STL_EXPORT_SEQ)
    VALUES
      (P_STATION_LOCATION_REC.STL_SEQ,
       P_STATION_LOCATION_REC.STL_STA_SEQ,
       P_STATION_LOCATION_REC.STL_SSW_SEQ,
       P_STATION_LOCATION_REC.STL_POINT_NAME,
       P_STATION_LOCATION_REC.STL_TYPE_CODE,
       P_STATION_LOCATION_REC.STL_SEQUENCE_NUMBER,
       P_STATION_LOCATION_REC.STL_ELEV_DATUM,
       P_STATION_LOCATION_REC.STL_ELEV_METHOD,
       P_STATION_LOCATION_REC.STL_ELEV_MSR_DATE,
       P_STATION_LOCATION_REC.STL_ELEVATION_MSR,
       P_STATION_LOCATION_REC.STL_ELEVATION_UNIT,
       P_STATION_LOCATION_REC.STL_GEOPOS_DATUM,
       P_STATION_LOCATION_REC.STL_GEOPOS_METHOD,
       P_STATION_LOCATION_REC.STL_GPS_SCALE,
       P_STATION_LOCATION_REC.STL_LAT_DIR,
       P_STATION_LOCATION_REC.STL_LAT_DEGREES,
       P_STATION_LOCATION_REC.STL_LAT_MINUTES,
       P_STATION_LOCATION_REC.STL_LAT_SECONDS,
       P_STATION_LOCATION_REC.STL_LONG_DIR,
       P_STATION_LOCATION_REC.STL_LONG_DEGREES,
       P_STATION_LOCATION_REC.STL_LONG_MINUTES,
       P_STATION_LOCATION_REC.STL_LONG_SECONDS,
       P_STATION_LOCATION_REC.STL_LAT_LONG_MSR_DATE,
       P_STATION_LOCATION_REC.STL_LATLONG_ACCURCY,
       P_STATION_LOCATION_REC.STL_LATLONG_ACCURCY_UNIT,
       P_STATION_LOCATION_REC.STL_TSMRRR_IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMRRR_ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMFHU_IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMFHU_ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMSTATN0IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMSTATN0ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMWELL_IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMWELL_ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMGEOPA_IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMGEOPA_ORG_ID,
       P_STATION_LOCATION_REC.STL_LAT_DEC_DEGREES, -- task 314
       P_STATION_LOCATION_REC.STL_LAT_DEC_MINUTES,
       P_STATION_LOCATION_REC.STL_LONG_DEC_DEGREES, -- task 314           
       P_STATION_LOCATION_REC.STL_LONG_DEC_MINUTES,
       P_STATION_LOCATION_REC.STL_TSMPIPE_IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMPIPE_ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMSTATN1ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMSTATN1IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMGEOPA0IS_NUMBER,
       P_STATION_LOCATION_REC.STL_TSMGEOPA0ORG_ID,
       P_STATION_LOCATION_REC.STL_TSMNAL_CD,
       P_STATION_LOCATION_REC.STL_TSMNAL_STATE,
       P_STATION_LOCATION_REC.STL_NRCS,
       P_STATION_LOCATION_REC.STL_STATUS,
       P_STATION_LOCATION_REC.STL_IMPORT_SEQ,
       P_STATION_LOCATION_REC.STL_EXPORT_SEQ);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                   ' - inserting station locations.',
                                  NULL);
      RETURN 0;
  END CREATE_STATION_LOCATION;

  /******************************************************************************************************/

  FUNCTION CHECK_TSMMAD_VALUE(P_ORIG_ITEM   IN VARCHAR2,
                              P_ITEM_NAME   IN VARCHAR2,
                              P_CATEGORY    IN VARCHAR2,
                              P_SUBCATEGORY IN VARCHAR2,
                              P_COL_SEQ     IN NUMBER,
                              P_STA_SEQ     IN NUMBER,
                              P_IMP_CFG_SEQ IN NUMBER,
                              P_IMP_LOG_SEQ IN NUMBER) RETURN VARCHAR2 IS
  
    -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.
  
    V_CONVERTED_ITEM VARCHAR2(120);
    V_TRANS_EXIST    BOOLEAN := TRUE;
    V_STORET_VALUE   VARCHAR2(12);
    V_INSTR_ID       VARCHAR2(8);
    V_INSTR_CODE     VARCHAR2(15);
    v_storet_pk      Sim_import_locations_pkg.t_storet_pk_type;  
      
    CURSOR C_CHECK_TSMMAD(P_ITEM IN VARCHAR2) IS
      SELECT ID_CODE
        FROM TSMMAD
       WHERE CATEGORY = RPAD(P_CATEGORY, 10) AND
             SUBCATEGORY = RPAD(P_SUBCATEGORY, 6) AND
             ID_CODE = RPAD(P_ITEM, 12);
  
  BEGIN
  
    V_STORET_PK := SIM_IMPORT_LOCATIONS_PKG.CHECK_STORET_TRANSLATION(P_ORIG_ITEM,
                                          P_IMP_CFG_SEQ,
                                          P_COL_SEQ,
                                          V_TRANS_EXIST,                                  
                                          V_CONVERTED_ITEM,
                                          P_STA_SEQ); 
                                          
    v_storet_value := v_converted_item; 
    IF v_storet_pk.is_number IS NULL AND NOT v_trans_exist THEN
      OPEN C_CHECK_TSMMAD(V_CONVERTED_ITEM);
      FETCH C_CHECK_TSMMAD
        INTO V_STORET_VALUE;
      IF C_CHECK_TSMMAD %NOTFOUND THEN
        V_STORET_VALUE := NULL;
      END IF;
      CLOSE C_CHECK_TSMMAD;
    
      IF V_STORET_VALUE IS NULL AND NOT V_TRANS_EXIST THEN
        V_INSTR_ID := 'LOC-TR';
        /* check instructions to see if translation record should be created */
        V_INSTR_CODE := GET_INSTRUCTIONS(P_IMP_CFG_SEQ, V_INSTR_ID);
        IF V_INSTR_CODE = 'LOC-TR-01' THEN
          /* if record could not be matched to anything in translations or STORET tables,
          create new record */
          CREATE_TRANS_RECORD(P_ORIG_ITEM, P_IMP_CFG_SEQ, P_COL_SEQ);
          SIM2_IMPORT_PKG.WRITE_ERROR(P_IMP_LOG_SEQ,
                                      'Line ' || V_LINE_CNT || ': ' ||
                                       P_ITEM_NAME || ', ' ||
                                       V_CONVERTED_ITEM ||
                                       ' is not a valid value - translation record has been created.',
                                      P_STA_SEQ);
        ELSE
          SIM2_IMPORT_PKG.WRITE_ERROR(P_IMP_LOG_SEQ,
                                      'Line ' || V_LINE_CNT || ': ' ||
                                       P_ITEM_NAME || ', ' ||
                                       V_CONVERTED_ITEM ||
                                       ' is not a valid value.',
                                      P_STA_SEQ);
        END IF;
      ELSIF V_STORET_VALUE IS NULL AND V_TRANS_EXIST THEN
        /*  item has been translated to an invalid storet value */
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMP_LOG_SEQ,
                                    P_ITEM_NAME || ', ' || P_ORIG_ITEM ||
                                    ' has been mapped to the invalid storet value: ' ||
                                    V_CONVERTED_ITEM,
                                    P_STA_SEQ);
      END IF;
    ELSIF v_storet_pk.is_number IS NULL AND V_TRANS_EXIST THEN
      /* item exists in translations table but is not mapped to anything, write error message.
      Status will be changed to 'N' later in the program. */
      V_STORET_VALUE := NULL;
    END IF;
  
    RETURN V_STORET_VALUE;
  
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMP_LOG_SEQ,
                                  SQLERRM || ' - checking STORET TSMMAD.',
                                  P_STA_SEQ);
      RETURN NULL;
  END CHECK_TSMMAD_VALUE;

  /******************************************************************************************************
    The import_locations procedure parses a text file line by line of data that is separated by
    a delimiter.  The procedure is activated by filling out and activating the import locations dialog box 
    in the SIM2 application. From there we will assign an ORG ID and procede with opening the text file to 
    extract the data.
  /******************************************************************************************************/
  PROCEDURE IMPORT_LOCATIONS(P_ORG_ID         IN VARCHAR2,
                           P_IMPORT_LOG_SEQ IN NUMBER,
                           P_IMP_CFG_SEQ    IN NUMBER,
                           P_TEST_COUNT     IN NUMBER) IS

    V_IMPORT_DESC    VARCHAR2(200);
    V_COMP           NUMBER(1);
    V_NUM_COLS       NUMBER(3);
    V_COL_SEQ        NUMBER(10);
    V_DFLT_POS       NUMBER(3);
    V_SUCCESS        NUMBER(1);
    V_CONVERTED_ITEM VARCHAR2(256);
    V_DELIMITER      CHAR(1);
    V_INLINE         VARCHAR2(1000);
    V_BAD_PARSE      BOOLEAN;
    V_ITEM_NAME      VARCHAR2(80);
    V_ITEM           VARCHAR2(256);
    V_TRANS_EXIST    BOOLEAN := TRUE;
    V_LINE_TABLE     SIM2_IMPORT_PKG.COL_TABLE_TYPE;
    V_DEFAULT_TABLE  SIM_IMPORT_LOCATIONS_PKG.COL_TABLE_TYPE;
    V_POS_TABLE      SIM_IMPORT_LOCATIONS_PKG.COL_TABLE_NUM_TYPE;
    V_ERROR_CNT      NUMBER(10);
    V_COMPLETE_CNT   NUMBER(10);
    V_DUMMY_SEQ      NUMBER(10);
    V_POINT_NAME     VARCHAR2(30);
    V_STA_SEQ        NUMBER(10);
    V_ID             VARCHAR2(15);
    V_STL_REC SIM_STATION_LOCATIONS%ROWTYPE;
    V_ELEVATION_UNIT      VARCHAR2(3);
    V_ELEVATION_METHOD    VARCHAR2(12);
    V_ELEVATION_DATUM     VARCHAR2(12);
    V_ELEVATION_DATE      DATE;
    V_TSMORGAN_IS_NUMBER  VARCHAR2(8);
    V_STATION_IS_NBR      NUMBER(12);
    V_ELEVATION_MSR       NUMBER(9, 4);
    V_LAT_DIR_SEQ         NUMBER(10);
    V_LONG_DIR_SEQ        NUMBER(10);
    V_LAT_LONG_DATE       DATE;
    V_LAT_DEGREES         NUMBER(2);
    V_LAT_MINUTES         NUMBER(2);
    V_LAT_SECONDS         NUMBER(6, 4);
    V_LAT_DIR             VARCHAR2(1);
    V_LAT_DIR_CHECK       VARCHAR2(1);
    V_LAT_DEC_MIN         NUMBER(6, 4);
    V_LAT_DEC_DEG         NUMBER(9, 7);     
    V_LONG_DEGREES        NUMBER(3);
    V_LONG_MINUTES        NUMBER(2);
    V_LONG_SECONDS        NUMBER(6, 4);
    V_LONG_DEC_MIN        NUMBER(8, 4);
    V_LONG_DEC_DEG        NUMBER(10, 7);
    V_LONG_DIR            VARCHAR2(1);
    V_LONG_DIR_CHECK      VARCHAR2(1);
    V_METHOD              VARCHAR2(12);
    V_DATUM               VARCHAR2(12);
    V_GPS_SCALE           VARCHAR2(20);
    V_SEQUENCE_NUMBER     NUMBER(4);
    V_LOCATION_SEQ        NUMBER(10);
    V_POINT_TYPE          VARCHAR2(16);
    V_NEXT                NUMBER(2) := 0;
    V_STATION_ID          SIM_STATIONS.STA_ID%TYPE;
    v_storet_key          SIM_IMPORT_LOCATIONS_PKG.T_STORET_PK_TYPE;     
  
    E_BAD_ORG EXCEPTION;
    E_NO_CFG EXCEPTION;
    E_NO_DELIMITER EXCEPTION;
    E_BAD_COLUMNS EXCEPTION;
    E_NO_STATION EXCEPTION;
  
    CURSOR C_GET_LOCATION_SEQ IS
      SELECT SIM_STATION_LOCATIONS_SEQ.NEXTVAL FROM DUAL;
  
    CURSOR C_GET_NEXT_LINE(P_IMPORT_SEQ IN NUMBER) IS
      SELECT SIPL_SEQ, SIPL_LINE_TEXT
        FROM SIM_IMP_LINES
       WHERE SIPL_SIL_SEQ = P_IMPORT_SEQ AND SIPL_LINE_TEXT IS NOT NULL -- bka  added to eliminate blank lines from the import
       ORDER BY SIPL_LINE_NO;
  
    CURSOR C_COLUMN_LOOP(P_IMP_CFG_SEQ IN NUMBER) IS
      SELECT SIC_PROC_ORDER, SIC_DFLT_POS, SIC_NAME, SIC_SEQ
        FROM SIM_IMP_COLS, SIM_IMP_DEFS, SIM_IMP_CFG
       WHERE SICC_SID_SEQ = SID_SEQ AND SIC_SID_SEQ = SID_SEQ AND
             SICC_SEQ = P_IMP_CFG_SEQ
       ORDER BY SIC_PROC_ORDER;
  
    CURSOR C_COMP_COUNT(P_IMPORT_LOG_SEQ IN NUMBER) IS
      SELECT COUNT(STL_SEQ)
        FROM SIM_STATION_LOCATIONS
       WHERE STL_IMPORT_SEQ = P_IMPORT_LOG_SEQ AND STL_STATUS = 'A';
  
  BEGIN
    /* GET sequence for import log */
    V_IMPORT_LOG_SEQ := P_IMPORT_LOG_SEQ;
    V_LINE_CNT       := 0;
  
    BEGIN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,'Start at ' ||TO_CHAR(SYSDATE, 'HH24:MI'),NULL);
      /* get tsmorgan_is_number */
      V_SUCCESS := GET_ORG_INFO(P_IMP_CFG_SEQ,P_ORG_ID,V_TSMORGAN_IS_NUMBER);
      IF V_SUCCESS != 1 OR V_ORG_IS_NBR IS NULL THEN
        RAISE E_BAD_ORG;
      END IF;
    
      /* get the delimiter */
      V_DELIMITER := NULL;
      V_SUCCESS   := GET_DELIMITER(P_IMP_CFG_SEQ, V_DELIMITER);
      IF V_SUCCESS != 1 THEN
        RAISE E_NO_DELIMITER;
      END IF;
    
      /* get actual positions and default information for items being imported */
      V_SUCCESS := GET_ITEM_POS(P_IMP_CFG_SEQ,V_POS_TABLE,V_DEFAULT_TABLE,V_NUM_COLS);
      IF V_SUCCESS != 1 THEN
        RAISE E_NO_CFG;
      END IF;
    
      /* loop through the lines in the text table */
      FOR V_TEXT_LINE IN C_GET_NEXT_LINE(V_IMPORT_LOG_SEQ) LOOP
      /* read one line */
    
      BEGIN
      
        /* get location sequence */
        OPEN C_GET_LOCATION_SEQ;
        FETCH C_GET_LOCATION_SEQ
          INTO V_LOCATION_SEQ;
        CLOSE C_GET_LOCATION_SEQ;
      
        V_INLINE   := V_TEXT_LINE.SIPL_LINE_TEXT;
        V_LINE_CNT := V_LINE_CNT + 1;
        V_STATUS   := 'A';
      
        -- send line to parsing routine:
        SIM2_IMPORT_PKG.PARSE_LINE_TABLE(V_INLINE,V_LINE_TABLE,V_NUM_COLS,V_DELIMITER,V_BAD_PARSE,V_IMPORT_LOG_SEQ);
        IF V_BAD_PARSE THEN
          RAISE E_BAD_COLUMNS;
        END IF;
      
        V_NEXT := 0;
      
        /* loop through all STATION columns */
        FOR V_COLUMNS IN C_COLUMN_LOOP(P_IMP_CFG_SEQ) LOOP
      
          /* get dflt position, item name, and column sequence */
          V_DFLT_POS  := V_COLUMNS.SIC_DFLT_POS;
          V_ITEM_NAME := V_COLUMNS.SIC_NAME;
          V_COL_SEQ   := V_COLUMNS.SIC_SEQ;
          V_NEXT      := V_NEXT + 1;
        
          /* if column is included on the import, continue. Don't look at positions <= 0 */
          IF V_DFLT_POS > 0 THEN
            IF V_POS_TABLE(V_DFLT_POS) IS NOT NULL THEN
            
              /* get the actual item, or use default if null */
              IF V_LINE_TABLE(V_POS_TABLE(V_DFLT_POS)) IS NOT NULL THEN
                V_ITEM := V_LINE_TABLE(V_POS_TABLE(V_DFLT_POS));
              ELSIF V_LINE_TABLE(V_POS_TABLE(V_DFLT_POS)) IS NULL AND
                    V_DEFAULT_TABLE(V_DFLT_POS) IS NOT NULL THEN
                V_ITEM := V_DEFAULT_TABLE(V_DFLT_POS);
              ELSE
                V_ITEM := NULL;
              END IF;
                      
              -- Does V_ITEM contain only spaces? (or NULL)
              IF (TRIM(V_ITEM) IS NULL) THEN
                -- if yes
                
                -- by definition V_ITEM is NULL!
                V_ITEM := NULL;
    
              END IF;
    
              /* assign item to appropriate variables and take care of any other procedural issues that need to be
              taken care of for or with a particular item */
            
              -------------
              -- Station ID
              -------------
              IF V_ITEM_NAME = 'Station ID' THEN
              
                IF V_ITEM IS NOT NULL THEN
                
                  -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                  V_ITEM := RTRIM(V_ITEM);
                  
                  -- Translate
                  v_storet_key := check_storet_translation(
                    v_item, 
                    p_imp_cfg_seq, 
                    v_col_seq, 
                    V_TRANS_EXIST,     -- OUT
                    V_CONVERTED_ITEM,  -- OUT
                    v_sta_seq
                  );

                    -- Check if station exists in SIM
                  V_COMP := CHECK_IF_STATION_EXISTS(
                    UPPER(v_converted_item),
                    P_ORG_ID,
                    V_STA_SEQ,
                    V_STATION_IS_NBR
                  );

                  -- Does the station exist in SIM?                  
                  IF V_COMP = 1 THEN
                    -- if yes

                    -- jah 3-18-2004 changed from v_item to v_converted_item
                    V_ID := V_CONVERTED_ITEM;
                    
                  ELSE
                    -- if no (i.e., the station does not exist in SIM)
                  
                    -- Prevent migration of this record                    
                    V_STATUS := 'N';
                    V_ID     := NULL;

                    -- jah 3-18-2004
                    -- Do we need to create a translation record?
                    IF (v_storet_key.is_number IS NULL) 
                      AND (NOT V_TRANS_EXIST) 
                      AND (GET_INSTRUCTIONS(p_imp_cfg_seq, 'LOC-TR') = 'LOC-TR-01') THEN
                      -- if yes

                      -- Create the translation record
                      CREATE_TRANS_RECORD(
                        v_item, 
                        P_IMP_CFG_SEQ, 
                        V_COL_SEQ
                      );

                      SIM2_IMPORT_PKG.WRITE_ERROR(
                        V_IMPORT_LOG_SEQ,
                        'Line ' || V_LINE_CNT || ': ' ||
                        V_ITEM_NAME || ' ' ||
                        V_ITEM ||
                        ' is not a valid - translation record has been created.',
                        V_STA_SEQ
                      );
                      
                    END IF;

                  END IF;
                  
                ELSE
                  -- if no (i.e., the station ID field does NOT contain anything)
                  
                  V_ID := NULL;
                  
                END IF;
                
                IF V_ITEM IS NOT NULL THEN
                  V_STATION_ID := UPPER(RTRIM(V_ITEM));
                END IF;
              
              -------------
              -- Point Name
              -------------  
              ELSIF V_ITEM_NAME = 'Point Name' THEN
              
                IF V_ITEM IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF LENGTH(V_ITEM) > 30 THEN
                      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,'Line: ' || V_LINE_CNT || ': ' || V_ITEM_NAME || ' is longer than 30 characters and will be truncated.',V_LOCATION_SEQ);
                      V_ITEM := SUBSTR(V_ITEM, 1, 30);
                    END IF;
                    V_POINT_NAME := V_ITEM;
                  ELSE
                    V_STATUS := 'N';
                  END IF;
                ELSE
                  V_POINT_NAME := NULL;
                END IF;
              
              -------------
              -- Point Type
              -------------
              ELSIF V_ITEM_NAME = 'Point Type' THEN

                -- Does the Point Type field of this record contain anything?
                IF V_ITEM IS NOT NULL THEN
                  -- if yes
                
                  -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                  V_ITEM      := RTRIM(V_ITEM);

                  -- Try to find the value in the allowable values table
                  V_DUMMY_SEQ := VALIDATE_ALVAL_ITEM(
                    V_ITEM_NAME,
                    V_ITEM,
                    P_IMP_CFG_SEQ,
                    V_COL_SEQ,
                    V_LOCATION_SEQ
                  );

                  -- did we find the value in the allowable values table?
                  IF (V_DUMMY_SEQ IS NOT NULL) THEN
                    -- if yes

                    -- set V_POINT_TYPE using the value from SIM_IMP_ALLOW_VALUES
                    V_POINT_TYPE := GET_ALLOWABLE_VALUE(
                      V_DUMMY_SEQ,
                      V_LOCATION_SEQ
                    );
                    
                    -- jah 2-24-04 Convert to uppercase because STORET GUI converts Point Type to uppercase
                    V_POINT_TYPE := UPPER(V_POINT_TYPE);
                    
                  ELSE
                    -- if no (we did NOT find the value in the allowable values table)
                    
                    -- prevent migration of this record to STORET
                    V_POINT_TYPE := NULL;
                    V_STATUS     := 'N';
                    
                    -- VALIDATE_ALVAL_ITEM has already display an error message
                  
                  END IF;
                  
                ELSE
                  -- if no (the Point Type field of this record does not contain anything)
                  
                  -- Prevent migration of this record to STORET  
                  V_POINT_TYPE := NULL;
                  V_STATUS     := 'N';
                  SIM2_IMPORT_PKG.WRITE_ERROR(
                    V_IMPORT_LOG_SEQ,
                    'Line ' || V_LINE_CNT || ': Point Type is required',
                    V_LOCATION_SEQ
                  );
                  
                END IF;

              ------------------
              -- Sequence Number
              ------------------              
              ELSIF V_ITEM_NAME = 'Sequence Number' THEN
                IF V_ITEM IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF V_ITEM IS NOT NULL THEN
                    V_SUCCESS := VALIDATE_NUMBER(V_ITEM,V_ITEM_NAME,'9999',V_LOCATION_SEQ);
                    IF V_SUCCESS = 1 THEN
                      V_SEQUENCE_NUMBER := VALIDATE_SEQUENCE_NUMBER(V_ITEM,V_STA_SEQ,V_POINT_TYPE,V_IMPORT_LOG_SEQ,V_LOCATION_SEQ);
                      IF V_SEQUENCE_NUMBER = 0 THEN
                        V_STATUS := 'N';
                      END IF;
                    END IF;
                  ELSE
                    V_STATUS := 'N';
                  END IF;
                END IF;

              -----------  
              -- Latitude
              ----------- 
              ELSIF V_ITEM_NAME = 'Latitude' THEN
                V_LAT_DIR := NULL;
                IF V_ITEM IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF V_ITEM IS NOT NULL THEN
                    v_success := get_latitude(v_item, v_col_seq,p_imp_cfg_seq, v_location_seq, v_line_cnt, v_lat_degrees,
                                              v_lat_minutes, v_lat_seconds, v_lat_dir, v_lat_dec_min, v_lat_dec_deg);
                    IF v_success = 2 THEN
                      v_status := 'N';
                      v_lat_dec_deg := NULL;
                      v_lat_dec_min := NULL;
                      v_lat_degrees := NULL;
                      v_lat_minutes := NULL;
                      v_lat_seconds := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;                  
                ELSE
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Latitude is required.',v_location_seq);  
                  v_status := 'N';
                  v_lat_dec_deg := NULL;
                  v_lat_dec_min := NULL;
                  v_lat_degrees := NULL;
                  v_lat_minutes := NULL;
                  v_lat_seconds := NULL;
                END IF;  
              
              ---------------------           
              -- Latitude Direction
              ---------------------     
              ELSIF v_item_name = 'Latitude Direction' THEN
                IF v_item IS NOT NULL THEN
                  v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
                  v_lat_dir_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_location_seq);
                  IF v_lat_dir_seq IS NOT NULL THEN
                    IF v_lat_dir IS NULL THEN
                      v_lat_dir := get_allowable_value(v_lat_dir_seq,v_location_seq);
                    ELSE  
                      v_lat_dir_check := get_allowable_value(v_lat_dir_seq,v_location_seq);
                      IF v_lat_dir_check <> v_lat_dir THEN
                        v_lat_dir := 'X'; -- so we don't default it
                        v_status := 'N';
                        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Given Latitude Direction is inconsistent with sign on Latitude.',v_location_seq);
                      END IF;
                    END IF;   
                      
                  ELSE
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item||' is not an allowable Latitude Direction.',v_location_seq);
                    v_status := 'N';
                    v_lat_dir := 'X'; -- so we don't default it
                  END IF;
                ELSE
                  IF v_lat_Dir IS NULL THEN
                    v_lat_dir := 'N';
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Latitude Direction is required and will be set to N',v_location_seq);
                  END IF;
                END IF;
      
              ------------
              -- Longitude
              ------------
              ELSIF V_ITEM_NAME = 'Longitude' THEN
                v_long_dir := NULL;
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := get_longitude(v_item, v_col_seq, p_imp_cfg_seq, v_location_seq, v_line_cnt, v_long_degrees, v_long_minutes,
                                      v_long_seconds, v_long_dec_min, v_long_dec_deg, v_long_dir);
                    IF v_success = 2 THEN
                      v_status := 'N';
                      v_long_dec_deg := NULL;
                      v_long_dec_min := NULL;
                      v_long_degrees := NULL;
                      v_long_minutes := NULL;
                      v_long_seconds := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Longitude is required.',v_location_seq);
                  v_status := 'N';
                  v_long_dec_deg := NULL;
                  v_long_dec_min := NULL;
                  v_long_degrees := NULL;
                  v_long_minutes := NULL;
                  v_long_seconds := NULL;
                END IF;
                  
              ----------------------
              -- Longitude Direction
              ----------------------
              ELSIF V_ITEM_NAME = 'Longitude Direction' THEN
                IF v_item IS NOT NULL THEN
                  v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
			            v_long_dir_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_location_seq);
                  IF v_long_dir_seq IS NOT NULL THEN
                    IF v_long_dir IS NULL THEN
                      v_long_dir := get_allowable_value(v_long_dir_seq,v_location_seq);
                    ELSE
                      v_long_dir_check := get_allowable_value(v_long_dir_seq,v_location_seq);
                      IF v_long_dir_check <> v_long_dir THEN
                        v_long_dir := 'X'; -- so we don't default it
                        v_status := 'N';
                        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Given Longitude Direction is inconsistent with sign on Longitude.',v_location_seq);
                      END IF;
                    END IF;                  
                  ELSE
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item||' is not an allowable Longitude Direction.',v_location_seq);
                    v_status := 'N';
                    v_long_dir := 'X';  --so we don't default it in this case
                  END IF;

                  -- v_item IS NULL, default to W
                  IF v_long_dir IS NULL THEN
                    v_long_dir := 'W';
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Longitude Direction is required and will be set to W',v_location_seq);
                  END IF;
                END IF;           
                
              ----------------------------
              -- Lat/Long Measurement Date
              ----------------------------
              ELSIF V_ITEM_NAME = 'Lat/Long Measurement Date' THEN
                IF V_ITEM IS NULL THEN
                  V_LAT_LONG_DATE := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF V_ITEM IS NOT NULL THEN
                    V_LAT_LONG_DATE := GET_DATE(V_ITEM,
                                                V_COL_SEQ,
                                                P_IMP_CFG_SEQ,
                                                V_LINE_CNT,
                                                V_LOCATION_SEQ);
                    IF V_LAT_LONG_DATE IS NULL THEN
                      --the date format is bad
                      V_STATUS := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                                  'Line: ' || V_LINE_CNT ||
                                                   ': - ' || V_ITEM_NAME ||
                                                   ' format does not match format in import configuration',
                                                  V_LOCATION_SEQ);
                    END IF;
              		ELSE
              		  V_STATUS := 'N';
              		END IF;
                END IF;
                
              ------------------------
              -- Geopositioning Method
              ------------------------
              ELSIF V_ITEM_NAME = 'Geopositioning Method' THEN
                IF V_ITEM IS NOT NULL THEN
                  V_ITEM   := RTRIM(V_ITEM); -- 061203 slp - Remove trailing spaces.
                  V_METHOD := CHECK_TSMMAD_VALUE(V_ITEM,
                                                 V_ITEM_NAME,
                                                 'HORIZONTAL',
                                                 'METHOD',
                                                 V_COL_SEQ,
                                                 V_LOCATION_SEQ,
                                                 P_IMP_CFG_SEQ,
                                                 V_IMPORT_LOG_SEQ);
                  IF V_METHOD IS NULL THEN
                    V_STATUS := 'N';
                  END IF;
                ELSE
                  V_METHOD := NULL;
                  V_STATUS := 'N';                  
                  SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                              'Line ' || V_LINE_CNT ||
                                               ': Geopositioning method is required.',
                                              V_LOCATION_SEQ);
                END IF;
              
              -----------------------
              -- Geopositioning Datum
              -----------------------
              ELSIF V_ITEM_NAME = 'Geopositioning Datum' THEN
                IF V_ITEM IS NOT NULL THEN
                  V_ITEM  := RTRIM(V_ITEM); -- 061203 slp - Remove trailing spaces.
                  V_DATUM := CHECK_TSMMAD_VALUE(V_ITEM,
                                                V_ITEM_NAME,
                                                'HORIZONTAL',
                                                'DATUM',
                                                V_COL_SEQ,
                                                V_LOCATION_SEQ,
                                                P_IMP_CFG_SEQ,
                                                V_IMPORT_LOG_SEQ);
                
                  IF V_DATUM IS NULL THEN
                    V_STATUS := 'N';
                  END IF;
                ELSE
                  V_DATUM := NULL;
                  V_STATUS := 'N';                  
                  SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                              'Line ' || V_LINE_CNT ||
                                               ': Geopositioning Datum is required.',
                                              V_LOCATION_SEQ);
                END IF;
                
              --------
              -- Scale
              --------
              ELSIF V_ITEM_NAME = 'Scale' THEN
                IF V_ITEM IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF V_ITEM IS NOT NULL THEN
                    IF LENGTH(V_ITEM) > 20 THEN
                      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                                  'Line: ' || V_LINE_CNT || ': ' ||
                                                   V_ITEM_NAME ||
                                                   ' is longer than 20 characters and will be truncated.',
                                                  V_LOCATION_SEQ);
                      V_ITEM := SUBSTR(V_ITEM, 1, 20);
                    END IF;
                    V_GPS_SCALE := V_ITEM;
              		ELSE
              		  V_STATUS := 'N';
              		END IF;
                ELSE
                  V_GPS_SCALE := NULL;
                END IF;

              ------------
              -- Elevation
              ------------
              ELSIF V_ITEM_NAME = 'Elevation' THEN
                IF V_ITEM IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF V_ITEM IS NOT NULL THEN
                    V_SUCCESS := VALIDATE_NUMBER(V_ITEM,
                                                 V_ITEM_NAME,
                                                 '99999.9999',
                                                 V_LOCATION_SEQ);
                    IF V_SUCCESS = 0 THEN
                      V_ELEVATION_MSR := NULL;
                      V_STATUS        := 'N';
                      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                                  'Line ' || V_LINE_CNT ||
                                                   ': Elevation value ' ||
                                                   V_ITEM ||
                                                   ' is not allowable. It must be a number.',
                                                  V_LOCATION_SEQ);
                    ELSE
                      V_ELEVATION_MSR := V_ITEM;
                    END IF;
              		ELSE
              		  V_STATUS := 'N';
              		END IF;
                ELSE
                  V_ELEVATION_MSR := NULL;
                END IF;
              
              ------------------
              -- Elevation Units
              ------------------
              ELSIF V_ITEM_NAME = 'Elevation Units' THEN
                IF V_ITEM IS NOT NULL THEN
                  V_ITEM      := RTRIM(V_ITEM); -- 061203 slp - Remove trailing spaces.
                  V_DUMMY_SEQ := VALIDATE_ALVAL_ITEM(V_ITEM_NAME,
                                                     V_ITEM,
                                                     P_IMP_CFG_SEQ,
                                                     V_COL_SEQ,
                                                     V_LOCATION_SEQ);
                  IF V_ELEVATION_MSR IS NOT NULL THEN
                    IF V_DUMMY_SEQ IS NULL THEN
                      V_ELEVATION_UNIT := NULL;
                      V_STATUS         := 'N';
                      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                                  'Line ' || V_LINE_CNT ||
                                                   ': Elevation Unit, ' ||
                                                   V_ITEM ||
                                                   ', is not valid.',
                                                  V_LOCATION_SEQ);
                    ELSE
                      V_ELEVATION_UNIT := GET_ALLOWABLE_VALUE(V_DUMMY_SEQ,
                                                              V_LOCATION_SEQ);
                    END IF;
                  ELSE
                    V_ELEVATION_UNIT := NULL;
                    SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                                'Warning: Line ' ||
                                                 V_LINE_CNT ||
                                                 ': Elevation Unit is not needed since Elevation is not provided.',
                                                V_LOCATION_SEQ);
                  END IF;
                ELSE
                  V_ELEVATION_UNIT := NULL;
                END IF;
              
              -------------------
              -- Elevation Method
              -------------------
              ELSIF V_ITEM_NAME = 'Elevation Method' THEN
                --sfp, this section rewritten 062701
                IF V_ITEM IS NOT NULL THEN
                  V_ITEM             := RTRIM(V_ITEM); -- 061203 slp - Remove trailing spaces.
                  V_ELEVATION_METHOD := CHECK_TSMMAD_VALUE(V_ITEM,
                                                           V_ITEM_NAME,
                                                           'VERTICAL',
                                                           'METHOD',
                                                           V_COL_SEQ,
                                                           V_LOCATION_SEQ,
                                                           P_IMP_CFG_SEQ,
                                                           V_IMPORT_LOG_SEQ);
                  IF V_ELEVATION_METHOD IS NULL THEN
                    V_STATUS := 'N';
                  END IF;
                ELSE
                  V_ELEVATION_METHOD := NULL;
                END IF;
              
              ------------------
              -- Elevation Datum
              ------------------
              ELSIF V_ITEM_NAME = 'Elevation Datum' THEN
                --sfp, this section rewritten 062701
                IF V_ITEM IS NOT NULL THEN
                  V_ITEM            := RTRIM(V_ITEM); -- 061203 slp - Remove trailing spaces.
                  V_ELEVATION_DATUM := CHECK_TSMMAD_VALUE(V_ITEM,
                                                          V_ITEM_NAME,
                                                          'VERTICAL',
                                                          'DATUM',
                                                          V_COL_SEQ,
                                                          V_LOCATION_SEQ,
                                                          P_IMP_CFG_SEQ,
                                                          V_IMPORT_LOG_SEQ);
                  IF V_ELEVATION_DATUM IS NULL THEN
                    V_STATUS := 'N';
                  END IF;
                ELSE
                  V_ELEVATION_DATUM := NULL;
                END IF;
              
              -----------------------------
              -- Elevation Measurement Date
              -----------------------------
              ELSIF V_ITEM_NAME = 'Elevation Measurement Date' THEN
                IF V_ITEM IS NULL THEN
                  V_ELEVATION_DATE := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF V_ITEM IS NOT NULL THEN
                    V_ELEVATION_DATE := GET_DATE(V_ITEM,
                                                 V_COL_SEQ,
                                                 P_IMP_CFG_SEQ,
                                                 V_LINE_CNT,
                                                 V_LOCATION_SEQ);
                    IF V_ELEVATION_DATE IS NULL THEN
                      --the date format is bad
                      V_STATUS := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                                  'Line: ' || V_LINE_CNT ||
                                                   ': - ' || V_ITEM_NAME ||
                                                   ' format does not match format in import configuration',
                                                  V_LOCATION_SEQ);
                    END IF;
      						ELSE
                    V_STATUS := 'N';
                  END IF;
                END IF;
              
              END IF; --end of v_item_names
            
            END IF;
          END IF;
        END LOOP;
              
        /*** END LOCATION LOOP ***/
            
        --Now check all the STORET business rules
      
        IF V_STATION_ID IS NULL THEN
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT ||
                                       ': Station ID is required.',
                                      V_LOCATION_SEQ);
          V_STATUS := 'N';
        END IF;
      
        IF V_ID IS NULL AND V_STATION_ID IS NOT NULL THEN
          SIM2_IMPORT_PKG.WRITE_ERROR(
            V_IMPORT_LOG_SEQ,
            'Line ' || V_LINE_CNT ||
            ': Station ID ' || v_station_id || ' is not in STORET.',
            V_LOCATION_SEQ
          );
          V_STATUS := 'N';
        END IF;

        -- Default Lat/Long to NW hemisphere if not included in import file
        IF v_lat_dir IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Latitude direction has not been provided and will be defaulted to N',v_location_seq);
          v_lat_dir := 'N';
        END IF;
  
        IF v_long_dir IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Longitude direction has not been provided and will be defaulted to W',v_location_seq);
          v_long_dir := 'W';
        END IF;
  
        -- Need to check this post loop as lat direction is not required in import file
        IF v_lat_dir = 'S' AND v_lat_dec_deg > 0 THEN
          IF v_lat_dec_deg < 10 THEN
            v_lat_dec_deg := '-'||TRUNC(v_lat_dec_deg,7);
          ELSE
            v_lat_dec_deg := '-'||TRUNC(v_lat_dec_deg,6);
          END IF;
        END IF;
  
        -- Need to check this post loop as long direction is not required in import file
        IF v_long_dir = 'W' AND v_long_dec_deg > 0 THEN
          IF v_long_dec_deg < 100 THEN
            v_long_dec_deg := '-'||TRUNC(v_long_dec_deg,6);
          ELSE
            v_long_dec_deg := '-'||TRUNC(v_long_dec_deg,5);
          END IF;
        END IF;
        
        --Elevation Rules
        IF V_ELEVATION_UNIT IS NULL AND V_ELEVATION_MSR IS NOT NULL THEN
          V_STATUS := 'N';
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT ||
                                       ': Elevation units are required when elevation is provided.',
                                      V_LOCATION_SEQ);
        END IF;
        IF V_ELEVATION_METHOD IS NULL AND V_ELEVATION_MSR IS NOT NULL THEN
          V_STATUS := 'N';
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT ||
                                       ': Elevation method is required when elevation is provided.',
                                      V_LOCATION_SEQ);
        END IF;
        IF V_ELEVATION_DATUM IS NULL AND V_ELEVATION_MSR IS NOT NULL THEN
          V_STATUS := 'N';
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT ||
                                       ': Elevation datum is required when elevation is provided.',
                                      V_LOCATION_SEQ);
        END IF;
        
      
      --Scale rules
      -- 018 is 'Interpolation-map'
      IF V_GPS_SCALE IS NULL AND V_METHOD = RPAD('018',12) THEN
        Sim2_Import_Pkg.WRITE_ERROR(
          v_import_log_seq,
          'Line '||v_line_cnt||' : Scale is required when Geopositioning Method is Interpolation-Map.',
          v_location_seq);
        V_STATUS := 'N';
      END IF;
        
        
        
      
        --If Sequence number is not included then generate one
        IF V_SEQUENCE_NUMBER IS NULL THEN
          V_SEQUENCE_NUMBER := VALIDATE_SEQUENCE_NUMBER(1,
                                                        V_STA_SEQ,
                                                        V_POINT_TYPE,
                                                        V_IMPORT_LOG_SEQ,
                                                        V_LOCATION_SEQ);                                                        
        END IF;
           
        /* create LOCATION records */
      
        BEGIN
          --LOCATION RECORD 
          V_STL_REC.STL_SEQ                  := V_LOCATION_SEQ;
          V_STL_REC.STL_STA_SEQ              := V_STA_SEQ;
          V_STL_REC.STL_SSW_SEQ              := NULL;
          V_STL_REC.STL_POINT_NAME           := V_POINT_NAME;
          V_STL_REC.STL_TYPE_CODE            := V_POINT_TYPE;
          V_STL_REC.STL_SEQUENCE_NUMBER      := V_SEQUENCE_NUMBER;
          V_STL_REC.STL_LAT_DEGREES          := V_LAT_DEGREES;
          V_STL_REC.STL_LAT_MINUTES          := V_LAT_MINUTES;
          V_STL_REC.STL_LAT_DEC_MINUTES      := V_LAT_DEC_MIN;
          V_STL_REC.STL_LAT_DEC_DEGREES      := V_LAT_DEC_DEG; -- task 314
          V_STL_REC.STL_LAT_SECONDS          := V_LAT_SECONDS;
          V_STL_REC.STL_LAT_DIR              := V_LAT_DIR;
          V_STL_REC.STL_LONG_DEGREES         := V_LONG_DEGREES;
          V_STL_REC.STL_LONG_MINUTES         := V_LONG_MINUTES;
          V_STL_REC.STL_LONG_DEC_MINUTES     := V_LONG_DEC_MIN;
          V_STL_REC.STL_LONG_DEC_DEGREES     := V_LONG_DEC_DEG; -- task 314
          V_STL_REC.STL_LONG_SECONDS         := V_LONG_SECONDS;
          V_STL_REC.STL_LONG_DIR             := V_LONG_DIR;
          V_STL_REC.STL_LAT_LONG_MSR_DATE    := V_LAT_LONG_DATE;
          V_STL_REC.STL_LATLONG_ACCURCY      := NULL;
          V_STL_REC.STL_LATLONG_ACCURCY_UNIT := NULL;
          V_STL_REC.STL_GPS_SCALE            := V_GPS_SCALE;
          V_STL_REC.STL_TSMGEOPA_IS_NUMBER   := NULL;
          V_STL_REC.STL_TSMGEOPA_ORG_ID      := NULL;
          V_STL_REC.STL_TSMFHU_IS_NUMBER     := NULL;
          V_STL_REC.STL_TSMFHU_ORG_ID        := NULL;
          V_STL_REC.STL_TSMRRR_IS_NUMBER     := NULL;
          V_STL_REC.STL_TSMRRR_ORG_ID        := NULL;
          V_STL_REC.STL_GEOPOS_METHOD        := V_METHOD;
          V_STL_REC.STL_GEOPOS_DATUM         := V_DATUM;
          V_STL_REC.STL_ELEVATION_MSR        := V_ELEVATION_MSR;
          V_STL_REC.STL_ELEVATION_UNIT       := V_ELEVATION_UNIT;
          V_STL_REC.STL_ELEV_METHOD          := V_ELEVATION_METHOD;
          V_STL_REC.STL_ELEV_DATUM           := V_ELEVATION_DATUM;
          V_STL_REC.STL_ELEV_MSR_DATE        := V_ELEVATION_DATE;
          V_STL_REC.STL_TSMNAL_CD            := NULL;
          V_STL_REC.STL_TSMNAL_STATE         := NULL;
          V_STL_REC.STL_NRCS                 := NULL;
          V_STL_REC.STL_TSMSTATN0IS_NUMBER   := V_STATION_IS_NBR;
          V_STL_REC.STL_TSMSTATN0ORG_ID      := RPAD(P_ORG_ID, 8);
          V_STL_REC.STL_IMPORT_SEQ           := V_IMPORT_LOG_SEQ;
          V_STL_REC.STL_STATUS               := V_STATUS;
        
          V_GPS_SCALE := NULL;
        
          V_SUCCESS := CREATE_STATION_LOCATION(V_STL_REC);
        
        EXCEPTION
          WHEN OTHERS THEN
            V_IMPORT_DESC := SQLERRM || ' - writing a station. Line: ' ||
                             V_LINE_CNT;
            SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                        V_IMPORT_DESC,
                                        V_LOCATION_SEQ);
        END;
      
        /* update import lines table with station sequence */
        UPDATE SIM_IMP_LINES
           SET SIPL_STL_SEQ = V_LOCATION_SEQ
         WHERE SIPL_SEQ = V_TEXT_LINE.SIPL_SEQ;
      
      EXCEPTION
        WHEN E_NO_STATION THEN
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      'Line ' || V_LINE_CNT ||
                                       ':Unable to Import Location.',
                                      NULL);
          V_STATUS := 'N';
        WHEN OTHERS THEN
          SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                      SQLERRM || ' Line ' || V_LINE_CNT ||
                                       ':Unable to Import Location.',
                                      NULL);
          V_STATUS := 'N';
      END;
    END LOOP; /* end of text loop */
      
    OPEN C_COMP_COUNT(V_IMPORT_LOG_SEQ);
    FETCH C_COMP_COUNT
      INTO V_COMPLETE_CNT;
    CLOSE C_COMP_COUNT;
  
    V_ERROR_CNT := V_LINE_CNT - V_COMPLETE_CNT;
  
    /* update counts in the import log */
    UPDATE SIM_IMPORT_LOG
       SET SIL_ROWS_IMPORTED = V_LINE_CNT,
           SIL_ROWS_ERRORS   = V_ERROR_CNT,
           SIL_ROWS_READY    = V_COMPLETE_CNT
     WHERE SIL_SEQ = V_IMPORT_LOG_SEQ;
    COMMIT;
    SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                TO_CHAR(V_LINE_CNT) ||
                                 ' lines processed /' ||
                                 TO_CHAR(V_COMPLETE_CNT) ||
                                 ' locations loaded',
                                NULL);
    SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                'Finished at ' ||
                                TO_CHAR(SYSDATE, 'HH24:MI'),
                                NULL);
  END;

  EXCEPTION
    WHEN E_NO_DELIMITER THEN
      RAISE_APPLICATION_ERROR(-20102,
                              'Error getting delimiter from import configuration.');
    WHEN E_BAD_ORG THEN
      RAISE_APPLICATION_ERROR(-20109,
                              'Unable to find the specified org id in local version of STORET.');
    WHEN E_NO_CFG THEN
      RAISE_APPLICATION_ERROR(-20102,
                              'Configuration invalid or does not exist.');
    WHEN E_BAD_COLUMNS THEN
      RAISE_APPLICATION_ERROR(-20197,
                              'The import file contains the wrong number of columns on line: ' ||
                              V_LINE_CNT);
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SUBSTR(SQLERRM ||
                                         ' - in sim2_import_station_pkg.import_station. Line: ' ||
                                         TO_CHAR(V_LINE_CNT),
                                         1,
                                         240),
                                  V_LOCATION_SEQ);
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  TO_CHAR(V_LINE_CNT) ||
                                   ' lines processed /' ||
                                   TO_CHAR(V_COMPLETE_CNT) ||
                                   ' stations loaded',
                                  NULL);
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  ' Finished at ' ||
                                  TO_CHAR(SYSDATE, 'HH24:MI'),
                                  NULL);
  END;
  /******************************************************************************************************/
END SIM_IMPORT_LOCATIONS_PKG;
/
