create or replace package Sim_import_result_checks is

  /* Modified 7/16/02 pwh - Added the v_userid_code variable, set it to 'SIM122  ' and inserted this variable
     into the D_USERID_CODE column in the STORET tables.
  */
  /*
  Modified 6/10/02 by SF. Expanded duplicate results to include detection limit UOM.
  Modified 2/25/02 by SF Added Habitat Assessment checks to check_type_category_match.
  
  Modified 2/25/02 by SF Changed check_parent so it checks that parents are on the same
  station visit as kids. Also improved error messages.

*/
  -- Author  : SUSANF
  -- Created : 12/26/2001 11:37:07 AM
  -- Purpose : Several check functions originally in the main Results import package

  -- Public type declarations
    TYPE t_lat_pk_type IS RECORD (
     degrees           NUMBER(2),
     minutes           NUMBER(2),
     seconds           NUMBER(8,6),
     dec_min           NUMBER(8,6)
     );

    TYPE t_lon_pk_type IS RECORD (
     degrees           NUMBER(3),
     minutes           NUMBER(2),
     seconds           NUMBER(8,6),
     dec_min           NUMBER(8,6)
     );

  -- Public function and procedure declarations
FUNCTION check_class_descriptor(p_char_is_nbr IN NUMBER, v_import_log_seq IN NUMBER)
         RETURN NUMBER;

FUNCTION check_duplicate_result(p_char_is_nbr IN NUMBER, p_fa_seq IN NUMBER, p_brg_id IN VARCHAR2,
         p_brgi_indiv IN NUMBER, p_br_type IN VARCHAR2, p_result_text IN VARCHAR2,
         p_uom_is_nbr IN NUMBER, p_species_number IN VARCHAR2, p_lower_bnd IN NUMBER,
         p_upper_bnd IN NUMBER, v_line_cnt IN NUMBER,
         v_char_result_type IN VARCHAR, v_import_log_seq IN NUMBER)
         RETURN BOOLEAN;

FUNCTION check_dup_class(p_fa_seq IN NUMBER, p_brg_seq IN NUMBER, p_lower_bnd_amt IN NUMBER,
                           p_upper_bnd_amt IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                           RETURN NUMBER;

FUNCTION check_dup_brgi (p_fa_seq IN NUMBER, p_brg_seq IN NUMBER, p_individual_number IN NUMBER,
                           p_char_is_nbr IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                           RETURN NUMBER;
                           
FUNCTION check_data_logger_item(p_item_name IN VARCHAR2,p_is_number IN NUMBER, p_line_cnt IN NUMBER, p_org_id IN VARCHAR2,p_fa_seq IN NUMBER,
                                  p_import_log_seq IN NUMBER)
                                  RETURN NUMBER;                       

FUNCTION check_parent (
           p_parent_id        IN VARCHAR2, 
           p_ssv_seq          IN NUMBER,
           p_line_cnt         IN NUMBER, 
           p_import_log_seq   IN NUMBER,
           p_parent_fa_seq    OUT NUMBER,
           p_parent_is_number OUT NUMBER,
           p_parent_org_id    OUT CHAR)
           RETURN NUMBER;

FUNCTION check_lat (p_inlat IN VARCHAR2, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                      RETURN t_lat_pk_type;

FUNCTION check_lon (p_inlon IN VARCHAR2, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                      RETURN t_lon_pk_type;

FUNCTION check_for_activity_in_sim(p_org_id IN VARCHAR2, p_activity_id IN VARCHAR2,p_trip_seq IN NUMBER,
                                       p_station_is_nbr IN NUMBER,p_visit_seq IN NUMBER,p_replicate_nbr IN NUMBER,
                                       p_import_log_seq IN NUMBER,p_fa_seq IN NUMBER,p_line_cnt IN NUMBER)
                                       RETURN NUMBER;

FUNCTION check_activity_id(p_activity_id    IN VARCHAR2, p_replicate_nbr IN VARCHAR2,
                           p_station_is_nbr IN NUMBER,   p_trip_seq      IN NUMBER,
                           p_visit_seq      IN NUMBER,   p_org_id        IN VARCHAR2,
                           p_type_name      IN VARCHAR2,   p_medium        IN VARCHAR2,
                           p_status         IN OUT VARCHAR2,
                           p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER,
                           v_max_assnd_seq_num OUT NUMBER) RETURN NUMBER;

FUNCTION check_storet_act_id(p_activity_id IN VARCHAR2, p_replicate_nbr IN NUMBER,
                               p_tsrstvst_is_nbr IN NUMBER, p_org_id IN VARCHAR2,
                               p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                               RETURN Sim_import_result_extras.t_storet_pk_type;

FUNCTION check_trawl_gear(p_proc_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER,
                            p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                            RETURN NUMBER;

FUNCTION check_gear_type(p_proc_is_nbr IN NUMBER, p_gear_is_nbr IN NUMBER,
                           p_org_id IN VARCHAR2, p_fa_seq IN NUMBER,
                           p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                           RETURN NUMBER;

FUNCTION check_trip_dates (p_trip_seq IN NUMBER, p_activity_date IN DATE, p_org_id IN VARCHAR2,
                             p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                             RETURN NUMBER;

FUNCTION check_visit_dates (p_visit_seq IN NUMBER, p_activity_date IN DATE, p_org_id IN VARCHAR2,
                              p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                              RETURN NUMBER;

FUNCTION check_group_type_med(p_group_pk IN Sim_import_result_extras.t_storet_pk_type,
                              p_type IN VARCHAR2, p_medium IN VARCHAR2,
                              p_intent_name IN VARCHAR2, p_bio_rslt_type_name IN VARCHAR2,
                              p_community_is_nbr IN NUMBER, p_community_org_id IN VARCHAR2,
                              p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER,
                              p_HA_group OUT NUMBER)
                              RETURN NUMBER;

FUNCTION check_chardet_comp(p_char_is_nbr IN NUMBER, p_char_org_id IN VARCHAR2,
                            p_anlpr_is_nbr IN NUMBER, p_uom_is_nbr IN NUMBER,
                            p_uom_org_id IN VARCHAR2, p_fraction IN NUMBER,
                            p_value_type IN VARCHAR2, p_units_ok IN BOOLEAN,
                            p_bio_rslt_type_name IN VARCHAR2,
                            p_fa_seq IN NUMBER, p_line_cnt IN NUMBER,
                            p_import_log_seq IN NUMBER,p_char_result_type IN VARCHAR2)
                            RETURN NUMBER;

FUNCTION check_type_category_match(p_type IN VARCHAR2, p_cat IN VARCHAR2,
                                   p_fa_seq IN NUMBER,p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER,
                                   p_habitat OUT NUMBER)
                                   RETURN NUMBER;

FUNCTION check_individual_type(p_char_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER,
                          p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                          RETURN NUMBER;
                          
end Sim_import_result_checks;
/
create or replace package body Sim_import_result_checks is

/******************************************************************/
FUNCTION check_class_descriptor(p_char_is_nbr IN NUMBER, v_import_log_seq IN NUMBER)
         RETURN NUMBER IS

  v_char_is_nbr NUMBER(8);

  CURSOR c_class_descriptor IS
    SELECT TSRCHAR_IS_NUMBER
    FROM TSRCTCA
    WHERE tsrchtyp_category = 'BIOLOGICAL'
    AND tsrchtyp_subcatgry = 'PHYSICAL MEASUREMENTS'
    AND tsrchar_is_number = p_char_is_nbr;

  BEGIN
    OPEN c_class_descriptor;
      FETCH c_class_descriptor INTO v_char_is_nbr;
      IF c_class_descriptor%FOUND THEN
        RETURN 1;
      ELSE
        RETURN 0;
      END IF;
    CLOSE c_class_descriptor;
  EXCEPTION
    WHEN OTHERS THEN
     Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check class descriptor',NULL);
     RETURN 0;

END check_class_descriptor;

/******************************************************************/
FUNCTION check_duplicate_result(
         p_char_is_nbr      IN NUMBER, 
         p_fa_seq           IN NUMBER, 
         p_brg_id           IN VARCHAR2,
         p_brgi_indiv       IN NUMBER, 
         p_br_type          IN VARCHAR2, 
         p_result_text      IN VARCHAR2,
         p_uom_is_nbr       IN NUMBER, 
         p_species_number   IN VARCHAR2, 
         p_lower_bnd        IN NUMBER,
         p_upper_bnd        IN NUMBER, 
         v_line_cnt         IN NUMBER,
         v_char_result_type IN VARCHAR, 
         v_import_log_seq   IN NUMBER)
         RETURN BOOLEAN IS
         
  /* SAIC has supplied these rules:
  For a given Field Activity, QC Sample, Biological Result Group, or
  Biological Result Group Individual, the following all must be the same
  before the duplicate warning message will appear.
  > *          Characteristic search_name
  > *          Result value_text
  > *          Result species_number
  > *          Unit of Measure
  
   SF 8/8/02
  */
  CURSOR c_check_dup_result(
         p_char_is_nbr IN NUMBER,
         p_brg_seq     IN NUMBER, 
         p_brgi_seq    IN NUMBER,
         p_fa_seq      IN NUMBER) IS
  SELECT rs_seq,rs_result,rs_tsruom_is_nbr,rs_species_number,rs_brg_seq,rs_brgi_seq
  FROM   SIM_RESULTS
  WHERE  RS_FA_SEQ = p_fa_seq
  AND    RS_TSRCHAR_IS_NUMBER = p_char_is_nbr
  AND    NVL(RS_BRG_SEQ,0) = NVL(p_brg_seq,0)
  AND    NVL(RS_BRGI_SEQ,0) = NVL(p_brgi_seq,0);

  CURSOR c_get_brg_seq(
         p_brg_id IN VARCHAR2,
         p_fa_seq IN NUMBER) IS
  SELECT BRG_SEQ FROM SIM_BIO_RESULTS_GRP
  WHERE  BRG_ID_CODE = p_brg_id
  AND    BRG_FA_SEQ = p_fa_seq;  --gets brg_seq if it exists

  CURSOR c_get_brgi_seq(
         p_brg_seq    IN NUMBER, 
         p_brgi_indiv IN NUMBER, 
         p_fa_seq     IN NUMBER) IS
  SELECT BRGI_SEQ FROM SIM_BIO_RESULTS_GRP_INDIVIDUAL
  WHERE  BRGI_BRG_SEQ = p_brg_seq
  AND    BRGI_INDIVIDUAL_NUMBER = p_brgi_indiv; -- finds a dup brgi individual_number

  CURSOR c_get_class_bounds(
         p_fa_seq    IN NUMBER, 
         p_brg_id    IN VARCHAR2, 
         p_lower_bnd IN NUMBER, 
         p_upper_bnd IN NUMBER) IS
  SELECT RCI_SEQ FROM SIM_RESULT_CLASS_INDICATOR, SIM_BIO_RESULTS_GRP
  WHERE  RCI_FA_SEQ = p_fa_seq
  AND    BRG_FA_SEQ = p_fa_seq
  AND    BRG_ID_CODE = p_brg_id
  AND    RCI_LOWER_BND_AMT = p_lower_bnd
  AND    RCI_UPPER_BND_AMT = p_upper_bnd;

  v_result_unique BOOLEAN;
  v_brg_seq       NUMBER(10) NULL;
  v_brgi_seq      NUMBER(10) NULL;
  v_rci_seq       NUMBER(10);
  v_result_rec    c_check_dup_result%ROWTYPE;

BEGIN
  -- kms - function that looks for results that have the same characteristic on the same activity.

  -- sfp: modified to allow for duplicate results on activity
  --      if they have different bio results groups
  
  -- does this activity have a bio results group id?
  IF p_brg_id IS NOT NULL THEN
    -- if yes
    -- get the primary key (aka seq) of the bio-results group
    OPEN c_get_brg_seq(p_brg_id,p_fa_seq);
    FETCH c_get_brg_seq INTO v_brg_seq;
    CLOSE c_get_brg_seq;
  END IF;

  -- did the caller pass a bio-results group individual number?
  IF p_brgi_indiv IS NOT NULL THEN
    -- if yes
    -- get the primary key (aka seq) of the bio-results group individual
    OPEN c_get_brgi_seq(v_brg_seq, p_brgi_indiv, p_fa_seq);
    FETCH c_get_brgi_seq INTO v_brgi_seq;
    CLOSE c_get_brgi_seq;
  END IF;

  -- Open a cursor full of duplicate results
  OPEN c_check_dup_result(
    p_char_is_nbr,
    NVL(v_brg_seq,0),
    NVL(v_brgi_seq,0), 
    p_fa_seq
  );
  
  -- prepare for loop below by first assuming result is unique
  v_result_unique := TRUE;
  
  LOOP
    FETCH c_check_dup_result INTO v_result_rec;

    -- did we run out of duplicate result records to check?    
    IF c_check_dup_result%NOTFOUND THEN
      -- if yes, exit the loop
      EXIT;

    ELSE
      -- if no (we found duplicate records)

      -- now assume the result is a duplicate
      v_result_unique := FALSE;

      -- are we sure everything is the same?
      IF NVL(v_char_result_type,'x') IN ('TEXT','PV  ') THEN
        -- if yes
        Sim2_Import_Pkg.write_error(
          v_import_log_seq,
          'Warning: Line '||v_line_cnt||
          ': Result with this characteristic already exists on this activity.',
          p_fa_seq
        );
        
      ELSE
        -- if no 
        
        -- we are not sure everything is the same yet 
        -- because one the following might differ

        -- is the result text different?
        IF NVL(v_result_rec.rs_result,'x') != NVL(p_result_text,'x') THEN
          -- if yes
          v_result_unique := TRUE;
        END IF;
  
        -- is the species number different?
        IF NVL(v_result_rec.rs_species_number,'x') != NVL(p_species_number,'x') THEN
          -- if yes
          v_result_unique := TRUE;
        END IF;
  
        -- is the unit of measure different?
        IF NVL(v_result_rec.rs_tsruom_is_nbr,-999) != NVL(p_uom_is_nbr,-999) THEN
          -- if yes
          v_result_unique := TRUE;
        END IF;
  
      END IF;
  
    END IF;

    -- Just one more thing to check...
    
    -- Do we think we have a duplicate result at this point?
    IF v_result_unique = FALSE THEN
      -- if yes
 
      -- Single Taxon Frequency Classes can have duplicates
      
      -- Is the class 'Single Taxon Frequency Classes'?
      IF p_br_type = 'Single Taxon Frequency Classes' THEN
        -- if yes
  
        -- try to get the class bound record
        OPEN c_get_class_bounds(p_fa_seq, v_brg_seq, p_lower_bnd, p_upper_bnd);
        FETCH c_get_class_bounds INTO v_rci_seq;
  
        -- did we get a class bounds record?
        IF c_get_class_bounds%NOTFOUND THEN
          -- if no, then the duplicate is okay
          v_result_unique := TRUE;
        END IF;
        CLOSE c_get_class_bounds;
  
      END IF;
  
    END IF;

    -- So, do we have a duplicate result?
    IF v_result_unique = FALSE THEN
      -- if yes
  
      Sim2_Import_Pkg.write_error(
        v_import_log_seq,
        'Warning: Line '||v_line_cnt||
        ': Duplicate result with this characteristic exists on this activity.',
        p_fa_seq
      );
      -- Leave the loop
      EXIT;

    END IF;

  END LOOP;
  CLOSE c_check_dup_result;

  RETURN v_result_unique;

EXCEPTION
  
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(
      v_import_log_seq, 
      SQLERRM||' - in check duplicate result.',
      p_fa_seq
    );
    RETURN FALSE;

END check_duplicate_result;

/******************************************************************************************************/

FUNCTION check_dup_brgi (p_fa_seq IN NUMBER, p_brg_seq IN NUMBER, p_individual_number IN NUMBER,
                         p_char_is_nbr IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                         RETURN NUMBER IS

  v_check         NUMBER(1);
  v_brgi_seq      NUMBER(10);

  CURSOR c_get_brgi_seq(p_brg_seq IN NUMBER, p_individual_number IN NUMBER ) IS
    SELECT BRGI_SEQ FROM SIM_BIO_RESULTS_GRP_INDIVIDUAL
     WHERE BRGI_BRG_SEQ = p_brg_seq
       AND BRGI_INDIVIDUAL_NUMBER = p_individual_number;

BEGIN
  /*sfp -  function that checks for duplicate individual_number in SIM_BIO_RESULTS_GROUP_INDIVIDUAL */

  OPEN c_get_brgi_seq(p_brg_seq,p_individual_number);
  FETCH c_get_brgi_seq INTO v_brgi_seq;
    IF c_get_brgi_seq%NOTFOUND THEN
      v_check := 1;  -- it is not a duplicate, it is first occurence of individual number, create new brgi record only for first occurrences
    ELSE
        v_check := 0; -- it is duplicate individual number
    END IF;

  RETURN v_check;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - getting individual number',NULL);
    v_check := 0;
    RETURN v_check;

END check_dup_brgi;

/***********************************************************************************************/
FUNCTION check_dup_class(p_fa_seq IN NUMBER, p_brg_seq IN NUMBER, p_lower_bnd_amt IN NUMBER,
                         p_upper_bnd_amt IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                         RETURN NUMBER IS

  v_check         NUMBER(1);
  v_rs_seq        NUMBER(10);
  v_rci_seq       NUMBER(10);

  CURSOR c_get_rci_seq(p_fa_seq IN NUMBER, p_brg_seq IN NUMBER, p_lower_bnd_amt IN NUMBER, p_upper_bnd_amt IN NUMBER) IS
    SELECT RCI_SEQ, RCI_RS_SEQ
      FROM SIM_RESULT_CLASS_INDICATOR
     WHERE RCI_BRG_SEQ = p_brg_seq
       AND RCI_FA_SEQ = p_fa_seq
       AND RCI_LOWER_BND_AMT = p_lower_bnd_amt
       AND RCI_UPPER_BND_AMT = p_upper_bnd_amt;

BEGIN
  /*sfp -  function that checks for duplicate upper/lower_bnd_amts in RCI record with same BRG_SEQ */

  OPEN c_get_rci_seq(p_fa_seq, p_brg_seq, p_lower_bnd_amt, p_upper_bnd_amt);
  FETCH c_get_rci_seq INTO v_rci_seq, v_rs_seq;
    IF c_get_rci_seq%NOTFOUND THEN
      v_check := 1;  -- it is not a duplicate, it is first occurence of a class
    ELSE
        v_check := 0; -- it is duplicate class
    END IF;

  RETURN v_check;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - getting individual number',NULL);
    v_check := 0;
    RETURN v_check;

END check_dup_class;


/***********************************************************************************************/
FUNCTION check_parent (
           p_parent_id        IN VARCHAR2,
           p_ssv_seq          IN NUMBER,
           p_line_cnt         IN NUMBER, 
           p_import_log_seq   IN NUMBER,
           p_parent_fa_seq    OUT NUMBER,
           p_parent_is_number OUT NUMBER,
           p_parent_org_id    OUT CHAR)
           RETURN NUMBER
  IS

  CURSOR c_sim_station_visits (
         cp_ssv_seq  IN NUMBER) 
  IS
  SELECT SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID
  FROM   SIM_STATION_VISITS
  WHERE  SSV_SEQ = cp_ssv_seq;
  
  CURSOR c_sim_field_activities(
         cp_ssv_seq    IN NUMBER, 
         cp_parent_id  IN VARCHAR2) 
  IS
  SELECT FA_SEQ, FA_IMPORT_SEQ
  FROM   SIM_FIELD_ACTIVITIES
  WHERE  FA_SSV_SEQ     = cp_ssv_seq
  AND    FA_ACTIVITY_ID = RPAD(cp_parent_id,12);
  
  CURSOR c_tsrfdact(
         cp_tsrstvst_is_nbr  IN NUMBER, 
         cp_tsrstvst_org_id  IN VARCHAR2,
         cp_parent_id        IN VARCHAR2) 
  IS
  SELECT TSRFDACT_IS_NUMBER, TSRFDACT_ORG_ID
  FROM   TSRFDACT
  WHERE  TSRSTVST_IS_NUMBER = cp_tsrstvst_is_nbr
  AND    TSRSTVST_ORG_ID    = cp_tsrstvst_org_id
  AND    ID_CODE            = RPAD(cp_parent_id,12);

  r_sim_station_visits    c_sim_station_visits%rowtype;
  r_sim_field_activities  c_sim_field_activities%rowtype;
  r_tsrfdact              c_tsrfdact%rowtype;

  e_wrong_import_log      exception;  -- RETURN 4
  e_no_sim_station_visit  exception;  -- RETURN 3
  e_no_parent_activity    exception;  -- RETURN 2

BEGIN

  -- Initialize the OUT parameters
  p_parent_fa_seq    := NULL;
  p_parent_is_number := NULL;
  p_parent_org_id    := NULL;
  
  -- First retrieve the station visit record
  OPEN  c_sim_station_visits (p_ssv_seq);
  FETCH c_sim_station_visits INTO r_sim_station_visits;

  -- Did we find the station visit record?
  IF c_sim_station_visits%NOTFOUND THEN
    -- if no

    -- close the station visit cursor
    CLOSE c_sim_station_visits;

    -- raise exception
    RAISE e_no_sim_station_visit;

  END IF;

  -- close the station visit cursor
  CLOSE c_sim_station_visits;

  -- try to retrieve the parent activity record from STORET    
  OPEN  c_tsrfdact(
    r_sim_station_visits.SSV_TSRSTVST_IS_NUMBER,
    r_sim_station_visits.SSV_TSRSTVST_ORG_ID,
    p_parent_id
  );
  FETCH c_tsrfdact INTO r_tsrfdact;

  -- did we find the parent field activities record in TSRFDACT?
  IF c_tsrfdact%FOUND THEN
    -- if yes
    p_parent_is_number := r_tsrfdact.TSRFDACT_IS_NUMBER;
    p_parent_org_id := r_tsrfdact.TSRFDACT_ORG_ID;

    -- close TSRFDACT    
    CLOSE c_tsrfdact;
    
  ELSE  
    -- if no, we did not find the parent in TSRFDACT
      
    -- close TSRFDACT    
    CLOSE c_tsrfdact;
  
    -- try to retrieve the parent activity record in SIM_FIELD_ACTIVITIES
    OPEN  c_sim_field_activities(p_ssv_seq,p_parent_id);
    FETCH c_sim_field_activities INTO r_sim_field_activities;
  
    -- did we find the parent field activities record in SIM_FIELD_ACTIVITIES?
    IF c_sim_field_activities%FOUND THEN
      -- if yes
      -- return the sequence number of the parent record to caller
      
      -- does the import log sequence match?
      IF (p_import_log_seq <> r_sim_field_activities.FA_IMPORT_SEQ) THEN
        -- if no
        -- close the sim_field_activities cursor
        CLOSE c_sim_field_activities;
        -- raise an exception
        raise e_wrong_import_log;
      END IF;
      
      p_parent_fa_seq := r_sim_field_activities.FA_SEQ;
    END IF;
  
    -- close the sim_field_activities cursor
    CLOSE c_sim_field_activities;
    
  END IF;
    
  -- did we find a parent activity in SIM or STORET?
  IF  p_parent_fa_seq IS NULL 
  AND p_parent_is_number IS NULL 
  AND p_parent_org_id IS NULL
  THEN
    -- if no
    -- raise an exception
    RAISE e_no_parent_activity;
  END IF;
  
  -- tell caller that no exception occurred
  RETURN NULL;

EXCEPTION

  WHEN e_wrong_import_log THEN

    Sim2_Import_Pkg.write_error(
      p_import_log_seq, 
      'Line ' || p_line_cnt || ': Parent activity ' || p_parent_id || ' found in a previous import. ' ||
      'Please migrate the previous import before importing this activity.',
      NULL
    );
    -- tell caller the import sequence does not match
    RETURN 4;

  WHEN e_no_parent_activity THEN

    Sim2_Import_Pkg.write_error(
      p_import_log_seq, 
      'Line ' || p_line_cnt || ': ' ||
      'Parent activity ' || p_parent_id || 
      ' not found in SIM or STORET on this visit.',
      NULL
    );
    -- tell caller we did NOT find a parent activity
    RETURN 3;

  WHEN e_no_sim_station_visit THEN

    Sim2_Import_Pkg.write_error(
      p_import_log_seq, 
      'Line ' || p_line_cnt || ': Station visit not found for this activity.',
      NULL
    );
    -- tell caller we did NOT find a parent activity
    RETURN 2;
      
  WHEN OTHERS THEN

    Sim2_Import_Pkg.write_error(
      p_import_log_seq, 
      'Line ' || p_line_cnt || ': ' || SQLERRM || 
      ' - in function SIM_IMPORT_RESULT_CHECKS.check_parent.',
      NULL
    );
    -- tell caller we did NOT find a parent activity because an exception occurred
    RETURN 1;

END check_parent;
/***********************************************************************************************/


FUNCTION check_lat (p_inlat IN VARCHAR2, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                    RETURN t_lat_pk_type IS

  v_lat_key             t_lat_pk_type;
  v_degrees             NUMBER(2);
  v_minutes             NUMBER(2);
  v_seconds             NUMBER(6,4);
  v_dec_min             NUMBER(6,4);

BEGIN
   IF TRANSLATE(p_inlat,'a1234567890.','a') IS NULL AND LENGTH(p_inlat) >=6 THEN
          v_degrees := TO_NUMBER(SUBSTR(p_inlat,1,2));
          v_minutes := TO_NUMBER(SUBSTR(p_inlat,3,2));
          v_seconds := TO_NUMBER(SUBSTR(p_inlat,5));

          IF v_degrees BETWEEN 0 AND 90 AND
             v_minutes BETWEEN 0 AND 60 AND
             v_seconds BETWEEN 0 AND 60 THEN
             v_dec_min := v_minutes + (v_seconds/60);
             v_lat_key.degrees := v_degrees;
             v_lat_key.minutes := v_minutes;
             v_lat_key.seconds := v_seconds;
             v_lat_key.dec_min := v_dec_min;
          ELSE
             v_lat_key.degrees := NULL;
             v_lat_key.minutes := NULL;
             v_lat_key.seconds := NULL;
             v_lat_key.dec_min := NULL;
          END IF;
   ELSE
          v_lat_key.degrees := NULL;
          v_lat_key.minutes := NULL;
          v_lat_key.seconds := NULL;
          v_lat_key.dec_min := NULL;
          Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq,' - Invalid character in latitude, line: '||p_line_cnt,NULL);

   END IF;

RETURN v_lat_key;

          EXCEPTION
            WHEN OTHERS THEN
                v_lat_key.degrees := NULL;
                v_lat_key.minutes := NULL;
                v_lat_key.seconds := NULL;
                v_lat_key.dec_min := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq,' - getting latitude degrees, line: '||p_line_cnt,NULL);
END check_lat;

/***********************************************************************************************/

FUNCTION check_lon (p_inlon IN VARCHAR2, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                    RETURN t_lon_pk_type IS

  v_lon_key             t_lon_pk_type;
  v_degrees             NUMBER(3);
  v_minutes             NUMBER(2);
  v_seconds             NUMBER(6,4);
  v_dec_min             NUMBER(6,4);

BEGIN
   IF TRANSLATE(p_inlon,'a1234567890.','a') IS NULL AND LENGTH(p_inlon) >=7 THEN
          v_degrees := TO_NUMBER(SUBSTR(p_inlon,1,3));
          v_minutes := TO_NUMBER(SUBSTR(p_inlon,4,2));
          v_seconds := TO_NUMBER(SUBSTR(p_inlon,6));

          IF v_degrees BETWEEN 0 AND 180 AND
             v_minutes BETWEEN 0 AND 60 AND
             v_seconds BETWEEN 0 AND 60 THEN
             v_dec_min := v_minutes + (v_seconds/60);
             v_lon_key.degrees := v_degrees;
             v_lon_key.minutes := v_minutes;
             v_lon_key.seconds := v_seconds;
             v_lon_key.dec_min := v_dec_min;
          ELSE
             v_lon_key.degrees := NULL;
             v_lon_key.minutes := NULL;
             v_lon_key.seconds := NULL;
             v_lon_key.dec_min := NULL;
          END IF;
   ELSE
          v_lon_key.degrees := NULL;
          v_lon_key.minutes := NULL;
          v_lon_key.seconds := NULL;
          v_lon_key.dec_min := NULL;
          Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq,' - Invalid character in longitude, line: '||p_line_cnt,NULL);

   END IF;

RETURN v_lon_key;

          EXCEPTION
            WHEN OTHERS THEN
                v_lon_key.degrees := NULL;
                v_lon_key.minutes := NULL;
                v_lon_key.seconds := NULL;
                v_lon_key.dec_min := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq,' - getting latitude degrees, line: '||p_line_cnt,NULL);
END check_lon;

/******************************************************************************************************/

FUNCTION check_activity_id(p_activity_id    IN VARCHAR2, p_replicate_nbr IN VARCHAR2,
                           p_station_is_nbr IN NUMBER,   p_trip_seq      IN NUMBER,
                           p_visit_seq      IN NUMBER,   p_org_id        IN VARCHAR2,
                           p_type_name      IN VARCHAR2,   p_medium        IN VARCHAR2,
                           p_status         IN OUT VARCHAR2,
                           p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER,
                           v_max_assnd_seq_num OUT NUMBER) RETURN NUMBER IS

  CURSOR c_check_activity_id(p_cursor_act_id  IN CHAR, p_replicate_nbr IN VARCHAR2,
                             p_station_is_nbr IN NUMBER,   p_trip_seq      IN NUMBER,
                             p_visit_seq      IN NUMBER,   p_cursor_org    IN VARCHAR2) IS
    SELECT FA_SEQ, FA_TYPE_NAME, FA_MEDIUM, FA_IMPORT_SEQ FROM SIM_FIELD_ACTIVITIES
      WHERE FA_ACTIVITY_ID                  = p_cursor_act_id
      AND   NVL(FA_REPLICATE_NBR,'   ')     = NVL(p_replicate_nbr,'   ')
      AND   NVL(FA_TSMSTATN_IS_NUMBER,'-1') = NVL(p_station_is_nbr,'-1')
      AND   NVL(FA_STP_SEQ,'-1')            = NVL(p_trip_seq,'-1')
      AND   NVL(FA_SSV_SEQ,'-1')            = NVL(p_visit_seq,'-1')
      AND   RPAD(FA_TSMORGAN_ORG_ID,8)      = RPAD(p_cursor_org,8);

    CURSOR c_get_D_ASSND_SEQ_NUM(act_seq IN NUMBER) IS
      SELECT MAX(RS_D_ASSND_SEQ_NUM)
      FROM SIM_RESULTS
      WHERE RS_FA_SEQ = act_seq;

  v_activity_rec c_check_activity_id%ROWTYPE;

BEGIN
  /* kms - function to see if the activity already exists in SIM. */

  OPEN c_check_activity_id(p_activity_id,p_replicate_nbr,p_station_is_nbr,p_trip_seq,p_visit_seq,p_org_id);
  FETCH c_check_activity_id INTO v_activity_rec;
  
    IF c_check_activity_id%NOTFOUND THEN
      v_activity_rec.fa_seq := NULL;      
    ELSE   
      IF v_activity_rec.fa_import_seq != p_import_log_seq THEN
        p_status := 'N';
        Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': This activity has already been imported into SIM in a different file.',NULL);
        RETURN NULL;
      ELSIF v_activity_rec.fa_type_name != p_type_name THEN
        p_status := 'N';
        Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': A new activity type for this activity has been found.  Only one activity type per activity is allowed.',NULL);
      ELSIF v_activity_rec.fa_medium != p_medium THEN
        p_status := 'N';
        Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': A new medium for this activity has been found.  Only one medium per activity is allowed.',NULL);
      END IF;
      --Get the biggest D_ASSND_SEQ_NUM on this activity
      OPEN c_get_d_assnd_seq_num(v_activity_rec.fa_seq);
        FETCH c_get_d_assnd_seq_num INTO v_max_assnd_seq_num;
      CLOSE c_get_d_assnd_seq_num;
    END IF;
  CLOSE c_check_activity_id;

  RETURN v_activity_rec.fa_seq;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check activity id.',NULL);
    RETURN NULL;

END check_activity_id;

/******************************************************************************************************/
FUNCTION check_for_activity_in_sim(p_org_id IN VARCHAR2, p_activity_id IN VARCHAR2,p_trip_seq IN NUMBER,
                                       p_station_is_nbr IN NUMBER,p_visit_seq IN NUMBER,p_replicate_nbr IN NUMBER,
                                       p_import_log_seq IN NUMBER,p_fa_seq IN NUMBER,p_line_cnt IN NUMBER)
                                       RETURN NUMBER IS

  v_fa_seq NUMBER(10);

  --We are only looking for this exact activity on a DIFFERENT import.
  CURSOR c_sim_activity IS
    SELECT FA_SEQ FROM SIM_FIELD_ACTIVITIES
      WHERE FA_ACTIVITY_ID = p_activity_id
      AND FA_STP_SEQ = p_trip_seq
      AND FA_SSV_SEQ = p_visit_seq
      AND FA_REPLICATE_NBR = p_replicate_nbr
      AND FA_TSMSTATN_IS_NUMBER = p_station_is_nbr
      AND FA_TSMSTATN_ORG_ID = RPAD(p_org_id,8)
      AND FA_IMPORT_SEQ != p_import_log_seq;

  BEGIN
    OPEN c_sim_activity;
      FETCH c_sim_activity INTO v_fa_seq;
      IF c_sim_activity%FOUND THEN
        RETURN 1;
      ELSE
        RETURN 0;
      END IF;
    CLOSE c_sim_activity;
  EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check activity in SIM.',p_fa_seq);
  END check_for_activity_in_sim;
/******************************************************************************************************/
FUNCTION check_storet_act_id(p_activity_id IN VARCHAR2, p_replicate_nbr IN NUMBER,
                             p_tsrstvst_is_nbr IN NUMBER, p_org_id IN VARCHAR2,
                             p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                             RETURN Sim_import_result_extras.t_storet_pk_type IS

  v_activity_pk Sim_import_result_extras.t_storet_pk_type;

  CURSOR c_check_storet_act_rep(p_activity_id IN VARCHAR2, p_replicate_nbr IN VARCHAR2,
                                p_tsrstvst_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT TSRFDACT_IS_NUMBER, TSRFDACT_ORG_ID FROM TSRFDACT
      WHERE ID_CODE                   = RPAD(p_activity_id,12)
      AND TSRFDACT_ORG_ID             = RPAD(p_org_id,8)
      AND REPLICATE_NUMBER            = p_replicate_nbr
      AND TSRSTVST_IS_NUMBER          = p_tsrstvst_is_nbr;

  CURSOR c_check_storet_act(p_activity_id IN VARCHAR2,
                            p_tsrstvst_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT TSRFDACT_IS_NUMBER, TSRFDACT_ORG_ID FROM TSRFDACT
      WHERE ID_CODE                   = RPAD(p_activity_id,12)
      AND TSRFDACT_ORG_ID             = RPAD(p_org_id,8)
      AND NVL(REPLICATE_NUMBER,0)     = 0
      AND TSRSTVST_IS_NUMBER          = p_tsrstvst_is_nbr;

BEGIN
  /* kms - function that looks for the activity id in STORET to make sure it's not
     duplicating a pre-existing activity.  If the activity is already in STORET, SIM
     will import it but will prevent it from being exported. */

  /*  DTW - 6/18/03  Corrected logic to allow the same activity ID as long
      as they are on seperate station visits.  Only checked combo of trip and
      station before but did not account for multiple visits in a trip. */

  IF p_replicate_nbr IS NOT NULL THEN
    OPEN c_check_storet_act_rep(p_activity_id,p_replicate_nbr,
                                p_tsrstvst_is_nbr, p_org_id);
    FETCH c_check_storet_act_rep INTO v_activity_pk;
      IF c_check_storet_act_rep%NOTFOUND THEN
        v_activity_pk.is_number := NULL;
        v_activity_pk.org_id := NULL;
      END IF;
    CLOSE c_check_storet_act_rep;
  ELSE
    OPEN c_check_storet_act(p_activity_id,
                            p_tsrstvst_is_nbr, p_org_id);
    FETCH c_check_storet_act INTO v_activity_pk;
      IF c_check_storet_act%NOTFOUND THEN
        v_activity_pk.is_number := NULL;
        v_activity_pk.org_id := NULL;
      END IF;
    CLOSE c_check_storet_act;
  END IF;

  RETURN v_activity_pk;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check activity id.',p_fa_seq);
    v_activity_pk.is_number := NULL;
    v_activity_pk.org_id := NULL;
    RETURN v_activity_pk;

END check_storet_act_id;

/******************************************************************************************************
This function checks to see what kind of gear type is represented by the chosen Sample Collection
Procedure. If it's a Net?Horizontal Tow then Location records must be created and a 1 is returned. If it's
a Net/Non Tow then a 2 is returned. */

FUNCTION check_individual_type(p_char_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER,
                          p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                          RETURN NUMBER IS
  TYPE t_char_type IS RECORD (
     tsrchtyp_category         CHAR(12),
     tsrchtyp_subcatgry      VARCHAR2(30)
     );

  v_good_char             NUMBER(1);
  v_char_type             t_char_type;

  CURSOR c_get_char_type IS
    SELECT tsrchtyp_category, tsrchtyp_subcatgry
    FROM TSRCTCA
    WHERE TSRCHAR_IS_NUMBER = p_char_is_nbr;

  BEGIN
    v_good_char := 0;
    FOR v_char_type IN c_get_char_type LOOP
      IF v_char_type.tsrchtyp_category = 'BIOLOGICAL  ' AND v_char_type.tsrchtyp_subcatgry = 'PHYSICAL MEASUREMENTS' THEN
        v_good_char := 1;
        RETURN v_good_char;
      END IF;
    END LOOP;

    RETURN v_good_char;

  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check individual type.',p_fa_seq);

END check_individual_type;
/**********************************************************************************************/
FUNCTION check_data_logger_item(p_item_name IN VARCHAR2,p_is_number IN NUMBER, p_line_cnt IN NUMBER, p_org_id IN VARCHAR2,p_fa_seq IN NUMBER,
                                p_import_log_seq IN NUMBER)
                                RETURN NUMBER IS
                                
  v_name          VARCHAR2(100);
                                  
  CURSOR c_tsrchar_name(p_char_is_nbr NUMBER,p_char_org_id VARCHAR2) IS
    SELECT A.display_name disp_name
      FROM TSRCTCA T,TSRCHAR A
      WHERE T.TSRCHTYP_SUBCATGRY = 'DATA-LOGGER PROBES'
        AND T.TSRCHAR_IS_NUMBER = A.TSRCHAR_IS_NUMBER
        AND T.TSRCHAR_ORG_ID = A.TSRCHAR_ORG_ID
        AND TSRCHTYP_CATEGORY = 'PHYSICAL'
        AND A.TSRCHAR_IS_NUMBER = p_char_is_nbr
        AND A.TSRCHAR_ORG_ID = p_char_org_id;

                             
BEGIN

  OPEN c_tsrchar_name(p_is_number,p_org_id);
    FETCH c_tsrchar_name INTO v_name;
  CLOSE c_tsrchar_name;

  IF v_name IS NOT NULL THEN
    RETURN(1);
  ELSE        
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||p_item_name||' - Characteristic is of wrong type for Portable Data Logger.',p_fa_seq);
    RETURN(0);
  END IF;
  
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in CHECK DATA LOGGER ITEM.',p_fa_seq);  
      RETURN (0);
END check_data_logger_item;
/**********************************************************************************************/
FUNCTION check_trawl_gear(p_proc_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER,
                          p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                          RETURN NUMBER IS

/* Determines if Activity is a trawl. If yes then 2 AAL records will be required. */
  v_trawl_gear_ind       NUMBER(1);
  v_trawl_gear_name      VARCHAR2(25);

  e_proc_not_found       EXCEPTION;

   CURSOR c_get_proc_gear_type (p_proc_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT FLD_GEAR_TYPE_NAME FROM TSRFLDPR WHERE TSRFLDPR_IS_NUMBER = p_proc_is_nbr
      AND TSRFLDPR_ORG_ID = RPAD(p_org_id,8);

BEGIN
  /* get gear type from field procedure */
  OPEN c_get_proc_gear_type(p_proc_is_nbr, p_org_id);
  FETCH c_get_proc_gear_type INTO v_trawl_gear_name;
    IF c_get_proc_gear_type%NOTFOUND THEN
      RAISE e_proc_not_found;
    END IF;
  CLOSE c_get_proc_gear_type;

  IF v_trawl_gear_name LIKE 'Net/Horizontal Tow%' THEN
    v_trawl_gear_ind := 1;
  ELSIF v_trawl_gear_name LIKE 'Net/Non-Tow%' THEN
    v_trawl_gear_ind := 2;
  ELSE
    v_trawl_gear_ind := 0;
  END IF;

  RETURN v_trawl_gear_ind;

EXCEPTION
  WHEN e_proc_not_found THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Error finding trawl gear type in TSRFLDPR, is number:'||p_proc_is_nbr,p_fa_seq);
    RETURN NULL;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check trawl gear.',p_fa_seq);
    RETURN NULL;

END check_trawl_gear;

/******************************************************************************************************/

FUNCTION check_gear_type(p_proc_is_nbr IN NUMBER, p_gear_is_nbr IN NUMBER,
                         p_org_id IN VARCHAR2, p_fa_seq IN NUMBER,
                         p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                         RETURN NUMBER IS

  v_gear_gear_type       VARCHAR2(25);
  v_proc_gear_type       VARCHAR2(25);
  v_gear_type_ok         NUMBER(1);

  e_proc_not_found       EXCEPTION;

  CURSOR c_get_gear_gear_type (p_gear_is_nbr IN NUMBER) IS
    SELECT TYPE_NAME FROM TSRFLDGR WHERE TSRFLDGR_IS_NUMBER = p_gear_is_nbr;

  CURSOR c_get_proc_gear_type (p_proc_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT FLD_GEAR_TYPE_NAME FROM TSRFLDPR WHERE TSRFLDPR_IS_NUMBER = p_proc_is_nbr
      AND TSRFLDPR_ORG_ID = RPAD(p_org_id,8);

BEGIN
  /* kms - function that makes sure the gear and field procedure selected are compatible -
     that they have the same gear type */

  /* kms (5/18/01) - Changed this so that it only checks the gear types if the collection has
     been assigned a gear type and gear has been given.  This function will be called
     whenever sample collection procedure is present, whether or not gear is also there. */

  /* get gear type from field procedure */
  OPEN c_get_proc_gear_type(p_proc_is_nbr, p_org_id);
  FETCH c_get_proc_gear_type INTO v_proc_gear_type;
    IF c_get_proc_gear_type%NOTFOUND THEN
      RAISE e_proc_not_found;
    END IF;
  CLOSE c_get_proc_gear_type;
  IF p_gear_is_nbr IS NOT NULL AND (RPAD(v_proc_gear_type,25) != RPAD(' ',25) AND v_proc_gear_type IS NOT NULL) THEN
    /* kms - if gear has been given, make sure gear types match */
    /* get gear type from gear */
    OPEN c_get_gear_gear_type(p_gear_is_nbr);
    FETCH c_get_gear_gear_type INTO v_gear_gear_type;
      IF c_get_gear_gear_type%NOTFOUND THEN
        v_gear_gear_type := NULL;
        Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Error finding gear type in TSRFLDGR, is number:'||p_gear_is_nbr,p_fa_seq);
      END IF;
    CLOSE c_get_gear_gear_type;
    /* compare gear types and set flag to true if ok, else set to false */
    IF v_gear_gear_type = v_proc_gear_type THEN
      v_gear_type_ok := 0;
    ELSE
      v_gear_type_ok := 1;
    END IF;

  ELSIF p_gear_is_nbr IS NULL AND (RPAD(v_proc_gear_type,25) != RPAD(' ',25) AND v_proc_gear_type IS NOT NULL) THEN
    /* kms - if gear has not been given and the procedure has a gear type, set flag to invalidate
       the activity and write an error message in import results procedure. */
     v_gear_type_ok := 2;

  END IF;

  RETURN v_gear_type_ok;

EXCEPTION
  WHEN e_proc_not_found THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Error finding gear type in TSRFLDPR, is number:'||p_proc_is_nbr,p_fa_seq);
    RETURN NULL;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check gear type.',p_fa_seq);
    RETURN NULL;

END check_gear_type;

/******************************************************************************************************/
FUNCTION check_trip_dates (p_trip_seq IN NUMBER, p_activity_date IN DATE, p_org_id IN VARCHAR2,
                           p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                           RETURN NUMBER IS

  v_start_date           DATE;
  v_end_date             DATE;
  v_after                NUMBER(10);
  v_before               NUMBER(10);
  v_return_value         NUMBER(1) := 1;

  CURSOR c_trip_dates (p_trip_seq IN NUMBER) IS
      SELECT STP_START_DATE, STP_END_DATE FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

BEGIN
  /* kms - function that makes sure the activity occured between the start and end dates of the trip.
     Called by main import procedure - import_results. */

  OPEN c_trip_dates(p_trip_seq);
  FETCH c_trip_dates INTO v_start_date, v_end_date;
    IF c_trip_dates%NOTFOUND THEN
      CLOSE c_trip_dates;
      RETURN 0;
    END IF;
  CLOSE c_trip_dates;

  /* added for STORET interface bug where some trip years have been entered as 0001 */
  IF TO_CHAR(p_activity_date,'RRRR') LIKE '20__' THEN
    IF TO_CHAR(v_start_date,'RRRR') LIKE '00__' THEN
      v_start_date := TO_DATE(TO_CHAR(v_start_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_start_date,'RRRR'),2,3),'MM-DD-RRRR');
    END IF;

    IF TO_CHAR(v_end_date,'RRRR') LIKE '00__' THEN
      v_end_date := TO_DATE(TO_CHAR(v_end_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_end_date,'RRRR'),2,3),'MM-DD-RRRR');
    END IF;
  END IF;

  /* make sure activity date is within trip date range */
  v_after := v_start_date - p_activity_date;  --will be negative if activity after start date

  /* if activity came after trip start date, make sure it came before trip end date */
  IF v_after <= 0 THEN

    IF TO_CHAR(v_end_date,'RRRR') IS NOT NULL THEN
      v_before := 0; -- bka  p_activity_date - v_end_date;  --will be negative if activity before end date

      IF v_before > 0 THEN
        /* activity occured after the trip ended, prevent export, write error message */
        v_return_value := 0;
        Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq, 'Line '||p_line_cnt||': Activity start date is after the end date for the indicated trip. The trip will be left off of the activity.',p_fa_seq);
      END IF;
    END IF;
  ELSE
    /* activity occured before the trip began, prevent export, write error message */
    v_return_value := 0;
    Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq, 'Line '||p_line_cnt||': Activity start date is before the start date for the indicated trip. The trip will be left off of the activity.',p_fa_seq);

  END IF;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check trip dates.',p_fa_seq);
    RETURN 0;

END check_trip_dates;

/******************************************************************************************************/

FUNCTION check_visit_dates (p_visit_seq IN NUMBER, p_activity_date IN DATE, p_org_id IN VARCHAR2,
                            p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER)
                            RETURN NUMBER IS

  v_start_date           DATE;
  v_end_date             DATE;
  v_after                NUMBER(10);
  v_return_value         NUMBER(1) := 1;

  CURSOR c_visit_dates (p_visit_seq IN NUMBER) IS
      SELECT SSV_START_DATE, SSV_END_DATE FROM SIM_STATION_VISITS WHERE SSV_SEQ = p_visit_seq;

BEGIN
  /* kms - function that makes sure the activity occured between the start and end dates of the visit.
     Called by main import procedure - import_results. */

  OPEN c_visit_dates(p_visit_seq);
  FETCH c_visit_dates INTO v_start_date, v_end_date;
    IF c_visit_dates%NOTFOUND THEN
      CLOSE c_visit_dates;
      RETURN 0;
    END IF;
  CLOSE c_visit_dates;

  /* added for STORET interface bug where some years have been entered as 0001 */
  IF TO_CHAR(p_activity_date,'RRRR') LIKE '20__' THEN
    IF TO_CHAR(v_start_date,'RRRR') LIKE '00__' THEN
      v_start_date := TO_DATE(TO_CHAR(v_start_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_start_date,'RRRR'),2,3),'MM-DD-RRRR');
    END IF;
  END IF;

  /* make sure activity date is within visit date range */
  v_after := v_start_date - p_activity_date;  --will be negative if activity after start date

  /* activity occured before the visit began, prevent export, write error message */
  IF v_after > 0 THEN
    v_return_value := 0;
    Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq, 'Line '||p_line_cnt||': Activity start date is before the start date for the indicated station visit. The station visit will be left off of the activity.',p_fa_seq);
  END IF;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check visit dates.',p_fa_seq);
    RETURN 0;

END check_visit_dates;

/*****************************************************************************************************/

FUNCTION check_group_type_med(p_group_pk IN Sim_import_result_extras.t_storet_pk_type,
                              p_type IN VARCHAR2, p_medium IN VARCHAR2,
                              p_intent_name IN VARCHAR2, p_bio_rslt_type_name IN VARCHAR2,
                              p_community_is_nbr IN NUMBER, p_community_org_id IN VARCHAR2,
                              p_fa_seq IN NUMBER, p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER,
                              p_HA_group OUT NUMBER)
                              RETURN NUMBER IS

  v_group_type              VARCHAR2(60);
  v_group_medium            VARCHAR2(60);
  v_act_type                VARCHAR2(13);
  v_act_medium              VARCHAR2(20);
  v_group_community         VARCHAR2(40);
  v_act_community           VARCHAR2(40);
  v_group_intent            VARCHAR2(20);
  v_group_res_type          VARCHAR2(32);
  v_return_value            NUMBER(1);
  v_hab_ass                 VARCHAR2(1);
  v_user_def                VARCHAR2(1);
  v_HA                      NUMBER(1);

  CURSOR c_get_group_type_med(p_group_pk IN Sim_import_result_extras.t_storet_pk_type) IS
    SELECT FLD_ACT_TYPE_NM, MEDIUM_TYPE_NAME, INTENT_TYPE_NAME, RES_GRP_TYPE_NM,
           COMMUNITY_NAME
      FROM TSRCHGRP
      WHERE TSRCHGRP_IS_NUMBER = p_group_pk.is_number AND TSRCHGRP_ORG_ID = p_group_pk.org_id;

  CURSOR c_get_community_name(p_community_is_nbr IN NUMBER, p_community_or_id  IN VARCHAR2) IS
    SELECT FIELD_VALUE FROM TSMPRMVL
      WHERE TSMPRMVL_IS_NUMBER = p_community_is_nbr
        AND TSMPRMVL_ORG_ID = p_community_org_id;

   CURSOR c_get_hab_assess(p_group_pk IN Sim_import_result_extras.t_storet_pk_type) IS
     SELECT HABITAT_ASSESS_IND, CHAR_DEFINED_IND FROM TSRCHGRP
     WHERE TSRCHGRP_IS_NUMBER = p_group_pk.is_number AND TSRCHGRP_ORG_ID = p_group_pk.org_id;

BEGIN
  /* kms (5/25/01) - function that checks the characterstic group activity type and medium
     against those on the activity being imported.  If they are the same, the group
     is valid and the function returns a 1, otherwise it will return a 2 and an error
     message will be written in the main program. */

  /* sfp (12/04/01) - modifiied this function to compare the additional variables of
     intent, community and bio results type  */

  /* sef (2/25/02) - modified to check for Habitat Assessment

  /* kms - get the activity type and medium from the characteristic group */
  --First see if it's a HA group
  p_HA_group := 0;
  v_HA := 0;
  OPEN c_get_hab_assess(p_group_pk);
  FETCH c_get_hab_assess INTO v_hab_ass, v_user_def;
    IF c_get_hab_assess%FOUND THEN
      IF v_hab_ass = 'Y' AND v_user_def = 'U' THEN
        v_act_type := p_type;

        IF RTRIM(v_act_type) != 'Field Msr/Obs' THEN
          v_return_value := 2;
          Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Habitat Assessment Characteristic must have an Activity Type of Field Msr/Obs.',p_fa_seq);
        ELSE
          v_return_value := 1;
        END IF;
        p_HA_group := 1;
        v_HA := 1;
      ELSIF v_hab_ass = 'Y' AND v_user_def = 'S' THEN
        v_return_value := 1;
        p_HA_group := 2;        
        v_HA := 1;
      ELSE
        p_HA_group := 0;
        v_HA := 2;
      END IF;
    ELSE
      p_ha_group := 0;
      v_HA := 2;
      v_return_value := 2;
      Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Error getting characteristic group activity type and medium - unable to compare these to values for activity.',p_fa_seq);
    END IF;

  CLOSE c_get_hab_assess;

  IF v_HA = 0 THEN  --go ahead and check the type and medium

    OPEN c_get_group_type_med(p_group_pk);
    FETCH c_get_group_type_med INTO v_group_type, v_group_medium, v_group_intent, v_group_res_type,
        v_group_community;
      IF c_get_group_type_med%FOUND THEN
        v_act_type := p_type;
        v_act_medium := p_medium;
        IF p_community_is_nbr IS NOT NULL THEN
         OPEN c_get_community_name(p_community_is_nbr, p_community_org_id);
         FETCH c_get_community_name INTO v_act_community;
         CLOSE c_get_community_name;
        ELSE
          v_act_community := NULL;
        END IF;

      /* kms - compare group and activity medium and type */
      /* bka - added check for Trip QC that allows 'Sample' char groups for activities of type Trip QC */
        IF RTRIM(v_act_type) = 'Trip QC' THEN
          IF RTRIM(v_group_type) != 'Sample' OR RPAD(v_group_medium,20) != RPAD(v_act_medium,20) THEN
            v_return_value := 2;
            Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Characteristic group does not have the same activity type and medium as the activity for this result.',p_fa_seq);
          ELSE
            v_return_value := 1;
          END IF;
        ELSIF  RTRIM(v_group_type) != RTRIM(v_act_type)
          OR RTRIM(v_group_medium) != RTRIM(v_act_medium)
          OR RTRIM(NVL(v_group_intent,'X')) != RTRIM(NVL(p_intent_name,'X'))
          OR RTRIM(NVL(v_group_community,'X')) != RTRIM(NVL(v_act_community,'X'))
          OR RTRIM(NVL(v_group_res_type,'X'))  != RTRIM(NVL(p_bio_rslt_type_name,'X'))  THEN
            v_return_value := 2;
            Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Characteristic group does not have the same activity type, medium, intent, community, or bio result type as the activity for this result.',p_fa_seq);
        ELSE
          v_return_value := 1;
        END IF;
      ELSE
        v_return_value := 2;
        Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Error getting characteristic group activity type and medium - unable to compare these to values for activity.',p_fa_seq);
      END IF;
    CLOSE c_get_group_type_med;
  END IF;
  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check group type med.',p_fa_seq);
    RETURN 0;

END check_group_type_med;

/******************************************************************************************************/

FUNCTION check_chardet_comp(p_char_is_nbr IN NUMBER, p_char_org_id IN VARCHAR2,
                            p_anlpr_is_nbr IN NUMBER, p_uom_is_nbr IN NUMBER,
                            p_uom_org_id IN VARCHAR2, p_fraction IN NUMBER,
                            p_value_type IN VARCHAR2, p_units_ok IN BOOLEAN,
                            p_bio_rslt_type_name IN VARCHAR2,
                            p_fa_seq IN NUMBER, p_line_cnt IN NUMBER,
                            p_import_log_seq IN NUMBER,p_char_result_type IN VARCHAR2)
                            RETURN NUMBER IS

  v_proc_req                VARCHAR2(1);
  v_samp_req                VARCHAR2(1);
  v_uom_type                VARCHAR2(5);
  v_uom_uom_type            VARCHAR2(5);
  v_char                    VARCHAR2(60);
  v_return_value            NUMBER(1) := 1;

  e_bad_char_pk             EXCEPTION;

  CURSOR c_get_char_reqs(p_char_is_number IN NUMBER, p_char_org_id IN VARCHAR2) IS
    SELECT PROC_REQ_IND_CD, SAMP_FRAC_REQ_CD, UOM_TYPE, DISPLAY_NAME FROM TSRCHAR
      WHERE TSRCHAR_IS_NUMBER = p_char_is_number AND TSRCHAR_ORG_ID = RPAD(p_char_org_id, 8);

  CURSOR c_get_uom_type(p_uom_is_nbr IN NUMBER, p_uom_org_id IN VARCHAR2) IS
    SELECT UOM_TYPE FROM TSRUOM WHERE TSRUOM_IS_NUMBER = p_uom_is_nbr
      AND TSRUOM_ORG_ID = p_uom_org_id;


BEGIN
  /* kms - function that checks characteristic requirements against what has been provided
     for the result being imported */

  /* get the requirements for this result's characteristic in order to make sure that they have
     been met */
  OPEN c_get_char_reqs(p_char_is_nbr, p_char_org_id);
    FETCH c_get_char_reqs INTO v_proc_req, v_samp_req, v_uom_type, v_char;
    IF c_get_char_reqs%NOTFOUND THEN
      /* if nothing found then char pk is bogus */
      CLOSE c_get_char_reqs;
      RAISE e_bad_char_pk;
    END IF;
  CLOSE c_get_char_reqs;

  /* make sure value type is not null  */
  IF p_value_type IS NULL AND p_char_result_type NOT IN ('TEXT') THEN  -- dcl If the char type is 'Text' then it's  OK not to have a value type
    v_return_value := 5;
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Warning: Line '||p_line_cnt||': Value type is required for all results. A value of "Actual" will be assigned.',p_fa_seq);
  END IF;

  /* check field/lab procedure */
  IF v_proc_req = 'Y' AND p_anlpr_is_nbr IS NULL THEN
    v_return_value := 2;
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Field/Lab Procedure has not been provided and is required for results with the characteristic '||v_char,p_fa_seq);
  END IF;

  /* check sample fraction */
  IF v_samp_req = 'Y' AND p_fraction IS NULL THEN
    v_return_value := 2;
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Sample Fraction has not been provided and is required for results with the characteristic '||v_char,p_fa_seq);
  END IF;

  /* check units and their type */
  IF v_uom_type = 'NONE ' AND p_uom_is_nbr IS NOT NULL THEN
    /* kms - units are not required, but have been provided, issue a warning. */
    IF p_uom_is_nbr != 1 THEN
      Sim2_Import_Pkg.write_error(p_import_log_seq, 'Warning: Line '||p_line_cnt||': Characteristic, '||v_char||', does not require a unit of measure but one has been provided.',p_fa_seq);
    END IF;
  ELSIF v_uom_type = 'NONE ' AND p_uom_is_nbr IS NULL THEN
    v_return_value := 4;
  ELSIF NOT p_units_ok AND p_uom_is_nbr IS NOT NULL THEN
    /* kms (5/29/01) - units should not be provided since this result is a
       non-detect (or similar).  Warn user that units will be removed from the
       result and set flag to do so. */
    v_return_value := 3;
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Warning: Line '||p_line_cnt||': Result Value has not been reported, unit of measure removed from result.',p_fa_seq);

  ELSIF v_uom_type != 'NONE ' AND p_uom_is_nbr IS NULL AND p_units_ok THEN
    /* unit of measure is required but has not been provided */
    v_return_value := 2;
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': Unit of Measure has not been provided and is required for results with the characteristic: '||v_char,p_fa_seq);

  ELSIF v_uom_type != 'NONE ' AND p_units_ok THEN
    /* if unit of measure is required and has been provided, make sure acode and char
       uom types match */
    OPEN c_get_uom_type(p_uom_is_nbr, p_uom_org_id);
    FETCH c_get_uom_type INTO v_uom_uom_type;
      IF c_get_uom_type%NOTFOUND THEN
        /* shouldn't happen since we have an ac_seq and an is number */
        RAISE_APPLICATION_ERROR(-20100,' - c_get_uom_type did not find a value.');
      ELSE
        IF    v_uom_uom_type LIKE 'TAX%' AND p_bio_rslt_type_name = 'Single Taxon Frequency Classes' THEN
              v_return_value := 1;
        END IF;
      END IF;
    CLOSE c_get_uom_type;

  END IF;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
  Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in check_chardet_comp.',p_fa_seq);
  RETURN 0;

END check_chardet_comp;

/******************************************************************************************************/

FUNCTION check_type_category_match(p_type IN VARCHAR2, p_cat IN VARCHAR2,
                                   p_fa_seq IN NUMBER,p_line_cnt IN NUMBER, p_import_log_seq IN NUMBER,
                                   p_habitat OUT NUMBER)
                                   RETURN NUMBER IS

  v_type_name                      SIM_IMP_ALLOW_VALUES.SIA_ITEM%TYPE;
  v_cat_name                       VARCHAR2(30);
  v_cat_ok                         NUMBER(1) := 0;
  e_bad_cat                        EXCEPTION;

BEGIN
  /* kms - function that makes sure category is valid for the activity type */

  /* get activity type */
  v_type_name := p_type;

  /* get activity category */
  v_cat_name := p_cat;

  p_habitat := 0;
  /* make sure category fits type */
  IF v_type_name = 'Sample' THEN

    IF v_cat_name = 'Routine Sample'
      OR v_cat_name = 'Field Replicate/Duplicate'
      OR v_cat_name = 'Depletion Replicate'
      OR v_cat_name = 'Integrated Time Series'
      OR v_cat_name = 'Integrated Flow Proportioned'
      OR v_cat_name = 'Integrated Horizontal Profile'
      OR v_cat_name = 'Integrated Vertical Profile'
      OR v_cat_name = 'Integrated Cross-Sectional Pro'
      OR v_cat_name = 'Composite w/o Parents'
      OR v_cat_name = 'Composite-with Parents'
      OR v_cat_name = 'Created from Sample'
      OR v_cat_name = 'Field Spike'
      OR v_cat_name = 'Field Equipment Rinsate Blank'
      OR v_cat_name = 'Field Blank'
      OR v_cat_name = 'Field Subsample'
      OR v_cat_name = 'Field Split'
      OR v_cat_name = 'Field Surrogate Spike'
      OR v_cat_name = 'Field Ambient Conditions Blank'
      THEN

      v_cat_ok := 1;
    END IF;

  ELSIF v_type_name = 'Field Msr/Obs' THEN

    IF v_cat_name = 'Routine Msr/Obs'
      OR v_cat_name = 'Replicate Msr/Obs'
      OR v_cat_name = 'Routine Habitat Assessment'
      OR v_cat_name = 'Replicate Habitat Assessment'
      OR v_cat_name = 'Field Calibration Check'
      OR v_cat_name = 'Portable Data Logger'
      THEN
      v_cat_ok := 1;
    END IF;
    IF v_cat_name = 'Routine Habitat Assessment'
      OR v_cat_name = 'Replicate Habitat Assessment' THEN
      p_habitat := 1;
    END IF;

  ELSIF v_type_name = 'Trip QC' THEN

    IF v_cat_name = 'Trip Blank'
      OR v_cat_name = 'Trip Reagent Blank'
      OR v_cat_name = 'Trip Equipment Blank'
      OR v_cat_name = 'Trip Pre-Preserv Blk'
      OR v_cat_name = 'Trip Post-Preserv Blk'
      OR v_cat_name = 'Trip Field Spike'
      OR v_cat_name = 'Trip Reference Sample'
      OR v_cat_name = 'Trip Bottle Blank'
      OR v_cat_name = 'Trip Perform Eval Sample'
      OR v_cat_name = 'Ship Container Temp Blk'
      OR v_cat_name = 'Trip Storage Blank'
      OR v_cat_name = 'Trip Calibration Blank'
      OR v_cat_name = 'Trip Control Blank'
      THEN
      v_cat_ok := 1;
    END IF;

  END IF;

  IF v_cat_ok != 1 THEN
    RAISE e_bad_cat;
  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_bad_cat THEN
    RETURN 2;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(p_import_log_seq, 'Line '||p_line_cnt||': '||SQLERRM||' - in get check type category match.',p_fa_seq);
    RETURN 0;

END check_type_category_match;

/******************************************************************************************************/

end Sim_import_result_checks;
/
