CREATE OR REPLACE PACKAGE SIM_IMPORT_RESULT_EXTRAS IS
  /* SIM 200 */

  /* Modified 7/16/02 pwh - Added the v_userid_code variable, set it to 'SIM122  ' and inserted this variable
     into the D_USERID_CODE column in the STORET tables.
  */
  /*
  Modified 3/12/02 by SF. If two ORGS have field/lab procedures with the same name SIM
  only finds the first one. So changed get_analproc_is_nbr so it first looks in the home
  ORG, then in ZZZZZZZZ for the procedure.

  Modified 2/26/02 by SF to add a function to get Habitat Assessment Row ID
  */
  -- Author  : SUSANF
  -- Created : 12/20/2001 2:24:05 PM
  -- Purpose : To store many small "get" functions originally in the main result import package

  -- Public type declarations
  TYPE T_STORET_PK_TYPE IS RECORD(
    IS_NUMBER NUMBER(12),
    ORG_ID    VARCHAR2(8));

  TYPE COL_TABLE_TYPE IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;

  TYPE COL_TABLE_NUM_TYPE IS TABLE OF NUMBER(8) INDEX BY BINARY_INTEGER;

  -- Public function and procedure declarations
  FUNCTION ISSUE_NOT_USED_WARNING(P_ITEM_NAME      IN VARCHAR2,
                                  P_LINE_CNT       IN NUMBER,
                                  P_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION GET_COMMUNITY_IS_NBR(P_COMMUNITY_NAME IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION VALIDATE_LOCATION(P_ITEM           IN VARCHAR2,
                             V_POINT_TYPE     IN VARCHAR2,
                             V_WHAT           IN VARCHAR2,
                             V_STATION_IS_NBR IN NUMBER,
                             V_STATION_ORG_ID IN VARCHAR2,
                             P_ACTIVITY_SEQ   IN NUMBER,
                             P_LINE_CNT       IN NUMBER,
                             P_IMPORT_LOG_SEQ IN NUMBER,
                             P_TSMALP_IS_NBR  OUT NUMBER,
                             P_TSMALP_ORG_ID  OUT VARCHAR2) RETURN NUMBER;

  FUNCTION VALIDATE_ALVAL_ITEM(P_ITEM_NAME      IN VARCHAR2,
                               P_ORIG_ITEM      IN VARCHAR2,
                               P_IMP_CFG_SEQ    IN NUMBER,
                               P_COL_SEQ        IN NUMBER,
                               P_FA_SEQ         IN NUMBER,
                               V_IMPORT_LOG_SEQ IN NUMBER,
                               V_LINE_CNT       IN NUMBER) RETURN NUMBER;

  FUNCTION VALIDATE_PV_RESULT(P_ORIG_ITEM      IN VARCHAR2,
                              P_IMP_CFG_SEQ    IN NUMBER,
                              P_CHAR_IS_NBR    IN NUMBER,
                              P_FA_SEQ         IN NUMBER,
                              P_OK             OUT BOOLEAN,
                              V_IMPORT_LOG_SEQ IN NUMBER,
                              V_LINE_CNT       IN NUMBER) RETURN VARCHAR2;

  FUNCTION VALIDATE_NUMBER(P_ITEM           IN VARCHAR2,
                           P_ITEM_NAME      IN VARCHAR2,
                           P_FORMAT         IN VARCHAR2,
                           P_IMPORT_LOG_SEQ IN NUMBER,
                           P_LINE_CNT       IN NUMBER) RETURN NUMBER;

  FUNCTION VALIDATE_STORET_ITEM(P_ITEM_NAME       IN VARCHAR2,
                                P_ORIG_ITEM       IN VARCHAR2,
                                P_IMP_CFG_SEQ     IN NUMBER,
                                P_COL_SEQ         IN NUMBER,
                                P_PARENT_PK       IN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE,
                                P_ORG_ID          IN VARCHAR2,
                                V_LINE_CNT        IN NUMBER,
                                P_FA_SEQ          IN NUMBER,
                                V_IMPORT_LOG_SEQ  IN NUMBER,
                                P_TRANSLATED_ITEM OUT VARCHAR2)
    RETURN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;

  FUNCTION VALIDATE_FRACTION(P_SAMP_FRAC_NM   IN VARCHAR2,
                             P_IMP_CFG_SEQ    IN NUMBER,
                             P_COL_SEQ        IN NUMBER,
                             P_TSRCHAR_IS_NBR IN NUMBER,
                             P_FA_SEQ         IN NUMBER,
                             V_IMPORT_LOG_SEQ IN NUMBER,
                             V_LINE_CNT       IN NUMBER,
                             V_TSMPRMVL_KEY   OUT T_STORET_PK_TYPE)
    RETURN VARCHAR2;

  FUNCTION GET_INSTRUCTIONS(P_IMP_CFG_SEQ      IN NUMBER,
                            P_INSTRUCTION_TYPE IN VARCHAR2,
                            P_FA_SEQ           IN NUMBER,
                            V_IMPORT_LOG_SEQ   IN NUMBER,
                            V_LINE_CNT         IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_TAXON_IS_NBR(P_CHAR_NAME      IN VARCHAR2,
                            P_FA_SEQ         IN NUMBER,
                            P_LINE_CNT       IN NUMBER,
                            P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_LIFE_IS_NBR(P_LIFE_NAME      IN VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;
  FUNCTION GET_SEX_IS_NBR(P_SEX_NAME       IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE;

  FUNCTION GET_BIOPT_IS_NBR(P_BIOPT_NAME     IN VARCHAR2,
                            P_FA_SEQ         IN NUMBER,
                            P_LINE_CNT       IN NUMBER,
                            P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_TSMALP_IS_NBR(P_TYPE_CODE       IN VARCHAR2,
                             P_SEQUENCE_NUMBER IN NUMBER,
                             P_TSMSTATN_IS_NBR IN NUMBER,
                             P_TSMSTATN_ORG_ID IN VARCHAR2,
                             P_LINE_CNT        IN NUMBER,
                             P_IMPORT_LOG_SEQ  IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_ITEM_VALUE(P_ALVAL_SEQ      IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_DELIMITER(P_IMP_CFG_SEQ    IN NUMBER,
                         P_LINE_CNT       IN NUMBER,
                         P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_ITEM_POS(P_SICC_SEQ       IN NUMBER,
                        P_POS_TABLE      OUT COL_TABLE_NUM_TYPE,
                        P_DEFAULT_TABLE  OUT COL_TABLE_TYPE,
                        P_COL_CNT        IN OUT NUMBER,
                        P_LINE_CNT       IN NUMBER,
                        P_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION GET_PROJ_IS_NBR(P_PROJ           VARCHAR2,
                           P_ORG_ID         VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_LAB_IS_NBR(P_LAB_ID         IN VARCHAR2,
                          P_ORG_ID         IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE;

  FUNCTION GET_GEAR_IS_NBR(P_GEAR           IN VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_GEAR_CFG_IS_NBR(P_GEAR_CFG       IN VARCHAR2,
                               P_GEAR_PK        IN T_STORET_PK_TYPE,
                               P_ORG_ID         IN VARCHAR2,
                               P_FA_SEQ         IN NUMBER,
                               P_LINE_CNT       IN NUMBER,
                               P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_SAMPLE_PRES_IS_NBR(P_PROC           IN VARCHAR2,
                                  P_ORG_ID         IN VARCHAR2,
                                  P_FA_SEQ         IN NUMBER,
                                  P_LINE_CNT       IN NUMBER,
                                  P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_GROUP_IS_NBR(P_GROUP_ID       IN VARCHAR2,
                            P_ORG_ID         IN VARCHAR2,
                            P_FA_SEQ         IN NUMBER,
                            P_LINE_CNT       IN NUMBER,
                            P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_ROW_IS_NBR(P_ROW_ID         IN VARCHAR2,
                          P_GROUP_KEY      IN T_STORET_PK_TYPE,
                          P_ORG_ID         IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE;

  FUNCTION GET_HA_ROW_IS_NBR(P_ROW_ID         IN VARCHAR2,
                             P_GROUP_KEY      IN T_STORET_PK_TYPE,
                             P_FA_SEQ         IN NUMBER,
                             P_LINE_CNT       IN NUMBER,
                             P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_HA_DEFAULT(V_TSRHCSC_PK     IN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE,
                          P_FA_SEQ         IN NUMBER,
                          V_IMPORT_LOG_SEQ IN NUMBER,
                          V_LINE_CNT       IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_LAB_SAMPLE_PREP_PROCEDURE(P_LABPREP_ID     IN VARCHAR2,
                                         P_LABPREP_SOURCE IN VARCHAR2,
                                         P_ORG_ID         IN VARCHAR2,
                                         P_IMP_CFG_SEQ    IN NUMBER,
                                         P_COL_SEQ        IN NUMBER,
                                         P_IMPORT_LOG_SEQ IN NUMBER,
                                         P_LINE_CNT       IN NUMBER,
                                         P_FA_SEQ         IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_FIELD_LAB_PROCEDURE(P_PROC_ID        IN VARCHAR2,
                                   P_SOURCE         IN VARCHAR2,
                                   P_ORG_ID         IN VARCHAR2,
                                   P_IMP_CFG_SEQ    IN NUMBER,
                                   P_COL_SEQ        IN NUMBER,
                                   P_IMPORT_LOG_SEQ IN NUMBER,
                                   P_LINE_CNT       IN NUMBER,
                                   P_FA_SEQ         IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_PROCEDURE_IS_NBR(P_PROC           IN VARCHAR2,
                                P_ORG_ID         IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_STATION_IS_NBR(P_STATION_ID     IN VARCHAR2,
                              P_ORG_ID         IN VARCHAR2,
                              P_FA_SEQ         IN NUMBER,
                              P_LINE_CNT       IN NUMBER,
                              P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_PERSONNEL_IS_NBR(P_CONTACT_ID     IN VARCHAR2,
                                P_ORG_ID         IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_REMARK_CODES_IS_NUM(P_SHORT_NAME     IN VARCHAR2,
                                   P_ORG_ID         IN VARCHAR2,
                                   P_FA_SEQ         IN NUMBER,
                                   P_LINE_CNT       IN NUMBER,
                                   P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_CHAR_IS_NBR(P_CHAR_NAME      IN VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER,
                           FROMWHERE        IN VARCHAR2)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_ANALPROC_IS_NBR(P_PROCEDURE_ID IN VARCHAR2,
                               P_ORG_ID              IN VARCHAR2,
                               P_FA_SEQ              IN NUMBER,
                               P_LINE_CNT            IN NUMBER,
                               P_IMPORT_LOG_SEQ      IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_LABPREP_IS_NBR(P_PREPARATION_ID IN VARCHAR2,
                              P_ORG_ID                IN VARCHAR2,
                              P_FA_SEQ                IN NUMBER,
                              P_LINE_CNT              IN NUMBER,
                              P_IMPORT_LOG_SEQ        IN NUMBER)
    RETURN T_STORET_PK_TYPE;

  FUNCTION GET_TRIP_ID(P_INSTR_CODE      IN VARCHAR2,
                       P_START_DATE      IN DATE,
                       P_TRIP_START_DATE OUT DATE,
                       P_TRIP_END_DATE   OUT DATE,
                       P_ORG_ID          IN VARCHAR2,
                       P_FA_SEQ          IN NUMBER,
                       P_LINE_CNT        IN NUMBER,
                       P_IMPORT_LOG_SEQ  IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_VISIT_START_DATE(P_VISIT_KEY      IN T_STORET_PK_TYPE,
                                P_TRIP_SEQ       IN NUMBER,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER) RETURN DATE;

  FUNCTION GET_VISIT_SEQ(
           P_VISIT_KEY         OUT T_STORET_PK_TYPE,
           P_STATION_KEY       IN T_STORET_PK_TYPE,
           P_VISIT_START_DATE  IN DATE,
           P_VISIT_ID          IN VARCHAR2,
           P_TRIP_SEQ          IN NUMBER,
           P_FA_SEQ            IN NUMBER,
           P_LINE_CNT          IN NUMBER,
           P_IMPORT_LOG_SEQ    IN NUMBER) 
           RETURN NUMBER;
    -- jah 3-8-2004 Added Parm P_VISIT_START_Date

  FUNCTION GET_ORG_INFO(P_ORG_ID IN VARCHAR2, P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN NUMBER;

  FUNCTION GET_ALLOWABLE_VALUE(P_ALVAL_SEQ      IN NUMBER,
                               P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2;

  FUNCTION CREATE_SIM_STATION_VISIT(P_VISIT_REC      IN SIM_STATION_VISITS%ROWTYPE,
                                    V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_SIM_TRIP(
    P_TRIP_REC       IN OUT SIM_TRIPS%ROWTYPE,
    P_FA_SEQ         IN NUMBER,
    V_IMPORT_LOG_SEQ IN NUMBER) 
    RETURN NUMBER;

  FUNCTION CREATE_FIELD_ACTIVITY(P_FA_REC         IN SIM_FIELD_ACTIVITIES%ROWTYPE,
                                 V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_FIELD_ACTIVITY_PARENT(
           p_fap_rec        IN SIM_FIELD_ACTIVITY_PARENT%ROWTYPE,
           p_import_log_seq IN NUMBER)
           RETURN NUMBER;
           
  FUNCTION CREATE_RCI(P_RCI_REC        IN SIM_RESULT_CLASS_INDICATOR%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_BRGI(P_BRGI_REC       IN SIM_BIO_RESULTS_GRP_INDIVIDUAL%ROWTYPE,
                       V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_BRG(P_BRG_REC        IN SIM_BIO_RESULTS_GRP%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_SAMPLE(P_SAMPLE_REC     IN SIM_SAMPLE%ROWTYPE,
                         V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_TOD(P_TOD_REC        IN SIM_TRAWL_OPS_DET%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_STL(P_STL_REC        IN SIM_STATION_LOCATIONS%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_AAL(P_AAL_REC        IN SIM_ACTUAL_ACTIVITY_LOC%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_TRAP_NET(P_STN_REC        IN SIM_TRAP_NET%ROWTYPE,
                           V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CREATE_RESULT(P_RESULT_REC     IN SIM_RESULTS%ROWTYPE,
                         V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER;

  FUNCTION CHECK_STORET_TRANSLATION(P_CODE IN VARCHAR2,
                                    P_IMP_CFG_SEQ IN NUMBER,
                                    P_COL_SEQ IN NUMBER,
                                    P_TRANS_EXIST OUT BOOLEAN,                                  
                                    V_IMPORT_LOG_SEQ IN NUMBER,
                                    V_LINE_CNT IN NUMBER,
                                    V_CONVERTED_ITEM OUT VARCHAR2,
                                    p_activity_seq IN NUMBER) RETURN T_STORET_PK_TYPE;
  
  FUNCTION CHECK_TRANSLATIONS(P_CODE           IN VARCHAR2,
                              P_IMP_CFG_SEQ    IN NUMBER,
                              P_COL_SEQ        IN NUMBER,
                              P_TRANS_EXIST    OUT BOOLEAN,
                              V_IMPORT_LOG_SEQ IN NUMBER,
                              V_LINE_CNT       IN NUMBER,
                              p_activity_seq   IN NUMBER) RETURN VARCHAR2;

  PROCEDURE CREATE_TRANS_RECORD(P_CODE           IN VARCHAR2,
                                P_IMP_CFG_SEQ    IN NUMBER,
                                P_COL_SEQ        IN NUMBER,
                                P_FA_SEQ         IN NUMBER,
                                V_IMPORT_LOG_SEQ IN NUMBER,
                                V_LINE_CNT       IN NUMBER);

  FUNCTION CHECK_ALLOWABLE_VALUES(P_ITEM           IN VARCHAR2,
                                  P_IMP_CFG_SEQ    IN NUMBER,
                                  P_COL_SEQ        IN NUMBER,
                                  P_FA_SEQ         IN NUMBER,
                                  V_IMPORT_LOG_SEQ IN NUMBER,
                                  V_LINE_CNT       IN NUMBER) RETURN NUMBER;

  FUNCTION GET_UOM_IS_NBR(P_UOM_NAME       IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          V_IMPORT_LOG_SEQ IN NUMBER,
                          V_LINE_CNT       IN NUMBER)
    RETURN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;

  FUNCTION UPDATE_RESULT(P_RESULT           IN VARCHAR2,
                         P_CHAR_IS_NBR      IN NUMBER,
                         P_UNITS_OK         OUT BOOLEAN,
                         P_RESULT_OK        OUT BOOLEAN,
                         P_IMP_CFG_SEQ      IN NUMBER,
                         P_FA_SEQ           IN NUMBER,
                         V_CHAR_RESULT_TYPE IN VARCHAR2,
                         V_IMPORT_LOG_SEQ   IN NUMBER,
                         V_LINE_CNT         IN NUMBER,
                         V_DETECT           IN VARCHAR2,
                         V_DETECT_CHECK     OUT NUMBER) RETURN VARCHAR2;

END SIM_IMPORT_RESULT_EXTRAS;
/
CREATE OR REPLACE PACKAGE BODY SIM_IMPORT_RESULT_EXTRAS IS

  /********************************************************************/
  FUNCTION ISSUE_NOT_USED_WARNING(P_ITEM_NAME      IN VARCHAR2,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

  BEGIN
    SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                'Warning: Line ' || P_LINE_CNT || ': ' ||
                                 P_ITEM_NAME ||
                                 ' is not currently imported.',
                                NULL);
    RETURN 0;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - issuing warning',
                                  NULL);
      RETURN 1;
  END ISSUE_NOT_USED_WARNING;

  /******************************************************************************************************/

  FUNCTION VALIDATE_SAMPLE_MATRIX(P_MATRIX         IN VARCHAR2,
                                  P_IMP_CFG_SEQ    IN NUMBER,
                                  P_COL_SEQ        IN NUMBER,
                                  P_LINE_CNT       IN NUMBER,
                                  P_IMPORT_LOG_SEQ IN NUMBER,
                                  P_FA_SEQ         IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_TRANS_EXIST    BOOLEAN;
    V_MATRIX_IS_NBR  NUMBER(12);
    V_MATRIX_ORG_ID  VARCHAR2(8);
    V_MATRIX_PK      T_STORET_PK_TYPE;
    V_CONVERTED_ITEM VARCHAR2(80);
    V_INSTR_CODE     VARCHAR2(10);

    CURSOR C_GET_MATRIX(V_CODE IN VARCHAR2) IS
      SELECT TSRMATRX_IS_NUMBER, TSRMATRX_ORG_ID
        FROM TSRMATRX
       WHERE CODE = RPAD(V_CODE, 4);

  BEGIN
    -- kms - validate sample matrix againsts values in STORET TSRMATRX table.
    V_MATRIX_PK.IS_NUMBER := NULL;
    V_MATRIX_PK.ORG_ID    := NULL;
    -- look in translations table to see if it needs to swapped - will return original item or translation
    V_CONVERTED_ITEM := CHECK_TRANSLATIONS(P_MATRIX,
                                           P_IMP_CFG_SEQ,
                                           P_COL_SEQ,
                                           V_TRANS_EXIST,
                                           P_IMPORT_LOG_SEQ,
                                           P_LINE_CNT,
                                           p_fa_seq);

    IF V_CONVERTED_ITEM IS NOT NULL THEN
      OPEN C_GET_MATRIX(V_CONVERTED_ITEM);
      FETCH C_GET_MATRIX
        INTO V_MATRIX_IS_NBR, V_MATRIX_ORG_ID;
      IF C_GET_MATRIX%NOTFOUND THEN

        IF NOT V_TRANS_EXIST THEN
          --check instructions to see if translation record should be created
          V_INSTR_CODE := GET_INSTRUCTIONS(P_IMP_CFG_SEQ,
                                           'RES-TR',
                                           P_FA_SEQ,
                                           P_IMPORT_LOG_SEQ,
                                           P_LINE_CNT);

          IF V_INSTR_CODE = 'RES-TR-01' THEN

            -- if record could not be matched to anything in translations or allowable values tables, create new record
            CREATE_TRANS_RECORD(P_MATRIX,
                                P_IMP_CFG_SEQ,
                                P_COL_SEQ,
                                P_FA_SEQ,
                                P_IMPORT_LOG_SEQ,
                                P_LINE_CNT);
            SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                        'Line ' || P_LINE_CNT ||
                                         ': Sample Matrix, ' ||
                                         V_CONVERTED_ITEM ||
                                         ' is not valid - translation record has been created.',
                                        P_FA_SEQ);
          ELSE
            SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                        'Line ' || P_LINE_CNT ||
                                         ': Sample Matrix, ' ||
                                         V_CONVERTED_ITEM ||
                                         ' is not valid.',
                                        P_FA_SEQ);
          END IF;

        ELSIF V_TRANS_EXIST THEN
          -- item has been translated to an invalid storet value
          SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                      'Line ' || P_LINE_CNT || ': ' ||
                                      P_MATRIX ||
                                      ' has been mapped to an invalid storet value: ' ||
                                      V_CONVERTED_ITEM,
                                      P_FA_SEQ);
        END IF;

      ELSE
        V_MATRIX_PK.IS_NUMBER := V_MATRIX_IS_NBR;
        V_MATRIX_PK.ORG_ID    := V_MATRIX_ORG_ID;
      END IF;
      CLOSE C_GET_MATRIX;

    ELSIF V_TRANS_EXIST AND V_CONVERTED_ITEM IS NULL THEN
      V_MATRIX_PK.IS_NUMBER := NULL;
      V_MATRIX_PK.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Sample Matrix ' || P_MATRIX ||
                                   ' exists in translations table but has not been mapped to a STORET code.',
                                  P_FA_SEQ);
    END IF;

    RETURN V_MATRIX_PK;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in validate sample matrix.',
                                  P_FA_SEQ);
      RETURN NULL;

  END VALIDATE_SAMPLE_MATRIX;

  /******************************************************************************************************/
  FUNCTION VALIDATE_LOCATION(P_ITEM           IN VARCHAR2,
                             V_POINT_TYPE     IN VARCHAR2,
                             V_WHAT           IN VARCHAR2,
                             V_STATION_IS_NBR IN NUMBER,
                             V_STATION_ORG_ID IN VARCHAR2,
                             P_ACTIVITY_SEQ   IN NUMBER,
                             P_LINE_CNT       IN NUMBER,
                             P_IMPORT_LOG_SEQ IN NUMBER,
                             P_TSMALP_IS_NBR  OUT NUMBER,
                             P_TSMALP_ORG_ID  OUT VARCHAR2) RETURN NUMBER IS

    V_RETURN       NUMBER(4);
		V_ID_CHECK     VARCHAR2(15);
    V_ITEM_NUM     NUMBER(4);
		V_WELL_PIPE_ID VARCHAR2(15);

    CURSOR C_FIND_LOCATION(V_LOC_NUM IN NUMBER) IS
      SELECT STL_SEQUENCE_NUMBER
        FROM SIM_STATION_LOCATIONS
       WHERE STL_SEQUENCE_NUMBER = V_LOC_NUM AND
             STL_TYPE_CODE LIKE V_POINT_TYPE || '%' AND
             STL_TSMSTATN0IS_NUMBER = V_STATION_IS_NBR AND
             STL_TSMSTATN0ORG_ID = RPAD(V_STATION_ORG_ID, 8);

    CURSOR C_FIND_WELL(V_WELL_NUM IN VARCHAR2) IS
      SELECT W.SSW_WELL_NO, L.STL_IS_NUMBER, L.STL_ORG_ID
        FROM SIM_STATION_WELLS W, SIM_STATION_LOCATIONS L
       WHERE W.SSW_IS_NUMBER = L.STL_TSMWELL_IS_NUMBER AND
             RTRIM(W.SSW_WELL_NO) = RTRIM(V_WELL_NUM) AND
             L.STL_TYPE_CODE LIKE V_POINT_TYPE || '%' AND
             L.STL_TSMSTATN0IS_NUMBER = V_STATION_IS_NBR AND
             L.STL_TSMSTATN0ORG_ID = RPAD(V_STATION_ORG_ID, 8);

    CURSOR C_FIND_PIPE(V_PIPE_NUM IN VARCHAR2) IS
      SELECT P.ID_NUMBER, L.STL_IS_NUMBER, L.STL_ORG_ID
        FROM TSMPIPE P, SIM_STATION_LOCATIONS L
       WHERE P.TSMPIPE_IS_NUMBER = L.STL_TSMPIPE_IS_NUMBER AND
             P.ID_NUMBER = V_PIPE_NUM AND
             L.STL_TYPE_CODE LIKE V_POINT_TYPE || '%' AND
             L.STL_TSMSTATN0IS_NUMBER = V_STATION_IS_NBR AND
             L.STL_TSMSTATN0ORG_ID = RPAD(V_STATION_ORG_ID, 8);

  BEGIN
    /* Initializing v_return to return the value passed in, 
       v_return will be reset to -1 if p_item is not valid
    */
    v_return := p_item;
    IF V_WHAT = 'SEQ' THEN
      BEGIN
        V_ITEM_NUM := TO_NUMBER(P_ITEM, '9999');
        OPEN C_FIND_LOCATION(V_ITEM_NUM);
        FETCH C_FIND_LOCATION
          INTO V_ID_CHECK;
        IF C_FIND_LOCATION%NOTFOUND THEN
          SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                      'Line ' || P_LINE_CNT ||
                                       ': Station Location,' || P_ITEM || ', ' ||
                                       V_POINT_TYPE ||
                                       ', not found in STORET on this station.',
                                      P_ACTIVITY_SEQ);
          V_RETURN := -1;
        END IF;
        CLOSE C_FIND_LOCATION;
      EXCEPTION
        WHEN OTHERS THEN
            SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                        'Line ' || P_LINE_CNT ||
                                         ': Sequence Number must be a number between 0 and 9999',
                                        P_ACTIVITY_SEQ);
         RETURN - 1;
      END;
    ELSE
      --First check for well
  	  V_WELL_PIPE_ID := RTRIM(P_ITEM);
      OPEN C_FIND_WELL(V_WELL_PIPE_ID);
      FETCH C_FIND_WELL
        INTO V_ID_CHECK, P_TSMALP_IS_NBR, P_TSMALP_ORG_ID;

			IF C_FIND_WELL%NOTFOUND THEN
        V_ITEM_NUM := TO_NUMBER(P_ITEM, '9999');
        OPEN C_FIND_PIPE(V_ITEM_NUM);
        FETCH C_FIND_PIPE
          INTO V_ID_CHECK, P_TSMALP_IS_NBR, P_TSMALP_ORG_ID;

				IF C_FIND_PIPE%NOTFOUND THEN
          SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                      'Line ' || P_LINE_CNT ||
                                       ': Well or Pipe ID,' || P_ITEM || ', ' ||
                                       V_POINT_TYPE ||
                                       ', not found in STORET on this station.',
                                      P_ACTIVITY_SEQ);
          V_RETURN := -1;
        END IF;
        CLOSE C_FIND_PIPE;
      END IF;
      CLOSE C_FIND_WELL;
    END IF;

    RETURN V_RETURN;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in validate location',
                                  P_ACTIVITY_SEQ);
      RETURN - 1;
  END VALIDATE_LOCATION;
  /******************************************************************************************************/
  FUNCTION GET_INSTRUCTIONS(P_IMP_CFG_SEQ      IN NUMBER,
                            P_INSTRUCTION_TYPE IN VARCHAR2,
                            P_FA_SEQ           IN NUMBER,
                            V_IMPORT_LOG_SEQ   IN NUMBER,
                            V_LINE_CNT         IN NUMBER) RETURN VARCHAR2 IS

    V_INSTR_CHOICE VARCHAR2(15);

    CURSOR C_GET_CFG_INSTR(P_CFG_SEQ IN NUMBER, P_INSTR_TYPE IN VARCHAR2) IS
      SELECT SIIC_CODE
        FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
       WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ AND
             SII_CODE = P_INSTR_TYPE AND SICI_SICC_SEQ = P_CFG_SEQ;

    CURSOR C_GET_INSTR_DEFAULT(P_INSTR_TYPE IN VARCHAR2) IS
      SELECT SIIC_CODE
        FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
       WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ AND
             SII_CODE = P_INSTR_TYPE AND SIIC_DFLT_FLAG = 1;

  BEGIN

    /* kms - get the instruction choice for this item */
    OPEN C_GET_CFG_INSTR(P_IMP_CFG_SEQ, P_INSTRUCTION_TYPE);
    FETCH C_GET_CFG_INSTR
      INTO V_INSTR_CHOICE;
    IF C_GET_CFG_INSTR%NOTFOUND THEN

      /* no choice could be found, get default choice */
      OPEN C_GET_INSTR_DEFAULT(P_INSTRUCTION_TYPE);
      FETCH C_GET_INSTR_DEFAULT
        INTO V_INSTR_CHOICE;
      CLOSE C_GET_INSTR_DEFAULT;

    END IF;
    CLOSE C_GET_CFG_INSTR;

    RETURN V_INSTR_CHOICE;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
        ' - in get_instructions.',
        P_FA_SEQ
      );
      RETURN NULL;

  END GET_INSTRUCTIONS;

  /******************************************************************************************************/

  FUNCTION GET_COMMUNITY_IS_NBR(P_COMMUNITY_NAME IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_COMMUNITY_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_IS_NBR(P_COMMUNITY_NAME IN VARCHAR2) IS
      SELECT TSMPRMVL_IS_NUMBER, TSMPRMVL_ORG_ID
        FROM TSMPRMVL
       WHERE TABLE_NAME = 'TSRFDACT' AND FIELD_NAME = 'COMMUNITY_NAME' AND
             FIELD_VALUE = P_COMMUNITY_NAME;

  BEGIN
    /*sfp -  function that checks the permitted value in STORET using the processed short_name.
    Called by validate_storet_item for 'Community' */

    OPEN C_GET_IS_NBR(P_COMMUNITY_NAME);
    FETCH C_GET_IS_NBR
      INTO V_COMMUNITY_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_COMMUNITY_KEY.IS_NUMBER := NULL;
      V_COMMUNITY_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Community, ' ||
                                   P_COMMUNITY_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_COMMUNITY_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting community is number',
                                  P_FA_SEQ);
      V_COMMUNITY_KEY.IS_NUMBER := NULL;
      V_COMMUNITY_KEY.ORG_ID    := NULL;
      RETURN V_COMMUNITY_KEY;

  END GET_COMMUNITY_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_TAXON_IS_NBR(P_CHAR_NAME      IN VARCHAR2,
                            P_FA_SEQ         IN NUMBER,
                            P_LINE_CNT       IN NUMBER,
                            P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_CHAR_KEY T_STORET_PK_TYPE;

    -- jah 5-27-2004 Task 477
    -- add OR SEARCH_NAME = UPPER(P_CHAR_NAME) to cursor c_get_is_nbr
    CURSOR C_GET_IS_NBR(P_CHAR_NAME IN VARCHAR2) IS
    SELECT TSRCHAR_IS_NUMBER, TSRCHAR_ORG_ID
    FROM   TSRCHAR
    WHERE  (DISPLAY_NAME = P_CHAR_NAME OR SEARCH_NAME = UPPER(P_CHAR_NAME))
    AND    (D_SCR_TYPE_CD = 'TAXA' OR D_SCR_TYPE_CD = 'VAR ');

  BEGIN
    /* 06/09/03 slp - Note that this function is used for checking 'TAXA' in the Subject Taxon column
    and also the characteristic column for Multi Taxon Population Census activities. */
    /* sfp - function that checks a TAXA characteristic in STORET using the processed display name.
    Called by validate storet item. */
    /* sfp - added restriction for D_SCR_TYPE_CD = 'TAXA'
    sf  - allow 'VAR' for V2 */
    OPEN C_GET_IS_NBR(P_CHAR_NAME);
    FETCH C_GET_IS_NBR
      INTO V_CHAR_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_CHAR_KEY.IS_NUMBER := NULL;
      V_CHAR_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Subject Taxon or Characteristic, ' ||
                                   P_CHAR_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_CHAR_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting subject taxon is number',
                                  P_FA_SEQ);
      V_CHAR_KEY.IS_NUMBER := NULL;
      V_CHAR_KEY.ORG_ID    := NULL;
      RETURN V_CHAR_KEY;

  END GET_TAXON_IS_NBR;

  /******************************************************************************************************/

  FUNCTION GET_SEX_IS_NBR(P_SEX_NAME       IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE IS

    V_SEX_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_IS_NBR(P_SEX_NAME IN VARCHAR2) IS
      SELECT TSRCPV_IS_NUMBER, TSRCPV_ORG_ID
        FROM TSRCPV
       WHERE SHORT_NAME = RPAD(P_SEX_NAME, 12) AND
             TSRCPV.TSRCHAR_IS_NUMBER =
             (SELECT TSRCHAR_IS_NUMBER
                FROM TSRCHAR
               WHERE DISPLAY_NAME = 'Sex (choice list)');
  BEGIN
    /*sfp -  function that checks the permitted value in STORET using the processed name.
    Called by validate_storet_item for 'Sex Name' */

    OPEN C_GET_IS_NBR(P_SEX_NAME);
    FETCH C_GET_IS_NBR
      INTO V_SEX_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_SEX_KEY.IS_NUMBER := NULL;
      V_SEX_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Sex, ' ||
                                   P_SEX_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_SEX_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting sex is number',
                                  P_FA_SEQ);
      V_SEX_KEY.IS_NUMBER := NULL;
      V_SEX_KEY.ORG_ID    := NULL;
      RETURN V_SEX_KEY;

  END GET_SEX_IS_NBR;

  /***********************************************************************************************/

  FUNCTION GET_LIFE_IS_NBR(P_LIFE_NAME      IN VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_LIFE_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_IS_NBR(P_LIFE_NAME IN VARCHAR2) IS
      SELECT TSRCPV_IS_NUMBER, TSRCPV_ORG_ID
        FROM TSRCPV
       WHERE SHORT_NAME = RPAD(P_LIFE_NAME, 12) AND
             TSRCPV.TSRCHAR_IS_NUMBER =
             (SELECT TSRCHAR_IS_NUMBER
                FROM TSRCHAR
               WHERE DISPLAY_NAME = 'Life Stage (choice list)');
  BEGIN
    /*sfp -  function that checks the permitted value in STORET using the processed name.
    Called by validate_storet_item for 'Life Stage Name' */

    OPEN C_GET_IS_NBR(P_LIFE_NAME);
    FETCH C_GET_IS_NBR
      INTO V_LIFE_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_LIFE_KEY.IS_NUMBER := NULL;
      V_LIFE_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Life Stage Name, ' || P_LIFE_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_LIFE_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting life is number',
                                  P_FA_SEQ);
      V_LIFE_KEY.IS_NUMBER := NULL;
      V_LIFE_KEY.ORG_ID    := NULL;
      RETURN V_LIFE_KEY;

  END GET_LIFE_IS_NBR;

  /***********************************************************************************************/

  FUNCTION GET_BIOPT_IS_NBR(P_BIOPT_NAME     IN VARCHAR2,
                            P_FA_SEQ         IN NUMBER,
                            P_LINE_CNT       IN NUMBER,
                            P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_BIOPT_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_IS_NBR(P_BIOPT_NAME IN VARCHAR2) IS
      SELECT TSRBIOPT_IS_NUMBER, TSRBIOPT_ORG_ID
        FROM TSRBIOPT
       WHERE NAME = P_BIOPT_NAME;

  BEGIN
    /*sfp -  function that checks the permitted value in STORET using the processed name.
    Called by validate_storet_item for 'Bio Part' */

    OPEN C_GET_IS_NBR(P_BIOPT_NAME);
    FETCH C_GET_IS_NBR
      INTO V_BIOPT_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_BIOPT_KEY.IS_NUMBER := NULL;
      V_BIOPT_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Bio Part, ' ||
                                   P_BIOPT_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_BIOPT_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting biopt is number',
                                  P_FA_SEQ);
      V_BIOPT_KEY.IS_NUMBER := NULL;
      V_BIOPT_KEY.ORG_ID    := NULL;
      RETURN V_BIOPT_KEY;

  END GET_BIOPT_IS_NBR;

  /**********************************************************************************************/

  FUNCTION GET_TSMALP_IS_NBR(P_TYPE_CODE       IN VARCHAR2,
                             P_SEQUENCE_NUMBER IN NUMBER,
                             P_TSMSTATN_IS_NBR IN NUMBER,
                             P_TSMSTATN_ORG_ID IN VARCHAR2,
                             P_LINE_CNT        IN NUMBER,
                             P_IMPORT_LOG_SEQ  IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_TSMALP_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_IS_NBR IS
      SELECT TSMALP_IS_NUMBER, TSMALP_ORG_ID
        FROM TSMALP
       WHERE TSMSTATN0IS_NUMBER = P_TSMSTATN_IS_NBR AND
             TSMSTATN0ORG_ID = RPAD(P_TSMSTATN_ORG_ID, 8) AND
             TYPE_CODE = RPAD(P_TYPE_CODE, 16) AND
             SEQUENCE_NUMBER = P_SEQUENCE_NUMBER;

  BEGIN
    /*sfp -  function that gets the tsmalp from STORET using the procedure is_nbr
    Called by import_result */

    OPEN C_GET_IS_NBR;
    FETCH C_GET_IS_NBR
      INTO V_TSMALP_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_TSMALP_KEY.IS_NUMBER := NULL;
      V_TSMALP_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': This station Location could not be found in STORET.',
                                  NULL);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_TSMALP_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting tsmalp is number',
                                  NULL);
      V_TSMALP_KEY.IS_NUMBER := NULL;
      V_TSMALP_KEY.ORG_ID    := NULL;
      RETURN V_TSMALP_KEY;

  END GET_TSMALP_IS_NBR;

  /***********************************************************************************************/

  FUNCTION GET_ITEM_VALUE(
           P_ALVAL_SEQ      IN NUMBER,
           P_LINE_CNT       IN NUMBER,
           P_IMPORT_LOG_SEQ IN NUMBER) 
           RETURN VARCHAR2 IS

  V_ITEM_VALUE VARCHAR2(32);

  -- Get the Item Value (i.e., the Name) in a particular record
  CURSOR C_GET_NAME(P_ALVAL_SEQ IN NUMBER) IS
  SELECT SIA_ITEM
  FROM   SIM_IMP_ALLOW_VALUES
  WHERE  SIA_SEQ = P_ALVAL_SEQ;

  /*  This function returns the item value (i.e., the name) 
      in a given record (P_ALVAL_SEQ) in the SIM Allowable 
      Values table.
      
      This function does not output an error message if the 
      calling routine passes an invalid record number, it 
      simply returns NULL. */

BEGIN

    -- try to retrieve the item name
    OPEN C_GET_NAME(P_ALVAL_SEQ);
    FETCH C_GET_NAME INTO V_ITEM_VALUE;
    
    -- did we retrieve the item name?
    IF C_GET_NAME%NOTFOUND THEN
      -- if no
      -- tell calling routine we did not retrieve the name
      V_ITEM_VALUE := NULL;
    END IF;
    CLOSE C_GET_NAME;

    RETURN V_ITEM_VALUE;

  EXCEPTION
 
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
        ' - getting item name',
        NULL
      );
      -- tell calling routine we did not retrieve the name
      RETURN NULL;

  END GET_ITEM_VALUE;

  /******************************************************************************************************/
  FUNCTION GET_DELIMITER(P_IMP_CFG_SEQ    IN NUMBER,
                         P_LINE_CNT       IN NUMBER,
                         P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2 IS

    V_DELIMITER VARCHAR2(1);

    CURSOR C_GET_IMP_CFG(P_SICC_SEQ IN NUMBER) IS
      SELECT SICC_DELIMITER FROM SIM_IMP_CFG WHERE SICC_SEQ = P_SICC_SEQ;

  BEGIN
    /* kms - function that retrieves the delimiter from the import configuration table.  Called by main
    import procedure - import_results. */

    OPEN C_GET_IMP_CFG(P_IMP_CFG_SEQ);
    FETCH C_GET_IMP_CFG
      INTO V_DELIMITER;
    IF C_GET_IMP_CFG%NOTFOUND THEN
      V_DELIMITER := NULL;
    END IF;
    CLOSE C_GET_IMP_CFG;

    RETURN V_DELIMITER;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - Failed to get delimiter.',
                                  NULL);
      RETURN NULL;

  END GET_DELIMITER;

  /******************************************************************************************************/

  FUNCTION GET_ITEM_POS(P_SICC_SEQ       IN NUMBER,
                        P_POS_TABLE      OUT COL_TABLE_NUM_TYPE,
                        P_DEFAULT_TABLE  OUT COL_TABLE_TYPE,
                        P_COL_CNT        IN OUT NUMBER,
                        P_LINE_CNT       IN NUMBER,
                        P_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

    V_IMPORT_TYPE_SEQ NUMBER(10);

    E_BAD_CFG EXCEPTION;

    CURSOR C_GET_IMPORT_TYPE(P_SICC_SEQ IN NUMBER) IS
      SELECT SICC_SID_SEQ FROM SIM_IMP_CFG WHERE SICC_SEQ = P_SICC_SEQ;

    CURSOR C_GET_DFLT_POS(P_TYPE_SEQ IN NUMBER) IS
      SELECT SIC_DFLT_POS FROM SIM_IMP_COLS WHERE SIC_SID_SEQ = P_TYPE_SEQ;

    CURSOR C_GET_COL_POS(P_SICC_SEQ IN NUMBER, P_DFLT_POS IN NUMBER) IS
      SELECT SICDT_POS, SICDT_DEFAULT
        FROM SIM_IMP_CFG_DETS, SIM_IMP_COLS
       WHERE SIC_SEQ = SICDT_SIC_SEQ AND SICDT_SICC_SEQ = P_SICC_SEQ AND
             SIC_DFLT_POS = P_DFLT_POS AND (SICDT_INCLUDED IS NOT NULL OR SICDT_GENERATE IS NOT NULL);

  BEGIN
    /* kms - function to populate the tables that will hold the import configuration information.
    This will populate tables that hold actual position information and default information
    indexed by the default column position.  It will also count the columns in order to insure
    that they match the import configuration.  Called by main import procedure - import_results. */

    /* get the import type in order to limit the columns selected to those that are for results
    and activities */
    OPEN C_GET_IMPORT_TYPE(P_SICC_SEQ);
    FETCH C_GET_IMPORT_TYPE
      INTO V_IMPORT_TYPE_SEQ;
    IF C_GET_IMPORT_TYPE%NOTFOUND THEN
      V_IMPORT_TYPE_SEQ := NULL;
      CLOSE C_GET_IMPORT_TYPE;
      RAISE E_BAD_CFG;
    END IF;
    CLOSE C_GET_IMPORT_TYPE;

    P_COL_CNT := 0;

    /* loop through import columns and assign actual position to a table (p_pos_table) indexed by
    default position */
    FOR V_DFLT_POS IN C_GET_DFLT_POS(V_IMPORT_TYPE_SEQ) LOOP

      /* get items' actual position and default value from configuration table */
      OPEN C_GET_COL_POS(P_SICC_SEQ, V_DFLT_POS.SIC_DFLT_POS);
      FETCH C_GET_COL_POS
        INTO P_POS_TABLE(V_DFLT_POS.SIC_DFLT_POS), P_DEFAULT_TABLE(V_DFLT_POS.SIC_DFLT_POS);
      IF C_GET_COL_POS%NOTFOUND THEN
        /* if the column has not been included in the import configuration, it's position and
        default will be null */
        P_POS_TABLE(V_DFLT_POS.SIC_DFLT_POS) := NULL;
        P_DEFAULT_TABLE(V_DFLT_POS.SIC_DFLT_POS) := NULL;
      ELSE
        P_COL_CNT := P_COL_CNT + 1;
      END IF;
      CLOSE C_GET_COL_POS;

    END LOOP;

    RETURN 1;

  EXCEPTION
    WHEN E_BAD_CFG THEN
      RAISE_APPLICATION_ERROR(-20103, 'Failed to get import sequence.');
      RETURN 0;

    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20103,
                              'Failed to get column positions, item names, and column sequences.');

  END GET_ITEM_POS;

  /******************************************************************************************************/
  FUNCTION GET_PROJ_IS_NBR(P_PROJ           VARCHAR2,
                           P_ORG_ID         VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_PROJ_KEY T_STORET_PK_TYPE;

    CURSOR C_PROJ_IS_NBR(P_PROJ VARCHAR2, P_ORG_ID VARCHAR2) IS
      SELECT TSMPROJ_IS_NUMBER, TSMPROJ_ORG_ID
        FROM TSMPROJ
       WHERE TSMPROJ_ORG_ID = RPAD(P_ORG_ID, 8) AND
             IDENTIFICATION_CD = RPAD(P_PROJ, 8);

  BEGIN
    /* kms - function that gets the project primary key using the processed project id - called by validate storet item*/
    OPEN C_PROJ_IS_NBR(P_PROJ, P_ORG_ID);
    FETCH C_PROJ_IS_NBR
      INTO V_PROJ_KEY;
    IF C_PROJ_IS_NBR%NOTFOUND THEN
      V_PROJ_KEY.IS_NUMBER := NULL;
      V_PROJ_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Project, ' ||
                                   P_PROJ ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_PROJ_IS_NBR;

    RETURN V_PROJ_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get project is number.',
                                  P_FA_SEQ);
      V_PROJ_KEY.IS_NUMBER := NULL;
      V_PROJ_KEY.ORG_ID    := NULL;
      RETURN V_PROJ_KEY;

  END GET_PROJ_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_LAB_IS_NBR(P_LAB_ID         IN VARCHAR2,
                          P_ORG_ID         IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE IS

    V_TSRLAB_KEY T_STORET_PK_TYPE;

    CURSOR C_FIND_LAB(P_LAB_ID IN VARCHAR2, P_ORG_ID IN VARCHAR2) IS
      SELECT TSRLAB_IS_NUMBER, TSRLAB_ORG_ID
        FROM TSRLAB
       WHERE ID_CODE = RPAD(P_LAB_ID, 8) AND
             TSRLAB_ORG_ID = RPAD(P_ORG_ID, 8);

  BEGIN
    /* kms - function that gets the lab primary key using the processed lab id - called by validate storet item */
    OPEN C_FIND_LAB(P_LAB_ID, P_ORG_ID);
    FETCH C_FIND_LAB
      INTO V_TSRLAB_KEY;
    IF C_FIND_LAB%NOTFOUND THEN
      V_TSRLAB_KEY.IS_NUMBER := NULL;
      V_TSRLAB_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Lab ID - ' ||
                                   P_LAB_ID ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_FIND_LAB;

    RETURN V_TSRLAB_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get lab is number.',
                                  P_FA_SEQ);
      V_TSRLAB_KEY.IS_NUMBER := NULL;
      V_TSRLAB_KEY.ORG_ID    := NULL;
      RETURN V_TSRLAB_KEY;

  END GET_LAB_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_GEAR_IS_NBR(P_GEAR           IN VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_GEAR_KEY T_STORET_PK_TYPE;

    CURSOR C_GEAR_IS_NBR IS
      SELECT TSRFLDGR_IS_NUMBER, TSRFLDGR_ORG_ID
        FROM TSRFLDGR
       WHERE ID_CODE = RPAD(P_GEAR, 10);
    /* sf - TSRFLDGR is a STORET table and contains distinct GEAR ID_CODEs.     */

  BEGIN
    /* kms - function that gets the gear primary key using the processed gear id - called by validate storet item */

    OPEN C_GEAR_IS_NBR;
    FETCH C_GEAR_IS_NBR
      INTO V_GEAR_KEY;
    IF C_GEAR_IS_NBR%NOTFOUND THEN
      V_GEAR_KEY.IS_NUMBER := NULL;
      V_GEAR_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Gear, ' ||
                                   P_GEAR ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GEAR_IS_NBR;

    RETURN V_GEAR_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get gear is number.',
                                  P_FA_SEQ);
      V_GEAR_KEY.IS_NUMBER := NULL;
      V_GEAR_KEY.ORG_ID    := NULL;
      RETURN V_GEAR_KEY;

  END GET_GEAR_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_GEAR_CFG_IS_NBR(
           P_GEAR_CFG       IN VARCHAR2,
           P_GEAR_PK        IN T_STORET_PK_TYPE,
           P_ORG_ID         IN VARCHAR2,
           P_FA_SEQ         IN NUMBER,
           P_LINE_CNT       IN NUMBER,
           P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_GEAR_CFG_KEY T_STORET_PK_TYPE;

    CURSOR C_GEAR_CFG_IS_NBR(P_GEAR_CFG IN VARCHAR2, P_GEAR_PK IN T_STORET_PK_TYPE, P_ORG_ID IN VARCHAR2) IS
    SELECT TSRGRCFG_IS_NUMBER, TSRGRCFG_ORG_ID
    FROM   TSRGRCFG
    WHERE  ID_CODE = RPAD(P_GEAR_CFG, 10)
    AND    TSRGRCFG_ORG_ID = RPAD(P_ORG_ID, 8) 
    AND    TSRFLDGR_IS_NUMBER = P_GEAR_PK.IS_NUMBER 
    AND    TSRFLDGR_ORG_ID = RPAD(P_GEAR_PK.ORG_ID, 8);

  BEGIN
  
    /* kms (5/21/01)
       This function gets the gear config primary key using
       the processed gear configuration id. Function
       VALIDATE_STORET_ITEM calls this function. */
       
    -- Try to retrieve the Gear Configuration ID from STORET
    OPEN C_GEAR_CFG_IS_NBR(P_GEAR_CFG, P_GEAR_PK, P_ORG_ID);
    FETCH C_GEAR_CFG_IS_NBR INTO V_GEAR_CFG_KEY;
    
    -- Did we find the Gear Configuration ID in STORET?
    IF C_GEAR_CFG_IS_NBR%NOTFOUND THEN
      -- if no
      /* kms - gear config either does not exist or is not right for this gear */
      
      -- Set the return value to NULL. This tells the caller
      -- that the gear configuration ID is not in STORET.
      V_GEAR_CFG_KEY.IS_NUMBER := NULL;
      V_GEAR_CFG_KEY.ORG_ID    := NULL;
      
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT ||
        ': Gear Configuration, ' || P_GEAR_CFG ||
        ', does not exist for the gear on this activity.',
        P_FA_SEQ
      );
    END IF;
    CLOSE C_GEAR_CFG_IS_NBR;

    RETURN V_GEAR_CFG_KEY;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
        ' - in get gear cfg is number.',
        P_FA_SEQ
      );
      V_GEAR_CFG_KEY.IS_NUMBER := NULL;
      V_GEAR_CFG_KEY.ORG_ID    := NULL;
      RETURN V_GEAR_CFG_KEY;

  END GET_GEAR_CFG_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_SAMPLE_PRES_IS_NBR(P_PROC           IN VARCHAR2,
                                  P_ORG_ID         IN VARCHAR2,
                                  P_FA_SEQ         IN NUMBER,
                                  P_LINE_CNT       IN NUMBER,
                                  P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_PROC_KEY T_STORET_PK_TYPE;

    CURSOR C_PROC_IS_NBR(P_PROC VARCHAR2, P_ORG_ID VARCHAR2) IS
      SELECT TSRSDP_IS_NUMBER, TSRSDP_ORG_ID
        FROM TSRSDP
       WHERE TSRSDP_ORG_ID = RPAD(P_ORG_ID, 8) AND
             ID_CODE = RPAD(P_PROC, 10);

  BEGIN
    /* kms - function that gets the sample preservation primary key using the processed id_code.
    Called by validate storet item */
    OPEN C_PROC_IS_NBR(P_PROC, P_ORG_ID);
    FETCH C_PROC_IS_NBR
      INTO V_PROC_KEY;
    IF C_PROC_IS_NBR%NOTFOUND THEN
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Sample Preservation, ' || P_PROC ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_PROC_IS_NBR;

    RETURN V_PROC_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get sample pres is number.',
                                  P_FA_SEQ);
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
      RETURN V_PROC_KEY;

  END GET_SAMPLE_PRES_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_GROUP_IS_NBR(P_GROUP_ID       IN VARCHAR2,
                            P_ORG_ID         IN VARCHAR2,
                            P_FA_SEQ         IN NUMBER,
                            P_LINE_CNT       IN NUMBER,
                            P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_GROUP_KEY T_STORET_PK_TYPE;

    CURSOR C_GROUP_IS_NBR(P_GROUP IN VARCHAR2) IS
      SELECT TSRCHGRP_IS_NUMBER, TSRCHGRP_ORG_ID
        FROM TSRCHGRP
       WHERE ID_CODE = RPAD(P_GROUP, 8) AND
             TSRCHGRP_ORG_ID = RPAD(P_ORG_ID, 8);

  BEGIN
    /* kms - function that gets the characteristic group primary key using the processed group id - called by validate storet item */
    OPEN C_GROUP_IS_NBR(P_GROUP_ID);
    FETCH C_GROUP_IS_NBR
      INTO V_GROUP_KEY;
    IF C_GROUP_IS_NBR%NOTFOUND THEN
      V_GROUP_KEY.IS_NUMBER := NULL;
      V_GROUP_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Characteristic Group ID, ' ||
                                   P_GROUP_ID ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GROUP_IS_NBR;

    RETURN V_GROUP_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get group is number.',
                                  P_FA_SEQ);
      V_GROUP_KEY.IS_NUMBER := NULL;
      V_GROUP_KEY.ORG_ID    := NULL;
      RETURN V_GROUP_KEY;

  END GET_GROUP_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_ROW_IS_NBR(P_ROW_ID         IN VARCHAR2,
                          P_GROUP_KEY      IN T_STORET_PK_TYPE,
                          P_ORG_ID         IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          P_LINE_CNT       IN NUMBER,
                          P_IMPORT_LOG_SEQ IN NUMBER) RETURN T_STORET_PK_TYPE IS

    V_ROW_ID_KEY T_STORET_PK_TYPE;
    V_COMP_CD    VARCHAR2(1);

    CURSOR C_GET_ROW_ID_IS_NBR(P_ROW_ID IN VARCHAR2, P_GROUP_PK IN T_STORET_PK_TYPE) IS
      SELECT TSRCHDEF_IS_NUMBER, TSRCHDEF_ORG_ID, COMP_IND_CD
        FROM TSRCHDEF, TSRCCGA
       WHERE TSRCHDEF.TSRCCGA_IS_NUMBER = TSRCCGA.TSRCCGA_IS_NUMBER AND
             TSRCHDEF.TSRCCGA_ORG_ID = TSRCCGA.TSRCCGA_ORG_ID AND
             ROW_ID = P_ROW_ID AND
             TSRCCGA.TSRCHGRP_IS_NUMBER = P_GROUP_PK.IS_NUMBER AND
             TSRCCGA.TSRCHGRP_ORG_ID = RPAD(P_GROUP_PK.ORG_ID, 8);

  BEGIN
    /* kms - function that gets the characteristic row primary key using the processed row id - called by validate storet item */
    OPEN C_GET_ROW_ID_IS_NBR(P_ROW_ID, P_GROUP_KEY);
    FETCH C_GET_ROW_ID_IS_NBR
      INTO V_ROW_ID_KEY.IS_NUMBER, V_ROW_ID_KEY.ORG_ID, V_COMP_CD;
    IF C_GET_ROW_ID_IS_NBR%NOTFOUND THEN
      V_ROW_ID_KEY.IS_NUMBER := NULL;
      V_ROW_ID_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Characteristic Row ID, ' || P_ROW_ID ||
                                   ', could not be found in STORET or is not a row in the provided characteristic group.',
                                  P_FA_SEQ);
    ELSE
      IF V_COMP_CD = 'N' THEN
        V_ROW_ID_KEY.IS_NUMBER := NULL;
        V_ROW_ID_KEY.ORG_ID    := NULL;
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                    'Line ' || P_LINE_CNT ||
                                     ': Characteristic Row ID, ' || P_ROW_ID ||
                                     ', is not a complete row.',
                                    P_FA_SEQ);
      END IF;
    END IF;
    CLOSE C_GET_ROW_ID_IS_NBR;

    RETURN V_ROW_ID_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get row_id is number.',
                                  P_FA_SEQ);
      V_ROW_ID_KEY.IS_NUMBER := NULL;
      V_ROW_ID_KEY.ORG_ID    := NULL;
      RETURN V_ROW_ID_KEY;

  END GET_ROW_IS_NBR;
  /******************************************************************************************************/
  FUNCTION GET_HA_ROW_IS_NBR(P_ROW_ID         IN VARCHAR2,
                             P_GROUP_KEY      IN T_STORET_PK_TYPE,
                             P_FA_SEQ         IN NUMBER,
                             P_LINE_CNT       IN NUMBER,
                             P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_ROW_ID_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_HA_ROW_ID_IS_NBR(P_ROW_ID IN VARCHAR2, P_GROUP_PK IN T_STORET_PK_TYPE) IS
      SELECT TSRHCSC_IS_NUMBER, TSRHCSC_ORG_ID
        FROM TSRHCSC
       WHERE TSRCHGRP_IS_NUMBER = P_GROUP_PK.IS_NUMBER AND
             TSRCHGRP_ORG_ID = RPAD(P_GROUP_PK.ORG_ID, 8) AND
             RTRIM(ROW_ID) = P_ROW_ID;

  BEGIN
    /* sf - function that gets the Habatat Assessment characteristic row primary key
    using the processed row id - called by validate storet item */
    OPEN C_GET_HA_ROW_ID_IS_NBR(P_ROW_ID, P_GROUP_KEY);
    FETCH C_GET_HA_ROW_ID_IS_NBR
      INTO V_ROW_ID_KEY.IS_NUMBER, V_ROW_ID_KEY.ORG_ID;
    IF C_GET_HA_ROW_ID_IS_NBR%NOTFOUND THEN
      V_ROW_ID_KEY.IS_NUMBER := NULL;
      V_ROW_ID_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Habitat Assessment Row ID, ' ||
                                   P_ROW_ID ||
                                   ', could not be found in STORET or is not a row in the provided HA characteristic group.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_HA_ROW_ID_IS_NBR;

    RETURN V_ROW_ID_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get_HA_row_id_is_nbr.',
                                  P_FA_SEQ);
      V_ROW_ID_KEY.IS_NUMBER := NULL;
      V_ROW_ID_KEY.ORG_ID    := NULL;
      RETURN V_ROW_ID_KEY;

  END GET_HA_ROW_IS_NBR;
  /******************************************************************************************************/
  FUNCTION GET_PROCEDURE_IS_NBR(P_PROC           IN VARCHAR2,
                                P_ORG_ID         IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_PROC_KEY T_STORET_PK_TYPE;

    CURSOR C_PROC_IS_NBR(P_PROC VARCHAR2, P_ORG_ID VARCHAR2) IS
      SELECT TSRFLDPR_IS_NUMBER, TSRFLDPR_ORG_ID
        FROM TSRFLDPR
       WHERE TSRFLDPR_ORG_ID = RPAD(P_ORG_ID, 8) AND
             ID_CODE = RPAD(P_PROC, 10);

  BEGIN
    /* kms - function that gets the field procedure primary key using the processed procedure id - called by validate storet item */
    OPEN C_PROC_IS_NBR(P_PROC, P_ORG_ID);
    FETCH C_PROC_IS_NBR
      INTO V_PROC_KEY;
    IF C_PROC_IS_NBR%NOTFOUND THEN
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Sample Collection Procedure, ' ||
                                   P_PROC ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_PROC_IS_NBR;

    RETURN V_PROC_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get_procedure_is_nbr.',
                                  P_FA_SEQ);
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
      RETURN V_PROC_KEY;

  END GET_PROCEDURE_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_STATION_IS_NBR(P_STATION_ID     IN VARCHAR2,
                              P_ORG_ID         IN VARCHAR2,
                              P_FA_SEQ         IN NUMBER,
                              P_LINE_CNT       IN NUMBER,
                              P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_STATION_KEY   T_STORET_PK_TYPE;
    V_COMPLETE_FLAG VARCHAR2(1);

    CURSOR C_FIND_STATION(P_STATION_ID IN VARCHAR2, P_ORG_ID IN VARCHAR2) IS
      SELECT TSMSTATN_IS_NUMBER, TSMSTATN_ORG_ID
        FROM TSMSTATN
       WHERE IDENTIFICATION_CD = RPAD(P_STATION_ID, 15) AND
             TSMSTATN_ORG_ID = RPAD(P_ORG_ID, 8);

  BEGIN

    /* kms - function that gets the station primary key using the processed station id - called by validate storet item */
    OPEN C_FIND_STATION(P_STATION_ID, P_ORG_ID);
    FETCH C_FIND_STATION
      INTO V_STATION_KEY.IS_NUMBER, V_STATION_KEY.ORG_ID;

    IF C_FIND_STATION%NOTFOUND THEN
      V_STATION_KEY.IS_NUMBER := NULL;
      V_STATION_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Station, ' ||
                                   P_STATION_ID ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    ELSIF V_COMPLETE_FLAG = 'N' THEN
      V_STATION_KEY.IS_NUMBER := NULL;
      V_STATION_KEY.ORG_ID    := 'XXX';
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Station, ' ||
                                   P_STATION_ID || ', is incomplete.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_FIND_STATION;

    RETURN V_STATION_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                  ' - PROC GET_STATION_IS_NBR, line: ' ||
                                  P_LINE_CNT,
                                  P_FA_SEQ);
      V_STATION_KEY.IS_NUMBER := NULL;
      V_STATION_KEY.ORG_ID    := NULL;
      RETURN V_STATION_KEY;

  END GET_STATION_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_PERSONNEL_IS_NBR(P_CONTACT_ID     IN VARCHAR2,
                                P_ORG_ID         IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_FIRST_NAME   VARCHAR2(30);
    V_LAST_NAME    VARCHAR2(40);
    V_TSMPERSN_KEY T_STORET_PK_TYPE;
    V_ACTIVE       VARCHAR2(1);

    CURSOR C_FIND_CONTACT(P_FIRST_NAME IN VARCHAR2, P_LAST_NAME IN VARCHAR2) IS
      SELECT TSMPERSN_IS_NUMBER, TSMPERSN_ORG_ID, ACTIVE_INDICATR_CD
        FROM TSMPERSN
       WHERE RPAD(FIRST_NAME, 30) = RPAD(P_FIRST_NAME, 30) AND
             RPAD(LAST_NAME, 40) = RPAD(P_LAST_NAME, 40) AND
             TSMPERSN_ORG_ID = RPAD(P_ORG_ID, 8);

  BEGIN
    /* kms - function that gets personnel primary key using the processed personnel name.  Expects to get last
    name first and that it is separated from the first name with a space.  If personnel cannot be found, this
    function will create it in the STORET table.  Called by validate_storet_item. */

    /* divide contact id into first and last names
    - assumes first space is break between first and last names */
    V_LAST_NAME  := RTRIM(SUBSTR(P_CONTACT_ID, 1, INSTR(P_CONTACT_ID, ' ')));
    V_FIRST_NAME := LTRIM(SUBSTR(P_CONTACT_ID, INSTR(P_CONTACT_ID, ' ')));

    OPEN C_FIND_CONTACT(V_FIRST_NAME, V_LAST_NAME);
    FETCH C_FIND_CONTACT
      INTO V_TSMPERSN_KEY.IS_NUMBER, V_TSMPERSN_KEY.ORG_ID, V_ACTIVE;
    IF C_FIND_CONTACT%NOTFOUND THEN
      V_TSMPERSN_KEY.IS_NUMBER := NULL;
      V_TSMPERSN_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Personnel, ' ||
                                   V_FIRST_NAME || ' ' || V_LAST_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
      /* create personnel record in STORET */
    ELSIF V_ACTIVE = 'N' THEN
      V_TSMPERSN_KEY.IS_NUMBER := NULL;
      V_TSMPERSN_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Contact, ' ||
                                   V_FIRST_NAME || ' ' || V_LAST_NAME ||
                                   ', is listed as inactive in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_FIND_CONTACT;

    RETURN V_TSMPERSN_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get personnel is nbr.',
                                  P_FA_SEQ);
      V_TSMPERSN_KEY.IS_NUMBER := NULL;
      V_TSMPERSN_KEY.ORG_ID    := NULL;
      RETURN V_TSMPERSN_KEY;

  END GET_PERSONNEL_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_CPORG_IS_NBR(P_CPORG_NAME     IN VARCHAR2,
                                P_ORG_ID         IN VARCHAR2,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_NAME           VARCHAR2(60);
    V_TSMCPORG_KEY   T_STORET_PK_TYPE;

    CURSOR C_TSMCPORG (P_NAME IN VARCHAR2) IS
    SELECT TSMCPORG_IS_NUMBER, TSMCPORG_ORG_ID
    FROM   TSMCPORG T
    WHERE  NAME = P_NAME;
       
  BEGIN
    V_NAME  := P_CPORG_NAME;

    OPEN C_TSMCPORG(V_NAME);
    FETCH C_TSMCPORG INTO V_TSMCPORG_KEY.IS_NUMBER, V_TSMCPORG_KEY.ORG_ID;
    IF C_TSMCPORG%NOTFOUND THEN
      V_TSMCPORG_KEY.IS_NUMBER := NULL;
      V_TSMCPORG_KEY.ORG_ID := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': Cooperating Organization, ' ||
                                   V_NAME || 
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_TSMCPORG;

    RETURN V_TSMCPORG_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get Cooperating Organization IS_NUMBER.',
                                  P_FA_SEQ);
      V_TSMCPORG_KEY.IS_NUMBER := NULL;
      V_TSMCPORG_KEY.ORG_ID    := NULL;
      RETURN V_TSMCPORG_KEY;

  END GET_CPORG_IS_NBR;


  /******************************************************************************************************/
  FUNCTION GET_REMARK_CODES_IS_NUM(P_SHORT_NAME     IN VARCHAR2,
                                   P_ORG_ID         IN VARCHAR2,
                                   P_FA_SEQ         IN NUMBER,
                                   P_LINE_CNT       IN NUMBER,
                                   P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    V_REMARK_KEY T_STORET_PK_TYPE;

    CURSOR C_GET_IS_NBR(P_SHORT_NAME IN VARCHAR2) IS
      SELECT TSRLBRMK_IS_NUMBER, TSRLBRMK_ORG_ID
        FROM TSRLBRMK
       WHERE UPPER(RTRIM(SHORT_NAME)) = UPPER(RTRIM(P_SHORT_NAME));

  BEGIN

    OPEN C_GET_IS_NBR(P_SHORT_NAME);
    FETCH C_GET_IS_NBR
      INTO V_REMARK_KEY.IS_NUMBER, V_REMARK_KEY.ORG_ID;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_REMARK_KEY.IS_NUMBER := NULL;
      V_REMARK_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Lab Remark Codes, ' || P_SHORT_NAME ||
                                   ', could not be found in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_REMARK_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get_remark_codes_is_num.',
                                  P_FA_SEQ);
      V_REMARK_KEY.IS_NUMBER := NULL;
      V_REMARK_KEY.ORG_ID    := NULL;
      RETURN V_REMARK_KEY;

  END GET_REMARK_CODES_IS_NUM;
  /******************************************************************************************************/

  FUNCTION GET_CHAR_IS_NBR(P_CHAR_NAME      IN VARCHAR2,
                           P_FA_SEQ         IN NUMBER,
                           P_LINE_CNT       IN NUMBER,
                           P_IMPORT_LOG_SEQ IN NUMBER,
                           FROMWHERE        IN VARCHAR2)
    RETURN T_STORET_PK_TYPE IS

    V_CHAR_KEY T_STORET_PK_TYPE;

    -- jah 5-27-2004 Task 477
    -- Added OR SEARCH_NAME = UPPER(P_CHAR_NAME) to cursor C_GET_IS_NBR
    
    CURSOR C_GET_IS_NBR(P_CHAR_NAME IN VARCHAR2) IS
    SELECT TSRCHAR_IS_NUMBER, TSRCHAR_ORG_ID
    FROM   TSRCHAR
    WHERE  DISPLAY_NAME = P_CHAR_NAME
    OR     SEARCH_NAME = UPPER(P_CHAR_NAME);

  BEGIN
    /* kms - function that checks the characteristic in STORET using the processed display name.
    Called by validate storet item. */

    OPEN C_GET_IS_NBR(P_CHAR_NAME);
    FETCH C_GET_IS_NBR
      INTO V_CHAR_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      V_CHAR_KEY.IS_NUMBER := NULL;
      V_CHAR_KEY.ORG_ID    := NULL;
      IF FROMWHERE = 'CHAR' THEN
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                    'Line ' || P_LINE_CNT ||
                                     ': Characteristic, ' || P_CHAR_NAME ||
                                     ', could not be found in STORET.',
                                    P_FA_SEQ);
      ELSIF FROMWHERE = 'SUBJ' THEN
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                    'Line ' || P_LINE_CNT ||
                                     ': Subject Taxon, ' || P_CHAR_NAME ||
                                     ', could not be found in STORET.',
                                    P_FA_SEQ);
      ELSE
        SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                    'Line ' || P_LINE_CNT ||
                                     ': Common Class Descriptor, ' ||
                                     P_CHAR_NAME ||
                                     ', could not be found in STORET.',
                                    P_FA_SEQ);
      END IF;
    END IF;
    CLOSE C_GET_IS_NBR;

    RETURN V_CHAR_KEY;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting characteristic is number',
                                  P_FA_SEQ);
      V_CHAR_KEY.IS_NUMBER := NULL;
      V_CHAR_KEY.ORG_ID    := NULL;
      RETURN V_CHAR_KEY;

  END GET_CHAR_IS_NBR;

  /******************************************************************************************************/
  FUNCTION GET_FIELD_LAB_PROCEDURE(P_PROC_ID        IN VARCHAR2,
                                   P_SOURCE         IN VARCHAR2,
                                   P_ORG_ID         IN VARCHAR2,
                                   P_IMP_CFG_SEQ    IN NUMBER,
                                   P_COL_SEQ        IN NUMBER,
                                   P_IMPORT_LOG_SEQ IN NUMBER,
                                   P_LINE_CNT       IN NUMBER,
                                   P_FA_SEQ         IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    /* kms - function that gets the analysis procedure primary key using the procedure id
    and the source acronym  */

    V_PROC_KEY    SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;
    V_ORG_IS_NBR  NUMBER(12);
    V_TEMP_IS_NBR NUMBER(12);
    V_TYPE_CD     VARCHAR2(2);

    E_NOT_YOURS EXCEPTION;
    E_BAD_PROC EXCEPTION;

    CURSOR C_GET_IS_NBR IS
      SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID
        FROM TSRANLPR
       WHERE PROCEDURE_ID = RPAD(P_PROC_ID, 15) AND
             SOURCE_ACR = RPAD(P_SOURCE, 12) AND
             TSRANLPR_ORG_ID = RPAD(P_ORG_ID, 8);

    CURSOR C_GET_IS_NBR_NATL IS
      SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID
        FROM TSRANLPR
       WHERE PROCEDURE_ID = RPAD(P_PROC_ID, 15) AND
             SOURCE_ACR = RPAD(P_SOURCE, 12) AND
             TSRANLPR_ORG_ID = 'ZZZZZZZZ';

    CURSOR C_CHECK_ADOPT(P_PROC_IS_NBR IN NUMBER, P_PROC_ORG_ID IN VARCHAR2, P_ORG_IS_NBR IN NUMBER) IS
      SELECT TSMORGAN_IS_NUMBER
        FROM TSROAPA
       WHERE TSRANLPR_IS_NUMBER = P_PROC_IS_NBR AND
             TSRANLPR_ORG_ID = RPAD(P_PROC_ORG_ID, 8) AND
             TSMORGAN_IS_NUMBER = P_ORG_IS_NBR;

    CURSOR C_GET_IS_NBR_OTHERS IS
      SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID
        FROM TSRANLPR
       WHERE PROCEDURE_ID = RPAD(P_PROC_ID, 15) AND
             SOURCE_ACR = RPAD(P_SOURCE, 12);

  BEGIN

    /* get prep is number for this id and org. If not found look in the national. */
    OPEN C_GET_IS_NBR;
    FETCH C_GET_IS_NBR
      INTO V_PROC_KEY;
    IF C_GET_IS_NBR%NOTFOUND THEN
      OPEN C_GET_IS_NBR_NATL;
      FETCH C_GET_IS_NBR_NATL
        INTO V_PROC_KEY;
      IF C_GET_IS_NBR_NATL%NOTFOUND THEN
        OPEN C_GET_IS_NBR_OTHERS;
        FETCH C_GET_IS_NBR_OTHERS
          INTO V_PROC_KEY;
        V_PROC_KEY.IS_NUMBER := NULL;
        V_PROC_KEY.ORG_ID    := NULL;
        IF C_GET_IS_NBR_OTHERS%NOTFOUND THEN
          RAISE E_BAD_PROC;
        ELSE
          RAISE E_NOT_YOURS;
        END IF;
        CLOSE C_GET_IS_NBR_OTHERS;
      END IF;
      CLOSE C_GET_IS_NBR_NATL;
    END IF;
    CLOSE C_GET_IS_NBR;

    /* get the org is number */
    V_ORG_IS_NBR := GET_ORG_INFO(P_ORG_ID, P_IMPORT_LOG_SEQ);

    /* make sure analysis procedure has been adopted by this org */
    IF V_PROC_KEY.IS_NUMBER IS NOT NULL THEN
      OPEN C_CHECK_ADOPT(V_PROC_KEY.IS_NUMBER,
                         V_PROC_KEY.ORG_ID,
                         V_ORG_IS_NBR);
      FETCH C_CHECK_ADOPT
        INTO V_TEMP_IS_NBR;
      IF C_CHECK_ADOPT%NOTFOUND THEN
        /* kms - set adoption type code to indicate whether or not the analysis procedure
        is owned by this organization */
        IF V_PROC_KEY.ORG_ID = 'ZZZZZZZZ' THEN
          V_TYPE_CD := 'AD';
        ELSE
          V_TYPE_CD := 'OW';
        END IF;

        /* ADDED BY SNG 3/5/01 - ADOPT THE PROCEDURE ...) */
        INSERT INTO TSROAPA
          (TSMORGAN_IS_NUMBER,
           TSRANLPR_IS_NUMBER,
           TSRANLPR_ORG_ID,
           TYPE_CODE,
           SUSPEND_INDICATOR,
           D_USERID_CODE,
           D_LAST_UPDATE_TS)
        VALUES
          (V_ORG_IS_NBR,
           V_PROC_KEY.IS_NUMBER,
           V_PROC_KEY.ORG_ID,
           V_TYPE_CD,
           'N',
           sim2_export_pkg.v_pkg_ownerid,
           TRUNC(SYSDATE));
      END IF;
      CLOSE C_CHECK_ADOPT;
    END IF;

    RETURN V_PROC_KEY;

  EXCEPTION
    WHEN E_BAD_PROC THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Field/Lab Procedure ' || P_PROC_ID ||
                                   ' with Source Acronym, ' || P_SOURCE ||
                                   ' not found in STORET.',
                                  P_FA_SEQ);
      RETURN V_PROC_KEY;
    WHEN E_NOT_YOURS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': The Field/Lab Procedure ' || P_PROC_ID ||
                                   ', is not valid: it is not a national procedure, nor does it belong to this organization.',
                                  P_FA_SEQ);
      RETURN V_PROC_KEY;
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting analysis procedure is number',
                                  P_FA_SEQ);
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
      RETURN V_PROC_KEY;

  END GET_FIELD_LAB_PROCEDURE;
  /*********************************************************************************************/
  FUNCTION GET_LAB_SAMPLE_PREP_PROCEDURE(P_LABPREP_ID     IN VARCHAR2,
                                         P_LABPREP_SOURCE IN VARCHAR2,
                                         P_ORG_ID         IN VARCHAR2,
                                         P_IMP_CFG_SEQ    IN NUMBER,
                                         P_COL_SEQ        IN NUMBER,
                                         P_IMPORT_LOG_SEQ IN NUMBER,
                                         P_LINE_CNT       IN NUMBER,
                                         P_FA_SEQ         IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    /* kms - function that gets the lab sample prep procedure primary key using the procedure id
    and the source acronym  */

    V_LABPREP_KEY SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;
    V_ORG_IS_NBR  NUMBER(12);
    V_TEMP_IS_NBR NUMBER(12);
    V_TYPE_CD     VARCHAR2(2);

    E_NOT_YOUR_LABPREP EXCEPTION;
    E_BAD_LABPREP EXCEPTION;

    CURSOR C_GET_PREP_SOURCE_IS_NBR IS
      SELECT TSRLSPP_IS_NUMBER, TSRLSPP_ORG_ID
        FROM TSRLSPP
       WHERE PREPARATION_ID = RPAD(P_LABPREP_ID, 15) AND
             SOURCE_ACR = RPAD(P_LABPREP_SOURCE, 12) AND
             TSRLSPP_ORG_ID = RPAD(P_ORG_ID, 8);

    CURSOR C_GET_PREP_SOURCE_IS_NBR_NATL IS
      SELECT TSRLSPP_IS_NUMBER, TSRLSPP_ORG_ID
        FROM TSRLSPP
       WHERE PREPARATION_ID = RPAD(P_LABPREP_ID, 15) AND
             SOURCE_ACR = RPAD(P_LABPREP_SOURCE, 12) AND
             TSRLSPP_ORG_ID = 'ZZZZZZZZ';

    CURSOR C_GET_PREP_IS_NBR_OTHERS IS
      SELECT TSRLSPP_IS_NUMBER, TSRLSPP_ORG_ID
        FROM TSRLSPP
       WHERE PREPARATION_ID = RPAD(P_LABPREP_ID, 15) AND
             SOURCE_ACR = RPAD(P_LABPREP_SOURCE, 12);

    CURSOR C_CHECK_ADOPT(P_PREP_IS_NBR IN NUMBER, P_PREP_ORG_ID IN VARCHAR2, P_ORG_IS_NBR IN NUMBER) IS
      SELECT TSMORGAN_IS_NUMBER
        FROM TSROLSPA
       WHERE TSRLSPP_IS_NUMBER = P_PREP_IS_NBR AND
             TSRLSPP_ORG_ID = RPAD(P_PREP_ORG_ID, 8) AND
             TSMORGAN_IS_NUMBER = P_ORG_IS_NBR;

  BEGIN

    /* get prep is number for this id and org. If not found look in the national. */
    OPEN C_GET_PREP_SOURCE_IS_NBR;
    FETCH C_GET_PREP_SOURCE_IS_NBR
      INTO V_LABPREP_KEY;
    IF C_GET_PREP_SOURCE_IS_NBR%NOTFOUND THEN
      OPEN C_GET_PREP_SOURCE_IS_NBR_NATL;
      FETCH C_GET_PREP_SOURCE_IS_NBR_NATL
        INTO V_LABPREP_KEY;
      IF C_GET_PREP_SOURCE_IS_NBR_NATL%NOTFOUND THEN
        OPEN C_GET_PREP_IS_NBR_OTHERS;
        FETCH C_GET_PREP_IS_NBR_OTHERS
          INTO V_LABPREP_KEY;
        V_LABPREP_KEY.IS_NUMBER := NULL;
        V_LABPREP_KEY.ORG_ID    := NULL;
        IF C_GET_PREP_IS_NBR_OTHERS%NOTFOUND THEN
          RAISE E_BAD_LABPREP;
        ELSE
          RAISE E_NOT_YOUR_LABPREP;
        END IF;
        CLOSE C_GET_PREP_IS_NBR_OTHERS;
      END IF;
      CLOSE C_GET_PREP_SOURCE_IS_NBR_NATL;
    END IF;
    CLOSE C_GET_PREP_SOURCE_IS_NBR;

    /* get the user's org is number */
    V_ORG_IS_NBR := GET_ORG_INFO(P_ORG_ID, P_IMPORT_LOG_SEQ);

    /* make sure analysis procedure has been adopted by this org */
    IF V_LABPREP_KEY.IS_NUMBER IS NOT NULL THEN
      OPEN C_CHECK_ADOPT(V_LABPREP_KEY.IS_NUMBER,
                         V_LABPREP_KEY.ORG_ID,
                         V_ORG_IS_NBR);
                         
      FETCH C_CHECK_ADOPT
        INTO V_TEMP_IS_NBR;
      IF C_CHECK_ADOPT%NOTFOUND THEN
        /* kms - set adoption type code to indicate whether or not the analysis procedure
        is owned by this organization */
        IF V_LABPREP_KEY.ORG_ID = 'ZZZZZZZZ' THEN
          V_TYPE_CD := 'AD';
        ELSE
          V_TYPE_CD := 'OW';
        END IF;

        /* Last updated by DTW 11/12/03 - ADOPT THE PROCEDURE ...) */
        INSERT INTO TSROLSPA
          (TSMORGAN_IS_NUMBER,
           TSRLSPP_IS_NUMBER,
           TSRLSPP_ORG_ID,
           TYPE_CODE,
           D_USERID_CODE,
           D_LAST_UPDATE_TS)
        VALUES
          (V_ORG_IS_NBR,
           V_LABPREP_KEY.IS_NUMBER,
           V_LABPREP_KEY.ORG_ID,
           V_TYPE_CD,
           sim2_export_pkg.v_pkg_ownerid,
           TRUNC(SYSDATE));
      END IF;
      CLOSE C_CHECK_ADOPT;
    END IF;

    RETURN V_LABPREP_KEY;

  EXCEPTION
    WHEN E_BAD_LABPREP THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Lab Sample Prep Procedure ' ||
                                   P_LABPREP_ID || ' with Source Acronym, ' ||
                                   P_LABPREP_SOURCE ||
                                   ' not found in STORET.',
                                  P_FA_SEQ);
      RETURN V_LABPREP_KEY;
    WHEN E_NOT_YOUR_LABPREP THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': The Lab Sample Prep Procedure ' ||
                                   P_LABPREP_ID ||
                                   ', is not valid: it is not a national procedure, nor does it belong to this organization.',
                                  P_FA_SEQ);
      RETURN V_LABPREP_KEY;
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting analysis procedure is number',
                                  P_FA_SEQ);
      V_LABPREP_KEY.IS_NUMBER := NULL;
      V_LABPREP_KEY.ORG_ID    := NULL;
      RETURN V_LABPREP_KEY;

  END GET_LAB_SAMPLE_PREP_PROCEDURE;
  /*********************************************************************************************/
  FUNCTION GET_ANALPROC_IS_NBR(P_PROCEDURE_ID        IN VARCHAR2,
                               P_ORG_ID              IN VARCHAR2,
                               P_FA_SEQ              IN NUMBER,
                               P_LINE_CNT            IN NUMBER,
                               P_IMPORT_LOG_SEQ      IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
    to see if the Procedure ID existis so translation stuff can be done. Actual validation
    happens after the source is read in get_field_lab_procedure */

    V_PROC_KEY     T_STORET_PK_TYPE;

    CURSOR C_CHECK_PROC_ID(P_PROC_ID IN VARCHAR2) IS
      SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID
        FROM TSRANLPR
       WHERE PROCEDURE_ID = RPAD(P_PROC_ID, 15)
         AND (TSRANLPR_ORG_ID = RPAD(P_ORG_ID,8)
          OR TSRANLPR_ORG_ID = 'ZZZZZZZZ');

  BEGIN

    OPEN C_CHECK_PROC_ID(P_PROCEDURE_ID);
    FETCH C_CHECK_PROC_ID INTO V_PROC_KEY.IS_NUMBER, V_PROC_KEY.ORG_ID;
    
    IF C_CHECK_PROC_ID%NOTFOUND THEN
    
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
    
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT ||
        ': Unable to find Field/Lab Procedure ' ||
        P_PROCEDURE_ID || ' in STORET.',
        P_FA_SEQ
      );
      
    END IF;
    
    CLOSE C_CHECK_PROC_ID;

    RETURN V_PROC_KEY;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
        ' - getting Field/Lab Procedure is number',
        P_FA_SEQ
      );
      V_PROC_KEY.IS_NUMBER := NULL;
      V_PROC_KEY.ORG_ID    := NULL;
      RETURN V_PROC_KEY;

  END GET_ANALPROC_IS_NBR;

  /*********************************************************************************************/
  FUNCTION GET_ANALPROC_SOURCE(P_PROCEDURE_SOURCE_ID IN VARCHAR2,
                               P_ORG_ID              IN VARCHAR2,
                               P_FA_SEQ              IN NUMBER,
                               P_LINE_CNT            IN NUMBER,
                               P_IMPORT_LOG_SEQ      IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
    to see if the Procedure Source exists so translation stuff can be done. Actual validation
    happens after the source is read in get_field_lab_procedure */

    V_SOURCE_KEY T_STORET_PK_TYPE;

    CURSOR C_CHECK_PROC_SOURCE(P_PROCEDURE_ACR IN VARCHAR2) IS
      SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID
        FROM TSRANLPR
       WHERE SOURCE_ACR = RPAD(P_PROCEDURE_ACR, 12);

  BEGIN

    OPEN C_CHECK_PROC_SOURCE(P_PROCEDURE_SOURCE_ID);
    FETCH C_CHECK_PROC_SOURCE
      INTO V_SOURCE_KEY.IS_NUMBER, V_SOURCE_KEY.ORG_ID;
    IF C_CHECK_PROC_SOURCE%NOTFOUND THEN
      V_SOURCE_KEY.IS_NUMBER := NULL;
      V_SOURCE_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Unable to find Field/Lab Procedure Source acronym ' ||
                                   P_PROCEDURE_SOURCE_ID || ' in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_CHECK_PROC_SOURCE;

    RETURN V_SOURCE_KEY;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting Field/Lab Procedure Source acronym is number',
                                  P_FA_SEQ);
      V_SOURCE_KEY.IS_NUMBER := NULL;
      V_SOURCE_KEY.ORG_ID    := NULL;
      RETURN V_SOURCE_KEY;

  END GET_ANALPROC_SOURCE;
  /******************************************************************************************************/
  FUNCTION GET_LABPREP_SOURCE(P_PROCEDURE_SOURCE_ID IN VARCHAR2,
                              P_ORG_ID              IN VARCHAR2,
                              P_FA_SEQ              IN NUMBER,
                              P_LINE_CNT            IN NUMBER,
                              P_IMPORT_LOG_SEQ      IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
    to see if the Lab Sample Prep Procedure Source existis so translation stuff can be done. Actual validation
    happens after the source is read in get_lab_sample_prep_procedure */

    V_PREP_SOURCE_KEY T_STORET_PK_TYPE;

    CURSOR C_CHECK_PREP_PROC_SOURCE(P_PROCEDURE_ACR IN VARCHAR2) IS
      SELECT TSRLSPP_IS_NUMBER, TSRLSPP_ORG_ID
        FROM TSRLSPP
       WHERE SOURCE_ACR = RPAD(P_PROCEDURE_ACR, 12);

  BEGIN

    OPEN C_CHECK_PREP_PROC_SOURCE(P_PROCEDURE_SOURCE_ID);
    FETCH C_CHECK_PREP_PROC_SOURCE
      INTO V_PREP_SOURCE_KEY.IS_NUMBER, V_PREP_SOURCE_KEY.ORG_ID;
    IF C_CHECK_PREP_PROC_SOURCE%NOTFOUND THEN
      V_PREP_SOURCE_KEY.IS_NUMBER := NULL;
      V_PREP_SOURCE_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Unable to find Lab Sample Prep Procedure Source acronym ' ||
                                   P_PROCEDURE_SOURCE_ID || ' in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_CHECK_PREP_PROC_SOURCE;

    RETURN V_PREP_SOURCE_KEY;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting Lab Sample Prep Procedure Source acronym is number',
                                  P_FA_SEQ);
      V_PREP_SOURCE_KEY.IS_NUMBER := NULL;
      V_PREP_SOURCE_KEY.ORG_ID    := NULL;
      RETURN V_PREP_SOURCE_KEY;

  END GET_LABPREP_SOURCE;
  /******************************************************************************************************/
  FUNCTION GET_LABPREP_IS_NBR(P_PREPARATION_ID        IN VARCHAR2,
                              P_ORG_ID                IN VARCHAR2,
                              P_FA_SEQ                IN NUMBER,
                              P_LINE_CNT              IN NUMBER,
                              P_IMPORT_LOG_SEQ        IN NUMBER)
    RETURN T_STORET_PK_TYPE IS

    /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
    to see if the Lab Sample Prep Procedure ID existis so translation stuff can be done. Actual validation
    happens after the source is read in get_lab_sample_prep_procedure */

    V_PREP_KEY       T_STORET_PK_TYPE;

    CURSOR C_CHECK_PREP_PROC_ID(P_PREP_PROCEDURE_ID IN VARCHAR2) IS
      SELECT TSRLSPP_IS_NUMBER, TSRLSPP_ORG_ID
        FROM TSRLSPP
       WHERE PREPARATION_ID = RPAD(P_PREP_PROCEDURE_ID, 15)
         AND (TSRLSPP_ORG_ID = RPAD(P_ORG_ID,8)
          OR TSRLSPP_ORG_ID = 'ZZZZZZZZ');

  BEGIN

    OPEN C_CHECK_PREP_PROC_ID(P_PREPARATION_ID);
    FETCH C_CHECK_PREP_PROC_ID
      INTO V_PREP_KEY.IS_NUMBER, V_PREP_KEY.ORG_ID;
    IF C_CHECK_PREP_PROC_ID%NOTFOUND THEN
      V_PREP_KEY.IS_NUMBER := NULL;
      V_PREP_KEY.ORG_ID    := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT ||
                                   ': Unable to find Lab Sample Prep Procedure ' ||
                                   P_PREPARATION_ID || ' in STORET.',
                                  P_FA_SEQ);
    END IF;
    CLOSE C_CHECK_PREP_PROC_ID;

    RETURN V_PREP_KEY;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - getting Lab Sample Prep Procedure is number',
                                  P_FA_SEQ);
      V_PREP_KEY.IS_NUMBER := NULL;
      V_PREP_KEY.ORG_ID    := NULL;
      RETURN V_PREP_KEY;

  END GET_LABPREP_IS_NBR;
  /******************************************************************************************************/
  FUNCTION GET_TRIP_ID(P_INSTR_CODE      IN VARCHAR2,
                     P_START_DATE      IN DATE,
                     P_TRIP_START_DATE OUT DATE,
                     P_TRIP_END_DATE   OUT DATE,
                     P_ORG_ID          IN VARCHAR2,
                     P_FA_SEQ          IN NUMBER,
                     P_LINE_CNT        IN NUMBER,
                     P_IMPORT_LOG_SEQ  IN NUMBER) RETURN VARCHAR2 IS

  V_TRIP_ID VARCHAR2(15);

BEGIN
  /* kms - function that will get the trip primary key based on instructions - id will be based either
     on year, month and year, date, or on what's in the text file.  Will create trip only if the id
    cannot be found in STORET. */

    /* one trip per year */
    IF P_INSTR_CODE = 'RES-TRIP-01' THEN
      V_TRIP_ID := TO_CHAR(P_START_DATE, 'RRRR');

      /* trip starts on the first day of the year */
      P_TRIP_START_DATE := TO_DATE('1-1-' ||
                                   TO_CHAR(TRUNC(P_START_DATE, 'RRRR'),
                                           'RRRR'),
                                   'MM-DD-RRRR');

      /* trip ends on the last day of the year */
      P_TRIP_END_DATE := TO_DATE('12-31-' ||
                                 TO_CHAR(TRUNC(P_START_DATE, 'RRRR'),
                                         'RRRR'),
                                 'MM-DD-RRRR');

      /* one trip per month */
    ELSIF P_INSTR_CODE = 'RES-TRIP-02' THEN
      V_TRIP_ID := TO_CHAR(P_START_DATE, 'MMRRRR');

      /* trip starts on the first day of the month */
      P_TRIP_START_DATE := TRUNC(P_START_DATE, 'MON');

      /* trip ends on the last day of the month */
      P_TRIP_END_DATE := LAST_DAY(P_START_DATE);

      /* one trip per day */
    ELSIF P_INSTR_CODE = 'RES-TRIP-03' THEN
      V_TRIP_ID := TO_CHAR(P_START_DATE, 'MMDDRRRR');

      /* trip starts the same day */
      P_TRIP_START_DATE := P_START_DATE;

      /* trip ends same day */
      P_TRIP_END_DATE := P_START_DATE;

    END IF;

    RETURN V_TRIP_ID;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get trip id.',
                                  P_FA_SEQ);
      P_TRIP_START_DATE := NULL;
      P_TRIP_END_DATE   := NULL;
      V_TRIP_ID         := NULL;
      RETURN V_TRIP_ID;

  END GET_TRIP_ID;

  /******************************************************************************************************/
  FUNCTION GET_VISIT_START_DATE(P_VISIT_KEY      IN T_STORET_PK_TYPE,
                                P_TRIP_SEQ       IN NUMBER,
                                P_FA_SEQ         IN NUMBER,
                                P_LINE_CNT       IN NUMBER,
                                P_IMPORT_LOG_SEQ IN NUMBER) RETURN DATE IS

    V_START_DATE DATE;

    -- retrieve trip from SIM
    CURSOR C_GET_DATE_FROM_TRIP(P_TRIP_SEQ IN NUMBER) IS
    SELECT STP_START_DATE 
    FROM   SIM_TRIPS
    WHERE  STP_SEQ = P_TRIP_SEQ;

  BEGIN
    /* kms (5/30/01)- function that gets the station visit start date from the station
    visit if it exists in STORET or from the trip. */
    
    /* kms (6/02/01 - changed this so it only gets it from the trip.  Since station
    visits are now loaded into SIM when a trip is first encountered by the import, we
    know that all of the station visits that exist in STORET are also in SIM.  The
    start date, at this point is only needed if the visit does not exist in SIM (and
    is therefore not in STORET). */

    /* kms - station visit does not exist yet, get date from trip in SIM table.
    This will be replaced later if the station visit start date has been
    included. */
    
    -- try to retrieve the Trip from SIM
    OPEN C_GET_DATE_FROM_TRIP(P_TRIP_SEQ);
    FETCH C_GET_DATE_FROM_TRIP INTO V_START_DATE;
    
    -- did we retrieve the Trip from SIM?
    IF C_GET_DATE_FROM_TRIP%NOTFOUND THEN
      -- if no
      
      -- Tell calling routine Trip not found      
      V_START_DATE := NULL;
      
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT || ': ' ||
        'Error retrieving start date from SIM_TRIPS.',
        P_FA_SEQ
      );
      
    END IF;
    CLOSE C_GET_DATE_FROM_TRIP;

    RETURN V_START_DATE;

  EXCEPTION
  
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT || ': ' 
        || SQLERRM || 
        ' - in get visit start date.',
        P_FA_SEQ
      );
      -- tell calling routine Trip not found
      RETURN NULL;

  END GET_VISIT_START_DATE;

  /******************************************************************************************************/
  FUNCTION GET_VISIT_SEQ(
           P_VISIT_KEY         OUT T_STORET_PK_TYPE,
           P_STATION_KEY       IN T_STORET_PK_TYPE,
           P_VISIT_START_DATE  IN DATE,
           P_VISIT_ID          IN VARCHAR2,
           P_TRIP_SEQ          IN NUMBER,
           P_FA_SEQ            IN NUMBER,
           P_LINE_CNT          IN NUMBER,
           P_IMPORT_LOG_SEQ    IN NUMBER) 
           RETURN NUMBER IS
    -- jah 3-8-2004 Added Parm P_VISIT_START_DATE
    
    /* 
       Purpose
         This function tries to find a station visit record in SIM_STATION_VISITS.
         In addition, it uses the data passed by the calling routine to update records 
         in SIM_STATION_VISITS that have an UNAssigned Station Visit ID.
         
         It tries twice to find the station visit record. First, it looks for a 
         record with a matching Trip, Station, and Station Visit ID Number. Second, it 
         looks for a record with a matching Trip, Station, Station Visit Start Date, and
         'UNA' in the first three letters of the Station Visit ID Number.
         
         If it finds a matching UNA record, this function changes the Station Visit ID
         Number in the record to the Station Visit ID passed by the calling routine.
         
       Parameters
         P_TRIP_SEQ          - Foreign Key to a Trip record
         P_VISIT_ID          - Station Visit ID from the import data record
         P_VISIT_START_DATE  - Station Visit Start Date from the import data record
         
       OUT Parameters
         P_VISIT_KEY         - STORET record key for the Station Visit record
         
       Return
         This function returns the key (sequence number) of the matching record in
         SIM_STATION_VISITS; or NULL if it cannot find a matching record.
         
       Remarks
         1. Pass the Station Visit ID Number and Station Visit Start Date from the 
            import data record. This means you should only call this function when the
            import instruction for the station visit is "Read existing station visit
            from (import) data file."
    */

    V_VISIT_SEQ NUMBER(10);

    -- Retrieve station visit using Trip, Station and Station Visit ID
    CURSOR C_FIND_STATION_VISIT(
      cp_visit_nbr        IN VARCHAR2,
      cp_tsmstatn_is_nbr  IN NUMBER,
      cp_tsmstatn_org_id  IN VARCHAR2,
      cp_trip_seq         IN NUMBER) IS
    SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID
    FROM   SIM_STATION_VISITS
    WHERE  SSV_ID                 = cp_visit_nbr
    AND    SSV_TSMSTATN_IS_NUMBER = cp_tsmstatn_is_nbr 
    AND    SSV_TSMSTATN_ORG_ID    = RPAD(cp_tsmstatn_org_id, 8) 
    AND    SSV_STP_SEQ            = cp_trip_seq;

    -- Retrieve UNAssigned station visit for a given date
    CURSOR C_FIND_UNA_STATION_VISIT(
      cp_visit_start_date IN DATE,
      cp_tsmstatn_is_nbr  IN NUMBER,
      cp_tsmstatn_org_id  IN VARCHAR2,
      cp_trip_seq         IN NUMBER) IS
    SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID
    FROM   SIM_STATION_VISITS
    WHERE  SUBSTR(SSV_ID,1,3)     = 'UNA'
    AND    SSV_START_DATE         = cp_visit_start_date
    AND    SSV_START_TIME         IS NULL
    AND    SSV_TSMSTATN_IS_NUMBER = cp_tsmstatn_is_nbr
    AND    SSV_TSMSTATN_ORG_ID    = RPAD(cp_tsmstatn_org_id, 8)
    AND    SSV_STP_SEQ            = cp_trip_seq;

  BEGIN

    -- Try to retrieve the station visit record using the trip station visit ID in the import data record.
    OPEN C_FIND_STATION_VISIT(
      P_VISIT_ID,
      P_STATION_KEY.IS_NUMBER,
      P_STATION_KEY.ORG_ID,
      P_TRIP_SEQ
    );
    FETCH C_FIND_STATION_VISIT INTO 
      V_VISIT_SEQ, 
      P_VISIT_KEY.IS_NUMBER, 
      P_VISIT_KEY.ORG_ID;

    -- did we find the station visit record using the visit ID?        
    IF C_FIND_STATION_VISIT%FOUND THEN
      -- if yes

      CLOSE C_FIND_STATION_VISIT;

    ELSE
      -- if no
      
      CLOSE C_FIND_STATION_VISIT;

      -- Try to find an UNAssigned station visit that matches
      OPEN C_FIND_UNA_STATION_VISIT(
        P_VISIT_START_DATE,
        P_STATION_KEY.IS_NUMBER,
        P_STATION_KEY.ORG_ID,
        P_TRIP_SEQ
      );
      FETCH C_FIND_UNA_STATION_VISIT INTO 
        V_VISIT_SEQ, 
        P_VISIT_KEY.IS_NUMBER, 
        P_VISIT_KEY.ORG_ID;
      
      -- did we find an UNAssigned station visit that matches?
      IF C_FIND_UNA_STATION_VISIT%FOUND THEN
        -- if yes            
                  
        /* jah 3-7-2004
           Update the "UNA"ssigned Station Visit record in SIM_STATION_VISITS. */
        UPDATE SIM_STATION_VISITS
        SET    SSV_ID = P_VISIT_ID
        WHERE  SSV_SEQ = V_VISIT_SEQ;

      ELSE            
        -- if no (i.e., we did NOT find an UNAssigned station visit that matches)
        
        -- tell calling routine we did not find a matching record
        V_VISIT_SEQ := NULL;

      END IF;
      CLOSE C_FIND_UNA_STATION_VISIT;

    END IF;

    RETURN V_VISIT_SEQ;

  EXCEPTION

    WHEN OTHERS THEN

      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line ' || P_LINE_CNT || ': ' || SQLERRM ||
        ' - in get visit seq.',
        P_FA_SEQ
      );

      -- Tell calling routine this function could not find a matching station visit record.
      RETURN NULL;

  END GET_VISIT_SEQ;

  /******************************************************************************************************/

  FUNCTION GET_ORG_INFO(P_ORG_ID IN VARCHAR2, P_IMPORT_LOG_SEQ IN NUMBER)
    RETURN NUMBER IS

    V_ORG_IS_NBR NUMBER(8);

    CURSOR C_GET_TSMORGAN_NBR(P_CURSOR_ORG_ID IN VARCHAR2) IS
      SELECT TSMORGAN_IS_NUMBER
        FROM TSMORGAN
       WHERE ORG_ID = RPAD(P_CURSOR_ORG_ID, 8);

  BEGIN
    /* kms - function that gets the is_number for the organization running the import.  Called by main
    import procedure - import_results */

    OPEN C_GET_TSMORGAN_NBR(P_ORG_ID);
    FETCH C_GET_TSMORGAN_NBR
      INTO V_ORG_IS_NBR;
    IF C_GET_TSMORGAN_NBR%NOTFOUND THEN
      V_ORG_IS_NBR := NULL;
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  'Failed to get organization is number.',
                                  NULL);
    END IF;
    CLOSE C_GET_TSMORGAN_NBR;

    RETURN V_ORG_IS_NBR;

  EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20199,
                              'Failed to get organization is number.');
      RETURN NULL;

  END GET_ORG_INFO;

  /******************************************************************************************************/
  FUNCTION GET_UOM_IS_NBR(P_UOM_NAME       IN VARCHAR2,
                          P_FA_SEQ         IN NUMBER,
                          V_IMPORT_LOG_SEQ IN NUMBER,
                          V_LINE_CNT       IN NUMBER)
    RETURN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE IS

    TYPE T_UOM_RECORD IS RECORD(
      UOM_IS_NBR NUMBER(8),
      UOM_ORG_ID VARCHAR2(8),
      UOM_NAME   VARCHAR2(10));

    V_UOM_INFO T_UOM_RECORD;
    V_UOM_KEY  SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;

    E_UNK_UOM EXCEPTION;
    E_UOM_SP EXCEPTION;

    CURSOR C_GET_IS_NBR(P_UOM_NAME IN VARCHAR2) IS
      SELECT TSRUOM_IS_NUMBER, TSRUOM_ORG_ID, SHORT_FORM_NAME
        FROM TSRUOM
       WHERE UPPER(SHORT_FORM_NAME) = UPPER(RPAD(P_UOM_NAME, 10));

  BEGIN
    /* kms - function that gets the uom primary key using the processed uom name - called by validate storet item
    Also checks the case of the unit of measure name and issues a warning if it is not correct */

    /* get tsruom key for unit of measure */
    OPEN C_GET_IS_NBR(P_UOM_NAME);
    FETCH C_GET_IS_NBR
      INTO V_UOM_INFO;
    IF C_GET_IS_NBR%NOTFOUND THEN

      /* if uom could not be found then raise an exception to write error to error log, will change status in main procedure */
      V_UOM_INFO.UOM_IS_NBR := NULL;
      V_UOM_INFO.UOM_ORG_ID := NULL;
      CLOSE C_GET_IS_NBR;
      RAISE E_UNK_UOM;

    ELSE

      IF RPAD(V_UOM_INFO.UOM_NAME, 10) != RPAD(P_UOM_NAME, 10) THEN
        CLOSE C_GET_IS_NBR;
        RAISE E_UOM_SP;
      END IF;

    END IF;
    CLOSE C_GET_IS_NBR;

    V_UOM_KEY.IS_NUMBER := V_UOM_INFO.UOM_IS_NBR;
    V_UOM_KEY.ORG_ID    := V_UOM_INFO.UOM_ORG_ID;

    RETURN V_UOM_KEY;

  EXCEPTION
    WHEN E_UNK_UOM THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                  ': Unknown Unit of Measure: ' ||
                                  P_UOM_NAME,
                                  P_FA_SEQ);
      RETURN V_UOM_KEY;

    WHEN E_UOM_SP THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                   ': Unit of Measure misspelled: ' ||
                                   P_UOM_NAME || ' - check capitalization.',
                                  P_FA_SEQ);
      RETURN V_UOM_KEY;

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in get uom is number.',
                                  P_FA_SEQ);
      V_UOM_KEY.IS_NUMBER := NULL;
      V_UOM_KEY.ORG_ID    := NULL;
      RETURN V_UOM_KEY;

  END GET_UOM_IS_NBR;

  /******************************************************************************************************/

  FUNCTION VALIDATE_NUMBER(P_ITEM         IN VARCHAR2,
                         P_ITEM_NAME      IN VARCHAR2,
                         P_FORMAT         IN VARCHAR2,
                         P_IMPORT_LOG_SEQ IN NUMBER,
                         P_LINE_CNT       IN NUMBER) RETURN NUMBER IS
                         
    -- Suspend numeric checking to facilitate changes to non-numerics
    -- V_RETURN NUMBER(10);

  BEGIN
    -- V_RETURN := TO_NUMBER(P_ITEM, P_FORMAT);
    RETURN 1;

  EXCEPTION
  
    WHEN OTHERS THEN
      
      SIM2_IMPORT_PKG.WRITE_ERROR(
        P_IMPORT_LOG_SEQ,
        'Line: ' || P_LINE_CNT || ': ' ||
        P_ITEM_NAME || ', ' || P_ITEM ||
        ', is not in an appropriate numeric format of ' ||
        P_FORMAT || '.',
        NULL
      );
      
      RETURN 0;
  
  END VALIDATE_NUMBER;
/******************************************************************************************************/
  FUNCTION GET_ALLOWABLE_VALUE(
           P_ALVAL_SEQ      IN NUMBER,
           P_IMPORT_LOG_SEQ IN NUMBER) 
           RETURN VARCHAR2 IS

    V_ITEM_NAME VARCHAR2(80);

    CURSOR C_GET_ALVAL_NAME(P_ALVAL_SEQ IN NUMBER) IS
    SELECT SIA_ITEM
    FROM   SIM_IMP_ALLOW_VALUES
    WHERE  SIA_SEQ = P_ALVAL_SEQ;

  BEGIN
    OPEN C_GET_ALVAL_NAME(P_ALVAL_SEQ);
    FETCH C_GET_ALVAL_NAME
      INTO V_ITEM_NAME;
    IF C_GET_ALVAL_NAME%NOTFOUND THEN
      V_ITEM_NAME := NULL;
    END IF;
    CLOSE C_GET_ALVAL_NAME;

    RETURN V_ITEM_NAME;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                   ' - getting allowable value from sequence.',
                                  NULL);
      RETURN NULL;
  END GET_ALLOWABLE_VALUE;
  /******************************************************************************************************/
  FUNCTION GET_HA_DEFAULT(V_TSRHCSC_PK     IN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE,
                          P_FA_SEQ         IN NUMBER,
                          V_IMPORT_LOG_SEQ IN NUMBER,
                          V_LINE_CNT       IN NUMBER) RETURN VARCHAR2 IS

    V_CHAR_NAME VARCHAR2(60);

    CURSOR C_GET_HA_INFO IS
      SELECT CHARACTERSTC_NAME
        FROM TSRHCSC
       WHERE TSRHCSC_IS_NUMBER = V_TSRHCSC_PK.IS_NUMBER AND
             TSRHCSC_ORG_ID = V_TSRHCSC_PK.ORG_ID;

  BEGIN

    OPEN C_GET_HA_INFO;
    FETCH C_GET_HA_INFO
      INTO V_CHAR_NAME;
    IF C_GET_HA_INFO%NOTFOUND THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                   ': HA Characteristic Name not found ',
                                  P_FA_SEQ);
      RETURN NULL;
    END IF;
    CLOSE C_GET_HA_INFO;

    RETURN V_CHAR_NAME;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                   ': Error getting HA Characteristic Name ',
                                  P_FA_SEQ);
      RETURN NULL;

  END GET_HA_DEFAULT;
  /******************************************************************************************************/
  FUNCTION UPDATE_RESULT(P_RESULT           IN VARCHAR2,
                       P_CHAR_IS_NBR      IN NUMBER,
                       P_UNITS_OK         OUT BOOLEAN,
                       P_RESULT_OK        OUT BOOLEAN,
                       P_IMP_CFG_SEQ      IN NUMBER,
                       P_FA_SEQ           IN NUMBER,
                       V_CHAR_RESULT_TYPE IN VARCHAR2,
                       V_IMPORT_LOG_SEQ   IN NUMBER,
                       V_LINE_CNT         IN NUMBER,
                       V_DETECT           IN VARCHAR2,
                       V_DETECT_CHECK     OUT NUMBER) RETURN VARCHAR2 IS

  V_RESULT  SIM_RESULTS.RS_RESULT%TYPE;

  BEGIN

    -- Initialize result flag
    P_RESULT_OK := TRUE;
    /* kms - function that gets characteristic type in order to see if it has permitted values
    or if other checks need to be performed on the result. */

    /* kms (7/11/01) - reorganized code structure below in order to make it easier to read
    and to insure that get value msr is being called when appropriate. */
    IF V_CHAR_RESULT_TYPE = 'PV  ' THEN
      /* kms (5/29/01) - result must be one of the permitted values for this characteristic. */
      V_RESULT   := VALIDATE_PV_RESULT(P_RESULT,
                                       P_IMP_CFG_SEQ,
                                       P_CHAR_IS_NBR,
                                       P_FA_SEQ,
                                       P_RESULT_OK,
                                       V_IMPORT_LOG_SEQ,
                                       V_LINE_CNT);

    ELSIF V_CHAR_RESULT_TYPE = 'TEXT' THEN
      /* dcl (08/29/01) - result must be one of the permitted values for this characteristic. */
      V_RESULT := '*Text';

    ELSE
    --sf - PHYS characteristics can't have *Non-detect, etc.
      IF V_CHAR_RESULT_TYPE != 'PHYS' THEN
        --check to see if their results match their detection condition
        V_DETECT_CHECK := 1;
        IF V_DETECT IS NOT NULL THEN
          IF V_DETECT = 'Detected, Not Quantified' AND
             P_RESULT != '*Present' THEN
            V_RESULT       := '*Present';
            V_DETECT_CHECK := 0;
          END IF;
          IF V_DETECT = 'Not Detected' AND P_RESULT != '*Non-detect' THEN
            V_RESULT       := '*Non-detect';
            V_DETECT_CHECK := 0;
          END IF;
          IF V_DETECT = 'Present above Quantification' AND
             P_RESULT != '*Present >QL' THEN
            V_RESULT       := '*Present >QL';
            V_DETECT_CHECK := 0;
          END IF;
          IF V_DETECT = 'Present below Quantification' AND
             P_RESULT != '*Present <QL' THEN
            V_RESULT       := '*Present <QL';
            V_DETECT_CHECK := 0;
          END IF;
          IF V_DETECT = 'Not Reported' AND P_RESULT != '*Not Reported' THEN
            V_RESULT       := '*Not Reported';
            V_DETECT_CHECK := 0;
          END IF;
          IF V_DETECT = 'Detected and Quantified' THEN
            V_RESULT := P_RESULT;
          END IF;
        ELSE
          V_RESULT   := P_RESULT;
        END IF;

        IF (P_RESULT = '*Non-detect' OR P_RESULT = '*Present <QL' OR
           P_RESULT = '*Present >QL' OR P_RESULT = '*Not Reported' OR
           P_RESULT = '*Present') THEN
          /* kms (5/29/01) - result value not reported, units are not ok. **/
          V_RESULT   := P_RESULT;
          P_UNITS_OK := FALSE;
        ELSIF (V_RESULT = '*Non-detect' OR V_RESULT = '*Present <QL' OR
              V_RESULT = '*Present >QL' OR V_RESULT = '*Not Reported' OR
              V_RESULT = '*Present') THEN
          P_UNITS_OK := FALSE;
        END IF;

      ELSE
        -- Physical Characteristics cannot have detection conditions for result

        IF (P_RESULT = '*Non-detect' OR P_RESULT = '*Present <QL' OR
           P_RESULT = '*Present >QL' OR P_RESULT = '*Not Reported' OR
           P_RESULT = '*Present') THEN

           V_RESULT    := P_RESULT;
           P_RESULT_OK := FALSE;
           SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                       'Line ' || V_LINE_CNT ||
                                        ': Physical Characteristics cannot have a value of : "' ||
                                        P_RESULT || '"',
                                        P_FA_SEQ);
        ELSE
           V_RESULT := P_RESULT;
        END IF;

      END IF;
     
    END IF;

    RETURN V_RESULT;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                  ' - in get char type. vres = ' ||
                                  V_RESULT,
                                  P_FA_SEQ);
      RETURN NULL;

  END UPDATE_RESULT;

  /******************************************************************************************************/
  FUNCTION VALIDATE_PV_RESULT(P_ORIG_ITEM      IN VARCHAR2,
                              P_IMP_CFG_SEQ    IN NUMBER,
                              P_CHAR_IS_NBR    IN NUMBER,
                              P_FA_SEQ         IN NUMBER,
                              P_OK             OUT BOOLEAN,
                              V_IMPORT_LOG_SEQ IN NUMBER,
                              V_LINE_CNT       IN NUMBER) RETURN VARCHAR2 IS

    V_CONVERTED_ITEM VARCHAR2(80);
    V_TRANS_EXIST    BOOLEAN;
    V_INSTR_CODE     VARCHAR2(15);

    V_DUMMY   NUMBER(8);
    V_RESULT  SIM_RESULTS.RS_RESULT%TYPE;
    V_COL_SEQ NUMBER(10);

    E_NO_COL_SEQ EXCEPTION;

    CURSOR C_GET_RESULT_COL_SEQ IS
      SELECT SIC_SEQ
        FROM SIM_IMP_COLS COL, SIM_IMP_CFG CFG
       WHERE COL.SIC_NAME = 'Result Value' AND
             COL.SIC_SID_SEQ = CFG.SICC_SID_SEQ AND
             SICC_SEQ = P_IMP_CFG_SEQ; /* make sure we use the right SIC Result Value column  */

    CURSOR C_CHECK_CHAR_PV(P_RESULT IN VARCHAR2, P_CHAR_IS_NBR IN NUMBER) IS
      SELECT TSRCPV_IS_NUMBER
        FROM TSRCPV
       WHERE UPPER(LTRIM(RTRIM(SHORT_NAME))) = UPPER(LTRIM(RTRIM(P_RESULT))) AND
             TSRCHAR_IS_NUMBER = P_CHAR_IS_NBR;

  BEGIN
    /* kms - function called if the result has a characteristic that indicates that it
    must be one of a list of allowable values.  Checks result to make sure it is has
    been checked for a translation. */
    /* kms (7/16/01) - added code that checks translations and will create them if the result
    value is not a permitted value and the instructions have been set to create translations. */

    /* kms (7/16/01) - get column sequence for results in order to see if any translations are
    set up for the result column. */

    P_OK := TRUE;

    OPEN C_GET_RESULT_COL_SEQ;
    FETCH C_GET_RESULT_COL_SEQ
      INTO V_COL_SEQ;
    IF C_GET_RESULT_COL_SEQ%NOTFOUND THEN
      CLOSE C_GET_RESULT_COL_SEQ;
      RAISE E_NO_COL_SEQ;
    END IF;
    CLOSE C_GET_RESULT_COL_SEQ;

    /* look in translations table to see if it needs to swapped
       10/1/03 pwh - added the trim function to this section.*/
    V_CONVERTED_ITEM := TRIM(CHECK_TRANSLATIONS(P_ORIG_ITEM,
                                           P_IMP_CFG_SEQ,
                                           V_COL_SEQ,
                                           V_TRANS_EXIST,
                                           V_IMPORT_LOG_SEQ,
                                           V_LINE_CNT,
                                           p_fa_seq));

    IF V_CONVERTED_ITEM IS NOT NULL THEN

      /* check result to see if it is one of the permitted values */
      OPEN C_CHECK_CHAR_PV(V_CONVERTED_ITEM, P_CHAR_IS_NBR);
      FETCH C_CHECK_CHAR_PV
        INTO V_DUMMY;
      IF C_CHECK_CHAR_PV%FOUND THEN
        V_RESULT := V_CONVERTED_ITEM;
      ELSE
        V_RESULT := NULL;
        P_OK     := FALSE;
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                    'Line ' || V_LINE_CNT ||
                                     ': Result Value, ' || V_CONVERTED_ITEM ||
                                     ' is not an allowed value for this characteristic.',
                                    P_FA_SEQ);
      END IF;
      CLOSE C_CHECK_CHAR_PV;

      IF V_DUMMY IS NULL AND NOT V_TRANS_EXIST THEN
        /* check instructions to see if translation record should be created */
        V_INSTR_CODE := GET_INSTRUCTIONS(P_IMP_CFG_SEQ,
                                         'RES-TR',
                                         P_FA_SEQ,
                                         V_IMPORT_LOG_SEQ,
                                         V_LINE_CNT);

        IF V_INSTR_CODE = 'RES-TR-01' THEN

          /* if record could not be matched to anything in translations or STORET tables,
          create new record */
          CREATE_TRANS_RECORD(P_ORIG_ITEM,
                              P_IMP_CFG_SEQ,
                              V_COL_SEQ,
                              P_FA_SEQ,
                              V_IMPORT_LOG_SEQ,
                              V_LINE_CNT);

        END IF;

      ELSIF V_DUMMY IS NULL AND V_TRANS_EXIST THEN

        /*  item has been translated to an invalid storet value */
        SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                    'Line ' || V_LINE_CNT ||
                                    ': Result Value, ' || P_ORIG_ITEM ||
                                    ', has been mapped to an invalid storet value: ' ||
                                    V_CONVERTED_ITEM,
                                    P_FA_SEQ);
        P_OK := FALSE;
      END IF;

    ELSIF V_TRANS_EXIST THEN
      /* item exists in translations table but is not mapped to anything, write error message.
      Status will be changed to 'N' later in the program. */
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' ||
                                   P_ORIG_ITEM ||
                                   ' exists in translation table but has not been mapped to a STORET code. ',
                                  P_FA_SEQ);
      P_OK := FALSE;
    END IF;

    RETURN V_RESULT;

  EXCEPTION
    WHEN E_NO_COL_SEQ THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                   ': Unable to find sequence for result column in function ''validate_pv_result''.',
                                  P_FA_SEQ);
      RETURN P_ORIG_ITEM;

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in validate pv_result.',
                                  P_FA_SEQ);
      RETURN P_ORIG_ITEM;

  END VALIDATE_PV_RESULT;

  /***********************************************************************************************/
  FUNCTION VALIDATE_FRACTION(P_SAMP_FRAC_NM   IN VARCHAR2,
                             P_IMP_CFG_SEQ    IN NUMBER,
                             P_COL_SEQ        IN NUMBER,
                             P_TSRCHAR_IS_NBR IN NUMBER,
                             P_FA_SEQ         IN NUMBER,
                             V_IMPORT_LOG_SEQ IN NUMBER,
                             V_LINE_CNT       IN NUMBER,
                             V_TSMPRMVL_KEY   OUT T_STORET_PK_TYPE)
    RETURN VARCHAR2 IS

    V_SAMPLE_FRAC      VARCHAR2(40);
    V_SAMP_REQ_CD      VARCHAR2(1);
    V_IS_NBR           NUMBER(12);
    V_ORG_ID           VARCHAR2(8);

    V_TRANS_EXIST      BOOLEAN := TRUE;
    V_CONVERTED_ITEM   VARCHAR2(80);
    V_INSTR_CODE       VARCHAR2(15);
    v_storet_pk        Sim_import_result_extras.t_storet_pk_type;
    
    E_INVALID_FRACTION EXCEPTION;
    E_SAMPLE_FRAC_SP EXCEPTION;

    CURSOR C_GET_SAMP_REQ_CD(P_TSRCHAR_IS_NBR IN NUMBER) IS
      SELECT SAMP_FRAC_REQ_CD
        FROM TSRCHAR
       WHERE TSRCHAR_IS_NUMBER = P_TSRCHAR_IS_NBR;

    CURSOR C_CHECK_DEF_FRAC(P_FRACTION IN VARCHAR2) IS
      SELECT FIELD_VALUE, TSMPRMVL_IS_NUMBER, TSMPRMVL_ORG_ID
        FROM TSMPRMVL
       WHERE FIELD_NAME = 'SMPL_FRAC_TYPE_NM' AND TABLE_NAME = 'TSRRSULT' AND
             FIELD_VALUE = P_FRACTION;

  BEGIN
                           
    /* get the char group is number - check translations and STORET tables */
    
    v_storet_pk := SIM_IMPORT_RESULT_EXTRAS.CHECK_STORET_TRANSLATION(
      P_SAMP_FRAC_NM,
      P_IMP_CFG_SEQ,
      P_COL_SEQ,
      V_TRANS_EXIST,                                  
      V_IMPORT_LOG_SEQ,
      V_LINE_CNT,
      V_CONVERTED_ITEM,
      P_FA_SEQ
    );
                                           
    IF V_CONVERTED_ITEM IS NOT NULL OR P_SAMP_FRAC_NM IS NULL THEN

      /* set sample fraction to null if it is full of spaces */
      IF RPAD(V_CONVERTED_ITEM, 15) = RPAD(' ', 15) OR
         P_SAMP_FRAC_NM IS NULL THEN
        V_SAMPLE_FRAC := NULL;
      ELSE
        V_SAMPLE_FRAC := V_CONVERTED_ITEM;
      END IF;

      OPEN C_GET_SAMP_REQ_CD(P_TSRCHAR_IS_NBR);
      FETCH C_GET_SAMP_REQ_CD
        INTO V_SAMP_REQ_CD;
      IF V_SAMP_REQ_CD = 'Y' OR V_SAMPLE_FRAC IS NOT NULL THEN

        /* see if fraction is valid if it is required or if it has been provided */
        OPEN C_CHECK_DEF_FRAC(V_CONVERTED_ITEM);
        FETCH C_CHECK_DEF_FRAC
          INTO V_SAMPLE_FRAC, V_IS_NBR, V_ORG_ID;
        IF C_CHECK_DEF_FRAC%NOTFOUND THEN
          /* if fraction could not be found in permitted values table, return null and write error */
          V_SAMPLE_FRAC := NULL;
          V_IS_NBR      := NULL;
          V_ORG_ID      := NULL;
          IF NOT V_TRANS_EXIST THEN
            /* check instructions to see if translation record should be created */
            V_INSTR_CODE := GET_INSTRUCTIONS(P_IMP_CFG_SEQ,
                                             'RES-TR',
                                             P_FA_SEQ,
                                             V_IMPORT_LOG_SEQ,
                                             V_LINE_CNT);

            IF V_INSTR_CODE = 'RES-TR-01' THEN

              /* if record could not be matched to anything in translations or allowable values tables, create new record */
              CREATE_TRANS_RECORD(P_SAMP_FRAC_NM,
                                  P_IMP_CFG_SEQ,
                                  P_COL_SEQ,
                                  P_FA_SEQ,
                                  V_IMPORT_LOG_SEQ,
                                  V_LINE_CNT);
              SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                          'Line ' || V_LINE_CNT ||
                                           ': Sample Fraction, ' ||
                                           V_CONVERTED_ITEM ||
                                           ' is not a valid sample fraction - translation record has been created.',
                                          P_FA_SEQ);

            ELSE
              SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                          'Line ' || V_LINE_CNT ||
                                           ': Sample Fraction, ' ||
                                           V_CONVERTED_ITEM ||
                                           ' is not a valid sample fraction.',
                                          P_FA_SEQ);
            END IF;

          ELSIF V_TRANS_EXIST THEN

            /* item has been translated to an invalid storet value */
            SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                        'Line ' || V_LINE_CNT || ': ' ||
                                        P_SAMP_FRAC_NM ||
                                        ' has been mapped to the invalid storet value: ' ||
                                        V_CONVERTED_ITEM,
                                        P_FA_SEQ);

          END IF;

        ELSE
          V_SAMPLE_FRAC := RTRIM(V_SAMPLE_FRAC);
        END IF;
        CLOSE C_CHECK_DEF_FRAC;
      ELSIF V_SAMP_REQ_CD = 'N' THEN
        V_SAMPLE_FRAC := 'NR'; --Not Required
      END IF;
      CLOSE C_GET_SAMP_REQ_CD;

    ELSIF V_TRANS_EXIST AND V_CONVERTED_ITEM IS NULL THEN
    
      /* if there is a code in the translations table but no value to translate to,
      write error message.
      status will be changed to 'N' later in the program */
      V_SAMPLE_FRAC := NULL;
      V_IS_NBR      := NULL;
      V_ORG_ID      := NULL;

    END IF;
    V_TSMPRMVL_KEY.IS_NUMBER := V_IS_NBR;
    V_TSMPRMVL_KEY.ORG_ID    := V_ORG_ID;
    RETURN V_SAMPLE_FRAC;

  EXCEPTION
    WHEN E_INVALID_FRACTION THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': Fraction, ' ||
                                   P_SAMP_FRAC_NM ||
                                   ', is not a valid STORET value.',
                                  P_FA_SEQ);
      RETURN NULL;

    WHEN E_SAMPLE_FRAC_SP THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                  ': Check fraction capitalization: (STORET/USER)' ||
                                  P_SAMP_FRAC_NM || '/' || V_SAMPLE_FRAC,
                                  P_FA_SEQ);
      RETURN V_SAMPLE_FRAC;

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in validate sample fraction.',
                                  P_FA_SEQ);
      RETURN NULL;

  END VALIDATE_FRACTION;
  /******************************************************************************************************/

  FUNCTION CHECK_ALLOWABLE_VALUES(P_ITEM           IN VARCHAR2,
                                  P_IMP_CFG_SEQ    IN NUMBER,
                                  P_COL_SEQ        IN NUMBER,
                                  P_FA_SEQ         IN NUMBER,
                                  V_IMPORT_LOG_SEQ IN NUMBER,
                                  V_LINE_CNT       IN NUMBER) RETURN NUMBER IS

    V_SIA_SEQ NUMBER(10);

    CURSOR C_CHECK_ALLOWABLE_VAL(P_ITEM IN VARCHAR2, P_IMP_CFG_SEQ IN NUMBER, P_COL_SEQ IN NUMBER) IS
      SELECT SIA_SEQ
        FROM SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS
       WHERE SIA_SIC_SEQ = SIC_SEQ AND SICDT_SICC_SEQ = P_IMP_CFG_SEQ AND
             SICDT_SIC_SEQ = SIC_SEQ AND SIC_SEQ = P_COL_SEQ AND
             SIA_ITEM = P_ITEM;

  BEGIN
    /* kms - function called for items that are listed as allowable values in SIM.  Checks
    processed item to make sure that it is one of the allowable values for that item and
    returns the sequence for that allowable value. */

    /* make sure item is an allowable value */
    OPEN C_CHECK_ALLOWABLE_VAL(P_ITEM, P_IMP_CFG_SEQ, P_COL_SEQ);
    FETCH C_CHECK_ALLOWABLE_VAL
      INTO V_SIA_SEQ;
    IF C_CHECK_ALLOWABLE_VAL%NOTFOUND THEN

      /* if not an allowable value, return null - will write error message, change status of activity */
      V_SIA_SEQ := NULL;

    END IF;
    CLOSE C_CHECK_ALLOWABLE_VAL;

    RETURN V_SIA_SEQ;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - in check allowable values',
                                  P_FA_SEQ);
      RETURN NULL;

  END CHECK_ALLOWABLE_VALUES;

  /******************************************************************************************************/
  FUNCTION VALIDATE_ALVAL_ITEM(
           P_ITEM_NAME      IN VARCHAR2,
           P_ORIG_ITEM      IN VARCHAR2,
           P_IMP_CFG_SEQ    IN NUMBER,
           P_COL_SEQ        IN NUMBER,
           P_FA_SEQ         IN NUMBER,
           V_IMPORT_LOG_SEQ IN NUMBER,
           V_LINE_CNT       IN NUMBER) 
           RETURN NUMBER IS

    V_CONVERTED_ITEM VARCHAR2(80);
    V_ITEM_SIA_SEQ   NUMBER(10);
    V_TRANS_EXIST    BOOLEAN := TRUE;
    V_INSTR_CODE     VARCHAR2(15);

  BEGIN

    -- if translation exists, return translation; otherwise return original value
    V_CONVERTED_ITEM := CHECK_TRANSLATIONS(
      P_ORIG_ITEM,
      P_IMP_CFG_SEQ,
      P_COL_SEQ,
      V_TRANS_EXIST,  -- OUT: if TRUE, then V_CONVERTED_ITEM is a translation
      V_IMPORT_LOG_SEQ,
      V_LINE_CNT,
      P_FA_SEQ
    );

    -- did the call to CHECK_TRANSLATIONS work properly?
    IF (V_CONVERTED_ITEM IS NOT NULL) THEN
      -- if yes
 
      -- jah 2-25-04
      -- Does the STORET GUI convert this item to uppercase?
      IF P_ITEM_NAME IN ('Point Type','Result Status') THEN
        -- if yes
        -- Convert to upper case to mimic STORET GUI
        V_CONVERTED_ITEM := UPPER(V_CONVERTED_ITEM);
      END IF;

      -- Get the primary key of the value in SIM_IMP_ALLOWABLE_VALUES table (NULL = invalid value)
      V_ITEM_SIA_SEQ := CHECK_ALLOWABLE_VALUES(
        V_CONVERTED_ITEM,
        P_IMP_CFG_SEQ,
        P_COL_SEQ,
        P_FA_SEQ,
        V_IMPORT_LOG_SEQ,
        V_LINE_CNT
      );

      -- is the value for this item invalid AND did CHECK_TRANSLATIONS return the original value?
      IF (V_ITEM_SIA_SEQ IS NULL) AND (NOT V_TRANS_EXIST) THEN
        -- if yes
        
        -- check instructions to see if translation record should be created
        V_INSTR_CODE := GET_INSTRUCTIONS(
          P_IMP_CFG_SEQ,
          'RES-TR',
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

        -- should translation record be created?
        IF V_INSTR_CODE = 'RES-TR-01' THEN
          -- if yes
          CREATE_TRANS_RECORD(
            P_ORIG_ITEM,
            P_IMP_CFG_SEQ,
            P_COL_SEQ,
            P_FA_SEQ,
            V_IMPORT_LOG_SEQ,
            V_LINE_CNT
          );
          
          -- Tell the end-user what happened
          SIM2_IMPORT_PKG.WRITE_ERROR(
            V_IMPORT_LOG_SEQ,
            'Line ' || V_LINE_CNT || ': ' ||P_ITEM_NAME || ', ' ||V_CONVERTED_ITEM || 
            ', is NOT an allowable value so a translation record has been created.',
            P_FA_SEQ
          );

        ELSE
          -- if no (a translation record should not be created)
          SIM2_IMPORT_PKG.WRITE_ERROR(
            V_IMPORT_LOG_SEQ,
            'Line ' || V_LINE_CNT || ': ' ||P_ITEM_NAME || ', ' ||V_CONVERTED_ITEM ||
            ', is NOT an allowable value.',
            P_FA_SEQ
          );
        END IF;

      -- is the value for this item invalid AND did CHECK_TRANSLATIONS return a translated value
      -- AND is the item name something other than 'Actual'?
      ELSIF (V_ITEM_SIA_SEQ IS NULL) AND (V_TRANS_EXIST) AND (P_ORIG_ITEM != 'Actual') THEN
        -- if yes
        -- tell end-user the original item value translated to an invalid STORET value
        SIM2_IMPORT_PKG.WRITE_ERROR(
          V_IMPORT_LOG_SEQ,
          'Line ' || V_LINE_CNT || ': ' ||P_ITEM_NAME || ', ' || P_ORIG_ITEM ||
          ', has been mapped to the invalid storet value: ' ||V_CONVERTED_ITEM,
          P_FA_SEQ
        );
      END IF;

    END IF;

    RETURN V_ITEM_SIA_SEQ;

  EXCEPTION
  
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT || ': ' || SQLERRM || ' - in validate allowable values.',
        P_FA_SEQ
      );
      RETURN NULL;

  END VALIDATE_ALVAL_ITEM;

  /******************************************************************************************************/

  FUNCTION VALIDATE_STORET_ITEM(
           P_ITEM_NAME       IN VARCHAR2,
           P_ORIG_ITEM       IN VARCHAR2,
           P_IMP_CFG_SEQ     IN NUMBER,
           P_COL_SEQ         IN NUMBER,
           P_PARENT_PK       IN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE,
           P_ORG_ID          IN VARCHAR2,
           V_LINE_CNT        IN NUMBER,
           P_FA_SEQ          IN NUMBER,
           V_IMPORT_LOG_SEQ  IN NUMBER,
           P_TRANSLATED_ITEM OUT VARCHAR2)  -- jah 3-3-2004
           RETURN SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE 
           IS

  V_CONVERTED_ITEM     VARCHAR2(4000);  -- jah 3-3-2004
  V_TRANS_EXIST        BOOLEAN := TRUE;
  V_STORET_PK          SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;
  V_INSTR_CODE         VARCHAR2(15);
  V_TRANS_OK           BOOLEAN := TRUE;
	SYS_DESC             VARCHAR2(40);

  E_NO_FUNCTION      EXCEPTION;
  E_FIELD_SIZE_ERROR EXCEPTION;

  BEGIN

  /* kms - function to look in the translations and STORET tables for items that should already exist as
  hard coded values in STORET tables */

  /* kms (5/19/01)- added p_parent_pk to parameters in order to allow validation to be dependent upon the
  presence of a valid foreign key to another STORET table, i.e. gear */

  /* look in translations table to see if it needs to swapped */
  IF p_item_name IN ('Characteristic Row ID','Habitat Assessment Row ID',
                     'Field/Lab Procedure','Field/Lab Procedure Source',
                     'Lab Sample Prep Procedure','Lab Sample Prep Procedure Source') THEN

    V_CONVERTED_ITEM := CHECK_TRANSLATIONS(
      P_ORIG_ITEM,
      P_IMP_CFG_SEQ,
      P_COL_SEQ,
      V_TRANS_EXIST,
      V_IMPORT_LOG_SEQ,
      V_LINE_CNT,
      P_FA_SEQ
    );

  ELSE

    V_STORET_PK := CHECK_STORET_TRANSLATION(
      P_ORIG_ITEM,
      P_IMP_CFG_SEQ,
      P_COL_SEQ,
      V_TRANS_EXIST,
      V_IMPORT_LOG_SEQ,
      V_LINE_CNT,
      V_CONVERTED_ITEM,  -- OUT
      P_FA_SEQ
    );

  END IF;

  /* kms - IF..THEN statement that calls function for the item being validated. */
  IF V_CONVERTED_ITEM IS NOT NULL THEN --PK IF START

    IF V_STORET_PK.IS_NUMBER IS NULL THEN --TRANS IF START

      /* call appropriate function to look in STORET table */
      IF P_ITEM_NAME = 'Project ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Project ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_PROJ_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Laboratory ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Laboratory ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_LAB_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Gear ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Gear ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_GEAR_IS_NBR(
          UPPER(v_converted_item),
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Gear Configuration ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Gear Configuration ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_GEAR_CFG_IS_NBR(
          UPPER(v_converted_item),
          P_PARENT_PK,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Sample Preservation Transport and Storage ID' OR
            P_ITEM_NAME = 'Sample Transport and Storage ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts these to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_SAMPLE_PRES_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Sample Collection Procedure ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Sample Collection Procedure ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_PROCEDURE_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Station ID' THEN
        /* sf translations should only be automatically created for stations that don't exist*/
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Station ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_STATION_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );
        IF V_STORET_PK.ORG_ID = 'XXX' THEN
          V_STORET_PK.ORG_ID := NULL;
          V_TRANS_OK         := FALSE;
        END IF;

      ELSIF P_ITEM_NAME = 'Personnel' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_PERSONNEL_IS_NBR(
          v_converted_item,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Cooperating Organization' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_CPORG_IS_NBR(
          v_converted_item,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Lab Remark Codes' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_REMARK_CODES_IS_NUM(
          v_converted_item,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Depth Units' THEN
        V_STORET_PK := GET_UOM_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

      ELSIF P_ITEM_NAME = 'Detection Limit Unit' OR
            P_ITEM_NAME = 'Laboratory Result Detection Limit Unit' THEN
        V_STORET_PK := GET_UOM_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

      ELSIF P_ITEM_NAME = 'Upper/Lower Depth Units' THEN
        V_STORET_PK := GET_UOM_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

      ELSIF P_ITEM_NAME = 'Characteristic Group ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Characteristic Group ID to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_GROUP_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Characteristic Row ID' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_ROW_IS_NBR(
          v_converted_item,
          P_PARENT_PK,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Habitat Assessment Row ID' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_HA_ROW_IS_NBR(
          v_converted_item,
          P_PARENT_PK,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Result Value Units'
         OR P_ITEM_NAME = 'Characteristic Result Value Unit' THEN
        V_STORET_PK := GET_UOM_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

      ELSIF P_ITEM_NAME = 'Characteristic Name' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_CHAR_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ,
          'CHAR'
          );

      ELSIF P_ITEM_NAME = 'Common Class Descriptor' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_CHAR_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ,
          'COMM'
        );

      ELSIF P_ITEM_NAME = 'Field/Lab Procedure' OR
            P_ITEM_NAME = 'Characteristic Field/Lab Analytical Procedure' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts these to uppercase
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_ANALPROC_IS_NBR(
          UPPER(v_converted_item),
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Field/Lab Procedure Source' OR
            P_ITEM_NAME =
            'Characteristic Field/Lab Analytical Procedure Source' THEN

        V_STORET_PK := GET_ANALPROC_SOURCE(
          v_converted_item,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Lab Sample Prep Procedure' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_LABPREP_IS_NBR(
          v_converted_item,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Lab Sample Prep Procedure Source' THEN
        V_STORET_PK := GET_LABPREP_SOURCE(
          v_converted_item,
          P_ORG_ID,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );
        
      ELSIF P_ITEM_NAME = 'Community' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_COMMUNITY_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Bio Part' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_BIOPT_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Subject Taxon' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_TAXON_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Sex' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_SEX_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Life Stage' THEN
        V_STORET_PK := SIM_IMPORT_RESULT_EXTRAS.GET_LIFE_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_LINE_CNT,
          V_IMPORT_LOG_SEQ
        );

      ELSIF P_ITEM_NAME = 'Upper/Lower Bound Units' THEN
        V_STORET_PK := GET_UOM_IS_NBR(
          v_converted_item,
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

      ELSE

        RAISE E_NO_FUNCTION;

      END IF;

      IF V_STORET_PK.IS_NUMBER IS NULL AND NOT V_TRANS_EXIST THEN
        /* check instructions to see if translation record should be created */
        
        V_INSTR_CODE := GET_INSTRUCTIONS(
          P_IMP_CFG_SEQ,
          'RES-TR',
          P_FA_SEQ,
          V_IMPORT_LOG_SEQ,
          V_LINE_CNT
        );

        IF V_INSTR_CODE = 'RES-TR-01' AND V_TRANS_OK THEN

          /* if record could not be matched to anything in translations or STORET tables,
          create new record */
          CREATE_TRANS_RECORD(
            P_ORIG_ITEM,
            P_IMP_CFG_SEQ,
            P_COL_SEQ,
            P_FA_SEQ,
            V_IMPORT_LOG_SEQ,
            V_LINE_CNT
          );
          
          SIM2_IMPORT_PKG.WRITE_ERROR(
            V_IMPORT_LOG_SEQ,
            'Line ' || V_LINE_CNT || ': ' ||
            P_ITEM_NAME ||
            '. Creating translation record for ' ||
            P_ORIG_ITEM || '.',
            P_FA_SEQ
          );
          
        END IF;

      ELSIF V_STORET_PK.IS_NUMBER IS NULL AND V_TRANS_EXIST THEN

        /*  item has been translated to an invalid storet value */
        SIM2_IMPORT_PKG.WRITE_ERROR(
          V_IMPORT_LOG_SEQ,
          'Line ' || V_LINE_CNT || ': ' ||
          P_ITEM_NAME || ', ' || P_ORIG_ITEM ||
          ' has been mapped to the invalid storet value: ' ||
          P_ORIG_ITEM,
          P_FA_SEQ
        );

      END IF;

     END IF; -- TRANS IF END                                 

    END IF;--PK IF END -- IF NOT NULL

    -- jah 3-3-2004
    BEGIN
      -- send the converted item back to the calling routine.
      P_TRANSLATED_ITEM := V_CONVERTED_ITEM;
    EXCEPTION
      WHEN OTHERS THEN
        -- Is the string buffer too small?
        IF SQLCODE = -6502 THEN
          -- if yes
          /* Do nothing. The value of p_item has too many characters, 
             therefore the value of p_item is not in STORET, therefore 
             this function has already displayed an error message.*/
          NULL;
        ELSE
          -- if no
          -- Re-raise the exception
          RAISE;
        END IF;
    END;

    RETURN V_STORET_PK;

  EXCEPTION
    
    WHEN E_NO_FUNCTION THEN
		  
      SELECT SC_DESC
			INTO   SYS_DESC
			FROM   SIM_CONFIG
			WHERE  SC_ITEM = '*BUILD_NUMBER';
       
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT ||
        SYS_DESC ||
        P_ITEM_NAME,
        P_FA_SEQ
      );
      
      V_STORET_PK.IS_NUMBER := NULL;
      V_STORET_PK.ORG_ID    := NULL;
      
      RETURN V_STORET_PK;

    WHEN OTHERS THEN

      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
        ' - in validate storet item. (' ||
        P_ITEM_NAME || ')',
        P_FA_SEQ
      );
      
      V_STORET_PK.IS_NUMBER := NULL;
      V_STORET_PK.ORG_ID    := NULL;
      
      RETURN V_STORET_PK;

  END VALIDATE_STORET_ITEM;

  /******************************************************************************************************/
FUNCTION CHECK_STORET_TRANSLATION(
         P_CODE           IN VARCHAR2,
         P_IMP_CFG_SEQ    IN NUMBER,
         P_COL_SEQ        IN NUMBER,
         P_TRANS_EXIST    OUT BOOLEAN,                                  
         V_IMPORT_LOG_SEQ IN NUMBER,
         V_LINE_CNT       IN NUMBER,
         V_CONVERTED_ITEM OUT VARCHAR2,
         p_activity_seq   IN NUMBER) 
  RETURN T_STORET_PK_TYPE 
  IS  
  
  v_storet_pk      SIM_IMPORT_RESULT_EXTRAS.T_STORET_PK_TYPE;
  v_seq            NUMBER(10);
  v_table_name     VARCHAR2(80);
  v_column_name    VARCHAR2(80);
  v_dyna_cursor    VARCHAR2(200);
  v_comma_pos      NUMBER(10);
  
  TYPE SORT_CURSOR_TYPE IS REF CURSOR;
  sort_cursor            sort_cursor_type;

  CURSOR c_get_storet_key(
         cp_code        VARCHAR2,
         cp_imp_cfg_seq NUMBER,
         cp_col_seq     NUMBER) IS
  SELECT st_seq, st_is_number, st_org_id, NULL
  FROM   sim_translations, sim_imp_cfg_dets
  WHERE  st_user_code   = cp_code
  AND    st_sicdt_seq   = sicdt_seq
  AND    sicdt_sicc_seq = cp_imp_cfg_seq
  AND    sicdt_sic_seq  = cp_col_seq;

  CURSOR c_imp_cols(cp_col_seq NUMBER) IS
  SELECT sic_storet_tab,sic_storet_col
  FROM   sim_imp_cols
  WHERE  sic_seq = cp_col_seq;  
      
  CURSOR c_comma(cp_col_seq NUMBER) IS
  SELECT INSTR(sic_storet_col,',') col_pos
  FROM   sim_imp_cols
  WHERE  sic_seq = cp_col_seq;      
      
  CURSOR c_inst_col(cp_col_seq NUMBER) IS
  SELECT SUBSTR(SIC_STORET_COL,1,INSTR(SIC_STORET_COL,',')-1) col_name
  FROM   sim_imp_cols
  WHERE  sic_seq = cp_col_seq;
        
BEGIN

  -- Try to retreve the storet key from the translations table
  OPEN  c_get_storet_key(p_code,p_imp_cfg_seq,p_col_seq);
  FETCH c_get_storet_key INTO v_seq,v_storet_pk.is_number,v_storet_pk.org_id,v_converted_item;
  CLOSE c_get_storet_key;

  -- did we retrieve a translation from the translations table?
  IF v_seq IS NULL THEN
    -- if no
    
    -- return the original value
    V_CONVERTED_ITEM := p_code;
    
    -- Tell the caller that the return value is the original value
    P_TRANS_EXIST  := FALSE;
    
  ELSIF v_seq IS NOT NULL THEN
    -- if yes
  
    /* if the code is in the translations table, flag so it's not recreated */
    
    -- Tell the caller that the return value is a translated value
    P_TRANS_EXIST := TRUE;
    
    -- dis we fail to get the storet key for the translated value?
    IF v_storet_pk.is_number IS NULL THEN
      -- if yes
      
      /*
        if there is a code in the translations table but no value to translate to,
        write error message. The calling routine is responsible for changing
        the migrate status to 'N'.
      */
      Sim2_Import_Pkg.write_error(
        v_import_log_seq, 
        'Line ' || v_line_cnt || ': ' || p_code || 
        ' exists in translations table but has not been translated to a value.',
        p_activity_seq
      );
      
    ELSE
    
      -- if no (we got the storet key for the translated value)

      -- Retrieve the STORET table name and column name of the item
      OPEN  c_imp_cols(p_col_seq);
      FETCH c_imp_cols INTO v_table_name,v_column_name;
      CLOSE c_imp_cols;

      OPEN  c_comma(p_col_seq);
      FETCH c_comma INTO v_comma_pos;
      CLOSE c_comma;
      
      IF v_comma_pos = 0 THEN

        v_dyna_cursor := 'SELECT RTRIM('||v_column_name||') '||
          ' FROM '||v_table_name||
          ' WHERE '||v_table_name||'_IS_NUMBER = '||v_storet_pk.is_number||
          ' AND '||v_table_name||'_ORG_ID = '''||v_storet_pk.org_id||'''';
          
      ELSE
      
        OPEN c_inst_col(p_col_seq);
        FETCH c_inst_col INTO v_column_name;
        CLOSE c_inst_col;

        v_dyna_cursor := 'SELECT RTRIM('||v_column_name||') '||
          ' FROM '||v_table_name||
          ' WHERE '||v_table_name||'_IS_NUMBER = '||v_storet_pk.is_number||
          ' AND '||v_table_name||'_ORG_ID = '''||v_storet_pk.org_id||'''';

      END IF; 
      
      OPEN sort_cursor FOR v_dyna_cursor;

      LOOP
        FETCH sort_cursor INTO v_converted_item;
        EXIT WHEN sort_cursor%NOTFOUND;
      END LOOP;      

      IF v_converted_item IS NOT NULL THEN
        P_TRANS_EXIST := TRUE;
      END IF;

		END IF;

  END IF;  

  RETURN v_storet_pk;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT || ': ' || SQLERRM || 
        ' - in check storet translations',
        p_activity_seq
      );
      RETURN NULL;
      
END CHECK_STORET_TRANSLATION;

/***********************************************************************************************************/
  FUNCTION CHECK_TRANSLATIONS(P_CODE           IN VARCHAR2,
                              P_IMP_CFG_SEQ    IN NUMBER,
                              P_COL_SEQ        IN NUMBER,
                              P_TRANS_EXIST    OUT BOOLEAN,
                              V_IMPORT_LOG_SEQ IN NUMBER,
                              V_LINE_CNT       IN NUMBER,
                              p_activity_seq   IN NUMBER) 
                              RETURN VARCHAR2 IS

    V_STORET_VALUE VARCHAR2(4000);

    CURSOR C_GET_STORET_CODE(
      P_CODE        IN VARCHAR2, 
      P_IMP_CFG_SEQ IN NUMBER, 
      P_COL_SEQ     IN NUMBER) IS
    SELECT ST_STORET_CODE
    FROM   SIM_TRANSLATIONS, SIM_IMP_CFG_DETS
    WHERE  ST_USER_CODE = P_CODE 
    AND    ST_SICDT_SEQ = SICDT_SEQ 
    AND    SICDT_SICC_SEQ = P_IMP_CFG_SEQ 
    AND    SICDT_SIC_SEQ = P_COL_SEQ;

  BEGIN

    -- try to find a translation for this item
    OPEN C_GET_STORET_CODE(P_CODE, P_IMP_CFG_SEQ, P_COL_SEQ);
    FETCH C_GET_STORET_CODE INTO V_STORET_VALUE;
    
    -- did we find a translation?
    IF C_GET_STORET_CODE%FOUND THEN
      -- if yes

      -- indicate that item IS translated
      P_TRANS_EXIST := TRUE;
      
      -- did the item translate to NULL?
      IF v_storet_value IS NULL THEN
        -- if yes
        -- write error message
        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'Line ' || v_line_cnt || ': ' || p_code ||
          ' exists in translations table but has not been translated to a value.',
          p_activity_seq
        );
        -- Notice that V_STATUS must be set to 'N' by the calling routine...
			END IF;
      
    ELSE
      -- if no (translation was not found)
      
      -- return original value
      V_STORET_VALUE := P_CODE;
      -- indicate that this function returned the original value
      P_TRANS_EXIST  := FALSE;
      
    END IF;
    
    CLOSE C_GET_STORET_CODE;
    
    RETURN V_STORET_VALUE;
    
  EXCEPTION
  
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        'Line ' || V_LINE_CNT || ': ' || SQLERRM || ' - in check translations',
        p_activity_seq
      );
      RETURN NULL;
      
  END CHECK_TRANSLATIONS;
  
  /******************************************************************************************************/
  PROCEDURE CREATE_TRANS_RECORD(P_CODE           IN VARCHAR2,
                              P_IMP_CFG_SEQ    IN NUMBER,
                              P_COL_SEQ        IN NUMBER,
                              P_FA_SEQ         IN NUMBER,
                              V_IMPORT_LOG_SEQ IN NUMBER,
                              V_LINE_CNT       IN NUMBER) IS

  V_ST_SEQ    NUMBER(10);
  V_SICDT_SEQ NUMBER(10);

  E_IMP_CFG_DETS EXCEPTION;
  E_TRANS_ERROR EXCEPTION;

  CURSOR C_GET_CFG_DET_SEQ(P_IMP_CFG_SEQ IN NUMBER, P_COL_SEQ IN NUMBER) IS
    SELECT SICDT_SEQ
      FROM SIM_IMP_CFG_DETS
     WHERE SICDT_SICC_SEQ = P_IMP_CFG_SEQ AND SICDT_SIC_SEQ = P_COL_SEQ;

  CURSOR C_GET_ST_SEQ IS
    SELECT SIM_TRANSLATIONS_SEQ.NEXTVAL FROM DUAL;

  BEGIN

  /* kms - if it has been determined that value won't work for STORET, that a translation
    record does not already exist, and that the user wants to autocreate them, this function
    will create half of a translation record.  The remaining half will need to be filled in
    by the user after the import has completed. */

    OPEN C_GET_CFG_DET_SEQ(P_IMP_CFG_SEQ, P_COL_SEQ);
    FETCH C_GET_CFG_DET_SEQ
      INTO V_SICDT_SEQ;
    IF C_GET_CFG_DET_SEQ%NOTFOUND THEN
      V_SICDT_SEQ := NULL;
      RAISE E_IMP_CFG_DETS;
    END IF;
    CLOSE C_GET_CFG_DET_SEQ;

    /* get the sequence for the new record in translations table */
    OPEN C_GET_ST_SEQ;
    FETCH C_GET_ST_SEQ
      INTO V_ST_SEQ;
    IF C_GET_ST_SEQ%NOTFOUND THEN
      V_ST_SEQ := NULL;
      RAISE E_TRANS_ERROR;
    END IF;
    CLOSE C_GET_ST_SEQ;

    INSERT INTO SIM_TRANSLATIONS
      (ST_SEQ, ST_SICDT_SEQ, ST_USER_CODE, ST_STORET_CODE)
    VALUES
      (V_ST_SEQ, /* st_seq */
       V_SICDT_SEQ, /* st_sicdt_seq */
       P_CODE, /* st_user_code */
       NULL /* st_storet_code */);

  EXCEPTION

    WHEN E_IMP_CFG_DETS THEN
      RAISE_APPLICATION_ERROR(-20101,
                              'Unable to find import configuration');

    WHEN E_TRANS_ERROR THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT ||
                                   ': Unable to create new record in translations table - error getting sequence',
                                  P_FA_SEQ);

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  'Line ' || V_LINE_CNT || ': ' || SQLERRM ||
                                   ' - adding record to translations table.',
                                  P_FA_SEQ);

  END CREATE_TRANS_RECORD;

  /******************************************************************************************************/

  FUNCTION CREATE_SIM_STATION_VISIT(
           P_VISIT_REC      IN SIM_STATION_VISITS%ROWTYPE,
           V_IMPORT_LOG_SEQ IN NUMBER) 
           RETURN NUMBER IS

  BEGIN
    /* create sim_station_visits using values in visit record */
    INSERT INTO SIM_STATION_VISITS
    ( 
      SSV_SEQ,
      SSV_STP_SEQ,
      SSV_ID,
      SSV_START_DATE,
      SSV_START_TIME,
      SSV_END_DATE,
      SSV_END_TIME,
      SSV_TSMSTATN_IS_NUMBER,
      SSV_TSMSTATN_ORG_ID,
      SSV_TSRSTVST_IS_NUMBER,
      SSV_TSRSTVST_ORG_ID,
      SSV_TSMORGAN_IS_NUMBER,
      SSV_TSMORGAN_ORG_ID,
      SSV_CREATED_BY_SIM,
      SSV_IMPORT_SEQ,
      SSV_DOCUMENT
    )
    VALUES
    (
      P_VISIT_REC.SSV_SEQ,                /* SSV_SEQ */
      P_VISIT_REC.SSV_STP_SEQ,            /* SSV_STP_SEQ */
      P_VISIT_REC.SSV_ID,                 /* SSV_ID */
      P_VISIT_REC.SSV_START_DATE,         /* SSV_START_DATE */
      P_VISIT_REC.SSV_START_TIME,         /* SSV_START_TIME */
      P_VISIT_REC.SSV_END_DATE,           /* SSV_END_DATE */
      P_VISIT_REC.SSV_END_TIME,           /* SSV_END_TIME */
      P_VISIT_REC.SSV_TSMSTATN_IS_NUMBER, /* SSV_TSMSTATN_IS_NUMBER */
      P_VISIT_REC.SSV_TSMSTATN_ORG_ID,    /* SSV_TSMSTATN_ORG_ID */
      P_VISIT_REC.SSV_TSRSTVST_IS_NUMBER, /* SSV_TSRSTVST_IS_NUMBER */
      P_VISIT_REC.SSV_TSRSTVST_ORG_ID,    /* SSV_TSRSTVST_ORG_ID */
      P_VISIT_REC.SSV_TSMORGAN_IS_NUMBER, /* SSV_TSMORGAN_IS_NUMBER */
      P_VISIT_REC.SSV_TSMORGAN_ORG_ID,    /* SSV_TSMORGAN_ORG_ID */
      P_VISIT_REC.SSV_CREATED_BY_SIM,     /* SSV_CREATED_BY_SIM */
      P_VISIT_REC.SSV_IMPORT_SEQ,         /* SSV_IMPORT_SEQ */
      P_VISIT_REC.SSV_DOCUMENT            /* SSV_DOCUMENT */
    );

    -- tell calling routine insert succeeded
    RETURN 1;

  EXCEPTION
  
    WHEN OTHERS THEN

      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM ||
        ' - creating sim station_visit.',
        NULL
      );
      
      -- tell calling routine insert failed
      RETURN NULL;

  END CREATE_SIM_STATION_VISIT;

  /******************************************************************************************************/
  FUNCTION CREATE_SIM_TRIP(P_TRIP_REC       IN OUT SIM_TRIPS%ROWTYPE,
                           P_FA_SEQ         IN NUMBER,
                           V_IMPORT_LOG_SEQ IN NUMBER) 
                           RETURN NUMBER IS
  /* jah 3-5-2004 
     Changed parameter P_TRIP_REC from IN to IN OUT 
     so that STP_SEQ passes back to the calling 
     routine. */
  
  BEGIN

    /* create sim_trips using values in trip record */
    INSERT INTO SIM_TRIPS
    (
      STP_SEQ,
      STP_TRIP_ID,
      STP_TRIP_NAME,
      STP_START_DATE,
      STP_END_DATE,
      STP_TSRTRIP_IS_NUMBER,
      STP_TSRTRIP_ORG_ID,
      STP_TSMORGAN_IS_NUMBER,
      STP_TSMORGAN_ORG_ID,
      STP_CREATED_BY_SIM,
      STP_IMPORT_SEQ
    )
    VALUES
    ( 
      P_TRIP_REC.STP_SEQ, /* STP_SEQ */
      P_TRIP_REC.STP_TRIP_ID, /* STP_TRIP_ID */
      P_TRIP_REC.STP_TRIP_NAME,
      P_TRIP_REC.STP_START_DATE, /* STP_START_DATE */
      P_TRIP_REC.STP_END_DATE, /* STP_END_DATE */
      P_TRIP_REC.STP_TSRTRIP_IS_NUMBER, /* STP_TSRTRIP_IS_NUMBER */
      P_TRIP_REC.STP_TSRTRIP_ORG_ID, /* STP_TSRTRIP_ORG_ID */
      P_TRIP_REC.STP_TSMORGAN_IS_NUMBER, /* STP_TSMORGAN_IS_NUMBER */
      P_TRIP_REC.STP_TSMORGAN_ORG_ID, /* STP_TSMORGAN_ORG_ID */
      P_TRIP_REC.STP_CREATED_BY_SIM, /* STP_CREATED_BY_SIM */
      P_TRIP_REC.STP_IMPORT_SEQ
    );

    -- tell the calling routine the insert succeeded
    RETURN 1;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM || ' - creating SIM trip.',
        P_FA_SEQ
      );
      
      -- tell the calling routine the insert failed
      RETURN NULL;

  END CREATE_SIM_TRIP;

  /******************************************************************************************************/
  FUNCTION CREATE_FIELD_ACTIVITY(P_FA_REC         IN SIM_FIELD_ACTIVITIES%ROWTYPE,
                                 V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

  BEGIN

    /* create field activity using values in record */
    INSERT INTO SIM_FIELD_ACTIVITIES
      (FA_SEQ,
       FA_TYPE_NAME,
       FA_CATEGORY,
       FA_ACTIVITY_ID,
       FA_START_DATE,
       FA_START_TIME,
       FA_STOP_DATE,
       FA_STOP_TIME,
       FA_DEPTH,
       FA_DEPTH_REF_POINT,
       FA_ACTIVITY_COMMENTS,
       FA_TSMSTATN_IS_NUMBER,
       FA_TSMSTATN_ORG_ID,
       FA_TSRTRIP_IS_NUMBER,
       FA_TSRTRIP_ORG_ID,
       FA_TSRSTVST_IS_NUMBER,
       FA_TSRSTVST_ORG_ID,
       FA_TSRFLDPR_IS_NUMBER,
       FA_TSRFLDPR_ORG_ID,
       FA_TSRFLDGR_IS_NUMBER,
       FA_TSRFLDGR_ORG_ID,
       FA_IMPORT_SEQ,
       FA_EXPORT_SEQ,
       FA_STATUS,
       FA_TSMORGAN_IS_NUMBER,
       FA_TSMORGAN_ORG_ID,
       FA_APPROVED_BY,
       FA_APPROVED_ON,
       FA_STVST_COMMENTS,
       FA_DEPTH_UOM,
       FA_MEDIUM,
       FA_TSRLAB_IS_NUMBER,
       FA_TSRLAB_ORG_ID,
       FA_IMPORT_LINE_NO,
       FA_TSRFDACT_IS_NUMBER,
       FA_TSRFDACT_ORG_ID,
       FA_REPLICATE_NBR,
       FA_FDSET_ID,
       FA_FDSET_NAME,
       FA_UPDEPTH,
       FA_LOWDEPTH,
       FA_THERMO,
       FA_HALO,
       FA_PYCNO,
       FA_REL_DEPTH,
       FA_DEPTH_ZONE,
       FA_DEPTH_RANGE_UOM,
       FA_START_TIME_ZONE,
       FA_STOP_TIME_ZONE,
       FA_TSRGRCFG_IS_NUMBER,
       FA_TSRGRCFG_ORG_ID,
       FA_TSRSDP_IS_NUMBER,
       FA_TSRSDP_ORG_ID,
       FA_STP_SEQ,
       FA_SSV_SEQ,
       FA_INTENT,
       FA_TSMPRMVL_COMMUNITY_IS_NBR,
       FA_TSMPRMVL_COMMUNITY_ORG_ID,
       FA_TSRBIOPT_IS_NUMBER,
       FA_TSRBIOPT_ORG_ID,
       FA_TSRCHAR_IS_NUMBER,
       FA_TSRCHAR_ORG_ID,
       FA_CHAIN_OF_CUSTODY_ID,
       FA_QC_INDICATOR,
       FA_TSRMATRX_IS_NUMBER,
       FA_TSRMATRX_ORG_ID,
       FA_SPECIES_NUMBER,
       FA_CHAR_GROUP,
       FA_DOCUMENT)
    VALUES
      (P_FA_REC.FA_SEQ,
       P_FA_REC.FA_TYPE_NAME,
       P_FA_REC.FA_CATEGORY,
       P_FA_REC.FA_ACTIVITY_ID,
       P_FA_REC.FA_START_DATE,
       P_FA_REC.FA_START_TIME,
       P_FA_REC.FA_STOP_DATE,
       P_FA_REC.FA_STOP_TIME,
       P_FA_REC.FA_DEPTH,
       P_FA_REC.FA_DEPTH_REF_POINT,
       P_FA_REC.FA_ACTIVITY_COMMENTS,
       P_FA_REC.FA_TSMSTATN_IS_NUMBER,
       P_FA_REC.FA_TSMSTATN_ORG_ID,
       P_FA_REC.FA_TSRTRIP_IS_NUMBER,
       P_FA_REC.FA_TSRTRIP_ORG_ID,
       P_FA_REC.FA_TSRSTVST_IS_NUMBER,
       P_FA_REC.FA_TSRSTVST_ORG_ID,
       P_FA_REC.FA_TSRFLDPR_IS_NUMBER,
       P_FA_REC.FA_TSRFLDPR_ORG_ID,
       P_FA_REC.FA_TSRFLDGR_IS_NUMBER,
       P_FA_REC.FA_TSRFLDGR_ORG_ID,
       P_FA_REC.FA_IMPORT_SEQ,
       P_FA_REC.FA_EXPORT_SEQ,
       P_FA_REC.FA_STATUS,
       P_FA_REC.FA_TSMORGAN_IS_NUMBER,
       P_FA_REC.FA_TSMORGAN_ORG_ID,
       P_FA_REC.FA_APPROVED_BY,
       P_FA_REC.FA_APPROVED_ON,
       P_FA_REC.FA_STVST_COMMENTS,
       P_FA_REC.FA_DEPTH_UOM,
       P_FA_REC.FA_MEDIUM,
       NULL, --p_fa_rec.FA_TSRLAB_IS_NUMBER,
       NULL, --p_fa_rec.FA_TSRLAB_ORG_ID,
       P_FA_REC.FA_IMPORT_LINE_NO,
       P_FA_REC.FA_TSRFDACT_IS_NUMBER,
       P_FA_REC.FA_TSRFDACT_ORG_ID,
       P_FA_REC.FA_REPLICATE_NBR,
       P_FA_REC.FA_FDSET_ID,
       P_FA_REC.FA_FDSET_NAME,
       P_FA_REC.FA_UPDEPTH,
       P_FA_REC.FA_LOWDEPTH,
       P_FA_REC.FA_THERMO,
       P_FA_REC.FA_HALO,
       P_FA_REC.FA_PYCNO,
       P_FA_REC.FA_REL_DEPTH,
       P_FA_REC.FA_DEPTH_ZONE,
       P_FA_REC.FA_DEPTH_RANGE_UOM,
       P_FA_REC.FA_START_TIME_ZONE,
       P_FA_REC.FA_STOP_TIME_ZONE,
       P_FA_REC.FA_TSRGRCFG_IS_NUMBER,
       P_FA_REC.FA_TSRGRCFG_ORG_ID,
       P_FA_REC.FA_TSRSDP_IS_NUMBER,
       P_FA_REC.FA_TSRSDP_ORG_ID,
       P_FA_REC.FA_STP_SEQ,
       P_FA_REC.FA_SSV_SEQ,
       P_FA_REC.FA_INTENT,
       P_FA_REC.FA_TSMPRMVL_COMMUNITY_IS_NBR,
       P_FA_REC.FA_TSMPRMVL_COMMUNITY_ORG_ID,
       P_FA_REC.FA_TSRBIOPT_IS_NUMBER,
       P_FA_REC.FA_TSRBIOPT_ORG_ID,
       P_FA_REC.FA_TSRCHAR_IS_NUMBER,
       P_FA_REC.FA_TSRCHAR_ORG_ID,
       P_FA_REC.FA_CHAIN_OF_CUSTODY_ID,
       P_FA_REC.FA_QC_INDICATOR,
       P_FA_REC.FA_TSRMATRX_IS_NUMBER,
       P_FA_REC.FA_TSRMATRX_ORG_ID,
       P_FA_REC.FA_SPECIES_NUMBER,
       P_FA_REC.FA_CHAR_GROUP,
       P_FA_REC.FA_DOCUMENT);

    COMMIT;

    RETURN 1;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting field activity.',
                                  P_FA_REC.FA_SEQ);
      RETURN 0;

  END CREATE_FIELD_ACTIVITY;
  /******************************************************************************************************/

  FUNCTION CREATE_FIELD_ACTIVITY_PARENT(
           p_fap_rec        IN SIM_FIELD_ACTIVITY_PARENT%ROWTYPE,
           p_import_log_seq IN NUMBER)
           RETURN NUMBER
  IS
  
  BEGIN
  
    INSERT INTO SIM_FIELD_ACTIVITY_PARENT
    (
      FAP_SEQ,
      FAP_FA_SEQ,
      FAP_PARENT_FA_SEQ,
      FAP_PARENT_IS_NUMBER,
      FAP_PARENT_ORG_ID
    )
    VALUES
    (
      p_fap_rec.FAP_SEQ,
      p_fap_rec.FAP_FA_SEQ,
      p_fap_rec.FAP_PARENT_FA_SEQ,
      p_fap_rec.FAP_PARENT_IS_NUMBER,
      p_fap_rec.FAP_PARENT_ORG_ID
    );
    
    COMMIT;

    RETURN NULL;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        p_import_log_seq,
        SQLERRM || ' - inserting field activity parent.',
        p_fap_rec.FAP_SEQ);
      RETURN 1;

  END Create_Field_Activity_Parent;
  
  
  /******************************************************************************************************/
  /* FUNCTION TO INSERT A SIM_RESULT_CLASS_INDICATOR RECORD  */

  FUNCTION CREATE_RCI(
           P_RCI_REC        IN SIM_RESULT_CLASS_INDICATOR%ROWTYPE,
           V_IMPORT_LOG_SEQ IN NUMBER) 
           RETURN NUMBER IS
           
  BEGIN
    /* create rci using values in record */
    INSERT INTO SIM_RESULT_CLASS_INDICATOR
      (RCI_SEQ,
       RCI_RS_SEQ,
       RCI_FA_SEQ,
       RCI_PRIMARY_CLASS_IND,
       RCI_SECOND_CLASS_IND,
       RCI_LOWER_BND_AMT,
       RCI_UPPER_BND_AMT,
       RCI_TSRUOM_IS_NUMBER,
       RCI_TSRUOM_ORG_ID,
       RCI_PARTICLE_SIZE_BASIS,  -- jah 2-23-04 Added Particle Size Basis
       RCI_BRG_SEQ,
       RCI_IMPORT_SEQ)
    VALUES
      (P_RCI_REC.RCI_SEQ,
       P_RCI_REC.RCI_RS_SEQ,
       P_RCI_REC.RCI_FA_SEQ,
       P_RCI_REC.RCI_PRIMARY_CLASS_IND,
       P_RCI_REC.RCI_SECOND_CLASS_IND,
       P_RCI_REC.RCI_LOWER_BND_AMT,
       P_RCI_REC.RCI_UPPER_BND_AMT,
       P_RCI_REC.RCI_TSRUOM_IS_NUMBER,
       P_RCI_REC.RCI_TSRUOM_ORG_ID,
       P_RCI_REC.RCI_PARTICLE_SIZE_BASIS,  -- jah 2-23-04 Added Particle Size Basis
       P_RCI_REC.RCI_BRG_SEQ,
       P_RCI_REC.RCI_IMPORT_SEQ);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(
        V_IMPORT_LOG_SEQ,
        SQLERRM || ' - inserting to SIM_RESULT_CLASS_INDICATOR.',
        P_RCI_REC.RCI_FA_SEQ);
      RETURN 0;

  END CREATE_RCI;

  /***********************************************************************************************/
  /* FUNCTION TO INSERT A SIM_BIO_RESULTS_GRP_INDIVIDUAL RECORD  */

  FUNCTION CREATE_BRGI(P_BRGI_REC       IN SIM_BIO_RESULTS_GRP_INDIVIDUAL%ROWTYPE,
                       V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS
  BEGIN
    /* create brgi using values in record */
    INSERT INTO SIM_BIO_RESULTS_GRP_INDIVIDUAL
      (BRGI_SEQ,
       BRGI_BRG_SEQ,
       BRGI_INDIVIDUAL_NUMBER,
       BRGI_FA_SEQ,
       BRGI_IMPORT_SEQ)
    VALUES
      (P_BRGI_REC.BRGI_SEQ,
       P_BRGI_REC.BRGI_BRG_SEQ,
       P_BRGI_REC.BRGI_INDIVIDUAL_NUMBER,
       P_BRGI_REC.BRGI_FA_SEQ,
       P_BRGI_REC.BRGI_IMPORT_SEQ);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                   ' - inserting to SIM_BIO_RESULTS_GRP_INDIVIDUAL.',
                                  P_BRGI_REC.BRGI_FA_SEQ);
      RETURN 0;
  END CREATE_BRGI;

  /***********************************************************************************************/
  /* FUNCTION TO INSERT A SIM_BIO_RESULTS_GRP RECORD */

  FUNCTION CREATE_BRG(P_BRG_REC        IN SIM_BIO_RESULTS_GRP%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS
  BEGIN
    /* create brg using values in record */

    INSERT INTO SIM_BIO_RESULTS_GRP
      (BRG_SEQ,
       BRG_FA_SEQ,
       BRG_ID_CODE,
       BRG_TYPE,
       BRG_SPECIES_NUMBER,
       BRG_DESCRIPTION_TEXT,
       BRG_FREQ_ANALYSIS,
       BRG_TSRCPV_SEX_IS_NUMBER,
       BRG_TSRCPV_SEX_ORG_ID,
       BRG_TSRCPV_LIFESTAGE_IS_NBR,
       BRG_TSRCPV_LIFESTAGE_ORG_ID,
       BRG_VALUE_TYPE_NAME,
       BRG_SUMMARY_GRP_COUNT,
       BRG_TSRCHAR_IS_NUMBER,
       BRG_TSRCHAR_ORG_ID,
       BRG_IMPORT_SEQ)
    VALUES
      (P_BRG_REC.BRG_SEQ,
       P_BRG_REC.BRG_FA_SEQ,
       P_BRG_REC.BRG_ID_CODE,
       P_BRG_REC.BRG_TYPE,
       P_BRG_REC.BRG_SPECIES_NUMBER,
       P_BRG_REC.BRG_DESCRIPTION_TEXT,
       P_BRG_REC.BRG_FREQ_ANALYSIS,
       P_BRG_REC.BRG_TSRCPV_SEX_IS_NUMBER,
       P_BRG_REC.BRG_TSRCPV_SEX_ORG_ID,
       P_BRG_REC.BRG_TSRCPV_LIFESTAGE_IS_NBR,
       P_BRG_REC.BRG_TSRCPV_LIFESTAGE_ORG_ID,
       P_BRG_REC.BRG_VALUE_TYPE_NAME,
       P_BRG_REC.BRG_SUMMARY_GRP_COUNT,
       P_BRG_REC.BRG_TSRCHAR_IS_NUMBER,
       P_BRG_REC.BRG_TSRCHAR_ORG_ID,
       P_BRG_REC.BRG_IMPORT_SEQ);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                   ' - inserting to SIM_BIO_RESULTS_GRP.',
                                  P_BRG_REC.BRG_FA_SEQ);
      RETURN 0;
  END CREATE_BRG;
  /***********************************************************************************************/
  /* FUNCTION TO INSERT A SIM_SAMPLE RECORD */

  FUNCTION CREATE_SAMPLE(P_SAMPLE_REC     IN SIM_SAMPLE%ROWTYPE,
                         V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS
  BEGIN
    /* create sim_sample using values in record */
    INSERT INTO SIM_SAMPLE
      (SS_SEQ,
       SS_FA_SEQ,
       SS_TOTAL_VOLUME_MSR,
       SS_TOTAL_VOL_UOM,
       SS_TOTAL_WEIGHT_MSR,
       SS_TOTAL_WT_UOM,
       SS_DATA_FILE_NAME_LOC,
       SS_CONTAINER_TYPE_NM,
       SS_CONTAINER_COLOR,
       SS_CONTAINER_SIZE_MSR,
       SS_CONTAINER_SIZE_UN,
       SS_TEMP_PRESERVN_TYPE,
       SS_PARENT_IND_CODE,
       SS_GEAR_DEPLOYMENT,
       SS_TRANSPORT_STORAGE,
       SS_PRESRV_STRGE_PRCDR,
       SS_DURATION_TIME,
       SS_DURATION_UNIT)
    VALUES
      (P_SAMPLE_REC.SS_SEQ,
       P_SAMPLE_REC.SS_FA_SEQ,
       P_SAMPLE_REC.SS_TOTAL_VOLUME_MSR,
       P_SAMPLE_REC.SS_TOTAL_VOL_UOM,
       P_SAMPLE_REC.SS_TOTAL_WEIGHT_MSR,
       P_SAMPLE_REC.SS_TOTAL_WT_UOM,
       P_SAMPLE_REC.SS_DATA_FILE_NAME_LOC,
       P_SAMPLE_REC.SS_CONTAINER_TYPE_NM,
       P_SAMPLE_REC.SS_CONTAINER_COLOR,
       P_SAMPLE_REC.SS_CONTAINER_SIZE_MSR,
       P_SAMPLE_REC.SS_CONTAINER_SIZE_UN,
       P_SAMPLE_REC.SS_TEMP_PRESERVN_TYPE,
       P_SAMPLE_REC.SS_PARENT_IND_CODE,
       P_SAMPLE_REC.SS_GEAR_DEPLOYMENT,
       P_SAMPLE_REC.SS_TRANSPORT_STORAGE,
       P_SAMPLE_REC.SS_PRESRV_STRGE_PRCDR,
       P_SAMPLE_REC.SS_DURATION_TIME,
       P_SAMPLE_REC.SS_DURATION_UNIT);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting to SIM_SAMPLE.',
                                  P_SAMPLE_REC.SS_FA_SEQ);
      RETURN 0;
  END CREATE_SAMPLE;

  /***********************************************************************************************/
  /* FUNCTION TO INSERT A SIM_TRAWL_OPS_DET RECORD  */

  FUNCTION CREATE_TOD(P_TOD_REC        IN SIM_TRAWL_OPS_DET%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS
  BEGIN
    /* create tod using values in record */
    INSERT INTO SIM_TRAWL_OPS_DET
      (TOD_SEQ,
       TOD_FA_SEQ,
       TOD_FISHED_DURATION,
       TOD_FISHED_DUR_UOM,
       TOD_BOAT_SPEED_MSR,
       TOD_BOAT_SPEED_UOM,
       TOD_FISHED_DISTANCE,
       TOD_FISHED_DISTANCE_UOM,
       TOD_REL_CURRENT_DIR,
       TOD_REL_WIND_DIR,
       TOD_COMMENT_TEXT)
    VALUES
      (P_TOD_REC.TOD_SEQ,
       P_TOD_REC.TOD_FA_SEQ,
       P_TOD_REC.TOD_FISHED_DURATION,
       P_TOD_REC.TOD_FISHED_DUR_UOM,
       P_TOD_REC.TOD_BOAT_SPEED_MSR,
       P_TOD_REC.TOD_BOAT_SPEED_UOM,
       P_TOD_REC.TOD_FISHED_DISTANCE,
       P_TOD_REC.TOD_FISHED_DISTANCE_UOM,
       P_TOD_REC.TOD_REL_CURRENT_DIR,
       P_TOD_REC.TOD_REL_WIND_DIR,
       P_TOD_REC.TOD_COMMENT_TEXT);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM ||
                                   ' - inserting to SIM_TRAWL_OPS_DET.',
                                  P_TOD_REC.TOD_FA_SEQ);
      RETURN 0;
  END CREATE_TOD;

  /***********************************************************************************************/
  /* FUNCTION TO INSERT A SIM_STATION_LOCATIONS (stl) RECORD */

  FUNCTION CREATE_STL(P_STL_REC        IN SIM_STATION_LOCATIONS%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

  BEGIN
    /* Insert location using values in record */

    INSERT INTO SIM_STATION_LOCATIONS
      (STL_SEQ,
       STL_TYPE_CODE,
       STL_SEQUENCE_NUMBER,
       STL_TSMSTATN0IS_NUMBER,
       STL_TSMSTATN0ORG_ID,
       STL_LAT_DEGREES,
       STL_LAT_MINUTES,
       STL_LAT_SECONDS,
       STL_LAT_DEC_MINUTES,
       STL_LONG_DEGREES,
       STL_LONG_MINUTES,
       STL_LONG_SECONDS,
       STL_LONG_DEC_MINUTES,
       STL_LAT_DIR,
       STL_LONG_DIR,
       STL_GEOPOS_METHOD,
       STL_GEOPOS_DATUM)
    VALUES
      (P_STL_REC.STL_SEQ,
       P_STL_REC.STL_TYPE_CODE,
       P_STL_REC.STL_SEQUENCE_NUMBER,
       P_STL_REC.STL_TSMSTATN0IS_NUMBER,
       P_STL_REC.STL_TSMSTATN0ORG_ID,
       P_STL_REC.STL_LAT_DEGREES,
       P_STL_REC.STL_LAT_MINUTES,
       P_STL_REC.STL_LAT_SECONDS,
       P_STL_REC.STL_LAT_DEC_MINUTES,
       P_STL_REC.STL_LONG_DEGREES,
       P_STL_REC.STL_LONG_MINUTES,
       P_STL_REC.STL_LONG_SECONDS,
       P_STL_REC.STL_LONG_DEC_MINUTES,
       P_STL_REC.STL_LAT_DIR,
       P_STL_REC.STL_LONG_DIR,
       P_STL_REC.STL_GEOPOS_METHOD,
       P_STL_REC.STL_GEOPOS_DATUM);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting stl.',
                                  NULL);
      RETURN 0;
  END CREATE_STL;

  /***********************************************************************************************/
  /* FUNCTION TO INSERT A SIM_ACTUAL_ACTIVITY_LOC (aal) RECORD */

  FUNCTION CREATE_AAL(P_AAL_REC        IN SIM_ACTUAL_ACTIVITY_LOC%ROWTYPE,
                      V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

  BEGIN
    /* Insert aal using values in record */

    INSERT INTO SIM_ACTUAL_ACTIVITY_LOC
      (AAL_SEQ,
       AAL_FA_SEQ,
       AAL_STL_SEQ,
       AAL_TSMALP_IS_NUMBER,
       AAL_TSMALP_ORG_ID,
       AAL_TSRAAL_TYPE_NAME,
       AAL_BOTTOM_DEPTH_MSR,
       AAL_BOTTOM_DEPTH_UOM,
       AAL_ADDTNL_LOC_INFO)
    VALUES
      (P_AAL_REC.AAL_SEQ,
       P_AAL_REC.AAL_FA_SEQ,
       P_AAL_REC.AAL_STL_SEQ,
       P_AAL_REC.AAL_TSMALP_IS_NUMBER,
       P_AAL_REC.AAL_TSMALP_ORG_ID,
       P_AAL_REC.AAL_TSRAAL_TYPE_NAME,
       P_AAL_REC.AAL_BOTTOM_DEPTH_MSR,
       P_AAL_REC.AAL_BOTTOM_DEPTH_UOM,
       P_AAL_REC.AAL_ADDTNL_LOC_INFO);
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting aal.',
                                  P_AAL_REC.AAL_FA_SEQ);
      RETURN 0;
  END CREATE_AAL;

  /***********************************************************************************************/
  FUNCTION CREATE_TRAP_NET(P_STN_REC        IN SIM_TRAP_NET%ROWTYPE,
                           V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

  BEGIN
    /* Insert stn using values in record */

    INSERT INTO SIM_TRAP_NET
      (STN_SEQ,
       STN_TSRFDACT_IS_NUMBER,
       STN_TSRFDACT_ORG_ID,
       STN_SAMPLING_DUR,
       STN_SAMPLING_DUR_UNIT,
       STN_ORIENT_TO_CUR,
       STN_REL_CURRENT,
       STN_REL_WIND,
       STN_COMMENT,
       STN_FA_SEQ)
    VALUES
      (P_STN_REC.STN_SEQ,
       P_STN_REC.STN_TSRFDACT_IS_NUMBER,
       P_STN_REC.STN_TSRFDACT_ORG_ID,
       P_STN_REC.STN_SAMPLING_DUR,
       P_STN_REC.STN_SAMPLING_DUR_UNIT,
       P_STN_REC.STN_ORIENT_TO_CUR,
       P_STN_REC.STN_REL_CURRENT,
       P_STN_REC.STN_REL_WIND,
       P_STN_REC.STN_COMMENT,
       P_STN_REC.STN_FA_SEQ);
    COMMIT;
    RETURN 1;

  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting trap_net.',
                                  P_STN_REC.STN_FA_SEQ);
      RETURN 0;
  END CREATE_TRAP_NET;

  /******************************************************************************************************/
  FUNCTION CREATE_RESULT(P_RESULT_REC     IN SIM_RESULTS%ROWTYPE,
                         V_IMPORT_LOG_SEQ IN NUMBER) RETURN NUMBER IS

  BEGIN

    /* create result using values in result record */
    INSERT INTO SIM_RESULTS
      (RS_SEQ,
       RS_AC_SEQ,
       RS_FA_SEQ,
       RS_BRG_SEQ,
       RS_BRGI_SEQ,
       RS_RESULT,
       RS_COMMENT,
       RS_DETECT_LIMIT,
       RS_IMPORT_LINE_NO,
       RS_VALUE_STATUS,
       RS_LAB_BATCH_ID,
       RS_LAB_CERT,
       RS_ANALYSIS_DATE,
       RS_ANALYSIS_TIME,
       RS_ANALYSIS_TIME_ZONE,
       RS_DET_LIMIT_UOM_NBR,
       RS_DET_LIMIT_UOM_ORG,
       RS_DETECT_COMMENT,
       RS_UPPER_QUANT_LIMIT,
       RS_LOWER_QUANT_LIMIT,
       RS_TSRCHAR_IS_NUMBER,
       RS_TSRCHAR_ORG_ID,
       RS_TSRCHAR0IS_NUMBER,
       RS_TSRCHAR0ORG_ID,
       RS_TSRUOM_IS_NBR,
       RS_TSRUOM_ORG_ID,
       RS_FRACTION,
       RS_VALUE_TYPE,
       RS_STAT_TYPE,
       RS_TSRANLPR_IS_NUMBER,
       RS_TSRANLPR_ORG_ID,
       RS_TSRLSPP_IS_NUMBER,
       RS_TSRLSPP_ORG_ID,
       RS_DURATION_BASIS,
       RS_TEMP_BASIS,
       RS_WEIGHT_BASIS,
       RS_PRECISION,
       RS_CONF_LEVEL,
       RS_TSRCHGRP_IS_NUMBER,
       RS_TSRCHGRP_ORG_ID,
       RS_TSRCHDEF_IS_NUMBER,
       RS_TSRCHDEF_ORG_ID,
       RS_TROPHIC_LEVEL,
       RS_FNCTIONAL_FEED_GRP,
       RS_TAXON_POLLUTION,
       RS_TSRLAB_IS_NUMBER,
       RS_TSRLAB_ORG_ID,
       RS_TSRHCSC_IS_NUMBER,
       RS_TSRHCSC_ORG_ID,
       RS_LOWER_RANGE_VALUE,
       RS_UPPER_RANGE_VALUE,
       RS_REPL_ANALYSIS_NUM,
       RS_TSRDLIN_ORG_ID,
       RS_TSRDLIN_IS_NUMBER,
       RS_FRACTION_TSMPRMVL_IS_NUMBER,
       RS_FRACTION_TSMPRMVL_ORG_ID,
       RS_HABIT_TSMPRMVL0IS_NUMBER,
       RS_HABIT_TSMPRMVL0ORG_ID,
       RS_VOLT_TSMPRMVL1IS_NUMBER,
       RS_VOLT_TSMPRMVL1ORG_ID,
       RS_D_ASSND_SEQ_NUM,
       RS_BIAS,
       RS_CONF_LVL_CORR_BIAS,
       RS_SPECIES_NUMBER,
       RS_REF_PT_FROM_NAME,
       RS_REF_PT_TO_NAME,
       RS_DILUTION_IND_CODE,
       RS_RECOVERY_IND_CODE,
       RS_CORRECTION_IND_CD,
       RS_CELL_TYPE_NM,
       RS_CELL_SHAPE_TYPE_NM,
       RS_IMPORT_SEQ,
       RS_DATA_LINE_NUMBER,
       RS_DATA_LINE_NAME,
       RS_DOCUMENT)
    VALUES
      (P_RESULT_REC.RS_SEQ, /* RS_SEQ */
       P_RESULT_REC.RS_AC_SEQ, /* RS_AC_SEQ */
       P_RESULT_REC.RS_FA_SEQ, /* RS_FA_SEQ */
       P_RESULT_REC.RS_BRG_SEQ,
       P_RESULT_REC.RS_BRGI_SEQ,
       P_RESULT_REC.RS_RESULT, /* RS_RESULT */
       P_RESULT_REC.RS_COMMENT, /* RS_COMMENT */
       P_RESULT_REC.RS_DETECT_LIMIT, /* RS_DETECT_LIMIT */
       P_RESULT_REC.RS_IMPORT_LINE_NO, /* RS_IMPORT_LINE_NO */
       P_RESULT_REC.RS_VALUE_STATUS, /* RS_VALUE_STATUS_SEQ */
       P_RESULT_REC.RS_LAB_BATCH_ID, /* RS_LAB_BATCH_ID */
       P_RESULT_REC.RS_LAB_CERT, /* RS_LAB_CERT */
       P_RESULT_REC.RS_ANALYSIS_DATE, /* RS_ANALYSIS_DATE */
       P_RESULT_REC.RS_ANALYSIS_TIME, /* RS_ANALYSIS_TIME */
       P_RESULT_REC.RS_ANALYSIS_TIME_ZONE, /* RS_ANALYSIS_TIME_ZONE */
       P_RESULT_REC.RS_DET_LIMIT_UOM_NBR, /* RS_DET_LIMIT_UOM_NBR */
       P_RESULT_REC.RS_DET_LIMIT_UOM_ORG, /* RS_DET_LIMIT_UOM_ORG */
       P_RESULT_REC.RS_DETECT_COMMENT, /* RS_DETECT_COMMENT */
       P_RESULT_REC.RS_UPPER_QUANT_LIMIT, /* RS_UPPER_QUANT_LIMIT */
       P_RESULT_REC.RS_LOWER_QUANT_LIMIT, /* RS_LOWER_QUANT_LIMIT */
       P_RESULT_REC.RS_TSRCHAR_IS_NUMBER, /* RS_TSRCHAR_IS_NUMBER */
       P_RESULT_REC.RS_TSRCHAR_ORG_ID, /* RS_TSRCHAR_ORG_ID */
       P_RESULT_REC.RS_TSRCHAR0IS_NUMBER, /* RS_TSRCHAR0IS_NUMBER */
       P_RESULT_REC.RS_TSRCHAR0ORG_ID, /* RS_TSRCHAR0ORG_ID */
       P_RESULT_REC.RS_TSRUOM_IS_NBR, /* RS_TSRUOM_IS_NBR */
       P_RESULT_REC.RS_TSRUOM_ORG_ID, /* RS_TSRUOM_ORG_ID */
       P_RESULT_REC.RS_FRACTION, /* RS_FRACTION */
       P_RESULT_REC.RS_VALUE_TYPE, /* RS_VALUE_TYPE_SEQ */
       P_RESULT_REC.RS_STAT_TYPE, /* RS_STAT_TYPE_SEQ */
       P_RESULT_REC.RS_TSRANLPR_IS_NUMBER, /* RS_TSRANLPR_IS_NUMBER */
       P_RESULT_REC.RS_TSRANLPR_ORG_ID, /* RS_TSRANLPR_ORG_ID */
       P_RESULT_REC.RS_TSRLSPP_IS_NUMBER, /* RS_TSRLSPP_IS_NUMBER */
       P_RESULT_REC.RS_TSRLSPP_ORG_ID, /* RS_TSRLSPP_ORG_ID */
       P_RESULT_REC.RS_DURATION_BASIS, /* RS_DURBASIS_SEQ */
       P_RESULT_REC.RS_TEMP_BASIS, /* RS_TEMPBASIS_SEQ */
       P_RESULT_REC.RS_WEIGHT_BASIS,
       P_RESULT_REC.RS_PRECISION, /* RS_PRECISION */
       P_RESULT_REC.RS_CONF_LEVEL, /* RS_CONFLEVEL_SEQ */
       P_RESULT_REC.RS_TSRCHGRP_IS_NUMBER, /* RS_TSRCHGRP_IS_NUMBER */
       P_RESULT_REC.RS_TSRCHGRP_ORG_ID, /* RS_TSRCHGRP_ORG_ID */
       P_RESULT_REC.RS_TSRCHDEF_IS_NUMBER, /* RS_TSRCHDEF_IS_NUMBER */
       P_RESULT_REC.RS_TSRCHDEF_ORG_ID, /* RS_TSRCHDEF_ORG_ID */
       P_RESULT_REC.RS_TROPHIC_LEVEL,
       P_RESULT_REC.RS_FNCTIONAL_FEED_GRP,
       P_RESULT_REC.RS_TAXON_POLLUTION,
       P_RESULT_REC.RS_TSRLAB_IS_NUMBER,
       P_RESULT_REC.RS_TSRLAB_ORG_ID,
       P_RESULT_REC.RS_TSRHCSC_IS_NUMBER,
       P_RESULT_REC.RS_TSRHCSC_ORG_ID,
       P_RESULT_REC.RS_LOWER_RANGE_VALUE,
       P_RESULT_REC.RS_UPPER_RANGE_VALUE,
       P_RESULT_REC.RS_REPL_ANALYSIS_NUM,
       P_RESULT_REC.RS_TSRDLIN_ORG_ID,
       P_RESULT_REC.RS_TSRDLIN_IS_NUMBER,
       P_RESULT_REC.RS_FRACTION_TSMPRMVL_IS_NUMBER,
       P_RESULT_REC.RS_FRACTION_TSMPRMVL_ORG_ID,
       P_RESULT_REC.RS_HABIT_TSMPRMVL0IS_NUMBER,
       P_RESULT_REC.RS_HABIT_TSMPRMVL0ORG_ID,
       P_RESULT_REC.RS_VOLT_TSMPRMVL1IS_NUMBER,
       P_RESULT_REC.RS_VOLT_TSMPRMVL1ORG_ID,
       P_RESULT_REC.RS_D_ASSND_SEQ_NUM,
       P_RESULT_REC.RS_BIAS,
       P_RESULT_REC.RS_CONF_LVL_CORR_BIAS,
       P_RESULT_REC.RS_SPECIES_NUMBER,
       P_RESULT_REC.RS_REF_PT_FROM_NAME,
       P_RESULT_REC.RS_REF_PT_TO_NAME,
       P_RESULT_REC.RS_DILUTION_IND_CODE,
       P_RESULT_REC.RS_RECOVERY_IND_CODE,
       P_RESULT_REC.RS_CORRECTION_IND_CD,
       P_RESULT_REC.RS_CELL_TYPE_NM,
       P_RESULT_REC.RS_CELL_SHAPE_TYPE_NM,
       P_RESULT_REC.RS_IMPORT_SEQ,
       P_RESULT_REC.RS_DATA_LINE_NUMBER,
       P_RESULT_REC.RS_DATA_LINE_NAME,
       P_RESULT_REC.RS_DOCUMENT);

    COMMIT;

    RETURN 1;

  EXCEPTION

    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
                                  SQLERRM || ' - inserting result.',
                                  P_RESULT_REC.RS_FA_SEQ);
      RETURN 0;

  END CREATE_RESULT;

/******************************************************************************************************/
END SIM_IMPORT_RESULT_EXTRAS;
/
