CREATE OR REPLACE PACKAGE sim2_export_result_pkg IS
/* Modified 9/15/03 tcj - Fixed coding to proprely display error when migration errors occur.

   Modified 7/16/02 pwh - Added the v_userid_code variable, set it to 'SIM122  ' and inserted this variable
   into the D_USERID_CODE column in the STORET tables.

   Modified 7/15/02 by SF to change one D_USERID_CODE from 'STORUSER' to USER.

   Modified 4/4/02 by SF to remove restricten on Detection limit UOM that Detection Limit
   has to exist. They are also used for Quantification limits.

   Modified 3/25/02 by SF - set v_detect_cond_cd := NULL when Habtat Assessment results

   Updated 2/22/02        by SF
   Made sure that TSRRSULT gets EITHER a TSRBRG_key OR a TSRFDACT_KEY, but not both.

   Updated 2/15/02        by SF
   Added code to always create a TSRAAL record instead of just for Trawl Points
   U
   pdated 2/13/02        by SF
   Activity, Trip-QC, and Trip Time Zone fixes
*/

  -- Author  : KHENAS
  -- Created : 11/16/00 4:24:36 PM
  -- Purpose : load results data from SIM into STORET

  -- Public variable declarations
  v_export_log_seq          NUMBER(10);
  v_stavst_no               NUMBER(3);
  v_last_station            VARCHAR2(15);

  -- Public function and procedure declarations
  FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_UOM_IS_NBR (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2, p_chdef_is_number IN NUMBER)
     RETURN NUMBER;

  FUNCTION validate_sample_frac (p_sample_frac_nm IN VARCHAR2, p_tsrchdef_is_nbr IN NUMBER,
                                 p_tsrchar_is_nbr IN NUMBER) RETURN VARCHAR2;

  FUNCTION get_next_visit_is_number (p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_next_activity_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_next_result_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_value_msr (p_result_text IN VARCHAR2) RETURN NUMBER;

  PROCEDURE add_station_to_trip(p_trip_is_nbr IN NUMBER,p_sta_is_nbr IN NUMBER,p_org_id IN VARCHAR2);

  PROCEDURE add_project_to_trip(p_trip_is_nbr IN NUMBER, p_proj_is_nbr IN NUMBER, p_org_id IN VARCHAR2);

  PROCEDURE SIM_EXPORT (p_org_id  IN VARCHAR2, p_import_seq IN NUMBER, p_success IN OUT NUMBER);

  PROCEDURE EXPORT_RESULTS  (p_fa_seq IN NUMBER, p_fat_desc IN VARCHAR2, p_activity_is_nbr IN NUMBER,
                             p_activity_org_id IN VARCHAR2, p_tripqc_is_nbr IN NUMBER, p_tripqc_org_id IN VARCHAR2,
                             p_lab_is_number IN NUMBER,p_lab_org_id IN VARCHAR2, p_org_id IN VARCHAR2,
                             p_export_log_seq IN NUMBER, p_act_result_cnt OUT NUMBER, p_time_zone IN VARCHAR2);

  PROCEDURE runtime;

  FUNCTION get_next_blob_is_number (p_org_id IN VARCHAR2) RETURN NUMBER;
  FUNCTION insert_result_binary_document(p_export_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER;
  FUNCTION insert_visit_binary_document(p_export_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER;
  FUNCTION insert_act_binary_document(p_export_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER;
  
END sim2_export_result_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim2_export_result_pkg IS

  e_a                EXCEPTION;
  v_success          NUMBER(1);
  t_success          NUMBER(1) := 1;

/******************************************************************************************************/
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS

  v_item_name         VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_alval_seq;

BEGIN

  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name INTO v_item_name;
    IF c_get_alval_name%NOTFOUND THEN
      v_item_name := NULL;
      sim2_export_pkg.write_error(v_export_log_seq, 'Could not find allowable value with sequence: '||p_alval_seq);
    END IF;
  CLOSE c_get_alval_name;

  RETURN v_item_name;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting allowable value sequence.');
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;

END get_allowable_value;
/******************************************************************************************************/
FUNCTION get_tsmprmvl_value(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) RETURN VARCHAR2 IS

  v_tsmprmvl_name         VARCHAR2(80);

  CURSOR c_get_tsmprmvl_name(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT FIELD_VALUE
    FROM TSMPRMVL
    WHERE TSMPRMVL_IS_NUMBER = p_is_nbr
    AND   TSMPRMVL_ORG_ID = RPAD(p_org_id,8);

BEGIN

  OPEN c_get_tsmprmvl_name(p_is_nbr, p_org_id);
  FETCH c_get_tsmprmvl_name INTO v_tsmprmvl_name;
    IF c_get_tsmprmvl_name%NOTFOUND THEN
      v_tsmprmvl_name := NULL;
      sim2_export_pkg.write_error(v_export_log_seq, 'Could not find TSMPRMVL value with COMMUNITY: '||p_is_nbr);
    END IF;
  CLOSE c_get_tsmprmvl_name;

  RETURN v_tsmprmvl_name;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting tsmprmvl value.');
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;

END get_tsmprmvl_value;
/******************************************************************************************************/
FUNCTION get_tsrcpv_value(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) RETURN VARCHAR2 IS

  v_tsrcpv_name         VARCHAR2(80);

  CURSOR c_get_tsrcpv_name(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT SHORT_NAME
    FROM TSRCPV
    WHERE TSRCPV_IS_NUMBER = p_is_nbr
    AND   TSRCPV_ORG_ID = RPAD(p_org_id,8);

BEGIN

  OPEN c_get_tsrcpv_name(p_is_nbr, p_org_id);
  FETCH c_get_tsrcpv_name INTO v_tsrcpv_name;
    IF c_get_tsrcpv_name%NOTFOUND THEN
      v_tsrcpv_name := NULL;
      sim2_export_pkg.write_error(v_export_log_seq, 'Could not find TSRCPV value  '||p_is_nbr);
    END IF;
  CLOSE c_get_tsrcpv_name;

  RETURN v_tsrcpv_name;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting tsrcpv value.');
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;

END get_tsrcpv_value;

/******************************************************************************************************/
FUNCTION get_parent_is_nbr(p_parent_fa_seq IN NUMBER) RETURN NUMBER IS

  v_parent_is_nbr         NUMBER(8);

  CURSOR c_get_parent_is_nbr(p_parent_fa_seq IN NUMBER) IS
    SELECT FA_TSRFDACT_IS_NUMBER
    FROM SIM_FIELD_ACTIVITIES
    WHERE FA_SEQ = p_parent_fa_seq;

BEGIN

  OPEN c_get_parent_is_nbr(p_parent_fa_seq);
  FETCH c_get_parent_is_nbr INTO v_parent_is_nbr;
    IF c_get_parent_is_nbr%NOTFOUND THEN
      v_parent_is_nbr := NULL;
      sim2_export_pkg.write_error(v_export_log_seq, 'Could not find parent_is_number for fa_parent_fa_seq: '||p_parent_fa_seq);
    END IF;
  CLOSE c_get_parent_is_nbr;

  RETURN v_parent_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting parent_is_nbr.');
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;

END get_parent_is_nbr;

/******************************************************************************************************/
FUNCTION GET_UOM_IS_NBR (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2, p_chdef_is_number IN NUMBER)
   RETURN NUMBER IS

   TYPE t_uom_record IS RECORD (
      uom_is_nbr        NUMBER(8),
      uom_name             VARCHAR2(10));

   v_uom_info              t_uom_record;

   e_unk_uom               EXCEPTION;
   e_uom_sp                EXCEPTION;

   CURSOR c_get_is_nbr (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2) IS
      SELECT TSRUOM_IS_NUMBER, SHORT_FORM_NAME FROM TSRUOM WHERE TSRUOM_ORG_ID = RPAD(p_org_id,8) AND
      UPPER(SHORT_FORM_NAME) = UPPER(RPAD(p_uom_name,10));

BEGIN

   OPEN c_get_is_nbr(p_org_id, p_uom_name);
   FETCH c_get_is_nbr INTO v_uom_info;
   IF c_get_is_nbr%NOTFOUND THEN
      v_uom_info.uom_is_nbr := NULL;
      RAISE e_unk_uom;
   ELSE
      IF v_uom_info.uom_name != p_uom_name THEN
         RAISE e_uom_sp;
      END IF;
   END IF;
   CLOSE c_get_is_nbr;

   RETURN v_uom_info.uom_is_nbr;

EXCEPTION
   WHEN e_unk_uom THEN
     v_success := 0;
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Unknown Unit of Measure: '||p_uom_name);
     IF t_success > v_success THEN 
 		   t_success := v_success;
     END IF;
     RETURN v_uom_info.uom_is_nbr;

   WHEN e_uom_sp THEN
     v_success := 0;
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Fix uom spelling in TSRCHDEF - is_number:'||p_chdef_is_number||', unit of measure misspelled: '||p_uom_name);
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN v_uom_info.uom_is_nbr;
   WHEN OTHERS THEN
     v_success := 0;
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'in get uom is_number:'||p_chdef_is_number||', unit of measure misspelled: '||p_uom_name);
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;

END GET_UOM_IS_NBR;
/******************************************************************************************************/
FUNCTION validate_sample_frac (p_sample_frac_nm IN VARCHAR2, p_tsrchdef_is_nbr IN NUMBER,
                               p_tsrchar_is_nbr IN NUMBER) RETURN VARCHAR2 IS

  v_sample_frac               VARCHAR2(40);
  v_samp_req_cd               VARCHAR2(1);

  e_sample_frac_sp            EXCEPTION;

  CURSOR c_get_samp_req_cd (p_tsrchar_is_nbr IN NUMBER) IS
    SELECT SAMP_FRAC_REQ_CD FROM TSRCHAR WHERE tsrchar_is_number = p_tsrchar_is_nbr;

  CURSOR c_check_def_frac IS
    SELECT FIELD_VALUE FROM TSMPRMVL
    WHERE FIELD_NAME = RPAD('SMPL_FRAC_TYPE_NM',20) AND TABLE_NAME = 'TSRCHDEF';

BEGIN

  /* set sample fraction to null if it is full of spaces */
  IF RPAD(p_sample_frac_nm,15) = RPAD(' ',15) OR p_sample_frac_nm IS NULL THEN
    v_sample_frac := NULL;
  ELSE
    v_sample_frac := p_sample_frac_nm;
  END IF;

  OPEN c_get_samp_req_cd(p_tsrchar_is_nbr);
  FETCH c_get_samp_req_cd INTO v_samp_req_cd;
    IF v_samp_req_cd = 'Y' OR v_sample_frac IS NOT NULL THEN

      FOR v_fraction IN c_check_def_frac LOOP
        IF v_fraction.FIELD_VALUE = RPAD(p_sample_frac_nm,40) THEN
          RETURN RTRIM(v_fraction.FIELD_VALUE);
        ELSIF UPPER(v_fraction.FIELD_VALUE) = UPPER(RPAD(p_sample_frac_nm,40)) THEN
          v_sample_frac := RTRIM(v_fraction.FIELD_VALUE);
          CLOSE c_get_samp_req_cd;
          RAISE e_sample_frac_sp;
        END IF;
      END LOOP;

      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Invalid fraction in TSRCHDEF_IS_NUMBER:'||p_tsrchdef_is_nbr||', sample fraction, '||p_sample_frac_nm||', not loaded into TSRRSULT');
      RETURN NULL;

    END IF;
  CLOSE c_get_samp_req_cd;

  RETURN NULL;

EXCEPTION
  WHEN e_sample_frac_sp THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Fix fraction spelling in TSRCHDEF - is_number:'||p_tsrchdef_is_nbr||', sample fraction: '||p_sample_frac_nm);
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN v_sample_frac;

  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in validate sample fraction.');
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN v_sample_frac;

END validate_sample_frac;

/******************************************************************************************************/
FUNCTION GET_FIELD_SET_IS_NBR (p_org_id VARCHAR2, p_visit_is_number NUMBER,
                               p_id_code IN VARCHAR2, p_fdset_name IN VARCHAR2)
                               RETURN NUMBER IS

   /* this function looks to see if there is already a field set with the same id-code set up for a visit.
      if it is, it returns the is-number of the field set.  If not, it creates a field set record and returns it's
      IS_number.    SNG 4/21/2000
   */

   v_field_set_is_nbr NUMBER(8);
   v_table_name       VARCHAR2(8);

   CURSOR c_field_set_is_nbr (p_org_id VARCHAR2, p_visit_is_number NUMBER, p_id_code VARCHAR2) IS
      SELECT tsrfdset_is_number FROM TSRFDSET WHERE
         ID_CODE = RPAD(p_id_code,10) AND
         TSRFDSET_ORG_ID = RPAD(p_org_id,8) AND
         TSRSTVST_IS_NUMBER = p_visit_is_number;


BEGIN

   OPEN c_field_set_is_nbr (p_org_id , p_visit_is_number , p_id_code );
   FETCH c_field_set_is_nbr INTO v_field_set_is_nbr;
   IF c_field_set_is_nbr%NOTFOUND THEN

       /* not found, so set one up */

       v_table_name := 'TSRFDSET';
       v_field_set_is_nbr :=  sim2_export_pkg.get_eisn_number(v_table_name,p_org_id);

       INSERT INTO TSRFDSET (
         TSRFDSET_ORG_ID,
         TSRFDSET_IS_NUMBER,
         ID_CODE,
         NAME,
         D_USERID_CODE,
         D_LAST_UPDATE_TS,
         TSRSTVST_IS_NUMBER,
         TSRSTVST_ORG_ID)
       VALUES (
         RPAD(p_org_id,8),
         v_field_set_is_nbr,
         p_id_code,
         p_fdset_name,
         sim2_export_pkg.v_pkg_ownerid,
         SYSDATE,
         p_visit_is_number,
         RPAD(p_org_id,8));

       COMMIT;

   END IF;

   CLOSE c_field_set_is_nbr;


   RETURN v_field_set_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get_field_set_is_nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN NULL;

END GET_FIELD_SET_IS_NBR;

/******************************************************************************************************/
FUNCTION get_sample_pres_info(p_tsrsdp_is_nbr IN NUMBER, p_tsrsdp_org_id IN VARCHAR2,
  p_activity_id IN VARCHAR2) RETURN TSRSMPLE%ROWTYPE IS

  v_sample_rec           TSRSMPLE%ROWTYPE;
  v_sample_copy_rec      TSRSMPLE%ROWTYPE;

  e_bad_sample_pres      EXCEPTION;

  CURSOR c_get_sample_pres_info IS
    SELECT TSMPRMVL_IS_NUMBER, TSMPRMVL_ORG_ID, TSMPRMVL0IS_NUMBER, TSMPRMVL0ORG_ID,
      CONTAINER_SIZE_MSR, CONTAINER_SIZE_UN,
      TSMPRMVL1IS_NUMBER, TSMPRMVL1ORG_ID, PRESRV_STRGE_PRCDR FROM TSRSDP
      WHERE TSRSDP_IS_NUMBER = p_tsrsdp_is_nbr AND TSRSDP_ORG_ID = RPAD(p_tsrsdp_org_id,8);

BEGIN
  /* kms (5/22/01) - this function gets all of the sample preservation, storage, and transport
    information on the activity and returns it in the form of a sample rowtype.  This is
    performed in this manner since STORET does not store this information relationally.

    sef (4/1/03) - new functionality in V2 - container type, color, and temp pres type are now
    stored as keys to the tsmprmvl table */

  OPEN c_get_sample_pres_info;
  FETCH c_get_sample_pres_info INTO v_sample_rec.tsmprmvl_is_number,v_sample_rec.tsmprmvl_org_id,
        v_sample_rec.tsmprmvl0is_number,v_sample_rec.tsmprmvl0org_id,
        v_sample_rec.container_size_msr,v_sample_rec.container_size_un,v_sample_rec.tsmprmvl1is_number,
        v_sample_rec.tsmprmvl1org_id, v_sample_rec.presrv_strge_prcdr;
    IF c_get_sample_pres_info%NOTFOUND THEN
      CLOSE c_get_sample_pres_info;
      RAISE e_bad_sample_pres;
    END IF;
  CLOSE c_get_sample_pres_info;

  RETURN v_sample_rec;

EXCEPTION
  WHEN e_bad_sample_pres THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Activity '||p_activity_id||': Unable to find sample preservation information - invalid foreign key.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN v_sample_copy_rec;

  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsmgntxt is nbr.');
     v_success := 0;
     RETURN v_sample_copy_rec;

END get_sample_pres_info;

/******************************************************************************************************/
FUNCTION get_instructions(p_import_seq IN NUMBER, p_instruction_type IN VARCHAR2) RETURN VARCHAR2 IS

  v_imp_cfg_seq          NUMBER(10);
  v_instr_choice         VARCHAR2(15);

  e_no_cfg               EXCEPTION;

  CURSOR c_get_cfg_seq(p_import_seq IN NUMBER) IS
    SELECT SIL_SICC_SEQ FROM SIM_IMPORT_LOG WHERE SIL_SEQ = p_import_seq;

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SICI_SICC_SEQ = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SIIC_DFLT_FLAG = 1;


BEGIN

  /* get the configuration for this import */
  OPEN c_get_cfg_seq(p_import_seq);
  FETCH c_get_cfg_seq INTO v_imp_cfg_seq;
    IF c_get_cfg_seq%NOTFOUND THEN
      v_imp_cfg_seq := NULL;
      RAISE e_no_cfg;
    END IF;
  CLOSE c_get_cfg_seq;

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(v_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr INTO v_instr_choice;
    IF c_get_cfg_instr%NOTFOUND THEN

      /* no choice could be found, get default choice */
      OPEN c_get_instr_default(p_instruction_type);
      FETCH c_get_instr_default INTO v_instr_choice;
      CLOSE c_get_instr_default;

    END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;

EXCEPTION
  WHEN e_no_cfg THEN
    RAISE_APPLICATION_ERROR(-20101, 'Configuration could not be found for this import.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN NULL;

  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get_instructions.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN NULL;

END get_instructions;

/******************************************************************************************************/
FUNCTION get_next_visit_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS

   v_next_visit_is_nbr NUMBER(8);

BEGIN
   /* get the next avail is nbr to use */
   v_next_visit_is_nbr := sim2_export_pkg.get_eisn_number('TSRSTVST',p_org_id);

   RETURN v_next_visit_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next visit is number.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
 		END IF;
    RETURN NULL;

END get_next_visit_is_number;

/******************************************************************************************************/
FUNCTION get_next_activity_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_activity_is_nbr     NUMBER(8);
BEGIN
   v_activity_is_nbr := sim2_export_pkg.get_eisn_number('TSRFDACT',p_org_id);

   RETURN v_activity_is_nbr ;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next activity is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
  	 END IF;
     RETURN NULL;

END get_next_activity_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_trip_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_trip_is_nbr     NUMBER(8);
BEGIN

   v_trip_is_nbr := sim2_export_pkg.get_eisn_number('TSRTRIP ',p_org_id);

   RETURN v_trip_is_nbr ;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next trip is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN NULL;

END get_next_trip_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_trip_qc_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_trip_qc_is_nbr     NUMBER(8);
BEGIN
    v_trip_qc_is_nbr := sim2_export_pkg.get_eisn_number('TSRFQS  ',p_org_id);

   RETURN v_trip_qc_is_nbr ;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next trip qc is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
 		 END IF;
     RETURN NULL;

END get_next_trip_qc_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_result_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_result_is_nbr     NUMBER(8);
BEGIN
     v_result_is_nbr := sim2_export_pkg.get_eisn_number('TSRRSULT',p_org_id);

   RETURN v_result_is_nbr ;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next result is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN NULL;

END get_next_result_is_nbr;
/******************************************************************************************************/
FUNCTION get_next_tsrbrg_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsrbrg_is_nbr     NUMBER(8);
BEGIN
   v_tsrbrg_is_nbr := sim2_export_pkg.get_eisn_number('TSRBRG',p_org_id);

   RETURN v_tsrbrg_is_nbr ;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsrbrg is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN NULL;

END get_next_tsrbrg_is_nbr;
/******************************************************************************************************/
FUNCTION get_next_tsrbrgi_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsrbrgi_is_nbr     NUMBER(8);
BEGIN
     v_tsrbrgi_is_nbr := sim2_export_pkg.get_eisn_number('TSRBRGI',p_org_id);

   RETURN v_tsrbrgi_is_nbr ;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsrbrgi is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN NULL;

END get_next_tsrbrgi_is_nbr;
/******************************************************************************************************/
FUNCTION get_next_tsmgntxt_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmgntxt_is_nbr     NUMBER(8);
BEGIN
    v_tsmgntxt_is_nbr := sim2_export_pkg.get_eisn_number('TSMGNTXT',p_org_id);

  RETURN v_tsmgntxt_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsmgntxt is nbr.');
     v_success := 0;
     IF t_success > v_success THEN 
 		   t_success := v_success;
		 END IF;
     RETURN NULL;

END get_next_tsmgntxt_nbr;

/******************************************************************************************************/
FUNCTION get_value_msr (p_result_text IN VARCHAR2) RETURN NUMBER IS

  v_result_value       TSRRSULT.VALUE_MEASURE%TYPE;

BEGIN
                   
    BEGIN
     v_result_value := ROUND(TO_NUMBER(p_result_text),5);
     IF v_result_value > 99999999.99999 THEN   
       v_result_value := 99999999.99999;          
       sim2_export_pkg.write_error(v_export_log_seq, 'Number too large for STORET value column, '||
                                      'set to ''99999999.99999'': '||p_result_text);       

     END IF;
        
    EXCEPTION
      WHEN OTHERS THEN
       v_result_value := NULL;
    END;
      


   RETURN v_result_value;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get value measure.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN NULL;

END get_value_msr;
/******************************************************************************************************/
PROCEDURE get_trip_info(p_trip_seq IN NUMBER, p_trip_rec OUT TSRTRIP%ROWTYPE) IS

  CURSOR c_get_trip_info (p_trip_seq IN NUMBER) IS
    SELECT STP_TRIP_ID, STP_START_DATE, --STP_START_TIME,
           STP_END_DATE, STP_TRIP_NAME,
           STP_TSMORGAN_IS_NUMBER
      FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

BEGIN

  /* kms (6/4/01) - get info about this trip from the SIM_TRIPS table */
  OPEN c_get_trip_info(p_trip_seq);
  FETCH c_get_trip_info INTO p_trip_rec.ID_CODE, p_trip_rec.start_date,
                             p_trip_rec.end_date, --p_trip_rec.end_time,
                             p_trip_rec.name,
                             p_trip_rec.TSMORGAN_IS_NUMBER;
  CLOSE c_get_trip_info;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get trip info.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;

END get_trip_info;

/******************************************************************************************************/
PROCEDURE get_visit_info(p_visit_seq IN NUMBER, p_visit_rec OUT TSRSTVST%ROWTYPE, p_visit_id OUT VARCHAR2) IS

  CURSOR c_get_visit_info (p_visit_seq IN NUMBER) IS
    SELECT SSV_ID, SSV_START_DATE, SSV_START_TIME, SSV_END_DATE, SSV_END_TIME, 
           sim2_export_pkg.get_file_type(SSV_DOCUMENT), sim2_export_pkg.get_file_name(SSV_DOCUMENT)    
      FROM SIM_STATION_VISITS WHERE SSV_SEQ = p_visit_seq;

BEGIN

  /* kms (6/4/01) - get info about this trip from the SIM_TRIPS table */
  OPEN c_get_visit_info(p_visit_seq);
  FETCH c_get_visit_info INTO p_visit_id, p_visit_rec.arrival_date,
                              p_visit_rec.arrival_time, p_visit_rec.departure_date,
                              p_visit_rec.departure_time, p_visit_rec.blob_type, p_visit_rec.blob_title;
  CLOSE c_get_visit_info;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get visit info.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;

END get_visit_info;

/******************************************************************************************************/
FUNCTION get_next_visit_id(
         p_trip_seq     IN NUMBER,
         p_trip_is_nbr  IN NUMBER, 
         p_trip_org_id  IN VARCHAR2,
         p_statn_is_nbr IN NUMBER, 
         p_statn_org_id IN VARCHAR2) 
         RETURN CHAR IS

  v_max_storet_id  NUMBER(5);
  v_max_sim_id     NUMBER(5);

  -- jah 3-8-2004
  -- Retrieve the largest STORET station visit number for a given trip and station
  CURSOR c_get_storet_max_visit_id (
         cp_trip_is_nbr  IN NUMBER, 
         cp_trip_org_id  IN VARCHAR2,
         cp_statn_is_nbr IN NUMBER, 
         cp_statn_org_id IN VARCHAR2) IS
  SELECT MAX(TO_NUMBER(TRIM(ID_NUMBER)))
  FROM   TSRSTVST 
  WHERE  TSRTRIP_IS_NUMBER  = cp_trip_is_nbr
  AND    TSRTRIP_ORG_ID     = RPAD(cp_trip_org_id,8) 
  AND    TSMSTATN_IS_NUMBER = cp_statn_is_nbr
  AND    TSMSTATN_ORG_ID    = RPAD(cp_statn_org_id,8)
  AND    TRANSLATE(ID_NUMBER, 'a0123456789 ', 'a') IS NULL; -- and is numeric

  -- jah 3-8-2004
  -- Retrieve the largest SIM station visit number for a given trip and station
  CURSOR c_get_sim_max_visit_id (
         cp_trip_seq     IN NUMBER, 
         cp_statn_is_nbr IN NUMBER, 
         cp_statn_org_id IN VARCHAR2) IS
  SELECT MAX(TO_NUMBER(TRIM(SSV_ID)))
  FROM   SIM_STATION_VISITS
  WHERE  SSV_STP_SEQ            = cp_trip_seq
  AND    SSV_TSMSTATN_IS_NUMBER = cp_statn_is_nbr
  AND    SSV_TSMSTATN_ORG_ID    = RPAD(cp_statn_org_id,8)
  AND    TRANSLATE(SSV_ID, 'a0123456789 ', 'a') IS NULL; -- and is numeric

BEGIN

  -- Get the largest STORET station visit ID for a given trip and station
  OPEN c_get_storet_max_visit_id(
    p_trip_is_nbr,
    p_trip_org_id,
    p_statn_is_nbr,
    p_statn_org_id
  );
  FETCH c_get_storet_max_visit_id INTO v_max_storet_id;
  CLOSE c_get_storet_max_visit_id;
  v_max_storet_id := NVL(v_max_storet_id,0);

  -- Get the largest SIM STATION visit ID for a given trip and station
  OPEN c_get_sim_max_visit_id(
    p_trip_seq,
    p_statn_is_nbr,
    p_statn_org_id
  );
  FETCH c_get_sim_max_visit_id INTO v_max_sim_id;
  CLOSE c_get_sim_max_visit_id;
  v_max_sim_id := NVL(v_max_sim_id,0);

  -- return the largest station visit ID
  RETURN TO_CHAR(GREATEST(v_max_storet_id, v_max_sim_id) + 1);
  
EXCEPTION

   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(
       v_export_log_seq, 
       SQLERRM || ' - in function get_next_visit_id.'
     );
     v_success := 0;
     
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     
     RETURN NULL;

END get_next_visit_id;

/******************************************************************************************************/
PROCEDURE add_station_to_trip(p_trip_is_nbr IN NUMBER,p_sta_is_nbr IN NUMBER,p_org_id IN VARCHAR2) IS

  v_tsrtsa                  NUMBER(8);

  CURSOR c_get_tsrtsa (p_cursor_trip IN NUMBER, p_cursor_sta IN NUMBER, p_cursor_org_id IN VARCHAR2) IS
    SELECT TSRTRIP_IS_NUMBER FROM tsrtsa WHERE TSMSTATN_IS_NUMBER = p_cursor_sta
    AND TSRTRIP_IS_NUMBER = p_cursor_trip
    AND TSRTRIP_ORG_ID = p_cursor_org_id;

BEGIN
  /* see if station is already on the trip */
  OPEN c_get_tsrtsa (p_trip_is_nbr,p_sta_is_nbr,RPAD(p_org_id,8));
  FETCH c_get_tsrtsa INTO v_tsrtsa;
     IF c_get_tsrtsa%NOTFOUND THEN
       /* Update TSRTSA to associate station with this trip if record does not already exist.  */
        INSERT INTO TSRTSA (
           TSRTRIP_IS_NUMBER,
           TSRTRIP_ORG_ID,
           TSMSTATN_IS_NUMBER,
           TSMSTATN_ORG_ID,
           D_USERID_CODE,
           D_LAST_UPDATE_TS
           )
        VALUES (
           p_trip_is_nbr,                     /* TSRTRIP_IS_NUMBER */
           RPAD(p_org_id,8),                  /* TSRTRIP_ORG_ID */
           p_sta_is_nbr,                      /* TSMSTATN_IS_NUMBER */
           RPAD(p_org_id,8),                  /* TSMSTATN_ORG_ID */
           sim2_export_pkg.v_pkg_ownerid,     
           SYSDATE                            /* D_LAST_UPDATE_TS */
           );

        COMMIT;

     END IF;
  CLOSE c_get_tsrtsa;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add station to trip.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;

END add_station_to_trip;

/******************************************************************************************************/
PROCEDURE add_project_to_activity(p_act_is_nbr IN NUMBER, p_proj_is_nbr IN NUMBER,
                                  p_org_id IN VARCHAR2) IS

  v_proj_is                  NUMBER(8);


  CURSOR c_check_for_project(p_cursor_act IN NUMBER,p_cursor_proj IN NUMBER,p_org_id IN VARCHAR2) IS
    SELECT TSMPROJ_IS_NUMBER FROM TSRFAPRA WHERE TSMPROJ_IS_NUMBER = p_cursor_proj
    AND TSRFDACT_IS_NUMBER = p_cursor_act AND TSRFDACT_ORG_ID = RPAD(p_org_id,8);

BEGIN
  /* see if the project is on the activity */
  OPEN c_check_for_project(p_act_is_nbr,p_proj_is_nbr,p_org_id);
  FETCH c_check_for_project INTO v_proj_is;
    IF c_check_for_project%NOTFOUND THEN
      /* if project is not on the activity then add */
      INSERT INTO TSRFAPRA(
        TSRFDACT_IS_NUMBER,
        TSRFDACT_ORG_ID,
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES(
        p_act_is_nbr,            /* TSRFDACT_IS_NUMBER */
        p_org_id,                /* TSRFDACT_ORG_ID */
        p_proj_is_nbr,           /* TSMPROJ_IS_NUMBER */
        p_org_id,                /* TSMPROJ_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid,  
        SYSDATE                  /* D_LAST_UPDATE_TS */
        );
      COMMIT;
    END IF;
  CLOSE c_check_for_project;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add project to activity.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;

END add_project_to_activity;

/******************************************************************************************************/
PROCEDURE add_project_to_trip(p_trip_is_nbr IN NUMBER, p_proj_is_nbr IN NUMBER,
                              p_org_id IN VARCHAR2) IS

  v_proj_is                  NUMBER(8);
  v_number_projects          NUMBER(8);


  CURSOR c_check_for_project(p_cursor_trip IN NUMBER,p_cursor_proj IN NUMBER,p_org_id IN VARCHAR2) IS
    SELECT TSMPROJ_IS_NUMBER FROM TSRTPA WHERE TSMPROJ_IS_NUMBER = p_cursor_proj
    AND TSRTRIP_IS_NUMBER = p_cursor_trip AND TSRTRIP_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_check_for_single_proj(p_cursor_trip IN NUMBER,p_cursor_org_id IN VARCHAR2) IS
    SELECT COUNT(TSMPROJ_IS_NUMBER) FROM TSRTPA WHERE TSRTRIP_IS_NUMBER = p_cursor_trip
    AND TSRTRIP_ORG_ID = p_cursor_org_id;


BEGIN
  /* see if the project is on the trip */
  OPEN c_check_for_project(p_trip_is_nbr,p_proj_is_nbr,p_org_id);
  FETCH c_check_for_project INTO v_proj_is;
    IF c_check_for_project%NOTFOUND THEN
      /* if project is not on the trip then add */
      INSERT INTO TSRTPA(
        TSRTRIP_IS_NUMBER,
        TSRTRIP_ORG_ID,
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES(
        p_trip_is_nbr,           /* TSRTRIP_IS_NUMBER */
        p_org_id,                /* TSRTRIP_ORG_ID */
        p_proj_is_nbr,           /* TSMPROJ_IS_NUMBER */
        p_org_id,                /* TSMPROJ_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid,
        SYSDATE                  /* D_LAST_UPDATE_TS */
        );
      COMMIT;
    END IF;
  CLOSE c_check_for_project;

  /* check for single project on trip */
  OPEN c_check_for_single_proj(p_trip_is_nbr,RPAD(p_org_id,8));
  FETCH c_check_for_single_proj INTO v_number_projects;
    /* set single project ind to 'N' if more than one project is on the trip */
    IF v_number_projects > 1 THEN
      UPDATE TSRTRIP SET SINGLE_PROJ_IND = 'N'
        WHERE TSRTRIP_IS_NUMBER = p_trip_is_nbr AND TSRTRIP_ORG_ID = RPAD(p_org_id,8);
    END IF;
  CLOSE c_check_for_single_proj;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add project to trip.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;

END add_project_to_trip;

/******************************************************************************************************/
FUNCTION add_station_to_project(p_proj_is_nbr IN VARCHAR2,p_sta_is_nbr IN VARCHAR2,
                                p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_sta_is                   NUMBER(8);

  CURSOR c_check_for_station(p_cursor_proj IN VARCHAR2, p_cursor_sta IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSMSTATN_IS_NUMBER FROM TSMPSA WHERE TSMSTATN_IS_NUMBER = p_cursor_sta
      AND TSMSTATN_ORG_ID = RPAD(p_org_id,8)
      AND TSMPROJ_IS_NUMBER = p_cursor_proj AND TSMPROJ_ORG_ID = RPAD(p_org_id,8);

BEGIN

  /* see if station is on project */
  OPEN c_check_for_station(p_proj_is_nbr,p_sta_is_nbr,p_org_id);
  FETCH c_check_for_station INTO v_sta_is;
  IF c_check_for_station%NOTFOUND THEN

    /* add station to project if not found */
    INSERT INTO TSMPSA (
      TSMSTATN_IS_NUMBER,
      TSMSTATN_ORG_ID,
      TSMPROJ_IS_NUMBER,
      TSMPROJ_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDT_TS
      )
    VALUES (
      p_sta_is_nbr,          /* TSMSTATN_IS_NUMBER */
      p_org_id,              /* TSMSTATN_ORG_ID */
      p_proj_is_nbr,         /* TSMPROJ_IS_NUMBER */
      p_org_id,              /* TSMPROJ_ORG_ID */
      sim2_export_pkg.v_pkg_ownerid, 
      SYSDATE                /* D_LAST_UPDT_TS */
      );
    END IF;
  CLOSE c_check_for_station;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in add station to project.');
    v_success := 0;
    RETURN 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;

END add_station_to_project;


/******************************************************************************************************/
FUNCTION add_person_to_project(p_person_is_nbr IN NUMBER,p_person_org_id IN VARCHAR2,
                               p_project_is_nbr IN NUMBER,p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_dummy                      NUMBER(8);

  CURSOR c_check_for_person_role(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2,
                                 p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSMPPRA WHERE TSMPERSN_IS_NUMBER = p_person_is_nbr
      AND TSMPERSN_ORG_ID = RPAD(p_person_org_id, 8) AND TSMPROJ_IS_NUMBER = p_project_is_nbr
      AND TSMPROJ_ORG_ID = RPAD(p_project_org_id, 8);

  CURSOR c_check_for_person(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2,
                            p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSMPPA WHERE TSMPERSN_IS_NUMBER = p_person_is_nbr
      AND TSMPERSN_ORG_ID = RPAD(p_person_org_id, 8) AND TSMPROJ_IS_NUMBER = p_project_is_nbr
      AND TSMPROJ_ORG_ID = RPAD(p_project_org_id, 8);

  CURSOR c_check_for_role(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSMPRA WHERE TSMPERSN_IS_NUMBER = p_person_is_nbr
      AND TSMPERSN_ORG_ID = RPAD(p_person_org_id, 8) AND TSMPROLE_IS_NUMBER = 10
      AND TSMPROLE_ORG_ID = 'ZZZZZZZZ';

BEGIN

  /* make sure project, person, role assignment has not already been made */
  OPEN c_check_for_person_role(p_person_is_nbr, p_person_org_id, p_project_is_nbr, p_org_id);
  FETCH c_check_for_person_role INTO v_dummy;
    IF c_check_for_person_role%NOTFOUND THEN

      /* if nothing found then create association using the staff/employee role */
      INSERT INTO TSMPPRA (
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        TSMPROLE_IS_NUMBER,
        TSMPROLE_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_project_is_nbr,                          /* TSMPROJ_IS_NUMBER */
        p_org_id,                                  /* TSMPROJ_ORG_ID */
        p_person_is_nbr,                           /* TSMPERSN_IS_NUMBER */
        p_person_org_id,                           /* TSMPERSN_ORG_ID */
        10,                                        /* TSMPROLE_IS_NUMBER */
        'ZZZZZZZZ',                                /* TSMPROLE_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid,
        SYSDATE                                    /* D_LAST_UPDATE_TS */
        );

    END IF;
  CLOSE c_check_for_person_role;

  /* make sure project, person assignment has not already been made */
  OPEN c_check_for_person(p_person_is_nbr, p_person_org_id, p_project_is_nbr, p_org_id);
  FETCH c_check_for_person INTO v_dummy;
    IF c_check_for_person%NOTFOUND THEN

      /* if nothing found then create association */
      INSERT INTO TSMPPA (
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_project_is_nbr,                          /* TSMPROJ_IS_NUMBER */
        p_org_id,                                  /* TSMPROJ_ORG_ID */
        p_person_is_nbr,                           /* TSMPERSN_IS_NUMBER */
        p_person_org_id,                           /* TSMPERSN_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid, 
        SYSDATE                                    /* D_LAST_UPDATE_TS */
        );

    END IF;
  CLOSE c_check_for_person;

  /* make sure role, person assignment has not already been made */
  OPEN c_check_for_role(p_person_is_nbr, p_person_org_id);
  FETCH c_check_for_role INTO v_dummy;
    IF c_check_for_role%NOTFOUND THEN

      /* if nothing found then create association */
      INSERT INTO TSMPRA (
        TSMPROLE_IS_NUMBER,
        TSMPROLE_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        10,                                        /* TSMPROLE_IS_NUMBER */
        'ZZZZZZZZ',                                /* TSMPROLE_ORG_ID */
        p_person_is_nbr,                           /* TSMPERSN_IS_NUMBER */
        p_person_org_id,                           /* TSMPERSN_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid,
        SYSDATE                                    /* D_LAST_UPDATE_TS */
        );

    END IF;
  CLOSE c_check_for_role;


  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in add person to project.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END add_person_to_project;

/******************************************************************************************************/
FUNCTION add_cporg_to_project(
         p_cporg_is_nbr    IN NUMBER,
         p_cporg_org_id    IN VARCHAR2,
         p_project_is_nbr  IN NUMBER,
         p_project_org_id  IN VARCHAR2)
         RETURN NUMBER
IS
 
  CURSOR c_tsmpcoa( 
         p_cporg_is_nbr    IN NUMBER, 
         p_cporg_org_id    IN VARCHAR2,
         p_project_is_nbr  IN NUMBER, 
         p_project_org_id  IN VARCHAR2) IS
  SELECT TSMPROJ_IS_NUMBER, TSMPROJ_ORG_ID, TSMCPORG_IS_NUMBER, TSMCPORG_ORG_ID
  FROM   TSMPCOA
  WHERE  TSMCPORG_IS_NUMBER = p_cporg_is_nbr
  AND    TSMCPORG_ORG_ID    = RPAD(p_cporg_org_id, 8)
  AND    TSMPROJ_IS_NUMBER  = p_project_is_nbr
  AND    TSMPROJ_ORG_ID     = RPAD(p_project_org_id, 8);
  
  r_tsmpcoa    c_tsmpcoa%rowtype;

BEGIN

  -- try to retrive the cooperating organization from STORET
  OPEN c_tsmpcoa(
       p_cporg_is_nbr, 
       p_cporg_org_id,
       p_project_is_nbr, 
       p_project_org_id
  );
  FETCH c_tsmpcoa INTO r_tsmpcoa;
  
  -- Do we need to make the Project Cooperating Organization Assignment (PCOA)?
  IF c_tsmpcoa%NOTFOUND THEN
    -- if yes
    
    -- assign the cooperating organization to the project
    INSERT INTO TSMPCOA
    (
      TSMPROJ_IS_NUMBER,
      TSMPROJ_ORG_ID,
      TSMCPORG_IS_NUMBER,
      TSMCPORG_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS
    )
    VALUES
    (
      p_project_is_nbr,              -- TSMPROJ_IS_NUMBER
      p_project_org_id,              -- TSMPROJ_ORG_ID
      p_cporg_is_nbr,                -- TSMPERSN_IS_NUMBER
      p_cporg_org_id,                -- TSMPERSN_ORG_ID
      sim2_export_pkg.v_pkg_ownerid, -- D_USERID_CODE
      sysdate                        -- D_LAST_UPDATE_TS
    );

  END IF;

  CLOSE c_tsmpcoa;

  COMMIT;

  -- return success
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    -- output error message
    sim2_export_pkg.write_error(
      v_export_log_seq, 
      SQLERRM ||' - in add person to project.'
    );
    
    -- eh?
    v_success := 0;

    -- eh?
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;

    -- return failure
    RETURN 0;

END add_cporg_to_project;


/******************************************************************************************************/
FUNCTION create_trip(p_trip_rec IN TSRTRIP%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create trip record */
  INSERT INTO TSRTRIP (
    TSRTRIP_IS_NUMBER,
    TSRTRIP_ORG_ID,
    ID_CODE,
    NAME,
    START_DATE,
    START_TIME,
    START_TIME_ZONE,
    END_DATE,
    END_TIME,
    END_TIME_ZONE,
    LEADER_NAME,
    VEHICLE_SHIP_NAME,
    TRIP_PLAN_TEXT,
    COMMENT_TEXT,
    SINGLE_PROJ_IND,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSMORGAN_IS_NUMBER
    )
  VALUES(
    p_trip_rec.TSRTRIP_IS_NUMBER,    /* TSRTRIP_IS_NUMBER */
    p_trip_rec.TSRTRIP_ORG_ID,       /* TSRTRIP_ORG_ID */
    p_trip_rec.ID_CODE,              /* ID_CODE */
    p_trip_rec.NAME,                 /* NAME */
    p_trip_rec.START_DATE,           /* START_DATE */
    p_trip_rec.START_TIME,           /* START_TIME */
    p_trip_rec.START_TIME_ZONE,      /* START_TIME_ZONE */
    p_trip_rec.END_DATE,             /* END_DATE */
    p_trip_rec.END_TIME,             /* END_TIME */
    p_trip_rec.END_TIME_ZONE,        /* END_TIME_ZONE */
    p_trip_rec.LEADER_NAME,          /* LEADER_NAME */
    p_trip_rec.VEHICLE_SHIP_NAME,    /* VEHICLE_SHIP_NAME */
    p_trip_rec.TRIP_PLAN_TEXT,       /* TRIP_PLAN_TEXT */
    p_trip_rec.COMMENT_TEXT,         /* COMMENT_TEXT */
    p_trip_rec.SINGLE_PROJ_IND,      /* SINGLE_PROJ_IND */
    p_trip_rec.D_USERID_CODE,        /* D_USERID_CODE */
    p_trip_rec.D_LAST_UPDATE_TS,     /* D_LAST_UPDATE_TS */
    p_trip_rec.TSMORGAN_IS_NUMBER    /* TSMORGAN_IS_NUMBER */
    );

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create activity location.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
    END IF;
    RETURN 0;

END create_trip;

/******************************************************************************************************/
FUNCTION create_activity_location(p_org_id IN VARCHAR2,p_activity_is_nbr IN NUMBER,
                                  p_activity_seq IN NUMBER,
                                  p_station_is_nbr IN NUMBER, p_station_org_id IN VARCHAR2)
                                  RETURN NUMBER IS

  CURSOR c_aal(p_activity_seq NUMBER) IS
    SELECT AAL_TSMALP_IS_NUMBER, AAL_TSRFDACT_IS_NUMBER,
           AAL_TSRAAL_TYPE_NAME, AAL_BOTTOM_DEPTH_MSR,AAL_BOTTOM_DEPTH_UOM,
           AAL_ADDTNL_LOC_INFO
    FROM   SIM_ACTUAL_ACTIVITY_LOC
    WHERE  AAL_FA_SEQ = p_activity_seq;

BEGIN

  FOR v_aal IN c_aal(p_activity_seq) LOOP

    IF v_aal.aal_bottom_depth_msr IS NULL THEN
      v_aal.aal_bottom_depth_msr := 0; --Default NULL Value
    END IF;

  /* create activity location record */

    INSERT INTO TSRAAL (
      TSMALP_IS_NUMBER,
      TSMALP_ORG_ID,
      TSRFDACT_IS_NUMBER,
      TSRFDACT_ORG_ID,
      TSRAAL_TYPE_NAME,
      BOTTOM_DEPTH_MSR,
      BOTTOM_DEPTH_UN_CD,
      ADDTNL_LOC_INFO,
      D_USERID_CODE,
      D_LAST_UPDATE_TS
      )
    VALUES(
      v_aal.aal_tsmalp_is_number,          /*tsmalp_is_number*/
      RPAD(p_org_id,8),                    /*tsmalp_org_id*/
      p_activity_is_nbr,                   /*tsrfdact_is_number*/
      RPAD(p_org_id,8),                    /*tsrfdact_org_id*/
      RPAD(v_aal.aal_tsraal_type_name,15), /* TSRAAL_TYPE_NAME */
      v_aal.aal_bottom_depth_msr,
      RPAD(v_aal.aal_bottom_depth_uom,3),
      v_aal.aal_addtnl_loc_info,
      sim2_export_pkg.v_pkg_ownerid,
      SYSDATE                              /* D_LAST_UPDATE_TS */
      );

      UPDATE SIM_ACTUAL_ACTIVITY_LOC
        SET AAL_TSRFDACT_IS_NUMBER = p_activity_is_nbr
        WHERE AAL_FA_SEQ = p_activity_seq;
    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  in create activity location.');
    v_success := 0;
    IF t_success > v_success THEN 
 	    t_success := v_success;
		END IF;
RETURN 0;

END create_activity_location;
/******************************************************************************************************/
FUNCTION create_tsrtod_record(p_org_id IN VARCHAR2,p_activity_is_nbr IN NUMBER,
                                  p_activity_seq IN NUMBER) RETURN NUMBER IS


  CURSOR c_tod(p_activity_seq NUMBER) IS
    SELECT TOD_TSRFDACT_IS_NUMBER, TOD_TSRFDACT_ORG_ID, TOD_FISHED_DURATION,
           TOD_FISHED_DUR_UOM, TOD_BOAT_SPEED_MSR, TOD_BOAT_SPEED_UOM,
           TOD_FISHED_DISTANCE, TOD_FISHED_DISTANCE_UOM, TOD_REL_CURRENT_DIR,
           TOD_REL_WIND_DIR, TOD_COMMENT_TEXT
    FROM   SIM_TRAWL_OPS_DET
    WHERE  TOD_FA_SEQ = p_activity_seq;

BEGIN

  FOR v_tod IN c_tod(p_activity_seq) LOOP
  --Default NULL values For TSRTOD
   IF v_tod.tod_boat_speed_msr IS NULL THEN
     v_tod.tod_boat_speed_msr := 0;
   END IF;
   IF v_tod.tod_fished_distance IS NULL THEN
     v_tod.tod_fished_distance := 0;
   END IF;
   IF v_tod.tod_rel_current_dir IS NULL THEN
     v_tod.tod_rel_current_dir := 0;
   END IF;
   IF v_tod.tod_rel_wind_dir IS NULL THEN
     v_tod.tod_rel_wind_dir := 0;
   END IF;

  /* create tsrtod record */
  INSERT INTO TSRTOD (
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    FISHED_DURATN_MSR,
    FISHD_DURTN_UNT_CD,
    BOAT_SPEED_MSR,
    BOAT_SPEED_UN_CD,
    FISHED_DISTANCE,
    FISHED_DISTANCE_UN,
    REL_CURRENT_DIR,
    REL_WIND_DIR,
    COMMENT_TEXT,
    D_USERID_CODE,
    D_LAST_UPDATE_TS
    )
  VALUES(
    v_tod.tod_tsrfdact_is_number,
    v_tod.tod_tsrfdact_org_id,
    v_tod.tod_fished_duration,
    v_tod.tod_fished_dur_uom,
    v_tod.tod_boat_speed_msr,
    v_tod.tod_boat_speed_uom,
    v_tod.tod_fished_distance,
    v_tod.tod_fished_distance_uom,
    v_tod.tod_rel_current_dir,
    v_tod.tod_rel_wind_dir,
    v_tod.tod_comment_text,
    sim2_export_pkg.v_pkg_ownerid,
    SYSDATE                              /* D_LAST_UPDATE_TS */
    );

  COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create tsrtod record.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_tsrtod_record;
/******************************************************************************************************/
FUNCTION create_tsrsmple_record(p_org_id IN VARCHAR2,p_activity_is_nbr IN NUMBER,
                                  p_activity_seq IN NUMBER,p_sample_rec IN TSRSMPLE%ROWTYPE) RETURN NUMBER IS

  v_duration_units       VARCHAR2(10);

  CURSOR c_sample(p_activity_seq NUMBER) IS
    SELECT SS_TSRFDACT_IS_NUMBER, SS_TSRFDACT_ORG_ID, SS_TOTAL_VOLUME_MSR,
           SS_TOTAL_VOL_UOM, SS_TOTAL_WEIGHT_MSR, SS_TOTAL_WT_UOM, SS_DATA_FILE_NAME_LOC,
           SS_CONTAINER_TYPE_NM, SS_CONTAINER_COLOR, SS_CONTAINER_SIZE_MSR,
           SS_CONTAINER_SIZE_UN, SS_TEMP_PRESERVN_TYPE, SS_PARENT_IND_CODE,
           SS_GEAR_DEPLOYMENT, SS_TRANSPORT_STORAGE, SS_PRESRV_STRGE_PRCDR,
           SS_DURATION_TIME, SS_DURATION_UNIT
    FROM   SIM_SAMPLE
    WHERE  SS_FA_SEQ = p_activity_seq;

BEGIN

  FOR v_sample IN c_sample(p_activity_seq) LOOP
  --Default NULL Values For TSRSMPLE
   IF v_sample.SS_TOTAL_VOLUME_MSR IS NULL THEN
     v_sample.SS_TOTAL_VOLUME_MSR := 0;
   END IF;
   IF v_sample.SS_TOTAL_WEIGHT_MSR IS NULL THEN
     v_sample.SS_TOTAL_WEIGHT_MSR := 0;
   END IF;
   IF v_sample.SS_CONTAINER_SIZE_MSR IS NULL THEN
     v_sample.SS_CONTAINER_SIZE_MSR := 0;
   END IF;
   IF v_sample.SS_DURATION_TIME IS NULL THEN
     v_sample.SS_DURATION_TIME := 0;
   END IF;

  /* create tsrsmple record */
  INSERT INTO TSRSMPLE (
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TOTAL_VOLUME_MSR,
    TOTAL_VOLUME_UN_CD,
    TOTAL_WEIGHT_MSR,
    TOTAL_WEIGHT_UN_CD,
    DATA_FILE_NAME_LOC,
    CONTAINER_TYPE_NM,
    CONTAINER_COLOR,
    CONTAINER_SIZE_MSR,
    CONTAINER_SIZE_UN,
    TEMP_PRESERVN_TYPE,
    PARENT_IND_CODE,
    GEAR_DEPLOYMENT,
    TRANSPORT_STORAGE,
    PRESRV_STRGE_PRCDR,
    DURATION_TIME,
    DURATION_UNITS,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSMPRMVL_IS_NUMBER,
    TSMPRMVL_ORG_ID,
    TSMPRMVL0IS_NUMBER,
    TSMPRMVL0ORG_ID,
    TSMPRMVL1IS_NUMBER,
    TSMPRMVL1ORG_ID
    )
  VALUES(
    v_sample.SS_TSRFDACT_IS_NUMBER,
    v_sample.SS_TSRFDACT_ORG_ID,
    v_sample.SS_TOTAL_VOLUME_MSR,
    v_sample.SS_TOTAL_VOL_UOM,
    v_sample.SS_TOTAL_WEIGHT_MSR,
    v_sample.SS_TOTAL_WT_UOM,
    v_sample.SS_DATA_FILE_NAME_LOC,
    v_sample.SS_CONTAINER_TYPE_NM,
    v_sample.SS_CONTAINER_COLOR,
    v_sample.SS_CONTAINER_SIZE_MSR,
    v_sample.SS_CONTAINER_SIZE_UN,
    v_sample.SS_TEMP_PRESERVN_TYPE,
    v_sample.SS_PARENT_IND_CODE,
    v_sample.SS_GEAR_DEPLOYMENT,
    v_sample.SS_TRANSPORT_STORAGE,
    v_sample.SS_PRESRV_STRGE_PRCDR,
    v_sample.SS_DURATION_TIME,
    v_duration_units,
    sim2_export_pkg.v_pkg_ownerid,
    SYSDATE,                              /* D_LAST_UPDATE_TS */
    p_sample_rec.TSMPRMVL_IS_NUMBER,
    p_sample_rec.TSMPRMVL_ORG_ID,
    p_sample_rec.TSMPRMVL0IS_NUMBER,
    p_sample_rec.TSMPRMVL0ORG_ID,
    p_sample_rec.TSMPRMVL1IS_NUMBER,
    p_sample_rec.TSMPRMVL1ORG_ID
    );

  COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create tsrsmple record.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_tsrsmple_record;


/******************************************************************************************************/
FUNCTION create_tsrbrg_records(p_org_id IN VARCHAR2,p_activity_is_nbr IN NUMBER,
                                  p_activity_seq IN NUMBER) RETURN NUMBER IS

  v_lifestage_name  VARCHAR2(25);
  v_sex_name        VARCHAR2(15);
  v_tsrbrg_is_nbr   NUMBER(8);

  CURSOR c_brg(p_activity_seq NUMBER) IS
    SELECT *
    FROM   SIM_BIO_RESULTS_GRP
    WHERE  BRG_FA_SEQ = p_activity_seq;

BEGIN

  FOR v_brg IN c_brg(p_activity_seq) LOOP
    IF v_brg.brg_tsrcpv_lifestage_is_nbr IS NOT NULL THEN
      v_lifestage_name := get_tsrcpv_value(v_brg.brg_tsrcpv_lifestage_is_nbr,v_brg.brg_tsrcpv_lifestage_org_id);
    END IF;

    IF v_brg.brg_tsrcpv_sex_is_number IS NOT NULL THEN
      v_sex_name := get_tsrcpv_value(v_brg.brg_tsrcpv_sex_is_number,v_brg.brg_tsrcpv_sex_org_id);
    END IF;

    IF v_brg.brg_summary_grp_count IS NULL THEN
      v_brg.brg_summary_grp_count := 0; --Default NULL Value
    END IF;

    v_tsrbrg_is_nbr := get_next_tsrbrg_is_nbr(p_org_id);

  /* create tsrbrg records */
    INSERT INTO TSRBRG (
      TSRBRG_IS_NUMBER,
      TSRBRG_ORG_ID,
      ID_CODE,
      TYPE_NAME,
      TYPE_INDICATOR,
      SEX_NAME,
      LIFE_STAGE_NAME,
      VALUE_TYPE_NAME,
      SUMMARY_GRP_COUNT,
      DESCRIPTION_TEXT,
      SPECIES_NUMBER,
      TSRFDACT_IS_NUMBER,
      TSRFDACT_ORG_ID,
      TSRCHAR_IS_NUMBER,
      TSRCHAR_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS
      )
    VALUES(
      v_tsrbrg_is_nbr,
      RPAD(p_org_id,8),
      v_brg.brg_id_code,
      v_brg.brg_type,
      v_brg.brg_freq_analysis,
      v_sex_name,
      v_lifestage_name,
      v_brg.brg_value_type_name,
      v_brg.brg_summary_grp_count,
      v_brg.brg_description_text,
      v_brg.brg_species_number,
      p_activity_is_nbr,
      RPAD(p_org_id,8),
      v_brg.brg_tsrchar_is_number,
      v_brg.brg_tsrchar_org_id,
      sim2_export_pkg.v_pkg_ownerid,
      SYSDATE                              /* D_LAST_UPDATE_TS */
      );

    UPDATE SIM_BIO_RESULTS_GRP
      SET BRG_TSRBRG_IS_NUMBER = v_tsrbrg_is_nbr,
         BRG_TSRBRG_ORG_ID = p_org_id
      WHERE BRG_SEQ = v_brg.brg_seq;

    UPDATE SIM_RESULTS
      SET RS_TSRBRG_IS_NUMBER = v_tsrbrg_is_nbr,
         RS_TSRBRG_ORG_ID = p_org_id
      WHERE RS_BRG_SEQ = v_brg.brg_seq
      AND RS_FA_SEQ = p_activity_seq;

    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create tsrbrg records.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_tsrbrg_records;

/******************************************************************************************************/
FUNCTION create_tsrbrgi_records(p_org_id IN VARCHAR2,p_activity_is_nbr IN NUMBER,
                                  p_activity_seq IN NUMBER) RETURN NUMBER IS
  v_tsrbrgi_is_nbr   NUMBER(8);

  CURSOR c_brgi(p_activity_seq NUMBER) IS
    SELECT BRGI.BRGI_SEQ BRGI_SEQ, BRGI.BRGI_INDIVIDUAL_NUMBER INDIVIDUAL_NUMBER,
           BRG.BRG_TSRBRG_IS_NUMBER TSRBRG_IS_NUMBER,
           BRG.BRG_TSRBRG_ORG_ID TSRBRG_ORG_ID
    FROM   SIM_BIO_RESULTS_GRP_INDIVIDUAL BRGI, SIM_BIO_RESULTS_GRP BRG
    WHERE  BRGI_FA_SEQ = p_activity_seq
    AND    BRG.BRG_SEQ = BRGI.BRGI_BRG_SEQ
    ORDER BY INDIVIDUAL_NUMBER;

BEGIN

  FOR v_brgi IN c_brgi(p_activity_seq) LOOP


   v_tsrbrgi_is_nbr := get_next_tsrbrgi_is_nbr(p_org_id);

  /* create tsrbrgi records */
  INSERT INTO TSRBRGI (
    TSRBRGI_IS_NUMBER,
    TSRBRGI_ORG_ID,
    INDIVIDUAL_NUMBER,
    TSRBRG_IS_NUMBER,
    TSRBRG_ORG_ID,
    D_USERID_CODE,
    D_LAST_UPDATE_TS
    )
  VALUES(
    v_tsrbrgi_is_nbr,
    RPAD(p_org_id,8),
    v_brgi.individual_number,
    v_brgi.tsrbrg_is_number,
    v_brgi.tsrbrg_org_id,
    sim2_export_pkg.v_pkg_ownerid,
    SYSDATE                              /* D_LAST_UPDATE_TS */
    );

   UPDATE SIM_RESULTS
     SET RS_TSRBRGI_IS_NUMBER = v_tsrbrgi_is_nbr,
         RS_TSRBRGI_ORG_ID = p_org_id
   WHERE RS_BRGI_SEQ = v_brgi.brgi_seq
     AND RS_FA_SEQ = p_activity_seq;


  COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create tsrbrgi records.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_tsrbrgi_records;
/******************************************************************************************************/
FUNCTION create_tsrrci_record(
         p_org_id        IN VARCHAR2, 
         p_result_is_nbr IN NUMBER) 
         RETURN NUMBER IS

  CURSOR c_rci(
    p_org_id        VARCHAR2, 
    p_result_is_nbr NUMBER) IS
  SELECT RCI_TSRRSULT_IS_NUMBER, RCI_TSRRSULT_ORG_ID, RCI_LOWER_BND_AMT,
         RCI_UPPER_BND_AMT, RCI_PRIMARY_CLASS_IND, RCI_SECOND_CLASS_IND,
         RCI_PARTICLE_SIZE_BASIS, RCI_TSRUOM_IS_NUMBER, RCI_TSRUOM_ORG_ID
  FROM   SIM_RESULT_CLASS_INDICATOR
  WHERE  RCI_TSRRSULT_IS_NUMBER = p_result_is_nbr
  AND    RCI_TSRRSULT_ORG_ID    = p_org_id;

BEGIN

  FOR v_rci IN c_rci(p_org_id, p_result_is_nbr) LOOP
    
    --Default NULL Values For TSRRCI
    IF v_rci.rci_lower_bnd_amt IS NULL THEN
      v_rci.rci_lower_bnd_amt := 0;
    END IF;
    
    IF v_rci.rci_upper_bnd_amt IS NULL THEN
      v_rci.rci_upper_bnd_amt := 0;
    END IF;

     /* create tsrrci records */
    INSERT INTO TSRRCI
    (
      TSRRSULT_IS_NUMBER,
      TSRRSULT_ORG_ID,
      LOWER_BND_AMT,
      UPPER_BND_AMT,
      PRIM_CLASS_DESC,
      SEC_CLASS_DESC,
      PARTICLE_SIZE_BASIS,
      TSRUOM_IS_NUMBER,
      TSRUOM_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS
    )
    VALUES
    (
      v_rci.rci_tsrrsult_is_number,
      v_rci.rci_tsrrsult_org_id,
      v_rci.rci_lower_bnd_amt,
      v_rci.rci_upper_bnd_amt,
      v_rci.rci_primary_class_ind,
      v_rci.rci_second_class_ind,
      v_rci.rci_particle_size_basis,
      v_rci.rci_tsruom_is_number,
      v_rci.rci_tsruom_org_id,
      sim2_export_pkg.v_pkg_ownerid,
      SYSDATE                              /* D_LAST_UPDATE_TS */
    );

    COMMIT;
    
  END LOOP;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(
      v_export_log_seq, 
      SQLERRM || ' - in create tsrrci records.'
    );
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_tsrrci_record;

/******************************************************************************************************/
FUNCTION add_lab_remarks_to_result(p_tsrrlra_rec IN TSRRLRA%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a Lab Remark record */

  INSERT INTO TSRRLRA (
                       tsrrsult_is_number,
                       tsrrsult_org_id,
                       tsrlbrmk_is_number,
                       tsrlbrmk_org_id,
                       d_userid_code,
                       d_last_update_ts
                       )
               VALUES (
                       p_tsrrlra_rec.tsrrsult_is_number,
                       p_tsrrlra_rec.tsrrsult_org_id,
                       p_tsrrlra_rec.tsrlbrmk_is_number,
                       p_tsrrlra_rec.tsrlbrmk_org_id,
                       p_tsrrlra_rec.d_userid_code,
                       p_tsrrlra_rec.d_last_update_ts
                       );

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create_tsrrlra_record.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END add_lab_remarks_to_result;
/******************************************************************************************************/
FUNCTION create_trap_net(p_tsrtnod_rec IN TSRTNOD%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a Trap/Net details record */

  INSERT INTO TSRTNOD (
                       TSRFDACT_IS_NUMBER,
                       TSRFDACT_ORG_ID,
                       SMPLNG_DURATN_MSR,
                       SMPLNG_DRTN_UNT_CD,
                       ORIENTN_TO_CURRENT,
                       REL_CURRENT_DIR,
                       REL_WIND_DIR,
                       COMMENT_TEXT,
                       D_USERID_CODE,
                       D_LAST_UPDATE_TS
                       )
               VALUES (
                       p_tsrtnod_rec.TSRFDACT_IS_NUMBER,             /* TSRFDACT_IS_NUMBER */
                       p_tsrtnod_rec.TSRFDACT_ORG_ID,
                       p_tsrtnod_rec.SMPLNG_DURATN_MSR,
                       p_tsrtnod_rec.SMPLNG_DRTN_UNT_CD,
                       p_tsrtnod_rec.ORIENTN_TO_CURRENT,
                       p_tsrtnod_rec.REL_CURRENT_DIR,
                       p_tsrtnod_rec.REL_WIND_DIR,
                       p_tsrtnod_rec.COMMENT_TEXT,
                       p_tsrtnod_rec.D_USERID_CODE,
                       p_tsrtnod_rec.D_LAST_UPDATE_TS
                       );

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create trap net.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_trap_net;
/******************************************************************************************************/

FUNCTION create_tsrcldes(p_tsrcldes_rec IN TSRCLDES%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a Trap/Net details record */

  INSERT INTO TSRCLDES (
                       TSRRSULT_IS_NUMBER,
                       TSRRSULT_ORG_ID,
                       CELL_TYPE_NM,
                       CELL_SHAPE_TYPE_NM,
                       D_USERID_CODE,
                       D_LAST_UPDATE_TS
                       )
               VALUES (
                        p_tsrcldes_rec.tsrrsult_is_number,
                        p_tsrcldes_rec.tsrrsult_org_id,
                        p_tsrcldes_rec.cell_type_nm,
                        p_tsrcldes_rec.cell_shape_type_nm,
                        p_tsrcldes_rec.d_userid_code,
                        p_tsrcldes_rec.d_last_update_ts
                       );

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create tsrcldes.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_tsrcldes;
/******************************************************************************************************/
FUNCTION create_station_visit(p_org_id IN VARCHAR2, p_stvst_rec IN TSRSTVST%ROWTYPE)
  RETURN NUMBER IS


BEGIN

  BEGIN
    INSERT INTO TSRSTVST (
      TSRSTVST_IS_NUMBER,
      TSRSTVST_ORG_ID,
      ID_NUMBER,
      ARRIVAL_DATE,
      ARRIVAL_TIME,
      ARRIVAL_TIME_ZONE,
      DEPARTURE_DATE,
      DEPARTURE_TIME,
      DEPRTURE_TIME_ZONE,
      COMMENT_TEXT,
      D_USERID_CODE,
      D_LAST_UPDATE_TS,
      TSRTRIP_IS_NUMBER,
      TSRTRIP_ORG_ID,
      TSMSTATN_IS_NUMBER,
      TSMSTATN_ORG_ID,
      BLOB_TYPE,
      BLOB_TITLE
      )
    VALUES (
      p_stvst_rec.tsrstvst_is_number                         /* TSRSTVST_IS_NUMBER */,
      p_stvst_rec.tsrstvst_org_id                            /* TSRSTVST_ORG_ID */,
      p_stvst_rec.id_number                                  /* ID_NUMBER */,
      p_stvst_rec.ARRIVAL_DATE                               /* ARRIVAL_DATE */,
      p_stvst_rec.ARRIVAL_TIME                               /* ARRIVAL_TIME */,
      p_stvst_rec.ARRIVAL_TIME_ZONE                          /* ARRIVAL_TIME_ZONE */,
      p_stvst_rec.DEPARTURE_DATE                             /* DEPARTURE_DATE */,
      p_stvst_rec.DEPARTURE_TIME                             /* DEPARTURE_TIME */,
      p_stvst_rec.DEPRTURE_TIME_ZONE                         /* DEPRTURE_TIME_ZONE */,
      p_stvst_rec.COMMENT_TEXT                               /* COMMENT_TEXT */,
      sim2_export_pkg.v_pkg_ownerid,
      SYSDATE                                                /* D_LAST_UPDATE_TS */,
      p_stvst_rec.TSRTRIP_IS_NUMBER                          /* TSRTRIP_IS_NUMBER */,
      p_stvst_rec.TSRTRIP_ORG_ID                             /* TSRTRIP_ORG_ID */,
      p_stvst_rec.TSMSTATN_IS_NUMBER                         /* TSMSTATN_IS_NUMBER */,
      p_stvst_rec.TSMSTATN_ORG_ID                            /* TSMSTATN_ORG_ID */,
      p_stvst_rec.BLOB_TYPE                                  /* BLOB_TYPE */,
      p_stvst_rec.BLOB_TITLE                                 /* BLOB_TITLE */
      );

  EXCEPTION
    WHEN OTHERS THEN
      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  inserting visit record into TSRSTVST');
      v_success := 0;
      IF t_success > v_success THEN 
		    t_success := v_success;
		  END IF;
  END;

  RETURN 1;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create station visit.');
     v_success := 0;
     IF t_success > v_success THEN 
		   t_success := v_success;
		 END IF;
     RETURN 0;

END create_station_visit;

/******************************************************************************************************/
FUNCTION create_activity(p_activity_rec IN OUT TSRFDACT%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create activity using info passed in through activity record */

  INSERT INTO TSRFDACT (
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TYPE_NAME,
    CATEGORY_TYPE_NAME,
    ID_CODE,
    REPLICATE_NUMBER,
    MEDIUM_TYPE_NAME,
    INTENT_TYPE_NAME,
    COMMUNITY_NAME,
    START_DATE,
    START_TIME,
    START_TIME_ZONE,
    STOP_DATE,
    STOP_TIME,
    STOP_TIME_ZONE,
    COMMENT_TEXT,
    ZONE_TYPE_NAME,
    RELTV_DEPTH_NAME,
    DEPTH_REF_POINT,
    DEPTH_TO_ACTIVITY,
    DEPTH_TO_ACT_UN_CD,
    UPPER_DEPTH_TO_ACT,
    DEPTH_MSR_UNT_CD,
    LOWER_DEPTH_TO_ACT,
    THERMOCLINE_LOC_CD,
    PYCNOCLINE_LOC_CD,
    HALOCLINE_LOC_CD,
    SPECIES_NUMBER,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSROPPRD_ORG_ID,
    TSROPPRD_IS_NUMBER,
    TSRFDACT0IS_NUMBER,
    TSRFDACT0ORG_ID,
    TSRFDACT1IS_NUMBER,
    TSRFDACT1ORG_ID,
    TSRSTVST_IS_NUMBER,
    TSRSTVST_ORG_ID,
    TSRBIOPT_IS_NUMBER,
    TSRBIOPT_ORG_ID,
    TSRCHAR_IS_NUMBER,
    TSRCHAR_ORG_ID,
    TSRFLDPR_IS_NUMBER,
    TSRFLDPR_ORG_ID,
    TSRFLDGR_IS_NUMBER,
    TSRFLDGR_ORG_ID,
    TSRGRCFG_IS_NUMBER,
    TSRGRCFG_ORG_ID,
    QC_INDICATOR,
    CHAIN_OF_CUSTODY_ID,
    TSRMATRX_IS_NUMBER,
    TSRMATRX_ORG_ID,
    BLOB_TYPE,
    BLOB_TITLE
    )
  VALUES (
    p_activity_rec.TSRFDACT_IS_NUMBER,          /* TSRFDACT_IS_NUMBER */
    p_activity_rec.TSRFDACT_ORG_ID,             /* TSRFDACT_ORG_ID */
    p_activity_rec.TYPE_NAME,                   /* TYPE_NAME */
    p_activity_rec.CATEGORY_TYPE_NAME,          /* CATEGORY_TYPE_NAME */
    p_activity_rec.ID_CODE,                     /* ID_CODE */
    p_activity_rec.REPLICATE_NUMBER,            /* REPLICATE_NUMBER */
    p_activity_rec.MEDIUM_TYPE_NAME,            /* MEDIUM_TYPE_NAME */
    p_activity_rec.INTENT_TYPE_NAME,            /* INTENT_TYPE_NAME */
    p_activity_rec.COMMUNITY_NAME,              /* COMMUNITY_NAME */
    p_activity_rec.START_DATE,                  /* START_DATE */
    p_activity_rec.START_TIME,                  /* START_TIME */
    p_activity_rec.START_TIME_ZONE,             /* START_TIME_ZONE */
    p_activity_rec.STOP_DATE,                   /* STOP_DATE */
    p_activity_rec.STOP_TIME,                   /* STOP_TIME */
    p_activity_rec.STOP_TIME_ZONE,              /* STOP_TIME_ZONE */
    p_activity_rec.COMMENT_TEXT,                /* COMMENT_TEXT */
    p_activity_rec.ZONE_TYPE_NAME,              /* ZONE_TYPE_NAME */
    p_activity_rec.RELTV_DEPTH_NAME,            /* RELTV_DEPTH_NAME */
    p_activity_rec.DEPTH_REF_POINT,             /* DEPTH_REF_POINT */
    p_activity_rec.DEPTH_TO_ACTIVITY,           /* DEPTH_TO_ACTIVITY */
    p_activity_rec.DEPTH_TO_ACT_UN_CD,          /* DEPTH_TO_ACT_UN_CD */
    p_activity_rec.UPPER_DEPTH_TO_ACT,          /* UPPER_DEPTH_TO_ACT */
    p_activity_rec.DEPTH_MSR_UNT_CD,            /* DEPTH_MSR_UNT_CD */
    p_activity_rec.LOWER_DEPTH_TO_ACT,          /* LOWER_DEPTH_TO_ACT */
    p_activity_rec.THERMOCLINE_LOC_CD,          /* THERMOCLINE_LOC_CD */
    p_activity_rec.PYCNOCLINE_LOC_CD,           /* PYCNOCLINE_LOC_CD */
    p_activity_rec.HALOCLINE_LOC_CD,            /* HALOCLINE_LOC_CD */
    p_activity_rec.SPECIES_NUMBER,              /* SPECIES_NUMBER */
    p_activity_rec.D_USERID_CODE,               /* D_USERID_CODE */
    p_activity_rec.D_LAST_UPDATE_TS,            /* D_LAST_UPDATE_TS */
    p_activity_rec.TSROPPRD_ORG_ID,             /* TSROPPRD_ORG_ID */
    p_activity_rec.TSROPPRD_IS_NUMBER,          /* TSROPPRD_IS_NUMBER */
    p_activity_rec.TSRFDACT0IS_NUMBER,          /* TSRFDACT0IS_NUMBER */
    p_activity_rec.TSRFDACT0ORG_ID,             /* TSRFDACT0ORG_ID */
    p_activity_rec.TSRFDACT1IS_NUMBER,          /* TSRFDACT1IS_NUMBER */
    p_activity_rec.TSRFDACT1ORG_ID,             /* TSRFDACT1ORG_ID */
    p_activity_rec.TSRSTVST_IS_NUMBER,          /* TSRSTVST_IS_NUMBER */
    p_activity_rec.TSRSTVST_ORG_ID,             /* TSRSTVST_ORG_ID */
    p_activity_rec.TSRBIOPT_IS_NUMBER,          /* TSRBIOPT_IS_NUMBER */
    p_activity_rec.TSRBIOPT_ORG_ID,             /* TSRBIOPT_ORG_ID */
    p_activity_rec.TSRCHAR_IS_NUMBER,           /* TSRCHAR_IS_NUMBER */
    p_activity_rec.TSRCHAR_ORG_ID,              /* TSRCHAR_ORG_ID */
    p_activity_rec.TSRFLDPR_IS_NUMBER,          /* TSRFLDPR_IS_NUMBER */
    p_activity_rec.TSRFLDPR_ORG_ID,             /* TSRFLDPR_ORG_ID */
    p_activity_rec.TSRFLDGR_IS_NUMBER,          /* TSRFLDGR_IS_NUMBER */
    p_activity_rec.TSRFLDGR_ORG_ID,             /* TSRFLDGR_ORG_ID */
    p_activity_rec.TSRGRCFG_IS_NUMBER,          /* TSRGRCFG_IS_NUMBER */
    p_activity_rec.TSRGRCFG_ORG_ID,             /* TSRGRCFG_ORG_ID */
    p_activity_rec.QC_INDICATOR,
    p_activity_rec.CHAIN_OF_CUSTODY_ID,
    p_activity_rec.TSRMATRX_IS_NUMBER,
    p_activity_rec.TSRMATRX_ORG_ID,
    p_activity_rec.BLOB_TYPE,                    /* BLOB_TYPE */
    p_activity_rec.BLOB_TITLE                    /* BLOB_TITLE */
    );

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create activity.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_activity;

/******************************************************************************************************/
FUNCTION create_sample(p_org_id IN VARCHAR2, p_sample_rec IN TSRSMPLE%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a sample record */

  INSERT INTO TSRSMPLE (
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TOTAL_VOLUME_MSR,
    TOTAL_VOLUME_UN_CD,
    TOTAL_WEIGHT_MSR,
    TOTAL_WEIGHT_UN_CD,
    DATA_FILE_NAME_LOC,
    CONTAINER_TYPE_NM,
    CONTAINER_COLOR,
    CONTAINER_SIZE_MSR,
    CONTAINER_SIZE_UN,
    TEMP_PRESERVN_TYPE,
    PARENT_IND_CODE,
    GEAR_DEPLOYMENT,
    TRANSPORT_STORAGE,
    PRESRV_STRGE_PRCDR,
    DURATION_TIME,
    DURATION_UNITS,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSMPRMVL_IS_NUMBER,
    TSMPRMVL_ORG_ID,
    TSMPRMVL0IS_NUMBER,
    TSMPRMVL0ORG_ID,
    TSMPRMVL1IS_NUMBER,
    TSMPRMVL1ORG_ID
    )
  VALUES (
    p_sample_rec.TSRFDACT_IS_NUMBER,             /* TSRFDACT_IS_NUMBER */
    p_sample_rec.TSRFDACT_ORG_ID,                /* TSRFDACT_ORG_ID */
    p_sample_rec.TOTAL_VOLUME_MSR,               /* TOTAL_VOLUME_MSR */
    p_sample_rec.TOTAL_VOLUME_UN_CD,             /* TOTAL_VOLUME_UN_CD */
    p_sample_rec.TOTAL_WEIGHT_MSR,               /* TOTAL_WEIGHT_MSR */
    p_sample_rec.TOTAL_WEIGHT_UN_CD,             /* TOTAL_WEIGHT_UN_CD */
    p_sample_rec.DATA_FILE_NAME_LOC,             /* DATA_FILE_NAME_LOC */
    p_sample_rec.CONTAINER_TYPE_NM,              /* CONTAINER_TYPE_NM */
    p_sample_rec.CONTAINER_COLOR,                /* CONTAINER_COLOR */
    p_sample_rec.CONTAINER_SIZE_MSR,             /* CONTAINER_SIZE_MSR */
    p_sample_rec.CONTAINER_SIZE_UN,              /* CONTAINER_SIZE_UN */
    p_sample_rec.TEMP_PRESERVN_TYPE,             /* TEMP_PRESERVN_TYPE */
    p_sample_rec.PARENT_IND_CODE,                /* PARENT_IND_CODE */
    p_sample_rec.GEAR_DEPLOYMENT,                /* GEAR_DEPLOYMENT */
    p_sample_rec.TRANSPORT_STORAGE,              /* TRANSPORT_STORAGE */
    p_sample_rec.PRESRV_STRGE_PRCDR,             /* PRESRV_STRGE_PRCDR */
    p_sample_rec.DURATION_TIME,                  /* DURATION_TIME */
    p_sample_rec.DURATION_UNITS,                 /* DURATION_UNITS */
    p_sample_rec.D_USERID_CODE,                  /* D_USERID_CODE */
    p_sample_rec.D_LAST_UPDATE_TS,                /* D_LAST_UPDATE_TS */
    p_sample_rec.TSMPRMVL_IS_NUMBER,
    p_sample_rec.TSMPRMVL_ORG_ID,
    p_sample_rec.TSMPRMVL0IS_NUMBER,
    p_sample_rec.TSMPRMVL0ORG_ID,
    p_sample_rec.TSMPRMVL1IS_NUMBER,
    p_sample_rec.TSMPRMVL1ORG_ID
    );

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create sample.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_sample;
/******************************************************************************************************/
FUNCTION create_trip_qc(p_trip_qc_rec IN OUT TSRFQS%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create trip qc sample using info passed in through trip qc record */

  INSERT INTO TSRFQS (
    TSRFQS_IS_NUMBER,
    TSRFQS_ORG_ID,
    ID_CODE,
    TYPE_NAME,
    NAME,
    CREATION_DATE,
    CREATION_TIME,
    TIME_ZONE,
    TOTAL_VOLUME_MSR,
    TOTAL_VOLUME_UN_CD,
    CONTAINER_TYPE_NM,
    CONTAINER_COLOR,
    CONTAINER_SIZE_MSR,
    CONTAINER_SIZE_UN,
    COMMENT_TEXT,
    CREATION_METHOD,
    TRANSPORT_STORAGE,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSRTRIP_IS_NUMBER,
    TSRTRIP_ORG_ID
    )
  VALUES (
    p_trip_qc_rec.TSRFQS_IS_NUMBER,          /* TSRFQS_IS_NUMBER */
    p_trip_qc_rec.TSRFQS_ORG_ID,             /* TSRFQS_ORG_ID */
    p_trip_qc_rec.ID_CODE,                   /* ID_CODE */
    p_trip_qc_rec.TYPE_NAME,                 /* TYPE_NAME */
    p_trip_qc_rec.NAME,                      /* NAME */
    p_trip_qc_rec.CREATION_DATE,             /* CREATION_DATE */
    p_trip_qc_rec.CREATION_TIME,             /* CREATION_TIME */
    p_trip_qc_rec.TIME_ZONE,                 /* TIME_ZONE */
    p_trip_qc_rec.TOTAL_VOLUME_MSR,          /* TOTAL_VOLUME_MSR */
    p_trip_qc_rec.TOTAL_VOLUME_UN_CD,        /* TOTAL_VOLUME_UN_CD */
    p_trip_qc_rec.CONTAINER_TYPE_NM,         /* CONTAINER_TYPE_NM */
    p_trip_qc_rec.CONTAINER_COLOR,           /* CONTAINER_COLOR */
    p_trip_qc_rec.CONTAINER_SIZE_MSR,        /* CONTAINER_SIZE_MSR */
    p_trip_qc_rec.CONTAINER_SIZE_UN,         /* CONTAINER_SIZE_UN */
    p_trip_qc_rec.COMMENT_TEXT,              /* COMMENT_TEXT */
    p_trip_qc_rec.CREATION_METHOD,           /* CREATION_METHOD */
    p_trip_qc_rec.TRANSPORT_STORAGE,         /* TRANSPORT_STORAGE */
    p_trip_qc_rec.D_USERID_CODE,             /* D_USERID_CODE */
    p_trip_qc_rec.D_LAST_UPDATE_TS,          /* D_LAST_UPDATE_TS */
    p_trip_qc_rec.TSRTRIP_IS_NUMBER,         /* TSRTRIP_IS_NUMBER */
    p_trip_qc_rec.TSRTRIP_ORG_ID             /* TSRTRIP_ORG_ID */
    );

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create trip qc.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_trip_qc;

/******************************************************************************************************/
FUNCTION add_person_to_activity(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2,
                                p_act_is_nbr IN NUMBER, p_act_org_id IN VARCHAR2) RETURN NUMBER IS

  v_persn_is_nbr             NUMBER(8);

  CURSOR c_check_for_person(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2,
                            p_act_is_nbr IN NUMBER, p_act_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSRFAPEA WHERE TSRFDACT_IS_NUMBER = p_act_is_nbr
      AND TSMPERSN_IS_NUMBER = p_persn_is_nbr AND TSRFDACT_ORG_ID = p_act_org_id
      AND TSMPERSN_ORG_ID = p_persn_org_id;

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_person(p_persn_is_nbr,p_persn_org_id,p_act_is_nbr,p_act_org_id);
  FETCH c_check_for_person INTO v_persn_is_nbr;
    IF c_check_for_person%NOTFOUND THEN

      /* add person to activity if not already there */
      INSERT INTO TSRFAPEA (
        TSRFDACT_IS_NUMBER,
        TSRFDACT_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_act_is_nbr,                           /* TSRFDACT_IS_NUMBER */
        p_act_org_id,                           /* TSRFDACT_ORG_ID */
        p_persn_is_nbr,                         /* TSMPERSN_IS_NUMBER */
        p_persn_org_id,                         /* TSMPERSN_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid,
        SYSDATE                                 /* D_LAST_UPDATE_TS */
        );

      COMMIT;

    END IF;
  CLOSE c_check_for_person;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add person to activity.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END add_person_to_activity;

/******************************************************************************************************/
FUNCTION add_cporg_to_activity(
         p_cporg_is_nbr IN NUMBER,
         p_cporg_org_id IN VARCHAR2,
         p_act_is_nbr   IN NUMBER, 
         p_act_org_id   IN VARCHAR2)
         RETURN NUMBER
IS

  CURSOR c_tsrfacoa(
         p_cporg_is_nbr IN NUMBER,
         p_cporg_org_id IN VARCHAR2,
         p_act_is_nbr   IN NUMBER,
         p_act_org_id   IN VARCHAR2) IS
  SELECT TSMCPORG_IS_NUMBER, TSMCPORG_ORG_ID, TSRFDACT_IS_NUMBER, TSRFDACT_ORG_ID
  FROM   TSRFACOA
  WHERE  TSMCPORG_IS_NUMBER = p_cporg_is_nbr 
  AND    TSMCPORG_ORG_ID    = p_cporg_org_id
  AND    TSRFDACT_IS_NUMBER = p_act_is_nbr
  AND    TSRFDACT_ORG_ID    = p_act_org_id;
  
  r_tsrfacoa     c_tsrfacoa%rowtype;

BEGIN

  -- try to retrieve the association
  OPEN c_tsrfacoa(p_cporg_is_nbr,p_cporg_org_id,p_act_is_nbr,p_act_org_id);
  FETCH c_tsrfacoa INTO r_tsrfacoa;
  
  -- Do we need to add the Cooperating Organization association?
  IF c_tsrfacoa%NOTFOUND THEN
    --  if yes

    -- Add Cooperating Organization association
    INSERT INTO TSRFACOA
    (
      TSMCPORG_IS_NUMBER,
      TSMCPORG_ORG_ID,
      TSRFDACT_IS_NUMBER,
      TSRFDACT_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS
    )
    VALUES
    (
      p_cporg_is_nbr,                  -- TSMCPORG_IS_NUMBER
      p_cporg_org_id,                  -- TSMCPORG_ORG_ID
      p_act_is_nbr,                    -- TSRFDACT_IS_NUMBER
      p_act_org_id,                    -- TSRFDACT_ORG_ID
      sim2_export_pkg.v_pkg_ownerid,   -- D_USERID_CODE
      SYSDATE                          -- D_LAST_UPDATE_TS
    );

    COMMIT;

  END IF;
  
  CLOSE c_tsrfacoa;

  -- return success
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    -- output error message
    sim2_export_pkg.WRITE_ERROR(
      v_export_log_seq, 
      SQLERRM ||' - in add Cooperating Organization to activity.'
    );
    
    v_success := 0;
    
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    
    -- return failure
    RETURN 0;

END add_cporg_to_activity;

/******************************************************************************************************/
FUNCTION add_person_to_trip_qc(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2,
                               p_trip_qc_is_nbr IN NUMBER, p_trip_qc_org_id IN VARCHAR2)
                               RETURN NUMBER IS

  v_persn_is_nbr             NUMBER(8);

  CURSOR c_check_for_person(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2,
                            p_trip_qc_is_nbr IN NUMBER, p_trip_qc_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSRFQSPA WHERE TSRFQS_IS_NUMBER = p_trip_qc_is_nbr
      AND TSMPERSN_IS_NUMBER = p_persn_is_nbr AND TSRFQS_ORG_ID = p_trip_qc_org_id
      AND TSMPERSN_ORG_ID = p_persn_org_id;

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_person(p_persn_is_nbr,p_persn_org_id,p_trip_qc_is_nbr,p_trip_qc_org_id);
  FETCH c_check_for_person INTO v_persn_is_nbr;
    IF c_check_for_person%NOTFOUND THEN

      /* add person to trip qc if not already there */
      INSERT INTO TSRFQSPA(
        TSRFQS_IS_NUMBER,
        TSRFQS_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_trip_qc_is_nbr,                       /* TSRFQS_IS_NUMBER */
        p_trip_qc_org_id,                       /* TSRFQS_ORG_ID */
        p_persn_is_nbr,                         /* TSMPERSN_IS_NUMBER */
        p_persn_org_id,                         /* TSMPERSN_ORG_ID */
        sim2_export_pkg.v_pkg_ownerid,
        SYSDATE                                 /* D_LAST_UPDATE_TS */
        );

      COMMIT;

    END IF;
  CLOSE c_check_for_person;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add person to trip qc.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END add_person_to_trip_qc;

/******************************************************************************************************/
FUNCTION add_field_set_to_activity(p_act_is_nbr IN NUMBER, p_act_org IN VARCHAR2,
                                   p_fld_set_is_nbr IN NUMBER, p_fld_set_org IN VARCHAR2)
                                   RETURN NUMBER IS

  v_fdset_is_nbr              NUMBER(8);

  CURSOR c_check_for_set(p_act_is_nbr IN NUMBER, p_act_org IN VARCHAR2,
                         p_fld_set_is_nbr IN NUMBER, p_fld_set_org IN VARCHAR2) IS
    SELECT TSRFDSET_IS_NUMBER FROM TSRFAFSA WHERE TSRFDACT_IS_NUMBER = p_act_is_nbr
      AND TSRFDSET_IS_NUMBER = p_fld_set_is_nbr AND TSRFDACT_ORG_ID = RPAD(p_act_org,8)
      AND TSRFDSET_ORG_ID = RPAD(p_fld_set_org,8);

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_set(p_act_is_nbr, p_act_org, p_fld_set_is_nbr, p_fld_set_org);
  FETCH c_check_for_set INTO v_fdset_is_nbr;
    IF c_check_for_set%NOTFOUND THEN

      /* if field set isn't on activity, add it */
      INSERT INTO TSRFAFSA (
        TSRFDACT_IS_NUMBER,
        TSRFDACT_ORG_ID,
        TSRFDSET_IS_NUMBER,
        TSRFDSET_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_act_is_nbr,
        p_act_org,
        p_fld_set_is_nbr,
        p_fld_set_org,
        sim2_export_pkg.v_pkg_ownerid,
        SYSDATE
        );

      COMMIT;

    END IF;
  CLOSE c_check_for_set;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add field set to activity.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END add_field_set_to_activity;

/******************************************************************************************************/
FUNCTION create_detection_record(p_detection_rec IN TSRDQL%ROWTYPE) RETURN NUMBER IS

BEGIN

  NULL;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create detection record.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_detection_record;

/******************************************************************************************************/
FUNCTION create_result(p_result_rec IN TSRRSULT%ROWTYPE, p_rs_seq IN NUMBER) RETURN NUMBER IS

BEGIN

  /* create result using info from result record */
  INSERT INTO TSRRSULT (
    TSRRSULT_IS_NUMBER,
    TSRRSULT_ORG_ID,
    D_ASSND_SEQ_NUM,
    COMP_IND_CD,
    LAB_CERT_IND_CODE,
    VALUE_TEXT,
    VALUE_MEASURE,
    VALUE_STATUS,
    PRECISION_AMT_TEXT,
    LAB_BATCH_ID_CODE,
    DILUTION_IND_CODE,
    RECOVERY_IND_CODE,
    CORRECTION_IND_CD,
    RSLT_TXT_IND_CD,
    CELL_DESC_IND_CD,
    BIAS,
    CONF_LVL_PCT_MSR,
    CONF_LVL_CORR_BIAS,
    PROC_EXCPTN_IND_CD,
    DUR_BASIS_TYPE_NM,
    DETECT_COND_CD,
    SMPL_FRAC_TYPE_NM,
    STATISTIC_TYPE_NM,
    VALUE_TYPE_NAME,
    WT_BASIS_TYPE_NM,
    TEMP_BASIS_LVL_NM,
    SPECIES_NUMBER,
    REF_PT_FROM_NAME,
    REF_PT_TO_NAME,
    LOWER_RANGE_VALUE,
    UPPER_RANGE_VALUE,
    TAXON_POLLUTION,
    FNCTIONAL_FEED_GRP,
    TROPHIC_LEVEL,
    ANALYSIS_DATE,
    ANALYSIS_TIME,
    ANALYSIS_TIME_ZONE,
    REPL_ANALYSIS_NUM,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TSRFQS_IS_NUMBER,
    TSRFQS_ORG_ID,
    TSRANLPR_IS_NUMBER,
    TSRANLPR_ORG_ID,
    TSRBRGI_IS_NUMBER,
    TSRBRGI_ORG_ID,
    TSRBRG_IS_NUMBER,
    TSRBRG_ORG_ID,
    TSRCHAR_IS_NUMBER,
    TSRCHAR_ORG_ID,
    TSRCHAR0IS_NUMBER,
    TSRCHAR0ORG_ID,
    TSRLAB_IS_NUMBER,
    TSRLAB_ORG_ID,
    TSRUOM_IS_NUMBER,
    TSRUOM_ORG_ID,
    TSRLSPP_IS_NUMBER,
    TSRLSPP_ORG_ID,
    TSRHCSC_IS_NUMBER,
    TSRHCSC_ORG_ID,
    TSMPRMVL_IS_NUMBER,
    TSMPRMVL_ORG_ID,
    TSMPRMVL0IS_NUMBER,
    TSMPRMVL0ORG_ID,
    TSMPRMVL1IS_NUMBER,
    TSMPRMVL1ORG_ID,
    TSRDLIN_IS_NUMBER,
    TSRDLIN_ORG_ID,
    BLOB_TYPE,
    BLOB_TITLE
    )
  VALUES (
    p_result_rec.TSRRSULT_IS_NUMBER,         /* TSRRSULT_IS_NUMBER */
    p_result_rec.TSRRSULT_ORG_ID,            /* TSRRSULT_ORG_ID */
    p_result_rec.D_ASSND_SEQ_NUM,            /* D_ASSND_SEQ_NUM */
    p_result_rec.COMP_IND_CD,                /* COMP_IND_CD */
    p_result_rec.LAB_CERT_IND_CODE,          /* LAB_CERT_IND_CODE */
    p_result_rec.VALUE_TEXT,                 /* VALUE_TEXT */
    p_result_rec.VALUE_MEASURE,              /* VALUE_MEASURE */
    p_result_rec.VALUE_STATUS,               /* VALUE_STATUS */
    p_result_rec.PRECISION_AMT_TEXT,         /* PRECISION_AMT_TEXT */
    p_result_rec.LAB_BATCH_ID_CODE,          /* LAB_BATCH_ID_CODE */
    p_result_rec.DILUTION_IND_CODE,          /* DILUTION_IND_CODE */
    p_result_rec.RECOVERY_IND_CODE,          /* RECOVERY_IND_CODE */
    p_result_rec.CORRECTION_IND_CD,          /* CORRECTION_IND_CD */
    p_result_rec.RSLT_TXT_IND_CD,            /* RSLT_TXT_IND_CD */
    p_result_rec.CELL_DESC_IND_CD,           /* CELL_DESC_IND_CD */
    p_result_rec.BIAS,
    p_result_rec.CONF_LVL_PCT_MSR,           /* CONF_LVL_PCT_MSR */
    p_result_rec.CONF_LVL_CORR_BIAS,
    p_result_rec.PROC_EXCPTN_IND_CD,         /* PROC_EXCPTN_IND_CD */
    p_result_rec.DUR_BASIS_TYPE_NM,          /* DUR_BASIS_TYPE_NM */
    p_result_rec.DETECT_COND_CD,             /* DETECT_COND_CD */
    p_result_rec.SMPL_FRAC_TYPE_NM,          /* SMPL_FRAC_TYPE_NM */
    p_result_rec.STATISTIC_TYPE_NM,          /* STATISTIC_TYPE_NM */
    p_result_rec.VALUE_TYPE_NAME,            /* VALUE_TYPE_NAME */
    p_result_rec.WT_BASIS_TYPE_NM,           /* WT_BASIS_TYPE_NM */
    p_result_rec.TEMP_BASIS_LVL_NM,          /* TEMP_BASIS_LVL_NM */
    p_result_rec.SPECIES_NUMBER,             /* SPECIES_NUMBER */
    p_result_rec.REF_PT_FROM_NAME,           /* REF_PT_FROM_NAME */
    p_result_rec.REF_PT_TO_NAME,             /* REF_PT_TO_NAME */
    p_result_rec.LOWER_RANGE_VALUE,          /* LOWER_RANGE_VALUE */
    p_result_rec.UPPER_RANGE_VALUE,          /* UPPER_RANGE_VALUE */
    p_result_rec.TAXON_POLLUTION,            /* TAXON_POLLUTION */
    p_result_rec.FNCTIONAL_FEED_GRP,         /* FNCTIONAL_FEED_GRP */
    p_result_rec.TROPHIC_LEVEL,              /* TROPHIC_LEVEL */
    p_result_rec.ANALYSIS_DATE,              /* ANALYSIS_DATE */
    p_result_rec.ANALYSIS_TIME,              /* ANALYSIS_TIME */
    p_result_rec.ANALYSIS_TIME_ZONE,         /* ANALYSIS_TIME_ZONE */
    p_result_rec.REPL_ANALYSIS_NUM,          /* REPL_ANALYSIS_NUM */
    p_result_rec.D_USERID_CODE,              /* D_USERID_CODE */
    p_result_rec.D_LAST_UPDATE_TS,           /* D_LAST_UPDATE_TS */
    p_result_rec.TSRFDACT_IS_NUMBER,         /* TSRFDACT_IS_NUMBER */
    p_result_rec.TSRFDACT_ORG_ID,            /* TSRFDACT_ORG_ID */
    p_result_rec.TSRFQS_IS_NUMBER,           /* TSRFQS_IS_NUMBER */
    p_result_rec.TSRFQS_ORG_ID,              /* TSRFQS_ORG_ID */
    p_result_rec.TSRANLPR_IS_NUMBER,         /* TSRANLPR_IS_NUMBER */
    p_result_rec.TSRANLPR_ORG_ID,            /* TSRANLPR_ORG_ID */
    p_result_rec.TSRBRGI_IS_NUMBER,          /* TSRBRGI_IS_NUMBER */
    p_result_rec.TSRBRGI_ORG_ID,             /* TSRBRGI_ORG_ID */
    p_result_rec.TSRBRG_IS_NUMBER,           /* TSRBRG_IS_NUMBER */
    p_result_rec.TSRBRG_ORG_ID,              /* TSRBRG_ORG_ID */
    p_result_rec.TSRCHAR_IS_NUMBER,          /* TSRCHAR_IS_NUMBER */
    p_result_rec.TSRCHAR_ORG_ID,             /* TSRCHAR_ORG_ID */
    p_result_rec.TSRCHAR0IS_NUMBER,          /* TSRCHAR0IS_NUMBER */
    p_result_rec.TSRCHAR0ORG_ID,             /* TSRCHAR0ORG_ID */
    p_result_rec.TSRLAB_IS_NUMBER,           /* TSRLAB_IS_NUMBER */
    p_result_rec.TSRLAB_ORG_ID,              /* TSRLAB_ORG_ID */
    p_result_rec.TSRUOM_IS_NUMBER,           /* TSRUOM_IS_NUMBER */
    p_result_rec.TSRUOM_ORG_ID,              /* TSRUOM_ORG_ID */
    p_result_rec.TSRLSPP_IS_NUMBER,          /* TSRLSPP_IS_NUMBER */
    p_result_rec.TSRLSPP_ORG_ID,             /* TSRLSPP_ORG_ID */
    p_result_rec.TSRHCSC_IS_NUMBER,          /* TSRHCSC_IS_NUMBER */
    p_result_rec.TSRHCSC_ORG_ID,              /* TSRHCSC_ORG_ID */
    p_result_rec.TSMPRMVL_IS_NUMBER,         --fraction
    p_result_rec.TSMPRMVL_ORG_ID,
    p_result_rec.TSMPRMVL0IS_NUMBER,         --habit
    p_result_rec.TSMPRMVL0ORG_ID,
    p_result_rec.TSMPRMVL1IS_NUMBER,         --voltinism
    p_result_rec.TSMPRMVL1ORG_ID,
    p_result_rec.TSRDLIN_IS_NUMBER,
    p_result_rec.TSRDLIN_ORG_ID,
    p_result_rec.BLOB_TYPE,                  /* BLOB_TYPE */
    p_result_rec.BLOB_TITLE                 /* BLOB_TITLE */
    );

    UPDATE SIM_RESULT_CLASS_INDICATOR
       SET RCI_TSRRSULT_IS_NUMBER = p_result_rec.tsrrsult_is_number,
           RCI_TSRRSULT_ORG_ID = p_result_rec.tsrrsult_org_id
     WHERE RCI_RS_SEQ = p_rs_seq;

    IF p_result_rec.TSRDLIN_IS_NUMBER IS NOT NULL THEN
      UPDATE SIM_RESULTS
       SET rs_tsrdlin_org_id = p_result_rec.TSRDLIN_ORG_ID,
           rs_tsrdlin_is_number = p_result_rec.TSRDLIN_IS_NUMBER
        WHERE rs_seq = p_rs_seq;
    END IF;        
        
  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create result.');
    v_success := 0;
    IF t_success > v_success THEN 
		  t_success := v_success;
		END IF;
    RETURN 0;

END create_result;

/******************************************************************************************************/
PROCEDURE export_activity (
          p_activity_seq IN NUMBER, 
          p_activity_rec IN OUT TSRFDACT%ROWTYPE,
          p_stvst_rec    IN OUT TSRSTVST%ROWTYPE, 
          p_import_seq   IN NUMBER,
          p_org_id       IN VARCHAR2,
          p_result_cnt   IN OUT NUMBER) IS

   v_time_zone                     VARCHAR2(3);
   v_activity_is_nbr               NUMBER(8);
   v_visit_is_number               NUMBER(8);
   v_field_set_is_number           NUMBER(8);
   v_trip_rec                      TSRTRIP%ROWTYPE;
   v_trap_net_rec                  SIM_TRAP_NET%ROWTYPE;
   v_tsrtnod_rec                   TSRTNOD%ROWTYPE;
   v_sample_rec                    TSRSMPLE%ROWTYPE;
   v_visit_id                      VARCHAR2(8);
   v_dummy_seq                     NUMBER(8);
   v_community_name                VARCHAR2(40);
   v_station_id                    VARCHAR2(15);
   v_success                       NUMBER(1);
   v_blob_success                  NUMBER(1);
   e_no_stvst                      EXCEPTION;
   e_missing_parent                EXCEPTION;

   CURSOR c_activity (p_org_id IN VARCHAR2, p_activity_seq IN NUMBER) IS
      SELECT
         FA_UPDEPTH, FA_LOWDEPTH, FA_DEPTH_RANGE_UOM, FA_SPECIES_NUMBER,
         FA_TYPE_NAME, FA_CATEGORY, FA_ACTIVITY_ID, FA_START_DATE, FA_START_TIME,
         FA_STOP_DATE, FA_STOP_TIME, FA_START_TIME_ZONE, FA_STOP_TIME_ZONE, FA_DEPTH, FA_DEPTH_REF_POINT,
         FA_THERMO, FA_HALO, FA_PYCNO, FA_DEPTH_ZONE,FA_CHAIN_OF_CUSTODY_ID,
         FA_QC_INDICATOR,FA_TSRMATRX_IS_NUMBER,FA_TSRMATRX_ORG_ID,
         FA_REL_DEPTH, FA_ACTIVITY_COMMENTS, FA_TSMSTATN_IS_NUMBER, FA_TSMSTATN_ORG_ID,
         FA_TSRTRIP_IS_NUMBER, FA_TSRTRIP_ORG_ID, FA_STP_SEQ, FA_TSRSTVST_IS_NUMBER,
         FA_TSRSTVST_ORG_ID, FA_SSV_SEQ, FA_TSRFLDPR_IS_NUMBER, FA_TSRFLDPR_ORG_ID,
         FA_TSRFLDGR_IS_NUMBER, FA_TSRFLDGR_ORG_ID, FA_FDSET_ID, FA_FDSET_NAME,
         FA_TSRLAB_IS_NUMBER, FA_TSRLAB_ORG_ID, FA_IMPORT_SEQ, FA_EXPORT_SEQ, FA_STATUS,
         FA_TSMORGAN_ORG_ID, FA_APPROVED_BY, FA_APPROVED_ON, FA_TSRGRCFG_IS_NUMBER,
         FA_TSRGRCFG_ORG_ID, FA_TSRSDP_IS_NUMBER, FA_TSRSDP_ORG_ID,
         TSRTRIP.ID_CODE TRIP_ID, FA_STVST_COMMENTS, FA_DEPTH_UOM, FA_MEDIUM, FA_REPLICATE_NBR,
         TSRFLDPR.ID_CODE PR_ID_CODE, TSRFLDGR.ID_CODE GR_ID_CODE,
         FA_INTENT, FA_TSMPRMVL_COMMUNITY_IS_NBR,
         FA_TSMPRMVL_COMMUNITY_ORG_ID, FA_TSRBIOPT_IS_NUMBER, FA_TSRBIOPT_ORG_ID,
         FA_TSRCHAR_IS_NUMBER, FA_TSRCHAR_ORG_ID, FA_DOCUMENT
      FROM
         SIM_FIELD_ACTIVITIES, TSRFLDPR, TSRFLDGR, TSRTRIP
      WHERE
         FA_TSRTRIP_IS_NUMBER = TSRTRIP_IS_NUMBER (+) AND
         RPAD(FA_TSRTRIP_ORG_ID,8) = TSRTRIP_ORG_ID (+) AND
         RPAD(FA_TSMORGAN_ORG_ID,8) = RPAD(p_org_id,8) AND
         FA_TSRFLDPR_IS_NUMBER = TSRFLDPR_IS_NUMBER(+) AND
         RPAD(FA_TSRFLDPR_ORG_ID,8) = TSRFLDPR_ORG_ID(+) AND
         FA_TSRFLDGR_IS_NUMBER = TSRFLDGR_IS_NUMBER(+) AND
         RPAD(FA_TSRFLDGR_ORG_ID,8) = TSRFLDGR_ORG_ID(+) AND
         FA_EXPORT_SEQ IS NULL AND
         FA_STATUS = 'A' AND
         FA_SEQ = p_activity_seq;

   CURSOR c_trap_net(p_activity_seq IN NUMBER) IS
      SELECT *
      FROM SIM_TRAP_NET
      WHERE STN_FA_SEQ = p_activity_seq;

   CURSOR c_get_station_id(p_org_id IN VARCHAR2, p_station_is_nbr IN NUMBER) IS
      SELECT IDENTIFICATION_CD FROM TSMSTATN WHERE TSMSTATN_ORG_ID = RPAD(p_org_id, 8)
        AND TSMSTATN_IS_NUMBER = p_station_is_nbr;

   CURSOR c_get_projects(p_activity_seq IN NUMBER) IS
     SELECT SAPJ_TSMPROJ_IS_NUMBER, SAPJ_TSMPROJ_ORG_ID FROM SIM_ACTIVITY_PROJECTS
       WHERE SAPJ_FA_SEQ = p_activity_seq;

   CURSOR c_get_personnel(p_activity_seq IN NUMBER) IS
     SELECT SAP_TSMPERSN_IS_NUMBER, SAP_TSMPERSN_ORG_ID FROM SIM_ACTIVITY_PERSONNEL
       WHERE SAP_FA_SEQ = p_activity_seq;

   CURSOR c_get_cporg(p_activity_seq IN NUMBER) IS
     SELECT SAC_TSMCPORG_IS_NUMBER, SAC_TSMCPORG_ORG_ID FROM SIM_ACTIVITY_CPORG
       WHERE SAC_FA_SEQ = p_activity_seq;

   CURSOR c_get_visit_info(p_visit_is_number IN NUMBER, p_visit_org_id IN VARCHAR2) IS
     SELECT TSRSTVST_IS_NUMBER FROM TSRSTVST T
       WHERE T.TSRSTVST_IS_NUMBER = p_visit_is_number
       AND   T.TSRSTVST_ORG_ID    = p_visit_org_id;

   CURSOR c_get_trip_info(p_trip_is_number IN NUMBER, p_trip_org_id IN VARCHAR2) IS
     SELECT TSRTRIP_IS_NUMBER FROM TSRTRIP T
       WHERE T.TSRTRIP_IS_NUMBER = p_trip_is_number
       AND   T.TSRTRIP_ORG_ID    = p_trip_org_id;
       
  -- jah
  CURSOR c_get_createdFromSample_parent (cp_fa_seq IN NUMBER) IS
  SELECT FAP_PARENT_IS_NUMBER, FAP_PARENT_ORG_ID, FA_IMPORT_SEQ
  FROM   SIM_FIELD_ACTIVITIES, SIM_FIELD_ACTIVITY_PARENT
  WHERE  FAP_FA_SEQ  = cp_fa_seq
  AND    FA_SEQ      = FAP_FA_SEQ
  AND    FA_CATEGORY = 'Created from Sample';

  -- jah
  CURSOR c_get_composWithParents_parent (p_fa_seq IN NUMBER) IS
  SELECT FAP_PARENT_IS_NUMBER, FAP_PARENT_ORG_ID
  FROM   SIM_FIELD_ACTIVITY_PARENT
  WHERE  FAP_FA_SEQ  = p_fa_seq;

  -- jah   
  r_get_createdFromSample_parent   c_get_createdFromSample_parent%ROWTYPE;
  r_get_composWithParents_parent   c_get_composWithParents_parent%ROWTYPE;

BEGIN

  FOR v_activity IN c_activity(p_org_id, p_activity_seq) LOOP

     /* get the allowable values */
     BEGIN

       IF v_activity.fa_tsmprmvl_community_is_nbr IS NOT NULL 
         AND v_activity.fa_tsmprmvl_community_org_id IS NOT NULL THEN
         v_community_name := get_tsmprmvl_value(v_activity.fa_tsmprmvl_community_is_nbr, v_activity.fa_tsmprmvl_community_org_id);
       ELSE
         v_community_name := NULL;
       END IF;

       IF v_activity.fa_tsrsdp_is_number IS NOT NULL THEN
         v_sample_rec := get_sample_pres_info(v_activity.fa_tsrsdp_is_number, v_activity.fa_tsrsdp_org_id,
                         v_activity.fa_activity_id);
       ELSE
         v_sample_rec.tsmprmvl_is_number := NULL;
         v_sample_rec.tsmprmvl_org_id := NULL;
         v_sample_rec.tsmprmvl0is_number := NULL;
         v_sample_rec.tsmprmvl0org_id := NULL;
         v_sample_rec.tsmprmvl1is_number := NULL;
         v_sample_rec.tsmprmvl1org_id := NULL;
         v_sample_rec.container_size_msr := NULL;
         v_sample_rec.container_size_un := NULL;
         v_sample_rec.presrv_strge_prcdr := NULL;
       END IF;
     END;

     /* Verify that trip still exists in STORET */
     IF v_activity.FA_TSRTRIP_IS_NUMBER IS NOT NULL THEN

         OPEN c_get_trip_info(v_activity.FA_TSRTRIP_IS_NUMBER, v_activity.FA_TSRTRIP_ORG_ID);
         FETCH c_get_trip_info INTO v_dummy_seq;

           /* If trip does not still exist, clear is_number and let SIM recreate it. */
           IF c_get_trip_info%NOTFOUND THEN
             v_activity.FA_TSRTRIP_IS_NUMBER := NULL;
             v_activity.FA_TSRTRIP_ORG_ID    := NULL;
           END IF;

         CLOSE c_get_trip_info;

     END IF;

     /* kms - if the trip does not already exist in STORET, get info needed to create it */
     IF v_activity.FA_TSRTRIP_IS_NUMBER IS NULL THEN
       get_trip_info(v_activity.FA_STP_SEQ,v_trip_rec);

       /* kms - call function to get next trip is number */
       v_activity.FA_TSRTRIP_IS_NUMBER := get_next_trip_is_nbr(p_org_id);

       v_trip_rec.TSRTRIP_IS_NUMBER := v_activity.FA_TSRTRIP_IS_NUMBER;
       v_trip_rec.TSRTRIP_ORG_ID := p_org_id;

       IF v_trip_rec.start_time IS NULL THEN
         v_trip_rec.start_time := TO_DATE('01-JAN-0001','DD-MON-YYYY'); --Default null value
       END IF;

       v_trip_rec.START_TIME_ZONE :=  NULL;   --v_time_zone; we don't import trip start time zone
       IF v_trip_rec.END_DATE IS NULL THEN
         v_trip_rec.END_DATE := TO_DATE('01-JAN-0001','DD-MON-YYYY'); --Default null value
       END IF;
       IF v_trip_rec.END_TIME IS NULL THEN
         v_trip_rec.end_time := TO_DATE('01-JAN-0001','DD-MON-YYYY'); --Default null value
       END IF;
       v_trip_rec.END_TIME_ZONE := NULL;      --v_time_zone;
       v_trip_rec.SINGLE_PROJ_IND := 'Y';
       v_trip_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
       v_trip_rec.D_LAST_UPDATE_TS := SYSDATE;

       /* kms - create the trip in STORET */
       v_success := create_trip(v_trip_rec);

       IF v_activity.FA_TSRTRIP_IS_NUMBER IS NOT NULL THEN
         v_activity.FA_TSRTRIP_ORG_ID := p_org_id;

         /* kms (6/6/01) - fill in the trip fk columns in the sim_trips record
            and in every activity with this trip. */
         UPDATE SIM_TRIPS
           SET  STP_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
                STP_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE STP_SEQ = v_activity.FA_STP_SEQ;

         UPDATE SIM_FIELD_ACTIVITIES
           SET FA_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
               FA_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE FA_STP_SEQ =  v_activity.FA_STP_SEQ;

         COMMIT;
       END IF;
     END IF;

     OPEN c_get_station_id (p_org_id, v_activity.fa_tsmstatn_is_number);
     FETCH c_get_station_id INTO v_station_id;
     CLOSE c_get_station_id;

     /* call function to get the next activity is number */
     v_activity_is_nbr := get_next_activity_is_nbr(p_org_id);

     UPDATE SIM_FIELD_ACTIVITIES
        SET FA_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
            FA_TSRFDACT_ORG_ID = p_org_id
      WHERE FA_SEQ =  p_activity_seq;

    UPDATE SIM_ACTUAL_ACTIVITY_LOC
       SET AAL_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
           AAL_TSRFDACT_ORG_ID = p_org_id
     WHERE AAL_FA_SEQ = p_activity_seq;

     UPDATE SIM_TRAWL_OPS_DET
        SET TOD_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
            TOD_TSRFDACT_ORG_ID = p_org_id
      WHERE TOD_FA_SEQ = p_activity_seq;

     UPDATE SIM_SAMPLE
        SET SS_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
            SS_TSRFDACT_ORG_ID = p_org_id
      WHERE SS_FA_SEQ = p_activity_seq;

     UPDATE SIM_TRAP_NET
        SET STN_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
            STN_TSRFDACT_ORG_ID = p_org_id
      WHERE STN_FA_SEQ = p_activity_seq;
          
     /* write the 'actual activity location' TSRAAL records */
     BEGIN
       v_success := create_activity_location(
         p_org_id,
         v_activity_is_nbr,
         p_activity_seq,
         v_activity.FA_TSMSTATN_IS_NUMBER,
         v_activity.FA_TSMSTATN_ORG_ID
       );

     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - Writing tsraal record.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* write the 'trawl ops det' TSRTOD record */
     BEGIN
       v_success := create_tsrtod_record(p_org_id,v_activity_is_nbr,p_activity_seq);
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - Writing tsrtod record.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
 		     END IF;
     END;

     /* write the 'sim_sample' TSRSMPLE record */
     BEGIN
       v_success := create_tsrsmple_record(p_org_id,v_activity_is_nbr,p_activity_seq,v_sample_rec);
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - Writing tsrsmple record.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* write the 'bio results grps' TSRBRG records */
     BEGIN
       v_success := create_tsrbrg_records(p_org_id,v_activity_is_nbr,p_activity_seq);
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - Writing tsrbrg record.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* write the 'bio results grp individual' TSRBRGI records */
     BEGIN
       v_success := create_tsrbrgi_records(p_org_id,v_activity_is_nbr,p_activity_seq);
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - Writing tsrbrgi record.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* kms (6/4/01) - if station visit does not already exist, move what's
        in SIM into a new station visit record in STORET. */
     BEGIN
         
       /* Verify that visit still exists in STORET */
       
       -- Do we have a Station Visit IS_NUMBER in SIM_FIELD_ACTIVITIES record?
       IF v_activity.FA_TSRSTVST_IS_NUMBER IS NOT NULL THEN
         -- if yes

         -- Try to retrieve the Station Visit record from STORET
         OPEN c_get_visit_info(v_activity.FA_TSRSTVST_IS_NUMBER, v_activity.FA_TSRSTVST_ORG_ID);
         FETCH c_get_visit_info INTO v_dummy_seq;

         /* If visit does not still exist, clear is_number and let SIM recreate it. */
         
         -- Does the Station Visit record still exist in STORET?
         IF c_get_visit_info%NOTFOUND THEN
           -- if no
           -- Tell subsequent processing to create a Station Visit Record in STORET for this activity
           v_activity.FA_TSRSTVST_IS_NUMBER := NULL;
           v_activity.FA_TSRSTVST_ORG_ID    := NULL;
         END IF;

         CLOSE c_get_visit_info;

       END IF;
         
       -- Do we need to create a Station Visit record in STORET for this activity?
       IF v_activity.FA_TSRSTVST_IS_NUMBER IS NULL THEN
         -- if yes
         
         /* kms - get the station visit info from SIM_STATION_VISITS */
         get_visit_info(v_activity.FA_SSV_SEQ,p_stvst_rec,v_visit_id);

         /* kms (6/8/01) - if the station visit id has not yet been assigned, use the
            max id nbr used in STORET and increment it by 1. */
         IF v_visit_id LIKE 'UNA%' THEN
           p_stvst_rec.ID_NUMBER := get_next_visit_id(
             v_activity.FA_STP_SEQ,
             v_activity.FA_TSRTRIP_IS_NUMBER,
             v_activity.FA_TSRTRIP_ORG_ID,
             v_activity.FA_TSMSTATN_IS_NUMBER,
             v_activity.FA_TSMSTATN_ORG_ID
           );
         ELSE
           p_stvst_rec.ID_NUMBER := v_visit_id;
         END IF;

         /* kms - get the next station visit is number */
         v_visit_is_number := get_next_visit_is_number(p_org_id);
         p_stvst_rec.tsrstvst_is_number := v_visit_is_number;
         p_stvst_rec.tsrstvst_org_id := RPAD(p_org_id,8);

         /* kms - set remaining variables for station visit fields and create station visit in STORET.
       Insert NULL defaults if NULL */
         IF p_stvst_rec.ARRIVAL_TIME IS NULL THEN
           p_stvst_rec.ARRIVAL_TIME := TO_DATE('01-JAN-0001','DD-MON-YYYY'); --Default null value
           p_stvst_rec.ARRIVAL_TIME_ZONE := NULL;
         END IF;
         IF p_stvst_rec.DEPARTURE_DATE IS NULL THEN
           p_stvst_rec.DEPARTURE_DATE := TO_DATE('01-JAN-0001','DD-MON-YYYY'); --Default null value
         END IF;
         IF p_stvst_rec.DEPARTURE_TIME IS NULL THEN
           p_stvst_rec.DEPARTURE_TIME := TO_DATE('01-JAN-0001','DD-MON-YYYY'); --Default null value
           p_stvst_rec.DEPRTURE_TIME_ZONE := NULL;
         END IF;

         p_stvst_rec.COMMENT_TEXT := v_activity.fa_stvst_comments;
         p_stvst_rec.TSRTRIP_IS_NUMBER := v_activity.fa_tsrtrip_is_number;
         p_stvst_rec.TSRTRIP_ORG_ID := v_activity.fa_tsrtrip_org_id;
         p_stvst_rec.TSMSTATN_IS_NUMBER := v_activity.fa_tsmstatn_is_number;
         p_stvst_rec.TSMSTATN_ORG_ID := v_activity.fa_tsmstatn_org_id;

         IF p_stvst_rec.ID_NUMBER IS NOT NULL THEN
           v_success := create_station_visit(p_org_id,p_stvst_rec);
         ELSE
           sim2_export_pkg.write_error(v_export_log_seq, 'Did not get a station visit id number, unable to create station visit.');
           RAISE e_no_stvst;
         END IF;
           
         IF p_stvst_rec.tsrstvst_is_number IS NOT NULL THEN

            /* export binary document for visit */
            IF p_stvst_rec.blob_title IS NOT NULL THEN
              v_blob_success := insert_visit_binary_document(v_export_log_seq, p_org_id, v_visit_is_number, p_stvst_rec.blob_title||'.'||p_stvst_rec.blob_type);
              IF v_blob_success = 0 THEN
                  v_success := 0;
                  IF t_success > v_success THEN 
              		  t_success := v_success;
          		   END IF;
              END IF;
            END IF;

           /* kms (6/6/01) - fill in the visit fk columns in the sim_trips record
              and in every activity with visit trip. */

           v_activity.FA_TSRSTVST_IS_NUMBER := p_stvst_rec.tsrstvst_is_number;
           v_activity.FA_TSRSTVST_ORG_ID := p_stvst_rec.tsrstvst_org_id;

           UPDATE SIM_STATION_VISITS
             SET  SSV_TSRSTVST_IS_NUMBER = v_activity.FA_TSRSTVST_IS_NUMBER,
                  SSV_TSRSTVST_ORG_ID = v_activity.FA_TSRSTVST_ORG_ID,
                  SSV_ID = RTRIM(p_stvst_rec.ID_NUMBER)
             WHERE SSV_SEQ = v_activity.FA_SSV_SEQ;

           UPDATE SIM_FIELD_ACTIVITIES
             SET FA_TSRSTVST_IS_NUMBER = v_activity.FA_TSRSTVST_IS_NUMBER,
                 FA_TSRSTVST_ORG_ID = v_activity.FA_TSRSTVST_ORG_ID
             WHERE FA_SSV_SEQ =  v_activity.FA_SSV_SEQ;

           COMMIT;
         ELSE
           RAISE e_no_stvst;
         END IF;

       ELSE

         v_visit_is_number := v_activity.FA_TSRSTVST_IS_NUMBER;
         
         -- jah 3-26-2004
         -- pass the station visit is number back to the calling routine
         p_stvst_rec.tsrstvst_is_number := v_visit_is_number;

       END IF;

     EXCEPTION
       WHEN e_no_stvst THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Did not find or create station visit');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;

       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing visit record for station');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* Add record to tsrtsa to associate station with trip  */

     BEGIN
       add_station_to_trip(v_activity.FA_TSRTRIP_IS_NUMBER,v_activity.FA_TSMSTATN_IS_NUMBER,p_org_id);
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing station visited on trip record: '||v_activity.FA_TSMSTATN_IS_NUMBER);
     END;

       /* Issue other STORET NULL Value defaults. */
      --FOR TSRFDACT
      IF v_activity.fa_replicate_nbr IS NULL THEN
        v_activity.fa_replicate_nbr := 0;
      END IF;
      IF v_activity.fa_stop_date IS NULL THEN
        v_activity.fa_stop_date := TO_DATE('01-JAN-0001','DD-MON-YYYY');
      END IF;
      IF v_activity.fa_start_time IS NULL THEN
        v_activity.fa_start_time := TO_DATE('01-JAN-0001','DD-MON-YYYY');
      END IF;
      IF v_activity.fa_stop_time IS NULL THEN
        v_activity.fa_stop_time := TO_DATE('01-JAN-0001','DD-MON-YYYY');
      END IF;


     BEGIN
        /* write the activity record */

        p_activity_rec.TSRFDACT_IS_NUMBER := v_activity_is_nbr;
        p_activity_rec.TSRFDACT_ORG_ID := RPAD(p_org_id,8);
        p_activity_rec.CATEGORY_TYPE_NAME := v_activity.fa_category;
        p_activity_rec.ID_CODE := RPAD(v_activity.fa_activity_id,12);
        p_activity_rec.REPLICATE_NUMBER := v_activity.fa_replicate_nbr;
        p_activity_rec.MEDIUM_TYPE_NAME := v_activity.fa_medium;
        p_activity_rec.INTENT_TYPE_NAME := v_activity.fa_intent;
        p_activity_rec.COMMUNITY_NAME := v_community_name;
        p_activity_rec.START_DATE := TRUNC(v_activity.fa_start_date);
        p_activity_rec.START_TIME := v_activity.fa_start_time;
        p_activity_rec.START_TIME_ZONE := v_activity.fa_start_time_zone;
        p_activity_rec.STOP_DATE := v_activity.fa_stop_date;
        p_activity_rec.STOP_TIME := v_activity.fa_stop_time;
        p_activity_rec.STOP_TIME_ZONE := v_activity.fa_stop_time_zone;
        p_activity_rec.COMMENT_TEXT := v_activity.fa_activity_comments;
        p_activity_rec.ZONE_TYPE_NAME := v_activity.fa_depth_zone;
        p_activity_rec.RELTV_DEPTH_NAME := RPAD(v_activity.fa_rel_depth,15);
        p_activity_rec.DEPTH_REF_POINT := v_activity.fa_depth_ref_point;
        p_activity_rec.DEPTH_TO_ACTIVITY := v_activity.fa_depth;
        p_activity_rec.DEPTH_TO_ACT_UN_CD := RPAD(v_activity.fa_depth_uom,3);
        p_activity_rec.UPPER_DEPTH_TO_ACT := v_activity.fa_updepth;
        p_activity_rec.DEPTH_MSR_UNT_CD := RPAD(v_activity.fa_depth_range_uom,3);
        p_activity_rec.LOWER_DEPTH_TO_ACT := v_activity.fa_lowdepth;
        p_activity_rec.THERMOCLINE_LOC_CD := v_activity.fa_thermo;
        p_activity_rec.PYCNOCLINE_LOC_CD := v_activity.fa_pycno;
        p_activity_rec.HALOCLINE_LOC_CD := v_activity.fa_halo;
        p_activity_rec.SPECIES_NUMBER := v_activity.fa_species_number;
        p_activity_rec.QC_INDICATOR := v_activity.fa_qc_indicator;
        p_activity_rec.CHAIN_OF_CUSTODY_ID := v_activity.fa_chain_of_custody_id;
        p_activity_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
        p_activity_rec.D_LAST_UPDATE_TS := SYSDATE;
        p_activity_rec.TSROPPRD_ORG_ID := NULL;
        p_activity_rec.TSROPPRD_IS_NUMBER := NULL;
        p_activity_rec.TSRFDACT0IS_NUMBER := NULL;
        p_activity_rec.TSRFDACT0ORG_ID := NULL;

        -- jah        
        --        p_activity_rec.TSRFDACT1IS_NUMBER := v_activity.fa_tsrfdact1is_number;
        --        p_activity_rec.TSRFDACT1ORG_ID := RPAD(v_activity.tsrfdact1org_id,8);
        -- Does this activity have a Parent Field Activity?
        IF (v_activity.FA_CATEGORY = 'Created from Sample') THEN
          -- if yes
          -- Retrieve the parent information
          OPEN  c_get_createdFromSample_parent (p_activity_seq);
          FETCH c_get_createdFromSample_parent INTO r_get_createdFromSample_parent;
          CLOSE c_get_createdFromSample_parent;
          
          -- is the parent already in STORET?
          -- Set the createdFromSample key
          p_activity_rec.TSRFDACT1IS_NUMBER := r_get_createdFromSample_parent.FAP_PARENT_IS_NUMBER;
          p_activity_rec.TSRFDACT1ORG_ID := RPAD(r_get_createdFromSample_parent.FAP_PARENT_ORG_ID,8);
        ELSE
          -- if no (this activity does not require a Parent Sample ID)
          -- set the createdFromSample key to NULL
          p_activity_rec.TSRFDACT1IS_NUMBER := NULL;
          p_activity_rec.TSRFDACT1ORG_ID := NULL;
        END IF;

        p_activity_rec.TSRSTVST_IS_NUMBER := v_visit_is_number;
        p_activity_rec.TSRSTVST_ORG_ID := RPAD(p_org_id,8);
        p_activity_rec.TSRBIOPT_IS_NUMBER := v_activity.fa_tsrbiopt_is_number;
        p_activity_rec.TSRBIOPT_ORG_ID := v_activity.fa_tsrbiopt_org_id;
        p_activity_rec.TSRCHAR_IS_NUMBER := v_activity.fa_tsrchar_is_number;
        p_activity_rec.TSRCHAR_ORG_ID := v_activity.fa_tsrchar_org_id;
        p_activity_rec.TSRFLDPR_IS_NUMBER := v_activity.FA_TSRFLDPR_IS_NUMBER;
        p_activity_rec.TSRFLDPR_ORG_ID := v_activity.FA_TSRFLDPR_ORG_ID;
        p_activity_rec.TSRFLDGR_IS_NUMBER := v_activity.FA_TSRFLDGR_IS_NUMBER;
        p_activity_rec.TSRFLDGR_ORG_ID := v_activity.FA_TSRFLDGR_ORG_ID;
        p_activity_rec.TSRGRCFG_IS_NUMBER := v_activity.FA_TSRGRCFG_IS_NUMBER;
        p_activity_rec.TSRGRCFG_ORG_ID := v_activity.FA_TSRGRCFG_ORG_ID;
        p_activity_rec.TSRMATRX_IS_NUMBER := v_activity.FA_TSRMATRX_IS_NUMBER;
        p_activity_rec.TSRMATRX_ORG_ID := v_activity.FA_TSRMATRX_ORG_ID;
        p_activity_rec.BLOB_TITLE := sim2_export_pkg.get_file_name(v_activity.FA_DOCUMENT);
        p_activity_rec.BLOB_TYPE := sim2_export_pkg.get_file_type(v_activity.FA_DOCUMENT);

        IF v_visit_is_number IS NOT NULL THEN
          v_success := create_activity(p_activity_rec);
        END IF;

        -- Did the Field Activity get inserted into TSRFDACT?        
        IF v_success = 1 THEN
          -- if yes
          
          /* 
            jah 2-13-04
            if p_activity_seq is in SIM_FIELD_ACTIVITY_PARENT, then this record is a 
            parent to some activity that occurs later in the import. So to prepare
            for inserting the Child, we find the child record in SIM_FIELD_ACTIVITY_PARENT
            and update the parent IS_NUMBER and ORG_ID.
          */
          -- Put the parent IS_NUMBER and the ORG_ID into SIM_FIELD_ACTIVITY_PARENT.
          UPDATE SIM_FIELD_ACTIVITY_PARENT
          SET    FAP_PARENT_IS_NUMBER = v_activity_is_nbr,
                 FAP_PARENT_ORG_ID    = RPAD(p_org_id,8)
          WHERE  FAP_PARENT_FA_SEQ    = p_activity_seq;
          COMMIT;
          
          -- jah 2-13-04
          -- Do we need to update a Parent Field Activity when we insert this Field Activity?
          IF (v_activity.FA_CATEGORY = 'Composite-with Parents') THEN
            -- if yes
  
            -- For each parent update the reference to this 'Composite...' child
            FOR r_get_composWithParents_parent IN c_get_composWithParents_parent (p_activity_seq) LOOP
  
              -- Is the parent in STORET?
              IF (r_get_composWithParents_parent.FAP_PARENT_IS_NUMBER IS NOT NULL) THEN
                -- if yes
                -- Update the 'Composite-with Parent' key in the parent activity
                UPDATE TSRFDACT
                SET
                  TSRFDACT0IS_NUMBER = p_activity_rec.TSRFDACT_IS_NUMBER,
                  TSRFDACT0ORG_ID = RPAD(p_activity_rec.TSRFDACT_ORG_ID,8)
                WHERE TSRFDACT_IS_NUMBER = r_get_composWithParents_parent.FAP_PARENT_IS_NUMBER
                AND   TSRFDACT_ORG_ID = RPAD(r_get_composWithParents_parent.FAP_PARENT_ORG_ID,8);
              ELSE
                RAISE e_missing_parent;
              END IF;
            END LOOP;
  
          END IF;
                
          
          /* export binary document for activity */
          IF v_activity.fa_document IS NOT NULL THEN
            v_blob_success := insert_act_binary_document(v_export_log_seq, p_org_id, v_activity_is_nbr, v_activity.fa_document);
            IF v_blob_success = 0 THEN
                v_success := 0;
                IF t_success > v_success THEN 
            		  t_success := v_success;
        		   END IF;
            END IF;
          END IF;

          OPEN c_trap_net(p_activity_seq);
            FETCH c_trap_net INTO v_trap_net_rec;
            IF c_trap_net%FOUND THEN

           --Default NULL values For TSRTNOD
            IF v_trap_net_rec.STN_REL_CURRENT IS NULL THEN
              v_trap_net_rec.STN_REL_CURRENT := 0;
            END IF;
            IF v_trap_net_rec.STN_REL_WIND IS NULL THEN
              v_trap_net_rec.STN_REL_WIND := 0;
            END IF;

              v_tsrtnod_rec.TSRFDACT_IS_NUMBER := v_trap_net_rec.STN_TSRFDACT_IS_NUMBER;
              v_tsrtnod_rec.TSRFDACT_ORG_ID := v_trap_net_rec.STN_TSRFDACT_ORG_ID;
              v_tsrtnod_rec.SMPLNG_DURATN_MSR := v_trap_net_rec.STN_SAMPLING_DUR;
              v_tsrtnod_rec.SMPLNG_DRTN_UNT_CD := v_trap_net_rec.STN_SAMPLING_DUR_UNIT;
              v_tsrtnod_rec.ORIENTN_TO_CURRENT := v_trap_net_rec.STN_ORIENT_TO_CUR;
              v_tsrtnod_rec.REL_CURRENT_DIR := v_trap_net_rec.STN_REL_CURRENT;
              v_tsrtnod_rec.REL_WIND_DIR := v_trap_net_rec.STN_REL_WIND;
              v_tsrtnod_rec.COMMENT_TEXT := v_trap_net_rec.STN_COMMENT;
              v_tsrtnod_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
              v_tsrtnod_rec.D_LAST_UPDATE_TS := SYSDATE;

              v_success := create_trap_net(v_tsrtnod_rec);
            END IF;
          CLOSE c_trap_net;
        END IF;

     EXCEPTION
        WHEN e_missing_parent THEN
          sim2_export_pkg.WRITE_ERROR(
            v_export_log_seq,
            'Error: Field Activity ' || v_activity.fa_activity_id || ' did not migrate, ' ||
            'because it requires a Parent Field Activity in STORET.'
          );

        WHEN OTHERS THEN
           sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing activity ');
           v_success := 0;
           IF t_success > v_success THEN 
		         t_success := v_success;
		       END IF;
     END;

     /* loop through projects, see if project is on the trip and activity, and add if it's not */
     BEGIN
       FOR v_project IN c_get_projects(p_activity_seq) LOOP
         add_project_to_trip(v_activity.FA_TSRTRIP_IS_NUMBER,v_project.SAPJ_TSMPROJ_IS_NUMBER,p_org_id);
         add_project_to_activity(v_activity_is_nbr,v_project.SAPJ_TSMPROJ_IS_NUMBER,p_org_id);

         /* make sure station is on project, add if it isn't */
         v_success:= add_station_to_project(v_project.SAPJ_TSMPROJ_IS_NUMBER,
                                            v_activity.FA_TSMSTATN_IS_NUMBER,p_org_id);

         /* loop through personnel, assign to role and project*/
         BEGIN
           FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

             v_success := add_person_to_project(
                       v_personnel.sap_tsmpersn_is_number,
                       RPAD(v_personnel.sap_tsmpersn_org_id,8),
                       v_project.SAPJ_TSMPROJ_IS_NUMBER,
                       RPAD(p_org_id,8));

           END LOOP;

         EXCEPTION
           WHEN OTHERS THEN
             sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to project.');
             v_success := 0;
             IF t_success > v_success THEN 
		           t_success := v_success;
		         END IF;
         END;

         /* loop through cooperating organizations and assign to project*/
         BEGIN
           FOR v_cporg IN c_get_cporg(p_activity_seq) LOOP

             v_success := add_cporg_to_project(
               v_cporg.sac_tsmcporg_is_number,
               RPAD(v_cporg.sac_tsmcporg_org_id,8),
               v_project.SAPJ_TSMPROJ_IS_NUMBER,
               RPAD(p_org_id,8)
             );

           END LOOP;

         EXCEPTION
           WHEN OTHERS THEN
             sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to project.');
             v_success := 0;
             IF t_success > v_success THEN 
		           t_success := v_success;
		         END IF;
         END;

       END LOOP;
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding project to trip. Trip_is_nbr: '||v_activity.FA_TSRTRIP_IS_NUMBER);
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;


     /* loop through personnel and add to the activity */
     BEGIN
       FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

         v_success := add_person_to_activity(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                             v_activity_is_nbr, RPAD(p_org_id,8));

       END LOOP;

     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to activity.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* loop through cooperating organizations and add to the activity */
     BEGIN
       FOR v_cporg IN c_get_cporg(p_activity_seq) LOOP

         v_success := add_cporg_to_activity(
           v_cporg.sac_tsmcporg_is_number,
           RPAD(v_cporg.sac_tsmcporg_org_id,8),
           v_activity_is_nbr, 
           RPAD(p_org_id,8)
         );

       END LOOP;

     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(
           v_export_log_seq, 
           SQLERRM ||' - adding cooperating organization to activity.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* get the field set is number from STORET if exists or create if not */
     IF v_activity.fa_fdset_id IS NOT NULL THEN
       v_field_set_is_number := GET_FIELD_SET_IS_NBR(P_org_id, v_visit_is_number,
                                                     v_activity.fa_fdset_id, v_activity.fa_fdset_name);

       BEGIN
         /* associate activity with field set */
         v_success := add_field_set_to_activity(v_activity_is_nbr,RPAD(p_org_id,8),v_field_set_is_number,
                                              RPAD(p_org_id,8));
       EXCEPTION
         WHEN OTHERS THEN
           sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding field set to activity.');
           v_success := 0;
           IF t_success > v_success THEN 
	  	       t_success := v_success;
		       END IF;
       END;
     ELSE
       v_field_set_is_number := NULL;
     END IF;

     /* IF QC RESULTS, THERE WON'T BE AN ACTIVITY IS NBR ... */
     BEGIN
    
       EXPORT_RESULTS(
         p_activity_seq, 
         p_activity_rec.type_name, 
         v_activity_is_nbr,
         p_org_id,
         null,
         null,
         v_activity.fa_tsrlab_is_number, 
         v_activity.fa_tsrlab_org_id,
         p_org_id, 
         v_export_log_seq, 
         p_result_cnt, 
         v_time_zone
       );
                       
     END;
  END LOOP;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in export activity.');
    v_success := 0;
    IF t_success > v_success THEN 
      t_success := v_success;
    END IF;

END export_activity;
/******************************************************************************************************/
PROCEDURE export_trip_qc (p_activity_seq IN NUMBER, p_trip_qc_rec IN OUT TSRFQS%ROWTYPE,
                          p_import_seq IN NUMBER, p_org_id IN VARCHAR2,p_result_cnt IN OUT NUMBER) IS


   v_time_zone                     VARCHAR2(3);
   v_trip_qc_is_nbr                NUMBER(8);
   v_trip_rec                      TSRTRIP%ROWTYPE;
   v_trip_copy_rec                 TSRTRIP%ROWTYPE;

   v_success                       NUMBER(1);

   CURSOR c_activity (p_org_id IN VARCHAR2, p_activity_seq IN NUMBER) IS
      SELECT
         FA_CATEGORY, FA_ACTIVITY_ID,
         FA_START_DATE, FA_START_TIME, FA_START_TIME_ZONE, FA_STOP_TIME_ZONE, FA_ACTIVITY_COMMENTS,
         FA_TSRTRIP_IS_NUMBER, FA_TSRTRIP_ORG_ID,
         FA_TSRLAB_IS_NUMBER, FA_TSRLAB_ORG_ID, FA_IMPORT_SEQ, FA_EXPORT_SEQ, FA_STATUS,
         FA_TSMORGAN_ORG_ID, FA_APPROVED_BY, FA_APPROVED_ON, FA_MEDIUM, FA_STP_SEQ
      FROM
         SIM_FIELD_ACTIVITIES, TSRTRIP
      WHERE
         FA_TSRTRIP_IS_NUMBER = TSRTRIP_IS_NUMBER (+) AND
         RPAD(FA_TSRTRIP_ORG_ID,8) = TSRTRIP_ORG_ID (+) AND
         RPAD(FA_TSMORGAN_ORG_ID,8) = RPAD(p_org_id,8) AND
         FA_EXPORT_SEQ IS NULL AND
         FA_STATUS = 'A' AND
         FA_SEQ = p_activity_seq;

   CURSOR c_get_projects(p_activity_seq IN NUMBER) IS
     SELECT SAPJ_TSMPROJ_IS_NUMBER, SAPJ_TSMPROJ_ORG_ID FROM SIM_ACTIVITY_PROJECTS
       WHERE SAPJ_FA_SEQ = p_activity_seq;

   CURSOR c_get_personnel(p_activity_seq IN NUMBER) IS
     SELECT SAP_TSMPERSN_IS_NUMBER, SAP_TSMPERSN_ORG_ID FROM SIM_ACTIVITY_PERSONNEL
       WHERE SAP_FA_SEQ = p_activity_seq;

BEGIN

  FOR v_activity IN c_activity(p_org_id, p_activity_seq) LOOP
     /* kms - if the trip does not already exist in STORET, get info needed to create it */
     IF v_activity.FA_TSRTRIP_IS_NUMBER IS NULL THEN
       get_trip_info(v_activity.FA_STP_SEQ,v_trip_rec);

       /* kms - call function to get next trip is number */
       v_activity.FA_TSRTRIP_IS_NUMBER := get_next_trip_is_nbr(p_org_id);

       v_trip_rec.TSRTRIP_IS_NUMBER := v_activity.FA_TSRTRIP_IS_NUMBER;
       v_trip_rec.TSRTRIP_ORG_ID := p_org_id;
       v_trip_rec.START_TIME_ZONE := NULL;   --v_time_zone;
       v_trip_rec.END_TIME_ZONE := NULL;     --v_time_zone;
       v_trip_rec.SINGLE_PROJ_IND := 'Y';
       v_trip_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
       v_trip_rec.D_LAST_UPDATE_TS := SYSDATE;

       /* kms - create the trip in STORET */
       v_success := create_trip(v_trip_rec);

       IF v_activity.FA_TSRTRIP_IS_NUMBER IS NOT NULL THEN
         v_activity.FA_TSRTRIP_ORG_ID := p_org_id;

         /* kms (6/6/01) - fill in the trip fk columns in the sim_trips record
            and in every activity with this trip. */
         UPDATE SIM_TRIPS
           SET  STP_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
                STP_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE STP_SEQ = v_activity.FA_STP_SEQ;

         UPDATE SIM_FIELD_ACTIVITIES
           SET FA_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
               FA_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE FA_STP_SEQ =  v_activity.FA_STP_SEQ;

         COMMIT;

       END IF;
     END IF;

     /* call function to get the next activity is number */
     v_trip_qc_is_nbr := get_next_trip_qc_is_nbr(p_org_id);

     BEGIN

       p_trip_qc_rec.TSRFQS_IS_NUMBER := v_trip_qc_is_nbr;
       p_trip_qc_rec.TSRFQS_ORG_ID := RPAD(p_org_id, 8);
       p_trip_qc_rec.ID_CODE := v_activity.fa_activity_id;
       p_trip_qc_rec.TYPE_NAME := v_activity.fa_category;
       p_trip_qc_rec.NAME := NULL;
       p_trip_qc_rec.CREATION_DATE := v_activity.fa_start_date;
       p_trip_qc_rec.CREATION_TIME := v_activity.fa_start_time;
       p_trip_qc_rec.TIME_ZONE := v_activity.fa_start_time_zone;
       p_trip_qc_rec.TOTAL_VOLUME_MSR := NULL;
       p_trip_qc_rec.TOTAL_VOLUME_UN_CD := NULL;
       p_trip_qc_rec.CONTAINER_TYPE_NM := NULL;
       p_trip_qc_rec.CONTAINER_COLOR := NULL;
       p_trip_qc_rec.CONTAINER_SIZE_MSR := NULL;
       p_trip_qc_rec.CONTAINER_SIZE_UN := NULL;
       p_trip_qc_rec.COMMENT_TEXT := v_activity.fa_activity_comments;
       p_trip_qc_rec.CREATION_METHOD := NULL;
       p_trip_qc_rec.TRANSPORT_STORAGE := NULL;
       p_trip_qc_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
       p_trip_qc_rec.D_LAST_UPDATE_TS := SYSDATE;
       p_trip_qc_rec.TSRTRIP_IS_NUMBER := v_activity.fa_tsrtrip_is_number;
       p_trip_qc_rec.TSRTRIP_ORG_ID := v_activity.fa_tsrtrip_org_id;

       /* write the trip qc record */
       v_success := create_trip_qc(p_trip_qc_rec);

     EXCEPTION
        WHEN OTHERS THEN
           sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing trip qc sample record.');
           v_success := 0;
           IF t_success > v_success THEN 
	  	       t_success := v_success;
		       END IF;
     END;

     /* loop through projects, see if project is on the trip, and add if it's not */
     BEGIN
       FOR v_project IN c_get_projects(p_activity_seq) LOOP
         add_project_to_trip(v_activity.FA_TSRTRIP_IS_NUMBER,v_project.SAPJ_TSMPROJ_IS_NUMBER,p_org_id);

         /* loop through personnel, assign to role and project*/
         BEGIN
           FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

             v_success := add_person_to_project(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                                v_project.SAPJ_TSMPROJ_IS_NUMBER,RPAD(p_org_id,8));

           END LOOP;

         EXCEPTION
           WHEN OTHERS THEN
             sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to project.');
             v_success := 0;
             IF t_success > v_success THEN 
    		       t_success := v_success;
		         END IF;
         END;

       END LOOP;
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding project to trip. Trip_is_nbr: '||v_activity.FA_TSRTRIP_IS_NUMBER);
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* loop through personnel, add to the sample */
     BEGIN
       FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

         v_success := add_person_to_trip_qc(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                            v_trip_qc_is_nbr, RPAD(p_org_id,8));

       END LOOP;

     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to trip qc sample.');
         v_success := 0;
         IF t_success > v_success THEN 
		       t_success := v_success;
		     END IF;
     END;

     /* IF QC RESULTS, THERE WON'T BE AN ACTIVITY IS NUBR ... */
     BEGIN
       EXPORT_RESULTS(
         p_activity_seq,
         p_trip_qc_rec.type_name, 
         null, 
         null,
         v_trip_qc_is_nbr, 
         p_org_id,
         v_activity.fa_tsrlab_is_number, 
         v_activity.fa_tsrlab_org_id,
         p_org_id, 
         v_export_log_seq,
         p_result_cnt, 
         v_time_zone
       );

     END;

     v_trip_rec := v_trip_copy_rec;

  END LOOP;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in export trip qc.');
    v_success := 0;
    IF t_success > v_success THEN 
      t_success := v_success;
    END IF;

END export_trip_qc;

/******************************************************************************************************/
PROCEDURE delete_exported_rec(p_fa_seq IN NUMBER) IS

BEGIN

  NULL;

END delete_exported_rec;

/******************************************************************************************************/
PROCEDURE SIM_EXPORT (p_org_id IN VARCHAR2, p_import_seq IN NUMBER, p_success IN OUT NUMBER) IS

  v_activity_cnt            NUMBER(10) := 0;
  v_result_cnt              NUMBER(10) := 0; --counter for each activity
  v_file_res_cnt            NUMBER(10) := 0; --counter for total results in file
  v_line_cnt                NUMBER(10) := 0;

   v_stvst_rec                     TSRSTVST%ROWTYPE;
   v_stvst_copy_rec                TSRSTVST%ROWTYPE;
   v_activity_rec                  TSRFDACT%ROWTYPE;
   v_activity_copy_rec             TSRFDACT%ROWTYPE;
   v_trip_qc_rec                   TSRFQS%ROWTYPE;

   v_activity_is_nbr               NUMBER(8);
   v_activity_org_id               VARCHAR2(8);
   v_trip_qc_is_nbr                NUMBER(8);
   v_trip_qc_org_id                VARCHAR(8);
   v_visit_is_number               NUMBER(8);
   v_visit_org_id                  VARCHAR2(8);


   v_act_type                      VARCHAR2(13);


   v_success                       NUMBER(1);
   v_export_test                   NUMBER(10);
   v_import_type                   VARCHAR2(4);
   v_export_desc                   VARCHAR2(25);

   ex_missing_info                 EXCEPTION;
   e_no_stvst                      EXCEPTION;

   CURSOR   c_activity (p_org_id IN VARCHAR2, p_export_test IN NUMBER) IS
   SELECT   FA_SEQ, FA_TYPE_NAME 
   FROM     SIM_FIELD_ACTIVITIES
   WHERE    RPAD(FA_TSMORGAN_ORG_ID,8) = RPAD(p_org_id,8)
   AND      FA_EXPORT_SEQ IS NULL
   AND      FA_STATUS = 'A' AND FA_IMPORT_SEQ = p_import_seq
   ORDER BY FA_SEQ;  -- jah 2-13-04 To ensure we export in the same order we imported! Needed for Parent/Child.

   CURSOR   c_import_type IS
   SELECT   SIL_IMPORT_TYPE
   FROM     SIM_IMPORT_LOG
   WHERE    SIL_SEQ = p_import_seq;


BEGIN
   /* kms - initialize activity and result counters */
   v_success := 1;
   v_activity_cnt := 0;
   v_result_cnt := 0;
   v_line_cnt := 0;
   v_file_res_cnt := 0;

   OPEN c_import_type;
     FETCH c_import_type INTO v_import_type;
   CLOSE c_import_type;

   BEGIN
   /* set up export log */
      v_export_log_seq := sim2_export_pkg.GET_EXPORT_LOG_SEQ;
      IF v_import_type in ('R', 'WR') THEN
        v_export_desc := 'Results : ';
      ELSIF v_import_type = 'XR' THEN
        v_export_desc := 'XML Results : ';
      ELSE
        v_export_desc := 'Biological : ';
      END IF;
      INSERT INTO sim_export_log(
        SEL_SEQ,
        SEL_EXPORTED_BY,
        SEL_EXPORTED_ON,
        SEL_FILENAME,
        SEL_ORG_ID,
        SEL_EXPORT_TYPE
        )
      VALUES(
        v_export_log_seq,
        USER,
        SYSDATE,
        v_export_desc||p_import_seq,
        p_org_id,
        v_import_type
        );

       COMMIT;

   EXCEPTION
      WHEN OTHERS THEN
        v_success := 0;
        IF t_success > v_success THEN 
		      t_success := v_success;
  	    END IF;
        p_success := t_success;
        sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - setting up export log record.');
   END;

   sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Start: '||to_char(sysdate,'HH24:MI:SS'));

   /* loop through the activities */
   FOR v_activity IN c_activity(p_org_id, v_export_test) LOOP
     BEGIN
       /* get the activity type */
       v_act_type := v_activity.fa_type_name;

     EXCEPTION
       WHEN OTHERS THEN
         v_success := 0;
         IF t_success > v_success THEN 
 		       t_success := v_success;
  	     END IF;
         p_success := t_success;
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  getting allowable values.');
     END;

     /* see if this is a quality control sample or an activity */
     IF v_act_type != 'Trip QC' THEN
       /* set trip qc key to null */
       v_trip_qc_is_nbr := NULL;
       v_trip_qc_org_id := NULL;

       v_activity_rec.TYPE_NAME := v_act_type;

       /* run the procedure that writes activity to TSRFDACT */
       export_activity(
         v_activity.fa_seq,
         v_activity_rec, 
         v_stvst_rec, 
         p_import_seq, 
         p_org_id, 
         v_result_cnt
       );

       v_activity_is_nbr := v_activity_rec.TSRFDACT_IS_NUMBER;
       v_activity_org_id := v_activity_rec.TSRFDACT_ORG_ID;
       v_visit_is_number := v_stvst_rec.TSRSTVST_IS_NUMBER;
       v_visit_org_id := v_stvst_rec.TSRSTVST_ORG_ID;

     ELSE
     
       /* set field activity keys to null */
       v_activity_is_nbr := NULL;
       v_activity_org_id := NULL;
       v_visit_is_number := NULL;
       v_visit_org_id := NULL;

       v_trip_qc_rec.TYPE_NAME := v_act_type;

       /* run the procedure that writes activity to TSRFQS */
       export_trip_qc (v_activity.fa_seq, v_trip_qc_rec, p_import_seq, p_org_id, v_result_cnt);

       v_trip_qc_is_nbr := v_trip_qc_rec.TSRFQS_IS_NUMBER;
       v_trip_qc_org_id := v_trip_qc_rec.TSRFQS_ORG_ID;

     END IF;

  /* update counters */
     IF v_result_cnt > 0 THEN --if no results in the activity, v_result_cnt = 0
       v_line_cnt := v_line_cnt + v_result_cnt;
     END IF;
     v_activity_cnt := v_activity_cnt + 1;
     v_file_res_cnt := v_file_res_cnt + v_result_cnt;
       
    /* update the activity as being exported */
    UPDATE SIM_FIELD_ACTIVITIES
    SET FA_EXPORT_SEQ = v_export_log_seq,
       FA_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
       FA_TSRFDACT_ORG_ID    = RPAD(p_org_id,8),
       FA_TSRSTVST_IS_NUMBER = v_visit_is_number,
       FA_TSRSTVST_ORG_ID = RPAD(p_org_id,8),
       FA_TSRFQS_IS_NUMBER = v_trip_qc_is_nbr,
       FA_TSRFQS_ORG_ID = RPAD(p_org_id,8),
       FA_STATUS     = 'E'
    WHERE FA_SEQ = v_activity.fa_seq;

    COMMIT;

    v_stvst_rec := v_stvst_copy_rec;
    v_activity_rec := v_activity_copy_rec;

  END LOOP;

  /* update export log with count */
  UPDATE SIM_EXPORT_LOG
    SET SEL_ACTIVITY_CNT = v_activity_cnt
  WHERE SEL_SEQ = v_export_log_seq;

  /* update the import log with export info
     SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE SIM_IMPORT_LOG SET
     SIL_EXPORTED_ON = SYSDATE,
     SIL_EXPORTED_BY = USER,
     SIL_ROWS_READY = SIL_ROWS_READY - v_line_cnt,
     SIL_ACTIVITIES_IMPORTED = SIL_ACTIVITIES_IMPORTED - v_activity_cnt
  WHERE SIL_SEQ = P_IMPORT_SEQ;

  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Migrated '||v_activity_cnt||' activities, '||v_file_res_cnt||' results.');
  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Finish: '||to_char(sysdate,'HH24:MI:SS'));

  COMMIT;
  p_success := t_success;

EXCEPTION
   WHEN OTHERS THEN
      v_success := 0;
      IF t_success > v_success THEN 
		    t_success := v_success;
  	  END IF;
      p_success := t_success;
      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  Error in sim2_export_pkg.EXPORT_SIM');

END SIM_EXPORT;

/******************************************************************************************************/
PROCEDURE EXPORT_RESULTS(
          p_fa_seq          IN NUMBER, 
          p_fat_desc        IN VARCHAR2, 
          p_activity_is_nbr IN NUMBER,
          p_activity_org_id IN VARCHAR2, 
          p_tripqc_is_nbr   IN NUMBER, 
          p_tripqc_org_id   IN VARCHAR2,
          p_lab_is_number   IN NUMBER,
          p_lab_org_id      IN VARCHAR2, 
          p_org_id          IN VARCHAR2,
          p_export_log_seq  IN NUMBER, 
          p_act_result_cnt  OUT NUMBER, 
          p_time_zone       IN VARCHAR2) IS

  v_success                       NUMBER(1);
  v_activity_id                   VARCHAR2(12);
  v_act_cat                       VARCHAR2(30);
  v_data_column_cnt               NUMBER(10);
  v_line_cnt                      NUMBER(10);
  v_result_rec                    TSRRSULT%ROWTYPE;
  v_result_copy_rec               TSRRSULT%ROWTYPE;
  v_cell_rec                      TSRCLDES%ROWTYPE;
  v_remark_rec                    TSRRLRA%ROWTYPE;
  v_remark_success                NUMBER(1);

  v_result_is_number              NUMBER(8);
  v_tsrdlin_is_number             NUMBER(8);
  v_lab_is_nbr                    NUMBER(8);
  v_lab_org_id                    VARCHAR2(8);
  v_tsrdlin_is_number2            NUMBER(12);
  v_tsrdlin_org_id2               VARCHAR2(8);

  v_result_value                  NUMBER(13,5);
  v_result_text                   VARCHAR2(15);
  v_det_limit                     VARCHAR2(12);
  v_min_detect                    VARCHAR2(12);
  v_max_detect                    VARCHAR2(12);
  v_detect_cond_cd                VARCHAR2(50);
  v_result_cmt                    VARCHAR2(1999);
  v_tsmgntxt_is_nbr               NUMBER(8);
  v_comp_ind_cd                   VARCHAR2(1);
  v_table                         VARCHAR2(8);
  v_desc                          VARCHAR2(8);
  v_blob_success                  NUMBER(1);

  -- jah 2-19-04
  v_prev_data_line_number         NUMBER(8);  -- tracks when rs_data_line_number changes
  v_prev_brgi_seq                 NUMBER(10); -- tracks when rs_brgi_seq changes
  v_assnd_seq_num                 NUMBER(8);

  e_def_not_complete              EXCEPTION;
  e_no_chgrp                      EXCEPTION;
  e_no_char                       EXCEPTION;
  e_no_rowid                      EXCEPTION;

  /* jah 2-19-04
     Get the results for a given activity.
     Order by BRGI_SEQ for biological activities.
     Order by RS_DATA_LINE_NUMBER for Portable Data Logger activities. */
  CURSOR   c_result (p_activity_seq IN NUMBER) IS
  SELECT   *
  FROM     SIM_RESULTS
  WHERE    RS_FA_SEQ = p_activity_seq
  ORDER BY RS_DATA_LINE_NUMBER, RS_BRGI_SEQ, RS_TSRCHAR_IS_NUMBER;

  CURSOR c_get_activity_id(p_activity_seq IN NUMBER) IS
    SELECT FA_ACTIVITY_ID
    FROM SIM_FIELD_ACTIVITIES
    WHERE FA_SEQ = p_activity_seq;
    
  CURSOR c_lab_remark_codes(p_seq NUMBER) IS
    SELECT srlr_seq,
           srlr_tsrlbrmk_is_number,
           srlr_tsrlbrmk_org_id
     FROM sim_result_lab_remarks
     WHERE srlr_rs_seq = p_seq;    

  CURSOR c_act_cat(p_seq NUMBER) IS
    SELECT fa_category
      FROM sim_field_activities
      WHERE fa_seq = p_seq;
           
  CURSOR c_data_column_cnt(p_seq NUMBER) IS
    SELECT COUNT(DISTINCT(rs_tsrchar_is_number))
      FROM sim_results
      WHERE rs_fa_seq = p_seq;
    
  --This cursor NEEDS to be ordered by rs_data_line_number                
  CURSOR c_data_lines(p_seq NUMBER) IS
    SELECT DISTINCT(RS_DATA_LINE_NUMBER) rs_data_line_number
      FROM sim_results
      WHERE rs_fa_seq = p_seq
      ORDER BY rs_data_line_number;

  CURSOR c_line_nbr_cnt(p_seq NUMBER,p_line_number NUMBER) IS
    SELECT COUNT (rs_tsrchar_is_number)
      FROM sim_results
      WHERE rs_fa_seq = p_seq
        AND rs_data_line_number = p_line_number;
        
  CURSOR c_missing_is_number(p_seq NUMBER,p_line_number NUMBER) IS
    SELECT DISTINCT rs_tsrchar_is_number,rs_tsrchar_org_id
      FROM sim_results
      WHERE rs_fa_seq = p_seq     
        AND rs_tsrchar_is_number NOT IN(SELECT rs_tsrchar_is_number
                                        FROM sim_results
                                        WHERE rs_fa_seq = p_seq
                                          AND rs_data_line_number = p_line_number);        
                      
  CURSOR c_tsrdlin(p_line_nbr NUMBER,p_is_nbr NUMBER,p_org_id VARCHAR2) IS
    SELECT TSRDLIN_IS_NUMBER,TSRDLIN_ORG_ID
      FROM TSRDLIN
      WHERE LINE_NUMBER = p_line_nbr
        AND TSRFDACT_IS_NUMBER = p_is_nbr
        AND TSRFDACT_ORG_ID = p_org_id;
        
BEGIN

  p_act_result_cnt := 0;

  -- jah 2-19-04  init display order sequence tracking variables
  v_assnd_seq_num := 0;
  v_prev_data_line_number := 0;
  v_prev_brgi_seq := 0;
  
  FOR v_result IN c_result (p_fa_seq) LOOP
    BEGIN
      /* initialize variables */
      v_result_value := NULL;
      v_det_limit := NULL;
      v_min_detect := NULL;
      v_max_detect := NULL;
      v_detect_cond_cd := NULL;
      v_comp_ind_cd := NULL;
      
      -- get the next IS_NUMBER for TSRRSULT
      v_result_is_number := get_next_result_is_nbr (p_org_id);

      v_lab_is_nbr :=  v_result.rs_tsrlab_is_number;    --p_lab_is_number;
      v_lab_org_id :=  v_result.rs_tsrlab_org_id;       --RPAD(p_lab_org_id,8);

      /* convert result text to number */
      v_result_text := v_result.rs_result;
    
      IF v_result.rs_tsrhcsc_is_number IS NOT NULL THEN  --If it's habitat Assessment mes is always 0
        v_result_value := 0;
      ELSE
        v_result_value := get_value_msr(v_result_text);
      END IF;
  
      IF v_result.rs_value_status IS NULL THEN
        v_result.rs_value_status := 'F';
      END IF;

      /* comp_ind_cd should always be 'Y' if the activity has been allowed to be exported */
      v_comp_ind_cd := 'Y';
      /* Issue other STORET NULL Value defaults. */
      --For TSRRSULT
      IF v_result.RS_ANALYSIS_DATE IS NULL THEN
        v_result.RS_ANALYSIS_DATE := TO_DATE('01-JAN-0001','DD-MON-YYYY');
      END IF;
      IF v_result.RS_ANALYSIS_TIME IS NULL THEN
        v_result.RS_ANALYSIS_TIME := TO_DATE('01-JAN-0001','DD-MON-YYYY');
      END IF;
      IF v_result.rs_lab_cert IS NULL THEN
        v_result.rs_lab_cert := 'N';
      END IF;

      /* kms - set detection condition to correct value in order for the correct radio
         button to be selected in the STORET result screen. */
      IF v_result.rs_tsrhcsc_is_number IS NOT NULL THEN  --If it's habitat Assessment no detect_cond
        v_detect_cond_cd := NULL;
      ELSE
        IF v_result_text = '*Non-detect' THEN
          v_detect_cond_cd := 'Not Detected';
        ELSIF v_result_text = '*Present <QL' THEN
          v_detect_cond_cd := 'Present below Quantification';
        ELSIF v_result_text = '*Present >QL' THEN
          v_detect_cond_cd := 'Present above Quantification';
        ELSIF v_result_text = '*Present' THEN
          v_detect_cond_cd := 'Detected not Quantified';
        ELSIF v_result_text = '*Not Reported' THEN
          v_detect_cond_cd := 'Not Reported';
        ELSE
          v_detect_cond_cd := 'Detected and Quantified';
        END IF;
      END IF;

      -- jah 2-19-04
      -- Does this result have a data line number?
      IF (v_result.RS_DATA_LINE_NUMBER IS NOT NULL) THEN
        -- if yes
        
        -- has the data line number changed?
        IF (v_prev_data_line_number <> v_result.RS_DATA_LINE_NUMBER) THEN
          -- if yes
          
          -- reset the display order sequence number
          v_assnd_seq_num := 0;
          
          -- set the previous line number to the current line number
          v_prev_data_line_number := v_result.RS_DATA_LINE_NUMBER;

          -- Get the IS_NUMBER for the next line number record in TSRDLIN
          v_tsrdlin_is_number := sim2_export_pkg.get_eisn_number('TSRDLIN',RPAD(p_org_id,8));

          -- insert the next current line number into TSRDLIN
          INSERT INTO TSRDLIN
            (tsrdlin_is_number,tsrdlin_org_id,line_number,tsrfdact_is_number,
             tsrfdact_org_id,d_userid_code,d_last_update_ts,line_name) 
           VALUES
            (v_tsrdlin_is_number,RPAD(p_org_id,8),v_result.RS_DATA_LINE_NUMBER,p_activity_is_nbr,
             p_activity_org_id,sim2_export_pkg.v_pkg_ownerid,sysdate,v_result.RS_DATA_LINE_NAME);
        
        END IF;

      -- Does this result have a Biological Result Group Individual sequence
      ELSIF (v_result.RS_BRGI_SEQ IS NOT NULL) THEN
        -- if yes
          
        -- has the Biological Result Group Individual sequence changed?
        IF (v_prev_brgi_seq <> v_result.RS_BRGI_SEQ) THEN
          -- if yes
          
          -- reset the display order sequence number
          v_assnd_seq_num := 0;
          
          -- reset the previous BRGI sequence number
          v_prev_brgi_seq := v_result.RS_BRGI_SEQ;
          
        END IF;             

      END IF;   

      /* write values to be exported to result record */
      BEGIN
        v_result_rec.TSRRSULT_IS_NUMBER := v_result_is_number;
        v_result_rec.TSRRSULT_ORG_ID := RPAD(p_org_id,8);

        -- jah 2-19-04
        -- increment the display order sequence number
        v_assnd_seq_num := v_assnd_seq_num + 1;
        v_result_rec.D_ASSND_SEQ_NUM := v_assnd_seq_num;

        v_result_rec.COMP_IND_CD := v_comp_ind_cd;
        v_result_rec.LAB_CERT_IND_CODE := v_result.rs_lab_cert;
        v_result_rec.VALUE_TEXT := RPAD(v_result_text,15);
        v_result_rec.VALUE_MEASURE := v_result_value;
        v_result_rec.VALUE_STATUS := v_result.rs_value_status;
        v_result_rec.PRECISION_AMT_TEXT := v_result.rs_precision;
        v_result_rec.LAB_BATCH_ID_CODE := v_result.rs_lab_batch_id;
        IF v_result_text = '*Text' THEN
          v_result_rec.RSLT_TXT_IND_CD := 'Y';
        ELSE
          v_result_rec.RSLT_TXT_IND_CD := 'N';
        END IF;
        v_result_rec.CELL_DESC_IND_CD := 'N';
        v_result_rec.CONF_LVL_PCT_MSR := v_result.rs_conf_level;
        v_result_rec.PROC_EXCPTN_IND_CD := 'N';
        v_result_rec.DUR_BASIS_TYPE_NM := v_result.rs_duration_basis;
        v_result_rec.DETECT_COND_CD := RPAD(v_detect_cond_cd,30);
        v_result_rec.SMPL_FRAC_TYPE_NM := v_result.rs_fraction;
        v_result_rec.STATISTIC_TYPE_NM := v_result.rs_stat_type;
        v_result_rec.VALUE_TYPE_NAME := v_result.rs_value_type;
        v_result_rec.WT_BASIS_TYPE_NM := v_result.rs_weight_basis;
        v_result_rec.TEMP_BASIS_LVL_NM := v_result.rs_temp_basis;
        v_result_rec.TAXON_POLLUTION := v_result.rs_taxon_pollution;
        v_result_rec.FNCTIONAL_FEED_GRP := v_result.rs_fnctional_feed_grp;
        v_result_rec.TROPHIC_LEVEL := v_result.rs_trophic_level;
        v_result_rec.ANALYSIS_DATE := v_result.rs_analysis_date;
        v_result_rec.ANALYSIS_TIME := v_result.rs_analysis_time;
        v_result_rec.ANALYSIS_TIME_ZONE := v_result.rs_analysis_time_zone;
        v_result_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
        v_result_rec.D_LAST_UPDATE_TS := SYSDATE;

        --02/22/02 SF - Either a TSRFDACT_IS_NUM or a TSRBRG_IS_NUM goes in, never both
        IF v_result.rs_tsrbrg_is_number IS NULL THEN
          v_result_rec.TSRFDACT_IS_NUMBER := p_activity_is_nbr;
          v_result_rec.TSRFDACT_ORG_ID := RPAD(p_activity_org_id,8);
        ELSE
          v_result_rec.TSRFDACT_IS_NUMBER := NULL;
          v_result_rec.TSRFDACT_ORG_ID := NULL;
        END IF;

        v_result_rec.TSRFQS_IS_NUMBER := p_tripqc_is_nbr;
        v_result_rec.TSRFQS_ORG_ID := RPAD(p_tripqc_org_id,8);
        v_result_rec.TSRANLPR_IS_NUMBER := v_result.rs_tsranlpr_is_number;
        v_result_rec.TSRANLPR_ORG_ID := RPAD(v_result.rs_tsranlpr_org_id,8);
        v_result_rec.TSRBRGI_IS_NUMBER := v_result.rs_tsrbrgi_is_number;
        v_result_rec.TSRBRGI_ORG_ID := RPAD(v_result.rs_tsrbrgi_org_id,8);
        v_result_rec.TSRBRG_IS_NUMBER := v_result.rs_tsrbrg_is_number;
        v_result_rec.TSRBRG_ORG_ID := RPAD(v_result.rs_tsrbrg_org_id,8);
        v_result_rec.TSRCHAR_IS_NUMBER := v_result.rs_tsrchar_is_number;
        v_result_rec.TSRCHAR_ORG_ID := v_result.rs_tsrchar_org_id;
        v_result_rec.TSRCHAR0IS_NUMBER :=v_result.rs_tsrchar0is_number;
        v_result_rec.TSRCHAR0ORG_ID := v_result.rs_tsrchar0org_id;
        v_result_rec.TSRLAB_IS_NUMBER := v_lab_is_nbr;
        v_result_rec.TSRLAB_ORG_ID := v_lab_org_id;
        v_result_rec.TSRUOM_IS_NUMBER := v_result.rs_tsruom_is_nbr;
        v_result_rec.TSRUOM_ORG_ID := RPAD(v_result.rs_tsruom_org_id,8);
        v_result_rec.TSRLSPP_IS_NUMBER := v_result.rs_tsrlspp_is_number;
        v_result_rec.TSRLSPP_ORG_ID := v_result.rs_tsrlspp_org_id;
        v_result_rec.TSRHCSC_IS_NUMBER := v_result.rs_tsrhcsc_is_number;
        v_result_rec.TSRHCSC_ORG_ID := v_result.rs_tsrhcsc_org_id;

        /*New stuff for V2 */

        v_result_rec.BIAS  := v_result.RS_BIAS;
        v_result_rec.CONF_LVL_CORR_BIAS := v_result.RS_CONF_LVL_CORR_BIAS;
        v_result_rec.CORRECTION_IND_CD := v_result.RS_CORRECTION_IND_CD;
        v_result_rec.DILUTION_IND_CODE := v_result.RS_DILUTION_IND_CODE;
        v_result_rec.TSMPRMVL_IS_NUMBER := v_result.RS_FRACTION_TSMPRMVL_IS_NUMBER;
        v_result_rec.TSMPRMVL_ORG_ID := v_result.RS_FRACTION_TSMPRMVL_ORG_ID;
        v_result_rec.TSMPRMVL0IS_NUMBER := v_result.RS_HABIT_TSMPRMVL0IS_NUMBER;
        v_result_rec.TSMPRMVL0ORG_ID := v_result.RS_HABIT_TSMPRMVL0ORG_ID;
        v_result_rec.TSMPRMVL1IS_NUMBER := v_result.RS_VOLT_TSMPRMVL1IS_NUMBER;
        v_result_rec.TSMPRMVL1ORG_ID := v_result.RS_VOLT_TSMPRMVL1ORG_ID;
        v_result_rec.LOWER_RANGE_VALUE := v_result.RS_LOWER_RANGE_VALUE;
        v_result_rec.UPPER_RANGE_VALUE := v_result.RS_UPPER_RANGE_VALUE;
        v_result_rec.RECOVERY_IND_CODE := v_result.RS_RECOVERY_IND_CODE;
        v_result_rec.REF_PT_FROM_NAME := v_result.RS_REF_PT_FROM_NAME;
        v_result_rec.REF_PT_TO_NAME := v_result.RS_REF_PT_TO_NAME;
        v_result_rec.REPL_ANALYSIS_NUM := v_result.RS_REPL_ANALYSIS_NUM;
        v_result_rec.SPECIES_NUMBER := v_result.RS_SPECIES_NUMBER;
        v_result_rec.TSRDLIN_IS_NUMBER := v_tsrdlin_is_number;
        v_result_rec.TSRDLIN_ORG_ID := RPAD(p_org_id,8);
        v_result_rec.BLOB_TITLE := sim2_export_pkg.get_file_name(v_result.RS_DOCUMENT);
        v_result_rec.BLOB_TYPE := sim2_export_pkg.get_file_type(v_result.RS_DOCUMENT);

        IF v_result.RS_CELL_SHAPE_TYPE_NM IS NOT NULL OR v_result.RS_CELL_TYPE_NM IS NOT NULL THEN
          v_result_rec.CELL_DESC_IND_CD := 'Y';
          v_cell_rec.TSRRSULT_IS_NUMBER := v_result_is_number;
          v_cell_rec.TSRRSULT_ORG_ID := RPAD(p_org_id,8);
          v_cell_rec.CELL_SHAPE_TYPE_NM := v_result.RS_CELL_SHAPE_TYPE_NM;
          v_cell_rec.CELL_TYPE_NM := v_result.RS_CELL_TYPE_NM;
          v_cell_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
          v_cell_rec.D_LAST_UPDATE_TS := SYSDATE;
        ELSE
          v_result_rec.CELL_DESC_IND_CD := 'N';
          v_cell_rec.TSRRSULT_IS_NUMBER := NULL;
        END IF;
      EXCEPTION
        WHEN OTHERS THEN
          v_success := 0;
          IF t_success > v_success THEN 
		        t_success := v_success;
  	      END IF;
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - assigning values to result record.');
      END;

      /* create STORET result record */
      v_success := create_result(v_result_rec, v_result.rs_seq);
      
      -- Create the Result Lab Remarks record 
      IF v_success = 1 THEN

        /* export binary document for result */
        IF v_result.rs_document IS NOT NULL THEN
          v_blob_success := insert_result_binary_document(v_export_log_seq, p_org_id, v_result_is_number, v_result.rs_document);
          IF v_blob_success = 0 THEN
            v_success := 0;
            IF t_success > v_success THEN 
              t_success := v_success;
      		  END IF;
          END IF;
        END IF;

        FOR c_lab_remark_codes_rec IN c_lab_remark_codes(v_result.rs_seq) LOOP
          v_remark_rec.tsrrsult_is_number := v_result_is_number;
          v_remark_rec.tsrrsult_org_id := RPAD(p_org_id,8);
          v_remark_rec.tsrlbrmk_is_number := c_lab_remark_codes_rec.srlr_tsrlbrmk_is_number;
          v_remark_rec.tsrlbrmk_org_id := c_lab_remark_codes_rec.srlr_tsrlbrmk_org_id;
          v_remark_rec.d_userid_code := sim2_export_pkg.v_pkg_ownerid;
          v_remark_rec.d_last_update_ts := SYSDATE;
          v_remark_success := add_lab_remarks_to_result(v_remark_rec);
        END LOOP;
      END IF;
      
      IF v_cell_rec.tsrrsult_is_number IS NOT NULL THEN
        v_success := create_tsrcldes(v_cell_rec);
      END IF;
      
      -- create tsrrci record for this result
      v_success := create_tsrrci_record(
        p_org_id,
        v_result_is_number
      ); 
  
      -- insert result comment    
      v_result_cmt := v_result.RS_COMMENT;
      IF v_result.RS_COMMENT = '*Text' THEN     -- If the comment indicates it is a text
        v_table := 'TSRRSULT';                  -- characteristic, insert data using the
        v_desc  := 'DESCRIPT';                  -- correct table and value for description
      ELSE                                      -- name
        v_table := 'TSRRSLT';
        v_desc  := 'RSLTCMNT';
      END IF;

      IF v_result_cmt IS NOT NULL THEN
        v_tsmgntxt_is_nbr := get_next_tsmgntxt_nbr(p_org_id);
        BEGIN
          INSERT INTO TSMGNTXT (
            TSMGNTXT_IS_NUMBER,
            TSMGNTXT_ORG_ID,
            TABLE_NAME,
            DESCRIPTION_NAME,
            DESCRIPTION_TEXT,
            D_USERID_CODE,
            D_LAST_UPDATE_TS,
            TSRRSULT_IS_NUMBER,
            TSRRSULT_ORG_ID,
            TSMPROJ_IS_NUMBER,
            TSMPROJ_ORG_ID,
            TSMPROGM_IS_NUMBER,
            TSMPROGM_ORG_ID,
            TSRGRCFG_IS_NUMBER,
            TSRGRCFG_ORG_ID
            )
          VALUES (
            v_tsmgntxt_is_nbr,                                 /* TSMGNTXT_IS_NUMBER */
            RPAD(p_org_id,8),                                  /* TSMGNTXT_ORG_ID */
            v_table,                                           /* TABLE_NAME */
            v_desc,                                            /* DESCRIPTION_NAME */
            v_result_cmt,                                      /* DESCRIPTION_TEXT */
            sim2_export_pkg.v_pkg_ownerid,
            SYSDATE,                                           /* D_LAST_UPDATE_TS */
            v_result_rec.TSRRSULT_IS_NUMBER,                   /* TSRRSULT_IS_NUMBER */
            v_result_rec.TSRRSULT_ORG_ID,                      /* TSRRSULT_ORG_ID */
            NULL,                                              /* TSMPROJ_IS_NUMBER */
            NULL,                                              /* TSMPROJ_ORG_ID */
            NULL,                                              /* TSMPROGM_IS_NUMBER */
            NULL,                                              /* TSMPROGM_ORG_ID */
            NULL,                                              /* TSRGRCFG_IS_NUMBER */
            NULL                                               /* TSRGRCFG_ORG_ID */
            );

        EXCEPTION
          WHEN OTHERS THEN
            sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - writing result comment.');
        END;
      END IF;

      /* figure out detect limit info */
      BEGIN
        v_result_text := v_result.RS_RESULT;
        v_det_limit := v_result.RS_DETECT_LIMIT;
        v_result_cmt := v_result.RS_COMMENT;
        v_min_detect := v_result.RS_LOWER_QUANT_LIMIT;
        v_max_detect := v_result.RS_UPPER_QUANT_LIMIT;

        IF v_det_limit IS NOT NULL OR v_min_detect IS NOT NULL OR v_max_detect IS NOT NULL THEN
          BEGIN
            INSERT INTO TSRDQL (
              TSRRSULT_IS_NUMBER,
              TSRRSULT_ORG_ID,
              MIN_DETECT_LIMIT,
              MIN_QUANT_LIMIT,
              MAX_QUANT_LIMIT,
              DESCRIPTION_TEXT,
              D_USERID_CODE,
              D_LAST_UPDATE_TS,
              TSRUOM_IS_NUMBER,
              TSRUOM_ORG_ID)
            VALUES (
              v_result_is_number,
              RPAD(p_org_id,8),
              v_det_limit,
              v_min_detect,                             /* MIN QUANT LIMIT */
              v_max_detect,                             /* MAX QUANT LIMIT */
              v_result.rs_detect_comment                /* DESC TEXT */,
              sim2_export_pkg.v_pkg_ownerid,
              SYSDATE                                   /* D_LAST_UPDATE_TS */,
              v_result.rs_det_limit_uom_nbr             /* TSRUOM_IS_NUMBER */,
              v_result.rs_det_limit_uom_org             /* TSRUOM_ORG_ID */
              );

            COMMIT;
          EXCEPTION
            WHEN OTHERS THEN
              sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - writing detect and quantification limit record.');
          END;

        END IF;

      EXCEPTION
        WHEN OTHERS THEN
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting detect and quantification limit information.');

      END;

      /* keep track of the number of results exported for an activity (incremented only
         when all inserts have been performed so that the value accurately reflects what has
         been loaded into STORET */
      p_act_result_cnt := p_act_result_cnt + 1;

    EXCEPTION
      WHEN OTHERS THEN
        sim2_export_pkg.WRITE_ERROR(p_export_log_seq, SQLERRM||' -  writing result ');
    END;

    v_result_rec := v_result_copy_rec;

  END LOOP;

  OPEN  c_act_cat(p_fa_seq);
  FETCH c_act_cat INTO v_act_cat;
  CLOSE c_act_cat;

  -- do we need to worry about missing characteric results for this activity?
  IF UPPER(v_act_cat) = 'PORTABLE DATA LOGGER' THEN
    -- if yes
   
    -- count the number of characteristics
    OPEN c_data_column_cnt(p_fa_seq);
    FETCH c_data_column_cnt INTO v_data_column_cnt;
    CLOSE c_data_column_cnt;
    
    -- loop through all the dataline for this activity
    FOR c_data_lines_rec IN c_data_lines(p_fa_seq) LOOP --c_data_lines_rec
    
      -- count the number of rows
      OPEN c_line_nbr_cnt(p_fa_seq,c_data_lines_rec.rs_data_line_number);
      FETCH c_line_nbr_cnt INTO v_line_cnt;
      CLOSE c_line_nbr_cnt;
      
      -- are any results missing?
      IF NVL(v_line_cnt,0) < v_data_column_cnt THEN
        -- if yes
        
        /*
          jah 2-19-04
          The following loop finds where the activity has missing results
          and inserts rows for the missing results into TSRRSULT. STORET
          requires a row for every characteristic whether the characteristic
          has data or not!
        */
        
        -- Loop through the missing characteristics
        FOR c_missing_is_number_rec IN c_missing_is_number(p_fa_seq,c_data_lines_rec.rs_data_line_number) LOOP --c_missing_is_number_rec
        
          -- Get the key of the TSRDLIN record for this activity
          OPEN c_tsrdlin(c_data_lines_rec.rs_data_line_number,p_activity_is_nbr,p_activity_org_id);
          FETCH c_tsrdlin INTO v_tsrdlin_is_number2,v_tsrdlin_org_id2;
          CLOSE c_tsrdlin;

          -- get the next IS_NUMBER for TSRRSULT
          v_result_is_number := get_next_result_is_nbr (p_org_id);
          
          BEGIN
            v_result_rec.TSRRSULT_IS_NUMBER := v_result_is_number;
            v_result_rec.TSRRSULT_ORG_ID := RPAD(p_org_id,8);
            
            /*
              jah 2-19-04
              TSRRSULT.D_ASSND_SEQ_NUM controls the display order of results.
              The following statement causes missing characteristics, which 
              this loop handles, to display first.
            */
            v_result_rec.D_ASSND_SEQ_NUM := 0;
            
            v_result_rec.COMP_IND_CD := v_comp_ind_cd;
            v_result_rec.VALUE_STATUS := 'F';     
            v_result_rec.TSRCHAR_IS_NUMBER := c_missing_is_number_rec.rs_tsrchar_is_number;
            v_result_rec.TSRCHAR_ORG_ID := c_missing_is_number_rec.rs_tsrchar_org_id;   
            v_result_rec.D_USERID_CODE := sim2_export_pkg.v_pkg_ownerid;
            v_result_rec.D_LAST_UPDATE_TS := SYSDATE;      
            v_result_rec.TSRFDACT_IS_NUMBER := p_activity_is_nbr;
            v_result_rec.TSRFDACT_ORG_ID := RPAD(p_activity_org_id,8);
            v_result_rec.TSRDLIN_IS_NUMBER := v_tsrdlin_is_number2;
            v_result_rec.TSRDLIN_ORG_ID := v_tsrdlin_org_id2;
            
          EXCEPTION
            WHEN OTHERS THEN
              v_success := 0;
              IF t_success > v_success THEN 
		            t_success := v_success;
      	      END IF;
              sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - assigning values to result record for the portable data logger.');
          END;
          
          -- insert the row for the missing result into STORET
          v_success := create_result(v_result_rec,NULL);

        END LOOP; --c_missing_is_number_rec
      END IF;
    
    END LOOP;--c_data_lines_rec
    
  END IF;
  /* find out if any results were exported on this activity and write error message if none were */
  IF p_act_result_cnt = 0 THEN
    OPEN c_get_activity_id(p_fa_seq);
    FETCH c_get_activity_id INTO v_activity_id;
    CLOSE c_get_activity_id;

    sim2_export_pkg.write_error(v_export_log_seq, 'No results exported for activity: '||v_activity_id);
  END IF;

END EXPORT_RESULTS;

/******************************************************************************************************/
PROCEDURE runtime IS

  v_temp  NUMBER(10);

  CURSOR c_temp IS
    SELECT (MAX(TSRFDACT_IS_NUMBER))
     FROM TSRFDACT
     WHERE TSRFDACT_ORG_ID = RPAD('UTAHDWQ ',8);

BEGIN



  OPEN c_temp;
    FETCH c_temp INTO v_temp;
  CLOSE c_temp;


END runtime;

/******************************************************************************************************/
FUNCTION insert_result_binary_document(p_export_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER IS

  v_is_number   NUMBER(8);
  v_bfile       BFILE;
  v_blob        BLOB;
  v_file_size   INTEGER;
  
BEGIN

  BEGIN
    v_bfile := bfilename('DOCUMENT_DIR', p_file_name);
    dbms_lob.fileopen(v_bfile, dbms_lob.file_readonly);
    v_file_size := dbms_lob.getlength(v_bfile);

    v_is_number := get_next_blob_is_number(p_org_id);
  
    INSERT INTO TSMBLOB (
      TSMBLOB_IS_NUMBER,
      TSMBLOB_ORG_ID,
      TABLE_NAME,
      TSRRSULT_IS_NUMBER,
      TSRRSULT_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS,
      BLOB_CONTENT)
    VALUES(
      v_is_number,
      RPAD(p_org_id,8),
      'TSRRSULT',
      p_is_nbr,
      RPAD(p_org_id,8),
      sim2_export_pkg.v_pkg_ownerid, 
      SYSDATE,
      EMPTY_BLOB())
    RETURN BLOB_CONTENT INTO v_blob;

    dbms_lob.loadfromfile(v_blob, v_bfile, v_file_size);
    dbms_lob.fileclose(v_bfile);
  EXCEPTION
    WHEN OTHERS THEN
      sim2_export_pkg.write_error(p_export_log_seq, 'Unable to load Document/Graphic file ' || '"' || p_file_name || '".  Please check directory and file permissions.');
      RETURN 0;
  END;

  RETURN 1;
    
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(p_export_log_seq, SQLERRM||' - in insert binary document.');
    RETURN 0;

END insert_result_binary_document;

/******************************************************************************************************/
FUNCTION insert_visit_binary_document(p_export_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER IS

  v_is_number   NUMBER(8);
  v_bfile       BFILE;
  v_blob        BLOB;
  v_file_size   INTEGER;
  
BEGIN

  BEGIN
    v_bfile := bfilename('DOCUMENT_DIR', p_file_name);
    dbms_lob.fileopen(v_bfile, dbms_lob.file_readonly);
    v_file_size := dbms_lob.getlength(v_bfile);

    v_is_number := get_next_blob_is_number(p_org_id);
  
    INSERT INTO TSMBLOB (
      TSMBLOB_IS_NUMBER,
      TSMBLOB_ORG_ID,
      TABLE_NAME,
      TSRSTVST_IS_NUMBER,
      TSRSTVST_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS,
      BLOB_CONTENT)
    VALUES(
      v_is_number,
      RPAD(p_org_id,8),
      'TSRSTVST',
      p_is_nbr,
      RPAD(p_org_id,8),
      sim2_export_pkg.v_pkg_ownerid,
      SYSDATE,
      EMPTY_BLOB())
    RETURN BLOB_CONTENT INTO v_blob;

    dbms_lob.loadfromfile(v_blob, v_bfile, v_file_size);
    dbms_lob.fileclose(v_bfile);
  EXCEPTION
    WHEN OTHERS THEN
      sim2_export_pkg.write_error(p_export_log_seq, 'Unable to load Document/Graphic file ' || '"' || p_file_name || '".  Please check directory and file permissions.');
      RETURN 0;
  END;

  RETURN 1;
    
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(p_export_log_seq, SQLERRM||' - in insert binary document.');
    RETURN 0;

END insert_visit_binary_document;

/******************************************************************************************************/
FUNCTION insert_act_binary_document(p_export_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER IS

  v_is_number   NUMBER(8);
  v_bfile       BFILE;
  v_blob        BLOB;
  v_file_size   INTEGER;
  
BEGIN

  BEGIN
    v_bfile := bfilename('DOCUMENT_DIR', p_file_name);
    dbms_lob.fileopen(v_bfile, dbms_lob.file_readonly);
    v_file_size := dbms_lob.getlength(v_bfile);

    v_is_number := get_next_blob_is_number(p_org_id);
  
    INSERT INTO TSMBLOB (
      TSMBLOB_IS_NUMBER,
      TSMBLOB_ORG_ID,
      TABLE_NAME,
      TSRFDACT_IS_NUMBER,
      TSRFDACT_ORG_ID,
      D_USERID_CODE,
      D_LAST_UPDATE_TS,
      BLOB_CONTENT)
    VALUES(
      v_is_number,
      RPAD(p_org_id,8),
      'TSRFDACT',
      p_is_nbr,
      RPAD(p_org_id,8),
      sim2_export_pkg.v_pkg_ownerid, 
      SYSDATE,
      EMPTY_BLOB())
    RETURN BLOB_CONTENT INTO v_blob;

    dbms_lob.loadfromfile(v_blob, v_bfile, v_file_size);
    dbms_lob.fileclose(v_bfile);
  EXCEPTION
    WHEN OTHERS THEN
      sim2_export_pkg.write_error(p_export_log_seq, 'Unable to load Document/Graphic file ' || '"' || p_file_name || '".  Please check directory and file permissions.');
      RETURN 0;
  END;

  RETURN 1;
    
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(p_export_log_seq, SQLERRM||' - in insert binary document.');
    RETURN 0;

END insert_act_binary_document;

/******************************************************************************************************/
FUNCTION get_next_blob_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS

   v_next_blob_is_nbr NUMBER(8);

BEGIN
   /* get the next avail is nbr to use */
   v_next_blob_is_nbr := sim2_export_pkg.get_eisn_number('TSMBLOB',p_org_id);

   RETURN v_next_blob_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next blob is number.');
     v_success := 0;
     RETURN NULL;

END get_next_blob_is_number;

/******************************************************************************************************/

END sim2_export_result_pkg;
/
