CREATE OR REPLACE PACKAGE sim2_export_station_pkg IS
/* Modified 7/16/02 pwh - Added the v_userid_code variable, set it to 'SIM122  ' and inserted this variable
   into the D_USERID_CODE column in the STORET tables.
*/
/*
  Modified 3/26/02 by SF to include NRCS in the Export. For some reason it was null

  Modified 2/14/02  by SF       To fix references to TSMALP.STATION_IS_NUMBER, etc
*/

  -- Author  : KHENAS - taken from old export package
  -- Created : 11/16/00 4:04:26 PM
  -- Purpose : export station data from SIM to STORET
  -- Public variable declarations
  v_export_log_seq      NUMBER(10);
  v_success             NUMBER(1);
  -- Public function and procedure declarations

  FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2;
  FUNCTION get_next_station_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER;
  FUNCTION GET_UOM_IS_NBR (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2, p_chdef_is_number IN NUMBER) RETURN NUMBER;
  FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER;
  PROCEDURE SIM2_EXPORT_STATIONS( p_imp_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_success IN OUT NUMBER);
  FUNCTION insert_binary_document(p_org_id IN VARCHAR2, p_station_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER;
  
END sim2_export_station_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim2_export_station_pkg IS
  -- Function and procedure implementations

/***************************************************************************************/
/******************************************************************************************************/
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS
  v_item_name         VARCHAR2(80);
  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_alval_seq;
BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name INTO v_item_name;
    IF c_get_alval_name%NOTFOUND THEN
      v_item_name := NULL;
      sim2_export_pkg.write_error(v_export_log_seq, 'Could not find allowable value with sequence: '||p_alval_seq);
    END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
   sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting allowable value sequence.');
   v_success := 0;
  RETURN NULL;
END get_allowable_value;
/******************************************************************************************************/
/******************************************************************************************************/
FUNCTION get_next_well_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS
   v_next_well_is_nbr NUMBER(8);
BEGIN
   /* get the next avail is nbr to use */
   v_next_well_is_nbr := sim2_export_pkg.get_eisn_number('TSMWELL',p_org_id);

   RETURN v_next_well_is_nbr;
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in get next well is number.');
     v_success := 0;
     RETURN NULL;
END get_next_well_is_number;
/******************************************************************************************************/
FUNCTION get_next_blob_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS

   v_next_blob_is_nbr NUMBER(8);

BEGIN
   /* get the next avail is nbr to use */
   v_next_blob_is_nbr := sim2_export_pkg.get_eisn_number('TSMBLOB',p_org_id);

   RETURN v_next_blob_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in get next blob is number.');
     v_success := 0;
     RETURN NULL;

END get_next_blob_is_number;
/******************************************************************************************************/
FUNCTION get_next_intvl_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS
   v_next_intvl_is_nbr NUMBER(8);
BEGIN
   /* get the next avail is nbr to use */
   v_next_intvl_is_nbr := sim2_export_pkg.get_eisn_number('TSMINTVL',p_org_id);

   RETURN v_next_intvl_is_nbr;
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in get next intvl is number.');
     v_success := 0;
     RETURN NULL;
END get_next_intvl_is_number;
/******************************************************************************************************/
/******************************************************************************************************/
FUNCTION get_next_igua_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS
   v_next_igua_is_number NUMBER(8);
BEGIN
   /* get the next avail is nbr to use */
   v_next_igua_is_number := sim2_export_pkg.get_eisn_number('TSMIGUA',p_org_id);

   RETURN v_next_igua_is_number;
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in get next igua is number.');
     v_success := 0;
     RETURN NULL;
END get_next_igua_is_number;
/******************************************************************************************************/
FUNCTION GET_UOM_IS_NBR (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2, p_chdef_is_number IN NUMBER)
   RETURN NUMBER IS
   TYPE t_uom_record IS RECORD (
      uom_is_nbr        NUMBER(8),
      uom_name             VARCHAR2(10));
   v_uom_info              t_uom_record;
   e_unk_uom               EXCEPTION;
   e_uom_sp                EXCEPTION;
   CURSOR c_get_is_nbr (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2) IS
      SELECT TSRUOM_IS_NUMBER, SHORT_FORM_NAME FROM TSRUOM WHERE TSRUOM_ORG_ID = RPAD(p_org_id,8) AND
      UPPER(SHORT_FORM_NAME) = UPPER(RPAD(p_uom_name,10));
BEGIN
   OPEN c_get_is_nbr(p_org_id, p_uom_name);
   FETCH c_get_is_nbr INTO v_uom_info;
   IF c_get_is_nbr%NOTFOUND THEN
      v_uom_info.uom_is_nbr := NULL;
      RAISE e_unk_uom;
   ELSE
      IF v_uom_info.uom_name != p_uom_name THEN
         RAISE e_uom_sp;
      END IF;
   END IF;
   CLOSE c_get_is_nbr;
   RETURN v_uom_info.uom_is_nbr;
EXCEPTION
   WHEN e_unk_uom THEN
     sim2_export_pkg.write_error(v_export_log_seq, 'Unknown Unit of Measure: '||p_uom_name);
     v_success := 0;
     RETURN v_uom_info.uom_is_nbr;
   WHEN e_uom_sp THEN
     sim2_export_pkg.write_error(v_export_log_seq, 'Fix uom spelling in TSRCHDEF - is_number:'||p_chdef_is_number||', unit of measure misspelled: '||p_uom_name);
     v_success := 0;
     RETURN v_uom_info.uom_is_nbr;
   WHEN OTHERS THEN
     sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||' in get uom is nbr');
     v_success := 0;
END GET_UOM_IS_NBR;

/******************************************************************************************************/
FUNCTION get_next_station_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_station_is_nbr     NUMBER(8);
BEGIN
   v_station_is_nbr := sim2_export_pkg.get_eisn_number('TSMSTATN',p_org_id);

   RETURN v_station_is_nbr ;
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' in get next station is nbr');
    v_success := 0;
END GET_NEXT_STATION_IS_NBR;

/******************************************************************************************************/
FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmalp_is_nbr     NUMBER(8);
BEGIN
   v_tsmalp_is_nbr := sim2_export_pkg.get_eisn_number('TSMALP',p_org_id);

   RETURN v_tsmalp_is_nbr ;
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' in get next tsmalp is nbr');
    v_success := 0;
END get_next_tsmalp_is_nbr;

/******************************************************************************************************/
PROCEDURE SIM2_EXPORT_STATIONS( p_imp_log_seq IN NUMBER, p_org_id IN VARCHAR2,
          p_success IN OUT NUMBER) IS

   v_export_log_seq    NUMBER(10);
   v_station_is_nbr    NUMBER(8);
   v_tsmalp_is_nbr     NUMBER(8);
   v_tsmwell_is_nbr    NUMBER(8);
   v_tsmwell_org_id    VARCHAR2(8);
   v_well_is_number    NUMBER(8);
   v_station_cnt       NUMBER(10) := 0;       /* ADDED SNG 4/18/01 */
   v_stl_seq           NUMBER(10);
   v_tsmnal_org_id     VARCHAR2(8);
   v_ocean             SIM_OCEAN%ROWTYPE;
   v_estuary           SIM_ESTUARY_LOC%ROWTYPE;
   v_lake              SIM_GREAT_LAKE%ROWTYPE;
   v_well              SIM_STATION_WELLS%ROWTYPE;
   v_import_type       VARCHAR2(4);
   v_blob_success      NUMBER;

   CURSOR c_stations (p_org_id IN VARCHAR2) IS
     SELECT *
     FROM  SIM_STATIONS
     WHERE RPAD(STA_ORG_ID,8) = RPAD(p_org_id,8)
     AND   STA_STATUS = 'A'
     AND STA_IMPORT_SEQ = p_imp_log_seq;

   CURSOR c_station_locations (p_sta_seq IN NUMBER) IS
     SELECT *
     FROM SIM_STATION_LOCATIONS
     WHERE  STL_STA_SEQ = p_sta_seq;

   CURSOR c_ocean(p_stl_seq IN NUMBER) IS
     SELECT
       SOC_SEQ,SOC_STL_SEQ,SOC_NAME,SOC_ADDITIONAL_LOC,SOC_SHORE_RELATION,
       SOC_SHORE_DISTANCE,SOC_SHORE_DIST_UNIT,SOC_REFERENCE_PT,SOC_LORAN_C_READING_1,
       SOC_LORAN_C_READING_2,SOC_BOTTOM_TOPOGRAPHY,SOC_TSMALP_IS_NUMBER,SOC_TSMALP_ORG_ID
     FROM SIM_OCEAN
     WHERE SOC_STL_SEQ = p_stl_seq;

  CURSOR c_estuary(p_stl_seq IN NUMBER) IS
    SELECT
       SEL_SEQ,SEL_TSMALP_IS_NUMBER,SEL_TSMALP_ORG_ID,SEL_OTHER_ESTUARY,SEL_ADD_LOC_NAME,
       SEL_SHORE_DIST,SEL_SHORE_DIST_UNIT,SEL_REF_PT,SEL_TSMESTRY_ORG_ID,SEL_TSMESTRY_IS_NUMBER,
       SEL_STL_SEQ
     FROM SIM_ESTUARY_LOC
     WHERE SEL_STL_SEQ = p_stl_seq;

   CURSOR c_lake(p_stl_seq IN NUMBER) IS
     SELECT
       SGL_SEQ,SGL_STL_SEQ,SGL_TSMALP_IS_NUMBER,SGL_TSMALP_ORG_ID,SGL_LAKE_NAME,
       SGL_ADD_LAKE_NAME,SGL_SHORE_DIST,SGL_SHORE_DIST_UNIT,SGL_REF_PT
     FROM SIM_GREAT_LAKE
     WHERE SGL_STL_SEQ = p_stl_seq;

   CURSOR c_well(p_sta_seq IN NUMBER) IS
     SELECT *
     FROM SIM_STATION_WELLS
     WHERE SSW_STA_SEQ = p_sta_seq;

   CURSOR c_import_type IS
     SELECT SIL_IMPORT_TYPE
     FROM SIM_IMPORT_LOG
     WHERE SIL_SEQ = p_imp_log_seq;

  BEGIN
    v_success := 1;

    BEGIN
    /* set up export log */

     OPEN c_import_type;
       FETCH c_import_type INTO v_import_type;
     CLOSE c_import_type;

      v_export_log_seq := sim2_export_pkg.GET_EXPORT_LOG_SEQ;
      INSERT INTO sim_export_log (SEL_SEQ, SEL_EXPORTED_BY, SEL_EXPORTED_ON,
                                  SEL_FILENAME, SEL_ORG_ID, sel_export_type)
         VALUES (v_export_log_seq, USER, SYSDATE, 'Stations '||p_imp_log_seq, p_org_id, v_import_type);
       COMMIT;
       p_success := v_success;

    EXCEPTION
      WHEN OTHERS THEN
         sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - setting up export log record.');
         v_success := 0;
         p_success := v_success;
    END;
    sim2_export_pkg.write_error(v_export_log_seq, 'Start at: '||to_char(sysdate,'HH24:MI:SS'));

    FOR v_station IN c_stations (p_org_id) LOOP
    --Default values
    --For TSMSTATN
      IF v_station.STA_ESTAB_DATE IS NULL THEN
        v_station.STA_ESTAB_DATE := TO_DATE('01-JAN-0001','DD-MON-YYYY');
      END IF;
      IF v_station.STA_WATER_DEPTH IS NULL THEN
        v_station.STA_WATER_DEPTH := 0;
      END IF;

      BEGIN
        v_station_is_nbr := get_next_station_is_nbr(p_org_id);
        INSERT INTO TSMSTATN (
            TSMSTATN_IS_NUMBER,
            TSMSTATN_ORG_ID,
            IDENTIFICATION_CD,
            EPA_KEY_IDENTIFIER,
            NAME,
            ESTABLISHMENT_DATE,
            ZID_RELATION_CODE,
            INFLUENCE_AREA,
            DESCRIPTION_TEXT,
            TRAVEL_DIR_TXT,
            WATER_DEPTH,
            WATER_DEPTH_UNIT,
            ECOREGION_NAME,
            D_COMPLETE_FLAG,
            D_DELETE_FLAG,
            D_LAST_STN_UPDT_TS,
            D_USERID_CODE,
            D_LAST_UPDT_TS,
            TSMORGAN_IS_NUMBER,
            TSMVSTC_IS_NUMBER,
            TSMVSTC_ORG_ID,
            BLOB_TYPE,
            BLOB_TITLE)
        VALUES (
            v_station_is_nbr                   /* TSMSTATN_IS_NUMBER */,
            RPAD(p_org_id,8)                   /* TSMSTATN_ORG_ID */,
            RPAD(v_station.sta_id,15)          /* IDENTIFICATION_CD */,
            v_station.STA_EPA_KEY_ID           /* EPA_KEY_IDENTIFIER */,
            v_station.STA_NAME                 /* NAME */,
            v_station.STA_ESTAB_DATE           /* ESTABLISHMENT_DATE */,
            v_station.STA_ZID_REL              /* ZID_RELATION_CODE */,
            v_station.STA_INFLUENCE_AREA             /* INFLUENCE_AREA */,
            v_station.STA_DESC                 /* DESCRIPTION_TEXT */,
            v_station.STA_TRAVEL_DIR           /* TRAVEL_DIR_TXT */,
            v_station.STA_WATER_DEPTH          /* WATER_DEPTH */,
            v_station.STA_WATER_DEPTH_UNIT     /* WATER_DEPTH_UNIT */,
            v_station.STA_ECOREGION            /* ECOREGION_NAME */,
            'Y'                                /* D_COMPLETE_FLAG */,
            'N'                                /* D_DELETE_FLAG */,
            SYSDATE                            /* D_LAST_STN_UPDT_TS */,
            sim2_export_pkg.v_pkg_ownerid,
            SYSDATE                            /* D_LAST_UPDT_TS */,
            v_station.STA_TSMORGAN_IS_NUMBER   /* TSMORGAN_IS_NUMBER */,
            v_station.STA_TSMVSTC_IS_NUMBER    /* TSMVSTC_IS_NUMBER */,
            'ZZZZZZZZ'                         /* TSMVSTC_ORG_ID */,
            sim2_export_pkg.get_file_type(v_station.sta_document),
            sim2_export_pkg.get_file_name(v_station.sta_document)
            );

        -- Facilities - if it's a facility must update the facility table
        IF v_station.STA_TSMVSTC_IS_NUMBER > 21 AND v_station.STA_TSMVSTC_IS_NUMBER < 27 THEN
          INSERT INTO TSMFCLTY(
            TSMSTATN_IS_NUMBER,
            TSMSTATN_ORG_ID,
            D_USERID_CODE,
            D_LAST_UPDT_TS)
          VALUES(
            v_station_is_nbr,
            RPAD(p_org_id,8),
            sim2_export_pkg.v_pkg_ownerid,
            SYSDATE);
        END IF;


        v_station_cnt := v_station_cnt + 1;
        v_well_is_number := NULL;
        -- Wells
        BEGIN
        --There will only be one well on a station import...but it has its own ALP record
          OPEN c_well(v_station.STA_SEQ);
            FETCH c_well INTO v_well;
            IF c_well%FOUND THEN
              v_well_is_number := get_next_well_is_number(RPAD(p_org_id,8));
              INSERT INTO TSMWELL (
                                 TSMWELL_IS_NUMBER,
                                 TSMWELL_ORG_ID,
                                 NAME,
                                 ID_NUMBER,
                                 USE_CODE,
                                 PRTCTN_AREA_NAME,
                                 DEVELOPMNT_MTHD_CD,
                                 CONSTRUCTN_MTHD_CD,
                                 WTR_PRIMRY_USE_CD,
                                 GRADIENT_TYPE_CODE,
                                 DEPTH_CMPLTN_MSR,
                                 DPTH_CMPLTN_UNT_CD,
                                 DEPTH_OF_HOLE_MSR,
                                 DEPTH_HOLE_UNT_CD,
                                 WELL_HEAD_HT_MSR,
                                 WL_HD_HT_UNIT_CD,
                                 CASING_HGT_MSR,
                                 CASING_HGT_UNIT_CD,
                                 DISINFECTED_IND_CD,
                                 CONSTRUCTN_STRT_DT,
                                 CONSTRUCTN_END_DT,
                                 INIT_PUMP_DUR,
                                 INT_PMP_DUR_UNT_CD,
                                 INIT_PUMP_RATE,
                                 INT_PMP_RAT_UNT_CD,
                                 NATURL_FLOW_IND_CD,
                                 STATUS_CODE,
                                 INIT_BOREHOLE_DIAM,
                                 INIT_BHOLE_DIAM_UN,
                                 D_USERID_CODE,
                                 D_LAST_UPDT_TS,
                                 TSMSTATN_IS_NUMBER,
                                 TSMSTATN_ORG_ID)
              VALUES            (
                                v_well_is_number,                    --TSMWELL_IS_NUMBER,
                                RPAD(p_org_id,8),                 --TSMWELL_ORG_ID,
                                v_well.SSW_NAME,                  --NAME,
                                v_well.SSW_WELL_NO,                --ID_NUMBER,
                                v_well.SSW_WELL_USE,               --USE_CODE,
                                NULL,                             --PRTCTN_AREA_NAME,
                                NULL,                             --DEVELOPMNT_MTHD_CD,
                                NULL,                             --CONSTRUCTN_MTHD_CD,
                                NULL,                             --WTR_PRIMRY_USE_CD,
                                NULL,                             --GRADIENT_TYPE_CODE,
                                NULL,                             --DEPTH_CMPLTN_MSR,
                                NULL,                             --DPTH_CMPLTN_UNT_CD,
                                NULL,                             --DEPTH_OF_HOLE_MSR,
                                NULL,                             --DEPTH_HOLE_UNT_CD,
                                NULL,                             --WELL_HEAD_HT_MSR,
                                NULL,                             --WL_HD_HT_UNIT_CD,
                                NULL,                             --CASING_HGT_MSR,
                                NULL,                             --CASING_HGT_UNIT_CD,
                                NULL,                             --DISINFECTED_IND_CD,
                                NULL,                             --CONSTRUCTN_STRT_DT,
                                NULL,                             --CONSTRUCTN_END_DT,
                                NULL,                             --INIT_PUMP_DUR,
                                NULL,                             --INT_PMP_DUR_UNT_CD,
                                NULL,                             --INIT_PUMP_RATE,
                                NULL,                             --INT_PMP_RAT_UNT_CD,
                                NULL,                             --NATURL_FLOW_IND_CD,
                                v_well.SSW_WELL_STATUS,           --STATUS_CODE,
                                NULL,                             --INIT_BOREHOLE_DIAM,
                                NULL,                             --INIT_BHOLE_DIAM_UN,
                                sim2_export_pkg.v_pkg_ownerid,
                                SYSDATE,                          --D_LAST_UPDT_TS,
                                v_station_is_nbr,                 --TSMSTATN_IS_NUMBER,
                                RPAD(p_org_id,8)                  --TSMSTATN_ORG_ID,
                                );

              UPDATE SIM_STATION_WELLS SET
                SSW_IS_NUMBER = v_well_is_number,
                SSW_ORG_ID = rpad(p_org_id,8),
                SSW_STATUS = 'E'
              WHERE
                SSW_SEQ = v_well.ssw_seq;

        --  END LOOP;
             
            END IF;
          CLOSE c_well;

        EXCEPTION
            WHEN OTHERS THEN
               sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||'migrating well');
               v_success := 0;
               p_success := v_success;
        END;

            /* STATION LOCATIONS */
          FOR v_location in c_station_locations (v_station.sta_seq) LOOP

         /* kms - gets the method, datum, and units for elevation based on foreign keys in the
                station locations table if elevation is present.  Otherwise, these are set to null */

            IF v_location.STL_ELEVATION_MSR IS NULL THEN
              v_location.STL_ELEVATION_MSR := 0; --Default NULL Value
            END IF;

            v_stl_seq := v_location.STL_SEQ;
            IF v_location.STL_TSMNAL_CD IS NOT NULL THEN
              v_tsmnal_org_id := 'ZZZZZZZZ';
            ELSE
              v_tsmnal_org_id := NULL;
            END IF;
            IF v_well_is_number IS NOT NULL AND v_location.STL_TYPE_CODE = 'WELL HEAD' THEN
              v_tsmwell_org_id := p_org_id;
              v_tsmwell_is_nbr := v_well_is_number;
            ELSE
              v_tsmwell_is_nbr := NULL;
              v_tsmwell_org_id := NULL;
            END IF;

            BEGIN
              v_tsmalp_is_nbr := get_next_tsmalp_is_nbr(RPAD(p_org_id,8));

              INSERT INTO TSMALP (
                TSMALP_IS_NUMBER,
                TSMALP_ORG_ID,
                TYPE_CODE,
                SEQUENCE_NUMBER,
                POINT_NAME,
                ELEVATION_DATUM_CD,
                ELVTN_METHOD_CD,
                ELVTN_MSR_DT,
                ELEVATION_MSR,
                ELVTN_UNT_CD,
                GEOPSTNG_DATUM_CD,
                GEOPSTNG_METHOD_CD,
                GEOPSTNG_SCALE_TXT,
                LAT_DIRECTION,
                LAT_DEGREE_MSR,
                GPS_LAT_DEGREE_MSR,
                GPS_LAT_MINUTE_MSR,
                GPS_LONG_DEG_MSR,
                GPS_LONG_MIN_MSR,
                LAT_MINUTE_MSR,
                LAT_SECOND_MSR,
                LONG_DIRECTION,
                LONG_DEGREE_MSR,
                LONG_MINUTE_MSR,
                LONG_SECOND_MSR,
                LAT_LONG_MSR_DT,
                LAT_DEC_DEG_MSR,
                LONG_DEC_DEG_MSR,
                RF3_RIVER_REACH_CD,
                RF1_MILEAGE,
                ON_RIVER_REACH_IND,
                NRCS_WTRSD_ID_NUM,
                D_USERID_CODE,
                D_LAST_UPDT_TS,
                TSMSTATN_ORG_ID,
                TSMSTATN_IS_NUMBER,
                TSMRRR_IS_NUMBER,
                TSMRRR_ORG_ID,
                TSMFHU_IS_NUMBER,
                TSMFHU_ORG_ID,
                TSMSTATN0IS_NUMBER,
                TSMSTATN0ORG_ID,
                TSMPIPE_IS_NUMBER,
                TSMPIPE_ORG_ID,
                TSMWELL_IS_NUMBER,
                TSMWELL_ORG_ID,
                TSMSTATN1ORG_ID,
                TSMSTATN1IS_NUMBER,
                TSMGEOPA_IS_NUMBER,
                TSMGEOPA_ORG_ID,
                TSMGEOPA0IS_NUMBER,
                TSMGEOPA0ORG_ID,
                TSMNAL_CD,
                TSMNAL_STATE,
                TSMNAL_ORG_ID,
                HORIZ_ACCURACY_MSR,
                HORIZ_ACCUR_UNIT,
                VERT_ACCURACY_MSR,
                VERT_ACCURACY_UNIT
                )
              VALUES (
                     v_tsmalp_is_nbr                            /* TSMALP_IS_NUMBER */,
                     RPAD(p_org_id,8)                           /* TSMALP_ORG_ID */,
                     RPAD(v_location.STL_TYPE_CODE,16)          /* TYPE_CODE */,
                     v_location.stl_sequence_number             /* SEQUENCE_NUMBER */,
                     v_location.stl_point_name,                 /*POINT NAME */
                     v_location.stl_elev_datum                  /* ELEVATION_DATUM_CD */,
                     v_location.stl_elev_method                 /* ELVTN_METHOD_CD */,
                     v_location.STL_ELEV_MSR_DATE               /* ELVTN_MSR_DATE */,
                     TO_NUMBER(v_location.STL_ELEVATION_MSR)    /* ELEVATION_MSR */,
                     v_location.stl_elevation_unit              /* ELVTN_UNT_CD */,
                     RPAD(v_location.stl_geopos_datum,12)        /* GEOPSTNG_DATUM_CD */,
                     RPAD(v_location.stl_geopos_method,12)       /* GEOPSTNG_METHOD_CD */,
                     v_location.STL_GPS_SCALE                   /* GEOPSTNG_SCALE_TXT */,
                     v_location.STL_LAT_DIR                     /* LAT_DIRECTION */,
                     v_location.STL_LAT_DEGREES                 /* LAT_DEGREE_MSR */,
                     v_location.STL_LAT_DEGREES                 /* GPS_LAT_DEGREE_MSR */,
                     v_location.STL_LAT_DEC_MINUTES             /* GPS_LAT_MINUTE_MSR */,
                     v_location.STL_LONG_DEGREES                /* GPS_LONG_DEG_MSR */,
                     v_location.STL_LONG_DEC_MINUTES            /* GPS_LONG_MIN_MSR */,
                     v_location.STL_LAT_MINUTES                 /* LAT_MINUTE_MSR */,
                     v_location.STL_LAT_SECONDS                 /* LAT_SECOND_MSR */,
                     v_location.STL_LONG_DIR                    /* LONG_DIRECTION */,
                     v_location.STL_LONG_DEGREES                /* LONG_DEGREE_MSR */,
                     v_location.STL_LONG_MINUTES                /* LONG_MINUTE_MSR */,
                     v_location.STL_LONG_SECONDS                /* LONG_SECOND_MSR */,
                     v_location.STL_LAT_LONG_MSR_DATE           /* LAT_LONG_MSR_DT */,
                     v_location.STL_LAT_DEC_DEGREES,            /* LAT_DEC_DEG_MSR */
                     v_location.STL_LONG_DEC_DEGREES,            /* LONG_DEC_DEG_MSR */
                     --NULL                                       /* LATLONG_ACCRCY_MSR */,
                     --NULL                                       /* LATLONG_ACCRCY_UNT */,
                     --NULL                                       /* LONG_ACCRCY_MSR */,
                     --NULL                                       /* LONG_ACCRCY_UNT_CD */,
                     NULL                                       /* RF3_RIVER_REACH_CD */,
                     0                                          /* RF1_MILEAGE */,
                     'N'                                        /* ON_RIVER_REACH_IND */,
                     v_location.STL_NRCS                        /* NRCS_WTRSD_ID_NUM */,
                     sim2_export_pkg.v_pkg_ownerid,
                     SYSDATE                                    /* D_LAST_UPDT_TS */,
                     NULL                                       /* TSMSTATN_ORG_ID */,
                     NULL                                       /* TSMSTATN_IS_NUMBER */,
                     NULL                                       /* TSMRRR_IS_NUMBER */,
                     NULL                                       /* TSMRRR_ORG_ID */,
                     v_location.STL_TSMFHU_IS_NUMBER            /* TSMFHU_IS_NUMBER */,
                     RPAD(v_location.STL_TSMFHU_ORG_ID,8)       /* TSMFHU_ORG_ID */,
                     v_station_is_nbr                           /* TSMSTATN0IS_NUMBER */,
                     RPAD(p_org_id,8)                           /* TSMSTATN0ORG_ID */,
                     NULL                                       /* TSMPIPE_IS_NUMBER */,
                     NULL                                       /* TSMPIPE_ORG_ID */,
                     v_tsmwell_is_nbr                           /* TSMWELL_IS_NUMBER */,
                     RPAD(v_tsmwell_org_id,8)                   /* TSMWELL_ORG_ID */,
                     NULL                                       /* TSMSTATN1ORG_ID */,
                     NULL                                       /* TSMSTATN1IS_NUMBER */,
                     v_location.STL_TSMGEOPA_IS_NUMBER          /* TSMGEOPA_IS_NUMBER */,
                     RPAD(v_location.STL_TSMGEOPA_ORG_ID,8)     /* TSMGEOPA_ORG_ID */,
                     NULL,                                      /* TSMGEOPA0IS_NUMBER */
                     NULL,                                      /* TSMGEOPA0ORG_ID */
                     v_location.STL_TSMNAL_CD                   /*TSMNAL_CD */,
                     v_location.STL_TSMNAL_STATE                /*TSMNAL_STATE */,
                     v_tsmnal_org_id,                            /*TSMNAL_ORG_ID */
                     NULL,                                       /*HORIZ_ACCURACY_MSR*/
                     NULL,                                       /*HORIZ_ACCUR_UNIT*/
                     NULL,                                       /*VERT_ACCURACY_MSR*/
                     NULL                                        /*VERT_ACCURACY_UNIT*/
                    );
          
          EXCEPTION
            WHEN OTHERS THEN
              sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||'migrating sta location');
              v_success := 0;
              p_success := v_success;
          END;

          --sf update the record in sim_station_locations to include is numbers
          UPDATE SIM_STATION_LOCATIONS
            SET STL_IS_NUMBER = v_tsmalp_is_nbr,
                STL_ORG_ID = p_org_id,
                STL_TSMSTATN0IS_NUMBER = v_station_is_nbr,
                STL_TSMSTATN0ORG_ID = p_org_id
            WHERE STL_SEQ = v_location.stl_seq;

          -- sfp (12/04/01) The following inserts record for TSMOCNLC if it is an Ocean station
          BEGIN
            OPEN c_ocean(v_stl_seq);
              FETCH c_ocean INTO v_ocean;
              IF c_ocean%FOUND THEN

                IF v_ocean.SOC_SHORE_DISTANCE IS NULL THEN
                  v_ocean.SOC_SHORE_DISTANCE := 0; -- Default NULL value
                END IF;

                INSERT INTO TSMOCNLC (
                 TSMALP_IS_NUMBER,
                 TSMALP_ORG_ID,
                 NAME,
                 SHORE_RELATION,
                 LORAN_C_READING_1,
                 LORAN_C_READING_2,
                 BOTTOM_TOPOGRAPHY,
                 ADDTNL_LOC_NAME,
                 SHORE_DISTANCE,
                 SHORE_DIST_UNIT_CD,
                 REFERENCE_PT,
                 D_USERID_CODE,
                 D_LAST_UPDT_TS)
                VALUES (
                     v_tsmalp_is_nbr,                            /* TSMALP_IS_NUMBER */
                     RPAD(p_org_id,8),                           /* TSMALP_ORG_ID */
                     v_ocean.soc_name,                           --NAME,
                     v_ocean.soc_shore_relation,                 --SHORE_RELATION
                     NULL,                                       --LORAN_C_READING_1
                     NULL,                                       --LORAN_C_READING_2
                     v_ocean.SOC_BOTTOM_TOPOGRAPHY,               --BOTTOM_TOPOGRAPHY
                     v_ocean.SOC_ADDITIONAL_LOC,                     --ADDTNL_LOC_NAME
                     v_ocean.SOC_SHORE_DISTANCE,                     --SHORE_DISTANCE
                     v_ocean.SOC_SHORE_DIST_UNIT,                    --SHORE_DIST_UNIT_CD
                     v_ocean.SOC_REFERENCE_PT,                       --REFERENCE_PT
                     sim2_export_pkg.v_pkg_ownerid,
                     SYSDATE                                     --D_LAST_UPDT_TS
                     );
              END IF;
            CLOSE c_ocean;
          EXCEPTION
            WHEN OTHERS THEN
              sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||'migrating ocean');
              v_success := 0;
              p_success := v_success;
          END;


          BEGIN
            OPEN c_estuary(v_stl_seq);
              FETCH c_estuary INTO v_estuary;
              IF c_estuary%FOUND THEN
                IF v_estuary.SEL_SHORE_DIST IS NULL THEN
                  v_estuary.SEL_SHORE_DIST := 0; --Default NULL value
                END IF;

                INSERT INTO TSMESTLC (
                 TSMALP_IS_NUMBER,
                 TSMALP_ORG_ID,
                 OTHER_ESTUARY_NAME,
                 ADDTNL_LOC_NAME,
                 SHORE_DISTANCE,
                 SHORE_DIST_UNIT_CD,
                 REFERENCE_PT,
                 D_USERID_CODE,
                 D_LAST_UPDT_TS,
                 TSMESTRY_ORG_ID,
                 TSMESTRY_IS_NUMBER
                 )
                VALUES
                 (
                 v_tsmalp_is_nbr,                     --TSMALP_IS_NUMBER
                 RPAD(p_org_id,8),                    --TSMALP_ORG_ID
                 v_estuary.SEL_OTHER_ESTUARY,         --OTHER_ESTUARY_NAME
                 v_estuary.SEL_ADD_LOC_NAME,          --ADDTNL_LOC_NAME
                 v_estuary.SEL_SHORE_DIST,            --SHORE_DISTANCE
                 v_estuary.SEL_SHORE_DIST_UNIT,       --SHORE_DIST_UNIT_CD
                 v_estuary.SEL_REF_PT,                --REFERENCE_PT
                 sim2_export_pkg.v_pkg_ownerid, 
                 SYSDATE,                             --D_LAST_UPDT_TS
                 'ZZZZZZZZ',                          --TSMESTRY_ORG_ID
                 v_estuary.SEL_TSMESTRY_IS_NUMBER     --TSMESTRY_IS_NUMBER
                );
              END IF;
            CLOSE c_estuary;
          EXCEPTION
            WHEN OTHERS THEN
              sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||'migrating estuary');
              v_success := 0;
              p_success := v_success;
          END;

          BEGIN
            OPEN c_lake(v_stl_seq);
              FETCH c_lake INTO v_lake;
              IF c_lake%FOUND THEN
                IF v_lake.SGL_SHORE_DIST IS NULL THEN
                  v_lake.SGL_SHORE_DIST := 0; -- Default NULL Value
                END IF;

                INSERT INTO TSMGLL (
                   TSMALP_IS_NUMBER,
                   TSMALP_ORG_ID,
                   NAME,
                   ADDTNL_LOC_NAME,
                   SHORE_DISTANCE,
                   SHORE_DIST_UNIT_CD,
                   REFERENCE_PT,
                   D_USERID_CODE,
                   D_LAST_UPDT_TS
                   )
                 VALUES
                   (
                   v_tsmalp_is_nbr,
                   RPAD(p_org_id,8),
                   v_lake.SGL_LAKE_NAME,
                   v_lake.SGL_ADD_LAKE_NAME,
                   v_lake.SGL_SHORE_DIST,
                   v_lake.SGL_SHORE_DIST_UNIT,
                   v_lake.SGL_REF_PT,
                   sim2_export_pkg.v_pkg_ownerid,
                   SYSDATE
                   );
               END IF;
             CLOSE c_lake;

          EXCEPTION
            WHEN OTHERS THEN
             sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||'migrating great lake');
             v_success := 0;
             p_success := v_success;
          END;

          /* export binary document */
          IF v_station.sta_document IS NOT NULL THEN
            v_blob_success := insert_binary_document(p_org_id, v_station_is_nbr, v_station.sta_document);
            IF v_blob_success = 0 THEN
               sim2_export_pkg.write_error(v_export_log_seq, 'Unable to load Document/Graphic file ' || '"' || v_station.sta_document || '".  Please check directory and file permissions.');
               v_success := 0;
               p_success := v_success;
            END IF;
          END IF;
          
        END LOOP; --end of locations loop

        /* update station with export seq */
        UPDATE SIM_STATIONS SET
            STA_EXPORT_SEQ = v_export_log_seq,
            STA_IS_NUMBER = v_station_is_nbr,
            STA_STATUS = 'E'
        WHERE
            STA_SEQ = v_station.sta_seq;
        COMMIT;


      EXCEPTION
         WHEN OTHERS THEN
         sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||' exporting station');
         v_success := 0;
         p_success := v_success;
      END;

    END LOOP;

    /* update export log with count - added SNG 4/18/01*/
    UPDATE SIM_EXPORT_LOG SET SEL_ACTIVITY_CNT = v_station_cnt
      WHERE SEL_SEQ = v_export_log_seq;
      /* update the import log with export info
      SNG 4/16/01 - added DECREMENT of sil_rows_ready */
    UPDATE SIM_IMPORT_LOG SET
      SIL_EXPORTED_ON = SYSDATE,
      SIL_EXPORTED_BY = USER,
      SIL_ROWS_READY = SIL_ROWS_READY - v_station_cnt
    WHERE SIL_SEQ = P_IMP_LOG_SEQ;
    
    /* all done! */
    sim2_export_pkg.write_error(v_export_log_seq, 'Migrated '||v_station_cnt||' stations.');
    sim2_export_pkg.write_error(v_export_log_seq, 'Finish at: '||to_char(sysdate,'HH24:MI:SS'));
    COMMIT;
    p_success := v_success;

  EXCEPTION
    WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM ||'IN sim2 export stations');
    v_success := 0;
    p_success := v_success;
  END SIM2_EXPORT_STATIONS;
/******************************************************************************************************/

/******************************************************************************************************/
FUNCTION insert_binary_document(p_org_id IN VARCHAR2, p_station_is_nbr IN NUMBER, p_file_name IN VARCHAR2) RETURN NUMBER IS

  v_is_number   NUMBER(8);
  v_bfile       BFILE;
  v_blob        BLOB;
  v_file_size   INTEGER;
  
BEGIN

  BEGIN
    v_bfile := bfilename('DOCUMENT_DIR', p_file_name);
    dbms_lob.fileopen(v_bfile, dbms_lob.file_readonly);
    v_file_size := dbms_lob.getlength(v_bfile);

  v_is_number := get_next_blob_is_number(p_org_id);

  INSERT INTO TSMBLOB (
    TSMBLOB_IS_NUMBER,
    TSMBLOB_ORG_ID,
    TABLE_NAME,
    TSMSTATN_IS_NUMBER,
    TSMSTATN_ORG_ID,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    BLOB_CONTENT)
  VALUES(
    v_is_number,                             
    RPAD(p_org_id,8),                        
    'TSMSTATN',                              
    p_station_is_nbr,                        
    RPAD(p_org_id,8),                        
    sim2_export_pkg.v_pkg_ownerid,
    SYSDATE,                                 
    EMPTY_BLOB())
  RETURN BLOB_CONTENT INTO v_blob;

    dbms_lob.loadfromfile(v_blob, v_bfile, v_file_size);
    dbms_lob.fileclose(v_bfile);

  EXCEPTION
    WHEN OTHERS THEN
      RETURN 0;
  END;

  RETURN 1;
    
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;

END insert_binary_document;

END sim2_export_station_pkg;
/
