CREATE OR REPLACE PACKAGE SIM2_IMPORT_PKG IS

	/* SIM200 */
	/* Modified 3/21/02 by SF. Added code at the end of Delete_import that looks for all the trips
  and station visits in SIM and delets them if they dont have any activities associated with them. */

	/* import utils by SNG Feb 2000 */

	TYPE COL_TABLE_TYPE IS TABLE OF VARCHAR2(2000) INDEX BY BINARY_INTEGER;

	TYPE T_ACTIVITY_COUNT IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;

	V_ACTIVITY_COUNT T_ACTIVITY_COUNT;

	PROCEDURE PARSE_LINE_TABLE(PV_LINE          IN LONG,
														 P_COL_TABLE      OUT COL_TABLE_TYPE,
														 P_NO_COLS        IN NUMBER,
														 P_DELIMITER      IN VARCHAR2 DEFAULT ',',
														 P_BAD_PARSE      IN OUT BOOLEAN,
														 P_IMPORT_LOG_SEQ IN NUMBER);

	FUNCTION GET_ONE_COL(PV_LINE          IN LONG,
											 P_COL_NO         IN NUMBER,
											 P_DELIMITER      IN VARCHAR2,
											 P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2;

	PROCEDURE WRITE_ERROR(P_IMPORT_LOG_SEQ IN NUMBER,
												P_DESC           IN VARCHAR2,
												P_IMPORT_SEQ     IN NUMBER);

	PROCEDURE DELETE_IMPORT(P_IMPORT_SEQ IN NUMBER);

	FUNCTION GET_IMPORT_LOG_SEQ RETURN NUMBER;

	PROCEDURE UPDATE_SIL_ROWS_READY(P_IMPORT_LOG_SEQ    IN NUMBER,
																	P_ROWS_READY_CHANGE IN NUMBER);

	PROCEDURE UPDATE_SIL_ROWS_ERRORS(P_IMPORT_LOG_SEQ IN NUMBER,
																	 P_ROWS_ERRORS    IN NUMBER);

	PROCEDURE UPDATE_SIL_ROWS_IMPORTED(P_IMPORT_LOG_SEQ IN NUMBER,
																		 P_ROWS_IMPORTED  IN NUMBER);

	FUNCTION IND_NBR_SORT(P_SEQ NUMBER) RETURN VARCHAR2;

	FUNCTION SIM200_READ_TEXT_FILE(P_DFLT_PATH      IN VARCHAR2,
																 P_FILE_NAME      IN OUT VARCHAR2,
																 P_IMP_TYPE       IN VARCHAR2,
																 P_ORG_ID         IN OUT VARCHAR2,
																 P_IMP_CFG_SEQ    IN NUMBER,
																 P_IMPORT_LOG_SEQ IN OUT NUMBER)
		RETURN VARCHAR2;

  FUNCTION CHECK_DOCUMENT_FILE(p_file_name IN VARCHAR2, p_import_log_seq IN NUMBER, p_line IN NUMBER) RETURN BOOLEAN;

END;
/
CREATE OR REPLACE PACKAGE BODY SIM2_IMPORT_PKG AS
	/*******************************************************/
	/* PROC TO PARSE one comma delimited line into a table */
	PROCEDURE PARSE_LINE_TABLE(PV_LINE          IN LONG,
														 P_COL_TABLE      OUT COL_TABLE_TYPE,
														 P_NO_COLS        IN NUMBER,
														 P_DELIMITER      IN VARCHAR2 DEFAULT ',',
														 P_BAD_PARSE      IN OUT BOOLEAN,
														 P_IMPORT_LOG_SEQ IN NUMBER) IS
		V_COL_CNT BINARY_INTEGER := 1;
		COL_CNT   INTEGER;
	BEGIN
		-- Get the actual number of columns in the import line
		COL_CNT := LENGTH(PV_LINE) -
							 LENGTH(TRANSLATE(PV_LINE, 'a' || P_DELIMITER, 'a')) + 1;
               
		-- Process the import line
		WHILE V_COL_CNT <= COL_CNT
		LOOP
			P_COL_TABLE(V_COL_CNT) := GET_ONE_COL(PV_LINE,
																						V_COL_CNT,
																						P_DELIMITER,
																						P_IMPORT_LOG_SEQ);
                                     
			V_COL_CNT := V_COL_CNT + 1;
		END LOOP;
	
		IF P_NO_COLS < COL_CNT THEN
			-- More columns found in inport line that in configuration
			SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
																	'Warning: There are more columns in the line than defined in the configuration',
																	NULL);
		ELSIF P_NO_COLS > COL_CNT THEN
			-- Add empty column at the end for the generate columns
			WHILE V_COL_CNT <= P_NO_COLS
			LOOP
				P_COL_TABLE(V_COL_CNT) := NULL;
				V_COL_CNT := V_COL_CNT + 1;
			END LOOP;
		END IF;
	
	EXCEPTION
		WHEN OTHERS THEN
			SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
																	SQLERRM ||
																	'- in parse_line_table function, column ' ||
																	P_NO_COLS,
																	NULL);
	END PARSE_LINE_TABLE;

	/************************************************************************/
	/* Internal function to get each column - used by parse_line_table proc */
	FUNCTION GET_ONE_COL(PV_LINE          IN LONG,
											 P_COL_NO         IN NUMBER,
											 P_DELIMITER      IN VARCHAR2,
											 P_IMPORT_LOG_SEQ IN NUMBER) RETURN VARCHAR2 IS
		V_FIRST_COMMA NUMBER;
		V_START_POS   NUMBER;
		V_LINE_LENGTH NUMBER;
	BEGIN
		V_LINE_LENGTH := LENGTH(PV_LINE);
	
		IF P_COL_NO = 1 THEN
			V_START_POS := 1;
		ELSE
			V_START_POS := INSTR(PV_LINE, P_DELIMITER, 1, P_COL_NO - 1) + 1;
		END IF;
	
		V_FIRST_COMMA := INSTR(PV_LINE, P_DELIMITER, 1, P_COL_NO);
	
		IF V_FIRST_COMMA = 0 THEN
			V_FIRST_COMMA := V_LINE_LENGTH + 1;
		END IF;
	
		RETURN SUBSTR(PV_LINE, V_START_POS, V_FIRST_COMMA - V_START_POS);
	EXCEPTION
		WHEN OTHERS THEN
			SIM2_IMPORT_PKG.WRITE_ERROR(P_IMPORT_LOG_SEQ,
																	SQLERRM ||
																	'- in get_one_col function, column ' ||
																	P_COL_NO,
																	NULL);
	END GET_ONE_COL;

	/******************************************************************************************************/
	PROCEDURE WRITE_ERROR(P_IMPORT_LOG_SEQ IN NUMBER,
												P_DESC           IN VARCHAR2,
												P_IMPORT_SEQ     IN NUMBER) IS
	BEGIN
		INSERT INTO SIM_IMPORT_DETAILS
			(SMD_SEQ,
			 SMD_SIL_SEQ,
			 SMD_DESC,
			 SMD_IMPORT_SEQ)
		VALUES
			(SIM_IMPORT_DETAILS_SEQ.NEXTVAL,
			 P_IMPORT_LOG_SEQ,
			 P_DESC,
			 P_IMPORT_SEQ);
	
		COMMIT;
	END WRITE_ERROR;

	/******************************************************************************************************/
	PROCEDURE DELETE_IMPORT(P_IMPORT_SEQ IN NUMBER) IS
	V_DELETE_CNT   NUMBER(10) := 1;
	V_IMPORT_TYPE  VARCHAR2(30);
	V_LOCATION_SEQ NUMBER(10);
	V_DUMMY_SEQ    NUMBER(10);
	V_VISIT_SEQ    NUMBER(10);
	V_TRIP_SEQ     NUMBER(10);

	CURSOR C_IMPORT_TYPE(P_IMPORT_SEQ IN NUMBER) IS
		SELECT SID_TYPE
			FROM SIM_IMPORT_LOG, SIM_IMP_CFG, SIM_IMP_DEFS
		 WHERE SIL_SEQ = P_IMPORT_SEQ AND SIL_SICC_SEQ = SICC_SEQ AND
					 SICC_SID_SEQ = SID_SEQ;

	CURSOR C_FIELD_ACTIVITIES(P_IMPORT_SEQ IN NUMBER) IS
		SELECT FA_SEQ,
					 FA_SSV_SEQ,
					 FA_STP_SEQ
			FROM SIM_FIELD_ACTIVITIES
		 WHERE FA_IMPORT_SEQ = P_IMPORT_SEQ;

	CURSOR C_GET_VISIT_INFO(P_VISIT_SEQ IN NUMBER) IS
		SELECT SSV_SEQ
			FROM SIM_STATION_VISITS SSV, TSRSTVST T
		 WHERE SSV.SSV_TSRSTVST_IS_NUMBER = T.TSRSTVST_IS_NUMBER AND
					 SSV.SSV_TSRSTVST_ORG_ID = T.TSRSTVST_ORG_ID AND
					 SSV.SSV_SEQ = P_VISIT_SEQ;

	CURSOR C_GET_ALL_TRIPS IS
		SELECT STP_SEQ FROM SIM_TRIPS;

	CURSOR C_GET_ALL_VISITS(P_TRIP_SEQ IN NUMBER) IS
		SELECT SSV_SEQ
			FROM SIM_STATION_VISITS
		 WHERE SSV_STP_SEQ = P_TRIP_SEQ;

	CURSOR C_GET_ALL_ACTIVITIES(P_VISIT_SEQ IN NUMBER, P_TRIP_SEQ IN NUMBER) IS
		SELECT FA_SEQ
			FROM SIM_FIELD_ACTIVITIES
		 WHERE FA_SSV_SEQ = P_VISIT_SEQ AND FA_STP_SEQ = P_TRIP_SEQ;

	CURSOR C_GET_TRIP_INFO(P_TRIP_SEQ IN NUMBER) IS
		SELECT STP_SEQ
			FROM SIM_TRIPS ST, TSRTRIP T
		 WHERE ST.STP_TSRTRIP_IS_NUMBER = T.TSRTRIP_IS_NUMBER AND
					 ST.STP_TSRTRIP_ORG_ID = T.TSRTRIP_ORG_ID AND
					 STP_SEQ = P_TRIP_SEQ;

	CURSOR C_CHECK_VISIT_IN_SIM(P_VISIT_SEQ IN NUMBER) IS
		SELECT FA_SEQ
			FROM SIM_FIELD_ACTIVITIES
		 WHERE FA_SSV_SEQ = P_VISIT_SEQ;

	CURSOR C_CHECK_TRIP_IN_SIM(P_TRIP_SEQ IN NUMBER) IS
		SELECT FA_SEQ
			FROM SIM_FIELD_ACTIVITIES
		 WHERE FA_STP_SEQ = P_TRIP_SEQ;

	CURSOR C_STATIONS(P_IMPORT_SEQ IN NUMBER) IS
		SELECT STA_SEQ FROM SIM_STATIONS WHERE STA_IMPORT_SEQ = P_IMPORT_SEQ;

	CURSOR C_RESULT_STATIONS(P_IMPORT_SEQ IN NUMBER) IS
		SELECT STA_SEQ
			FROM SIM_STATIONS S
		 WHERE S.STA_STATUS = 'U'
     AND NOT EXISTS (SELECT 1 FROM SIM_FIELD_ACTIVITIES F
                     WHERE F.FA_TSMSTATN_IS_NUMBER = S.STA_IS_NUMBER 
                     AND F.FA_TSMSTATN_ORG_ID = S.STA_ORG_ID)
     AND NOT EXISTS (SELECT 1 FROM SIM_STATION_LOCATIONS L
                     WHERE L.STL_STA_SEQ = S.STA_SEQ
                     AND L.STL_STATUS != 'U')
     AND NOT EXISTS (SELECT 1 FROM SIM_STATION_WELLS W
                     WHERE W.SSW_STA_SEQ = S.STA_SEQ
                     AND W.SSW_STATUS != 'U');                   

	CURSOR C_GET_STATION_LOCATION(P_STATION_SEQ IN NUMBER) IS
		SELECT STL_SEQ
			FROM SIM_STATION_LOCATIONS
		 WHERE STL_STA_SEQ = P_STATION_SEQ;

	CURSOR C_WELLS(P_IMPORT_SEQ IN NUMBER) IS
		SELECT SSW_SEQ
			FROM SIM_STATION_WELLS
		 WHERE SSW_IMPORT_SEQ = P_IMPORT_SEQ;

	CURSOR C_CHECK_IMPORT_LINE(P_IMPORT_SEQ IN NUMBER) IS
		SELECT SIPL_SEQ FROM SIM_IMP_LINES WHERE SIPL_SIL_SEQ = P_IMPORT_SEQ;

	CURSOR C_CHECK_IMPORT_DETAIL(P_IMPORT_SEQ IN NUMBER) IS
		SELECT SMD_SEQ
			FROM SIM_IMPORT_DETAILS
		 WHERE SMD_SIL_SEQ = P_IMPORT_SEQ;

BEGIN
	OPEN C_IMPORT_TYPE(P_IMPORT_SEQ);

	FETCH C_IMPORT_TYPE
		INTO V_IMPORT_TYPE;

	CLOSE C_IMPORT_TYPE;

	IF V_IMPORT_TYPE = 'STATIONS' OR V_IMPORT_TYPE = 'WEB STATIONS' OR
	   V_IMPORT_TYPE = 'WELLS' OR V_IMPORT_TYPE = 'INTERVALS' OR
	   V_IMPORT_TYPE = 'LOCATIONS' THEN
  	/*
      DELETING A STATIONS IMPORT  This will also delete a single well and location that came in with
      a station. Multiple wells and locations MIGHT have brought in some existing STORET Stations and
      they, too will be deleted here. 
    */
  	FOR V_STATIONS IN C_STATIONS(P_IMPORT_SEQ)
  	LOOP
  		OPEN C_GET_STATION_LOCATION(V_STATIONS.STA_SEQ);
  	
  		FETCH C_GET_STATION_LOCATION
  			INTO V_LOCATION_SEQ;
  	
  		CLOSE C_GET_STATION_LOCATION;
  	
  		DELETE SIM_OCEAN WHERE SOC_STL_SEQ = V_LOCATION_SEQ;
  	
  		DELETE SIM_ESTUARY_LOC WHERE SEL_STL_SEQ = V_LOCATION_SEQ;
  	
  		DELETE SIM_GREAT_LAKE WHERE SGL_STL_SEQ = V_LOCATION_SEQ;
  	
  		DELETE SIM_STATION_LOCATIONS
  		 WHERE STL_STA_SEQ = V_STATIONS.STA_SEQ;
  	
  		DELETE SIM_STATION_WELLS WHERE SSW_STA_SEQ = V_STATIONS.STA_SEQ;
  	
  		DELETE SIM_STATIONS WHERE STA_SEQ = V_STATIONS.STA_SEQ;
  	
  		IF V_DELETE_CNT > 100 THEN
  			COMMIT;
  			V_DELETE_CNT := 0;
  		END IF;
  	
  		V_DELETE_CNT := V_DELETE_CNT + 2;
  	END LOOP;
  
  	/* Now we need to delete individual wells and locations that have their own import_log_seq. */
  	DELETE SIM_STATION_LOCATIONS WHERE STL_IMPORT_SEQ = P_IMPORT_SEQ;
  
  	COMMIT;
  
  	FOR V_WELLS IN C_WELLS(P_IMPORT_SEQ)
  	LOOP
  		DELETE SIM_STATION_LOCATIONS WHERE STL_SSW_SEQ = V_WELLS.SSW_SEQ;
  	
  		V_DELETE_CNT := V_DELETE_CNT + 1;
  	
  		IF V_DELETE_CNT > 100 THEN
  			COMMIT;
  			V_DELETE_CNT := 0;
  		END IF;
  	END LOOP;
  
  	DELETE SIM_STATION_WELLS WHERE SSW_IMPORT_SEQ = P_IMPORT_SEQ;
  
  	COMMIT;
  ELSIF V_IMPORT_TYPE = 'PROJECTS' THEN
  	/* DELETING A PROJECTS IMPORT */
  	DELETE SIM_PROJECTS WHERE SPJ_IMPORT_SEQ = P_IMPORT_SEQ;
  
  	COMMIT;
  ELSE
  	/* DELETING A RESULTS IMPORT */
	  FOR V_FIELD_ACTIVITY IN C_FIELD_ACTIVITIES(P_IMPORT_SEQ)
    LOOP
    	DELETE SIM_RESULT_CLASS_INDICATOR
    	 WHERE RCI_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_BIO_RESULTS_GRP_INDIVIDUAL
    	 WHERE BRGI_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_BIO_RESULTS_GRP
    	 WHERE BRG_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_RESULT_LAB_REMARKS
    	 WHERE SRLR_RS_SEQ IN
    				 (SELECT RS_SEQ
    						FROM SIM_RESULTS
    					 WHERE RS_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ);
    
    	DELETE SIM_RESULTS WHERE RS_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_TRAP_NET WHERE STN_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_ACTIVITY_PERSONNEL
    	 WHERE SAP_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;

    	DELETE SIM_ACTIVITY_CPORG
    	 WHERE SAC_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_ACTIVITY_PROJECTS
    	 WHERE SAPJ_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_ACTUAL_ACTIVITY_LOC
    	 WHERE AAL_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_TRAWL_OPS_DET
    	 WHERE TOD_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_SAMPLE WHERE SS_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
      
      -- jah 2-13-04
      DELETE SIM_FIELD_ACTIVITY_PARENT 
      WHERE  FAP_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ
      OR     FAP_PARENT_FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	DELETE SIM_FIELD_ACTIVITIES WHERE FA_SEQ = V_FIELD_ACTIVITY.FA_SEQ;
    
    	COMMIT;
    
    	/* kms (6/11/01) - see if visit created by SIM, if so, see if any other activities
         are on the visit.  If there aren't and the visit is not still in STORET, delete
         the visit from the SIM_STATION_VISITS table. */
		  /* This cursor looks only for visits that are both in SIM and STORET */
			OPEN C_GET_VISIT_INFO(V_FIELD_ACTIVITY.FA_SSV_SEQ);
			
			FETCH C_GET_VISIT_INFO
			 INTO V_DUMMY_SEQ;
			
			IF C_GET_VISIT_INFO%NOTFOUND THEN
					/* kms - visit created by SIM and not associated with a visit in STORET,
          see if there are any activities associated with it. */
				OPEN C_CHECK_VISIT_IN_SIM(V_FIELD_ACTIVITY.FA_SSV_SEQ);
				
				FETCH C_CHECK_VISIT_IN_SIM
			  	INTO V_DUMMY_SEQ;
				
				IF C_CHECK_VISIT_IN_SIM%NOTFOUND THEN
					/* kms - no activities associated with this visit, delete it. */
					DELETE SIM_STATION_VISITS
					 WHERE SSV_SEQ = V_FIELD_ACTIVITY.FA_SSV_SEQ;
					
					/* kms (6/11/01) - see if trip can be deleted.  Only do this when the station
           visit is deleted since we know the possibility exists that there is nothing
           left on the trip. */
					OPEN C_GET_TRIP_INFO(V_FIELD_ACTIVITY.FA_STP_SEQ);
					
					FETCH C_GET_TRIP_INFO
				  	INTO V_DUMMY_SEQ;
					
					IF C_GET_TRIP_INFO%NOTFOUND THEN
						/* kms - trip created by SIM and not associated with a trip in STORET,
             see if there are any other station visits associated with it and
             delete if not */
						OPEN C_CHECK_TRIP_IN_SIM(V_FIELD_ACTIVITY.FA_STP_SEQ);
					
						FETCH C_CHECK_TRIP_IN_SIM
							INTO V_DUMMY_SEQ;
						
						IF C_CHECK_TRIP_IN_SIM%NOTFOUND THEN
							/* kms - no station visits associated with the trip, delete.
               sf - there could still be other visits on this trip, though, so
               don't raise an error if it won't go */
							BEGIN
								DELETE SIM_TRIPS
								 WHERE STP_SEQ = V_FIELD_ACTIVITY.FA_STP_SEQ;
				 			EXCEPTION
								WHEN OTHERS THEN
									NULL;
							END;
						END IF;
						
						CLOSE C_CHECK_TRIP_IN_SIM;
					END IF;
					
					CLOSE C_GET_TRIP_INFO;
				END IF;
				
				CLOSE C_CHECK_VISIT_IN_SIM;
			END IF;
			
			CLOSE C_GET_VISIT_INFO;
			
			COMMIT;
		END LOOP;
		
		COMMIT;
		
		OPEN C_GET_ALL_TRIPS;

		LOOP
		FETCH C_GET_ALL_TRIPS
		 INTO V_TRIP_SEQ;

		  IF C_GET_ALL_TRIPS%FOUND THEN
				OPEN C_GET_ALL_VISITS(V_TRIP_SEQ);

				LOOP
          FETCH C_GET_ALL_VISITS
			     INTO V_VISIT_SEQ;
	
	        IF C_GET_ALL_VISITS%FOUND THEN
					  OPEN C_GET_ALL_ACTIVITIES(V_VISIT_SEQ, V_TRIP_SEQ);
		  			FETCH C_GET_ALL_ACTIVITIES
			       INTO V_DUMMY_SEQ;
		
  			    IF C_GET_ALL_ACTIVITIES%NOTFOUND THEN
  				    --There are no activities on this visit, so delete
				      DELETE SIM_STATION_VISITS WHERE SSV_SEQ = V_VISIT_SEQ;
			      END IF;
		
			      CLOSE C_GET_ALL_ACTIVITIES;
		      ELSE
			      EXIT;
		      END IF;
	      END LOOP;

	      CLOSE C_GET_ALL_VISITS;

	      BEGIN
	        DELETE SIM_TRIPS WHERE STP_SEQ = V_TRIP_SEQ;
        EXCEPTION
	        WHEN OTHERS THEN
	          NULL;
	          --DO nothing. Can't delete this trip cuz there are visits on it still.
				END;
			ELSE
				EXIT;
			END IF;
		END LOOP;
		
		CLOSE C_GET_ALL_TRIPS;
		
		COMMIT;
		
  	/* Clean up and Stations that were created when importing this file. */
  	FOR V_RESULT_STATIONS IN C_RESULT_STATIONS(P_IMPORT_SEQ)
  	LOOP
  		DELETE SIM_STATION_WELLS
  		 WHERE SSW_STA_SEQ = V_RESULT_STATIONS.STA_SEQ;
  	
  		DELETE SIM_STATION_LOCATIONS
  		 WHERE STL_STA_SEQ = V_RESULT_STATIONS.STA_SEQ;
  	
  		DELETE SIM_STATIONS
  		 WHERE STA_SEQ = V_RESULT_STATIONS.STA_SEQ;
  	END LOOP;
  END IF;
	
  /* clean out import lines table */
  /* DTW 09/27/01 Delete in batches of 100 to avoid rollback issues. */
  LOOP
  	OPEN C_CHECK_IMPORT_LINE(P_IMPORT_SEQ);
  
  	FETCH C_CHECK_IMPORT_LINE
  		INTO V_DUMMY_SEQ;
  
  	IF C_CHECK_IMPORT_LINE%NOTFOUND THEN
  		CLOSE C_CHECK_IMPORT_LINE;
  	
  		EXIT;
  	ELSE
  		DELETE SIM_IMP_LINES
  		 WHERE SIPL_SIL_SEQ = P_IMPORT_SEQ AND ROWNUM < 100;
  	
  		COMMIT;
  	
  		CLOSE C_CHECK_IMPORT_LINE;
  	END IF;
  END LOOP;
  
  DELETE SIM_IMP_LINES WHERE SIPL_SIL_SEQ = P_IMPORT_SEQ;
  
  /* remove from import details */
  /* DTW 09/27/01 Delete in batches of 100 to avoid rollback issues. */
  LOOP
  	OPEN C_CHECK_IMPORT_DETAIL(P_IMPORT_SEQ);
  
  	FETCH C_CHECK_IMPORT_DETAIL
  		INTO V_DUMMY_SEQ;
  
  	IF C_CHECK_IMPORT_DETAIL%NOTFOUND THEN
  		CLOSE C_CHECK_IMPORT_DETAIL;
  	
  		EXIT;
  	ELSE
  		DELETE SIM_IMPORT_DETAILS
  		 WHERE SMD_SIL_SEQ = P_IMPORT_SEQ AND ROWNUM < 100;
  	
  		COMMIT;
  	
  		CLOSE C_CHECK_IMPORT_DETAIL;
  	END IF;
  END LOOP;
  
  /* remove from import log */
  DELETE SIM_IMPORT_LOG WHERE SIL_SEQ = P_IMPORT_SEQ;
  
  COMMIT;
END DELETE_IMPORT;

	/******************************************************************************************************/
	FUNCTION GET_IMPORT_LOG_SEQ RETURN NUMBER IS
		CURSOR C_IMPORT_LOG_SEQ IS
			SELECT SIM_IMPORT_LOG_SEQ.NEXTVAL FROM DUAL;
	
		V_IMPORT_LOG_SEQ NUMBER;
	BEGIN
		/* GET sequence for import log */
		OPEN C_IMPORT_LOG_SEQ;
	
		FETCH C_IMPORT_LOG_SEQ
			INTO V_IMPORT_LOG_SEQ;
	
		CLOSE C_IMPORT_LOG_SEQ;
	
		RETURN V_IMPORT_LOG_SEQ;
	EXCEPTION
		WHEN OTHERS THEN
			SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,
																	SQLERRM ||
																	 ' error in sim2_import_pkg.get_import_log_seq',
																	NULL);
	END GET_IMPORT_LOG_SEQ;

	PROCEDURE UPDATE_SIL_ROWS_READY(P_IMPORT_LOG_SEQ    IN NUMBER,
																	P_ROWS_READY_CHANGE IN NUMBER) IS
	BEGIN
		UPDATE SIM_IMPORT_LOG
			 SET SIL_ROWS_READY = NVL(SIL_ROWS_READY, 0) +
														NVL(P_ROWS_READY_CHANGE, 0)
		 WHERE SIL_SEQ = P_IMPORT_LOG_SEQ;
	
		COMMIT;
	END UPDATE_SIL_ROWS_READY;

	/******************************************************************************************************/
	PROCEDURE UPDATE_SIL_ROWS_ERRORS(P_IMPORT_LOG_SEQ IN NUMBER,
																	 P_ROWS_ERRORS    IN NUMBER) IS
	BEGIN
		UPDATE SIM_IMPORT_LOG
			 SET SIL_ROWS_ERRORS = NVL(P_ROWS_ERRORS, 0)
		 WHERE SIL_SEQ = P_IMPORT_LOG_SEQ;
	
		COMMIT;
	END UPDATE_SIL_ROWS_ERRORS;

	/******************************************************************************************************/
	PROCEDURE UPDATE_SIL_ROWS_IMPORTED(P_IMPORT_LOG_SEQ IN NUMBER,
																		 P_ROWS_IMPORTED  IN NUMBER) IS
	BEGIN
		UPDATE SIM_IMPORT_LOG
			 SET SIL_ROWS_IMPORTED = NVL(P_ROWS_IMPORTED, 0)
		 WHERE SIL_SEQ = P_IMPORT_LOG_SEQ;
	
		COMMIT;
	END UPDATE_SIL_ROWS_IMPORTED;

	/******************************************************************************************************/
	FUNCTION IND_NBR_SORT(P_SEQ NUMBER) RETURN VARCHAR2 IS
		V_INDV_NBR VARCHAR2(10);
		V_SEQ NUMBER(10) := P_SEQ;
	
		CURSOR C_BRGI(P_SEQ NUMBER) IS
			SELECT BRGI_INDIVIDUAL_NUMBER
				FROM SIM_BIO_RESULTS_GRP_INDIVIDUAL
			 WHERE BRGI_SEQ = P_SEQ;
	BEGIN
		OPEN C_BRGI(V_SEQ);
	
		FETCH C_BRGI
			INTO V_INDV_NBR;
	
		CLOSE C_BRGI;
	
		RETURN(V_INDV_NBR);
	END IND_NBR_SORT;

	/******************************************************************************************************/
	FUNCTION SIM200_READ_TEXT_FILE(P_DFLT_PATH      IN VARCHAR2,
																 P_FILE_NAME      IN OUT VARCHAR2,
																 P_IMP_TYPE       IN VARCHAR2,
																 P_ORG_ID         IN OUT VARCHAR2,
																 P_IMP_CFG_SEQ    IN NUMBER,
																 P_IMPORT_LOG_SEQ IN OUT NUMBER)
		RETURN VARCHAR2 IS
		V_FILE_HANDLE    UTL_FILE.FILE_TYPE;
		V_LINE_SEQ       NUMBER(10);
		V_LINE_CNT       NUMBER(10) := 0;
		V_TEXT_LINE      LONG;
		V_DELIMITER      VARCHAR2(1);
		V_NO_FIELDS      NUMBER(3);
		V_NO_IMP_FIELDS  NUMBER(3);
		V_NO_GEN_FIELDS  NUMBER(3);
		V_ORG_IS_NBR     NUMBER(8);
		V_FILE_NAME      VARCHAR2(100);
		V_FILE_PATH      VARCHAR2(300);
		EX_BAD_FILE_EXT EXCEPTION;
		EX_BAD_FILE_HANDLE EXCEPTION;
		EX_NO_SEQUENCE EXCEPTION;
		EX_BAD_CFG EXCEPTION;
		EX_WRONG_DELIMITER EXCEPTION;
		EX_WRONG_NO_FIELDS EXCEPTION;
		EX_NO_FIELDS EXCEPTION;
		E_NO_DATA EXCEPTION;
		E_NO_FILE EXCEPTION;
		E_ALL_DONE EXCEPTION;
	
		CURSOR C_GET_ORG_IS IS
			SELECT TSMORGAN_IS_NUMBER
				FROM TSMORGAN
			 WHERE ORG_ID = RPAD(P_ORG_ID, 8);
	
		CURSOR C_GET_DELIMITER(P_IMP_CFG_SEQ IN NUMBER) IS
			SELECT SICC_DELIMITER
				FROM SIM_IMP_CFG
			 WHERE SICC_SEQ = P_IMP_CFG_SEQ;
	
		CURSOR C_GET_NEXT_LINE_SEQ IS
			SELECT SIM_IMP_LINES_SEQ.NEXTVAL FROM DUAL;
	
		CURSOR C_GET_NO_IMP_FIELDS(P_IMP_CFG_SEQ IN NUMBER) IS
			SELECT COUNT(1)
				FROM SIM_IMP_CFG, SIM_IMP_CFG_DETS
			 WHERE SICDT_INCLUDED IS NOT NULL AND SICC_SEQ = P_IMP_CFG_SEQ AND
						 SICDT_SICC_SEQ = SICC_SEQ;
	
		CURSOR C_GET_NO_GEN_FIELDS(P_IMP_CFG_SEQ IN NUMBER) IS
			SELECT COUNT(1)
				FROM SIM_IMP_CFG, SIM_IMP_CFG_DETS
			 WHERE SICDT_GENERATE IS NOT NULL AND SICC_SEQ = P_IMP_CFG_SEQ AND
						 SICDT_SICC_SEQ = SICC_SEQ;
	BEGIN
		IF P_FILE_NAME IS NULL THEN
			RAISE E_NO_FILE;
		ELSE
			/* Break out the file path and file name */
			V_FILE_NAME := UPPER(SUBSTR(P_FILE_NAME,
																	INSTR(P_FILE_NAME, '\', -1) + 1));
			V_FILE_PATH := UPPER(SUBSTR(P_FILE_NAME,
																	1,
																	INSTR(P_FILE_NAME, '\', -1) - 1)) || '\';
		END IF;
	
		BEGIN
			V_FILE_HANDLE := UTL_FILE.FOPEN(V_FILE_PATH, V_FILE_NAME, 'R', 5000);
		EXCEPTION
			WHEN OTHERS THEN
				RAISE EX_BAD_FILE_HANDLE;
		END;
	
		P_IMPORT_LOG_SEQ := SIM2_IMPORT_PKG.GET_IMPORT_LOG_SEQ;
	
		/* get org is number */
		OPEN C_GET_ORG_IS;
	
		FETCH C_GET_ORG_IS
			INTO V_ORG_IS_NBR;
	
		CLOSE C_GET_ORG_IS;
	
		/* start import log record */
		BEGIN
			INSERT INTO SIM_IMPORT_LOG
				(SIL_SEQ,
				 SIL_SICC_SEQ,
				 SIL_IMPORT_ON,
				 SIL_IMPORTED_BY,
				 SIL_IMPORT_FILE,
				 SIL_LAB_ID,
				 SIL_ROWS_IMPORTED,
				 SIL_ROWS_ERRORS,
				 SIL_ROWS_READY,
				 SIL_DELETED_ON,
				 SIL_DELETED_BY,
				 SIL_EXPORTED_ON,
				 SIL_EXPORTED_BY,
				 SIL_IMPORT_TYPE,
				 SIL_TSMORGAN_IS_NUMBER,
				 SIL_TSMORGAN_ORG_ID)
			VALUES
				(P_IMPORT_LOG_SEQ, /* SIL_SEQ */
				 P_IMP_CFG_SEQ, /* SIL_SICC_SEQ */
				 SYSDATE, /* SIL_IMPORT_ON */
				 USER, /* SIL_IMPORTED_BY */
				 P_FILE_NAME, /* SIL_IMPORT_FILE */
				 NULL, /* SIL_LAB_ID */
				 NULL, /* SIL_ROWS_IMPORTED */
				 NULL, /* SIL_ROWS_ERRORS */
				 NULL, /* SIL_ROWS_READY */
				 NULL, /* SIL_DELETED_ON */
				 NULL, /* SIL_DELETED_BY */
				 NULL, /* SIL_EXPORTED_ON */
				 NULL, /* SIL_EXPORTED_BY */
				 P_IMP_TYPE, /* SIL_IMPORT_TYPE */
				 V_ORG_IS_NBR, /* SIL_TSMORGAN_IS_NUMBER */
				 P_ORG_ID /* SIL_TSMORGAN_ORG_ID */);
		
			SIM2_COMMIT;
		END;
	
		BEGIN
			OPEN C_GET_DELIMITER(P_IMP_CFG_SEQ);
		
			FETCH C_GET_DELIMITER
				INTO V_DELIMITER;
		
			IF C_GET_DELIMITER%NOTFOUND THEN
				RAISE EX_BAD_CFG;
			END IF;
		
			CLOSE C_GET_DELIMITER;
		EXCEPTION
			WHEN OTHERS THEN
				RETURN SQLERRM || ' - trying to get the delimiter';
		END;
	
		LOOP
			OPEN C_GET_NEXT_LINE_SEQ;
		
			FETCH C_GET_NEXT_LINE_SEQ
				INTO V_LINE_SEQ;
		
			IF C_GET_NEXT_LINE_SEQ%NOTFOUND THEN
				RAISE EX_NO_SEQUENCE;
			END IF;
		
			CLOSE C_GET_NEXT_LINE_SEQ;
		
			V_LINE_CNT := V_LINE_CNT + 1;
		
			BEGIN
				UTL_FILE.GET_LINE(V_FILE_HANDLE, V_TEXT_LINE);
			EXCEPTION
				WHEN NO_DATA_FOUND THEN
					RAISE E_ALL_DONE;
				WHEN OTHERS THEN
					RETURN ' - Line ' || V_LINE_CNT || ' Line is too long. Approximately 5000 characters is the maximum length allowed.';
			END;
		
			IF V_LINE_CNT = 1 THEN
				IF INSTR(V_TEXT_LINE, V_DELIMITER) = 0 THEN
					RAISE EX_WRONG_DELIMITER;
				END IF;
			
				/* count the number of fields in first line */
				V_NO_FIELDS := LENGTH(V_TEXT_LINE) -
											 LENGTH(TRANSLATE(V_TEXT_LINE,
																				'a' || V_DELIMITER,
																				'a')) + 1;
				/* Raise exception if import file contains no lines */
				IF V_NO_FIELDS = 0 THEN
					RAISE EX_NO_FIELDS;
				END IF;
				/* count the number of fields in import configuration */
				OPEN C_GET_NO_IMP_FIELDS(P_IMP_CFG_SEQ);
			
				FETCH C_GET_NO_IMP_FIELDS
					INTO V_NO_IMP_FIELDS;
			
				IF C_GET_NO_IMP_FIELDS%NOTFOUND THEN
					RAISE EX_BAD_CFG;
				END IF;
			
				CLOSE C_GET_NO_IMP_FIELDS;
			
				/* count the number of generated fields in import configuration */
				OPEN C_GET_NO_GEN_FIELDS(P_IMP_CFG_SEQ);
			
				FETCH C_GET_NO_GEN_FIELDS
					INTO V_NO_GEN_FIELDS;
			
				IF C_GET_NO_GEN_FIELDS%NOTFOUND THEN
					RAISE EX_BAD_CFG;
				END IF;
			
				CLOSE C_GET_NO_GEN_FIELDS;
			
				IF V_NO_FIELDS != (V_NO_IMP_FIELDS - V_NO_GEN_FIELDS) THEN
					RAISE EX_WRONG_NO_FIELDS;
				END IF;
			END IF;
		
			INSERT INTO SIM_IMP_LINES
				(SIPL_SEQ,
				 SIPL_SIL_SEQ,
				 SIPL_LINE_TEXT,
				 SIPL_LINE_NO,
				 SIPL_STA_SEQ,
				 SIPL_FA_SEQ,
				 SIPL_RS_SEQ,
				 SIPL_AC_SEQ,
				 SIPL_SPJ_SEQ)
			VALUES
				(V_LINE_SEQ, /* SIPL_SEQ */
				 P_IMPORT_LOG_SEQ, /* SIPL_SIL_SEQ */
				 V_TEXT_LINE, /* SIPL_LINE_TEXT */
				 V_LINE_CNT, /* SIPL_LINE_NO */
				 NULL, /* SIPL_STA_SEQ */
				 NULL, /* SIPL_FA_SEQ */
				 NULL, /* SIPL_RS_SEQ */
				 NULL, /* SIPL_AC_SEQ */
				 NULL /* SIPL_SPJ_SEQ */);
		
			IF MOD(V_LINE_CNT, 100) = 0 THEN
				SIM2_COMMIT;
			END IF;
		END LOOP;
	
		RETURN(NULL);
	EXCEPTION
		WHEN EX_BAD_FILE_EXT THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'Import file must be a text file and end with the .txt extension.';
		WHEN EX_BAD_FILE_HANDLE THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN SQLERRM || 'Invalid file name or path.';
		WHEN EX_BAD_CFG THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'Configuration is incomplete or invalid.';
		WHEN EX_WRONG_DELIMITER THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'Delimiter could not be found in the text file. Make sure the file is in text format and the delimiter is right in the import configuration.';
		WHEN EX_WRONG_NO_FIELDS THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'Import file contains the wrong number of fields, check the import configuration to make sure that it has been set up correctly.';
		WHEN EX_NO_FIELDS THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'Import file contains no fields, the import configuration must contain at lease one field.';
		WHEN EX_NO_SEQUENCE THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'No sequence returned for sim_imp_lines.';
		WHEN E_NO_FILE THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN 'A import path and file must be declared.';
		WHEN E_NO_DATA THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			SIM2_COMMIT;
			RETURN 'Text line is too long.';
		WHEN E_ALL_DONE THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			SIM2_COMMIT;
			RETURN NULL;
		WHEN OTHERS THEN
			UTL_FILE.FCLOSE(V_FILE_HANDLE);
			RETURN SQLERRM || ' - in SIM200_READ_TEXT_FILE function.';
	END SIM200_READ_TEXT_FILE;
	/******************************************************************************************************/

FUNCTION CHECK_DOCUMENT_FILE(p_file_name IN VARCHAR2, p_import_log_seq IN NUMBER, p_line IN NUMBER) RETURN BOOLEAN IS

  v_bfile               BFILE;
  v_count               INTEGER;

BEGIN

  -- Check DOCUMENT_DIR configuration
  select count(1)
    into v_count
    from all_directories
    where directory_name = 'DOCUMENT_DIR';
  
  IF v_count = 0 THEN
    sim2_import_pkg.write_error(p_import_log_seq ,'Line '||p_line||': The DOCUMENT_DIR Oracle directory is not configured.  Unable to support Document/Graphic files at this time.',NULL);
    RETURN FALSE;
  END IF;

  -- Check minimum length
  IF length(p_file_name) <= 4 THEN
    sim2_import_pkg.write_error(p_import_log_seq ,'Line '||p_line||': Document/Graphic file name "' || p_file_name || '" is invalid.',NULL);
    RETURN FALSE;
  END IF;

  -- Check for supported file type (PDF, JPG, BMP, GIF, or TXT)
  IF upper(substr(p_file_name, length(p_file_name) - 3)) NOT IN ('.PDF', '.JPG', '.BMP', '.GIF', '.TXT') THEN
    sim2_import_pkg.write_error(p_import_log_seq ,'Line '||p_line||': Document/Graphic "' || p_file_name || '" file type is not PDF, JPG, BMP, GIF, or TXT.',NULL);
    RETURN FALSE;
  END IF;

  -- Check for valid file
  BEGIN
    v_bfile := bfilename('DOCUMENT_DIR', p_file_name);
    dbms_lob.fileopen(v_bfile, dbms_lob.file_readonly);
    dbms_lob.fileclose(v_bfile);
  EXCEPTION
    WHEN OTHERS THEN
      sim2_import_pkg.write_error(p_import_log_seq ,'Line '||p_line||': Document/Graphic file name "' || p_file_name || '" is invalid.',NULL);
      RETURN FALSE;
  END;

  RETURN TRUE;
  
END CHECK_DOCUMENT_FILE;

END SIM2_IMPORT_PKG;
/
