CREATE OR REPLACE PACKAGE sim2_import_project_pkg IS
 /* Modified 3/20/02 by SF - Added an exception to deal with  bad dates in the data file */

  -- Author  : KHENAS
  -- Created : 2/5/01 10:37:43 AM
  -- Purpose : import STORET projects

  -- Public type declarations
   TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;

   TYPE col_table_num_type IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;

   TYPE t_storet_pk_type IS RECORD (
     is_number         NUMBER(8),
     org_id            VARCHAR2(8));

  -- Public constant declarations
  -- Public variable declarations

   v_org_is_nbr         NUMBER(8) := 0;
   v_import_log_seq     NUMBER(10);
   v_line_cnt           NUMBER(10) := 0;

  -- Public function and procedure declarations
   FUNCTION get_org_info(p_imp_cfg_seq IN NUMBER, p_org_id IN VARCHAR2,p_org_is_nbr IN OUT NUMBER) RETURN NUMBER;

   FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER,p_delimiter IN OUT VARCHAR2) RETURN NUMBER;

   FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table IN OUT col_table_num_type,
                         p_default_table IN OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER;

   FUNCTION get_column_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) RETURN NUMBER;

   FUNCTION check_translations(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                               p_trans_exist OUT BOOLEAN) RETURN VARCHAR2;

   PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER,p_proj_seq IN NUMBER);

   FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,p_proj_seq IN NUMBER)
     RETURN NUMBER;

   FUNCTION validate_alval_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                                p_col_seq IN NUMBER, p_proj_seq IN NUMBER) RETURN NUMBER;

   FUNCTION validate_storet_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                                 p_col_seq IN NUMBER, p_req IN BOOLEAN, p_org_id IN VARCHAR2,
                                 p_line_nbr IN NUMBER, p_proj_seq IN NUMBER) RETURN t_storet_pk_type;

   FUNCTION write_error_message(p_item_name IN VARCHAR2,p_req IN BOOLEAN,p_line_nbr IN NUMBER,
                                p_provided IN BOOLEAN,p_status IN OUT VARCHAR2, p_comment IN OUT VARCHAR2,
                                p_proj_seq IN NUMBER) RETURN NUMBER;

   FUNCTION get_date(p_column_name IN VARCHAR2,p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,
                     p_cfg_seq IN NUMBER,p_proj_seq IN NUMBER) RETURN DATE;

   PROCEDURE import_projects (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                              p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER);

END sim2_import_project_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim2_import_project_pkg IS

  -- Private type declarations

  -- Private constant declarations

  -- Private variable declarations

  -- Function and procedure implementations
  e_a EXCEPTION;

FUNCTION get_org_info(p_imp_cfg_seq IN NUMBER, p_org_id IN VARCHAR2,p_org_is_nbr IN OUT NUMBER) RETURN NUMBER IS

  CURSOR c_get_tsmorgan_nbr (p_cursor_org_id IN VARCHAR2) IS
    SELECT TSMORGAN_IS_NUMBER FROM TSMORGAN WHERE ORG_ID = RPAD(p_cursor_org_id,8);

BEGIN

  OPEN c_get_tsmorgan_nbr (p_org_id);
  FETCH c_get_tsmorgan_nbr INTO p_org_is_nbr;
    IF c_get_tsmorgan_nbr%NOTFOUND THEN
      p_org_is_nbr := NULL;
    END IF;
  CLOSE c_get_tsmorgan_nbr;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20199, 'Failed to get organization is number.');
    RETURN 0;

END get_org_info;

/******************************************************************************************************/
FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER, p_delimiter IN OUT VARCHAR2) RETURN NUMBER IS

  e_no_cfg EXCEPTION;

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT SICC_DELIMITER FROM sim_imp_cfg WHERE SICC_SEQ = p_sicc_seq;

BEGIN

  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg INTO p_delimiter;
    IF c_get_imp_cfg%NOTFOUND THEN
      RAISE e_no_cfg;
    END IF;
  CLOSE c_get_imp_cfg;

  RETURN 1;

EXCEPTION

  WHEN e_no_cfg THEN
    RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');

  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20198 , 'Failed to get delimiter.');
    RETURN 0;

END get_delimiter;

/******************************************************************************************************/
FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table IN OUT col_table_num_type,
                      p_default_table IN OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER IS

  v_import_type_seq              NUMBER(10);

  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT SICC_SID_SEQ FROM SIM_IMP_CFG WHERE SICC_SEQ = p_sicc_seq;

  CURSOR c_get_dflt_pos (p_type_seq IN NUMBER) IS
    SELECT SIC_DFLT_POS FROM SIM_IMP_COLS WHERE SIC_SID_SEQ = p_type_seq;

  CURSOR c_get_col_pos (p_sicc_seq IN NUMBER,p_dflt_pos IN NUMBER) IS
    SELECT SICDT_POS, SICDT_DEFAULT FROM SIM_IMP_CFG_DETS, SIM_IMP_COLS
      WHERE SIC_SEQ = SICDT_SIC_SEQ AND SICDT_SICC_SEQ = p_sicc_seq AND SIC_DFLT_POS = p_dflt_pos
      AND (SICDT_INCLUDED IS NOT NULL OR SICDT_GENERATE IS NOT NULL);

BEGIN

  /* get the import type in order to limit the columns selected to the right type */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type INTO v_import_type_seq;
   IF c_get_import_type%NOTFOUND THEN
     v_import_type_seq := NULL;
     RAISE_APPLICATION_ERROR(-20102, 'Unable to get import type from import configuration.');
   END IF;
  CLOSE c_get_import_type;

  p_col_cnt := 0;

  /* loop through import columns and assign actual position to table indexed by
     default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP

    /* get actual position and item default from configuration table */
    OPEN c_get_col_pos(p_sicc_seq,v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
      IF c_get_col_pos%NOTFOUND THEN
        /* if there is nothing in the configuration table for this column, position and default will be null */
        p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
        p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
      ELSE
        p_col_cnt := p_col_cnt + 1;
      END IF;
    CLOSE c_get_col_pos;

  END LOOP;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20103 , 'Failed to get column positions, item names, and column sequences.');

END get_item_pos;

/******************************************************************************************************/
FUNCTION get_column_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) RETURN NUMBER IS

  v_imp_col_seq  NUMBER(10);
  e_no_cfg_found EXCEPTION;

  CURSOR c_get_col_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) IS
    SELECT sicdt_sic_seq FROM SIM_IMP_CFG, SIM_IMP_CFG_DETS
      WHERE sicdt_sicc_seq = sicc_seq AND sicc_seq = p_cfg_seq
      AND sicdt_pos = p_cfg_pos;

BEGIN

  OPEN c_get_col_seq(p_cfg_seq,p_cfg_pos);
  FETCH c_get_col_seq INTO v_imp_col_seq;
    IF c_get_col_seq%NOTFOUND THEN
      v_imp_col_seq := NULL;
      RAISE e_no_cfg_found;
    END IF;
  CLOSE c_get_col_seq;

  RETURN v_imp_col_seq;

EXCEPTION
  WHEN e_no_cfg_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - unable to find sequence in sim_imp_cfg_dets for pos: '||p_cfg_pos,NULL);
    RAISE_APPLICATION_ERROR(-20100, 'Import configuration is not complete');

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - getting column sequence.', NULL);
    RETURN NULL;

END get_column_seq;
/******************************************************************************************************/
FUNCTION check_sim_proj(p_project_id IN VARCHAR2, p_org_id IN VARCHAR2, p_project_name IN VARCHAR2)
  RETURN NUMBER IS

  
  v_proj_name_cnt         NUMBER(10);
  v_proj_id_cnt           NUMBER(10);  
  v_success               NUMBER(1) := 0;
  
  CURSOR c_project_id(p_id VARCHAR2,p_org_id VARCHAR2) IS
    SELECT COUNT(spj_seq)
      FROM sim_projects
      WHERE spj_id = p_id
        AND RTRIM(spj_org_id) = RTRIM(p_org_id);
        
  CURSOR c_project_name(p_name VARCHAR2,p_org_id VARCHAR2) IS
    SELECT COUNT(spj_seq)
      FROM sim_projects
      WHERE spj_name = p_name
        AND RTRIM(spj_org_id) = RTRIM(p_org_id);
  
BEGIN

  OPEN c_project_id(p_project_id,p_org_id);
    FETCH c_project_id INTO v_proj_id_cnt;
  CLOSE c_project_id;
  
  IF NVL(v_proj_id_cnt,0) > 0 THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||':  - Duplicate Project ID in SIM.',NULL);
    v_success := 1;
  END IF;
  
  OPEN c_project_name(p_project_name,p_org_id);
    FETCH c_project_name INTO v_proj_name_cnt;
  CLOSE c_project_name;
  
  IF NVL(v_proj_name_cnt,0) > 0 THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||':  - Duplicate Project Name in SIM.',NULL);    
    v_success := 1;
  END IF;
  
  RETURN v_success;    
  
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check_sim_proj.',NULL); 
    v_success := 1;
    RETURN v_success;     
END check_sim_proj;
/******************************************************************************************************/
FUNCTION check_storet_proj(p_project_id IN VARCHAR2, p_org_id IN VARCHAR2, p_project_name IN VARCHAR2, p_fa_seq IN NUMBER)
  RETURN t_storet_pk_type IS

  v_project_pk t_storet_pk_type;

  CURSOR c_check_storet_proj_id(p_project_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number, tsmproj_org_id FROM TSMPROJ
      WHERE identification_cd = p_project_id
      AND   tsmproj_org_id    = p_org_id;

  CURSOR c_check_storet_proj_name(p_project_name IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number, tsmproj_org_id FROM TSMPROJ
      WHERE name           = p_project_name
      AND   rtrim(tsmproj_org_id) = p_org_id;

BEGIN
  /* bka - function that looks for the project id in STORET to make sure it's not
     duplicating a pre-existing project.  If the project is already in STORET, SIM
     will import it but will prevent it from being exported. */

  OPEN c_check_storet_proj_name(p_project_name, p_org_id);
  FETCH c_check_storet_proj_name INTO v_project_pk;
    IF c_check_storet_proj_name%NOTFOUND THEN
      v_project_pk.is_number := NULL;
      v_project_pk.org_id    := NULL;
    END IF;
  CLOSE c_check_storet_proj_name;

  IF v_project_pk.is_number IS NULL THEN
    OPEN c_check_storet_proj_id(p_project_id, p_org_id);
    FETCH c_check_storet_proj_id INTO v_project_pk;
      IF c_check_storet_proj_id%NOTFOUND THEN
        v_project_pk.is_number := NULL;
        v_project_pk.org_id    := NULL;
      END IF;
    CLOSE c_check_storet_proj_id;
  END IF;

  RETURN v_project_pk;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check activity id.',NULL);
    v_project_pk.is_number := NULL;
    v_project_pk.org_id    := NULL;
    RETURN v_project_pk;

END check_storet_proj;

/******************************************************************************************************/
FUNCTION get_instructions(p_imp_cfg_seq IN NUMBER, p_instruction_type IN VARCHAR2, p_proj_seq IN NUMBER)
  RETURN VARCHAR2 IS

  v_instr_choice         VARCHAR2(15);

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SICI_SICC_SEQ = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN NUMBER) IS
    SELECT SIIC_CODE FROM SIM_IMP_INSTR_CHOICES WHERE SIIC_SII_SEQ = p_instr_type
      AND SIIC_DFLT_FLAG = 1;

BEGIN

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr INTO v_instr_choice;
    IF c_get_cfg_instr%NOTFOUND THEN

      /* no choice could be found, get default choice */
      OPEN c_get_instr_default(p_instruction_type);
      FETCH c_get_instr_default INTO v_instr_choice;
      CLOSE c_get_instr_default;

    END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;

EXCEPTION
  WHEN OTHERS THEN
    sim2_import_pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get_instructions.',p_proj_seq);
    RETURN NULL;

END get_instructions;

/******************************************************************************************************/
FUNCTION get_date(p_column_name IN VARCHAR2,p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,
                  p_cfg_seq IN NUMBER,p_proj_seq IN NUMBER) RETURN DATE IS

  v_start_date                  DATE;
  v_fmt                         VARCHAR2(80);

  e_no_cfg_found                EXCEPTION;

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;

BEGIN

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      v_fmt := NULL;
      /* no format specified, use default */
      BEGIN
        v_start_date := TRUNC(TO_DATE(p_raw_date,'MM-DD-RRRR'));
      EXCEPTION
        WHEN OTHERS THEN
          sim2_import_pkg.write_error(v_import_log_seq, 'Line: '||v_line_cnt||' - Date '||p_raw_date||' is not in a valid date format. ',p_proj_seq);
      END;
    END IF;
  CLOSE c_get_date_fmt;

  BEGIN
    IF v_fmt IS NOT NULL THEN
      v_start_date := TRUNC(TO_DATE(p_raw_date,v_fmt));

    END IF;
  EXCEPTION
    WHEN OTHERS THEN
      v_start_date := NULL;
      sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - check date format in import configuration. '||p_column_name||' '||p_raw_date,p_proj_seq);
  END;

  RETURN v_start_date;

EXCEPTION

  WHEN e_no_cfg_found THEN
    RAISE_APPLICATION_ERROR(-20100,'Import configuration for start or end date is incomplete or does not exist.');
    RETURN NULL;

  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - in get date, column '||p_column_name||' '||p_raw_date,p_proj_seq);
    RETURN NULL;

END get_date;

/******************************************************************************************************/
FUNCTION check_translations(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                            p_trans_exist OUT BOOLEAN) RETURN VARCHAR2 IS

  v_storet_value             VARCHAR2(2000);
  v_proj_seq								 NUMBER(10);
	
  CURSOR c_get_project_seq IS SELECT SIM_PROJECTS_SEQ.CURRVAL FROM DUAL;

  CURSOR c_get_storet_code(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT st_storet_code FROM SIM_TRANSLATIONS, SIM_IMP_CFG_DETS WHERE st_user_code = p_code
      AND st_sicdt_seq = sicdt_seq AND sicdt_sicc_seq = p_imp_cfg_seq
      AND sicdt_sic_seq = p_col_seq;

BEGIN

  /* check translations table to see if the code should be swapped with another value */
  OPEN c_get_storet_code(p_code,p_imp_cfg_seq,p_col_seq);
  FETCH c_get_storet_code INTO v_storet_value;
    IF c_get_storet_code%NOTFOUND THEN

      /* if not in translations table return original value - will check elsewhere for sequences */
      v_storet_value := p_code;
      p_trans_exist := FALSE;

    ELSIF c_get_storet_code%FOUND THEN
      /* if the code is in the translations table, flag so it's not recreated */
      p_trans_exist := TRUE;
      IF v_storet_value IS NULL THEN
        /* if there is a code in the translations table but no value to translate to, write error message
           status will be changed to 'N' later in the program */
				OPEN c_get_project_seq;
				FETCH c_get_project_seq INTO v_proj_seq;
				CLOSE c_get_project_seq;
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_code||' exists in translations table but has not been mapped to a STORET code.',v_proj_seq);
			END IF;
 
    END IF;
  CLOSE c_get_storet_code;
  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - in check translations',NULL);

END check_translations;

/******************************************************************************************************/
PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER,
                              p_proj_seq IN NUMBER) IS

  v_st_seq          NUMBER(10);
  v_sicdt_seq       NUMBER(10);

  e_imp_cfg_dets    EXCEPTION;
  e_trans_error     EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SICDT_SEQ FROM SIM_IMP_CFG_DETS WHERE SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL FROM DUAL;

BEGIN

  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq INTO v_sicdt_seq;
    IF c_get_cfg_det_seq%NOTFOUND THEN
      v_sicdt_seq := NULL;
      RAISE e_imp_cfg_dets;
    END IF;
  CLOSE c_get_cfg_det_seq;

  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq INTO v_st_seq;
    IF c_get_st_seq%NOTFOUND THEN
      v_st_seq := NULL;
      RAISE e_trans_error;
    END IF;
  CLOSE c_get_st_seq;

  INSERT INTO SIM_TRANSLATIONS (
    ST_SEQ,
    ST_SICDT_SEQ,
    ST_USER_CODE,
    ST_STORET_CODE
    )
  VALUES (
    v_st_seq,                  /* st_seq */
    v_sicdt_seq,               /* st_sicdt_seq */
    p_code,                    /* st_user_code */
    NULL                       /* st_storet_code */
    );

    sim2_import_pkg.write_error(v_import_log_seq, 'Created translation record for '||p_code,p_proj_seq);

EXCEPTION

  WHEN e_imp_cfg_dets THEN
    sim2_import_pkg.write_error(v_import_log_seq, 'Unable to find import configuration - stopped import',p_proj_seq);
    RAISE_APPLICATION_ERROR(-20101,'Unable to find import configuration');

  WHEN e_trans_error THEN
    sim2_import_pkg.write_error(v_import_log_seq, 'Unable to create new record in translations table - error getting sequence',p_proj_seq);

  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - adding record to translations table.',p_proj_seq);

END create_trans_record;

/******************************************************************************************************/
FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                                p_proj_seq IN NUMBER) RETURN NUMBER IS

  v_sia_seq         NUMBER(10);

  CURSOR c_check_allowable_val(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SIA_SEQ FROM SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS
      WHERE SIA_SIC_SEQ = SIC_SEQ AND SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = SIC_SEQ AND SIC_SEQ = p_col_seq AND SIA_ITEM = p_item;

BEGIN

  /* make sure item is an allowable value */
  OPEN c_check_allowable_val(p_item, p_imp_cfg_seq, p_col_seq);
  FETCH c_check_allowable_val INTO v_sia_seq;
    IF c_check_allowable_val%NOTFOUND THEN

      /* if not an allowable value, return null - will write error message, change status of activity */
      v_sia_seq := NULL;

    END IF;
  CLOSE c_check_allowable_val;
  RETURN v_sia_seq;

EXCEPTION
  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - in check allowable values',p_proj_seq);
    RETURN NULL;

END check_allowable_values;

/******************************************************************************************************/
FUNCTION validate_alval_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                             p_col_seq IN NUMBER, p_proj_seq IN NUMBER) RETURN NUMBER IS

   v_converted_item           VARCHAR2(80);
   v_item_sia_seq             NUMBER(10);
   v_trans_exist              BOOLEAN := TRUE;
   v_instr_code               VARCHAR2(15);

BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
     hard coded values in STORET */

  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN

    /* look in allowable values */
    v_item_sia_seq := check_allowable_values(v_converted_item, p_imp_cfg_seq, p_col_seq, p_proj_seq);

    IF v_item_sia_seq IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'PRJ-TR', p_proj_seq);

      IF v_instr_code = 'PRJ-TR-01' THEN

        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_proj_seq);
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value - translation record has been created.',p_proj_seq);

      ELSE
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value.',p_proj_seq);
      END IF;

    ELSIF v_item_sia_seq IS NULL AND v_trans_exist THEN

      /* item has been translated to an invalid storet value */
      sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_proj_seq);

    END IF;
  END IF;
  RETURN v_item_sia_seq;

EXCEPTION
  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate allowable values.',p_proj_seq);
    RETURN NULL;

END validate_alval_item;

/******************************************************************************************************/
FUNCTION validate_storet_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                              p_col_seq IN NUMBER, p_req IN BOOLEAN, p_org_id IN VARCHAR2,
                              p_line_nbr IN NUMBER, p_proj_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  v_storet_pk                 t_storet_pk_type;
  v_instr_code                VARCHAR2(8);

  e_no_function               EXCEPTION;


BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
     hard coded values in STORET tables */

  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN

    /* call appropriate function to look in STORET table */
    IF p_item_name = 'Contact' THEN
      null;
    END IF;

    IF v_storet_pk.is_number = -1 THEN
      RAISE e_no_function;
    END IF;

    IF v_storet_pk.is_number IS NULL AND NOT v_trans_exist THEN

      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'PRJ-TR', p_proj_seq);

      IF v_instr_code = 'PRJ-TR-01' THEN

        /* if record could not be matched to anything in translations or STORET tables,
           create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_proj_seq);
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value - translation record has been created.',p_proj_seq);

      ELSE
        sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value.',p_proj_seq);
      END IF;


    ELSIF v_storet_pk.is_number IS NULL AND v_trans_exist THEN

      /*  item has been translated to an invalid storet value */
      sim2_import_pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_proj_seq);

    END IF;

  ELSIF v_trans_exist THEN

    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_storet_pk.is_number := NULL;
    sim2_import_pkg.write_error(v_import_log_seq,p_orig_item||' exists in translation table but has not been mapped to a STORET code. ',p_proj_seq);

  END IF;

  RETURN v_storet_pk;

EXCEPTION

  WHEN e_no_function THEN
    sim2_import_pkg.write_error(v_import_log_seq, 'SIM2 does not currently handle '||p_item_name,p_proj_seq);
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
    RETURN v_storet_pk;

  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - in validate storet item.',p_proj_seq);
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
    RETURN v_storet_pk;

END validate_storet_item;

/******************************************************************************************************/
FUNCTION write_error_message(p_item_name IN VARCHAR2,p_req IN BOOLEAN,p_line_nbr IN NUMBER,
                             p_provided IN BOOLEAN,p_status IN OUT VARCHAR2, p_comment IN OUT VARCHAR2,
                             p_proj_seq IN NUMBER) RETURN NUMBER IS

BEGIN

  IF p_provided THEN

    p_status := 'N';

    p_comment := RTRIM(p_item_name||' could not be found or mapped to valid STORET value. '||NVL(p_comment,' '));
    sim2_import_pkg.write_error(v_import_log_seq, SUBSTR(INITCAP(p_item_name)||' could not be mapped to a valid STORET value, line: '||p_line_nbr,1,240),p_proj_seq);

  ELSE

    IF p_req THEN
      p_status := 'N';
      p_comment := RTRIM(p_item_name||' not included. '||NVL(p_comment,' '));
      sim2_import_pkg.write_error(
        v_import_log_seq, 
        SUBSTR('Line ' || p_line_nbr || ': Required item ' || p_item_name || ' not provided. Project will not export.',1,240),
        p_proj_seq
      );
    ELSE
      p_status := p_status;
      p_comment := RTRIM(p_item_name||' not included. '||NVL(p_comment,' '));
      sim2_import_pkg.write_error(
        v_import_log_seq, 
        SUBSTR('Line ' || p_line_nbr || ': Included item ' || p_item_name || ' not provided.',1,240),
        NULL
      );
    END IF;

  END IF;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - in write error message.',p_proj_seq);
    RETURN 0;

END write_error_message;

/******************************************************************************************************/
FUNCTION create_project (p_project_rec sim_projects%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create project using values in project record */
  INSERT INTO SIM_PROJECTS (
    SPJ_SEQ,
    SPJ_TSMORGAN_IS_NUMBER,
    SPJ_ORG_ID,
    SPJ_ID,
    SPJ_TSMPROJ_IS_NUMBER,
    SPJ_TSMPROJ_ORG_ID,
    SPJ_NAME,
    SPJ_START_DATE,
    SPJ_DURATION,
    SPJ_PURPOSE,
    SPJ_CONTACT,
    SPJ_DOCUMENT,
    SPJ_EXPORT_SEQ,
    SPJ_IMPORT_SEQ,
    SPJ_STATUS
    )
  VALUES (
    p_project_rec.SPJ_SEQ,                     /* SPJ_SEQ */
    p_project_rec.SPJ_TSMORGAN_IS_NUMBER,      /* SPJ_TSMORGAN_IS_NUMBER */
    p_project_rec.SPJ_ORG_ID,                  /* SPJ_ORG_ID */
    p_project_rec.SPJ_ID,                      /* SPJ_ID */
    p_project_rec.SPJ_TSMPROJ_IS_NUMBER,       /* SPJ_TSMPROJ_IS_NUMBER */
    p_project_rec.SPJ_TSMPROJ_ORG_ID,          /* SPJ_TSMPROJ_ORG_ID */
    p_project_rec.SPJ_NAME,                    /* SPJ_NAME */
    p_project_rec.SPJ_START_DATE,              /* SPJ_START_DATE */
    p_project_rec.SPJ_DURATION,                /* SPJ_DURATION */
    p_project_rec.SPJ_PURPOSE,                 /* SPJ_PURPOSE */
    p_project_rec.SPJ_CONTACT,                 /* SPJ_CONTACT */
    p_project_rec.SPJ_DOCUMENT,                /* SPJ_DOCUMENT */
    p_project_rec.SPJ_EXPORT_SEQ,              /* SPJ_EXPORT_SEQ */
    p_project_rec.SPJ_IMPORT_SEQ,              /* SPJ_IMPORT_SEQ */
    p_project_rec.SPJ_STATUS                   /* SPJ_STATUS */
    );

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - inserting project.',p_project_rec.SPJ_SEQ);
    RETURN 0;

END create_project;

/******************************************************************************************************/
PROCEDURE import_projects (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                           p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER) IS


  v_status              VARCHAR2(1);
  v_sim_proj_success    NUMBER(1);

  v_InLine              VARCHAR2(3000);
  v_bad_parse           BOOLEAN;
  v_line_table          sim2_import_pkg.col_table_type;
  v_default_table       sim2_import_project_pkg.col_table_type;
  v_pos_table           sim2_import_project_pkg.col_table_num_type;
  v_delimiter           VARCHAR2(1);
  v_num_cols            NUMBER(3);
  v_col_seq             NUMBER(10);
  v_dflt_pos            NUMBER(3);

  v_item_name           VARCHAR2(30);
  v_item                VARCHAR2(1999);
  v_comp                NUMBER(1);
	v_trans_exist         BOOLEAN;
  v_success             NUMBER(1);
  v_storet_key          t_storet_pk_type;
  v_comment             VARCHAR2(1999);

  v_project_rec         SIM_PROJECTS%ROWTYPE;
  v_project_copy_rec    SIM_PROJECTS%ROWTYPE;

  v_project_cnt         NUMBER(10) := 0;
  v_error_cnt           NUMBER(10) := 0;
  v_complete_cnt        NUMBER(10) := 0;

  v_project_seq         NUMBER(10);
  v_start_date          DATE;
  v_project_id          VARCHAR2(8);
  v_project_name        VARCHAR2(60);
  v_duration            VARCHAR2(15);
  v_purpose             VARCHAR2(1999);
  v_contact             VARCHAR2(1999);
  v_document            VARCHAR2(1999);

  e_fileread            EXCEPTION;
  e_bad_org             EXCEPTION;
  e_no_cfg              EXCEPTION;
  e_e_bad_delimiter     EXCEPTION;
  e_bad_delimiter       EXCEPTION;
  e_NO_DATA_FOUND       EXCEPTION;
  e_project_seq         EXCEPTION;

  CURSOR c_get_project_seq IS SELECT SIM_PROJECTS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT SIPL_SEQ, SIPL_LINE_TEXT
      FROM SIM_IMP_LINES
     WHERE SIPL_SIL_SEQ = p_import_seq
       and sipl_line_text is not null  -- bka  added to eliminate blank lines from the import
      ORDER BY SIPL_LINE_NO;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_PROC_ORDER, SIC_DFLT_POS, SIC_NAME, SIC_SEQ
      FROM SIM_IMP_COLS, SIM_IMP_DEFS, SIM_IMP_CFG
      WHERE SICC_SID_SEQ = SID_SEQ AND SIC_SID_SEQ = SID_SEQ AND SICC_SEQ = p_imp_cfg_seq
      ORDER BY SIC_PROC_ORDER;

 CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(SPJ_SEQ) FROM SIM_PROJECTS
      WHERE SPJ_IMPORT_SEQ = p_import_log_seq
      AND SPJ_STATUS = 'A';


BEGIN


  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  v_line_cnt := 0;

  BEGIN

    sim2_import_pkg.WRITE_ERROR (v_import_log_seq , 'Start at '||to_char(SYSDATE,'HH24:MI'),NULL);

    /* get tsmorgan_is_number */
    BEGIN
      v_success := get_org_info(p_imp_cfg_seq,p_org_id,v_org_is_nbr);
      IF v_success != 1 OR v_org_is_nbr IS NULL THEN
        RAISE e_bad_org;
      END IF;

    EXCEPTION
      WHEN e_bad_org THEN
        RAISE_APPLICATION_ERROR(-20109,'Unable to find the specified org id in local version of STORET.');

      WHEN OTHERS THEN
        sim2_import_pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting org is number.',NULL);
        RAISE NO_DATA_FOUND;
    END;

    /*  get the delimiter */
    BEGIN
      v_success := get_delimiter(p_imp_cfg_seq,v_delimiter);
      IF v_success != 1 THEN
        RAISE e_no_cfg;
      END IF;

    EXCEPTION
      WHEN e_no_cfg THEN
        RAISE_APPLICATION_ERROR(-20102,'Error getting delimiter from import configuration.');

      WHEN OTHERS THEN
        sim2_import_pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||'- getting delimiter.',NULL);
        RAISE NO_DATA_FOUND;
    END;

    /* get actual positions and default information for items being imported */
    BEGIN

      v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table, v_num_cols);
      IF v_success != 1 THEN
        RAISE e_no_cfg;
      END IF;
    EXCEPTION
      WHEN e_no_cfg THEN
        RAISE_APPLICATION_ERROR(-20102, 'Unable to get item information from the configuration table.');
    END;

    /* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
      BEGIN
        /* read one line */
        v_inline := v_text_line.SIPL_LINE_TEXT;

        v_comment := NULL;

        v_line_cnt := v_line_cnt + 1;
        v_status := 'A';

        -- send line to parsing routine:
        BEGIN
          sim2_import_pkg.PARSE_LINE_TABLE(v_inline, v_line_table, v_num_cols, v_delimiter,
                                           v_bad_parse,v_import_log_seq);

          IF v_bad_parse THEN
            RAISE e_e_bad_delimiter;
          END IF;

        EXCEPTION
          WHEN e_e_bad_delimiter THEN
            RAISE e_bad_delimiter;

          WHEN OTHERS THEN
            sim2_import_pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' parsing one line.',NULL);
            RAISE e_fileread;
        END;
      END;

      /* loop through all project columns */
      FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP

        /* get dflt position, item name, and column sequence */
        v_dflt_pos := v_columns.SIC_DFLT_POS;
        v_item_name := v_columns.SIC_NAME;
        v_col_seq := v_columns.SIC_SEQ;
				v_trans_exist := TRUE;

        /* if column is included on the import, continue */
        IF v_pos_table(v_dflt_pos) IS NOT NULL THEN

          /* get the actual item, or use default if null */
          IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
            v_item := v_line_table(v_pos_table(v_dflt_pos));
          ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL AND v_default_table(v_dflt_pos) IS NOT NULL THEN
            v_item := v_default_table(v_dflt_pos);
          ELSE
            v_item := NULL;
          END IF;

          /* assign item to appropriate variables and take care of any other procedural issues that need to be
             taken care of for or with a particular item */

          -- Does V_ITEM contain only spaces? (or NULL)
          IF (TRIM(V_ITEM) IS NULL) THEN
            -- if yes
            
            -- by definition V_ITEM is NULL!
            V_ITEM := NULL;

          END IF;

          IF v_item_name = 'ProjectID' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := UPPER(RTRIM(v_item)); -- 061203 slp - Set ID to uppercase and remove trailing spaces.
  								/* bka - Check to make sure that the project ID does not exceed 8 characters */
                  IF substr(v_item,9,9) IS NULL THEN
                    v_project_id := v_item;
                  ELSE
                    v_project_id := substr(v_item,1,8);
                    v_status := 'N';
                    sim2_import_pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Project ID, '||v_item||' is longer than 8 characters.',v_project_seq);
                  END IF;
  
                  /* get the sequence for this project */
                  OPEN c_get_project_seq;
                  FETCH c_get_project_seq INTO v_project_seq;
                    IF c_get_project_seq%NOTFOUND THEN
                      v_status := 'N'; -- SET STATUS TO N
                      v_project_seq := NULL;
                      RAISE e_project_seq;
                    END IF;
                  CLOSE c_get_project_seq;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
  
              ELSE
                /* write error message if project id has not been provided */
                v_project_id := NULL;
                v_comp := write_error_message(v_item_name,TRUE/*p_req*/,v_line_cnt,FALSE/*p_provided*/,v_status,v_comment,v_project_seq);
                v_status := 'N'; -- SET STATUS TO N
              END IF;
            EXCEPTION
              WHEN e_project_seq THEN
                RAISE_APPLICATION_ERROR(-20150,'Unable to get sequence for sim_projects table.');

              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting project id, line: '||v_line_cnt,v_project_seq);
            END;

          ELSIF v_item_name = 'Name' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_project_name := v_item;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if project name has not been provided */
                v_project_name := NULL;
                v_comp := write_error_message(v_item_name,TRUE/*p_req*/,v_line_cnt,FALSE/*p_provided*/,v_status,v_comment,v_project_seq);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting project name, line: '||v_line_cnt,v_project_seq);
            END;

          ELSIF v_item_name = 'StartDate' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_start_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_project_seq);
                  IF v_start_date IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if project start date has not been provided */
                v_start_date := NULL;
                v_comp := write_error_message(v_item_name,TRUE/*p_req*/,v_line_cnt,FALSE/*p_provided*/,v_status,v_comment,v_project_seq);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting start date, line: '||v_line_cnt,v_project_seq);
            END;

          ELSIF v_item_name = 'Duration' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
                  v_duration := v_item;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if duration has not been provided */
                v_duration := NULL;
                v_comp := write_error_message(v_item_name,TRUE/*p_req*/,v_line_cnt,FALSE/*p_provided*/,v_status,v_comment,v_project_seq);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting duration, line: '||v_line_cnt,v_project_seq);
            END;

          ELSIF v_item_name = 'Purpose' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_purpose := v_item;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if purpose has not been provided */
                v_purpose := NULL;
                v_comp := write_error_message(v_item_name,TRUE/*p_req*/,v_line_cnt,FALSE/*p_provided*/,v_status,v_comment,v_project_seq);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting purpose, line: '||v_line_cnt,v_project_seq);
            END;

          ELSIF v_item_name = 'Contact' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_contact := v_item;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if project contact has not been provided */
                v_contact := NULL;
                v_comp := write_error_message(v_item_name,FALSE/*p_req*/,v_line_cnt,FALSE/*p_provided*/,v_status,v_comment,v_project_seq);
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting contact, line: '||v_line_cnt,v_project_seq);
            END;

          ELSIF v_item_name = 'Document/Graphic' THEN
            BEGIN
              IF v_item IS NOT NULL THEN  
                v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);   
                IF v_item IS NOT NULL THEN       
                  IF sim2_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                    v_document := v_item;
                  ELSE
                    v_status := 'N'; -- Invalid document.
                    v_document := NULL;
                  END IF;  
                ELSE
                  v_status := 'N';   --Translated to a NULL value.  Set to 'N'.
                  v_document := NULL;
                END IF;
              ELSE
                v_document := NULL;
              END IF;  
            EXCEPTION
              WHEN OTHERS THEN
                sim2_import_pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' - getting project document/graphic, line: '||v_line_cnt,v_project_seq);
            END;
          END IF;
        END IF;
      END LOOP; -- end loop through columns

      -- Checks if there is a duplicate Project ID or Project Name in SIM
      v_sim_proj_success := check_sim_proj(v_project_id, p_org_id, v_project_name);
      IF v_sim_proj_success = 1 THEN
        v_status := 'N';
      END IF;        
      
      v_storet_key := check_storet_proj(v_project_id, p_org_id, v_project_name, v_project_seq);

    --  insert into bka_test values(v_storet_key.is_number);

      IF v_storet_key.is_number IS NOT NULL THEN
        /* bka - project already exists in STORET, create in SIM but do not allow it to be exported */
        v_status := 'N';
        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'Line '||v_line_cnt||': ' ||
          'Project ' || v_project_id || ' already exists in STORET.',
          v_project_seq
        );
      END IF;

      /* create project record in SIM */
      BEGIN

        v_project_rec.SPJ_SEQ := v_project_seq;
        v_project_rec.SPJ_TSMORGAN_IS_NUMBER := v_org_is_nbr;
        v_project_rec.SPJ_ORG_ID := p_org_id;
        v_project_rec.SPJ_ID := v_project_id;
        v_project_rec.SPJ_TSMPROJ_IS_NUMBER := NULL;
        v_project_rec.SPJ_TSMPROJ_ORG_ID := NULL;
        v_project_rec.SPJ_NAME := v_project_name;
        v_project_rec.SPJ_START_DATE := v_start_date;
        v_project_rec.SPJ_DURATION := v_duration;
        v_project_rec.SPJ_PURPOSE := v_purpose;
        v_project_rec.SPJ_CONTACT := v_contact;
        v_project_rec.SPJ_DOCUMENT := v_document;
        v_project_rec.SPJ_EXPORT_SEQ := NULL;
        v_project_rec.SPJ_IMPORT_SEQ := v_import_log_seq;
        v_project_rec.SPJ_STATUS := v_status;

        /* write data to sim table */
        v_success := create_project(v_project_rec);

        v_project_cnt := v_project_cnt + 1;

      EXCEPTION
        WHEN OTHERS THEN
          sim2_import_pkg.write_error(v_import_log_seq, SQLERRM||' - creating project record.',v_project_seq);
      END;

      /* update import lines table with project sequence */
      UPDATE SIM_IMP_LINES SET SIPL_SPJ_SEQ = v_project_seq
        WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;

      /* wipe out record containing project data */
      v_project_rec := v_project_copy_rec;

    END LOOP; -- end loop through text lines

    /* count the number of projects with errors */
    OPEN c_comp_count(v_import_log_seq);
    FETCH c_comp_count INTO v_complete_cnt;
    CLOSE c_comp_count;

    v_error_cnt := v_line_cnt - v_complete_cnt;

    /* update counts in the import log */
    UPDATE SIM_IMPORT_LOG
       SET SIL_ROWS_IMPORTED = v_line_cnt,
           SIL_ROWS_ERRORS = v_error_cnt,
           SIL_ROWS_READY = v_complete_cnt
       WHERE SIL_SEQ = v_import_log_seq;

    COMMIT;

    sim2_import_pkg.WRITE_ERROR (v_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_project_cnt)||' projects loaded',null);
    sim2_import_pkg.WRITE_ERROR (v_import_log_seq ,'Finished at '||to_char(SYSDATE,'HH24:MI'),null);

  END;

EXCEPTION

   WHEN e_no_cfg THEN
     RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');

   WHEN e_bad_delimiter THEN
     RAISE_APPLICATION_ERROR(-20197,'The import file contains the wrong number of columns on line: '||v_line_cnt);

   WHEN OTHERS THEN
       sim2_import_pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR(SQLERRM||' - in sim2_import_project_pkg.import_projects. Line: '||TO_CHAR(v_line_cnt),1,240),v_project_seq);
       sim2_import_pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_project_cnt)||' projects loaded',null);
       sim2_import_pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||to_char(sysdate,'HH24:MI'),null);


END import_projects;

END sim2_import_project_pkg;
/
