CREATE OR REPLACE PACKAGE Sim2_Import_Station_Pkg IS

  /* SIM 201*/

  -- Author  : KHENA S,Dan L
  -- Major modification 4/02 by SF

  -- Created : 11/15/00 11:35:05 AM
  -- Purpose : import stations data into SIM tables from a text file
   TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;
   TYPE col_table_num_type IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;
   TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;
   TYPE t_storet_pk_type IS RECORD (
     is_number         NUMBER(8),
     org_id            VARCHAR2(8)
     );
   TYPE t_storet_tsmvstc_type IS RECORD (
     tsmvstc_is_number         NUMBER(8),
     tsmvstc_org_id            VARCHAR2(8)
     );
    TYPE t_storet_tsmmad_type IS RECORD (
     tsmmad_is_number         NUMBER(8),
     tsmmad_org_id            VARCHAR2(8)
     );
   TYPE t_aquifer_type    IS RECORD (
     tsmgeoun_cd          VARCHAR2(12),
     aquifer_name         VARCHAR2(80)
     );
   v_activity_count     t_activity_count;
   v_org_is_nbr         NUMBER(8) := 0;
   v_import_log_seq     NUMBER(10);
   v_status             VARCHAR2(1);
   v_line_cnt           NUMBER(10) := 0;

  -- Public function and procedure declarations

   PROCEDURE import_station (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                          p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER);

   FUNCTION  check_tsmmad_value(p_orig_item IN VARCHAR2,p_item_name IN VARCHAR2,
                             p_category IN VARCHAR2,p_subcategory IN VARCHAR2,
                             p_col_seq IN NUMBER,p_sta_seq IN NUMBER,p_imp_cfg_seq IN NUMBER,
                             p_imp_log_seq IN NUMBER)
                              RETURN VARCHAR2;

END Sim2_Import_Station_Pkg;
/
CREATE OR REPLACE PACKAGE BODY Sim2_Import_Station_Pkg IS

   FUNCTION check_translations(
            p_code IN VARCHAR2, 
            p_imp_cfg_seq IN NUMBER, 
            p_col_seq IN NUMBER,
            p_trans_exist OUT BOOLEAN) 
            RETURN VARCHAR2;

   PROCEDURE create_trans_record(
            p_code IN VARCHAR2,
            p_imp_cfg_seq IN NUMBER,
            p_col_seq IN NUMBER);
   
   FUNCTION CHECK_STORET_TRANSLATION(
            P_CODE IN VARCHAR2,
            P_IMP_CFG_SEQ IN NUMBER,
            P_COL_SEQ IN NUMBER,
            P_TRANS_EXIST OUT BOOLEAN,
            V_CONVERTED_ITEM OUT VARCHAR2,
            P_STA_SEQ IN NUMBER) 
            RETURN T_STORET_PK_TYPE;

/*********************************************************************************************************/
FUNCTION Validate_number(
         p_item IN VARCHAR2,
         p_item_name IN VARCHAR2,
         p_format IN VARCHAR2, 
         p_sta_seq IN NUMBER) 
         RETURN NUMBER IS
         
  -- 4/12/02 - SF
  -- This function is to test any value that is supposed to be a number. 
  -- If it's not a number a 0 is returned.

  v_return   NUMBER(10);

  BEGIN
    v_return := TO_NUMBER(p_item,p_format);
    RETURN 1;

  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||p_item_name||', '||P_item||', is not in an appropriate numeric format of '||p_format||'.' ,p_sta_seq);
      RETURN 0;
END Validate_number;

/*********************************************************************************************************/

/** This function checks to see if a station already exists by checking for existing station ID**/
FUNCTION check_if_station_exists(p_station_cd IN VARCHAR2,p_org_id IN VARCHAR2, p_sta_seq IN NUMBER) RETURN NUMBER IS

  v_station_cd     VARCHAR2(15);

  CURSOR c_compare_station_id(p_station_cd IN VARCHAR2,p_station_org_id IN VARCHAR2) IS
    SELECT IDENTIFICATION_CD
    FROM TSMSTATN
    WHERE  IDENTIFICATION_CD = RPAD(p_station_cd,15)
      AND TSMSTATN_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_compare_sim_stat_id(p_station_cd IN VARCHAR2,p_station_org_id IN VARCHAR2) IS
    SELECT STA_ID
    FROM SIM_STATIONS
    WHERE  STA_ID = p_station_cd
      AND STA_TSMORGAN_ORG_ID = p_org_id;

BEGIN

  /* Check STORET */
  OPEN c_compare_station_id(p_station_cd,p_org_id);
    FETCH c_compare_station_id INTO v_station_cd;
  CLOSE c_compare_station_id;

  IF v_station_cd IS NOT NULL THEN
    Sim2_Import_Pkg.WRITE_ERROR(
      v_import_log_seq,
      'Line ' || v_line_cnt || ': Station ' || TRIM(v_station_cd) || ' already exists in STORET.',
      p_sta_seq
    );
  END IF;

  /* Check SIM */
  IF v_station_cd IS NULL THEN
    OPEN c_compare_sim_stat_id(p_station_cd,p_org_id);
      FETCH c_compare_sim_stat_id INTO v_station_cd;
    CLOSE c_compare_sim_stat_id;

    IF v_station_cd IS NOT NULL THEN
      Sim2_Import_Pkg.WRITE_ERROR(
        v_import_log_seq,
        'Line ' || v_line_cnt || ': Station ' || v_station_cd || ' already exists in SIM.',
        NULL
      );
    END IF;
  END IF;

  /** Station is unique **/
  IF v_station_cd IS NULL THEN
    RETURN 1;
  ELSE
    RETURN 0;
  END IF;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check_if_station_exists.',NULL);
    RETURN 0;
END check_if_station_exists;

/*********************************************************************************************************/
FUNCTION get_org_info(p_tsmorgan_org_id IN VARCHAR2,p_tsmorgan_is_nbr OUT NUMBER) RETURN NUMBER IS
  CURSOR c_get_tsmorgan_nbr (p_cursor_org_id IN VARCHAR2) IS
    SELECT TSMORGAN_IS_NUMBER FROM TSMORGAN WHERE ORG_ID = RPAD(p_cursor_org_id,8);
BEGIN
  OPEN c_get_tsmorgan_nbr (p_tsmorgan_org_id);
    FETCH c_get_tsmorgan_nbr INTO p_tsmorgan_is_nbr;
    IF c_get_tsmorgan_nbr%NOTFOUND THEN
      p_tsmorgan_is_nbr := NULL;
    END IF;
  CLOSE c_get_tsmorgan_nbr;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END get_org_info;

/******************************************************************************************************/
FUNCTION get_instructions(p_imp_cfg_seq IN NUMBER, p_instruction_type IN VARCHAR2)
  RETURN VARCHAR2 IS
  v_instr_choice         VARCHAR2(15);
  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SICI_SICC_SEQ = p_cfg_seq;
  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SIIC_DFLT_FLAG = 1;
BEGIN
  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr INTO v_instr_choice;
    IF c_get_cfg_instr%NOTFOUND THEN
      /* no choice could be found, get default choice */
      OPEN c_get_instr_default(p_instruction_type);
      FETCH c_get_instr_default INTO v_instr_choice;
      CLOSE c_get_instr_default;
    END IF;
  CLOSE c_get_cfg_instr;
  RETURN v_instr_choice;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get_instructions.',NULL);
    RETURN NULL;
END get_instructions;

/******************************************************************************************************/
FUNCTION get_allowable_value(p_alval_seq IN NUMBER, p_sta_seq IN NUMBER) RETURN VARCHAR2 IS
  v_item_name         VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_alval_seq;

BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name INTO v_item_name;
    IF c_get_alval_name%NOTFOUND THEN
      v_item_name := NULL;
    END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - getting allowable value from sequence.',p_sta_seq);
END get_allowable_value;


/******************************************************************************************************/
/* The get_latitude and get_longitude functions are used by stations, wells, 
and locations imports to interpret the values sent in to the Latitude and 
Longitude fields respectively. Several assumptions are made and they are 
noted in the code. */
FUNCTION get_latitude(v_item_in IN VARCHAR2, 
                      v_column_seq IN NUMBER, 
                      p_imp_cfg_seq IN NUMBER,
                      v_sta_seq IN NUMBER, 
                      v_line_cnt IN NUMBER, 
                      p_lat_degrees OUT NUMBER,
                      p_lat_minutes OUT NUMBER, 
                      p_lat_seconds OUT NUMBER, 
                      p_lat_dir OUT VARCHAR2,
                      p_lat_dec_min OUT NUMBER, 
                      p_lat_dec_deg OUT NUMBER) RETURN NUMBER IS

  v_lat_format     VARCHAR2(15);
  v_latitude       VARCHAR2(20);
  WhatsBad         VARCHAR2(15);
  v_item           VARCHAR2(60);
  v_temp_min       NUMBER(10,7);

  e_bad_format     EXCEPTION;
  e_bad_number     EXCEPTION;

  CURSOR c_get_format IS
    SELECT SIFC_FMT_CHOICE
      FROM SIM_IMP_FMT_CHOICES C, SIM_IMP_CFG_DETS D
      WHERE D.SICDT_SIC_SEQ = v_column_seq
      AND D.SICDT_SICC_SEQ = p_imp_cfg_seq
      AND D.SICDT_SIFC_SEQ = C.SIFC_SEQ;

BEGIN
  v_item := v_item_in;
  OPEN c_get_format;
    FETCH c_get_format INTO v_lat_format;      
    IF c_get_format%NOTFOUND THEN --require format
      Sim2_Import_Pkg.WRITE_ERROR(
        v_import_log_seq,
        'Line ' || v_line_cnt || ': Please choose a format for Latitude.',
        v_sta_seq
      );
    END IF;      
  CLOSE c_get_format;
  
  IF v_lat_format = 'DD-MM-SS.SSSS' THEN
    IF SUBSTR(v_item,3,1) != '-' OR SUBSTR(v_item,6,1) != '-'  THEN
      IF SUBSTR(v_item,2,1) = '-' AND SUBSTR(v_item,5,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item,1,2));
    p_lat_minutes := NVL(TO_NUMBER(SUBSTR(v_item,4,2)),0);
    p_lat_seconds := NVL(TO_NUMBER(SUBSTR(v_item,7,7)),0);
    v_temp_min := p_lat_minutes + (p_lat_seconds/60); --use this to keep extra precision
    p_lat_dec_min := TRUNC(v_temp_min,4);
    p_lat_dec_deg := TRUNC(p_lat_degrees + (v_temp_min/60),7);
    p_lat_dir := NULL;
    
  ELSIF v_lat_format = 'DD MM SS.SSSS' THEN
    IF SUBSTR(v_item,3,1) != ' ' OR SUBSTR(v_item,6,1) != ' '  THEN
      IF SUBSTR(v_item,2,1) = ' ' AND SUBSTR(v_item,5,1) = ' ' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item,1,2));
    p_lat_minutes := NVL(TO_NUMBER(SUBSTR(v_item,4,2)),0);
    p_lat_seconds := NVL(TO_NUMBER(SUBSTR(v_item,7,7)),0);
    v_temp_min := p_lat_minutes + (p_lat_seconds/60); --use this to keep extra precision
    p_lat_dec_min := TRUNC(v_temp_min,4);
    p_lat_dec_deg := TRUNC(p_lat_degrees + (v_temp_min/60),7);
    p_lat_dir := NULL;    
    
  ELSIF v_lat_format = 'DD-MM.MMMM' THEN
    IF SUBSTR(v_item,3,1) != '-' THEN
      IF SUBSTR(v_item,2,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item,1,2));
    p_lat_minutes := NVL(TO_NUMBER(SUBSTR(v_item,4,2)),0);
    p_lat_dec_min := NVL(TO_NUMBER(SUBSTR(v_item,4,7)),0);
    p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
    p_lat_dec_deg := TRUNC(p_lat_degrees + (p_lat_dec_min/60),7);
    p_lat_dir := NULL;

  ELSIF v_lat_format = 'DD.DDDDDDD' THEN
    IF SUBSTR(v_item,3,1) != '.' THEN
      IF SUBSTR(v_item,2,1) = '.' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    
    p_lat_dec_deg := TRUNC(TO_NUMBER(v_item),7);
    p_lat_degrees := TO_NUMBER(SUBSTR(v_item, 1,2));
    p_lat_dec_min := 60 * (p_lat_dec_deg - p_lat_degrees);
    p_lat_minutes := FLOOR(p_lat_dec_min);
    p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
    p_lat_dir := NULL;

  ELSIF v_lat_format = '-DD.DDDDDD' THEN
    
    -- start by making sure there is a sign on the front
    IF SUBSTR(v_item,1,1) <> '+' AND SUBSTR(v_item,1,1) <> '-' THEN 
      -- assume - (Northern hemisphere)
      v_latitude := '+'||v_item;
    ELSE
      v_latitude := v_item;
    END IF;
    
    IF LENGTH(v_latitude) > 3 THEN
      IF SUBSTR(v_latitude,4,1) <> '.' THEN
        RAISE e_bad_format;
      END IF;
    END IF;
    
    IF SUBSTR(v_latitude,1,1) = '-' THEN
      p_lat_dir := 'S';
    ELSE
      p_lat_dir := 'N';
    END IF;
    
    p_lat_dec_deg := TRUNC(TO_NUMBER(v_latitude),7);
    p_lat_degrees := ABS(TO_NUMBER(SUBSTR(v_latitude,2,2)));
    p_lat_dec_min := 60 * (ABS(p_lat_dec_deg) - p_lat_degrees);
    p_lat_minutes := FLOOR(p_lat_dec_min);
    p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
  END IF;

  --truncate dec degrees
  IF p_lat_dir = 'S' THEN
    IF abs(p_lat_dec_deg) < 10 THEN
      p_lat_dec_deg := TRUNC(p_lat_dec_deg,7);
    ELSE
      p_lat_dec_deg := TRUNC(p_lat_dec_deg,6);
    END IF;
  ELSE
    p_lat_dec_deg := TRUNC(p_lat_dec_deg,7);
  END IF;
  
  --Check to see if any of the values are out of reasonable range.
  IF p_lat_degrees < 0 OR p_lat_degrees > 90 THEN
    WhatsBad := 'Lat Degrees';
    RAISE e_bad_number;
  ELSIF p_lat_minutes < 0 OR p_lat_minutes > 60 THEN
    WhatsBad := 'Lat Minutes';
    RAISE e_bad_number;
  ELSIF p_lat_seconds < 0 OR p_lat_seconds > 60 THEN
    WhatsBad := 'Lat Seconds';
    RAISE e_bad_number;
  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_bad_format THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Latitude does not match chosen format.',v_sta_seq);
     RETURN 2;
  WHEN e_bad_number THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': '||WhatsBad||' is out of range.',v_sta_seq);
      RETURN 2;
  WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Latitude does not match chosen format.',v_sta_seq);
      RETURN 2;
END get_latitude;

/******************************************************************************************************/
FUNCTION get_longitude(v_item_in IN VARCHAR2, 
                       v_column_seq IN NUMBER, 
                       p_imp_cfg_seq IN NUMBER,
                       v_sta_seq IN NUMBER, 
                       v_line_cnt IN NUMBER, 
                       p_long_degrees OUT NUMBER,
                       p_long_minutes OUT NUMBER, 
                       p_long_seconds OUT NUMBER, 
                       p_long_dec_min OUT NUMBER,
                       p_long_dec_deg OUT NUMBER, 
                       p_long_dir OUT VARCHAR2) RETURN NUMBER IS

  v_long_format VARCHAR2(15);
  v_longitude   VARCHAR2(20);
  WhatsBad      VARCHAR2(15);
  v_item        VARCHAR2(60);
  
  e_bad_format  EXCEPTION;
  e_bad_number  EXCEPTION;
  
  CURSOR c_get_format IS
    SELECT SIFC_FMT_CHOICE
      FROM SIM_IMP_FMT_CHOICES C, SIM_IMP_CFG_DETS D
      WHERE D.SICDT_SIC_SEQ = v_column_seq
      AND D.SICDT_SICC_SEQ = p_imp_cfg_seq
      AND D.SICDT_SIFC_SEQ = C.SIFC_SEQ;

BEGIN
  v_item := v_item_in;
  OPEN c_get_format;
    FETCH c_get_format INTO v_long_format;
    IF c_get_format%NOTFOUND THEN --require a format
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,
        'Line ' || v_line_cnt || '" Please choose a format for Longitude.',
        v_sta_seq
      );
      RETURN 2;
    END IF;
  CLOSE c_get_format;

  IF v_long_format = 'DDD-MM-SS.SSSS' THEN
    IF SUBSTR(v_item,4,1) != '-' OR SUBSTR(v_item,7,1) != '-' THEN
      IF SUBSTR(v_item,3,1) = '-' AND SUBSTR(v_item,6,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_degrees := TO_NUMBER(SUBSTR(v_item,1,3));
    p_long_minutes := NVL(TO_NUMBER(SUBSTR(v_item,5,2)),0);
    p_long_seconds := NVL(TO_NUMBER(SUBSTR(v_item,8,7)),0);
    p_long_dec_min := p_long_minutes + (p_long_seconds/60);
    p_long_dec_deg := TRUNC(p_long_degrees + (p_long_dec_min/60),7);
    p_long_dir := NULL;
    
  ELSIF v_long_format = 'DDD MM SS.SSSS' THEN
    IF SUBSTR(v_item,4,1) != ' ' OR SUBSTR(v_item,7,1) != ' ' THEN
      IF SUBSTR(v_item,3,1) = ' ' AND SUBSTR(v_item,6,1) = ' ' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_degrees := TO_NUMBER(SUBSTR(v_item,1,3));
    p_long_minutes := NVL(TO_NUMBER(SUBSTR(v_item,5,2)),0);
    p_long_seconds := NVL(TO_NUMBER(SUBSTR(v_item,8,7)),0);
    p_long_dec_min := p_long_minutes + (p_long_seconds/60);
    p_long_dec_deg := TRUNC(p_long_degrees + (p_long_dec_min/60),7);
    p_long_dir := NULL;    

  ELSIF v_long_format = 'DDD-MM.MMMM' THEN
    IF SUBSTR(v_item,4,1) != '-' THEN
      IF SUBSTR(v_item,3,1) = '-' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_degrees := TO_NUMBER(SUBSTR(v_item,1,3));
    p_long_minutes := NVL(TO_NUMBER(SUBSTR(v_item,5,2)),0);
    p_long_dec_min := NVL(TO_NUMBER(SUBSTR(v_item,5,7)),0);
    p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
    p_long_dec_deg := TRUNC(p_long_degrees + (p_long_dec_min/60),7);
    p_long_dir := NULL;

  ELSIF v_long_format = 'DDD.DDDDDD' THEN
    IF SUBSTR(v_item,4,1) != '.' THEN
      IF SUBSTR(v_item,3,1) = '.' THEN
        v_item := '0'||v_item;
      ELSE
        RAISE e_bad_format;
      END IF;
    END IF;
    p_long_dec_deg := TRUNC(TO_NUMBER(v_item),7);
    p_long_degrees := TO_NUMBER(SUBSTR(v_item, 1,3));
    p_long_dec_min := 60 * (p_long_dec_deg - p_long_degrees);
    p_long_minutes := FLOOR(p_long_dec_min);
    p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
    p_long_dir := NULL;

  ELSIF v_long_format = '-DDD.DDDDD' THEN
    
    --start by making sure there is a sign on the front
    IF SUBSTR(v_item,1,1) <> '+' AND SUBSTR(v_item,1,1) <> '-' THEN --assume + (Eastern hemisphere)
      v_longitude := '+'||v_item;
    ELSE
      v_longitude := v_item;
    END IF;
    
    -- Pad longitude with 0 for 2 digit longitudes
    IF LENGTH(v_longitude) > 4 THEN
      IF SUBSTR(v_longitude,5,1) <> '.' THEN
        IF SUBSTR(v_longitude,4,1) = '.' THEN           
          v_longitude := SUBSTR(v_longitude,1,1)||'0'||SUBSTR(v_longitude,2,LENGTH(v_longitude)) ;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
    ELSE
      IF LENGTH(v_longitude) <> 4 THEN
        IF LENGTH(v_longitude) = 3 THEN
          v_longitude := SUBSTR(v_longitude,1,1)||'0'||SUBSTR(v_longitude,2,LENGTH(v_longitude)) ;
        ELSE 
          RAISE e_bad_format;
        END IF;
      END IF;  
    END IF;
    
    IF SUBSTR(v_longitude,1,1) = '+' THEN
      p_long_dir := 'E';
    ELSE
      p_long_dir := 'W';
    END IF;
    
    p_long_dec_deg := TRUNC(TO_NUMBER(v_longitude),7);
    p_long_degrees := ABS(TO_NUMBER(SUBSTR(v_longitude, 2,3)));
    p_long_dec_min := 60 * (ABS(p_long_dec_deg) - p_long_degrees);
    p_long_minutes := FLOOR(p_long_dec_min);
    p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
  END IF;

  --truncate dec_degrees
  IF p_long_dir = 'W' THEN
    IF abs(p_long_dec_deg) < 100 THEN
      p_long_dec_deg := TRUNC(p_long_dec_deg,6);
    ELSE
      p_long_dec_deg := TRUNC(p_long_dec_deg,5);
    END IF;
  ELSE
    IF p_long_dec_deg < 100 THEN
      p_long_dec_deg := TRUNC(p_long_dec_deg,7);
    ELSE
      p_long_dec_deg := TRUNC(p_long_dec_deg,6);
    END IF;
  END IF;

  --Check to see if any of the values are out of reasonable range.
  IF p_long_degrees < 0 OR p_long_degrees > 180 THEN
    WhatsBad := 'Long Degrees';
    RAISE e_bad_number;
  ELSIF p_long_minutes < 0 OR p_long_minutes > 60 THEN
    WhatsBad := 'Long Minutes';
    RAISE e_bad_number;
  ELSIF p_long_seconds < 0 OR p_long_seconds > 60 THEN
    WhatsBad := 'Long Seconds';
    RAISE e_bad_number;
  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_bad_format THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Longitude does not match chosen format.',v_sta_seq);
     RETURN 2;
  WHEN e_bad_number THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': '||WhatsBad||' is out of range.',v_sta_seq);
      RETURN 2;
  WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,SQLERRM||' Line '||v_line_cnt||': Longitude does not match chosen format.',v_sta_seq);
      RETURN 2;
END get_longitude;

/******************************************************************************************************
    The validate_primary_type function takes the original item and
    checks it against existing translations. If there is a translation 
    the translation is returned to IMPORT_STATIONS where it is used to 
    find the tsmvstc_is_nbr and org_id.
/****************************************************************************************************/

FUNCTION validate_primary_type(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER,
                              p_item_name IN VARCHAR2) RETURN VARCHAR2 IS

  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN;
  e_no_function               EXCEPTION;
  v_primary_type              VARCHAR2(30);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);

  CURSOR c_check_primary_type (p_primary_type IN VARCHAR2)IS
       SELECT PRIMARY_TYPE_CD
       FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_primary_type;
BEGIN
  /* function to look in the translations and STORET tables for items that 
     should already exist in hard coded values in STORET tables
     look in translations table to see if it needs to be swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_primary_type (v_converted_item);
      FETCH c_check_primary_type INTO v_primary_type;
      IF c_check_primary_type%NOTFOUND THEN
        v_primary_type := NULL;
      END IF;
    CLOSE c_check_primary_type;
    /*  call function to look in storet table for primary type */
    IF v_primary_type IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_id  := 'STA-TR';
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN --create translations
        /* if record could not be matched to anything in translations or STORET tables,
           create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
      END IF;
    ELSIF v_primary_type IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item||'.',p_sta_seq);
      v_status := 'N';
    END IF;
  ELSIF v_trans_exist THEN
    v_status := 'N';
  END IF;
  RETURN v_primary_type;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate primary type.line: '||v_line_cnt,p_sta_seq);
    RETURN NULL;
    v_status := 'N';
END validate_primary_type;

/******************************************************************************************************
    The validate_secondary_type function takes the original item and
    checks it against existing translations. If there is a translation 
    the translation is returned to IMPORT_STATIONS where it is used to 
    find the tsmvstc_is_nbr and org_id. If there is a valid secondary 
    type, the is number and org id from the primary type calculation
    will be overwritten.
    
    Updated for v2 as secondary type is now required
/****************************************************************************************************/
FUNCTION validate_secondary_type(p_orig_item IN VARCHAR2, p_primary_type IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                              p_col_seq IN NUMBER, p_sta_seq IN NUMBER, p_item_name IN VARCHAR2) RETURN VARCHAR2 IS

  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_secondary_type            VARCHAR2(30);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);

CURSOR c_check_secondary_type  (p_secondary_type IN VARCHAR2,p_primary_type IN VARCHAR2)
       IS SELECT SECONDARY_TYPE_CD FROM TSMVSTC
       WHERE SECONDARY_TYPE_CD = p_secondary_type AND PRIMARY_TYPE_CD = p_primary_type;
BEGIN
  /* function to look in the translations and STORET tables for items 
     that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_secondary_type (v_converted_item,p_primary_type);
      FETCH c_check_secondary_type INTO v_secondary_type;
      IF c_check_secondary_type%NOTFOUND THEN
        v_secondary_type := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Secondary Station Type: '||p_orig_item||' does not match with Primary Type: '||p_primary_type||'.',p_sta_seq);
      END IF;
    CLOSE c_check_secondary_type;

    /* call function to look in storet table for primary/secondary type */
    IF v_secondary_type IS NULL AND NOT v_trans_exist THEN
      v_instr_id  := 'STA-TR';

      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' OR v_instr_code = 'GWS-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
      END IF;
    ELSIF v_secondary_type IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item||'.',p_sta_seq);
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to N later in the program. */
    v_secondary_type := NULL;
  END IF;
  RETURN v_secondary_type;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate secondary type. line: '||v_line_cnt,p_sta_seq);
    RETURN NULL;
END validate_secondary_type;

/******************************************************************************************************
    The validate_county function takes the original item and checks 
    it against existing translations. cursor c_check_county uses 
    character data to search, while c_check_county_fips checks with the
    county FIPS code. First the p_orig_item is tested to see if it is 
    of character or numeric format, then it is checked against known 
    translations. If there is a translation the translation is returned 
    and used in a cursor to find the IS number org id from TSMGEOPA.
/****************************************************************************************************/
FUNCTION validate_county(
         p_orig_item   IN VARCHAR2, 
         p_imp_cfg_seq IN NUMBER, 
         p_item_name   IN VARCHAR2,
         p_col_seq     IN NUMBER, 
         p_sta_seq     IN NUMBER) 
         RETURN VARCHAR2 IS
         
  v_converted_item  VARCHAR2(25);
  v_trans_exist     BOOLEAN := TRUE;
  e_no_function     EXCEPTION;
  v_county          VARCHAR2(25);
  v_instr_code      VARCHAR2(15);
  v_instr_id        VARCHAR2(8);

  -- Get County from STORET using "text" County
  -- jah 2-26-04 Convert p_county to uppercase because STORET GUI converts County to uppercase
  CURSOR c_check_county (p_county IN VARCHAR2) IS 
  SELECT COUNTY_NAME 
  FROM   TSMGEOPA
  WHERE  COUNTY_NAME = UPPER(p_county);
  
  -- Get County from STORET using "FIPS" County
  -- jah 2-26-04 Convert p_county to uppercase because STORET GUI converts County to uppercase
  CURSOR c_check_county_fips (p_county IN VARCHAR2) IS 
  SELECT FIPS_COUNTY_CODE 
  FROM   TSMGEOPA
  WHERE  FIPS_COUNTY_CODE = UPPER(p_county);
         
BEGIN
  
  -- Translate the item value (return original value if no translation exists)
  v_converted_item := check_translations(
    p_orig_item, 
    p_imp_cfg_seq, 
    p_col_seq, 
    v_trans_exist
  );
  
  -- did CHECK_TRANSLATIONS work properly?
  IF (v_converted_item IS NOT NULL) THEN
    -- if yes
  
    -- is this a FIPS county code?
    IF (TRANSLATE(v_converted_item, 'a1234567890.- ', 'a') IS NULL) THEN
      -- if yes

      -- Try to retrieve the county using FIPS county
      OPEN c_check_county_fips (v_converted_item);
      FETCH c_check_county_fips INTO v_county;

      -- Did we retrieve the county using FIPS county?
      IF c_check_county_fips%NOTFOUND THEN
        -- if no
        
        -- return NULL
        v_county := NULL;
        
        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'FIPS county code is invalid.',
          p_sta_seq
        );
        
      END IF;
      CLOSE c_check_county_fips;
      
    ELSE
      -- if no (it is not a FIPS county code, therefore it is a text county code)

      -- Try to retrieve the text county record
      OPEN c_check_county (v_converted_item);
      FETCH c_check_county INTO v_county;
      
      -- Did we retrieve the text county record?
      IF c_check_county%NOTFOUND THEN
        -- if no
        
        -- return NULL
        v_county := NULL;
        
        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'County name is invalid.',
          p_sta_seq
        );
        
      END IF;
      CLOSE c_check_county;
      
    END IF;
    
    -- did the calling routine pass a NULL value?
    IF (v_county IS NULL) AND (NOT v_trans_exist) THEN
      -- if yes
      
      v_instr_id  := 'STA-TR';
      
      -- check instructions to see if translation record should be created
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      
      -- should we create a translation record?
      IF v_instr_code = 'STA-TR-01' THEN
        -- if yes

        -- create translation record and tell the end-user        
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||
          ' is not a valid value - translation record has been created.',
          p_sta_seq
        );

      ELSE

        Sim2_Import_Pkg.write_error(
          v_import_log_seq, 
          'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||
          ' is not a valid value.',
          p_sta_seq
        );

      END IF;
      
    -- did CHECK_TRASNSLATIONS translate to NULL?
    ELSIF v_county IS NULL AND v_trans_exist THEN
      -- if yes

      Sim2_Import_Pkg.write_error(
        v_import_log_seq,
        'Line '||v_line_cnt||': '||p_orig_item||
        ' has been mapped to the invalid storet value: '||v_converted_item,
        p_sta_seq
      );
      
    END IF;
    
  ELSIF v_trans_exist THEN
    -- if no, CHECK_TRANSLATIONS did NOT work properly (i.e., it translated to NULL)
    
    -- return NULL.
    -- Calling routine is responsible for error message, if any,
    -- and for changing Status to 'N' to prevent migration to STORET.
    v_county := NULL;

  END IF;

  RETURN v_county;
  
EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(
      v_import_log_seq, 
      SQLERRM||' - in validate county name.',
      p_sta_seq
    );
    RETURN NULL;

END validate_county;

/******************************************************************************************************
    The validate_state function takes the original item and checks 
    it against existing translations. If there is a translation the 
    translation is returned and used in a cursor to find the IS number
    org id from TSMGEOPA.
/****************************************************************************************************/
FUNCTION validate_state(
         p_orig_item   IN VARCHAR2, 
         p_imp_cfg_seq IN NUMBER, 
         p_item_name   IN VARCHAR2,
         p_col_seq     IN NUMBER, 
         p_sta_seq     IN NUMBER) 
         RETURN VARCHAR2 IS
         
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_state                     VARCHAR2(2);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);

  -- Get State using "text" State
  -- jah 2-26-04 Convert p_state to uppercase because STORET GUI converts State to uppercase
  CURSOR c_check_state (p_state IN VARCHAR2) IS 
  SELECT STATE_POSTAL_CODE 
  FROM   TSMGEOPA
  WHERE  STATE_POSTAL_CODE = UPPER(p_state);

  -- Get State using "FIPS" State
  -- jah 2-26-04 Convert p_state to uppercase because STORET GUI converts State to uppercase
  CURSOR c_check_state_fips (p_state IN VARCHAR2) IS 
  SELECT STATE_FIPS_CODE 
  FROM   TSMGEOPA
  WHERE  STATE_FIPS_CODE = p_state;
  
BEGIN
  /* function to look in the translations and STORET tables for 
     items that should already exist in hard coded values in 
     STORET tables */
  /* look in translations table to see if it needs to swapped */

  v_converted_item := check_translations(
    p_orig_item, 
    p_imp_cfg_seq, 
    p_col_seq, 
    v_trans_exist
   );
   
   -- did check_translations work properly?
   IF v_converted_item IS NOT NULL THEN
     -- if yes
     
     -- is converted item numeric?
     IF TRANSLATE(v_converted_item, 'a1234567890.- ', 'a') IS NULL THEN
       -- if yes (the it's a FIPS county code)
       
       OPEN c_check_state_fips (v_converted_item);
       FETCH c_check_state_fips INTO v_state;
       IF c_check_state_fips%NOTFOUND THEN
         v_state := NULL;
       END IF;
       CLOSE c_check_state_fips;
       
    ELSE
      -- if no (i.e., converted item is text
      
      OPEN c_check_state (v_converted_item);
      FETCH c_check_state INTO v_state;
      IF c_check_state%NOTFOUND THEN
        v_state := NULL;
      END IF;
      CLOSE c_check_state;
      
    END IF;
    
    /* call function to look in storet table for primary/secondary type */
    IF v_state IS NULL AND NOT v_trans_exist THEN
      v_instr_id  := 'STA-TR';

      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
      /* if record could not be matched to anything in translations or STORET tables,
         create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
      END IF;
      
    ELSIF v_state IS NULL AND v_trans_exist THEN
      
      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(
        v_import_log_seq,
        'Line '||v_line_cnt||': '||
        p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,
        p_sta_seq
      );
      
    END IF;
    
  ELSIF v_trans_exist THEN
  
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_state := NULL;
  
  END IF;
  
  RETURN v_state;
  
EXCEPTION
  
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(
      v_import_log_seq, 
      SQLERRM||' - in validate state.',
      p_sta_seq
    );
    
    RETURN NULL;
    
END validate_state;

/******************************************************************************************************
    The validate_estuary_state function takes the original item 
    and checks it against existing translations.  Validate the 
    converted item against the state name in the TSMESTRY table.  
    We could not use the State FIPS code or Postal Code as the 
    state names in the TSMGEOPA table do not match the valid 
    state names in the TSMESTRY table.  For this reason, we had 
    to accept only the complete state name as they are defined
    in the TSMESTRY table.
/****************************************************************************************************/
FUNCTION validate_estuary_state(
         p_orig_item   IN VARCHAR2, 
         p_imp_cfg_seq IN NUMBER, 
         p_item_name   IN VARCHAR2,
         p_col_seq     IN NUMBER, 
         p_sta_seq     IN NUMBER) 
         RETURN VARCHAR2 IS
         
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_estuary_state             VARCHAR2(40);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_storet_pk                 T_STORET_PK_TYPE;

CURSOR c_check_state(p_converted_item IN VARCHAR2) IS
  SELECT state_province_name
  FROM tsmestry
  WHERE state_province_name = p_converted_item;

BEGIN
  
  -- jah 4-29-04 Changed the following to check_translations (task 472)
  v_converted_item := check_translations(
    p_orig_item,
    p_imp_cfg_seq,
    p_col_seq,
    v_trans_exist
  );

  -- did the call to check_translation work?  
  IF (v_converted_item IS NOT NULL) THEN
    -- if yes

    -- try to retrieve the estuary state
    OPEN c_check_state (v_converted_item);
    FETCH c_check_state INTO v_estuary_state;

    -- did we retrieve the estuary state?
    IF c_check_state%NOTFOUND THEN
      -- if no
      v_estuary_state := NULL;
    END IF;
    CLOSE c_check_state;

    IF v_estuary_state IS NULL AND NOT v_trans_exist THEN

      v_instr_id  := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);

      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
          create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
      END IF;

    ELSIF v_estuary_state IS NULL AND v_trans_exist THEN

      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq);

    END IF;

  ELSIF v_trans_exist THEN

    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_estuary_state := NULL;
    
  END IF;

  RETURN v_estuary_state;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate estuary state.',p_sta_seq);
    RETURN NULL;
END validate_estuary_state;

/******************************************************************************************************
    The validate_huc function takes the original item and checks 
    it against existing translations.  If there is a translation 
    the translation is returned and used in a cursor to find the 
    IS number org id from TSMFHU.
/****************************************************************************************************/
FUNCTION validate_huc(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER,p_sta_seq IN NUMBER) RETURN VARCHAR2 IS
  v_converted_item            VARCHAR2(8);
  v_trans_exist               BOOLEAN := TRUE;
  e_no_function               EXCEPTION;
  v_huc                       VARCHAR2(8);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);
  v_storet_key                SIM2_IMPORT_STATION_PKG.T_STORET_PK_TYPE;    

CURSOR c_check_huc  (p_huc IN VARCHAR2)
       IS SELECT HYDROLOGIC_UNIT_CD FROM TSMFHU
       WHERE HYDROLOGIC_UNIT_CD = RPAD(p_huc,8);
BEGIN
  /* function to look in the translations and STORET tables for items 
     that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
   v_storet_key := check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist,v_converted_item,p_sta_seq);
 
   IF v_converted_item IS NOT NULL THEN
     OPEN c_check_huc (v_converted_item);
       FETCH c_check_huc INTO v_huc;
       IF c_check_huc%NOTFOUND THEN
         v_huc := NULL;
       END IF;
     CLOSE c_check_huc;
     IF v_huc IS NULL AND NOT v_trans_exist THEN
       v_instr_id  := 'STA-TR';
       /* check instructions to see if translation record should be created */
       v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
       IF v_instr_code = 'STA-TR-01' THEN
       /* if record could not be matched to anything in translations or STORET tables,
         create new record */
         create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
       ELSE
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
       END IF;
     ELSIF v_huc IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       Sim2_Import_Pkg.write_error(v_import_log_seq, p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq);
     END IF;
   ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
     v_huc := NULL;
   END IF;
   RETURN v_huc;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate huc.',p_sta_seq);
    RETURN NULL;
END validate_huc;

/******************************************************************************************************/
FUNCTION validate_primary_estuary(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_est_st_name IN VARCHAR2, p_tsmestry_is_nbr OUT NUMBER, p_sta_seq IN NUMBER)
                              RETURN VARCHAR2 IS

  v_converted_item            VARCHAR2(30);
  v_trans_exist               BOOLEAN := TRUE;

  v_estuary                   VARCHAR2(30);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);


CURSOR c_check_estuary (p_estuary IN VARCHAR2, p_estuary_state IN VARCHAR2) IS
       SELECT TSMESTRY_IS_NUMBER FROM TSMESTRY
       WHERE NAME = p_estuary
       AND STATE_PROVINCE_NAME = p_estuary_state
       AND PRIMARY_INDICATOR = 'Y';

BEGIN
  /* function to look in the translations and STORET tables for items 
     that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */

   v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
     OPEN c_check_estuary (v_converted_item, p_est_st_name);
       FETCH c_check_estuary INTO p_tsmestry_is_nbr;
       IF c_check_estuary%NOTFOUND THEN
         v_estuary := NULL;
       ELSE
         v_estuary := v_converted_item;
       END IF;
     CLOSE c_check_estuary;

     IF v_estuary IS NULL AND NOT v_trans_exist THEN
       v_instr_id  := 'STA-TR';
       /* check instructions to see if translation record should be created */
       v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
       IF v_instr_code = 'STA-TR-01' THEN
       /* if record could not be matched to anything in translations or STORET tables,
         create new record */
         create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
       ELSE
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',NULL);
       END IF;
     ELSIF v_estuary IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL);
     END IF;

   ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
     v_estuary := NULL;
   END IF;

RETURN v_estuary;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate primary estuary.',NULL);
    RETURN NULL;
END validate_primary_estuary;

/******************************************************************************************************/
FUNCTION validate_secondary_estuary(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_item_name IN VARCHAR2,
                              p_col_seq IN NUMBER, p_primary_is_nbr IN NUMBER, p_tsmestry_is_nbr OUT NUMBER,
                              p_sta_seq IN NUMBER)
                              RETURN VARCHAR2 IS

  v_converted_item            VARCHAR2(30);
  v_trans_exist               BOOLEAN := TRUE;

  v_estuary                   VARCHAR2(30);
  v_instr_code                VARCHAR2(15);
  v_instr_id                  VARCHAR2(8);


CURSOR c_check_estuary (p_estuary IN VARCHAR2, p_primary IN NUMBER) IS
       SELECT TSMESTRY_IS_NUMBER
       FROM TSMESTRY
       WHERE TSMESTRY0IS_NUMBER = p_primary
       AND NAME = p_estuary;

BEGIN
  /* function to look in the translations and STORET tables for items 
     that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
   v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
   IF v_converted_item IS NOT NULL THEN
     OPEN c_check_estuary (v_converted_item, p_primary_is_nbr);
       FETCH c_check_estuary INTO p_tsmestry_is_nbr;
       IF c_check_estuary%NOTFOUND THEN
         v_estuary := NULL;
       ELSE
         v_estuary := v_converted_item;
       END IF;
     CLOSE c_check_estuary;

     IF v_estuary IS NULL AND NOT v_trans_exist THEN
       v_instr_id  := 'STA-TR';
       /* check instructions to see if translation record should be created */
       v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
       IF v_instr_code = 'STA-TR-01' THEN
       /* if record could not be matched to anything in translations or STORET tables,
         create new record */
         create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
       ELSE
         Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',NULL);
       END IF;
     ELSIF v_estuary IS NULL AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL);
     END IF;

   ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
     v_estuary := NULL;
   END IF;

RETURN v_estuary;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in validate secondary estuary.',NULL);
    RETURN NULL;
END validate_secondary_estuary;

/******************************************************************************************************/
FUNCTION  validate_nal(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
          p_item_name IN VARCHAR2,p_col_seq IN NUMBER,v_native_name IN VARCHAR,
          v_native_state IN VARCHAR2, v_which IN NUMBER,p_sta_seq IN NUMBER) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given State and 
  -- NAL code are in the STORET TSMNAL table.

  v_nal_name                  VARCHAR2(40);
  v_nal_code                  VARCHAR2(5);
  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  v_nal                       VARCHAR2(40);
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);

  CURSOR c_check_nal(p_native_name IN VARCHAR2, p_native_state IN VARCHAR2) IS
    SELECT TSMNAL_CD
      FROM TSMNAL
      WHERE NAME = p_native_name
      AND TSMNAL_STATE = p_native_state;

  CURSOR c_check_name(p_native_name IN VARCHAR2) IS
    SELECT NAME
      FROM TSMNAL
      WHERE NAME = p_native_name;

  CURSOR c_check_state(p_native_state IN VARCHAR2) IS
    SELECT NAME
      FROM TSMNAL
      WHERE TSMNAL_STATE = p_native_state;


   BEGIN
     v_nal := NULL;
     IF v_which <> 3 THEN
       v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
     ELSE
       v_converted_item := p_orig_item;
     END IF;

     IF v_converted_item IS NOT NULL THEN
       IF v_which = 1 THEN --check code
         OPEN c_check_name(v_converted_item);
           FETCH c_check_name INTO v_nal_name;
           IF c_check_name %NOTFOUND THEN
             v_nal := NULL;
           ELSE
             v_nal := v_nal_name;
           END IF;
         CLOSE c_check_name;
       ELSIF v_which = 2 THEN
             --check state
         OPEN c_check_state(v_converted_item);
           FETCH c_check_state INTO v_nal_name;
           IF c_check_state %NOTFOUND THEN
             v_nal := NULL;
           ELSE
             v_nal := v_converted_item;
           END IF;
         CLOSE c_check_state;
       ELSE  --check them both together
         OPEN c_check_nal(v_native_name, v_native_state);
           FETCH c_check_nal INTO v_nal_code;
           IF c_check_nal %NOTFOUND THEN
             RETURN 'BAD';
           ELSE
             RETURN v_nal_code;
           END IF;
         CLOSE c_check_nal;
       END IF;

       IF v_nal IS NULL AND NOT v_trans_exist THEN
         v_instr_id  := 'STA-TR';
         /* check instructions to see if translation record should be created */
         v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
         IF v_instr_code = 'STA-TR-01' THEN
         /* if record could not be matched to anything in translations or STORET tables,
           create new record */
           create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
         ELSE
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
         END IF;
       ELSIF v_nal IS NULL AND v_trans_exist THEN
        /*  item has been translated to an invalid storet value */
         Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||p_item_name||', '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq);
       END IF;
     ELSIF v_trans_exist THEN
      /* item exists in translations table but is not mapped to anything, write error message.
         Status will be changed to 'N' later in the program. */
       v_nal := NULL;
     END IF;

     RETURN v_nal;

   EXCEPTION
     WHEN OTHERS THEN
       Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - checking NAL.',p_sta_seq);
       RETURN 'BAD';
END validate_nal;

/******************************************************************************************************/

FUNCTION  validate_prmvl_item(p_orig_item IN CHAR, p_imp_cfg_seq IN NUMBER,
          p_item_name IN VARCHAR2,p_col_seq IN NUMBER,p_table_name IN VARCHAR,
          p_column_name IN VARCHAR2,p_sta_seq IN NUMBER) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item            VARCHAR2(40);
  v_trans_exist               BOOLEAN := TRUE;
  v_storet_value             VARCHAR2(40);
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);

  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT FIELD_VALUE
      FROM TSMPRMVL
      WHERE TABLE_NAME = RPAD(p_table_name,8)
      AND FIELD_NAME = RPAD(p_column_name,20)
      AND UPPER(FIELD_VALUE) = RPAD(UPPER(p_item),40);


   BEGIN
     v_storet_value := NULL;
     v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

     IF v_converted_item IS NOT NULL THEN
       OPEN c_check_prmvl(v_converted_item);
           FETCH c_check_prmvl INTO v_storet_value;
           IF c_check_prmvl %NOTFOUND THEN
             v_storet_value := NULL;
           END IF;
         CLOSE c_check_prmvl;

       IF v_storet_value IS NULL AND NOT v_trans_exist THEN
         v_instr_id  := 'STA-TR';
         /* check instructions to see if translation record should be created */
         v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
         IF v_instr_code = 'STA-TR-01' THEN
         /* if record could not be matched to anything in translations or STORET tables,
           create new record */
           create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
         ELSE
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
         END IF;
       ELSIF v_storet_value IS NULL AND v_trans_exist THEN
        /*  item has been translated to an invalid storet value */
         Sim2_Import_Pkg.write_error(v_import_log_seq,p_item_name||', '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq);
       END IF;
     ELSIF v_trans_exist THEN
      /* item exists in translations table but is not mapped to anything, write error message.
         Status will be changed to 'N' later in the program. */
       v_storet_value := NULL;
       Sim2_Import_Pkg.write_error(v_import_log_seq,p_item_name||', '||p_orig_item||' exists in translation table but has not been mapped to a STORET code. ',p_sta_seq);
     END IF;

     RETURN v_storet_value;

   EXCEPTION
     WHEN OTHERS THEN
       Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - checking STORET PRMVL.',p_sta_seq);
       RETURN NULL;
END validate_prmvl_item;

/******************************************************************************************************/
FUNCTION  check_tsmmad_value(p_orig_item IN VARCHAR2,p_item_name IN VARCHAR2,
                             p_category IN VARCHAR2,p_subcategory IN VARCHAR2,
                             p_col_seq IN NUMBER,p_sta_seq IN NUMBER,p_imp_cfg_seq IN NUMBER,
                             p_imp_log_seq IN NUMBER)
                              RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item            VARCHAR2(120);
  v_trans_exist               BOOLEAN := TRUE;
  v_storet_value              VARCHAR2(12);
  v_instr_id                  VARCHAR2(8);
  v_instr_code                VARCHAR2(15);
  v_storet_key                SIM2_IMPORT_STATION_PKG.T_STORET_PK_TYPE;  

  CURSOR c_check_tsmmad(p_item IN VARCHAR2) IS
    SELECT ID_CODE
      FROM TSMMAD
      WHERE CATEGORY = RPAD(p_category,10)
      AND SUBCATEGORY = RPAD(p_subcategory,6)
      AND ID_CODE = RPAD(p_item,12);

   BEGIN
     v_storet_value := NULL;
     v_storet_key := check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist,v_converted_item,p_sta_seq);

     IF v_converted_item IS NOT NULL THEN
     
       OPEN c_check_tsmmad(v_converted_item);
         FETCH c_check_tsmmad INTO v_storet_value;
         IF c_check_tsmmad %NOTFOUND THEN
           v_storet_value := NULL;
         END IF;
       CLOSE c_check_tsmmad;

       IF v_storet_value IS NULL AND NOT v_trans_exist THEN
         v_instr_id  := 'STA-TR';
         /* check instructions to see if translation record should be created */
         v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
         IF v_instr_code = 'STA-TR-01' THEN
         /* if record could not be matched to anything in translations or STORET tables,
           create new record */
           create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
           Sim2_Import_Pkg.write_error(p_imp_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value - translation record has been created.',p_sta_seq);
         ELSE
           Sim2_Import_Pkg.write_error(p_imp_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not a valid value.',p_sta_seq);
         END IF;
       ELSIF v_storet_value IS NULL AND v_trans_exist THEN
        /*  item has been translated to an invalid storet value */
         Sim2_Import_Pkg.write_error(p_imp_log_seq,'Line '||v_line_cnt||': '||p_item_name||', '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq);
       END IF;
     ELSIF v_trans_exist THEN
      /* item exists in translations table but is not mapped to anything, write error message.
         Status will be changed to 'N' later in the program. */
       v_storet_value := NULL;
     END IF;

     RETURN v_storet_value;

   EXCEPTION
     WHEN OTHERS THEN
       Sim2_Import_Pkg.write_error(p_imp_log_seq,SQLERRM||' - checking STORET TSMMAD.',p_sta_seq);
       RETURN NULL;
END check_tsmmad_value;

/******************************************************************************************************
    The get_delimiter function checks the table sim_imp_config 
    to find the value entered for the delimiter.
/****************************************************************************************************/
FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER,p_delimiter OUT VARCHAR2) RETURN NUMBER IS

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT SICC_DELIMITER FROM sim_imp_cfg WHERE SICC_SEQ = p_sicc_seq;

BEGIN
  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg INTO p_delimiter;
    IF c_get_imp_cfg%NOTFOUND THEN
      RETURN 0;
    END IF;
  CLOSE c_get_imp_cfg;
  RETURN 1;
EXCEPTION
    WHEN OTHERS THEN
    RETURN 0;
END get_delimiter;

/******************************************************************************************************
    The get_item_pos function checks the table sim_imp_config 
    SIM_IMP_CFG_DETS SIM_IMP_COLS to find the value entered 
    for the the position of the item in the list
/****************************************************************************************************/
FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table IN OUT col_table_num_type,
                      p_default_table IN OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER IS
  v_import_type_seq              NUMBER(10);
  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT SICC_SID_SEQ FROM SIM_IMP_CFG WHERE SICC_SEQ = p_sicc_seq;
  CURSOR c_get_dflt_pos (p_type_seq IN NUMBER) IS
    SELECT SIC_DFLT_POS FROM SIM_IMP_COLS WHERE SIC_SID_SEQ = p_type_seq;
  CURSOR c_get_col_pos (p_sicc_seq IN NUMBER,p_dflt_pos IN NUMBER) IS
    SELECT SICDT_POS, SICDT_DEFAULT FROM SIM_IMP_CFG_DETS, SIM_IMP_COLS
      WHERE SIC_SEQ = SICDT_SIC_SEQ AND SICDT_SICC_SEQ = p_sicc_seq AND SIC_DFLT_POS = p_dflt_pos
      AND (SICDT_INCLUDED IS NOT NULL OR SICDT_GENERATE IS NOT NULL);
BEGIN
  /* get the import type in order to limit the columns selected to the right type */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type INTO v_import_type_seq;
   IF c_get_import_type%NOTFOUND THEN
     v_import_type_seq := NULL;
     RAISE_APPLICATION_ERROR(-20102, 'Unable to get import type from import configuration.');
   END IF;
  CLOSE c_get_import_type;
  p_col_cnt := 0;
  /* loop through import columns and assign actual position to table indexed by
     default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP
    /* get actual position and item default from configuration table */
    OPEN c_get_col_pos(p_sicc_seq,v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
      IF c_get_col_pos%NOTFOUND THEN
        /* if there is nothing in the configuration table for this column, position and default will be null */
        p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
        p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
      ELSE
        p_col_cnt := p_col_cnt + 1;
      END IF;
    CLOSE c_get_col_pos;
  END LOOP;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20103 , 'Failed to get column positions, item names, and column sequences.');
END get_item_pos;

/******************************************************************************************************/
FUNCTION get_column_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) RETURN NUMBER IS
  v_imp_col_seq                   NUMBER(10);
  e_no_cfg_found                  EXCEPTION;
  CURSOR c_get_col_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) IS
    SELECT SICDT_SIC_SEQ FROM sim_imp_cfg, sim_imp_cfg_dets
      WHERE sicdt_sicc_seq = sicc_seq AND SICC_SEQ = p_cfg_seq
      AND sicdt_pos = p_cfg_pos;
BEGIN
  OPEN c_get_col_seq(p_cfg_seq,p_cfg_pos);
  FETCH c_get_col_seq INTO v_imp_col_seq;
    IF c_get_col_seq%NOTFOUND THEN
      v_imp_col_seq := NULL;
      RAISE e_no_cfg_found;
    END IF;
  CLOSE c_get_col_seq;
  RETURN v_imp_col_seq;
EXCEPTION
  WHEN e_no_cfg_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - unable to find sequence in sim_imp_cfg_dets for pos: '||p_cfg_pos,NULL);
    RAISE_APPLICATION_ERROR(-20100,'Import configuration is not complete');
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - getting column sequence.',NULL);
    RETURN NULL;
END get_column_seq;

/******************************************************************************************************/
FUNCTION check_translations(
         p_code        IN VARCHAR2, 
         p_imp_cfg_seq IN NUMBER, 
         p_col_seq     IN NUMBER,
         p_trans_exist OUT BOOLEAN) 
         RETURN VARCHAR2 IS
                            
  v_storet_value             VARCHAR2(4000);
  v_station_seq							 NUMBER(10);
	
  CURSOR c_get_station_seq IS
  SELECT SIM_STATIONS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_storet_code(
         p_code        IN VARCHAR2,
         p_imp_cfg_seq IN NUMBER,
         p_col_seq     IN NUMBER) IS
  SELECT st_storet_code 
  FROM   SIM_TRANSLATIONS, SIM_IMP_CFG_DETS 
  WHERE  st_user_code = p_code
  AND    st_sicdt_seq = sicdt_seq AND sicdt_sicc_seq = p_imp_cfg_seq
  AND    sicdt_sic_seq = p_col_seq;

BEGIN

  -- Try to retrieve a translation for the item value
  OPEN c_get_storet_code(p_code,p_imp_cfg_seq,p_col_seq);
  FETCH c_get_storet_code INTO v_storet_value;

  -- did we retrieve a translation?
  IF c_get_storet_code%FOUND THEN
    -- if yes

    -- tell calling routine the return value is a translation
    p_trans_exist := TRUE;

    -- did we encounter a translation problem?
    IF v_storet_value IS NULL THEN
      -- if yes

      /* if p_code is in the translations table, but the translation value is NULL, 
         then write an error message. However, the calling routine is responsible for 
         changing the status 'N' to prevent migration to STORET. */
      
      -- Get the next station sequence number sfor display in the error message   
  		OPEN c_get_station_seq;
	    FETCH c_get_station_seq INTO v_station_seq;
			CLOSE c_get_station_seq;
      Sim2_Import_Pkg.write_error(
        v_import_log_seq, 
        'Line '||v_line_cnt||': '||p_code||
        ' exists in translations table but has not been translated to a value.',
        v_station_seq
      );
		END IF;
    
  ELSE
    -- if no (we did NOT find a translation)

    -- telling calling routine the return value is the original value
    p_trans_exist := FALSE;

    -- return the original value
    v_storet_value := p_code;

  END IF;
  CLOSE c_get_storet_code;

  RETURN v_storet_value;
  
EXCEPTION
  
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(
      v_import_log_seq, 
      SQLERRM||' - in check translations',
      NULL
    );
    
END check_translations;

/******************************************************************************************************/
FUNCTION check_storet_translation(P_CODE IN VARCHAR2,
                                  P_IMP_CFG_SEQ IN NUMBER,
                                  P_COL_SEQ IN NUMBER,
                                  P_TRANS_EXIST OUT BOOLEAN,
                                  V_CONVERTED_ITEM OUT VARCHAR2,
                                  P_STA_SEQ IN NUMBER) 
  RETURN T_STORET_PK_TYPE IS  
  
  v_storet_pk      SIM2_IMPORT_STATION_PKG.T_STORET_PK_TYPE;
  v_seq            NUMBER(10);
  v_table_name     VARCHAR2(80);
  v_column_name    VARCHAR2(80);
  v_dyna_cursor    VARCHAR2(200);
  v_comma_pos      NUMBER(10);
  
  TYPE SORT_CURSOR_TYPE IS REF CURSOR;
  sort_cursor            sort_cursor_type;
  
  CURSOR c_get_storet_key(p_c_code VARCHAR2,p_c_imp_cfg_seq NUMBER,
                          p_c_col_seq NUMBER) IS
    SELECT st_seq,st_is_number,st_org_id,st_storet_code
      FROM sim_translations,sim_imp_cfg_dets
      WHERE st_user_code = p_code
        AND st_sicdt_seq = sicdt_seq
        AND sicdt_sicc_seq = p_imp_cfg_seq
        AND sicdt_sic_seq = p_col_seq;

  CURSOR c_imp_cols(p_col_seq NUMBER) IS
    SELECT sic_storet_tab,sic_storet_col
      FROM sim_imp_cols
      WHERE sic_seq = p_col_seq;  
      
  CURSOR c_comma(p_col_seq NUMBER) IS
    SELECT INSTR(sic_storet_col,',') col_pos
      FROM sim_imp_cols
      WHERE sic_seq = p_col_seq;      
      
  CURSOR c_inst_col(p_col_seq NUMBER) IS
    SELECT SUBSTR(SIC_STORET_COL,1,INSTR(SIC_STORET_COL,',')-1) col_name
      FROM sim_imp_cols
      WHERE sic_seq = p_col_seq;             
        
BEGIN

  OPEN c_get_storet_key(p_code,p_imp_cfg_seq,p_col_seq);
  FETCH c_get_storet_key INTO v_seq,v_storet_pk.is_number,v_storet_pk.org_id,v_converted_item;
  CLOSE c_get_storet_key;
  
  IF v_seq IS NULL THEN
    v_converted_item := p_code;
    P_TRANS_EXIST  := FALSE;
  ELSIF v_seq IS NOT NULL THEN
    /* if the code is in the translations table, flag so it's not recreated */
    P_TRANS_EXIST := TRUE;
    IF v_storet_pk.is_number IS NULL THEN
      /* if there is a code in the translations table but no value to translate to, write error message
         status will be changed to 'N' later in the program */
       Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_code||' exists in translations table but has not been translated to a value.',p_sta_seq);
    ELSE
      OPEN c_imp_cols(p_col_seq);
      FETCH c_imp_cols INTO v_table_name,v_column_name;
      CLOSE c_imp_cols;
      OPEN c_comma(p_col_seq);
      FETCH c_comma INTO v_comma_pos;
      CLOSE c_comma;
      IF v_comma_pos = 0 THEN
        v_dyna_cursor := 'SELECT RTRIM('||v_column_name||') '||
                         ' FROM '||v_table_name||
                         ' WHERE '||v_table_name||'_IS_NUMBER = '||v_storet_pk.is_number||
                         ' AND '||v_table_name||'_ORG_ID = '''||v_storet_pk.org_id||'''';
      ELSE
        OPEN c_inst_col(p_col_seq);
          FETCH c_inst_col INTO v_column_name;
        CLOSE c_inst_col;
        v_dyna_cursor := 'SELECT RTRIM('||v_column_name||') '||
                         ' FROM '||v_table_name||
                         ' WHERE '||v_table_name||'_IS_NUMBER = '||v_storet_pk.is_number||
                         ' AND '||v_table_name||'_ORG_ID = '''||v_storet_pk.org_id||'''';
      END IF; 

      OPEN sort_cursor FOR v_dyna_cursor;
      LOOP
        FETCH sort_cursor INTO v_converted_item;
      EXIT WHEN sort_cursor%NOTFOUND;
      END LOOP;      
      IF v_converted_item IS NOT NULL THEN
        P_TRANS_EXIST := TRUE;
      END IF;
		END IF;
  END IF;  
  RETURN v_storet_pk;
  
  EXCEPTION
    WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR(V_IMPORT_LOG_SEQ,'Line ' || V_LINE_CNT || ': ' || SQLERRM || ' - in check storet translations',NULL);
      RETURN NULL;
END CHECK_STORET_TRANSLATION;
/*******************************************************************************************/
PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER) IS

  v_st_seq          NUMBER(10);
  v_sicdt_seq       NUMBER(10);
  e_imp_cfg_dets    EXCEPTION;
  e_trans_error     EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SICDT_SEQ FROM SIM_IMP_CFG_DETS WHERE SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL FROM DUAL;

BEGIN
  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq INTO v_sicdt_seq;
    IF c_get_cfg_det_seq%NOTFOUND THEN
      v_sicdt_seq := NULL;
      RAISE e_imp_cfg_dets;
    END IF;
  CLOSE c_get_cfg_det_seq;
  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq INTO v_st_seq;
    IF c_get_st_seq%NOTFOUND THEN
      v_st_seq := NULL;
      RAISE e_trans_error;
    END IF;
  CLOSE c_get_st_seq;
  INSERT INTO SIM_TRANSLATIONS (
    ST_SEQ,
    ST_SICDT_SEQ,
    ST_USER_CODE,
    ST_STORET_CODE
    )
  VALUES (
    v_st_seq,                  /* st_seq */
    v_sicdt_seq,               /* st_sicdt_seq */
    p_code,                    /* st_user_code */
    NULL                       /* st_storet_code */
    );
EXCEPTION
  WHEN e_imp_cfg_dets THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to find import configuration - stopped import',NULL);
    RAISE_APPLICATION_ERROR(-20101,'Unable to find import configuration');
  WHEN e_trans_error THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to create new record in translations table - error getting sequence',NULL);
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - adding record to translations table.',NULL);
END create_trans_record;

/******************************************************************************************************
    The check allowable values function uses the item input 
    from test and SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, 
    SIM_IMP_CFG_DETS tables are queried to obtain the
    sia_seq
/****************************************************************************************************/
FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,p_sta_seq IN NUMBER)
  RETURN NUMBER IS
  v_sia_seq         NUMBER(10);
  CURSOR c_check_allowable_val(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SIA_SEQ FROM SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS
      WHERE SIA_SIC_SEQ = SIC_SEQ AND SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = SIC_SEQ AND SIC_SEQ = p_col_seq AND SIA_ITEM = p_item;
BEGIN
  /* make sure item is an allowable value */
  OPEN c_check_allowable_val(p_item, p_imp_cfg_seq, p_col_seq);
  FETCH c_check_allowable_val INTO v_sia_seq;
    IF c_check_allowable_val%NOTFOUND THEN
      /* if not an allowable value, return null - will write error message, change status of activity */
      v_sia_seq := NULL;
    END IF;
  CLOSE c_check_allowable_val;
  RETURN v_sia_seq;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check allowable values',NULL);
    RETURN NULL;
END check_allowable_values;

/******************************************************************************************************
  function to look in the translations and allowable values 
  tables for items that are hard coded values in STORET
********************************************************************************************************/
FUNCTION validate_alval_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                             p_col_seq IN NUMBER, p_sta_seq IN NUMBER) RETURN NUMBER IS
   v_converted_item           VARCHAR2(80);
   v_item_sia_seq             NUMBER(10);
   v_trans_exist              BOOLEAN;
   v_instr_code               VARCHAR2(15);
BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
     hard coded values in STORET */
  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    /* look in allowable values */
    v_item_sia_seq := check_allowable_values(v_converted_item, p_imp_cfg_seq, p_col_seq, p_sta_seq);
    IF v_item_sia_seq IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */

      v_instr_code := get_instructions(p_imp_cfg_seq, 'STA-TR');
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value - translation record has been created.',p_sta_seq);
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value.',p_sta_seq);
      END IF;
    ELSIF v_item_sia_seq IS NULL AND v_trans_exist THEN
      /* item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,p_sta_seq);
    END IF;
  END IF;
  RETURN v_item_sia_seq;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate allowable values.',p_sta_seq);
    RETURN NULL;
END validate_alval_item;

/******************************************************************************************************/
FUNCTION get_date(p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,p_cfg_seq IN NUMBER,p_line_cnt IN NUMBER,p_seq IN NUMBER)
  RETURN DATE IS
  v_start_date                  DATE;
  v_fmt                         VARCHAR2(80);

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      v_fmt := NULL;
      /* no format specified, use default */
      v_start_date := TRUNC(TO_DATE(p_raw_date,'MM-DD-RRRR'));
    END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_date := TRUNC(TO_DATE(p_raw_date,v_fmt));
  END IF;
  RETURN v_start_date;

EXCEPTION
    WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,' Line '||p_line_cnt||' - date, '||p_raw_date||', format does not match given import format ',p_seq);
    RETURN NULL;
END get_date;

/******************************************************************************************************/
FUNCTION get_time(p_raw_time IN VARCHAR2,p_col_seq IN NUMBER,p_cfg_seq IN NUMBER)
  RETURN DATE IS
  v_start_time                  DATE;
  v_fmt                         VARCHAR2(80);
  e_no_cfg_found                EXCEPTION;
  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      v_fmt := NULL;
      /* no format specified, use default */
      v_start_time := TO_DATE(SUBSTR(p_raw_time,1,8),'HH24:MI:SS');
    END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_time := TO_DATE(SUBSTR(p_raw_time,1,8),v_fmt);
  END IF;
  RETURN v_start_time;
EXCEPTION
  WHEN e_no_cfg_found THEN
    RAISE_APPLICATION_ERROR(-20100,'Import configuration for start or end time is incomplete or does not exist.');
    RETURN NULL;
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in get time.',NULL);
    RETURN NULL;
END get_time;

/************************************************************************************************************
  The get_station_is_nbr function finds the station IS number 
  and ORG id from  the Storet table TSMSTATN The values are 
  returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS 
  BOTH THE ORG AND IS #)
/************************************************************************************************************/
FUNCTION get_station_is_nbr (p_station_id IN VARCHAR2, p_org_id IN VARCHAR2,p_item_name IN VARCHAR2, p_sta_seq IN NUMBER)
  RETURN T_STORET_PK_TYPE IS
  v_station_pk          T_STORET_PK_TYPE;
  CURSOR c_find_station (p_station_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSMSTATN_IS_NUMBER,TSMSTATN_ORG_ID FROM TSMSTATN
    WHERE IDENTIFICATION_CD = RPAD(p_station_id,15) AND TSMSTATN_ORG_ID = RPAD(p_org_id,8);
BEGIN
  OPEN c_find_station (p_station_id ,p_org_id);
  FETCH c_find_station INTO v_station_pk;
    IF c_find_station%NOTFOUND THEN
      v_station_pk.is_number := NULL;
      v_station_pk.org_id := NULL;
    END IF;
  CLOSE c_find_station;
  RETURN v_station_pk;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||
                                 ' - PROC get_station_is_nbr, line: '||v_line_cnt,NULL);
      v_station_pk.is_number := NULL;
      v_station_pk.org_id := NULL;
      RETURN v_station_pk;
END get_station_is_nbr;

/******************************************************************************************************
  The get_tsmvstc_is_nbr function finds the primary/secondary type 
  IS number and ORG id from the Storet table TSMVSTC The values are 
  returned in a variable that is TYPE T_STORET_TSMVSTC_TYPE (HOLDS 
  BOTH THE ORG AND IS #)
/************************************************************************************************************/
FUNCTION get_tsmvstc_is_nbr (p_vstc_primary_id IN VARCHAR2, p_vstc_secondary_id IN VARCHAR2,
                            p_item_name IN VARCHAR2, p_sta_seq IN NUMBER)
                            RETURN T_STORET_TSMVSTC_TYPE IS

  v_storet_tsmvstc         T_STORET_TSMVSTC_TYPE;
  v_rowcount         NUMBER(10);
  v_trans_2ndary     VARCHAR2(30);
CURSOR c_count_primary_types  IS
       SELECT COUNT(PRIMARY_TYPE_CD) FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_vstc_primary_id;
CURSOR c_find_vstc (p_vstc_primary_id IN VARCHAR2, p_vstc_secondary_id IN VARCHAR2) IS
       SELECT TSMVSTC_IS_NUMBER,TSMVSTC_ORG_ID FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_vstc_primary_id AND SECONDARY_TYPE_CD = p_vstc_secondary_id;
CURSOR c_find_vstc2 (p_vstc_primary_id IN VARCHAR2) IS
       SELECT TSMVSTC_IS_NUMBER,TSMVSTC_ORG_ID FROM TSMVSTC
       WHERE PRIMARY_TYPE_CD = p_vstc_primary_id;
BEGIN
v_trans_2ndary := TRANSLATE(p_vstc_secondary_id,'1 ', '1'); -- If v_trans_secondary is nothing but spaces then this will be a null
  BEGIN
  OPEN c_count_primary_types;
  FETCH c_count_primary_types INTO v_rowcount;
     IF c_count_primary_types%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
     ELSIF v_rowcount > 1 AND v_trans_2ndary IS NULL THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
      v_storet_tsmvstc.tsmvstc_org_id := NULL;
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||p_vstc_primary_id||' -Primary Station Type Needs Secondary Type',p_sta_seq);
      v_status := 'N';
      RETURN v_storet_tsmvstc;
     END IF;
  CLOSE c_count_primary_types;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||P_VSTC_PRIMARY_ID||' - count primary type line:'||v_line_cnt,p_sta_seq);
  END;
 IF p_vstc_secondary_id IS NOT NULL THEN
  BEGIN
  OPEN c_find_vstc (p_vstc_primary_id ,p_vstc_secondary_id);
  FETCH c_find_vstc INTO v_storet_tsmvstc;
    IF c_find_vstc%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
    END IF;
  CLOSE c_find_vstc;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' -Error: Cannot find TSMVSTC IS number or Org Id line: '||v_line_cnt,p_sta_seq);
  END;
 END IF;
IF p_vstc_secondary_id IS NULL THEN
  BEGIN
  OPEN c_find_vstc2 (p_vstc_primary_id);
  FETCH c_find_vstc2 INTO v_storet_tsmvstc;
    IF c_find_vstc2%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
    END IF;
  CLOSE c_find_vstc2;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' -Primaty Type error in find vstc2, line:'||v_line_cnt,NULL);
  END;
 END IF;
  RETURN v_storet_tsmvstc;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - In Function GET_VSTC_IS_NBR line: '||v_line_cnt,NULL);
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
      v_storet_tsmvstc.tsmvstc_org_id := NULL;
      RETURN v_storet_tsmvstc;
END GET_TSMVSTC_IS_NBR;

/***************************************************************************************/
/* Get Primary Type is used to determine if we are importing 
   a well, and therefore if we Need to process the code for 
   aquifer and casing */
/***************************************************************************************/
FUNCTION get_primary_type(p_tsmvstc_is_number IN NUMBER,p_tsmvstc_org_id IN VARCHAR2) RETURN NUMBER IS
  v_primary_type     VARCHAR2(30);
  CURSOR c_station_primary_type (p_tsmvstc_is_number IN NUMBER,p_tsmvstc_org_id IN VARCHAR2) IS SELECT
        PRIMARY_TYPE_CD
        FROM TSMVSTC
        WHERE TSMVSTC_IS_NUMBER = p_tsmvstc_is_number AND
              TSMVSTC_ORG_ID    = p_tsmvstc_org_id;
BEGIN
  OPEN c_station_primary_type(p_tsmvstc_is_number,p_tsmvstc_org_id);
  FETCH c_station_primary_type INTO v_primary_type;
  CLOSE c_station_primary_type;
  IF v_primary_type = 'Well' THEN
   RETURN 1;
  ELSE
   RETURN 0;
  END IF;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END;

/************************************************************************************************************
  The get_tsmfhu_is_nbr function finds the hydrologic unit 
  IS number and ORG id from the Storet table TSMFHU The 
  values are returned in a variable that is 
  TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #)
/************************************************************************************************************/
FUNCTION get_tsmfhu_is_nbr (p_fhu IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  v_storet_tsmfhu    T_STORET_PK_TYPE;
CURSOR c_find_fhu (p_fhu IN VARCHAR2) IS
       SELECT TSMFHU_IS_NUMBER,TSMFHU_ORG_ID FROM TSMFHU
       WHERE HYDROLOGIC_UNIT_CD = RPAD(p_fhu,8);
BEGIN
  BEGIN
  OPEN c_find_fhu (p_fhu);
  FETCH c_find_fhu INTO v_storet_tsmfhu;
    IF c_find_fhu%NOTFOUND THEN
      v_storet_tsmfhu.is_number := NULL;
    END IF;
  CLOSE c_find_fhu;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - error finding fhu',NULL);
  END;
  RETURN v_storet_tsmfhu;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - Function get_fhu_is_nbr ',NULL);
      v_storet_tsmfhu.is_number := NULL;
      v_storet_tsmfhu.org_id := NULL;
      RETURN v_storet_tsmfhu;
END GET_TSMFHU_IS_NBR;

/************************************************************************************************************
  The get_tsmgeopa_is_nbr function finds the station 
  IS number and ORG id from the Storet table TSMGEOPA.
  The values are returned in a variable that is 
  TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #)
  This function accounts for the 4 possible combinations 
  of state and county ID's.
  1 It's a state postal code and a county name
  2 It's a state FIPS and a county name
  3 It's a state postal code and  county FIPS
  4 It's a  state FIPS and county FIPS
/************************************************************************************************************/
FUNCTION get_tsmgeopa_is_nbr (p_state IN VARCHAR2, p_county IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  v_storet_tsmgeopa    T_STORET_PK_TYPE;
/*** four cursors for the four possible combinations of state and county id's *****/
CURSOR c_find_geopa_1(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(p_state,2) AND COUNTY_NAME = p_county;
CURSOR c_find_geopa_2(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMgeopa
       WHERE STATE_FIPS_CODE = RPAD(p_state,2) AND COUNTY_NAME = p_county;
CURSOR c_find_geopa_3(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMGEOPA
       WHERE STATE_POSTAL_CODE = RPAD(p_state,2) AND FIPS_COUNTY_CODE = RPAD(p_county,3);
CURSOR c_find_geopa_4(p_state IN VARCHAR2,p_county IN VARCHAR2) IS
       SELECT TSMGEOPA_IS_NUMBER,TSMGEOPA_ORG_ID FROM TSMGEOPA
       WHERE STATE_FIPS_CODE = RPAD(p_state,2) AND FIPS_COUNTY_CODE = RPAD(p_county,3);
BEGIN
  BEGIN
    IF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NOT NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NOT NULL THEN -- it's a state postal and a county NAME
        OPEN c_find_geopa_1 (p_state,p_county);
        FETCH c_find_geopa_1 INTO v_storet_tsmgeopa;
          IF c_find_geopa_1%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_1;
    ELSIF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NOT NULL THEN -- It's a state fips and a county name
        OPEN c_find_geopa_2 (p_state,p_county);
        FETCH c_find_geopa_2 INTO v_storet_tsmgeopa;
          IF c_find_geopa_2%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_2;
    ELSIF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NOT NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NULL THEN -- it's a state postal and  county fips
        OPEN c_find_geopa_3 (p_state,p_county);
        FETCH c_find_geopa_3 INTO v_storet_tsmgeopa;
          IF c_find_geopa_3%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_3;
    ELSIF TRANSLATE(p_state,'a1234567890.- ', 'a') IS NULL AND TRANSLATE(p_county,'a1234567890.- ', 'a') IS NULL THEN -- it's a  state fips and county FIPS
        OPEN c_find_geopa_4 (p_state,p_county);
        FETCH c_find_geopa_4 INTO v_storet_tsmgeopa;
          IF c_find_geopa_4%NOTFOUND THEN
           v_storet_tsmgeopa.is_number := NULL;
          END IF;
        CLOSE c_find_geopa_4;
    ELSE
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'get TSMgeopa IS COMBO NOT FOUND: '||v_storet_tsmgeopa.is_number ||'p_state -> '||p_state||' p_county -> '||p_county,NULL);
    END IF;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - find geopa',NULL);
  END;
  RETURN v_storet_tsmgeopa;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||
                               ' - PROC GET_geopa_IS_NBR ',NULL);
    v_storet_tsmgeopa.is_number := NULL;
    v_storet_tsmgeopa.org_id := NULL;
    RETURN v_storet_tsmgeopa;
END GET_TSMgeopa_IS_NBR;

/****************************************************************************************/
/*** river reach ***/
/************************************************************************************************************
  The get_tsmrrr_is_nbr function finds the station IS number and ORG id from  the Storet
  table TSMRRR The values are returned in a variable that is TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #
  This function
/************************************************************************************************************/
FUNCTION get_tsmrrr_is_nbr (p_huc IN VARCHAR2, p_rf1 IN VARCHAR2)
  RETURN T_STORET_PK_TYPE IS
  v_storet_tsmrrr    T_STORET_PK_TYPE;
CURSOR c_find_tsmrrr(p_huc IN VARCHAR2,p_rf1 IN VARCHAR2) IS
       SELECT TSMRRR_IS_NUMBER,TSMRRR_ORG_ID FROM TSMRRR
       WHERE HYDROLOGIC_UNIT_CD = RPAD(p_huc,8) AND SEGMENT_CODE = RPAD(p_rf1,3);
BEGIN
  BEGIN
        OPEN c_find_tsmrrr (p_huc,p_rf1);
        FETCH c_find_tsmrrr INTO v_storet_tsmrrr;
          IF c_find_tsmrrr%NOTFOUND THEN
           v_storet_tsmrrr.is_number := NULL;
          END IF;
        CLOSE c_find_tsmrrr;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||' - find tsmrrr_is_number',NULL);
  END;
  RETURN v_storet_tsmrrr;
EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, SQLERRM||
                                 ' - PROC GET_TSMRRR_IS_NBR ',NULL);
      v_storet_tsmrrr.is_number := NULL;
      v_storet_tsmrrr.org_id := NULL;
      RETURN v_storet_tsmrrr;
END GET_TSMRRR_IS_NBR;
/******************************************************************************************************
  The create_station function inserts known good data into the SIM table SIM_STATIONS
  After the station has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_station (p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER IS
BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATIONS (
                            STA_SEQ,
                            STA_ID,
                            STA_NAME,
                            STA_DESC,
                            STA_ORG_ID,
                            STA_TSMORGAN_ORG_ID,
                            STA_TSMORGAN_IS_NUMBER,
                            STA_STATUS,
                            STA_ESTAB_DATE,
                            STA_EXPORT_SEQ,
                            STA_IMPORT_SEQ,
                            STA_TSMVSTC_IS_NUMBER,
                            STA_TSMVSTC_ORG_ID,
                            STA_EPA_KEY_ID,
                            STA_ZID_REL,
                            STA_INFLUENCE_AREA,
                            STA_TRAVEL_DIR,
                            STA_WATER_DEPTH,
                            STA_WATER_DEPTH_UNIT,
                            STA_ECOREGION,
                            STA_DOCUMENT
                           )
  VALUES (
    p_station_rec.STA_SEQ,                               /* STA_SEQ            */
    p_station_rec.STA_ID,                                /* STA_ID             */
    p_station_rec.STA_NAME,                              /* STA_NAME           */
    p_station_rec.STA_DESC,                              /* STA_DESC           */
    p_station_rec.STA_ORG_ID,                            /* STATION ORG ID     */
    p_station_rec.STA_TSMORGAN_ORG_ID,                   /* TSMORGAN_ORG_ID    */
    p_station_rec.STA_TSMORGAN_IS_NUMBER,                /* TSMORGAN_IS_NUMBER */
    p_station_rec.STA_STATUS,                            /* STA_STATUS         */
    p_station_rec.STA_ESTAB_DATE,                        /* STA_ESTAB_DATE     */
    p_station_rec.STA_EXPORT_SEQ,                        /* STA_EXPORT_SEQ     */
    p_station_rec.STA_IMPORT_SEQ,                        /* STA_IMPORT_SEQ     */
    p_station_rec.STA_TSMVSTC_IS_NUMBER,                 /* STA_TSMVSTC_IS_NUMBER     */
    p_station_rec.STA_TSMVSTC_ORG_ID,                     /* STA_TSMVSTC_ORG_ID */
    p_station_rec.STA_EPA_KEY_ID,
    p_station_rec.STA_ZID_REL,
    p_station_rec.STA_INFLUENCE_AREA,
    p_station_rec.STA_TRAVEL_DIR,
    p_station_rec.STA_WATER_DEPTH,
    p_station_rec.STA_WATER_DEPTH_UNIT,
    p_station_rec.STA_ECOREGION,
    p_station_rec.STA_DOCUMENT
    );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station.',p_station_rec.sta_seq);
    RETURN 0;
END create_station;
/******************************************************************************************************
  The create_station_location function inserts known good data into the SIM table SIM_STATIONS
  After the station_location has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_station_location (p_station_location_rec IN sim_station_locations%ROWTYPE) RETURN NUMBER IS
BEGIN
  /* create station using values in station record */
  INSERT INTO SIM_STATION_LOCATIONS (
                                     STL_SEQ
                                     ,STL_STA_SEQ
                                     ,STL_TYPE_CODE
                                     ,STL_SEQUENCE_NUMBER
                                     ,STL_ELEV_DATUM
                                     ,STL_ELEV_METHOD
                                     ,STL_ELEV_MSR_DATE
                                     ,STL_ELEVATION_MSR
                                     ,STL_ELEVATION_UNIT
                                     ,STL_GEOPOS_DATUM
                                     ,STL_GEOPOS_METHOD
                                     ,STL_GPS_SCALE
                                     ,STL_LAT_DIR
                                     ,STL_LAT_DEGREES
                                     ,STL_LAT_MINUTES
                                     ,STL_LAT_SECONDS
                                     ,STL_LONG_DIR
                                     ,STL_LONG_DEGREES
                                     ,STL_LONG_MINUTES
                                     ,STL_LONG_SECONDS
                                     ,STL_LAT_LONG_MSR_DATE
                                     ,STL_LATLONG_ACCURCY
                                     ,STL_LATLONG_ACCURCY_UNIT
                                     ,STL_TSMRRR_IS_NUMBER
                                     ,STL_TSMRRR_ORG_ID
                                     ,STL_TSMFHU_IS_NUMBER
                                     ,STL_TSMFHU_ORG_ID
                                     ,STL_TSMSTATN0IS_NUMBER
                                     ,STL_TSMSTATN0ORG_ID
                                     ,STL_TSMWELL_IS_NUMBER
                                     ,STL_TSMWELL_ORG_ID
                                     ,STL_TSMGEOPA_IS_NUMBER
                                     ,STL_TSMGEOPA_ORG_ID
                                     ,STL_LAT_DEC_MINUTES
                                     ,STL_LONG_DEC_MINUTES
                                     ,STL_TSMPIPE_IS_NUMBER
                                     ,STL_TSMPIPE_ORG_ID
                                     ,STL_TSMSTATN1ORG_ID
                                     ,STL_TSMSTATN1IS_NUMBER
                                     ,STL_TSMGEOPA0IS_NUMBER
                                     ,STL_TSMGEOPA0ORG_ID
                                     ,STL_TSMNAL_CD
                                     ,STL_TSMNAL_STATE
                                     ,STL_NRCS
                                     ,STL_STATUS
                                     ,STL_IMPORT_SEQ
                                     ,STL_EXPORT_SEQ
                                     ,STL_LAT_DEC_DEGREES
                                     ,STL_LONG_DEC_DEGREES
                                     ,STL_POINT_NAME
                                     )
  VALUES (
         p_station_location_rec.STL_SEQ,
         p_station_location_rec.STL_STA_SEQ,
         p_station_location_rec.STL_TYPE_CODE,
         p_station_location_rec.STL_SEQUENCE_NUMBER,
         p_station_location_rec.STL_ELEV_DATUM,
         p_station_location_rec.STL_ELEV_METHOD,
         p_station_location_rec.STL_ELEV_MSR_DATE,
         p_station_location_rec.STL_ELEVATION_MSR,
         p_station_location_rec.STL_ELEVATION_UNIT,
         p_station_location_rec.STL_GEOPOS_DATUM
         ,p_station_location_rec.STL_GEOPOS_METHOD
         ,p_station_location_rec.STL_GPS_SCALE
         ,p_station_location_rec.STL_LAT_DIR
         ,p_station_location_rec.STL_LAT_DEGREES
         ,p_station_location_rec.STL_LAT_MINUTES
         ,p_station_location_rec.STL_LAT_SECONDS
         ,p_station_location_rec.STL_LONG_DIR
         ,p_station_location_rec.STL_LONG_DEGREES
         ,p_station_location_rec.STL_LONG_MINUTES
         ,p_station_location_rec.STL_LONG_SECONDS
         ,p_station_location_rec.STL_LAT_LONG_MSR_DATE
         ,p_station_location_rec.STL_LATLONG_ACCURCY
         ,p_station_location_rec.STL_LATLONG_ACCURCY_UNIT
         ,p_station_location_rec.STL_TSMRRR_IS_NUMBER
         ,p_station_location_rec.STL_TSMRRR_ORG_ID
         ,p_station_location_rec.STL_TSMFHU_IS_NUMBER
         ,p_station_location_rec.STL_TSMFHU_ORG_ID
         ,p_station_location_rec.STL_TSMSTATN0IS_NUMBER
         ,p_station_location_rec.STL_TSMSTATN0ORG_ID
         ,p_station_location_rec.STL_TSMWELL_IS_NUMBER
         ,p_station_location_rec.STL_TSMWELL_ORG_ID
         ,p_station_location_rec.STL_TSMGEOPA_IS_NUMBER
         ,p_station_location_rec.STL_TSMGEOPA_ORG_ID
         ,p_station_location_rec.STL_LAT_DEC_MINUTES
         ,p_station_location_rec.STL_LONG_DEC_MINUTES
         ,p_station_location_rec.STL_TSMPIPE_IS_NUMBER
         ,p_station_location_rec.STL_TSMPIPE_ORG_ID
         ,p_station_location_rec.STL_TSMSTATN1ORG_ID
         ,p_station_location_rec.STL_TSMSTATN1IS_NUMBER
         ,p_station_location_rec.STL_TSMGEOPA0IS_NUMBER
         ,p_station_location_rec.STL_TSMGEOPA0ORG_ID
         ,p_station_location_rec.STL_TSMNAL_CD
         ,p_station_location_rec.STL_TSMNAL_STATE
         ,p_station_location_rec.STL_NRCS
         ,p_station_location_rec.STL_STATUS
         ,p_station_location_rec.STL_IMPORT_SEQ
         ,p_station_location_rec.STL_EXPORT_SEQ
         ,p_station_location_rec.STL_LAT_DEC_DEGREES
         ,p_station_location_rec.STL_LONG_DEC_DEGREES
         ,p_station_location_rec.STL_POINT_NAME
         );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting station locations.',NULL);
    RETURN 0;
END create_station_location;
/******************************************************************************************************
  The create_ocean function inserts known good data into the SIM table SIM_OCEAN
  After the ocean has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_ocean (p_ocean_rec IN sim_ocean%ROWTYPE) RETURN NUMBER IS
  BEGIN
    /* create ocean using values in ocean record */
    INSERT INTO SIM_OCEAN ( SOC_SEQ,
                            SOC_STL_SEQ,
                            SOC_NAME,
                            SOC_ADDITIONAL_LOC,
                            SOC_SHORE_RELATION,
                            SOC_SHORE_DISTANCE,
                            SOC_SHORE_DIST_UNIT,
                            SOC_REFERENCE_PT,
                            SOC_LORAN_C_READING_1,
                            SOC_LORAN_C_READING_2,
                            SOC_BOTTOM_TOPOGRAPHY,
                            SOC_TSMALP_IS_NUMBER,
                            SOC_TSMALP_ORG_ID
                             )
                    VALUES (
                            p_ocean_rec.SOC_SEQ,
                            p_ocean_rec.SOC_STL_SEQ,
                            p_ocean_rec.SOC_NAME,
                            p_ocean_rec.SOC_ADDITIONAL_LOC,
                            p_ocean_rec.SOC_SHORE_RELATION,
                            p_ocean_rec.SOC_SHORE_DISTANCE,
                            p_ocean_rec.SOC_SHORE_DIST_UNIT,
                            p_ocean_rec.SOC_REFERENCE_PT,
                            p_ocean_rec.SOC_LORAN_C_READING_1,
                            p_ocean_rec.SOC_LORAN_C_READING_2,
                            p_ocean_rec.SOC_BOTTOM_TOPOGRAPHY,
                            p_ocean_rec.SOC_TSMALP_IS_NUMBER,
                            p_ocean_rec.SOC_TSMALP_ORG_ID
                            );
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting ocean.',NULL);
      RETURN 0;
END create_ocean;
/******************************************************************************************************
  The create_estuary function inserts known good data into the SIM table SIM_ESTUARY_LOC
  After the estuary has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_estuary (p_est_rec IN sim_estuary_loc%ROWTYPE) RETURN NUMBER IS
  BEGIN
    /* create ocean using values in ocean record */
    INSERT INTO SIM_ESTUARY_LOC (SEL_SEQ,
                                 SEL_TSMALP_IS_NUMBER,
                                 SEL_TSMALP_ORG_ID,
                                 SEL_OTHER_ESTUARY,
                                 SEL_ADD_LOC_NAME,
                                 SEL_SHORE_DIST,
                                 SEL_SHORE_DIST_UNIT,
                                 SEL_REF_PT,
                                 SEL_TSMESTRY_ORG_ID,
                                 SEL_TSMESTRY_IS_NUMBER,
                                 SEL_STL_SEQ
                                 )
                    VALUES (
                                 p_est_rec.SEL_SEQ,
                                 p_est_rec.SEL_TSMALP_IS_NUMBER,
                                 p_est_rec.SEL_TSMALP_ORG_ID,
                                 p_est_rec.SEL_OTHER_ESTUARY,
                                 p_est_rec.SEL_ADD_LOC_NAME,
                                 p_est_rec.SEL_SHORE_DIST,
                                 p_est_rec.SEL_SHORE_DIST_UNIT,
                                 p_est_rec.SEL_REF_PT,
                                 p_est_rec.SEL_TSMESTRY_ORG_ID,
                                 p_est_rec.SEL_TSMESTRY_IS_NUMBER,
                                 p_est_rec.SEL_STL_SEQ
                            );
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting estuary.',NULL);
      RETURN 0;
END create_estuary;
/******************************************************************************************************
  The create_lake function inserts known good data into the SIM table SIM_GREAT_LAKE
  After the great lake has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_lake (p_lake_rec IN sim_great_lake%ROWTYPE) RETURN NUMBER IS
  BEGIN
    /* create ocean using values in ocean record */
    INSERT INTO SIM_GREAT_LAKE  (SGL_SEQ,
                                 SGL_TSMALP_IS_NUMBER,
                                 SGL_TSMALP_ORG_ID,
                                 SGL_LAKE_NAME,
                                 SGL_ADD_LAKE_NAME,
                                 SGL_SHORE_DIST,
                                 SGL_SHORE_DIST_UNIT,
                                 SGL_REF_PT,
                                 SGL_STL_SEQ                                             )
                    VALUES (
                                 p_lake_rec.SGL_SEQ,
                                 p_lake_rec.SGL_TSMALP_IS_NUMBER,
                                 p_lake_rec.SGL_TSMALP_ORG_ID,
                                 p_lake_rec.SGL_LAKE_NAME,
                                 p_lake_rec.SGL_ADD_LAKE_NAME,
                                 p_lake_rec.SGL_SHORE_DIST,
                                 p_lake_rec.SGL_SHORE_DIST_UNIT,
                                 p_lake_rec.SGL_REF_PT,
                                 p_lake_rec.SGL_STL_SEQ
                            );
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting great lake.',NULL);
      RETURN 0;
END create_lake;
/******************************************************************************************************
  The create_well function inserts known good data into the SIM table SIM_STATION_WELLS
  After the well has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
/************************************************************************************************************/
FUNCTION create_well (p_well_rec IN sim_station_wells%ROWTYPE) RETURN NUMBER IS
  BEGIN
    /* create well using values in well record */
    INSERT INTO SIM_STATION_WELLS  (SSW_SEQ,
                                    SSW_STA_SEQ,
                                    SSW_WELL_STATUS,
                                    SSW_IS_NUMBER,
                                    SSW_ORG_ID,
                                    SSW_NAME,
                                    SSW_WELL_NO,
                                    SSW_WELL_USE,
                                    SSW_DEVELOP_METHOD,
                                    SSW_WELL_CONST_METH,
                                    SSW_WATER_USE,
                                    SSW_GRADIENT_TYPE,
                                    SSW_DEPTH_COMPLETION,
                                    SSW_DEPTH_COMPLETION_UNIT,
                                    SSW_DEPTH_HOLE,
                                    SSW_DEPTH_HOLE_UNIT ,
                                    SSW_WELL_HEIGHT,
                                    SSW_WELL_HEIGHT_UNIT,
                                    SSW_CASING_HEIGHT,
                                    SSW_CASING_HEIGHT_UNIT,
                                    SSW_DISINFECTED,
                                    SSW_CONST_START_DATE,
                                    SSW_CONST_END_DATE,
                                    SSW_INIT_PUMP_DUR,
                                    SSW_INIT_PUMP_DUR_UNIT,
                                    SSW_INIT_PUMP_RATE,
                                    SSW_INIT_PUMP_RATE_UNIT,
                                    SSW_NATURAL_FLOW,
                                    SSW_INIT_BOREHOLE_DIAM,
                                    SSW_INIT_BOREHOLE_DIAM_UNIT,
                                    SSW_TSMSTATN_IS_NUMBER,
                                    SSW_TSMSTATN_ORG_ID,
                                    SSW_STATUS,
                                    SSW_IMPORT_SEQ,
                                    SSW_EXPORT_SEQ
                                   )
                    VALUES (
                                    p_well_rec.SSW_SEQ,
                                    p_well_rec.SSW_STA_SEQ,
                                    p_well_rec.SSW_WELL_STATUS,
                                    p_well_rec.SSW_IS_NUMBER ,
                                    p_well_rec.SSW_ORG_ID,
                                    p_well_rec.SSW_NAME,
                                    p_well_rec.SSW_WELL_NO,
                                    p_well_rec.SSW_WELL_USE,
                                    p_well_rec.SSW_DEVELOP_METHOD,
                                    p_well_rec.SSW_WELL_CONST_METH,
                                    p_well_rec.SSW_WATER_USE,
                                    p_well_rec.SSW_GRADIENT_TYPE,
                                    p_well_rec.SSW_DEPTH_COMPLETION,
                                    p_well_rec.SSW_DEPTH_COMPLETION_UNIT,
                                    p_well_rec.SSW_DEPTH_HOLE,
                                    p_well_rec.SSW_DEPTH_HOLE_UNIT,
                                    p_well_rec.SSW_WELL_HEIGHT,
                                    p_well_rec.SSW_WELL_HEIGHT_UNIT,
                                    p_well_rec.SSW_CASING_HEIGHT,
                                    p_well_rec.SSW_CASING_HEIGHT_UNIT,
                                    p_well_rec.SSW_DISINFECTED,
                                    p_well_rec.SSW_CONST_START_DATE,
                                    p_well_rec.SSW_CONST_END_DATE,
                                    p_well_rec.SSW_INIT_PUMP_DUR,
                                    p_well_rec.SSW_INIT_PUMP_DUR_UNIT,
                                    p_well_rec.SSW_INIT_PUMP_RATE,
                                    p_well_rec.SSW_INIT_PUMP_RATE_UNIT,
                                    p_well_rec.SSW_NATURAL_FLOW,
                                    p_well_rec.SSW_INIT_BOREHOLE_DIAM,
                                    p_well_rec.SSW_INIT_BOREHOLE_DIAM_UNIT,
                                    p_well_rec.SSW_TSMSTATN_IS_NUMBER,
                                    p_well_rec.SSW_TSMSTATN_ORG_ID,
                                    p_well_rec.SSW_STATUS,
                                    p_well_rec.SSW_IMPORT_SEQ,
                                    p_well_rec.SSW_EXPORT_SEQ
                            );
    COMMIT;
    RETURN 1;
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq,SQLERRM||' - inserting well.',NULL);
      RETURN 0;
END create_well;
/******************************************************************************************************
  The import_station procedure parses a text file line by line of data that is separated by
  a delimiter typically a pipe |
  the procedure is activated by filling out and activating the import station dialog box in the SIM2
  application. From there we will assign an ORG ID and procede with opening the text file to extract the
  data
/******************************************************************************************************/
PROCEDURE import_station (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                       p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER) IS


  v_import_desc                          VARCHAR2(200);
  v_comp                                 NUMBER(1);
  v_num_cols                             NUMBER(3);
  v_col_seq                              NUMBER(10);
  v_dflt_pos                             NUMBER(3);
  v_success                              NUMBER(1);
  v_check_primary_type                   NUMBER(1);
  v_primary_checked                      VARCHAR2(1);
  v_converted_item                       VARCHAR2(256);
  v_delimiter                            CHAR(1);
  v_InLine                               LONG;
  v_bad_parse                            BOOLEAN;
  v_item_name                            VARCHAR2(80);
  v_item                                 VARCHAR2(2000);
  v_trans_exist                          BOOLEAN := TRUE;
  v_line_table                           Sim2_Import_Pkg.col_table_type;
  v_default_table                        Sim2_Import_Station_Pkg.col_table_type;
  v_pos_table                            Sim2_Import_Station_Pkg.col_table_num_type;

  v_storet_pk                            t_storet_pk_type;
  v_storet_tsmvstc                       t_storet_tsmvstc_type;
  v_storet_tsmgeopa                      t_storet_pk_type;
  v_storet_tsmfhu                        t_storet_pk_type;

  v_rows                                 NUMBER(2);
  v_error_cnt                            NUMBER(10);
  v_complete_cnt                         NUMBER(10);
  v_dummy_seq                            NUMBER(10);

  v_stl_rec              SIM_STATION_LOCATIONS%ROWTYPE;
  v_sta_rec              SIM_STATIONS%ROWTYPE;
  v_soc_rec              SIM_OCEAN%ROWTYPE;
  v_estuary_rec          SIM_ESTUARY_LOC%ROWTYPE;
  v_lake_rec             SIM_GREAT_LAKE%ROWTYPE;
  v_well_rec             SIM_STATION_WELLS%ROWTYPE;

  v_county                               VARCHAR2(25);
  v_elevation_unit                       VARCHAR2(3);
  v_point_name                           VARCHAR2(30);
  v_elevation_method                     VARCHAR2(12);
  v_elevation_datum                      VARCHAR2(12);
  v_elevation_date                       DATE;
  v_tsmvstc_org_id                       VARCHAR2(8) := NULL;
  v_tsmvstc_is_number                    VARCHAR2(8) := NULL;
  v_tsmorgan_is_number                   VARCHAR2(8);
  v_tsmgeopa_org_id                      VARCHAR2(8);
  v_tsmgeopa_is_number                   NUMBER(8);
  v_tsmfhu_org_id                        VARCHAR2(8);
  v_tsmfhu_is_number                     NUMBER(8);
  v_name                                 VARCHAR2(60);
  v_type1                                VARCHAR2(30);
  v_secondary_type                       VARCHAR2(30);
  v_estab_date                           DATE;
  v_seq                                  NUMBER(10);
  v_stl_seq                              NUMBER(10);
  v_id                                   VARCHAR2(15);
  v_nrcs                                 VARCHAR2(8);
  v_sta_desc                             VARCHAR2(1999);
  v_ecoregion                            VARCHAR2(60);
  v_water_depth                          NUMBER(8,3);
  v_water_depth_unit                     VARCHAR2(2);
  v_elevation_msr                        NUMBER(9,4);
  v_lat_long_date                        DATE;  
  v_lat_degrees                          NUMBER(2);
  v_lat_minutes                          NUMBER(2);
  v_lat_seconds                          NUMBER(6,4);
  v_lat_dir                              VARCHAR2(1);
  v_check_lat_dir                        VARCHAR2(1); 
  v_lat_dec_min                          NUMBER(6,4);
  v_lat_dec_deg                          NUMBER(10,7);
  v_long_degrees                         NUMBER(3);
  v_long_minutes                         NUMBER(2);
  v_long_seconds                         NUMBER(6,4);
  v_long_dec_min                         NUMBER(6,4);
  v_long_dec_deg                         NUMBER(10,7);
  v_long_dir                             VARCHAR2(1);
  v_check_long_dir                       VARCHAR2(1);
  v_datum                                VARCHAR2(12);
  v_method                               VARCHAR2(12);
  v_gps_scale                            VARCHAR2(20);
  v_state                                VARCHAR2(2);
  v_ocean_seq                            NUMBER(10);
  v_ocean_name                           VARCHAR2(30);
  v_ocean_ref                            VARCHAR2(30);
  v_ocean_bottom                         VARCHAR2(256);
  v_add_ocean                            VARCHAR2(30);
  v_ocean_dist                           NUMBER(6,2);
  v_ocean_dist_unit                      VARCHAR2(3);
  v_shore_relation                       VARCHAR2(10);
  v_native_name                          VARCHAR2(40);
  v_native_code                          CHAR(5);
  v_native                               VARCHAR2(5);
  v_native_state                         CHAR(2);
  v_huc                                  VARCHAR2(8);
  v_comment                              VARCHAR2(254);
  v_EPA_key                              VARCHAR2(36);
  v_influence                            VARCHAR2(120);
  v_travel_dir                           VARCHAR2(1999);
  v_zid_rel                              VARCHAR2(2);
  v_lake_seq                             NUMBER(10);
  v_great_lake                           VARCHAR2(15);
  v_lake_ref                             VARCHAR2(30);
  v_add_lake                             VARCHAR2(30);
  v_lake_dist                            NUMBER(6,2);
  v_lake_dist_unit                       VARCHAR2(3);
  v_estuary_seq                          NUMBER(10);
  v_estuary_state                        VARCHAR2(40);
  v_primary_est                          VARCHAR2(30);
  v_secondary_est                        VARCHAR2(30);
  v_primary_is_nbr                       NUMBER(8);
  v_secondary_is_nbr                     NUMBER(8);
  v_tsmestry_is_nbr                      NUMBER(8);
  v_estuary_dist                         NUMBER(6,2);
  v_est_dist_unit                        VARCHAR2(3);
  v_estuary_ref                          VARCHAR2(30);
  v_add_estuary                          VARCHAR2(30);
  v_other_estuary                        VARCHAR2(30);
  v_document                             VARCHAR2(1999);
  v_well_seq                             NUMBER(10);
  v_well_name                            VARCHAR2(35);
  v_well_num                             NUMBER(3);
  v_well_use                             VARCHAR2(30);
  v_well_status                          VARCHAR2(9);
  v_next                                 NUMBER(2) := 0;
  
  e_bad_org                              EXCEPTION;
  e_no_cfg                               EXCEPTION;
  e_no_delimiter                         EXCEPTION;
  e_bad_columns                          EXCEPTION;

  CURSOR c_get_station_seq IS
         SELECT SIM_STATIONS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_station_location_seq IS
         SELECT SIM_STATION_LOCATIONS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_ocean_seq IS
         SELECT SIM_OCEAN_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_estuary_seq IS
         SELECT SIM_ESTUARY_LOC_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_great_lake_seq IS
         SELECT SIM_GREAT_LAKE_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_well_seq IS
         SELECT SIM_STATION_WELLS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT SIPL_SEQ, SIPL_LINE_TEXT
       FROM SIM_IMP_LINES
       WHERE SIPL_SIL_SEQ = p_import_seq
       AND sipl_line_text IS NOT NULL  -- bka  added to eliminate blank lines from the import
       ORDER BY SIPL_LINE_NO;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_PROC_ORDER, SIC_DFLT_POS, SIC_NAME, SIC_SEQ
      FROM SIM_IMP_COLS, SIM_IMP_DEFS, SIM_IMP_CFG
      WHERE SICC_SID_SEQ = SID_SEQ AND SIC_SID_SEQ = SID_SEQ AND SICC_SEQ = p_imp_cfg_seq
      ORDER BY SIC_PROC_ORDER;

  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(sta_seq)
      FROM SIM_STATIONS
      WHERE STA_IMPORT_SEQ = p_import_log_seq
      AND STA_STATUS = 'A';

  CURSOR c_count_primary_types  IS                     -- DETERMINES IF THERE IS A SECONDARY TYPE
    SELECT COUNT(PRIMARY_TYPE_CD)
      FROM TSMVSTC
      WHERE PRIMARY_TYPE_CD = v_type1;


BEGIN
  /** Find out import type **/
  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  v_line_cnt := 0;
  BEGIN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Start at '||TO_CHAR(SYSDATE,'HH24:MI'),NULL);
    /* get tsmorgan_is_number */
    v_success := get_org_info(p_org_id,v_tsmorgan_is_number);
    IF v_success != 1 OR v_org_is_nbr IS NULL THEN
      RAISE e_bad_org;
    END IF;

   /* get the delimiter */
    v_delimiter := NULL;
    v_success := get_delimiter(p_imp_cfg_seq,v_delimiter);
    IF v_success != 1 THEN
      RAISE e_no_delimiter;
    END IF;

    /* get actual positions and default information for items being imported */
    v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table,v_num_cols);
    IF v_success != 1 THEN
      RAISE e_no_cfg;
    END IF;

    /* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
      /* read one line */
      /**** get station sequence ******/
      OPEN c_get_station_seq;
        FETCH c_get_station_seq INTO v_seq;
      CLOSE c_get_station_seq;
      v_inline := v_text_line.SIPL_LINE_TEXT;
      v_comment := NULL;
      v_line_cnt := v_line_cnt + 1;
      v_status := 'A';
      v_primary_checked := 'N';
      v_lat_dir := NULL;
      v_long_dir := NULL;

      -- send line to parsing routine:
      Sim2_Import_Pkg.PARSE_LINE_TABLE(v_inline, v_line_table, v_num_cols, v_delimiter,
                                         v_bad_parse,v_import_log_seq);
      IF v_bad_parse THEN
        RAISE e_bad_columns;
      END IF;
      v_next := 0;
      /* loop through all STATION columns */
      FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP

        /* get dflt position, item name, and column sequence */
        v_dflt_pos := v_columns.SIC_DFLT_POS;
        v_item_name := v_columns.SIC_NAME;
        v_col_seq := v_columns.SIC_SEQ;
        v_next := v_next + 1;

        /* if column is included on the import, continue. Don't look at positions <= 0 */
        IF v_dflt_pos > 0 THEN
          IF v_pos_table(v_dflt_pos) IS NOT NULL THEN
            /* get the actual item, or use default if null */
            IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
              v_item := v_line_table(v_pos_table(v_dflt_pos));
            ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL AND v_default_table(v_dflt_pos) IS NOT NULL THEN
              v_item := v_default_table(v_dflt_pos);
            ELSE
              v_item := NULL;
            END IF;
                  
            -- Does V_ITEM contain only spaces? (or NULL)
            IF (TRIM(V_ITEM) IS NULL) THEN
              -- if yes
              
              -- by definition V_ITEM is NULL!
              V_ITEM := NULL;
  
            END IF;


          /* assign item to appropriate variables and take care of any other procedural issues that need to be
             taken care of for or with a particular item */

          IF v_item_name = 'Station ID' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_item := UPPER(RTRIM(v_item)); -- 061203 slp - Storet only does searches on station id's in upper case
                                                       --  and without trailing spaces.
                IF LENGTH(v_item) > 15 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(
                    v_import_log_seq,
                    'Warning: Line '||v_line_cnt||': '||
                    v_item_name||' is longer than 15 characters and will be truncated.',
                    v_seq
                  );
                  v_item := SUBSTR(v_item,1,15);
                END IF;
                v_comp := CHECK_IF_STATION_EXISTS(v_item, p_org_id,v_seq);
                IF v_comp = 1 THEN
                  v_id := v_item;
                ELSE
                  v_id := v_item;
                  v_status := 'N';
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            ELSE
              v_id := NULL;
            END IF;

          ELSIF v_item_name = 'Point Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 30 characters and will be truncated.',v_seq);
                  v_item := SUBSTR(v_item,1,30);
                END IF;
                v_point_name := v_item;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            ELSE
              v_point_name := NULL;
            END IF;

          ELSIF v_item_name = 'Station Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 60 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||' is longer than 60 characters and will be truncated.',v_seq);
                  v_item := SUBSTR(v_item,1,60);
                END IF;
                v_name := v_item;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            ELSE
              v_name := NULL;
            END IF;

          ELSIF v_item_name = 'Primary Type' THEN
            IF v_item IS NOT NULL THEN
         
              v_type1 := validate_primary_type(v_item,p_imp_cfg_seq,v_col_seq,v_seq,v_item_name);
              OPEN c_count_primary_types; -- Sees if there is a secondary type required for this primary type.
                FETCH c_count_primary_types INTO v_rows;
                IF c_count_primary_types%NOTFOUND THEN
                  v_type1 := NULL;
                ELSIF v_rows = 1 THEN -- Primary type does not need a secondary type
                  v_storet_tsmvstc := get_tsmvstc_is_nbr(v_type1,NULL,v_item_name,v_seq);
                  v_check_primary_type := get_primary_type(v_storet_tsmvstc.tsmvstc_is_number,v_storet_tsmvstc.tsmvstc_org_id);
                  v_primary_checked := 'Y';
                ELSIF v_rows > 1 THEN
                  v_primary_checked := 'N'; -- You will need a secondary Type
                ELSE
                  v_type1 := NULL;
                  v_storet_pk.org_id := NULL;
                  v_primary_checked := 'Y';
                  v_status := 'N';
                END IF;
              CLOSE c_count_primary_types;
              IF v_type1 IS NULL THEN
                 v_storet_pk.org_id := NULL;
              END IF;
           ELSE
              v_storet_pk.org_id := NULL;
              v_status := 'N';
              v_primary_checked := 'Y';
              v_type1 := NULL;
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' - Missing primary type.',v_seq);
           END IF;

          ELSIF v_item_name = 'Secondary Type' AND v_type1 IS NOT NULL THEN
            
            IF v_item IS NULL THEN
              v_item := 'None';
            END IF; 
            
            v_secondary_type := validate_secondary_type(v_item,v_type1,p_imp_cfg_seq,v_col_seq,v_seq,v_item_name);
            IF v_secondary_type IS NULL AND v_primary_checked = 'N' THEN
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_type1||' - has incorrect Secondary Type.',v_seq);
              v_status := 'N';
              v_secondary_type := NULL;
            END IF;
            v_storet_tsmvstc := get_tsmvstc_is_nbr(v_type1,v_secondary_type,v_item_name,v_seq);
            v_primary_checked := 'Y';
            v_tsmvstc_is_number := v_storet_tsmvstc.tsmvstc_is_number;
            v_tsmvstc_org_id := v_storet_tsmvstc.tsmvstc_org_id;


          ELSIF v_item_name = 'Establishment Date'  THEN
            IF v_item IS NULL THEN
              v_estab_date := NULL;
            ELSE
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_estab_date := GET_DATE(v_item,v_col_seq,p_imp_cfg_seq,v_line_cnt,v_seq);
                IF v_estab_date IS NULL THEN
                --the date format is bad
                  v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': - Establishment date format does not match format in import configuration' ,v_seq);
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            END IF;

          ELSIF v_item_name = 'Water Depth' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := validate_number(v_item,v_item_name,'9999.999', v_seq);
                IF v_success = 1 THEN
                  v_water_depth := TO_NUMBER(v_item);
                ELSE
                  v_water_depth := NULL;
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            ELSE
              v_water_depth := NULL;
            END IF;

          ELSIF v_item_name = 'Water Depth Units' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq,
                             v_col_seq, v_seq);
              v_water_depth_unit := get_allowable_value(v_dummy_seq, v_seq);
            ELSE
              v_water_depth_unit := NULL;
            END IF;

          ELSIF v_item_name = 'Latitude' THEN
            v_lat_dir := NULL;
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := get_latitude(v_item, v_col_seq,p_imp_cfg_seq, v_seq, v_line_cnt, v_lat_degrees,
                                  v_lat_minutes, v_lat_seconds, v_lat_dir, v_lat_dec_min, v_lat_dec_deg);
                IF v_success = 2 THEN
                  v_status := 'N';
                  v_lat_dec_deg := NULL;
                  v_lat_dec_min := NULL;
                  v_lat_degrees := NULL;
                  v_lat_minutes := NULL;
                  v_lat_seconds := NULL;
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            ELSE
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Latitude is required.',v_seq);
              v_status := 'N';
              v_lat_dec_deg := NULL;
              v_lat_dec_min := NULL;
              v_lat_degrees := NULL;
              v_lat_minutes := NULL;
              v_lat_seconds := NULL;
            END IF;

          ELSIF v_item_name = 'Latitude Direction' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
			        v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NOT NULL THEN
                IF v_lat_dir IS NULL THEN
                  v_lat_dir := get_allowable_value(v_dummy_seq,v_seq);
                ELSE
                  v_check_lat_dir := get_allowable_value(v_dummy_seq,v_seq);
                  IF v_check_lat_dir <> v_lat_dir THEN
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Given Latitude Direction is inconsistent with sign on Latitude.',v_seq);
                    v_status := 'N';
                    v_lat_dir := 'X';  -- so we don't default it in
                  END IF;
                END IF;
              ELSE
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item||' is not an allowable Latitude Direction.',v_seq);
                v_status := 'N';
                v_lat_dir := 'X';  -- so we don't default it in
              END IF;
            ELSE  
              -- v_item IS NULL, default to N
              IF v_lat_Dir IS NULL THEN
                v_lat_dir := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Latitude Direction is required and will be set to N',v_seq);
              END IF;
            END IF;

          ELSIF v_item_name = 'Longitude' THEN
            v_long_dir := NULL;
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := get_longitude(v_item, v_col_seq, p_imp_cfg_seq, v_seq, v_line_cnt, v_long_degrees, v_long_minutes,
                                  v_long_seconds, v_long_dec_min, v_long_dec_deg, v_long_dir);
                IF v_success = 2 THEN
                  v_status := 'N';
                  v_long_dec_deg := NULL;
                  v_long_dec_min := NULL;
                  v_long_degrees := NULL;
                  v_long_minutes := NULL;
                  v_long_seconds := NULL;
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            ELSE
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Longitude is required.',v_seq);
              v_status := 'N';
              v_long_dec_deg := NULL;
              v_long_dec_min := NULL;
              v_long_degrees := NULL;
              v_long_minutes := NULL;
              v_long_seconds := NULL;
            END IF;

          ELSIF v_item_name = 'Longitude Direction' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
			        v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NOT NULL THEN
                IF v_long_dir IS NULL THEN
                  v_long_dir := get_allowable_value(v_dummy_seq,v_seq);
                ELSE
                  v_check_long_dir := get_allowable_value(v_dummy_seq,v_seq);
                  IF v_check_long_dir <> v_long_dir THEN
                    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Given Longitude Direction is inconsistent with sign on Longitude.',v_seq);
                    v_status := 'N';
                    v_long_dir := 'X'; -- so we don't default it in
                  END IF;
                END IF;
              ELSE
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item||' is not an allowable Longitude Direction.',v_seq);
                v_status := 'N';
                v_long_dir := 'X'; -- so we don't default it in
              END IF;
            ELSE  
              -- v_item is NULL, default to W
              IF v_long_dir IS NULL THEN
                v_long_dir := 'W';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Longitude Direction is required and will be set to W',v_seq);
              END IF;
            END IF;

          ELSIF v_item_name = 'Lat/Long Measurement Date' THEN
            IF v_item IS NULL THEN
              v_lat_long_date := NULL;
            ELSE
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_lat_long_date := GET_DATE(v_item,v_col_seq,p_imp_cfg_seq,v_line_cnt,v_seq);
                IF v_lat_long_date IS NULL THEN
                --the date format is bad
                  v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||' format does not match format in import configuration' ,v_seq);
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            END IF;

          ELSIF v_item_name = 'Geopositioning Method' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_method := check_tsmmad_value(v_item,v_item_name,'HORIZONTAL','METHOD',v_col_seq,v_seq,p_imp_cfg_seq,v_import_log_seq);
              IF v_method IS NULL THEN
                v_status := 'N';
              END IF;
            ELSE
              v_method := NULL;
              v_status := 'N';              
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Geopositioning method is required.',v_seq);
            END IF;

          ELSIF v_item_name = 'Geopositioning Datum' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_datum := check_tsmmad_value(v_item,v_item_name,'HORIZONTAL','DATUM',v_col_seq,v_seq,p_imp_cfg_seq,v_import_log_seq);
              IF v_datum IS NULL THEN
                v_status := 'N';
              END IF;
            ELSE
              v_datum := NULL;
              v_status := 'N';              
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Geopositioning Datum is required.',v_seq);
            END IF;

          ELSIF v_item_name = 'Scale' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 20 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||' is longer than 20 characters and will be truncated.',v_seq);
                  v_item := SUBSTR(v_item,1,20);
                END IF;
                 v_gps_scale := v_item;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_gps_scale := NULL;
            END IF;


          ---------
          -- County
          ---------
          ELSIF v_item_name = 'County' THEN

            IF v_item IS NOT NULL THEN
            
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := RTRIM(v_item);
              
              -- translate and validate county
              v_county := validate_county(
                v_item,
                p_imp_cfg_seq,
                v_item_name,
                v_col_seq,
                v_seq
              );
              
              -- did item translate and validate?
              IF (v_county IS NULL) THEN  
                -- if no, county is not valid or translated to NULL             
                -- prevent export to STORET
                v_status := 'N';
              END IF;

            ELSE

                -- County not required for Ocean, Estuary, or Great Lake
              IF v_type1 NOT IN ('Ocean','Estuary','Great Lake') THEN
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': County is required.',v_seq);
              END IF; 
              v_county := NULL;             
            END IF;


          --------            
          -- State
          --------            
          ELSIF v_item_name = 'State' THEN
          
            IF v_item IS NOT NULL THEN
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := RTRIM(v_item);
              
              -- do we already have the county?
              IF v_county IS NOT NULL THEN
                -- if yes
                -- translate and validate the state
                v_state := validate_state(
                  v_item,
                  p_imp_cfg_seq,
                  v_item_name,
                  v_col_seq,
                  v_seq
                );
                
                -- did the state translate and validate?
                IF (v_state IS NOT NULL) THEN
                  -- if yes
                  -- jah 2-24-04 Convert to uppercase because STORET GUI converts state to uppercase
                  v_state := UPPER(v_state);
                  v_storet_tsmgeopa := get_tsmgeopa_is_nbr(v_state,v_county);
                  v_tsmgeopa_is_number := v_storet_tsmgeopa.is_number;

                  -- jah 5-12-04 (Task LOG 474)
                  -- Added following statement because the
                  -- org id is required to identify the county!
                  v_tsmgeopa_org_id := v_storet_tsmgeopa.org_id;

                  IF v_tsmgeopa_is_number IS NULL THEN
                    -- Valid Combination of State and County not found
                     v_status := 'N';
                     Sim2_Import_Pkg.write_error(
                       v_import_log_seq,
                       'Line '||v_line_cnt||': The county of '||v_county||
                       ' is not valid for the State of '||v_state||'.',
                       v_seq
                     );
                  END IF;
                  
                ELSE
                  -- State was invalid or translated to NULL             
                  v_status := 'N';
                END IF;
              
              ELSE
              
                -- No County
                v_status := 'N';
                Sim2_Import_Pkg.write_error(
                  v_import_log_seq,
                  'Line '||v_line_cnt||': State not loaded since county has not been provided.',
                  v_seq
                );
              END IF;
              
            ELSE
              -- State is not required for Ocean, Estuary, or Great Lake
              IF v_type1  IN ('Ocean','Estuary','Great Lake') THEN
                IF v_county IS NOT NULL THEN
                  -- If county provided with out State give error
                  Sim2_Import_Pkg.WRITE_ERROR(
                    v_import_log_seq,
                    'Line '||v_line_cnt||': State is required for all stations when county is provided.',
                    v_seq
                  );
                  v_status := 'N';
                END IF;
              ELSE
                Sim2_Import_Pkg.WRITE_ERROR(
                  v_import_log_seq,
                  'Line '||v_line_cnt||': State is required.',
                  v_seq
                );
                v_status := 'N';
              END IF;

              -- No State or County will be loaded
              v_tsmgeopa_is_number := NULL;
              v_tsmgeopa_org_id := NULL; 

            END IF;

          ELSIF v_item_name = 'HUC' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_huc := validate_huc(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq);
              IF v_huc IS NOT NULL THEN
                v_storet_tsmfhu := get_tsmfhu_is_nbr(v_huc);
                v_tsmfhu_is_number := v_storet_tsmfhu.is_number;
                v_tsmfhu_org_id    := v_storet_tsmfhu.org_id;
              ELSE
                v_tsmfhu_is_number := NULL;
                v_tsmfhu_org_id    := NULL;
                v_status := 'N';
              END IF;
            ELSE
             v_tsmfhu_is_number := NULL;
              v_tsmfhu_org_id    := NULL;
            END IF;

          ELSIF v_item_name = 'NRCS Watershed ID' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
                v_trans_exist := TRUE;
                v_converted_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                IF length(v_converted_item) > 8 THEN
                  v_nrcs := SUBSTR(v_converted_item,1,8);
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': NRCS ID, '||v_converted_item||', is longer than 8 characters and will be truncated.',v_seq);
                ELSE
                  v_nrcs := v_converted_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_nrcs := NULL;
            END IF;
            IF v_nrcs = 'NULL' THEN
              v_nrcs := NULL;
            END IF;

          ELSIF v_item_name = 'Station Description' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 4000 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||' is longer than 4000 characters and will be truncated.',v_seq);
                  v_item := SUBSTR(v_item,1,4000);
                END IF;
                v_sta_desc := v_item;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_sta_desc := NULL;
            END IF;

          ELSIF v_item_name = 'Ocean Name' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL AND v_type1 = 'Ocean' THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Ocean Name is required when primary type is Ocean',v_seq);
                v_status := 'N';
              ELSIF v_type1 != 'Ocean' THEN
                v_ocean_name := NULL;
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Ocean Name Ignored - only used when primary type is Ocean',v_seq);
              ELSE
                v_ocean_name := get_allowable_value(v_dummy_seq, v_seq);
              END IF;
            ELSE
              v_ocean_name := NULL;
            END IF;

          ELSIF v_item_name = 'Shore Relation' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL AND v_type1 = 'Ocean'  THEN
                v_status := 'N';
                v_shore_relation := NULL;
              ELSIF v_type1 != 'Ocean' THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': Shore Relation ignored. It is only used when primary type is Ocean' ,v_seq);
                v_shore_relation := NULL;
              ELSE
                v_shore_relation := get_allowable_value(v_dummy_seq, v_seq);
              END IF;
            ELSE
              v_shore_relation := NULL;
            END IF;

          ELSIF v_item_name = 'Elevation' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := validate_number(v_item,v_item_name,'99999.9999',v_seq);
                IF v_success = 0 THEN
                  v_elevation_msr := NULL;
                  v_status := 'N';
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': Elevation value '||v_item||' is not allowable. It must be a number.',v_seq);
                ELSE
                  v_elevation_msr := v_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
             ELSE
               v_elevation_msr := NULL;
             END IF;

          ELSIF v_item_name = 'Elevation Units' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_elevation_msr IS NOT NULL THEN
                IF v_dummy_seq IS NULL THEN
                  v_elevation_unit := NULL;
                  v_status := 'N';
                ELSE
                  v_elevation_unit := get_allowable_value(v_dummy_seq,v_seq);
                END IF;
              ELSE
                v_elevation_unit := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Elevation Unit is not needed since Elevation is not provided.',v_seq);
              END IF;
            ELSE
              v_elevation_unit := NULL;
            END IF;

          ELSIF v_item_name = 'Elevation Method' THEN --sfp, this section rewritten 062701
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_elevation_method := check_tsmmad_value(v_item,v_item_name,'VERTICAL','METHOD',v_col_seq,v_seq,p_imp_cfg_seq,v_import_log_seq);
              IF v_elevation_method IS NULL THEN
                v_status := 'N';
              END IF;
            ELSE
              v_elevation_method := NULL;
            END IF;

          ELSIF v_item_name = 'Elevation Datum' THEN --sfp, this section rewritten 062701
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_elevation_datum := check_tsmmad_value(v_item,v_item_name,'VERTICAL','DATUM',v_col_seq,v_seq,p_imp_cfg_seq,v_import_log_seq);
              IF v_elevation_datum IS NULL THEN
                v_status := 'N';
              END IF;
            ELSE
              v_elevation_datum := NULL;
            END IF;

          ELSIF v_item_name = 'Elevation Measurement Date' THEN
            IF v_item IS NULL THEN
              v_elevation_date := NULL;
            ELSE
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_elevation_date := GET_DATE(v_item,v_col_seq,p_imp_cfg_seq,v_line_cnt,v_seq);
                IF v_elevation_date IS NULL THEN
                --the date format is bad
                  v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': - '||v_item_name||' format does not match format in import configuration' ,v_seq);
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to null - set status to 'N'
              END IF;
            END IF;

          ELSIF v_item_name = 'EPA Key Identifier' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 36 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': EPA Key Identifier is longer than 36 characters and will be truncated.',v_seq);
                  v_EPA_Key := substr(v_item,1,36);
                ELSE
                  v_EPA_key := v_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_EPA_key := NULL;
            END IF;

          ELSIF v_item_name = 'Ecoregion Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 60 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 60 characters and will be truncated.',v_seq);
                  v_ecoregion := substr(v_item,1,60);
                ELSE
                  v_ecoregion := v_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_ecoregion := NULL;
            END IF;

          ELSIF v_item_name = 'Influence Area' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 120 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 120 characters and will be truncated.',v_seq);
                  v_influence := substr(v_item,1,120);
                ELSE
                  v_influence := v_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_influence := NULL;
            END IF;

          ELSIF v_item_name = 'Travel Directions' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 1999 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||'are longer than 1999 characters and will be truncated.',v_seq);
                  v_travel_dir := substr(v_item,1,1999);
                ELSE
                  v_travel_dir := v_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_travel_dir := NULL;
            END IF;


          ELSIF v_item_name = 'ZID Relation' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL THEN
                v_status := 'N';
                v_zid_rel := NULL;
              ELSE
                v_zid_rel := get_allowable_value(v_dummy_seq, v_seq);
              END IF;
            ELSE
              v_zid_rel := NULL;
            END IF;


          ELSIF v_item_name = 'Native American Land Name' THEN
            IF v_item IS NOT NULL THEN
              v_native_name := validate_nal(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,NULL, NULL,1,v_seq);
              IF v_native_name IS NULL THEN
                v_status := 'N';
              END IF;
            ELSE
              v_native_name := NULL;
            END IF;

          ELSIF v_item_name = 'Native American Land State' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_native_state := validate_nal(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,NULL, NULL,2,v_seq);
              IF v_native_state IS NOT NULL THEN
                IF v_native_name IS NOT NULL THEN
                  v_native := validate_nal(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_native_name,v_native_state,3,v_seq);
                  IF v_native = 'BAD' THEN
                     v_status := 'N';
                     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_native_name||'/'||v_native_state||' is an invalid Native Americal Land Name/State combination.',v_seq);
                  END IF;
                ELSE --state but no Name
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Native Americal Land State is invalid without a Name and will be ignored.',v_seq);
                END IF;
              ELSE
                v_status := 'N';
              END IF;
            ELSE  --no state
              IF v_native_name IS NOT NULL THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': Native Americal Land Name is invalid without a State and will be ignored.',v_seq);
              END IF;
              v_native_state := NULL;
              v_native_name := NULL;
              v_native_code := NULL;
            END IF;

          ELSIF v_item_name = 'Additional Ocean Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_add_ocean := substr(v_item,1,30);
                ELSE
                  v_add_ocean := v_item;
                END IF;
                IF v_type1 != 'Ocean' THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - only used when primary type is Ocean' ,v_seq);
                  v_add_ocean := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_add_ocean := NULL;
            END IF;

          ELSIF v_item_name = 'Ocean Station Dist to Shore' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := validate_number(v_item,v_item_name,'9999.99',v_seq);
                IF v_success = 0 THEN
                  v_ocean_dist := NULL;
                  v_status := 'N';
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a number in the allowed format of ZZZZ.ZZ.',v_seq);
                ELSE
                  v_ocean_dist := v_item;
                END IF;
                IF v_type1 != 'Ocean' THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - only used when primary type is Ocean' ,v_seq);
                  v_ocean_dist := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
             ELSE
               v_ocean_dist := NULL;
             END IF;

          ELSIF v_item_name = 'Ocean Station Dist to Shore Units' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL AND v_ocean_dist IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a valid value.' ,v_seq);
                v_status := 'N';
                v_ocean_dist_unit := NULL;
              ELSIF v_dummy_seq IS NOT NULL AND v_ocean_dist IS NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not needed if Distance is not given.' ,v_seq);
              ELSIF v_dummy_seq IS NOT NULL AND v_ocean_dist IS NOT NULL THEN
                v_ocean_dist_unit := get_allowable_value(v_dummy_seq,v_seq);
              END IF;
            ELSE
              v_ocean_dist_unit := NULL;
            END IF;

          ELSIF v_item_name = 'Ocean Station Ref Point' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_Ocean_ref := substr(v_item,1,30);
                ELSE
                  v_ocean_ref := v_item;
                END IF;
                IF v_type1 != 'Ocean' THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is only used when primary type is Ocean' ,v_seq);
                  v_ocean_ref := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_ocean_ref := NULL;
            END IF;

          ELSIF v_item_name = 'Ocean Station Bottom Topography' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 254 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' is longer than 254 characters and will be truncated.',v_seq);
                  v_ocean_bottom := substr(v_item,1,254);
                ELSE
                  v_ocean_bottom := v_item;
                END IF;
                IF v_type1 != 'Ocean' THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - only used when primary type is Ocean' ,v_seq);
                  v_ocean_bottom := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_ocean_bottom := NULL;
            END IF;

         ELSIF v_item_name = 'Primary Estuary State' THEN
            IF v_item IS NOT NULL THEN
              IF v_type1 != 'Estuary' THEN
               Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
               v_estuary_state := NULL;
              ELSE
               v_estuary_state := validate_estuary_state(v_item,p_imp_cfg_seq,v_item_name,v_col_seq,v_seq);
              END IF;
           ELSE
              v_estuary_state := NULL;
           END IF;

          ELSIF v_item_name = 'Primary Estuary' THEN
            IF v_item IS NOT NULL THEN
              IF v_type1 != 'Estuary' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
                  v_primary_is_nbr := NULL;
                  v_primary_est := NULL;
              ELSIF v_estuary_state IS NULL THEN
                  v_primary_is_nbr := NULL;
                  v_primary_est := NULL;
              ELSE
                  v_primary_est := validate_primary_estuary(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_estuary_state, v_primary_is_nbr, v_seq);
                  IF v_primary_est IS NULL THEN
                    v_status := 'N';
                    v_primary_is_nbr := NULL;
                  END IF;
              END IF;
            ELSE
              v_primary_est := NULL;
              v_primary_is_nbr := NULL;
            END IF;

          ELSIF v_item_name = 'Secondary Estuary' THEN
            IF v_item IS NOT NULL THEN
              IF v_type1 != 'Estuary' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
                  v_secondary_is_nbr := NULL;
                  v_secondary_est := NULL;
              ELSIF v_primary_is_nbr IS NULL THEN
                  v_secondary_is_nbr := NULL;
                  v_secondary_est := NULL;
              ELSE
                  v_secondary_est := validate_secondary_estuary(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_primary_is_nbr, v_secondary_is_nbr, v_seq);
                  IF v_secondary_est IS NULL THEN
                    v_secondary_is_nbr := NULL;
                    v_status := 'N';
                  END IF;
              END IF;
            ELSE
              v_secondary_est := NULL;
              v_secondary_is_nbr := NULL;
            END IF;

          ELSIF v_item_name = 'Other Estuary' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_other_estuary := substr(v_item,1,30);
                ELSE
                  v_other_estuary := v_item;
                END IF;
                IF v_type1 != 'Estuary' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
                  v_other_estuary := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_other_estuary := NULL;
            END IF;

          ELSIF v_item_name = 'Additional Estuary Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_add_estuary := substr(v_item,1,30);
                ELSE
                  v_add_estuary := v_item;
                END IF;
                IF v_type1 != 'Estuary' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
                  v_add_estuary := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_add_estuary := NULL;
            END IF;

          ELSIF v_item_name = 'Estuary Distance to Shore' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := validate_number(v_item,v_item_name,'9999.99',v_seq);
                IF v_success = 0 THEN
                  v_estuary_dist := NULL;
                  v_status := 'N';
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a number in the allowed format of ZZZZ.ZZ.',v_seq);
                ELSIF v_type1 != 'Estuary' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
                  v_estuary_dist := NULL;
                ELSE
                  v_estuary_dist := v_item;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
             ELSE
               v_estuary_dist := NULL;
             END IF;

          ELSIF v_item_name = 'Estuary Dist to Shore Units' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL AND v_estuary_dist IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a valid value.' ,v_seq);
                v_status := 'N';
                v_est_dist_unit := NULL;
              ELSIF v_dummy_seq IS NOT NULL AND v_estuary_dist IS NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', ignored - it is not needed if Distance is not given.' ,v_seq);
              ELSIF v_dummy_seq IS NOT NULL AND v_estuary_dist IS NOT NULL THEN
                v_est_dist_unit := get_allowable_value(v_dummy_seq,v_seq);
              END IF;
            ELSE
              v_est_dist_unit := NULL;
            END IF;

          ELSIF v_item_name = 'Estuary Reference Point' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_estuary_ref := substr(v_item,1,30);
                ELSE
                  v_estuary_ref := v_item;
                END IF;
                IF v_type1 != 'Estuary' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Estuary.',v_seq);
                  v_estuary_ref := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_estuary_ref := NULL;
            END IF;

          ELSIF v_item_name = 'Great Lake' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL  THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a valid value.' ,v_seq);
                v_status := 'N';
                v_great_lake := NULL;
              ELSIF v_dummy_seq IS NOT NULL AND v_type1 != 'Great Lake' THEN
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. Not used unless Primary Type is Great Lake.',v_seq);
                v_great_lake := NULL;
              ELSE
                v_great_lake := get_allowable_value(v_dummy_seq,v_seq);
              END IF;
            ELSE
              v_great_lake := NULL;
            END IF;

          ELSIF v_item_name = 'Additional Great Lake Name' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_add_lake := substr(v_item,1,30);
                ELSE
                  v_add_lake := v_item;
                END IF;
                IF v_type1 != 'Great Lake' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Great Lake.',v_seq);
                  v_add_lake := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_add_lake := NULL;
            END IF;

          ELSIF v_item_name = 'Great Lake Dist to Shore' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                v_success := validate_number(v_item,v_item_name,'9999.99',v_seq);
                IF v_success = 0 THEN
                  v_lake_dist := NULL;
                  v_status := 'N';
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a number in the allowed format of ZZZZ.ZZ.',v_seq);
                ELSE
                  v_lake_dist := v_item;
                END IF;
                IF v_type1 != 'Great Lake' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Great Lake.',v_seq);
                  v_lake_dist := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
             ELSE
               v_lake_dist := NULL;
             END IF;

          ELSIF v_item_name = 'Great Lake Dist to Shore Units' THEN
            IF v_item IS NOT NULL THEN
              v_item := RTRIM(v_item); -- 061203 slp - Remove trailing spaces.
              v_dummy_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_seq);
              IF v_dummy_seq IS NULL AND v_lake_dist IS NOT NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||v_item_name||', '||v_item||' is not a valid value.' ,v_seq);
                v_status := 'N';
                v_lake_dist_unit := NULL;
              ELSIF v_dummy_seq IS NOT NULL AND v_lake_dist IS NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored - it is not needed if Distance is not given.' ,v_seq);
              ELSIF v_dummy_seq IS NOT NULL AND v_lake_dist IS NOT NULL THEN
                v_lake_dist_unit := get_allowable_value(v_dummy_seq,v_seq);
              END IF;
            ELSE
              v_lake_dist_unit := NULL;
            END IF;

          ELSIF v_item_name = 'Great Lake Reference Point' THEN
            IF v_item IS NOT NULL THEN
              -- 09/30/03 TCJ - add code to check for translations
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN
                IF LENGTH(v_item) > 30 THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||', is longer than 30 characters and will be truncated.',v_seq);
                  v_lake_ref := substr(v_item,1,30);
                ELSE
                  v_lake_ref := v_item;
                END IF;
                IF v_type1 != 'Great Lake' THEN
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Warning: Line '||v_line_cnt||': '||v_item_name||' ignored. It is not used unless Station Type is Great Lake.',v_seq);
                  v_lake_ref := NULL;
                END IF;
              ELSE							 
                v_status := 'N'; -- value was translated to null
							END IF;
            ELSE
              v_lake_ref := NULL;
            END IF;

          ELSIF v_item_name = 'Document/Graphic' THEN
            IF v_item IS NOT NULL THEN
              v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
              IF v_item IS NOT NULL THEN  
                IF sim2_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN 
                  v_document := v_item;
                ELSE
                  v_status := 'N'; -- Invalid document - SET STATUS TO N 
                  v_document := NULL;
                END IF;
              ELSE
                v_status := 'N'; -- v_item translated to NULL - SET STATUS TO N 
                v_document := NULL; 
              END IF;                 
            ELSE
              v_document := NULL;
            END IF;

          END IF; --end of v_item_names

          END IF;
        END IF;
      END LOOP;

      /*** END STATION LOOP ***/
      --Now check all the STORET business rules
      IF v_id IS NULL THEN
         Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Station ID is required.',v_seq);
         v_status := 'N';
      END IF;

      IF v_primary_checked = 'N' THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Primary type, '||v_type1||', needs secondary type.',v_seq);
        v_status := 'N';
        v_tsmvstc_is_number              := NULL;
        v_tsmvstc_org_id                 := NULL;
      END IF;

      -- Default Lat/Long to NW hemisphere if not included in import file
      IF v_lat_dir IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Latitude direction has not been provided and will be defaulted to N',v_seq);
        v_lat_dir := 'N';
      END IF;

      IF v_long_dir IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Longitude direction has not been provided and will be defaulted to W',v_seq);
        v_long_dir := 'W';
      END IF;

      -- Need to check this post loop as lat direction is not required in import file
      IF v_lat_dir = 'S' AND v_lat_dec_deg > 0 THEN
        IF v_lat_dec_deg < 10 THEN
          v_lat_dec_deg := '-'||TRUNC(v_lat_dec_deg,7);
        ELSE
          v_lat_dec_deg := '-'||TRUNC(v_lat_dec_deg,6);
        END IF;
      END IF;

      -- Need to check this post loop as long direction is not required in import file
      IF v_long_dir = 'W' AND v_long_dec_deg > 0 THEN
        IF v_long_dec_deg < 100 THEN
          v_long_dec_deg := '-'||TRUNC(v_long_dec_deg,6);
        ELSE
          v_long_dec_deg := '-'||TRUNC(v_long_dec_deg,5);
        END IF;
      END IF;
      
      --Elevation Rules
      IF v_elevation_unit IS NULL AND v_elevation_msr IS NOT NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Elevation units are required when elevation is provided.',v_seq);
      END IF;
      IF v_elevation_method IS NULL AND v_elevation_msr IS NOT NULL THEN
         v_status := 'N';
         Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Elevation method is required when elevation is provided.',v_seq);
      END IF;
      IF v_elevation_datum IS NULL AND v_elevation_msr IS NOT NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Elevation datum is required when elevation is provided.',v_seq);
      END IF;

      --Estuary rules
      IF v_type1 = 'Estuary' THEN
        IF v_estuary_state IS NULL THEN
          Sim2_Import_Pkg.WRITE_ERROR(
            v_import_log_seq,
            'Line '||v_line_cnt||' : Estuary State is required when Station type is Estuary.',
            v_seq
          );
          v_status := 'N';
        ELSIF v_primary_is_nbr IS NULL THEN
          Sim2_Import_Pkg.WRITE_ERROR(
            v_import_log_seq,
            'Line '||v_line_cnt||' : Primary Estuary is required when Station type is Estuary.',
            v_seq
          );
          v_status := 'N';
        ELSIF v_secondary_is_nbr IS NULL THEN
          v_tsmestry_is_nbr := v_primary_is_nbr;
        ELSE
          v_tsmestry_is_nbr := v_secondary_is_nbr;
        END IF;
      END IF;

      -- Ocean rules
      IF v_type1 = 'Ocean' AND v_ocean_name IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Ocean Name is required when Station type is Ocean.',v_seq);
      END IF;
      IF v_type1 = 'Ocean' AND v_shore_relation IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Shore Relation is required when Station type is Ocean.',v_seq);
      END IF;

      -- Lake rules
      IF v_type1 = 'Great Lake' AND v_great_lake IS NULL THEN
        v_status := 'N';
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Great Lake is required when Station type is Great Lake.',v_seq);
      END IF;

      --NAL Rules
      IF v_native_name IS NOT NULL AND v_native_state IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Native American Land State is required when Name is given.',v_seq);
        v_status := 'N';
        v_native_name := NULL;
      END IF;

     -- Well Rules (One can import a well without the details, but if there is one you better have the all)\
      IF v_type1 = 'Well' THEN
        IF v_well_num IS NOT NULL OR v_well_name IS NOT NULL OR v_well_use IS NOT NULL OR v_well_status IS NOT NULL THEN
          IF v_well_num IS NULL THEN
            v_status := 'N';
            Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Well Number is required when any other Well-specific data are given. (These and other well features can be added using the Wells Import Type.)',v_seq);
          END IF;
          IF v_well_name IS NULL THEN
            Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Well Name is required when any other Well-specific data are given. (These and other well features can be added using the Wells Import Type.)',v_seq);
            v_status := 'N';
            v_well_num := NULL;
          END IF;
          IF v_well_use IS NULL THEN
            Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Well Use is required when any other Well-specific data are given. (These and other well features can be added using the Wells Import Type.)',v_seq);
            v_status := 'N';
            v_well_num := NULL;
          END IF;
          IF v_well_status IS NULL THEN
            Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||' : Well Status is required when any other Well-specific data are given. (These and other well features can be added using the Wells Import Type.)',v_seq);
            v_status := 'N';
            v_well_num := NULL;
          END IF;
        END IF;
      END IF;

      --Units rules
      IF v_water_depth_unit IS NOT NULL AND v_water_depth IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Water Depth Units ignored since Water Depth is not given.',v_seq);
      END IF;
      IF v_ocean_dist_unit IS NOT NULL AND v_ocean_dist IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Ocean Station Distance to Shore Units ignored since distance is not given.',v_seq);
      END IF;
      IF v_est_dist_unit IS NOT NULL AND v_estuary_dist IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Estuary Distance to Shore Units ignored since distance is not given.',v_seq);
      END IF;
      IF v_lake_dist_unit IS NOT NULL AND v_lake_dist IS NULL THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Warning: Line '||v_line_cnt||': Great Lake Distance to Shore Units ignored since distance is not given.',v_seq);
      END IF;
      
      --Scale rules
      -- 018 is 'Interpolation-map'
      IF v_gps_scale IS NULL AND v_Method = RPAD('018',12) THEN
        Sim2_Import_Pkg.WRITE_ERROR(
          v_import_log_seq,
          'Line '||v_line_cnt||' : Scale is required when Geopositioning Method is Interpolation-Map.',
          v_seq);
        v_status := 'N';
      END IF;
        
      

      /* create STATION */
      BEGIN
        /* write data to record and send to create station function */
        /* get station location sequence */
        BEGIN
          OPEN c_get_station_location_seq;
          FETCH c_get_station_location_seq INTO v_stl_seq;
          CLOSE c_get_station_location_seq;
        EXCEPTION
          WHEN OTHERS THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq,' - getting station location sequence.',v_seq);
        END;

        v_sta_rec.STA_SEQ                := v_seq;
        v_sta_rec.STA_ID                 := v_id;
        v_sta_rec.STA_TSMORGAN_ORG_ID    := p_org_id;
        v_sta_rec.STA_TSMORGAN_IS_NUMBER := v_tsmorgan_is_number;
        v_sta_rec.STA_ORG_ID             := p_org_id;
        v_sta_rec.STA_STATUS             := v_status;
        v_sta_rec.STA_TSMVSTC_IS_NUMBER  := v_tsmvstc_is_number;
        v_sta_rec.STA_TSMVSTC_ORG_ID     := v_tsmvstc_org_id;
        v_sta_rec.STA_NAME               := v_name;
        v_sta_rec.STA_IMPORT_SEQ         := v_import_log_seq;
        v_sta_rec.STA_ESTAB_DATE         := v_estab_date;
        v_sta_rec.STA_DESC               := v_sta_desc;
        v_sta_rec.STA_ECOREGION          := v_ecoregion;
        v_sta_rec.STA_WATER_DEPTH        := v_water_depth;
        v_sta_rec.STA_WATER_DEPTH_UNIT   := v_water_depth_unit;
        v_sta_rec.STA_EPA_KEY_ID         := v_EPA_key;
        v_sta_rec.STA_ZID_REL            := v_zid_rel;
        v_sta_rec.STA_INFLUENCE_AREA     := v_influence;
        v_sta_rec.STA_TRAVEL_DIR         := v_travel_dir;
        v_sta_rec.STA_DOCUMENT           := v_document;

        v_stl_rec.STL_SEQ                := v_stl_seq;
        v_stl_rec.STL_STA_SEQ            := v_seq;
        v_stl_rec.STL_TYPE_CODE          := '*POINT OF RECORD';
        v_stl_rec.STL_SEQUENCE_NUMBER    := 0;
        v_stl_rec.STL_POINT_NAME         := v_point_name;
        v_stl_rec.STL_LAT_DEGREES        := v_lat_degrees;
        v_stl_rec.STL_LAT_MINUTES        := v_lat_minutes;
        v_stl_rec.STL_LAT_DEC_DEGREES    := v_lat_dec_deg;
        v_stl_rec.STL_LAT_DEC_MINUTES    := v_lat_dec_min;
        v_stl_rec.STL_LAT_SECONDS        := v_lat_seconds;
        v_stl_rec.STL_LAT_DIR            := v_lat_dir;
        v_stl_rec.STL_LONG_DEGREES       := v_long_degrees;
        v_stl_rec.STL_LONG_MINUTES       := v_long_minutes;
        v_stl_rec.STL_LONG_DEC_DEGREES   := v_long_dec_deg;
        v_stl_rec.STL_LONG_DEC_MINUTES   := v_long_dec_min;
        v_stl_rec.STL_LONG_SECONDS       := v_long_seconds;
        v_stl_rec.STL_LONG_DIR           := v_long_dir;
        v_stl_rec.STL_LAT_LONG_MSR_DATE  := v_lat_long_date;
        v_stl_rec.STL_LATLONG_ACCURCY    := NULL;
        v_stl_rec.STL_LATLONG_ACCURCY_UNIT := NULL;
        v_stl_rec.STL_GPS_SCALE          := v_gps_scale;
        v_stl_rec.STL_TSMGEOPA_IS_NUMBER := v_tsmgeopa_is_number;
        v_stl_rec.STL_TSMGEOPA_ORG_ID    := v_tsmgeopa_org_id;
        v_stl_rec.STL_TSMFHU_IS_NUMBER   := v_tsmfhu_is_number;
        v_stl_rec.STL_TSMFHU_ORG_ID      := v_tsmfhu_org_id;
        v_stl_rec.STL_TSMRRR_IS_NUMBER   := NULL;
        v_stl_rec.STL_TSMRRR_ORG_ID      := NULL;
        v_stl_rec.STL_GEOPOS_METHOD      := v_method;
        v_stl_rec.STL_GEOPOS_DATUM       := v_datum;
        v_stl_rec.STL_ELEVATION_MSR      := v_elevation_msr;
        v_stl_rec.STL_ELEVATION_UNIT     := v_elevation_unit;
        v_stl_rec.STL_ELEV_METHOD        := v_elevation_method;
        v_stl_rec.STL_ELEV_DATUM         := v_elevation_datum;
        v_stl_rec.STL_ELEV_MSR_DATE      := v_elevation_date;
        v_stl_rec.STL_TSMNAL_CD          := v_native_code;
        v_stl_rec.STL_TSMNAL_STATE       := v_native_state;
        v_stl_rec.STL_NRCS               := v_nrcs;

        /* Go ahead and create the records for the station and point of record */
        /* insert station */
        IF v_id IS NOT NULL THEN
           v_success := create_station(v_sta_rec);
        END IF;
        /* if insert function failed, write error and prevent station from being exported */

        IF v_success != 1 THEN
          UPDATE sim_stations
          SET sta_status = 'N'
          WHERE sta_seq = v_seq;
          Sim2_Import_Pkg.write_error(
            v_import_log_seq, 
            'Line ' || v_line_cnt || ': Failed to create station in sim_stations. Activity will not be exported.',
            v_seq
          );
        ELSE --continue - create location
          v_success := create_station_location(v_stl_rec);
            /* if insert function failed, write error and prevent activity from being exported */
          IF v_success != 1 THEN
            UPDATE sim_stations
            SET sta_status = 'N'
            WHERE sta_seq = v_seq;
            Sim2_Import_Pkg.write_error(
              v_import_log_seq, 
              'Line ' || v_line_cnt || ': Failed to create station location in sim_station_locations. Station will not be exported.',
              v_seq
            );
          END IF;
        END IF;

        IF v_type1 = 'Ocean' THEN
          BEGIN
            OPEN c_get_ocean_seq;
            FETCH c_get_ocean_seq INTO v_ocean_seq;
            CLOSE c_get_ocean_seq;
          EXCEPTION
            WHEN OTHERS THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq,' - getting ocean sequence.',v_seq);
          END;

          v_soc_rec.SOC_SEQ                := v_ocean_seq;
          v_soc_rec.SOC_STL_SEQ            := v_stl_seq;
          v_soc_rec.SOC_NAME               := v_ocean_name;
          v_soc_rec.SOC_ADDITIONAL_LOC     := v_add_ocean;
          v_soc_rec.SOC_SHORE_RELATION     := v_shore_relation;
          v_soc_rec.SOC_SHORE_DISTANCE     := v_ocean_dist;
          v_soc_rec.SOC_SHORE_DIST_UNIT    := v_ocean_dist_unit;
          v_soc_rec.SOC_REFERENCE_PT       := v_ocean_ref;
          v_soc_rec.SOC_BOTTOM_TOPOGRAPHY  := v_ocean_bottom;

        ELSIF v_type1 = 'Estuary' THEN

          OPEN c_get_estuary_seq;
          FETCH c_get_estuary_seq INTO v_estuary_seq;
          CLOSE c_get_estuary_seq;

          v_estuary_rec.SEL_SEQ                  := v_estuary_seq;
          v_estuary_rec.SEL_TSMALP_IS_NUMBER     := NULL;
          v_estuary_rec.SEL_TSMALP_ORG_ID        := NULL;
          v_estuary_rec.SEL_OTHER_ESTUARY        := v_other_estuary;
          v_estuary_rec.SEL_ADD_LOC_NAME         := v_add_estuary;
          v_estuary_rec.SEL_SHORE_DIST           := v_estuary_dist;
          v_estuary_rec.SEL_SHORE_DIST_UNIT      := v_est_dist_unit;
          v_estuary_rec.SEL_REF_PT               := v_estuary_ref;
          v_estuary_rec.SEL_TSMESTRY_ORG_ID      := 'ZZZZZZZZ';
          v_estuary_rec.SEL_TSMESTRY_IS_NUMBER   := v_tsmestry_is_nbr;
          v_estuary_rec.SEL_STL_SEQ              := v_stl_seq;

        ELSIF v_type1 = 'Great Lake' THEN
          OPEN c_get_great_lake_seq;
          FETCH c_get_great_lake_seq INTO v_lake_seq;
          CLOSE c_get_great_lake_seq;

          v_lake_rec.sgl_seq                   := v_lake_seq;
          v_lake_rec.SGL_TSMALP_IS_NUMBER      := NULL;
          v_lake_rec.SGL_TSMALP_ORG_ID         := NULL;
          v_lake_rec.SGL_LAKE_NAME             := v_great_lake;
          v_lake_rec.SGL_ADD_LAKE_NAME         := v_add_lake;
          v_lake_rec.SGL_SHORE_DIST            := v_lake_dist;
          v_lake_rec.SGL_SHORE_DIST_UNIT       := v_lake_dist_unit;
          v_lake_rec.SGL_REF_PT                := v_lake_ref;
          v_lake_rec.SGL_STL_SEQ               := v_stl_seq;

        ELSIF v_type1 = 'Well' AND v_well_num IS NOT NULL THEN

          OPEN c_get_well_seq;
            FETCH c_get_well_seq INTO v_well_seq;
          CLOSE c_get_well_seq;
          v_well_rec.SSW_SEQ                   := v_well_seq;
          v_well_rec.SSW_STA_SEQ               := v_seq;
          v_well_rec.SSW_WELL_STATUS           := v_well_status;
          v_well_rec.SSW_IS_NUMBER             := NULL;
          v_well_rec.SSW_ORG_ID                := NULL;
          v_well_rec.SSW_NAME                  := v_well_name;
          v_well_rec.SSW_WELL_NO               := v_well_num;
          v_well_rec.SSW_WELL_USE              := v_well_use;
          v_well_rec.SSW_DEVELOP_METHOD        := NULL;
          v_well_rec.SSW_WELL_CONST_METH       := NULL;
          v_well_rec.SSW_WATER_USE            := NULL;
          v_well_rec.SSW_GRADIENT_TYPE         := NULL;
          v_well_rec.SSW_DEPTH_COMPLETION      := NULL;
          v_well_rec.SSW_DEPTH_COMPLETION_UNIT := NULL;
          v_well_rec.SSW_DEPTH_HOLE            := NULL;
          v_well_rec.SSW_DEPTH_HOLE_UNIT       := NULL;
          v_well_rec.SSW_WELL_HEIGHT           := NULL;
          v_well_rec.SSW_WELL_HEIGHT_UNIT      := NULL;
          v_well_rec.SSW_CASING_HEIGHT         := NULL;
          v_well_rec.SSW_CASING_HEIGHT_UNIT    := NULL;
          v_well_rec.SSW_DISINFECTED           := NULL;
          v_well_rec.SSW_CONST_START_DATE      := NULL;
          v_well_rec.SSW_CONST_END_DATE        := NULL;
          v_well_rec.SSW_INIT_PUMP_DUR         := NULL;
          v_well_rec.SSW_INIT_PUMP_DUR_UNIT    := NULL;
          v_well_rec.SSW_INIT_PUMP_RATE        := NULL;
          v_well_rec.SSW_INIT_PUMP_RATE_UNIT   := NULL;
          v_well_rec.SSW_NATURAL_FLOW          := NULL;
          v_well_rec.SSW_INIT_BOREHOLE_DIAM    := NULL;
          v_well_rec.SSW_INIT_BOREHOLE_DIAM_UNIT  := NULL;
          v_well_rec.SSW_TSMSTATN_IS_NUMBER       := NULL;
          v_well_rec.SSW_TSMSTATN_ORG_ID          := NULL;
          v_well_rec.SSW_STATUS                   := NULL;
          v_well_rec.SSW_IMPORT_SEQ               := NULL;
          v_well_rec.SSW_EXPORT_SEQ               := NULL;

          /* Create a location record for the well head */
          OPEN c_get_station_location_seq;
            FETCH c_get_station_location_seq INTO v_stl_seq;
          CLOSE c_get_station_location_seq;

          v_stl_rec.STL_SEQ                := v_stl_seq;
          v_stl_rec.STL_STA_SEQ            := v_seq;
          v_stl_rec.STL_SSW_SEQ            := v_well_rec.SSW_SEQ;
          v_stl_rec.STL_TYPE_CODE          := 'WELL HEAD';
          v_stl_rec.STL_SEQUENCE_NUMBER    := 0;
          v_stl_rec.STL_LAT_DEGREES        := v_lat_degrees;
          v_stl_rec.STL_LAT_MINUTES        := v_lat_minutes;
          v_stl_rec.STL_LAT_DEC_MINUTES    := v_lat_dec_min;
          v_stl_rec.STL_LAT_SECONDS        := v_lat_seconds;
          v_stl_rec.STL_LAT_DIR            := v_lat_dir;
          v_stl_rec.STL_LONG_DEGREES       := v_long_degrees;
          v_stl_rec.STL_LONG_MINUTES       := v_long_minutes;
          v_stl_rec.STL_LONG_DEC_MINUTES   := v_long_dec_min;
          v_stl_rec.STL_LONG_SECONDS       := v_long_seconds;
          v_stl_rec.STL_LONG_DIR           := v_long_dir;
          v_stl_rec.STL_LAT_LONG_MSR_DATE  := v_lat_long_date;
          v_stl_rec.STL_LATLONG_ACCURCY    := NULL;
          v_stl_rec.STL_LATLONG_ACCURCY_UNIT := NULL;
          v_stl_rec.STL_GPS_SCALE          := v_gps_scale;
          v_stl_rec.STL_TSMGEOPA_IS_NUMBER := NULL;
          v_stl_rec.STL_TSMGEOPA_ORG_ID    := NULL;
          v_stl_rec.STL_TSMFHU_IS_NUMBER   := NULL;
          v_stl_rec.STL_TSMFHU_ORG_ID      := NULL;
          v_stl_rec.STL_TSMRRR_IS_NUMBER   := NULL;
          v_stl_rec.STL_TSMRRR_ORG_ID      := NULL;
          v_stl_rec.STL_GEOPOS_METHOD      := v_method;
          v_stl_rec.STL_GEOPOS_DATUM       := v_datum;
          v_stl_rec.STL_ELEVATION_MSR      := v_elevation_msr;
          v_stl_rec.STL_ELEVATION_UNIT := v_elevation_unit;
          v_stl_rec.STL_ELEV_METHOD      := v_elevation_method;
          v_stl_rec.STL_ELEV_DATUM       := v_elevation_datum;
          v_stl_rec.STL_ELEV_MSR_DATE      := v_elevation_date;
          v_stl_rec.STL_TSMNAL_CD          := NULL;
          v_stl_rec.STL_TSMNAL_STATE       := NULL;
          v_stl_rec.STL_NRCS               := NULL;

        END IF;

        v_gps_scale := NULL;


        IF v_type1 = 'Ocean' AND v_soc_rec.soc_seq IS NOT NULL THEN
            v_success := create_ocean(v_soc_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
              SET sta_status = 'N'
              WHERE sta_seq = v_seq;
              Sim2_Import_Pkg.write_error(
                v_import_log_seq, 
                'Line ' || v_line_cnt || ': Failed to create Ocean in sim_ocean. Station will not be exported.',
                v_seq
              );
            END IF;
        ELSIF v_type1 = 'Estuary' AND v_estuary_rec.sel_seq IS NOT NULL THEN
            v_success := create_estuary(v_estuary_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
              SET sta_status = 'N'
              WHERE sta_seq = v_seq;
              Sim2_Import_Pkg.write_error(
                v_import_log_seq, 
                'Line ' || v_line_cnt || 'Failed to create Estuary in sim_estuary_loc. Station will not be exported.',
                v_seq
              );
            END IF;
        ELSIF v_type1 = 'Great Lake' AND v_lake_rec.SGL_SEQ IS NOT NULL THEN
            v_success := create_lake(v_lake_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
              SET sta_status = 'N'
              WHERE sta_seq = v_seq;
              Sim2_Import_Pkg.write_error(
                v_import_log_seq, 
                'Line ' || v_line_cnt || ': Failed to create Great Lake in sim_great_lake. Station will not be exported.',
                v_seq
              );
            END IF;
        ELSIF v_type1 = 'Well' AND v_well_rec.SSW_SEQ IS NOT NULL THEN
            v_success := create_well(v_well_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
              SET sta_status = 'N'
              WHERE sta_seq = v_seq;
              Sim2_Import_Pkg.write_error(
                v_import_log_seq, 
                'Line ' || v_line_cnt || ': Failed to create Well in sim_station_wells. Station will not be exported.',
                v_seq
              );
            ELSE
              v_success := create_station_location(v_stl_rec);
            END IF;
        END IF;
      EXCEPTION
        WHEN OTHERS THEN
          v_import_desc := SQLERRM||' - writing a station. Line: '||v_line_cnt;
          Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,v_seq);
      END;

      /* update import lines table with station sequence */
      UPDATE SIM_IMP_LINES SET SIPL_STA_SEQ = v_seq
        WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;
    END LOOP; /* end of text loop */

    OPEN c_comp_count(v_import_log_seq);
      FETCH c_comp_count INTO v_complete_cnt;
    CLOSE c_comp_count;
    v_error_cnt := v_line_cnt - v_complete_cnt;
    /* update counts in the import log */
    UPDATE SIM_IMPORT_LOG
       SET SIL_ROWS_IMPORTED = v_line_cnt,
           SIL_ROWS_ERRORS = v_error_cnt,
           SIL_ROWS_READY = v_complete_cnt
       WHERE SIL_SEQ = v_import_log_seq;
    COMMIT;
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_complete_cnt)||' stations loaded',null);
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq ,'Finished at '||TO_CHAR(SYSDATE,'HH24:MI'),null);
  END;

EXCEPTION
   WHEN e_no_delimiter THEN
     RAISE_APPLICATION_ERROR(-20102,'Error getting delimiter from import configuration.');
    WHEN e_bad_org THEN
     RAISE_APPLICATION_ERROR(-20109,'Unable to find the specified org id in local version of STORET.');
   WHEN e_no_cfg THEN
     RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');
   WHEN e_bad_columns THEN
     RAISE_APPLICATION_ERROR(-20197,'The import file contains the wrong number of columns on line: '||v_line_cnt);
   WHEN OTHERS THEN
       Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR(SQLERRM||' - in sim2_import_station_pkg.import_station. Line: '||TO_CHAR(v_line_cnt),1,240),v_seq);
       Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_complete_cnt)||' stations loaded',null);
       Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||TO_CHAR(SYSDATE,'HH24:MI'),null);
END;

/******************************************************************************************************/
END Sim2_Import_Station_Pkg;
/
