CREATE OR REPLACE PACKAGE sim_export_location_pkg IS
  /*******************************************************************************************
  Author  : KHENAS - taken from old export package
  Created : 11/16/00 4:04:26 PM
  Purpose : export station data from SIM to STORET

  Modified 7/16/02 pwh - Added the v_userid_code variable, set it to 'SIM122  ' 
  and inserted this variable into the D_USERID_CODE column in the STORET tables.

  Modified from Well import on 06/10/02 by SF
  **********************************************************************************************/

  -- Public variable declarations
  v_export_log_seq      NUMBER(10);
  v_success             NUMBER(1);
  
  -- Public function and procedure declarations
  PROCEDURE migrate_locations(p_imp_log_seq IN NUMBER, 
                              p_org_id IN VARCHAR2, 
                              p_success IN OUT NUMBER);
  
  PROCEDURE WRITE_ERROR (p_export_seq IN NUMBER, 
                         p_desc IN VARCHAR2);

END sim_export_location_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_export_location_pkg IS

/******************************************************************************************************/
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS
  v_item_name         VARCHAR2(80);
  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_alval_seq;
BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name INTO v_item_name;
    IF c_get_alval_name%NOTFOUND THEN
      v_item_name := NULL;
      write_error(v_export_log_seq, 'Could not find allowable value with sequence: '||p_alval_seq);
    END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
   write_error(v_export_log_seq, SQLERRM||' - getting allowable value sequence.');
   v_success := 0;
  RETURN NULL;
END get_allowable_value;
/******************************************************************************************************/
FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmalp_is_nbr     NUMBER(8);
BEGIN
  v_tsmalp_is_nbr := sim2_export_pkg.get_eisn_number('TSMALP',p_org_id);

  RETURN v_tsmalp_is_nbr ;
EXCEPTION
  WHEN OTHERS THEN
    WRITE_ERROR(v_export_log_seq, SQLERRM||' in get next tsmalp is nbr');
    v_success := 0;
END get_next_tsmalp_is_nbr;

PROCEDURE WRITE_ERROR (p_export_seq IN NUMBER, p_desc IN VARCHAR2) IS
  BEGIN
     INSERT INTO SIM_EXPORT_DETAILS (SED_SEQ , SED_SEL_SEQ, SED_DESC )
       VALUES (SIM_EXPORT_DETAILS_SEQ.NEXTVAL, p_export_seq, p_desc);
     COMMIT;
END WRITE_ERROR;
/******************************************************************************************************/
PROCEDURE migrate_locations(p_imp_log_seq IN NUMBER, p_org_id IN VARCHAR2, p_success IN OUT NUMBER) IS

   v_export_log_seq    NUMBER(10);
   v_tsmalp_is_nbr     NUMBER(8);
   v_location_cnt      NUMBER(10) := 0;
   v_location          SIM_STATION_LOCATIONS%ROWTYPE;

   e_no_location       EXCEPTION;

   CURSOR c_station_locations IS
     SELECT *
     FROM SIM_STATION_LOCATIONS
     WHERE  STL_STATUS = 'A'
     AND STL_IMPORT_SEQ = p_imp_log_seq;

  BEGIN
    v_success := 1;

    BEGIN
    /* set up export log */

      v_export_log_seq := sim2_export_pkg.GET_EXPORT_LOG_SEQ;

      INSERT INTO sim_export_log (SEL_SEQ, SEL_EXPORTED_BY, SEL_EXPORTED_ON,
                                  SEL_FILENAME, SEL_ORG_ID, sel_export_type)
         VALUES (v_export_log_seq, USER, SYSDATE, 'Locations '||p_imp_log_seq, p_org_id, 'L');
       COMMIT;

    EXCEPTION
      WHEN OTHERS THEN
         WRITE_ERROR(v_export_log_seq, SQLERRM||' - setting up migration log record.');
         v_success := 0;
         p_success := v_success;
    END;

    WRITE_ERROR(v_export_log_seq, 'Start at: '||to_char(sysdate,'HH24:MI:SS'));

    BEGIN
      BEGIN
          FOR v_location IN c_station_locations LOOP

              IF v_location.STL_ELEVATION_MSR IS NULL THEN
                v_location.STL_ELEVATION_MSR := 0; --Default NULL Value
              END IF;

              v_tsmalp_is_nbr := get_next_tsmalp_is_nbr(RPAD(p_org_id,8));

              INSERT INTO TSMALP (
                TSMALP_IS_NUMBER,
                TSMALP_ORG_ID,
                TYPE_CODE,
                POINT_NAME,
                SEQUENCE_NUMBER,
                ELEVATION_DATUM_CD,
                ELVTN_METHOD_CD,
                ELVTN_MSR_DT,
                ELEVATION_MSR,
                ELVTN_UNT_CD,
                GEOPSTNG_DATUM_CD,
                GEOPSTNG_METHOD_CD,
                GEOPSTNG_SCALE_TXT,
                LAT_DIRECTION,
                LAT_DEGREE_MSR,
                GPS_LAT_DEGREE_MSR,
                GPS_LAT_MINUTE_MSR,
                LAT_MINUTE_MSR,
                LAT_SECOND_MSR,
                LONG_DIRECTION,
                LONG_DEGREE_MSR,
                GPS_LONG_DEG_MSR,
                GPS_LONG_MIN_MSR,
                LONG_MINUTE_MSR,
                LONG_SECOND_MSR,
                LAT_LONG_MSR_DT,
                HORIZ_ACCURACY_MSR,
                HORIZ_ACCUR_UNIT,
                VERT_ACCURACY_MSR,
                VERT_ACCURACY_UNIT,
                RF3_RIVER_REACH_CD,
                RF1_MILEAGE,
                ON_RIVER_REACH_IND,
                NRCS_WTRSD_ID_NUM,
                D_USERID_CODE,
                D_LAST_UPDT_TS,
                TSMSTATN_ORG_ID,
                TSMSTATN_IS_NUMBER,
                TSMRRR_IS_NUMBER,
                TSMRRR_ORG_ID,
                TSMFHU_IS_NUMBER,
                TSMFHU_ORG_ID,
                TSMSTATN0IS_NUMBER,
                TSMSTATN0ORG_ID,
                TSMPIPE_IS_NUMBER,
                TSMPIPE_ORG_ID,
                TSMWELL_IS_NUMBER,
                TSMWELL_ORG_ID,
                TSMSTATN1ORG_ID,
                TSMSTATN1IS_NUMBER,
                TSMGEOPA_IS_NUMBER,
                TSMGEOPA_ORG_ID,
                TSMGEOPA0IS_NUMBER,
                TSMGEOPA0ORG_ID,
                TSMNAL_CD,
                TSMNAL_STATE,
                TSMNAL_ORG_ID,
                LAT_DEC_DEG_MSR,
                LONG_DEC_DEG_MSR
                )
              VALUES (
                     v_tsmalp_is_nbr                            /* TSMALP_IS_NUMBER */,
                     RPAD(p_org_id,8)                           /* TSMALP_ORG_ID */,
                     v_location.stl_type_code                   /* TYPE_CODE */,
                     v_location.STL_POINT_NAME                  /*POINT_NAME */,
                     v_location.STL_SEQUENCE_NUMBER             /* SEQUENCE_NUMBER */,
                     v_location.stl_elev_datum                  /* ELEVATION_DATUM_CD */,
                     v_location.stl_elev_method                 /* ELVTN_METHOD_CD */,
                     v_location.STL_ELEV_MSR_DATE               /* ELVTN_MSR_DATE */,
                     TO_NUMBER(v_location.STL_ELEVATION_MSR)    /* ELEVATION_MSR */,
                     v_location.stl_elevation_unit              /* ELVTN_UNT_CD */,
                     v_location.stl_geopos_datum                /* GEOPSTNG_DATUM_CD */,
                     v_location.stl_geopos_method               /* GEOPSTNG_METHOD_CD */,
                     v_location.STL_GPS_SCALE                   /* GEOPSTNG_SCALE_TXT */,
                     v_location.STL_LAT_DIR                     /* LAT_DIRECTION */,
                     v_location.STL_LAT_DEGREES                 /* LAT_DEGREE_MSR */,
                     v_location.STL_LAT_DEGREES                 /* GPS_LAT_DEGREE_MSR */,
                     v_location.STL_LAT_DEC_MINUTES             /* GPS_LAT_MINUTE_MSR */,
                     v_location.STL_LAT_MINUTES                 /* LAT_MINUTE_MSR */,
                     v_location.STL_LAT_SECONDS                 /* LAT_SECOND_MSR */,
                     v_location.STL_LONG_DIR                    /* LONG_DIRECTION */,
                     v_location.STL_LONG_DEGREES                /* LONG_DEGREE_MSR */,
                     v_location.STL_LONG_DEGREES                /* GPS_LONG_DEG_MSR */,
                     v_location.STL_LONG_DEC_MINUTES            /* GPS_LONG_MIN_MSR */,
                     v_location.STL_LONG_MINUTES                /* LONG_MINUTE_MSR */,
                     v_location.STL_LONG_SECONDS                /* LONG_SECOND_MSR */,
                     v_location.STL_LAT_LONG_MSR_DATE           /* LAT_LONG_MSR_DT */,
                     NULL                                       /* LATLONG_ACCRCY_MSR */,
                     NULL                                       /* LATLONG_ACCRCY_UNT */,
                     NULL,                                      /* VERT ACCURACY */
                     NULL,                                      /* VERT ACCURACY */
                     NULL,                                      /* RF3_RIVER_REACH_CD */
                     0                                          /* RF1_MILEAGE */,
                     'N'                                        /* ON_RIVER_REACH_IND */,
                     NULL                                       /* NRCS_WTRSD_ID_NUM */,
                     sim2_export_pkg.v_pkg_ownerid,
                     SYSDATE                                    /* D_LAST_UPDT_TS */,
                     NULL                                       /* TSMSTATN_ORG_ID */,
                     NULL                                       /* TSMSTATN_IS_NUMBER */,
                     NULL                                       /* TSMRRR_IS_NUMBER */,
                     NULL                                       /* TSMRRR_ORG_ID */,
                     NULL                                       /* TSMFHU_IS_NUMBER */,
                     NULL                                       /* TSMFHU_ORG_ID */,
                     v_location.stl_tsmstatn0is_number          /* TSMSTATN0IS_NUMBER */,
                     RPAD(p_org_id,8)                           /* TSMSTATN0ORG_ID */,
                     NULL                                       /* TSMPIPE_IS_NUMBER */,
                     NULL                                       /* TSMPIPE_ORG_ID */,
                     NULL                                       /* TSMWELL_IS_NUMBER */,
                     NULL                                       /* TSMWELL_ORG_ID */,
                     NULL                                       /* TSMSTATN1ORG_ID */,
                     NULL                                       /* TSMSTATN1IS_NUMBER */,
                     NULL                                       /* TSMGEOPA_IS_NUMBER */,
                     NULL                                       /* TSMGEOPA_ORG_ID */,
                     NULL,                                      /* TSMGEOPA0IS_NUMBER */
                     NULL,                                      /* TSMGEOPA0ORG_ID */
                     NULL                                       /*TSMNAL_CD */,
                     NULL                                       /*TSMNAL_STATE */,
                     NULL                                       /*TSMNAL_ORG_ID */,
                     v_location.STL_LAT_DEC_DEGREES             /* LAT_DEC_DEGREE_MSR */,
                     v_location.STL_LONG_DEC_DEGREES            /* LONG_DEC_DEGREE_MSR */
                    );

              UPDATE SIM_STATION_LOCATIONS
                SET STL_IS_NUMBER = v_tsmalp_is_nbr, STL_ORG_ID = p_org_id,
                STL_EXPORT_SEQ = v_export_log_seq, STL_STATUS = 'E'
                WHERE STL_SEQ = v_location.stl_seq;
              COMMIT;
              v_location_cnt := v_location_cnt + 1;
          END LOOP;

      EXCEPTION
          WHEN e_no_location THEN
            WRITE_ERROR(v_export_log_seq, 'Location record not found');
            v_success := 0;
            p_success := v_success;

          WHEN OTHERS THEN
            WRITE_ERROR(v_export_log_seq, SQLERRM ||' migrating location');
            v_success := 0;
            p_success := v_success;

        END;

    EXCEPTION
      WHEN OTHERS THEN
         WRITE_ERROR(v_export_log_seq, SQLERRM ||'migrating location');
         v_success := 0;
         p_success := v_success;
    END;

    /* update export log with count - added SNG 4/18/01*/
    UPDATE SIM_EXPORT_LOG SET SEL_ACTIVITY_CNT = v_location_cnt
      WHERE SEL_SEQ = v_export_log_seq;

      /* update the import log with export info
      SNG 4/16/01 - added DECREMENT of sil_rows_ready */
    UPDATE SIM_IMPORT_LOG SET
      SIL_EXPORTED_ON = SYSDATE,
      SIL_EXPORTED_BY = USER,
      SIL_ROWS_READY = SIL_ROWS_READY - v_location_cnt
      WHERE SIL_SEQ = P_IMP_LOG_SEQ;

    /* all done! */
    WRITE_ERROR(v_export_log_seq, 'Migrated '||v_location_cnt||' locations.');
    WRITE_ERROR(v_export_log_seq, 'Finish at: '||to_char(sysdate,'HH24:MI:SS'));
    COMMIT;
    p_success := v_success;

  EXCEPTION
    WHEN OTHERS THEN
      WRITE_ERROR(v_export_log_seq, SQLERRM ||' in migrate locations');
      v_success := 0;
      p_success := v_success;

  END migrate_locations;
/******************************************************************************************************/
END sim_export_location_pkg;
/
