CREATE OR REPLACE PACKAGE sim_event_log_pkg IS

/*--------------------------------------------------------------------------------------
  SIM Event Logging
  --------------------------------------------------------------------------------------*/

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  /* Cursor to count the records in event details table based on the reference seq and
    the event details type*/
  CURSOR c_event_details_cnt
  (
    p_sevl_ref_seq   sim_event_log.sevl_ref_seq%TYPE, 
    p_seldt_seq  sim_event_log_details_type.seldt_seq%TYPE := NULL
  ) IS
    SELECT COUNT(1)
      FROM sim_event_log_details, sim_event_log
     WHERE seld_sevl_seq = sevl_seq
       AND sevl_ref_seq = p_sevl_ref_seq
       AND seld_seldt_seq = decode(p_seldt_seq,null,seld_seldt_seq,p_seldt_seq);
       
  /* Cursor to return the count of event details excluding type Warnings*/
  CURSOR c_event_dtl_cnt_ex_warnings
  (
   p_sevl_ref_seq   sim_event_log.sevl_ref_seq%TYPE
  ) IS
    SELECT COUNT(1)
      FROM sim_event_log_details, sim_event_log
     WHERE seld_sevl_seq = sevl_seq
       AND sevl_ref_seq = p_sevl_ref_seq
       AND seld_seldt_seq != sim_lib_pkg.c_seldt_warning;
      
  /*Cursor to get the message from the event details table for a given reference seq and 
    event type*/ 
  CURSOR c_event_details(p_sevl_ref_seq sim_event_log.sevl_ref_seq%TYPE,
                       p_selt_seq    sim_event_log_type.selt_seq%TYPE) IS
    SELECT seld_message
      FROM sim_event_log_details, sim_event_log
     WHERE seld_sevl_seq = sevl_seq
       AND sevl_ref_seq = p_sevl_ref_seq
       AND sevl_selt_seq = p_selt_seq;    
     

/****************************************************************************************
*                             PUBLIC PROCEDURES AND FUNCTIONS                          *
****************************************************************************************/
PROCEDURE write_event (
  p_selt_seq        IN sim_event_log.sevl_selt_seq%TYPE,
  p_ref_seq         IN sim_event_log.sevl_ref_seq%TYPE := NULL,
  p_tsmorgan_org_id IN sim_event_log.sevl_tsmorgan_org_id%TYPE := NULL,
  p_sid_seq         IN sim_event_log.sevl_sid_seq%TYPE := NULL,
  p_set_end_time    IN NUMBER := 0);

PROCEDURE write_event_detail (
  p_seldt_seq     IN sim_event_log_details.seld_seldt_seq%TYPE,
  p_seld_context  IN sim_event_log_details.seld_context%TYPE := NULL,
  p_seld_message  IN sim_event_log_details.seld_message%TYPE);

PROCEDURE log_error (
  p_seld_message  IN sim_event_log_details.seld_message%TYPE,
  p_seld_context  IN sim_event_log_details.seld_context%TYPE := NULL);

PROCEDURE update_stop_time;

FUNCTION get_sevl_seq 
  RETURN NUMBER;

FUNCTION get_seld_seq 
  RETURN NUMBER;

PROCEDURE set_sevl_seq (
  p_sevl_seq IN NUMBER);
  
PROCEDURE set_sevl_ref_seq ( 
  p_sevl_seq     IN NUMBER,
  p_sevl_ref_seq IN NUMBER);

FUNCTION insert_event_log_lines (
  p_sell_line_text IN sim_event_log_lines.sell_line_text%TYPE,
  p_sell_line_no   IN sim_event_log_lines.sell_line_no%TYPE,
  p_sqlcode        IN OUT VARCHAR2,
  p_sqlerrm        IN OUT VARCHAR2) 
  RETURN NUMBER;
  
FUNCTION clear_event_log(
  p_no_days IN NUMBER) 
  RETURN NUMBER;

END sim_event_log_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_event_log_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/

/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  v_sevl_seq NUMBER := NULL;
  v_seld_seq NUMBER := NULL;

/****************************************************************************************
 *                                  PROCEDURES AND FUNCTIONS                            *
 ****************************************************************************************/

/****************************************************************************************
  Create a new event                                                                                      
 ****************************************************************************************/
PROCEDURE write_event (
  p_selt_seq        IN sim_event_log.sevl_selt_seq%TYPE,
  p_ref_seq         IN sim_event_log.sevl_ref_seq%TYPE := NULL,
  p_tsmorgan_org_id IN sim_event_log.sevl_tsmorgan_org_id%TYPE := NULL,
  p_sid_seq         IN sim_event_log.sevl_sid_seq%TYPE := NULL,
  p_set_end_time    IN NUMBER := 0) IS    --Set to 1 if you want to set end time

  PRAGMA AUTONOMOUS_TRANSACTION;
  
BEGIN
  SELECT sim_event_log_seq.NEXTVAL
    INTO v_sevl_seq
    FROM dual;
  INSERT INTO sim_event_log (
      sevl_seq,
      sevl_selt_seq,
      sevl_tsmorgan_org_id,
      sevl_ref_seq,
      sevl_sid_seq,
      sevl_user_id,
      sevl_start)
    VALUES (
      v_sevl_seq,
      p_selt_seq,
      p_tsmorgan_org_id,
      p_ref_seq,
      p_sid_seq,
      USER,
      SYSDATE);
  COMMIT;
  IF p_set_end_time = 1 THEN
    update_stop_time;
  END IF;
END write_event;

/****************************************************************************************
  Insert a new event log detail row (related to the current event)
 ****************************************************************************************/
PROCEDURE write_event_detail (
  p_seldt_seq     IN sim_event_log_details.seld_seldt_seq%TYPE,
  p_seld_context  IN sim_event_log_details.seld_context%TYPE := NULL,
  p_seld_message  IN sim_event_log_details.seld_message%TYPE) IS

  PRAGMA AUTONOMOUS_TRANSACTION;
    
BEGIN
  SELECT sim_event_log_details_seq.NEXTVAL
    INTO v_seld_seq
    FROM dual;  
  INSERT INTO sim_event_log_details (
      seld_seq,
      seld_sevl_seq,
      seld_seldt_seq,
      seld_context,
      seld_message)
    VALUES (
      v_seld_seq,
      v_sevl_seq,
      p_seldt_seq,
      p_seld_context,
      substr(p_seld_message,1,4000));
  COMMIT;
END write_event_detail;

/****************************************************************************************
  LOG AN APPLICATION ERROR
  If there is a current event, use it.
  Otherwise, create a new event for this error.
 ****************************************************************************************/
PROCEDURE log_error (
  p_seld_message  IN sim_event_log_details.seld_message%TYPE,
  p_seld_context  IN sim_event_log_details.seld_context%TYPE := NULL) IS

BEGIN
  IF v_sevl_seq IS NOT NULL THEN 
    -- event is still active so continue to use it
    write_event_detail(sim_lib_pkg.c_seldt_error, p_seld_context, p_seld_message);
  ELSE
    -- no current event so create a new on just for this error
    write_event(sim_lib_pkg.c_selt_app_error);
    write_event_detail(sim_lib_pkg.c_seldt_error, p_seld_context, p_seld_message);
    update_stop_time;
  END IF;
END log_error;

/****************************************************************************************
  End the current event
 ****************************************************************************************/
PROCEDURE update_stop_time IS

  PRAGMA AUTONOMOUS_TRANSACTION;
  
BEGIN
  IF v_sevl_seq IS NOT NULL THEN 
    UPDATE sim_event_log
      SET sevl_stop = SYSDATE
      WHERE sevl_seq = v_sevl_seq;
    COMMIT;
    set_sevl_seq(NULL);
  END IF;
END update_stop_time;

/****************************************************************************************
  Get current event log seq
 ****************************************************************************************/
FUNCTION get_sevl_seq RETURN NUMBER IS

BEGIN
  RETURN v_sevl_seq;
END get_sevl_seq;

/****************************************************************************************
  Get last event log detail seq
 ****************************************************************************************/
FUNCTION get_seld_seq 
  RETURN NUMBER IS

BEGIN
  RETURN v_seld_seq;
END get_seld_seq;

/****************************************************************************************
  Set current event log seq
  Is this really used?
  It doesn't seem like it should, at least not outside this package
 ****************************************************************************************/
PROCEDURE set_sevl_seq ( 
  p_sevl_seq IN NUMBER) IS

BEGIN
  v_sevl_seq := p_sevl_seq;
END set_sevl_seq;

/****************************************************************************************
  Update the SEVL_REF_SEQ.  In certain situations the event will be created before
  the SEVL_REF_SEQ is known, and we need to go back and set this.
 ****************************************************************************************/
PROCEDURE set_sevl_ref_seq ( 
  p_sevl_seq     IN NUMBER,
  p_sevl_ref_seq IN NUMBER) IS
  
  PRAGMA AUTONOMOUS_TRANSACTION;
  
BEGIN
  UPDATE sim_event_log
    SET sevl_ref_seq = p_sevl_ref_seq
    WHERE sevl_seq = p_sevl_seq;
  COMMIT;
END set_sevl_ref_seq;

/****************************************************************************************
  Add a new event log line (related to the current event)
 ****************************************************************************************/
FUNCTION insert_event_log_lines (
  p_sell_line_text IN     sim_event_log_lines.sell_line_text%TYPE,
  p_sell_line_no   IN     sim_event_log_lines.sell_line_no%TYPE,
  p_sqlcode        IN OUT VARCHAR2,
  p_sqlerrm        IN OUT VARCHAR2) 
  
  RETURN NUMBER IS

  PRAGMA AUTONOMOUS_TRANSACTION;
  
  v_sell_seq        NUMBER;
  v_sell_line_text  sim_event_log_lines.sell_line_text%TYPE;

BEGIN
  -- remove trailing CRLF from lines 
  -- (a quirk when reading files from linux via a Samba mount to a Windows box)
  v_sell_line_text := REPLACE(REPLACE(TRIM(p_sell_line_text), chr(13), ''), chr(10), '');
  SELECT sim_event_log_lines_seq.NEXTVAL
    INTO v_sell_seq
    FROM dual;
  INSERT INTO sim_event_log_lines (
      sell_seq,
      sell_sevl_seq,
      sell_line_text,
      sell_line_no)
    VALUES (
      v_sell_seq,
      v_sevl_seq,
      v_sell_line_text,
      p_sell_line_no);
  COMMIT;
  RETURN v_sell_seq;

EXCEPTION
  WHEN OTHERS THEN
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN -1; -- Return error code

END insert_event_log_lines;

/****************************************************************************************
  Clear SIM events older than a specified number of days (90 is the minimum # of days)
 ****************************************************************************************/
FUNCTION clear_event_log (
  p_no_days IN NUMBER) 
  
  RETURN NUMBER IS
  
BEGIN
  IF p_no_days < 90 THEN
    RETURN 0; --unsuccessful
  ELSE
    DELETE sim_event_log sevl
     WHERE trunc(sevl_start) < trunc(SYSDATE - p_no_days)
       AND (sevl.sevl_tsmorgan_org_id IN
            (SELECT tsmorgan.org_id
               FROM tsmorgan,
                    tsmuoa
              WHERE tsmuoa.tsmuser_userid = rpad(USER, 8)
                AND tsmuoa.tsmorgan_is_number = tsmorgan.tsmorgan_is_number) OR
            sevl.sevl_tsmorgan_org_id IS NULL);
    COMMIT;
    RETURN 1; --successful return
  END IF;
END clear_event_log;
  
END sim_event_log_pkg;
/
