CREATE OR REPLACE PACKAGE sim_export_location_pkg IS
  
/*--------------------------------------------------------------------------------------
  Export station data from SIM to STORET
  --------------------------------------------------------------------------------------*/
 
/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/


/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_export_log_seq NUMBER(10);
  v_success        NUMBER(1);

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ****************************************************************************************/

  PROCEDURE migrate_locations
  (
    p_imp_log_seq IN NUMBER,
    p_org_id      IN VARCHAR2,
    p_success     IN OUT NUMBER
  );

END sim_export_location_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_export_location_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/
  c_migrate_true    CONSTANT BOOLEAN := TRUE;
  c_migrate_false   CONSTANT BOOLEAN := FALSE;
  c_new_event_no    CONSTANT VARCHAR2(1) := 'N';
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  
  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS
  v_item_name VARCHAR2(80);
  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
     
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                       'Could not find allowable value with sequence: ' ||
                                        p_alval_seq);
   
        
  END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - getting allowable value sequence.');
     
          
    v_success := 0;
    RETURN NULL;
END get_allowable_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmalp_is_nbr NUMBER(8);

BEGIN
  v_tsmalp_is_nbr := sim_export_pkg.get_eisn_number('TSMALP', p_org_id);

  RETURN v_tsmalp_is_nbr;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' in get next tsmalp is nbr');
     
          
    v_success := 0;
END get_next_tsmalp_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE migrate_locations
(
  p_imp_log_seq IN NUMBER,
  p_org_id      IN VARCHAR2,
  p_success     IN OUT NUMBER
) IS

  v_export_log_seq NUMBER(10);
  v_tsmalp_is_nbr  NUMBER(8);
  v_location_cnt   NUMBER(10) := 0;
  v_location       sim_station_locations%ROWTYPE;
  v_location_rec   sim_station_locations%ROWTYPE;
  v_sid_seq        NUMBER(10);  
  v_rows_ready     NUMBER(10); --sjj 10/24/05
  v_action_code    VARCHAR2(1); 
  v_filename       VARCHAR2(265);
  
  e_no_location EXCEPTION;

  CURSOR c_station_locations IS
    SELECT *
      FROM sim_station_locations
     WHERE stl_status = 'A'
       AND stl_import_seq = p_imp_log_seq;

BEGIN
  v_success := 1;
  BEGIN
    /* set up export log */
    v_export_log_seq := p_imp_log_seq;
    SELECT substr(sil_import_file,instr(sil_import_file, '\', -1) +1)
      INTO v_filename
      FROM sim_import_log
      WHERE sil_seq = v_export_log_seq;
    INSERT INTO sim_export_log
      (sel_seq,
       sel_exported_by,
       sel_exported_on,
       sel_filename,
       sel_org_id,
       sel_export_type)
    VALUES
      (v_export_log_seq,
       USER,
       SYSDATE,
       v_filename,
       p_org_id,
       'L');
    COMMIT;
  
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           SQLERRM || ' - setting up migration log record.');
      v_success := 0;
      p_success := v_success;
  END;
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = (SELECT sil_sicc_seq 
                       FROM sim_import_log 
                      WHERE sil_seq =  p_imp_log_seq);
  sim_event_log_pkg.write_event(4,v_export_log_seq,p_org_id,v_sid_seq,NULL);
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Start: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  SELECT sid_action_code
    INTO v_action_code
    FROM sim_imp_defs
    WHERE sid_seq = v_sid_seq;
            
  BEGIN
    BEGIN
      FOR v_location IN c_station_locations LOOP
      
        IF v_location.stl_elevation_msr IS NULL THEN
          v_location.stl_elevation_msr := 0; --Default NULL Value
        END IF;
        
        IF v_action_code = 'U' THEN
          BEGIN 
            SELECT *
              INTO v_location_rec
              FROM sim_station_locations
              WHERE stl_seq = v_location.stl_seq
              AND stl_sequence_number = v_location.stl_sequence_number
              AND stl_import_seq = p_imp_log_seq;
               
            SELECT tsmalp_is_number
              INTO v_tsmalp_is_nbr
              FROM tsmalp
              WHERE tsmstatn0is_number = v_location_rec.stl_tsmstatn0is_number
              AND tsmstatn0org_id = rpad(p_org_id, 8)
              AND sequence_number = v_location_rec.stl_sequence_number;
                        
            UPDATE tsmalp
              SET
                sequence_number = v_location_rec.stl_sequence_number,
                point_name = v_location_rec.stl_point_name,
                elevation_datum_cd = v_location_rec.stl_elev_datum,
                elvtn_method_cd = v_location_rec.stl_elev_method,
                elvtn_msr_dt = v_location_rec.stl_elev_msr_date,
                elevation_msr = v_location_rec.stl_elevation_msr,
                elvtn_unt_cd = v_location_rec.stl_elevation_unit,
                geopstng_datum_cd = v_location_rec.stl_geopos_datum,
                geopstng_method_cd = v_location_rec.stl_geopos_method,
                geopstng_scale_txt = v_location_rec.stl_gps_scale,
                lat_direction = v_location_rec.stl_lat_dir,
                lat_degree_msr = v_location_rec.stl_lat_degrees,
                gps_lat_degree_msr = v_location_rec.stl_lat_degrees,
                gps_lat_minute_msr = v_location_rec.stl_lat_dec_minutes,
                gps_long_deg_msr = v_location_rec.stl_long_degrees,
                gps_long_min_msr = v_location_rec.stl_long_dec_minutes,
                lat_minute_msr = v_location_rec.stl_lat_minutes,
                lat_second_msr = v_location_rec.stl_lat_seconds,
                long_direction = v_location_rec.stl_long_dir,
                long_degree_msr = v_location_rec.stl_long_degrees,
                long_minute_msr = v_location_rec.stl_long_minutes,
                long_second_msr = v_location_rec.stl_long_seconds,
                lat_long_msr_dt = v_location_rec.stl_lat_long_msr_date,
                lat_dec_deg_msr = v_location_rec.stl_lat_dec_degrees, 
                long_dec_deg_msr = v_location_rec.stl_long_dec_degrees,
                nrcs_wtrsd_id_num = v_location_rec.stl_nrcs,
                d_userid_code = sim_export_pkg.v_pkg_ownerid,
                d_last_updt_ts = SYSDATE,
                tsmfhu_is_number = v_location_rec.stl_tsmfhu_is_number,
                tsmfhu_org_id = v_location_rec.stl_tsmfhu_org_id,
                tsmstatn0is_number = v_location_rec.stl_tsmstatn0is_number,
                tsmstatn0org_id = v_location_rec.stl_tsmstatn0org_id,
                tsmwell_is_number = v_location_rec.stl_tsmwell_is_number,
                tsmwell_org_id = v_location_rec.stl_tsmwell_org_id,
                tsmgeopa_is_number = v_location_rec.stl_tsmgeopa_is_number,
                tsmgeopa_org_id = v_location_rec.stl_tsmgeopa_org_id,
                tsmnal_cd = v_location_rec.stl_tsmnal_cd,
                tsmnal_state = v_location_rec.stl_tsmnal_state
              WHERE tsmalp_is_number = v_tsmalp_is_nbr
              AND tsmalp_org_id = rpad(p_org_id, 8)
              AND type_code = v_location_rec.stl_type_code; 
          
          v_location_cnt := v_location_cnt + 1;
        EXCEPTION
          WHEN no_data_found THEN
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                 SQLERRM || ' - in updating Location data.');
            v_success := 0;
            p_success := v_success;
                                                             
          WHEN OTHERS THEN
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM || ' In update stations.');
            v_success := 0;
            p_success := v_success;                                               
        END;
        
        ELSE
        
          v_tsmalp_is_nbr := get_next_tsmalp_is_nbr(rpad(p_org_id, 8));
          INSERT INTO tsmalp
            (tsmalp_is_number,
             tsmalp_org_id,
             type_code,
             point_name,
             sequence_number,
             elevation_datum_cd,
             elvtn_method_cd,
             elvtn_msr_dt,
             elevation_msr,
             elvtn_unt_cd,
             geopstng_datum_cd,
             geopstng_method_cd,
             geopstng_scale_txt,
             lat_direction,
             lat_degree_msr,
             gps_lat_degree_msr,
             gps_lat_minute_msr,
             lat_minute_msr,
             lat_second_msr,
             long_direction,
             long_degree_msr,
             gps_long_deg_msr,
             gps_long_min_msr,
             long_minute_msr,
             long_second_msr,
             lat_long_msr_dt,
             horiz_accuracy_msr,
             horiz_accur_unit,
             vert_accuracy_msr,
             vert_accuracy_unit,
             rf3_river_reach_cd,
             rf1_mileage,
             on_river_reach_ind,
             nrcs_wtrsd_id_num,
             d_userid_code,
             d_last_updt_ts,
             tsmstatn_org_id,
             tsmstatn_is_number,
             tsmrrr_is_number,
             tsmrrr_org_id,
             tsmfhu_is_number,
             tsmfhu_org_id,
             tsmstatn0is_number,
             tsmstatn0org_id,
             tsmpipe_is_number,
             tsmpipe_org_id,
             tsmwell_is_number,
             tsmwell_org_id,
             tsmstatn1org_id,
             tsmstatn1is_number,
             tsmgeopa_is_number,
             tsmgeopa_org_id,
             tsmgeopa0is_number,
             tsmgeopa0org_id,
             tsmnal_cd,
             tsmnal_state,
             tsmnal_org_id,
             lat_dec_deg_msr,
             long_dec_deg_msr)
          VALUES
            (v_tsmalp_is_nbr /* TSMALP_IS_NUMBER */,
             rpad(p_org_id, 8) /* TSMALP_ORG_ID */,
             v_location.stl_type_code /* TYPE_CODE */,
             v_location.stl_point_name /*POINT_NAME */,
             v_location.stl_sequence_number /* SEQUENCE_NUMBER */,
             v_location.stl_elev_datum /* ELEVATION_DATUM_CD */,
             v_location.stl_elev_method /* ELVTN_METHOD_CD */,
             v_location.stl_elev_msr_date /* ELVTN_MSR_DATE */,
             to_number(v_location.stl_elevation_msr) /* ELEVATION_MSR */,
             v_location.stl_elevation_unit /* ELVTN_UNT_CD */,
             v_location.stl_geopos_datum /* GEOPSTNG_DATUM_CD */,
             v_location.stl_geopos_method /* GEOPSTNG_METHOD_CD */,
             v_location.stl_gps_scale /* GEOPSTNG_SCALE_TXT */,
             v_location.stl_lat_dir /* LAT_DIRECTION */,
             v_location.stl_lat_degrees /* LAT_DEGREE_MSR */,
             v_location.stl_lat_degrees /* GPS_LAT_DEGREE_MSR */,
             v_location.stl_lat_dec_minutes /* GPS_LAT_MINUTE_MSR */,
             v_location.stl_lat_minutes /* LAT_MINUTE_MSR */,
             v_location.stl_lat_seconds /* LAT_SECOND_MSR */,
             v_location.stl_long_dir /* LONG_DIRECTION */,
             v_location.stl_long_degrees /* LONG_DEGREE_MSR */,
             v_location.stl_long_degrees /* GPS_LONG_DEG_MSR */,
             v_location.stl_long_dec_minutes /* GPS_LONG_MIN_MSR */,
             v_location.stl_long_minutes /* LONG_MINUTE_MSR */,
             v_location.stl_long_seconds /* LONG_SECOND_MSR */,
             v_location.stl_lat_long_msr_date /* LAT_LONG_MSR_DT */,
             NULL /* LATLONG_ACCRCY_MSR */,
             NULL /* LATLONG_ACCRCY_UNT */,
             NULL, /* VERT ACCURACY */
             NULL, /* VERT ACCURACY */
             NULL, /* RF3_RIVER_REACH_CD */
             0 /* RF1_MILEAGE */,
             'N' /* ON_RIVER_REACH_IND */,
             NULL /* NRCS_WTRSD_ID_NUM */,
             sim_export_pkg.v_pkg_ownerid,
             SYSDATE /* D_LAST_UPDT_TS */,
             NULL /* TSMSTATN_ORG_ID */,
             NULL /* TSMSTATN_IS_NUMBER */,
             NULL /* TSMRRR_IS_NUMBER */,
             NULL /* TSMRRR_ORG_ID */,
             NULL /* TSMFHU_IS_NUMBER */,
             NULL /* TSMFHU_ORG_ID */,
             v_location.stl_tsmstatn0is_number /* TSMSTATN0IS_NUMBER */,
             rpad(p_org_id, 8) /* TSMSTATN0ORG_ID */,
             NULL /* TSMPIPE_IS_NUMBER */,
             NULL /* TSMPIPE_ORG_ID */,
             NULL /* TSMWELL_IS_NUMBER */,
             NULL /* TSMWELL_ORG_ID */,
             NULL /* TSMSTATN1ORG_ID */,
             NULL /* TSMSTATN1IS_NUMBER */,
             NULL /* TSMGEOPA_IS_NUMBER */,
             NULL /* TSMGEOPA_ORG_ID */,
             NULL, /* TSMGEOPA0IS_NUMBER */
             NULL, /* TSMGEOPA0ORG_ID */
             NULL /*TSMNAL_CD */,
             NULL /*TSMNAL_STATE */,
             NULL /*TSMNAL_ORG_ID */,
             v_location.stl_lat_dec_degrees /* LAT_DEC_DEGREE_MSR */,
             v_location.stl_long_dec_degrees /* LONG_DEC_DEGREE_MSR */);
        
          UPDATE sim_station_locations
             SET stl_is_number = v_tsmalp_is_nbr,
                 stl_org_id = p_org_id,
                 stl_export_seq = v_export_log_seq,
                 stl_status = 'E'
           WHERE stl_seq = v_location.stl_seq;
          COMMIT;
          v_location_cnt := v_location_cnt + 1;
        END IF;
      END LOOP;
    
    EXCEPTION
      WHEN e_no_location THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             'Location record not found');
         
              
        v_success := 0;
        p_success := v_success;
      
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM || ' migrating location');
         
              
        v_success := 0;
        p_success := v_success;
      
    END;
  
  EXCEPTION
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           SQLERRM || 'migrating location');
       
            
      v_success := 0;
      p_success := v_success;
  END;

  /* update export log with count - added SNG 4/18/01*/
  UPDATE sim_export_log
     SET sel_activity_cnt = v_location_cnt
   WHERE sel_seq = v_export_log_seq;

  /* update the import log with export info
  SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE sim_import_log
     SET sil_exported_on = SYSDATE,
         sil_exported_by = USER,
         sil_rows_ready = sil_rows_ready - v_location_cnt
   WHERE sil_seq = p_imp_log_seq;

  /* all done! */
   
  COMMIT;
  p_success := v_success;
  
  /* If the action is an Update then remove data from sim regardless of success or not*/
  IF v_action_code = 'U' THEN
    sim_import_pkg.delete_import(p_imp_log_seq, c_migrate_false, c_new_event_no);
    DELETE FROM sim_export_log
     WHERE sel_seq = v_export_log_seq;
     
  ELSE
    /* SJJ- Added following code to delete location data in SIM when successfully exported
       to Storet then a call to delete_import will delete records from sim and insert rows 
       into sim_export_storet_ids as the link to un-migrate data from storet if needed*/
    IF p_success = 1 THEN
      SELECT sil_rows_ready
        INTO v_rows_ready
        FROM sim_import_log
        WHERE sil_seq = p_imp_log_seq;
      
      IF v_rows_ready IS NULL OR v_rows_ready = 0 THEN
        sim_import_pkg.delete_import(p_imp_log_seq, c_migrate_true, c_new_event_no);
      END IF;
    ELSE
      /* If export is not successful still remove data from SIM*/
      sim_import_pkg.delete_import(p_imp_log_seq, c_migrate_true, c_new_event_no);
                                                
    END IF;
  END IF;
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Migrated ' || v_location_cnt || ' locations.');
   
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Finish at: ' || to_char(SYSDATE, 'HH24:MI:SS'));
   
  sim_event_log_pkg.update_stop_time;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' in migrate locations');
     
          
    v_success := 0;
    p_success := v_success;
  
END migrate_locations;

END sim_export_location_pkg;
/
