CREATE OR REPLACE PACKAGE sim_export_pkg IS

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_pkg_ownerid CHAR(8) := 'SIM202  ';
  
/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ****************************************************************************************/
  FUNCTION get_eisn_number
  (
    p_table_name IN VARCHAR2,
    p_org_id     IN VARCHAR2
  ) RETURN NUMBER;

  FUNCTION chk_if_result_delete
  (
    p_tsrfdact_is_number IN NUMBER,
    p_tsrfdact_org_id    IN VARCHAR2,
    p_exp_id             VARCHAR2,
    p_exp_log_seq        NUMBER,
    p_activity_id        IN VARCHAR2 DEFAULT NULL
  ) RETURN VARCHAR2;

  FUNCTION get_tsmorgan_is_number(p_org_id IN VARCHAR2) RETURN NUMBER;

  PROCEDURE clear_storet_export
  (
    p_export_log_seq    IN NUMBER,
    p_remove_by_file_yn IN VARCHAR2,
    p_success           OUT NUMBER
  );

  PROCEDURE clear_station_export
  (
    p_export_log_seq    IN NUMBER,
    p_remove_by_file_yn IN VARCHAR2,
    p_success           OUT NUMBER
  );

  FUNCTION clear_export
(
  p_export_log_seq    IN NUMBER, 
  p_remove_by_file_yn IN VARCHAR2 := 'N'
) RETURN NUMBER;

  FUNCTION clear_export_all
  (
    p_org_id           IN sim_export_log.sel_org_id%TYPE,
    p_include_projects IN BOOLEAN := FALSE
  ) RETURN NUMBER;

  FUNCTION get_file_type(p_path IN VARCHAR2) RETURN VARCHAR2;

  FUNCTION get_file_name(p_path IN VARCHAR2) RETURN VARCHAR2;
  
  FUNCTION get_next_blob_is_number(p_org_id IN VARCHAR2) RETURN NUMBER;
  
  FUNCTION insert_binary_document 
  (
    p_export_log_seq      IN NUMBER,
    p_table_name          IN VARCHAR2,
    p_userid_code         IN CHAR,
    p_file_name           IN VARCHAR2,
    p_tsmproj_is_number   IN sim_projects.spj_tsmproj_is_number%TYPE := NULL,
    p_tsmproj_org_id      IN sim_projects.spj_tsmproj_org_id%TYPE := NULL,
    p_tsrrsult_is_number  IN sim_result_class_indicator.rci_tsrrsult_is_number%TYPE := NULL,  
    p_tsrrsult_org_id     IN sim_result_class_indicator.rci_tsrrsult_org_id%TYPE := NULL,
    p_tsmstatn_is_number  IN sim_stations.sta_is_number%TYPE := NULL,
    p_tsmstatn_org_id     IN sim_stations.sta_org_id%TYPE := NULL,
    p_tsrstvst_is_number  IN sim_station_visits.ssv_tsrstvst_is_number%TYPE := NULL,
    p_tsrstvst_org_id     IN sim_station_visits.ssv_tsrstvst_org_id%TYPE := NULL,
    p_tsrfdact_is_number  IN sim_field_activities.fa_tsrfdact_is_number%TYPE := NULL,  
    p_tsrfdact_org_id     IN sim_field_activities.fa_tsrfdact_org_id%TYPE := NULL
  ) RETURN NUMBER;
    
  PROCEDURE get_clear_export_status
  (
    p_export_log_seq IN NUMBER,
    p_export_type    IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fnd_cnt        OUT NUMBER,
    p_notfnd_cnt     OUT NUMBER
  );
  
  PROCEDURE insert_into_storet_ids
  (
    p_export_log_seq    IN NUMBER,
    p_storet_table_name IN VARCHAR2,
    p_org_id            IN VARCHAR2,
    p_is_number         IN NUMBER
  );
  
  FUNCTION get_storet_ids_from_file
  (
    p_export_log_seq    IN NUMBER,
    p_event_log_seq     IN NUMBER,
    p_org_id            IN VARCHAR2,
    p_export_type       IN VARCHAR2,
    p_delimiter         IN VARCHAR2,
    p_skip_header       IN NUMBER DEFAULT 0
  ) RETURN NUMBER;
END;
/
CREATE OR REPLACE PACKAGE BODY sim_export_pkg AS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/
 
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
 
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/
 
/****************************************************************************************

 ****************************************************************************************/ 
FUNCTION get_eisn_number
(
  p_table_name IN VARCHAR2,
  p_org_id     IN VARCHAR2
) RETURN NUMBER IS

  v_is_number NUMBER(8);

  CURSOR c_get_number(p_table_name IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT is_number
      FROM tsmeisn
     WHERE tsmeisn_name = rpad(p_table_name, 8)
       AND tsmeisn_org_id = rpad(p_org_id, 8)
       FOR UPDATE; -- Locks table while being used

BEGIN
  OPEN c_get_number(p_table_name, p_org_id);
  FETCH c_get_number
    INTO v_is_number;
  IF c_get_number%NOTFOUND THEN
    v_is_number := 1;
    INSERT INTO tsmeisn
      (tsmeisn_name,
       tsmeisn_org_id,
       is_number,
       d_userid_code,
       d_last_updt_ts)
    VALUES
      (p_table_name,
       p_org_id,
       v_is_number,
       USER,
       SYSDATE);
  
    COMMIT;
  ELSE
    v_is_number := v_is_number + 1;
    UPDATE tsmeisn
       SET is_number = v_is_number
     WHERE tsmeisn_name = rpad(p_table_name, 8)
       AND tsmeisn_org_id = rpad(p_org_id, 8);
  
    COMMIT;
  END IF;
  CLOSE c_get_number;

  RETURN v_is_number;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL,
                                         SQLERRM || ' - in get eisn'); 
END get_eisn_number;

/****************************************************************************************

 ****************************************************************************************/ 
FUNCTION get_tsmorgan_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_is_nbr NUMBER(8);

  -- Table of Storet Metadata for ORGANIZATION (TSMORGAN)
  CURSOR c_tsmorgan_is_nbr(p_org_id IN VARCHAR2) IS
    SELECT tsmorgan_is_number
      FROM tsmorgan
     WHERE org_id = rpad(p_org_id, 8);

BEGIN
  OPEN c_tsmorgan_is_nbr(p_org_id);
  FETCH c_tsmorgan_is_nbr
    INTO v_is_nbr;
  CLOSE c_tsmorgan_is_nbr;
  RETURN v_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get tsmorgan');
END get_tsmorgan_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION delete_result_children
(
  p_tsrfdact_is_number IN NUMBER,
  p_tsrfdact_org_id    IN VARCHAR2,
  p_exp_log_seq        IN NUMBER,
  p_activity_id        IN VARCHAR2 DEFAULT NULL
) RETURN VARCHAR2 IS

  CURSOR c_tsrbrg_id(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2) IS
    SELECT tsrbrg_is_number,
           tsrbrg_org_id
      FROM tsrbrg
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  CURSOR c_brg_tsrrsult_id(p_tsrbrg_is_number NUMBER, p_tsrbrg_org_id VARCHAR2) IS
    SELECT tsrrsult_is_number,
           tsrrsult_org_id
      FROM tsrrsult
     WHERE tsrbrg_is_number = p_tsrbrg_is_number
       AND tsrbrg_org_id = rpad(p_tsrbrg_org_id, 8);

  CURSOR c_act_tsrrsult_id(cp_tsrfdact_is_number NUMBER, cp_tsrfdact_org_id VARCHAR2) IS
    SELECT tsrrsult_is_number,
           tsrrsult_org_id
      FROM tsrrsult
     WHERE tsrfdact_is_number = cp_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(cp_tsrfdact_org_id, 8);

BEGIN

  DELETE tsrrlra
   WHERE tsrrsult_is_number IN
         (SELECT tsrrsult_is_number
            FROM tsrrsult
           WHERE tsrfdact_is_number = p_tsrfdact_is_number)
     AND tsrrsult_org_id IN
         (SELECT tsrrsult_org_id
            FROM tsrrsult
           WHERE tsrfdact_is_number = p_tsrfdact_org_id);

  DELETE tsraal
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsreod
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrfafsa
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrfapea
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrfapra
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrfaqsa
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrsmple
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrtnod
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  DELETE tsrtod
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  -- Remove Biological records
  FOR c_tsrbrg_id_rec IN c_tsrbrg_id(p_tsrfdact_is_number, p_tsrfdact_org_id) LOOP
  
    DELETE tsrbrgi
     WHERE tsrbrg_is_number = c_tsrbrg_id_rec.tsrbrg_is_number
       AND tsrbrg_org_id = rpad(c_tsrbrg_id_rec.tsrbrg_org_id, 8);
  
    FOR c_brg_tsrrsult_id_rec IN c_brg_tsrrsult_id(c_tsrbrg_id_rec.tsrbrg_is_number, c_tsrbrg_id_rec.tsrbrg_org_id) LOOP
    
      DELETE tsmgntxt
       WHERE tsrrsult_is_number = c_brg_tsrrsult_id_rec.tsrrsult_is_number
         AND tsrrsult_org_id = rpad(c_brg_tsrrsult_id_rec.tsrrsult_org_id, 8);
    
      DELETE tsrdql
       WHERE tsrrsult_is_number = c_brg_tsrrsult_id_rec.tsrrsult_is_number
         AND tsrrsult_org_id = rpad(c_brg_tsrrsult_id_rec.tsrrsult_org_id, 8);
    
      DELETE tsrrlra
       WHERE tsrrsult_is_number = c_brg_tsrrsult_id_rec.tsrrsult_is_number
         AND tsrrsult_org_id = rpad(c_brg_tsrrsult_id_rec.tsrrsult_org_id, 8);
    
      DELETE tsrcldes
       WHERE tsrrsult_is_number = c_brg_tsrrsult_id_rec.tsrrsult_is_number
         AND tsrrsult_org_id = rpad(c_brg_tsrrsult_id_rec.tsrrsult_org_id, 8);
    
      DELETE tsrrci
       WHERE tsrrsult_is_number = c_brg_tsrrsult_id_rec.tsrrsult_is_number
         AND tsrrsult_org_id = rpad(c_brg_tsrrsult_id_rec.tsrrsult_org_id, 8);
    
      DELETE tsrrqcaf
       WHERE tsrrsult_is_number = c_brg_tsrrsult_id_rec.tsrrsult_is_number
         AND tsrrsult_org_id = rpad(c_brg_tsrrsult_id_rec.tsrrsult_org_id, 8);
    
    END LOOP; --c_brg_tsrrsult_id_rec
  
    DELETE tsrrsult
     WHERE tsrbrg_is_number = c_tsrbrg_id_rec.tsrbrg_is_number
       AND tsrbrg_org_id = rpad(c_tsrbrg_id_rec.tsrbrg_org_id, 8);
  
  END LOOP; --c_tsrbrg_id_rec

  FOR c_act_tsrrsult_id_rec IN c_act_tsrrsult_id(p_tsrfdact_is_number, p_tsrfdact_org_id) LOOP
  
    DELETE tsmgntxt
     WHERE tsrrsult_is_number = c_act_tsrrsult_id_rec.tsrrsult_is_number
       AND tsrrsult_org_id = rpad(c_act_tsrrsult_id_rec.tsrrsult_org_id, 8);
  
    DELETE tsrdql
     WHERE tsrrsult_is_number = c_act_tsrrsult_id_rec.tsrrsult_is_number
       AND tsrrsult_org_id = rpad(c_act_tsrrsult_id_rec.tsrrsult_org_id, 8);
  
    DELETE tsrrlra
     WHERE tsrrsult_is_number = c_act_tsrrsult_id_rec.tsrrsult_is_number
       AND tsrrsult_org_id = rpad(c_act_tsrrsult_id_rec.tsrrsult_org_id, 8);
  
    DELETE tsrrci
     WHERE tsrrsult_is_number = c_act_tsrrsult_id_rec.tsrrsult_is_number
       AND tsrrsult_org_id = rpad(c_act_tsrrsult_id_rec.tsrrsult_org_id, 8);
  
    DELETE tsrrqcaf
     WHERE tsrrsult_is_number = c_act_tsrrsult_id_rec.tsrrsult_is_number
       AND tsrrsult_org_id = rpad(c_act_tsrrsult_id_rec.tsrrsult_org_id, 8);
  
  END LOOP; -- c_act_tsrrsult_id_rec

  DELETE tsrbrg
   WHERE tsrfdact_is_number = p_tsrfdact_is_number
     AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8);

  -- jah: remove 'Composite...' child records
  DELETE tsrfdact
   WHERE tsrfdact0is_number = p_tsrfdact_is_number
     AND tsrfdact0org_id = rpad(p_tsrfdact_org_id, 8);

  -- jah: remove 'Created from Sample' child records
  DELETE tsrfdact
   WHERE tsrfdact1is_number = p_tsrfdact_is_number
     AND tsrfdact1org_id = rpad(p_tsrfdact_org_id, 8);

  COMMIT;
  RETURN('Y');
EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                           
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in DELETE_RESULT_CHILDREN');
     
       
    RETURN('N');
END delete_result_children;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION chk_if_station_delete
(
  p_sta_is_number       NUMBER,
  p_sta_tsmorgan_org_id VARCHAR2,
  p_exp_id              VARCHAR2,
  p_export_log_seq      NUMBER
) RETURN VARCHAR2 IS

  v_cnt     NUMBER(10) := 0;
  v_station VARCHAR2(30);
  v_chk     VARCHAR2(1) := 'Y';
  v_success VARCHAR2(1);

  e_bad_chk EXCEPTION;

  -- Table of Storet Metadata for STATioN (TSMSTATN)
  CURSOR c_station(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT identification_cd
      FROM tsmstatn
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  -- Table of Storet Metadata for PIPE (TSMPIPE)    
  CURSOR c_pipe(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmpipe
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8)
       AND d_userid_code != p_exp_id;

  /* Cursors of tables that are children from table tsmpipe */
  CURSOR c_pipe_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsmpipe_is_number,
           tsmpipe_org_id
      FROM tsmpipe
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsmalp_pipe_id(p_tsmpipe_is_number IN NUMBER, p_tsmpipe_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsmalp_is_number,
           tsmalp_org_id
      FROM tsmalp
     WHERE tsmpipe_is_number = p_tsmpipe_is_number
       AND tsmpipe_org_id = rpad(p_tsmpipe_org_id, 8);
  /* End tables that are children from table tsmpipe */

  CURSOR c_tsmalp(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmalp
     WHERE tsmstatn0is_number = p_tsmstatn_is_number
       AND tsmstatn0org_id = rpad(p_tsmstatn_org_id, 8)
       AND d_userid_code != p_exp_id;

  /* Cursors of tables that are children from table tsmalp */
  CURSOR c_tsmalp_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsmalp_is_number,
           tsmalp_org_id
      FROM tsmalp
     WHERE tsmstatn0is_number = p_tsmstatn_is_number
       AND tsmstatn0org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsmestlc(p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmestlc
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_tsmalp_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmgll(p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmgll
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_tsmalp_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmocnlc(p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmocnlc
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_tsmalp_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsraal(p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsraal
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_tsmalp_org_id, 8)
       AND d_userid_code != p_exp_id;

  /* Cursor to find child Field Activities */
  CURSOR c_tsraal_id(p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsrfdact_is_number,
           tsrfdact_org_id
      FROM tsraal
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_tsmalp_org_id, 8);
  /* End tables that are children from table tsmalp */

  CURSOR c_tsradl(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsradl
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8)
       AND d_userid_code != p_exp_id;

  /*  Cursors of tables that are children from table tsradl */
  CURSOR c_tsradl_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsradl_is_number,
           tsradl_org_id
      FROM tsradl
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsropprd(p_tsradl_is_number IN NUMBER, p_tsradl_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsradl
     WHERE tsradl_is_number = p_tsradl_is_number
       AND tsradl_org_id = rpad(p_tsradl_org_id, 8)
       AND d_userid_code != p_exp_id;
  /* End tables that are children from table tsradl */

  CURSOR c_tsmwell(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmwell
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8)
       AND d_userid_code != p_exp_id;

  /*  Cursors of tables that are children from table tsmwell */
  CURSOR c_tsmwell_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsmwell_is_number,
           tsmwell_org_id
      FROM tsmwell
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsmaddr(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2, p_exp_id IN VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmaddr
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmlog(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2, p_exp_id IN VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmlog
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmpump(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2, p_exp_id IN VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmpump
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmwle(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2, p_exp_id IN VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmwle
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmintvl(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmintvl
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmintvl_id(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsmintvl_is_number,
           tsmintvl_org_id
      FROM tsmintvl
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8);

  CURSOR c_tsmcsng(p_tsmintvl_is_number IN NUMBER, p_tsmintvl_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmcsng
     WHERE tsmintvl_is_number = p_tsmintvl_is_number
       AND tsmintvl_org_id = rpad(p_tsmintvl_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmopng(p_tsmintvl_is_number IN NUMBER, p_tsmintvl_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmopng
     WHERE tsmintvl_is_number = p_tsmintvl_is_number
       AND tsmintvl_org_id = rpad(p_tsmintvl_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmfill(p_tsmintvl_is_number IN NUMBER, p_tsmintvl_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmfill
     WHERE tsmintvl_is_number = p_tsmintvl_is_number
       AND tsmintvl_org_id = rpad(p_tsmintvl_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmhole(p_tsmintvl_is_number IN NUMBER, p_tsmintvl_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmhole
     WHERE tsmintvl_is_number = p_tsmintvl_is_number
       AND tsmintvl_org_id = rpad(p_tsmintvl_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsmigua(p_tsmintvl_is_number IN NUMBER, p_tsmintvl_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmigua
     WHERE tsmintvl_is_number = p_tsmintvl_is_number
       AND tsmintvl_org_id = rpad(p_tsmintvl_org_id, 8)
       AND d_userid_code != p_exp_id;

BEGIN

  OPEN c_station(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id);
  FETCH c_station
    INTO v_station;
  CLOSE c_station;

  /* Check if there are any TSMPIPE children */
  OPEN c_pipe(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id);
  FETCH c_pipe
    INTO v_cnt;
  CLOSE c_pipe;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  FOR c_pipe_id_rec IN c_pipe_id(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id) LOOP
    FOR c_tsmalp_pipe_id_rec IN c_tsmalp_pipe_id(c_pipe_id_rec.tsmpipe_is_number, c_pipe_id_rec.tsmpipe_org_id, p_exp_id) LOOP
      OPEN c_tsmestlc(c_tsmalp_pipe_id_rec.tsmalp_is_number, c_tsmalp_pipe_id_rec.tsmalp_org_id, p_exp_id);
      FETCH c_tsmestlc
        INTO v_cnt;
      CLOSE c_tsmestlc;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsmgll(c_tsmalp_pipe_id_rec.tsmalp_is_number, c_tsmalp_pipe_id_rec.tsmalp_org_id, p_exp_id);
      FETCH c_tsmgll
        INTO v_cnt;
      CLOSE c_tsmgll;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsmocnlc(c_tsmalp_pipe_id_rec.tsmalp_is_number, c_tsmalp_pipe_id_rec.tsmalp_org_id, p_exp_id);
      FETCH c_tsmocnlc
        INTO v_cnt;
      CLOSE c_tsmocnlc;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsraal(c_tsmalp_pipe_id_rec.tsmalp_is_number, c_tsmalp_pipe_id_rec.tsmalp_org_id, p_exp_id);
      FETCH c_tsraal
        INTO v_cnt;
      CLOSE c_tsraal;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      FOR c_tsraal_id_rec IN c_tsraal_id(c_tsmalp_pipe_id_rec.tsmalp_is_number, c_tsmalp_pipe_id_rec.tsmalp_org_id, p_exp_id) LOOP
        v_success := chk_if_result_delete(c_tsraal_id_rec.tsrfdact_is_number, c_tsraal_id_rec.tsrfdact_org_id, p_exp_id, p_export_log_seq, NULL);
        IF v_success = 'N' THEN
          RAISE e_bad_chk;
        END IF;
      END LOOP; -- c_tsraal_id_rec
    END LOOP; --c_tsmalp_pip_id_rec
  
  END LOOP; --c_pipe_id_rec

  /* Check if there are any TSMMALP children */
  OPEN c_tsmalp(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id);
  FETCH c_tsmalp
    INTO v_cnt;
  CLOSE c_tsmalp;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  FOR c_tsmalp_id_rec IN c_tsmalp_id(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id) LOOP
  
    OPEN c_tsmestlc(c_tsmalp_id_rec.tsmalp_is_number, c_tsmalp_id_rec.tsmalp_org_id, p_exp_id);
    FETCH c_tsmestlc
      INTO v_cnt;
    CLOSE c_tsmestlc;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsmgll(c_tsmalp_id_rec.tsmalp_is_number, c_tsmalp_id_rec.tsmalp_org_id, p_exp_id);
    FETCH c_tsmgll
      INTO v_cnt;
    CLOSE c_tsmgll;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsmocnlc(c_tsmalp_id_rec.tsmalp_is_number, c_tsmalp_id_rec.tsmalp_org_id, p_exp_id);
    FETCH c_tsmocnlc
      INTO v_cnt;
    CLOSE c_tsmocnlc;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsraal(c_tsmalp_id_rec.tsmalp_is_number, c_tsmalp_id_rec.tsmalp_org_id, p_exp_id);
    FETCH c_tsraal
      INTO v_cnt;
    CLOSE c_tsraal;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    FOR c_tsraal_id_rec IN c_tsraal_id(c_tsmalp_id_rec.tsmalp_is_number, c_tsmalp_id_rec.tsmalp_org_id, p_exp_id) LOOP
      v_success := chk_if_result_delete(c_tsraal_id_rec.tsrfdact_is_number, c_tsraal_id_rec.tsrfdact_org_id, p_exp_id, p_export_log_seq, NULL);
      IF v_success = 'N' THEN
        RAISE e_bad_chk;
      END IF;
    END LOOP; -- c_tsraal_id_rec
  END LOOP; -- c_tsmalp_id_rec

  /* Check if there are any TSRADL children */
  OPEN c_tsradl(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id);
  FETCH c_tsradl
    INTO v_cnt;
  CLOSE c_tsradl;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  FOR c_tsradl_id_rec IN c_tsradl_id(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id) LOOP
    OPEN c_tsropprd(c_tsradl_id_rec.tsradl_is_number, c_tsradl_id_rec.tsradl_org_id, p_exp_id);
    FETCH c_tsropprd
      INTO v_cnt;
    CLOSE c_tsropprd;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  END LOOP; -- c_tsradl_id_rec

  /* Check if there are any TSMWELL children */
  OPEN c_tsmwell(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id);
  FETCH c_tsmwell
    INTO v_cnt;
  CLOSE c_tsmwell;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  FOR c_tsmwell_id_rec IN c_tsmwell_id(p_sta_is_number, p_sta_tsmorgan_org_id, p_exp_id) LOOP
    OPEN c_tsmaddr(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id, p_exp_id);
    FETCH c_tsmaddr
      INTO v_cnt;
    CLOSE c_tsmaddr;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsmlog(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id, p_exp_id);
    FETCH c_tsmlog
      INTO v_cnt;
    CLOSE c_tsmlog;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsmpump(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id, p_exp_id);
    FETCH c_tsmpump
      INTO v_cnt;
    CLOSE c_tsmpump;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsmwle(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id, p_exp_id);
    FETCH c_tsmwle
      INTO v_cnt;
    CLOSE c_tsmwle;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsmintvl(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id, p_exp_id);
    FETCH c_tsmintvl
      INTO v_cnt;
    CLOSE c_tsmintvl;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    FOR c_tsmintvl_id_rec IN c_tsmintvl_id(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id, p_exp_id) LOOP
    
      OPEN c_tsmcsng(c_tsmintvl_id_rec.tsmintvl_is_number, c_tsmintvl_id_rec.tsmintvl_org_id, p_exp_id);
      FETCH c_tsmcsng
        INTO v_cnt;
      CLOSE c_tsmcsng;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsmfill(c_tsmintvl_id_rec.tsmintvl_is_number, c_tsmintvl_id_rec.tsmintvl_org_id, p_exp_id);
      FETCH c_tsmfill
        INTO v_cnt;
      CLOSE c_tsmfill;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsmhole(c_tsmintvl_id_rec.tsmintvl_is_number, c_tsmintvl_id_rec.tsmintvl_org_id, p_exp_id);
      FETCH c_tsmhole
        INTO v_cnt;
      CLOSE c_tsmhole;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsmigua(c_tsmintvl_id_rec.tsmintvl_is_number, c_tsmintvl_id_rec.tsmintvl_org_id, p_exp_id);
      FETCH c_tsmigua
        INTO v_cnt;
      CLOSE c_tsmigua;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsmopng(c_tsmintvl_id_rec.tsmintvl_is_number, c_tsmintvl_id_rec.tsmintvl_org_id, p_exp_id);
      FETCH c_tsmopng
        INTO v_cnt;
      CLOSE c_tsmopng;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
    END LOOP; -- c_tsmintvl_id_rec
  
  END LOOP; -- c_tsmwell_id_rec

  RETURN(v_chk);
EXCEPTION
  WHEN e_bad_chk THEN
    v_chk := 'N';
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Associated records have been found in STORET that were not migrated by SIM.' ||
                                         ' Please use the STORET interface to delete Station ' ||
                                         v_station);
     
       
    RETURN(v_chk);
  
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                               
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in chk_if_station_delete');
     
       
END chk_if_station_delete;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION chk_if_result_delete
(
  p_tsrfdact_is_number IN NUMBER,
  p_tsrfdact_org_id    IN VARCHAR2,
  p_exp_id             VARCHAR2,
  p_exp_log_seq        NUMBER,
  p_activity_id        IN VARCHAR2 DEFAULT NULL
) RETURN VARCHAR2 IS

  v_cnt NUMBER(10) := 0;
  e_bad_chk EXCEPTION;
  e_parent EXCEPTION;

  CURSOR c_tsraal(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsraal
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != rpad(p_exp_id, 8);

  CURSOR c_tsreod(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsreod
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code = p_exp_id;

  CURSOR c_tsrfafsa(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfafsa
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  -- Field Activity PErsonnel Association (FAPEA)
  CURSOR c_tsrfapea(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfapea
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  -- Field Activity Cooperating Organization Assocciation (FACOA)
  CURSOR c_tsrfacoa(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfacoa
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  -- Field Activity PRoject Association (FAPRA)
  CURSOR c_tsrfapra(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfapra
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrfaqsa(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfaqsa
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrsmple(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrsmple
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrtnod(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrtnod
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrtod(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrtod
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrbrg(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrbrg
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrbrg_id(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsrbrg_is_number,
           tsrbrg_org_id
      FROM tsrbrg
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code = p_exp_id;

  CURSOR c_brg_tsrbrgi(p_tsrbrg_is_number NUMBER, p_tsrbrg_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrbrgi
     WHERE tsrbrg_is_number = p_tsrbrg_is_number
       AND tsrbrg_org_id = rpad(p_tsrbrg_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_brg_tsrrsult(p_tsrbrg_is_number NUMBER, p_tsrbrg_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrrsult
     WHERE tsrbrg_is_number = p_tsrbrg_is_number
       AND tsrbrg_org_id = rpad(p_tsrbrg_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_act_tsrrsult(p_tsrfdact_is_number NUMBER, p_tsrfdact_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrrsult
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_act_tsrrsult_id(p_tsrfdact_is_number NUMBER, p_tsrfdact_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsrrsult_is_number,
           tsrrsult_org_id
      FROM tsrrsult
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code = p_exp_id;

  CURSOR c_brg_tsrrsult_id(p_tsrbrg_is_number NUMBER, p_tsrbrg_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT tsrrsult_is_number,
           tsrrsult_org_id
      FROM tsrrsult
     WHERE tsrbrg_is_number = p_tsrbrg_is_number
       AND tsrbrg_org_id = rpad(p_tsrbrg_org_id, 8)
       AND d_userid_code = p_exp_id;

  -- Need to call these cursors twice once for tsrfdac and for tsrbrg
  CURSOR c_tsmgntxt(p_tsrrsult_is_number NUMBER, p_tsrrsult_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsmgntxt
     WHERE tsrrsult_is_number = p_tsrrsult_is_number
       AND tsrrsult_org_id = rpad(p_tsrrsult_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrdql(p_tsrrsult_is_number NUMBER, p_tsrrsult_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrdql
     WHERE tsrrsult_is_number = p_tsrrsult_is_number
       AND tsrrsult_org_id = rpad(p_tsrrsult_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrrlra(p_tsrrsult_is_number NUMBER, p_tsrrsult_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrrlra
     WHERE tsrrsult_is_number = p_tsrrsult_is_number
       AND tsrrsult_org_id = rpad(p_tsrrsult_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrcldes(p_tsrrsult_is_number NUMBER, p_tsrrsult_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrcldes
     WHERE tsrrsult_is_number = p_tsrrsult_is_number
       AND tsrrsult_org_id = rpad(p_tsrrsult_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrrci(p_tsrrsult_is_number NUMBER, p_tsrrsult_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrrci
     WHERE tsrrsult_is_number = p_tsrrsult_is_number
       AND tsrrsult_org_id = rpad(p_tsrrsult_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrrqcaf(p_tsrrsult_is_number NUMBER, p_tsrrsult_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrrqcaf
     WHERE tsrrsult_is_number = p_tsrrsult_is_number
       AND tsrrsult_org_id = rpad(p_tsrrsult_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_tsrdlin(p_tsrfdact_is_number NUMBER, p_tsrfdact_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrdlin
     WHERE tsrfdact_is_number = p_tsrfdact_is_number
       AND tsrfdact_org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  -- These cursors are for samples and composite samples

  CURSOR c_samp(p_tsrfdact_is_number NUMBER, p_tsrfdact_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfdact
     WHERE tsrfdact0is_number = p_tsrfdact_is_number
       AND tsrfdact0org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

  CURSOR c_comp_samp(p_tsrfdact_is_number NUMBER, p_tsrfdact_org_id VARCHAR2, p_exp_id VARCHAR2) IS
    SELECT COUNT(1)
      FROM tsrfdact
     WHERE tsrfdact1is_number = p_tsrfdact_is_number
       AND tsrfdact1org_id = rpad(p_tsrfdact_org_id, 8)
       AND d_userid_code != p_exp_id;

BEGIN

  OPEN c_tsraal(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsraal
    INTO v_cnt;
  CLOSE c_tsraal;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsreod(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsreod
    INTO v_cnt;
  CLOSE c_tsreod;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrfafsa(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrfafsa
    INTO v_cnt;
  CLOSE c_tsrfafsa;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrfapea(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrfapea
    INTO v_cnt;
  CLOSE c_tsrfapea;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  -- count changes to Field Activity cooperating Organization Association
  OPEN c_tsrfacoa(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrfacoa
    INTO v_cnt;
  CLOSE c_tsrfacoa;
  -- did we find any changes?
  IF nvl(v_cnt, 0) > 0 THEN
    -- if yes
    -- raise an exception
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrdlin(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrdlin
    INTO v_cnt;
  CLOSE c_tsrdlin;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrfapra(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrfapra
    INTO v_cnt;
  CLOSE c_tsrfapra;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrfaqsa(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrfaqsa
    INTO v_cnt;
  CLOSE c_tsrfaqsa;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrsmple(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrsmple
    INTO v_cnt;
  CLOSE c_tsrsmple;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrtnod(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrtnod
    INTO v_cnt;
  CLOSE c_tsrtnod;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrtod(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrtod
    INTO v_cnt;
  CLOSE c_tsrtod;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_tsrbrg(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_tsrbrg
    INTO v_cnt;
  CLOSE c_tsrbrg;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  FOR c_tsrbrg_id_rec IN c_tsrbrg_id(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id) LOOP
    OPEN c_brg_tsrbrgi(c_tsrbrg_id_rec.tsrbrg_is_number, c_tsrbrg_id_rec.tsrbrg_org_id, p_exp_id);
    FETCH c_brg_tsrbrgi
      INTO v_cnt;
    CLOSE c_brg_tsrbrgi;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_brg_tsrrsult(c_tsrbrg_id_rec.tsrbrg_is_number, c_tsrbrg_id_rec.tsrbrg_org_id, p_exp_id);
    FETCH c_brg_tsrrsult
      INTO v_cnt;
    CLOSE c_brg_tsrrsult;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    FOR c_brg_tsrrsult_id_rec IN c_brg_tsrrsult_id(c_tsrbrg_id_rec.tsrbrg_is_number, c_tsrbrg_id_rec.tsrbrg_org_id, p_exp_id) LOOP
      OPEN c_tsmgntxt(c_brg_tsrrsult_id_rec.tsrrsult_is_number, c_brg_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
      FETCH c_tsmgntxt
        INTO v_cnt;
      CLOSE c_tsmgntxt;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsrdql(c_brg_tsrrsult_id_rec.tsrrsult_is_number, c_brg_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
      FETCH c_tsrdql
        INTO v_cnt;
      CLOSE c_tsrdql;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsrrlra(c_brg_tsrrsult_id_rec.tsrrsult_is_number, c_brg_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
      FETCH c_tsrrlra
        INTO v_cnt;
      CLOSE c_tsrrlra;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsrcldes(c_brg_tsrrsult_id_rec.tsrrsult_is_number, c_brg_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
      FETCH c_tsrcldes
        INTO v_cnt;
      CLOSE c_tsrcldes;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsrrci(c_brg_tsrrsult_id_rec.tsrrsult_is_number, c_brg_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
      FETCH c_tsrrci
        INTO v_cnt;
      CLOSE c_tsrrci;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    
      OPEN c_tsrrqcaf(c_brg_tsrrsult_id_rec.tsrrsult_is_number, c_brg_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
      FETCH c_tsrrqcaf
        INTO v_cnt;
      CLOSE c_tsrrqcaf;
      IF nvl(v_cnt, 0) > 0 THEN
        RAISE e_bad_chk;
      END IF;
    END LOOP;
  END LOOP;

  OPEN c_act_tsrrsult(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_act_tsrrsult
    INTO v_cnt;
  CLOSE c_act_tsrrsult;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  FOR c_act_tsrrsult_id_rec IN c_act_tsrrsult_id(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id) LOOP
    OPEN c_tsmgntxt(c_act_tsrrsult_id_rec.tsrrsult_is_number, c_act_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
    FETCH c_tsmgntxt
      INTO v_cnt;
    CLOSE c_tsmgntxt;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsrdql(c_act_tsrrsult_id_rec.tsrrsult_is_number, c_act_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
    FETCH c_tsrdql
      INTO v_cnt;
    CLOSE c_tsrdql;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsrrlra(c_act_tsrrsult_id_rec.tsrrsult_is_number, c_act_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
    FETCH c_tsrrlra
      INTO v_cnt;
    CLOSE c_tsrrlra;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsrcldes(c_act_tsrrsult_id_rec.tsrrsult_is_number, c_act_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
    FETCH c_tsrcldes
      INTO v_cnt;
    CLOSE c_tsrcldes;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsrrci(c_act_tsrrsult_id_rec.tsrrsult_is_number, c_act_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
    FETCH c_tsrrci
      INTO v_cnt;
    CLOSE c_tsrrci;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  
    OPEN c_tsrrqcaf(c_act_tsrrsult_id_rec.tsrrsult_is_number, c_act_tsrrsult_id_rec.tsrrsult_org_id, p_exp_id);
    FETCH c_tsrrqcaf
      INTO v_cnt;
    CLOSE c_tsrrqcaf;
    IF nvl(v_cnt, 0) > 0 THEN
      RAISE e_bad_chk;
    END IF;
  END LOOP;

  OPEN c_samp(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_samp
    INTO v_cnt;
  CLOSE c_samp;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  OPEN c_comp_samp(p_tsrfdact_is_number, p_tsrfdact_org_id, p_exp_id);
  FETCH c_comp_samp
    INTO v_cnt;
  CLOSE c_comp_samp;
  IF nvl(v_cnt, 0) > 0 THEN
    RAISE e_bad_chk;
  END IF;

  -- return 'okay to delete'
  RETURN('Y');

EXCEPTION

  -- did we find a record that was not migrated by SIM?
  WHEN e_bad_chk THEN
    -- if yes
  
    -- Did the calling routine pass an activity id?     
    IF p_activity_id IS NOT NULL THEN
      -- if yes
      -- Output an exception message
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           'Associated records have been found in STORET that were not migrated by SIM. ' ||
                                           'Please use the STORET interface to delete Activity ' ||
                                           p_activity_id || '.');
       
         
    END IF;
  
    -- return 'do not delete'
    RETURN('N');
  
  WHEN OTHERS THEN
    -- output exception message
     
                                                                                                                                                                                                                                                   
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in Procedure CHK_IF_RESULT_DELETE.');
     
       
  
END chk_if_result_delete;

/****************************************************************************************
  Remove all data from STORET that was part of one SIM Import file
 ****************************************************************************************/ 
PROCEDURE clear_storet_export
(
  p_export_log_seq    IN NUMBER,
  p_remove_by_file_yn IN VARCHAR2,
  p_success           OUT NUMBER
) IS

  v_tsrfdact_is_number NUMBER(8);
  v_tsrfdact_org_id    VARCHAR2(8);
  v_tsrfqs_is_number   NUMBER(8);
  v_tsrfqs_org_id      VARCHAR2(8);
  v_visit_created_by   VARCHAR2(10);
  v_trip_created_by    VARCHAR2(10);
  v_visit_is_nbr       NUMBER(8);
  v_visit_org_id       CHAR(8);
  v_trip_is_nbr        NUMBER(8);
  v_trip_org_id        CHAR(8);
  v_dummy_is_nbr       NUMBER(8);
  v_result_cnt         NUMBER(10) := 0;
  v_import_seq         NUMBER(10);
  v_chk_storet_on_del  VARCHAR2(5);
  v_delete_ok          VARCHAR2(1);
  v_activity_cnt       NUMBER(10) := 0;
  
  e_storet_child EXCEPTION;
  e_sim_child EXCEPTION;
  e_no_delete EXCEPTION;

  CURSOR c_get_visit_info(p_tsrfdact_is_number IN NUMBER, p_tsrfdact_org_id IN VARCHAR2) IS
    SELECT v.d_userid_code, v.tsrstvst_is_number, v.tsrstvst_org_id
       FROM tsrstvst v,
            tsrfdact fa
      WHERE fa.tsrfdact_is_number = p_tsrfdact_is_number
        AND fa.tsrfdact_org_id = p_tsrfdact_org_id
        AND v.tsrstvst_is_number = fa.tsrstvst_is_number
        AND v.tsrstvst_org_id = fa.tsrstvst_org_id;

  CURSOR c_get_trip_info(p_visit_is_nbr IN NUMBER, p_visit_org_id IN VARCHAR2) IS
    SELECT d_userid_code, tsrtrip_is_number, tsrtrip_org_id
      FROM tsrstvst
      WHERE tsrstvst_is_number = p_visit_is_nbr
      AND tsrstvst_org_id = p_visit_org_id;

  CURSOR c_check_visit_in_storet(p_visit_is_nbr IN NUMBER, p_visit_org_id IN VARCHAR2) IS
    SELECT tsrfdact_is_number
      FROM tsrfdact
     WHERE tsrstvst_is_number = p_visit_is_nbr
       AND tsrstvst_org_id = rpad(p_visit_org_id, 8);

  CURSOR c_check_trip_in_storet(p_trip_is_nbr IN NUMBER, p_trip_org_id IN VARCHAR2) IS
    SELECT tsrstvst_is_number
      FROM tsrstvst
     WHERE tsrtrip_is_number = p_trip_is_nbr
       AND tsrtrip_org_id = rpad(p_trip_org_id, 8);

  CURSOR c_check_trip_qc(p_trip_is_nbr IN NUMBER, p_trip_org_id IN VARCHAR2) IS
    SELECT tsrfqs_is_number
      FROM tsrfqs
     WHERE tsrtrip_is_number = p_trip_is_nbr
       AND tsrtrip_org_id = rpad(p_trip_org_id, 8);

  CURSOR c_result_is_nbr(p_org_id VARCHAR2, p_activity_is NUMBER) IS
    SELECT tsrrsult_is_number,
           tsrrsult_org_id
      FROM tsrrsult
     WHERE tsrfdact_is_number = p_activity_is
       AND tsrfdact_org_id = rpad(p_org_id, 8);

  CURSOR c_brg_result_is_nbr(p_org_id VARCHAR2, p_activity_is NUMBER) IS
    SELECT r.tsrrsult_is_number,
           r.tsrrsult_org_id,
           r.tsrbrg_is_number,
           r.tsrbrg_org_id,
           r.tsrbrgi_is_number,
           r.tsrbrgi_org_id
      FROM tsrrsult r,
           tsrbrg b
     WHERE r.tsrbrg_is_number = b.tsrbrg_is_number
       AND r.tsrbrg_org_id = b.tsrbrg_org_id
       AND b.tsrfdact_is_number = p_activity_is
       AND b.tsrfdact_org_id = rpad(p_org_id, 8);

  CURSOR c_qc_result_is_nbr(p_org_id VARCHAR2, p_trip_qc_is NUMBER) IS
    SELECT tsrrsult_is_number,
           tsrrsult_org_id
      FROM tsrrsult
     WHERE tsrfqs_is_number = p_trip_qc_is
       AND tsrfqs_org_id = rpad(p_org_id, 8);

  CURSOR c_chk_storet_on_del IS
    SELECT sc_value
      FROM sim_config
     WHERE sc_item = 'CHK_STORET_ON_DEL';
 
  CURSOR c_get_storet_children(cp_is_number IN NUMBER, cp_org_id IN VARCHAR2, cp_export_seq IN NUMBER) IS
    SELECT 1
      FROM tsrfdact,
           sim_export_storet_ids
     WHERE tsrfdact0is_number = sesi_is_number(+)
       AND tsrfdact0org_id = sesi_org_id(+)
       AND tsrfdact0is_number IS NOT NULL
       AND nvl(sesi_sel_seq, 0) != cp_export_seq
       AND tsrfdact_org_id = rpad(cp_org_id, 8)
       AND tsrfdact_is_number = cp_is_number
    UNION
    SELECT 1
      FROM tsrfdact,
           sim_export_storet_ids
     WHERE tsrfdact_is_number = sesi_is_number(+)
       AND tsrfdact_org_id = sesi_org_id(+)
       AND tsrfdact1is_number = cp_is_number
       AND tsrfdact1org_id = rpad(cp_org_id, 8)
       AND nvl(sesi_sel_seq, 0) != cp_export_seq;
       
  r_get_storet_children c_get_storet_children%ROWTYPE;

BEGIN
  OPEN c_chk_storet_on_del;
  FETCH c_chk_storet_on_del
    INTO v_chk_storet_on_del;
  CLOSE c_chk_storet_on_del;

  IF v_chk_storet_on_del IS NULL
     OR upper(v_chk_storet_on_del) = 'YES' THEN
  
    -- Loop through all the records in this export
    FOR v_fa_export IN (SELECT sesi_is_number tsrfdact_is_number,
                               sesi_org_id tsrfdact_org_id,
                               id_code,
                               category_type_name
                          FROM sim_export_storet_ids,
                               tsrfdact
                         WHERE sesi_is_number = tsrfdact_is_number
                           AND sesi_org_id = tsrfdact_org_id
                           AND sesi_sel_seq = p_export_log_seq) LOOP
      
      /*If the remove process is by file then don't do the following code to allow
       removal of records regardless of modified user.*/
      IF p_remove_by_file_yn = 'N' THEN
        -- Check for altered STORET child records that were not altered by SIM
        v_delete_ok := chk_if_result_delete(v_fa_export.tsrfdact_is_number, 
                                            v_fa_export.tsrfdact_org_id, 
                                            sim_export_pkg.v_pkg_ownerid, 
                                            p_export_log_seq, 
                                            v_fa_export.id_code);
      
        -- Did we find any altered STORET child records that were NOT altered by SIM?
        IF v_delete_ok = 'N' THEN
          -- if so
          -- Terminate the un-migrate to avoid deleting child activity records that someone else entered.
          RAISE e_no_delete;
        END IF;
      END IF;
      -- jah 2-18-04
      -- Try to find any STORET children of this activity that are NOT part of this export
      OPEN c_get_storet_children(v_fa_export.tsrfdact_is_number, v_fa_export.tsrfdact_org_id, p_export_log_seq);
      FETCH c_get_storet_children
        INTO r_get_storet_children;
    
      -- jah 2-18-04
      -- Did we find any children in STORET that are NOT part of this export?
      IF (c_get_storet_children%FOUND) THEN
        -- if yes
        -- Terminate the un-migrate because it would leave an orphan child activity in STORET
        RAISE e_storet_child;
      END IF;
      CLOSE c_get_storet_children;
    END LOOP; --c_fa_export
  END IF;

  -- Get the name of the storet table in sim to determine if this is a Trip QC activity
  FOR tablename_rec IN (SELECT DISTINCT sesi.sesi_table_name 
                          FROM sim_export_storet_ids sesi
                          WHERE sesi_sel_seq = p_export_log_seq) LOOP
    
    IF tablename_rec.sesi_table_name = 'TSRFQS' THEN
      FOR v_qc_export IN (SELECT sesi_is_number tsrfqs_is_number, 
                                 sesi_org_id tsrfqs_org_id,
                                 id_code
                            FROM sim_export_storet_ids,
                                 tsrfqs
                            WHERE sesi_is_number = tsrfqs_is_number
                              AND sesi_org_id = tsrfqs_org_id
                              AND sesi_sel_seq = p_export_log_seq) LOOP
       
      /* different set of deletes for trip qc data */
        FOR v_result_is_nbr IN c_qc_result_is_nbr(v_qc_export.tsrfqs_org_id, 
                                                    v_qc_export.tsrfqs_is_number) LOOP
          
          --Delete the Result Lab Remarks
          DELETE tsrrlra
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
             
          /* delete the tsrdql for each result*/
          DELETE tsrdql
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
          
          /* delete the tsmgntxt for each result*/
          /*rk - 09/01/2005*/
          DELETE tsmgntxt
            WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
              AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
                               
          END LOOP;
        /* delete the results */
        DELETE tsrrsult
         WHERE tsrfqs_is_number = v_qc_export.tsrfqs_is_number
           AND tsrfqs_org_id = rpad(v_qc_export.tsrfqs_org_id, 8);
      
        COMMIT;
      
        /* delete trip qc, person assignment */
        DELETE tsrfqspa
         WHERE tsrfqs_is_number = v_qc_export.tsrfqs_is_number
           AND tsrfqs_org_id = rpad(v_qc_export.tsrfqs_org_id, 8);
      
        /* delete the activity */
        DELETE tsrfqs
         WHERE tsrfqs_is_number = v_qc_export.tsrfqs_is_number
           AND tsrfqs_org_id = rpad(v_qc_export.tsrfqs_org_id, 8);
      
        COMMIT;
      END LOOP;
      COMMIT;  
    ELSIF tablename_rec.sesi_table_name = 'TSRFDACT' THEN                              
      -- Loop through field activity records that have the given export log sequence number
      /*FOR v_fa_export IN c_fa_export(p_export_log_seq) LOOP*/
      FOR v_fa_export IN (SELECT sesi_is_number tsrfdact_is_number,
                                 sesi_org_id tsrfdact_org_id,
                                 id_code,
                                 category_type_name,
                                 tsrstvst_is_number,
                                 tsrstvst_org_id
                            FROM sim_export_storet_ids,
                                 tsrfdact
                           WHERE sesi_is_number = tsrfdact_is_number
                             AND sesi_org_id = tsrfdact_org_id
                             AND sesi_sel_seq = p_export_log_seq) LOOP
                               
        -- get the IS NUMBER and ORG_ID
        v_tsrfdact_is_number := v_fa_export.tsrfdact_is_number;
        v_tsrfdact_org_id := v_fa_export.tsrfdact_org_id;
        
        OPEN c_get_visit_info(v_fa_export.tsrfdact_is_number, v_fa_export.tsrfdact_org_id);
          FETCH c_get_visit_info
            INTO v_visit_created_by, v_visit_is_nbr, v_visit_org_id;
          CLOSE c_get_visit_info;
      
        OPEN c_get_trip_info(v_visit_is_nbr, v_visit_org_id);
              FETCH c_get_trip_info
                INTO v_trip_created_by, v_trip_is_nbr, v_trip_org_id;
              CLOSE c_get_trip_info;
        
        --Delete non-brg results
        FOR v_result_is_nbr IN c_result_is_nbr(v_tsrfdact_org_id, v_tsrfdact_is_number) LOOP
        
          --Delete the Result Lab Remarks
          DELETE tsrrlra
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
        
          /* delete the tsrdql for each result*/
          DELETE tsrdql
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
        
          /* delete the tsmgntxt for each result */
          DELETE tsmgntxt
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
        
          /* delete the tsrcldes for each result */
          DELETE tsrcldes
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
        
          -- jah 2-23-04 delete the tsrrci for each non-brg result
          DELETE tsrrci
           WHERE tsrrsult_is_number = v_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_result_is_nbr.tsrrsult_org_id, 8);
        
        END LOOP;
        COMMIT;
      
        -- Do the same thing for brg results
        FOR v_brg_result_is_nbr IN c_brg_result_is_nbr(v_tsrfdact_org_id, v_tsrfdact_is_number) LOOP
        
          /* delete the tsrdql for each result*/
          DELETE tsrdql
           WHERE tsrrsult_is_number = v_brg_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_brg_result_is_nbr.tsrrsult_org_id, 8);
        
          /* delete the tsmgntxt for each result */
          DELETE tsmgntxt
           WHERE tsrrsult_is_number = v_brg_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_brg_result_is_nbr.tsrrsult_org_id, 8);
        
          /* delete the tsrbrgi for each result */
          DELETE tsrbrgi
           WHERE tsrbrgi_is_number = v_brg_result_is_nbr.tsrbrgi_is_number
             AND tsrbrgi_org_id = rpad(v_brg_result_is_nbr.tsrbrgi_org_id, 8);
        
          /* delete the tsrbrg for each result */
          DELETE tsrbrg
           WHERE tsrbrg_is_number = v_brg_result_is_nbr.tsrbrg_is_number
             AND tsrbrg_org_id = rpad(v_brg_result_is_nbr.tsrbrg_org_id, 8);
        
          /* delete the tsrrci for each result */
          DELETE tsrrci
           WHERE tsrrsult_is_number = v_brg_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_brg_result_is_nbr.tsrrsult_org_id, 8);
        
          /* delete the tsrcldes for each result */
          DELETE tsrcldes
           WHERE tsrrsult_is_number = v_brg_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_brg_result_is_nbr.tsrrsult_org_id, 8);
        
          DELETE tsrrsult
           WHERE tsrrsult_is_number = v_brg_result_is_nbr.tsrrsult_is_number
             AND tsrrsult_org_id = rpad(v_brg_result_is_nbr.tsrrsult_org_id, 8);
        
        END LOOP;
        COMMIT;
      
        /* delete the non-brg results */
        DELETE tsrrsult
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
       
       -- Delete the Data Lines
        DELETE tsrdlin
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        -- delete Field Activity PErsonnel Assignment (FAPEA)
        DELETE tsrfapea
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        -- delete Field Activity Cooperating Organization Assignment (FACOA)
        DELETE tsrfacoa
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete the sample */
        DELETE tsrsmple
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete the trawl op detail (tod) */
        DELETE tsrtod
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete the trap/net details (tsrtnod) */
        DELETE tsrtnod
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete the actual location record */
        DELETE tsraal
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete the field set assmt */
        DELETE tsrfafsa
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete activity, project assignment */
        DELETE tsrfapra
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        /* delete the activity */
        DELETE tsrfdact
         WHERE tsrfdact_is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact_org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
      
        -- jah 2-17-04
        -- Remove references to this activity in parent activities
        UPDATE tsrfdact
           SET tsrfdact0is_number = NULL,
               tsrfdact0org_id = NULL
         WHERE tsrfdact0is_number = v_fa_export.tsrfdact_is_number
           AND tsrfdact0org_id = rpad(v_fa_export.tsrfdact_org_id, 8);
        COMMIT;
        -- Using variables set at the start of the loop, find visits and trips to be deleted     
        IF v_visit_created_by LIKE 'SIM%' OR p_remove_by_file_yn = 'Y' THEN
          /* kms - if visit is not used by any other activities in STORET, delete it from STORET */
          OPEN c_check_visit_in_storet(v_visit_is_nbr, v_visit_org_id);
          FETCH c_check_visit_in_storet
            INTO v_dummy_is_nbr;
        
          -- Do other activities in STORET use this visit number?
          IF c_check_visit_in_storet%NOTFOUND THEN
            -- if no
          
            /* delete field set for sta visit */
            DELETE tsrfdset
             WHERE tsrstvst_is_number = v_visit_is_nbr
               AND tsrstvst_org_id = rpad(v_visit_org_id, 8);
          
            /* kms - delete the station visit */
            DELETE tsrstvst
             WHERE tsrstvst_is_number = v_visit_is_nbr
               AND tsrstvst_org_id = v_visit_org_id;
            /*
              kms - since we know there are no more activities on this station visit, see if there
              are more station visits on the trip and delete if not.  We don't do this when the
              visit was not created by SIM since we know there is at least station visit on it. 
            */    
            IF v_trip_created_by LIKE 'SIM%' OR p_remove_by_file_yn = 'Y' THEN
            
              OPEN c_check_trip_in_storet(v_trip_is_nbr, v_trip_org_id);
              FETCH c_check_trip_in_storet
                INTO v_dummy_is_nbr;
              IF c_check_trip_in_storet%NOTFOUND THEN
              
                /*
                  kms - see if there are any trip qc samples on this trip 
                */
                OPEN c_check_trip_qc(v_trip_is_nbr, v_trip_org_id);
                FETCH c_check_trip_qc
                  INTO v_dummy_is_nbr;
                IF c_check_trip_qc%NOTFOUND THEN
                
                  /*
                    kms - no other station visits or trip qcs on this trip, delete trip and its associations 
                  */
                  DELETE tsrtsa
                   WHERE tsrtrip_is_number = v_trip_is_nbr
                     AND tsrtrip_org_id = v_trip_org_id;
                  DELETE tsrtpa
                   WHERE tsrtrip_is_number = v_trip_is_nbr
                     AND tsrtrip_org_id = v_trip_org_id;
                  DELETE tsrtrip
                   WHERE tsrtrip_is_number = v_trip_is_nbr
                     AND tsrtrip_org_id = v_trip_org_id;  
                END IF;
                CLOSE c_check_trip_qc;
              END IF;
              CLOSE c_check_trip_in_storet;
            END IF;
          END IF;
          CLOSE c_check_visit_in_storet;
        END IF;
        COMMIT;
       END LOOP;
     END IF; 
   END LOOP;
 
  UPDATE sim_field_activities
     SET fa_export_seq = NULL,
         fa_status = 'A'
   WHERE fa_export_seq = p_export_log_seq;

  --sjj - This is a line of code that need to change for event logging  
   
  DELETE sim_export_storet_ids
    WHERE sesi_sel_seq = p_export_log_seq;
    
  DELETE sim_export_log
   WHERE sel_seq = p_export_log_seq;
  COMMIT;
  -- return success [p_success OUT VARCHAR2]
  p_success := 1;

EXCEPTION
  WHEN e_storet_child THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
      ' -- Could not Delete Export -- Child activities in STORET would be orphaned. ' ||
      'To clear this migration, you must first delete all child activities from STORET and SIM.');
    p_success := 0;
  
  WHEN e_sim_child THEN    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
      ' -- Could not Delete Export -- Child activities in SIM would be orphaned. ' ||
      'To clear this migration, you must first delete all child activities from SIM.');
    p_success := 0;
  
  WHEN e_no_delete THEN     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
      ' -- Could not Delete Export -- Found Child records not created by SIM. ');
    p_success := 0;
  
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
      SQLERRM || ' - Failure in Clear_STORET_Export');
    p_success := 0;
  
END clear_storet_export;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION delete_station_children
(
  p_tsmstatn_is_number IN NUMBER,
  p_tsmstatn_org_id    IN VARCHAR2,
  p_exp_log_seq        NUMBER
) RETURN VARCHAR2 IS

  v_success VARCHAR2(1) := 'Y';

  CURSOR c_tsmpipe_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT tsmpipe_is_number,
           tsmpipe_org_id
      FROM tsmpipe
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsmpipe_tsmalp_id(p_tsmpipe_is_number IN NUMBER, p_tsmpipe_org_id IN VARCHAR2) IS
    SELECT tsmalp_is_number,
           tsmalp_org_id
      FROM tsmalp
     WHERE tsmpipe_is_number = p_tsmpipe_is_number
       AND tsmpipe_org_id = rpad(p_tsmpipe_org_id, 8);

  CURSOR c_tsmalp_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT tsmalp_is_number,
           tsmalp_org_id
      FROM tsmalp
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsradl_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT tsradl_is_number,
           tsradl_org_id
      FROM tsradl
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_tsmwell_id(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT tsmwell_is_number,
           tsmwell_org_id
      FROM tsmwell
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_interval_id(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2) IS
    SELECT tsmintvl_is_number,
           tsmintvl_org_id,
           type_code
      FROM tsmintvl
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8);

  CURSOR c_tsraal_id(p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2, p_exp_log_seq NUMBER) IS
    SELECT tsrfdact_is_number,
           tsrfdact_org_id
      FROM tsraal
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_tsmalp_org_id, 8);

BEGIN

  /* Delete the tsmpipe child and child records of tsmpipe */
  FOR c_tsmpipe_id_rec IN c_tsmpipe_id(p_tsmstatn_is_number, p_tsmstatn_org_id) LOOP
    FOR c_tsmpipe_tsmalp_id_rec IN c_tsmpipe_tsmalp_id(c_tsmpipe_id_rec.tsmpipe_is_number, c_tsmpipe_id_rec.tsmpipe_org_id) LOOP
      DELETE tsmestlc
       WHERE tsmalp_is_number = c_tsmpipe_tsmalp_id_rec.tsmalp_is_number
         AND tsmalp_org_id = rpad(c_tsmpipe_tsmalp_id_rec.tsmalp_org_id, 8);
    
      DELETE tsmgll
       WHERE tsmalp_is_number = c_tsmpipe_tsmalp_id_rec.tsmalp_is_number
         AND tsmalp_org_id = rpad(c_tsmpipe_tsmalp_id_rec.tsmalp_org_id, 8);
    
      DELETE tsmocnlc
       WHERE tsmalp_is_number = c_tsmpipe_tsmalp_id_rec.tsmalp_is_number
         AND tsmalp_org_id = rpad(c_tsmpipe_tsmalp_id_rec.tsmalp_org_id, 8);
    
      FOR c_tsraal_id_rec IN c_tsraal_id(c_tsmpipe_tsmalp_id_rec.tsmalp_is_number, c_tsmpipe_tsmalp_id_rec.tsmalp_is_number, p_exp_log_seq) LOOP
        v_success := delete_result_children(c_tsraal_id_rec.tsrfdact_is_number, c_tsraal_id_rec.tsrfdact_org_id, p_exp_log_seq, NULL);
      END LOOP; -- c_tsraal_id_rec
    END LOOP; --c_tsmpipe_tsmalp_id_rec
  END LOOP; --c_tsmpipe_id_rec

  DELETE tsmpipe
   WHERE tsmstatn_is_number = p_tsmstatn_is_number
     AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);
  /* End tsmpipe delete */

  -- Delete the TSMALP table and children
  FOR c_tsmalp_id_rec IN c_tsmalp_id(p_tsmstatn_is_number, p_tsmstatn_org_id) LOOP
    DELETE tsmestlc
     WHERE tsmalp_is_number = c_tsmalp_id_rec.tsmalp_is_number
       AND tsmalp_org_id = rpad(c_tsmalp_id_rec.tsmalp_org_id, 8);
  
    DELETE tsmgll
     WHERE tsmalp_is_number = c_tsmalp_id_rec.tsmalp_is_number
       AND tsmalp_org_id = rpad(c_tsmalp_id_rec.tsmalp_org_id, 8);
  
    DELETE tsmocnlc
     WHERE tsmalp_is_number = c_tsmalp_id_rec.tsmalp_is_number
       AND tsmalp_org_id = rpad(c_tsmalp_id_rec.tsmalp_org_id, 8);
  
    FOR c_tsraal_id_rec IN c_tsraal_id(c_tsmalp_id_rec.tsmalp_is_number, c_tsmalp_id_rec.tsmalp_is_number, p_exp_log_seq) LOOP
      v_success := delete_result_children(c_tsraal_id_rec.tsrfdact_is_number, c_tsraal_id_rec.tsrfdact_org_id, p_exp_log_seq, NULL);
    
    END LOOP; -- c_tsraal_id_rec
  END LOOP; --c_tsmalp_id_rec

  DELETE tsmalp
   WHERE tsmstatn_is_number = p_tsmstatn_is_number
     AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);
  /* End tsmalp delete */

  -- Delete the TSRADL table and children
  FOR c_tsradl_id_rec IN c_tsradl_id(p_tsmstatn_is_number, p_tsmstatn_org_id) LOOP
    DELETE tsropprd
     WHERE tsradl_is_number = c_tsradl_id_rec.tsradl_is_number
       AND tsradl_org_id = rpad(c_tsradl_id_rec.tsradl_org_id, 8);
  END LOOP; --c_tsradl_id_rec

  DELETE tsradl
   WHERE tsmstatn_is_number = p_tsmstatn_is_number
     AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);
  /* End tsradl delete */

  -- Delete the tsmwell table and children
  FOR c_tsmwell_id_rec IN c_tsmwell_id(p_tsmstatn_is_number, p_tsmstatn_org_id) LOOP
    DELETE tsmaddr
     WHERE tsmwell_is_number = c_tsmwell_id_rec.tsmwell_is_number
       AND tsmwell_org_id = rpad(c_tsmwell_id_rec.tsmwell_org_id, 8);
  
    DELETE tsmlog
     WHERE tsmwell_is_number = c_tsmwell_id_rec.tsmwell_is_number
       AND tsmwell_org_id = rpad(c_tsmwell_id_rec.tsmwell_org_id, 8);
  
    DELETE tsmpump
     WHERE tsmwell_is_number = c_tsmwell_id_rec.tsmwell_is_number
       AND tsmwell_org_id = rpad(c_tsmwell_id_rec.tsmwell_org_id, 8);
  
    DELETE tsmwle
     WHERE tsmwell_is_number = c_tsmwell_id_rec.tsmwell_is_number
       AND tsmwell_org_id = rpad(c_tsmwell_id_rec.tsmwell_org_id, 8);
  
    -- Delete the tsmintvl children
    FOR c_interval_id_rec IN c_interval_id(c_tsmwell_id_rec.tsmwell_is_number, c_tsmwell_id_rec.tsmwell_org_id) LOOP
      DELETE tsmcsng
       WHERE tsmintvl_is_number = c_interval_id_rec.tsmintvl_is_number
         AND tsmintvl_org_id = rpad(c_interval_id_rec.tsmintvl_org_id, 8);
      DELETE tsmfill
       WHERE tsmintvl_is_number = c_interval_id_rec.tsmintvl_is_number
         AND tsmintvl_org_id = rpad(c_interval_id_rec.tsmintvl_org_id, 8);
      DELETE tsmhole
       WHERE tsmintvl_is_number = c_interval_id_rec.tsmintvl_is_number
         AND tsmintvl_org_id = rpad(c_interval_id_rec.tsmintvl_org_id, 8);
      DELETE tsmigua
       WHERE tsmintvl_is_number = c_interval_id_rec.tsmintvl_is_number
         AND tsmintvl_org_id = rpad(c_interval_id_rec.tsmintvl_org_id, 8);
      DELETE tsmopng
       WHERE tsmintvl_is_number = c_interval_id_rec.tsmintvl_is_number
         AND tsmintvl_org_id = rpad(c_interval_id_rec.tsmintvl_org_id, 8);
    END LOOP; --c_interval_id_rec
  
    DELETE tsmintvl
     WHERE tsmwell_is_number = c_tsmwell_id_rec.tsmwell_is_number
       AND tsmwell_org_id = rpad(c_tsmwell_id_rec.tsmwell_org_id, 8);
  
  END LOOP; --c_tsmwell_id_rec

  DELETE tsmwell
   WHERE tsmstatn_is_number = p_tsmstatn_is_number
     AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);
  /* End tsmwell delete */

  /* Delete Facility record*/
  DELETE tsmfclty
   WHERE tsmstatn_is_number = p_tsmstatn_is_number
     AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  COMMIT;
  RETURN('Y');
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in DELETE_STATION_CHILDREN');
    RETURN('N');
  
END delete_station_children;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE clear_station_export
(
  p_export_log_seq    IN NUMBER,
  p_remove_by_file_yn IN VARCHAR2,
  p_success           OUT NUMBER
) IS

  CURSOR c_export_org(p_export_log_seq IN NUMBER) IS
    SELECT sel_org_id
      FROM sim_export_log
     WHERE sel_seq = p_export_log_seq;
  
  /*Changed to use sim_export_storet_ids table 10/24/05- sjj*/    
  CURSOR c_stations(p_export_log_seq IN NUMBER) IS
    SELECT sesi_is_number,
           sesi_org_id
      FROM sim_export_storet_ids
     WHERE sesi_sel_seq = p_export_log_seq;      

  CURSOR c_abs_loc_point(p_sta_is_number IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT tsmalp_is_number,
           tsmalp_org_id
      FROM tsmalp
     WHERE tsmstatn0is_number = p_sta_is_number
       AND tsmstatn0org_id = rpad(p_org_id, 8);
 
  /*Changed to use sim_export_storet_ids table 10/24/05- sjj*/     
  CURSOR c_statn_visit(p_export_log_seq IN NUMBER) IS
    SELECT v.tsrstvst_is_number
      FROM tsmstatn s,
           tsrstvst v,
           sim_export_storet_ids ss
     WHERE s.tsmstatn_is_number = v.tsmstatn_is_number
       AND s.tsmstatn_org_id = v.tsmstatn_org_id
       AND s.tsmstatn_is_number = ss.sesi_is_number
       AND v.tsmstatn_org_id = rpad(ss.sesi_org_id, 8)
       AND ss.sesi_sel_seq = p_export_log_seq;

 /*Changed to use sim_export_storet_ids table 10/24/05- sjj*/ 
  CURSOR c_well_on_station(p_export_log_seq IN NUMBER) IS
     SELECT w.tsmwell_is_number
       FROM tsmwell w,
            tsmstatn s,
            sim_export_storet_ids ss
      WHERE w.tsmstatn_is_number = s.tsmstatn_is_number
        AND w.tsmstatn_org_id = s.tsmstatn_org_id
        AND s.tsmstatn_is_number = ss.sesi_is_number
        AND s.tsmstatn_org_id = rpad(ss.sesi_org_id, 8)
        AND ss.sesi_sel_seq = p_export_log_seq;

  /*Changed to use sim_export_storet_ids table 10/24/05- sjj*/       
  CURSOR c_tsmalp_id(p_export_log_seq IN NUMBER) IS
    SELECT type_code,
           NAME
      FROM tsmalp a,
           tsmstatn s,
           sim_export_storet_ids ss
     WHERE s.tsmstatn_is_number = ss.sesi_is_number
       AND s.tsmstatn_org_id = rpad(ss.sesi_org_id, 8)
       AND a.tsmstatn0is_number = s.tsmstatn_is_number
       AND a.tsmstatn0org_id = s.tsmstatn_org_id
       AND ss.sesi_sel_seq = p_export_log_seq;          

  /*Changed to use sim_export_storet_ids table 10/24/05- sjj*/      
  CURSOR c_well(p_station_is_nbr IN NUMBER) IS
    SELECT tsmwell_is_number,
           tsmwell_org_id
      FROM tsmwell
     WHERE tsmstatn_is_number = p_station_is_nbr;

  CURSOR c_chk_storet_on_del IS
    SELECT sc_value
      FROM sim_config
     WHERE sc_item = 'CHK_STORET_ON_DEL';

  v_org_id               VARCHAR2(8);
  v_station_cnt          NUMBER(10) := 0;
  v_tsrstvst             NUMBER(12); -- SJJ-replaced v_station
  v_tsmwell_is_number    NUMBER(12); -- SJJ-replaced v_well_on_station
  v_station_child_delete VARCHAR2(1);
  v_chk_storet_on_del    VARCHAR2(5);
  v_delete_ok            VARCHAR2(1);

  e_visitsfound        EXCEPTION;
  e_no_delete          EXCEPTION;
  e_child_delete_error EXCEPTION;

BEGIN

  /* bka - Check to make sure that no stations from this migration already have visits assigned to them in STORET */
  OPEN c_statn_visit(p_export_log_seq);
  FETCH c_statn_visit
    INTO v_tsrstvst;
  IF c_statn_visit%FOUND THEN
    sim_event_log_pkg.write_event_detail(
      3, 
      NULL, 
      'Station ' || v_tsrstvst || ' has station visits assigned to it.  '
        || 'This migration cannot be removed from STORET');
    RAISE e_visitsfound;
  END IF;
  CLOSE c_statn_visit;
  OPEN c_well_on_station(p_export_log_seq);
  FETCH c_well_on_station
    INTO v_tsmwell_is_number;
  IF c_well_on_station%FOUND THEN     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Station has at least one well, ' ||
                                         rtrim(v_tsmwell_is_number) ||
                                         ', assigned to it. Please remove WELLS migration from STORET first.');
    RAISE e_visitsfound;
  END IF;
  CLOSE c_well_on_station;

  FOR c_tsmalp_id_rec IN c_tsmalp_id(p_export_log_seq) LOOP
    IF c_tsmalp_id_rec.type_code <> '*POINT OF RECORD' THEN       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           'Station ' || c_tsmalp_id_rec.name ||
                                           ' has at least one Location other than *POINT OF RECORD, assigned to it. Please remove Location migration from STORET first.');
      RAISE e_visitsfound;
    END IF;
  END LOOP;

  OPEN c_chk_storet_on_del;
  FETCH c_chk_storet_on_del
    INTO v_chk_storet_on_del;
  CLOSE c_chk_storet_on_del;

  IF v_chk_storet_on_del IS NULL
     OR upper(v_chk_storet_on_del) = 'YES' THEN
    /* Check if there are any child records that have not been exported by SIM */
    FOR v_stations IN c_stations(p_export_log_seq) LOOP
      -- only check if normal clear export skip if remove by file
      IF p_remove_by_file_yn = 'N' THEN
        v_delete_ok := chk_if_station_delete(v_stations.sesi_is_number, v_stations.sesi_org_id, sim_export_pkg.v_pkg_ownerid, p_export_log_seq);
      
        IF v_delete_ok = 'N' THEN
          RAISE e_no_delete;
        END IF;
      END IF;
    END LOOP; -- v_stations
  END IF;

  /* Get the org id, so can pass to seq update process */
  OPEN c_export_org(p_export_log_seq);
  FETCH c_export_org
    INTO v_org_id;
  CLOSE c_export_org;

  FOR v_stations IN c_stations(p_export_log_seq) LOOP
  
    v_station_child_delete := delete_station_children(v_stations.sesi_is_number, v_stations.sesi_org_id, p_export_log_seq);
  
    IF v_station_child_delete = 'N' THEN
      RAISE e_child_delete_error;
    END IF;
  
    FOR v_well IN c_well(v_stations.sesi_is_number) LOOP
    
      DELETE tsmwell
       WHERE tsmwell_is_number = v_well.tsmwell_is_number
         AND tsmwell_org_id = rpad(v_stations.sesi_org_id, 8);
     
    END LOOP;
  
    /* delete ocean, lake, and estuary location data associated with migrated stations */
    FOR v_abs_loc_point IN c_abs_loc_point(v_stations.sesi_is_number, v_stations.sesi_org_id) LOOP
      DELETE tsmocnlc
       WHERE tsmalp_org_id = rpad(v_abs_loc_point.tsmalp_org_id, 8)
         AND tsmalp_is_number = v_abs_loc_point.tsmalp_is_number;
      DELETE tsmestlc
       WHERE tsmalp_org_id = rpad(v_abs_loc_point.tsmalp_org_id, 8)
         AND tsmalp_is_number = v_abs_loc_point.tsmalp_is_number;
      DELETE tsmgll
       WHERE tsmalp_org_id = rpad(v_abs_loc_point.tsmalp_org_id, 8)
         AND tsmalp_is_number = v_abs_loc_point.tsmalp_is_number;
    
    END LOOP;
  
    DELETE tsmalp
     WHERE tsmstatn0org_id = rpad(v_stations.sesi_org_id, 8)
       AND tsmstatn0is_number = v_stations.sesi_is_number;
    
    DELETE tsmpsa
     WHERE tsmstatn_is_number = v_stations.sesi_is_number
       AND tsmstatn_org_id = rpad(v_stations.sesi_org_id, 8);
    
    DELETE tsmstatn
     WHERE tsmstatn_is_number = v_stations.sesi_is_number
       AND tsmstatn_org_id = rpad(v_stations.sesi_org_id, 8);
   
    v_station_cnt := v_station_cnt + 1;
  
    COMMIT;
  
  END LOOP;

  -- This first delete statement needs to be removed/changed for event logging SJJ
  DELETE FROM sim_export_storet_ids
    WHERE sesi_sel_seq = p_export_log_seq;
    
  DELETE sim_export_log
   WHERE sel_seq = p_export_log_seq;
   
  COMMIT;
  p_success := 1;
  
EXCEPTION
  WHEN e_visitsfound THEN
    p_success := 0;
    RAISE;
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
      SQLERRM || ' - in clear_station_export');
    p_success := 0;
  
END clear_station_export;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE clear_well_export(p_export_log_seq IN NUMBER) IS

  CURSOR c_export_org(p_export_log_seq IN NUMBER) IS
    SELECT sel_org_id
      FROM sim_export_log
     WHERE sel_seq = p_export_log_seq;

  v_org_id     VARCHAR2(8);
  v_well_cnt   NUMBER(10) := 0;
  v_well       VARCHAR2(60);
  e_visitsfound EXCEPTION;

BEGIN

  /* Get the org id, so can pass to seq update process */
  OPEN c_export_org(p_export_log_seq);
  FETCH c_export_org
    INTO v_org_id;
  CLOSE c_export_org;

  FOR v_well IN (SELECT sesi_is_number tsmwell_is_number,
                        sesi_org_id tsmwell_org_id
                             FROM sim_export_storet_ids
                             WHERE sesi_sel_seq = p_export_log_seq) LOOP
    DELETE tsmalp
     WHERE tsmwell_is_number = v_well.tsmwell_is_number
       AND tsmwell_org_id = rpad(v_well.tsmwell_org_id, 8);
    DELETE tsmwell
     WHERE tsmwell_is_number = v_well.tsmwell_is_number
       AND tsmwell_org_id = rpad(v_well.tsmwell_org_id, 8);
    
    v_well_cnt := v_well_cnt + 1;
    IF v_well_cnt = 100 THEN
      COMMIT;
    END IF;
  END LOOP;

  /*This first delete statement needs to be removed/changed for event logging -SJJ*/
  
  --sjj Added 10/24/05 
  DELETE FROM sim_export_storet_ids
    WHERE sesi_sel_seq = p_export_log_seq;
     
  DELETE sim_export_log
   WHERE sel_seq = p_export_log_seq;

  COMMIT;

EXCEPTION
  WHEN OTHERS THEN    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in Clear_well_export');
 
END clear_well_export;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE clear_location_export(p_export_log_seq IN NUMBER) IS

  CURSOR c_export_org(p_export_log_seq IN NUMBER) IS
    SELECT sel_org_id
      FROM sim_export_log
     WHERE sel_seq = p_export_log_seq;

  v_org_id       VARCHAR2(8);
  v_location_cnt NUMBER(10) := 0;

BEGIN

  /* Get the org id, so can pass to seq update process */
  OPEN c_export_org(p_export_log_seq);
  FETCH c_export_org
    INTO v_org_id;
  CLOSE c_export_org;

  FOR v_location IN (SELECT sesi_is_number,
                            sesi_org_id
                       FROM sim_export_storet_ids
                      WHERE sesi_sel_seq = p_export_log_seq) LOOP
  
    DELETE tsmalp
     WHERE tsmalp_is_number = v_location.sesi_is_number
       AND tsmalp_org_id = rpad(v_location.sesi_org_id, 8);
  
   v_location_cnt := v_location_cnt + 1;
    IF v_location_cnt = 100 THEN
      COMMIT;
    END IF;
  END LOOP;

  /*This first delete statement needs to be removed/changed for event logging -SJJ*/
  --SJJ 10/24/05 
  DELETE FROM sim_export_storet_ids
    WHERE sesi_sel_seq = p_export_log_seq; 
  
  DELETE sim_export_log
   WHERE sel_seq = p_export_log_seq;

  COMMIT;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in Clear_location_export');
     
END clear_location_export;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE clear_project_export(p_export_log_seq IN NUMBER,
                               p_success        OUT NUMBER) IS

  v_project_cnt               NUMBER(10) := 0;
  v_project_is_number         sim_export_storet_ids.sesi_is_number%TYPE;
  v_activity_exists           NUMBER(1);
  v_specimen_exists           NUMBER(1);
  v_operating_period_exists   NUMBER(1);
  e_activity_found            EXCEPTION;
  e_specimen_found            EXCEPTION;
  e_operating_period_found    EXCEPTION;
  
  
BEGIN

  FOR v_project_export IN (SELECT sesi_is_number tsmproj_is_number,
                                  sesi_org_id tsmproj_org_id
                             FROM sim_export_storet_ids
                             WHERE sesi_sel_seq = p_export_log_seq) LOOP
    
    v_project_is_number := v_project_export.tsmproj_is_number;
    
    /* check to see if activities belong to this project and reject delete if exists */
    SELECT COUNT(*)
      INTO v_activity_exists
      FROM tsrfapra t
     WHERE t.tsmproj_is_number = v_project_is_number
       AND t.tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
       
    IF nvl(v_activity_exists,0) > 0 THEN
      RAISE e_activity_found;
    END IF;
    
    /* check to see if voucher specimen collection is setup in STORET and reject delete if exists */
    SELECT COUNT(*)
      INTO v_specimen_exists
      FROM tsmvsc t
     WHERE t.tsmproj_is_number = v_project_is_number
       AND t.tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
       
    IF nvl(v_specimen_exists,0) > 0 THEN
      RAISE e_specimen_found;
    END IF;
    
    /* check to see if data logger operating period exists on project and reject delete if exists */
    SELECT COUNT(*)
      INTO v_operating_period_exists
      FROM tsrpopa t
     WHERE t.tsmproj_is_number = v_project_is_number
       AND t.tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
       
    IF nvl(v_operating_period_exists,0) > 0 THEN
      RAISE e_operating_period_found;
    END IF;
    
    /* delete tsmblob for each project record */
    DELETE tsmblob
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);

    /* delete the tsmgntxt for each project record */
    DELETE tsmgntxt
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
    
    /* delete tsmpcoa for each project record */
    DELETE tsmpcoa
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);  
         
    /* delete project, station association */
    DELETE tsmpsa
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
  
    /* delete project, person association */
    DELETE tsmppa
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
  
    /* delete project, person, role association */
    DELETE tsmppra
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
    
    /* delete tsmprpra for each project record */
    DELETE tsmprpra
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);  
    
    /* delete tsmpsw for each project record */
    DELETE tsmpsw
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);  
    
    /* delete tsrcpa for each project record */
    DELETE tsrcpa
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);  
    
    /* delete tsrtpa for each project record */
    DELETE tsrtpa
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);  
                      
    /* delete the project */
    DELETE tsmproj
     WHERE tsmproj_is_number = v_project_is_number
       AND tsmproj_org_id = rpad(v_project_export.tsmproj_org_id, 8);
  
    COMMIT;
    p_success := 1;
  
    v_project_cnt := v_project_cnt + 1;
  
  END LOOP;

  /*This first delete statement needs to be removed/changed for event logging -SJJ*/
 
  DELETE FROM sim_export_storet_ids
    WHERE sesi_sel_seq = p_export_log_seq;
  
  DELETE sim_export_log
   WHERE sel_seq = p_export_log_seq;

  COMMIT;

EXCEPTION
  WHEN e_activity_found THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Project '||v_project_is_number|| 
                                         ' has activites assigned to it.  This migration cannot be removed from STORET');
    p_success := 0;
    RAISE;
  WHEN e_specimen_found THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Project '||v_project_is_number||
                                         ' has voucher specimens assigned to it.  You must remove these voucher specimens in STORET before this project can be removed');
    
    p_success := 0;RAISE;
  WHEN e_operating_period_found THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Project '||v_project_is_number||
                                         'has data logger operating period(s) assigned to it.  You must remove these assignments in STORET before this project can be removed');
    p_success := 0;
    RAISE;
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in Clear_project_export');  
    p_success := 0;
END clear_project_export;

/****************************************************************************************
  Delete all data from STORET that's part of one migration
 ****************************************************************************************/ 
FUNCTION clear_export (
  p_export_log_seq    IN NUMBER, 
  p_remove_by_file_yn IN VARCHAR2 := 'N') 
  
  RETURN NUMBER IS

  v_export_type VARCHAR2(4);
  v_success     NUMBER(1) := 1;
  v_sel_org_id  CHAR(8);  

BEGIN
  SELECT sel_org_id, sel_export_type
    INTO v_sel_org_id, v_export_type
    FROM sim_export_log sel
    WHERE sel_seq = p_export_log_seq;
  IF p_remove_by_file_yn = 'N' THEN
    -- create event
    sim_event_log_pkg.write_event(
      sim_lib_pkg.c_selt_delete_migration,
      p_export_log_seq,
      v_sel_org_id,
      NULL,
      NULL);
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
      'Start: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  END IF;
  IF v_export_type = 'S' THEN
    clear_station_export(p_export_log_seq, p_remove_by_file_yn, v_success);
  ELSIF v_export_type = 'W' THEN
    clear_well_export(p_export_log_seq);
  ELSIF v_export_type = 'L' THEN
    clear_location_export(p_export_log_seq);
  ELSIF v_export_type = 'P' THEN
    clear_project_export(p_export_log_seq, v_success);
  ELSIF v_export_type IN ('R','B','XR') THEN
    clear_storet_export(p_export_log_seq, p_remove_by_file_yn, v_success);
  END IF;
  IF p_remove_by_file_yn = 'N' THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
      'Finish at: ' || to_char(SYSDATE, 'HH24:MI:SS'));                                  
    sim_event_log_pkg.update_stop_time;
  END IF;     
  RETURN v_success;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.log_error(SQLERRM,'sim_export_pkg.clear_export');
    RETURN 0;
  
END clear_export;

/****************************************************************************************
 * Remove all data from STORET that came from a SIM Import/Migrate for one particular Org.
 ****************************************************************************************/
FUNCTION clear_export_all
(
  p_org_id           IN sim_export_log.sel_org_id%TYPE,
  p_include_projects IN BOOLEAN := FALSE
)

 RETURN NUMBER IS

BEGIN
  FOR rec IN (SELECT sel.sel_seq,
                     sel.sel_export_type
                FROM sim_export_log sel
                WHERE sel.sel_org_id = p_org_id
                ORDER BY sel.sel_exported_on DESC) LOOP
    IF rec.sel_export_type <> 'P'
       OR p_include_projects = TRUE THEN
      IF clear_export(rec.sel_seq) = 0 THEN
        RETURN 0;
      END IF;
    END IF;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
  
END clear_export_all;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_file_type(p_path IN VARCHAR2) RETURN VARCHAR2 IS

BEGIN

  IF p_path IS NULL THEN
    RETURN NULL;
  END IF;

  RETURN upper(substr(p_path, length(p_path) - 2));

END get_file_type;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_file_name(p_path IN VARCHAR2) RETURN VARCHAR2 IS

BEGIN

  IF p_path IS NULL THEN
    RETURN NULL;
  END IF;

  RETURN upper(substr(p_path, 1, length(p_path) - 4));

END get_file_name;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_blob_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_blob_is_nbr NUMBER(8);

BEGIN
  /* get the next avail is nbr to use */
  v_next_blob_is_nbr := sim_export_pkg.get_eisn_number('TSMBLOB', p_org_id);

  RETURN v_next_blob_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next blob is number.');
     
       
    RETURN NULL;
  
END get_next_blob_is_number;

/****************************************************************************************
  Insert the binary contents of a file into the TSMBLOB table
 ****************************************************************************************/ 
FUNCTION insert_binary_document (
  p_export_log_seq      IN NUMBER,
  p_table_name          IN VARCHAR2,
  p_userid_code         IN CHAR,
  p_file_name           IN VARCHAR2,
  p_tsmproj_is_number   IN sim_projects.spj_tsmproj_is_number%TYPE := NULL,
  p_tsmproj_org_id      IN sim_projects.spj_tsmproj_org_id%TYPE := NULL,
  p_tsrrsult_is_number  IN sim_result_class_indicator.rci_tsrrsult_is_number%TYPE := NULL,  
  p_tsrrsult_org_id     IN sim_result_class_indicator.rci_tsrrsult_org_id%TYPE := NULL,
  p_tsmstatn_is_number  IN sim_stations.sta_is_number%TYPE := NULL,
  p_tsmstatn_org_id     IN sim_stations.sta_org_id%TYPE := NULL,
  p_tsrstvst_is_number  IN sim_station_visits.ssv_tsrstvst_is_number%TYPE := NULL,
  p_tsrstvst_org_id     IN sim_station_visits.ssv_tsrstvst_org_id%TYPE := NULL,
  p_tsrfdact_is_number  IN sim_field_activities.fa_tsrfdact_is_number%TYPE := NULL,  
  p_tsrfdact_org_id     IN sim_field_activities.fa_tsrfdact_org_id%TYPE := NULL) 
  
  RETURN NUMBER IS

  v_is_number   NUMBER(8);
  v_blob        BLOB;
  v_org_id      tsmblob.tsmblob_org_id%TYPE;
  v_file_type   utl_file.file_type;
  v_bfile       BFILE;
  v_dml         VARCHAR2(400);
  v_dir         VARCHAR2(30);
  
BEGIN
  -- create oracle directory to read the document into blob in STORET
  v_dir := 'DOCUMENT_DIR_' || substr(USER, 1, 17);
  v_dml := 'CREATE OR REPLACE DIRECTORY ' || v_dir || ' AS ''' || sim_lib_pkg.get_document_import_path || ''''; 
  EXECUTE IMMEDIATE v_dml;
  -- grant user read priv. on directory
  v_dml := 'GRANT READ ON DIRECTORY ' || v_dir || ' TO ' || USER;
  EXECUTE IMMEDIATE v_dml;
  -- get the org. ID
  IF p_tsmproj_org_id IS NOT NULL THEN
    v_org_id := p_tsmproj_org_id;
  ELSIF p_tsrrsult_org_id IS NOT NULL THEN
    v_org_id := p_tsrrsult_org_id;
  ELSIF p_tsmstatn_org_id IS NOT NULL THEN
    v_org_id := p_tsmstatn_org_id;
  ELSIF p_tsrstvst_org_id IS NOT NULL THEN
    v_org_id := p_tsrstvst_org_id;
  ELSIF p_tsrfdact_org_id IS NOT NULL THEN
    v_org_id := p_tsrfdact_org_id;
  ELSE
    v_org_id := NULL;
  END IF;  
  v_is_number := get_next_blob_is_number(v_org_id);
  INSERT INTO tsmblob (
      tsmblob_is_number, 
      tsmblob_org_id,  
      table_name,
      tsmproj_is_number,
      tsmproj_org_id,
      tsrrsult_is_number,  
      tsrrsult_org_id,
      tsmstatn_is_number,
      tsmstatn_org_id,
      tsrstvst_is_number,
      tsrstvst_org_id,
      tsrfdact_is_number,
      tsrfdact_org_id,
      d_userid_code,
      d_last_update_ts, 
      blob_content)
    VALUES (
      v_is_number,
      v_org_id,
      p_table_name,
      p_tsmproj_is_number,
      p_tsmproj_org_id,
      p_tsrrsult_is_number,  
      p_tsrrsult_org_id,
      p_tsmstatn_is_number,
      p_tsmstatn_org_id,
      p_tsrstvst_is_number,
      p_tsrstvst_org_id,
      p_tsrfdact_is_number,
      p_tsrfdact_org_id,
      p_userid_code,
      SYSDATE, 
      empty_blob()) 
    RETURN blob_content 
    INTO v_blob;    

  BEGIN
    v_bfile := bfilename(v_dir, p_file_name);
    dbms_lob.fileopen(v_bfile, dbms_lob.file_readonly);
    dbms_lob.loadfromfile(v_blob, v_bfile, dbms_lob.getlength(v_bfile));
    dbms_lob.fileclose(v_bfile);
    
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.log_error(SQLERRM ||' Unable to load Document/Graphic file ' 
        || '"' || p_file_name || '".  Please check directory and file permissions.');
      RETURN 0;
      
  END;
  -- drop directory now (to keep things clean)
  v_dml := 'DROP DIRECTORY ' || v_dir;
  EXECUTE IMMEDIATE v_dml;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.log_error(SQLERRM || ' - in insert_binary_document.');
    RETURN 0;
  
END insert_binary_document;

/****************************************************************************************
 * Display counts of exported records prior to removing export from Storet              *
 ****************************************************************************************/ 
PROCEDURE get_clear_export_status
(
  p_export_log_seq IN NUMBER,
  p_export_type    IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fnd_cnt        OUT NUMBER,
  p_notfnd_cnt     OUT NUMBER
) IS

  v_sesi_cnt   NUMBER;
  v_storet_cnt NUMBER;
  v_qc_cnt     NUMBER;
  
BEGIN
  IF p_export_type = 'P' THEN
    SELECT COUNT(*)
      INTO v_sesi_cnt
      FROM sim_export_storet_ids
     WHERE sesi_table_name = 'TSMPROJ'
       AND sesi_sel_seq = p_export_log_seq;
  
    SELECT COUNT(*)
      INTO v_storet_cnt
      FROM tsmproj
     WHERE tsmproj_is_number IN
           (SELECT sesi_is_number
              FROM sim_export_storet_ids
             WHERE sesi_table_name = 'TSMPROJ'
               AND sesi_sel_seq = p_export_log_seq)
        AND tsmproj_org_id = rpad(p_org_id,8);
  
    IF v_sesi_cnt <> v_storet_cnt THEN
      p_fnd_cnt := v_storet_cnt;
      p_notfnd_cnt := (v_sesi_cnt - v_storet_cnt);
    ELSE
      p_fnd_cnt := v_sesi_cnt;
      p_notfnd_cnt := 0;
    END IF;
  ELSIF p_export_type = 'S' THEN
    SELECT COUNT(*)
      INTO v_sesi_cnt
      FROM sim_export_storet_ids
     WHERE sesi_table_name = 'TSMSTATN'
       AND sesi_sel_seq = p_export_log_seq;
  
    SELECT COUNT(*)
      INTO v_storet_cnt
      FROM tsmstatn
     WHERE tsmstatn_is_number IN
           (SELECT sesi_is_number
              FROM sim_export_storet_ids
             WHERE sesi_table_name = 'TSMSTATN'
               AND sesi_sel_seq = p_export_log_seq)
        AND tsmstatn_org_id = rpad(p_org_id,8);
  
    IF v_sesi_cnt <> v_storet_cnt THEN
      p_fnd_cnt := v_storet_cnt;
      p_notfnd_cnt := (v_sesi_cnt - v_storet_cnt);
    ELSE
      p_fnd_cnt := v_sesi_cnt;
      p_notfnd_cnt := 0;
    END IF;
  ELSIF p_export_type = 'W' THEN
    SELECT COUNT(*)
      INTO v_sesi_cnt
      FROM sim_export_storet_ids
     WHERE sesi_table_name = 'TSMWELL'
       AND sesi_sel_seq = p_export_log_seq;
  
    SELECT COUNT(*)
      INTO v_storet_cnt
      FROM tsmwell
     WHERE tsmwell_is_number IN
           (SELECT sesi_is_number
              FROM sim_export_storet_ids
             WHERE sesi_table_name = 'TSMWELL'
               AND sesi_sel_seq = p_export_log_seq)
        AND tsmwell_org_id = rpad(p_org_id,8);
  
    IF v_sesi_cnt <> v_storet_cnt THEN
      p_fnd_cnt := v_storet_cnt;
      p_notfnd_cnt := (v_sesi_cnt - v_storet_cnt);
    ELSE
      p_fnd_cnt := v_sesi_cnt;
      p_notfnd_cnt := 0;
    END IF;
  ELSIF p_export_type = 'L' THEN
    SELECT COUNT(*)
      INTO v_sesi_cnt
      FROM sim_export_storet_ids
     WHERE sesi_table_name = 'TSMALP'
       AND sesi_sel_seq = p_export_log_seq;
  
    SELECT COUNT(*)
      INTO v_storet_cnt
      FROM tsmalp
     WHERE tsmalp_is_number IN
           (SELECT sesi_is_number
              FROM sim_export_storet_ids
             WHERE sesi_table_name = 'TSMALP'
               AND sesi_sel_seq = p_export_log_seq)
       AND tsmalp_org_id = rpad(p_org_id,8);
  
    IF v_sesi_cnt != v_storet_cnt THEN
      p_fnd_cnt := v_storet_cnt;
      p_notfnd_cnt := (v_sesi_cnt - v_storet_cnt);
    ELSE
      p_fnd_cnt := v_sesi_cnt;
      p_notfnd_cnt := 0;
    END IF;
    
  ELSIF p_export_type IN ('R','B','XR') THEN
    SELECT COUNT(*)
      INTO v_sesi_cnt
      FROM sim_export_storet_ids
     WHERE sesi_sel_seq = p_export_log_seq;
  
    SELECT COUNT(*)
      INTO v_storet_cnt
      FROM tsrfdact act, sim_export_storet_ids sesi
     WHERE act.tsrfdact_is_number = sesi.sesi_is_number
       AND act.tsrfdact_org_id = rpad(p_org_id,8)
       AND sesi.sesi_table_name = 'TSRFDACT'
       AND sesi.sesi_sel_seq = p_export_log_seq;
       
    SELECT COUNT(*)
      INTO v_qc_cnt
      FROM tsrfqs qc, sim_export_storet_ids sesi
      WHERE qc.tsrfqs_is_number = sesi.sesi_is_number
      AND qc.tsrfqs_org_id = rpad(p_org_id,8)
      AND sesi.sesi_table_name = 'TSRFQS'
      AND sesi.sesi_sel_seq = p_export_log_seq;
     
    p_fnd_cnt := v_qc_cnt + v_storet_cnt;
    p_notfnd_cnt := v_sesi_cnt - p_fnd_cnt;
    
  END IF;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get clear export status.'); 
END get_clear_export_status;

/****************************************************************************************
* Procedure to reuse code to insert into the sim_export_storet_ids table                                                                                     *
****************************************************************************************/
PROCEDURE insert_into_storet_ids(
  p_export_log_seq    IN NUMBER,
  p_storet_table_name IN VARCHAR2,
  p_org_id            IN VARCHAR2,
  p_is_number         IN NUMBER) IS
  
BEGIN

  INSERT INTO sim_export_storet_ids(
    SESI_SEQ,
    SESI_SEL_SEQ,
    SESI_TABLE_NAME,
    SESI_ORG_ID,
	  SESI_IS_NUMBER)
  VALUES (
    SIM_EXPORT_STORET_IDS_SEQ.NEXTVAL,
    p_export_log_seq,
    p_storet_table_name,
    p_org_id,
    p_is_number);
    
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in insert into storet ids.'); 
END insert_into_storet_ids;

/****************************************************************************************
 Procedure to remove records from Storet using a text file.
 ****************************************************************************************/
FUNCTION get_storet_ids_from_file(
  p_export_log_seq    IN NUMBER,
  p_event_log_seq     IN NUMBER,
  p_org_id            IN VARCHAR2,
  p_export_type       IN VARCHAR2,
  p_delimiter         IN VARCHAR2,
  p_skip_header       IN NUMBER := 0) RETURN NUMBER IS
  
  v_parsed_line        sim_lib_pkg.t_varchar_table;
  v_storet_table_name   VARCHAR2(8);
  v_index               NUMBER(1);
  v_is_number           NUMBER(12);
  v_file_id             VARCHAR2(30);
  v_station_id          tsmstatn.identification_cd%TYPE;
  v_seq_nbr             tsmalp.sequence_number%TYPE;
  v_type_code           tsmalp.type_code%TYPE;
  v_id_number           tsmwell.id_number%TYPE;
  v_trip_id             tsrtrip.id_code%TYPE;
  v_visit_id            tsrstvst.id_number%TYPE;
  v_activity_id         tsrfdact.id_code%TYPE;
  v_replicate_id        tsrfdact.replicate_number%TYPE;
  v_success             NUMBER(1) := 1;
  v_error_msg           VARCHAR2(200);
  v_line_no             NUMBER(8) := 0;
  v_skip_header         NUMBER(2) := p_skip_header;
  v_delimiter           VARCHAR2(5);
    
BEGIN
  IF p_delimiter IS NULL AND p_export_type NOT IN ('P', 'S') THEN
    raise_application_error(-20000,'Delimiter is required');
  END IF;
  IF p_delimiter = 'l' THEN
    v_delimiter := chr(9);
  ELSE
    v_delimiter := p_delimiter;
  END IF;
  sim_event_log_pkg.write_event(1,p_export_log_seq,p_org_id,NULL,NULL);
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Start: ' ||
                                         to_char(SYSDATE, 'HH24:MI:SS'));
  IF v_skip_header IS NULL THEN 
    v_skip_header := 0;
  END IF;                                         
  IF v_skip_header = 1 THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Header skipped');
    v_line_no := 1;
  END IF;
  -- Loop through each row in the lines table.
  FOR rec IN (SELECT sell_seq, sell_line_text, sell.sell_line_no
                FROM sim_event_log_lines sell
               WHERE sell_sevl_seq = p_event_log_seq
               ORDER BY sell.sell_line_no) LOOP
    
    BEGIN
      IF v_skip_header = 0
         OR rec.sell_line_no <> 1 THEN
        v_line_no := v_line_no + 1;
        -- parse one line into a plsql table of elements
        v_parsed_line := sim_lib_pkg.parse_to_table(rec.sell_line_text, v_delimiter);
        --------------- PROJECTS --------------------
        IF p_export_type = 'P' THEN 
          v_file_id := upper(v_parsed_line(1)); 
          BEGIN
            SELECT tsmproj_is_number 
              INTO v_is_number
              FROM tsmproj
              WHERE identification_cd = rpad(v_file_id,8)
                AND tsmproj_org_id = rpad(p_org_id, 8);
          EXCEPTION 
            WHEN no_data_found THEN
              -- use a dummy value (that won't be found)
              v_is_number := -1;
          END;
          v_storet_table_name := 'TSMPROJ';                                
          insert_into_storet_ids(
            p_export_log_seq,
            v_storet_table_name,
            p_org_id,
            v_is_number);   
        ---------------- STATIONS -------------------- 
        ELSIF p_export_type = 'S' THEN
          v_station_id := upper(v_parsed_line(1));
          BEGIN
            SELECT tsmstatn_is_number 
              INTO v_is_number
              FROM tsmstatn
              WHERE identification_cd = rpad(v_station_id,15)
                AND tsmstatn_org_id = rpad(p_org_id, 8);
          EXCEPTION 
            WHEN no_data_found THEN
              -- use a dummy value (that won't be found)
              v_is_number := -1;
          END;    
          v_storet_table_name := 'TSMSTATN';
          insert_into_storet_ids(
            p_export_log_seq,
            v_storet_table_name,
            p_org_id,
            v_is_number); 
        -------------- ADDITIONAL LOCATIONS -----------------  
        ELSIF p_export_type = 'L' THEN
          IF v_parsed_line.COUNT <> 3 THEN
            raise_application_error(-20000,'Wrong number of columns in file or wrong delimiter specified.');
          ELSE
            v_station_id  := upper(v_parsed_line(1));
            v_type_code   := upper(v_parsed_line(2));
            v_seq_nbr     := v_parsed_line(3);
            IF sim_import_pkg.check_allowable_values(v_type_code,10336,v_line_no) IS NULL THEN
              raise_application_error(-20000,v_type_code || ' is not an allowable value for Point Type');
            END IF;
            BEGIN                
              SELECT tsmalp_is_number 
                INTO v_is_number
                FROM tsmalp l, tsmstatn s
                WHERE l.tsmstatn0org_id = s.tsmstatn_org_id 
                AND l.tsmstatn0is_number = s.tsmstatn_is_number                      
                AND s.identification_cd = rpad(v_station_id,15)
                AND l.sequence_number = v_seq_nbr
                AND l.tsmalp_org_id = rpad(p_org_id, 8)
                AND l.type_code = rpad(v_type_code,16);
            EXCEPTION 
              WHEN no_data_found THEN
                -- use a dummy value (that won't be found)
                v_is_number := -1;
            END;
            v_storet_table_name := 'TSMALP';
            insert_into_storet_ids(
              p_export_log_seq,
              v_storet_table_name,
              p_org_id,
              v_is_number);    
          END IF;  
        ----------------------- WELLS ----------------------  
        ELSIF p_export_type = 'W' THEN     
          IF v_parsed_line.COUNT <> 2 THEN
            raise_application_error(-20000,'Wrong number of columns in file or wrong delimiter specified.');
          ELSE
            v_station_id := upper(v_parsed_line(1));
            v_id_number  := upper(v_parsed_line(2));
            BEGIN
              SELECT tsmwell_is_number 
                INTO v_is_number
                FROM tsmwell w, tsmstatn s
                WHERE w.tsmstatn_org_id = s.tsmstatn_org_id
                AND w.tsmstatn_is_number = s.tsmstatn_is_number
                AND s.identification_cd = rpad(v_station_id,15)
                AND w.id_number = rpad(v_id_number,15)
                AND w.tsmwell_org_id = rpad(p_org_id,8);             
            EXCEPTION 
              WHEN no_data_found THEN
                -- use a dummy value (that won't be found)
                v_is_number := -1;
            END; 
            v_storet_table_name := 'TSMWELL';        
            insert_into_storet_ids(
              p_export_log_seq,
              v_storet_table_name,
              p_org_id,
              v_is_number);
          END IF;
        ------------------ RESULTS/BIOLOGICAL -------------------  
        ELSIF p_export_type IN ('R', 'B') THEN
          IF v_parsed_line.COUNT <> 5 THEN
            raise_application_error(-20000,'Wrong number of columns in file or wrong delimiter specified.');
          ELSE
            v_trip_id      := upper(v_parsed_line(1));
            v_station_id   := upper(v_parsed_line(2));
            v_visit_id     := upper(v_parsed_line(3));
            v_activity_id  := upper(v_parsed_line(4));
            v_replicate_id := upper(v_parsed_line(5)); 
            BEGIN
              SELECT tsrfdact_is_number
                INTO v_is_number
                FROM tsrfdact a, tsrstvst v, tsmstatn s, tsrtrip t
                WHERE a.tsrstvst_is_number = v.tsrstvst_is_number
                  AND a.tsrstvst_org_id = v.tsrstvst_org_id
                  AND a.tsrfdact_org_id = rpad(p_org_id, 8)
                  AND v.id_number = rpad(v_visit_id,3)
                  AND v.tsmstatn_is_number = s.tsmstatn_is_number
                  AND v.tsmstatn_org_id = s.tsmstatn_org_id
                  AND s.identification_cd = rpad(v_station_id,15)
                  AND v.tsrtrip_is_number = t.tsrtrip_is_number
                  AND v.tsrtrip_org_id = t.tsrtrip_org_id
                  AND t.id_code = rpad(v_trip_id,15)
                  AND a.id_code = rpad(v_activity_id,12)
                  AND a.replicate_number = v_replicate_id;
            EXCEPTION 
              WHEN no_data_found THEN
                -- use a dummy value (that won't be found)
                v_is_number := -1;
            END;            
            v_storet_table_name := 'TSRFDACT';    
            insert_into_storet_ids(
              p_export_log_seq,
              v_storet_table_name,
              p_org_id,
              v_is_number); 
          END IF;          
        END IF;
      END IF;  
      
    EXCEPTION
      WHEN OTHERS THEN
        v_success := 0;
        sim_event_log_pkg.log_error(SQLERRM,'Line ' || v_line_no);
        
    END;
  END LOOP;
  COMMIT;
  RETURN v_success;
  
EXCEPTION          
  WHEN OTHERS THEN  
    sim_event_log_pkg.log_error(SQLERRM,'get_storet_ids_from_file');
    RETURN 0;                                          

END get_storet_ids_from_file;

END sim_export_pkg;
/
