CREATE OR REPLACE PACKAGE sim_export_project_pkg IS

/*--------------------------------------------------------------------------------------
  Export project data from SIM to STORET
  --------------------------------------------------------------------------------------*/
 
/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                               *
 ****************************************************************************************/

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_export_log_seq NUMBER(10);
  v_mig_success    NUMBER(1);
  
/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                           *
 ****************************************************************************************/
PROCEDURE export_projects (
  p_org_id     IN VARCHAR2,
  p_import_seq IN NUMBER,
  p_success    IN OUT NUMBER);

END sim_export_project_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_export_project_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                             *
 ****************************************************************************************/
  c_migrate_true    CONSTANT BOOLEAN := TRUE;
  c_new_event_no    CONSTANT VARCHAR2(1) := 'N';
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  
  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************
  Get the next IS Number from STORET for TSMPROJ
 ****************************************************************************************/ 
FUNCTION get_next_project_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_project_is_nbr NUMBER(8);

BEGIN
  v_next_project_is_nbr := sim_export_pkg.get_eisn_number('TSMPROJ', p_org_id);

  RETURN v_next_project_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - in get next project is number.');       
    v_mig_success := 0;
    RETURN NULL;
  
END get_next_project_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsmgntxt_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_tsmgntxt_is_nbr NUMBER(8);

BEGIN
  /* get the next avail is nbr to use */
  v_next_tsmgntxt_is_nbr := sim_export_pkg.get_eisn_number('TSMGNTXT', p_org_id);

  RETURN v_next_tsmgntxt_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - in get next tsmgntxt is number.');
    v_mig_success := 0;
    RETURN NULL;
  
END get_next_tsmgntxt_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsmgntxt_rec(p_tsmgntxt_rec IN tsmgntxt%ROWTYPE) RETURN NUMBER IS
  v_is_number NUMBER(8);
BEGIN

  v_is_number := get_next_tsmgntxt_is_number(p_tsmgntxt_rec.tsmgntxt_org_id);
  INSERT INTO tsmgntxt
    (tsmgntxt_is_number,
     tsmgntxt_org_id,
     table_name,
     description_name,
     description_text,
     d_userid_code,
     d_last_update_ts,
     tsrrsult_is_number,
     tsrrsult_org_id,
     tsmproj_is_number,
     tsmproj_org_id,
     tsmprogm_is_number,
     tsmprogm_org_id,
     tsrgrcfg_is_number,
     tsrgrcfg_org_id)
  VALUES
    (v_is_number, /* TSMGNTXT_IS_NUMBER */
     p_tsmgntxt_rec.tsmgntxt_org_id, /* TSMGNTXT_ORG_ID */
     p_tsmgntxt_rec.table_name, /* TABLE_NAME */
     p_tsmgntxt_rec.description_name, /* DESCRIPTION_NAME */
     p_tsmgntxt_rec.description_text, /* DESCRIPTION_TEXT */
     p_tsmgntxt_rec.d_userid_code, /* D_USERID_CODE */
     p_tsmgntxt_rec.d_last_update_ts, /* D_LAST_UPDATE_TS */
     p_tsmgntxt_rec.tsrrsult_is_number, /* TSRRSULT_IS_NUMBER */
     p_tsmgntxt_rec.tsrrsult_org_id, /* TSRRSULT_ORG_ID */
     p_tsmgntxt_rec.tsmproj_is_number, /* TSMPROJ_IS_NUMBER */
     p_tsmgntxt_rec.tsmproj_org_id, /* TSMPROJ_ORG_ID */
     p_tsmgntxt_rec.tsmprogm_is_number, /* TSMPROGM_IS_NUMBER */
     p_tsmgntxt_rec.tsmprogm_org_id, /* TSMPROGM_ORG_ID */
     p_tsmgntxt_rec.tsrgrcfg_is_number, /* TSRGRCFG_IS_NUMBER */
     p_tsmgntxt_rec.tsrgrcfg_org_id /* TSRGRCFG_ORG_ID */);

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                          ' - in create tsmgntxt rec.');
    v_mig_success := 0;
    RETURN 0;
  
END create_tsmgntxt_rec;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_project(p_project_rec IN tsmproj%ROWTYPE) RETURN NUMBER IS

BEGIN
  INSERT INTO tsmproj
    (tsmproj_is_number,
     tsmproj_org_id,
     identification_cd,
     NAME,
     start_date,
     planned_duration,
     d_userid_code,
     d_last_updt_ts,
     tsmorgan_is_number,
     blob_type,
     blob_title)
  VALUES
    (p_project_rec.tsmproj_is_number, /* TSMPROJ_IS_NUMBER */
     p_project_rec.tsmproj_org_id, /* TSMPROJ_ORG_ID */
     p_project_rec.identification_cd, /* IDENTIFICATION_CD */
     p_project_rec.NAME, /* NAME */
     p_project_rec.start_date, /* START_DATE */
     p_project_rec.planned_duration, /* PLANNED_DURATION */
     p_project_rec.d_userid_code, /* D_USERID_CODE */
     p_project_rec.d_last_updt_ts, /* D_LAST_UPDT_TS */
     p_project_rec.tsmorgan_is_number, /* TSMORGAN_IS_NUMBER */
     p_project_rec.blob_type, /* BLOB_TYPE */
     p_project_rec.blob_title /* BLOB_TITLE */);

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in create project.');
    v_mig_success := 0;
    RETURN 0;
  
END create_project;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE export_projects
(
  p_org_id     IN VARCHAR2,
  p_import_seq IN NUMBER,
  p_success    IN OUT NUMBER
) IS

  v_success NUMBER(1);

  v_project_cnt       NUMBER(10) := 0;
  v_project_is_nbr    NUMBER(8);
  v_proj_rec          tsmproj%ROWTYPE;
  v_proj_copy_rec     tsmproj%ROWTYPE;
  v_tsmgntxt_rec      tsmgntxt%ROWTYPE;
  v_tsmgntxt_copy_rec tsmgntxt%ROWTYPE;
  v_blob_success      NUMBER(1);
  v_rows_ready        NUMBER(10);
  v_sid_seq           NUMBER(10); 
  v_filename          VARCHAR2(265); 

  CURSOR c_get_projects(p_import_seq IN NUMBER) IS
    SELECT spj_seq,
           spj_id,
           spj_name,
           spj_start_date,
           spj_duration,
           spj_purpose,
           spj_contact,
           spj_org_id,
           spj_tsmorgan_is_number,
           spj_document
      FROM sim_projects
     WHERE spj_import_seq = p_import_seq
       AND spj_status = 'A';

BEGIN
  v_mig_success := 1;
  BEGIN
    v_export_log_seq := p_import_seq;
    SELECT substr(sil_import_file,instr(sil_import_file, '\', -1) +1)
      INTO v_filename
      FROM sim_import_log
      WHERE sil_seq = v_export_log_seq;
      
    INSERT INTO sim_export_log
      (sel_seq,
       sel_exported_by,
       sel_exported_on,
       sel_filename,
       sel_org_id,
       sel_export_type)
    VALUES
      (v_export_log_seq,
       USER,
       SYSDATE,
       v_filename,
       p_org_id,
       'P');
    COMMIT;
  
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           SQLERRM ||
                                          ' - setting up export log record.');  
      v_mig_success := 0;
      p_success := v_mig_success;
  END;
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = (SELECT sil_sicc_seq 
                       FROM sim_import_log 
                      WHERE sil_seq = p_import_seq);
  
  sim_event_log_pkg.write_event(4,v_export_log_seq,p_org_id,v_sid_seq,NULL);
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Start: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  /* loop through projects */
  FOR v_project IN c_get_projects(p_import_seq) LOOP
    /* create project record */
    BEGIN
      v_project_is_nbr := get_next_project_is_number(p_org_id);
      v_proj_rec.tsmproj_is_number := v_project_is_nbr;
      v_proj_rec.tsmproj_org_id := p_org_id;
      v_proj_rec.identification_cd := v_project.spj_id;
      v_proj_rec.NAME := v_project.spj_name;
      v_proj_rec.start_date := v_project.spj_start_date;
      v_proj_rec.planned_duration := v_project.spj_duration;
      v_proj_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
      v_proj_rec.d_last_updt_ts := SYSDATE;
      v_proj_rec.tsmorgan_is_number := v_project.spj_tsmorgan_is_number;
      v_proj_rec.blob_type := sim_export_pkg.get_file_type(v_project.spj_document);
      v_proj_rec.blob_title := sim_export_pkg.get_file_name(v_project.spj_document);
      v_success := create_project(v_proj_rec);
      v_project_cnt := v_project_cnt + 1;
    
    EXCEPTION
      WHEN OTHERS THEN 
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                              ' - creating project record.');   
        v_mig_success := 0;
        p_success := v_mig_success;
    END;
    /* create tsmgntxt record - project purpose */
    IF v_project.spj_purpose IS NOT NULL THEN
      BEGIN
        v_tsmgntxt_rec.tsmgntxt_org_id := p_org_id;
        v_tsmgntxt_rec.table_name := 'TSMPROJ';
        v_tsmgntxt_rec.description_name := 'PURPOSE';
        v_tsmgntxt_rec.description_text := v_project.spj_purpose;
        v_tsmgntxt_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
        v_tsmgntxt_rec.d_last_update_ts := SYSDATE;
        v_tsmgntxt_rec.tsrrsult_is_number := NULL;
        v_tsmgntxt_rec.tsrrsult_org_id := NULL;
        v_tsmgntxt_rec.tsmproj_is_number := v_project_is_nbr;
        v_tsmgntxt_rec.tsmproj_org_id := p_org_id;
        v_tsmgntxt_rec.tsmprogm_is_number := NULL;
        v_tsmgntxt_rec.tsmprogm_org_id := NULL;
        v_tsmgntxt_rec.tsrgrcfg_is_number := NULL;
        v_tsmgntxt_rec.tsrgrcfg_org_id := NULL;
        v_success := create_tsmgntxt_rec(v_tsmgntxt_rec);
      
      EXCEPTION
        WHEN OTHERS THEN
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM ||
                                                ' - creating project purpose record.');
          v_mig_success := 0;
          p_success := v_mig_success;
      END;
    END IF;
    /* create tsmgntxt record - project contact */
    IF v_project.spj_contact IS NOT NULL THEN
      BEGIN
        v_tsmgntxt_rec := v_tsmgntxt_copy_rec;
        v_tsmgntxt_rec.tsmgntxt_org_id := p_org_id;
        v_tsmgntxt_rec.table_name := 'TSMPROJ';
        v_tsmgntxt_rec.description_name := 'WORKPLAN';
        v_tsmgntxt_rec.description_text := v_project.spj_contact;
        v_tsmgntxt_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
        v_tsmgntxt_rec.d_last_update_ts := SYSDATE;
        v_tsmgntxt_rec.tsrrsult_is_number := NULL;
        v_tsmgntxt_rec.tsrrsult_org_id := NULL;
        v_tsmgntxt_rec.tsmproj_is_number := v_project_is_nbr;
        v_tsmgntxt_rec.tsmproj_org_id := p_org_id;
        v_tsmgntxt_rec.tsmprogm_is_number := NULL;
        v_tsmgntxt_rec.tsmprogm_org_id := NULL;
        v_tsmgntxt_rec.tsrgrcfg_is_number := NULL;
        v_tsmgntxt_rec.tsrgrcfg_org_id := NULL;
        v_success := create_tsmgntxt_rec(v_tsmgntxt_rec);
      
      EXCEPTION
        WHEN OTHERS THEN
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM ||
                                                ' - creating project contact record.');   
          v_mig_success := 0;
          p_success := v_mig_success;
      END;
    END IF;
    /* export binary document */
    IF v_proj_rec.blob_type IS NOT NULL THEN
      v_blob_success := sim_export_pkg.insert_binary_document(
        p_export_log_seq => v_export_log_seq, -- number
        p_table_name => 'TSMPROJ', -- varchar2
        p_userid_code => v_proj_rec.d_userid_code, -- char
        p_file_name => v_project.spj_document, -- varchar2
        p_tsmproj_is_number => v_proj_rec.tsmproj_is_number, -- number(12)
        p_tsmproj_org_id => v_proj_rec.tsmproj_org_id); -- char(8)
      IF v_blob_success = 0 THEN
        v_mig_success := 0;
        p_success := v_mig_success;
      END IF;
    END IF;
    /* update project table with project is number and org id, export info*/
    UPDATE sim_projects
       SET spj_tsmproj_is_number = v_project_is_nbr,
           spj_tsmproj_org_id = rpad(p_org_id, 8),
           spj_export_seq = v_export_log_seq,
           spj_status = 'E'
     WHERE spj_seq = v_project.spj_seq;
    /* wipe out records holding project info so they are clean for the next one */
    v_proj_rec := v_proj_copy_rec;
    v_tsmgntxt_rec := v_tsmgntxt_copy_rec;
  END LOOP;
  /* update export log with count */
  UPDATE sim_export_log
     SET sel_activity_cnt = v_project_cnt
   WHERE sel_seq = v_export_log_seq;
  /* update the import log with export info
  SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE sim_import_log
     SET sil_exported_on = SYSDATE,
         sil_exported_by = USER,
         sil_rows_ready = sil_rows_ready - v_project_cnt
   WHERE sil_seq = p_import_seq; 
  COMMIT;
  p_success := v_mig_success;
  IF p_success = 1 THEN
    SELECT sil_rows_ready
      INTO v_rows_ready
      FROM sim_import_log
      WHERE sil_seq = p_import_seq;
    IF v_rows_ready IS NULL OR v_rows_ready = 0 THEN
      sim_import_pkg.delete_import(p_import_seq, c_migrate_true, c_new_event_no);
    END IF;
  ELSE 
    /* If export is not successful still remove data from SIM*/
    sim_import_pkg.delete_import(p_import_seq, c_migrate_true, c_new_event_no);                                      
  END IF;
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Migrated ' || v_project_cnt ||
                                       ' projects.');
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Finish: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS')); 
  sim_event_log_pkg.update_stop_time;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         SQLERRM ||
                                         ' -  Error in sim_export_project_pkg.export_projects'); 
    v_mig_success := 0;
    p_success := v_mig_success;
  
END export_projects;

END sim_export_project_pkg;
/
