CREATE OR REPLACE PACKAGE sim_export_result_pkg IS

/*--------------------------------------------------------------------------------------
  Export Trips, Visits, Activities, and Results to STORET
  --------------------------------------------------------------------------------------*/
 
/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_export_log_seq NUMBER(10);
  v_stavst_no      NUMBER(3);
  v_last_station   VARCHAR2(15);

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ****************************************************************************************/

  FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2;

  FUNCTION get_uom_is_nbr
  (
    p_org_id          IN VARCHAR2,
    p_uom_name        IN VARCHAR2,
    p_chdef_is_number IN NUMBER
  ) RETURN NUMBER;

  FUNCTION validate_sample_frac
  (
    p_sample_frac_nm  IN VARCHAR2,
    p_tsrchdef_is_nbr IN NUMBER,
    p_tsrchar_is_nbr  IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION get_next_visit_is_number(p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_next_activity_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_next_result_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_value_msr(p_result_text IN VARCHAR2) RETURN NUMBER;

  PROCEDURE add_station_to_trip
  (
    p_trip_is_nbr IN NUMBER,
    p_sta_is_nbr  IN NUMBER,
    p_org_id      IN VARCHAR2
  );

  PROCEDURE add_project_to_trip
  (
    p_trip_is_nbr IN NUMBER,
    p_proj_is_nbr IN NUMBER,
    p_org_id      IN VARCHAR2
  );

  PROCEDURE sim_export
  (
    p_org_id     IN VARCHAR2,
    p_import_seq IN NUMBER,
    p_success    IN OUT NUMBER
  );

  PROCEDURE export_results
  (
    p_fa_seq          IN NUMBER,
    p_fat_desc        IN VARCHAR2,
    p_activity_is_nbr IN NUMBER,
    p_activity_org_id IN VARCHAR2,
    p_tripqc_is_nbr   IN NUMBER,
    p_tripqc_org_id   IN VARCHAR2,
    p_lab_is_number   IN NUMBER,
    p_lab_org_id      IN VARCHAR2,
    p_org_id          IN VARCHAR2,
    p_export_log_seq  IN NUMBER,
    p_act_result_cnt  OUT NUMBER,
    p_time_zone       IN VARCHAR2
  );

  PROCEDURE runtime;

END sim_export_result_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_export_result_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                             *
 ****************************************************************************************/
  c_migrate_true    CONSTANT BOOLEAN := TRUE;
  c_new_event_no    CONSTANT VARCHAR2(1) := 'N';
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  e_a         EXCEPTION;
  
  v_success   NUMBER(1);
  t_success   NUMBER(1) := 1;
  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS

  v_item_name VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

BEGIN

  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Could not find allowable value with sequence: ' ||
                                          p_alval_seq);
     
        
  END IF;
  CLOSE c_get_alval_name;

  RETURN v_item_name;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - getting allowable value sequence.');
     
        
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_allowable_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_tsmprmvl_value
(
  p_is_nbr IN NUMBER,
  p_org_id IN VARCHAR2
) RETURN VARCHAR2 IS

  v_tsmprmvl_name VARCHAR2(80);

  CURSOR c_get_tsmprmvl_name(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT field_value
      FROM tsmprmvl
     WHERE tsmprmvl_is_number = p_is_nbr
       AND tsmprmvl_org_id = rpad(p_org_id, 8);

BEGIN

  OPEN c_get_tsmprmvl_name(p_is_nbr, p_org_id);
  FETCH c_get_tsmprmvl_name
    INTO v_tsmprmvl_name;
  IF c_get_tsmprmvl_name%NOTFOUND THEN
    v_tsmprmvl_name := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Could not find TSMPRMVL value with COMMUNITY: ' ||
                                          p_is_nbr);
     
       
  END IF;
  CLOSE c_get_tsmprmvl_name;

  RETURN v_tsmprmvl_name;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - getting tsmprmvl value.');
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_tsmprmvl_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_tsrcpv_value
(
  p_is_nbr IN NUMBER,
  p_org_id IN VARCHAR2
) RETURN VARCHAR2 IS

  v_tsrcpv_name VARCHAR2(80);

  CURSOR c_get_tsrcpv_name(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT short_name
      FROM tsrcpv
     WHERE tsrcpv_is_number = p_is_nbr
       AND tsrcpv_org_id = rpad(p_org_id, 8);

BEGIN

  OPEN c_get_tsrcpv_name(p_is_nbr, p_org_id);
  FETCH c_get_tsrcpv_name
    INTO v_tsrcpv_name;
  IF c_get_tsrcpv_name%NOTFOUND THEN
    v_tsrcpv_name := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Could not find TSRCPV value  ' ||
                                          p_is_nbr);
     
       
  END IF;
  CLOSE c_get_tsrcpv_name;

  RETURN v_tsrcpv_name;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - getting tsrcpv value.');
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_tsrcpv_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_parent_is_nbr(p_parent_fa_seq IN NUMBER) RETURN NUMBER IS

  v_parent_is_nbr NUMBER(8);

  CURSOR c_get_parent_is_nbr(p_parent_fa_seq IN NUMBER) IS
    SELECT fa_tsrfdact_is_number
      FROM sim_field_activities
     WHERE fa_seq = p_parent_fa_seq;

BEGIN

  OPEN c_get_parent_is_nbr(p_parent_fa_seq);
  FETCH c_get_parent_is_nbr
    INTO v_parent_is_nbr;
  IF c_get_parent_is_nbr%NOTFOUND THEN
    v_parent_is_nbr := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Could not find parent_is_number for fa_parent_fa_seq: ' ||
                                          p_parent_fa_seq);
     
       
  END IF;
  CLOSE c_get_parent_is_nbr;

  RETURN v_parent_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - getting parent_is_nbr.');
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_parent_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_uom_is_nbr
(
  p_org_id          IN VARCHAR2,
  p_uom_name        IN VARCHAR2,
  p_chdef_is_number IN NUMBER
) RETURN NUMBER IS

  TYPE t_uom_record IS RECORD(
    uom_is_nbr NUMBER(8),
    uom_name   VARCHAR2(10));

  v_uom_info t_uom_record;

  e_unk_uom EXCEPTION;
  e_uom_sp EXCEPTION;

  CURSOR c_get_is_nbr(p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2) IS
    SELECT tsruom_is_number,
           short_form_name
      FROM tsruom
     WHERE tsruom_org_id = rpad(p_org_id, 8)
       AND upper(short_form_name) = upper(rpad(p_uom_name, 10));

BEGIN

  OPEN c_get_is_nbr(p_org_id, p_uom_name);
  FETCH c_get_is_nbr
    INTO v_uom_info;
  IF c_get_is_nbr%NOTFOUND THEN
    v_uom_info.uom_is_nbr := NULL;
    RAISE e_unk_uom;
  ELSE
    IF v_uom_info.uom_name != p_uom_name THEN
      RAISE e_uom_sp;
    END IF;
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_uom_info.uom_is_nbr;

EXCEPTION
  WHEN e_unk_uom THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Unknown Unit of Measure: ' ||
                                         p_uom_name);
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN v_uom_info.uom_is_nbr;
  
  WHEN e_uom_sp THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Fix uom spelling in TSRCHDEF - is_number:' ||
                                         p_chdef_is_number ||
                                         ', unit of measure misspelled: ' ||
                                         p_uom_name);
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN v_uom_info.uom_is_nbr;
  WHEN OTHERS THEN
    v_success := 0;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'in get uom is_number:' ||
                                         p_chdef_is_number ||
                                         ', unit of measure misspelled: ' ||
                                         p_uom_name);
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_uom_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_sample_frac
(
  p_sample_frac_nm  IN VARCHAR2,
  p_tsrchdef_is_nbr IN NUMBER,
  p_tsrchar_is_nbr  IN NUMBER
) RETURN VARCHAR2 IS

  v_sample_frac VARCHAR2(40);
  v_samp_req_cd VARCHAR2(1);

  e_sample_frac_sp EXCEPTION;

  CURSOR c_get_samp_req_cd(p_tsrchar_is_nbr IN NUMBER) IS
    SELECT samp_frac_req_cd
      FROM tsrchar
     WHERE tsrchar_is_number = p_tsrchar_is_nbr;

  CURSOR c_check_def_frac IS
    SELECT field_value
      FROM tsmprmvl
     WHERE field_name = rpad('SMPL_FRAC_TYPE_NM', 20)
       AND table_name = 'TSRCHDEF';

BEGIN

  /* set sample fraction to null if it is full of spaces */
  IF rpad(p_sample_frac_nm, 15) = rpad(' ', 15)
     OR p_sample_frac_nm IS NULL THEN
    v_sample_frac := NULL;
  ELSE
    v_sample_frac := p_sample_frac_nm;
  END IF;

  OPEN c_get_samp_req_cd(p_tsrchar_is_nbr);
  FETCH c_get_samp_req_cd
    INTO v_samp_req_cd;
  IF v_samp_req_cd = 'Y'
     OR v_sample_frac IS NOT NULL THEN
  
    FOR v_fraction IN c_check_def_frac LOOP
      IF v_fraction.field_value = rpad(p_sample_frac_nm, 40) THEN
        RETURN rtrim(v_fraction.field_value);
      ELSIF upper(v_fraction.field_value) = upper(rpad(p_sample_frac_nm, 40)) THEN
        v_sample_frac := rtrim(v_fraction.field_value);
        CLOSE c_get_samp_req_cd;
        RAISE e_sample_frac_sp;
      END IF;
    END LOOP;
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Invalid fraction in TSRCHDEF_IS_NUMBER:' ||
                                         p_tsrchdef_is_nbr ||
                                         ', sample fraction, ' ||
                                         p_sample_frac_nm ||
                                         ', not loaded into TSRRSULT');
     
       
    RETURN NULL;
  
  END IF;
  CLOSE c_get_samp_req_cd;

  RETURN NULL;

EXCEPTION
  WHEN e_sample_frac_sp THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Fix fraction spelling in TSRCHDEF - is_number:' ||
                                         p_tsrchdef_is_nbr ||
                                         ', sample fraction: ' ||
                                         p_sample_frac_nm);
     
       
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN v_sample_frac;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in validate sample fraction.');
     
       
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN v_sample_frac;
  
END validate_sample_frac;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_field_set_is_nbr
(
  p_org_id          VARCHAR2,
  p_visit_is_number NUMBER,
  p_id_code         IN VARCHAR2,
  p_fdset_name      IN VARCHAR2
) RETURN NUMBER IS

  /* this function looks to see if there is already a field set with the same id-code set up for a visit.
     if it is, it returns the is-number of the field set.  If not, it creates a field set record and returns it's
     IS_number.    SNG 4/21/2000
  */

  v_field_set_is_nbr NUMBER(8);
  v_table_name       VARCHAR2(8);

  CURSOR c_field_set_is_nbr(p_org_id VARCHAR2, p_visit_is_number NUMBER, p_id_code VARCHAR2) IS
    SELECT tsrfdset_is_number
      FROM tsrfdset
     WHERE id_code = rpad(p_id_code, 10)
       AND tsrfdset_org_id = rpad(p_org_id, 8)
       AND tsrstvst_is_number = p_visit_is_number;

BEGIN

  OPEN c_field_set_is_nbr(p_org_id, p_visit_is_number, p_id_code);
  FETCH c_field_set_is_nbr
    INTO v_field_set_is_nbr;
  IF c_field_set_is_nbr%NOTFOUND THEN
  
    /* not found, so set one up */
  
    v_table_name := 'TSRFDSET';
    v_field_set_is_nbr := sim_export_pkg.get_eisn_number(v_table_name, p_org_id);
  
    INSERT INTO tsrfdset
      (tsrfdset_org_id,
       tsrfdset_is_number,
       id_code,
       NAME,
       d_userid_code,
       d_last_update_ts,
       tsrstvst_is_number,
       tsrstvst_org_id)
    VALUES
      (rpad(p_org_id, 8),
       v_field_set_is_nbr,
       p_id_code,
       p_fdset_name,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE,
       p_visit_is_number,
       rpad(p_org_id, 8));
  
    COMMIT;
  
  END IF;

  CLOSE c_field_set_is_nbr;

  RETURN v_field_set_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get_field_set_is_nbr.');
     
       
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_field_set_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_sample_pres_info
(
  p_tsrsdp_is_nbr IN NUMBER,
  p_tsrsdp_org_id IN VARCHAR2,
  p_activity_id   IN VARCHAR2
) RETURN tsrsmple%ROWTYPE IS

  v_sample_rec      tsrsmple%ROWTYPE;
  v_sample_copy_rec tsrsmple%ROWTYPE;

  e_bad_sample_pres EXCEPTION;

  CURSOR c_get_sample_pres_info IS
    SELECT tsmprmvl_is_number,
           tsmprmvl_org_id,
           tsmprmvl0is_number,
           tsmprmvl0org_id,
           container_size_msr,
           container_size_un,
           tsmprmvl1is_number,
           tsmprmvl1org_id,
           presrv_strge_prcdr
      FROM tsrsdp
     WHERE tsrsdp_is_number = p_tsrsdp_is_nbr
       AND tsrsdp_org_id = rpad(p_tsrsdp_org_id, 8);

BEGIN
  /* kms (5/22/01) - this function gets all of the sample preservation, storage, and transport
  information on the activity and returns it in the form of a sample rowtype.  This is
  performed in this manner since STORET does not store this information relationally.
  
  sef (4/1/03) - new functionality in V2 - container type, color, and temp pres type are now
  stored as keys to the tsmprmvl table */

  OPEN c_get_sample_pres_info;
  FETCH c_get_sample_pres_info
    INTO v_sample_rec.tsmprmvl_is_number, v_sample_rec.tsmprmvl_org_id, v_sample_rec.tsmprmvl0is_number, v_sample_rec.tsmprmvl0org_id, v_sample_rec.container_size_msr, v_sample_rec.container_size_un, v_sample_rec.tsmprmvl1is_number, v_sample_rec.tsmprmvl1org_id, v_sample_rec.presrv_strge_prcdr;
  IF c_get_sample_pres_info%NOTFOUND THEN
    CLOSE c_get_sample_pres_info;
    RAISE e_bad_sample_pres;
  END IF;
  CLOSE c_get_sample_pres_info;

  RETURN v_sample_rec;

EXCEPTION
  WHEN e_bad_sample_pres THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Activity ' || p_activity_id ||
                                         ': Unable to find sample preservation information - invalid foreign key.');
     
       
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN v_sample_copy_rec;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - in get next tsmgntxt is nbr.');
     
       
    v_success := 0;
    RETURN v_sample_copy_rec;
  
END get_sample_pres_info;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_instructions
(
  p_import_seq       IN NUMBER,
  p_instruction_type IN VARCHAR2
) RETURN VARCHAR2 IS

  v_imp_cfg_seq  NUMBER(10);
  v_instr_choice VARCHAR2(15);

  e_no_cfg EXCEPTION;

  CURSOR c_get_cfg_seq(p_import_seq IN NUMBER) IS
    SELECT sil_sicc_seq
      FROM sim_import_log
     WHERE sil_seq = p_import_seq;

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_cfg_instrs,
           sim_imp_instr_choices,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND sici_sicc_seq = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_instr_choices,
           sim_imp_cfg_instrs,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND siic_dflt_flag = 1;

BEGIN

  /* get the configuration for this import */
  OPEN c_get_cfg_seq(p_import_seq);
  FETCH c_get_cfg_seq
    INTO v_imp_cfg_seq;
  IF c_get_cfg_seq%NOTFOUND THEN
    v_imp_cfg_seq := NULL;
    RAISE e_no_cfg;
  END IF;
  CLOSE c_get_cfg_seq;

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(v_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr
    INTO v_instr_choice;
  IF c_get_cfg_instr%NOTFOUND THEN
  
    /* no choice could be found, get default choice */
    OPEN c_get_instr_default(p_instruction_type);
    FETCH c_get_instr_default
      INTO v_instr_choice;
    CLOSE c_get_instr_default;
  
  END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;

EXCEPTION
  WHEN e_no_cfg THEN
    raise_application_error(-20101, 'Configuration could not be found for this import.');
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get_instructions.');
     
       
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_instructions;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_visit_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_visit_is_nbr NUMBER(8);

BEGIN
  /* get the next avail is nbr to use */
  v_next_visit_is_nbr := sim_export_pkg.get_eisn_number('TSRSTVST', p_org_id);

  RETURN v_next_visit_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - in get next visit is number.');
     
       
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_visit_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_activity_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_activity_is_nbr NUMBER(8);
BEGIN
  v_activity_is_nbr := sim_export_pkg.get_eisn_number('TSRFDACT', p_org_id);

  RETURN v_activity_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next activity is nbr.');
     
       
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_activity_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_trip_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_trip_is_nbr NUMBER(8);
BEGIN

  v_trip_is_nbr := sim_export_pkg.get_eisn_number('TSRTRIP ', p_org_id);

  RETURN v_trip_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next trip is nbr.');
     
    
       v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_trip_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_trip_qc_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_trip_qc_is_nbr NUMBER(8);
BEGIN
  v_trip_qc_is_nbr := sim_export_pkg.get_eisn_number('TSRFQS  ', p_org_id);

  RETURN v_trip_qc_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next trip qc is nbr.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_trip_qc_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_result_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_result_is_nbr NUMBER(8);
BEGIN
  v_result_is_nbr := sim_export_pkg.get_eisn_number('TSRRSULT', p_org_id);

  RETURN v_result_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next result is nbr.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_result_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsrbrg_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsrbrg_is_nbr NUMBER(8);

BEGIN
  v_tsrbrg_is_nbr := sim_export_pkg.get_eisn_number('TSRBRG', p_org_id);

  RETURN v_tsrbrg_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next tsrbrg is nbr.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_tsrbrg_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsrbrgi_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsrbrgi_is_nbr NUMBER(8);

BEGIN
  v_tsrbrgi_is_nbr := sim_export_pkg.get_eisn_number('TSRBRGI', p_org_id);

  RETURN v_tsrbrgi_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                          ' - in get next tsrbrgi is nbr.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_tsrbrgi_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsmgntxt_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmgntxt_is_nbr NUMBER(8);
BEGIN
  v_tsmgntxt_is_nbr := sim_export_pkg.get_eisn_number('TSMGNTXT', p_org_id);

  RETURN v_tsmgntxt_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - in get next tsmgntxt is nbr.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_next_tsmgntxt_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_value_msr(p_result_text IN VARCHAR2) RETURN NUMBER IS

  v_result_value tsrrsult.value_measure%TYPE;

BEGIN

  BEGIN
    v_result_value := round(to_number(p_result_text), 5);
    IF v_result_value > 99999999.99999 THEN
      v_result_value := 99999999.99999;
       
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Number too large for STORET value column, ' ||
                                         'set to ''99999999.99999'': ' ||
                                         p_result_text);
     
    
    END IF;
  
  EXCEPTION
    WHEN OTHERS THEN
      v_result_value := NULL;
  END;

  RETURN v_result_value;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get value measure.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN NULL;
  
END get_value_msr;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE get_trip_info
(
  p_trip_seq IN NUMBER,
  p_trip_rec OUT tsrtrip%ROWTYPE
) IS

  CURSOR c_get_trip_info(p_trip_seq IN NUMBER) IS
    SELECT stp_trip_id,
           stp_start_date, --STP_START_TIME,
           stp_end_date,
           stp_trip_name,
           stp_tsmorgan_is_number
      FROM sim_trips
     WHERE stp_seq = p_trip_seq;

BEGIN

  /* kms (6/4/01) - get info about this trip from the SIM_TRIPS table */
  OPEN c_get_trip_info(p_trip_seq);
  FETCH c_get_trip_info
    INTO p_trip_rec.id_code, p_trip_rec.start_date, p_trip_rec.end_date, --p_trip_rec.end_time,
  p_trip_rec.NAME, p_trip_rec.tsmorgan_is_number;
  CLOSE c_get_trip_info;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get trip info.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_trip_info;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE get_visit_info
(
  p_visit_seq    IN NUMBER,
  p_visit_rec    OUT tsrstvst%ROWTYPE,
  p_visit_id     OUT VARCHAR2,
  p_ssv_document OUT sim_station_visits.ssv_document%TYPE
) IS

  CURSOR c_get_visit_info(p_visit_seq IN NUMBER) IS
    SELECT ssv_id,
           ssv_start_date,
           ssv_start_time,
           ssv_end_date,
           ssv_end_time,
           sim_export_pkg.get_file_type(ssv_document),
           sim_export_pkg.get_file_name(ssv_document),
           ssv_arrival_time_zone,
           ssv_document
      FROM sim_station_visits
     WHERE ssv_seq = p_visit_seq;

BEGIN

  /* kms (6/4/01) - get info about this trip from the SIM_TRIPS table */
  OPEN c_get_visit_info(p_visit_seq);
  FETCH c_get_visit_info
    INTO p_visit_id, p_visit_rec.arrival_date, p_visit_rec.arrival_time, p_visit_rec.departure_date, p_visit_rec.departure_time, p_visit_rec.blob_type, p_visit_rec.blob_title, p_visit_rec.arrival_time_zone, p_ssv_document;
  CLOSE c_get_visit_info;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get visit info.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END get_visit_info;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_visit_id
(
  p_trip_seq     IN NUMBER,
  p_trip_is_nbr  IN NUMBER,
  p_trip_org_id  IN VARCHAR2,
  p_statn_is_nbr IN NUMBER,
  p_statn_org_id IN VARCHAR2
) RETURN CHAR IS

  v_max_storet_id NUMBER(5);
  v_max_sim_id    NUMBER(5);

  -- jah 3-8-2004
  -- Retrieve the largest STORET station visit number for a given trip and station
  CURSOR c_get_storet_max_visit_id(cp_trip_is_nbr IN NUMBER, cp_trip_org_id IN VARCHAR2, cp_statn_is_nbr IN NUMBER, cp_statn_org_id IN VARCHAR2) IS
    SELECT MAX(to_number(TRIM(id_number)))
      FROM tsrstvst
     WHERE tsrtrip_is_number = cp_trip_is_nbr
       AND tsrtrip_org_id = rpad(cp_trip_org_id, 8)
       AND tsmstatn_is_number = cp_statn_is_nbr
       AND tsmstatn_org_id = rpad(cp_statn_org_id, 8)
       AND translate(id_number, 'a0123456789 ', 'a') IS NULL; -- and is numeric

  -- jah 3-8-2004
  -- Retrieve the largest SIM station visit number for a given trip and station
  CURSOR c_get_sim_max_visit_id(cp_trip_seq IN NUMBER, cp_statn_is_nbr IN NUMBER, cp_statn_org_id IN VARCHAR2) IS
    SELECT MAX(to_number(TRIM(ssv_id)))
      FROM sim_station_visits
     WHERE ssv_stp_seq = cp_trip_seq
       AND ssv_tsmstatn_is_number = cp_statn_is_nbr
       AND ssv_tsmstatn_org_id = rpad(cp_statn_org_id, 8)
       AND translate(ssv_id, 'a0123456789 ', 'a') IS NULL; -- and is numeric

BEGIN

  -- Get the largest STORET station visit ID for a given trip and station
  OPEN c_get_storet_max_visit_id(p_trip_is_nbr, p_trip_org_id, p_statn_is_nbr, p_statn_org_id);
  FETCH c_get_storet_max_visit_id
    INTO v_max_storet_id;
  CLOSE c_get_storet_max_visit_id;
  v_max_storet_id := nvl(v_max_storet_id, 0);

  -- Get the largest SIM STATION visit ID for a given trip and station
  OPEN c_get_sim_max_visit_id(p_trip_seq, p_statn_is_nbr, p_statn_org_id);
  FETCH c_get_sim_max_visit_id
    INTO v_max_sim_id;
  CLOSE c_get_sim_max_visit_id;
  v_max_sim_id := nvl(v_max_sim_id, 0);

  -- return the largest station visit ID
  IF v_max_storet_id = 999
     OR v_max_sim_id = 999 THEN
    raise_application_error(-20000, 'Exceeded the maximum number of visits allowed for one trip (999).');
  END IF;
  RETURN to_char(greatest(v_max_storet_id, v_max_sim_id) + 1);

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in function get_next_visit_id.');
     
    
    v_success := 0;
  
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
    RETURN NULL;
  
END get_next_visit_id;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE add_station_to_trip
(
  p_trip_is_nbr IN NUMBER,
  p_sta_is_nbr  IN NUMBER,
  p_org_id      IN VARCHAR2
) IS

  v_tsrtsa NUMBER(8);

  CURSOR c_get_tsrtsa(p_cursor_trip IN NUMBER, p_cursor_sta IN NUMBER, p_cursor_org_id IN VARCHAR2) IS
    SELECT tsrtrip_is_number
      FROM tsrtsa
     WHERE tsmstatn_is_number = p_cursor_sta
       AND tsrtrip_is_number = p_cursor_trip
       AND tsrtrip_org_id = p_cursor_org_id;

BEGIN
  /* see if station is already on the trip */
  OPEN c_get_tsrtsa(p_trip_is_nbr, p_sta_is_nbr, rpad(p_org_id, 8));
  FETCH c_get_tsrtsa
    INTO v_tsrtsa;
  IF c_get_tsrtsa%NOTFOUND THEN
    /* Update TSRTSA to associate station with this trip if record does not already exist.  */
    INSERT INTO tsrtsa
      (tsrtrip_is_number,
       tsrtrip_org_id,
       tsmstatn_is_number,
       tsmstatn_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_trip_is_nbr, /* TSRTRIP_IS_NUMBER */
       rpad(p_org_id, 8), /* TSRTRIP_ORG_ID */
       p_sta_is_nbr, /* TSMSTATN_IS_NUMBER */
       rpad(p_org_id, 8), /* TSMSTATN_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    COMMIT;
  
  END IF;
  CLOSE c_get_tsrtsa;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in add station to trip.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END add_station_to_trip;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE add_project_to_activity
(
  p_act_is_nbr  IN NUMBER,
  p_proj_is_nbr IN NUMBER,
  p_org_id      IN VARCHAR2
) IS

  v_proj_is NUMBER(8);

  CURSOR c_check_for_project(p_cursor_act IN NUMBER, p_cursor_proj IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number
      FROM tsrfapra
     WHERE tsmproj_is_number = p_cursor_proj
       AND tsrfdact_is_number = p_cursor_act
       AND tsrfdact_org_id = rpad(p_org_id, 8);

BEGIN
  /* see if the project is on the activity */
  OPEN c_check_for_project(p_act_is_nbr, p_proj_is_nbr, p_org_id);
  FETCH c_check_for_project
    INTO v_proj_is;
  IF c_check_for_project%NOTFOUND THEN
    /* if project is not on the activity then add */
    INSERT INTO tsrfapra
      (tsrfdact_is_number,
       tsrfdact_org_id,
       tsmproj_is_number,
       tsmproj_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_act_is_nbr, /* TSRFDACT_IS_NUMBER */
       p_org_id, /* TSRFDACT_ORG_ID */
       p_proj_is_nbr, /* TSMPROJ_IS_NUMBER */
       p_org_id, /* TSMPROJ_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
    COMMIT;
  END IF;
  CLOSE c_check_for_project;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in add project to activity.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END add_project_to_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE add_project_to_trip
(
  p_trip_is_nbr IN NUMBER,
  p_proj_is_nbr IN NUMBER,
  p_org_id      IN VARCHAR2
) IS

  v_proj_is         NUMBER(8);
  v_number_projects NUMBER(8);

  CURSOR c_check_for_project(p_cursor_trip IN NUMBER, p_cursor_proj IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number
      FROM tsrtpa
     WHERE tsmproj_is_number = p_cursor_proj
       AND tsrtrip_is_number = p_cursor_trip
       AND tsrtrip_org_id = rpad(p_org_id, 8);

  CURSOR c_check_for_single_proj(p_cursor_trip IN NUMBER, p_cursor_org_id IN VARCHAR2) IS
    SELECT COUNT(tsmproj_is_number)
      FROM tsrtpa
     WHERE tsrtrip_is_number = p_cursor_trip
       AND tsrtrip_org_id = p_cursor_org_id;

BEGIN
  /* see if the project is on the trip */
  OPEN c_check_for_project(p_trip_is_nbr, p_proj_is_nbr, p_org_id);
  FETCH c_check_for_project
    INTO v_proj_is;
  IF c_check_for_project%NOTFOUND THEN
    /* if project is not on the trip then add */
    INSERT INTO tsrtpa
      (tsrtrip_is_number,
       tsrtrip_org_id,
       tsmproj_is_number,
       tsmproj_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_trip_is_nbr, /* TSRTRIP_IS_NUMBER */
       p_org_id, /* TSRTRIP_ORG_ID */
       p_proj_is_nbr, /* TSMPROJ_IS_NUMBER */
       p_org_id, /* TSMPROJ_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
    COMMIT;
  END IF;
  CLOSE c_check_for_project;

  /* check for single project on trip */
  OPEN c_check_for_single_proj(p_trip_is_nbr, rpad(p_org_id, 8));
  FETCH c_check_for_single_proj
    INTO v_number_projects;
  /* set single project ind to 'N' if more than one project is on the trip */
  IF v_number_projects > 1 THEN
    UPDATE tsrtrip
       SET single_proj_ind = 'N'
     WHERE tsrtrip_is_number = p_trip_is_nbr
       AND tsrtrip_org_id = rpad(p_org_id, 8);
  END IF;
  CLOSE c_check_for_single_proj;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in add project to trip.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END add_project_to_trip;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_station_to_project
(
  p_proj_is_nbr IN VARCHAR2,
  p_sta_is_nbr  IN VARCHAR2,
  p_org_id      IN VARCHAR2
) RETURN NUMBER IS

  v_sta_is NUMBER(8);

  CURSOR c_check_for_station(p_cursor_proj IN VARCHAR2, p_cursor_sta IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmstatn_is_number
      FROM tsmpsa
     WHERE tsmstatn_is_number = p_cursor_sta
       AND tsmstatn_org_id = rpad(p_org_id, 8)
       AND tsmproj_is_number = p_cursor_proj
       AND tsmproj_org_id = rpad(p_org_id, 8);

BEGIN

  /* see if station is on project */
  OPEN c_check_for_station(p_proj_is_nbr, p_sta_is_nbr, p_org_id);
  FETCH c_check_for_station
    INTO v_sta_is;
  IF c_check_for_station%NOTFOUND THEN
  
    /* add station to project if not found */
    INSERT INTO tsmpsa
      (tsmstatn_is_number,
       tsmstatn_org_id,
       tsmproj_is_number,
       tsmproj_org_id,
       d_userid_code,
       d_last_updt_ts)
    VALUES
      (p_sta_is_nbr, /* TSMSTATN_IS_NUMBER */
       p_org_id, /* TSMSTATN_ORG_ID */
       p_proj_is_nbr, /* TSMPROJ_IS_NUMBER */
       p_org_id, /* TSMPROJ_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDT_TS */);
  END IF;
  CLOSE c_check_for_station;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                       ' - in add station to project.');
     
    
    v_success := 0;
    RETURN 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END add_station_to_project;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_person_to_project
(
  p_person_is_nbr  IN NUMBER,
  p_person_org_id  IN VARCHAR2,
  p_project_is_nbr IN NUMBER,
  p_org_id         IN VARCHAR2
) RETURN NUMBER IS

  v_dummy NUMBER(8);

  CURSOR c_check_for_person_role(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2, p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT tsmpersn_is_number
      FROM tsmppra
     WHERE tsmpersn_is_number = p_person_is_nbr
       AND tsmpersn_org_id = rpad(p_person_org_id, 8)
       AND tsmproj_is_number = p_project_is_nbr
       AND tsmproj_org_id = rpad(p_project_org_id, 8);

  CURSOR c_check_for_person(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2, p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT tsmpersn_is_number
      FROM tsmppa
     WHERE tsmpersn_is_number = p_person_is_nbr
       AND tsmpersn_org_id = rpad(p_person_org_id, 8)
       AND tsmproj_is_number = p_project_is_nbr
       AND tsmproj_org_id = rpad(p_project_org_id, 8);

  CURSOR c_check_for_role(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2) IS
    SELECT tsmpersn_is_number
      FROM tsmpra
     WHERE tsmpersn_is_number = p_person_is_nbr
       AND tsmpersn_org_id = rpad(p_person_org_id, 8)
       AND tsmprole_is_number = 10
       AND tsmprole_org_id = 'ZZZZZZZZ';

BEGIN

   /* make sure project, person assignment has not already been made */
  OPEN c_check_for_person(p_person_is_nbr, p_person_org_id, p_project_is_nbr, p_org_id);
  FETCH c_check_for_person
    INTO v_dummy;
  IF c_check_for_person%NOTFOUND THEN
  
    /* if nothing found then create association */
    INSERT INTO tsmppa
      (tsmproj_is_number,
       tsmproj_org_id,
       tsmpersn_is_number,
       tsmpersn_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_project_is_nbr, /* TSMPROJ_IS_NUMBER */
       p_org_id, /* TSMPROJ_ORG_ID */
       p_person_is_nbr, /* TSMPERSN_IS_NUMBER */
       p_person_org_id, /* TSMPERSN_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
  END IF;
  CLOSE c_check_for_person;

  /* make sure project, person, role assignment has not already been made */
  OPEN c_check_for_person_role(p_person_is_nbr, p_person_org_id, p_project_is_nbr, p_org_id);
  FETCH c_check_for_person_role
    INTO v_dummy;
  IF c_check_for_person_role%NOTFOUND THEN
  
    /* if nothing found then create association using the staff/employee role */
    INSERT INTO tsmppra
      (tsmproj_is_number,
       tsmproj_org_id,
       tsmpersn_is_number,
       tsmpersn_org_id,
       tsmprole_is_number,
       tsmprole_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_project_is_nbr, /* TSMPROJ_IS_NUMBER */
       p_org_id, /* TSMPROJ_ORG_ID */
       p_person_is_nbr, /* TSMPERSN_IS_NUMBER */
       p_person_org_id, /* TSMPERSN_ORG_ID */
       10, /* TSMPROLE_IS_NUMBER */
       'ZZZZZZZZ', /* TSMPROLE_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
  END IF;
  CLOSE c_check_for_person_role;

 
  /* make sure role, person assignment has not already been made */
  OPEN c_check_for_role(p_person_is_nbr, p_person_org_id);
  FETCH c_check_for_role
    INTO v_dummy;
  IF c_check_for_role%NOTFOUND THEN
  
    /* if nothing found then create association */
    INSERT INTO tsmpra
      (tsmprole_is_number,
       tsmprole_org_id,
       tsmpersn_is_number,
       tsmpersn_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (10, /* TSMPROLE_IS_NUMBER */
       'ZZZZZZZZ', /* TSMPROLE_ORG_ID */
       p_person_is_nbr, /* TSMPERSN_IS_NUMBER */
       p_person_org_id, /* TSMPERSN_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
  END IF;
  CLOSE c_check_for_role;

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in add person to project.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END add_person_to_project;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_cporg_to_project
(
  p_cporg_is_nbr   IN NUMBER,
  p_cporg_org_id   IN VARCHAR2,
  p_project_is_nbr IN NUMBER,
  p_project_org_id IN VARCHAR2
) RETURN NUMBER IS

  CURSOR c_tsmpcoa(p_cporg_is_nbr IN NUMBER, p_cporg_org_id IN VARCHAR2, p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number,
           tsmproj_org_id,
           tsmcporg_is_number,
           tsmcporg_org_id
      FROM tsmpcoa
     WHERE tsmcporg_is_number = p_cporg_is_nbr
       AND tsmcporg_org_id = rpad(p_cporg_org_id, 8)
       AND tsmproj_is_number = p_project_is_nbr
       AND tsmproj_org_id = rpad(p_project_org_id, 8);

  r_tsmpcoa c_tsmpcoa%ROWTYPE;

BEGIN

  -- try to retrive the cooperating organization from STORET
  OPEN c_tsmpcoa(p_cporg_is_nbr, p_cporg_org_id, p_project_is_nbr, p_project_org_id);
  FETCH c_tsmpcoa
    INTO r_tsmpcoa;

  -- Do we need to make the Project Cooperating Organization Assignment (PCOA)?
  IF c_tsmpcoa%NOTFOUND THEN
    -- if yes
  
    -- assign the cooperating organization to the project
    INSERT INTO tsmpcoa
      (tsmproj_is_number,
       tsmproj_org_id,
       tsmcporg_is_number,
       tsmcporg_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_project_is_nbr, -- TSMPROJ_IS_NUMBER
       p_project_org_id, -- TSMPROJ_ORG_ID
       p_cporg_is_nbr, -- TSMPERSN_IS_NUMBER
       p_cporg_org_id, -- TSMPERSN_ORG_ID
       sim_export_pkg.v_pkg_ownerid, -- D_USERID_CODE
       SYSDATE -- D_LAST_UPDATE_TS
       );
  
  END IF;

  CLOSE c_tsmpcoa;

  COMMIT;

  -- return success
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    -- output error message
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in add person to project.');
     
    
    -- eh?
    v_success := 0;
  
    -- eh?
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
    -- return failure
    RETURN 0;
  
END add_cporg_to_project;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_trip(p_trip_rec IN tsrtrip%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create trip record */
  INSERT INTO tsrtrip
    (tsrtrip_is_number,
     tsrtrip_org_id,
     id_code,
     NAME,
     start_date,
     start_time,
     start_time_zone,
     end_date,
     end_time,
     end_time_zone,
     leader_name,
     vehicle_ship_name,
     trip_plan_text,
     comment_text,
     single_proj_ind,
     d_userid_code,
     d_last_update_ts,
     tsmorgan_is_number)
  VALUES
    (p_trip_rec.tsrtrip_is_number, /* TSRTRIP_IS_NUMBER */
     p_trip_rec.tsrtrip_org_id, /* TSRTRIP_ORG_ID */
     p_trip_rec.id_code, /* ID_CODE */
     p_trip_rec.NAME, /* NAME */
     p_trip_rec.start_date, /* START_DATE */
     p_trip_rec.start_time, /* START_TIME */
     p_trip_rec.start_time_zone, /* START_TIME_ZONE */
     p_trip_rec.end_date, /* END_DATE */
     p_trip_rec.end_time, /* END_TIME */
     p_trip_rec.end_time_zone, /* END_TIME_ZONE */
     p_trip_rec.leader_name, /* LEADER_NAME */
     p_trip_rec.vehicle_ship_name, /* VEHICLE_SHIP_NAME */
     p_trip_rec.trip_plan_text, /* TRIP_PLAN_TEXT */
     p_trip_rec.comment_text, /* COMMENT_TEXT */
     p_trip_rec.single_proj_ind, /* SINGLE_PROJ_IND */
     p_trip_rec.d_userid_code, /* D_USERID_CODE */
     p_trip_rec.d_last_update_ts, /* D_LAST_UPDATE_TS */
     p_trip_rec.tsmorgan_is_number /* TSMORGAN_IS_NUMBER */);

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create activity location.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_trip;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_activity_location
(
  p_org_id          IN VARCHAR2,
  p_activity_is_nbr IN NUMBER,
  p_activity_seq    IN NUMBER,
  p_station_is_nbr  IN NUMBER,
  p_station_org_id  IN VARCHAR2
) RETURN NUMBER IS

  CURSOR c_aal(p_activity_seq NUMBER) IS
    SELECT aal_tsmalp_is_number,
           aal_tsrfdact_is_number,
           aal_tsraal_type_name,
           aal_bottom_depth_msr,
           aal_bottom_depth_uom,
           aal_addtnl_loc_info
      FROM sim_actual_activity_loc
     WHERE aal_fa_seq = p_activity_seq;

BEGIN

  FOR v_aal IN c_aal(p_activity_seq) LOOP
  
    IF v_aal.aal_bottom_depth_msr IS NULL THEN
      v_aal.aal_bottom_depth_msr := 0; --Default NULL Value
    END IF;
  
    /* create activity location record */
  
    INSERT INTO tsraal
      (tsmalp_is_number,
       tsmalp_org_id,
       tsrfdact_is_number,
       tsrfdact_org_id,
       tsraal_type_name,
       bottom_depth_msr,
       bottom_depth_un_cd,
       addtnl_loc_info,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (v_aal.aal_tsmalp_is_number, /*tsmalp_is_number*/
       rpad(p_org_id, 8), /*tsmalp_org_id*/
       p_activity_is_nbr, /*tsrfdact_is_number*/
       rpad(p_org_id, 8), /*tsrfdact_org_id*/
       rpad(v_aal.aal_tsraal_type_name, 15), /* TSRAAL_TYPE_NAME */
       v_aal.aal_bottom_depth_msr,
       rpad(v_aal.aal_bottom_depth_uom, 3),
       v_aal.aal_addtnl_loc_info,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    UPDATE sim_actual_activity_loc
       SET aal_tsrfdact_is_number = p_activity_is_nbr
     WHERE aal_fa_seq = p_activity_seq;
    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' -  in create activity location.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_activity_location;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsrtod_record
(
  p_org_id          IN VARCHAR2,
  p_activity_is_nbr IN NUMBER,
  p_activity_seq    IN NUMBER
) RETURN NUMBER IS

  CURSOR c_tod(p_activity_seq NUMBER) IS
    SELECT tod_tsrfdact_is_number,
           tod_tsrfdact_org_id,
           tod_fished_duration,
           tod_fished_dur_uom,
           tod_boat_speed_msr,
           tod_boat_speed_uom,
           tod_fished_distance,
           tod_fished_distance_uom,
           tod_rel_current_dir,
           tod_rel_wind_dir,
           tod_comment_text
      FROM sim_trawl_ops_det
     WHERE tod_fa_seq = p_activity_seq;

BEGIN

  FOR v_tod IN c_tod(p_activity_seq) LOOP
    --Default NULL values For TSRTOD
    IF v_tod.tod_boat_speed_msr IS NULL THEN
      v_tod.tod_boat_speed_msr := 0;
    END IF;
    IF v_tod.tod_fished_distance IS NULL THEN
      v_tod.tod_fished_distance := 0;
    END IF;
    IF v_tod.tod_rel_current_dir IS NULL THEN
      v_tod.tod_rel_current_dir := 0;
    END IF;
    IF v_tod.tod_rel_wind_dir IS NULL THEN
      v_tod.tod_rel_wind_dir := 0;
    END IF;
  
    /* create tsrtod record */
    INSERT INTO tsrtod
      (tsrfdact_is_number,
       tsrfdact_org_id,
       fished_duratn_msr,
       fishd_durtn_unt_cd,
       boat_speed_msr,
       boat_speed_un_cd,
       fished_distance,
       fished_distance_un,
       rel_current_dir,
       rel_wind_dir,
       comment_text,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (v_tod.tod_tsrfdact_is_number,
       v_tod.tod_tsrfdact_org_id,
       v_tod.tod_fished_duration,
       v_tod.tod_fished_dur_uom,
       v_tod.tod_boat_speed_msr,
       v_tod.tod_boat_speed_uom,
       v_tod.tod_fished_distance,
       v_tod.tod_fished_distance_uom,
       v_tod.tod_rel_current_dir,
       v_tod.tod_rel_wind_dir,
       v_tod.tod_comment_text,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create tsrtod record.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_tsrtod_record;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsrsmple_record
(
  p_org_id          IN VARCHAR2,
  p_activity_is_nbr IN NUMBER,
  p_activity_seq    IN NUMBER,
  p_sample_rec      IN tsrsmple%ROWTYPE
) RETURN NUMBER IS

  v_duration_units VARCHAR2(10);

  CURSOR c_sample(p_activity_seq NUMBER) IS
    SELECT ss_tsrfdact_is_number,
           ss_tsrfdact_org_id,
           ss_total_volume_msr,
           ss_total_vol_uom,
           ss_total_weight_msr,
           ss_total_wt_uom,
           ss_data_file_name_loc,
           ss_container_type_nm,
           ss_container_color,
           ss_container_size_msr,
           ss_container_size_un,
           ss_temp_preservn_type,
           ss_parent_ind_code,
           ss_gear_deployment,
           ss_transport_storage,
           ss_presrv_strge_prcdr,
           ss_duration_time,
           ss_duration_unit
      FROM sim_sample
     WHERE ss_fa_seq = p_activity_seq;

BEGIN

  FOR v_sample IN c_sample(p_activity_seq) LOOP
    --Default NULL Values For TSRSMPLE
    IF v_sample.ss_total_volume_msr IS NULL THEN
      v_sample.ss_total_volume_msr := 0;
    END IF;
    IF v_sample.ss_total_weight_msr IS NULL THEN
      v_sample.ss_total_weight_msr := 0;
    END IF;
    IF v_sample.ss_container_size_msr IS NULL THEN
      v_sample.ss_container_size_msr := 0;
    END IF;
    IF v_sample.ss_duration_time IS NULL THEN
      v_sample.ss_duration_time := 0;
    END IF;
  
    /* create tsrsmple record */
    INSERT INTO tsrsmple
      (tsrfdact_is_number,
       tsrfdact_org_id,
       total_volume_msr,
       total_volume_un_cd,
       total_weight_msr,
       total_weight_un_cd,
       data_file_name_loc,
       container_type_nm,
       container_color,
       container_size_msr,
       container_size_un,
       temp_preservn_type,
       parent_ind_code,
       gear_deployment,
       transport_storage,
       presrv_strge_prcdr,
       duration_time,
       duration_units,
       d_userid_code,
       d_last_update_ts,
       tsmprmvl_is_number,
       tsmprmvl_org_id,
       tsmprmvl0is_number,
       tsmprmvl0org_id,
       tsmprmvl1is_number,
       tsmprmvl1org_id)
    VALUES
      (v_sample.ss_tsrfdact_is_number,
       v_sample.ss_tsrfdact_org_id,
       v_sample.ss_total_volume_msr,
       v_sample.ss_total_vol_uom,
       v_sample.ss_total_weight_msr,
       v_sample.ss_total_wt_uom,
       v_sample.ss_data_file_name_loc,
       v_sample.ss_container_type_nm,
       v_sample.ss_container_color,
       v_sample.ss_container_size_msr,
       v_sample.ss_container_size_un,
       v_sample.ss_temp_preservn_type,
       v_sample.ss_parent_ind_code,
       v_sample.ss_gear_deployment,
       v_sample.ss_transport_storage,
       v_sample.ss_presrv_strge_prcdr,
       v_sample.ss_duration_time,
       v_duration_units,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE, /* D_LAST_UPDATE_TS */
       p_sample_rec.tsmprmvl_is_number,
       p_sample_rec.tsmprmvl_org_id,
       p_sample_rec.tsmprmvl0is_number,
       p_sample_rec.tsmprmvl0org_id,
       p_sample_rec.tsmprmvl1is_number,
       p_sample_rec.tsmprmvl1org_id);
  
    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                       ' - in create tsrsmple record.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_tsrsmple_record;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsrbrg_records
(
  p_org_id          IN VARCHAR2,
  p_activity_is_nbr IN NUMBER,
  p_activity_seq    IN NUMBER
) RETURN NUMBER IS

  v_lifestage_name VARCHAR2(25);
  v_sex_name       VARCHAR2(15);
  v_tsrbrg_is_nbr  NUMBER(8);

  CURSOR c_brg(p_activity_seq NUMBER) IS
    SELECT *
      FROM sim_bio_results_grp
     WHERE brg_fa_seq = p_activity_seq;

BEGIN

  FOR v_brg IN c_brg(p_activity_seq) LOOP
    IF v_brg.brg_tsrcpv_lifestage_is_nbr IS NOT NULL THEN
      v_lifestage_name := get_tsrcpv_value(v_brg.brg_tsrcpv_lifestage_is_nbr, v_brg.brg_tsrcpv_lifestage_org_id);
    END IF;
  
    IF v_brg.brg_tsrcpv_sex_is_number IS NOT NULL THEN
      v_sex_name := get_tsrcpv_value(v_brg.brg_tsrcpv_sex_is_number, v_brg.brg_tsrcpv_sex_org_id);
    END IF;
  
    IF v_brg.brg_summary_grp_count IS NULL THEN
      v_brg.brg_summary_grp_count := 0; --Default NULL Value
    END IF;
  
    v_tsrbrg_is_nbr := get_next_tsrbrg_is_nbr(p_org_id);
  
    /* create tsrbrg records */
    INSERT INTO tsrbrg
      (tsrbrg_is_number,
       tsrbrg_org_id,
       id_code,
       type_name,
       type_indicator,
       sex_name,
       life_stage_name,
       value_type_name,
       summary_grp_count,
       description_text,
       species_number,
       tsrfdact_is_number,
       tsrfdact_org_id,
       tsrchar_is_number,
       tsrchar_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (v_tsrbrg_is_nbr,
       rpad(p_org_id, 8),
       v_brg.brg_id_code,
       v_brg.brg_type,
       v_brg.brg_freq_analysis,
       v_sex_name,
       v_lifestage_name,
       v_brg.brg_value_type_name,
       v_brg.brg_summary_grp_count,
       v_brg.brg_description_text,
       v_brg.brg_species_number,
       p_activity_is_nbr,
       rpad(p_org_id, 8),
       v_brg.brg_tsrchar_is_number,
       v_brg.brg_tsrchar_org_id,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    UPDATE sim_bio_results_grp
       SET brg_tsrbrg_is_number = v_tsrbrg_is_nbr,
           brg_tsrbrg_org_id = p_org_id
     WHERE brg_seq = v_brg.brg_seq;
  
    UPDATE sim_results
       SET rs_tsrbrg_is_number = v_tsrbrg_is_nbr,
           rs_tsrbrg_org_id = p_org_id
     WHERE rs_brg_seq = v_brg.brg_seq
       AND rs_fa_seq = p_activity_seq;
  
    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create tsrbrg records.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_tsrbrg_records;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsrbrgi_records
(
  p_org_id          IN VARCHAR2,
  p_activity_is_nbr IN NUMBER,
  p_activity_seq    IN NUMBER
) RETURN NUMBER IS
  v_tsrbrgi_is_nbr NUMBER(8);

  CURSOR c_brgi(p_activity_seq NUMBER) IS
    SELECT brgi.brgi_seq brgi_seq,
           brgi.brgi_individual_number individual_number,
           brg.brg_tsrbrg_is_number tsrbrg_is_number,
           brg.brg_tsrbrg_org_id tsrbrg_org_id
      FROM sim_bio_results_grp_individual brgi,
           sim_bio_results_grp brg
     WHERE brgi_fa_seq = p_activity_seq
       AND brg.brg_seq = brgi.brgi_brg_seq
     ORDER BY individual_number;

BEGIN

  FOR v_brgi IN c_brgi(p_activity_seq) LOOP
  
    v_tsrbrgi_is_nbr := get_next_tsrbrgi_is_nbr(p_org_id);
  
    /* create tsrbrgi records */
    INSERT INTO tsrbrgi
      (tsrbrgi_is_number,
       tsrbrgi_org_id,
       individual_number,
       tsrbrg_is_number,
       tsrbrg_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (v_tsrbrgi_is_nbr,
       rpad(p_org_id, 8),
       v_brgi.individual_number,
       v_brgi.tsrbrg_is_number,
       v_brgi.tsrbrg_org_id,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    UPDATE sim_results
       SET rs_tsrbrgi_is_number = v_tsrbrgi_is_nbr,
           rs_tsrbrgi_org_id = p_org_id
     WHERE rs_brgi_seq = v_brgi.brgi_seq
       AND rs_fa_seq = p_activity_seq;
  
    COMMIT;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create tsrbrgi records.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_tsrbrgi_records;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsrrci_record
(
  p_org_id        IN VARCHAR2,
  p_result_is_nbr IN NUMBER
) RETURN NUMBER IS

  CURSOR c_rci(p_org_id VARCHAR2, p_result_is_nbr NUMBER) IS
    SELECT rci_tsrrsult_is_number,
           rci_tsrrsult_org_id,
           rci_lower_bnd_amt,
           rci_upper_bnd_amt,
           rci_primary_class_ind,
           rci_second_class_ind,
           rci_particle_size_basis,
           rci_tsruom_is_number,
           rci_tsruom_org_id
      FROM sim_result_class_indicator
     WHERE rci_tsrrsult_is_number = p_result_is_nbr
       AND rci_tsrrsult_org_id = p_org_id;

BEGIN

  FOR v_rci IN c_rci(p_org_id, p_result_is_nbr) LOOP
  
    --Default NULL Values For TSRRCI
    IF v_rci.rci_lower_bnd_amt IS NULL THEN
      v_rci.rci_lower_bnd_amt := 0;
    END IF;
  
    IF v_rci.rci_upper_bnd_amt IS NULL THEN
      v_rci.rci_upper_bnd_amt := 0;
    END IF;
  
    /* create tsrrci records */
    INSERT INTO tsrrci
      (tsrrsult_is_number,
       tsrrsult_org_id,
       lower_bnd_amt,
       upper_bnd_amt,
       prim_class_desc,
       sec_class_desc,
       particle_size_basis,
       tsruom_is_number,
       tsruom_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (v_rci.rci_tsrrsult_is_number,
       v_rci.rci_tsrrsult_org_id,
       v_rci.rci_lower_bnd_amt,
       v_rci.rci_upper_bnd_amt,
       v_rci.rci_primary_class_ind,
       v_rci.rci_second_class_ind,
       v_rci.rci_particle_size_basis,
       v_rci.rci_tsruom_is_number,
       v_rci.rci_tsruom_org_id,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    COMMIT;
  
  END LOOP;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create tsrrci records.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_tsrrci_record;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_lab_remarks_to_result(p_tsrrlra_rec IN tsrrlra%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a Lab Remark record */

  INSERT INTO tsrrlra
    (tsrrsult_is_number,
     tsrrsult_org_id,
     tsrlbrmk_is_number,
     tsrlbrmk_org_id,
     d_userid_code,
     d_last_update_ts)
  VALUES
    (p_tsrrlra_rec.tsrrsult_is_number,
     p_tsrrlra_rec.tsrrsult_org_id,
     p_tsrrlra_rec.tsrlbrmk_is_number,
     p_tsrrlra_rec.tsrlbrmk_org_id,
     p_tsrrlra_rec.d_userid_code,
     p_tsrrlra_rec.d_last_update_ts);

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create_tsrrlra_record.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END add_lab_remarks_to_result;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_trap_net(p_tsrtnod_rec IN tsrtnod%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a Trap/Net details record */

  INSERT INTO tsrtnod
    (tsrfdact_is_number,
     tsrfdact_org_id,
     smplng_duratn_msr,
     smplng_drtn_unt_cd,
     orientn_to_current,
     rel_current_dir,
     rel_wind_dir,
     comment_text,
     d_userid_code,
     d_last_update_ts)
  VALUES
    (p_tsrtnod_rec.tsrfdact_is_number, /* TSRFDACT_IS_NUMBER */
     p_tsrtnod_rec.tsrfdact_org_id,
     p_tsrtnod_rec.smplng_duratn_msr,
     p_tsrtnod_rec.smplng_drtn_unt_cd,
     p_tsrtnod_rec.orientn_to_current,
     p_tsrtnod_rec.rel_current_dir,
     p_tsrtnod_rec.rel_wind_dir,
     p_tsrtnod_rec.comment_text,
     p_tsrtnod_rec.d_userid_code,
     p_tsrtnod_rec.d_last_update_ts);

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM || ' - in create trap net.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_trap_net;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_tsrcldes(p_tsrcldes_rec IN tsrcldes%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a Trap/Net details record */

  INSERT INTO tsrcldes
    (tsrrsult_is_number,
     tsrrsult_org_id,
     cell_type_nm,
     cell_shape_type_nm,
     d_userid_code,
     d_last_update_ts)
  VALUES
    (p_tsrcldes_rec.tsrrsult_is_number,
     p_tsrcldes_rec.tsrrsult_org_id,
     p_tsrcldes_rec.cell_type_nm,
     p_tsrcldes_rec.cell_shape_type_nm,
     p_tsrcldes_rec.d_userid_code,
     p_tsrcldes_rec.d_last_update_ts);

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM || ' - in create tsrcldes.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_tsrcldes;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_station_visit
(
  p_org_id    IN VARCHAR2,
  p_stvst_rec IN tsrstvst%ROWTYPE
) RETURN NUMBER IS

BEGIN

  BEGIN
    INSERT INTO tsrstvst
      (tsrstvst_is_number,
       tsrstvst_org_id,
       id_number,
       arrival_date,
       arrival_time,
       arrival_time_zone,
       departure_date,
       departure_time,
       deprture_time_zone,
       comment_text,
       d_userid_code,
       d_last_update_ts,
       tsrtrip_is_number,
       tsrtrip_org_id,
       tsmstatn_is_number,
       tsmstatn_org_id,
       blob_type,
       blob_title)
    VALUES
      (p_stvst_rec.tsrstvst_is_number /* TSRSTVST_IS_NUMBER */,
       p_stvst_rec.tsrstvst_org_id /* TSRSTVST_ORG_ID */,
       p_stvst_rec.id_number /* ID_NUMBER */,
       p_stvst_rec.arrival_date /* ARRIVAL_DATE */,
       p_stvst_rec.arrival_time /* ARRIVAL_TIME */,
       p_stvst_rec.arrival_time_zone /* ARRIVAL_TIME_ZONE */,
       p_stvst_rec.departure_date /* DEPARTURE_DATE */,
       p_stvst_rec.departure_time /* DEPARTURE_TIME */,
       p_stvst_rec.deprture_time_zone /* DEPRTURE_TIME_ZONE */,
       p_stvst_rec.comment_text /* COMMENT_TEXT */,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */,
       p_stvst_rec.tsrtrip_is_number /* TSRTRIP_IS_NUMBER */,
       p_stvst_rec.tsrtrip_org_id /* TSRTRIP_ORG_ID */,
       p_stvst_rec.tsmstatn_is_number /* TSMSTATN_IS_NUMBER */,
       p_stvst_rec.tsmstatn_org_id /* TSMSTATN_ORG_ID */,
       p_stvst_rec.blob_type /* BLOB_TYPE */,
       p_stvst_rec.blob_title /* BLOB_TITLE */);
  
  EXCEPTION
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                          SQLERRM ||
                                          ' -  inserting visit record into TSRSTVST');
       
      
      v_success := 0;
      IF t_success > v_success THEN
        t_success := v_success;
      END IF;
  END;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        SQLERRM ||
                                        ' - in create station visit.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_station_visit;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_activity(p_activity_rec IN OUT tsrfdact%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create activity using info passed in through activity record */

  INSERT INTO tsrfdact
    (tsrfdact_is_number,
     tsrfdact_org_id,
     type_name,
     category_type_name,
     id_code,
     replicate_number,
     medium_type_name,
     intent_type_name,
     community_name,
     start_date,
     start_time,
     start_time_zone,
     stop_date,
     stop_time,
     stop_time_zone,
     comment_text,
     zone_type_name,
     reltv_depth_name,
     depth_ref_point,
     depth_to_activity,
     depth_to_act_un_cd,
     upper_depth_to_act,
     depth_msr_unt_cd,
     lower_depth_to_act,
     thermocline_loc_cd,
     pycnocline_loc_cd,
     halocline_loc_cd,
     species_number,
     d_userid_code,
     d_last_update_ts,
     tsropprd_org_id,
     tsropprd_is_number,
     tsrfdact0is_number,
     tsrfdact0org_id,
     tsrfdact1is_number,
     tsrfdact1org_id,
     tsrstvst_is_number,
     tsrstvst_org_id,
     tsrbiopt_is_number,
     tsrbiopt_org_id,
     tsrchar_is_number,
     tsrchar_org_id,
     tsrfldpr_is_number,
     tsrfldpr_org_id,
     tsrfldgr_is_number,
     tsrfldgr_org_id,
     tsrgrcfg_is_number,
     tsrgrcfg_org_id,
     qc_indicator,
     chain_of_custody_id,
     tsrmatrx_is_number,
     tsrmatrx_org_id,
     blob_type,
     blob_title)
  VALUES
    (p_activity_rec.tsrfdact_is_number, /* TSRFDACT_IS_NUMBER */
     p_activity_rec.tsrfdact_org_id, /* TSRFDACT_ORG_ID */
     p_activity_rec.type_name, /* TYPE_NAME */
     p_activity_rec.category_type_name, /* CATEGORY_TYPE_NAME */
     p_activity_rec.id_code, /* ID_CODE */
     p_activity_rec.replicate_number, /* REPLICATE_NUMBER */
     p_activity_rec.medium_type_name, /* MEDIUM_TYPE_NAME */
     p_activity_rec.intent_type_name, /* INTENT_TYPE_NAME */
     p_activity_rec.community_name, /* COMMUNITY_NAME */
     p_activity_rec.start_date, /* START_DATE */
     p_activity_rec.start_time, /* START_TIME */
     p_activity_rec.start_time_zone, /* START_TIME_ZONE */
     p_activity_rec.stop_date, /* STOP_DATE */
     p_activity_rec.stop_time, /* STOP_TIME */
     p_activity_rec.stop_time_zone, /* STOP_TIME_ZONE */
     p_activity_rec.comment_text, /* COMMENT_TEXT */
     p_activity_rec.zone_type_name, /* ZONE_TYPE_NAME */
     p_activity_rec.reltv_depth_name, /* RELTV_DEPTH_NAME */
     p_activity_rec.depth_ref_point, /* DEPTH_REF_POINT */
     p_activity_rec.depth_to_activity, /* DEPTH_TO_ACTIVITY */
     p_activity_rec.depth_to_act_un_cd, /* DEPTH_TO_ACT_UN_CD */
     p_activity_rec.upper_depth_to_act, /* UPPER_DEPTH_TO_ACT */
     p_activity_rec.depth_msr_unt_cd, /* DEPTH_MSR_UNT_CD */
     p_activity_rec.lower_depth_to_act, /* LOWER_DEPTH_TO_ACT */
     p_activity_rec.thermocline_loc_cd, /* THERMOCLINE_LOC_CD */
     p_activity_rec.pycnocline_loc_cd, /* PYCNOCLINE_LOC_CD */
     p_activity_rec.halocline_loc_cd, /* HALOCLINE_LOC_CD */
     p_activity_rec.species_number, /* SPECIES_NUMBER */
     p_activity_rec.d_userid_code, /* D_USERID_CODE */
     p_activity_rec.d_last_update_ts, /* D_LAST_UPDATE_TS */
     p_activity_rec.tsropprd_org_id, /* TSROPPRD_ORG_ID */
     p_activity_rec.tsropprd_is_number, /* TSROPPRD_IS_NUMBER */
     p_activity_rec.tsrfdact0is_number, /* TSRFDACT0IS_NUMBER */
     p_activity_rec.tsrfdact0org_id, /* TSRFDACT0ORG_ID */
     p_activity_rec.tsrfdact1is_number, /* TSRFDACT1IS_NUMBER */
     p_activity_rec.tsrfdact1org_id, /* TSRFDACT1ORG_ID */
     p_activity_rec.tsrstvst_is_number, /* TSRSTVST_IS_NUMBER */
     p_activity_rec.tsrstvst_org_id, /* TSRSTVST_ORG_ID */
     p_activity_rec.tsrbiopt_is_number, /* TSRBIOPT_IS_NUMBER */
     p_activity_rec.tsrbiopt_org_id, /* TSRBIOPT_ORG_ID */
     p_activity_rec.tsrchar_is_number, /* TSRCHAR_IS_NUMBER */
     p_activity_rec.tsrchar_org_id, /* TSRCHAR_ORG_ID */
     p_activity_rec.tsrfldpr_is_number, /* TSRFLDPR_IS_NUMBER */
     p_activity_rec.tsrfldpr_org_id, /* TSRFLDPR_ORG_ID */
     p_activity_rec.tsrfldgr_is_number, /* TSRFLDGR_IS_NUMBER */
     p_activity_rec.tsrfldgr_org_id, /* TSRFLDGR_ORG_ID */
     p_activity_rec.tsrgrcfg_is_number, /* TSRGRCFG_IS_NUMBER */
     p_activity_rec.tsrgrcfg_org_id, /* TSRGRCFG_ORG_ID */
     p_activity_rec.qc_indicator,
     p_activity_rec.chain_of_custody_id,
     p_activity_rec.tsrmatrx_is_number,
     p_activity_rec.tsrmatrx_org_id,
     p_activity_rec.blob_type, /* BLOB_TYPE */
     p_activity_rec.blob_title /* BLOB_TITLE */);

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                       SQLERRM || ' - in create activity.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_sample
(
  p_org_id     IN VARCHAR2,
  p_sample_rec IN tsrsmple%ROWTYPE
) RETURN NUMBER IS

BEGIN

  /* create a sample record */

  INSERT INTO tsrsmple
    (tsrfdact_is_number,
     tsrfdact_org_id,
     total_volume_msr,
     total_volume_un_cd,
     total_weight_msr,
     total_weight_un_cd,
     data_file_name_loc,
     container_type_nm,
     container_color,
     container_size_msr,
     container_size_un,
     temp_preservn_type,
     parent_ind_code,
     gear_deployment,
     transport_storage,
     presrv_strge_prcdr,
     duration_time,
     duration_units,
     d_userid_code,
     d_last_update_ts,
     tsmprmvl_is_number,
     tsmprmvl_org_id,
     tsmprmvl0is_number,
     tsmprmvl0org_id,
     tsmprmvl1is_number,
     tsmprmvl1org_id)
  VALUES
    (p_sample_rec.tsrfdact_is_number, /* TSRFDACT_IS_NUMBER */
     p_sample_rec.tsrfdact_org_id, /* TSRFDACT_ORG_ID */
     p_sample_rec.total_volume_msr, /* TOTAL_VOLUME_MSR */
     p_sample_rec.total_volume_un_cd, /* TOTAL_VOLUME_UN_CD */
     p_sample_rec.total_weight_msr, /* TOTAL_WEIGHT_MSR */
     p_sample_rec.total_weight_un_cd, /* TOTAL_WEIGHT_UN_CD */
     p_sample_rec.data_file_name_loc, /* DATA_FILE_NAME_LOC */
     p_sample_rec.container_type_nm, /* CONTAINER_TYPE_NM */
     p_sample_rec.container_color, /* CONTAINER_COLOR */
     p_sample_rec.container_size_msr, /* CONTAINER_SIZE_MSR */
     p_sample_rec.container_size_un, /* CONTAINER_SIZE_UN */
     p_sample_rec.temp_preservn_type, /* TEMP_PRESERVN_TYPE */
     p_sample_rec.parent_ind_code, /* PARENT_IND_CODE */
     p_sample_rec.gear_deployment, /* GEAR_DEPLOYMENT */
     p_sample_rec.transport_storage, /* TRANSPORT_STORAGE */
     p_sample_rec.presrv_strge_prcdr, /* PRESRV_STRGE_PRCDR */
     p_sample_rec.duration_time, /* DURATION_TIME */
     p_sample_rec.duration_units, /* DURATION_UNITS */
     p_sample_rec.d_userid_code, /* D_USERID_CODE */
     p_sample_rec.d_last_update_ts, /* D_LAST_UPDATE_TS */
     p_sample_rec.tsmprmvl_is_number,
     p_sample_rec.tsmprmvl_org_id,
     p_sample_rec.tsmprmvl0is_number,
     p_sample_rec.tsmprmvl0org_id,
     p_sample_rec.tsmprmvl1is_number,
     p_sample_rec.tsmprmvl1org_id);

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in create sample.');
     
        
     v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_sample;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_trip_qc(p_trip_qc_rec IN OUT tsrfqs%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create trip qc sample using info passed in through trip qc record */

  INSERT INTO tsrfqs
    (tsrfqs_is_number,
     tsrfqs_org_id,
     id_code,
     type_name,
     NAME,
     creation_date,
     creation_time,
     time_zone,
     total_volume_msr,
     total_volume_un_cd,
     container_type_nm,
     container_color,
     container_size_msr,
     container_size_un,
     comment_text,
     creation_method,
     transport_storage,
     d_userid_code,
     d_last_update_ts,
     tsrtrip_is_number,
     tsrtrip_org_id)
  VALUES
    (p_trip_qc_rec.tsrfqs_is_number, /* TSRFQS_IS_NUMBER */
     p_trip_qc_rec.tsrfqs_org_id, /* TSRFQS_ORG_ID */
     p_trip_qc_rec.id_code, /* ID_CODE */
     p_trip_qc_rec.type_name, /* TYPE_NAME */
     p_trip_qc_rec.NAME, /* NAME */
     p_trip_qc_rec.creation_date, /* CREATION_DATE */
     p_trip_qc_rec.creation_time, /* CREATION_TIME */
     p_trip_qc_rec.time_zone, /* TIME_ZONE */
     p_trip_qc_rec.total_volume_msr, /* TOTAL_VOLUME_MSR */
     p_trip_qc_rec.total_volume_un_cd, /* TOTAL_VOLUME_UN_CD */
     p_trip_qc_rec.container_type_nm, /* CONTAINER_TYPE_NM */
     p_trip_qc_rec.container_color, /* CONTAINER_COLOR */
     p_trip_qc_rec.container_size_msr, /* CONTAINER_SIZE_MSR */
     p_trip_qc_rec.container_size_un, /* CONTAINER_SIZE_UN */
     p_trip_qc_rec.comment_text, /* COMMENT_TEXT */
     p_trip_qc_rec.creation_method, /* CREATION_METHOD */
     p_trip_qc_rec.transport_storage, /* TRANSPORT_STORAGE */
     p_trip_qc_rec.d_userid_code, /* D_USERID_CODE */
     p_trip_qc_rec.d_last_update_ts, /* D_LAST_UPDATE_TS */
     p_trip_qc_rec.tsrtrip_is_number, /* TSRTRIP_IS_NUMBER */
     p_trip_qc_rec.tsrtrip_org_id /* TSRTRIP_ORG_ID */);

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in create sample.');
     
    
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_trip_qc;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_person_to_activity
(
  p_persn_is_nbr IN NUMBER,
  p_persn_org_id IN VARCHAR2,
  p_act_is_nbr   IN NUMBER,
  p_act_org_id   IN VARCHAR2
) RETURN NUMBER IS

  v_persn_is_nbr NUMBER(8);

  CURSOR c_check_for_person(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2, p_act_is_nbr IN NUMBER, p_act_org_id IN VARCHAR2) IS
    SELECT tsmpersn_is_number
      FROM tsrfapea
     WHERE tsrfdact_is_number = p_act_is_nbr
       AND tsmpersn_is_number = p_persn_is_nbr
       AND tsrfdact_org_id = p_act_org_id
       AND tsmpersn_org_id = p_persn_org_id;

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_person(p_persn_is_nbr, p_persn_org_id, p_act_is_nbr, p_act_org_id);
  FETCH c_check_for_person
    INTO v_persn_is_nbr;
  IF c_check_for_person%NOTFOUND THEN
  
    /* add person to activity if not already there */
    INSERT INTO tsrfapea
      (tsrfdact_is_number,
       tsrfdact_org_id,
       tsmpersn_is_number,
       tsmpersn_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_act_is_nbr, /* TSRFDACT_IS_NUMBER */
       p_act_org_id, /* TSRFDACT_ORG_ID */
       p_persn_is_nbr, /* TSMPERSN_IS_NUMBER */
       p_persn_org_id, /* TSMPERSN_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    COMMIT;
  
  END IF;
  CLOSE c_check_for_person;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in add person to activity.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END add_person_to_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_cporg_to_activity
(
  p_cporg_is_nbr IN NUMBER,
  p_cporg_org_id IN VARCHAR2,
  p_act_is_nbr   IN NUMBER,
  p_act_org_id   IN VARCHAR2
) RETURN NUMBER IS

  CURSOR c_tsrfacoa(p_cporg_is_nbr IN NUMBER, p_cporg_org_id IN VARCHAR2, p_act_is_nbr IN NUMBER, p_act_org_id IN VARCHAR2) IS
    SELECT tsmcporg_is_number,
           tsmcporg_org_id,
           tsrfdact_is_number,
           tsrfdact_org_id
      FROM tsrfacoa
     WHERE tsmcporg_is_number = p_cporg_is_nbr
       AND tsmcporg_org_id = p_cporg_org_id
       AND tsrfdact_is_number = p_act_is_nbr
       AND tsrfdact_org_id = p_act_org_id;

  r_tsrfacoa c_tsrfacoa%ROWTYPE;

BEGIN

  -- try to retrieve the association
  OPEN c_tsrfacoa(p_cporg_is_nbr, p_cporg_org_id, p_act_is_nbr, p_act_org_id);
  FETCH c_tsrfacoa
    INTO r_tsrfacoa;

  -- Do we need to add the Cooperating Organization association?
  IF c_tsrfacoa%NOTFOUND THEN
    --  if yes
  
    -- Add Cooperating Organization association
    INSERT INTO tsrfacoa
      (tsmcporg_is_number,
       tsmcporg_org_id,
       tsrfdact_is_number,
       tsrfdact_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_cporg_is_nbr, -- TSMCPORG_IS_NUMBER
       p_cporg_org_id, -- TSMCPORG_ORG_ID
       p_act_is_nbr, -- TSRFDACT_IS_NUMBER
       p_act_org_id, -- TSRFDACT_ORG_ID
       sim_export_pkg.v_pkg_ownerid, -- D_USERID_CODE
       SYSDATE -- D_LAST_UPDATE_TS
       );
  
    COMMIT;
  
  END IF;

  CLOSE c_tsrfacoa;

  -- return success
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    -- output error message
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in add Cooperating Organization to activity.');
     
        
    v_success := 0;
  
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
    -- return failure
    RETURN 0;
  
END add_cporg_to_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_person_to_trip_qc
(
  p_persn_is_nbr   IN NUMBER,
  p_persn_org_id   IN VARCHAR2,
  p_trip_qc_is_nbr IN NUMBER,
  p_trip_qc_org_id IN VARCHAR2
) RETURN NUMBER IS

  v_persn_is_nbr NUMBER(8);

  CURSOR c_check_for_person(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2, p_trip_qc_is_nbr IN NUMBER, p_trip_qc_org_id IN VARCHAR2) IS
    SELECT tsmpersn_is_number
      FROM tsrfqspa
     WHERE tsrfqs_is_number = p_trip_qc_is_nbr
       AND tsmpersn_is_number = p_persn_is_nbr
       AND tsrfqs_org_id = p_trip_qc_org_id
       AND tsmpersn_org_id = p_persn_org_id;

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_person(p_persn_is_nbr, p_persn_org_id, p_trip_qc_is_nbr, p_trip_qc_org_id);
  FETCH c_check_for_person
    INTO v_persn_is_nbr;
  IF c_check_for_person%NOTFOUND THEN
  
    /* add person to trip qc if not already there */
    INSERT INTO tsrfqspa
      (tsrfqs_is_number,
       tsrfqs_org_id,
       tsmpersn_is_number,
       tsmpersn_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_trip_qc_is_nbr, /* TSRFQS_IS_NUMBER */
       p_trip_qc_org_id, /* TSRFQS_ORG_ID */
       p_persn_is_nbr, /* TSMPERSN_IS_NUMBER */
       p_persn_org_id, /* TSMPERSN_ORG_ID */
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE /* D_LAST_UPDATE_TS */);
  
    COMMIT;
  
  END IF;
  CLOSE c_check_for_person;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in add person to trip qc.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END add_person_to_trip_qc;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION add_field_set_to_activity
(
  p_act_is_nbr     IN NUMBER,
  p_act_org        IN VARCHAR2,
  p_fld_set_is_nbr IN NUMBER,
  p_fld_set_org    IN VARCHAR2
) RETURN NUMBER IS

  v_fdset_is_nbr NUMBER(8);

  CURSOR c_check_for_set(p_act_is_nbr IN NUMBER, p_act_org IN VARCHAR2, p_fld_set_is_nbr IN NUMBER, p_fld_set_org IN VARCHAR2) IS
    SELECT tsrfdset_is_number
      FROM tsrfafsa
     WHERE tsrfdact_is_number = p_act_is_nbr
       AND tsrfdset_is_number = p_fld_set_is_nbr
       AND tsrfdact_org_id = rpad(p_act_org, 8)
       AND tsrfdset_org_id = rpad(p_fld_set_org, 8);

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_set(p_act_is_nbr, p_act_org, p_fld_set_is_nbr, p_fld_set_org);
  FETCH c_check_for_set
    INTO v_fdset_is_nbr;
  IF c_check_for_set%NOTFOUND THEN
  
    /* if field set isn't on activity, add it */
    INSERT INTO tsrfafsa
      (tsrfdact_is_number,
       tsrfdact_org_id,
       tsrfdset_is_number,
       tsrfdset_org_id,
       d_userid_code,
       d_last_update_ts)
    VALUES
      (p_act_is_nbr,
       p_act_org,
       p_fld_set_is_nbr,
       p_fld_set_org,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE);
  
    COMMIT;
  
  END IF;
  CLOSE c_check_for_set;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in add field set to activity.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END add_field_set_to_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_detection_record(p_detection_rec IN tsrdql%ROWTYPE) RETURN NUMBER IS

BEGIN

  NULL;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in create detection record.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_detection_record;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_result
(
  p_result_rec IN tsrrsult%ROWTYPE,
  p_rs_seq     IN NUMBER
) RETURN NUMBER IS

BEGIN

  /* create result using info from result record */
  INSERT INTO tsrrsult
    (tsrrsult_is_number,
     tsrrsult_org_id,
     d_assnd_seq_num,
     comp_ind_cd,
     lab_cert_ind_code,
     value_text,
     value_measure,
     value_status,
     precision_amt_text,
     lab_batch_id_code,
     dilution_ind_code,
     recovery_ind_code,
     correction_ind_cd,
     rslt_txt_ind_cd,
     cell_desc_ind_cd,
     bias,
     conf_lvl_pct_msr,
     conf_lvl_corr_bias,
     proc_excptn_ind_cd,
     dur_basis_type_nm,
     detect_cond_cd,
     smpl_frac_type_nm,
     statistic_type_nm,
     value_type_name,
     wt_basis_type_nm,
     temp_basis_lvl_nm,
     species_number,
     ref_pt_from_name,
     ref_pt_to_name,
     lower_range_value,
     upper_range_value,
     taxon_pollution,
     fnctional_feed_grp,
     trophic_level,
     analysis_date,
     analysis_time,
     analysis_time_zone,
     repl_analysis_num,
     d_userid_code,
     d_last_update_ts,
     tsrfdact_is_number,
     tsrfdact_org_id,
     tsrfqs_is_number,
     tsrfqs_org_id,
     tsranlpr_is_number,
     tsranlpr_org_id,
     tsrbrgi_is_number,
     tsrbrgi_org_id,
     tsrbrg_is_number,
     tsrbrg_org_id,
     tsrchar_is_number,
     tsrchar_org_id,
     tsrchar0is_number,
     tsrchar0org_id,
     tsrlab_is_number,
     tsrlab_org_id,
     tsruom_is_number,
     tsruom_org_id,
     tsrlspp_is_number,
     tsrlspp_org_id,
     tsrhcsc_is_number,
     tsrhcsc_org_id,
     tsmprmvl_is_number,
     tsmprmvl_org_id,
     tsmprmvl0is_number,
     tsmprmvl0org_id,
     tsmprmvl1is_number,
     tsmprmvl1org_id,
     tsrdlin_is_number,
     tsrdlin_org_id,
     blob_type,
     blob_title)
  VALUES
    (p_result_rec.tsrrsult_is_number, /* TSRRSULT_IS_NUMBER */
     p_result_rec.tsrrsult_org_id, /* TSRRSULT_ORG_ID */
     p_result_rec.d_assnd_seq_num, /* D_ASSND_SEQ_NUM */
     p_result_rec.comp_ind_cd, /* COMP_IND_CD */
     p_result_rec.lab_cert_ind_code, /* LAB_CERT_IND_CODE */
     p_result_rec.value_text, /* VALUE_TEXT */
     p_result_rec.value_measure, /* VALUE_MEASURE */
     p_result_rec.value_status, /* VALUE_STATUS */
     p_result_rec.precision_amt_text, /* PRECISION_AMT_TEXT */
     p_result_rec.lab_batch_id_code, /* LAB_BATCH_ID_CODE */
     p_result_rec.dilution_ind_code, /* DILUTION_IND_CODE */
     p_result_rec.recovery_ind_code, /* RECOVERY_IND_CODE */
     p_result_rec.correction_ind_cd, /* CORRECTION_IND_CD */
     p_result_rec.rslt_txt_ind_cd, /* RSLT_TXT_IND_CD */
     p_result_rec.cell_desc_ind_cd, /* CELL_DESC_IND_CD */
     p_result_rec.bias,
     p_result_rec.conf_lvl_pct_msr, /* CONF_LVL_PCT_MSR */
     p_result_rec.conf_lvl_corr_bias,
     p_result_rec.proc_excptn_ind_cd, /* PROC_EXCPTN_IND_CD */
     p_result_rec.dur_basis_type_nm, /* DUR_BASIS_TYPE_NM */
     p_result_rec.detect_cond_cd, /* DETECT_COND_CD */
     p_result_rec.smpl_frac_type_nm, /* SMPL_FRAC_TYPE_NM */
     p_result_rec.statistic_type_nm, /* STATISTIC_TYPE_NM */
     p_result_rec.value_type_name, /* VALUE_TYPE_NAME */
     p_result_rec.wt_basis_type_nm, /* WT_BASIS_TYPE_NM */
     p_result_rec.temp_basis_lvl_nm, /* TEMP_BASIS_LVL_NM */
     p_result_rec.species_number, /* SPECIES_NUMBER */
     p_result_rec.ref_pt_from_name, /* REF_PT_FROM_NAME */
     p_result_rec.ref_pt_to_name, /* REF_PT_TO_NAME */
     p_result_rec.lower_range_value, /* LOWER_RANGE_VALUE */
     p_result_rec.upper_range_value, /* UPPER_RANGE_VALUE */
     p_result_rec.taxon_pollution, /* TAXON_POLLUTION */
     p_result_rec.fnctional_feed_grp, /* FNCTIONAL_FEED_GRP */
     p_result_rec.trophic_level, /* TROPHIC_LEVEL */
     p_result_rec.analysis_date, /* ANALYSIS_DATE */
     p_result_rec.analysis_time, /* ANALYSIS_TIME */
     p_result_rec.analysis_time_zone, /* ANALYSIS_TIME_ZONE */
     p_result_rec.repl_analysis_num, /* REPL_ANALYSIS_NUM */
     p_result_rec.d_userid_code, /* D_USERID_CODE */
     p_result_rec.d_last_update_ts, /* D_LAST_UPDATE_TS */
     p_result_rec.tsrfdact_is_number, /* TSRFDACT_IS_NUMBER */
     p_result_rec.tsrfdact_org_id, /* TSRFDACT_ORG_ID */
     p_result_rec.tsrfqs_is_number, /* TSRFQS_IS_NUMBER */
     p_result_rec.tsrfqs_org_id, /* TSRFQS_ORG_ID */
     p_result_rec.tsranlpr_is_number, /* TSRANLPR_IS_NUMBER */
     p_result_rec.tsranlpr_org_id, /* TSRANLPR_ORG_ID */
     p_result_rec.tsrbrgi_is_number, /* TSRBRGI_IS_NUMBER */
     p_result_rec.tsrbrgi_org_id, /* TSRBRGI_ORG_ID */
     p_result_rec.tsrbrg_is_number, /* TSRBRG_IS_NUMBER */
     p_result_rec.tsrbrg_org_id, /* TSRBRG_ORG_ID */
     p_result_rec.tsrchar_is_number, /* TSRCHAR_IS_NUMBER */
     p_result_rec.tsrchar_org_id, /* TSRCHAR_ORG_ID */
     p_result_rec.tsrchar0is_number, /* TSRCHAR0IS_NUMBER */
     p_result_rec.tsrchar0org_id, /* TSRCHAR0ORG_ID */
     p_result_rec.tsrlab_is_number, /* TSRLAB_IS_NUMBER */
     p_result_rec.tsrlab_org_id, /* TSRLAB_ORG_ID */
     p_result_rec.tsruom_is_number, /* TSRUOM_IS_NUMBER */
     p_result_rec.tsruom_org_id, /* TSRUOM_ORG_ID */
     p_result_rec.tsrlspp_is_number, /* TSRLSPP_IS_NUMBER */
     p_result_rec.tsrlspp_org_id, /* TSRLSPP_ORG_ID */
     p_result_rec.tsrhcsc_is_number, /* TSRHCSC_IS_NUMBER */
     p_result_rec.tsrhcsc_org_id, /* TSRHCSC_ORG_ID */
     p_result_rec.tsmprmvl_is_number, --fraction
     p_result_rec.tsmprmvl_org_id,
     p_result_rec.tsmprmvl0is_number, --habit
     p_result_rec.tsmprmvl0org_id,
     p_result_rec.tsmprmvl1is_number, --voltinism
     p_result_rec.tsmprmvl1org_id,
     p_result_rec.tsrdlin_is_number,
     p_result_rec.tsrdlin_org_id,
     p_result_rec.blob_type, /* BLOB_TYPE */
     p_result_rec.blob_title /* BLOB_TITLE */);

  UPDATE sim_result_class_indicator
     SET rci_tsrrsult_is_number = p_result_rec.tsrrsult_is_number,
         rci_tsrrsult_org_id = p_result_rec.tsrrsult_org_id
   WHERE rci_rs_seq = p_rs_seq;

  IF p_result_rec.tsrdlin_is_number IS NOT NULL THEN
    UPDATE sim_results
       SET rs_tsrdlin_org_id = p_result_rec.tsrdlin_org_id,
           rs_tsrdlin_is_number = p_result_rec.tsrdlin_is_number
     WHERE rs_seq = p_rs_seq;
  END IF;

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in create result.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    RETURN 0;
  
END create_result;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE export_activity
(
  p_activity_seq IN NUMBER,
  p_activity_rec IN OUT tsrfdact%ROWTYPE,
  p_stvst_rec    IN OUT tsrstvst%ROWTYPE,
  p_import_seq   IN NUMBER,
  p_org_id       IN VARCHAR2,
  p_result_cnt   IN OUT NUMBER
) IS

  v_project_id          sim_projects.spj_id%TYPE;
  v_time_zone           VARCHAR2(3);
  v_activity_is_nbr     NUMBER(8);
  v_visit_is_number     NUMBER(8);
  v_field_set_is_number NUMBER(8);
  v_trip_rec            tsrtrip%ROWTYPE;
  v_trap_net_rec        sim_trap_net%ROWTYPE;
  v_tsrtnod_rec         tsrtnod%ROWTYPE;
  v_sample_rec          tsrsmple%ROWTYPE;
  v_visit_id            VARCHAR2(8);
  v_dummy_seq           NUMBER(8);
  v_community_name      VARCHAR2(40);
  v_station_id          VARCHAR2(15);
  v_success             NUMBER(1);
  v_blob_success        NUMBER(1);
  v_ssv_document        sim_station_visits.ssv_document%TYPE;
  e_no_stvst EXCEPTION;
  e_missing_parent EXCEPTION;

  CURSOR c_activity(p_org_id IN VARCHAR2, p_activity_seq IN NUMBER) IS
    SELECT fa_updepth,
           fa_lowdepth,
           fa_depth_range_uom,
           fa_species_number,
           fa_type_name,
           fa_category,
           fa_activity_id,
           fa_start_date,
           fa_start_time,
           fa_stop_date,
           fa_stop_time,
           fa_start_time_zone,
           fa_stop_time_zone,
           fa_depth,
           fa_depth_ref_point,
           fa_thermo,
           fa_halo,
           fa_pycno,
           fa_depth_zone,
           fa_chain_of_custody_id,
           fa_qc_indicator,
           fa_tsrmatrx_is_number,
           fa_tsrmatrx_org_id,
           fa_rel_depth,
           fa_activity_comments,
           fa_tsmstatn_is_number,
           fa_tsmstatn_org_id,
           fa_tsrtrip_is_number,
           fa_tsrtrip_org_id,
           fa_stp_seq,
           fa_tsrstvst_is_number,
           fa_tsrstvst_org_id,
           fa_ssv_seq,
           fa_tsrfldpr_is_number,
           fa_tsrfldpr_org_id,
           fa_tsrfldgr_is_number,
           fa_tsrfldgr_org_id,
           fa_fdset_id,
           fa_fdset_name,
           fa_tsrlab_is_number,
           fa_tsrlab_org_id,
           fa_import_seq,
           fa_export_seq,
           fa_status,
           fa_tsmorgan_org_id,
           fa_approved_by,
           fa_approved_on,
           fa_tsrgrcfg_is_number,
           fa_tsrgrcfg_org_id,
           fa_tsrsdp_is_number,
           fa_tsrsdp_org_id,
           tsrtrip.id_code trip_id,
           fa_stvst_comments,
           fa_depth_uom,
           fa_medium,
           fa_replicate_nbr,
           tsrfldpr.id_code pr_id_code,
           tsrfldgr.id_code gr_id_code,
           fa_intent,
           fa_tsmprmvl_community_is_nbr,
           fa_tsmprmvl_community_org_id,
           fa_tsrbiopt_is_number,
           fa_tsrbiopt_org_id,
           fa_tsrchar_is_number,
           fa_tsrchar_org_id,
           fa_document
      FROM sim_field_activities,
           tsrfldpr,
           tsrfldgr,
           tsrtrip
     WHERE fa_tsrtrip_is_number = tsrtrip_is_number(+)
       AND rpad(fa_tsrtrip_org_id, 8) = tsrtrip_org_id(+)
       AND rpad(fa_tsmorgan_org_id, 8) = rpad(p_org_id, 8)
       AND fa_tsrfldpr_is_number = tsrfldpr_is_number(+)
       AND rpad(fa_tsrfldpr_org_id, 8) = tsrfldpr_org_id(+)
       AND fa_tsrfldgr_is_number = tsrfldgr_is_number(+)
       AND rpad(fa_tsrfldgr_org_id, 8) = tsrfldgr_org_id(+)
       AND fa_export_seq IS NULL
       AND fa_status = 'A'
       AND fa_seq = p_activity_seq;

  CURSOR c_trap_net(p_activity_seq IN NUMBER) IS
    SELECT *
      FROM sim_trap_net
     WHERE stn_fa_seq = p_activity_seq;

  CURSOR c_get_station_id(p_org_id IN VARCHAR2, p_station_is_nbr IN NUMBER) IS
    SELECT identification_cd
      FROM tsmstatn
     WHERE tsmstatn_org_id = rpad(p_org_id, 8)
       AND tsmstatn_is_number = p_station_is_nbr;

  CURSOR c_get_projects(p_activity_seq IN NUMBER) IS
    SELECT sapj_tsmproj_is_number,
           sapj_tsmproj_org_id
      FROM sim_activity_projects
     WHERE sapj_fa_seq = p_activity_seq;

  CURSOR c_get_personnel(p_activity_seq IN NUMBER) IS
    SELECT sap_tsmpersn_is_number,
           sap_tsmpersn_org_id
      FROM sim_activity_personnel
     WHERE sap_fa_seq = p_activity_seq;

  CURSOR c_get_cporg(p_activity_seq IN NUMBER) IS
    SELECT sac_tsmcporg_is_number,
           sac_tsmcporg_org_id
      FROM sim_activity_cporg
     WHERE sac_fa_seq = p_activity_seq;

  CURSOR c_get_visit_info(p_visit_is_number IN NUMBER, p_visit_org_id IN VARCHAR2) IS
    SELECT tsrstvst_is_number
      FROM tsrstvst t
     WHERE t.tsrstvst_is_number = p_visit_is_number
       AND t.tsrstvst_org_id = p_visit_org_id;

  CURSOR c_get_trip_info(p_trip_is_number IN NUMBER, p_trip_org_id IN VARCHAR2) IS
    SELECT tsrtrip_is_number
      FROM tsrtrip t
     WHERE t.tsrtrip_is_number = p_trip_is_number
       AND t.tsrtrip_org_id = p_trip_org_id;

  -- jah
  CURSOR c_get_createdfromsample_parent(cp_fa_seq IN NUMBER) IS
    SELECT fap_parent_is_number,
           fap_parent_org_id,
           fa_import_seq
      FROM sim_field_activities,
           sim_field_activity_parent
     WHERE fap_fa_seq = cp_fa_seq
       AND fa_seq = fap_fa_seq
       AND fa_category = 'Created from Sample';

  -- jah
  CURSOR c_get_composwithparents_parent(p_fa_seq IN NUMBER) IS
    SELECT fap_parent_is_number,
           fap_parent_org_id
      FROM sim_field_activity_parent
     WHERE fap_fa_seq = p_fa_seq;

  -- jah   
  r_get_createdfromsample_parent c_get_createdfromsample_parent%ROWTYPE;
  r_get_composwithparents_parent c_get_composwithparents_parent%ROWTYPE;

BEGIN

  FOR v_activity IN c_activity(p_org_id, p_activity_seq) LOOP
  
    /* get the allowable values */
    BEGIN
    
      IF v_activity.fa_tsmprmvl_community_is_nbr IS NOT NULL
         AND v_activity.fa_tsmprmvl_community_org_id IS NOT NULL THEN
        v_community_name := get_tsmprmvl_value(v_activity.fa_tsmprmvl_community_is_nbr, v_activity.fa_tsmprmvl_community_org_id);
      ELSE
        v_community_name := NULL;
      END IF;
    
      IF v_activity.fa_tsrsdp_is_number IS NOT NULL THEN
        v_sample_rec := get_sample_pres_info(v_activity.fa_tsrsdp_is_number, v_activity.fa_tsrsdp_org_id, v_activity.fa_activity_id);
      ELSE
        v_sample_rec.tsmprmvl_is_number := NULL;
        v_sample_rec.tsmprmvl_org_id := NULL;
        v_sample_rec.tsmprmvl0is_number := NULL;
        v_sample_rec.tsmprmvl0org_id := NULL;
        v_sample_rec.tsmprmvl1is_number := NULL;
        v_sample_rec.tsmprmvl1org_id := NULL;
        v_sample_rec.container_size_msr := NULL;
        v_sample_rec.container_size_un := NULL;
        v_sample_rec.presrv_strge_prcdr := NULL;
      END IF;
    END;
  
    /* Verify that trip still exists in STORET */
    IF v_activity.fa_tsrtrip_is_number IS NOT NULL THEN
    
      OPEN c_get_trip_info(v_activity.fa_tsrtrip_is_number, v_activity.fa_tsrtrip_org_id);
      FETCH c_get_trip_info
        INTO v_dummy_seq;
    
      /* If trip does not still exist, clear is_number and let SIM recreate it. */
      IF c_get_trip_info%NOTFOUND THEN
        v_activity.fa_tsrtrip_is_number := NULL;
        v_activity.fa_tsrtrip_org_id := NULL;
      END IF;
    
      CLOSE c_get_trip_info;
    
    END IF;
  
    /* kms - if the trip does not already exist in STORET, get info needed to create it */
    IF v_activity.fa_tsrtrip_is_number IS NULL THEN
      get_trip_info(v_activity.fa_stp_seq, v_trip_rec);
    
      /* kms - call function to get next trip is number */
      v_activity.fa_tsrtrip_is_number := get_next_trip_is_nbr(p_org_id);
    
      v_trip_rec.tsrtrip_is_number := v_activity.fa_tsrtrip_is_number;
      v_trip_rec.tsrtrip_org_id := p_org_id;
    
      IF v_trip_rec.start_time IS NULL THEN
        v_trip_rec.start_time := to_date('01-JAN-0001', 'DD-MON-YYYY'); --Default null value
      END IF;
    
      v_trip_rec.start_time_zone := NULL; --v_time_zone; we don't import trip start time zone
      IF v_trip_rec.end_date IS NULL THEN
        v_trip_rec.end_date := to_date('01-JAN-0001', 'DD-MON-YYYY'); --Default null value
      END IF;
      IF v_trip_rec.end_time IS NULL THEN
        v_trip_rec.end_time := to_date('01-JAN-0001', 'DD-MON-YYYY'); --Default null value
      END IF;
      v_trip_rec.end_time_zone := NULL; --v_time_zone;
      v_trip_rec.single_proj_ind := 'Y';
      v_trip_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
      v_trip_rec.d_last_update_ts := SYSDATE;
    
      /* kms - create the trip in STORET */
      v_success := create_trip(v_trip_rec);
    
      IF v_activity.fa_tsrtrip_is_number IS NOT NULL THEN
        v_activity.fa_tsrtrip_org_id := p_org_id;
      
        /* kms (6/6/01) - fill in the trip fk columns in the sim_trips record
        and in every activity with this trip. */
        UPDATE sim_trips
           SET stp_tsrtrip_is_number = v_activity.fa_tsrtrip_is_number,
               stp_tsrtrip_org_id = v_activity.fa_tsrtrip_org_id
         WHERE stp_seq = v_activity.fa_stp_seq;
      
        UPDATE sim_field_activities
           SET fa_tsrtrip_is_number = v_activity.fa_tsrtrip_is_number,
               fa_tsrtrip_org_id = v_activity.fa_tsrtrip_org_id
         WHERE fa_stp_seq = v_activity.fa_stp_seq;
      
        COMMIT;
      END IF;
    END IF;
  
    OPEN c_get_station_id(p_org_id, v_activity.fa_tsmstatn_is_number);
    FETCH c_get_station_id
      INTO v_station_id;
    -- make sure the station still exists in STORET
    IF c_get_station_id%NOTFOUND THEN
      BEGIN
        SELECT s.sta_id
          INTO v_station_id
          FROM sim_stations s
          WHERE s.sta_is_number = v_activity.fa_tsmstatn_is_number;
      EXCEPTION
        WHEN OTHERS THEN 
          NULL;
      END;
      sim_event_log_pkg.write_event_detail(
        sim_lib_pkg.c_seldt_error,
        'sim_export_result_pkg.export_activity',
        'Station with ID "' || v_station_id || '" cannot be found in STORET.  '
          || 'Data from this migration will be in an invalid state in STORET.  '
          || 'You should delete this entire migration '
          || 'and re-import the file again after adding the station back into STORET');
      v_success := 0;
    END IF;
    CLOSE c_get_station_id;
    /* call function to get the next activity is number */
    v_activity_is_nbr := get_next_activity_is_nbr(p_org_id);
  
    UPDATE sim_field_activities
       SET fa_tsrfdact_is_number = v_activity_is_nbr,
           fa_tsrfdact_org_id = p_org_id
     WHERE fa_seq = p_activity_seq;
  
    UPDATE sim_actual_activity_loc
       SET aal_tsrfdact_is_number = v_activity_is_nbr,
           aal_tsrfdact_org_id = p_org_id
     WHERE aal_fa_seq = p_activity_seq;
  
    UPDATE sim_trawl_ops_det
       SET tod_tsrfdact_is_number = v_activity_is_nbr,
           tod_tsrfdact_org_id = p_org_id
     WHERE tod_fa_seq = p_activity_seq;
  
    UPDATE sim_sample
       SET ss_tsrfdact_is_number = v_activity_is_nbr,
           ss_tsrfdact_org_id = p_org_id
     WHERE ss_fa_seq = p_activity_seq;
  
    UPDATE sim_trap_net
       SET stn_tsrfdact_is_number = v_activity_is_nbr,
           stn_tsrfdact_org_id = p_org_id
     WHERE stn_fa_seq = p_activity_seq;
    
    /* Add record to tsrtsa to associate station with trip  */
    BEGIN
      add_station_to_trip(v_activity.fa_tsrtrip_is_number, v_activity.fa_tsmstatn_is_number, p_org_id);
    EXCEPTION
      WHEN OTHERS THEN 
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' -  writing station visited on trip record: ' ||
                                             v_activity.fa_tsmstatn_is_number);
    END;
  
    /* kms (6/4/01) - if station visit does not already exist, move what's
    in SIM into a new station visit record in STORET. */
    BEGIN
    
      /* Verify that visit still exists in STORET */
    
      -- Do we have a Station Visit IS_NUMBER in SIM_FIELD_ACTIVITIES record?
      IF v_activity.fa_tsrstvst_is_number IS NOT NULL THEN
        -- if yes
      
        -- Try to retrieve the Station Visit record from STORET
        OPEN c_get_visit_info(v_activity.fa_tsrstvst_is_number, v_activity.fa_tsrstvst_org_id);
        FETCH c_get_visit_info
          INTO v_dummy_seq;
      
        /* If visit does not still exist, clear is_number and let SIM recreate it. */
      
        -- Does the Station Visit record still exist in STORET?
        IF c_get_visit_info%NOTFOUND THEN
          -- if no
          -- Tell subsequent processing to create a Station Visit Record in STORET for this activity
          v_activity.fa_tsrstvst_is_number := NULL;
          v_activity.fa_tsrstvst_org_id := NULL;
        END IF;
      
        CLOSE c_get_visit_info;
      
      END IF;
    
      -- Do we need to create a Station Visit record in STORET for this activity?
      IF v_activity.fa_tsrstvst_is_number IS NULL THEN
        -- if yes
      
        /* kms - get the station visit info from SIM_STATION_VISITS */
        -- 04/04/2004 (RyanJ) added v_ssv_document param so that I can get the case sensitive file name (for Linux compatibility)
        get_visit_info(v_activity.fa_ssv_seq, p_stvst_rec, v_visit_id, v_ssv_document);
      
        /* kms (6/8/01) - if the station visit id has not yet been assigned, use the
        max id nbr used in STORET and increment it by 1. */
        IF v_visit_id LIKE 'UNA%' THEN
          p_stvst_rec.id_number := get_next_visit_id(v_activity.fa_stp_seq, v_activity.fa_tsrtrip_is_number, v_activity.fa_tsrtrip_org_id, v_activity.fa_tsmstatn_is_number, v_activity.fa_tsmstatn_org_id);
        ELSE
          p_stvst_rec.id_number := v_visit_id;
        END IF;
      
        /* kms - get the next station visit is number */
        v_visit_is_number := get_next_visit_is_number(p_org_id);
        p_stvst_rec.tsrstvst_is_number := v_visit_is_number;
        p_stvst_rec.tsrstvst_org_id := rpad(p_org_id, 8);
      
        /* kms - set remaining variables for station visit fields and create station visit in STORET.
        Insert NULL defaults if NULL */
        IF p_stvst_rec.arrival_time IS NULL THEN
          p_stvst_rec.arrival_time := to_date('01-JAN-0001', 'DD-MON-YYYY'); --Default null value
          p_stvst_rec.arrival_time_zone := NULL;
        END IF;
        IF p_stvst_rec.departure_date IS NULL THEN
          p_stvst_rec.departure_date := to_date('01-JAN-0001', 'DD-MON-YYYY'); --Default null value
        END IF;
        IF p_stvst_rec.departure_time IS NULL THEN
          p_stvst_rec.departure_time := to_date('01-JAN-0001', 'DD-MON-YYYY'); --Default null value
          p_stvst_rec.deprture_time_zone := NULL;
        END IF;
      
        p_stvst_rec.comment_text := v_activity.fa_stvst_comments;
        p_stvst_rec.tsrtrip_is_number := v_activity.fa_tsrtrip_is_number;
        p_stvst_rec.tsrtrip_org_id := v_activity.fa_tsrtrip_org_id;
        p_stvst_rec.tsmstatn_is_number := v_activity.fa_tsmstatn_is_number;
        p_stvst_rec.tsmstatn_org_id := v_activity.fa_tsmstatn_org_id;
      
        IF p_stvst_rec.id_number IS NOT NULL THEN
          v_success := create_station_visit(p_org_id, p_stvst_rec);
        ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             'Did not get a station visit id number, unable to create station visit.');
         
            
          RAISE e_no_stvst;
        END IF;
      
        IF p_stvst_rec.tsrstvst_is_number IS NOT NULL THEN
        
          /* export binary document for visit */
          IF p_stvst_rec.blob_title IS NOT NULL THEN
            -- 04/04/2005 (RyanJ) replace blob_title.blob_type with v_ssv_document 
            -- because title/type are upper-case and are not appropriate for Linux compatibility
            -- 11/02/05 (ScottJ) Changed call to insert_binary_document to use one function
            -- in the export pkg instead of each export having it's own binary_doc function
            v_blob_success := sim_export_pkg.insert_binary_document(
              p_export_log_seq => v_export_log_seq, 
              p_table_name => 'TSRSTVST', 
              p_userid_code => p_stvst_rec.d_userid_code, 
              p_file_name => v_ssv_document, --p_stvst_rec.blob_title||'.'||p_stvst_rec.blob_type);
              p_tsrstvst_is_number => v_visit_is_number, 
              p_tsrstvst_org_id => p_org_id); 
            IF v_blob_success = 0 THEN
              v_success := 0;
              IF t_success > v_success THEN
                t_success := v_success;
              END IF;
            END IF;
          END IF;
        
          /* kms (6/6/01) - fill in the visit fk columns in the sim_trips record
          and in every activity with visit trip. */
        
          v_activity.fa_tsrstvst_is_number := p_stvst_rec.tsrstvst_is_number;
          v_activity.fa_tsrstvst_org_id := p_stvst_rec.tsrstvst_org_id;
        
          UPDATE sim_station_visits
             SET ssv_tsrstvst_is_number = v_activity.fa_tsrstvst_is_number,
                 ssv_tsrstvst_org_id = v_activity.fa_tsrstvst_org_id,
                 ssv_id = rtrim(p_stvst_rec.id_number)
           WHERE ssv_seq = v_activity.fa_ssv_seq;
        
          UPDATE sim_field_activities
             SET fa_tsrstvst_is_number = v_activity.fa_tsrstvst_is_number,
                 fa_tsrstvst_org_id = v_activity.fa_tsrstvst_org_id
           WHERE fa_ssv_seq = v_activity.fa_ssv_seq;
        
          COMMIT;
        ELSE
          RAISE e_no_stvst;
        END IF;
      
      ELSE
      
        v_visit_is_number := v_activity.fa_tsrstvst_is_number;
      
        -- jah 3-26-2004
        -- pass the station visit is number back to the calling routine
        p_stvst_rec.tsrstvst_is_number := v_visit_is_number;
      
      END IF;
    
    EXCEPTION
      WHEN e_no_stvst THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             'Did not find or create station visit');
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
      
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' -  writing visit record for station');
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    
  
    /* Issue other STORET NULL Value defaults. */
    --FOR TSRFDACT
    IF v_activity.fa_replicate_nbr IS NULL THEN
      v_activity.fa_replicate_nbr := 0;
    END IF;
    IF v_activity.fa_stop_date IS NULL THEN
      v_activity.fa_stop_date := to_date('01-JAN-0001', 'DD-MON-YYYY');
    END IF;
    IF v_activity.fa_start_time IS NULL THEN
      v_activity.fa_start_time := to_date('01-JAN-0001', 'DD-MON-YYYY');
    END IF;
    IF v_activity.fa_stop_time IS NULL THEN
      v_activity.fa_stop_time := to_date('01-JAN-0001', 'DD-MON-YYYY');
    END IF;
  
    BEGIN
      /* write the activity record */
    
      p_activity_rec.tsrfdact_is_number := v_activity_is_nbr;
      p_activity_rec.tsrfdact_org_id := rpad(p_org_id, 8);
      p_activity_rec.category_type_name := v_activity.fa_category;
      p_activity_rec.id_code := rpad(v_activity.fa_activity_id, 12);
      p_activity_rec.replicate_number := v_activity.fa_replicate_nbr;
      p_activity_rec.medium_type_name := v_activity.fa_medium;
      p_activity_rec.intent_type_name := v_activity.fa_intent;
      p_activity_rec.community_name := v_community_name;
      p_activity_rec.start_date := trunc(v_activity.fa_start_date);
      p_activity_rec.start_time := v_activity.fa_start_time;
      p_activity_rec.start_time_zone := v_activity.fa_start_time_zone;
      p_activity_rec.stop_date := v_activity.fa_stop_date;
      p_activity_rec.stop_time := v_activity.fa_stop_time;
      p_activity_rec.stop_time_zone := v_activity.fa_stop_time_zone;
      p_activity_rec.comment_text := v_activity.fa_activity_comments;
      p_activity_rec.zone_type_name := v_activity.fa_depth_zone;
      p_activity_rec.reltv_depth_name := rpad(v_activity.fa_rel_depth, 15);
      p_activity_rec.depth_ref_point := v_activity.fa_depth_ref_point;
      p_activity_rec.depth_to_activity := v_activity.fa_depth;
      p_activity_rec.depth_to_act_un_cd := rpad(v_activity.fa_depth_uom, 3);
      p_activity_rec.upper_depth_to_act := v_activity.fa_updepth;
      p_activity_rec.depth_msr_unt_cd := rpad(v_activity.fa_depth_range_uom, 3);
      p_activity_rec.lower_depth_to_act := v_activity.fa_lowdepth;
      p_activity_rec.thermocline_loc_cd := v_activity.fa_thermo;
      p_activity_rec.pycnocline_loc_cd := v_activity.fa_pycno;
      p_activity_rec.halocline_loc_cd := v_activity.fa_halo;
      p_activity_rec.species_number := v_activity.fa_species_number;
      p_activity_rec.qc_indicator := v_activity.fa_qc_indicator;
      p_activity_rec.chain_of_custody_id := v_activity.fa_chain_of_custody_id;
      p_activity_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
      p_activity_rec.d_last_update_ts := SYSDATE;
      p_activity_rec.tsropprd_org_id := NULL;
      p_activity_rec.tsropprd_is_number := NULL;
      p_activity_rec.tsrfdact0is_number := NULL;
      p_activity_rec.tsrfdact0org_id := NULL;
    
      -- jah        
      --        p_activity_rec.TSRFDACT1IS_NUMBER := v_activity.fa_tsrfdact1is_number;
      --        p_activity_rec.TSRFDACT1ORG_ID := RPAD(v_activity.tsrfdact1org_id,8);
      -- Does this activity have a Parent Field Activity?
      IF (v_activity.fa_category = 'Created from Sample') THEN
        -- if yes
        -- Retrieve the parent information
        OPEN c_get_createdfromsample_parent(p_activity_seq);
        FETCH c_get_createdfromsample_parent
          INTO r_get_createdfromsample_parent;
        CLOSE c_get_createdfromsample_parent;
      
        -- is the parent already in STORET?
        -- Set the createdFromSample key
        p_activity_rec.tsrfdact1is_number := r_get_createdfromsample_parent.fap_parent_is_number;
        p_activity_rec.tsrfdact1org_id := rpad(r_get_createdfromsample_parent.fap_parent_org_id, 8);
      ELSE
        -- if no (this activity does not require a Parent Sample ID)
        -- set the createdFromSample key to NULL
        p_activity_rec.tsrfdact1is_number := NULL;
        p_activity_rec.tsrfdact1org_id := NULL;
      END IF;
    
      p_activity_rec.tsrstvst_is_number := v_visit_is_number;
      p_activity_rec.tsrstvst_org_id := rpad(p_org_id, 8);
      p_activity_rec.tsrbiopt_is_number := v_activity.fa_tsrbiopt_is_number;
      p_activity_rec.tsrbiopt_org_id := v_activity.fa_tsrbiopt_org_id;
      p_activity_rec.tsrchar_is_number := v_activity.fa_tsrchar_is_number;
      p_activity_rec.tsrchar_org_id := v_activity.fa_tsrchar_org_id;
      p_activity_rec.tsrfldpr_is_number := v_activity.fa_tsrfldpr_is_number;
      p_activity_rec.tsrfldpr_org_id := v_activity.fa_tsrfldpr_org_id;
      p_activity_rec.tsrfldgr_is_number := v_activity.fa_tsrfldgr_is_number;
      p_activity_rec.tsrfldgr_org_id := v_activity.fa_tsrfldgr_org_id;
      p_activity_rec.tsrgrcfg_is_number := v_activity.fa_tsrgrcfg_is_number;
      p_activity_rec.tsrgrcfg_org_id := v_activity.fa_tsrgrcfg_org_id;
      p_activity_rec.tsrmatrx_is_number := v_activity.fa_tsrmatrx_is_number;
      p_activity_rec.tsrmatrx_org_id := v_activity.fa_tsrmatrx_org_id;
      p_activity_rec.blob_title := sim_export_pkg.get_file_name(v_activity.fa_document);
      p_activity_rec.blob_type := sim_export_pkg.get_file_type(v_activity.fa_document);
    
      IF v_visit_is_number IS NOT NULL THEN
        v_success := create_activity(p_activity_rec);
      END IF;
    
      -- Did the Field Activity get inserted into TSRFDACT?        
      IF v_success = 1 THEN
        -- if yes
      
        /* 
          jah 2-13-04
          if p_activity_seq is in SIM_FIELD_ACTIVITY_PARENT, then this record is a 
          parent to some activity that occurs later in the import. So to prepare
          for inserting the Child, we find the child record in SIM_FIELD_ACTIVITY_PARENT
          and update the parent IS_NUMBER and ORG_ID.
        */
        -- Put the parent IS_NUMBER and the ORG_ID into SIM_FIELD_ACTIVITY_PARENT.
        UPDATE sim_field_activity_parent
           SET fap_parent_is_number = v_activity_is_nbr,
               fap_parent_org_id = rpad(p_org_id, 8)
         WHERE fap_parent_fa_seq = p_activity_seq;
        COMMIT;
      
        -- jah 2-13-04
        -- Do we need to update a Parent Field Activity when we insert this Field Activity?
        IF (v_activity.fa_category = 'Composite-with Parents') THEN
          -- if yes
        
          -- For each parent update the reference to this 'Composite...' child
          FOR r_get_composwithparents_parent IN c_get_composwithparents_parent(p_activity_seq) LOOP
          
            -- Is the parent in STORET?
            IF (r_get_composwithparents_parent.fap_parent_is_number IS NOT NULL) THEN
              -- if yes
              -- Update the 'Composite-with Parent' key in the parent activity
              UPDATE tsrfdact
                 SET tsrfdact0is_number = p_activity_rec.tsrfdact_is_number,
                     tsrfdact0org_id = rpad(p_activity_rec.tsrfdact_org_id, 8)
               WHERE tsrfdact_is_number =
                     r_get_composwithparents_parent.fap_parent_is_number
                 AND tsrfdact_org_id =
                     rpad(r_get_composwithparents_parent.fap_parent_org_id, 8);
            ELSE
              RAISE e_missing_parent;
            END IF;
          END LOOP;
        
        END IF;
      
        /* export binary document for activity */
        IF v_activity.fa_document IS NOT NULL THEN
          v_blob_success := sim_export_pkg.insert_binary_document(
          p_export_log_seq => v_export_log_seq, 
          p_table_name => 'TSRFDACT', 
          p_userid_code => p_activity_rec.d_userid_code, 
          p_file_name => v_activity.fa_document, 
          p_tsrfdact_is_number => v_activity_is_nbr, 
          p_tsrfdact_org_id => p_org_id);
          IF v_blob_success = 0 THEN
            v_success := 0;
            IF t_success > v_success THEN
              t_success := v_success;
            END IF;
          END IF;
        END IF;
      
        OPEN c_trap_net(p_activity_seq);
        FETCH c_trap_net
          INTO v_trap_net_rec;
        IF c_trap_net%FOUND THEN
        
          --Default NULL values For TSRTNOD
          IF v_trap_net_rec.stn_rel_current IS NULL THEN
            v_trap_net_rec.stn_rel_current := 0;
          END IF;
          IF v_trap_net_rec.stn_rel_wind IS NULL THEN
            v_trap_net_rec.stn_rel_wind := 0;
          END IF;
        
          v_tsrtnod_rec.tsrfdact_is_number := v_trap_net_rec.stn_tsrfdact_is_number;
          v_tsrtnod_rec.tsrfdact_org_id := v_trap_net_rec.stn_tsrfdact_org_id;
          v_tsrtnod_rec.smplng_duratn_msr := v_trap_net_rec.stn_sampling_dur;
          v_tsrtnod_rec.smplng_drtn_unt_cd := v_trap_net_rec.stn_sampling_dur_unit;
          v_tsrtnod_rec.orientn_to_current := v_trap_net_rec.stn_orient_to_cur;
          v_tsrtnod_rec.rel_current_dir := v_trap_net_rec.stn_rel_current;
          v_tsrtnod_rec.rel_wind_dir := v_trap_net_rec.stn_rel_wind;
          v_tsrtnod_rec.comment_text := v_trap_net_rec.stn_comment;
          v_tsrtnod_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
          v_tsrtnod_rec.d_last_update_ts := SYSDATE;
        
          v_success := create_trap_net(v_tsrtnod_rec);
        END IF;
        CLOSE c_trap_net;
      END IF;
    
    EXCEPTION
      WHEN e_missing_parent THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             'Error: Field Activity ' ||
                                             v_activity.fa_activity_id ||
                                             ' did not migrate, ' ||
                                             'because it requires a Parent Field Activity in STORET.');
         
            
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' -  writing activity ');   
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
    
    /* write the 'actual activity location' TSRAAL records */
    BEGIN
      v_success := create_activity_location(p_org_id, v_activity_is_nbr, p_activity_seq, v_activity.fa_tsmstatn_is_number, v_activity.fa_tsmstatn_org_id);
    
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - Writing tsraal record.');   
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* write the 'sim_sample' TSRSMPLE record */
    BEGIN
      v_success := create_tsrsmple_record(p_org_id, v_activity_is_nbr, p_activity_seq, v_sample_rec);
    EXCEPTION
      WHEN OTHERS THEN
          
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                            ' - Writing tsrsmple record.');   
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
    
    /* write the 'trawl ops det' TSRTOD record */
    BEGIN
      v_success := create_tsrtod_record(p_org_id, v_activity_is_nbr, p_activity_seq);
    EXCEPTION
      WHEN OTHERS THEN
                                                                                                                                                                                                                                                                                                                                                                                                                                                               
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - Writing tsrtod record.');     
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* write the 'bio results grps' TSRBRG records */
    BEGIN
      v_success := create_tsrbrg_records(p_org_id, v_activity_is_nbr, p_activity_seq);
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - Writing tsrbrg record.');
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* write the 'bio results grp individual' TSRBRGI records */
    BEGIN
      v_success := create_tsrbrgi_records(p_org_id, v_activity_is_nbr, p_activity_seq);
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - Writing tsrbrgi record.');
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
    /* loop through projects, see if project is on the trip and activity, and add if it's not */
    BEGIN
      FOR v_project IN c_get_projects(p_activity_seq) LOOP
        BEGIN
          SELECT p.tsmproj_is_number
            INTO v_project.sapj_tsmproj_is_number
            FROM tsmproj p
            WHERE p.tsmproj_is_number = v_project.sapj_tsmproj_is_number;
        EXCEPTION
          WHEN OTHERS THEN
              BEGIN
                SELECT p.spj_id
                  INTO v_project_id
                  FROM sim_projects p
                  WHERE p.spj_tsmproj_is_number = v_project.sapj_tsmproj_is_number;
              EXCEPTION
                WHEN OTHERS THEN 
                  NULL;
              END;
              sim_event_log_pkg.write_event_detail(
                sim_lib_pkg.c_seldt_error,
                'sim_export_result_pkg.export_activity',
                'One or more projects in your migration cannot be found in STORET.  '
                  || 'Data from this migration will be in an invalid state in STORET.  '
                  || 'You should delete this entire migration '
                  || 'and re-import the file again after adding the project back into STORET');
              v_success := 0;
        END;
        add_project_to_trip(v_activity.fa_tsrtrip_is_number, v_project.sapj_tsmproj_is_number, p_org_id);
        add_project_to_activity(v_activity_is_nbr, v_project.sapj_tsmproj_is_number, p_org_id);
      
        /* make sure station is on project, add if it isn't */
        v_success := add_station_to_project(v_project.sapj_tsmproj_is_number, v_activity.fa_tsmstatn_is_number, p_org_id);
      
        /* loop through personnel, assign to role and project*/
        BEGIN
          FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP
          
            v_success := add_person_to_project(v_personnel.sap_tsmpersn_is_number, rpad(v_personnel.sap_tsmpersn_org_id, 8), v_project.sapj_tsmproj_is_number, rpad(p_org_id, 8));
          
          END LOOP;
        
        EXCEPTION
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                 SQLERRM ||
                                                 ' - adding person to project.');
             
                
            v_success := 0;
            IF t_success > v_success THEN
              t_success := v_success;
            END IF;
        END;
      
        /* loop through cooperating organizations and assign to project*/
        BEGIN
          FOR v_cporg IN c_get_cporg(p_activity_seq) LOOP
          
            v_success := add_cporg_to_project(v_cporg.sac_tsmcporg_is_number, rpad(v_cporg.sac_tsmcporg_org_id, 8), v_project.sapj_tsmproj_is_number, rpad(p_org_id, 8));
          
          END LOOP;
        
        EXCEPTION
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                 SQLERRM ||
                                                ' - adding person to project.');
             
                
            v_success := 0;
            IF t_success > v_success THEN
              t_success := v_success;
            END IF;
        END;
        
      END LOOP;
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - adding project to trip. Trip_is_nbr: ' ||
                                             v_activity.fa_tsrtrip_is_number);
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* loop through personnel and add to the activity */
    BEGIN
      FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP
      
        v_success := add_person_to_activity(v_personnel.sap_tsmpersn_is_number, rpad(v_personnel.sap_tsmpersn_org_id, 8), v_activity_is_nbr, rpad(p_org_id, 8));
      
      END LOOP;
    
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                              ' - adding person to activity.');
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* loop through cooperating organizations and add to the activity */
    BEGIN
      FOR v_cporg IN c_get_cporg(p_activity_seq) LOOP
      
        v_success := add_cporg_to_activity(v_cporg.sac_tsmcporg_is_number, rpad(v_cporg.sac_tsmcporg_org_id, 8), v_activity_is_nbr, rpad(p_org_id, 8));
      
      END LOOP;
    
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - adding cooperating organization to activity.');
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* get the field set is number from STORET if exists or create if not */
    IF v_activity.fa_fdset_id IS NOT NULL THEN
      v_field_set_is_number := get_field_set_is_nbr(p_org_id, v_visit_is_number, v_activity.fa_fdset_id, v_activity.fa_fdset_name);
    
      BEGIN
        /* associate activity with field set */
        v_success := add_field_set_to_activity(v_activity_is_nbr, rpad(p_org_id, 8), v_field_set_is_number, rpad(p_org_id, 8));
      EXCEPTION
        WHEN OTHERS THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM ||
                                               ' - adding field set to activity.');
           
              
          v_success := 0;
          IF t_success > v_success THEN
            t_success := v_success;
          END IF;
      END;
    ELSE
      v_field_set_is_number := NULL;
    END IF;
  
    /* IF QC RESULTS, THERE WON'T BE AN ACTIVITY IS NBR ... */
    BEGIN
    
      export_results(p_activity_seq, p_activity_rec.type_name, v_activity_is_nbr, p_org_id, NULL, NULL, v_activity.fa_tsrlab_is_number, v_activity.fa_tsrlab_org_id, p_org_id, v_export_log_seq, p_result_cnt, v_time_zone);
    
    END;
  END LOOP;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in export activity.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END export_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE export_trip_qc
(
  p_activity_seq IN NUMBER,
  p_trip_qc_rec  IN OUT tsrfqs%ROWTYPE,
  p_import_seq   IN NUMBER,
  p_org_id       IN VARCHAR2,
  p_result_cnt   IN OUT NUMBER
) IS

  v_time_zone      VARCHAR2(3);
  v_trip_qc_is_nbr NUMBER(8);
  v_trip_rec       tsrtrip%ROWTYPE;
  v_trip_copy_rec  tsrtrip%ROWTYPE;

  v_success NUMBER(1);

  CURSOR c_activity(p_org_id IN VARCHAR2, p_activity_seq IN NUMBER) IS
    SELECT fa_category,
           fa_activity_id,
           fa_start_date,
           fa_start_time,
           fa_start_time_zone,
           fa_stop_time_zone,
           fa_activity_comments,
           fa_tsrtrip_is_number,
           fa_tsrtrip_org_id,
           fa_tsrlab_is_number,
           fa_tsrlab_org_id,
           fa_import_seq,
           fa_export_seq,
           fa_status,
           fa_tsmorgan_org_id,
           fa_approved_by,
           fa_approved_on,
           fa_medium,
           fa_stp_seq
      FROM sim_field_activities,
           tsrtrip
     WHERE fa_tsrtrip_is_number = tsrtrip_is_number(+)
       AND rpad(fa_tsrtrip_org_id, 8) = tsrtrip_org_id(+)
       AND rpad(fa_tsmorgan_org_id, 8) = rpad(p_org_id, 8)
       AND fa_export_seq IS NULL
       AND fa_status = 'A'
       AND fa_seq = p_activity_seq;

  CURSOR c_get_projects(p_activity_seq IN NUMBER) IS
    SELECT sapj_tsmproj_is_number,
           sapj_tsmproj_org_id
      FROM sim_activity_projects
     WHERE sapj_fa_seq = p_activity_seq;

  CURSOR c_get_personnel(p_activity_seq IN NUMBER) IS
    SELECT sap_tsmpersn_is_number,
           sap_tsmpersn_org_id
      FROM sim_activity_personnel
     WHERE sap_fa_seq = p_activity_seq;

BEGIN

  FOR v_activity IN c_activity(p_org_id, p_activity_seq) LOOP
    /* kms - if the trip does not already exist in STORET, get info needed to create it */
    IF v_activity.fa_tsrtrip_is_number IS NULL THEN
      get_trip_info(v_activity.fa_stp_seq, v_trip_rec);
    
      /* kms - call function to get next trip is number */
      v_activity.fa_tsrtrip_is_number := get_next_trip_is_nbr(p_org_id);
    
      v_trip_rec.tsrtrip_is_number := v_activity.fa_tsrtrip_is_number;
      v_trip_rec.tsrtrip_org_id := p_org_id;
      v_trip_rec.start_time_zone := NULL; --v_time_zone;
      v_trip_rec.end_time_zone := NULL; --v_time_zone;
      v_trip_rec.single_proj_ind := 'Y';
      v_trip_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
      v_trip_rec.d_last_update_ts := SYSDATE;
    
      /* kms - create the trip in STORET */
      v_success := create_trip(v_trip_rec);
    
      IF v_activity.fa_tsrtrip_is_number IS NOT NULL THEN
        v_activity.fa_tsrtrip_org_id := p_org_id;
      
        /* kms (6/6/01) - fill in the trip fk columns in the sim_trips record
        and in every activity with this trip. */
        UPDATE sim_trips
           SET stp_tsrtrip_is_number = v_activity.fa_tsrtrip_is_number,
               stp_tsrtrip_org_id = v_activity.fa_tsrtrip_org_id
         WHERE stp_seq = v_activity.fa_stp_seq;
      
        UPDATE sim_field_activities
           SET fa_tsrtrip_is_number = v_activity.fa_tsrtrip_is_number,
               fa_tsrtrip_org_id = v_activity.fa_tsrtrip_org_id
         WHERE fa_stp_seq = v_activity.fa_stp_seq;
      
        COMMIT;
      
      END IF;
    END IF;
  
    /* call function to get the next activity is number */
    v_trip_qc_is_nbr := get_next_trip_qc_is_nbr(p_org_id);
  
    BEGIN
    
      p_trip_qc_rec.tsrfqs_is_number := v_trip_qc_is_nbr;
      p_trip_qc_rec.tsrfqs_org_id := rpad(p_org_id, 8);
      p_trip_qc_rec.id_code := v_activity.fa_activity_id;
      p_trip_qc_rec.type_name := v_activity.fa_category;
      p_trip_qc_rec.NAME := NULL;
      p_trip_qc_rec.creation_date := v_activity.fa_start_date;
      p_trip_qc_rec.creation_time := v_activity.fa_start_time;
      p_trip_qc_rec.time_zone := v_activity.fa_start_time_zone;
      p_trip_qc_rec.total_volume_msr := NULL;
      p_trip_qc_rec.total_volume_un_cd := NULL;
      p_trip_qc_rec.container_type_nm := NULL;
      p_trip_qc_rec.container_color := NULL;
      p_trip_qc_rec.container_size_msr := NULL;
      p_trip_qc_rec.container_size_un := NULL;
      p_trip_qc_rec.comment_text := v_activity.fa_activity_comments;
      p_trip_qc_rec.creation_method := NULL;
      p_trip_qc_rec.transport_storage := NULL;
      p_trip_qc_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
      p_trip_qc_rec.d_last_update_ts := SYSDATE;
      p_trip_qc_rec.tsrtrip_is_number := v_activity.fa_tsrtrip_is_number;
      p_trip_qc_rec.tsrtrip_org_id := v_activity.fa_tsrtrip_org_id;
    
      /* write the trip qc record */
      v_success := create_trip_qc(p_trip_qc_rec);
    
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                              ' -  writing trip qc sample record.');
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* loop through projects, see if project is on the trip, and add if it's not */
    BEGIN
      FOR v_project IN c_get_projects(p_activity_seq) LOOP
        add_project_to_trip(v_activity.fa_tsrtrip_is_number, v_project.sapj_tsmproj_is_number, p_org_id);
      
        /* loop through personnel, assign to role and project*/
        BEGIN
          FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP
          
            v_success := add_person_to_project(v_personnel.sap_tsmpersn_is_number, rpad(v_personnel.sap_tsmpersn_org_id, 8), v_project.sapj_tsmproj_is_number, rpad(p_org_id, 8));
          
          END LOOP;
        
        EXCEPTION
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                 SQLERRM ||
                                                 ' - adding person to project.');
             
                
            v_success := 0;
            IF t_success > v_success THEN
              t_success := v_success;
            END IF;
        END;
      
      END LOOP;
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - adding project to trip. Trip_is_nbr: ' ||
                                             v_activity.fa_tsrtrip_is_number);
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* loop through personnel, add to the sample */
    BEGIN
      FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP
      
        v_success := add_person_to_trip_qc(v_personnel.sap_tsmpersn_is_number, rpad(v_personnel.sap_tsmpersn_org_id, 8), v_trip_qc_is_nbr, rpad(p_org_id, 8));
      
      END LOOP;
    
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' - adding person to trip qc sample.');
         
            
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
    END;
  
    /* IF QC RESULTS, THERE WON'T BE AN ACTIVITY IS NUBR ... */
    BEGIN
      export_results(p_activity_seq, p_trip_qc_rec.type_name, NULL, NULL, v_trip_qc_is_nbr, p_org_id, v_activity.fa_tsrlab_is_number, v_activity.fa_tsrlab_org_id, p_org_id, v_export_log_seq, p_result_cnt, v_time_zone);
    
    END;
  
    v_trip_rec := v_trip_copy_rec;
  
  END LOOP;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in export trip qc.');
     
        
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
  
END export_trip_qc;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE delete_exported_rec(p_fa_seq IN NUMBER) IS

BEGIN
  NULL;
END delete_exported_rec;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE sim_export
(
  p_org_id     IN VARCHAR2,
  p_import_seq IN NUMBER,
  p_success    IN OUT NUMBER
) IS

  v_activity_cnt NUMBER(10) := 0;
  v_result_cnt   NUMBER(10) := 0; --counter for each activity
  v_file_res_cnt NUMBER(10) := 0; --counter for total results in file
  v_line_cnt     NUMBER(10) := 0;

  v_stvst_rec         tsrstvst%ROWTYPE;
  v_stvst_copy_rec    tsrstvst%ROWTYPE;
  v_activity_rec      tsrfdact%ROWTYPE;
  v_activity_copy_rec tsrfdact%ROWTYPE;
  v_trip_qc_rec       tsrfqs%ROWTYPE;

  v_activity_is_nbr NUMBER(8);
  v_activity_org_id VARCHAR2(8);
  v_trip_qc_is_nbr  NUMBER(8);
  v_trip_qc_org_id  VARCHAR(8);
  v_visit_is_number NUMBER(8);
  v_visit_org_id    VARCHAR2(8);

  v_act_type VARCHAR2(13);

  v_success     NUMBER(1);
  v_export_test NUMBER(10);
  v_import_type VARCHAR2(4);
  v_filename    VARCHAR2(265);
  v_sid_seq     NUMBER(10);  
  v_rows_ready  NUMBER(10);
  v_sevl_seq    NUMBER(10);
  v_err_cnt     NUMBER(10);

  ex_missing_info EXCEPTION;
  e_no_stvst EXCEPTION;

  CURSOR c_activity(p_org_id IN VARCHAR2, p_export_test IN NUMBER) IS
    SELECT fa_seq,
           fa_type_name
      FROM sim_field_activities
     WHERE rpad(fa_tsmorgan_org_id, 8) = rpad(p_org_id, 8)
       AND fa_export_seq IS NULL
       AND fa_status = 'A'
       AND fa_import_seq = p_import_seq
     ORDER BY fa_seq; -- jah 2-13-04 To ensure we export in the same order we imported! Needed for Parent/Child.

  CURSOR c_import_type IS
    SELECT sil_import_type
      FROM sim_import_log
     WHERE sil_seq = p_import_seq;

BEGIN
  sim_event_log_pkg.write_event(sim_lib_pkg.c_selt_migrate,p_import_seq,p_org_id,v_sid_seq,NULL);
  /* kms - initialize activity and result counters */
  v_success := 1;
  v_activity_cnt := 0;
  v_result_cnt := 0;
  v_line_cnt := 0;
  v_file_res_cnt := 0;

  OPEN c_import_type;
  FETCH c_import_type
    INTO v_import_type;
  CLOSE c_import_type;

  BEGIN
    /* set up export log */
    v_export_log_seq := p_import_seq;
    SELECT substr(sil_import_file,instr(sil_import_file, '\', -1) +1)
      INTO v_filename
      FROM sim_import_log
      WHERE sil_seq = v_export_log_seq;
    
    INSERT INTO sim_export_log
      (sel_seq,
       sel_exported_by,
       sel_exported_on,
       sel_filename,
       sel_org_id,
       sel_export_type)
    VALUES
      (v_export_log_seq,
       USER,
       SYSDATE,
       v_filename,
       p_org_id,
       v_import_type);
  
    COMMIT;
  
  EXCEPTION
    WHEN OTHERS THEN
      v_success := 0;
      IF t_success > v_success THEN
        t_success := v_success;
      END IF;
      p_success := t_success;
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           SQLERRM ||
                                           ' - setting up export log record.'); 
  END;
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = (SELECT sil_sicc_seq 
                       FROM sim_import_log 
                      WHERE sil_seq =  p_import_seq);
  
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Start: ' ||
                                          to_char(SYSDATE, 'HH24:MI:SS'));
  /* loop through the activities */
  FOR v_activity IN c_activity(p_org_id, v_export_test) LOOP
    BEGIN
      /* get the activity type */
      v_act_type := v_activity.fa_type_name;
    
    EXCEPTION
      WHEN OTHERS THEN
        v_success := 0;
        IF t_success > v_success THEN
          t_success := v_success;
        END IF;
        p_success := t_success;  
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM ||
                                             ' -  getting allowable values.');          
    END;

    /* see if this is a quality control sample or an activity */
    IF v_act_type != 'Trip QC' THEN
      /* set trip qc key to null */
      v_trip_qc_is_nbr := NULL;
      v_trip_qc_org_id := NULL;
    
      v_activity_rec.type_name := v_act_type;
    
      /* run the procedure that writes activity to TSRFDACT */
      export_activity(v_activity.fa_seq, v_activity_rec, v_stvst_rec, p_import_seq, p_org_id, v_result_cnt);
    
      v_activity_is_nbr := v_activity_rec.tsrfdact_is_number;
      v_activity_org_id := v_activity_rec.tsrfdact_org_id;
      v_visit_is_number := v_stvst_rec.tsrstvst_is_number;
      v_visit_org_id := v_stvst_rec.tsrstvst_org_id;
    
    ELSE
    
      /* set field activity keys to null */
      v_activity_is_nbr := NULL;
      v_activity_org_id := NULL;
      v_visit_is_number := NULL;
      v_visit_org_id := NULL;
    
      v_trip_qc_rec.type_name := v_act_type;
    
      /* run the procedure that writes activity to TSRFQS */
      export_trip_qc(v_activity.fa_seq, v_trip_qc_rec, p_import_seq, p_org_id, v_result_cnt);
    
      v_trip_qc_is_nbr := v_trip_qc_rec.tsrfqs_is_number;
      v_trip_qc_org_id := v_trip_qc_rec.tsrfqs_org_id;
    
    END IF;
  
    /* update counters */
    IF v_result_cnt > 0 THEN
      --if no results in the activity, v_result_cnt = 0
      v_line_cnt := v_line_cnt + v_result_cnt;
    END IF;
    v_activity_cnt := v_activity_cnt + 1;
    v_file_res_cnt := v_file_res_cnt + v_result_cnt;
  
    /* update the activity as being exported */
    UPDATE sim_field_activities
       SET fa_export_seq = v_export_log_seq,
           fa_tsrfdact_is_number = v_activity_is_nbr,
           fa_tsrfdact_org_id = rpad(p_org_id, 8),
           fa_tsrstvst_is_number = v_visit_is_number,
           fa_tsrstvst_org_id = rpad(p_org_id, 8),
           fa_tsrfqs_is_number = v_trip_qc_is_nbr,
           fa_tsrfqs_org_id = rpad(p_org_id, 8),
           fa_status = 'E'
     WHERE fa_seq = v_activity.fa_seq;
  
    COMMIT;
  
    v_stvst_rec := v_stvst_copy_rec;
    v_activity_rec := v_activity_copy_rec;
  
  END LOOP;

  /* update export log with count */
  UPDATE sim_export_log
     SET sel_activity_cnt = v_activity_cnt
   WHERE sel_seq = v_export_log_seq;

  /* update the import log with export info
  SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE sim_import_log
     SET sil_exported_on = SYSDATE,
         sil_exported_by = USER,
         sil_rows_ready = sil_rows_ready - v_line_cnt,
         sil_activities_imported = sil_activities_imported - v_activity_cnt
   WHERE sil_seq = p_import_seq;
  
  COMMIT;
  p_success := t_success;
  
  v_sevl_seq := sim_event_log_pkg.get_sevl_seq;
  
  IF p_success = 1 THEN
    SELECT COUNT(*)
      INTO v_err_cnt
      FROM sim_event_log_details d
      WHERE d.seld_sevl_seq = v_sevl_seq
        AND d.seld_seldt_seq = sim_lib_pkg.c_seldt_error;
    IF v_err_cnt > 0 THEN 
      p_success := 0;
    END IF;
  END IF;
  sim_import_pkg.delete_import(p_import_seq, c_migrate_true, c_new_event_no);
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Migrated ' || v_activity_cnt ||
                                         ' activities, ' || v_file_res_cnt ||
                                         ' results.'); 
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Finish: ' ||
                                       to_char(SYSDATE, 'HH24:MI:SS')); 
  sim_event_log_pkg.update_stop_time;
    
EXCEPTION
  WHEN OTHERS THEN
    v_success := 0;
    IF t_success > v_success THEN
      t_success := v_success;
    END IF;
    p_success := t_success;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' -  Error in sim_export_pkg.EXPORT_SIM');
     
        
END sim_export;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE export_results
(
  p_fa_seq          IN NUMBER,
  p_fat_desc        IN VARCHAR2,
  p_activity_is_nbr IN NUMBER,
  p_activity_org_id IN VARCHAR2,
  p_tripqc_is_nbr   IN NUMBER,
  p_tripqc_org_id   IN VARCHAR2,
  p_lab_is_number   IN NUMBER,
  p_lab_org_id      IN VARCHAR2,
  p_org_id          IN VARCHAR2,
  p_export_log_seq  IN NUMBER,
  p_act_result_cnt  OUT NUMBER,
  p_time_zone       IN VARCHAR2
) IS

  v_success         NUMBER(1);
  v_activity_id     VARCHAR2(12);
  v_act_cat         VARCHAR2(30);
  v_data_column_cnt NUMBER(10);
  v_line_cnt        NUMBER(10);
  v_result_rec      tsrrsult%ROWTYPE;
  v_result_copy_rec tsrrsult%ROWTYPE;
  v_cell_rec        tsrcldes%ROWTYPE;
  v_remark_rec      tsrrlra%ROWTYPE;
  v_remark_success  NUMBER(1);

  v_result_is_number   NUMBER(8);
  v_tsrdlin_is_number  NUMBER(8);
  v_lab_is_nbr         NUMBER(8);
  v_lab_org_id         VARCHAR2(8);
  v_tsrdlin_is_number2 NUMBER(12);
  v_tsrdlin_org_id2    VARCHAR2(8);

  v_result_value    NUMBER(13, 5);
  v_result_text     VARCHAR2(15);
  v_det_limit       VARCHAR2(12);
  v_min_detect      VARCHAR2(12);
  v_max_detect      VARCHAR2(12);
  v_detect_cond_cd  VARCHAR2(50);
  v_result_cmt      VARCHAR2(1999);
  v_tsmgntxt_is_nbr NUMBER(8);
  v_comp_ind_cd     VARCHAR2(1);
  v_table           VARCHAR2(8);
  v_desc            VARCHAR2(8);
  v_blob_success    NUMBER(1);

  -- jah 2-19-04
  v_prev_data_line_number NUMBER(8); -- tracks when rs_data_line_number changes
  v_prev_brgi_seq         NUMBER(10); -- tracks when rs_brgi_seq changes
  v_assnd_seq_num         NUMBER(8);

  e_def_not_complete EXCEPTION;
  e_no_chgrp EXCEPTION;
  e_no_char EXCEPTION;
  e_no_rowid EXCEPTION;

  /* jah 2-19-04
  Get the results for a given activity.
  Order by BRGI_SEQ for biological activities.
  Order by RS_DATA_LINE_NUMBER for Portable Data Logger activities. */
  CURSOR c_result(p_activity_seq IN NUMBER) IS
    SELECT *
      FROM sim_results
     WHERE rs_fa_seq = p_activity_seq
     ORDER BY rs_data_line_number,
              rs_brgi_seq,
              rs_tsrchar_is_number;

  CURSOR c_get_activity_id(p_activity_seq IN NUMBER) IS
    SELECT fa_activity_id
      FROM sim_field_activities
     WHERE fa_seq = p_activity_seq;

  CURSOR c_lab_remark_codes(p_seq NUMBER) IS
    SELECT srlr_seq,
           srlr_tsrlbrmk_is_number,
           srlr_tsrlbrmk_org_id
      FROM sim_result_lab_remarks
     WHERE srlr_rs_seq = p_seq;

  CURSOR c_act_cat(p_seq NUMBER) IS
    SELECT fa_category
      FROM sim_field_activities
     WHERE fa_seq = p_seq;

  CURSOR c_data_column_cnt(p_seq NUMBER) IS
    SELECT COUNT(DISTINCT(rs_tsrchar_is_number))
      FROM sim_results
     WHERE rs_fa_seq = p_seq;

  --This cursor NEEDS to be ordered by rs_data_line_number                
  CURSOR c_data_lines(p_seq NUMBER) IS
    SELECT DISTINCT (rs_data_line_number) rs_data_line_number
      FROM sim_results
     WHERE rs_fa_seq = p_seq
     ORDER BY rs_data_line_number;

  CURSOR c_line_nbr_cnt(p_seq NUMBER, p_line_number NUMBER) IS
    SELECT COUNT(rs_tsrchar_is_number)
      FROM sim_results
     WHERE rs_fa_seq = p_seq
       AND rs_data_line_number = p_line_number;

  CURSOR c_missing_is_number(p_seq NUMBER, p_line_number NUMBER) IS
    SELECT DISTINCT rs_tsrchar_is_number,
                    rs_tsrchar_org_id
      FROM sim_results
     WHERE rs_fa_seq = p_seq
       AND rs_tsrchar_is_number NOT IN
           (SELECT rs_tsrchar_is_number
              FROM sim_results
             WHERE rs_fa_seq = p_seq
               AND rs_data_line_number = p_line_number);

  CURSOR c_tsrdlin(p_line_nbr NUMBER, p_is_nbr NUMBER, p_org_id VARCHAR2) IS
    SELECT tsrdlin_is_number,
           tsrdlin_org_id
      FROM tsrdlin
     WHERE line_number = p_line_nbr
       AND tsrfdact_is_number = p_is_nbr
       AND tsrfdact_org_id = p_org_id;

BEGIN

  p_act_result_cnt := 0;

  -- jah 2-19-04  init display order sequence tracking variables
  v_assnd_seq_num := 0;
  v_prev_data_line_number := 0;
  v_prev_brgi_seq := 0;

  FOR v_result IN c_result(p_fa_seq) LOOP
    BEGIN
      /* initialize variables */
      v_result_value := NULL;
      v_det_limit := NULL;
      v_min_detect := NULL;
      v_max_detect := NULL;
      v_detect_cond_cd := NULL;
      v_comp_ind_cd := NULL;
    
      -- get the next IS_NUMBER for TSRRSULT
      v_result_is_number := get_next_result_is_nbr(p_org_id);
    
      v_lab_is_nbr := v_result.rs_tsrlab_is_number; --p_lab_is_number;
      v_lab_org_id := v_result.rs_tsrlab_org_id; --RPAD(p_lab_org_id,8);
    
      /* convert result text to number */
      v_result_text := v_result.rs_result;
    
      IF v_result.rs_tsrhcsc_is_number IS NOT NULL THEN
        --If it's habitat Assessment mes is always 0
        v_result_value := 0;
      ELSE
        v_result_value := get_value_msr(v_result_text);
      END IF;
    
      IF v_result.rs_value_status IS NULL THEN
        v_result.rs_value_status := 'F';
      END IF;
    
      /* comp_ind_cd should always be 'Y' if the activity has been allowed to be exported */
      v_comp_ind_cd := 'Y';
      /* Issue other STORET NULL Value defaults. */
      --For TSRRSULT
      IF v_result.rs_analysis_date IS NULL THEN
        v_result.rs_analysis_date := to_date('01-JAN-0001', 'DD-MON-YYYY');
      END IF;
      IF v_result.rs_analysis_time IS NULL THEN
        v_result.rs_analysis_time := to_date('01-JAN-0001', 'DD-MON-YYYY');
      END IF;
      IF v_result.rs_lab_cert IS NULL THEN
        v_result.rs_lab_cert := 'N';
      END IF;
    
      /* kms - set detection condition to correct value in order for the correct radio
      button to be selected in the STORET result screen. */
      IF v_result.rs_tsrhcsc_is_number IS NOT NULL THEN
        --If it's habitat Assessment no detect_cond
        v_detect_cond_cd := NULL;
      ELSE
        IF v_result_text = '*Non-detect' THEN
          v_detect_cond_cd := 'Not Detected';
        ELSIF v_result_text = '*Present <QL' THEN
          v_detect_cond_cd := 'Present below Quantification';
        ELSIF v_result_text = '*Present >QL' THEN
          v_detect_cond_cd := 'Present above Quantification';
        ELSIF v_result_text = '*Present' THEN
          v_detect_cond_cd := 'Detected not Quantified';
        ELSIF v_result_text = '*Not Reported' THEN
          v_detect_cond_cd := 'Not Reported';
        ELSE
          v_detect_cond_cd := 'Detected and Quantified';
        END IF;
      END IF;
    
      -- jah 2-19-04
      -- Does this result have a data line number?
      IF (v_result.rs_data_line_number IS NOT NULL) THEN
        -- if yes
      
        -- has the data line number changed?
        IF (v_prev_data_line_number <> v_result.rs_data_line_number) THEN
          -- if yes
        
          -- reset the display order sequence number
          v_assnd_seq_num := 0;
        
          -- set the previous line number to the current line number
          v_prev_data_line_number := v_result.rs_data_line_number;
        
          -- Get the IS_NUMBER for the next line number record in TSRDLIN
          v_tsrdlin_is_number := sim_export_pkg.get_eisn_number('TSRDLIN', rpad(p_org_id, 8));
        
          -- insert the next current line number into TSRDLIN
          INSERT INTO tsrdlin
            (tsrdlin_is_number,
             tsrdlin_org_id,
             line_number,
             tsrfdact_is_number,
             tsrfdact_org_id,
             d_userid_code,
             d_last_update_ts,
             line_name)
          VALUES
            (v_tsrdlin_is_number,
             rpad(p_org_id, 8),
             v_result.rs_data_line_number,
             p_activity_is_nbr,
             p_activity_org_id,
             sim_export_pkg.v_pkg_ownerid,
             SYSDATE,
             v_result.rs_data_line_name);
        
        END IF;
      
        -- Does this result have a Biological Result Group Individual sequence
      ELSIF (v_result.rs_brgi_seq IS NOT NULL) THEN
        -- if yes
      
        -- has the Biological Result Group Individual sequence changed?
        IF (v_prev_brgi_seq <> v_result.rs_brgi_seq) THEN
          -- if yes
        
          -- reset the display order sequence number
          v_assnd_seq_num := 0;
        
          -- reset the previous BRGI sequence number
          v_prev_brgi_seq := v_result.rs_brgi_seq;
        
        END IF;
      
      END IF;
    
      /* write values to be exported to result record */
      BEGIN
        v_result_rec.tsrrsult_is_number := v_result_is_number;
        v_result_rec.tsrrsult_org_id := rpad(p_org_id, 8);
      
        -- jah 2-19-04
        -- increment the display order sequence number
        v_assnd_seq_num := v_assnd_seq_num + 1;
        v_result_rec.d_assnd_seq_num := v_assnd_seq_num;
      
        v_result_rec.comp_ind_cd := v_comp_ind_cd;
        v_result_rec.lab_cert_ind_code := v_result.rs_lab_cert;
        v_result_rec.value_text := rpad(v_result_text, 15);
        v_result_rec.value_measure := v_result_value;
        v_result_rec.value_status := v_result.rs_value_status;
        v_result_rec.precision_amt_text := v_result.rs_precision;
        v_result_rec.lab_batch_id_code := v_result.rs_lab_batch_id;
        IF v_result_text = '*Text' THEN
          v_result_rec.rslt_txt_ind_cd := 'Y';
        ELSE
          v_result_rec.rslt_txt_ind_cd := 'N';
        END IF;
        v_result_rec.cell_desc_ind_cd := 'N';
        v_result_rec.conf_lvl_pct_msr := v_result.rs_conf_level;
        v_result_rec.proc_excptn_ind_cd := 'N';
        v_result_rec.dur_basis_type_nm := v_result.rs_duration_basis;
        v_result_rec.detect_cond_cd := rpad(v_detect_cond_cd, 30);
        v_result_rec.smpl_frac_type_nm := v_result.rs_fraction;
        v_result_rec.statistic_type_nm := v_result.rs_stat_type;
        v_result_rec.value_type_name := v_result.rs_value_type;
        v_result_rec.wt_basis_type_nm := v_result.rs_weight_basis;
        v_result_rec.temp_basis_lvl_nm := v_result.rs_temp_basis;
        v_result_rec.taxon_pollution := v_result.rs_taxon_pollution;
        v_result_rec.fnctional_feed_grp := v_result.rs_fnctional_feed_grp;
        v_result_rec.trophic_level := v_result.rs_trophic_level;
        v_result_rec.analysis_date := v_result.rs_analysis_date;
        v_result_rec.analysis_time := v_result.rs_analysis_time;
        v_result_rec.analysis_time_zone := v_result.rs_analysis_time_zone;
        v_result_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
        v_result_rec.d_last_update_ts := SYSDATE;
      
        --02/22/02 SF - Either a TSRFDACT_IS_NUM or a TSRBRG_IS_NUM goes in, never both
        IF v_result.rs_tsrbrg_is_number IS NULL THEN
          v_result_rec.tsrfdact_is_number := p_activity_is_nbr;
          v_result_rec.tsrfdact_org_id := rpad(p_activity_org_id, 8);
        ELSE
          v_result_rec.tsrfdact_is_number := NULL;
          v_result_rec.tsrfdact_org_id := NULL;
        END IF;
      
        v_result_rec.tsrfqs_is_number := p_tripqc_is_nbr;
        v_result_rec.tsrfqs_org_id := rpad(p_tripqc_org_id, 8);
        v_result_rec.tsranlpr_is_number := v_result.rs_tsranlpr_is_number;
        v_result_rec.tsranlpr_org_id := rpad(v_result.rs_tsranlpr_org_id, 8);
        v_result_rec.tsrbrgi_is_number := v_result.rs_tsrbrgi_is_number;
        v_result_rec.tsrbrgi_org_id := rpad(v_result.rs_tsrbrgi_org_id, 8);
        v_result_rec.tsrbrg_is_number := v_result.rs_tsrbrg_is_number;
        v_result_rec.tsrbrg_org_id := rpad(v_result.rs_tsrbrg_org_id, 8);
        v_result_rec.tsrchar_is_number := v_result.rs_tsrchar_is_number;
        v_result_rec.tsrchar_org_id := v_result.rs_tsrchar_org_id;
        v_result_rec.tsrchar0is_number := v_result.rs_tsrchar0is_number;
        v_result_rec.tsrchar0org_id := v_result.rs_tsrchar0org_id;
        v_result_rec.tsrlab_is_number := v_lab_is_nbr;
        v_result_rec.tsrlab_org_id := v_lab_org_id;
        v_result_rec.tsruom_is_number := v_result.rs_tsruom_is_nbr;
        v_result_rec.tsruom_org_id := rpad(v_result.rs_tsruom_org_id, 8);
        v_result_rec.tsrlspp_is_number := v_result.rs_tsrlspp_is_number;
        v_result_rec.tsrlspp_org_id := v_result.rs_tsrlspp_org_id;
        v_result_rec.tsrhcsc_is_number := v_result.rs_tsrhcsc_is_number;
        v_result_rec.tsrhcsc_org_id := v_result.rs_tsrhcsc_org_id;
      
        /*New stuff for V2 */
      
        v_result_rec.bias := v_result.rs_bias;
        v_result_rec.conf_lvl_corr_bias := v_result.rs_conf_lvl_corr_bias;
        v_result_rec.correction_ind_cd := v_result.rs_correction_ind_cd;
        v_result_rec.dilution_ind_code := v_result.rs_dilution_ind_code;
        v_result_rec.tsmprmvl_is_number := v_result.rs_fraction_tsmprmvl_is_number;
        v_result_rec.tsmprmvl_org_id := v_result.rs_fraction_tsmprmvl_org_id;
        v_result_rec.tsmprmvl0is_number := v_result.rs_habit_tsmprmvl0is_number;
        v_result_rec.tsmprmvl0org_id := v_result.rs_habit_tsmprmvl0org_id;
        v_result_rec.tsmprmvl1is_number := v_result.rs_volt_tsmprmvl1is_number;
        v_result_rec.tsmprmvl1org_id := v_result.rs_volt_tsmprmvl1org_id;
        v_result_rec.lower_range_value := v_result.rs_lower_range_value;
        v_result_rec.upper_range_value := v_result.rs_upper_range_value;
        v_result_rec.recovery_ind_code := v_result.rs_recovery_ind_code;
        v_result_rec.ref_pt_from_name := v_result.rs_ref_pt_from_name;
        v_result_rec.ref_pt_to_name := v_result.rs_ref_pt_to_name;
        v_result_rec.repl_analysis_num := v_result.rs_repl_analysis_num;
        v_result_rec.species_number := v_result.rs_species_number;
        v_result_rec.tsrdlin_is_number := v_tsrdlin_is_number;
        v_result_rec.tsrdlin_org_id := rpad(p_org_id, 8);
        v_result_rec.blob_title := sim_export_pkg.get_file_name(v_result.rs_document);
        v_result_rec.blob_type := sim_export_pkg.get_file_type(v_result.rs_document);
      
        IF v_result.rs_cell_shape_type_nm IS NOT NULL
           OR v_result.rs_cell_type_nm IS NOT NULL THEN
          v_result_rec.cell_desc_ind_cd := 'Y';
          v_cell_rec.tsrrsult_is_number := v_result_is_number;
          v_cell_rec.tsrrsult_org_id := rpad(p_org_id, 8);
          v_cell_rec.cell_shape_type_nm := v_result.rs_cell_shape_type_nm;
          v_cell_rec.cell_type_nm := v_result.rs_cell_type_nm;
          v_cell_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
          v_cell_rec.d_last_update_ts := SYSDATE;
        ELSE
          v_result_rec.cell_desc_ind_cd := 'N';
          v_cell_rec.tsrrsult_is_number := NULL;
        END IF;
      EXCEPTION
        WHEN OTHERS THEN
          v_success := 0;
          IF t_success > v_success THEN
            t_success := v_success;
          END IF;
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM ||
                                              ' - assigning values to result record.');
           
              
      END;
    
      /* create STORET result record */
      v_success := create_result(v_result_rec, v_result.rs_seq);
    
      -- Create the Result Lab Remarks record 
      IF v_success = 1 THEN
      
        /* export binary document for result */
        IF v_result.rs_document IS NOT NULL THEN
          v_blob_success := sim_export_pkg.insert_binary_document(
              p_export_log_seq => v_export_log_seq,
              p_table_name => 'TSRRSULT',
              p_userid_code => v_result_rec.d_userid_code,
              p_file_name => v_result.rs_document, 
              p_tsrrsult_is_number => v_result_is_number, 
              p_tsrrsult_org_id => p_org_id);
          IF v_blob_success = 0 THEN
            v_success := 0;
            IF t_success > v_success THEN
              t_success := v_success;
            END IF;
          END IF;
        END IF;
      
        FOR c_lab_remark_codes_rec IN c_lab_remark_codes(v_result.rs_seq) LOOP
          v_remark_rec.tsrrsult_is_number := v_result_is_number;
          v_remark_rec.tsrrsult_org_id := rpad(p_org_id, 8);
          v_remark_rec.tsrlbrmk_is_number := c_lab_remark_codes_rec.srlr_tsrlbrmk_is_number;
          v_remark_rec.tsrlbrmk_org_id := c_lab_remark_codes_rec.srlr_tsrlbrmk_org_id;
          v_remark_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
          v_remark_rec.d_last_update_ts := SYSDATE;
          v_remark_success := add_lab_remarks_to_result(v_remark_rec);
        END LOOP;
      END IF;
    
      IF v_cell_rec.tsrrsult_is_number IS NOT NULL THEN
        v_success := create_tsrcldes(v_cell_rec);
      END IF;
    
      -- create tsrrci record for this result
      v_success := create_tsrrci_record(p_org_id, v_result_is_number);
    
      -- insert result comment    
      v_result_cmt := v_result.rs_comment;
      IF v_result.rs_comment = '*Text' THEN
        -- If the comment indicates it is a text
        v_table := 'TSRRSULT'; -- characteristic, insert data using the
        v_desc := 'DESCRIPT'; -- correct table and value for description
      ELSE
        -- name
        v_table := 'TSRRSLT';
        v_desc := 'RSLTCMNT';
      END IF;
    
      IF v_result_cmt IS NOT NULL THEN
        v_tsmgntxt_is_nbr := get_next_tsmgntxt_nbr(p_org_id);
        BEGIN
          INSERT INTO tsmgntxt
            (tsmgntxt_is_number,
             tsmgntxt_org_id,
             table_name,
             description_name,
             description_text,
             d_userid_code,
             d_last_update_ts,
             tsrrsult_is_number,
             tsrrsult_org_id,
             tsmproj_is_number,
             tsmproj_org_id,
             tsmprogm_is_number,
             tsmprogm_org_id,
             tsrgrcfg_is_number,
             tsrgrcfg_org_id)
          VALUES
            (v_tsmgntxt_is_nbr, /* TSMGNTXT_IS_NUMBER */
             rpad(p_org_id, 8), /* TSMGNTXT_ORG_ID */
             v_table, /* TABLE_NAME */
             v_desc, /* DESCRIPTION_NAME */
             v_result_cmt, /* DESCRIPTION_TEXT */
             sim_export_pkg.v_pkg_ownerid,
             SYSDATE, /* D_LAST_UPDATE_TS */
             v_result_rec.tsrrsult_is_number, /* TSRRSULT_IS_NUMBER */
             v_result_rec.tsrrsult_org_id, /* TSRRSULT_ORG_ID */
             NULL, /* TSMPROJ_IS_NUMBER */
             NULL, /* TSMPROJ_ORG_ID */
             NULL, /* TSMPROGM_IS_NUMBER */
             NULL, /* TSMPROGM_ORG_ID */
             NULL, /* TSRGRCFG_IS_NUMBER */
             NULL /* TSRGRCFG_ORG_ID */);
        
        EXCEPTION
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                 SQLERRM ||
                                                 ' - writing result comment.');
             
                
        END;
      END IF;
    
      /* figure out detect limit info */
      BEGIN
        v_result_text := v_result.rs_result;
        v_det_limit := v_result.rs_detect_limit;
        v_result_cmt := v_result.rs_comment;
        v_min_detect := v_result.rs_lower_quant_limit;
        v_max_detect := v_result.rs_upper_quant_limit;
      
        IF v_det_limit IS NOT NULL
           OR v_min_detect IS NOT NULL
           OR v_max_detect IS NOT NULL THEN
          BEGIN
            INSERT INTO tsrdql
              (tsrrsult_is_number,
               tsrrsult_org_id,
               min_detect_limit,
               min_quant_limit,
               max_quant_limit,
               description_text,
               d_userid_code,
               d_last_update_ts,
               tsruom_is_number,
               tsruom_org_id)
            VALUES
              (v_result_is_number,
               rpad(p_org_id, 8),
               v_det_limit,
               v_min_detect, /* MIN QUANT LIMIT */
               v_max_detect, /* MAX QUANT LIMIT */
               v_result.rs_detect_comment /* DESC TEXT */,
               sim_export_pkg.v_pkg_ownerid,
               SYSDATE /* D_LAST_UPDATE_TS */,
               v_result.rs_det_limit_uom_nbr /* TSRUOM_IS_NUMBER */,
               v_result.rs_det_limit_uom_org /* TSRUOM_ORG_ID */);
          
            COMMIT;
          EXCEPTION
            WHEN OTHERS THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                   SQLERRM ||
                                                   ' - writing detect and quantification limit record.');
               
                  
          END;
        
        END IF;
      
      EXCEPTION
        WHEN OTHERS THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM ||
                                              ' - getting detect and quantification limit information.');
           
              
      END;
    
      /* keep track of the number of results exported for an activity (incremented only
      when all inserts have been performed so that the value accurately reflects what has
      been loaded into STORET */
      p_act_result_cnt := p_act_result_cnt + 1;
    
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                             SQLERRM || ' -  writing result ');
         
        
    END;
  
    v_result_rec := v_result_copy_rec;
  
  END LOOP;

  OPEN c_act_cat(p_fa_seq);
  FETCH c_act_cat
    INTO v_act_cat;
  CLOSE c_act_cat;

  -- do we need to worry about missing characteric results for this activity?
  IF upper(v_act_cat) = 'PORTABLE DATA LOGGER' THEN
    -- if yes
  
    -- count the number of characteristics
    OPEN c_data_column_cnt(p_fa_seq);
    FETCH c_data_column_cnt
      INTO v_data_column_cnt;
    CLOSE c_data_column_cnt;
  
    -- loop through all the dataline for this activity
    FOR c_data_lines_rec IN c_data_lines(p_fa_seq) LOOP
      --c_data_lines_rec
    
      -- count the number of rows
      OPEN c_line_nbr_cnt(p_fa_seq, c_data_lines_rec.rs_data_line_number);
      FETCH c_line_nbr_cnt
        INTO v_line_cnt;
      CLOSE c_line_nbr_cnt;
    
      -- are any results missing?
      IF nvl(v_line_cnt, 0) < v_data_column_cnt THEN
        -- if yes
      
        /*
          jah 2-19-04
          The following loop finds where the activity has missing results
          and inserts rows for the missing results into TSRRSULT. STORET
          requires a row for every characteristic whether the characteristic
          has data or not!
        */
      
        -- Loop through the missing characteristics
        FOR c_missing_is_number_rec IN c_missing_is_number(p_fa_seq, c_data_lines_rec.rs_data_line_number) LOOP
          --c_missing_is_number_rec
        
          -- Get the key of the TSRDLIN record for this activity
          OPEN c_tsrdlin(c_data_lines_rec.rs_data_line_number, p_activity_is_nbr, p_activity_org_id);
          FETCH c_tsrdlin
            INTO v_tsrdlin_is_number2, v_tsrdlin_org_id2;
          CLOSE c_tsrdlin;
        
          -- get the next IS_NUMBER for TSRRSULT
          v_result_is_number := get_next_result_is_nbr(p_org_id);
        
          BEGIN
            v_result_rec.tsrrsult_is_number := v_result_is_number;
            v_result_rec.tsrrsult_org_id := rpad(p_org_id, 8);
          
            /*
              jah 2-19-04
              TSRRSULT.D_ASSND_SEQ_NUM controls the display order of results.
              The following statement causes missing characteristics, which 
              this loop handles, to display first.
            */
            v_result_rec.d_assnd_seq_num := 0;
          
            v_result_rec.comp_ind_cd := v_comp_ind_cd;
            v_result_rec.value_status := 'F';
            v_result_rec.tsrchar_is_number := c_missing_is_number_rec.rs_tsrchar_is_number;
            v_result_rec.tsrchar_org_id := c_missing_is_number_rec.rs_tsrchar_org_id;
            v_result_rec.d_userid_code := sim_export_pkg.v_pkg_ownerid;
            v_result_rec.d_last_update_ts := SYSDATE;
            v_result_rec.tsrfdact_is_number := p_activity_is_nbr;
            v_result_rec.tsrfdact_org_id := rpad(p_activity_org_id, 8);
            v_result_rec.tsrdlin_is_number := v_tsrdlin_is_number2;
            v_result_rec.tsrdlin_org_id := v_tsrdlin_org_id2;
          
          EXCEPTION
            WHEN OTHERS THEN
              v_success := 0;
              IF t_success > v_success THEN
                t_success := v_success;
              END IF;
              
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                                   SQLERRM ||
                                                   ' - assigning values to result record for the portable data logger.');
               
              
          END;
        
          -- insert the row for the missing result into STORET
          v_success := create_result(v_result_rec, NULL);
        
        END LOOP; --c_missing_is_number_rec
      END IF;
    
    END LOOP; --c_data_lines_rec
  
  END IF;
  /* find out if any results were exported on this activity and write error message if none were */
  IF p_act_result_cnt = 0 THEN
    OPEN c_get_activity_id(p_fa_seq);
    FETCH c_get_activity_id
      INTO v_activity_id;
    CLOSE c_get_activity_id;
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'No results exported for activity: ' ||
                                          v_activity_id);
     
    
  END IF;

END export_results;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE runtime IS

  v_temp NUMBER(10);

  CURSOR c_temp IS
    SELECT (MAX(tsrfdact_is_number))
      FROM tsrfdact
     WHERE tsrfdact_org_id = rpad('UTAHDWQ ', 8);

BEGIN

  OPEN c_temp;
  FETCH c_temp
    INTO v_temp;
  CLOSE c_temp;

END runtime;

END sim_export_result_pkg;
/
