CREATE OR REPLACE PACKAGE sim_export_station_pkg IS

/*--------------------------------------------------------------------------------------
  Export Stations to STORET
  --------------------------------------------------------------------------------------*/
  
/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/


/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_export_log_seq NUMBER(10);
  v_success        NUMBER(1);

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ****************************************************************************************/  
 
  FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2;
  FUNCTION get_next_station_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER;
  FUNCTION get_uom_is_nbr
  (
    p_org_id          IN VARCHAR2,
    p_uom_name        IN VARCHAR2,
    p_chdef_is_number IN NUMBER
  ) RETURN NUMBER;
  FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER;
  PROCEDURE sim_export_stations
  (
    p_imp_log_seq IN NUMBER,
    p_org_id      IN VARCHAR2,
    p_success     IN OUT NUMBER
  );
  

END sim_export_station_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_export_station_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                            *
 ****************************************************************************************/
  c_type_great_lake CONSTANT tsmvstc.tsmvstc_is_number%TYPE := 3;
  c_type_ocean      CONSTANT tsmvstc.tsmvstc_is_number%TYPE := 4;
  c_type_estuary    CONSTANT tsmvstc.tsmvstc_is_number%TYPE := 5;
  
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  
  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/
 
/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS
  
  v_item_name VARCHAR2(80);
  
  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;
     
BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Could not find allowable value with sequence: ' ||
                                         p_alval_seq);
  END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - getting allowable value sequence.');
    v_success := 0;
    RETURN NULL;
END get_allowable_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_well_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS
  
  v_next_well_is_nbr NUMBER(8);
  
BEGIN
  /* get the next avail is nbr to use */
  v_next_well_is_nbr := sim_export_pkg.get_eisn_number('TSMWELL', p_org_id);
  RETURN v_next_well_is_nbr;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next well is number.');
    v_success := 0;
    RETURN NULL;
END get_next_well_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_intvl_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_intvl_is_nbr NUMBER(8);
  
BEGIN
  /* get the next avail is nbr to use */
  v_next_intvl_is_nbr := sim_export_pkg.get_eisn_number('TSMINTVL', p_org_id);
  RETURN v_next_intvl_is_nbr;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         ' - in get next intvl is number.');
    v_success := 0;
    RETURN NULL;
    
END get_next_intvl_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_igua_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_igua_is_number NUMBER(8);
  
BEGIN
  /* get the next avail is nbr to use */
  v_next_igua_is_number := sim_export_pkg.get_eisn_number('TSMIGUA', p_org_id);
  RETURN v_next_igua_is_number;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' - in get next igua is number.');
    v_success := 0;
    RETURN NULL;
END get_next_igua_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_uom_is_nbr
(
  p_org_id          IN VARCHAR2,
  p_uom_name        IN VARCHAR2,
  p_chdef_is_number IN NUMBER
) RETURN NUMBER IS

  TYPE t_uom_record IS RECORD(
    uom_is_nbr NUMBER(8),
    uom_name   VARCHAR2(10));
    
  v_uom_info t_uom_record;
  e_unk_uom EXCEPTION;
  e_uom_sp EXCEPTION;
  
  CURSOR c_get_is_nbr(p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2) IS
    SELECT tsruom_is_number,
           short_form_name
      FROM tsruom
     WHERE tsruom_org_id = rpad(p_org_id, 8)
       AND upper(short_form_name) = upper(rpad(p_uom_name, 10));
       
BEGIN
  OPEN c_get_is_nbr(p_org_id, p_uom_name);
  FETCH c_get_is_nbr
    INTO v_uom_info;
  IF c_get_is_nbr%NOTFOUND THEN
    v_uom_info.uom_is_nbr := NULL;
    RAISE e_unk_uom;
  ELSE
    IF v_uom_info.uom_name != p_uom_name THEN
      RAISE e_uom_sp;
    END IF;
  END IF;
  CLOSE c_get_is_nbr;
  RETURN v_uom_info.uom_is_nbr;
  
EXCEPTION
  WHEN e_unk_uom THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Unknown Unit of Measure: ' ||
                                          p_uom_name);
    v_success := 0;
    RETURN v_uom_info.uom_is_nbr;
    
  WHEN e_uom_sp THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Fix uom spelling in TSRCHDEF - is_number:' ||
                                         p_chdef_is_number ||
                                         ', unit of measure misspelled: ' ||
                                         p_uom_name);
    v_success := 0;
    RETURN v_uom_info.uom_is_nbr;
    
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' in get uom is nbr');
    v_success := 0;
END get_uom_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_station_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_station_is_nbr NUMBER(8);
  
BEGIN
  v_station_is_nbr := sim_export_pkg.get_eisn_number('TSMSTATN', p_org_id);
  RETURN v_station_is_nbr;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' in get uom is nbr');
    v_success := 0;
END get_next_station_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmalp_is_nbr NUMBER(8);
  
BEGIN
  v_tsmalp_is_nbr := sim_export_pkg.get_eisn_number('TSMALP', p_org_id);
  RETURN v_tsmalp_is_nbr;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                        ' in get next tsmalp is nbr');
    v_success := 0;
    
END get_next_tsmalp_is_nbr;

/****************************************************************************************
 * Delete procedure to delete from ocean, estuary, and great lakes when type changes.                                                                                      *
 ****************************************************************************************/ 
PROCEDURE check_type_delete
(
  p_tsmalp_is_number      IN tsmalp.tsmalp_is_number%TYPE,
  p_org_id                IN VARCHAR2,
  p_tsmvstc_is_number_old IN tsmvstc.tsmvstc_is_number%TYPE
) IS

BEGIN
  IF p_tsmvstc_is_number_old IN (c_type_ocean, c_type_estuary, c_type_great_lake) THEN
    IF p_tsmvstc_is_number_old = c_type_estuary THEN
      DELETE FROM tsmestlc
       WHERE tsmalp_is_number = p_tsmalp_is_number
         AND tsmalp_org_id = rpad(p_org_id, 8);
         
    ELSIF p_tsmvstc_is_number_old = c_type_great_lake THEN
      DELETE FROM tsmgll
       WHERE tsmalp_is_number = p_tsmalp_is_number
         AND tsmalp_org_id = rpad(p_org_id, 8);
         
    ELSIF p_tsmvstc_is_number_old = c_type_ocean THEN
      DELETE FROM tsmocnlc
       WHERE tsmalp_is_number = p_tsmalp_is_number
         AND tsmalp_org_id = rpad(p_org_id, 8);
    END IF;
  END IF;
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || 'in check_type_delete');
    v_success := 0;
  
END check_type_delete;

/****************************************************************************************
 * Function to update or insert into the ocean table retruns 1 if success.                                                                                     *
 ****************************************************************************************/
FUNCTION create_ocean
(
  p_location_seq          IN sim_station_locations.stl_seq%TYPE,
  p_tsmalp_is_number      IN tsmalp.tsmalp_is_number%TYPE,
  p_org_id                IN VARCHAR2,
  p_tsmvstc_is_number_new IN sim_stations.sta_tsmvstc_is_number%TYPE,
  p_tsmvstc_is_number_old IN tsmvstc.tsmvstc_is_number%TYPE
) RETURN NUMBER IS

  v_ocean_rec sim_ocean%ROWTYPE;

BEGIN
  v_success := 1;

  SELECT *
    INTO v_ocean_rec
    FROM sim_ocean
   WHERE soc_stl_seq = p_location_seq;

  IF p_tsmvstc_is_number_new = p_tsmvstc_is_number_old THEN
    UPDATE tsmocnlc
       SET NAME = v_ocean_rec.soc_name,
           shore_relation = v_ocean_rec.soc_shore_relation,
           loran_c_reading_1 = v_ocean_rec.soc_loran_c_reading_1,
           loran_c_reading_2 = v_ocean_rec.soc_loran_c_reading_2,
           bottom_topography = v_ocean_rec.soc_bottom_topography,
           addtnl_loc_name = v_ocean_rec.soc_additional_loc,
           shore_distance = v_ocean_rec.soc_shore_distance,
           shore_dist_unit_cd = v_ocean_rec.soc_shore_dist_unit,
           reference_pt = v_ocean_rec.soc_reference_pt,
           d_userid_code = sim_export_pkg.v_pkg_ownerid,
           d_last_updt_ts = SYSDATE
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_org_id, 8);
  ELSE
    IF p_tsmvstc_is_number_old IS NOT NULL THEN
      check_type_delete(p_tsmalp_is_number, p_org_id, p_tsmvstc_is_number_old);
    END IF;
    IF v_ocean_rec.soc_shore_distance IS NULL THEN
      v_ocean_rec.soc_shore_distance := 0; -- Default NULL value
    END IF;
  
    INSERT INTO tsmocnlc
      (tsmalp_is_number,
       tsmalp_org_id,
       NAME,
       shore_relation,
       loran_c_reading_1,
       loran_c_reading_2,
       bottom_topography,
       addtnl_loc_name,
       shore_distance,
       shore_dist_unit_cd,
       reference_pt,
       d_userid_code,
       d_last_updt_ts)
    VALUES
      (p_tsmalp_is_number, 
       rpad(p_org_id, 8),
       v_ocean_rec.soc_name,
       v_ocean_rec.soc_shore_relation,
       NULL, 
       NULL, 
       v_ocean_rec.soc_bottom_topography,
       v_ocean_rec.soc_additional_loc,
       v_ocean_rec.soc_shore_distance,
       v_ocean_rec.soc_shore_dist_unit,
       v_ocean_rec.soc_reference_pt,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE 
       );
  END IF;

  RETURN v_success;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || 'migrating ocean');
    v_success := 0;
  
END create_ocean;

/****************************************************************************************
 * Function to insert or update Estuary in Storet Returns 1 if success.                                                                                     *
 ****************************************************************************************/
FUNCTION create_estuary
(
  p_location_seq          IN sim_station_locations.stl_seq%TYPE,
  p_tsmalp_is_number      IN tsmalp.tsmalp_is_number%TYPE,
  p_org_id                IN VARCHAR2,
  p_tsmvstc_is_number_new IN sim_stations.sta_tsmvstc_is_number%TYPE,
  p_tsmvstc_is_number_old IN tsmvstc.tsmvstc_is_number%TYPE
) RETURN NUMBER IS

  v_estuary_rec sim_estuary_loc%ROWTYPE;
BEGIN
  v_success := 1;
  SELECT *
    INTO v_estuary_rec
    FROM sim_estuary_loc
   WHERE sel_stl_seq = p_location_seq;

  IF p_tsmvstc_is_number_new = p_tsmvstc_is_number_old THEN
    UPDATE tsmestlc
       SET other_estuary_name = v_estuary_rec.sel_other_estuary,
           addtnl_loc_name = v_estuary_rec.sel_add_loc_name,
           shore_distance = v_estuary_rec.sel_shore_dist,
           shore_dist_unit_cd = v_estuary_rec.sel_shore_dist_unit,
           reference_pt = v_estuary_rec.sel_ref_pt,
           d_userid_code = sim_export_pkg.v_pkg_ownerid,
           d_last_updt_ts = SYSDATE,
           tsmestry_org_id = v_estuary_rec.sel_tsmestry_org_id,
           tsmestry_is_number = v_estuary_rec.sel_tsmestry_is_number
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_org_id, 8);
  ELSE
    IF p_tsmvstc_is_number_old IS NOT NULL THEN
      check_type_delete(p_tsmalp_is_number, p_org_id, p_tsmvstc_is_number_old);
    END IF;
    IF v_estuary_rec.sel_shore_dist IS NULL THEN
      v_estuary_rec.sel_shore_dist := 0; 
    END IF;
  
    INSERT INTO tsmestlc
      (tsmalp_is_number,
       tsmalp_org_id,
       other_estuary_name,
       addtnl_loc_name,
       shore_distance,
       shore_dist_unit_cd,
       reference_pt,
       d_userid_code,
       d_last_updt_ts,
       tsmestry_org_id,
       tsmestry_is_number)
    VALUES
      (p_tsmalp_is_number,
       rpad(p_org_id, 8),
       v_estuary_rec.sel_other_estuary,
       v_estuary_rec.sel_add_loc_name,
       v_estuary_rec.sel_shore_dist,
       v_estuary_rec.sel_shore_dist_unit,
       v_estuary_rec.sel_ref_pt,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE,
       'ZZZZZZZZ',
       v_estuary_rec.sel_tsmestry_is_number);
  END IF;
  RETURN v_success;
EXCEPTION
  WHEN OTHERS THEN
  
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || 'migrating estuary');
    v_success := 0;
END create_estuary;

/****************************************************************************************
 * Function to insert or update Great Lake table in storet returns 1 if success.                                                                                    *
 ****************************************************************************************/
FUNCTION create_great_lake
(
  p_location_seq          IN sim_station_locations.stl_seq%TYPE,
  p_tsmalp_is_number      IN tsmalp.tsmalp_is_number%TYPE,
  p_org_id                IN VARCHAR2,
  p_tsmvstc_is_number_new IN sim_stations.sta_tsmvstc_is_number%TYPE,
  p_tsmvstc_is_number_old IN tsmvstc.tsmvstc_is_number%TYPE
) RETURN NUMBER IS

  v_great_lake_rec sim_great_lake%ROWTYPE;
BEGIN
  v_success := 1;
  SELECT *
    INTO v_great_lake_rec
    FROM sim_great_lake
   WHERE sgl_stl_seq = p_location_seq;

  IF p_tsmvstc_is_number_new = p_tsmvstc_is_number_old THEN
    UPDATE tsmgll
       SET NAME = v_great_lake_rec.sgl_lake_name,
           addtnl_loc_name = v_great_lake_rec.sgl_add_lake_name,
           shore_distance = v_great_lake_rec.sgl_shore_dist,
           shore_dist_unit_cd = v_great_lake_rec.sgl_shore_dist_unit,
           reference_pt = v_great_lake_rec.sgl_ref_pt,
           d_userid_code = sim_export_pkg.v_pkg_ownerid,
           d_last_updt_ts = SYSDATE
     WHERE tsmalp_is_number = p_tsmalp_is_number
       AND tsmalp_org_id = rpad(p_org_id, 8);
  ELSE
    IF p_tsmvstc_is_number_old IS NOT NULL THEN
      check_type_delete(p_tsmalp_is_number, p_org_id, p_tsmvstc_is_number_old);
    END IF;
    IF v_great_lake_rec.sgl_shore_dist IS NULL THEN
      v_great_lake_rec.sgl_shore_dist := 0; 
    END IF;
  
    INSERT INTO tsmgll
      (tsmalp_is_number,
       tsmalp_org_id,
       NAME,
       addtnl_loc_name,
       shore_distance,
       shore_dist_unit_cd,
       reference_pt,
       d_userid_code,
       d_last_updt_ts)
    VALUES
      (p_tsmalp_is_number,
       rpad(p_org_id, 8),
       v_great_lake_rec.sgl_lake_name,
       v_great_lake_rec.sgl_add_lake_name,
       v_great_lake_rec.sgl_shore_dist,
       v_great_lake_rec.sgl_shore_dist_unit,
       v_great_lake_rec.sgl_ref_pt,
       sim_export_pkg.v_pkg_ownerid,
       SYSDATE);
  END IF;

  RETURN v_success;

EXCEPTION
  WHEN OTHERS THEN
  
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || 'migrating Great Lake');
    v_success := 0;
END create_great_lake;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE sim_export_stations
(
  p_imp_log_seq IN NUMBER,
  p_org_id      IN VARCHAR2,
  p_success     IN OUT NUMBER
) IS

  v_export_log_seq    NUMBER(10);
  v_station_is_nbr    NUMBER(8);
  v_tsmalp_is_nbr     NUMBER(8);
  v_station_cnt       NUMBER(10) := 0; /* ADDED SNG 4/18/01 */
  v_stl_seq           NUMBER(10);
  v_tsmnal_org_id     VARCHAR2(8);
  v_sid_seq           NUMBER(10);
  v_import_type       VARCHAR2(4);
  v_blob_success      NUMBER;
  v_rows_ready        NUMBER(10); 
  v_action_code       VARCHAR2(1);
  v_filename          VARCHAR2(265);
  v_tsmvstc_is_number tsmvstc.tsmvstc_is_number%TYPE;
  v_tsmvstc_org_id    tsmvstc.tsmvstc_org_id%TYPE;

  v_station_rec  sim_stations%ROWTYPE;
  v_location_rec sim_station_locations%ROWTYPE;

  CURSOR c_stations(p_org_id IN VARCHAR2) IS
    SELECT *
      FROM sim_stations
     WHERE rpad(sta_org_id, 8) = rpad(p_org_id, 8)
       AND sta_status = 'A'
       AND sta_import_seq = p_imp_log_seq;

  CURSOR c_station_locations(p_sta_seq IN NUMBER) IS
    SELECT *
      FROM sim_station_locations
     WHERE stl_sta_seq = p_sta_seq;

  CURSOR c_import_type IS
    SELECT sil_import_type
      FROM sim_import_log
     WHERE sil_seq = p_imp_log_seq;

BEGIN
  v_success := 1;

  BEGIN
    /* set up export log */
    OPEN c_import_type;
    FETCH c_import_type
      INTO v_import_type;
    CLOSE c_import_type;
  
    v_export_log_seq := p_imp_log_seq;
    SELECT substr(sil_import_file,instr(sil_import_file, '\', -1) +1)
      INTO v_filename
      FROM sim_import_log
      WHERE sil_seq = v_export_log_seq;
    INSERT INTO sim_export_log
      (sel_seq,
       sel_exported_by,
       sel_exported_on,
       sel_filename,
       sel_org_id,
       sel_export_type)
    VALUES
      (v_export_log_seq,
       USER,
       SYSDATE,
       v_filename,
       p_org_id,
       v_import_type);
    COMMIT;
    p_success := v_success;
  
  EXCEPTION
    WHEN OTHERS THEN
    
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                            ' - setting up export log record.');
    
      v_success := 0;
      p_success := v_success;
  END;

  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = (SELECT sil_sicc_seq
                       FROM sim_import_log
                      WHERE sil_seq = p_imp_log_seq);

  sim_event_log_pkg.write_event(4, v_export_log_seq, p_org_id, v_sid_seq, NULL);

  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message, NULL, 'Start: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
  SELECT sid_action_code
    INTO v_action_code
    FROM sim_imp_defs
   WHERE sid_seq = v_sid_seq;

  FOR v_station IN c_stations(p_org_id) LOOP
    --Default values
    --For TSMSTATN
    IF v_station.sta_estab_date IS NULL THEN
      v_station.sta_estab_date := to_date('01-JAN-0001', 'DD-MON-YYYY');
    END IF;
    IF v_station.sta_water_depth IS NULL THEN
      v_station.sta_water_depth := 0;
    END IF;
  
    /*If the Action is update then update Storet instead of Insert*/
    IF v_action_code = 'U' THEN
      BEGIN
        SELECT *
          INTO v_station_rec
          FROM sim_stations
         WHERE sta_id = v_station.sta_id
           AND sta_org_id = rpad(p_org_id, 8)
           AND sta_import_seq = p_imp_log_seq;
      
        SELECT t.tsmstatn_is_number,
               t.tsmvstc_is_number,
               t.tsmvstc_org_id
          INTO v_station_is_nbr,
               v_tsmvstc_is_number,
               v_tsmvstc_org_id
          FROM tsmstatn t
         WHERE t.identification_cd = rpad(v_station_rec.sta_id, 15)
           AND t.tsmstatn_org_id = rpad(p_org_id, 8);
      
        UPDATE tsmstatn
           SET epa_key_identifier = v_station_rec.sta_epa_key_id,
               NAME = v_station_rec.sta_name,
               establishment_date = v_station_rec.sta_estab_date,
               zid_relation_code = v_station_rec.sta_zid_rel,
               influence_area = v_station_rec.sta_influence_area,
               description_text = v_station_rec.sta_desc,
               travel_dir_txt = v_station_rec.sta_travel_dir,
               water_depth = v_station_rec.sta_water_depth,
               water_depth_unit = v_station_rec.sta_water_depth_unit,
               ecoregion_name = v_station_rec.sta_ecoregion,
               d_last_stn_updt_ts = SYSDATE,
               d_userid_code = sim_export_pkg.v_pkg_ownerid,
               d_last_updt_ts = SYSDATE,
               tsmorgan_is_number = v_station_rec.sta_tsmorgan_is_number,
               tsmvstc_is_number = v_station_rec.sta_tsmvstc_is_number,
               tsmvstc_org_id = v_station_rec.sta_tsmvstc_org_id,
               blob_type = sim_export_pkg.get_file_type(v_station_rec.sta_document),
               blob_title = sim_export_pkg.get_file_name(v_station_rec.sta_document)
         WHERE tsmstatn_is_number = v_station_is_nbr
           AND tsmstatn_org_id = rpad(p_org_id, 8);
      
        v_station_cnt := v_station_cnt + 1;
        
        /*Update additional location points*/
        BEGIN
          SELECT *
            INTO v_location_rec
            FROM sim_station_locations
           WHERE stl_sta_seq = v_station_rec.sta_seq;
        
          SELECT tsmalp_is_number
            INTO v_tsmalp_is_nbr
            FROM tsmalp
           WHERE tsmstatn0is_number = v_station_is_nbr
             AND tsmstatn0org_id = rpad(p_org_id, 8)
             AND type_code LIKE '*POINT%';
          
          IF v_location_rec.stl_elevation_msr IS NULL THEN
            v_location_rec.stl_elevation_msr := 0; --Default NULL Value
          END IF;
             
          IF v_location_rec.stl_seq IS NOT NULL THEN
            UPDATE tsmalp
               SET type_code = v_location_rec.stl_type_code,
                   sequence_number = v_location_rec.stl_sequence_number,
                   point_name = v_location_rec.stl_point_name,
                   elevation_datum_cd = v_location_rec.stl_elev_datum,
                   elvtn_method_cd = v_location_rec.stl_elev_method,
                   elvtn_msr_dt = v_location_rec.stl_elev_msr_date,
                   elevation_msr = v_location_rec.stl_elevation_msr,
                   elvtn_unt_cd = v_location_rec.stl_elevation_unit,
                   geopstng_datum_cd = v_location_rec.stl_geopos_datum,
                   geopstng_method_cd = v_location_rec.stl_geopos_method,
                   geopstng_scale_txt = v_location_rec.stl_gps_scale,
                   lat_direction = v_location_rec.stl_lat_dir,
                   lat_degree_msr = v_location_rec.stl_lat_degrees,
                   gps_lat_degree_msr = v_location_rec.stl_lat_degrees,
                   gps_lat_minute_msr = v_location_rec.stl_lat_dec_minutes,
                   gps_long_deg_msr = v_location_rec.stl_long_degrees,
                   gps_long_min_msr = v_location_rec.stl_long_dec_minutes,
                   lat_minute_msr = v_location_rec.stl_lat_minutes,
                   lat_second_msr = v_location_rec.stl_lat_seconds,
                   long_direction = v_location_rec.stl_long_dir,
                   long_degree_msr = v_location_rec.stl_long_degrees,
                   long_minute_msr = v_location_rec.stl_long_minutes,
                   long_second_msr = v_location_rec.stl_long_seconds,
                   lat_long_msr_dt = v_location_rec.stl_lat_long_msr_date,
                   lat_dec_deg_msr = v_location_rec.stl_lat_dec_degrees,
                   long_dec_deg_msr = v_location_rec.stl_long_dec_degrees,
                   nrcs_wtrsd_id_num = v_location_rec.stl_nrcs,
                   d_userid_code = sim_export_pkg.v_pkg_ownerid,
                   d_last_updt_ts = SYSDATE,
                   tsmfhu_is_number = v_location_rec.stl_tsmfhu_is_number,
                   tsmfhu_org_id = v_location_rec.stl_tsmfhu_org_id,
                   tsmstatn0is_number = v_location_rec.stl_tsmstatn0is_number,
                   tsmstatn0org_id = v_location_rec.stl_tsmstatn0org_id,
                   tsmwell_is_number = v_location_rec.stl_tsmwell_is_number,
                   tsmwell_org_id = v_location_rec.stl_tsmwell_org_id,
                   tsmgeopa_is_number = v_location_rec.stl_tsmgeopa_is_number,
                   tsmgeopa_org_id = v_location_rec.stl_tsmgeopa_org_id,
                   tsmnal_cd = v_location_rec.stl_tsmnal_cd,
                   tsmnal_state = v_location_rec.stl_tsmnal_state
             WHERE tsmalp_is_number = v_tsmalp_is_nbr
               AND tsmalp_org_id = rpad(p_org_id, 8);
          END IF;
        EXCEPTION
          WHEN no_data_found THEN
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                                  ' - in updating Location data.');
            v_success := 0;
            p_success := v_success;
          
          WHEN OTHERS THEN
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                                  ' In update stations.');
            v_success := 0;
            p_success := v_success;
        END;
        /*If the station type in Sim is either an ocean, estuary or great lake then
          call the create function to either insert or update*/
        IF v_station_rec.sta_tsmvstc_is_number = c_type_ocean THEN
          v_success := create_ocean(v_location_rec.stl_seq, 
                                    v_tsmalp_is_nbr, 
                                    p_org_id, 
                                    v_station_rec.sta_tsmvstc_is_number, 
                                    v_tsmvstc_is_number);
        
        ELSIF v_station_rec.sta_tsmvstc_is_number = c_type_estuary THEN
          v_success := create_estuary(v_location_rec.stl_seq, 
                                      v_tsmalp_is_nbr, 
                                      p_org_id, 
                                      v_station_rec.sta_tsmvstc_is_number, 
                                      v_tsmvstc_is_number);
        
        ELSIF v_station_rec.sta_tsmvstc_is_number = c_type_great_lake THEN
          v_success := create_great_lake(v_location_rec.stl_seq, 
                                         v_tsmalp_is_nbr, 
                                         p_org_id, 
                                         v_station_rec.sta_tsmvstc_is_number, 
                                         v_tsmvstc_is_number);
        
        ELSIF v_station_rec.sta_tsmvstc_is_number != v_tsmvstc_is_number THEN
          check_type_delete(v_tsmalp_is_nbr, p_org_id, v_tsmvstc_is_number);
        END IF;
      
      EXCEPTION
        WHEN no_data_found THEN
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                                ' - Station update could not find id');
          v_success := 0;
          p_success := v_success;
        WHEN OTHERS THEN
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || ' In update stations.');
          v_success := 0;
          p_success := v_success;
      END;
    
    ELSE
      BEGIN
        v_station_is_nbr := get_next_station_is_nbr(p_org_id);
        INSERT INTO tsmstatn
          (tsmstatn_is_number,
           tsmstatn_org_id,
           identification_cd,
           epa_key_identifier,
           NAME,
           establishment_date,
           zid_relation_code,
           influence_area,
           description_text,
           travel_dir_txt,
           water_depth,
           water_depth_unit,
           ecoregion_name,
           d_complete_flag,
           d_delete_flag,
           d_last_stn_updt_ts,
           d_userid_code,
           d_last_updt_ts,
           tsmorgan_is_number,
           tsmvstc_is_number,
           tsmvstc_org_id,
           blob_type,
           blob_title)
        VALUES
          (v_station_is_nbr /* TSMSTATN_IS_NUMBER */,
           rpad(p_org_id, 8) /* TSMSTATN_ORG_ID */,
           rpad(v_station.sta_id, 15) /* IDENTIFICATION_CD */,
           v_station.sta_epa_key_id /* EPA_KEY_IDENTIFIER */,
           v_station.sta_name /* NAME */,
           v_station.sta_estab_date /* ESTABLISHMENT_DATE */,
           v_station.sta_zid_rel /* ZID_RELATION_CODE */,
           v_station.sta_influence_area /* INFLUENCE_AREA */,
           v_station.sta_desc /* DESCRIPTION_TEXT */,
           v_station.sta_travel_dir /* TRAVEL_DIR_TXT */,
           v_station.sta_water_depth /* WATER_DEPTH */,
           v_station.sta_water_depth_unit /* WATER_DEPTH_UNIT */,
           v_station.sta_ecoregion /* ECOREGION_NAME */,
           'Y' /* D_COMPLETE_FLAG */,
           'N' /* D_DELETE_FLAG */,
           SYSDATE /* D_LAST_STN_UPDT_TS */,
           sim_export_pkg.v_pkg_ownerid,
           SYSDATE /* D_LAST_UPDT_TS */,
           v_station.sta_tsmorgan_is_number /* TSMORGAN_IS_NUMBER */,
           v_station.sta_tsmvstc_is_number /* TSMVSTC_IS_NUMBER */,
           'ZZZZZZZZ' /* TSMVSTC_ORG_ID */,
           sim_export_pkg.get_file_type(v_station.sta_document),
           sim_export_pkg.get_file_name(v_station.sta_document));
      
        -- Facilities - if it's a facility must update the facility table
        IF v_station.sta_tsmvstc_is_number > 21
           AND v_station.sta_tsmvstc_is_number < 27 THEN
          INSERT INTO tsmfclty
            (tsmstatn_is_number,
             tsmstatn_org_id,
             d_userid_code,
             d_last_updt_ts)
          VALUES
            (v_station_is_nbr,
             rpad(p_org_id, 8),
             sim_export_pkg.v_pkg_ownerid,
             SYSDATE);
        END IF;
      
        v_station_cnt := v_station_cnt + 1;
      
        /* STATION LOCATIONS */
        FOR v_location IN c_station_locations(v_station.sta_seq) LOOP
        
          /* kms - gets the method, datum, and units for elevation based on foreign keys in the
          station locations table if elevation is present.  Otherwise, these are set to null */
        
          IF v_location.stl_elevation_msr IS NULL THEN
            v_location.stl_elevation_msr := 0; --Default NULL Value
          END IF;
        
          v_stl_seq := v_location.stl_seq;
          IF v_location.stl_tsmnal_cd IS NOT NULL THEN
            v_tsmnal_org_id := 'ZZZZZZZZ';
          ELSE
            v_tsmnal_org_id := NULL;
          END IF;
        
          BEGIN
            v_tsmalp_is_nbr := get_next_tsmalp_is_nbr(rpad(p_org_id, 8));
          
            INSERT INTO tsmalp
              (tsmalp_is_number,
               tsmalp_org_id,
               type_code,
               sequence_number,
               point_name,
               elevation_datum_cd,
               elvtn_method_cd,
               elvtn_msr_dt,
               elevation_msr,
               elvtn_unt_cd,
               geopstng_datum_cd,
               geopstng_method_cd,
               geopstng_scale_txt,
               lat_direction,
               lat_degree_msr,
               gps_lat_degree_msr,
               gps_lat_minute_msr,
               gps_long_deg_msr,
               gps_long_min_msr,
               lat_minute_msr,
               lat_second_msr,
               long_direction,
               long_degree_msr,
               long_minute_msr,
               long_second_msr,
               lat_long_msr_dt,
               lat_dec_deg_msr,
               long_dec_deg_msr,
               rf3_river_reach_cd,
               rf1_mileage,
               on_river_reach_ind,
               nrcs_wtrsd_id_num,
               d_userid_code,
               d_last_updt_ts,
               tsmstatn_org_id,
               tsmstatn_is_number,
               tsmrrr_is_number,
               tsmrrr_org_id,
               tsmfhu_is_number,
               tsmfhu_org_id,
               tsmstatn0is_number,
               tsmstatn0org_id,
               tsmpipe_is_number,
               tsmpipe_org_id,
               tsmstatn1org_id,
               tsmstatn1is_number,
               tsmgeopa_is_number,
               tsmgeopa_org_id,
               tsmgeopa0is_number,
               tsmgeopa0org_id,
               tsmnal_cd,
               tsmnal_state,
               tsmnal_org_id,
               horiz_accuracy_msr,
               horiz_accur_unit,
               vert_accuracy_msr,
               vert_accuracy_unit)
            VALUES
              (v_tsmalp_is_nbr /* TSMALP_IS_NUMBER */,
               rpad(p_org_id, 8) /* TSMALP_ORG_ID */,
               rpad(v_location.stl_type_code, 16) /* TYPE_CODE */,
               v_location.stl_sequence_number /* SEQUENCE_NUMBER */,
               v_location.stl_point_name, /*POINT NAME */
               v_location.stl_elev_datum /* ELEVATION_DATUM_CD */,
               v_location.stl_elev_method /* ELVTN_METHOD_CD */,
               v_location.stl_elev_msr_date /* ELVTN_MSR_DATE */,
               to_number(v_location.stl_elevation_msr) /* ELEVATION_MSR */,
               v_location.stl_elevation_unit /* ELVTN_UNT_CD */,
               rpad(v_location.stl_geopos_datum, 12) /* GEOPSTNG_DATUM_CD */,
               rpad(v_location.stl_geopos_method, 12) /* GEOPSTNG_METHOD_CD */,
               v_location.stl_gps_scale /* GEOPSTNG_SCALE_TXT */,
               v_location.stl_lat_dir /* LAT_DIRECTION */,
               v_location.stl_lat_degrees /* LAT_DEGREE_MSR */,
               v_location.stl_lat_degrees /* GPS_LAT_DEGREE_MSR */,
               v_location.stl_lat_dec_minutes /* GPS_LAT_MINUTE_MSR */,
               v_location.stl_long_degrees /* GPS_LONG_DEG_MSR */,
               v_location.stl_long_dec_minutes /* GPS_LONG_MIN_MSR */,
               v_location.stl_lat_minutes /* LAT_MINUTE_MSR */,
               v_location.stl_lat_seconds /* LAT_SECOND_MSR */,
               v_location.stl_long_dir /* LONG_DIRECTION */,
               v_location.stl_long_degrees /* LONG_DEGREE_MSR */,
               v_location.stl_long_minutes /* LONG_MINUTE_MSR */,
               v_location.stl_long_seconds /* LONG_SECOND_MSR */,
               v_location.stl_lat_long_msr_date /* LAT_LONG_MSR_DT */,
               v_location.stl_lat_dec_degrees, /* LAT_DEC_DEG_MSR */
               v_location.stl_long_dec_degrees, /* LONG_DEC_DEG_MSR */
               --NULL                                       /* LATLONG_ACCRCY_MSR */,
               --NULL                                       /* LATLONG_ACCRCY_UNT */,
               --NULL                                       /* LONG_ACCRCY_MSR */,
               --NULL                                       /* LONG_ACCRCY_UNT_CD */,
               NULL /* RF3_RIVER_REACH_CD */,
               0 /* RF1_MILEAGE */,
               'N' /* ON_RIVER_REACH_IND */,
               v_location.stl_nrcs /* NRCS_WTRSD_ID_NUM */,
               sim_export_pkg.v_pkg_ownerid,
               SYSDATE /* D_LAST_UPDT_TS */,
               NULL /* TSMSTATN_ORG_ID */,
               NULL /* TSMSTATN_IS_NUMBER */,
               NULL /* TSMRRR_IS_NUMBER */,
               NULL /* TSMRRR_ORG_ID */,
               v_location.stl_tsmfhu_is_number /* TSMFHU_IS_NUMBER */,
               rpad(v_location.stl_tsmfhu_org_id, 8) /* TSMFHU_ORG_ID */,
               v_station_is_nbr /* TSMSTATN0IS_NUMBER */,
               rpad(p_org_id, 8) /* TSMSTATN0ORG_ID */,
               NULL /* TSMPIPE_IS_NUMBER */,
               NULL /* TSMPIPE_ORG_ID */,
               NULL /* TSMSTATN1ORG_ID */,
               NULL /* TSMSTATN1IS_NUMBER */,
               v_location.stl_tsmgeopa_is_number /* TSMGEOPA_IS_NUMBER */,
               rpad(v_location.stl_tsmgeopa_org_id, 8) /* TSMGEOPA_ORG_ID */,
               NULL, /* TSMGEOPA0IS_NUMBER */
               NULL, /* TSMGEOPA0ORG_ID */
               v_location.stl_tsmnal_cd /*TSMNAL_CD */,
               v_location.stl_tsmnal_state /*TSMNAL_STATE */,
               v_tsmnal_org_id, /*TSMNAL_ORG_ID */
               NULL, /*HORIZ_ACCURACY_MSR*/
               NULL, /*HORIZ_ACCUR_UNIT*/
               NULL, /*VERT_ACCURACY_MSR*/
               NULL /*VERT_ACCURACY_UNIT*/);
          
          EXCEPTION
            WHEN OTHERS THEN
            
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                                    'migrating sta location');
            
              v_success := 0;
              p_success := v_success;
          END;
        
          --sf update the record in sim_station_locations to include is numbers
          UPDATE sim_station_locations
             SET stl_is_number = v_tsmalp_is_nbr,
                 stl_org_id = p_org_id,
                 stl_tsmstatn0is_number = v_station_is_nbr,
                 stl_tsmstatn0org_id = p_org_id
           WHERE stl_seq = v_location.stl_seq;
        
          IF v_station.sta_tsmvstc_is_number = c_type_ocean THEN
            v_success := create_ocean(v_location.stl_seq, 
                                      v_tsmalp_is_nbr, 
                                      p_org_id, 
                                      v_station.sta_tsmvstc_is_number, 
                                      NULL);
          ELSIF v_station.sta_tsmvstc_is_number = c_type_estuary THEN
            v_success := create_estuary(v_location.stl_seq, 
                                        v_tsmalp_is_nbr, 
                                        p_org_id, 
                                        v_station.sta_tsmvstc_is_number, 
                                        NULL);
          ELSIF v_station.sta_tsmvstc_is_number = c_type_great_lake THEN
            v_success := create_great_lake(v_location.stl_seq, 
                                           v_tsmalp_is_nbr, 
                                           p_org_id, 
                                           v_station.sta_tsmvstc_is_number, 
                                           NULL);
          END IF;
        
          /* export binary document */
          IF v_station.sta_document IS NOT NULL THEN
            v_blob_success := sim_export_pkg.insert_binary_document
            (
              p_export_log_seq => v_export_log_seq, 
              p_table_name => 'TSMSTATN', 
              p_userid_code => sim_export_pkg.v_pkg_ownerid, 
              p_file_name => v_station.sta_document, 
              p_tsmstatn_is_number => v_station_is_nbr, 
              p_tsmstatn_org_id => p_org_id
            );
            IF v_blob_success = 0 THEN
            
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, 'Unable to load Document/Graphic file ' || '"' ||
                                                    v_station.sta_document ||
                                                    '".  Please check directory and file permissions.');
            
              v_success := 0;
              p_success := v_success;
            END IF;
          END IF;
        
        END LOOP; --end of locations loop
      
        /* update station with export seq */
        UPDATE sim_stations
           SET sta_export_seq = v_export_log_seq,
               sta_is_number = v_station_is_nbr,
               sta_status = 'E'
         WHERE sta_seq = v_station.sta_seq;
        COMMIT;
      
      EXCEPTION
        WHEN OTHERS THEN
        
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || ' exporting station');
        
          v_success := 0;
          p_success := v_success;
      END;
    END IF; --End of Update condition block*/
  END LOOP;

  /* update export log with count - added SNG 4/18/01*/
  UPDATE sim_export_log
     SET sel_activity_cnt = v_station_cnt
   WHERE sel_seq = v_export_log_seq;
  /* update the import log with export info
  SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE sim_import_log
     SET sil_exported_on = SYSDATE,
         sil_exported_by = USER,
         sil_rows_ready = sil_rows_ready - v_station_cnt
   WHERE sil_seq = p_imp_log_seq;

  /* all done! */

  COMMIT;
  p_success := v_success;

  /* If the action is an Update then remove data from sim regardless of success or not*/
  IF v_action_code = 'U' THEN
    sim_import_pkg.delete_import(p_imp_log_seq, FALSE, 'N');
    DELETE FROM sim_export_log
     WHERE sel_seq = v_export_log_seq;
  
  ELSE
    /* This code added to remove data from SIM database when successfully exported to 
    Storet - SJJ 10/21/05*/
    IF p_success = 1 THEN
      SELECT sil_rows_ready
        INTO v_rows_ready
        FROM sim_import_log
       WHERE sil_seq = p_imp_log_seq;
    
      IF v_rows_ready IS NULL
         OR v_rows_ready = 0 THEN
        sim_import_pkg.delete_import(p_imp_log_seq, TRUE, 'N');
      END IF;
    
    ELSE
      /* If p_success does not equal 1 then enter rows into sim_export_storet_ids table
      but do not delete sim data.  This will allow un-migrate then migrate without 
      the need to import records again*/
      sim_import_pkg.delete_import(p_imp_log_seq, TRUE, 'N');
      
      /*FOR v_rec IN (SELECT sta_is_number
                      FROM sim_stations
                     WHERE sta_is_number IS NOT NULL
                       AND sta_export_seq = v_export_log_seq) LOOP
        sim_import_pkg.sim_cleanup_post_export(p_export_log_seq => v_export_log_seq, 
                                               p_table_name => 'TSMSTATN', 
                                               p_org_id => p_org_id, 
                                               p_is_number => v_rec.sta_is_number);
      END LOOP;*/
    END IF;
  END IF;

  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message, NULL, 'Migrated ' || v_station_cnt ||
                                        ' stations.');

  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message, NULL, 'Finish at: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));

  sim_event_log_pkg.update_stop_time;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM || 'In sim export stations');
    v_success := 0;
    p_success := v_success;
  
END sim_export_stations;

END sim_export_station_pkg;
/
