CREATE OR REPLACE PACKAGE sim_export_well_pkg IS

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/


/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_export_log_seq NUMBER(10);
  v_success        NUMBER(1);

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ****************************************************************************************/

  PROCEDURE migrate_wells
  (
    p_imp_log_seq IN NUMBER,
    p_org_id      IN VARCHAR2,
    p_success     IN OUT NUMBER
  );
END sim_export_well_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_export_well_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/
  c_migrate_true    CONSTANT BOOLEAN := TRUE;
  c_new_event_no    CONSTANT VARCHAR2(1) := 'N';
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  
  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS
  v_item_name VARCHAR2(80);
  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Could not find allowable value with sequence: ' ||
                                          p_alval_seq);
     
        
  END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - getting allowable value sequence.');
     
        
    v_success := 0;
    RETURN NULL;
END get_allowable_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_well_is_number(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_next_well_is_nbr NUMBER(8);

BEGIN
  /* get the next avail is nbr to use */
  v_next_well_is_nbr := sim_export_pkg.get_eisn_number('TSMWELL', p_org_id);
  RETURN v_next_well_is_nbr;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get next well is number.');
     
        
    v_success := 0;
    RETURN NULL;
END get_next_well_is_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_next_tsmalp_is_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_tsmalp_is_nbr NUMBER(8);
BEGIN
  v_tsmalp_is_nbr := sim_export_pkg.get_eisn_number('TSMALP', p_org_id);

  RETURN v_tsmalp_is_nbr;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' in get next tsmalp is nbr');
     
        
    v_success := 0;
END get_next_tsmalp_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE migrate_wells
(
  p_imp_log_seq IN NUMBER,
  p_org_id      IN VARCHAR2,
  p_success     IN OUT NUMBER
) IS

  v_export_log_seq NUMBER(10);
  v_tsmalp_is_nbr  NUMBER(8);
  v_well_is_nbr    NUMBER(8);
  v_well_cnt       NUMBER(10) := 0;
  v_sid_seq        NUMBER(10); 
  v_filename       VARCHAR2(265); 
  v_well           sim_station_wells%ROWTYPE;
  v_location       sim_station_locations%ROWTYPE;
  v_rows_ready     NUMBER(10); --Scott 10/21/05
  
  e_migration_failed EXCEPTION;
  e_no_location EXCEPTION;

  CURSOR c_station_locations(p_well_seq IN NUMBER) IS
    SELECT *
      FROM sim_station_locations
     WHERE stl_ssw_seq = p_well_seq;

  CURSOR c_well IS
    SELECT *
      FROM sim_station_wells
     WHERE ssw_import_seq = p_imp_log_seq
       AND ssw_status = 'A';

BEGIN
  v_success := 1;
  BEGIN
    /* set up export log */
    v_export_log_seq := p_imp_log_seq;
    SELECT substr(sil_import_file,instr(sil_import_file, '\', -1) +1)
      INTO v_filename
      FROM sim_import_log
      WHERE sil_seq = v_export_log_seq;
    INSERT INTO sim_export_log
      (sel_seq,
       sel_exported_by,
       sel_exported_on,
       sel_filename,
       sel_org_id,
       sel_export_type)
    VALUES
      (v_export_log_seq,
       USER,
       SYSDATE,
       v_filename,
       p_org_id,
       'W');
    COMMIT;
  
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL,
        SQLERRM || ' - setting up migration log record.');
      RAISE e_migration_failed;
  END;
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = (SELECT sil_sicc_seq 
                       FROM sim_import_log 
                      WHERE sil_seq = p_imp_log_seq);
  sim_event_log_pkg.write_event(4,v_export_log_seq,p_org_id,v_sid_seq,NULL); 
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Start: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  -- Wells
  BEGIN
    FOR v_well IN c_well LOOP
      v_well_is_nbr := get_next_well_is_number(rpad(p_org_id, 8));
      INSERT INTO tsmwell
        (tsmwell_is_number,
         tsmwell_org_id,
         NAME,
         id_number,
         use_code,
         prtctn_area_name,
         developmnt_mthd_cd,
         constructn_mthd_cd,
         wtr_primry_use_cd,
         gradient_type_code,
         depth_cmpltn_msr,
         dpth_cmpltn_unt_cd,
         depth_of_hole_msr,
         depth_hole_unt_cd,
         well_head_ht_msr,
         wl_hd_ht_unit_cd,
         casing_hgt_msr,
         casing_hgt_unit_cd,
         disinfected_ind_cd,
         constructn_strt_dt,
         constructn_end_dt,
         init_pump_dur,
         int_pmp_dur_unt_cd,
         init_pump_rate,
         int_pmp_rat_unt_cd,
         naturl_flow_ind_cd,
         status_code,
         init_borehole_diam,
         init_bhole_diam_un,
         d_userid_code,
         d_last_updt_ts,
         tsmstatn_is_number,
         tsmstatn_org_id)
      VALUES
        (v_well_is_nbr, --TSMWELL_IS_NUMBER,
         rpad(p_org_id, 8), --TSMWELL_ORG_ID,
         v_well.ssw_name, --NAME,
         v_well.ssw_well_no, --ID_NUMBER,
         v_well.ssw_well_use, --USE_CODE,
         NULL, --PRTCTN_AREA_NAME,
         NULL, --DEVELOPMNT_MTHD_CD,
         NULL, --CONSTRUCTN_MTHD_CD,
         NULL, --WTR_PRIMRY_USE_CD,
         NULL, --GRADIENT_TYPE_CODE,
         NULL, --DEPTH_CMPLTN_MSR,
         NULL, --DPTH_CMPLTN_UNT_CD,
         NULL, --DEPTH_OF_HOLE_MSR,
         NULL, --DEPTH_HOLE_UNT_CD,
         NULL, --WELL_HEAD_HT_MSR,
         NULL, --WL_HD_HT_UNIT_CD,
         NULL, --CASING_HGT_MSR,
         NULL, --CASING_HGT_UNIT_CD,
         NULL, --DISINFECTED_IND_CD,
         NULL, --CONSTRUCTN_STRT_DT,
         NULL, --CONSTRUCTN_END_DT,
         NULL, --INIT_PUMP_DUR,
         NULL, --INT_PMP_DUR_UNT_CD,
         NULL, --INIT_PUMP_RATE,
         NULL, --INT_PMP_RAT_UNT_CD,
         NULL, --NATURL_FLOW_IND_CD,
         v_well.ssw_well_status, --STATUS_CODE,
         NULL, --INIT_BOREHOLE_DIAM,
         NULL, --INIT_BHOLE_DIAM_UN,
         sim_export_pkg.v_pkg_ownerid,
         SYSDATE, --D_LAST_UPDT_TS,
         v_well.ssw_tsmstatn_is_number, --TSMSTATN_IS_NUMBER,
         rpad(p_org_id, 8) --TSMSTATN_ORG_ID,
         );
    
      UPDATE sim_station_wells
         SET ssw_is_number = v_well_is_nbr,
             ssw_org_id = rpad(p_org_id, 8),
             ssw_export_seq = v_export_log_seq,
             ssw_status = 'E'
       WHERE ssw_seq = v_well.ssw_seq;
    
      v_well_cnt := v_well_cnt + 1;
      --CLOSE c_well;
    
      --For each well there is a single location record
      BEGIN
        /* WELL LOCATIONS */
        OPEN c_station_locations(v_well.ssw_seq);
        FETCH c_station_locations
          INTO v_location;
        IF c_station_locations%NOTFOUND THEN
          RAISE e_no_location;
        ELSE
        
          IF v_location.stl_elevation_msr IS NULL THEN
            v_location.stl_elevation_msr := 0; --Default NULL Value
          END IF;
        
          v_tsmalp_is_nbr := get_next_tsmalp_is_nbr(rpad(p_org_id, 8));
        
          INSERT INTO tsmalp
            (tsmalp_is_number,
             tsmalp_org_id,
             type_code,
             sequence_number,
             elevation_datum_cd,
             elvtn_method_cd,
             elvtn_msr_dt,
             elevation_msr,
             elvtn_unt_cd,
             geopstng_datum_cd,
             geopstng_method_cd,
             geopstng_scale_txt,
             lat_direction,
             lat_degree_msr,
             gps_lat_degree_msr,
             gps_lat_minute_msr,
             gps_long_deg_msr,
             gps_long_min_msr,
             lat_minute_msr,
             lat_second_msr,
             long_direction,
             long_degree_msr,
             long_minute_msr,
             long_second_msr,
             lat_long_msr_dt,
             horiz_accuracy_msr,
             horiz_accur_unit,
             vert_accuracy_msr,
             vert_accuracy_unit,
             rf3_river_reach_cd,
             rf1_mileage,
             on_river_reach_ind,
             nrcs_wtrsd_id_num,
             d_userid_code,
             d_last_updt_ts,
             tsmstatn_org_id,
             tsmstatn_is_number,
             tsmrrr_is_number,
             tsmrrr_org_id,
             tsmfhu_is_number,
             tsmfhu_org_id,
             tsmstatn0is_number,
             tsmstatn0org_id,
             tsmpipe_is_number,
             tsmpipe_org_id,
             tsmwell_is_number,
             tsmwell_org_id,
             tsmstatn1org_id,
             tsmstatn1is_number,
             tsmgeopa_is_number,
             tsmgeopa_org_id,
             tsmgeopa0is_number,
             tsmgeopa0org_id,
             tsmnal_cd,
             tsmnal_state,
             tsmnal_org_id,
             lat_dec_deg_msr,
             long_dec_deg_msr)
          VALUES
            (v_tsmalp_is_nbr /* TSMALP_IS_NUMBER */,
             rpad(p_org_id, 8) /* TSMALP_ORG_ID */,
             rpad('WELL HEAD', 16) /* TYPE_CODE */,
             v_location.stl_sequence_number /* SEQUENCE_NUMBER */,
             v_location.stl_elev_datum /* ELEVATION_DATUM_CD */,
             v_location.stl_elev_method /* ELVTN_METHOD_CD */,
             v_location.stl_elev_msr_date /* ELVTN_MSR_DATE */,
             to_number(v_location.stl_elevation_msr) /* ELEVATION_MSR */,
             v_location.stl_elevation_unit /* ELVTN_UNT_CD */,
             v_location.stl_geopos_datum /* GEOPSTNG_DATUM_CD */,
             v_location.stl_geopos_method /* GEOPSTNG_METHOD_CD */,
             v_location.stl_gps_scale /* GEOPSTNG_SCALE_TXT */,
             v_location.stl_lat_dir /* LAT_DIRECTION */,
             v_location.stl_lat_degrees /* LAT_DEGREE_MSR */,
             v_location.stl_lat_degrees /* GPS_LAT_DEGREE_MSR */,
             v_location.stl_lat_dec_minutes /* GPS_LAT_MINUTE_MSR */,
             v_location.stl_long_degrees /* GPS_LONG_DEG_MSR */,
             v_location.stl_long_dec_minutes /* GPS_LONG_MIN_MSR */,
             v_location.stl_lat_minutes /* LAT_MINUTE_MSR */,
             v_location.stl_lat_seconds /* LAT_SECOND_MSR */,
             v_location.stl_long_dir /* LONG_DIRECTION */,
             v_location.stl_long_degrees /* LONG_DEGREE_MSR */,
             v_location.stl_long_minutes /* LONG_MINUTE_MSR */,
             v_location.stl_long_seconds /* LONG_SECOND_MSR */,
             v_location.stl_lat_long_msr_date /* LAT_LONG_MSR_DT */,
             NULL /* LATLONG_ACCRCY_MSR */,
             NULL /* LATLONG_ACCRCY_UNT */,
             NULL /* LONG_ACCRCY_MSR */,
             NULL /* LONG_ACCRCY_UNT_CD */,
             NULL /* RF3_RIVER_REACH_CD */,
             0 /* RF1_MILEAGE */,
             'N' /* ON_RIVER_REACH_IND */,
             v_location.stl_nrcs /* NRCS_WTRSD_ID_NUM */,
             sim_export_pkg.v_pkg_ownerid,
             SYSDATE /* D_LAST_UPDT_TS */,
             NULL /* TSMSTATN_ORG_ID */,
             NULL /* TSMSTATN_IS_NUMBER */,
             v_location.stl_tsmrrr_is_number /* TSMRRR_IS_NUMBER */,
             rpad(v_location.stl_tsmrrr_org_id, 8) /* TSMRRR_ORG_ID */,
             v_location.stl_tsmfhu_is_number /* TSMFHU_IS_NUMBER */,
             rpad(v_location.stl_tsmfhu_org_id, 8) /* TSMFHU_ORG_ID */,
             v_well.ssw_tsmstatn_is_number /* TSMSTATN0IS_NUMBER */,
             rpad(p_org_id, 8) /* TSMSTATN0ORG_ID */,
             NULL /* TSMPIPE_IS_NUMBER */,
             NULL /* TSMPIPE_ORG_ID */,
             v_well_is_nbr /* TSMWELL_IS_NUMBER */,
             rpad(p_org_id, 8) /* TSMWELL_ORG_ID */,
             NULL /* TSMSTATN1ORG_ID */,
             NULL /* TSMSTATN1IS_NUMBER */,
             NULL /* TSMGEOPA_IS_NUMBER */,
             NULL /* TSMGEOPA_ORG_ID */,
             NULL, /* TSMGEOPA0IS_NUMBER */
             NULL, /* TSMGEOPA0ORG_ID */
             NULL /*TSMNAL_CD */,
             NULL /*TSMNAL_STATE */,
             NULL /*TSMNAL_ORG_ID */,
             v_location.stl_lat_dec_degrees,
             v_location.stl_long_dec_degrees);
        
          UPDATE sim_station_locations
             SET stl_tsmwell_is_number = v_well_is_nbr,
                 stl_tsmwell_org_id = p_org_id,
                 stl_is_number = v_tsmalp_is_nbr,
                 stl_org_id = p_org_id
           WHERE stl_seq = v_location.stl_seq;
          COMMIT;
        END IF;
        CLOSE c_station_locations;
      EXCEPTION
        WHEN e_no_location THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               'Well location record not found');
           
              
          CLOSE c_station_locations;
          RAISE e_migration_failed;
        WHEN OTHERS THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                               SQLERRM || ' migrating well location');
           
              
          CLOSE c_station_locations;
          RAISE e_migration_failed;
      END;
    
    END LOOP;
  
  EXCEPTION
    WHEN OTHERS THEN
        
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                           SQLERRM || ' migrating well');
       
          
      RAISE e_migration_failed;
  END;

  /* update export log with count - added SNG 4/18/01*/
  UPDATE sim_export_log
     SET sel_activity_cnt = v_well_cnt
   WHERE sel_seq = v_export_log_seq;

  /* update the import log with export info
  SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE sim_import_log
     SET sil_exported_on = SYSDATE,
         sil_exported_by = USER,
         sil_rows_ready = sil_rows_ready - v_well_cnt
   WHERE sil_seq = p_imp_log_seq;

  /* all done! */
  COMMIT;
  p_success := v_success;
  
  /* Following code added to call delete_import procedure to delete SIM well records
     based on import_seq when successful export of Wells*/
  IF p_success = 1 THEN
    SELECT sil_rows_ready
      INTO v_rows_ready
      FROM sim_import_log
      WHERE sil_seq = p_imp_log_seq;
      
    IF v_rows_ready IS NULL OR v_rows_ready = 0 THEN
      sim_import_pkg.delete_import(p_imp_log_seq, c_migrate_true, c_new_event_no);
    END IF;
  ELSE
    /* If export is not successful still remove data from SIM*/
    sim_import_pkg.delete_import(p_imp_log_seq, c_migrate_true, c_new_event_no);
                                         
  END IF;
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Migrated ' || v_well_cnt || ' wells.');
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Finish at: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  sim_event_log_pkg.update_stop_time;
     
EXCEPTION
  WHEN e_migration_failed THEN
    v_success := 0;
    p_success := v_success;
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' in migrate wells');
     
        
    v_success := 0;
    p_success := v_success;
  
END migrate_wells;

END sim_export_well_pkg;
/
