CREATE OR REPLACE PACKAGE sim_import_pkg IS

  /* SIM200 */
  /* Modified 3/21/02 by SF. Added code at the end of Delete_import that looks for all the trips
  and station visits in SIM and delets them if they dont have any activities associated with them. */

  /* import utils by SNG Feb 2000 */

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                               *
 ****************************************************************************************/
  TYPE col_table_type IS TABLE OF VARCHAR2(4000) INDEX BY BINARY_INTEGER;

  TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_activity_count t_activity_count;
  --flag that we've already warned logged the warning when removing wrapping quotes.
  v_strip_quote NUMBER := NULL; 

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                           *
 ****************************************************************************************/

  PROCEDURE parse_line_table (
    pv_line          IN LONG,
    p_col_table      OUT col_table_type,
    p_no_cols        IN NUMBER,
    p_delimiter      IN VARCHAR2 DEFAULT ',',
    p_bad_parse      IN OUT BOOLEAN,
    p_import_log_seq IN NUMBER
  );

  FUNCTION get_one_col
  (
    pv_line          IN LONG,
    p_col_no         IN NUMBER,
    p_delimiter      IN VARCHAR2,
    p_import_log_seq IN NUMBER
  ) RETURN VARCHAR2;

  PROCEDURE delete_import(
    p_import_seq    IN NUMBER,
    p_migrate       IN BOOLEAN := FALSE,
    p_new_event_yn  IN VARCHAR2 := 'Y');

  PROCEDURE delete_import_all
  (
    p_org_id           IN sim_import_log.sil_tsmorgan_org_id%TYPE,
    p_include_projects IN BOOLEAN := FALSE
  );

  FUNCTION get_import_log_seq RETURN NUMBER;

  PROCEDURE update_sil_rows_ready
  (
    p_import_log_seq    IN NUMBER,
    p_rows_ready_change IN NUMBER
  );

  PROCEDURE update_sil_rows_errors
  (
    p_import_log_seq IN NUMBER,
    p_rows_errors    IN NUMBER
  );

  FUNCTION ind_nbr_sort(p_seq NUMBER) RETURN VARCHAR2;

  FUNCTION check_document_file
  (
    p_file_name      IN VARCHAR2,
    p_import_log_seq IN NUMBER,
    p_line           IN NUMBER
  ) RETURN BOOLEAN;

  FUNCTION get_latitude
  (
    v_import_log_seq  IN NUMBER,
    v_item_in         IN VARCHAR2,
    v_column_seq      IN NUMBER,
    p_imp_cfg_seq     IN NUMBER,
    v_sta_seq         IN NUMBER,
    v_line_cnt        IN NUMBER,
    p_lat_degrees     OUT NUMBER,
    p_lat_minutes     OUT NUMBER,
    p_lat_seconds     OUT NUMBER,
    p_lat_dir         OUT VARCHAR2,
    p_lat_dec_min     OUT NUMBER,
    p_lat_dec_deg     OUT NUMBER
  ) RETURN NUMBER;
  
  FUNCTION get_longitude
  (
    v_import_log_seq  IN NUMBER,
    v_item_in         IN VARCHAR2,
    v_column_seq      IN NUMBER,
    p_imp_cfg_seq     IN NUMBER,
    v_sta_seq         IN NUMBER,
    v_line_cnt        IN NUMBER,
    p_long_degrees    OUT NUMBER,
    p_long_minutes    OUT NUMBER,
    p_long_seconds    OUT NUMBER,
    p_long_dec_min    OUT NUMBER,
    p_long_dec_deg    OUT NUMBER,
    p_long_dir        OUT VARCHAR2
  ) RETURN NUMBER;
  
  PROCEDURE sim_cleanup_post_export 
  (
  p_export_log_seq  IN  NUMBER,
  p_table_name      IN  VARCHAR2,
  p_org_id          IN  VARCHAR2,
  p_is_number       IN  NUMBER);
  
  FUNCTION check_allowable_values(
    p_item           IN VARCHAR2, 
    p_sic_seq        IN NUMBER, 
    v_line_cnt       IN NUMBER) 
    RETURN NUMBER;
  
  FUNCTION check_translations
  (
    p_code        IN VARCHAR2,
    p_imp_cfg_seq IN NUMBER,
    p_col_seq     IN NUMBER,
    p_line_cnt    IN NUMBER,
    p_trans_exist OUT BOOLEAN
  ) RETURN VARCHAR2;
  
  FUNCTION get_rows_with_errors(
    p_sil_seq IN sim_event_log.sevl_ref_seq%TYPE) 
    RETURN NUMBER;

END;
/
CREATE OR REPLACE PACKAGE BODY sim_import_pkg AS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                             *
 ****************************************************************************************/
 
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/

  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/
 
/****************************************************************************************
  PROC TO PARSE one comma delimited line into a table  
 ****************************************************************************************/ 
PROCEDURE parse_line_table (
  pv_line          IN LONG,
  p_col_table      OUT col_table_type,
  p_no_cols        IN NUMBER,
  p_delimiter      IN VARCHAR2 DEFAULT ',',
  p_bad_parse      IN OUT BOOLEAN,
  p_import_log_seq IN NUMBER) IS

  v_col_cnt BINARY_INTEGER := 1;
  col_cnt   INTEGER;

BEGIN
  -- Get the actual number of columns in the import line
  col_cnt := length(pv_line) - length(translate(pv_line, 'a' || p_delimiter, 'a')) + 1;
  -- Process the import line
  WHILE v_col_cnt <= col_cnt LOOP
    p_col_table(v_col_cnt) := get_one_col(pv_line, v_col_cnt, p_delimiter, p_import_log_seq);
    v_col_cnt := v_col_cnt + 1;
  END LOOP;
  IF p_no_cols < col_cnt THEN
    -- More columns found in inport line that in configuration
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,NULL,
                                         'Warning: There are more columns in the line than defined in the configuration');
  ELSIF p_no_cols > col_cnt THEN
    -- Add empty column at the end for the generate columns
    WHILE v_col_cnt <= p_no_cols LOOP
      p_col_table(v_col_cnt) := NULL;
      v_col_cnt := v_col_cnt + 1;
    END LOOP;
  END IF;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                         '- in parse_line_table function, column ' ||
                                         p_no_cols);
      
END parse_line_table;

/****************************************************************************************
  Internal function to get each column - used by parse_line_table proc 
 ****************************************************************************************/ 
FUNCTION get_one_col
(
  pv_line          IN LONG,
  p_col_no         IN NUMBER,
  p_delimiter      IN VARCHAR2,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  v_first_comma NUMBER;
  v_start_pos   NUMBER;
  v_line_length NUMBER;
  v_col_data    VARCHAR2(4000); 

BEGIN
  v_line_length := length(pv_line);
  IF p_col_no = 1 THEN
    v_start_pos := 1;
  ELSE
    v_start_pos := instr(pv_line, p_delimiter, 1, p_col_no - 1) + 1;
  END IF;
  v_first_comma := instr(pv_line, p_delimiter, 1, p_col_no);
  IF v_first_comma = 0 THEN
    v_first_comma := v_line_length + 1;
  END IF;
  v_col_data := TRIM(substr(pv_line, v_start_pos, v_first_comma - v_start_pos));
  -- trim preceeding/trailing quotes from the item (if neccessary)
  IF substr(v_col_data, 1, 1 ) IN ('"', '''') 
      AND substr(v_col_data, length(v_col_data), 1) IN ('"', '''') THEN
    v_col_data := TRIM(substr(v_col_data, 2, length(v_col_data) - 2));
    IF v_strip_quote IS NULL THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning, NULL, 'Warning: Import file contains '
        || 'quotes around column values. Quotes have been removed.');
       
      v_strip_quote := 1;
    END IF;
  END IF;
  RETURN v_col_data;
  
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
      SQLERRM || '- in get_one_col function, column ' || p_col_no);
     
       
END get_one_col;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE delete_import(
  p_import_seq    IN NUMBER,
  p_migrate       IN BOOLEAN := FALSE,
  p_new_event_yn  IN VARCHAR2 := 'Y') IS

  v_import_type  VARCHAR2(30);
  v_dummy_seq    NUMBER(10);
  v_visit_seq    NUMBER(10);
  v_trip_seq     NUMBER(10);
  v_sid_seq      NUMBER(10);
  v_org_id       VARCHAR2(8);

  CURSOR c_get_visit_info(p_visit_seq IN NUMBER) IS
    SELECT ssv_seq
      FROM sim_station_visits ssv,
           tsrstvst t
     WHERE ssv.ssv_tsrstvst_is_number = t.tsrstvst_is_number
       AND ssv.ssv_tsrstvst_org_id = t.tsrstvst_org_id
       AND ssv.ssv_seq = p_visit_seq;

  CURSOR c_get_all_trips IS
    SELECT stp_seq
      FROM sim_trips;

  CURSOR c_get_all_visits(p_trip_seq IN NUMBER) IS
    SELECT ssv_seq
      FROM sim_station_visits
     WHERE ssv_stp_seq = p_trip_seq;

  CURSOR c_get_all_activities(p_visit_seq IN NUMBER, p_trip_seq IN NUMBER) IS
    SELECT fa_seq
      FROM sim_field_activities
     WHERE fa_ssv_seq = p_visit_seq
       AND fa_stp_seq = p_trip_seq;

  CURSOR c_get_trip_info(p_trip_seq IN NUMBER) IS
    SELECT stp_seq
      FROM sim_trips st,
           tsrtrip t
     WHERE st.stp_tsrtrip_is_number = t.tsrtrip_is_number
       AND st.stp_tsrtrip_org_id = t.tsrtrip_org_id
       AND stp_seq = p_trip_seq;

  CURSOR c_check_visit_in_sim(p_visit_seq IN NUMBER) IS
    SELECT fa_seq
      FROM sim_field_activities
     WHERE fa_ssv_seq = p_visit_seq;

  CURSOR c_check_trip_in_sim(p_trip_seq IN NUMBER) IS
    SELECT fa_seq
      FROM sim_field_activities
     WHERE fa_stp_seq = p_trip_seq;

  CURSOR c_result_stations IS
    SELECT sta_seq
      FROM sim_stations s
      WHERE s.sta_status = 'U'
        AND NOT EXISTS (SELECT 1
              FROM sim_field_activities f
              WHERE f.fa_tsmstatn_is_number = s.sta_is_number
                AND f.fa_tsmstatn_org_id = s.sta_org_id)
        AND NOT EXISTS (SELECT 1
              FROM sim_station_locations l
              WHERE l.stl_sta_seq = s.sta_seq
                AND l.stl_status != 'U')
        AND NOT EXISTS (SELECT 1
              FROM sim_station_wells w
              WHERE w.ssw_sta_seq = s.sta_seq
                AND w.ssw_status != 'U');

  CURSOR c_check_import_line(p_import_seq IN NUMBER) IS
    SELECT sipl_seq
      FROM sim_imp_lines
     WHERE sipl_sil_seq = p_import_seq;

BEGIN   
  SELECT sicc.sicc_sid_seq, sil.sil_tsmorgan_org_id, sid.sid_type
    INTO v_sid_seq, v_org_id, v_import_type
    FROM sim_imp_cfg sicc, sim_import_log sil, sim_imp_defs sid
    WHERE sil.sil_seq = p_import_seq
      AND sil.sil_sicc_seq = sicc.sicc_seq
      AND sicc.sicc_sid_seq = sid.sid_seq; 
  IF p_new_event_yn = 'Y' THEN
    sim_event_log_pkg.write_event(sim_lib_pkg.c_selt_delete_import,p_import_seq,v_org_id,v_sid_seq,NULL);    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Start: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  END IF;           
  /********************* STATIONS, LOCATIONS, WELLS ***********************************/
  IF v_import_type IN ('STATIONS', 'WELLS', 'LOCATIONS') THEN
    -- DELETING A STATIONS IMPORT  This will also delete a single well and location that came in with
    -- a station. Multiple wells and locations MIGHT have brought in some existing STORET Stations and
    -- they, too will be deleted here. 
    FOR v_stations IN (
        SELECT sta_seq, sta_is_number, sta_org_id, sta_export_seq
          FROM sim_stations
          WHERE sta_import_seq = p_import_seq) LOOP  
      FOR v_sta_loc_rec IN (
          SELECT stl_seq
            FROM sim_station_locations
            WHERE stl_sta_seq = v_stations.sta_seq) LOOP
        DELETE sim_ocean
          WHERE soc_stl_seq = v_sta_loc_rec.stl_seq;
        DELETE sim_estuary_loc
          WHERE sel_stl_seq = v_sta_loc_rec.stl_seq;
        DELETE sim_great_lake
          WHERE sgl_stl_seq = v_sta_loc_rec.stl_seq;
        -- Delete locations (as long as they didn't come from a LOCATIONS Import)
        IF v_import_type <> 'LOCATIONS' THEN        
          DELETE sim_station_locations
            WHERE stl_seq = v_sta_loc_rec.stl_seq;
        END IF;
      END LOOP;
      
      -- Delete any wells (as long as they didn't come from a WELLS Import)
      -- Does this ever happen?
      IF v_import_type <> 'WELLS' THEN
        DELETE sim_station_wells
          WHERE ssw_sta_seq = v_stations.sta_seq;
        -- Do not allow insert to sim_export_storet_ids if type is wells or locations
        IF v_import_type <> 'LOCATIONS' THEN
          IF p_migrate = TRUE THEN
            sim_cleanup_post_export(
              v_stations.sta_export_seq,
              'TSMSTATN',
              v_stations.sta_org_id,
              v_stations.sta_is_number);
          END IF;    
          DELETE sim_stations
            WHERE sta_seq = v_stations.sta_seq;
        END IF;
      END IF;
      COMMIT;
    END LOOP;
  
    -- Now delete individual wells and locations that have their own import_log_seq.
    FOR v_loc_rec IN (
        SELECT stl_seq, stl_is_number, stl_org_id, stl_export_seq
          FROM sim_station_locations
          WHERE stl_import_seq = p_import_seq) LOOP
      IF p_migrate = TRUE THEN 
        sim_cleanup_post_export(
          v_loc_rec.stl_export_seq,
          'TSMALP',
          v_loc_rec.stl_org_id,
          v_loc_rec.stl_is_number);
      END IF; 
      DELETE sim_station_locations
        WHERE stl_seq = v_loc_rec.stl_seq;
    END LOOP;
    DELETE sim_stations
      WHERE sta_import_seq = p_import_seq;
    COMMIT;
    FOR v_well IN (
        SELECT ssw_seq, ssw_is_number, ssw_org_id, ssw_export_seq 
          FROM sim_station_wells
          WHERE ssw_import_seq = p_import_seq) LOOP
      IF p_migrate = TRUE THEN
         sim_cleanup_post_export(
           v_well.ssw_export_seq,
           'TSMWELL',
           v_well.ssw_org_id,
           v_well.ssw_is_number);
      END IF;    
      DELETE sim_station_locations
        WHERE stl_ssw_seq = v_well.ssw_seq;                       
      DELETE sim_station_wells
        WHERE ssw_seq = v_well.ssw_seq; 
    END LOOP;
    COMMIT;
  
  /***************************** PROJECTS ***********************************************/
  ELSIF v_import_type = 'PROJECTS' THEN
    FOR proj_rec IN (
        SELECT spj_seq, spj_tsmproj_is_number, spj_tsmproj_org_id, spj_export_seq
          FROM sim_projects
          WHERE spj_import_seq = p_import_seq) LOOP
      IF p_migrate = TRUE THEN
        sim_cleanup_post_export(
          proj_rec.spj_export_seq,
          'TSMPROJ',
          proj_rec.spj_tsmproj_org_id,
          proj_rec.spj_tsmproj_is_number);
      END IF;    
      DELETE sim_projects
        WHERE spj_seq = proj_rec.spj_seq;
    END LOOP;
    COMMIT;
       
  /********************************* RESULTS ********************************************/     
  ELSE
    FOR v_field_activity IN (SELECT fa_seq, 
                                    fa_ssv_seq, 
                                    fa_stp_seq, 
                                    fa_type_name,
                                    fa_tsrfdact_is_number, 
                                    fa_tsrfdact_org_id, 
                                    fa_tsrfqs_is_number,
                                    fa_tsrfqs_org_id,
                                    fa_export_seq
                                FROM sim_field_activities
                                WHERE fa_import_seq = p_import_seq) LOOP 
        
             
      DELETE sim_result_class_indicator
        WHERE rci_fa_seq = v_field_activity.fa_seq;    
      DELETE sim_bio_results_grp_individual
        WHERE brgi_fa_seq = v_field_activity.fa_seq;
      DELETE sim_bio_results_grp
        WHERE brg_fa_seq = v_field_activity.fa_seq;    
      DELETE sim_result_lab_remarks
        WHERE srlr_rs_seq IN (SELECT rs_seq
                                FROM sim_results
                                WHERE rs_fa_seq = v_field_activity.fa_seq);   
      DELETE sim_results
        WHERE rs_fa_seq = v_field_activity.fa_seq;
      DELETE sim_trap_net
        WHERE stn_fa_seq = v_field_activity.fa_seq;
      DELETE sim_activity_personnel
        WHERE sap_fa_seq = v_field_activity.fa_seq;
      DELETE sim_activity_cporg
        WHERE sac_fa_seq = v_field_activity.fa_seq;
      DELETE sim_activity_projects
        WHERE sapj_fa_seq = v_field_activity.fa_seq;
      DELETE sim_actual_activity_loc
        WHERE aal_fa_seq = v_field_activity.fa_seq;   
      DELETE sim_trawl_ops_det
        WHERE tod_fa_seq = v_field_activity.fa_seq;   
      DELETE sim_sample
        WHERE ss_fa_seq = v_field_activity.fa_seq;   
      DELETE sim_field_activity_parent
        WHERE fap_fa_seq = v_field_activity.fa_seq
          OR fap_parent_fa_seq = v_field_activity.fa_seq;   
      -- Insert record into sim_export_storet_ids if called during migration
      IF p_migrate = TRUE THEN
        IF v_field_activity.fa_type_name = 'Trip QC' THEN
          sim_cleanup_post_export(
            v_field_activity.fa_export_seq,
            'TSRFQS',
            v_field_activity.fa_tsrfqs_org_id,
            v_field_activity.fa_tsrfqs_is_number);
        ELSE
          sim_cleanup_post_export(
            v_field_activity.fa_export_seq,
            'TSRFDACT',
            v_field_activity.fa_tsrfdact_org_id,
            v_field_activity.fa_tsrfdact_is_number);
        END IF;
      END IF;
      DELETE sim_field_activities
        WHERE fa_seq = v_field_activity.fa_seq;   
      COMMIT;
    
      /* see if visit created by SIM, if so, see if any other activities
      are on the visit.  If there aren't and the visit is not still in STORET, delete
      the visit from the SIM_STATION_VISITS table. */
      /* This cursor looks only for visits that are both in SIM and STORET */
      OPEN c_get_visit_info(v_field_activity.fa_ssv_seq);
      FETCH c_get_visit_info
        INTO v_dummy_seq;
      IF c_get_visit_info%NOTFOUND THEN
        /* visit created by SIM and not associated with a visit in STORET,
        see if there are any activities associated with it. */
        OPEN c_check_visit_in_sim(v_field_activity.fa_ssv_seq);
        FETCH c_check_visit_in_sim
          INTO v_dummy_seq;
        IF c_check_visit_in_sim%NOTFOUND THEN
          /* no activities associated with this visit, delete it. */
          /*sjj - If p_migrate is true then insert a record into the sim_export_storet_ids
            table then delete visit*/
           DELETE sim_station_visits
              WHERE ssv_seq = v_field_activity.fa_ssv_seq;
          /* see if trip can be deleted.  Only do this when the station
          visit is deleted since we know the possibility exists that there is nothing
          left on the trip. */
          OPEN c_get_trip_info(v_field_activity.fa_stp_seq);
          FETCH c_get_trip_info
            INTO v_dummy_seq;
          IF c_get_trip_info%NOTFOUND THEN
            /* trip created by SIM and not associated with a trip in STORET,
            see if there are any other station visits associated with it and
            delete if not */
            OPEN c_check_trip_in_sim(v_field_activity.fa_stp_seq);
            FETCH c_check_trip_in_sim
              INTO v_dummy_seq;
            IF c_check_trip_in_sim%NOTFOUND THEN
              /* no station visits associated with the trip, delete.
              sf - there could still be other visits on this trip, though, so
              don't raise an error if it won't go */
              BEGIN
                DELETE sim_trips
                   WHERE stp_seq = v_field_activity.fa_stp_seq;                
              EXCEPTION
                WHEN OTHERS THEN
                  NULL;
              END;
            END IF;
            CLOSE c_check_trip_in_sim;
          END IF;
          CLOSE c_get_trip_info;
        END IF;
        CLOSE c_check_visit_in_sim;
      END IF;
      CLOSE c_get_visit_info;
      COMMIT;
    END LOOP;
    
    COMMIT;
    OPEN c_get_all_trips;
    LOOP
      FETCH c_get_all_trips
        INTO v_trip_seq;
      IF c_get_all_trips%FOUND THEN
        OPEN c_get_all_visits(v_trip_seq);
        LOOP
          FETCH c_get_all_visits
            INTO v_visit_seq;
          IF c_get_all_visits%FOUND THEN
            OPEN c_get_all_activities(v_visit_seq, v_trip_seq);
            FETCH c_get_all_activities
              INTO v_dummy_seq;
            IF c_get_all_activities%NOTFOUND THEN
              --There are no activities on this visit, so delete
              DELETE sim_station_visits
                WHERE ssv_seq = v_visit_seq;     
            END IF;  
            CLOSE c_get_all_activities;
          ELSE
            EXIT;
          END IF;
        END LOOP;
        CLOSE c_get_all_visits;
        BEGIN 
          DELETE sim_trips
            WHERE stp_seq = v_trip_seq;
        EXCEPTION
          WHEN OTHERS THEN
            NULL;
            --Do nothing. Can't delete this trip cuz there are visits on it still.
        END;
      ELSE
        EXIT;
      END IF;
    END LOOP;
    CLOSE c_get_all_trips;
    COMMIT;
  END IF;
  /* Clean up any Stations that were created when importing this file. */
  FOR v_result_stations IN c_result_stations LOOP
    DELETE sim_station_wells
      WHERE ssw_sta_seq = v_result_stations.sta_seq;
    DELETE sim_station_locations
      WHERE stl_sta_seq = v_result_stations.sta_seq;
    DELETE sim_stations
      WHERE sta_seq = v_result_stations.sta_seq;
  END LOOP;
  
  
  -- clean out import lines table
  LOOP
    OPEN c_check_import_line(p_import_seq);
    FETCH c_check_import_line
      INTO v_dummy_seq;
    IF c_check_import_line%NOTFOUND THEN
      CLOSE c_check_import_line;
      EXIT;
    ELSE
      DELETE sim_imp_lines
        WHERE sipl_sil_seq = p_import_seq
          AND rownum < 100;
      COMMIT;
      CLOSE c_check_import_line;
    END IF;
  END LOOP;
  DELETE sim_imp_lines
    WHERE sipl_sil_seq = p_import_seq;
  /* remove from import log */
  DELETE sim_import_log
    WHERE sil_seq = p_import_seq;                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       
  IF p_new_event_yn = 'Y' THEN  
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Finished: ' ||
                                         to_char(SYSDATE, 'HH24:MI:SS'));
    sim_event_log_pkg.update_stop_time;
  END IF;
  COMMIT;
END delete_import;

/****************************************************************************************
  Remove all SIM Import data for a particular Org.
 ****************************************************************************************/
PROCEDURE delete_import_all
(
  p_org_id           IN sim_import_log.sil_tsmorgan_org_id%TYPE,
  p_include_projects IN BOOLEAN := FALSE
) IS

  v_success BOOLEAN := TRUE;

BEGIN
  FOR rec IN (SELECT sil.sil_seq,
                     sil.sil_import_type
                FROM sim_import_log sil
                WHERE sil.sil_tsmorgan_org_id = p_org_id
                ORDER BY sil.sil_import_on DESC) LOOP
    IF rec.sil_import_type <> 'P'
       OR p_include_projects = TRUE THEN
      BEGIN
        delete_import(rec.sil_seq);
      EXCEPTION
        WHEN OTHERS THEN
          v_success := FALSE;
      END;
    END IF;
  END LOOP;
  IF NOT v_success THEN
    raise_application_error(-20000, 'Failed to delete all SIM Import Data');
  END IF;
END delete_import_all;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_import_log_seq RETURN NUMBER IS
  CURSOR c_import_log_seq IS
    SELECT sim_import_log_seq.NEXTVAL
      FROM dual;

  v_import_log_seq NUMBER;

BEGIN
  /* GET sequence for import log */
  OPEN c_import_log_seq;

  FETCH c_import_log_seq
    INTO v_import_log_seq;

  CLOSE c_import_log_seq;

  RETURN v_import_log_seq;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM ||
                                          ' error in sim_import_pkg.get_import_log_seq');
     
       
END get_import_log_seq;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE update_sil_rows_ready
(
  p_import_log_seq    IN NUMBER,
  p_rows_ready_change IN NUMBER
) IS

BEGIN
  UPDATE sim_import_log
    SET sil_rows_ready = nvl(sil_rows_ready, 0) + nvl(p_rows_ready_change, 0)
    WHERE sil_seq = p_import_log_seq;
  COMMIT;
END update_sil_rows_ready;

/****************************************************************************************
  Update the import log with the # of lines imported and the # of lines with errors
 ****************************************************************************************/ 
PROCEDURE update_sil_rows_errors (
  p_import_log_seq IN NUMBER,
  p_rows_errors    IN NUMBER) IS
  
  v_line_cnt  NUMBER(10);
  
BEGIN
  SELECT COUNT(*)
    INTO v_line_cnt
    FROM sim_imp_lines
    WHERE sipl_sil_seq = p_import_log_seq;
  UPDATE sim_import_log
    SET sil_rows_errors = nvl(p_rows_errors, 0),
        sil_rows_imported = v_line_cnt
    WHERE sil_seq = p_import_log_seq;
  COMMIT;
END update_sil_rows_errors;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION ind_nbr_sort(p_seq NUMBER) RETURN VARCHAR2 IS

  v_indv_nbr VARCHAR2(10);
  v_seq      NUMBER(10) := p_seq;

  CURSOR c_brgi(p_seq NUMBER) IS
    SELECT brgi_individual_number
      FROM sim_bio_results_grp_individual
     WHERE brgi_seq = p_seq;

BEGIN
  OPEN c_brgi(v_seq);

  FETCH c_brgi
    INTO v_indv_nbr;

  CLOSE c_brgi;

  RETURN(v_indv_nbr);
END ind_nbr_sort;

/****************************************************************************************
 Check to make sure the document/image file referenced in import exists and can be opened
 ****************************************************************************************/ 
FUNCTION check_document_file
(
  p_file_name      IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_line           IN NUMBER
) RETURN BOOLEAN IS

  v_file_type   utl_file.file_type;

BEGIN
  -- Check minimum length
  IF length(p_file_name) <= 4 THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || p_line,
                                         'Document/Graphic file name "' ||
                                          p_file_name || '" is invalid.');       
    RETURN FALSE;
  END IF;
  -- Check for supported file type (PDF, JPG, BMP, GIF, or TXT)
  IF upper(substr(p_file_name, length(p_file_name) - 3)) NOT IN
     ('.PDF', '.JPG', '.BMP', '.GIF', '.TXT') THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || p_line,
                                         'Line ' || p_line ||
                                         ': Document/Graphic "' || p_file_name ||
                                         '" file type is not PDF, JPG, BMP, GIF, or TXT.');
    RETURN FALSE;
  END IF;
  -- Check for valid file
  BEGIN
    v_file_type := utl_file.fopen(sim_lib_pkg.get_document_import_path, p_file_name, 'r');
    utl_file.fclose(v_file_type);

  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || p_line,
                                           'Line ' || p_line ||
                                           ': Document/Graphic file name "' ||
                                           p_file_name || '" is invalid. Verify that a file by this name exists in "' || sim_lib_pkg.get_document_import_path || '" on the database server.');
      RETURN FALSE;
  END;

  RETURN TRUE;

END check_document_file;    

  /****************************************************************************************
   The get_latitude and get_longitude functions are used by stations, wells, and locations 
   imports to interpret the values sent in to the Latitude and Longitude fields 
   respectively. Several assumptions are made and they are noted in the code.                                                                                      *
  ****************************************************************************************/
  FUNCTION get_latitude
  (
    v_import_log_seq  IN NUMBER,
    v_item_in         IN VARCHAR2,
    v_column_seq      IN NUMBER,
    p_imp_cfg_seq     IN NUMBER,
    v_sta_seq         IN NUMBER,
    v_line_cnt        IN NUMBER,
    p_lat_degrees     OUT NUMBER,
    p_lat_minutes     OUT NUMBER,
    p_lat_seconds     OUT NUMBER,
    p_lat_dir         OUT VARCHAR2,
    p_lat_dec_min     OUT NUMBER,
    p_lat_dec_deg     OUT NUMBER
  ) RETURN NUMBER IS
  
    v_lat_format      VARCHAR2(15);
    v_latitude        VARCHAR2(20);
    whatsbad          VARCHAR2(15);
    v_item            VARCHAR2(60);
    v_temp_min        NUMBER(10, 7);
  
    e_bad_format EXCEPTION;
    e_bad_number EXCEPTION;
  
    CURSOR c_get_format IS
      SELECT sifc_fmt_choice
        FROM sim_imp_fmt_choices c,
             sim_imp_cfg_dets d
       WHERE d.sicdt_sic_seq = v_column_seq
         AND d.sicdt_sicc_seq = p_imp_cfg_seq
         AND d.sicdt_sifc_seq = c.sifc_seq;
  
  BEGIN
    v_item := v_item_in;
    OPEN c_get_format;
    FETCH c_get_format
      INTO v_lat_format;
    IF c_get_format%NOTFOUND THEN
      --require format
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           'Line ' || v_line_cnt ||
                                           ': Please choose a format for Latitude.');
       
         
    END IF;
    CLOSE c_get_format;
  
    IF v_lat_format IN ('DD-MM-SS.SSSS', 'DD MM SS.SSSS') THEN
      v_item := REPLACE(v_item, ' ', '-');
      IF substr(v_item, 3, 1) != '-'
         OR substr(v_item, 6, 1) != '-' THEN
        IF substr(v_item, 2, 1) = '-'
           AND substr(v_item, 5, 1) = '-' THEN
          v_item := '0' || v_item;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
    
      p_lat_degrees := to_number(substr(v_item, 1, 2));
      p_lat_minutes := nvl(to_number(substr(v_item, 4, 2)), 0);
      p_lat_seconds := nvl(to_number(substr(v_item, 7, 7)), 0);
      v_temp_min    := p_lat_minutes + (p_lat_seconds / 60);
      p_lat_dec_min := round(v_temp_min, 4);
      p_lat_dec_deg := trunc(p_lat_degrees + (v_temp_min / 60), 7);
      p_lat_dir := NULL;
    
    ELSIF v_lat_format IN ('DD-MM.MMMM', 'DD MM.MMMM') THEN
      v_item := REPLACE(v_item, ' ', '-');
      IF substr(v_item, 3, 1) != '-' THEN
        IF substr(v_item, 2, 1) = '-' THEN
          v_item := '0' || v_item;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
    
      p_lat_degrees := to_number(substr(v_item, 1, 2));
      p_lat_minutes := nvl(to_number(substr(v_item, 4, 2)), 0);
      p_lat_dec_min := trunc(nvl(to_number(substr(v_item, 4, 7)), 0), 4);
      p_lat_seconds := 60 * (p_lat_dec_min - p_lat_minutes);
      p_lat_dec_deg := trunc(p_lat_degrees + (p_lat_dec_min / 60), 7);
      p_lat_dir := NULL;
    
    ELSIF v_lat_format = 'DD.DDDDDDD' THEN
      IF substr(v_item, 3, 1) != '.' THEN
        IF substr(v_item, 2, 1) = '.' THEN
          v_item := '0' || v_item;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
    
      p_lat_dec_deg := trunc(to_number(v_item), 7);
      p_lat_degrees := to_number(substr(v_item, 1, 2));
      v_temp_min    := 60 * (p_lat_dec_deg - p_lat_degrees);
      p_lat_dec_min := trunc(v_temp_min, 4);
      p_lat_minutes := floor(p_lat_dec_min);
      p_lat_seconds := trunc(60 * (v_temp_min - p_lat_minutes), 4);
      p_lat_dir := NULL;
    
    ELSIF v_lat_format = '-DD.DDDDDDD' THEN
      -- start by making sure there is a sign on the front
      IF substr(v_item, 1, 1) <> '+'
         AND substr(v_item, 1, 1) <> '-' THEN
        -- assume - (Northern hemisphere)
        v_latitude := '+' || v_item;
      ELSE
        v_latitude := v_item;
      END IF;
    
      IF length(v_latitude) > 3 THEN
        IF substr(v_latitude, 4, 1) <> '.' THEN
          RAISE e_bad_format;
        END IF;
      END IF;
    
      IF substr(v_latitude, 1, 1) = '-' THEN
        p_lat_dir := 'S';
      ELSE
        p_lat_dir := 'N';
      END IF;
    
      p_lat_dec_deg := trunc(to_number(v_latitude), 7);
      p_lat_degrees := abs(to_number(substr(v_latitude, 2, 2)));
      v_temp_min    := 60 * (abs(p_lat_dec_deg) - p_lat_degrees);
      p_lat_dec_min := trunc(v_temp_min, 4);
      p_lat_minutes := floor(p_lat_dec_min);
      p_lat_seconds := trunc(60 * (v_temp_min - p_lat_minutes), 4);
    
    ELSE
      -- we haven't coded for this format (it's not supported yet)
      RAISE e_bad_format;    
      
    END IF;
  
    --Check to see if any of the values are out of reasonable range.
    IF p_lat_degrees < 0
       OR p_lat_degrees >= 90 THEN
      whatsbad := 'Lat Degrees';
      RAISE e_bad_number;
    ELSIF p_lat_minutes < 0
          OR p_lat_minutes >= 60 THEN
      whatsbad := 'Lat Minutes';
      RAISE e_bad_number;
    ELSIF p_lat_seconds < 0
          OR p_lat_seconds >= 60 THEN
      whatsbad := 'Lat Seconds';
      RAISE e_bad_number;
    END IF;
  
    RETURN 1;
  
  EXCEPTION
    WHEN e_bad_format THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           SQLERRM || ' Line ' || v_line_cnt ||
                                           ': Latitude does not match chosen format.');
       
         
      RETURN 2;
    WHEN e_bad_number THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           SQLERRM || ' Line ' || v_line_cnt || ': ' ||
                                           whatsbad || ' is out of range.');
       
         
      RETURN 2;
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           SQLERRM || ' Line ' || v_line_cnt ||
                                           ': Latitude does not match chosen format.');
       
         
      RETURN 2;
  END get_latitude;

  /****************************************************************************************
  
  ****************************************************************************************/
  FUNCTION get_longitude
  (
    v_import_log_seq  IN NUMBER,
    v_item_in         IN VARCHAR2,
    v_column_seq      IN NUMBER,
    p_imp_cfg_seq     IN NUMBER,
    v_sta_seq         IN NUMBER,
    v_line_cnt        IN NUMBER,
    p_long_degrees    OUT NUMBER,
    p_long_minutes    OUT NUMBER,
    p_long_seconds    OUT NUMBER,
    p_long_dec_min    OUT NUMBER,
    p_long_dec_deg    OUT NUMBER,
    p_long_dir        OUT VARCHAR2
  ) RETURN NUMBER IS
  
    v_long_format     VARCHAR2(15);
    v_longitude       VARCHAR2(20);
    whatsbad          VARCHAR2(15);
    v_item            VARCHAR2(60);
    v_temp_min        NUMBER(10, 7);
    
    e_bad_format EXCEPTION;
    e_bad_number EXCEPTION;
  
    CURSOR c_get_format IS
      SELECT sifc_fmt_choice
        FROM sim_imp_fmt_choices c,
             sim_imp_cfg_dets d
       WHERE d.sicdt_sic_seq = v_column_seq
         AND d.sicdt_sicc_seq = p_imp_cfg_seq
         AND d.sicdt_sifc_seq = c.sifc_seq;
  
  BEGIN
    v_item := v_item_in;
    OPEN c_get_format;
    FETCH c_get_format
      INTO v_long_format;
    IF c_get_format%NOTFOUND THEN
      --require a format
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           'Line ' || v_line_cnt ||
                                            '" Please choose a format for Longitude.');
       
         
      RETURN 2;
    END IF;
    CLOSE c_get_format;
  
    IF v_long_format IN ('DDD-MM-SS.SSSS', 'DDD MM SS.SSSS') THEN
      v_item := REPLACE(v_item, ' ', '-');
      IF substr(v_item, 4, 1) != '-'
         OR substr(v_item, 7, 1) != '-' THEN
        IF substr(v_item, 3, 1) = '-'
           AND substr(v_item, 6, 1) = '-' THEN
          v_item := '0' || v_item;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
      p_long_degrees := to_number(substr(v_item, 1, 3));
      p_long_minutes := nvl(to_number(substr(v_item, 5, 2)), 0);
      p_long_seconds := nvl(to_number(substr(v_item, 8, 7)), 0);
      v_temp_min     := p_long_minutes + (p_long_seconds / 60);
      p_long_dec_min := trunc(v_temp_min, 4);
      p_long_dec_deg := trunc(p_long_degrees + (v_temp_min / 60), 7);
      p_long_dir := NULL;
   
    ELSIF v_long_format IN ('DDD-MM.MMMM', 'DDD MM.MMMM') THEN
      v_item := REPLACE(v_item, ' ', '-');
      IF substr(v_item, 4, 1) != '-' THEN
        IF substr(v_item, 3, 1) = '-' THEN
          v_item := '0' || v_item;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
      p_long_degrees := to_number(substr(v_item, 1, 3));
      p_long_minutes := nvl(to_number(substr(v_item, 5, 2)), 0);
      p_long_dec_min := nvl(to_number(substr(v_item, 5, 7)), 0);
      p_long_seconds := 60 * (p_long_dec_min - p_long_minutes);
      p_long_dec_deg := trunc(p_long_degrees + (p_long_dec_min / 60), 7);
      p_long_dir := NULL;
    
    ELSIF v_long_format = 'DDD.DDDDDDD' THEN
      IF substr(v_item, 4, 1) != '.' THEN
        IF substr(v_item, 3, 1) = '.' THEN
          v_item := '0' || v_item;
        ELSE
          RAISE e_bad_format;
        END IF;
      END IF;
      p_long_dec_deg := trunc(to_number(v_item), 7);
      p_long_degrees := to_number(substr(v_item, 1, 3));
      v_temp_min     := 60 * (p_long_dec_deg - p_long_degrees);
      p_long_dec_min := trunc(v_temp_min, 4);
      p_long_minutes := floor(p_long_dec_min);
      p_long_seconds := trunc(60 * (v_temp_min - p_long_minutes), 4);
      p_long_dir := NULL;
    
    ELSIF v_long_format = '-DDD.DDDDDDD' THEN
      --start by making sure there is a sign on the front
      IF substr(v_item, 1, 1) <> '+'
         AND substr(v_item, 1, 1) <> '-' THEN
        --assume + (Eastern hemisphere)
        v_longitude := '+' || v_item;
      ELSE
        v_longitude := v_item;
      END IF;
    
      -- Pad longitude with 0 for 2 digit longitudes
      IF length(v_longitude) > 4 THEN
        IF substr(v_longitude, 5, 1) <> '.' THEN
          IF substr(v_longitude, 4, 1) = '.' THEN
            v_longitude := substr(v_longitude, 1, 1) || '0' ||
                           substr(v_longitude, 2, length(v_longitude));
          ELSE
            RAISE e_bad_format;
          END IF;
        END IF;
      ELSE
        IF length(v_longitude) <> 4 THEN
          IF length(v_longitude) = 3 THEN
            v_longitude := substr(v_longitude, 1, 1) || '0' ||
                           substr(v_longitude, 2, length(v_longitude));
          ELSE
            RAISE e_bad_format;
          END IF;
        END IF;
      END IF;
    
      IF substr(v_longitude, 1, 1) = '+' THEN
        p_long_dir := 'E';
      ELSE
        p_long_dir := 'W';
      END IF;
    
      p_long_dec_deg := trunc(to_number(v_longitude), 7);
      p_long_degrees := abs(to_number(substr(v_longitude, 2, 3)));
      v_temp_min     := 60 * (abs(p_long_dec_deg) - p_long_degrees);
      p_long_dec_min := trunc(v_temp_min, 4);
      p_long_minutes := floor(p_long_dec_min);
      p_long_seconds := trunc(60 * (v_temp_min - p_long_minutes), 4);

    ELSE
      -- we haven't coded for this format (it's not supported yet)
      RAISE e_bad_format;      
      
    END IF;
  
    --Check to see if any of the values are out of reasonable range.
    IF p_long_degrees < 0
       OR p_long_degrees >= 180 THEN
      whatsbad := 'Long Degrees';
      RAISE e_bad_number;
    ELSIF p_long_minutes < 0
          OR p_long_minutes >= 60 THEN
      whatsbad := 'Long Minutes';
      RAISE e_bad_number;
    ELSIF p_long_seconds < 0
          OR p_long_seconds >= 60 THEN
      whatsbad := 'Long Seconds';
      RAISE e_bad_number;
    END IF;
  
    RETURN 1;
  
  EXCEPTION
    WHEN e_bad_format THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           SQLERRM || ' Line ' || v_line_cnt ||
                                           ': Longitude does not match chosen format.');
       
         
      RETURN 2;
    WHEN e_bad_number THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           SQLERRM || ' Line ' || v_line_cnt || ': ' ||
                                           whatsbad || ' is out of range.');
       
         
      RETURN 2;
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line ' || v_line_cnt,
                                           SQLERRM || ' Line ' || v_line_cnt ||
                                           ': Longitude does not match chosen format.');
       
         
      RETURN 2;
  END get_longitude;
  
/****************************************************************************************

****************************************************************************************/  
  PROCEDURE sim_cleanup_post_export (
    p_export_log_seq  IN  NUMBER,
    p_table_name      IN  VARCHAR2,
    p_org_id          IN  VARCHAR2,
    p_is_number       IN  NUMBER) IS
    
  BEGIN
    INSERT INTO sim_export_storet_ids (
      sesi_seq,
      sesi_sel_seq,
      sesi_table_name,
      sesi_org_id,
      sesi_is_number)
    VALUES (
      sim_export_storet_ids_seq.NEXTVAL,
      p_export_log_seq,
      p_table_name,
      p_org_id,
      p_is_number);
      
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
      NULL;
      
  END sim_cleanup_post_export;
  
/****************************************************************************************
  Check whether a value is one of the allowable values for a column
  and return the allowable value ID
 ****************************************************************************************/
FUNCTION check_allowable_values(
  p_item           IN VARCHAR2, 
  p_sic_seq        IN NUMBER, 
  v_line_cnt       IN NUMBER) 
  
  RETURN NUMBER IS
  
  v_sia_seq NUMBER(10);

  CURSOR c_check_allowable_val IS
    SELECT sia_seq
      FROM sim_imp_allow_values
     WHERE sia_sic_seq = p_sic_seq
       AND sia_item = trim(p_item);

  BEGIN
    OPEN c_check_allowable_val;
    FETCH c_check_allowable_val
      INTO v_sia_seq;
    CLOSE c_check_allowable_val;
    RETURN v_sia_seq;
  
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 
        'Line ' || v_line_cnt, SQLERRM || ' - in check allowable values');
      RETURN NULL;
    
  END check_allowable_values;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION check_translations
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_line_cnt    IN NUMBER,
  p_trans_exist OUT BOOLEAN
) RETURN VARCHAR2 IS

  v_storet_value VARCHAR2(4000);
  v_proj_seq     NUMBER(10);

  CURSOR c_get_project_seq IS
    SELECT sim_projects_seq.CURRVAL
      FROM dual;

  CURSOR c_get_storet_code(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT st_storet_code
      FROM sim_translations,
           sim_imp_cfg_dets
     WHERE st_user_code = p_code
       AND st_sicdt_seq = sicdt_seq
       AND sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

BEGIN

  /* check translations table to see if the code should be swapped with another value */
  OPEN c_get_storet_code(p_code, p_imp_cfg_seq, p_col_seq);
  FETCH c_get_storet_code
    INTO v_storet_value;
  IF c_get_storet_code%NOTFOUND THEN
  
    /* if not in translations table return original value - will check elsewhere for sequences */
    v_storet_value := p_code;
    p_trans_exist := FALSE;
  
  ELSIF c_get_storet_code%FOUND THEN
    /* if the code is in the translations table, flag so it's not recreated */
    p_trans_exist := TRUE;
    IF v_storet_value IS NULL THEN
      /* if there is a code in the translations table but no value to translate to, write error message
      status will be changed to 'N' later in the program */
      OPEN c_get_project_seq;
      FETCH c_get_project_seq
        INTO v_proj_seq;
      CLOSE c_get_project_seq;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                           p_code ||
                                           ' exists in translations table but has not been mapped to a STORET code.');
       
   
    END IF;
  
  END IF;
  CLOSE c_get_storet_code;
  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM || ' - in check translations');
     
END check_translations;

/****************************************************************************************
  Get the distinct count of lines in the import file that had errors
 ****************************************************************************************/ 
FUNCTION get_rows_with_errors(
  p_sil_seq IN sim_event_log.sevl_ref_seq%TYPE) 
  
  RETURN NUMBER IS

  v_error_count  NUMBER(10);
  
BEGIN
  SELECT COUNT(*)
    INTO v_error_count
    FROM (SELECT 1
            FROM sim_event_log sevl,
                 sim_event_log_details seld
           WHERE seld.seld_sevl_seq = sevl.sevl_seq
             AND sevl.sevl_ref_seq = p_sil_seq
             AND sevl.sevl_selt_seq = sim_lib_pkg.c_selt_import_file
             AND seld.seld_seldt_seq = sim_lib_pkg.c_seldt_error
           GROUP BY seld.seld_context);
  RETURN v_error_count;
END get_rows_with_errors;



END sim_import_pkg;
/
