CREATE OR REPLACE PACKAGE sim_import_project_pkg IS
  /* Modified 3/20/02 by SF - Added an exception to deal with  bad dates in the data file */

  -- Author  : KHENAS
  -- Created : 2/5/01 10:37:43 AM
  -- Purpose : import STORET projects

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                               *
 ****************************************************************************************/
  TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;

  TYPE col_table_num_type IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;

  TYPE t_storet_pk_type IS RECORD(
    is_number NUMBER(8),
    org_id    VARCHAR2(8));

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_org_is_nbr     NUMBER(8) := 0;
  v_import_log_seq NUMBER(10);
  v_line_cnt       NUMBER(10) := 0;

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                           *
 ****************************************************************************************/  

  FUNCTION get_org_info
  (
    p_imp_cfg_seq IN NUMBER,
    p_org_id      IN VARCHAR2,
    p_org_is_nbr  IN OUT NUMBER
  ) RETURN NUMBER;

  FUNCTION get_delimiter
  (
    p_imp_cfg_seq IN NUMBER,
    p_delimiter   IN OUT VARCHAR2
  ) RETURN NUMBER;

  FUNCTION get_item_pos
  (
    p_sicc_seq      IN NUMBER,
    p_pos_table     IN OUT col_table_num_type,
    p_default_table IN OUT col_table_type,
    p_col_cnt       IN OUT NUMBER
  ) RETURN NUMBER;

  FUNCTION get_column_seq
  (
    p_cfg_seq IN NUMBER,
    p_cfg_pos IN NUMBER
  ) RETURN NUMBER;

  PROCEDURE create_trans_record
  (
    p_code        IN VARCHAR2,
    p_imp_cfg_seq IN NUMBER,
    p_col_seq     IN NUMBER,
    p_proj_seq    IN NUMBER
  );

  FUNCTION validate_alval_item
  (
    p_item_name   IN VARCHAR2,
    p_orig_item   IN VARCHAR2,
    p_imp_cfg_seq IN NUMBER,
    p_col_seq     IN NUMBER,
    p_proj_seq    IN NUMBER
  ) RETURN NUMBER;

  FUNCTION validate_storet_item
  (
    p_item_name   IN VARCHAR2,
    p_orig_item   IN VARCHAR2,
    p_imp_cfg_seq IN NUMBER,
    p_col_seq     IN NUMBER,
    p_req         IN BOOLEAN,
    p_org_id      IN VARCHAR2,
    p_line_nbr    IN NUMBER,
    p_proj_seq    IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION write_error_message
  (
    p_item_name IN VARCHAR2,
    p_req       IN BOOLEAN,
    p_line_nbr  IN NUMBER,
    p_provided  IN BOOLEAN,
    p_status    IN OUT VARCHAR2,
    p_comment   IN OUT VARCHAR2,
    p_proj_seq  IN NUMBER
  ) RETURN NUMBER;

  FUNCTION get_date
  (
    p_column_name IN VARCHAR2,
    p_raw_date    IN VARCHAR2,
    p_col_seq     IN NUMBER,
    p_cfg_seq     IN NUMBER,
    p_proj_seq    IN NUMBER
  ) RETURN DATE;

  PROCEDURE import_projects
  (
    p_org_id         IN VARCHAR2,
    p_import_log_seq IN NUMBER,
    p_imp_cfg_seq    IN NUMBER,
    p_test_count     IN NUMBER,
    p_skip_header    IN NUMBER DEFAULT 0       -- CLM 08/02/2005 Skip header option    
    );                                         --   1 = Skip  0 = Don't Skip

END sim_import_project_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_import_project_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                             *
 ****************************************************************************************/
 
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  e_a EXCEPTION;
  
/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_org_info
(
  p_imp_cfg_seq IN NUMBER,
  p_org_id      IN VARCHAR2,
  p_org_is_nbr  IN OUT NUMBER
) RETURN NUMBER IS

  CURSOR c_get_tsmorgan_nbr(p_cursor_org_id IN VARCHAR2) IS
    SELECT tsmorgan_is_number
      FROM tsmorgan
     WHERE org_id = rpad(p_cursor_org_id, 8);

BEGIN

  OPEN c_get_tsmorgan_nbr(p_org_id);
  FETCH c_get_tsmorgan_nbr
    INTO p_org_is_nbr;
  IF c_get_tsmorgan_nbr%NOTFOUND THEN
    p_org_is_nbr := NULL;
  END IF;
  CLOSE c_get_tsmorgan_nbr;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    raise_application_error(-20199, 'Failed to get organization is number.');
    RETURN 0;
  
END get_org_info;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_delimiter
(
  p_imp_cfg_seq IN NUMBER,
  p_delimiter   IN OUT VARCHAR2
) RETURN NUMBER IS

  e_no_cfg EXCEPTION;

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT sicc_delimiter
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

BEGIN

  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg
    INTO p_delimiter;
  IF c_get_imp_cfg%NOTFOUND THEN
    RAISE e_no_cfg;
  END IF;
  CLOSE c_get_imp_cfg;

  RETURN 1;

EXCEPTION

  WHEN e_no_cfg THEN
    raise_application_error(-20102, 'Configuration invalid or does not exist.');
  
  WHEN OTHERS THEN
    raise_application_error(-20198, 'Failed to get delimiter.');
    RETURN 0;
  
END get_delimiter;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_item_pos
(
  p_sicc_seq      IN NUMBER,
  p_pos_table     IN OUT col_table_num_type,
  p_default_table IN OUT col_table_type,
  p_col_cnt       IN OUT NUMBER
) RETURN NUMBER IS

  v_import_type_seq NUMBER(10);

  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT sicc_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

  CURSOR c_get_dflt_pos(p_type_seq IN NUMBER) IS
    SELECT sic_dflt_pos
      FROM sim_imp_cols
     WHERE sic_sid_seq = p_type_seq;

  CURSOR c_get_col_pos(p_sicc_seq IN NUMBER, p_dflt_pos IN NUMBER) IS
    SELECT sicdt_pos,
           sicdt_default
      FROM sim_imp_cfg_dets,
           sim_imp_cols
     WHERE sic_seq = sicdt_sic_seq
       AND sicdt_sicc_seq = p_sicc_seq
       AND sic_dflt_pos = p_dflt_pos
       AND (sicdt_included IS NOT NULL OR sicdt_generate IS NOT NULL);

BEGIN

  /* get the import type in order to limit the columns selected to the right type */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type
    INTO v_import_type_seq;
  IF c_get_import_type%NOTFOUND THEN
    v_import_type_seq := NULL;
    raise_application_error(-20102, 'Unable to get import type from import configuration.');
  END IF;
  CLOSE c_get_import_type;

  p_col_cnt := 0;

  /* loop through import columns and assign actual position to table indexed by
  default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP
  
    /* get actual position and item default from configuration table */
    OPEN c_get_col_pos(p_sicc_seq, v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos
      INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
    IF c_get_col_pos%NOTFOUND THEN
      /* if there is nothing in the configuration table for this column, position and default will be null */
      p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
      p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
    ELSE
      p_col_cnt := p_col_cnt + 1;
    END IF;
    CLOSE c_get_col_pos;
  
  END LOOP;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    raise_application_error(-20103, 'Failed to get column positions, item names, and column sequences.');
  
END get_item_pos;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_column_seq
(
  p_cfg_seq IN NUMBER,
  p_cfg_pos IN NUMBER
) RETURN NUMBER IS

  v_imp_col_seq NUMBER(10);
  e_no_cfg_found EXCEPTION;

  CURSOR c_get_col_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) IS
    SELECT sicdt_sic_seq
      FROM sim_imp_cfg,
           sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = sicc_seq
       AND sicc_seq = p_cfg_seq
       AND sicdt_pos = p_cfg_pos;

BEGIN

  OPEN c_get_col_seq(p_cfg_seq, p_cfg_pos);
  FETCH c_get_col_seq
    INTO v_imp_col_seq;
  IF c_get_col_seq%NOTFOUND THEN
    v_imp_col_seq := NULL;
    RAISE e_no_cfg_found;
  END IF;
  CLOSE c_get_col_seq;

  RETURN v_imp_col_seq;

EXCEPTION
  WHEN e_no_cfg_found THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - unable to find sequence in sim_imp_cfg_dets for pos: ' ||
                                         p_cfg_pos);
     
 
    raise_application_error(-20100, 'Import configuration is not complete');
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                          ' - getting column sequence.');
     
 
    RETURN NULL;
  
END get_column_seq;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION check_sim_proj
(
  p_project_id   IN VARCHAR2,
  p_org_id       IN VARCHAR2,
  p_project_name IN VARCHAR2
) RETURN NUMBER IS

  v_proj_name_cnt NUMBER(10);
  v_proj_id_cnt   NUMBER(10);
  v_success       NUMBER(1) := 0;

  CURSOR c_project_id(p_id VARCHAR2, p_org_id VARCHAR2) IS
    SELECT COUNT(spj_seq)
      FROM sim_projects
     WHERE spj_id = p_id
       AND rtrim(spj_org_id) = rtrim(p_org_id);

  CURSOR c_project_name(p_name VARCHAR2, p_org_id VARCHAR2) IS
    SELECT COUNT(spj_seq)
      FROM sim_projects
     WHERE spj_name = p_name
       AND rtrim(spj_org_id) = rtrim(p_org_id);

BEGIN

  OPEN c_project_id(p_project_id, p_org_id);
  FETCH c_project_id
    INTO v_proj_id_cnt;
  CLOSE c_project_id;

  IF nvl(v_proj_id_cnt, 0) > 0 THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         '- Duplicate Project ID in SIM.');
     
 
    v_success := 1;
  END IF;

  OPEN c_project_name(p_project_name, p_org_id);
  FETCH c_project_name
    INTO v_proj_name_cnt;
  CLOSE c_project_name;

  IF nvl(v_proj_name_cnt, 0) > 0 THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         '- Duplicate Project Name in SIM.');
     
 
    v_success := 1;
  END IF;

  RETURN v_success;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in check_sim_proj.');
     
 
    v_success := 1;
    RETURN v_success;
END check_sim_proj;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION check_storet_proj
(
  p_project_id   IN VARCHAR2,
  p_org_id       IN VARCHAR2,
  p_project_name IN VARCHAR2,
  p_fa_seq       IN NUMBER
) RETURN t_storet_pk_type IS

  v_project_pk t_storet_pk_type;

  CURSOR c_check_storet_proj_id(p_project_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number,
           tsmproj_org_id
      FROM tsmproj
     WHERE identification_cd = p_project_id
       AND tsmproj_org_id = p_org_id;

  CURSOR c_check_storet_proj_name(p_project_name IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmproj_is_number,
           tsmproj_org_id
      FROM tsmproj
     WHERE NAME = p_project_name
       AND rtrim(tsmproj_org_id) = p_org_id;

BEGIN
  /* bka - function that looks for the project id in STORET to make sure it's not
  duplicating a pre-existing project.  If the project is already in STORET, SIM
  will import it but will prevent it from being exported. */

  OPEN c_check_storet_proj_name(p_project_name, p_org_id);
  FETCH c_check_storet_proj_name
    INTO v_project_pk;
  IF c_check_storet_proj_name%NOTFOUND THEN
    v_project_pk.is_number := NULL;
    v_project_pk.org_id := NULL;
  END IF;
  CLOSE c_check_storet_proj_name;

  IF v_project_pk.is_number IS NULL THEN
    OPEN c_check_storet_proj_id(p_project_id, p_org_id);
    FETCH c_check_storet_proj_id
      INTO v_project_pk;
    IF c_check_storet_proj_id%NOTFOUND THEN
      v_project_pk.is_number := NULL;
      v_project_pk.org_id := NULL;
    END IF;
    CLOSE c_check_storet_proj_id;
  END IF;

  RETURN v_project_pk;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in check activity id.');
     
 
    v_project_pk.is_number := NULL;
    v_project_pk.org_id := NULL;
    RETURN v_project_pk;
  
END check_storet_proj;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_instructions
(
  p_imp_cfg_seq      IN NUMBER,
  p_instruction_type IN VARCHAR2,
  p_proj_seq         IN NUMBER
) RETURN VARCHAR2 IS

  v_instr_choice VARCHAR2(15);

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_cfg_instrs,
           sim_imp_instr_choices,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND sici_sicc_seq = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN NUMBER) IS
    SELECT siic_code
      FROM sim_imp_instr_choices
     WHERE siic_sii_seq = p_instr_type
       AND siic_dflt_flag = 1;

BEGIN

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr
    INTO v_instr_choice;
  IF c_get_cfg_instr%NOTFOUND THEN
  
    /* no choice could be found, get default choice */
    OPEN c_get_instr_default(p_instruction_type);
    FETCH c_get_instr_default
      INTO v_instr_choice;
    CLOSE c_get_instr_default;
  
  END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get_instructions.');
     
 
    RETURN NULL;
  
END get_instructions;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_date
(
  p_column_name IN VARCHAR2,
  p_raw_date    IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_cfg_seq     IN NUMBER,
  p_proj_seq    IN NUMBER
) RETURN DATE IS

  v_start_date DATE;
  v_fmt        VARCHAR2(80);

  e_no_cfg_found EXCEPTION;

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;

BEGIN

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    v_fmt := NULL;
    /* no format specified, use default */
    BEGIN
      v_start_date := trunc(to_date(p_raw_date, 'MM-DD-RRRR'));
    EXCEPTION
      WHEN OTHERS THEN
         
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Date ' || p_raw_date ||
                                          ' is not in a valid date format. ');
     
 
    END;
  END IF;
  CLOSE c_get_date_fmt;

  BEGIN
    IF v_fmt IS NOT NULL THEN
      v_start_date := trunc(to_date(p_raw_date, v_fmt));
    
    END IF;
  EXCEPTION
    WHEN OTHERS THEN
      v_start_date := NULL;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'check date format in import configuration. ' ||
                                            p_column_name || ' ' || p_raw_date);
       
   
  END;

  RETURN v_start_date;

EXCEPTION

  WHEN e_no_cfg_found THEN
    raise_application_error(-20100, 'Import configuration for start or end date is incomplete or does not exist.');
    RETURN NULL;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get date, column ' ||
                                         p_column_name || ' ' || p_raw_date);
     
 
    RETURN NULL;
  
END get_date;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE create_trans_record
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_proj_seq    IN NUMBER
) IS

  v_st_seq    NUMBER(10);
  v_sicdt_seq NUMBER(10);

  e_imp_cfg_dets EXCEPTION;
  e_trans_error EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT sicdt_seq
      FROM sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL
      FROM dual;

BEGIN

  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq
    INTO v_sicdt_seq;
  IF c_get_cfg_det_seq%NOTFOUND THEN
    v_sicdt_seq := NULL;
    RAISE e_imp_cfg_dets;
  END IF;
  CLOSE c_get_cfg_det_seq;

  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq
    INTO v_st_seq;
  IF c_get_st_seq%NOTFOUND THEN
    v_st_seq := NULL;
    RAISE e_trans_error;
  END IF;
  CLOSE c_get_st_seq;

  INSERT INTO sim_translations
    (st_seq,
     st_sicdt_seq,
     st_user_code,
     st_storet_code)
  VALUES
    (v_st_seq, /* st_seq */
     v_sicdt_seq, /* st_sicdt_seq */
     p_code, /* st_user_code */
     NULL /* st_storet_code */);

   
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                      'Created translation record for ' ||
                                       p_code);
   

EXCEPTION

  WHEN e_imp_cfg_dets THEN
     
                                                      
   
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                      'Unable to find import configuration - stopped import');
   

    raise_application_error(-20101, 'Unable to find import configuration');
  
  WHEN e_trans_error THEN
     
  
                                                        
   
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                      'Unable to create new record in translations table - error getting sequence');
   
    
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        SQLERRM ||
                                        ' - adding record to translations table.');
     
      
END create_trans_record;


/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_alval_item
(
  p_item_name   IN VARCHAR2,
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_proj_seq    IN NUMBER
) RETURN NUMBER IS

  v_converted_item VARCHAR2(80);
  v_item_sia_seq   NUMBER(10);
  v_trans_exist    BOOLEAN := TRUE;
  v_instr_code     VARCHAR2(15);

BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
  hard coded values in STORET */

  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := sim_import_pkg.check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_line_cnt, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN
  
    /* look in allowable values */
    v_item_sia_seq := sim_import_pkg.check_allowable_values(v_converted_item, p_col_seq, v_line_cnt);
  
    IF v_item_sia_seq IS NULL
       AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'PRJ-TR', p_proj_seq);
    
      IF v_instr_code = 'PRJ-TR-01' THEN
      
        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_proj_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            p_item_name || ', ' ||
                                           v_converted_item ||
                                           ' is not an allowable value - translation record has been created.');
         
          
      ELSE
          
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not an allowable value.');
         
          
      END IF;
    
    ELSIF v_item_sia_seq IS NULL
          AND v_trans_exist THEN
    
      /* item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                          p_orig_item ||
                                         ' has been mapped to the invalid storet value: ' ||
                                         v_converted_item);
       
        
    END IF;
  END IF;
  RETURN v_item_sia_seq;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        SQLERRM ||
                                        ' - in validate allowable values.');
     
      
    RETURN NULL;
  
END validate_alval_item;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_storet_item
(
  p_item_name   IN VARCHAR2,
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_req         IN BOOLEAN,
  p_org_id      IN VARCHAR2,
  p_line_nbr    IN NUMBER,
  p_proj_seq    IN NUMBER
) RETURN t_storet_pk_type IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_pk      t_storet_pk_type;
  v_instr_code     VARCHAR2(8);

  e_no_function EXCEPTION;

BEGIN
  /* function to look in the translations and STORET tables for items that should already exist in
  hard coded values in STORET tables */

  /* look in translations table to see if it needs to swapped */
  v_converted_item := sim_import_pkg.check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_line_cnt, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN
  
    /* call appropriate function to look in STORET table */
    IF p_item_name = 'Contact' THEN
      NULL;
    END IF;
  
    IF v_storet_pk.is_number = -1 THEN
      RAISE e_no_function;
    END IF;
  
    IF v_storet_pk.is_number IS NULL
       AND NOT v_trans_exist THEN
    
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'PRJ-TR', p_proj_seq);
    
      IF v_instr_code = 'PRJ-TR-01' THEN
      
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_proj_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            p_item_name || ', ' ||
                                           v_converted_item ||
                                           ' is not an allowable value - translation record has been created.');
         
          
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_item_name || ', ' ||
                                           v_converted_item ||
                                           ' is not an allowable value.');
         
          
      END IF;
    
    ELSIF v_storet_pk.is_number IS NULL
          AND v_trans_exist THEN
    
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                          p_orig_item ||
                                         ' has been mapped to the invalid storet value: ' ||
                                         v_converted_item);
       
        
    END IF;
  
  ELSIF v_trans_exist THEN
  
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_pk.is_number := NULL;
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        p_orig_item ||
                                        ' exists in translation table but has not been mapped to a STORET code. ');
     
      
  END IF;

  RETURN v_storet_pk;

EXCEPTION

  WHEN e_no_function THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        'SIM2 does not currently handle ' ||
                                        p_item_name);
     
      
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
    RETURN v_storet_pk;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        SQLERRM ||
                                        ' - in validate storet item.');
     
      
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
    RETURN v_storet_pk;
  
END validate_storet_item;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION write_error_message
(
  p_item_name IN VARCHAR2,
  p_req       IN BOOLEAN,
  p_line_nbr  IN NUMBER,
  p_provided  IN BOOLEAN,
  p_status    IN OUT VARCHAR2,
  p_comment   IN OUT VARCHAR2,
  p_proj_seq  IN NUMBER
) RETURN NUMBER IS

BEGIN

  IF p_provided THEN
  
    p_status := 'N';
  
    p_comment := rtrim(p_item_name ||
                       ' could not be found or mapped to valid STORET value. ' ||
                       nvl(p_comment, ' '));
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        substr(initcap(p_item_name) ||
                                        ' could not be mapped to a valid STORET value', 1, 240));
     
      
  ELSE
  
    IF p_req THEN
      p_status := 'N';
      p_comment := rtrim(p_item_name || ' not included. ' || nvl(p_comment, ' '));
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                          substr('Line ' || p_line_nbr ||
                                          ': Required item ' ||
                                          p_item_name ||
                                          ' not provided. Project will not export.', 1, 240));
       
        
    ELSE
      p_status := p_status;
      p_comment := rtrim(p_item_name || ' not included. ' || nvl(p_comment, ' '));
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         substr('Line ' || p_line_nbr ||
                                          ': Included item ' ||
                                          p_item_name ||
                                          ' not provided.', 1, 240));
       
        
    END IF;
  
  END IF;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        SQLERRM ||
                                        ' - in write error message.');
     
      
    RETURN 0;
  
END write_error_message;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_project(p_project_rec sim_projects%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create project using values in project record */
  INSERT INTO sim_projects
    (spj_seq,
     spj_tsmorgan_is_number,
     spj_org_id,
     spj_id,
     spj_tsmproj_is_number,
     spj_tsmproj_org_id,
     spj_name,
     spj_start_date,
     spj_duration,
     spj_purpose,
     spj_contact,
     spj_document,
     spj_export_seq,
     spj_import_seq,
     spj_status)
  VALUES
    (p_project_rec.spj_seq, /* SPJ_SEQ */
     p_project_rec.spj_tsmorgan_is_number, /* SPJ_TSMORGAN_IS_NUMBER */
     p_project_rec.spj_org_id, /* SPJ_ORG_ID */
     p_project_rec.spj_id, /* SPJ_ID */
     p_project_rec.spj_tsmproj_is_number, /* SPJ_TSMPROJ_IS_NUMBER */
     p_project_rec.spj_tsmproj_org_id, /* SPJ_TSMPROJ_ORG_ID */
     p_project_rec.spj_name, /* SPJ_NAME */
     p_project_rec.spj_start_date, /* SPJ_START_DATE */
     p_project_rec.spj_duration, /* SPJ_DURATION */
     p_project_rec.spj_purpose, /* SPJ_PURPOSE */
     p_project_rec.spj_contact, /* SPJ_CONTACT */
     p_project_rec.spj_document, /* SPJ_DOCUMENT */
     p_project_rec.spj_export_seq, /* SPJ_EXPORT_SEQ */
     p_project_rec.spj_import_seq, /* SPJ_IMPORT_SEQ */
     p_project_rec.spj_status /* SPJ_STATUS */);

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
     
                                                                                                                       
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        SQLERRM || ' - inserting project.');
     
      
    RETURN 0;
  
END create_project;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE import_projects
(
  p_org_id         IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_imp_cfg_seq    IN NUMBER,
  p_test_count     IN NUMBER,
  p_skip_header    IN NUMBER DEFAULT 0       -- CLM 08/02/2005 Skip header option    
  ) IS                                       --   1 = Skip  0 = Don't Skip

  v_status           VARCHAR2(1);
  v_sim_proj_success NUMBER(1);

  v_inline        VARCHAR2(10000);
  v_bad_parse     BOOLEAN;
  v_line_table    sim_import_pkg.col_table_type;
  v_default_table sim_import_project_pkg.col_table_type;
  v_pos_table     sim_import_project_pkg.col_table_num_type;
  v_delimiter     VARCHAR2(1);
  v_num_cols      NUMBER(3);
  v_col_seq       NUMBER(10);
  v_dflt_pos      NUMBER(3);

  v_item_name   VARCHAR2(30);
  v_item        VARCHAR2(4000);
  v_comp        NUMBER(1);
  v_trans_exist BOOLEAN;
  v_success     NUMBER(1);
  v_storet_key  t_storet_pk_type;
  v_comment     VARCHAR2(1999);

  v_project_rec      sim_projects%ROWTYPE;
  v_project_copy_rec sim_projects%ROWTYPE;

  v_project_cnt  NUMBER(10) := 0;
  v_error_cnt    NUMBER(10) := 0;
  v_complete_cnt NUMBER(10) := 0;

  v_project_seq  NUMBER(10);
  v_start_date   DATE;
  v_project_id   VARCHAR2(8);
  v_project_name VARCHAR2(60);
  v_duration     VARCHAR2(15);
  v_purpose      VARCHAR2(4000);
  v_contact      VARCHAR2(4000);
  v_document     VARCHAR2(1999);
  v_sid_seq      NUMBER(10);  

  e_fileread EXCEPTION;
  e_bad_org EXCEPTION;
  e_no_cfg EXCEPTION;
  e_e_bad_delimiter EXCEPTION;
  e_bad_delimiter EXCEPTION;
  e_no_data_found EXCEPTION;
  e_project_seq EXCEPTION;
  
  CURSOR c_get_project_seq IS
    SELECT sim_projects_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT sipl_seq,
           sipl_line_text,
           sipl_line_no
      FROM sim_imp_lines
     WHERE sipl_sil_seq = p_import_seq
       AND sipl_line_text IS NOT NULL -- bka  added to eliminate blank lines from the import
     ORDER BY sipl_line_no;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT sic_proc_order,
           sic_dflt_pos,
           sic_name,
           sic_seq
      FROM sim_imp_cols,
           sim_imp_defs,
           sim_imp_cfg
     WHERE sicc_sid_seq = sid_seq
       AND sic_sid_seq = sid_seq
       AND sicc_seq = p_imp_cfg_seq
     ORDER BY sic_proc_order;

  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(spj_seq)
      FROM sim_projects
     WHERE spj_import_seq = p_import_log_seq
       AND spj_status = 'A';

BEGIN

  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  v_line_cnt := 0;

  BEGIN
    SELECT sicc_sid_seq
      INTO v_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_imp_cfg_seq;
    
    sim_event_log_pkg.write_event(1,v_import_log_seq,p_org_id,v_sid_seq,NULL);                                            
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Start: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
            
    IF p_skip_header = 1 THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                          'Header skipped');
      v_line_cnt := 1;
    END IF;
    /* get tsmorgan_is_number */
    BEGIN
      v_success := get_org_info(p_imp_cfg_seq, p_org_id, v_org_is_nbr);
      IF v_success != 1
         OR v_org_is_nbr IS NULL THEN
        RAISE e_bad_org;
      END IF;
    
    EXCEPTION
      WHEN e_bad_org THEN
        raise_application_error(-20109, 'Unable to find the specified org id in local version of STORET.');
      
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            SQLERRM ||
                                            ' - getting org is number.');
         
          
        RAISE no_data_found;
    END;
  
    /*  get the delimiter */
    BEGIN
      v_success := get_delimiter(p_imp_cfg_seq, v_delimiter);
      IF v_success != 1 THEN
        RAISE e_no_cfg;
      END IF;
    
    EXCEPTION
      WHEN e_no_cfg THEN
        raise_application_error(-20102, 'Error getting delimiter from import configuration.');
      
      WHEN OTHERS THEN
         
                                                                                                                                             
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            SQLERRM || '- getting delimiter.');
         
          
        RAISE no_data_found;
    END;
  
    /* get actual positions and default information for items being imported */
    BEGIN
    
      v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table, v_num_cols);
      IF v_success != 1 THEN
        RAISE e_no_cfg;
      END IF;
    EXCEPTION
      WHEN e_no_cfg THEN
        raise_application_error(-20102, 'Unable to get item information from the configuration table.');
    END;
  
    /* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
    /* CLM 08/02/2005 If skip header = 'No' or this is not the first row */
     IF p_skip_header = 0 OR v_text_line.sipl_line_no <> 1 THEN
      BEGIN
        /* read one line */
        v_inline := v_text_line.sipl_line_text;
      
        v_comment := NULL;
      
        v_line_cnt := v_line_cnt + 1;
        v_status := 'A';
      
        -- send line to parsing routine:
        BEGIN
          sim_import_pkg.parse_line_table(v_inline, v_line_table, v_num_cols, v_delimiter, v_bad_parse, v_import_log_seq);
        
          IF v_bad_parse THEN
            RAISE e_e_bad_delimiter;
          END IF;
        
        EXCEPTION
          WHEN e_e_bad_delimiter THEN
            RAISE e_bad_delimiter;
          
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                SQLERRM ||
                                                ' parsing one line.');
             
              
            RAISE e_fileread;
        END;
      END;
    
      /* loop through all project columns */
      FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP
      
        /* get dflt position, item name, and column sequence */
        v_dflt_pos := v_columns.sic_dflt_pos;
        v_item_name := v_columns.sic_name;
        v_col_seq := v_columns.sic_seq;
        v_trans_exist := TRUE;
      
        /* if column is included on the import, continue */
        IF v_pos_table(v_dflt_pos) IS NOT NULL THEN
        
          /* get the actual item, or use default if null */
          IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
            v_item := v_line_table(v_pos_table(v_dflt_pos));
          ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL
                AND v_default_table(v_dflt_pos) IS NOT NULL THEN
            v_item := v_default_table(v_dflt_pos);
          ELSE
            v_item := NULL;
          END IF;
        
          /* assign item to appropriate variables and take care of any other procedural issues that need to be
          taken care of for or with a particular item */
        
          -- Does V_ITEM contain only spaces? (or NULL)
          IF (TRIM(v_item) IS NULL) THEN
            -- if yes
          
            -- by definition V_ITEM is NULL!
            v_item := NULL;
          
          END IF;
        
          IF v_item_name = 'ProjectID' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := upper(rtrim(v_item)); -- 061203 slp - Set ID to uppercase and remove trailing spaces.
                  /* bka - Check to make sure that the project ID does not exceed 8 characters */
                  IF substr(v_item, 9, 9) IS NULL THEN
                    v_project_id := v_item;
                  ELSE
                    v_project_id := substr(v_item, 1, 8);
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        'Project ID, ' ||
                                                         v_item ||
                                                         ' is longer than 8 characters.');
                     
                      
                  END IF;
                
                  /* get the sequence for this project */
                  OPEN c_get_project_seq;
                  FETCH c_get_project_seq
                    INTO v_project_seq;
                  IF c_get_project_seq%NOTFOUND THEN
                    v_status := 'N'; -- SET STATUS TO N
                    v_project_seq := NULL;
                    RAISE e_project_seq;
                  END IF;
                  CLOSE c_get_project_seq;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              
              ELSE
                /* write error message if project id has not been provided */
                v_project_id := NULL;
                v_comp := write_error_message(v_item_name, TRUE /*p_req*/, v_line_cnt, FALSE /*p_provided*/, v_status, v_comment, v_project_seq);
                v_status := 'N'; -- SET STATUS TO N
              END IF;
            EXCEPTION
              WHEN e_project_seq THEN
                raise_application_error(-20150, 'Unable to get sequence for sim_projects table.');
              
              WHEN OTHERS THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting project id');
                v_status := 'N';  
            END;
          
          ELSIF v_item_name = 'Name' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_project_name := v_item;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if project name has not been provided */
                v_project_name := NULL;
                v_comp := write_error_message(v_item_name, TRUE /*p_req*/, v_line_cnt, FALSE /*p_provided*/, v_status, v_comment, v_project_seq);
              
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting project name');
                v_status := 'N'; 
                  
            END;
          
          ELSIF v_item_name = 'StartDate' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_start_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_project_seq);
                  IF v_start_date IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if project start date has not been provided */
                v_start_date := NULL;
                v_comp := write_error_message(v_item_name, TRUE /*p_req*/, v_line_cnt, FALSE /*p_provided*/, v_status, v_comment, v_project_seq);
              
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting start date,');
                v_status := 'N'; 
                  
            END;
          
          ELSIF v_item_name = 'Duration' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_duration := v_item;
                  
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if duration has not been provided */
                v_duration := NULL;
                v_comp := write_error_message(v_item_name, TRUE /*p_req*/, v_line_cnt, FALSE /*p_provided*/, v_status, v_comment, v_project_seq);
                
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting duration,');
                v_status := 'N'; 
                  
            END;
          
          ELSIF v_item_name = 'Purpose' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_purpose := v_item;
                  
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if purpose has not been provided */
                v_purpose := NULL;
                v_comp := write_error_message(v_item_name, TRUE /*p_req*/, v_line_cnt, FALSE /*p_provided*/, v_status, v_comment, v_project_seq);
                
              END IF;
            EXCEPTION 
              WHEN OTHERS THEN
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting purpose');
                v_status := 'N'; 
                  
            END;
          
          ELSIF v_item_name = 'Contact' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_contact := v_item;
                  
                ELSE
                  v_status := 'N'; -- Item translated to null - SET STATUS TO N 
                END IF;
              ELSE
                /* write error message if project contact has not been provided */
                v_contact := NULL;
                v_comp := write_error_message(v_item_name, FALSE /*p_req*/, v_line_cnt, FALSE /*p_provided*/, v_status, v_comment, v_project_seq);
                
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting contact');
                v_status := 'N'; 
                  
            END;
          
          ELSIF v_item_name = 'Document/Graphic' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF sim_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                    v_document := v_item;
                  ELSE
                    v_status := 'N'; -- Invalid document.
                    v_document := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; --Translated to a NULL value.  Set to 'N'.
                  v_document := NULL;
                END IF;
              ELSE
                v_document := NULL;
              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     SQLERRM ||
                                                     ' - getting project document/graphic');
                v_status := 'N'; 
                  
            END;
          END IF;
        END IF;
      END LOOP; -- end loop through columns
    
      -- Checks if there is a duplicate Project ID or Project Name in SIM
      v_sim_proj_success := check_sim_proj(v_project_id, p_org_id, v_project_name);
      IF v_sim_proj_success = 1 THEN
        v_status := 'N';
      END IF;
    
      v_storet_key := check_storet_proj(v_project_id, p_org_id, v_project_name, v_project_seq);
    
      --  insert into bka_test values(v_storet_key.is_number);
    
      IF v_storet_key.is_number IS NOT NULL THEN
        /* bka - project already exists in STORET, create in SIM but do not allow it to be exported */
        v_status := 'N';
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             'Project ' || v_project_id ||
                                             ' already exists in STORET.');
         
          
      END IF;
    
      /* create project record in SIM */
      BEGIN
      
        v_project_rec.spj_seq := v_project_seq;
        v_project_rec.spj_tsmorgan_is_number := v_org_is_nbr;
        v_project_rec.spj_org_id := p_org_id;
        v_project_rec.spj_id := v_project_id;
        v_project_rec.spj_tsmproj_is_number := NULL;
        v_project_rec.spj_tsmproj_org_id := NULL;
        v_project_rec.spj_name := v_project_name;
        v_project_rec.spj_start_date := v_start_date;
        v_project_rec.spj_duration := v_duration;
        v_project_rec.spj_purpose := v_purpose;
        v_project_rec.spj_contact := v_contact;
        v_project_rec.spj_document := v_document;
        v_project_rec.spj_export_seq := NULL;
        v_project_rec.spj_import_seq := v_import_log_seq;
        v_project_rec.spj_status := v_status;
      
        /* write data to sim table */
        v_success := create_project(v_project_rec);
      
        v_project_cnt := v_project_cnt + 1;
      
      EXCEPTION
        WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             SQLERRM ||
                                              ' - creating project record.');
         
          
      END;
    
      /* update import lines table with project sequence */
      UPDATE sim_imp_lines
         SET sipl_spj_seq = v_project_seq
       WHERE sipl_seq = v_text_line.sipl_seq;
    
      /* wipe out record containing project data */
      v_project_rec := v_project_copy_rec;
     END IF;
    END LOOP; -- end loop through text lines
  
    /* count the number of projects with errors */
    OPEN c_comp_count(v_import_log_seq);
    FETCH c_comp_count
      INTO v_complete_cnt;
    CLOSE c_comp_count;
    v_error_cnt := sim_import_pkg.get_rows_with_errors(v_import_log_seq);
    /* update counts in the import log */
    UPDATE sim_import_log
       SET sil_rows_imported = v_line_cnt,
           sil_rows_errors = v_error_cnt,
           sil_rows_ready = v_complete_cnt
     WHERE sil_seq = v_import_log_seq;
  
    COMMIT;
    -- Ryan J. 03/08/05 Changed the "projects loaded" text to use v_complete_cnt rather than v_project_cnt
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         to_char(v_line_cnt) ||
                                         ' lines processed / ' ||
                                         to_char(v_complete_cnt) ||
                                         ' projects loaded');
     
      
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Finished: ' ||
                                          to_char(SYSDATE, 'HH24:MI:SS'));
     
      
     
    sim_event_log_pkg.update_stop_time;
     
                                 
    sim_import_pkg.v_strip_quote := NULL; --10/04/2005 --RWK 
  
  END;

EXCEPTION

  WHEN e_no_cfg THEN
    raise_application_error(-20102, 'Configuration invalid or does not exist.');
  
  WHEN e_bad_delimiter THEN
    raise_application_error(-20197, 'The import file contains the wrong number of columns');
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                         substr(SQLERRM ||
                                        ' - in sim_import_project_pkg.import_projects.', 1, 240));
     
      
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         to_char(v_line_cnt) ||
                                         ' lines processed / ' ||
                                         to_char(v_complete_cnt) ||
                                         ' projects loaded');
     
      
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Finished: ' ||
                                          to_char(SYSDATE, 'HH24:MI:SS'));
     
      
    -- Ryan J. 03/08/05 - Raise error to the calling routine
    raise_application_error(-20000, substr(SQLERRM ||
                                    ' - in sim_import_project_pkg.import_projects.', 1, 240));
  
END import_projects;

END sim_import_project_pkg;
/
