CREATE OR REPLACE PACKAGE sim_import_result_extras IS
  
  -- Purpose: 
  -- To store many small "get" functions originally in the main result import package

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                               *
 ****************************************************************************************/
  TYPE t_storet_pk_type IS RECORD(
    is_number NUMBER(12),
    org_id    VARCHAR2(8));

  TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;

  TYPE col_table_num_type IS TABLE OF NUMBER(8) INDEX BY BINARY_INTEGER;

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/


/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                           *
 ****************************************************************************************/
  FUNCTION issue_not_used_warning
  (
    p_item_name      IN VARCHAR2,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION get_community_is_nbr
  (
    p_community_name IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION validate_location
  (
    p_item           IN VARCHAR2,
    v_point_type     IN VARCHAR2,
    v_what           IN VARCHAR2,
    v_station_is_nbr IN NUMBER,
    v_station_org_id IN VARCHAR2,
    p_activity_seq   IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER,
    p_tsmalp_is_nbr  OUT NUMBER,
    p_tsmalp_org_id  OUT VARCHAR2
  ) RETURN VARCHAR2;              -- 09/13/05 -RWK -Changed return from NUMBER to VARCHAR2

  FUNCTION validate_alval_item
  (
    p_item_name      IN VARCHAR2,
    p_orig_item      IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_fa_seq         IN NUMBER,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER
  ) RETURN NUMBER;

  FUNCTION validate_pv_result
  (
    p_orig_item      IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_char_is_nbr    IN NUMBER,
    p_fa_seq         IN NUMBER,
    p_ok             OUT BOOLEAN,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION validate_number
  (
    p_item           IN VARCHAR2,
    p_item_name      IN VARCHAR2,
    p_format         IN VARCHAR2,
    p_import_log_seq IN NUMBER,
    p_line_cnt       IN NUMBER
  ) RETURN NUMBER;

  FUNCTION validate_storet_item
  (
    p_item_name       IN VARCHAR2,
    p_orig_item       IN VARCHAR2,
    p_imp_cfg_seq     IN NUMBER,
    p_col_seq         IN NUMBER,
    p_parent_pk       IN sim_import_result_extras.t_storet_pk_type,
    p_org_id          IN VARCHAR2,
    v_line_cnt        IN NUMBER,
    p_fa_seq          IN NUMBER,
    v_import_log_seq  IN NUMBER,
    p_translated_item OUT VARCHAR2
  ) RETURN sim_import_result_extras.t_storet_pk_type;

  FUNCTION validate_fraction
  (
    p_samp_frac_nm   IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_tsrchar_is_nbr IN NUMBER,
    p_fa_seq         IN NUMBER,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER,
    v_tsmprmvl_key   OUT t_storet_pk_type
  ) RETURN VARCHAR2;

  FUNCTION get_instructions
  (
    p_imp_cfg_seq      IN NUMBER,
    p_instruction_type IN VARCHAR2,
    p_fa_seq           IN NUMBER,
    v_import_log_seq   IN NUMBER,
    v_line_cnt         IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION get_taxon_is_nbr
  (
    p_char_name      IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_life_is_nbr
  (
    p_life_name      IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;
  FUNCTION get_sex_is_nbr
  (
    p_sex_name       IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_biopt_is_nbr
  (
    p_biopt_name     IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_tsmalp_is_nbr
  (
    p_type_code       IN VARCHAR2,
    p_sequence_number IN NUMBER,
    p_tsmstatn_is_nbr IN NUMBER,
    p_tsmstatn_org_id IN VARCHAR2,
    p_line_cnt        IN NUMBER,
    p_import_log_seq  IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_item_value
  (
    p_alval_seq      IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION get_delimiter
  (
    p_imp_cfg_seq    IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION get_item_pos
  (
    p_sicc_seq       IN NUMBER,
    p_pos_table      OUT col_table_num_type,
    p_default_table  OUT col_table_type,
    p_col_cnt        IN OUT NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION get_proj_is_nbr
  (
    p_proj           VARCHAR2,
    p_org_id         VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_lab_is_nbr
  (
    p_lab_id         IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_gear_is_nbr
  (
    p_gear           IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_gear_cfg_is_nbr
  (
    p_gear_cfg       IN VARCHAR2,
    p_gear_pk        IN t_storet_pk_type,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_sample_pres_is_nbr
  (
    p_proc           IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_group_is_nbr
  (
    p_group_id       IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_row_is_nbr
  (
    p_row_id         IN VARCHAR2,
    p_group_key      IN t_storet_pk_type,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_ha_row_is_nbr
  (
    p_row_id         IN VARCHAR2,
    p_group_key      IN t_storet_pk_type,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_ha_default
  (
    v_tsrhcsc_pk     IN sim_import_result_extras.t_storet_pk_type,
    p_fa_seq         IN NUMBER,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION get_lab_sample_prep_procedure
  (
    p_labprep_id     IN VARCHAR2,
    p_labprep_source IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_import_log_seq IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_fa_seq         IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_field_lab_procedure
  (
    p_proc_id        IN VARCHAR2,
    p_source         IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_import_log_seq IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_fa_seq         IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_procedure_is_nbr
  (
    p_proc           IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_station_is_nbr
  (
    p_station_id     IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_personnel_is_nbr
  (
    p_contact_id     IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_remark_codes_is_num
  (
    p_short_name     IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_char_is_nbr
  (
    p_char_name      IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER,
    fromwhere        IN VARCHAR2
  ) RETURN t_storet_pk_type;

  FUNCTION get_analproc_is_nbr
  (
    p_procedure_id   IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_labprep_is_nbr
  (
    p_preparation_id IN VARCHAR2,
    p_org_id         IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN t_storet_pk_type;

  FUNCTION get_trip_id
  (
    p_instr_code      IN VARCHAR2,
    p_start_date      IN DATE,
    p_trip_start_date OUT DATE,
    p_trip_end_date   OUT DATE,
    p_org_id          IN VARCHAR2,
    p_fa_seq          IN NUMBER,
    p_line_cnt        IN NUMBER,
    p_import_log_seq  IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION get_visit_start_date
  (
    p_visit_key      IN t_storet_pk_type,
    p_trip_seq       IN NUMBER,
    p_fa_seq         IN NUMBER,
    p_line_cnt       IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN DATE;

  FUNCTION get_visit_seq
  (
    p_visit_key        OUT t_storet_pk_type,
    p_station_key      IN t_storet_pk_type,
    p_visit_start_date IN DATE,
    p_visit_id         IN VARCHAR2,
    p_trip_seq         IN NUMBER,
    p_fa_seq           IN NUMBER,
    p_line_cnt         IN NUMBER,
    p_import_log_seq   IN NUMBER
  ) RETURN NUMBER;
  -- jah 3-8-2004 Added Parm P_VISIT_START_Date

  FUNCTION get_org_info
  (
    p_org_id         IN VARCHAR2,
    p_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION get_allowable_value
  (
    p_alval_seq      IN NUMBER,
    p_import_log_seq IN NUMBER
  ) RETURN VARCHAR2;

  FUNCTION create_sim_station_visit
  (
    p_visit_rec      IN sim_station_visits%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_sim_trip
  (
    p_trip_rec       IN OUT sim_trips%ROWTYPE,
    p_fa_seq         IN NUMBER,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_field_activity
  (
    p_fa_rec         IN sim_field_activities%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_field_activity_parent
  (
    p_fap_rec        IN sim_field_activity_parent%ROWTYPE,
    p_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_rci
  (
    p_rci_rec        IN sim_result_class_indicator%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_brgi
  (
    p_brgi_rec       IN sim_bio_results_grp_individual%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_brg
  (
    p_brg_rec        IN sim_bio_results_grp%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_sample
  (
    p_sample_rec     IN sim_sample%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_tod
  (
    p_tod_rec        IN sim_trawl_ops_det%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_stl
  (
    p_stl_rec        IN sim_station_locations%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_aal
  (
    p_aal_rec        IN sim_actual_activity_loc%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_trap_net
  (
    p_stn_rec        IN sim_trap_net%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION create_result
  (
    p_result_rec     IN sim_results%ROWTYPE,
    v_import_log_seq IN NUMBER
  ) RETURN NUMBER;

  FUNCTION check_storet_translation
  (
    p_code           IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_trans_exist    OUT BOOLEAN,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER,
    v_converted_item OUT VARCHAR2,
    p_activity_seq   IN NUMBER
  ) RETURN t_storet_pk_type;

  PROCEDURE create_trans_record
  (
    p_code           IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_fa_seq         IN NUMBER,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER
  );

  FUNCTION get_uom_is_nbr
  (
    p_uom_name       IN VARCHAR2,
    p_fa_seq         IN NUMBER,
    v_import_log_seq IN NUMBER,
    v_line_cnt       IN NUMBER
  ) RETURN sim_import_result_extras.t_storet_pk_type;

  FUNCTION update_result
  (
    p_result           IN VARCHAR2,
    p_char_is_nbr      IN NUMBER,
    p_units_ok         OUT BOOLEAN,
    p_result_ok        OUT BOOLEAN,
    p_imp_cfg_seq      IN NUMBER,
    p_fa_seq           IN NUMBER,
    v_char_result_type IN VARCHAR2,
    v_import_log_seq   IN NUMBER,
    v_line_cnt         IN NUMBER,
    v_detect           IN VARCHAR2,
    v_detect_check     OUT NUMBER
  ) RETURN VARCHAR2;

END sim_import_result_extras;
/
CREATE OR REPLACE PACKAGE BODY sim_import_result_extras IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/
  c_adult_is_number CONSTANT NUMBER(12) := 559;
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  
  
/****************************************************************************************
 *                                  PROCEDURES AND FUNCTIONS                            *
****************************************************************************************/

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION issue_not_used_warning
(
  p_item_name      IN VARCHAR2,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN 
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| p_line_cnt,
                                       p_item_name ||
                                      ' is not currently imported.');
  RETURN 0;

EXCEPTION
  WHEN OTHERS THEN 
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| p_line_cnt,
                                         SQLERRM || ' - issuing warning');
    RETURN 1;
END issue_not_used_warning;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_sample_matrix
(
  p_matrix         IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_fa_seq         IN NUMBER
) RETURN t_storet_pk_type IS

  v_trans_exist    BOOLEAN;
  v_matrix_is_nbr  NUMBER(12);
  v_matrix_org_id  VARCHAR2(8);
  v_matrix_pk      t_storet_pk_type;
  v_converted_item VARCHAR2(80);
  v_instr_code     VARCHAR2(10);

  CURSOR c_get_matrix(v_code IN VARCHAR2) IS
    SELECT tsrmatrx_is_number,
           tsrmatrx_org_id
      FROM tsrmatrx
     WHERE code = rpad(v_code, 4);

BEGIN
  -- kms - validate sample matrix againsts values in STORET TSRMATRX table.
  v_matrix_pk.is_number := NULL;
  v_matrix_pk.org_id := NULL;
  -- look in translations table to see if it needs to swapped - will return original item or translation
  v_converted_item := sim_import_pkg.check_translations(p_matrix, p_imp_cfg_seq, p_col_seq, p_line_cnt, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN
    OPEN c_get_matrix(v_converted_item);
    FETCH c_get_matrix
      INTO v_matrix_is_nbr, v_matrix_org_id;
    IF c_get_matrix%NOTFOUND THEN
    
      IF NOT v_trans_exist THEN
        --check instructions to see if translation record should be created
        v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, p_import_log_seq, p_line_cnt);
      
        IF v_instr_code = 'RES-TR-01' THEN
        
          -- if record could not be matched to anything in translations or allowable values tables, create new record
          create_trans_record(p_matrix, p_imp_cfg_seq, p_col_seq, p_fa_seq, p_import_log_seq, p_line_cnt);
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                               'Sample Matrix, ' ||
                                               v_converted_item ||
                                               ' is not valid - translation record has been created.');
           
        
        ELSE
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                               'Sample Matrix, ' ||
                                               v_converted_item ||
                                               ' is not valid.');
           
        
        END IF;
      
      ELSIF v_trans_exist THEN
        -- item has been translated to an invalid storet value
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                             p_matrix ||
                                             ' has been mapped to an invalid storet value: ' ||
                                             v_converted_item);
         
      
      END IF;
    
    ELSE
      v_matrix_pk.is_number := v_matrix_is_nbr;
      v_matrix_pk.org_id := v_matrix_org_id;
    END IF;
    CLOSE c_get_matrix;
  
  ELSIF v_trans_exist
        AND v_converted_item IS NULL THEN
    v_matrix_pk.is_number := NULL;
    v_matrix_pk.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Sample Matrix ' || p_matrix ||
                                         ' exists in translations table but has not been mapped to a STORET code.');
     
          
  END IF;

  RETURN v_matrix_pk;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                          ' - in validate sample matrix.');
     
          
    RETURN NULL;
  
END validate_sample_matrix;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_location
(
  p_item           IN VARCHAR2,
  v_point_type     IN VARCHAR2,
  v_what           IN VARCHAR2,
  v_station_is_nbr IN NUMBER,
  v_station_org_id IN VARCHAR2,
  p_activity_seq   IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_tsmalp_is_nbr  OUT NUMBER,
  p_tsmalp_org_id  OUT VARCHAR2
  ) RETURN VARCHAR2 IS -- 09/13/05 -RWK - Changed Return from NUBMER to VARCHAR2 to
                       -- accomodate alphanumeric well numbers.

  v_return       VARCHAR2(15); -- 09/13/05 -RWK - Changed from NUBMER to VARCHAR2
  v_id_check     VARCHAR2(15);
  v_item_num      VARCHAR2(15); -- 09/13/05 -RWK - Changed from NUBMER to VARCHAR2
  v_well_pipe_id VARCHAR2(15);

  -- 09/13/05 -RWK - Changed parameter from NUBMER to VARCHAR2
  CURSOR c_find_location(v_loc_num IN VARCHAR2) IS 
    SELECT stl_sequence_number
      FROM sim_station_locations
     WHERE stl_sequence_number = v_loc_num
       AND stl_type_code LIKE v_point_type || '%'
       AND stl_tsmstatn0is_number = v_station_is_nbr
       AND stl_tsmstatn0org_id = rpad(v_station_org_id, 8);

  CURSOR c_find_well(v_well_num IN VARCHAR2) IS
    SELECT w.ssw_well_no,
           l.stl_is_number,
           l.stl_org_id
      FROM sim_station_wells w,
           sim_station_locations l
     WHERE w.ssw_is_number = l.stl_tsmwell_is_number
       AND rtrim(w.ssw_well_no) = rtrim(v_well_num)
       AND l.stl_type_code LIKE v_point_type || '%'
       AND l.stl_tsmstatn0is_number = v_station_is_nbr
       AND l.stl_tsmstatn0org_id = rpad(v_station_org_id, 8);

  CURSOR c_find_pipe(v_pipe_num IN VARCHAR2) IS
    SELECT p.id_number,
           l.stl_is_number,
           l.stl_org_id
      FROM tsmpipe p,
           sim_station_locations l
     WHERE p.tsmpipe_is_number = l.stl_tsmpipe_is_number
       AND p.id_number = v_pipe_num
       AND l.stl_type_code LIKE v_point_type || '%'
       AND l.stl_tsmstatn0is_number = v_station_is_nbr
       AND l.stl_tsmstatn0org_id = rpad(v_station_org_id, 8);

BEGIN
  /* Initializing v_return to return the value passed in, 
     v_return will be reset to -1 if p_item is not valid
  */
  v_return := rtrim(p_item);
  IF v_what = 'SEQ' THEN
    BEGIN
      v_item_num := rtrim(p_item); -- 09/13/05 -RWK - added rtrim() to assignment
      OPEN c_find_location(v_item_num);
      FETCH c_find_location
        INTO v_id_check;
      IF c_find_location%NOTFOUND THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                             'Station Location,' || p_item || ', ' ||
                                             v_point_type ||
                                             ', not found in STORET on this station.');
         
             
        v_return := '-1'; -- 09/13/05 -RWK - Changed -1 from NUBMER to String
      END IF;
      CLOSE c_find_location;
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                             'Sequence Number must be a number between 0 and 9999');
         
             
        RETURN '-1'; -- 09/13/05 -RWK - Changed -1 from NUBMER to String
    END;
  ELSE
    --First check for well
    v_well_pipe_id := rtrim(p_item);
    OPEN c_find_well(v_well_pipe_id);
    FETCH c_find_well
      INTO v_id_check, p_tsmalp_is_nbr, p_tsmalp_org_id;
  
    IF c_find_well%NOTFOUND THEN
      v_item_num := rtrim(p_item); -- 09/13/05 -RWK - added rtrim() to assignment
      OPEN c_find_pipe(v_item_num);
      FETCH c_find_pipe
        INTO v_id_check, p_tsmalp_is_nbr, p_tsmalp_org_id;
    
      IF c_find_pipe%NOTFOUND THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                             'Well or Pipe Number,' || p_item || ', ' ||
                                             v_point_type ||
                                             ', not found in STORET on this station.');
         
             
        v_return := '-1'; -- 09/13/05 -RWK - Changed -1 from NUBMER to String
      END IF;
      CLOSE c_find_pipe;
    END IF;
    CLOSE c_find_well;
  END IF;

  RETURN v_return;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM || ' - in validate location');
     
         
    RETURN '-1'; -- 09/13/05 -RWK - Changed -1 from NUBMER to String
END validate_location;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_instructions
(
  p_imp_cfg_seq      IN NUMBER,
  p_instruction_type IN VARCHAR2,
  p_fa_seq           IN NUMBER,
  v_import_log_seq   IN NUMBER,
  v_line_cnt         IN NUMBER
) RETURN VARCHAR2 IS

  v_instr_choice VARCHAR2(15);

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_cfg_instrs,
           sim_imp_instr_choices,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND sici_sicc_seq = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_instr_choices,
           sim_imp_cfg_instrs,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND siic_dflt_flag = 1;

BEGIN

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr
    INTO v_instr_choice;
  IF c_get_cfg_instr%NOTFOUND THEN
  
    /* no choice could be found, get default choice */
    OPEN c_get_instr_default(p_instruction_type);
    FETCH c_get_instr_default
      INTO v_instr_choice;
    CLOSE c_get_instr_default;
  
  END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;

EXCEPTION
  WHEN OTHERS THEN
      
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get_instructions.');
     
         
    RETURN NULL;
  
END get_instructions;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_community_is_nbr
(
  p_community_name IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_community_key t_storet_pk_type;

  CURSOR c_get_is_nbr(p_community_name IN VARCHAR2) IS
    SELECT tsmprmvl_is_number,
           tsmprmvl_org_id
      FROM tsmprmvl
     WHERE table_name = 'TSRFDACT'
       AND field_name = 'COMMUNITY_NAME'
       AND field_value = p_community_name;

BEGIN
  /*sfp -  function that checks the permitted value in STORET using the processed short_name.
  Called by validate_storet_item for 'Community' */

  OPEN c_get_is_nbr(p_community_name);
  FETCH c_get_is_nbr
    INTO v_community_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_community_key.is_number := NULL;
    v_community_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Community, ' || p_community_name ||
                                          ', could not be found in STORET.');
     
         
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_community_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                          ' - getting community is number');
     
         
    v_community_key.is_number := NULL;
    v_community_key.org_id := NULL;
    RETURN v_community_key;
  
END get_community_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_taxon_is_nbr
(
  p_char_name      IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_char_key t_storet_pk_type;

  -- jah 5-27-2004 Task 477
  -- add OR SEARCH_NAME = UPPER(P_CHAR_NAME) to cursor c_get_is_nbr
  CURSOR c_get_is_nbr(p_char_name IN VARCHAR2) IS
    SELECT tsrchar_is_number,
           tsrchar_org_id
      FROM tsrchar
     WHERE (display_name = p_char_name OR search_name = upper(p_char_name))
       AND (d_scr_type_cd = 'TAXA' OR d_scr_type_cd = 'VAR ');

BEGIN
  /* 06/09/03 slp - Note that this function is used for checking 'TAXA' in the Subject Taxon column
  and also the characteristic column for Multi Taxon Population Census activities. */
  /* sfp - function that checks a TAXA characteristic in STORET using the processed display name.
  Called by validate storet item. */
  /* sfp - added restriction for D_SCR_TYPE_CD = 'TAXA'
  sf  - allow 'VAR' for V2 */
  OPEN c_get_is_nbr(p_char_name);
  FETCH c_get_is_nbr
    INTO v_char_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_char_key.is_number := NULL;
    v_char_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Subject Taxon or Characteristic, ' ||
                                         p_char_name ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_char_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                          ' - getting subject taxon is number');
     
        
    v_char_key.is_number := NULL;
    v_char_key.org_id := NULL;
    RETURN v_char_key;
  
END get_taxon_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_sex_is_nbr
(
  p_sex_name       IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_sex_key t_storet_pk_type;

  CURSOR c_get_is_nbr(p_sex_name IN VARCHAR2) IS
    SELECT tsrcpv_is_number,
           tsrcpv_org_id
      FROM tsrcpv
     WHERE short_name = rpad(p_sex_name, 12)
       AND tsrcpv.tsrchar_is_number =
           (SELECT tsrchar_is_number
              FROM tsrchar
             WHERE display_name = 'Sex (choice list)');
BEGIN
  /*sfp -  function that checks the permitted value in STORET using the processed name.
  Called by validate_storet_item for 'Sex Name' */

  OPEN c_get_is_nbr(p_sex_name);
  FETCH c_get_is_nbr
    INTO v_sex_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_sex_key.is_number := NULL;
    v_sex_key.org_id := NULL;
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Sex, ' ||
                                         p_sex_name ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_sex_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM || ' - getting sex is number');
     
        
    v_sex_key.is_number := NULL;
    v_sex_key.org_id := NULL;
    RETURN v_sex_key;
  
END get_sex_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_life_is_nbr
(
  p_life_name      IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_life_key t_storet_pk_type;

  CURSOR c_get_is_nbr(p_life_name IN VARCHAR2) IS
    SELECT tsrcpv_is_number,
           tsrcpv_org_id
      FROM tsrcpv
     WHERE short_name = rpad(p_life_name, 12)
       AND tsrcpv.tsrchar_is_number =
           (SELECT tsrchar_is_number
              FROM tsrchar
             WHERE display_name = 'Lifestage (choice list)');
BEGIN
  /*sfp -  function that checks the permitted value in STORET using the processed name.
  Called by validate_storet_item for 'Life Stage Name' */

  OPEN c_get_is_nbr(p_life_name);
  FETCH c_get_is_nbr
    INTO v_life_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_life_key.is_number := NULL;
    v_life_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Lifestage Name, ' || p_life_name ||
                                          ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_life_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - getting life is number');
     
        
    v_life_key.is_number := NULL;
    v_life_key.org_id := NULL;
    RETURN v_life_key;
  
END get_life_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_biopt_is_nbr
(
  p_biopt_name     IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_biopt_key t_storet_pk_type;

  CURSOR c_get_is_nbr(p_biopt_name IN VARCHAR2) IS
    SELECT tsrbiopt_is_number,
           tsrbiopt_org_id
      FROM tsrbiopt
     WHERE NAME = p_biopt_name;

BEGIN
  /*sfp -  function that checks the permitted value in STORET using the processed name.
  Called by validate_storet_item for 'Bio Part' */

  OPEN c_get_is_nbr(p_biopt_name);
  FETCH c_get_is_nbr
    INTO v_biopt_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_biopt_key.is_number := NULL;
    v_biopt_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Bio Part, ' || p_biopt_name ||
                                          ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_biopt_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting biopt is number');
     
        
    v_biopt_key.is_number := NULL;
    v_biopt_key.org_id := NULL;
    RETURN v_biopt_key;
  
END get_biopt_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_tsmalp_is_nbr
(
  p_type_code       IN VARCHAR2,
  p_sequence_number IN NUMBER,
  p_tsmstatn_is_nbr IN NUMBER,
  p_tsmstatn_org_id IN VARCHAR2,
  p_line_cnt        IN NUMBER,
  p_import_log_seq  IN NUMBER
) RETURN t_storet_pk_type IS

  v_tsmalp_key t_storet_pk_type;

  CURSOR c_get_is_nbr IS
    SELECT tsmalp_is_number,
           tsmalp_org_id
      FROM tsmalp
     WHERE tsmstatn0is_number = p_tsmstatn_is_nbr
       AND tsmstatn0org_id = rpad(p_tsmstatn_org_id, 8)
       AND type_code = rpad(p_type_code, 16)
       AND sequence_number = p_sequence_number;

BEGIN
  /*sfp -  function that gets the tsmalp from STORET using the procedure is_nbr
  Called by import_result */

  OPEN c_get_is_nbr;
  FETCH c_get_is_nbr
    INTO v_tsmalp_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_tsmalp_key.is_number := NULL;
    v_tsmalp_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'This station Location could not be found in STORET.');
     
        
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_tsmalp_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - getting tsmalp is number');
     
        
    v_tsmalp_key.is_number := NULL;
    v_tsmalp_key.org_id := NULL;
    RETURN v_tsmalp_key;
  
END get_tsmalp_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_item_value
(
  p_alval_seq      IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  v_item_value VARCHAR2(32);

  -- Get the Item Value (i.e., the Name) in a particular record
  CURSOR c_get_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

  /*  This function returns the item value (i.e., the name) 
  in a given record (P_ALVAL_SEQ) in the SIM Allowable 
  Values table.
  
  This function does not output an error message if the 
  calling routine passes an invalid record number, it 
  simply returns NULL. */

BEGIN

  -- try to retrieve the item name
  OPEN c_get_name(p_alval_seq);
  FETCH c_get_name
    INTO v_item_value;

  -- did we retrieve the item name?
  IF c_get_name%NOTFOUND THEN
    -- if no
    -- tell calling routine we did not retrieve the name
    v_item_value := NULL;
  END IF;
  CLOSE c_get_name;

  RETURN v_item_value;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM || ' - getting item name');
     
        
    -- tell calling routine we did not retrieve the name
    RETURN NULL;
  
END get_item_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_delimiter
(
  p_imp_cfg_seq    IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  v_delimiter VARCHAR2(1);

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT sicc_delimiter
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

BEGIN
  /* kms - function that retrieves the delimiter from the import configuration table.  Called by main
  import procedure - import_results. */

  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg
    INTO v_delimiter;
  IF c_get_imp_cfg%NOTFOUND THEN
    v_delimiter := NULL;
  END IF;
  CLOSE c_get_imp_cfg;

  RETURN v_delimiter;

EXCEPTION

  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - Failed to get delimiter.');
     
        
    RETURN NULL;
  
END get_delimiter;

/****************************************************************************************
  Populate the tables that will hold the import configuration information.
  This will populate tables that hold actual position information and default information
  indexed by the default column position.  It will also count the columns in order to insure
  that they match the import configuration.  Called by main import procedure - import_results.
 ****************************************************************************************/ 
FUNCTION get_item_pos
(
  p_sicc_seq       IN NUMBER,
  p_pos_table      OUT col_table_num_type,
  p_default_table  OUT col_table_type,
  p_col_cnt        IN OUT NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN NUMBER IS

  v_import_type_seq NUMBER(10);

  e_bad_cfg EXCEPTION;

  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT sicc_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

  CURSOR c_get_dflt_pos(p_type_seq IN NUMBER) IS
    SELECT sic_dflt_pos
      FROM sim_imp_cols
     WHERE sic_sid_seq = p_type_seq;

  CURSOR c_get_col_pos(p_sicc_seq IN NUMBER, p_dflt_pos IN NUMBER) IS
    SELECT sicdt_pos,
           sicdt_default
      FROM sim_imp_cfg_dets,
           sim_imp_cols
     WHERE sic_seq = sicdt_sic_seq
       AND sicdt_sicc_seq = p_sicc_seq
       AND sic_dflt_pos = p_dflt_pos
       AND (sicdt_included IS NOT NULL OR sicdt_generate IS NOT NULL);

BEGIN
  /* get the import type in order to limit the columns selected to those that are for results
  and activities */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type
    INTO v_import_type_seq;
  IF c_get_import_type%NOTFOUND THEN
    v_import_type_seq := NULL;
    CLOSE c_get_import_type;
    RAISE e_bad_cfg;
  END IF;
  CLOSE c_get_import_type;
  -- get the count of included or generated columns 
  SELECT COUNT(*)
    INTO p_col_cnt
    FROM sim_imp_cfg_dets d
    WHERE (d.sicdt_included IS NOT NULL OR d.sicdt_generate IS NOT NULL)
    AND d.sicdt_sicc_seq = p_sicc_seq;
  /* loop through import columns and assign actual position to a table (p_pos_table) indexed by
  default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP
    /* get items' actual position and default value from configuration table */
    OPEN c_get_col_pos(p_sicc_seq, v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos
      INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
    IF c_get_col_pos%NOTFOUND THEN
      /* if the column has not been included in the import configuration, it's position and
      default will be null */
      p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
      p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
    END IF;
    CLOSE c_get_col_pos;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN e_bad_cfg THEN
    raise_application_error(-20103, 'Failed to get import sequence.');
    RETURN 0;
  
  WHEN OTHERS THEN
    raise_application_error(-20103, 'Failed to get column positions, item names, and column sequences.');
  
END get_item_pos;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_proj_is_nbr
(
  p_proj           VARCHAR2,
  p_org_id         VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_proj_key t_storet_pk_type;

  CURSOR c_proj_is_nbr(p_proj VARCHAR2, p_org_id VARCHAR2) IS
    SELECT tsmproj_is_number,
           tsmproj_org_id
      FROM tsmproj
     WHERE tsmproj_org_id = rpad(p_org_id, 8)
       AND identification_cd = rpad(p_proj, 8);

BEGIN
  /* kms - function that gets the project primary key using the processed project id - called by validate storet item*/
  OPEN c_proj_is_nbr(p_proj, p_org_id);
  FETCH c_proj_is_nbr
    INTO v_proj_key;
  IF c_proj_is_nbr%NOTFOUND THEN
    v_proj_key.is_number := NULL;
    v_proj_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Project, ' || p_proj ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_proj_is_nbr;

  RETURN v_proj_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                          ' - in get project is number.');
     
        
    v_proj_key.is_number := NULL;
    v_proj_key.org_id := NULL;
    RETURN v_proj_key;
  
END get_proj_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_lab_is_nbr
(
  p_lab_id         IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_tsrlab_key t_storet_pk_type;

  CURSOR c_find_lab(p_lab_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsrlab_is_number,
           tsrlab_org_id
      FROM tsrlab
     WHERE id_code = rpad(p_lab_id, 8)
       AND tsrlab_org_id = rpad(p_org_id, 8);

BEGIN
  /* kms - function that gets the lab primary key using the processed lab id - called by validate storet item */
  OPEN c_find_lab(p_lab_id, p_org_id);
  FETCH c_find_lab
    INTO v_tsrlab_key;
  IF c_find_lab%NOTFOUND THEN
    v_tsrlab_key.is_number := NULL;
    v_tsrlab_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Lab ID - ' || p_lab_id ||
                                        ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_find_lab;

  RETURN v_tsrlab_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM || ' - in get lab is number.');
     
        
    v_tsrlab_key.is_number := NULL;
    v_tsrlab_key.org_id := NULL;
    RETURN v_tsrlab_key;
  
END get_lab_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_gear_is_nbr
(
  p_gear           IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_gear_key t_storet_pk_type;

  CURSOR c_gear_is_nbr IS
    SELECT tsrfldgr_is_number,
           tsrfldgr_org_id
      FROM tsrfldgr
     WHERE id_code = rpad(p_gear, 10);
  /* sf - TSRFLDGR is a STORET table and contains distinct GEAR ID_CODEs.     */

BEGIN
  /* kms - function that gets the gear primary key using the processed gear id - called by validate storet item */

  OPEN c_gear_is_nbr;
  FETCH c_gear_is_nbr
    INTO v_gear_key;
  IF c_gear_is_nbr%NOTFOUND THEN
    v_gear_key.is_number := NULL;
    v_gear_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Gear, ' ||
                                         p_gear ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_gear_is_nbr;

  RETURN v_gear_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - in get gear is number.');
     
        
    v_gear_key.is_number := NULL;
    v_gear_key.org_id := NULL;
    RETURN v_gear_key;
  
END get_gear_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_gear_cfg_is_nbr
(
  p_gear_cfg       IN VARCHAR2,
  p_gear_pk        IN t_storet_pk_type,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_gear_cfg_key t_storet_pk_type;

  CURSOR c_gear_cfg_is_nbr(p_gear_cfg IN VARCHAR2, p_gear_pk IN t_storet_pk_type, p_org_id IN VARCHAR2) IS
    SELECT tsrgrcfg_is_number,
           tsrgrcfg_org_id
      FROM tsrgrcfg
     WHERE id_code = rpad(p_gear_cfg, 10)
       AND tsrgrcfg_org_id = rpad(p_org_id, 8)
       AND tsrfldgr_is_number = p_gear_pk.is_number
       AND tsrfldgr_org_id = rpad(p_gear_pk.org_id, 8);

BEGIN

  /* kms (5/21/01)
  This function gets the gear config primary key using
  the processed gear configuration id. Function
  VALIDATE_STORET_ITEM calls this function. */

  -- Try to retrieve the Gear Configuration ID from STORET
  OPEN c_gear_cfg_is_nbr(p_gear_cfg, p_gear_pk, p_org_id);
  FETCH c_gear_cfg_is_nbr
    INTO v_gear_cfg_key;

  -- Did we find the Gear Configuration ID in STORET?
  IF c_gear_cfg_is_nbr%NOTFOUND THEN
    -- if no
    /* kms - gear config either does not exist or is not right for this gear */
  
    -- Set the return value to NULL. This tells the caller
    -- that the gear configuration ID is not in STORET.
    v_gear_cfg_key.is_number := NULL;
    v_gear_cfg_key.org_id := NULL;
  
      
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Gear Configuration, ' ||
                                         p_gear_cfg ||
                                         ', does not exist for the gear on this activity.');
     
        
  END IF;
  CLOSE c_gear_cfg_is_nbr;

  RETURN v_gear_cfg_key;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM ||
                                        ' - in get gear cfg is number.');
     
        
    v_gear_cfg_key.is_number := NULL;
    v_gear_cfg_key.org_id := NULL;
    RETURN v_gear_cfg_key;
  
END get_gear_cfg_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_sample_pres_is_nbr
(
  p_proc           IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_proc_key t_storet_pk_type;

  CURSOR c_proc_is_nbr(p_proc VARCHAR2, p_org_id VARCHAR2) IS
    SELECT tsrsdp_is_number,
           tsrsdp_org_id
      FROM tsrsdp
     WHERE tsrsdp_org_id = rpad(p_org_id, 8)
       AND id_code = rpad(p_proc, 10);

BEGIN
  /* kms - function that gets the sample preservation primary key using the processed id_code.
  Called by validate storet item */
  OPEN c_proc_is_nbr(p_proc, p_org_id);
  FETCH c_proc_is_nbr
    INTO v_proc_key;
  IF c_proc_is_nbr%NOTFOUND THEN
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Sample Preservation, ' || p_proc ||
                                        ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_proc_is_nbr;

  RETURN v_proc_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM ||
                                        ' - in get sample pres is number.');
     
        
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;
  
END get_sample_pres_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_group_is_nbr
(
  p_group_id       IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_group_key t_storet_pk_type;

  CURSOR c_group_is_nbr(p_group IN VARCHAR2) IS
    SELECT tsrchgrp_is_number,
           tsrchgrp_org_id
      FROM tsrchgrp
     WHERE id_code = rpad(p_group, 8)
       AND tsrchgrp_org_id = rpad(p_org_id, 8);

BEGIN
  /* kms - function that gets the characteristic group primary key using the processed group id - called by validate storet item */
  OPEN c_group_is_nbr(p_group_id);
  FETCH c_group_is_nbr
    INTO v_group_key;
  IF c_group_is_nbr%NOTFOUND THEN
    v_group_key.is_number := NULL;
    v_group_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Characteristic Group ID, ' ||
                                         p_group_id ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_group_is_nbr;

  RETURN v_group_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM ||
                                        ' - in get group is number.');
     
        
    v_group_key.is_number := NULL;
    v_group_key.org_id := NULL;
    RETURN v_group_key;
  
END get_group_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_row_is_nbr
(
  p_row_id         IN VARCHAR2,
  p_group_key      IN t_storet_pk_type,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_row_id_key t_storet_pk_type;
  v_comp_cd    VARCHAR2(1);

  CURSOR c_get_row_id_is_nbr(p_row_id IN VARCHAR2, p_group_pk IN t_storet_pk_type) IS
    SELECT tsrchdef_is_number,
           tsrchdef_org_id,
           comp_ind_cd
      FROM tsrchdef,
           tsrccga
     WHERE tsrchdef.tsrccga_is_number = tsrccga.tsrccga_is_number
       AND tsrchdef.tsrccga_org_id = tsrccga.tsrccga_org_id
       AND row_id = p_row_id
       AND tsrccga.tsrchgrp_is_number = p_group_pk.is_number
       AND tsrccga.tsrchgrp_org_id = rpad(p_group_pk.org_id, 8);

BEGIN
  /* kms - function that gets the characteristic row primary key using the processed row id - called by validate storet item */
  OPEN c_get_row_id_is_nbr(p_row_id, p_group_key);
  FETCH c_get_row_id_is_nbr
    INTO v_row_id_key.is_number, v_row_id_key.org_id, v_comp_cd;
  IF c_get_row_id_is_nbr%NOTFOUND THEN
    v_row_id_key.is_number := NULL;
    v_row_id_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Characteristic Row ID, ' ||
                                         p_row_id ||
                                         ', could not be found in STORET or is not a row in the provided characteristic group.');
     
        
  ELSE
    IF v_comp_cd = 'N' THEN
      v_row_id_key.is_number := NULL;
      v_row_id_key.org_id := NULL;
      
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Characteristic Row ID, ' ||
                                         p_row_id ||
                                         ', is not a complete row.');
     
        
    END IF;
  END IF;
  CLOSE c_get_row_id_is_nbr;

  RETURN v_row_id_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM ||
                                        ' - in get row_id is number.');
     
        
    v_row_id_key.is_number := NULL;
    v_row_id_key.org_id := NULL;
    RETURN v_row_id_key;
  
END get_row_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_ha_row_is_nbr
(
  p_row_id         IN VARCHAR2,
  p_group_key      IN t_storet_pk_type,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_row_id_key t_storet_pk_type;

  CURSOR c_get_ha_row_id_is_nbr(p_row_id IN VARCHAR2, p_group_pk IN t_storet_pk_type) IS
    SELECT tsrhcsc_is_number,
           tsrhcsc_org_id
      FROM tsrhcsc
     WHERE tsrchgrp_is_number = p_group_pk.is_number
       AND tsrchgrp_org_id = rpad(p_group_pk.org_id, 8)
       AND rtrim(row_id) = p_row_id;

BEGIN
  /* sf - function that gets the Habatat Assessment characteristic row primary key
  using the processed row id - called by validate storet item */
  OPEN c_get_ha_row_id_is_nbr(p_row_id, p_group_key);
  FETCH c_get_ha_row_id_is_nbr
    INTO v_row_id_key.is_number, v_row_id_key.org_id;
  IF c_get_ha_row_id_is_nbr%NOTFOUND THEN
    v_row_id_key.is_number := NULL;
    v_row_id_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Habitat Assessment Row ID, ' ||
                                         p_row_id ||
                                         ', could not be found in STORET or is not a row in the provided HA characteristic group.');
     
        
  END IF;
  CLOSE c_get_ha_row_id_is_nbr;

  RETURN v_row_id_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM ||
                                        ' - in get_HA_row_id_is_nbr.');
     
        
    v_row_id_key.is_number := NULL;
    v_row_id_key.org_id := NULL;
    RETURN v_row_id_key;
  
END get_ha_row_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_procedure_is_nbr
(
  p_proc           IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_proc_key t_storet_pk_type;

  CURSOR c_proc_is_nbr(p_proc VARCHAR2, p_org_id VARCHAR2) IS
    SELECT tsrfldpr_is_number,
           tsrfldpr_org_id
      FROM tsrfldpr
     WHERE tsrfldpr_org_id = rpad(p_org_id, 8)
       AND id_code = rpad(p_proc, 10);

BEGIN
  /* kms - function that gets the field procedure primary key using the processed procedure id - called by validate storet item */
  OPEN c_proc_is_nbr(p_proc, p_org_id);
  FETCH c_proc_is_nbr
    INTO v_proc_key;
  IF c_proc_is_nbr%NOTFOUND THEN
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Sample Collection Procedure, ' ||
                                         p_proc ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_proc_is_nbr;

  RETURN v_proc_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        SQLERRM ||
                                        ' - in get_procedure_is_nbr.');
     
        
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;
  
END get_procedure_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_station_is_nbr
(
  p_station_id     IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_station_key   t_storet_pk_type;
  v_complete_flag VARCHAR2(1);

  CURSOR c_find_station(p_station_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmstatn_is_number,
           tsmstatn_org_id
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_id, 15)
       AND tsmstatn_org_id = rpad(p_org_id, 8);

BEGIN

  /* kms - 4 that gets the station primary key using the processed station id - called by validate storet item */
  OPEN c_find_station(p_station_id, p_org_id);
  FETCH c_find_station
    INTO v_station_key.is_number, v_station_key.org_id;

  IF c_find_station%NOTFOUND THEN
    v_station_key.is_number := NULL;
    v_station_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Station, ' || p_station_id ||
                                        ', could not be found in STORET.');
     
        
  ELSIF v_complete_flag = 'N' THEN
    v_station_key.is_number := NULL;
    v_station_key.org_id := 'XXX';
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                        'Station, ' || p_station_id ||
                                        ', is incomplete.');
     
        
  END IF;
  CLOSE c_find_station;

  RETURN v_station_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - PROC GET_STATION_IS_NBR,');
     
        
    v_station_key.is_number := NULL;
    v_station_key.org_id := NULL;
    RETURN v_station_key;
  
END get_station_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_personnel_is_nbr
(
  p_contact_id     IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_first_name   VARCHAR2(30);
  v_last_name    VARCHAR2(40);
  v_tsmpersn_key t_storet_pk_type;
  v_active       VARCHAR2(1);

  CURSOR c_find_contact(p_first_name IN VARCHAR2, p_last_name IN VARCHAR2) IS
    SELECT tsmpersn_is_number,
           tsmpersn_org_id,
           active_indicatr_cd
      FROM tsmpersn
     WHERE rpad(first_name, 30) = rpad(p_first_name, 30)
       AND rpad(last_name, 40) = rpad(p_last_name, 40)
       AND tsmpersn_org_id = rpad(p_org_id, 8);

BEGIN
  /* kms - function that gets personnel primary key using the processed personnel name.  Expects to get last
  name first and that it is separated from the first name with a space.  If personnel cannot be found, this
  function will create it in the STORET table.  Called by validate_storet_item. */

  /* divide contact id into first and last names
  - assumes first space is break between first and last names */
  v_last_name := rtrim(substr(p_contact_id, 1, instr(p_contact_id, ' ')));
  v_first_name := ltrim(substr(p_contact_id, instr(p_contact_id, ' ')));

  OPEN c_find_contact(v_first_name, v_last_name);
  FETCH c_find_contact
    INTO v_tsmpersn_key.is_number, v_tsmpersn_key.org_id, v_active;
  IF c_find_contact%NOTFOUND THEN
    v_tsmpersn_key.is_number := NULL;
    v_tsmpersn_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Personnel, ' || v_first_name || ' ' ||
                                         v_last_name ||
                                         ', could not be found in STORET.');
     
        
    /* create personnel record in STORET */
  ELSIF v_active = 'N' THEN
    v_tsmpersn_key.is_number := NULL;
    v_tsmpersn_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Contact, ' || v_first_name || ' ' ||
                                         v_last_name ||
                                         ', is listed as inactive in STORET.');
     
        
  END IF;
  CLOSE c_find_contact;

  RETURN v_tsmpersn_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - in get personnel is nbr.');
     
        
    v_tsmpersn_key.is_number := NULL;
    v_tsmpersn_key.org_id := NULL;
    RETURN v_tsmpersn_key;
  
END get_personnel_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_cporg_is_nbr
(
  p_cporg_name     IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_name         VARCHAR2(60);
  v_tsmcporg_key t_storet_pk_type;

  CURSOR c_tsmcporg(p_name IN VARCHAR2) IS
    SELECT tsmcporg_is_number,
           tsmcporg_org_id
      FROM tsmcporg t
     WHERE NAME = p_name;

BEGIN
  v_name := p_cporg_name;

  OPEN c_tsmcporg(v_name);
  FETCH c_tsmcporg
    INTO v_tsmcporg_key.is_number, v_tsmcporg_key.org_id;
  IF c_tsmcporg%NOTFOUND THEN
    v_tsmcporg_key.is_number := NULL;
    v_tsmcporg_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Cooperating Organization, ' ||
                                         v_name ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_tsmcporg;

  RETURN v_tsmcporg_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - in get Cooperating Organization IS_NUMBER.');
     
        
    v_tsmcporg_key.is_number := NULL;
    v_tsmcporg_key.org_id := NULL;
    RETURN v_tsmcporg_key;
  
END get_cporg_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_remark_codes_is_num
(
  p_short_name     IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  v_remark_key t_storet_pk_type;

  CURSOR c_get_is_nbr(p_short_name IN VARCHAR2) IS
    SELECT tsrlbrmk_is_number,
           tsrlbrmk_org_id
      FROM tsrlbrmk
     WHERE upper(rtrim(short_name)) = upper(rtrim(p_short_name));

BEGIN

  OPEN c_get_is_nbr(p_short_name);
  FETCH c_get_is_nbr
    INTO v_remark_key.is_number, v_remark_key.org_id;
  IF c_get_is_nbr%NOTFOUND THEN
    v_remark_key.is_number := NULL;
    v_remark_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Lab Remark Codes, ' ||
                                         p_short_name ||
                                         ', could not be found in STORET.');
     
        
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_remark_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - in get_remark_codes_is_num.');
     
        
    v_remark_key.is_number := NULL;
    v_remark_key.org_id := NULL;
    RETURN v_remark_key;
  
END get_remark_codes_is_num;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_char_is_nbr
(
  p_char_name      IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER,
  fromwhere        IN VARCHAR2
) RETURN t_storet_pk_type IS

  v_char_key t_storet_pk_type;

  -- jah 5-27-2004 Task 477
  -- Added OR SEARCH_NAME = UPPER(P_CHAR_NAME) to cursor C_GET_IS_NBR

  CURSOR c_get_is_nbr(p_char_name IN VARCHAR2) IS
    SELECT tsrchar_is_number,
           tsrchar_org_id
      FROM tsrchar
     WHERE display_name = p_char_name
        OR search_name = upper(p_char_name);

BEGIN
  /* kms - function that checks the characteristic in STORET using the processed display name.
  Called by validate storet item. */

  OPEN c_get_is_nbr(p_char_name);
  FETCH c_get_is_nbr
    INTO v_char_key;
  IF c_get_is_nbr%NOTFOUND THEN
    v_char_key.is_number := NULL;
    v_char_key.org_id := NULL;
    IF fromwhere = 'CHAR' THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                           'Characteristic, ' || p_char_name ||
                                            ', could not be found in STORET.');
       
        
    ELSIF fromwhere = 'SUBJ' THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                           'Subject Taxon, ' || p_char_name ||
                                            ', could not be found in STORET.');
       
          
    ELSE
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                           'Common Class Descriptor, ' ||
                                           p_char_name ||
                                           ', could not be found in STORET.');
       
          
    END IF;
  END IF;
  CLOSE c_get_is_nbr;

  RETURN v_char_key;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting characteristic is number');
     
        
    v_char_key.is_number := NULL;
    v_char_key.org_id := NULL;
    RETURN v_char_key;
  
END get_char_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_field_lab_procedure
(
  p_proc_id        IN VARCHAR2,
  p_source         IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_fa_seq         IN NUMBER
) RETURN t_storet_pk_type IS

  /* kms - function that gets the analysis procedure primary key using the procedure id
  and the source acronym  */

  v_proc_key    sim_import_result_extras.t_storet_pk_type;
  v_org_is_nbr  NUMBER(12);
  v_temp_is_nbr NUMBER(12);
  v_type_cd     VARCHAR2(2);

  e_not_yours EXCEPTION;
  e_bad_proc EXCEPTION;

  CURSOR c_get_is_nbr IS
    SELECT tsranlpr_is_number,
           tsranlpr_org_id
      FROM tsranlpr
     WHERE procedure_id = rpad(p_proc_id, 15)
       AND source_acr = rpad(p_source, 12)
       AND tsranlpr_org_id = rpad(p_org_id, 8);

  CURSOR c_get_is_nbr_natl IS
    SELECT tsranlpr_is_number,
           tsranlpr_org_id
      FROM tsranlpr
     WHERE procedure_id = rpad(p_proc_id, 15)
       AND source_acr = rpad(p_source, 12)
       AND tsranlpr_org_id = 'ZZZZZZZZ';

  CURSOR c_check_adopt(p_proc_is_nbr IN NUMBER, p_proc_org_id IN VARCHAR2, p_org_is_nbr IN NUMBER) IS
    SELECT tsmorgan_is_number
      FROM tsroapa
     WHERE tsranlpr_is_number = p_proc_is_nbr
       AND tsranlpr_org_id = rpad(p_proc_org_id, 8)
       AND tsmorgan_is_number = p_org_is_nbr;

  CURSOR c_get_is_nbr_others IS
    SELECT tsranlpr_is_number,
           tsranlpr_org_id
      FROM tsranlpr
     WHERE procedure_id = rpad(p_proc_id, 15)
       AND source_acr = rpad(p_source, 12);

BEGIN

  /* get prep is number for this id and org. If not found look in the national. */
  OPEN c_get_is_nbr;
  FETCH c_get_is_nbr
    INTO v_proc_key;
  IF c_get_is_nbr%NOTFOUND THEN
    OPEN c_get_is_nbr_natl;
    FETCH c_get_is_nbr_natl
      INTO v_proc_key;
    IF c_get_is_nbr_natl%NOTFOUND THEN
      OPEN c_get_is_nbr_others;
      FETCH c_get_is_nbr_others
        INTO v_proc_key;
      v_proc_key.is_number := NULL;
      v_proc_key.org_id := NULL;
      IF c_get_is_nbr_others%NOTFOUND THEN
        RAISE e_bad_proc;
      ELSE
        RAISE e_not_yours;
      END IF;
      CLOSE c_get_is_nbr_others;
    END IF;
    CLOSE c_get_is_nbr_natl;
  END IF;
  CLOSE c_get_is_nbr;

  /* get the org is number */
  v_org_is_nbr := get_org_info(p_org_id, p_import_log_seq);

  /* make sure analysis procedure has been adopted by this org */
  IF v_proc_key.is_number IS NOT NULL THEN
    OPEN c_check_adopt(v_proc_key.is_number, v_proc_key.org_id, v_org_is_nbr);
    FETCH c_check_adopt
      INTO v_temp_is_nbr;
    IF c_check_adopt%NOTFOUND THEN
      /* kms - set adoption type code to indicate whether or not the analysis procedure
      is owned by this organization */
      IF v_proc_key.org_id = 'ZZZZZZZZ' THEN
        v_type_cd := 'AD';
      ELSE
        v_type_cd := 'OW';
      END IF;
    
      /* ADDED BY SNG 3/5/01 - ADOPT THE PROCEDURE ...) */
      INSERT INTO tsroapa
        (tsmorgan_is_number,
         tsranlpr_is_number,
         tsranlpr_org_id,
         type_code,
         suspend_indicator,
         d_userid_code,
         d_last_update_ts)
      VALUES
        (v_org_is_nbr,
         v_proc_key.is_number,
         v_proc_key.org_id,
         v_type_cd,
         'N',
         sim_export_pkg.v_pkg_ownerid,
         trunc(SYSDATE));
    END IF;
    CLOSE c_check_adopt;
  END IF;

  RETURN v_proc_key;

EXCEPTION
  WHEN e_bad_proc THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Field/Lab Procedure ' || p_proc_id ||
                                         ' with Source Acronym, ' || p_source ||
                                         ' not found in STORET.');
     
        
    RETURN v_proc_key;
  WHEN e_not_yours THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'The Field/Lab Procedure ' ||
                                         p_proc_id ||
                                         ', is not valid: it is not a national procedure, nor does it belong to this organization.');
     
        
    RETURN v_proc_key;
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting analysis procedure is number');
     
        
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;
  
END get_field_lab_procedure;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_lab_sample_prep_procedure
(
  p_labprep_id     IN VARCHAR2,
  p_labprep_source IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_fa_seq         IN NUMBER
) RETURN t_storet_pk_type IS

  /* kms - function that gets the lab sample prep procedure primary key using the procedure id
  and the source acronym  */

  v_labprep_key sim_import_result_extras.t_storet_pk_type;
  v_org_is_nbr  NUMBER(12);
  v_temp_is_nbr NUMBER(12);
  v_type_cd     VARCHAR2(2);

  e_not_your_labprep EXCEPTION;
  e_bad_labprep EXCEPTION;

  CURSOR c_get_prep_source_is_nbr IS
    SELECT tsrlspp_is_number,
           tsrlspp_org_id
      FROM tsrlspp
     WHERE preparation_id = rpad(p_labprep_id, 15)
       AND source_acr = rpad(p_labprep_source, 12)
       AND tsrlspp_org_id = rpad(p_org_id, 8);

  CURSOR c_get_prep_source_is_nbr_natl IS
    SELECT tsrlspp_is_number,
           tsrlspp_org_id
      FROM tsrlspp
     WHERE preparation_id = rpad(p_labprep_id, 15)
       AND source_acr = rpad(p_labprep_source, 12)
       AND tsrlspp_org_id = 'ZZZZZZZZ';

  CURSOR c_get_prep_is_nbr_others IS
    SELECT tsrlspp_is_number,
           tsrlspp_org_id
      FROM tsrlspp
     WHERE preparation_id = rpad(p_labprep_id, 15)
       AND source_acr = rpad(p_labprep_source, 12);

  CURSOR c_check_adopt(p_prep_is_nbr IN NUMBER, p_prep_org_id IN VARCHAR2, p_org_is_nbr IN NUMBER) IS
    SELECT tsmorgan_is_number
      FROM tsrolspa
     WHERE tsrlspp_is_number = p_prep_is_nbr
       AND tsrlspp_org_id = rpad(p_prep_org_id, 8)
       AND tsmorgan_is_number = p_org_is_nbr;

BEGIN

  /* get prep is number for this id and org. If not found look in the national. */
  OPEN c_get_prep_source_is_nbr;
  FETCH c_get_prep_source_is_nbr
    INTO v_labprep_key;
  IF c_get_prep_source_is_nbr%NOTFOUND THEN
    OPEN c_get_prep_source_is_nbr_natl;
    FETCH c_get_prep_source_is_nbr_natl
      INTO v_labprep_key;
    IF c_get_prep_source_is_nbr_natl%NOTFOUND THEN
      OPEN c_get_prep_is_nbr_others;
      FETCH c_get_prep_is_nbr_others
        INTO v_labprep_key;
      v_labprep_key.is_number := NULL;
      v_labprep_key.org_id := NULL;
      IF c_get_prep_is_nbr_others%NOTFOUND THEN
        RAISE e_bad_labprep;
      ELSE
        RAISE e_not_your_labprep;
      END IF;
      CLOSE c_get_prep_is_nbr_others;
    END IF;
    CLOSE c_get_prep_source_is_nbr_natl;
  END IF;
  CLOSE c_get_prep_source_is_nbr;

  /* get the user's org is number */
  v_org_is_nbr := get_org_info(p_org_id, p_import_log_seq);

  /* make sure analysis procedure has been adopted by this org */
  IF v_labprep_key.is_number IS NOT NULL THEN
    OPEN c_check_adopt(v_labprep_key.is_number, v_labprep_key.org_id, v_org_is_nbr);
  
    FETCH c_check_adopt
      INTO v_temp_is_nbr;
    IF c_check_adopt%NOTFOUND THEN
      /* kms - set adoption type code to indicate whether or not the analysis procedure
      is owned by this organization */
      IF v_labprep_key.org_id = 'ZZZZZZZZ' THEN
        v_type_cd := 'AD';
      ELSE
        v_type_cd := 'OW';
      END IF;
    
      /* Last updated by DTW 11/12/03 - ADOPT THE PROCEDURE ...) */
      INSERT INTO tsrolspa
        (tsmorgan_is_number,
         tsrlspp_is_number,
         tsrlspp_org_id,
         type_code,
         d_userid_code,
         d_last_update_ts)
      VALUES
        (v_org_is_nbr,
         v_labprep_key.is_number,
         v_labprep_key.org_id,
         v_type_cd,
         sim_export_pkg.v_pkg_ownerid,
         trunc(SYSDATE));
    END IF;
    CLOSE c_check_adopt;
  END IF;

  RETURN v_labprep_key;

EXCEPTION
  WHEN e_bad_labprep THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Lab Sample Prep Procedure ' ||
                                         p_labprep_id ||
                                         ' with Source Acronym, ' ||
                                         p_labprep_source ||
                                         ' not found in STORET.');
     
        
    RETURN v_labprep_key;
  WHEN e_not_your_labprep THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'The Lab Sample Prep Procedure ' ||
                                         p_labprep_id ||
                                         ', is not valid: it is not a national procedure, nor does it belong to this organization.');
     
        
    RETURN v_labprep_key;
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting analysis procedure is number');
     
        
    v_labprep_key.is_number := NULL;
    v_labprep_key.org_id := NULL;
    RETURN v_labprep_key;
  
END get_lab_sample_prep_procedure;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_analproc_is_nbr
(
  p_procedure_id   IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
  to see if the Procedure ID existis so translation stuff can be done. Actual validation
  happens after the source is read in get_field_lab_procedure */

  v_proc_key t_storet_pk_type;

  CURSOR c_check_proc_id(p_proc_id IN VARCHAR2) IS
    SELECT tsranlpr_is_number,
           tsranlpr_org_id
      FROM tsranlpr
     WHERE procedure_id = rpad(p_proc_id, 15)
       AND (tsranlpr_org_id = rpad(p_org_id, 8) OR tsranlpr_org_id = 'ZZZZZZZZ');

BEGIN

  OPEN c_check_proc_id(p_procedure_id);
  FETCH c_check_proc_id
    INTO v_proc_key.is_number, v_proc_key.org_id;

  IF c_check_proc_id%NOTFOUND THEN
  
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Unable to find Field/Lab Procedure ' ||
                                          p_procedure_id || ' in STORET.');
     
        
  END IF;

  CLOSE c_check_proc_id;

  RETURN v_proc_key;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting Field/Lab Procedure is number');
     
        
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;
  
END get_analproc_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_analproc_source
(
  p_procedure_source_id IN VARCHAR2,
  p_org_id              IN VARCHAR2,
  p_fa_seq              IN NUMBER,
  p_line_cnt            IN NUMBER,
  p_import_log_seq      IN NUMBER
) RETURN t_storet_pk_type IS

  /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
  to see if the Procedure Source exists so translation stuff can be done. Actual validation
  happens after the source is read in get_field_lab_procedure */

  v_source_key t_storet_pk_type;

  CURSOR c_check_proc_source(p_procedure_acr IN VARCHAR2) IS
    SELECT tsranlpr_is_number,
           tsranlpr_org_id
      FROM tsranlpr
     WHERE source_acr = rpad(p_procedure_acr, 12);

BEGIN

  OPEN c_check_proc_source(p_procedure_source_id);
  FETCH c_check_proc_source
    INTO v_source_key.is_number, v_source_key.org_id;
  IF c_check_proc_source%NOTFOUND THEN
    v_source_key.is_number := NULL;
    v_source_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Unable to find Field/Lab Procedure Source acronym ' ||
                                         p_procedure_source_id ||
                                         ' in STORET.');
     
        
  END IF;
  CLOSE c_check_proc_source;

  RETURN v_source_key;

EXCEPTION

  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting Field/Lab Procedure Source acronym is number');
     
        
    v_source_key.is_number := NULL;
    v_source_key.org_id := NULL;
    RETURN v_source_key;
  
END get_analproc_source;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_labprep_source
(
  p_procedure_source_id IN VARCHAR2,
  p_org_id              IN VARCHAR2,
  p_fa_seq              IN NUMBER,
  p_line_cnt            IN NUMBER,
  p_import_log_seq      IN NUMBER
) RETURN t_storet_pk_type IS

  /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
  to see if the Lab Sample Prep Procedure Source existis so translation stuff can be done. Actual validation
  happens after the source is read in get_lab_sample_prep_procedure */

  v_prep_source_key t_storet_pk_type;

  CURSOR c_check_prep_proc_source(p_procedure_acr IN VARCHAR2) IS
    SELECT tsrlspp_is_number,
           tsrlspp_org_id
      FROM tsrlspp
     WHERE source_acr = rpad(p_procedure_acr, 12);

BEGIN

  OPEN c_check_prep_proc_source(p_procedure_source_id);
  FETCH c_check_prep_proc_source
    INTO v_prep_source_key.is_number, v_prep_source_key.org_id;
  IF c_check_prep_proc_source%NOTFOUND THEN
    v_prep_source_key.is_number := NULL;
    v_prep_source_key.org_id := NULL;
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Unable to find Lab Sample Prep Procedure Source acronym ' ||
                                         p_procedure_source_id ||
                                         ' in STORET.');
     
        
  END IF;
  CLOSE c_check_prep_proc_source;

  RETURN v_prep_source_key;

EXCEPTION

  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting Lab Sample Prep Procedure Source acronym is number');
     
        
    v_prep_source_key.is_number := NULL;
    v_prep_source_key.org_id := NULL;
    RETURN v_prep_source_key;
  
END get_labprep_source;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_labprep_is_nbr
(
  p_preparation_id IN VARCHAR2,
  p_org_id         IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN t_storet_pk_type IS

  /* SF - 10/7/02 - Now we are using the source as well as the ID. So this routine JUST checks
  to see if the Lab Sample Prep Procedure ID existis so translation stuff can be done. Actual validation
  happens after the source is read in get_lab_sample_prep_procedure */

  v_prep_key t_storet_pk_type;

  CURSOR c_check_prep_proc_id(p_prep_procedure_id IN VARCHAR2) IS
    SELECT tsrlspp_is_number,
           tsrlspp_org_id
      FROM tsrlspp
     WHERE preparation_id = rpad(p_prep_procedure_id, 15)
       AND (tsrlspp_org_id = rpad(p_org_id, 8) OR tsrlspp_org_id = 'ZZZZZZZZ');

BEGIN

  OPEN c_check_prep_proc_id(p_preparation_id);
  FETCH c_check_prep_proc_id
    INTO v_prep_key.is_number, v_prep_key.org_id;
  IF c_check_prep_proc_id%NOTFOUND THEN
    v_prep_key.is_number := NULL;
    v_prep_key.org_id := NULL;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Unable to find Lab Sample Prep Procedure ' ||
                                          p_preparation_id || ' in STORET.');
     
        
  END IF;
  CLOSE c_check_prep_proc_id;

  RETURN v_prep_key;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                        ' - getting Lab Sample Prep Procedure is number');
     
        
    v_prep_key.is_number := NULL;
    v_prep_key.org_id := NULL;
    RETURN v_prep_key;
  
END get_labprep_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_trip_id
(
  p_instr_code      IN VARCHAR2,
  p_start_date      IN DATE,
  p_trip_start_date OUT DATE,
  p_trip_end_date   OUT DATE,
  p_org_id          IN VARCHAR2,
  p_fa_seq          IN NUMBER,
  p_line_cnt        IN NUMBER,
  p_import_log_seq  IN NUMBER
) RETURN VARCHAR2 IS

  v_trip_id VARCHAR2(15);

BEGIN
  /* kms - function that will get the trip primary key based on instructions - id will be based either
   on year, month and year, date, or on what's in the text file.  Will create trip only if the id
  cannot be found in STORET. */

  /* one trip per year */
  IF p_instr_code = 'RES-TRIP-01' THEN
    v_trip_id := to_char(p_start_date, 'RRRR');
  
    /* trip starts on the first day of the year */
    p_trip_start_date := to_date('1-1-' || to_char(trunc(p_start_date, 'RRRR'), 'RRRR'), 'MM-DD-RRRR');
  
    /* trip ends on the last day of the year */
    p_trip_end_date := to_date('12-31-' || to_char(trunc(p_start_date, 'RRRR'), 'RRRR'), 'MM-DD-RRRR');
  
    /* one trip per month */
  ELSIF p_instr_code = 'RES-TRIP-02' THEN
    v_trip_id := to_char(p_start_date, 'MMRRRR');
  
    /* trip starts on the first day of the month */
    p_trip_start_date := trunc(p_start_date, 'MON');
  
    /* trip ends on the last day of the month */
    p_trip_end_date := last_day(p_start_date);
  
    /* one trip per day */
  ELSIF p_instr_code = 'RES-TRIP-03' THEN
    v_trip_id := to_char(p_start_date, 'MMDDRRRR');
  
    /* trip starts the same day */
    p_trip_start_date := p_start_date;
  
    /* trip ends same day */
    p_trip_end_date := p_start_date;
  
  END IF;

  RETURN v_trip_id;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM || ' - in get trip id.');
     
        
    p_trip_start_date := NULL;
    p_trip_end_date := NULL;
    v_trip_id := NULL;
    RETURN v_trip_id;
  
END get_trip_id;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_visit_start_date
(
  p_visit_key      IN t_storet_pk_type,
  p_trip_seq       IN NUMBER,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN DATE IS

  v_start_date DATE;

  -- retrieve trip from SIM
  CURSOR c_get_date_from_trip(p_trip_seq IN NUMBER) IS
    SELECT stp_start_date
      FROM sim_trips
     WHERE stp_seq = p_trip_seq;

BEGIN
  /* kms (5/30/01)- function that gets the station visit start date from the station
  visit if it exists in STORET or from the trip. */

  /* kms (6/02/01 - changed this so it only gets it from the trip.  Since station
  visits are now loaded into SIM when a trip is first encountered by the import, we
  know that all of the station visits that exist in STORET are also in SIM.  The
  start date, at this point is only needed if the visit does not exist in SIM (and
  is therefore not in STORET). */

  /* kms - station visit does not exist yet, get date from trip in SIM table.
  This will be replaced later if the station visit start date has been
  included. */

  -- try to retrieve the Trip from SIM
  OPEN c_get_date_from_trip(p_trip_seq);
  FETCH c_get_date_from_trip
    INTO v_start_date;

  -- did we retrieve the Trip from SIM?
  IF c_get_date_from_trip%NOTFOUND THEN
    -- if no
  
    -- Tell calling routine Trip not found      
    v_start_date := NULL;
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         'Error retrieving start date from SIM_TRIPS.');
     
        
  END IF;
  CLOSE c_get_date_from_trip;

  RETURN v_start_date;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM ||
                                         ' - in get visit start date.');
     
        
    -- tell calling routine Trip not found
    RETURN NULL;
  
END get_visit_start_date;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_visit_seq
(
  p_visit_key        OUT t_storet_pk_type,
  p_station_key      IN t_storet_pk_type,
  p_visit_start_date IN DATE,
  p_visit_id         IN VARCHAR2,
  p_trip_seq         IN NUMBER,
  p_fa_seq           IN NUMBER,
  p_line_cnt         IN NUMBER,
  p_import_log_seq   IN NUMBER
) RETURN NUMBER IS
  -- jah 3-8-2004 Added Parm P_VISIT_START_DATE

  /* 
     Purpose
       This function tries to find a station visit record in SIM_STATION_VISITS.
       In addition, it uses the data passed by the calling routine to update records 
       in SIM_STATION_VISITS that have an UNAssigned Station Visit ID.
       
       It tries twice to find the station visit record. First, it looks for a 
       record with a matching Trip, Station, and Station Visit ID Number. Second, it 
       looks for a record with a matching Trip, Station, Station Visit Start Date, and
       'UNA' in the first three letters of the Station Visit ID Number.
       
       If it finds a matching UNA record, this function changes the Station Visit ID
       Number in the record to the Station Visit ID passed by the calling routine.
       
     Parameters
       P_TRIP_SEQ          - Foreign Key to a Trip record
       P_VISIT_ID          - Station Visit ID from the import data record
       P_VISIT_START_DATE  - Station Visit Start Date from the import data record
       
     OUT Parameters
       P_VISIT_KEY         - STORET record key for the Station Visit record
       
     Return
       This function returns the key (sequence number) of the matching record in
       SIM_STATION_VISITS; or NULL if it cannot find a matching record.
       
     Remarks
       1. Pass the Station Visit ID Number and Station Visit Start Date from the 
          import data record. This means you should only call this function when the
          import instruction for the station visit is "Read existing station visit
          from (import) data file."
  */

  v_visit_seq NUMBER(10);

  -- Retrieve station visit using Trip, Station and Station Visit ID
  CURSOR c_find_station_visit(cp_visit_nbr IN VARCHAR2, cp_tsmstatn_is_nbr IN NUMBER, cp_tsmstatn_org_id IN VARCHAR2, cp_trip_seq IN NUMBER) IS
    SELECT ssv_seq,
           ssv_tsrstvst_is_number,
           ssv_tsrstvst_org_id
      FROM sim_station_visits
     WHERE ssv_id = cp_visit_nbr
       AND ssv_tsmstatn_is_number = cp_tsmstatn_is_nbr
       AND ssv_tsmstatn_org_id = rpad(cp_tsmstatn_org_id, 8)
       AND ssv_stp_seq = cp_trip_seq;

  -- Retrieve UNAssigned station visit for a given date
  CURSOR c_find_una_station_visit(cp_visit_start_date IN DATE, cp_tsmstatn_is_nbr IN NUMBER, cp_tsmstatn_org_id IN VARCHAR2, cp_trip_seq IN NUMBER) IS
    SELECT ssv_seq,
           ssv_tsrstvst_is_number,
           ssv_tsrstvst_org_id
      FROM sim_station_visits
     WHERE substr(ssv_id, 1, 3) = 'UNA'
       AND ssv_start_date = cp_visit_start_date
       AND ssv_start_time IS NULL
       AND ssv_tsmstatn_is_number = cp_tsmstatn_is_nbr
       AND ssv_tsmstatn_org_id = rpad(cp_tsmstatn_org_id, 8)
       AND ssv_stp_seq = cp_trip_seq;

BEGIN

  -- Try to retrieve the station visit record using the trip station visit ID in the import data record.
  OPEN c_find_station_visit(p_visit_id, p_station_key.is_number, p_station_key.org_id, p_trip_seq);
  FETCH c_find_station_visit
    INTO v_visit_seq, p_visit_key.is_number, p_visit_key.org_id;

  -- did we find the station visit record using the visit ID?        
  IF c_find_station_visit%FOUND THEN
    -- if yes
  
    CLOSE c_find_station_visit;
  
  ELSE
    -- if no
  
    CLOSE c_find_station_visit;
  
    -- Try to find an UNAssigned station visit that matches
    OPEN c_find_una_station_visit(p_visit_start_date, p_station_key.is_number, p_station_key.org_id, p_trip_seq);
    FETCH c_find_una_station_visit
      INTO v_visit_seq, p_visit_key.is_number, p_visit_key.org_id;
  
    -- did we find an UNAssigned station visit that matches?
    IF c_find_una_station_visit%FOUND THEN
      -- if yes            
    
      /* jah 3-7-2004
      Update the "UNA"ssigned Station Visit record in SIM_STATION_VISITS. */
      UPDATE sim_station_visits
         SET ssv_id = p_visit_id
       WHERE ssv_seq = v_visit_seq;
    
    ELSE
      -- if no (i.e., we did NOT find an UNAssigned station visit that matches)
    
      -- tell calling routine we did not find a matching record
      v_visit_seq := NULL;
    
    END IF;
    CLOSE c_find_una_station_visit;
  
  END IF;

  RETURN v_visit_seq;

EXCEPTION

  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         SQLERRM || ' - in get visit seq.');
     
        
    -- Tell calling routine this function could not find a matching station visit record.
    RETURN NULL;
  
END get_visit_seq;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_org_info
(
  p_org_id         IN VARCHAR2,
  p_import_log_seq IN NUMBER
) RETURN NUMBER IS

  v_org_is_nbr NUMBER(8);

  CURSOR c_get_tsmorgan_nbr(p_cursor_org_id IN VARCHAR2) IS
    SELECT tsmorgan_is_number
      FROM tsmorgan
     WHERE org_id = rpad(p_cursor_org_id, 8);

BEGIN
  /* kms - function that gets the is_number for the organization running the import.  Called by main
  import procedure - import_results */

  OPEN c_get_tsmorgan_nbr(p_org_id);
  FETCH c_get_tsmorgan_nbr
    INTO v_org_is_nbr;
  IF c_get_tsmorgan_nbr%NOTFOUND THEN
    v_org_is_nbr := NULL;
     
                                                                                                                                                                                                                                                                                          
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         'Failed to get organization is number.');
     
        
  END IF;
  CLOSE c_get_tsmorgan_nbr;

  RETURN v_org_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    raise_application_error(-20199, 'Failed to get organization is number.');
    RETURN NULL;
  
END get_org_info;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_uom_is_nbr
(
  p_uom_name       IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER
) RETURN sim_import_result_extras.t_storet_pk_type IS

  TYPE t_uom_record IS RECORD(
    uom_is_nbr NUMBER(8),
    uom_org_id VARCHAR2(8),
    uom_name   VARCHAR2(10));

  v_uom_info t_uom_record;
  v_uom_key  sim_import_result_extras.t_storet_pk_type;

  e_unk_uom EXCEPTION;
  e_uom_sp EXCEPTION;

  CURSOR c_get_is_nbr(p_uom_name IN VARCHAR2) IS
    SELECT tsruom_is_number,
           tsruom_org_id,
           short_form_name
      FROM tsruom
     WHERE upper(short_form_name) = upper(rpad(p_uom_name, 10));

BEGIN
  /* kms - function that gets the uom primary key using the processed uom name - called by validate storet item
  Also checks the case of the unit of measure name and issues a warning if it is not correct */

  /* get tsruom key for unit of measure */
  OPEN c_get_is_nbr(p_uom_name);
  FETCH c_get_is_nbr
    INTO v_uom_info;
  IF c_get_is_nbr%NOTFOUND THEN
  
    /* if uom could not be found then raise an exception to write error to error log, will change status in main procedure */
    v_uom_info.uom_is_nbr := NULL;
    v_uom_info.uom_org_id := NULL;
    CLOSE c_get_is_nbr;
    RAISE e_unk_uom;
  
  ELSE
  
    IF rpad(v_uom_info.uom_name, 10) != rpad(p_uom_name, 10) THEN
      CLOSE c_get_is_nbr;
      RAISE e_uom_sp;
    END IF;
  
  END IF;
  CLOSE c_get_is_nbr;

  v_uom_key.is_number := v_uom_info.uom_is_nbr;
  v_uom_key.org_id := v_uom_info.uom_org_id;

  RETURN v_uom_key;

EXCEPTION
  WHEN e_unk_uom THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unknown Unit of Measure: ' ||
                                         p_uom_name);
     
        
    RETURN v_uom_key;
  
  WHEN e_uom_sp THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unit of Measure misspelled: ' ||
                                         p_uom_name ||
                                         ' - check capitalization.');
     
        
    RETURN v_uom_key;
  
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get uom is number.');
     
        
    v_uom_key.is_number := NULL;
    v_uom_key.org_id := NULL;
    RETURN v_uom_key;
  
END get_uom_is_nbr;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_number
(
  p_item           IN VARCHAR2,
  p_item_name      IN VARCHAR2,
  p_format         IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_line_cnt       IN NUMBER
) RETURN NUMBER IS

  -- Suspend numeric checking to facilitate changes to non-numerics
  -- V_RETURN NUMBER(10);

BEGIN
  -- V_RETURN := TO_NUMBER(P_ITEM, P_FORMAT);
  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| p_line_cnt,
                                         p_item_name || ', ' || p_item ||
                                         ', is not in an appropriate numeric format of ' ||
                                         p_format || '.');
     
        
    RETURN 0;
  
END validate_number;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_allowable_value
(
  p_alval_seq      IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  v_item_name VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
  END IF;
  CLOSE c_get_alval_name;

  RETURN v_item_name;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                         ' - getting allowable value from sequence.');
     
        
    RETURN NULL;
END get_allowable_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_ha_default
(
  v_tsrhcsc_pk     IN sim_import_result_extras.t_storet_pk_type,
  p_fa_seq         IN NUMBER,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER
) RETURN VARCHAR2 IS

  v_char_name VARCHAR2(60);

  CURSOR c_get_ha_info IS
    SELECT characterstc_name
      FROM tsrhcsc
     WHERE tsrhcsc_is_number = v_tsrhcsc_pk.is_number
       AND tsrhcsc_org_id = v_tsrhcsc_pk.org_id;

BEGIN

  OPEN c_get_ha_info;
  FETCH c_get_ha_info
    INTO v_char_name;
  IF c_get_ha_info%NOTFOUND THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'HA Characteristic Name not found ');
     
        
    RETURN NULL;
  END IF;
  CLOSE c_get_ha_info;

  RETURN v_char_name;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Error getting HA Characteristic Name ');
     
        
    RETURN NULL;
  
END get_ha_default;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION update_result
(
  p_result           IN VARCHAR2,
  p_char_is_nbr      IN NUMBER,
  p_units_ok         OUT BOOLEAN,
  p_result_ok        OUT BOOLEAN,
  p_imp_cfg_seq      IN NUMBER,
  p_fa_seq           IN NUMBER,
  v_char_result_type IN VARCHAR2,
  v_import_log_seq   IN NUMBER,
  v_line_cnt         IN NUMBER,
  v_detect           IN VARCHAR2,
  v_detect_check     OUT NUMBER
) RETURN VARCHAR2 IS

  v_result sim_results.rs_result%TYPE;

BEGIN
  -- default to TRUE then set to FALSE where applicable
  p_units_ok := TRUE;
  p_result_ok := TRUE;
  /* kms - function that gets characteristic type in order to see if it has permitted values
  or if other checks need to be performed on the result. */

  IF v_char_result_type = 'PV  ' THEN
    /* kms (5/29/01) - result must be one of the permitted values for this characteristic. */
    v_result := validate_pv_result(p_result, p_imp_cfg_seq, p_char_is_nbr, p_fa_seq, p_result_ok, v_import_log_seq, v_line_cnt);
    p_units_ok := FALSE;
      
  ELSIF v_char_result_type = 'TEXT' THEN
    /* dcl (08/29/01) - result must be one of the permitted values for this characteristic. */
    v_result := '*Text';
    p_units_ok := FALSE;
    
  ELSE
    --sf - PHYS characteristics can't have *Non-detect, etc.
    IF v_char_result_type != 'PHYS' THEN
      --check to see if their results match their detection condition
      v_detect_check := 1;
      IF v_detect IS NOT NULL THEN
        IF v_detect = 'Detected not Quantified'
           AND p_result != '*Present' THEN
          v_result := '*Present';
          v_detect_check := 0;
        END IF;
        IF v_detect = 'Not Detected'
           AND p_result != '*Non-detect' THEN
          v_result := '*Non-detect';
          v_detect_check := 0;
        END IF;
        IF v_detect = 'Present above Quantification'
           AND p_result != '*Present >QL' THEN
          v_result := '*Present >QL';
          v_detect_check := 0;
        END IF;
        IF v_detect = 'Present below Quantification'
           AND p_result != '*Present <QL' THEN
          v_result := '*Present <QL';
          v_detect_check := 0;
        END IF;
        IF v_detect = 'Not Reported'
           AND p_result != '*Not Reported' THEN
          v_result := '*Not Reported';
          v_detect_check := 0;
        END IF;
        IF v_detect = 'Detected and Quantified' THEN
          v_result := p_result;
        END IF;
      ELSE
        v_result := p_result;
      END IF;
    
      IF (p_result = '*Non-detect' OR p_result = '*Present <QL' OR
         p_result = '*Present >QL' OR p_result = '*Not Reported' OR
         p_result = '*Present') THEN
        /* kms (5/29/01) - result value not reported, units are not ok. **/
        v_result := p_result;
        p_units_ok := FALSE;
      ELSIF (v_result = '*Non-detect' OR v_result = '*Present <QL' OR
            v_result = '*Present >QL' OR v_result = '*Not Reported' OR
            v_result = '*Present') THEN
        p_units_ok := FALSE;
      END IF;
    
    ELSE
      -- Physical Characteristics cannot have detection conditions for result
    
      IF (p_result = '*Non-detect' OR p_result = '*Present <QL' OR
         p_result = '*Present >QL' OR p_result = '*Not Reported' OR
         p_result = '*Present') THEN
      
        v_result := p_result;
        p_result_ok := FALSE;
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             'Physical Characteristics cannot have a value of : "' ||
                                              p_result || '"');
         
            
      ELSE
        v_result := p_result;
      END IF;
    
    END IF;
  
  END IF;

  RETURN v_result;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in get char type. vres = ' ||
                                         v_result);
     
        
    RETURN NULL;
  
END update_result;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_pv_result
(
  p_orig_item      IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_char_is_nbr    IN NUMBER,
  p_fa_seq         IN NUMBER,
  p_ok             OUT BOOLEAN,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN;
  v_instr_code     VARCHAR2(15);

  v_dummy   NUMBER(8);
  v_result  sim_results.rs_result%TYPE;
  v_col_seq NUMBER(10);

  e_no_col_seq EXCEPTION;

  CURSOR c_get_result_col_seq IS
    SELECT sic_seq
      FROM sim_imp_cols col,
           sim_imp_cfg cfg
     WHERE col.sic_name = 'Result Value'
       AND col.sic_sid_seq = cfg.sicc_sid_seq
       AND sicc_seq = p_imp_cfg_seq; /* make sure we use the right SIC Result Value column  */

  CURSOR c_check_char_pv(p_result IN VARCHAR2, p_char_is_nbr IN NUMBER) IS
    SELECT tsrcpv_is_number
      FROM tsrcpv
     WHERE upper(ltrim(rtrim(short_name))) = upper(ltrim(rtrim(p_result)))
       AND tsrchar_is_number = p_char_is_nbr;

BEGIN
  /* kms - function called if the result has a characteristic that indicates that it
  must be one of a list of allowable values.  Checks result to make sure it is has
  been checked for a translation. */
  /* kms (7/16/01) - added code that checks translations and will create them if the result
  value is not a permitted value and the instructions have been set to create translations. */

  /* kms (7/16/01) - get column sequence for results in order to see if any translations are
  set up for the result column. */

  p_ok := TRUE;

  OPEN c_get_result_col_seq;
  FETCH c_get_result_col_seq
    INTO v_col_seq;
  IF c_get_result_col_seq%NOTFOUND THEN
    CLOSE c_get_result_col_seq;
    RAISE e_no_col_seq;
  END IF;
  CLOSE c_get_result_col_seq;

  /* look in translations table to see if it needs to swapped
  10/1/03 pwh - added the trim function to this section.*/
  v_converted_item := TRIM(sim_import_pkg.check_translations(p_orig_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist));

  IF v_converted_item IS NOT NULL THEN
  
    /* check result to see if it is one of the permitted values */
    OPEN c_check_char_pv(v_converted_item, p_char_is_nbr);
    FETCH c_check_char_pv
      INTO v_dummy;
    IF c_check_char_pv%FOUND THEN
      v_result := v_converted_item;
    ELSE
      v_result := NULL;
      p_ok := FALSE;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Result Value, ' ||
                                           v_converted_item ||
                                           ' is not an allowed value for this characteristic.');
       
          
    END IF;
    CLOSE c_check_char_pv;
  
    IF v_dummy IS NULL
       AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
    
      IF v_instr_code = 'RES-TR-01' THEN
      
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, v_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
      
      END IF;
    
    ELSIF v_dummy IS NULL
          AND v_trans_exist THEN
    
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Result Value, ' || p_orig_item ||
                                           ', has been mapped to an invalid storet value: ' ||
                                           v_converted_item);
       
          
      p_ok := FALSE;
    END IF;
  
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         p_orig_item ||
                                        ' exists in translation table but has not been mapped to a STORET code. ');
     
        
    p_ok := FALSE;
  END IF;

  RETURN v_result;

EXCEPTION
  WHEN e_no_col_seq THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unable to find sequence for result column in function ''validate_pv_result''.');
     
        
    RETURN p_orig_item;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                        ' - in validate pv_result.');
     
        
    RETURN p_orig_item;
  
END validate_pv_result;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_fraction
(
  p_samp_frac_nm   IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_tsrchar_is_nbr IN NUMBER,
  p_fa_seq         IN NUMBER,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER,
  v_tsmprmvl_key   OUT t_storet_pk_type
) RETURN VARCHAR2 IS

  v_sample_frac VARCHAR2(40);
  v_samp_req_cd VARCHAR2(1);
  v_is_nbr      NUMBER(12);
  v_org_id      VARCHAR2(8);

  v_trans_exist    BOOLEAN := TRUE;
  v_converted_item VARCHAR2(80);
  v_instr_code     VARCHAR2(15);
  v_storet_pk      sim_import_result_extras.t_storet_pk_type;

  e_invalid_fraction EXCEPTION;
  e_sample_frac_sp EXCEPTION;

  CURSOR c_get_samp_req_cd(p_tsrchar_is_nbr IN NUMBER) IS
    SELECT samp_frac_req_cd
      FROM tsrchar
     WHERE tsrchar_is_number = p_tsrchar_is_nbr;

  CURSOR c_check_def_frac(p_fraction IN VARCHAR2) IS
    SELECT field_value,
           tsmprmvl_is_number,
           tsmprmvl_org_id
      FROM tsmprmvl
     WHERE field_name = 'SMPL_FRAC_TYPE_NM'
       AND table_name = 'TSRRSULT'
       AND field_value = p_fraction;

BEGIN

  /* get the char group is number - check translations and STORET tables */

  v_storet_pk := sim_import_result_extras.check_storet_translation(p_samp_frac_nm, p_imp_cfg_seq, p_col_seq, v_trans_exist, v_import_log_seq, v_line_cnt, v_converted_item, p_fa_seq);

  IF v_converted_item IS NOT NULL
     OR p_samp_frac_nm IS NULL THEN
  
    /* set sample fraction to null if it is full of spaces */
    IF rpad(v_converted_item, 15) = rpad(' ', 15)
       OR p_samp_frac_nm IS NULL THEN
      v_sample_frac := NULL;
    ELSE
      v_sample_frac := v_converted_item;
    END IF;
  
    OPEN c_get_samp_req_cd(p_tsrchar_is_nbr);
    FETCH c_get_samp_req_cd
      INTO v_samp_req_cd;
    IF v_samp_req_cd = 'Y'
       OR v_sample_frac IS NOT NULL THEN
    
      /* see if fraction is valid if it is required or if it has been provided */
      OPEN c_check_def_frac(v_converted_item);
      FETCH c_check_def_frac
        INTO v_sample_frac, v_is_nbr, v_org_id;
      IF c_check_def_frac%NOTFOUND THEN
        /* if fraction could not be found in permitted values table, return null and write error */
        v_sample_frac := NULL;
        v_is_nbr := NULL;
        v_org_id := NULL;
        IF NOT v_trans_exist THEN
          /* check instructions to see if translation record should be created */
          v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
        
          IF v_instr_code = 'RES-TR-01' THEN
          
            /* if record could not be matched to anything in translations or allowable values tables, create new record */
            create_trans_record(p_samp_frac_nm, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                 'Sample Fraction, ' ||
                                                 v_converted_item ||
                                                 ' is not a valid sample fraction - translation record has been created.');
             
                
          ELSE
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                 'Sample Fraction, ' ||
                                                 v_converted_item ||
                                                 ' is not a valid sample fraction.');
             
                
          END IF;
        
        ELSIF v_trans_exist THEN
        
          /* item has been translated to an invalid storet value */
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                               p_samp_frac_nm ||
                                               ' has been mapped to the invalid storet value: ' ||
                                               v_converted_item);
           
              
        END IF;
      
      ELSE
        v_sample_frac := rtrim(v_sample_frac);
      END IF;
      CLOSE c_check_def_frac;
    ELSIF v_samp_req_cd = 'N' THEN
      v_sample_frac := 'NR'; --Not Required
    END IF;
    CLOSE c_get_samp_req_cd;
  
  ELSIF v_trans_exist
        AND v_converted_item IS NULL THEN
  
    /* if there is a code in the translations table but no value to translate to,
    write error message.
    status will be changed to 'N' later in the program */
    v_sample_frac := NULL;
    v_is_nbr := NULL;
    v_org_id := NULL;
  
  END IF;
  v_tsmprmvl_key.is_number := v_is_nbr;
  v_tsmprmvl_key.org_id := v_org_id;
  RETURN v_sample_frac;

EXCEPTION
  WHEN e_invalid_fraction THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Fraction, ' || p_samp_frac_nm ||
                                          ', is not a valid STORET value.');
     
        
    RETURN NULL;
  
  WHEN e_sample_frac_sp THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Check fraction capitalization: (STORET/USER)' ||
                                         p_samp_frac_nm || '/' ||
                                         v_sample_frac);
     
        
    RETURN v_sample_frac;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                        ' - in validate sample fraction.');
     
        
    RETURN NULL;
  
END validate_fraction;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_alval_item
(
  p_item_name      IN VARCHAR2,
  p_orig_item      IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_fa_seq         IN NUMBER,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER
) RETURN NUMBER IS

  v_converted_item VARCHAR2(80);
  v_item_sia_seq   NUMBER(10);
  v_trans_exist    BOOLEAN := TRUE;
  v_instr_code     VARCHAR2(15);

BEGIN

  -- if translation exists, return translation; otherwise return original value
  v_converted_item := sim_import_pkg.check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_line_cnt, v_trans_exist);

  -- did the call to CHECK_TRANSLATIONS work properly?
  IF (v_converted_item IS NOT NULL) THEN
    -- if yes
  
    -- jah 2-25-04
    -- Does the STORET GUI convert this item to uppercase?
    IF p_item_name IN ('Point Type', 'Result Status') THEN
      -- if yes
      -- Convert to upper case to mimic STORET GUI
      v_converted_item := upper(v_converted_item);
    END IF;
  
    -- Get the primary key of the value in SIM_IMP_ALLOWABLE_VALUES table (NULL = invalid value)
    v_item_sia_seq := sim_import_pkg.check_allowable_values(v_converted_item, p_col_seq, v_line_cnt);
  
    -- is the value for this item invalid AND did CHECK_TRANSLATIONS return the original value?
    IF (v_item_sia_seq IS NULL)
       AND (NOT v_trans_exist) THEN
      -- if yes
    
      -- check instructions to see if translation record should be created
      v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
    
      -- should translation record be created?
      IF v_instr_code = 'RES-TR-01' THEN
        -- if yes
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
      
        -- Tell the end-user what happened
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ', is NOT an allowable value so a translation record has been created.');
         
            
      ELSE
        -- if no (a translation record should not be created)
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ', is NOT an allowable value.');
         
            
      END IF;
    
      -- is the value for this item invalid AND did CHECK_TRANSLATIONS return a translated value
      -- AND is the item name something other than 'Actual'?
    ELSIF (v_item_sia_seq IS NULL)
          AND (v_trans_exist)
          AND (p_orig_item != 'Actual') THEN
      -- if yes
      -- tell end-user the original item value translated to an invalid STORET value
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            p_item_name || ', ' || p_orig_item ||
                                           ', has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
       
          
    END IF;
  
  END IF;

  RETURN v_item_sia_seq;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate allowable values.');
     
        
    RETURN NULL;
  
END validate_alval_item;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_storet_item
(
  p_item_name       IN VARCHAR2,
  p_orig_item       IN VARCHAR2,
  p_imp_cfg_seq     IN NUMBER,
  p_col_seq         IN NUMBER,
  p_parent_pk       IN sim_import_result_extras.t_storet_pk_type,
  p_org_id          IN VARCHAR2,
  v_line_cnt        IN NUMBER,
  p_fa_seq          IN NUMBER,
  v_import_log_seq  IN NUMBER,
  p_translated_item OUT VARCHAR2
) -- jah 3-3-2004
 RETURN sim_import_result_extras.t_storet_pk_type IS

  v_converted_item VARCHAR2(4000); -- jah 3-3-2004
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_pk      sim_import_result_extras.t_storet_pk_type;
  v_instr_code     VARCHAR2(15);
  v_trans_ok       BOOLEAN := TRUE;
  sys_desc         VARCHAR2(40);

  e_no_function EXCEPTION;
  e_field_size_error EXCEPTION;

BEGIN

  /* kms - function to look in the translations and STORET tables for items that should already exist as
  hard coded values in STORET tables */

  /* kms (5/19/01)- added p_parent_pk to parameters in order to allow validation to be dependent upon the
  presence of a valid foreign key to another STORET table, i.e. gear */

  /* look in translations table to see if it needs to swapped */
  IF p_item_name IN ('Characteristic Row ID', 'Habitat Assessment Row ID',
      'Field/Lab Procedure', 'Field/Lab Procedure Source',
      'Lab Sample Prep Procedure', 'Lab Sample Prep Procedure Source') THEN
  
    v_converted_item := sim_import_pkg.check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_line_cnt, v_trans_exist);
  
  ELSE
  
    v_storet_pk := check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, v_import_log_seq, v_line_cnt, v_converted_item, -- OUT
                                            p_fa_seq);
  
  END IF;

  /* kms - IF..THEN statement that calls function for the item being validated. */
  IF v_converted_item IS NOT NULL THEN
    --PK IF START
  
    IF v_storet_pk.is_number IS NULL THEN
      --TRANS IF START
    
      /* call appropriate function to look in STORET table */
      IF p_item_name = 'Project ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Project ID to uppercase
        v_storet_pk := sim_import_result_extras.get_proj_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Laboratory ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Laboratory ID to uppercase
        v_storet_pk := sim_import_result_extras.get_lab_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Gear ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Gear ID to uppercase
        v_storet_pk := sim_import_result_extras.get_gear_is_nbr(upper(v_converted_item), p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Gear Configuration ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Gear Configuration ID to uppercase
        v_storet_pk := sim_import_result_extras.get_gear_cfg_is_nbr(upper(v_converted_item), p_parent_pk, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Sample Preservation Transport and Storage ID'
            OR p_item_name = 'Sample Transport and Storage ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts these to uppercase
        v_storet_pk := sim_import_result_extras.get_sample_pres_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Sample Collection Procedure ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Sample Collection Procedure ID to uppercase
        v_storet_pk := sim_import_result_extras.get_procedure_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Station ID' THEN
        /* sf translations should only be automatically created for stations that don't exist*/
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Station ID to uppercase
        v_storet_pk := sim_import_result_extras.get_station_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
        IF v_storet_pk.org_id = 'XXX' THEN
          v_storet_pk.org_id := NULL;
          v_trans_ok := FALSE;
        END IF;
      
      ELSIF p_item_name = 'Personnel' THEN
        v_storet_pk := sim_import_result_extras.get_personnel_is_nbr(v_converted_item, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Cooperating Organization' THEN
        v_storet_pk := sim_import_result_extras.get_cporg_is_nbr(v_converted_item, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Lab Remark Codes' THEN
        v_storet_pk := sim_import_result_extras.get_remark_codes_is_num(v_converted_item, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Depth Units' THEN
        v_storet_pk := get_uom_is_nbr(v_converted_item, p_fa_seq, v_import_log_seq, v_line_cnt);
      
      ELSIF p_item_name = 'Detection Limit Unit'
            OR p_item_name = 'Laboratory Result Detection Limit Unit' THEN
        v_storet_pk := get_uom_is_nbr(v_converted_item, p_fa_seq, v_import_log_seq, v_line_cnt);
      
      ELSIF p_item_name = 'Upper/Lower Depth Units' THEN
        v_storet_pk := get_uom_is_nbr(v_converted_item, p_fa_seq, v_import_log_seq, v_line_cnt);
      
      ELSIF p_item_name = 'Characteristic Group ID' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts Characteristic Group ID to uppercase
        v_storet_pk := sim_import_result_extras.get_group_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Characteristic Row ID' THEN
        v_storet_pk := sim_import_result_extras.get_row_is_nbr(v_converted_item, p_parent_pk, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Habitat Assessment Row ID' THEN
        v_storet_pk := sim_import_result_extras.get_ha_row_is_nbr(v_converted_item, p_parent_pk, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Result Value Units'
            OR p_item_name = 'Characteristic Result Value Unit' THEN
        v_storet_pk := get_uom_is_nbr(v_converted_item, p_fa_seq, v_import_log_seq, v_line_cnt);
      
      ELSIF p_item_name = 'Characteristic Name' THEN
        v_storet_pk := sim_import_result_extras.get_char_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq, 'CHAR');
      
      ELSIF p_item_name = 'Common Class Descriptor' THEN
        v_storet_pk := sim_import_result_extras.get_char_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq, 'COMM');
      
      ELSIF p_item_name = 'Field/Lab Procedure'
            OR p_item_name = 'Characteristic Field/Lab Analytical Procedure' THEN
        -- jah 2-24-04 Upper the converted item because STORET GUI converts these to uppercase
        v_storet_pk := sim_import_result_extras.get_analproc_is_nbr(upper(v_converted_item), p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Field/Lab Procedure Source'
            OR p_item_name = 'Characteristic Field/Lab Analytical Procedure Source' THEN
      
        v_storet_pk := get_analproc_source(v_converted_item, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Lab Sample Prep Procedure' THEN
        v_storet_pk := sim_import_result_extras.get_labprep_is_nbr(v_converted_item, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Lab Sample Prep Procedure Source' THEN
        v_storet_pk := get_labprep_source(v_converted_item, p_org_id, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Community' THEN
        v_storet_pk := sim_import_result_extras.get_community_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Bio Part' THEN
        v_storet_pk := sim_import_result_extras.get_biopt_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Subject Taxon' THEN
        v_storet_pk := sim_import_result_extras.get_taxon_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Sex' THEN
        v_storet_pk := sim_import_result_extras.get_sex_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Life Stage' THEN
        v_storet_pk := sim_import_result_extras.get_life_is_nbr(v_converted_item, p_fa_seq, v_line_cnt, v_import_log_seq);
      
      ELSIF p_item_name = 'Upper/Lower Bound Units' THEN
        v_storet_pk := get_uom_is_nbr(v_converted_item, p_fa_seq, v_import_log_seq, v_line_cnt);
      
      ELSE
      
        RAISE e_no_function;
      
      END IF;
    
      IF v_storet_pk.is_number IS NULL
         AND NOT v_trans_exist THEN
        /* check instructions to see if translation record should be created */
      
        v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
      
        IF v_instr_code = 'RES-TR-01'
           AND v_trans_ok THEN
        
          /* if record could not be matched to anything in translations or STORET tables,
          create new record */
          create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
        
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                               p_item_name ||
                                               '. Creating translation record for ' ||
                                               p_orig_item || '.');
           
              
        END IF;
      
      ELSIF v_storet_pk.is_number IS NULL
            AND v_trans_exist THEN
      
        /*  item has been translated to an invalid storet value */
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             p_orig_item ||
                                             ' has been mapped to the invalid storet value: ' ||
                                             p_orig_item);
         
            
      END IF;
    
    END IF; -- TRANS IF END                                 
  
  END IF; --PK IF END -- IF NOT NULL

  -- jah 3-3-2004
  BEGIN
    -- send the converted item back to the calling routine.
    p_translated_item := v_converted_item;
  EXCEPTION
    WHEN OTHERS THEN
      -- Is the string buffer too small?
      IF SQLCODE = -6502 THEN
        -- if yes
        /* Do nothing. The value of p_item has too many characters, 
        therefore the value of p_item is not in STORET, therefore 
        this function has already displayed an error message.*/
        NULL;
      ELSE
        -- if no
        -- Re-raise the exception
        RAISE;
      END IF;
  END;

  RETURN v_storet_pk;

EXCEPTION

  WHEN e_no_function THEN
  
    SELECT sc_desc
      INTO sys_desc
      FROM sim_config
     WHERE sc_item = '*BUILD_NUMBER';
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Line ' || v_line_cnt || sys_desc ||
                                          p_item_name);
     
        
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
  
    RETURN v_storet_pk;
  
  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate storet item. (' ||
                                         p_item_name || ')');
     
        
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
  
    RETURN v_storet_pk;
  
END validate_storet_item;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION check_storet_translation
(
  p_code           IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_trans_exist    OUT BOOLEAN,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER,
  v_converted_item OUT VARCHAR2,
  p_activity_seq   IN NUMBER
) RETURN t_storet_pk_type IS

  v_storet_pk   sim_import_result_extras.t_storet_pk_type;
  v_seq         NUMBER(10);
  v_table_name  VARCHAR2(80);
  v_column_name VARCHAR2(80);
  v_dyna_cursor VARCHAR2(200);
  v_comma_pos   NUMBER(10);

  TYPE sort_cursor_type IS REF CURSOR;
  sort_cursor sort_cursor_type;

  CURSOR c_get_storet_key(cp_code VARCHAR2, cp_imp_cfg_seq NUMBER, cp_col_seq NUMBER) IS
    SELECT st_seq,
           st_is_number,
           st_org_id,
           NULL
      FROM sim_translations,
           sim_imp_cfg_dets
     WHERE st_user_code = cp_code
       AND st_sicdt_seq = sicdt_seq
       AND sicdt_sicc_seq = cp_imp_cfg_seq
       AND sicdt_sic_seq = cp_col_seq;

  CURSOR c_imp_cols(cp_col_seq NUMBER) IS
    SELECT sic_storet_tab,
           sic_storet_col
      FROM sim_imp_cols
     WHERE sic_seq = cp_col_seq;

  CURSOR c_comma(cp_col_seq NUMBER) IS
    SELECT instr(sic_storet_col, ',') col_pos
      FROM sim_imp_cols
     WHERE sic_seq = cp_col_seq;

  CURSOR c_inst_col(cp_col_seq NUMBER) IS
    SELECT substr(sic_storet_col, 1, instr(sic_storet_col, ',') - 1) col_name
      FROM sim_imp_cols
     WHERE sic_seq = cp_col_seq;

BEGIN

  -- Try to retreve the storet key from the translations table
  OPEN c_get_storet_key(p_code, p_imp_cfg_seq, p_col_seq);
  FETCH c_get_storet_key
    INTO v_seq, v_storet_pk.is_number, v_storet_pk.org_id, v_converted_item;
  CLOSE c_get_storet_key;

  -- did we retrieve a translation from the translations table?
  IF v_seq IS NULL THEN
    -- if no
  
    -- return the original value
    v_converted_item := p_code;
  
    -- Tell the caller that the return value is the original value
    p_trans_exist := FALSE;
  
  ELSIF v_seq IS NOT NULL THEN
    -- if yes
  
    /* if the code is in the translations table, flag so it's not recreated */
  
    -- Tell the caller that the return value is a translated value
    p_trans_exist := TRUE;
  
    -- dis we fail to get the storet key for the translated value?
    IF v_storet_pk.is_number IS NULL THEN
      -- if yes
    
      /*
        if there is a code in the translations table but no value to translate to,
        write error message. The calling routine is responsible for changing
        the migrate status to 'N'.
      */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_code ||
                                          ' exists in translations table but has not been translated to a value.');
       
          
    ELSE
    
      -- if no (we got the storet key for the translated value)
    
      -- Retrieve the STORET table name and column name of the item
      OPEN c_imp_cols(p_col_seq);
      FETCH c_imp_cols
        INTO v_table_name, v_column_name;
      CLOSE c_imp_cols;
    
      OPEN c_comma(p_col_seq);
      FETCH c_comma
        INTO v_comma_pos;
      CLOSE c_comma;
    
      IF v_comma_pos = 0 THEN
      
        v_dyna_cursor := 'SELECT RTRIM(' || v_column_name || ') ' || ' FROM ' ||
                         v_table_name || ' WHERE ' || v_table_name || '_IS_NUMBER = ' ||
                         v_storet_pk.is_number || ' AND ' || v_table_name ||
                         '_ORG_ID = ''' || v_storet_pk.org_id || '''';
      
      ELSE
      
        OPEN c_inst_col(p_col_seq);
        FETCH c_inst_col
          INTO v_column_name;
        CLOSE c_inst_col;
      
        v_dyna_cursor := 'SELECT RTRIM(' || v_column_name || ') ' || ' FROM ' ||
                         v_table_name || ' WHERE ' || v_table_name || '_IS_NUMBER = ' ||
                         v_storet_pk.is_number || ' AND ' || v_table_name ||
                         '_ORG_ID = ''' || v_storet_pk.org_id || '''';
      
      END IF;
    
      OPEN sort_cursor FOR v_dyna_cursor;
    
      LOOP
        FETCH sort_cursor
          INTO v_converted_item;
        EXIT WHEN sort_cursor%NOTFOUND;
      END LOOP;
    
      IF v_converted_item IS NOT NULL THEN
        p_trans_exist := TRUE;
      END IF;
    
    END IF;
  
  END IF;

  RETURN v_storet_pk;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                        ' - in check storet translations');
     
        
    RETURN NULL;
  
END check_storet_translation;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE create_trans_record
(
  p_code           IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_fa_seq         IN NUMBER,
  v_import_log_seq IN NUMBER,
  v_line_cnt       IN NUMBER
) IS

  v_st_seq    NUMBER(10);
  v_sicdt_seq NUMBER(10);

  e_imp_cfg_dets EXCEPTION;
  e_trans_error EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT sicdt_seq
      FROM sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL
      FROM dual;

BEGIN

  /* kms - if it has been determined that value won't work for STORET, that a translation
  record does not already exist, and that the user wants to autocreate them, this function
  will create half of a translation record.  The remaining half will need to be filled in
  by the user after the import has completed. */

  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq
    INTO v_sicdt_seq;
  IF c_get_cfg_det_seq%NOTFOUND THEN
    v_sicdt_seq := NULL;
    RAISE e_imp_cfg_dets;
  END IF;
  CLOSE c_get_cfg_det_seq;

  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq
    INTO v_st_seq;
  IF c_get_st_seq%NOTFOUND THEN
    v_st_seq := NULL;
    RAISE e_trans_error;
  END IF;
  CLOSE c_get_st_seq;

  INSERT INTO sim_translations
    (st_seq,
     st_sicdt_seq,
     st_user_code,
     st_storet_code)
  VALUES
    (v_st_seq, /* st_seq */
     v_sicdt_seq, /* st_sicdt_seq */
     p_code, /* st_user_code */
     NULL /* st_storet_code */);

EXCEPTION

  WHEN e_imp_cfg_dets THEN
    raise_application_error(-20101, 'Unable to find import configuration');
  
  WHEN e_trans_error THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unable to create new record in translations table - error getting sequence');
     
        
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                        ' - adding record to translations table.');
     
        
END create_trans_record;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_sim_station_visit
(
  p_visit_rec      IN sim_station_visits%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN
  /* create sim_station_visits using values in visit record */
  INSERT INTO sim_station_visits
    (ssv_seq,
     ssv_stp_seq,
     ssv_id,
     ssv_start_date,
     ssv_start_time,
     ssv_end_date,
     ssv_end_time,
     ssv_tsmstatn_is_number,
     ssv_tsmstatn_org_id,
     ssv_tsrstvst_is_number,
     ssv_tsrstvst_org_id,
     ssv_tsmorgan_is_number,
     ssv_tsmorgan_org_id,
     ssv_created_by_sim,
     ssv_arrival_time_zone,
     ssv_document)
  VALUES
    (p_visit_rec.ssv_seq, /* SSV_SEQ */
     p_visit_rec.ssv_stp_seq, /* SSV_STP_SEQ */
     p_visit_rec.ssv_id, /* SSV_ID */
     p_visit_rec.ssv_start_date, /* SSV_START_DATE */
     p_visit_rec.ssv_start_time, /* SSV_START_TIME */
     p_visit_rec.ssv_end_date, /* SSV_END_DATE */
     p_visit_rec.ssv_end_time, /* SSV_END_TIME */
     p_visit_rec.ssv_tsmstatn_is_number, /* SSV_TSMSTATN_IS_NUMBER */
     p_visit_rec.ssv_tsmstatn_org_id, /* SSV_TSMSTATN_ORG_ID */
     p_visit_rec.ssv_tsrstvst_is_number, /* SSV_TSRSTVST_IS_NUMBER */
     p_visit_rec.ssv_tsrstvst_org_id, /* SSV_TSRSTVST_ORG_ID */
     p_visit_rec.ssv_tsmorgan_is_number, /* SSV_TSMORGAN_IS_NUMBER */
     p_visit_rec.ssv_tsmorgan_org_id, /* SSV_TSMORGAN_ORG_ID */
     p_visit_rec.ssv_created_by_sim, /* SSV_CREATED_BY_SIM */
     p_visit_rec.ssv_arrival_time_zone, /* SSV_ARRIVAL_TIME_ZONE */
     p_visit_rec.ssv_document /* SSV_DOCUMENT */);

  -- tell calling routine insert succeeded
  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                         ' - creating sim station_visit.');
     
        
    -- tell calling routine insert failed
    RETURN NULL;
  
END create_sim_station_visit;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_sim_trip
(
  p_trip_rec       IN OUT sim_trips%ROWTYPE,
  p_fa_seq         IN NUMBER,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS
  /* jah 3-5-2004 
  Changed parameter P_TRIP_REC from IN to IN OUT 
  so that STP_SEQ passes back to the calling 
  routine. */

BEGIN

  /* create sim_trips using values in trip record */
  INSERT INTO sim_trips
    (stp_seq,
     stp_trip_id,
     stp_trip_name,
     stp_start_date,
     stp_end_date,
     stp_tsrtrip_is_number,
     stp_tsrtrip_org_id,
     stp_tsmorgan_is_number,
     stp_tsmorgan_org_id,
     stp_created_by_sim)
  VALUES
    (p_trip_rec.stp_seq, /* STP_SEQ */
     p_trip_rec.stp_trip_id, /* STP_TRIP_ID */
     p_trip_rec.stp_trip_name,
     p_trip_rec.stp_start_date, /* STP_START_DATE */
     p_trip_rec.stp_end_date, /* STP_END_DATE */
     p_trip_rec.stp_tsrtrip_is_number, /* STP_TSRTRIP_IS_NUMBER */
     p_trip_rec.stp_tsrtrip_org_id, /* STP_TSRTRIP_ORG_ID */
     p_trip_rec.stp_tsmorgan_is_number, /* STP_TSMORGAN_IS_NUMBER */
     p_trip_rec.stp_tsmorgan_org_id, /* STP_TSMORGAN_ORG_ID */
     p_trip_rec.stp_created_by_sim);

  -- tell the calling routine the insert succeeded
  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
     
  
                                                                                                                                                                                                                                                                                                                                                                                                                                                                    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM || ' - creating SIM trip.');
     
        
    -- tell the calling routine the insert failed
    RETURN NULL;
  
END create_sim_trip;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_field_activity
(
  p_fa_rec         IN sim_field_activities%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN

  /* create field activity using values in record */
  INSERT INTO sim_field_activities
    (fa_seq,
     fa_type_name,
     fa_category,
     fa_activity_id,
     fa_start_date,
     fa_start_time,
     fa_stop_date,
     fa_stop_time,
     fa_depth,
     fa_depth_ref_point,
     fa_activity_comments,
     fa_tsmstatn_is_number,
     fa_tsmstatn_org_id,
     fa_tsrtrip_is_number,
     fa_tsrtrip_org_id,
     fa_tsrstvst_is_number,
     fa_tsrstvst_org_id,
     fa_tsrfldpr_is_number,
     fa_tsrfldpr_org_id,
     fa_tsrfldgr_is_number,
     fa_tsrfldgr_org_id,
     fa_import_seq,
     fa_export_seq,
     fa_status,
     fa_tsmorgan_is_number,
     fa_tsmorgan_org_id,
     fa_approved_by,
     fa_approved_on,
     fa_stvst_comments,
     fa_depth_uom,
     fa_medium,
     fa_tsrlab_is_number,
     fa_tsrlab_org_id,
     fa_import_line_no,
     fa_tsrfdact_is_number,
     fa_tsrfdact_org_id,
     fa_replicate_nbr,
     fa_fdset_id,
     fa_fdset_name,
     fa_updepth,
     fa_lowdepth,
     fa_thermo,
     fa_halo,
     fa_pycno,
     fa_rel_depth,
     fa_depth_zone,
     fa_depth_range_uom,
     fa_start_time_zone,
     fa_stop_time_zone,
     fa_tsrgrcfg_is_number,
     fa_tsrgrcfg_org_id,
     fa_tsrsdp_is_number,
     fa_tsrsdp_org_id,
     fa_stp_seq,
     fa_ssv_seq,
     fa_intent,
     fa_tsmprmvl_community_is_nbr,
     fa_tsmprmvl_community_org_id,
     fa_tsrbiopt_is_number,
     fa_tsrbiopt_org_id,
     fa_tsrchar_is_number,
     fa_tsrchar_org_id,
     fa_chain_of_custody_id,
     fa_qc_indicator,
     fa_tsrmatrx_is_number,
     fa_tsrmatrx_org_id,
     fa_species_number,
     fa_char_group,
     fa_document)
  VALUES
    (p_fa_rec.fa_seq,
     p_fa_rec.fa_type_name,
     p_fa_rec.fa_category,
     p_fa_rec.fa_activity_id,
     p_fa_rec.fa_start_date,
     p_fa_rec.fa_start_time,
     p_fa_rec.fa_stop_date,
     p_fa_rec.fa_stop_time,
     p_fa_rec.fa_depth,
     p_fa_rec.fa_depth_ref_point,
     p_fa_rec.fa_activity_comments,
     p_fa_rec.fa_tsmstatn_is_number,
     p_fa_rec.fa_tsmstatn_org_id,
     p_fa_rec.fa_tsrtrip_is_number,
     p_fa_rec.fa_tsrtrip_org_id,
     p_fa_rec.fa_tsrstvst_is_number,
     p_fa_rec.fa_tsrstvst_org_id,
     p_fa_rec.fa_tsrfldpr_is_number,
     p_fa_rec.fa_tsrfldpr_org_id,
     p_fa_rec.fa_tsrfldgr_is_number,
     p_fa_rec.fa_tsrfldgr_org_id,
     p_fa_rec.fa_import_seq,
     p_fa_rec.fa_export_seq,
     p_fa_rec.fa_status,
     p_fa_rec.fa_tsmorgan_is_number,
     p_fa_rec.fa_tsmorgan_org_id,
     p_fa_rec.fa_approved_by,
     p_fa_rec.fa_approved_on,
     p_fa_rec.fa_stvst_comments,
     p_fa_rec.fa_depth_uom,
     p_fa_rec.fa_medium,
     NULL, --p_fa_rec.FA_TSRLAB_IS_NUMBER,
     NULL, --p_fa_rec.FA_TSRLAB_ORG_ID,
     p_fa_rec.fa_import_line_no,
     p_fa_rec.fa_tsrfdact_is_number,
     p_fa_rec.fa_tsrfdact_org_id,
     p_fa_rec.fa_replicate_nbr,
     p_fa_rec.fa_fdset_id,
     p_fa_rec.fa_fdset_name,
     p_fa_rec.fa_updepth,
     p_fa_rec.fa_lowdepth,
     p_fa_rec.fa_thermo,
     p_fa_rec.fa_halo,
     p_fa_rec.fa_pycno,
     p_fa_rec.fa_rel_depth,
     p_fa_rec.fa_depth_zone,
     p_fa_rec.fa_depth_range_uom,
     p_fa_rec.fa_start_time_zone,
     p_fa_rec.fa_stop_time_zone,
     p_fa_rec.fa_tsrgrcfg_is_number,
     p_fa_rec.fa_tsrgrcfg_org_id,
     p_fa_rec.fa_tsrsdp_is_number,
     p_fa_rec.fa_tsrsdp_org_id,
     p_fa_rec.fa_stp_seq,
     p_fa_rec.fa_ssv_seq,
     p_fa_rec.fa_intent,
     p_fa_rec.fa_tsmprmvl_community_is_nbr,
     p_fa_rec.fa_tsmprmvl_community_org_id,
     p_fa_rec.fa_tsrbiopt_is_number,
     p_fa_rec.fa_tsrbiopt_org_id,
     p_fa_rec.fa_tsrchar_is_number,
     p_fa_rec.fa_tsrchar_org_id,
     p_fa_rec.fa_chain_of_custody_id,
     p_fa_rec.fa_qc_indicator,
     p_fa_rec.fa_tsrmatrx_is_number,
     p_fa_rec.fa_tsrmatrx_org_id,
     p_fa_rec.fa_species_number,
     p_fa_rec.fa_char_group,
     p_fa_rec.fa_document);

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                         ' - inserting field activity.');
     
        
    RETURN 0;
  
END create_field_activity;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_field_activity_parent
(
  p_fap_rec        IN sim_field_activity_parent%ROWTYPE,
  p_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN

  INSERT INTO sim_field_activity_parent
    (fap_seq,
     fap_fa_seq,
     fap_parent_fa_seq,
     fap_parent_is_number,
     fap_parent_org_id)
  VALUES
    (p_fap_rec.fap_seq,
     p_fap_rec.fap_fa_seq,
     p_fap_rec.fap_parent_fa_seq,
     p_fap_rec.fap_parent_is_number,
     p_fap_rec.fap_parent_org_id);

  COMMIT;

  RETURN NULL;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                         ' - inserting field activity parent.');
     
        
    RETURN 1;
  
END create_field_activity_parent;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_RESULT_CLASS_INDICATOR RECORD 
 ****************************************************************************************/ 
FUNCTION create_rci
(
  p_rci_rec        IN sim_result_class_indicator%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN
  /* create rci using values in record */
  INSERT INTO sim_result_class_indicator
    (rci_seq,
     rci_rs_seq,
     rci_fa_seq,
     rci_primary_class_ind,
     rci_second_class_ind,
     rci_lower_bnd_amt,
     rci_upper_bnd_amt,
     rci_tsruom_is_number,
     rci_tsruom_org_id,
     rci_particle_size_basis, -- jah 2-23-04 Added Particle Size Basis
     rci_brg_seq,
     rci_import_seq)
  VALUES
    (p_rci_rec.rci_seq,
     p_rci_rec.rci_rs_seq,
     p_rci_rec.rci_fa_seq,
     p_rci_rec.rci_primary_class_ind,
     p_rci_rec.rci_second_class_ind,
     p_rci_rec.rci_lower_bnd_amt,
     p_rci_rec.rci_upper_bnd_amt,
     p_rci_rec.rci_tsruom_is_number,
     p_rci_rec.rci_tsruom_org_id,
     p_rci_rec.rci_particle_size_basis, -- jah 2-23-04 Added Particle Size Basis
     p_rci_rec.rci_brg_seq,
     p_rci_rec.rci_import_seq);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                        ' - inserting to SIM_RESULT_CLASS_INDICATOR.');
     
        
    RETURN 0;
  
END create_rci;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_BIO_RESULTS_GRP_INDIVIDUAL RECORD  
 ****************************************************************************************/ 
FUNCTION create_brgi
(
  p_brgi_rec       IN sim_bio_results_grp_individual%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS
BEGIN
  /* create brgi using values in record */
  INSERT INTO sim_bio_results_grp_individual
    (brgi_seq,
     brgi_brg_seq,
     brgi_individual_number,
     brgi_fa_seq,
     brgi_import_seq)
  VALUES
    (p_brgi_rec.brgi_seq,
     p_brgi_rec.brgi_brg_seq,
     p_brgi_rec.brgi_individual_number,
     p_brgi_rec.brgi_fa_seq,
     p_brgi_rec.brgi_import_seq);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                         ' - inserting to SIM_BIO_RESULTS_GRP_INDIVIDUAL.');
     
        
    RETURN 0;
END create_brgi;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_BIO_RESULTS_GRP RECORD 
 ****************************************************************************************/ 
FUNCTION create_brg
(
  p_brg_rec        IN sim_bio_results_grp%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS
BEGIN
  /* create brg using values in record */

  INSERT INTO sim_bio_results_grp
    (brg_seq,
     brg_fa_seq,
     brg_id_code,
     brg_type,
     brg_species_number,
     brg_description_text,
     brg_freq_analysis,
     brg_tsrcpv_sex_is_number,
     brg_tsrcpv_sex_org_id,
     brg_tsrcpv_lifestage_is_nbr,
     brg_tsrcpv_lifestage_org_id,
     brg_value_type_name,
     brg_summary_grp_count,
     brg_tsrchar_is_number,
     brg_tsrchar_org_id,
     brg_import_seq)
  VALUES
    (p_brg_rec.brg_seq,
     p_brg_rec.brg_fa_seq,
     p_brg_rec.brg_id_code,
     p_brg_rec.brg_type,
     p_brg_rec.brg_species_number,
     p_brg_rec.brg_description_text,
     p_brg_rec.brg_freq_analysis,
     p_brg_rec.brg_tsrcpv_sex_is_number,
     p_brg_rec.brg_tsrcpv_sex_org_id,
     p_brg_rec.brg_tsrcpv_lifestage_is_nbr,
     p_brg_rec.brg_tsrcpv_lifestage_org_id,
     p_brg_rec.brg_value_type_name,
     p_brg_rec.brg_summary_grp_count,
     p_brg_rec.brg_tsrchar_is_number,
     p_brg_rec.brg_tsrchar_org_id,
     p_brg_rec.brg_import_seq);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                         ' - inserting to SIM_BIO_RESULTS_GRP.');
     
        
    RETURN 0;
END create_brg;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_SAMPLE RECORD 
 ****************************************************************************************/ 
FUNCTION create_sample
(
  p_sample_rec     IN sim_sample%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS
BEGIN
  /* create sim_sample using values in record */
  INSERT INTO sim_sample
    (ss_seq,
     ss_fa_seq,
     ss_total_volume_msr,
     ss_total_vol_uom,
     ss_total_weight_msr,
     ss_total_wt_uom,
     ss_data_file_name_loc,
     ss_container_type_nm,
     ss_container_color,
     ss_container_size_msr,
     ss_container_size_un,
     ss_temp_preservn_type,
     ss_parent_ind_code,
     ss_gear_deployment,
     ss_transport_storage,
     ss_presrv_strge_prcdr,
     ss_duration_time,
     ss_duration_unit)
  VALUES
    (p_sample_rec.ss_seq,
     p_sample_rec.ss_fa_seq,
     p_sample_rec.ss_total_volume_msr,
     p_sample_rec.ss_total_vol_uom,
     p_sample_rec.ss_total_weight_msr,
     p_sample_rec.ss_total_wt_uom,
     p_sample_rec.ss_data_file_name_loc,
     p_sample_rec.ss_container_type_nm,
     p_sample_rec.ss_container_color,
     p_sample_rec.ss_container_size_msr,
     p_sample_rec.ss_container_size_un,
     p_sample_rec.ss_temp_preservn_type,
     p_sample_rec.ss_parent_ind_code,
     p_sample_rec.ss_gear_deployment,
     p_sample_rec.ss_transport_storage,
     p_sample_rec.ss_presrv_strge_prcdr,
     p_sample_rec.ss_duration_time,
     p_sample_rec.ss_duration_unit);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                        ' - inserting to SIM_SAMPLE.');
     
        
    RETURN 0;
END create_sample;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_TRAWL_OPS_DET RECORD  
 ****************************************************************************************/ 
FUNCTION create_tod
(
  p_tod_rec        IN sim_trawl_ops_det%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS
BEGIN
  /* create tod using values in record */
  INSERT INTO sim_trawl_ops_det
    (tod_seq,
     tod_fa_seq,
     tod_fished_duration,
     tod_fished_dur_uom,
     tod_boat_speed_msr,
     tod_boat_speed_uom,
     tod_fished_distance,
     tod_fished_distance_uom,
     tod_rel_current_dir,
     tod_rel_wind_dir,
     tod_comment_text)
  VALUES
    (p_tod_rec.tod_seq,
     p_tod_rec.tod_fa_seq,
     p_tod_rec.tod_fished_duration,
     p_tod_rec.tod_fished_dur_uom,
     p_tod_rec.tod_boat_speed_msr,
     p_tod_rec.tod_boat_speed_uom,
     p_tod_rec.tod_fished_distance,
     p_tod_rec.tod_fished_distance_uom,
     p_tod_rec.tod_rel_current_dir,
     p_tod_rec.tod_rel_wind_dir,
     p_tod_rec.tod_comment_text);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM ||
                                        ' - inserting to SIM_TRAWL_OPS_DET.');
     
        
    RETURN 0;
END create_tod;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_STATION_LOCATIONS (stl) RECORD 
 ****************************************************************************************/ 
FUNCTION create_stl
(
  p_stl_rec        IN sim_station_locations%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN
  /* Insert location using values in record */

  INSERT INTO sim_station_locations
    (stl_seq,
     stl_type_code,
     stl_sequence_number,
     stl_tsmstatn0is_number,
     stl_tsmstatn0org_id,
     stl_lat_degrees,
     stl_lat_minutes,
     stl_lat_seconds,
     stl_lat_dec_minutes,
     stl_long_degrees,
     stl_long_minutes,
     stl_long_seconds,
     stl_long_dec_minutes,
     stl_lat_dir,
     stl_long_dir,
     stl_geopos_method,
     stl_geopos_datum)
  VALUES
    (p_stl_rec.stl_seq,
     p_stl_rec.stl_type_code,
     p_stl_rec.stl_sequence_number,
     p_stl_rec.stl_tsmstatn0is_number,
     p_stl_rec.stl_tsmstatn0org_id,
     p_stl_rec.stl_lat_degrees,
     p_stl_rec.stl_lat_minutes,
     p_stl_rec.stl_lat_seconds,
     p_stl_rec.stl_lat_dec_minutes,
     p_stl_rec.stl_long_degrees,
     p_stl_rec.stl_long_minutes,
     p_stl_rec.stl_long_seconds,
     p_stl_rec.stl_long_dec_minutes,
     p_stl_rec.stl_lat_dir,
     p_stl_rec.stl_long_dir,
     p_stl_rec.stl_geopos_method,
     p_stl_rec.stl_geopos_datum);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM || ' - inserting stl.');
     
        
    RETURN 0;
END create_stl;

/****************************************************************************************
 * FUNCTION TO INSERT A SIM_ACTUAL_ACTIVITY_LOC (aal) RECORD 
 ****************************************************************************************/ 
FUNCTION create_aal
(
  p_aal_rec        IN sim_actual_activity_loc%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN
  /* Insert aal using values in record */

  INSERT INTO sim_actual_activity_loc
    (aal_seq,
     aal_fa_seq,
     aal_stl_seq,
     aal_tsmalp_is_number,
     aal_tsmalp_org_id,
     aal_tsraal_type_name,
     aal_bottom_depth_msr,
     aal_bottom_depth_uom,
     aal_addtnl_loc_info)
  VALUES
    (p_aal_rec.aal_seq,
     p_aal_rec.aal_fa_seq,
     p_aal_rec.aal_stl_seq,
     p_aal_rec.aal_tsmalp_is_number,
     p_aal_rec.aal_tsmalp_org_id,
     p_aal_rec.aal_tsraal_type_name,
     p_aal_rec.aal_bottom_depth_msr,
     p_aal_rec.aal_bottom_depth_uom,
     p_aal_rec.aal_addtnl_loc_info);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM || ' - inserting aal.');
     
        
    RETURN 0;
END create_aal;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_trap_net
(
  p_stn_rec        IN sim_trap_net%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN
  /* Insert stn using values in record */

  INSERT INTO sim_trap_net
    (stn_seq,
     stn_tsrfdact_is_number,
     stn_tsrfdact_org_id,
     stn_sampling_dur,
     stn_sampling_dur_unit,
     stn_orient_to_cur,
     stn_rel_current,
     stn_rel_wind,
     stn_comment,
     stn_fa_seq)
  VALUES
    (p_stn_rec.stn_seq,
     p_stn_rec.stn_tsrfdact_is_number,
     p_stn_rec.stn_tsrfdact_org_id,
     p_stn_rec.stn_sampling_dur,
     p_stn_rec.stn_sampling_dur_unit,
     p_stn_rec.stn_orient_to_cur,
     p_stn_rec.stn_rel_current,
     p_stn_rec.stn_rel_wind,
     p_stn_rec.stn_comment,
     p_stn_rec.stn_fa_seq);
  COMMIT;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM || ' - inserting trap_net.');
     
        
    RETURN 0;
END create_trap_net;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION create_result
(
  p_result_rec     IN sim_results%ROWTYPE,
  v_import_log_seq IN NUMBER
) RETURN NUMBER IS

BEGIN

  /* create result using values in result record */
  INSERT INTO sim_results
    (rs_seq,
     rs_ac_seq,
     rs_fa_seq,
     rs_brg_seq,
     rs_brgi_seq,
     rs_result,
     rs_comment,
     rs_detect_limit,
     rs_import_line_no,
     rs_value_status,
     rs_lab_batch_id,
     rs_lab_cert,
     rs_analysis_date,
     rs_analysis_time,
     rs_analysis_time_zone,
     rs_det_limit_uom_nbr,
     rs_det_limit_uom_org,
     rs_detect_comment,
     rs_upper_quant_limit,
     rs_lower_quant_limit,
     rs_tsrchar_is_number,
     rs_tsrchar_org_id,
     rs_tsrchar0is_number,
     rs_tsrchar0org_id,
     rs_tsruom_is_nbr,
     rs_tsruom_org_id,
     rs_fraction,
     rs_value_type,
     rs_stat_type,
     rs_tsranlpr_is_number,
     rs_tsranlpr_org_id,
     rs_tsrlspp_is_number,
     rs_tsrlspp_org_id,
     rs_duration_basis,
     rs_temp_basis,
     rs_weight_basis,
     rs_precision,
     rs_conf_level,
     rs_tsrchgrp_is_number,
     rs_tsrchgrp_org_id,
     rs_tsrchdef_is_number,
     rs_tsrchdef_org_id,
     rs_trophic_level,
     rs_fnctional_feed_grp,
     rs_taxon_pollution,
     rs_tsrlab_is_number,
     rs_tsrlab_org_id,
     rs_tsrhcsc_is_number,
     rs_tsrhcsc_org_id,
     rs_lower_range_value,
     rs_upper_range_value,
     rs_repl_analysis_num,
     rs_tsrdlin_org_id,
     rs_tsrdlin_is_number,
     rs_fraction_tsmprmvl_is_number,
     rs_fraction_tsmprmvl_org_id,
     rs_habit_tsmprmvl0is_number,
     rs_habit_tsmprmvl0org_id,
     rs_volt_tsmprmvl1is_number,
     rs_volt_tsmprmvl1org_id,
     rs_d_assnd_seq_num,
     rs_bias,
     rs_conf_lvl_corr_bias,
     rs_species_number,
     rs_ref_pt_from_name,
     rs_ref_pt_to_name,
     rs_dilution_ind_code,
     rs_recovery_ind_code,
     rs_correction_ind_cd,
     rs_cell_type_nm,
     rs_cell_shape_type_nm,
     rs_import_seq,
     rs_data_line_number,
     rs_data_line_name,
     rs_document)
  VALUES
    (p_result_rec.rs_seq, /* RS_SEQ */
     p_result_rec.rs_ac_seq, /* RS_AC_SEQ */
     p_result_rec.rs_fa_seq, /* RS_FA_SEQ */
     p_result_rec.rs_brg_seq,
     p_result_rec.rs_brgi_seq,
     p_result_rec.rs_result, /* RS_RESULT */
     p_result_rec.rs_comment, /* RS_COMMENT */
     p_result_rec.rs_detect_limit, /* RS_DETECT_LIMIT */
     p_result_rec.rs_import_line_no, /* RS_IMPORT_LINE_NO */
     p_result_rec.rs_value_status, /* RS_VALUE_STATUS_SEQ */
     p_result_rec.rs_lab_batch_id, /* RS_LAB_BATCH_ID */
     p_result_rec.rs_lab_cert, /* RS_LAB_CERT */
     p_result_rec.rs_analysis_date, /* RS_ANALYSIS_DATE */
     p_result_rec.rs_analysis_time, /* RS_ANALYSIS_TIME */
     p_result_rec.rs_analysis_time_zone, /* RS_ANALYSIS_TIME_ZONE */
     p_result_rec.rs_det_limit_uom_nbr, /* RS_DET_LIMIT_UOM_NBR */
     p_result_rec.rs_det_limit_uom_org, /* RS_DET_LIMIT_UOM_ORG */
     p_result_rec.rs_detect_comment, /* RS_DETECT_COMMENT */
     p_result_rec.rs_upper_quant_limit, /* RS_UPPER_QUANT_LIMIT */
     p_result_rec.rs_lower_quant_limit, /* RS_LOWER_QUANT_LIMIT */
     p_result_rec.rs_tsrchar_is_number, /* RS_TSRCHAR_IS_NUMBER */
     p_result_rec.rs_tsrchar_org_id, /* RS_TSRCHAR_ORG_ID */
     p_result_rec.rs_tsrchar0is_number, /* RS_TSRCHAR0IS_NUMBER */
     p_result_rec.rs_tsrchar0org_id, /* RS_TSRCHAR0ORG_ID */
     p_result_rec.rs_tsruom_is_nbr, /* RS_TSRUOM_IS_NBR */
     p_result_rec.rs_tsruom_org_id, /* RS_TSRUOM_ORG_ID */
     p_result_rec.rs_fraction, /* RS_FRACTION */
     p_result_rec.rs_value_type, /* RS_VALUE_TYPE_SEQ */
     p_result_rec.rs_stat_type, /* RS_STAT_TYPE_SEQ */
     p_result_rec.rs_tsranlpr_is_number, /* RS_TSRANLPR_IS_NUMBER */
     p_result_rec.rs_tsranlpr_org_id, /* RS_TSRANLPR_ORG_ID */
     p_result_rec.rs_tsrlspp_is_number, /* RS_TSRLSPP_IS_NUMBER */
     p_result_rec.rs_tsrlspp_org_id, /* RS_TSRLSPP_ORG_ID */
     p_result_rec.rs_duration_basis, /* RS_DURBASIS_SEQ */
     p_result_rec.rs_temp_basis, /* RS_TEMPBASIS_SEQ */
     p_result_rec.rs_weight_basis,
     p_result_rec.rs_precision, /* RS_PRECISION */
     p_result_rec.rs_conf_level, /* RS_CONFLEVEL_SEQ */
     p_result_rec.rs_tsrchgrp_is_number, /* RS_TSRCHGRP_IS_NUMBER */
     p_result_rec.rs_tsrchgrp_org_id, /* RS_TSRCHGRP_ORG_ID */
     p_result_rec.rs_tsrchdef_is_number, /* RS_TSRCHDEF_IS_NUMBER */
     p_result_rec.rs_tsrchdef_org_id, /* RS_TSRCHDEF_ORG_ID */
     p_result_rec.rs_trophic_level,
     p_result_rec.rs_fnctional_feed_grp,
     p_result_rec.rs_taxon_pollution,
     p_result_rec.rs_tsrlab_is_number,
     p_result_rec.rs_tsrlab_org_id,
     p_result_rec.rs_tsrhcsc_is_number,
     p_result_rec.rs_tsrhcsc_org_id,
     p_result_rec.rs_lower_range_value,
     p_result_rec.rs_upper_range_value,
     p_result_rec.rs_repl_analysis_num,
     p_result_rec.rs_tsrdlin_org_id,
     p_result_rec.rs_tsrdlin_is_number,
     p_result_rec.rs_fraction_tsmprmvl_is_number,
     p_result_rec.rs_fraction_tsmprmvl_org_id,
     p_result_rec.rs_habit_tsmprmvl0is_number,
     p_result_rec.rs_habit_tsmprmvl0org_id,
     p_result_rec.rs_volt_tsmprmvl1is_number,
     p_result_rec.rs_volt_tsmprmvl1org_id,
     p_result_rec.rs_d_assnd_seq_num,
     p_result_rec.rs_bias,
     p_result_rec.rs_conf_lvl_corr_bias,
     p_result_rec.rs_species_number,
     p_result_rec.rs_ref_pt_from_name,
     p_result_rec.rs_ref_pt_to_name,
     p_result_rec.rs_dilution_ind_code,
     p_result_rec.rs_recovery_ind_code,
     p_result_rec.rs_correction_ind_cd,
     p_result_rec.rs_cell_type_nm,
     p_result_rec.rs_cell_shape_type_nm,
     p_result_rec.rs_import_seq,
     p_result_rec.rs_data_line_number,
     p_result_rec.rs_data_line_name,
     p_result_rec.rs_document);
  
  COMMIT;

  RETURN 1;
  


EXCEPTION

  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| NULL,
                                         SQLERRM || ' - inserting result.');
     
        
    RETURN 0;
  
END create_result;

END sim_import_result_extras;
/
