CREATE OR REPLACE PACKAGE sim_import_result_pkg IS

-----------------------------------------------------------------------------------------
-- Package that imports activity and result records into SIM2 from a text file.
-----------------------------------------------------------------------------------------

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/
TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;

TYPE t_rci_pk_type IS RECORD(
  is_number NUMBER(8),
  org_id    VARCHAR2(8),
  indicator NUMBER(1));

TYPE t_stvst_id_rec IS RECORD(
  trip_seq       NUMBER(10),
  station_is_nbr NUMBER(8),
  visit_id       NUMBER(3));

TYPE t_stvst_id_type IS TABLE OF t_stvst_id_rec INDEX BY BINARY_INTEGER;

TYPE t_stvst_id_key_type IS TABLE OF NUMBER(19) INDEX BY BINARY_INTEGER;

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ***************************************************************************************/

FUNCTION get_date
(
  p_column_name IN VARCHAR2,
  p_raw_date    IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_cfg_seq     IN NUMBER,
  p_fa_seq      IN NUMBER,
  p_sid_seq     IN NUMBER
) RETURN DATE;

FUNCTION get_time
(
  p_column_name IN VARCHAR2,
  p_raw_time    IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_cfg_seq     IN NUMBER,
  p_fa_seq      IN NUMBER
) RETURN DATE;

PROCEDURE import_results
(
  p_org_id         IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_imp_cfg_seq    IN NUMBER,
  p_test_num       OUT NUMBER,
  p_skip_header    IN NUMBER := 0
);

FUNCTION validate_class_descriptor
(
  p_orig_item      IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_char_is_nbr    IN NUMBER,
  p_item_name      IN VARCHAR2,
  p_col_seq        IN NUMBER,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2;

FUNCTION create_station(
  p_station_rec IN sim_stations%ROWTYPE) 
  RETURN NUMBER;

FUNCTION create_well(
  p_well_rec IN sim_station_wells%ROWTYPE) 
  RETURN NUMBER;

PROCEDURE update_rows_ready (
  p_sil_seq     IN  sim_import_log.sil_seq%TYPE,
  p_result_cnt  OUT sim_import_log.sil_rows_ready%TYPE,
  p_act_cnt     OUT sim_import_log.sil_activities_imported%TYPE);

FUNCTION create_station_location(
  p_station_location_rec IN sim_station_locations%ROWTYPE)
  RETURN NUMBER;

END sim_import_result_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_import_result_pkg IS

/*--------------------------------------------------------------------------------------
  Import samples and results into SIM
  --------------------------------------------------------------------------------------*/
  
/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/
  TYPE t_acode_defs_rec IS RECORD(
    char_pk             sim_import_result_extras.t_storet_pk_type,
    anlpr_pk            sim_import_result_extras.t_storet_pk_type,
    labprep_pk          sim_import_result_extras.t_storet_pk_type,
    uom_pk              sim_import_result_extras.t_storet_pk_type,
    lab_pk              sim_import_result_extras.t_storet_pk_type,
    fraction_is         NUMBER(12),
    voltinism_is        NUMBER(12),
    habit_is            NUMBER(12),
    value_type          VARCHAR2(10),
    stat_type           VARCHAR2(18),
    dur_basis           VARCHAR2(10),
    temp_basis          VARCHAR2(8),
    weight_basis        VARCHAR2(12),
    pollution_tol       VARCHAR2(4),
    funct_feed_grp      VARCHAR2(6),
    trophic_level       VARCHAR2(4),
    species             VARCHAR2(8),
    particle_size_basis VARCHAR2(40), -- jah 02-20-04  Added Particle Size Basis
    lower_range_value   NUMBER(13, 5), -- Added Lower/Upper range support for Groups and Rows
    upper_range_value   NUMBER(13, 5) -- Added Lower/Upper range support for Groups and Rows
    );

/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  v_line_cnt         NUMBER(10) := 0;
  v_acode_defs_rec   t_acode_defs_rec;
  v_acode_init_rec   t_acode_defs_rec;
  v_org_is_nbr       NUMBER(12);
  v_import_log_seq   NUMBER(10);
  v_trip_qc          NUMBER(1) := 0;
  v_cur_trip_is_nbr  NUMBER(12) := NULL;
  gv_last_visit_id   NUMBER(5) := 0;
  v_char_result_type VARCHAR2(4) := NULL;
  v_sid_seq          NUMBER(10) := NULL;

/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************
  
 ****************************************************************************************/
FUNCTION get_date
(
  p_column_name IN VARCHAR2,
  p_raw_date    IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_cfg_seq     IN NUMBER,
  p_fa_seq      IN NUMBER,
  p_sid_seq     IN NUMBER
) RETURN DATE IS

  v_start_date DATE;
  v_fmt        VARCHAR2(80);

  e_no_cfg_found EXCEPTION;

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;

BEGIN
  /* kms - function that uses the value passed in from the text file to get a date based
  on the date format selected in the import configuration. */

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    /* no format specified, use default */
    IF p_sid_seq = 10007 THEN
      v_fmt := 'RRRR-MM-DD';
    ELSE
      v_fmt := 'MM-DD-RRRR';
    END IF;
  END IF;
  CLOSE c_get_date_fmt;

  IF v_fmt IS NOT NULL THEN
    BEGIN
      v_start_date := trunc(to_date(p_raw_date, v_fmt));
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_column_name ||
                                             ', data does not match the format chosen in the import configuration.');
         
     
        v_start_date := NULL;
    END;
  END IF;

  RETURN v_start_date;

EXCEPTION

  WHEN e_no_cfg_found THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Import configuration for start, end, or analysis date is incomplete or does not exist.');
     
 
    RETURN NULL;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get date, column ' ||
                                         p_column_name || ' ' || p_raw_date);
     
 
    RETURN NULL;
  
END get_date;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_time
(
  p_column_name IN VARCHAR2,
  p_raw_time    IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_cfg_seq     IN NUMBER,
  p_fa_seq      IN NUMBER
) RETURN DATE IS

  v_start_time DATE;
  v_fmt        VARCHAR2(80);
  v_raw_time   VARCHAR2(12);

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;

BEGIN
  /* kms - function that converts the value in the text file to a time based on the the format
  selected in the import configuration */

  --SF 11/4/02 Check for lack of leading zeros
  IF substr(p_raw_time, 2, 1) = ':' THEN
    v_raw_time := '0' || p_raw_time;
  ELSE
    v_raw_time := p_raw_time;
  END IF;

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    /* no format specified, use default */
    v_fmt := 'HH24:MI:SS';
  END IF;
  CLOSE c_get_date_fmt;

  BEGIN
    --now support HHMM for STORET V2.  In STORET, this format is specified and means 'HH24MI' in Oracle speak.
    IF v_fmt = 'HHMM' THEN
      v_fmt := 'HH24MI';
    END IF;
    v_start_time := to_date('01/01/0001 ' || v_raw_time, 'MM/DD/RRRR ' ||
                             v_fmt);
  EXCEPTION
    WHEN OTHERS THEN
      v_start_time := NULL;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_column_name ||
                                           ', data does not match the format chosen in the import configuration.');
       
   
      RETURN v_start_time;
  END;

  RETURN v_start_time;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get time ' ||
                                         p_raw_time || ', column ' ||
                                         p_column_name);
     
 
    RETURN NULL;
  
END get_time;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_value_msr
(
  p_result_text    IN VARCHAR2,
  p_result_ok      OUT BOOLEAN,
  p_fa_seq         IN NUMBER,
  v_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  v_result_num NUMBER;
  v_result     VARCHAR2(50);

BEGIN
  BEGIN
    v_result := p_result_text;
    p_result_ok := TRUE;
    /* Check if the result is a number and round to 5 places */
    v_result_num := round(to_number(v_result), 5);
  
    IF v_result_num > 99999999.99999 THEN
      /* Check if number is too large for STORET */
      p_result_ok := FALSE;
      v_result := '* TO LARGE';
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Result is too large for STORET: ' ||
                                            p_result_text);
       
   
    ELSIF v_result_num <> p_result_text THEN
      /* if result tail is longer than 5 decimal places, warn user that it will be rounded */
      p_result_ok := TRUE;
      v_result := v_result_num;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                           'Result, ' || p_result_text ||
                                           ' will be rounded to 5 places.');
       
   
    END IF;
  
  EXCEPTION
    WHEN OTHERS THEN
      /* If the result is not a number then fail*/
      p_result_ok := FALSE;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Result, ' || p_result_text ||
                                           ', is not a number.');
       
   
      /*Returns the failed result, gives error and trims the result to fit into SIM*/
      v_result := substr(p_result_text, 1, 15);
  END;

  RETURN(v_result);

EXCEPTION
  WHEN OTHERS THEN
    p_result_ok := FALSE;
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get value measure.');
     
 
    RETURN NULL;
END get_value_msr;

/****************************************************************************************

****************************************************************************************/
PROCEDURE get_or_add_trip_rec
(
  p_trip_rec        IN OUT sim_trips%ROWTYPE,
  p_org_id          IN VARCHAR2,
  p_fa_seq          IN NUMBER,
  p_trip_start_date IN DATE
) IS

  v_trip_seq        NUMBER(10);
  v_trip_start_date DATE;
  v_trip_end_date   DATE;
  v_success         NUMBER(1);

  e_no_start_date EXCEPTION;

  -- Retrieve Trip from SIM
  CURSOR c_get_trip_seq(p_trip_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT stp_seq,
           stp_tsrtrip_is_number,
           stp_tsrtrip_org_id
      FROM sim_trips
     WHERE stp_trip_id = p_trip_id
       AND stp_tsmorgan_org_id = rpad(p_org_id, 8);

  -- Retrieve the trip from STORET
  CURSOR c_get_trip_key(p_trip_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsrtrip_is_number,
           tsrtrip_org_id,
           start_date,
           start_time
      FROM tsrtrip
     WHERE id_code = rpad(p_trip_id, 15)
       AND tsrtrip_org_id = rpad(p_org_id, 8);

  -- Retrieve the next Trip Sequence
  CURSOR c_get_next_trip_seq IS
    SELECT sim_trips_seq.NEXTVAL
      FROM dual;

  /* This function looks for the Trip in the SIM table.
  If it is not there, then it looks in STORET.
  
  If the Trip ID is in STORET, this function returns 
  the IS_NUMBER and ORG_ID to the trip to the calling 
  routine.
  
  If the Trip ID is not in SIM, this function inserts 
  the trip into SIM_TRIPS. */

BEGIN

  -- Try to retrieve the Trip from SIM
  OPEN c_get_trip_seq(p_trip_rec.stp_trip_id, p_org_id);
  FETCH c_get_trip_seq
    INTO v_trip_seq, p_trip_rec.stp_tsrtrip_is_number, p_trip_rec.stp_tsrtrip_org_id;

  -- did we find the Trip in SIM?
  IF c_get_trip_seq%NOTFOUND THEN
    -- if no
  
    -- Indicate the Trip is NOT in SIM
    v_trip_seq := NULL;
  
    -- Put default values in the IS_NUMBER and ORG_ID in case Trip is not in STORET.
    p_trip_rec.stp_tsmorgan_is_number := v_org_is_nbr; -- v_org_is_number is a global variable
    p_trip_rec.stp_tsmorgan_org_id := p_org_id;
  
    -- Try to retrieve the Trip from STORET
    OPEN c_get_trip_key(p_trip_rec.stp_trip_id, p_org_id);
    FETCH c_get_trip_key
      INTO p_trip_rec.stp_tsrtrip_is_number, p_trip_rec.stp_tsrtrip_org_id, v_trip_start_date, v_trip_end_date;
  
    -- did we find the Trip ID in STORET?
    IF c_get_trip_key%NOTFOUND THEN
      -- if no
    
      -- tell calling routine that SIM is setting the Trip start date
      p_trip_rec.stp_created_by_sim := 1;
    
      -- does the Trip record already contain the Trip start date?
      IF p_trip_rec.stp_start_date IS NULL THEN
        -- if no
      
        -- did the calling routine pass the Trip start date?
        IF p_trip_start_date IS NOT NULL THEN
          -- if yes
        
          -- return the Trip start date back to the calling routine
          p_trip_rec.stp_start_date := p_trip_start_date;
        
        ELSE
          -- if no (i.e., the calling routine did not pass a Trip start date)
        
          RAISE e_no_start_date;
        
        END IF;
      
      END IF;
    
    ELSE
      -- if yes (i.e., we found the Trip ID in STORET)
    
      -- tell calling routine that SIM is not setting the Trip start date
      p_trip_rec.stp_created_by_sim := 0;
    
      -- return the the Trip start and end dates to the calling routine
      p_trip_rec.stp_start_date := v_trip_start_date;
      p_trip_rec.stp_end_date := v_trip_end_date;
    
    END IF;
    CLOSE c_get_trip_key;
  
  END IF;
  CLOSE c_get_trip_seq;

  -- Is the Trip in SIM?
  IF v_trip_seq IS NULL THEN
    -- if no
  
    -- retrieve the next Trip sequence
    OPEN c_get_next_trip_seq;
    FETCH c_get_next_trip_seq
      INTO p_trip_rec.stp_seq;
    CLOSE c_get_next_trip_seq;
  
    -- Insert the Trip into SIM_TRIPS */
    /* jah 3-5-2004 
    Changed this parameter from IN to IN OUT so STP_SEQ passes back to calling routine. */
    v_success := sim_import_result_extras.create_sim_trip(p_trip_rec, p_fa_seq, v_import_log_seq);
  
  ELSE
    -- if yes (i.e., the Trip is in SIM)
  
    -- return the trip sequence to the calling routine    
    p_trip_rec.stp_seq := v_trip_seq;
  
  END IF;

EXCEPTION

  WHEN e_no_start_date THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Could not find Trip ID in SIM or STORET.  ' ||
                                         'Could not create Trip ID because Trip Start Date is not in the import record.');
     
 
    p_trip_rec.stp_tsrtrip_org_id := NULL;
    p_trip_rec.stp_tsrtrip_is_number := NULL;
    p_trip_rec.stp_seq := NULL;
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in get or add trip rec.');
     
 
    p_trip_rec.stp_tsrtrip_org_id := NULL;
    p_trip_rec.stp_tsrtrip_is_number := NULL;
    p_trip_rec.stp_seq := NULL;
  
END get_or_add_trip_rec;

/****************************************************************************************

****************************************************************************************/
PROCEDURE load_station_visits
(
  p_trip_seq       IN NUMBER,
  p_trip_key       IN sim_import_result_extras.t_storet_pk_type,
  p_org_id         IN VARCHAR2,
  p_stvst_id_table IN OUT t_stvst_id_type,
  p_fa_seq         IN NUMBER
) IS

  v_visit_seq         NUMBER(10);
  v_visit_rec         sim_station_visits%ROWTYPE;
  v_success           NUMBER(1);

  -- Retrieve station visits from STORET
  CURSOR c_get_trip_station_visits(p_trip_is_nbr IN NUMBER, p_trip_org_id IN VARCHAR2) IS
    SELECT id_number vst_id,
           arrival_date,
           arrival_time,
           arrival_time_zone,
           departure_date,
           departure_time,
           tsrstvst_is_number is_nbr,
           tsrstvst_org_id org_id,
           tsmstatn_is_number statn_is_nbr,
           tsmstatn_org_id statn_org_id,
           blob_title
      FROM tsrstvst
     WHERE tsrtrip_is_number = p_trip_is_nbr
       AND tsrtrip_org_id = rpad(p_trip_org_id, 8)
     ORDER BY tsmstatn_is_number;

  -- Retrieve station visits from SIM
  CURSOR c_check_visit(p_visit_is_nbr IN NUMBER, p_visit_org_id IN VARCHAR2) IS
    SELECT ssv_seq
      FROM sim_station_visits
     WHERE ssv_tsrstvst_is_number = p_visit_is_nbr
       AND ssv_tsrstvst_org_id = rpad(p_visit_org_id, 8);

  -- Retrieve next Station Visit Sequence
  CURSOR c_get_next_visit_seq IS
    SELECT sim_station_visits_seq.NEXTVAL
      FROM dual;

  /* kms (6/8/01) - get all the station visits that exist on this trip in STORET and load
  them into the SIM table if they are not already there.  Update visits that are already
  in SIM with what's in STORET. */

  /* kms - loop through all station visits on the current trip */

BEGIN

  -- loop thru the Station Visits in STORET
  FOR v_visit IN c_get_trip_station_visits(p_trip_key.is_number, p_trip_key.org_id) LOOP
  
    -- try to find the station visit in SIM
    OPEN c_check_visit(v_visit.is_nbr, v_visit.org_id);
    FETCH c_check_visit
      INTO v_visit_seq;
  
    -- is the station visit in SIM?
    IF c_check_visit%NOTFOUND THEN
      -- if no
    
      -- get sequence for new record
      OPEN c_get_next_visit_seq;
      FETCH c_get_next_visit_seq
        INTO v_visit_seq;
      CLOSE c_get_next_visit_seq;
    
      -- create record to insert station visit into sim
      v_visit_rec.ssv_seq := v_visit_seq;
      v_visit_rec.ssv_stp_seq := p_trip_seq;
      v_visit_rec.ssv_id := rtrim(v_visit.vst_id); -- jah 3-10-2004 Removed trailing spaces
      v_visit_rec.ssv_start_date := v_visit.arrival_date;
      v_visit_rec.ssv_start_time := v_visit.arrival_time;
      v_visit_rec.ssv_arrival_time_zone := v_visit.arrival_time_zone;
      v_visit_rec.ssv_end_date := v_visit.departure_date;
      v_visit_rec.ssv_end_time := v_visit.departure_time;
      v_visit_rec.ssv_tsrstvst_is_number := v_visit.is_nbr;
      v_visit_rec.ssv_tsrstvst_org_id := v_visit.org_id;
      v_visit_rec.ssv_tsmstatn_is_number := v_visit.statn_is_nbr;
      v_visit_rec.ssv_tsmstatn_org_id := v_visit.statn_org_id;
      v_visit_rec.ssv_tsmorgan_is_number := v_org_is_nbr;
      v_visit_rec.ssv_tsmorgan_org_id := p_org_id;
      v_visit_rec.ssv_created_by_sim := 0;
      v_visit_rec.ssv_document := v_visit.blob_title;
    
      -- insert station visit into sim
      v_success := sim_import_result_extras.create_sim_station_visit(v_visit_rec, v_import_log_seq);
    
    ELSE
      -- if yes (i.e., the station visit is in SIM)
    
      -- update the station visit in SIM with the values from STORET.
      UPDATE sim_station_visits
         SET ssv_id = rtrim(v_visit.vst_id), -- jah 3-10-2004 Removed trailing spaces
             ssv_start_date = v_visit.arrival_date,
             ssv_start_time = v_visit.arrival_time,
             ssv_end_date = v_visit.departure_date,
             ssv_end_time = v_visit.departure_time
       WHERE ssv_seq = v_visit_seq;
    
    END IF;
    CLOSE c_check_visit;
  
  END LOOP;

EXCEPTION

  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in load_station_visits.');
     
 
END load_station_visits;

/****************************************************************************************

****************************************************************************************/
PROCEDURE get_next_stavst_id(p_stavst_id OUT VARCHAR2) IS

  /* This function returns the next available station visit number.
  Notice that the station visit number begins with 'UNA'.
  UNA is short for UNAssigned. This tells the migration code to
  determines the actual station visit number. */

BEGIN

  gv_last_visit_id := gv_last_visit_id + 1;
  IF gv_last_visit_id > 999 THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Exceeded the maximum number of visits allowed for one trip (999).');
     
 
    p_stavst_id := 'ERROR';
  ELSE
    p_stavst_id := 'UNA' || to_char(gv_last_visit_id);
  END IF;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get next visit id.');
     
 
END get_next_stavst_id;

/****************************************************************************************

****************************************************************************************/
FUNCTION check_existing_visit
(
  p_instr_choice     IN VARCHAR2,
  p_act_start_date   IN DATE,
  p_act_start_time   IN DATE,
  p_trip_seq         IN NUMBER,
  p_station_key      IN sim_import_result_extras.t_storet_pk_type,
  p_visit_start_date OUT DATE,
  p_visit_start_time OUT DATE,
  p_visit_end_date   OUT DATE,
  p_visit_end_time   OUT DATE,
  p_visit_key        OUT sim_import_result_extras.t_storet_pk_type,
  p_visit_seq        IN OUT NUMBER,
  p_visit_id_tab     IN OUT t_stvst_id_type,
  p_visit_id         OUT VARCHAR2
) RETURN NUMBER IS

  -- Retrieve station visit record by station, trip, and date
  CURSOR c_find_station_visit(cp_stat_is_nbr IN NUMBER, cp_stat_org_id IN VARCHAR2, cp_trip_seq IN NUMBER, cp_date IN DATE) IS
    SELECT ssv_seq,
           ssv_tsrstvst_is_number,
           ssv_tsrstvst_org_id,
           ssv_id
      FROM sim_station_visits
     WHERE trunc(ssv_start_date, 'DD') = trunc(cp_date, 'DD')
       AND ssv_tsmstatn_is_number = cp_stat_is_nbr
       AND ssv_tsmstatn_org_id = rpad(cp_stat_org_id, 8)
       AND ssv_stp_seq = cp_trip_seq;

  -- Retrieve station visit record by station, trip, date, and time
  CURSOR c_find_station_visit_dt(p_stat_is_nbr IN NUMBER, p_stat_org_id IN VARCHAR2, p_trip_seq IN NUMBER, p_date IN DATE, p_time IN DATE) IS
    SELECT ssv_seq,
           ssv_tsrstvst_is_number,
           ssv_tsrstvst_org_id,
           ssv_id
      FROM sim_station_visits
     WHERE nvl(to_char(ssv_start_time, 'HH24:MI:SS'), 'x') =
           nvl(to_char(p_time, 'HH24:MI:SS'), 'x')
       AND trunc(ssv_start_date, 'DD') = trunc(p_date, 'DD')
       AND ssv_tsmstatn_is_number = p_stat_is_nbr
       AND ssv_tsmstatn_org_id = rpad(p_stat_org_id, 8)
       AND ssv_stp_seq = p_trip_seq;

  /* Call this function when you need to find a station visit record in SIM. */

BEGIN

  -- Is the import instruction "Create one station visit per day"?
  IF p_instr_choice = 'RES-STVST-01' THEN
    -- if yes
  
    -- try to retrieve a station visit for this Station, Trip, and Date
    OPEN c_find_station_visit(p_station_key.is_number, p_station_key.org_id, p_trip_seq, p_act_start_date);
    FETCH c_find_station_visit
      INTO p_visit_seq, p_visit_key.is_number, p_visit_key.org_id, p_visit_id; -- jah 3-26-04
  
    -- Does a station visit exist for this Station, Trip, and Date?
    IF c_find_station_visit%NOTFOUND THEN
      -- if no
    
      -- tell subsequent code in this function to determine the next station visit number
      -- and also tell the calling routine the station visit record is not in SIM
      p_visit_seq := NULL;
    
      -- tell the calling routine the station visit record is not in STORET
      p_visit_key.is_number := NULL;
      p_visit_key.org_id := NULL;
      p_visit_id := NULL; -- jah 3-26-04
    
    END IF;
    CLOSE c_find_station_visit;
  
    -- return station visit start and end parameters to the calling routine
    p_visit_start_date := p_act_start_date;
    p_visit_end_date := NULL;
    p_visit_start_time := NULL;
    p_visit_end_time := NULL;
  
    -- Is the instruction "Create one station visit per date and time"?  
  ELSIF p_instr_choice = 'RES-STVST-03' THEN
    -- if yes
  
    -- try to retrieve a station visit for this Station, Trip, Date, and Time
    OPEN c_find_station_visit_dt(p_station_key.is_number, p_station_key.org_id, p_trip_seq, p_act_start_date, nvl(p_act_start_time, to_date('00:00', 'HH24:MI')) -- jah 3-9-2004 Changed from '00:01' to '00:00'
                                 );
    FETCH c_find_station_visit_dt
      INTO p_visit_seq, p_visit_key.is_number, p_visit_key.org_id, p_visit_id; -- jah 3-26-04
  
    -- Does a station visit exist for this Station, Trip, Date, and Time?
    IF c_find_station_visit_dt%NOTFOUND THEN
      -- if no
    
      -- tell subsequent code in this function to determine the next station visit number
      -- and also tell the calling routine the station visit record is not in SIM
      p_visit_seq := NULL;
    
      -- tell the calling routine the station visit record is not in STORET
      p_visit_key.is_number := NULL;
      p_visit_key.org_id := NULL;
      p_visit_id := NULL; -- jah 3-26-04
    
    END IF;
    CLOSE c_find_station_visit_dt;
  
    -- return station visit start and end parameters to the calling routine
    p_visit_start_date := p_act_start_date;
    p_visit_end_date := NULL;
    p_visit_end_time := NULL;
    p_visit_start_time := nvl(p_act_start_time, to_date('00:00', 'HH24:MI')); -- jah 3-9-2004 Changed from '00:01' to '00:00'
  
    IF (p_act_start_time IS NULL) THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Missing Activity Start Time is equivalent to Midnight.');
       
   
    END IF;
  
  END IF;

  -- Did we find a matching station visit record?
  IF p_visit_seq IS NULL THEN
    -- if no
  
    -- determine the next station visit number and return it to the calling routine
    get_next_stavst_id(p_visit_id);
  
  END IF;

  -- Tell calling routine NO exceptions
  RETURN NULL;

EXCEPTION

  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in check existing visit.');
     
 
    -- Tell calling routine OTHERS exception
    RETURN 1;
  
END check_existing_visit;

/****************************************************************************************

****************************************************************************************/
FUNCTION add_project_to_activity
(
  p_fa_seq         IN NUMBER,
  p_project_is_nbr IN NUMBER,
  p_project_org    IN VARCHAR2
) RETURN NUMBER IS

  v_act_project_seq NUMBER(10);

  CURSOR c_get_act_project_seq IS
    SELECT sim_activity_projects_seq.NEXTVAL
      FROM dual;

BEGIN
  /* kms - function that associates projects with activities in SIM */

  /* get next sequence */
  OPEN c_get_act_project_seq;
  FETCH c_get_act_project_seq
    INTO v_act_project_seq;
  IF c_get_act_project_seq%NOTFOUND THEN
    v_act_project_seq := NULL;
  END IF;
  CLOSE c_get_act_project_seq;

  INSERT INTO sim_activity_projects
    (sapj_seq,
     sapj_fa_seq,
     sapj_tsmproj_is_number,
     sapj_tsmproj_org_id)
  VALUES
    (v_act_project_seq, /* SAPJ_SEQ */
     p_fa_seq, /* SAPJ_FA_SEQ */
     p_project_is_nbr, /* SAPJ_TSMPROJ_IS_NUMBER */
     p_project_org /* SAPJ_TSMPROJ_ORG_ID */);

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in add project to activity.');
     
 
    RETURN 0;
  
END add_project_to_activity;

/****************************************************************************************

****************************************************************************************/
PROCEDURE remove_lab_remark_dups
(
  p_remark_codes_table IN OUT sim_import_pkg.col_table_type,
  p_no_remark_codes    IN OUT NUMBER
) IS
  -- DTW 10/21/03 - If duplicate lab remark codes are imported, remove them with a warning
  v_remark_codes_temp    sim_import_pkg.col_table_type;
  v_no_remark_codes_temp NUMBER := 0;
  v_dup_found            BOOLEAN;
BEGIN
  FOR v_cntr IN 1 .. p_no_remark_codes LOOP
    v_dup_found := FALSE;
    FOR v_inner_cntr IN 1 .. (v_cntr - 1) LOOP
      IF p_remark_codes_table(v_cntr) = p_remark_codes_table(v_inner_cntr) THEN
        v_dup_found := TRUE;
      END IF;
    END LOOP;
    IF v_dup_found = TRUE THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                           'Duplicate Lab Remark Code Will be Ignored.');
       
   
    ELSE
      v_no_remark_codes_temp := v_no_remark_codes_temp + 1;
      v_remark_codes_temp(v_no_remark_codes_temp) := p_remark_codes_table(v_cntr);
    END IF;
  END LOOP;

  -- Return remark code table and count with duplicates removed
  p_remark_codes_table := v_remark_codes_temp;
  p_no_remark_codes := v_no_remark_codes_temp;

END remove_lab_remark_dups;

/****************************************************************************************

****************************************************************************************/
FUNCTION add_remark_to_results
(
  p_rs_seq        IN NUMBER,
  p_remark_is_nbr IN NUMBER,
  p_remark_org_id IN VARCHAR2
) RETURN NUMBER IS

BEGIN

  INSERT INTO sim_result_lab_remarks
    (srlr_seq,
     srlr_rs_seq,
     srlr_tsrlbrmk_is_number,
     srlr_tsrlbrmk_org_id)
  VALUES
    (sim_srlr_seq.NEXTVAL,
     p_rs_seq,
     p_remark_is_nbr,
     p_remark_org_id);

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in add_remark_to_results.');
     
 
    RETURN 0;
END add_remark_to_results;

/****************************************************************************************

****************************************************************************************/
FUNCTION add_personnel_to_activity
(
  p_fa_seq       IN NUMBER,
  p_persn_is_nbr IN NUMBER,
  p_persn_org    IN VARCHAR2
) RETURN NUMBER IS

  v_act_persn_seq NUMBER(10);

  CURSOR c_get_act_persn_seq IS
    SELECT sim_activity_personnel_seq.NEXTVAL
      FROM dual;

BEGIN
  /* kms - function that associates personnel with activities in SIM. */

  /* get next sequence */
  OPEN c_get_act_persn_seq;
  FETCH c_get_act_persn_seq
    INTO v_act_persn_seq;
  CLOSE c_get_act_persn_seq;

  INSERT INTO sim_activity_personnel
    (sap_seq,
     sap_fa_seq,
     sap_tsmpersn_is_number,
     sap_tsmpersn_org_id)
  VALUES
    (v_act_persn_seq, /* SAP_SEQ */
     p_fa_seq, /* SAP_FA_SEQ */
     p_persn_is_nbr, /* SAP_TSMPERSN_IS_NUMBER */
     p_persn_org /* SAP_TSMPERSN_ORG_ID */);

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in add personnel to activity.');
     
 
    RETURN 0;
  
END add_personnel_to_activity;

/****************************************************************************************

****************************************************************************************/
FUNCTION add_cporg_to_activity
(
  p_fa_seq       IN NUMBER,
  p_cporg_is_nbr IN NUMBER,
  p_cporg_org_id IN VARCHAR2
) RETURN NUMBER IS

  v_sac_seq NUMBER(10);

  CURSOR c_sim_activity_cporg_seq IS
    SELECT sim_activity_cporg_seq.NEXTVAL
      FROM dual;

BEGIN

  -- get next sequence
  OPEN c_sim_activity_cporg_seq;
  FETCH c_sim_activity_cporg_seq
    INTO v_sac_seq;
  CLOSE c_sim_activity_cporg_seq;

  -- insert Cooperating Organization
  INSERT INTO sim_activity_cporg
    (sac_seq,
     sac_fa_seq,
     sac_tsmcporg_is_number,
     sac_tsmcporg_org_id)
  VALUES
    (v_sac_seq, /* SAC_SEQ */
     p_fa_seq, /* SAC_FA_SEQ */
     p_cporg_is_nbr, /* SAC_TSMCPORG_IS_NUMBER */
     p_cporg_org_id /* SAC_TSMCPORG_ORG_ID */);
  COMMIT;

  -- Return success
  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    -- output error message
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in add cooperating organization to activity.');
     
 
    -- return failure
    RETURN 0;
  
END add_cporg_to_activity;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_char_result_type(p_char_is_nbr IN NUMBER) RETURN VARCHAR2 IS

  v_char_result_type VARCHAR2(4);

  CURSOR c_get_char_type(p_char_is_nbr IN NUMBER) IS
    SELECT d_scr_type_cd
      FROM tsrchar
     WHERE tsrchar_is_number = p_char_is_nbr
       AND tsrchar_org_id = 'ZZZZZZZZ';

BEGIN
  /* kms - function that gets characteristic type in order to see if it has permitted values
  or if other checks need to be performed on the result. */

  OPEN c_get_char_type(p_char_is_nbr);
  FETCH c_get_char_type
    INTO v_char_result_type;
  IF c_get_char_type%NOTFOUND THEN
    v_char_result_type := NULL;
  END IF;
  CLOSE c_get_char_type;

  RETURN v_char_result_type;

END get_char_result_type;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_col_seq
(
  p_col_name    IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_fa_seq      IN NUMBER
) RETURN NUMBER IS

  v_col_seq NUMBER(10);

  CURSOR c_get_col_seq(cp_col_name IN VARCHAR2, cp_imp_cfg_seq IN NUMBER) IS
    SELECT sic_seq
      FROM sim_imp_cols,
           sim_imp_cfg_dets,
           sim_imp_cfg
     WHERE sic_seq = sicdt_sic_seq
       AND sicc_seq = sicdt_sicc_seq
       AND sicc_seq = cp_imp_cfg_seq
       AND sic_name = cp_col_name;

BEGIN

  OPEN c_get_col_seq(p_col_name, p_imp_cfg_seq);
  FETCH c_get_col_seq
    INTO v_col_seq;
  CLOSE c_get_col_seq;

  RETURN v_col_seq;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - In get col seq.');
     
 
    RETURN NULL;
  
END get_col_seq;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_char_defaults
(
  p_analysis_seq IN NUMBER,
  p_row_pk       IN sim_import_result_extras.t_storet_pk_type,
  p_imp_cfg_seq  IN NUMBER,
  p_col_seq      IN NUMBER,
  p_fa_seq       IN NUMBER
) RETURN NUMBER IS

  v_uom_name   VARCHAR2(10);
  
  e_bad_uom EXCEPTION;
  e_bad_def_map EXCEPTION;
  e_bad_def_pk EXCEPTION;

  CURSOR c_get_storet_row_defs(p_row_pk IN sim_import_result_extras.t_storet_pk_type) IS
    SELECT def.tsmprmvl_is_number fraction_is,
           def.tsmprmvl0is_number voltinism_is,
           def.tsmprmvl1is_number habit_is,
           def.uom_name uom_name,
           def.dur_basis_type_nm dur_basis,
           def.statistic_type_nm stat_type,
           def.value_type_name value_type,
           def.temp_basis_lvl_nm temp_basis,
           def.wt_basis_type_nm wieght_basis,
           def.species_number species,
           def.taxon_pollution pollution_tol,
           def.fnctional_feed_grp funct_feed_group,
           def.trophic_level trophic_level,
           def.particle_size_basis particle_size_basis, -- jah 02-20-04  Added Particle Size Basis
           def.tsranlpr_is_number anlpr_is_nbr,
           def.tsranlpr_org_id anlpr_org_id,
           def.tsrlspp_is_number lspp_is_number,
           def.tsrlspp_org_id lspp_org_id,
           def.lower_range_value lower_range_value, -- Added Lower/Upper range support for Groups and Rows
           def.upper_range_value upper_range_value, -- Added Lower/Upper range support for Groups and Rows
           ccga.tsrchar_is_number char_is_nbr,
           ccga.tsrchar_org_id char_org_id
      FROM tsrchdef def,
           tsrccga ccga
     WHERE def.tsrccga_is_number = ccga.tsrccga_is_number
       AND def.tsrccga_org_id = ccga.tsrccga_org_id
       AND def.tsrchdef_is_number = p_row_pk.is_number
       AND def.tsrchdef_org_id = rpad(p_row_pk.org_id, 8);

BEGIN
  /* function that gets characteristic defaults from STORET to be stored with results in SIM */
  IF p_row_pk.is_number IS NOT NULL THEN
    /* kms - used characteristic group and row */
    OPEN c_get_storet_row_defs(p_row_pk);
    FETCH c_get_storet_row_defs
      INTO v_acode_defs_rec.fraction_is, v_acode_defs_rec.voltinism_is, v_acode_defs_rec.habit_is, v_uom_name, v_acode_defs_rec.dur_basis, v_acode_defs_rec.stat_type, v_acode_defs_rec.value_type, v_acode_defs_rec.temp_basis, v_acode_defs_rec.weight_basis, v_acode_defs_rec.species, v_acode_defs_rec.pollution_tol, v_acode_defs_rec.funct_feed_grp, v_acode_defs_rec.trophic_level, v_acode_defs_rec.particle_size_basis, -- jah 02-20-04  Added Particle Size Basis
    v_acode_defs_rec.anlpr_pk.is_number, v_acode_defs_rec.anlpr_pk.org_id, v_acode_defs_rec.labprep_pk.is_number, v_acode_defs_rec.labprep_pk.org_id, v_acode_defs_rec.lower_range_value, -- Added Lower/Upper range support for Groups and Rows
    v_acode_defs_rec.upper_range_value, -- Added Lower/Upper range support for Groups and Rows
    v_acode_defs_rec.char_pk.is_number, v_acode_defs_rec.char_pk.org_id;
  
    IF c_get_storet_row_defs%NOTFOUND THEN
      /* kms - row primary key could not be found in STORET, raise exception */
      CLOSE c_get_storet_row_defs;
      RAISE e_bad_def_pk;
    END IF;
  
    CLOSE c_get_storet_row_defs;
  
    v_acode_defs_rec.lab_pk.is_number := NULL;
    v_acode_defs_rec.lab_pk.org_id := NULL;
  
  END IF;

  /* look up pk, allowable value sequences */
  /* get uom pk */
  v_char_result_type := get_char_result_type(v_acode_defs_rec.char_pk.is_number);

  IF v_uom_name LIKE '    %' THEN
    v_uom_name := 'None      ';
  END IF;

  -- Get the Unit of Measure
  v_acode_defs_rec.uom_pk := sim_import_result_extras.get_uom_is_nbr(v_uom_name, p_fa_seq, v_import_log_seq, v_line_cnt);

  -- jah 3-9-2004
  -- Is the unit of measure valid?  
  IF v_acode_defs_rec.uom_pk.is_number IS NULL THEN
    -- if no
    RAISE e_bad_uom;
  END IF;

  /*  SF 8/6/2002 Having changed all this for V2 we are no longer storing these sequences.
      * get duration basis sequence
      * get stat type sequence
      * get value type sequence
      * get temp basis sequence
  */

  -- Tell calling routine NO ERRORS occurred
  RETURN 1;

EXCEPTION

  WHEN e_bad_uom THEN
  
    /* jah 3-9-2004
    Function sim_import_result_extras.get_uom_is_nbr has already output the error message for this.
    This error happens when the EPA updates the Unit of Measure table (TSROUM) but not the
    characteristic default table (TSMCHDEF) */
  
    -- re-initialize v_acode_def_recs
    v_acode_defs_rec := v_acode_init_rec;
  
    -- Tell calling routine an error occurred
    RETURN NULL;
  
  WHEN e_bad_def_map THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unable to get char defaults from STORET due to faulty mapping.');
     
 
    -- re-initialize v_acode_def_recs
    v_acode_defs_rec := v_acode_init_rec;
  
    -- Tell calling routine an error occurred
    RETURN NULL;
  
  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get char defaults.');
     
 
    -- re-initialize v_acode_def_recs
    v_acode_defs_rec := v_acode_init_rec;
  
    -- Tell calling routine an error occurred
    RETURN NULL;
  
END get_char_defaults;

/****************************************************************************************

 ****************************************************************************************/
FUNCTION check_allow_species(p_char_is_number IN NUMBER) RETURN NUMBER IS

  -- This function checks if the taxon provided can have an associated species based
  -- on the rank of the taxon.  The function returns a '0' if a species is NOT allowed
  -- and a 1 if it is.
  v_rank_code CHAR(8);

  CURSOR c_check_species IS
    SELECT taxon_rank_code
      FROM tsrchar
     WHERE tsrchar_is_number = p_char_is_number;

BEGIN
  OPEN c_check_species;
  FETCH c_check_species
    INTO v_rank_code;
  IF v_rank_code IN ('SPE     ', 'FOR     ', 'SSP     ', 'VAR     ') THEN
    RETURN 0;
  ELSE
    RETURN 1;
  END IF;
  CLOSE c_check_species;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
  
END check_allow_species;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_class_descriptor
(
  p_orig_item      IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_char_is_nbr    IN NUMBER,
  p_item_name      IN VARCHAR2,
  p_col_seq        IN NUMBER,
  p_fa_seq         IN NUMBER,
  p_line_cnt       IN NUMBER,
  p_import_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(40);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_value   VARCHAR2(12);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);
  v_storet_pk      sim_import_result_extras.t_storet_pk_type;

  CURSOR c_check_char_pv(p_result IN VARCHAR2, p_char_is_nbr IN NUMBER) IS
    SELECT short_name
      FROM tsrcpv
     WHERE short_name = rpad(p_result, 12)
       AND tsrchar_is_number = p_char_is_nbr;

BEGIN

  v_storet_value := NULL;

  /* get the char group is number - check translations and STORET tables */
  v_storet_pk := sim_import_result_extras.check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, p_import_log_seq, p_line_cnt, v_converted_item, p_fa_seq);

  v_storet_value := v_converted_item;

  IF v_storet_pk.is_number IS NULL
     AND NOT v_trans_exist THEN
    OPEN c_check_char_pv(v_converted_item, p_char_is_nbr);
    FETCH c_check_char_pv
      INTO v_storet_value;
    IF c_check_char_pv%NOTFOUND THEN
      v_storet_value := NULL;
    END IF;
    CLOSE c_check_char_pv;
  
    IF v_storet_value IS NULL THEN
      v_instr_id := 'RES-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
      IF v_instr_code = 'RES-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
         
     
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
         
     
      END IF;
    ELSIF v_storet_value IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                            p_item_name || ', ' || p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
         
     
    END IF;
  ELSIF v_storet_pk.is_number IS NULL
        AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - checking class descriptor value.');
     
 
    RETURN NULL;
END validate_class_descriptor;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_prmvl_item
(
  p_orig_item      IN CHAR,
  p_imp_cfg_seq    IN NUMBER,
  p_item_name      IN VARCHAR2,
  p_col_seq        IN NUMBER,
  p_table_name     IN VARCHAR,
  p_column_name    IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_line_cnt       IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(40);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_value   VARCHAR2(40);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);
  v_storet_pk      sim_import_result_extras.t_storet_pk_type;

  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT field_value
      FROM tsmprmvl
     WHERE table_name = rpad(p_table_name, 8)
       AND field_name = p_column_name
       AND field_value = p_item;

BEGIN

  v_storet_value := NULL;

  /* get the char group is number - check translations and STORET tables */
  v_storet_pk := sim_import_result_extras.check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, p_import_log_seq, p_line_cnt, v_converted_item, p_fa_seq);

  v_storet_value := v_converted_item;

  IF v_storet_pk.is_number IS NULL
     AND NOT v_trans_exist THEN
    OPEN c_check_prmvl(v_converted_item);
    FETCH c_check_prmvl
      INTO v_storet_value;
    IF c_check_prmvl %NOTFOUND THEN
      v_storet_value := NULL;
    END IF;
    CLOSE c_check_prmvl;
  
    IF v_storet_value IS NULL THEN
      v_instr_id := 'RES-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
      IF v_instr_code = 'RES-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
         
     
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
         
     
      END IF;
    END IF;
  ELSIF v_storet_pk.is_number IS NULL
        AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - checking STORET PRMVL.');
     
 
    RETURN NULL;
END validate_prmvl_item;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_prmvl_key
(
  p_orig_item      IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_item_name      IN VARCHAR2,
  p_col_seq        IN NUMBER,
  p_table_name     IN VARCHAR,
  p_column_name    IN VARCHAR2,
  p_fa_seq         IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_line_cnt       IN NUMBER
) RETURN sim_import_result_extras.t_storet_pk_type IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(40);
  v_trans_exist    BOOLEAN := TRUE;
  --v_storet_value             VARCHAR2(40);
  v_instr_id   VARCHAR2(8);
  v_instr_code VARCHAR2(15);
  v_storet_key sim_import_result_extras.t_storet_pk_type;

  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT tsmprmvl_is_number,
           tsmprmvl_org_id
      FROM tsmprmvl
     WHERE table_name = rpad(p_table_name, 8)
       AND field_name = p_column_name
       AND field_value = p_item;

BEGIN

  /* get the char group is number - check translations and STORET tables */
  v_storet_key := sim_import_result_extras.check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, p_import_log_seq, p_line_cnt, v_converted_item, p_fa_seq);

  IF v_storet_key.is_number IS NULL
     AND NOT v_trans_exist THEN
    OPEN c_check_prmvl(v_converted_item);
    FETCH c_check_prmvl
      INTO v_storet_key.is_number, v_storet_key.org_id;
    IF c_check_prmvl %NOTFOUND THEN
      v_storet_key.is_number := NULL;
      v_storet_key.org_id := NULL;
    END IF;
    CLOSE c_check_prmvl;
  
    IF v_storet_key.is_number IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'RES-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
      IF v_instr_code = 'RES-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
         
     
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
         
     
      END IF;
    ELSIF v_storet_key.is_number IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
      
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
       
   
    END IF;
  ELSIF v_storet_key.is_number IS NULL
        AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_key.is_number := NULL;
    v_storet_key.org_id := NULL;
  END IF;

  RETURN v_storet_key;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in get STORET PRMVL key.');
     
 
    RETURN NULL;
END get_prmvl_key;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_sample_matrix
(
  p_orig_item      IN CHAR,
  p_imp_cfg_seq    IN NUMBER,
  p_fa_seq         IN NUMBER,
  p_import_log_seq IN NUMBER,
  p_col_seq        IN NUMBER,
  p_line_cnt       IN NUMBER
) RETURN sim_import_result_extras.t_storet_pk_type IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(40);
  v_trans_exist    BOOLEAN := TRUE;
  v_matrx_key      sim_import_result_extras.t_storet_pk_type;
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);

  CURSOR c_check_matrx(p_item IN VARCHAR2) IS
    SELECT tsrmatrx_is_number,
           tsrmatrx_org_id
      FROM tsrmatrx
     WHERE NAME = p_item;

BEGIN

  v_matrx_key.is_number := NULL;

  /* get the char group is number - check translations and STORET tables */
  v_matrx_key := sim_import_result_extras.check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, p_import_log_seq, p_line_cnt, v_converted_item, p_fa_seq);

  IF v_matrx_key.is_number IS NULL
     AND NOT v_trans_exist THEN
    OPEN c_check_matrx(v_converted_item);
    FETCH c_check_matrx
      INTO v_matrx_key.is_number, v_matrx_key.org_id;
    IF c_check_matrx %NOTFOUND THEN
      v_matrx_key.is_number := NULL;
    END IF;
    CLOSE c_check_matrx;
  
    IF v_matrx_key.is_number IS NULL THEN
      v_instr_id := 'RES-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq, v_import_log_seq, v_line_cnt);
      IF v_instr_code = 'RES-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        sim_import_result_extras.create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq, v_import_log_seq, v_line_cnt);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             'Sample Matrix, ' ||
                                             v_converted_item ||
                                             ', is not a valid value - translation record has been created.');
         
     
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             'Sample Matrix, ' ||
                                             v_converted_item ||
                                             ', is not a valid value.');
         
     
      END IF;
    ELSIF v_matrx_key.is_number IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Sample Matrix, ' || p_orig_item ||
                                           ', has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
       
   
    END IF;
  ELSIF v_matrx_key.is_number IS NULL
        AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_matrx_key.is_number := NULL;
    v_matrx_key.org_id := NULL;
  END IF;

  RETURN v_matrx_key;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - checking STORET TSRMATRX.');
    RETURN NULL;
END validate_sample_matrix;

/****************************************************************************************

 ****************************************************************************************/
FUNCTION check_existing_mtpc
(
  p_brg_type IN VARCHAR2,
  p_fa_seq   IN NUMBER
) RETURN NUMBER IS

  v_return NUMBER(1);
  v_seq    NUMBER(10);

  CURSOR c_check_mtpc IS
    SELECT brg_seq
      FROM sim_bio_results_grp
     WHERE brg_fa_seq = p_fa_seq
       AND brg_type = 'Multi-Taxon Population Census';

BEGIN
  OPEN c_check_mtpc;
  FETCH c_check_mtpc
    INTO v_seq;
  IF c_check_mtpc%FOUND THEN
    v_return := 2;
  ELSE
    v_return := 0;
  END IF;
  CLOSE c_check_mtpc;
  RETURN v_return;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in check existing MTPC.');
     
 
    RETURN 1;
END check_existing_mtpc;

/****************************************************************************************

****************************************************************************************/
FUNCTION check_if_station_exists
(
  p_station_cd IN VARCHAR2,
  p_org_id     IN VARCHAR2,
  p_sta_seq    OUT NUMBER
) RETURN NUMBER IS

  --  SF-12/11/02
  -- In this function we check to see if the station existis in SIM, and if all the wells and locations
  -- that exist in storet also are reflected in SIM.
  
  v_storet_station_cd VARCHAR2(15) := p_station_cd;
  v_sim_station_cd    VARCHAR2(15) := p_station_cd;
  v_return            NUMBER(1) := 0;
  v_seq               NUMBER(10);
  v_is_nbr            NUMBER(10);
  v_success           NUMBER(1);
  v_sta_rec           sim_stations%ROWTYPE;
  v_stor_sta_rec      tsmstatn%ROWTYPE;
  v_tsmstatn_is_nbr   NUMBER(12);
  v_type              NUMBER(8);
  v_well_rec          sim_station_wells%ROWTYPE;
  v_storet_wells      tsmwell%ROWTYPE;
  v_loc_rec           sim_station_locations%ROWTYPE;
  v_storet_loc_rec    tsmalp%ROWTYPE;
  v_well_seq          NUMBER(10);
  v_loc_seq           NUMBER(10);
  
  e_bad_well EXCEPTION;
  e_bad_station EXCEPTION;

  CURSOR c_compare_station_id IS
    SELECT identification_cd,
           tsmstatn_is_number,
           tsmvstc_is_number
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_cd, 15)
       AND tsmstatn_org_id = rpad(p_org_id, 8);

  CURSOR c_compare_sim_stat_id(p_station_cd IN VARCHAR2, p_station_org_id IN VARCHAR2, p_tsmstatn_is_nbr IN NUMBER) IS
    SELECT sta_seq,
           sta_id
      FROM sim_stations
     WHERE sta_id = p_station_cd
       AND sta_is_number = p_tsmstatn_is_nbr
       AND sta_org_id = rpad(p_org_id, 8);

  CURSOR c_compare_sim_well(p_station_seq IN NUMBER, p_well_is_nbr IN NUMBER, p_well_org_id IN VARCHAR2) IS
    SELECT ssw_is_number
      FROM sim_station_wells
     WHERE ssw_sta_seq = p_station_seq
       AND ssw_is_number = p_well_is_nbr
       AND ssw_org_id = rpad(p_well_org_id, 8);

  CURSOR c_compare_sim_loc(p_station_seq IN NUMBER, p_tsmalp_is_number IN NUMBER, p_tsmalp_org_id IN VARCHAR2) IS
    SELECT stl_is_number
      FROM sim_station_locations
     WHERE stl_sta_seq = p_station_seq
       AND stl_is_number = p_tsmalp_is_number
       AND stl_org_id = rpad(p_tsmalp_org_id, 8);

  CURSOR c_get_station_seq IS
    SELECT sim_stations_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_well_seq IS
    SELECT sim_station_wells_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_location_seq IS
    SELECT sim_station_locations_seq.NEXTVAL
      FROM dual;

  CURSOR c_storet_station IS
    SELECT *
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_cd, 15)
       AND tsmstatn_org_id = rpad(p_org_id, 8);

  CURSOR c_storet_wells(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT *
      FROM tsmwell
     WHERE tsmstatn_is_number = v_tsmstatn_is_nbr
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_storet_loc(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT *
      FROM tsmalp
     WHERE tsmstatn0is_number = p_tsmstatn_is_number
       AND tsmstatn0org_id = rpad(p_tsmstatn_org_id, 8);

BEGIN
  v_return := 1;

  -- Check STORET 
  OPEN c_compare_station_id;
  FETCH c_compare_station_id
    INTO v_storet_station_cd, v_tsmstatn_is_nbr, v_type;

  IF c_compare_station_id%NOTFOUND THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Station, ' || p_station_cd ||
                                         ', does not exist in STORET.');
     
 
    v_return := 0;
  
  ELSE
    --Check to see if it's in SIM, too
    BEGIN
      OPEN c_compare_sim_stat_id(p_station_cd, p_org_id, v_tsmstatn_is_nbr);
      FETCH c_compare_sim_stat_id
        INTO v_seq, v_sim_station_cd;
      IF c_compare_sim_stat_id%NOTFOUND THEN
        OPEN c_get_station_seq;
        FETCH c_get_station_seq
          INTO v_seq;
        CLOSE c_get_station_seq;
        OPEN c_storet_station;
        FETCH c_storet_station
          INTO v_stor_sta_rec;
        CLOSE c_storet_station;
        v_sta_rec.sta_seq := v_seq;
        v_sta_rec.sta_is_number := v_stor_sta_rec.tsmstatn_is_number;
        v_sta_rec.sta_org_id := rtrim(v_stor_sta_rec.tsmstatn_org_id);
        v_sta_rec.sta_id := rtrim(v_stor_sta_rec.identification_cd);
        v_sta_rec.sta_name := v_stor_sta_rec.NAME;
        v_sta_rec.sta_estab_date := v_stor_sta_rec.establishment_date;
        v_sta_rec.sta_desc := v_stor_sta_rec.description_text;
        v_sta_rec.sta_tsmorgan_org_id := p_org_id;
        v_sta_rec.sta_tsmorgan_is_number := v_stor_sta_rec.tsmorgan_is_number;
        v_sta_rec.sta_tsmvstc_is_number := v_stor_sta_rec.tsmvstc_is_number;
        v_sta_rec.sta_tsmvstc_org_id := v_stor_sta_rec.tsmvstc_org_id;
        v_sta_rec.sta_status := 'U';
        v_sta_rec.sta_export_seq := NULL;
        v_sta_rec.sta_epa_key_id := v_stor_sta_rec.epa_key_identifier;
        v_sta_rec.sta_zid_rel := v_stor_sta_rec.zid_relation_code;
        v_sta_rec.sta_influence_area := v_stor_sta_rec.influence_area;
        v_sta_rec.sta_travel_dir := v_stor_sta_rec.travel_dir_txt;
        v_sta_rec.sta_water_depth := v_stor_sta_rec.water_depth;
        v_sta_rec.sta_water_depth_unit := v_stor_sta_rec.water_depth_unit;
        v_sta_rec.sta_ecoregion := v_stor_sta_rec.ecoregion_name;
        v_sta_rec.sta_document := v_stor_sta_rec.blob_title;
        v_success := create_station(v_sta_rec);
        IF v_success = 1 THEN
          -- go ahead and get the wells
          FOR v_storet_wells IN c_storet_wells(v_tsmstatn_is_nbr, p_org_id) LOOP
            OPEN c_get_well_seq;
            FETCH c_get_well_seq
              INTO v_well_seq;
            CLOSE c_get_well_seq;
          
            v_well_rec.ssw_seq := v_well_seq;
            v_well_rec.ssw_sta_seq := v_seq;
            v_well_rec.ssw_well_status := v_storet_wells.status_code;
            v_well_rec.ssw_is_number := v_storet_wells.tsmwell_is_number;
            v_well_rec.ssw_org_id := v_storet_wells.tsmwell_org_id;
            v_well_rec.ssw_name := v_storet_wells.NAME;
            v_well_rec.ssw_well_no := v_storet_wells.id_number;
            v_well_rec.ssw_well_use := v_storet_wells.use_code;
            v_well_rec.ssw_develop_method := v_storet_wells.developmnt_mthd_cd;
            v_well_rec.ssw_well_const_meth := v_storet_wells.constructn_mthd_cd;
            v_well_rec.ssw_water_use := v_storet_wells.wtr_primry_use_cd;
            v_well_rec.ssw_gradient_type := v_storet_wells.gradient_type_code;
            v_well_rec.ssw_depth_completion := v_storet_wells.depth_cmpltn_msr;
            v_well_rec.ssw_depth_completion_unit := v_storet_wells.dpth_cmpltn_unt_cd;
            v_well_rec.ssw_depth_hole := v_storet_wells.depth_of_hole_msr;
            v_well_rec.ssw_depth_hole_unit := v_storet_wells.depth_hole_unt_cd;
            v_well_rec.ssw_well_height := v_storet_wells.well_head_ht_msr;
            v_well_rec.ssw_well_height_unit := v_storet_wells.wl_hd_ht_unit_cd;
            v_well_rec.ssw_casing_height := v_storet_wells.casing_hgt_msr;
            v_well_rec.ssw_casing_height_unit := v_storet_wells.casing_hgt_unit_cd;
            v_well_rec.ssw_disinfected := v_storet_wells.disinfected_ind_cd;
            v_well_rec.ssw_const_start_date := v_storet_wells.constructn_strt_dt;
            v_well_rec.ssw_const_end_date := v_storet_wells.constructn_end_dt;
            v_well_rec.ssw_init_pump_dur := v_storet_wells.init_pump_dur;
            v_well_rec.ssw_init_pump_dur_unit := v_storet_wells.int_pmp_dur_unt_cd;
            v_well_rec.ssw_init_pump_rate := v_storet_wells.init_pump_rate;
            v_well_rec.ssw_init_pump_rate_unit := v_storet_wells.int_pmp_rat_unt_cd;
            v_well_rec.ssw_natural_flow := v_storet_wells.naturl_flow_ind_cd;
            v_well_rec.ssw_init_borehole_diam := v_storet_wells.init_borehole_diam;
            v_well_rec.ssw_init_borehole_diam_unit := v_storet_wells.init_bhole_diam_un;
            v_well_rec.ssw_tsmstatn_is_number := v_storet_wells.tsmstatn_is_number;
            v_well_rec.ssw_tsmstatn_org_id := v_storet_wells.tsmstatn_org_id;
            v_well_rec.ssw_status := 'U';
            v_well_rec.ssw_import_seq := NULL;
            v_well_rec.ssw_export_seq := NULL;
            v_success := create_well(v_well_rec);
            IF v_success != 1 THEN
              RAISE e_bad_well;
            END IF;
          END LOOP;
        
          --now locations
          FOR v_storet_loc IN c_storet_loc(v_tsmstatn_is_nbr, p_org_id) LOOP
            OPEN c_get_location_seq;
            FETCH c_get_location_seq
              INTO v_loc_seq;
            CLOSE c_get_location_seq;
            v_loc_rec.stl_seq := v_loc_seq;
            v_loc_rec.stl_sta_seq := v_seq;
            v_loc_rec.stl_is_number := v_storet_loc.tsmalp_is_number;
            v_loc_rec.stl_org_id := v_storet_loc.tsmalp_org_id;
            v_loc_rec.stl_point_name := v_storet_loc.point_name;
            v_loc_rec.stl_type_code := v_storet_loc.type_code;
            v_loc_rec.stl_sequence_number := v_storet_loc.sequence_number;
            v_loc_rec.stl_elev_datum := NULL;
            v_loc_rec.stl_elev_method := NULL;
            v_loc_rec.stl_elev_msr_date := v_storet_loc.elvtn_msr_dt;
            v_loc_rec.stl_elevation_msr := v_storet_loc.elevation_msr;
            v_loc_rec.stl_elevation_unit := NULL;
            v_loc_rec.stl_geopos_datum := NULL;
            v_loc_rec.stl_geopos_method := NULL;
            v_loc_rec.stl_gps_scale := v_storet_loc.geopstng_scale_txt;
            v_loc_rec.stl_lat_dir := v_storet_loc.lat_direction;
            v_loc_rec.stl_lat_degrees := v_storet_loc.lat_degree_msr;
            v_loc_rec.stl_lat_minutes := v_storet_loc.lat_minute_msr;
            v_loc_rec.stl_lat_seconds := v_storet_loc.lat_second_msr;
            v_loc_rec.stl_long_dir := v_storet_loc.long_direction;
            v_loc_rec.stl_long_degrees := v_storet_loc.long_degree_msr;
            v_loc_rec.stl_long_minutes := v_storet_loc.long_minute_msr;
            v_loc_rec.stl_long_seconds := v_storet_loc.long_second_msr;
            v_loc_rec.stl_lat_long_msr_date := v_storet_loc.lat_long_msr_dt;
            v_loc_rec.stl_latlong_accurcy := NULL;
            v_loc_rec.stl_latlong_accurcy_unit := NULL;
            v_loc_rec.stl_tsmrrr_is_number := NULL;
            v_loc_rec.stl_tsmrrr_org_id := NULL;
            v_loc_rec.stl_tsmfhu_is_number := v_storet_loc.tsmfhu_is_number;
            v_loc_rec.stl_tsmfhu_org_id := v_storet_loc.tsmfhu_org_id;
            v_loc_rec.stl_tsmstatn0is_number := v_storet_loc.tsmstatn0is_number;
            v_loc_rec.stl_tsmstatn0org_id := v_storet_loc.tsmstatn0org_id;
            v_loc_rec.stl_tsmwell_is_number := v_storet_loc.tsmwell_is_number;
            v_loc_rec.stl_tsmwell_org_id := v_storet_loc.tsmwell_org_id;
            v_loc_rec.stl_tsmgeopa_is_number := v_storet_loc.tsmgeopa_is_number;
            v_loc_rec.stl_tsmgeopa_org_id := v_storet_loc.tsmgeopa_org_id;
            v_loc_rec.stl_lat_dec_minutes := NULL;
            v_loc_rec.stl_long_dec_minutes := NULL;
            v_loc_rec.stl_tsmpipe_is_number := NULL;
            v_loc_rec.stl_tsmpipe_org_id := NULL;
            v_loc_rec.stl_tsmstatn1org_id := NULL;
            v_loc_rec.stl_tsmstatn1is_number := NULL;
            v_loc_rec.stl_tsmgeopa0is_number := NULL;
            v_loc_rec.stl_tsmgeopa0org_id := NULL;
            v_loc_rec.stl_tsmnal_cd := v_storet_loc.tsmnal_cd;
            v_loc_rec.stl_tsmnal_state := v_storet_loc.tsmnal_state;
            v_loc_rec.stl_nrcs := v_storet_loc.nrcs_wtrsd_id_num;
            v_loc_rec.stl_status := 'U';
            v_loc_rec.stl_import_seq := NULL;
            v_loc_rec.stl_export_seq := NULL;
          
            v_success := create_station_location(v_loc_rec);
          END LOOP;
        ELSE
          v_return := 0;
          RAISE e_bad_station;
        END IF;
      ELSE
        --the station is there, check its wells and locations
        FOR v_storet_wells IN c_storet_wells(v_tsmstatn_is_nbr, p_org_id) LOOP
          OPEN c_compare_sim_well(v_seq, v_storet_wells.tsmwell_is_number, v_storet_wells.tsmwell_org_id);
          FETCH c_compare_sim_well
            INTO v_is_nbr;
          IF c_compare_sim_well%NOTFOUND THEN
            OPEN c_get_well_seq;
            FETCH c_get_well_seq
              INTO v_well_seq;
            CLOSE c_get_well_seq;
            v_well_rec.ssw_seq := v_well_seq;
            v_well_rec.ssw_sta_seq := v_seq;
            v_well_rec.ssw_well_status := v_storet_wells.status_code;
            v_well_rec.ssw_is_number := v_storet_wells.tsmwell_is_number;
            v_well_rec.ssw_org_id := v_storet_wells.tsmwell_org_id;
            v_well_rec.ssw_name := v_storet_wells.NAME;
            v_well_rec.ssw_well_no := v_storet_wells.id_number;
            v_well_rec.ssw_well_use := v_storet_wells.use_code;
            v_well_rec.ssw_develop_method := v_storet_wells.developmnt_mthd_cd;
            v_well_rec.ssw_well_const_meth := v_storet_wells.constructn_mthd_cd;
            v_well_rec.ssw_water_use := v_storet_wells.wtr_primry_use_cd;
            v_well_rec.ssw_gradient_type := v_storet_wells.gradient_type_code;
            v_well_rec.ssw_depth_completion := v_storet_wells.depth_cmpltn_msr;
            v_well_rec.ssw_depth_completion_unit := v_storet_wells.dpth_cmpltn_unt_cd;
            v_well_rec.ssw_depth_hole := v_storet_wells.depth_of_hole_msr;
            v_well_rec.ssw_depth_hole_unit := v_storet_wells.depth_hole_unt_cd;
            v_well_rec.ssw_well_height := v_storet_wells.well_head_ht_msr;
            v_well_rec.ssw_well_height_unit := v_storet_wells.wl_hd_ht_unit_cd;
            v_well_rec.ssw_casing_height := v_storet_wells.casing_hgt_msr;
            v_well_rec.ssw_casing_height_unit := v_storet_wells.casing_hgt_unit_cd;
            v_well_rec.ssw_disinfected := v_storet_wells.disinfected_ind_cd;
            v_well_rec.ssw_const_start_date := v_storet_wells.constructn_strt_dt;
            v_well_rec.ssw_const_end_date := v_storet_wells.constructn_end_dt;
            v_well_rec.ssw_init_pump_dur := v_storet_wells.init_pump_dur;
            v_well_rec.ssw_init_pump_dur_unit := v_storet_wells.int_pmp_dur_unt_cd;
            v_well_rec.ssw_init_pump_rate := v_storet_wells.init_pump_rate;
            v_well_rec.ssw_init_pump_rate_unit := v_storet_wells.int_pmp_rat_unt_cd;
            v_well_rec.ssw_natural_flow := v_storet_wells.naturl_flow_ind_cd;
            v_well_rec.ssw_init_borehole_diam := v_storet_wells.init_borehole_diam;
            v_well_rec.ssw_init_borehole_diam_unit := v_storet_wells.init_bhole_diam_un;
            v_well_rec.ssw_tsmstatn_is_number := v_storet_wells.tsmstatn_is_number;
            v_well_rec.ssw_tsmstatn_org_id := v_storet_wells.tsmstatn_org_id;
            v_well_rec.ssw_status := 'U';
            v_well_rec.ssw_import_seq := NULL;
            v_well_rec.ssw_export_seq := NULL;
          
            v_success := create_well(v_well_rec);
            IF v_success != 1 THEN
              RAISE e_bad_well;
            END IF;
          END IF;
          CLOSE c_compare_sim_well;
        END LOOP;
        FOR v_storet_loc_rec IN c_storet_loc(v_tsmstatn_is_nbr, p_org_id) LOOP
          OPEN c_compare_sim_loc(v_seq, v_storet_loc_rec.tsmalp_is_number, v_storet_loc_rec.tsmalp_org_id);
          FETCH c_compare_sim_loc
            INTO v_is_nbr;
          IF c_compare_sim_loc%NOTFOUND THEN
          
            OPEN c_get_location_seq;
            FETCH c_get_location_seq
              INTO v_loc_seq;
            CLOSE c_get_location_seq;
            v_loc_rec.stl_seq := v_loc_seq;
            v_loc_rec.stl_sta_seq := v_seq;
            v_loc_rec.stl_is_number := v_storet_loc_rec.tsmalp_is_number;
            v_loc_rec.stl_org_id := v_storet_loc_rec.tsmalp_org_id;
            v_loc_rec.stl_point_name := v_storet_loc_rec.point_name;
            v_loc_rec.stl_type_code := v_storet_loc_rec.type_code;
            v_loc_rec.stl_sequence_number := v_storet_loc_rec.sequence_number;
            v_loc_rec.stl_elev_datum := NULL;
            v_loc_rec.stl_elev_method := NULL;
            v_loc_rec.stl_elev_msr_date := v_storet_loc_rec.elvtn_msr_dt;
            v_loc_rec.stl_elevation_msr := v_storet_loc_rec.elevation_msr;
            v_loc_rec.stl_elevation_unit := NULL;
            v_loc_rec.stl_geopos_datum := NULL;
            v_loc_rec.stl_geopos_method := NULL;
            v_loc_rec.stl_gps_scale := v_storet_loc_rec.geopstng_scale_txt;
            v_loc_rec.stl_lat_dir := v_storet_loc_rec.lat_direction;
            v_loc_rec.stl_lat_degrees := v_storet_loc_rec.lat_degree_msr;
            v_loc_rec.stl_lat_minutes := v_storet_loc_rec.lat_minute_msr;
            v_loc_rec.stl_lat_seconds := v_storet_loc_rec.lat_second_msr;
            v_loc_rec.stl_long_dir := v_storet_loc_rec.long_direction;
            v_loc_rec.stl_long_degrees := v_storet_loc_rec.long_degree_msr;
            v_loc_rec.stl_long_minutes := v_storet_loc_rec.long_minute_msr;
            v_loc_rec.stl_long_seconds := v_storet_loc_rec.long_second_msr;
            v_loc_rec.stl_lat_long_msr_date := v_storet_loc_rec.lat_long_msr_dt;
            v_loc_rec.stl_latlong_accurcy := NULL;
            v_loc_rec.stl_latlong_accurcy_unit := NULL;
            v_loc_rec.stl_tsmrrr_is_number := NULL;
            v_loc_rec.stl_tsmrrr_org_id := NULL;
            v_loc_rec.stl_tsmfhu_is_number := v_storet_loc_rec.tsmfhu_is_number;
            v_loc_rec.stl_tsmfhu_org_id := v_storet_loc_rec.tsmfhu_org_id;
            v_loc_rec.stl_tsmstatn0is_number := v_storet_loc_rec.tsmstatn0is_number;
            v_loc_rec.stl_tsmstatn0org_id := v_storet_loc_rec.tsmstatn0org_id;
            v_loc_rec.stl_tsmwell_is_number := v_storet_loc_rec.tsmwell_is_number;
            v_loc_rec.stl_tsmwell_org_id := v_storet_loc_rec.tsmwell_org_id;
            v_loc_rec.stl_tsmgeopa_is_number := v_storet_loc_rec.tsmgeopa_is_number;
            v_loc_rec.stl_tsmgeopa_org_id := v_storet_loc_rec.tsmgeopa_org_id;
            v_loc_rec.stl_lat_dec_minutes := NULL;
            v_loc_rec.stl_long_dec_minutes := NULL;
            v_loc_rec.stl_tsmpipe_is_number := NULL;
            v_loc_rec.stl_tsmpipe_org_id := NULL;
            v_loc_rec.stl_tsmstatn1org_id := NULL;
            v_loc_rec.stl_tsmstatn1is_number := NULL;
            v_loc_rec.stl_tsmgeopa0is_number := NULL;
            v_loc_rec.stl_tsmgeopa0org_id := NULL;
            v_loc_rec.stl_tsmnal_cd := v_storet_loc_rec.tsmnal_cd;
            v_loc_rec.stl_tsmnal_state := v_storet_loc_rec.tsmnal_state;
            v_loc_rec.stl_nrcs := v_storet_loc_rec.nrcs_wtrsd_id_num;
            v_loc_rec.stl_status := 'U';
            v_loc_rec.stl_import_seq := NULL;
            v_loc_rec.stl_export_seq := NULL;
          
            v_success := create_station_location(v_loc_rec);
          
          END IF;
          CLOSE c_compare_sim_loc;
        END LOOP;
      END IF;
      CLOSE c_compare_sim_stat_id;
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             SQLERRM ||
                                             ' in c_compare_station_id in check_if_station_exists.');
         
     
        v_return := 0;
        CLOSE c_compare_sim_stat_id;
    END;
  END IF;

  CLOSE c_compare_station_id;

  RETURN v_return;

EXCEPTION
  WHEN e_bad_well THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' grabbing STORET wells in check_if_station_exists.');
     
 
    RETURN 0;
  WHEN e_bad_station THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' grabbing STORET station in check_if_station_exists.');
     
 
    RETURN 0;
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in check_if_station_exists.');
     
 
    RETURN 0;
  
END check_if_station_exists;

/****************************************************************************************
 The create_station_location function inserts known good data into the SIM table SIM_STATIONS
 After the station_location has loaded sucessfully the function returns a 1
 If there was a problem the function returns a zero.
****************************************************************************************/
FUNCTION create_station_location(p_station_location_rec IN sim_station_locations%ROWTYPE)
  RETURN NUMBER IS

BEGIN
  /* create station using values in station record */

  INSERT INTO sim_station_locations
    (stl_seq,
     stl_sta_seq,
     stl_ssw_seq,
     stl_is_number,
     stl_org_id,
     stl_point_name,
     stl_type_code,
     stl_sequence_number,
     stl_elev_datum,
     stl_elev_method,
     stl_elev_msr_date,
     stl_elevation_msr,
     stl_elevation_unit,
     stl_geopos_datum,
     stl_geopos_method,
     stl_gps_scale,
     stl_lat_dir,
     stl_lat_degrees,
     stl_lat_minutes,
     stl_lat_seconds,
     stl_long_dir,
     stl_long_degrees,
     stl_long_minutes,
     stl_long_seconds,
     stl_lat_long_msr_date,
     stl_latlong_accurcy,
     stl_latlong_accurcy_unit,
     stl_tsmrrr_is_number,
     stl_tsmrrr_org_id,
     stl_tsmfhu_is_number,
     stl_tsmfhu_org_id,
     stl_tsmstatn0is_number,
     stl_tsmstatn0org_id,
     stl_tsmwell_is_number,
     stl_tsmwell_org_id,
     stl_tsmgeopa_is_number,
     stl_tsmgeopa_org_id,
     stl_lat_dec_minutes,
     stl_long_dec_minutes,
     stl_tsmpipe_is_number,
     stl_tsmpipe_org_id,
     stl_tsmstatn1org_id,
     stl_tsmstatn1is_number,
     stl_tsmgeopa0is_number,
     stl_tsmgeopa0org_id,
     stl_tsmnal_cd,
     stl_tsmnal_state,
     stl_nrcs,
     stl_status,
     stl_import_seq,
     stl_export_seq,
     stl_lat_dec_degrees,
     stl_long_dec_degrees)
  VALUES
    (p_station_location_rec.stl_seq,
     p_station_location_rec.stl_sta_seq,
     p_station_location_rec.stl_ssw_seq,
     p_station_location_rec.stl_is_number,
     p_station_location_rec.stl_org_id,
     p_station_location_rec.stl_point_name,
     p_station_location_rec.stl_type_code,
     p_station_location_rec.stl_sequence_number,
     p_station_location_rec.stl_elev_datum,
     p_station_location_rec.stl_elev_method,
     p_station_location_rec.stl_elev_msr_date,
     p_station_location_rec.stl_elevation_msr,
     p_station_location_rec.stl_elevation_unit,
     p_station_location_rec.stl_geopos_datum,
     p_station_location_rec.stl_geopos_method,
     p_station_location_rec.stl_gps_scale,
     p_station_location_rec.stl_lat_dir,
     p_station_location_rec.stl_lat_degrees,
     p_station_location_rec.stl_lat_minutes,
     p_station_location_rec.stl_lat_seconds,
     p_station_location_rec.stl_long_dir,
     p_station_location_rec.stl_long_degrees,
     p_station_location_rec.stl_long_minutes,
     p_station_location_rec.stl_long_seconds,
     p_station_location_rec.stl_lat_long_msr_date,
     p_station_location_rec.stl_latlong_accurcy,
     p_station_location_rec.stl_latlong_accurcy_unit,
     p_station_location_rec.stl_tsmrrr_is_number,
     p_station_location_rec.stl_tsmrrr_org_id,
     p_station_location_rec.stl_tsmfhu_is_number,
     p_station_location_rec.stl_tsmfhu_org_id,
     p_station_location_rec.stl_tsmstatn0is_number,
     p_station_location_rec.stl_tsmstatn0org_id,
     p_station_location_rec.stl_tsmwell_is_number,
     p_station_location_rec.stl_tsmwell_org_id,
     p_station_location_rec.stl_tsmgeopa_is_number,
     p_station_location_rec.stl_tsmgeopa_org_id,
     p_station_location_rec.stl_lat_dec_minutes,
     p_station_location_rec.stl_long_dec_minutes,
     p_station_location_rec.stl_tsmpipe_is_number,
     p_station_location_rec.stl_tsmpipe_org_id,
     p_station_location_rec.stl_tsmstatn1org_id,
     p_station_location_rec.stl_tsmstatn1is_number,
     p_station_location_rec.stl_tsmgeopa0is_number,
     p_station_location_rec.stl_tsmgeopa0org_id,
     p_station_location_rec.stl_tsmnal_cd,
     p_station_location_rec.stl_tsmnal_state,
     p_station_location_rec.stl_nrcs,
     p_station_location_rec.stl_status,
     p_station_location_rec.stl_import_seq,
     p_station_location_rec.stl_export_seq,
     p_station_location_rec.stl_lat_dec_degrees,
     p_station_location_rec.stl_long_dec_degrees);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - inserting station locations.');
     
 
    RETURN 0;
END create_station_location;

/****************************************************************************************
 The create_well function inserts known good data into the SIM table SIM_STATION_WELLS
 After the well has loaded sucessfully the function returns a 1
 If there was a problem the function returns a zero.
****************************************************************************************/
FUNCTION create_well(p_well_rec IN sim_station_wells%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create well using values in well record */
  INSERT INTO sim_station_wells
    (ssw_seq,
     ssw_sta_seq,
     ssw_well_status,
     ssw_is_number,
     ssw_org_id,
     ssw_name,
     ssw_well_no,
     ssw_well_use,
     ssw_develop_method,
     ssw_well_const_meth,
     ssw_water_use,
     ssw_gradient_type,
     ssw_depth_completion,
     ssw_depth_completion_unit,
     ssw_depth_hole,
     ssw_depth_hole_unit,
     ssw_well_height,
     ssw_well_height_unit,
     ssw_casing_height,
     ssw_casing_height_unit,
     ssw_disinfected,
     ssw_const_start_date,
     ssw_const_end_date,
     ssw_init_pump_dur,
     ssw_init_pump_dur_unit,
     ssw_init_pump_rate,
     ssw_init_pump_rate_unit,
     ssw_natural_flow,
     ssw_init_borehole_diam,
     ssw_init_borehole_diam_unit,
     ssw_tsmstatn_is_number,
     ssw_tsmstatn_org_id,
     ssw_status,
     ssw_import_seq,
     ssw_export_seq)
  VALUES
    (p_well_rec.ssw_seq,
     p_well_rec.ssw_sta_seq,
     p_well_rec.ssw_well_status,
     p_well_rec.ssw_is_number,
     p_well_rec.ssw_org_id,
     p_well_rec.ssw_name,
     p_well_rec.ssw_well_no,
     p_well_rec.ssw_well_use,
     p_well_rec.ssw_develop_method,
     p_well_rec.ssw_well_const_meth,
     p_well_rec.ssw_water_use,
     p_well_rec.ssw_gradient_type,
     p_well_rec.ssw_depth_completion,
     p_well_rec.ssw_depth_completion_unit,
     p_well_rec.ssw_depth_hole,
     p_well_rec.ssw_depth_hole_unit,
     p_well_rec.ssw_well_height,
     p_well_rec.ssw_well_height_unit,
     p_well_rec.ssw_casing_height,
     p_well_rec.ssw_casing_height_unit,
     p_well_rec.ssw_disinfected,
     p_well_rec.ssw_const_start_date,
     p_well_rec.ssw_const_end_date,
     p_well_rec.ssw_init_pump_dur,
     p_well_rec.ssw_init_pump_dur_unit,
     p_well_rec.ssw_init_pump_rate,
     p_well_rec.ssw_init_pump_rate_unit,
     p_well_rec.ssw_natural_flow,
     p_well_rec.ssw_init_borehole_diam,
     p_well_rec.ssw_init_borehole_diam_unit,
     p_well_rec.ssw_tsmstatn_is_number,
     p_well_rec.ssw_tsmstatn_org_id,
     p_well_rec.ssw_status,
     p_well_rec.ssw_import_seq,
     p_well_rec.ssw_export_seq
     
     );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - inserting well.');
     
 
    RETURN 0;
END create_well;

/****************************************************************************************

****************************************************************************************/
FUNCTION create_station(p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create station using values in station record */
  INSERT INTO sim_stations
    (sta_seq,
     sta_id,
     sta_name,
     sta_is_number,
     sta_org_id,
     sta_desc,
     sta_tsmorgan_org_id,
     sta_tsmorgan_is_number,
     sta_status,
     sta_estab_date,
     sta_export_seq,
     sta_tsmvstc_is_number,
     sta_tsmvstc_org_id,
     sta_epa_key_id,
     sta_zid_rel,
     sta_influence_area,
     sta_travel_dir,
     sta_water_depth,
     sta_water_depth_unit,
     sta_ecoregion)
  VALUES
    (p_station_rec.sta_seq, /* STA_SEQ            */
     p_station_rec.sta_id, /* STA_ID             */
     p_station_rec.sta_name, /* STA_NAME           */
     p_station_rec.sta_is_number,
     p_station_rec.sta_org_id, /* STATION ORG ID     */
     p_station_rec.sta_desc, /* STA_DESC           */
     p_station_rec.sta_tsmorgan_org_id, /* TSMORGAN_ORG_ID    */
     p_station_rec.sta_tsmorgan_is_number, /* TSMORGAN_IS_NUMBER */
     p_station_rec.sta_status, /* STA_STATUS         */
     p_station_rec.sta_estab_date, /* STA_ESTAB_DATE     */
     p_station_rec.sta_export_seq, /* STA_EXPORT_SEQ     */
     p_station_rec.sta_tsmvstc_is_number, /* STA_TSMVSTC_IS_NUMBER     */
     p_station_rec.sta_tsmvstc_org_id, /* STA_TSMVSTC_ORG_ID */
     p_station_rec.sta_epa_key_id,
     p_station_rec.sta_zid_rel,
     p_station_rec.sta_influence_area,
     p_station_rec.sta_travel_dir,
     p_station_rec.sta_water_depth,
     p_station_rec.sta_water_depth_unit,
     p_station_rec.sta_ecoregion);
  COMMIT;
  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - inserting station.');
    RETURN 0;
END create_station;

/****************************************************************************************
  Update Rows Ready and Activities Ready in the Import Log
 ****************************************************************************************/
PROCEDURE update_rows_ready (
  p_sil_seq     IN  sim_import_log.sil_seq%TYPE,
  p_result_cnt  OUT sim_import_log.sil_rows_ready%TYPE,
  p_act_cnt     OUT sim_import_log.sil_activities_imported%TYPE) AS

BEGIN
  SELECT COUNT(*)
    INTO p_result_cnt
    FROM sim_field_activities,
         sim_results
   WHERE rs_fa_seq = fa_seq
     AND fa_import_seq = p_sil_seq
     AND fa_status = 'A';

  SELECT COUNT(*)
    INTO p_act_cnt
    FROM sim_field_activities
   WHERE fa_import_seq = p_sil_seq
     AND fa_status = 'A';

  UPDATE sim_import_log
     SET sil_rows_ready = p_result_cnt,
         sil_activities_imported = p_act_cnt
   WHERE sil_seq = p_sil_seq;
END update_rows_ready;

/****************************************************************************************

 ****************************************************************************************/
PROCEDURE import_results
(
  p_org_id         IN  VARCHAR2,
  p_import_log_seq IN  NUMBER,
  p_imp_cfg_seq    IN  NUMBER,
  p_test_num       OUT NUMBER,
  p_skip_header    IN  NUMBER := 0 -- 1 = Skip  0 = Don't Skip
) IS

  v_result_cnt  sim_import_log.sil_rows_ready%TYPE := 0;
  v_act_cnt     sim_import_log.sil_activities_imported%TYPE := 0;
  
  v_group       VARCHAR2(10);
  v_ha_group_id VARCHAR2(10);

  v_status      VARCHAR2(1);
  v_approved_by VARCHAR2(20);
  v_approved_on DATE;

  v_groups     BOOLEAN := FALSE;
  v_char_names BOOLEAN := FALSE;

  v_inline    LONG;
  v_bad_parse BOOLEAN;

  v_trans_exist BOOLEAN := TRUE;

  v_error_code NUMBER(3);

  -- TABLES
  v_line_table             sim_import_pkg.col_table_type;
  v_personnel_table        sim_import_pkg.col_table_type;
  v_cporg_table            sim_import_pkg.col_table_type;
  v_parent_id_table        sim_import_pkg.col_table_type;
  v_parent_fa_seq_table    sim_import_pkg.col_table_type;
  v_parent_is_number_table sim_import_pkg.col_table_type;

  v_parent_org_id_table sim_import_pkg.col_table_type;

  v_remark_codes_table sim_import_pkg.col_table_type;
  v_remark_is_table    sim_import_result_extras.col_table_num_type;
  v_remark_org_table   sim_import_result_extras.col_table_type;
  v_persn_is_table     sim_import_result_extras.col_table_num_type;
  v_persn_org_table    sim_import_result_extras.col_table_type;
  v_cporg_is_table     sim_import_result_extras.col_table_num_type;
  v_cporg_org_table    sim_import_result_extras.col_table_type;
  v_project_table      sim_import_pkg.col_table_type;
  v_proj_is_table      sim_import_result_extras.col_table_num_type;
  v_proj_org_table     sim_import_result_extras.col_table_type;
  v_default_table      sim_import_result_extras.col_table_type;
  v_pos_table          sim_import_result_extras.col_table_num_type;
  v_trip_table         sim_import_result_extras.col_table_type;
  v_stvst_id_table     sim_import_result_pkg.t_stvst_id_type;

  v_delimiter VARCHAR2(1);
  v_num_cols  NUMBER(3);
  v_col_seq   NUMBER(10);
  v_dflt_pos  NUMBER(3);

  v_item_name      VARCHAR2(55);
  v_item           VARCHAR2(4000);
  v_converted_item VARCHAR2(4000);
  v_instr_code     VARCHAR2(15);
  v_gear_type_ok   NUMBER(1);
  v_success        NUMBER(1);
  v_dummy_seq      NUMBER(10);
  v_dummy_char     VARCHAR2(60);
  v_detection      VARCHAR2(40);
  v_detect_check   NUMBER(1);

  v_storet_key sim_import_result_extras.t_storet_pk_type;
  v_temp_key   sim_import_result_extras.t_storet_pk_type;
  v_dummy_key  sim_import_result_extras.t_storet_pk_type;

  v_fa_rec     sim_field_activities%ROWTYPE;
  v_fap_rec    sim_field_activity_parent%ROWTYPE;
  v_result_rec sim_results%ROWTYPE;
  v_trip_rec   sim_trips%ROWTYPE;
  v_visit_rec  sim_station_visits%ROWTYPE;
  v_sample_rec sim_sample%ROWTYPE;
  v_tsrsdp_rec tsrsdp%ROWTYPE;
  v_brg_rec    sim_bio_results_grp%ROWTYPE;
  v_brgi_rec   sim_bio_results_grp_individual%ROWTYPE;
  v_tod_rec    sim_trawl_ops_det%ROWTYPE;
  v_rci_rec    sim_result_class_indicator%ROWTYPE;
  v_aal_rec    sim_actual_activity_loc%ROWTYPE;
  v_stn_rec    sim_trap_net%ROWTYPE;

  v_fap_seq            NUMBER(10);
  p_sta_seq            NUMBER(10);
  v_result_seq_num     NUMBER(5);
  v_assnd_seq_num      NUMBER(5);
  v_sequence_number    VARCHAR2(15); 
  v_well_pipe_num      VARCHAR2(15); 
  v_point_type         VARCHAR2(16);
  v_result_seq         NUMBER(10);
  v_stn_seq            NUMBER(10);
  v_stl_seq            NUMBER(10);
  v_activity_cat       VARCHAR2(30);
  v_rel_depth          VARCHAR2(15);
  v_field_set_name     VARCHAR2(30);
  v_field_set_nbr      VARCHAR2(10);
  v_lowq_limit         VARCHAR2(12);
  v_upq_limit          VARCHAR2(12);
  v_detect_limit       VARCHAR2(12);
  v_det_comment        VARCHAR2(254);
  v_det_uom_is_nbr     NUMBER(12);
  v_det_uom_org_id     VARCHAR2(8);
  v_result_cmt         VARCHAR2(4000) := NULL;
  v_result_status      CHAR(1);
  v_batch_id           VARCHAR2(12);
  v_lab_cert           VARCHAR2(1);
  v_stvst_cmt          VARCHAR2(4000);
  v_up_depth           VARCHAR2(8);
  v_low_depth          VARCHAR2(8);
  v_chain_of_custody   VARCHAR2(30);
  v_class_count        VARCHAR2(12);
  v_thermo             VARCHAR2(5);
  v_halo               VARCHAR2(5);
  v_pycno              VARCHAR2(5);
  v_depth_zone         VARCHAR2(15);
  v_depth_ref_unit     VARCHAR2(3);
  v_qc_indicator       VARCHAR2(1);
  v_cell_type          VARCHAR2(11);
  v_cell_shape         VARCHAR2(18);
  v_bias               VARCHAR2(12);
  v_cl_corrected       VARCHAR2(1);
  v_add_loc_info       VARCHAR2(254);
  v_tsrmatrx_is_number NUMBER(12);
  v_tsrmatrx_org_id    VARCHAR2(8);

  v_last_activity_id VARCHAR2(12) := '$$$$$$$$$$$$';
  v_activity_id      VARCHAR2(12);
  v_new_activity     NUMBER(1);
  v_activity_seq     NUMBER(10);
  v_replicate_nbr    VARCHAR2(3);
  v_last_rep_nbr     VARCHAR2(3);
  v_station_id       VARCHAR2(15);
  v_last_trip_seq    NUMBER(10);
  v_last_visit_seq   NUMBER(10);
  v_station_visit    VARCHAR2(8);
  v_act_is_nbr       NUMBER(12);
  v_act_org_id       VARCHAR2(8);

  v_error_cnt        NUMBER(10) := 0;
  v_import_desc      VARCHAR2(200);

  v_medium          VARCHAR2(20);
  v_last_medium     VARCHAR2(20);
  v_comment         VARCHAR2(1999);
  v_tsmpersn_is_nbr NUMBER(12);
  v_tsmpersn_org_id VARCHAR2(8);
  v_tsmcporg_is_nbr NUMBER(12);
  v_tsmcporg_org_id VARCHAR2(8);
  v_tsrlbrmk_is_nbr NUMBER(12);
  v_tsrlbrmk_org_id VARCHAR2(8);
  v_type            VARCHAR2(20);
  v_last_type       VARCHAR2(20);

  v_intent               VARCHAR2(20);
  v_taxon_is_nbr         NUMBER(12);
  v_taxon_org_id         VARCHAR2(8);
  v_char_species_number  VARCHAR2(8);
  v_allow_species        NUMBER(1);
  v_taxon_species_number VARCHAR2(8);
  v_result_species       VARCHAR2(8);
  v_brg_species          VARCHAR2(8);
  v_act_species          VARCHAR2(8);
  v_community_is_nbr     NUMBER(12);
  v_community_org_id     VARCHAR2(8);
  v_biopart_is_nbr       NUMBER(12);
  v_biopart_org_id       VARCHAR2(8);
  v_data_file_name_loc   VARCHAR2(60);
  v_gear_deployment      VARCHAR2(1999);
  v_transport_storage    VARCHAR2(1999);
  v_sample_seq           VARCHAR2(10);
  v_ttl_smpl_vol         NUMBER(7, 2);
  v_smpl_vol_unit_seq    NUMBER(10);
  v_smpl_wt              NUMBER(7, 2);
  v_total_wt_uom         VARCHAR2(3);
  v_parent_ind_code      VARCHAR2(1);

  v_check_trawl_gear NUMBER(1);
  v_tsmalp_is_nbr    NUMBER(12);
  v_tsmalp_org_id    VARCHAR2(8);
  v_fished_duration  NUMBER(4, 2);
  v_fished_dur_uom   VARCHAR2(10);
  v_distance_fished  NUMBER(4, 2);
  v_dist_fished_uom  VARCHAR2(3);
  v_boat_speed       NUMBER(3, 1);
  v_boat_speed_uom   VARCHAR2(10);
  v_curr_group       VARCHAR2(8);
  v_new_group        NUMBER(1) := 1;
  v_brg_id           VARCHAR2(8);
  v_brg_seq          NUMBER(10);
  v_brg_text         VARCHAR2(1999);

  v_brgi_seq          NUMBER(10);
  v_brgi_brg_seq      NUMBER(10);
  v_brg_type          VARCHAR2(32);
  v_value_type_name   VARCHAR2(10);
  v_ttl_nbr_in_grp    NUMBER(8);
  v_ttl_nbr_indiv     NUMBER(5);
  v_brg_freq_analysis CHAR(1);
  v_sexname_is_nbr    NUMBER(12);
  v_sexname_org_id    VARCHAR2(8);
  v_lifestage_is_nbr  NUMBER(12);
  v_lifestage_org_id  VARCHAR2(8);

  -- SIM_RESULT_CLASS_INDICATOR variables
  v_primary_class         VARCHAR2(9);
  v_secondary_class       VARCHAR2(9);
  v_primary_class_value   VARCHAR2(12);
  v_secondary_class_value VARCHAR2(12);
  v_particle_size_basis   VARCHAR2(40);
  v_lower_bnd_amt         NUMBER(7, 2);
  v_upper_bnd_amt         NUMBER(7, 2);
  v_individual_nbr        NUMBER(4);

  v_depth_ref_seq      NUMBER(10);
  v_depth_ref          VARCHAR2(30);
  v_depth_uom          VARCHAR2(3);
  v_depth_range_uom    VARCHAR2(3);
  v_tsmstatn_is_nbr    NUMBER(12);
  v_tsmstatn_org_id    VARCHAR2(8);
  v_visit_start_date   DATE;
  v_visit_start_time   DATE;
  v_visit_end_date     DATE;
  v_visit_end_time     DATE;
  v_visit_seq          NUMBER(10);
  v_tsrstvst_is_nbr    NUMBER(12);
  v_tsrstvst_org_id    VARCHAR2(8);
  v_tsrlab_is_nbr      NUMBER(12);
  v_tsrlab_org_id      VARCHAR2(8);
  v_trip_id            VARCHAR2(15);
  v_trip_name          VARCHAR2(60);
  v_trip_seq           NUMBER(10);
  v_trip_is_nbr        NUMBER(12);
  v_trip_org_id        VARCHAR2(8);
  v_trip_start_date    DATE;
  v_trip_stop_date     DATE;
  v_proj_is_nbr        NUMBER(12);
  v_proj_org_id        VARCHAR2(8);
  v_gear_is_nbr        NUMBER(12);
  v_gear_org_id        VARCHAR2(8);
  v_gearcfg_is_nbr     NUMBER(12);
  v_gearcfg_org_id     VARCHAR2(8);
  v_proc_is_nbr        NUMBER(12);
  v_proc_org_id        VARCHAR2(8);
  v_proc_name          VARCHAR(30);
  v_sample_pres_is_nbr NUMBER(12);
  v_sample_pres_org_id VARCHAR2(8);
  v_depth              VARCHAR2(8) := NULL;
  v_dummy_num          NUMBER(8);

  v_start_date      DATE;
  v_start_date_val  VARCHAR2(15);
  v_start_time      DATE;
  v_start_time_zone VARCHAR2(3);
  v_stop_time_zone  VARCHAR2(3);
  v_end_date        DATE;
  v_end_time        DATE;

  v_trap_net_duration NUMBER(4, 2);
  v_trap_net_dur_unit VARCHAR2(10);
  v_relative_current  NUMBER(3);
  v_relative_wind     NUMBER(3);
  v_orient_to_cur     VARCHAR2(15);
  v_trap_net_comment  VARCHAR2(254);

  v_func_feed_grp   VARCHAR2(6);
  v_trophic_level   VARCHAR2(4);
  v_taxon_pollution VARCHAR2(4);

  v_analysis_date      DATE;
  v_analysis_time      DATE;
  v_analysis_time_zone VARCHAR2(3);
  v_result             VARCHAR2(256);
  v_result_save        VARCHAR2(256); -- may not be large enough

  v_units_ok        BOOLEAN;
  v_result_ok       BOOLEAN;
  v_tsrchgrp_is_nbr NUMBER(12);
  v_tsrchgrp_org_id VARCHAR2(8);
  v_tsrhcsc_is_nbr  NUMBER(12);
  v_tsrhcsc_org_id  VARCHAR2(8);
  v_tsrchdef_is_nbr NUMBER(12);
  v_tsrchdef_org_id VARCHAR2(8);
  v_char_is_nbr     NUMBER(12);
  v_char_org_id     VARCHAR2(8);
  v_uom_is_nbr      NUMBER(12);
  v_uom_org_id      VARCHAR2(8);
  v_fraction        VARCHAR2(40);
  v_fraction_is_nbr NUMBER(12);
  v_fraction_org_id VARCHAR2(8);
  v_value_type      VARCHAR2(10);
  v_stat_type       VARCHAR2(18);
  v_anlpr_is_nbr    NUMBER(12);
  v_anlpr_org_id    VARCHAR2(8);

  v_lab_procedure VARCHAR2(15); 
  v_lab_source    VARCHAR2(12); 

  v_labprep_is_nbr       NUMBER(12);
  v_labprep_org_id       VARCHAR2(8);
  v_labprep_id           VARCHAR2(15);
  v_labprep_source       VARCHAR2(12);
  v_duration_basis       VARCHAR2(10);
  v_temp_basis           VARCHAR2(8);
  v_weight_basis         VARCHAR2(12);
  v_precision            VARCHAR2(12);
  v_conf_level           VARCHAR2(8);
  v_habit_is_number      NUMBER(12);
  v_volt_is_number       NUMBER(12);
  v_habit_org_id         VARCHAR2(8);
  v_volt_org_id          VARCHAR2(8);
  v_bio_condition_is_nbr NUMBER(12);
  v_bio_condition_org_id VARCHAR2(8);
  v_sec_bio_cond_is_nbr  NUMBER(12);
  v_sec_bio_cond_org_id  VARCHAR2(8);
  v_upper_range          NUMBER := 0;
  v_lower_range          NUMBER := 0;

  v_result_value_msr NUMBER(1);
  v_first_lab_id     VARCHAR2(8);
  v_no_person        NUMBER(2);
  v_no_cporgs        NUMBER(2);
  v_no_parent_ids    NUMBER(2);
  v_no_remark_codes  NUMBER(2);
  v_no_projects      NUMBER(2);

  v_habitat          NUMBER(1);
  v_ha_group         NUMBER(1);
  v_ha_char_name     VARCHAR2(60);

  v_pre_check        NUMBER(1);
  v_pre_check_status VARCHAR2(1);
  v_dup_brgi_success NUMBER(1);
  v_data_line_number NUMBER(8);
  v_data_line_name   VARCHAR2(25);

  v_visit_document    VARCHAR2(1999);
  v_activity_document VARCHAR2(1999);
  v_result_document   VARCHAR2(1999);
  v_sid_seq           NUMBER(10);  

  e_fileread EXCEPTION;
  e_bad_org EXCEPTION;
  e_no_cfg EXCEPTION;
  e_e_bad_delimiter EXCEPTION;
  e_bad_delimiter EXCEPTION;
  e_no_data_found EXCEPTION;
  e_project_is EXCEPTION;
  e_no_sid_seq EXCEPTION;
  e_bad_station EXCEPTION;

  CURSOR c_activity_seq IS
    SELECT sim_field_activities_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_result_seq IS
    SELECT sim_results_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_visit_seq IS
    SELECT sim_station_visits_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_trap_net_seq IS
    SELECT sim_trap_net_seq.NEXTVAL
      FROM dual;

  /* cursors for ocsim */
  CURSOR c_get_next_sample_seq IS
    SELECT sim_sample_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_tsrsdp_values(p_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT *
      FROM tsrsdp
     WHERE tsrsdp_is_number = p_is_nbr
       AND tsrsdp_org_id = rpad(p_org_id, 8);

  CURSOR c_get_next_brg_seq IS
    SELECT sim_brg_seq.NEXTVAL
      FROM dual;

  CURSOR c_find_brg_seq(p_curr_group IN VARCHAR2, p_activity_seq IN NUMBER) IS
    SELECT brg_seq
      FROM sim_bio_results_grp
     WHERE brg_id_code = p_curr_group
       AND brg_fa_seq = p_activity_seq;
      
  -- c_get_brg and c_get_uom were added to allow Biological result files to 
  -- avoid having to be grouped within the import file.
  CURSOR c_get_brg(p_curr_group IN VARCHAR2, p_activity_seq IN NUMBER) IS
    SELECT a.brg_seq brg_seq,
           a.brg_type brg_type,
           a.brg_freq_analysis brg_freq_analysis,
           a.brg_summary_grp_count brg_summary_grp_count,
           a.brg_value_type_name brg_value_type_name,
           a.brg_description_text brg_description_text,
           a.brg_tsrchar_is_number brg_tsrchar_is_number,
           a.brg_tsrchar_org_id brg_tsrchar_org_id,
           a.brg_tsrcpv_lifestage_is_nbr brg_tsrcpv_lifestage_is_nbr,
           a.brg_tsrcpv_lifestage_org_id brg_tsrcpv_lifestage_org_id,
           a.brg_tsrcpv_sex_is_number brg_tsrcpv_sex_is_number,
           a.brg_tsrcpv_sex_org_id brg_tsrcpv_sex_org_id
      FROM sim_bio_results_grp a
     WHERE a.brg_id_code = p_curr_group
       AND a.brg_fa_seq = p_activity_seq;
       
  CURSOR c_get_uom(p_rci_brg_seq IN NUMBER) IS
    SELECT rci_tsruom_is_number,
           rci_tsruom_org_id
     FROM  sim_result_class_indicator
    WHERE  rci_brg_seq = p_rci_brg_seq;
 
  CURSOR c_get_next_brgi_seq IS
    SELECT sim_brgi_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_tod_seq IS
    SELECT sim_trawl_ops_det_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_rci_seq IS
    SELECT sim_result_class_indicator_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_aal_seq IS
    SELECT sim_activity_loc_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_import_type(p_imp_cfg_seq IN NUMBER) IS
    SELECT sicc_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_imp_cfg_seq;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT sipl_seq,
           sipl_line_text,
           sipl_line_no
      FROM sim_imp_lines
     WHERE sipl_sil_seq = p_import_seq
       AND sipl_line_text IS NOT NULL
     ORDER BY sipl_line_no;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT sic_proc_order,
           sic_dflt_pos,
           sic_name,
           sic_seq
      FROM sim_imp_cols,
           sim_imp_defs,
           sim_imp_cfg
     WHERE sicc_sid_seq = sid_seq
       AND sic_sid_seq = sid_seq
       AND sicc_seq = p_imp_cfg_seq
     ORDER BY sic_proc_order;

  CURSOR c_trip_dates(p_trip_seq IN NUMBER) IS
    SELECT stp_start_date
      FROM sim_trips
     WHERE stp_seq = p_trip_seq;

  CURSOR c_get_location_info IS
    SELECT stl_seq
      FROM sim_station_locations
     WHERE stl_tsmstatn0is_number = v_tsmstatn_is_nbr
       AND stl_tsmstatn0org_id = rpad(v_tsmstatn_org_id, 8)
       AND stl_is_number = v_tsmalp_is_nbr
       AND stl_org_id = rpad(v_tsmalp_org_id, 8);

  CURSOR c_matching_config(p_proc_name IN VARCHAR2) IS
    SELECT tsrgrcfg_is_number,
           tsrgrcfg_org_id,
           tsrfldgr_is_number,
           tsrfldgr_org_id
      FROM tsrgrcfg
     WHERE id_code = rpad(p_proc_name, 10)
       AND tsrgrcfg_org_id = rpad(p_org_id, 8);

  CURSOR c_get_pre_check IS
    SELECT sc_value
      FROM sim_config
     WHERE sc_item = 'DO_PRE_CHECK';

  CURSOR c_char_group(p_act_seq IN NUMBER) IS
    SELECT fa_char_group
      FROM sim_field_activities
     WHERE fa_seq = p_act_seq;

  CURSOR c_tsrfdact(p_is_number IN NUMBER, p_org_id IN CHAR) IS
    SELECT tsrfdact0is_number,
           tsrfdact0org_id,
           tsrstvst_is_number,
           tsrstvst_org_id
      FROM tsrfdact
     WHERE tsrfdact_is_number = p_is_number
       AND tsrfdact_org_id = p_org_id;

  CURSOR c_station_visit(p_ssv_seq IN NUMBER) IS
    SELECT ssv_tsrstvst_is_number,
           ssv_tsrstvst_org_id
      FROM sim_station_visits
     WHERE ssv_seq = p_ssv_seq;

  CURSOR c_activity_parent(p_parent_seq IN NUMBER, p_is_number IN NUMBER, p_org_id IN CHAR, p_category IN VARCHAR2) IS
    SELECT fap_seq
      FROM sim_field_activity_parent,
           sim_field_activities
     WHERE fap_fa_seq = fa_seq
       AND (fap_parent_fa_seq = p_parent_seq)
        OR (fap_parent_is_number = p_is_number AND
           fap_parent_org_id = rpad(p_org_id, 8))
       AND fa_category = p_category
       AND fa_status = 'A';

  -- Cursor to determine is a Gear Configuration ID is valid for a given Gear ID
  CURSOR c_get_gear_configuration(cp_gear_is_number NUMBER, cp_gear_org_id VARCHAR2, cp_gearcfg_is_number NUMBER, cp_gearcfg_org_id VARCHAR2) IS
    SELECT 1
      FROM tsrgrcfg
     WHERE tsrgrcfg_is_number = cp_gearcfg_is_number
       AND tsrgrcfg_org_id = rpad(cp_gearcfg_org_id, 8)
       AND tsrfldgr_is_number = cp_gear_is_number
       AND tsrfldgr_org_id = rpad(cp_gear_org_id, 8);

  r_tsrfdact               c_tsrfdact%ROWTYPE;
  r_station_visit          c_station_visit%ROWTYPE;
  r_activity_parent        c_activity_parent%ROWTYPE;
  r_get_gear_configuration c_get_gear_configuration%ROWTYPE;
  v_dummy_brg              c_get_brg%ROWTYPE;
  v_dummy_uom              c_get_uom%ROWTYPE;
  
BEGIN
  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  
  /* For now we'll delete all pre-check import log info right now. */  
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = p_imp_cfg_seq;
  
  sim_event_log_pkg.write_event(1,v_import_log_seq,p_org_id,v_sid_seq,NULL);
  BEGIN
    DELETE sim_import_log
     WHERE sil_tsmorgan_org_id = rpad(p_org_id, 8)
       AND nvl(sil_pre_check_flag, 0) = 1;
  
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || ' in c_find_pre_check');
  END;
  /* Find out if we are pre-checking or not. */
  BEGIN
    OPEN c_get_pre_check;
    FETCH c_get_pre_check
      INTO v_dummy_char;
    IF c_get_pre_check%NOTFOUND THEN
      v_pre_check := 0;
    ELSE
      IF upper(v_dummy_char) = 'YES' THEN
        v_pre_check := 1;
      ELSE
        v_pre_check := 0;
      END IF;
    END IF;
    v_dummy_char := NULL;
    CLOSE c_get_pre_check;
  EXCEPTION
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || ' in c_get_pre_check');
       
   
  END;

  <<start_checking>>
  /* Update the import log with Pre-check info */
  BEGIN
    UPDATE sim_import_log
       SET sil_pre_check_flag = v_pre_check
     WHERE sil_seq = v_import_log_seq;
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || ' in update log');
  END;

  -- initialize global variables for this time through the package
  v_line_cnt := 0;
  v_acode_defs_rec := v_acode_init_rec;
  v_org_is_nbr := NULL;
  v_trip_qc := 0;
  v_habitat := 0;
  v_cur_trip_is_nbr := NULL;
  v_pre_check_status := 'A';
  gv_last_visit_id := 0;
  v_new_activity := 1;
  v_activity_seq := NULL;
  v_last_activity_id := '$$$$$$$$$$$$';
  
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                       'Start: ' ||
                                       to_char(SYSDATE, 'HH24:MI:SS'));
   
  IF p_skip_header = 1 THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Header skipped');
    v_line_cnt := 1; 
  END IF;

  /* get tsmorgan_is_number */
  BEGIN
    v_org_is_nbr := sim_import_result_extras.get_org_info(p_org_id, v_import_log_seq);
    IF v_org_is_nbr IS NULL THEN
      RAISE e_bad_org;
    END IF;
  
  EXCEPTION
    WHEN e_bad_org THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unable to find the specified org id in local version of STORET.');
     
 
      RAISE no_data_found;
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM ||
                                           ' - getting org is number.');
       
    
      RAISE no_data_found;
  END;

  /*  get the delimiter */
  BEGIN
    v_delimiter := sim_import_result_extras.get_delimiter(p_imp_cfg_seq, v_line_cnt, v_import_log_seq);
    IF v_delimiter IS NULL THEN
      RAISE e_no_cfg;
    END IF;
  
  EXCEPTION
    WHEN e_no_cfg THEN       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Configuration invalid or does not exist - select new configuration or modify existing one.');  
      RAISE no_data_found;
      
    WHEN OTHERS THEN 
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || ' - getting delimiter.');
      RAISE no_data_found;
  END;

  OPEN c_get_import_type(p_imp_cfg_seq);
  FETCH c_get_import_type
    INTO v_sid_seq;
  CLOSE c_get_import_type;
  IF v_sid_seq IS NULL THEN
    RAISE e_no_sid_seq;
  END IF;

  /* get actual positions and default information for items being imported */
  v_success := sim_import_result_extras.get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table, v_num_cols, v_line_cnt, v_import_log_seq);
  IF v_success != 1 THEN     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unable to find the import type in the import configurations - select or create a new configuration.');
  END IF;
  /* loop through the lines in the text table */
  FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
    IF p_skip_header = 0
        OR v_text_line.sipl_line_no <> 1 THEN
      BEGIN
        /* read one line */
        v_inline := v_text_line.sipl_line_text;
        /* null values which might be carried through to the next line */
        v_comment := NULL;
        v_result_cmt := NULL;
        v_result_ok := TRUE;
        v_result := NULL;
        v_last_type := v_type;
        v_type := NULL;
        v_line_cnt := v_line_cnt + 1;
        v_new_activity := 1; -- set to create new activity at beginning of loop
        v_result_seq_num := 0;
        v_status := 'A';
        v_check_trawl_gear := 0;
        v_data_line_number := NULL;
        v_data_line_name := NULL;
        v_trip_rec.stp_start_date := NULL;
        v_trip_start_date := NULL;
        v_visit_start_date := NULL;
        v_particle_size_basis := NULL; -- jah 2-23-04 Added Particle Size Basis     
      
        -- These variables are initialized to NULL at the end of the loop.
        -- v_uom_is_nbr := NULL;
        -- v_uom_org_id := NULL;
        -- v_char_is_nbr := NULL;
        -- v_char_org_id := NULL;
      
        -- send line to parsing routine:
        BEGIN
          sim_import_pkg.parse_line_table(v_inline, v_line_table, v_num_cols, v_delimiter, v_bad_parse, v_import_log_seq);
        
          IF v_bad_parse THEN
            RAISE e_e_bad_delimiter;
          END IF;
        
        EXCEPTION
          WHEN e_e_bad_delimiter THEN
            RAISE e_bad_delimiter;
          
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                 SQLERRM ||
                                                 ' parsing one line.');          
            RAISE e_fileread;
        END;
      
        /* loop through all result columns */
        FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP
          /* get dflt position, item name, and column sequence */
          v_dflt_pos := v_columns.sic_dflt_pos;
          v_item_name := v_columns.sic_name;
          v_col_seq := v_columns.sic_seq;
        
          /* if column is included or generated, continue */
          IF v_pos_table(v_dflt_pos) IS NOT NULL THEN
          
            /* get the actual item, or use default if null */
            IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
              IF length(v_line_table(v_pos_table(v_dflt_pos))) > 4000 THEN
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                     'too many characters, truncating to 4000');
                 
              
              END IF;
              v_item := substr(v_line_table(v_pos_table(v_dflt_pos)), 1, 4000);
            ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL
                  AND v_default_table(v_dflt_pos) IS NOT NULL THEN
              IF length(v_line_table(v_pos_table(v_dflt_pos))) > 4000 THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'too many characters, truncating to 4000');
                 
              
              END IF;
              v_item := substr(v_default_table(v_dflt_pos), 1, 4000);
            ELSE
              v_item := NULL;
            END IF;
          
            -- Does v_item contain only spaces? (or NULL)
            IF (TRIM(v_item) IS NULL) THEN
              -- if yes
              -- by definition V_ITEM is NULL!
              v_item := NULL;
            END IF;
          
            /* assign item to appropriate variables and take care of any other procedural issues that need to be
            taken care of for or with a particular item */
            IF v_item_name = 'Replicate Number'
               OR v_item_name = 'Activity Replicate Number' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF length(v_item) > 3 THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Replicate Number, ' ||
                                                         v_item ||
                                                         ', is invalid. It is longer than 3 characters.');                  
                    v_replicate_nbr := NULL;
                    v_status := 'N';
                  ELSE
                    v_replicate_nbr := v_item;
                    
                    BEGIN
                    --check to see if v_replicate_number is a number.  If not throw an excpetion.
                    IF to_number(v_replicate_nbr) IS NOT NULL THEN
                      NULL;
                    END IF;
                    EXCEPTION
                      WHEN OTHERS THEN
                          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                               'Replicate Number, '|| v_item ||
                                                               ', must be a number.'); 
                          v_status := 'N';
                     END;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_replicate_nbr := NULL;
                END IF;
              ELSE
                v_replicate_nbr := NULL;
              END IF;
            
              ----------------                    
              -- Activity Type
              ----------------                    
            ELSIF v_item_name = 'Activity Type' THEN
              /* Business Rules
              
              Each activity id must be unique on a station visit.  
              
              Activity type and category have been moved up in the processing order 
              so that quality control activities (aka "Trip qcs") do not causes error 
              messages. (Trip qcs do not have station ids or station visits. */
            
              -- Does the Activity Type field contain anything?
              IF v_item IS NOT NULL THEN
                -- if yes
              
                -- Remove trailing spaces
                v_item := rtrim(v_item);
              
                -- Translate the Activity Type and then determine if the Activity Type is valid
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, NULL, v_import_log_seq, v_line_cnt);
              
                -- Is the Activity Type valid?
                IF v_dummy_seq IS NULL THEN
                  -- if no
                  -- prevent migration of this record
                  v_status := 'N';
                  v_type := NULL;
                
                ELSE
                
                  -- if yes, the Activity Type is valid
                
                  -- retrieve the item value from the SIM Allowable Value table.
                  v_type := sim_import_result_extras.get_item_value(v_dummy_seq, -- this parm contains the record number in the SIM Allowable Value table.
                                                                    v_line_cnt, v_import_log_seq);
                
                  -- is this a record for a Qualify Control activity?
                  IF v_type = 'Trip QC' THEN
                    -- if yes
                    -- tell subsequent processing this record is for a Quality Control activity
                    v_trip_qc := 1;
                  ELSE
                    -- if no
                    -- tell subsequent processing this record is NOT for a Quality Control activity
                    v_trip_qc := 0;
                  END IF;
                
                END IF;
              
              ELSE
                -- if no, the Activity Type field does not contain anything
              
                -- Prevent migration of this record              
                v_type := NULL;
                v_status := 'N';
              
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'Activity Type is required, but is not in the import data file.');
                 
              
              END IF;
            
              --------------------
              -- Activity Category
              --------------------
            ELSIF v_item_name = 'Activity Category' THEN
            
              IF v_item IS NOT NULL THEN
                IF v_trip_qc != 1 THEN
                  /* get category - check translations and STORET TSMPRMVL table */
                  v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSRFDACT', 'CATEGORY_TYPE_NAME', v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_char IS NOT NULL THEN
                    v_activity_cat := substr(rtrim(v_dummy_char), 1, 30);
                    IF v_type IS NOT NULL THEN
                      v_success := sim_import_result_checks.check_type_category_match(v_type, v_activity_cat, NULL, v_line_cnt, v_import_log_seq, v_habitat);
                      IF v_success = 2 THEN
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             'Activity category is not valid for the chosen activity type.');
                         
                      
                      END IF;
                    ELSIF v_sid_seq = 10007 THEN
                      --for WEBSIM the type is inferred from the category
                      IF v_activity_cat = 'Routine Sample'
                         OR v_activity_cat = 'Field Replicate/Duplicate'
                         OR v_activity_cat = 'Depletion Replicate'
                         OR v_activity_cat = 'Integrated Time Series'
                         OR v_activity_cat = 'Integrated Flow Proportioned'
                         OR v_activity_cat = 'Integrated Horizontal Profile'
                         OR v_activity_cat = 'Integrated Vertical Profile'
                         OR v_activity_cat = 'Composite w/o Parents'
                         OR v_activity_cat = 'Field Spike'
                         OR v_activity_cat = 'Field Equipment Rinsate Blank'
                         OR v_activity_cat = 'Field Blank'
                         OR v_activity_cat = 'Field Subsample'
                         OR v_activity_cat = 'Field Split'
                         OR v_activity_cat = 'Field Surrogate Spike'
                         OR v_activity_cat = 'Field Ambient Conditions Blank' THEN
                        v_type := 'Sample';
                      ELSIF v_activity_cat = 'Routine Msr/Obs'
                            OR v_activity_cat = 'Routine Habitat Assessment'
                            OR v_activity_cat = 'Replicate Msr/Obs'
                            OR v_activity_cat = 'Replicate Habitat Assessment'
                            OR v_activity_cat = 'Field Calibration Check' THEN
                        v_type := 'Field Msr/Obs';
                      ELSE
                        v_type := NULL;
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             'Activity category is not valid.');
                         
                      
                      END IF;
                    ELSE
                      v_status := 'N';
                    END IF;
                    /* replicate check */
                    IF v_replicate_nbr IS NULL
                       AND (v_activity_cat = 'Field Replicate/Duplicate' OR
                       v_activity_cat = 'Depletion replicate' OR
                       v_activity_cat = 'Replicate Msr/Obs' OR
                       v_activity_cat = 'Replicate Habitat Assessment') THEN
                    
                      /* if activity indicates a replicate and it has been left blank, write error message */
                      v_status := 'N';
                      
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Activity category indicates the activity is a replicate, but no replicate number has been provided.');
                       
                    
                    ELSIF v_replicate_nbr IS NOT NULL
                          AND (v_activity_cat != 'Field Replicate/Duplicate' AND
                          v_activity_cat != 'Depletion Replicate' AND
                          v_activity_cat != 'Replicate Msr/Obs' AND
                          v_activity_cat != 'Replicate Habitat Assessment') THEN
                    
                      /* if activity is not a replicate and a replicate number has been given, write error message */
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           'Activity is not a replicate, but a replicate number has been provided.');
                       
                    
                      v_replicate_nbr := NULL;
                    END IF;
                  
                  ELSE
                    v_status := 'N';
                    v_activity_cat := NULL;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Activity Category ignored. It is not used when Activity Type = Trip QC.');
                   
                
                  v_activity_cat := NULL;
                END IF;
              ELSE
                IF nvl(v_type, 'x') != 'Trip QC' THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'Activity Category is required and has not been provided.');
                   
                
                  v_activity_cat := NULL;
                  v_status := 'N';
                END IF;
              END IF;
            
              ---------------              
              -- Trip QC Type
              ---------------              
            ELSIF v_item_name = 'Trip QC Type' THEN
              IF v_type = 'Trip QC' THEN
                IF v_item IS NOT NULL THEN
                  /* get trip QC type - check translations and STORET TSMPRMVL table */
                  v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSRFQS', 'TYPE_NAME', v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_char IS NOT NULL THEN
                    v_activity_cat := rtrim(v_dummy_char);
                  
                    /* make sure valid for activity type */
                    v_success := sim_import_result_checks.check_type_category_match(v_type, v_activity_cat, NULL, v_line_cnt, v_import_log_seq, v_habitat);
                    IF v_success = 2 THEN
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Trip QC Type is not valid for the chosen activity type.');
                       
                    
                    END IF;
                  ELSE
                    v_status := 'N';
                  END IF;
                ELSE
                  -- Error for null trip QC Type handled after the loop.
                  NULL;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Trip QC Type ignored. It is only used when Activity Type is "Trip QC".');
                   
                
                END IF;
              END IF;
            
              ----------
              -- Trip ID
              ----------
            ELSIF v_item_name = 'Trip ID' THEN
            
              -- jah 2-23-04 trim trailing spaces (but do NOT upper case until after translation!)
              v_item := rtrim(v_item);
            
              /* check instructions to see what SIM should do */
              v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-TRIP', v_dummy_seq, v_import_log_seq, v_line_cnt);
            
              -- Is there a trip instruction?
              IF v_instr_code IS NOT NULL THEN
                -- if yes
              
                -- Is the instruction "Read existing trip ID from data file"?
                IF v_instr_code = 'RES-TRIP-04' THEN
                  -- if yes
                
                  -- does the data file contain a trip ID?
                  IF v_item IS NOT NULL THEN
                    -- if yes
                  
                    /* kms (6/4/01) 
                    set trip id equal to whatever is in the column (check for translation), 
                    will validate later. */
                  
                    -- translate
                    v_trip_id := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  
                    -- did translate work properly?
                    IF (v_trip_id IS NOT NULL) THEN
                      -- if yes
                    
                      -- jah 2-23-04 Convert to uppercase because STORET GUI converts Trip ID to uppercase
                      v_trip_id := upper(v_trip_id);
                    
                    ELSE
                      -- if no
                    
                      -- prevent migration of this record
                      v_status := 'N';
                      v_trip_id := NULL;
                      v_trip_seq := NULL;
                    
                    END IF;
                  
                  ELSE
                    -- if no (i.e., the data file does not contain a trip ID)
                  
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Trip ID is required and has not been provided. ' ||
                                                         'Provide the trip id or select an instruction to create trips.');
                     
                  
                    -- prevent migration of this record
                    v_status := 'N';
                    v_trip_id := NULL;
                    v_trip_seq := NULL;
                  
                  END IF;
                
                ELSE
                  -- if no (i.e., the instruction is NOT "Read existing trip ID from data file")
                
                  -- does the data file contain an activity start date?
                  IF v_start_date IS NOT NULL THEN
                    -- if yes
                  
                    -- use the activity start date to get the trip ID
                    v_trip_id := sim_import_result_extras.get_trip_id(v_instr_code, v_start_date, v_trip_rec.stp_start_date, v_trip_rec.stp_end_date, p_org_id, v_dummy_seq, v_line_cnt, v_import_log_seq);
                  
                  ELSE
                    -- if no (i.e., the data file does NOT contain an activity start date)
                  
                    -- prevent migration of this record
                    v_status := 'N';
                    v_trip_seq := NULL;
                    v_trip_id := NULL;
                  
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Activity Start Date is required to create Trip ID.');
                     
                  
                  END IF;
                
                END IF;
              
                -- do we have the trip ID yet?  
                IF v_trip_id IS NOT NULL THEN
                  -- if yes
                
                  /* kms (6/4/01)
                  Check SIM table to see if trip is there, 
                  if not, check STORET. Either way, add
                  the trip to SIM. */
                
                  v_trip_rec.stp_trip_id := v_trip_id;
                  v_trip_rec.stp_trip_name := v_trip_name;
                  v_trip_rec.stp_end_date := v_trip_stop_date;
                  v_trip_rec.stp_tsrtrip_is_number := NULL;
                  v_trip_rec.stp_tsrtrip_org_id := NULL;
                  v_trip_rec.stp_import_seq := v_import_log_seq;
                
                  -- try to ensure the trip is in SIM
                  get_or_add_trip_rec(v_trip_rec, p_org_id, v_dummy_seq, v_trip_start_date);
                
                  -- is the Trip in SIM?
                  IF v_trip_rec.stp_seq IS NULL THEN
                    -- if no
                  
                    -- prevent migration of this record
                    v_status := 'N';
                    v_trip_seq := NULL;
                    v_trip_id := NULL;
                  
                  ELSE
                    -- if yes (i.e., the Trip is in SIM)
                  
                    v_trip_seq := v_trip_rec.stp_seq;
                    v_storet_key.is_number := v_trip_rec.stp_tsrtrip_is_number;
                    v_storet_key.org_id := v_trip_rec.stp_tsrtrip_org_id;
                  
                    /* kms (6/4/01) - if this is the first time the trip has been processed by this import
                    and it already exists in STORET, load all station visits on this trip in STORET
                    into the SIM table. */
                  
                    -- is the Trip in STORET
                    -- AND do we need to copy the station visits into SIM?
                    IF (v_storet_key.is_number IS NOT NULL)
                       AND (NOT v_trip_table.EXISTS(v_trip_seq)) THEN
                      -- if yes
                    
                      -- load the station visits into SIM
                      load_station_visits(v_trip_seq, v_storet_key, p_org_id, v_stvst_id_table, v_dummy_seq);
                    
                    END IF;
                  
                    -- indicate the Trip is in SIM, so we load everything again for subsequent records
                    v_trip_table(v_trip_seq) := v_trip_id;
                  
                    -- do we need to see if the activity falls within the start and end dates?
                    IF v_trip_seq IS NOT NULL
                       AND v_start_date IS NOT NULL THEN
                      -- if yes
                    
                      v_success := sim_import_result_checks.check_trip_dates(v_trip_seq, v_start_date, p_org_id, v_dummy_seq, v_line_cnt, v_import_log_seq);
                    
                      -- does the activity falls within trip start and end dates?
                      IF v_success != 0 THEN
                        -- if yes
                      
                        v_trip_is_nbr := v_storet_key.is_number;
                        v_trip_org_id := v_storet_key.org_id;
                      
                      ELSE
                        -- if no
                      
                        -- prevent migration of this record
                        v_status := 'N';
                        v_trip_is_nbr := NULL;
                        v_trip_is_nbr := NULL;
                        v_trip_seq := NULL;
                      
                      END IF;
                    
                    ELSIF v_trip_seq IS NOT NULL
                          AND v_start_date IS NULL THEN
                      /* kms whenever the activity start date is null, set it to be the start date for the trip */
                      OPEN c_trip_dates(v_trip_seq);
                      FETCH c_trip_dates
                        INTO v_start_date;
                      CLOSE c_trip_dates;
                    
                      v_trip_is_nbr := v_storet_key.is_number;
                      v_trip_org_id := v_storet_key.org_id;
                    
                      /* issue warning telling user the start date has been added */
                       
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         'Activity Start Date has been set to ' ||
                                                         v_start_date ||
                                                         ', the trip start date.');
                     
                  
                    ELSE
                      /* no trip, invalidate activity and make sure error message
                      written to the error log */
                      v_trip_is_nbr := NULL;
                      v_trip_is_nbr := NULL;
                      v_trip_seq := NULL;
                      v_status := 'N';
                    
                      IF v_start_date IS NULL
                         AND v_start_date_val IS NULL THEN
                         
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Start Date is required and has not been provided.');
                     
                  
                      END IF;
                    
                    END IF;
                  
                  END IF;
                
                END IF;
              
              ELSE
                /* if no instructions provided, write error message and set status to 'N' */
                v_trip_is_nbr := NULL;
                v_trip_org_id := NULL;
                v_trip_seq := NULL;
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'no instructions provided for trip, update current import configuration or select a new one.');
                 
              
              END IF;
            
              ------------------
              -- Trip Start Date
              ------------------
            ELSIF v_item_name = 'Trip Start Date' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_trip_start_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq, v_sid_seq);
                  IF v_trip_start_date IS NULL THEN
                    v_status := 'N';
                     
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'Trip Start Date format not recognized.');
                 
              
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_trip_start_date := NULL;
                END IF;
              ELSE
                v_trip_start_date := NULL;
              END IF;
            
              -----------------
              -- Trip Stop Date
              -----------------
            ELSIF v_item_name = 'Trip Stop Date' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_trip_stop_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq, v_sid_seq);
                  IF v_trip_stop_date IS NULL THEN
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Trip Stop Date format not recognized.');
                     
                  
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_trip_stop_date := NULL;
                END IF;
              ELSE
                v_trip_stop_date := NULL;
              END IF;
            
            ELSIF v_item_name = 'Trip Name' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF length(v_item) > 60 THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Trip Name is longer that 60 characters.');
                     
                  
                    v_status := 'N';
                    v_trip_name := NULL;
                  ELSE
                    v_trip_name := v_item;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_trip_name := NULL;
                END IF;
              ELSE
                v_trip_name := NULL;
              END IF;
            
            ELSIF v_item_name = 'Station ID' THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
              
                -- jah 2-23-04 trim trailing spaces (but do NOT upper case until after translation!)
                v_item := rtrim(v_item);
              
                /* get the station is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_dummy_seq, v_import_log_seq, v_station_id);
              
                -- did item translate okay?
                IF v_storet_key.is_number IS NOT NULL THEN
                  -- if yes
                  v_tsmstatn_is_nbr := v_storet_key.is_number;
                  v_tsmstatn_org_id := v_storet_key.org_id;
                  -- jah 2-23-04 Convert to uppercase because STORET GUI converts Station ID to uppercase
                  v_station_id := upper(v_station_id);
                ELSE
                  -- if no
                  /* station could not be found in storet, prevent export. */
                  v_tsmstatn_is_nbr := NULL;
                  v_tsmstatn_org_id := NULL;
                  v_station_id := NULL;
                  v_status := 'N';
                END IF;
              ELSIF v_trip_qc != 1 THEN
                v_tsmstatn_org_id := NULL;
                v_tsmstatn_is_nbr := NULL;
                v_station_id := NULL;
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'Station ID is required and has not been provided.');
                 
              
              ELSE
                v_tsmstatn_org_id := NULL;
                v_tsmstatn_is_nbr := NULL;
                v_station_id := NULL;
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                     'Station ID Ignored - it is not used for Trip QC.');
              END IF;
            
              -------------
              -- Point Type
              -------------
            ELSIF v_item_name = 'Point Type'
                  AND v_new_activity = 1 THEN
            
              -- jah 2-23-04 trim trailing spaces (but do NOT upper case until after translation!)
              v_item := rtrim(v_item);
            
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
              
                IF v_activity_cat = 'Created from Sample' THEN
                  v_point_type := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Point Type is not used on child (Created from) Samples and will be ignored.');
                   
                
                ELSE
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NULL THEN
                    v_status := 'N';
                    v_point_type := NULL;
                  ELSE
                    v_point_type := sim_import_result_extras.get_allowable_value(v_dummy_seq, v_import_log_seq);
                    -- jah 2-24-04 Convert to uppercase because STORET GUI converts Point Type to uppercase
                    v_point_type := upper(v_point_type);
                  END IF;
                END IF;
              ELSE
                v_point_type := NULL;
              END IF;
            
            ELSIF v_item_name = 'Sequence Number'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_activity_cat = 'Created from Sample' THEN
                    v_sequence_number := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         'Sequence Number is not used on child (Created from) Samples and will be ignored.');
                     
                  
                  ELSE
                    IF v_point_type IS NULL THEN
                      v_sequence_number := NULL;
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Point Type is required if Sequence Number is provided.');
                       
                    
                    ELSE
                      IF v_point_type != 'WELL HEAD'
                         AND v_point_type != 'PIPE END' THEN
                        v_sequence_number := sim_import_result_extras.validate_location(v_item, v_point_type, 'SEQ', v_tsmstatn_is_nbr, v_tsmstatn_org_id, v_activity_seq, v_line_cnt, v_import_log_seq, v_tsmalp_is_nbr, v_tsmalp_org_id);
                        IF v_sequence_number = '-1' THEN
                          --09/13/05 -RWK - Changed -1 to String
                          v_status := 'N';
                        END IF;
                      ELSE
                        v_sequence_number := '0'; --09/13/05 -RWK - Changed 0 to String
                      END IF;
                    END IF;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'  
                  v_sequence_number := NULL;
                END IF;
              ELSE
                v_sequence_number := NULL;
              END IF;
            
              --09/13/05 -RWK - Chnaged 'Well or Pipe ID' to 'Well or Pipe Number'
            ELSIF v_item_name = 'Well or Pipe Number'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_activity_cat = 'Created from Sample' THEN
                    v_sequence_number := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         'Well or Pipe Number is not used on child (Created from) Samples and will be ignored.');
                     
                  
                  ELSE
                    IF v_point_type = 'WELL HEAD'
                       OR v_point_type = 'PIPE END' THEN
                      v_well_pipe_num := sim_import_result_extras.validate_location(v_item, v_point_type, 'WELL', v_tsmstatn_is_nbr, v_tsmstatn_org_id, v_activity_seq, v_line_cnt, v_import_log_seq, v_tsmalp_is_nbr, v_tsmalp_org_id);
                      IF v_well_pipe_num = '-1' THEN
                        --09/13/05 -RWK - Changed -1 to String
                        v_status := 'N';
                        v_well_pipe_num := NULL;
                      END IF;
                    ELSE
                      v_well_pipe_num := NULL;
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           'Well or Pipe Number is not used unless point type is WELL HEAD or PIPE END and will be ignored.');
                       
                    
                    END IF;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_well_pipe_num := NULL;
                END IF;
              ELSE
                v_well_pipe_num := NULL;
              END IF;
            
            ELSIF v_item_name = 'Additional Location Information'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF length(v_item) > 254 THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                         ' is longer that 254 characters and will be truncated.');
                     
                  
                    v_add_loc_info := substr(v_item, 1, 254);
                  ELSE
                    v_add_loc_info := v_item;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_add_loc_info := NULL;
                END IF;
              ELSE
                v_add_loc_info := NULL;
              END IF;
            
            ELSIF v_item_name = 'Station Visit Arrival Date' THEN
              v_start_date_val := v_item;
            
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
                /* kms 6/2/01) - this date will be replaced later if the station visit already exists. */
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
              
                IF v_item IS NOT NULL THEN
                  v_visit_start_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq, v_sid_seq);
                  IF v_start_date IS NULL THEN
                    v_status := 'N';
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Station Visit Start Date format not recognized.');
                     
                  
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_visit_start_date := NULL;
                END IF;
              ELSE
                /* kms - visit start date has not been included, wait to see if it can be obtained from
                the station visit (if it already exists) or the trip. */
                v_visit_start_date := NULL;
              END IF;
            
              -----------------------
              -- Station Visit Number
              -----------------------
            ELSIF v_item_name = 'Station Visit Number' THEN
            
              -- SLP 06/13/02 Remove trailing spaces
              v_item := rtrim(v_item);
            
              /* Business Rule
              Activity records must specify a Station
              and the Station must already be in STORET. */
            
              -- does the import data record specify a Station and is the Station in STORET?
              -- AND is this a quality control record?
              IF (v_tsmstatn_is_nbr IS NOT NULL)
                 AND (v_trip_qc != 1) THEN
                -- if YES (the station is specified and in STORET)
                -- and NO (this is not a quality control record)
              
                /* kms (5/22/01) - set temporary key to station key used in preparation
                for insuring the station visit is valid for this activity. */
                v_temp_key.is_number := v_tsmstatn_is_nbr;
                v_temp_key.org_id := v_tsmstatn_org_id;
              
                -- initialize the STORET Station Visit key
                v_storet_key.is_number := NULL;
                v_storet_key.org_id := NULL;
              
                -- is this an "XML Results" record?
                IF v_sid_seq = 10007 THEN
                  -- if yes
                
                  -- Business Rule: 
                  --   XML Results import data records must specify the Station Visit Number
                
                  -- Force the import instruction to be "Read existing station visit from data file"
                  v_instr_code := 'RES-STVST-04';
                
                ELSE
                  -- if no (i.e., this is NOT an "XML Results" record)
                
                  -- get the import instruction for the station visit number
                  v_instr_code := sim_import_result_extras.get_instructions(p_imp_cfg_seq, 'RES-STVST', v_dummy_seq, v_import_log_seq, v_line_cnt);
                
                END IF;
              
                -- do we have a Station Visits instruction?
                IF v_instr_code IS NOT NULL THEN
                  -- if yes
                
                  -- Is the instruction "Read existing station visit from data file"?
                  IF v_instr_code = 'RES-STVST-04' THEN
                    -- if yes
                  
                    -- does the import data record contain the station visit number?
                    IF (v_item IS NOT NULL) THEN
                      -- if yes
                    
                      -- translate
                      v_station_visit := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                    
                      -- did the translate work properly?
                      IF (v_station_visit IS NULL) THEN
                        -- if no
                      
                        -- sim_import_result_extras.check_translations is responsible for the error message.
                      
                        -- prevent migration of this record
                        v_status := 'N';
                      
                        -- set associated variables to NULL
                        v_visit_seq := NULL;
                        v_station_visit := NULL;
                        v_visit_start_date := NULL;
                        v_storet_key.is_number := NULL;
                        v_storet_key.org_id := NULL;
                        v_tsrstvst_is_nbr := NULL;
                        v_tsrstvst_org_id := NULL;
                        v_temp_key.is_number := NULL;
                        v_temp_key.org_id := NULL;
                      
                      ELSE
                        -- if yes (i.e., the translate worked properly)
                      
                        -- try to retrieve the key to the station visit in SIM_STATION_VISITS table
                        v_visit_seq := sim_import_result_extras.get_visit_seq(v_storet_key, v_temp_key, v_visit_start_date, v_station_visit, v_trip_seq, v_dummy_seq, v_line_cnt, v_import_log_seq);
                      
                        -- Do we need to create a Station Visit record in SIM_STATION_VISITS
                        -- AND is the visit start date NOT in the import data file?
                        IF (v_visit_seq IS NULL)
                           AND (v_visit_start_date IS NULL) THEN
                          -- if yes
                        
                          -- try to use the SIM_TRIPS start date for the Station Visit start date
                          v_visit_start_date := sim_import_result_extras.get_visit_start_date(v_storet_key, v_trip_seq, v_dummy_seq, v_line_cnt, v_import_log_seq);
                        
                        END IF;
                      
                        -- Do we need to create a Station Visit record in SIM_STATION_VISITS
                        -- AND do we still NOT have the visit start date?
                        IF v_visit_seq IS NULL
                           AND v_visit_start_date IS NULL THEN
                          -- if so
                        
                          -- prevent migration of this record
                          v_status := 'N';
                        
                          -- set associated variables to NULL
                          v_visit_seq := NULL;
                          v_station_visit := NULL;
                          v_visit_start_date := NULL;
                          v_storet_key.is_number := NULL;
                          v_storet_key.org_id := NULL;
                          v_tsrstvst_is_nbr := NULL;
                          v_tsrstvst_org_id := NULL;
                          v_temp_key.is_number := NULL;
                          v_temp_key.org_id := NULL;
                        
                           
                          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                               'Station Visit Start Date is required when the station visit does not already exist in STORET.  ' ||
                                                               'Unable to get start date from trip.');
                           
                        
                        END IF;
                      
                      END IF;
                    
                    ELSE
                      -- if no (i.e., the import data record does NOT contain the station visit number)
                    
                      -- prevent migration of this record
                      v_status := 'N';
                    
                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_station_visit := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                    
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Station Visit is required when instruction to read visits from the import file is chosen.');
                       
                    
                    END IF;
                  
                  ELSE
                    -- if no (i.e., the instruction is NOT "Read existing station visit from data file")
                  
                    -- figure out the station visit number and other data
                    v_error_code := check_existing_visit(v_instr_code, -- IN
                                                         v_start_date, -- IN          activity start date
                                                         v_start_time, -- IN          activity start time
                                                         v_trip_seq, -- IN
                                                         v_temp_key, -- IN
                                                         v_visit_start_date, -- OUT
                                                         v_visit_start_time, -- OUT         if v_start_time is NULL, return value is 00:00 (midnight)
                                                         v_visit_end_date, -- OUT
                                                         v_visit_end_time, -- OUT
                                                         v_storet_key, -- OUT
                                                         v_visit_seq, -- IN OUT
                                                         v_stvst_id_table, -- IN OUT
                                                         v_station_visit -- OUT
                                                         );
                  
                    IF v_station_visit = 'ERROR' THEN
                      v_status := 'N';
                    END IF;
                  
                    -- did check_existing_visit work properly?
                    IF (v_error_code IS NOT NULL) THEN
                      -- if no
                    
                      -- Prevent migration of this record
                      v_status := 'N';
                    
                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_station_visit := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                    
                    END IF;
                  
                  END IF;
                
                  -- Do we know the station visit ID?
                  IF (v_station_visit IS NOT NULL) THEN
                    -- if yes
                  
                    -- set the STORET station visit key for subsequent processing
                    v_tsrstvst_is_nbr := v_storet_key.is_number;
                    v_tsrstvst_org_id := v_storet_key.org_id;
                  
                    -- re-initialize the temp key
                    v_temp_key.is_number := NULL;
                    v_temp_key.org_id := NULL;
                  
                  END IF;
                
                  -- Do we need to create a station visit record?
                  IF (v_visit_seq IS NULL)
                     AND (v_station_visit IS NOT NULL) THEN
                    -- if yes
                  
                    /* To create a station visit you need the following.
                    o  The Trip on which the station visit occurs
                    o  The Station Visit Start Time (returned by FUNCTION check_existing_visit)
                    o  Activity Start Time Zone
                       (Business Rule:)
                       but ONLY if the import instruction is 
                       'Create one station visit per date and time'
                       and the station visit start time is not midnight. */
                  
                    -- Is the Trip missing?
                    IF (v_trip_seq IS NULL) THEN
                      -- if yes
                    
                      -- Prevent migration of this record
                      v_status := 'N';
                    
                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_station_visit := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                    
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Could not create the Station Visit because the Trip ID is missing.');
                       
                    
                      -- Do we need the Activity Start Time Zone?
                      -- AND is the Visit Start Time something other than midnight?
                      -- AND is the Activity Start Time Zone missing?
                    ELSIF (v_instr_code = 'RES-STVST-03')
                          AND (v_visit_start_time <> to_date('00:00', 'HH24:MI'))
                          AND (v_start_time_zone IS NULL) THEN
                      -- if yes
                    
                      -- Prevent migration of this record
                      v_status := 'N';
                    
                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_station_visit := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                    
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Could not create the Station Visit because the Activity Start Time Zone is missing.');
                       
                    
                    ELSE
                      -- We have what we need to create a station visit record
                    
                      -- get the get next station visit key
                      OPEN c_get_next_visit_seq;
                      FETCH c_get_next_visit_seq
                        INTO v_visit_rec.ssv_seq;
                      CLOSE c_get_next_visit_seq;
                    
                      -- populate the columns in the Station Visit record
                      v_visit_rec.ssv_stp_seq := v_trip_seq;
                      v_visit_rec.ssv_id := v_station_visit;
                      v_visit_rec.ssv_start_date := v_visit_start_date;
                      v_visit_rec.ssv_start_time := v_visit_start_time;
                      v_visit_rec.ssv_arrival_time_zone := v_start_time_zone; -- jah 3-9-2004
                      v_visit_rec.ssv_end_date := v_visit_end_date;
                      v_visit_rec.ssv_end_time := v_visit_end_time;
                      v_visit_rec.ssv_tsmstatn_is_number := v_tsmstatn_is_nbr;
                      v_visit_rec.ssv_tsmstatn_org_id := v_tsmstatn_org_id;
                      v_visit_rec.ssv_tsrstvst_is_number := v_storet_key.is_number;
                      v_visit_rec.ssv_tsrstvst_org_id := v_storet_key.org_id;
                      v_visit_rec.ssv_tsmorgan_is_number := v_org_is_nbr;
                      v_visit_rec.ssv_tsmorgan_org_id := p_org_id;
                      v_visit_rec.ssv_created_by_sim := 1;
                      v_visit_rec.ssv_import_seq := v_import_log_seq;
                      v_visit_rec.ssv_document := v_visit_document;
                    
                      -- Try to insert the Station Visit into SIM_STATION_VISITS
                      v_success := sim_import_result_extras.create_sim_station_visit(v_visit_rec, p_import_log_seq);
                    
                      -- Were we able to insert the Station Visit into SIM_STATION_VISITS?
                      IF (v_success IS NOT NULL) THEN
                        -- if yes
                      
                        -- set the SIM_STATION_VISITS key for subsequent processing
                        v_visit_seq := v_visit_rec.ssv_seq;
                      ELSE
                        -- if no (i.e., we were NOT able to insert the Station Visit into SIM_STATION_VISITS)
                      
                        -- Prevent migration of this record
                        v_status := 'N';
                      
                        -- set associated variables to NULL
                        v_visit_seq := NULL;
                        v_station_visit := NULL;
                        v_visit_start_date := NULL;
                        v_storet_key.is_number := NULL;
                        v_storet_key.org_id := NULL;
                        v_tsrstvst_is_nbr := NULL;
                        v_tsrstvst_org_id := NULL;
                        v_temp_key.is_number := NULL;
                        v_temp_key.org_id := NULL;
                      
                      END IF;
                    
                    END IF;
                  
                  END IF;
                
                ELSE
                  -- if no (i.e., we do NOT have a Station Visits instruction)
                
                  -- Prevent migration of this record
                  v_status := 'N';
                
                  -- set associated variables to NULL
                  v_visit_seq := NULL;
                  v_station_visit := NULL;
                  v_visit_start_date := NULL;
                  v_storet_key.is_number := NULL;
                  v_storet_key.org_id := NULL;
                  v_tsrstvst_is_nbr := NULL;
                  v_tsrstvst_org_id := NULL;
                  v_temp_key.is_number := NULL;
                  v_temp_key.org_id := NULL;
                
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'No instructions provided for station visit, update import configuration or select a new one.');
                   
                
                END IF;
              
                -- Do we know the Station Visit Number?
                IF (v_station_visit IS NOT NULL) THEN
                  -- if yes
                
                  /* kms (6/6/01)- make sure station visit start date comes before or is the same as
                  activity start date */
                
                  -- Do we know the key of the Station Visit record in SIM?
                  IF v_visit_seq IS NOT NULL THEN
                    -- if yes
                  
                    -- Double check the station visit dates
                    v_success := sim_import_result_checks.check_visit_dates(v_visit_seq, v_start_date, p_org_id, v_dummy_seq, v_line_cnt, v_import_log_seq);
                  
                    -- Is there a problem with the station visit dates?
                    IF v_success != 1 THEN
                      -- if yes
                    
                      -- Prevent migration of this import data record
                      v_status := 'N';
                    
                      -- set associated variables to NULL
                      v_visit_seq := NULL;
                      v_visit_start_date := NULL;
                      v_storet_key.is_number := NULL;
                      v_storet_key.org_id := NULL;
                      v_tsrstvst_is_nbr := NULL;
                      v_tsrstvst_org_id := NULL;
                      v_temp_key.is_number := NULL;
                      v_temp_key.org_id := NULL;
                    
                    END IF;
                  
                  END IF;
                
                END IF;
              
                -- is this a quality control record?
                -- (we already know the Station is not in the import record or is NOT in STORET)
              ELSIF v_trip_qc != 1 THEN
                -- if no
              
                -- Prevent migration of this record
                v_status := 'N';
              
                -- set associated variables to NULL
                v_visit_seq := NULL;
                v_station_visit := NULL;
                v_visit_start_date := NULL;
                v_storet_key.is_number := NULL;
                v_storet_key.org_id := NULL;
                v_tsrstvst_is_nbr := NULL;
                v_tsrstvst_org_id := NULL;
                v_temp_key.is_number := NULL;
                v_temp_key.org_id := NULL;
              
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'Unable to load Station Visit on this activity since Station was either not provided or is invalid.');
                 
              
              ELSE
                -- otherwise (i.e., this is a quality control record)
              
                -- Note: 
                -- DO NOT put any code here to prevent migration of this record, 
                -- it is a quality control record!
              
                -- set associated variables to NULL
                v_visit_seq := NULL;
                v_station_visit := NULL;
                v_visit_start_date := NULL;
                v_storet_key.is_number := NULL;
                v_storet_key.org_id := NULL;
                v_tsrstvst_is_nbr := NULL;
                v_tsrstvst_org_id := NULL;
                v_temp_key.is_number := NULL;
                v_temp_key.org_id := NULL;
              
              END IF;
            
              --------------            
              -- Activity ID
              --------------            
            ELSIF v_item_name = 'Activity ID' THEN
            
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
              
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF length(v_item) > 12 THEN
                    v_activity_id := upper(rpad(substr(v_item, 1, 12), 12)); -- truncate to allowed length
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Activity ID is longer than 12 characters.');
                     
                  
                    v_status := 'N';
                  ELSE
                    v_activity_id := upper(v_item); -- dcl id's need to always be uppers
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_activity_id := '????????';
                END IF;
              ELSE
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                     'Activity ID is required and has not been provided.');
                 
              
                v_activity_id := '????????';
                v_status := 'N';
              END IF;
            
              IF v_last_activity_id IS NULL
                 OR v_activity_id != v_last_activity_id
                 OR nvl(v_replicate_nbr, '`') != nvl(v_last_rep_nbr, '`')
                 OR nvl(v_visit_seq, '-1') != nvl(v_last_visit_seq, '-1')
                 OR nvl(v_trip_seq, '-1') != nvl(v_last_trip_seq, '-1') THEN
              
                /* check for activity in sim_field_activities and decide if it needs to be made */
                v_activity_seq := sim_import_result_checks.check_activity_id(v_activity_id, v_replicate_nbr, v_tsmstatn_is_nbr, v_trip_seq, v_visit_seq, rpad(p_org_id, 8), v_type, v_medium, v_status, v_line_cnt, v_import_log_seq, v_assnd_seq_num);
              
                IF v_activity_seq IS NULL THEN
                  /* not first line, but new activity id */
                  v_last_activity_id := v_activity_id;
                  v_last_rep_nbr := v_replicate_nbr;
                  v_last_visit_seq := v_visit_seq;
                  v_last_trip_seq := v_trip_seq;
                  v_new_activity := 1;
                  v_result_seq_num := 0;
                ELSE
                  /* activity already exists, do not create */
                  v_new_activity := 0;
                  v_result_seq_num := v_assnd_seq_num;
                  v_last_activity_id := v_activity_id;
                  v_last_rep_nbr := v_replicate_nbr;
                  v_last_visit_seq := v_visit_seq;
                  v_last_trip_seq := v_trip_seq;
                END IF;
              
              ELSE
              
                /* Check the medium to make sure they are the same */
                IF v_habitat <> 1 THEN
                  -- If Habitat Assessment then Medium is not required
                  IF v_last_medium != v_medium THEN
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'A new medium for this activity has been found.  Only one medium per activity is allowed.');
                     
                  
                  END IF;
                END IF;
              
                /* Check the activity type to make sure they are the same */
                IF v_last_type != v_type THEN
                  v_status := 'N';
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'A new activity type for this activity has been found.  Only one activity type per activity is allowed.');
                   
                
                END IF;
              
                -- another line, same activity
                v_new_activity := 0;
                v_last_activity_id := v_activity_id;
                v_last_rep_nbr := v_replicate_nbr;
                v_last_visit_seq := v_visit_seq;
                v_last_trip_seq := v_trip_seq;
              END IF;
            
              IF v_new_activity = 1 THEN
                -- Get sequence for field activity record (save so can use for related results)
                OPEN c_activity_seq;
                FETCH c_activity_seq
                  INTO v_activity_seq;
                IF c_activity_seq%NOTFOUND THEN
                  v_activity_seq := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'Failed to get new activity sequence.');
                   
                
                END IF;
                CLOSE c_activity_seq;
              
              END IF;
            
           
              /*SF - 12/12/02 Check for stations, wells, and locations in SIM. Doing this now because
              we only want to do it for each NEW activity */
              IF v_tsmstatn_is_nbr IS NOT NULL
                 AND v_new_activity = 1
                 AND v_trip_qc != 1 THEN
                v_success := check_if_station_exists(v_station_id, p_org_id, p_sta_seq);
                IF v_success = 0 THEN
                  RAISE e_bad_station;
                  v_status := 'N';
                END IF;
              END IF;
            
            ELSIF (v_item_name = 'Visit Comments' OR
                  v_item_name = 'Station Visit Conditions and Other Comments')
                  AND v_new_activity = 1 THEN
              IF v_trip_qc != 1
                 AND v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_stvst_cmt := v_item;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_stvst_cmt := NULL;
                END IF;
              ELSE
                v_stvst_cmt := NULL;
              END IF;
            
            ELSIF v_item_name = 'Visit Document/Graphic' THEN
              IF v_item IS NOT NULL THEN
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF sim_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                    v_visit_document := v_item;
                  ELSE
                    v_status := 'N';
                    v_visit_document := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_visit_document := NULL;
                END IF;
              ELSE
                v_visit_document := NULL;
              END IF;
            
            ELSIF v_item_name = 'Medium'
                  OR v_item_name = 'Activity Medium' THEN
              --need medium on every result to check char group id
              v_last_medium := v_medium;
              IF v_item IS NOT NULL THEN
                /* get medium - check translations and allowable values tables */
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_medium := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_status := 'N';
                  v_medium := NULL;
                END IF;
                /* SF (2/4/02) Make sure Bio Medium is not on a Field Measurement */
                IF v_medium = 'Biological'
                   AND v_type = 'Field Msr/Obs' THEN
                  v_status := 'N';
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'Field Measurements cannot be made in a Biological Medium');
                   
                
                END IF;
              ELSE
                v_medium := NULL;
                END IF;
            
            ELSIF v_item_name = 'Activity Start Date'
                  AND v_new_activity = 1 THEN
            
              BEGIN
                /* if there is no start date, set status to 'N' so it will not be exported */
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    /* get start date using formatting options */
                    v_start_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq, v_sid_seq);
                  
                    IF v_start_date IS NULL THEN
                      v_status := 'N';
                      v_start_date_val := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                    v_start_date := NULL;
                  END IF;
                ELSE
                  v_start_date := NULL;
                END IF;
              
              EXCEPTION
                WHEN OTHERS THEN
                  v_status := 'N';
                  v_start_date := NULL;
                  v_start_time := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       SQLERRM ||
                                                       ' - getting start date.');
                   
                
              END;
            
              ----------------------  
              -- Activity Start Time
              ----------------------  
            ELSIF v_item_name = 'Activity Start Time'
                  AND v_new_activity = 1 THEN
            
              BEGIN
                /* if a start time is supplied, format it for STORET */
                IF v_item IS NOT NULL /* AND v_trip_qc != 1 */
                 THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    /* get start time using formatting options */
                    v_start_time := get_time(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq);
                    IF v_start_time IS NULL THEN
                      v_status := 'N';
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                    v_start_time := NULL;
                  END IF;
                ELSE
                  v_start_time := NULL;
                END IF;
              
              EXCEPTION
                WHEN OTHERS THEN
                  v_start_time := NULL;
                  v_start_date := NULL;
                
                  -- Prevent migration of this record
                  v_status := 'N';
                
                  v_import_desc := 'Line ' || v_line_cnt || ': ' ||
                                   'Error in date/time "' || v_start_date || '/' ||
                                   v_item || '".';
                
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       v_import_desc);
                   
                
              END;
            
              ---------------------------
              -- Activity Start Time Zone
              ---------------------------
            ELSIF v_item_name = 'Activity Start Time Zone'
                  AND v_new_activity = 1 THEN
            
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                /* get allowable value sequence for time zone */
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_start_time_zone := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_status := 'N';
                  v_start_time_zone := NULL;
                END IF;
              ELSE
                v_start_time_zone := NULL;
              END IF;
            
            ELSIF v_item_name = 'Activity End Date'
                  AND v_new_activity = 1 THEN
              BEGIN
                IF v_item IS NOT NULL
                   AND v_trip_qc != 1 THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_end_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq, v_sid_seq);
                  
                    IF v_end_date IS NULL THEN
                      v_status := 'N';
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                    v_end_date := NULL;
                  END IF;
                ELSE
                  v_end_date := NULL;
                  IF v_trip_qc = 1 THEN
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored - it is not used for Trip QC.'); 
                  END IF;  
                END IF;
              
              EXCEPTION
                WHEN OTHERS THEN
                  v_status := 'N';
                  v_end_date := NULL;
                  v_end_time := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       SQLERRM ||
                                                       ' - getting stop date.');
                   
                
              END;
            
            ELSIF v_item_name = 'Activity End Time'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                AND v_trip_qc != 1 THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_end_time := get_time(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq);
                
                  IF v_end_time IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_end_time := NULL;
                END IF;
              ELSE
                v_end_time := NULL;
                IF v_trip_qc = 1 THEN
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored - it is not used for Trip QC.');   
                END IF; 
              END IF;
            
            ELSIF v_item_name = 'Activity End Time Zone'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                /* get allowable value sequence for time zone */
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_stop_time_zone := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_status := 'N';
                  v_stop_time_zone := NULL;
                END IF;
              ELSE
                v_stop_time_zone := NULL;
              END IF;
            
            ELSIF v_item_name = 'Activity Comments'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1 THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                -- 09/12/05 RWK - If statement added to check for Activity Comments
                -- that exceed 254 characters.  If this situation exists a "friendly" 
                -- message is given to the user.
                IF length(v_item) > 254 THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'Activity Comment value cannot exceed' ||
                                                       ' 254 charaters.');
                   
                
                  v_item := NULL;
                END IF;
              
                IF v_item IS NOT NULL THEN
                  /* comments */
                  v_comment := rtrim(v_item);
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_comment := NULL;
                END IF;
              ELSE
                v_comment := NULL;
              END IF;
            
            ELSIF v_item_name = 'Depth to Activity'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  BEGIN
                    --check to make sure it's a valid number
                    IF v_trip_qc != 1
                       AND v_habitat != 1 THEN
                      v_dummy_num := to_number(v_item);
                      v_depth := v_item;
                    ELSE
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                           ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                       
                    
                      v_depth := NULL;
                    END IF;
                  EXCEPTION
                    WHEN OTHERS THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is not a valid number of 8 or fewer characters.');
                       
                    
                      v_status := 'N';
                  END;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_depth := NULL;
                END IF;
              ELSE
                v_depth := NULL;
              END IF;
            
            ELSIF v_item_name = 'Depth to Activity Units'
                  AND v_new_activity = 1 THEN
              v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1 THEN
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_depth_uom := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_depth_uom := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_depth_uom := NULL;
              END IF;
            
            ELSIF (v_item_name = 'Relative Depth' OR
                  v_item_name = 'Depth to Activity Relative Depth Indicator')
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_rel_depth := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_rel_depth := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                   
                
                END IF;
                v_rel_depth := NULL;
              END IF;
            
            ELSIF v_item_name = 'Upper Depth'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  BEGIN
                    --check to make sure it's a valid number
                    IF (v_depth IS NOT NULL OR v_rel_depth IS NOT NULL OR
                       v_depth_uom IS NOT NULL)
                       AND v_item IS NOT NULL THEN
                      v_status := 'N';
                      
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          'Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.');
                       
                    
                      v_up_depth := NULL;
                    ELSE
                      IF v_trip_qc != 1
                         AND v_habitat != 1 THEN
                        v_dummy_num := to_number(v_item);
                        v_up_depth := v_item;
                      ELSE
                        IF v_item IS NOT NULL THEN
                           
                          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                              v_item_name ||
                                                              ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                           
                        
                          v_up_depth := NULL;
                        END IF;
                      END IF;
                    END IF;
                  EXCEPTION
                    WHEN OTHERS THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is not a valid number of 8 or fewer characters.');
                       
                    
                      v_status := 'N';
                  END;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_up_depth := NULL;
                END IF;
              ELSE
                v_up_depth := NULL;
              END IF;
            
            ELSIF v_item_name = 'Lower Depth'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  BEGIN
                    --check to make sure it's a valid number
                    IF (v_depth IS NOT NULL OR v_rel_depth IS NOT NULL OR
                       v_depth_uom IS NOT NULL)
                       AND v_item IS NOT NULL THEN
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          'Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.');
                       
                    
                      v_low_depth := NULL;
                    ELSE
                      IF v_trip_qc != 1
                         AND v_habitat != 1 THEN
                        v_dummy_num := to_number(v_item);
                        v_low_depth := v_item;
                      ELSE
                        IF v_item IS NOT NULL THEN
                           
                          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                              v_item_name ||
                                                              ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                           
                        
                        END IF;
                        v_low_depth := NULL;
                      END IF;
                    END IF;
                  EXCEPTION
                    WHEN OTHERS THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is not a valid number of 8 or fewer characters.');
                       
                    
                      v_status := 'N';
                  END;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_low_depth := NULL;
                END IF;
              ELSE
                v_low_depth := NULL;
              END IF;
            
            ELSIF v_item_name = 'Upper/Lower Depth Units'
                  AND v_new_activity = 1 THEN
              v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
              IF (v_depth IS NOT NULL OR v_rel_depth IS NOT NULL OR
                 v_depth_uom IS NOT NULL)
                 AND v_item IS NOT NULL THEN
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.');
                 
              
                v_depth_range_uom := NULL;
              ELSE
                IF v_item IS NOT NULL
                   AND v_trip_qc != 1
                   AND v_habitat != 1 THEN
                  /* get the depth range units allowable value - check translations table */
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_depth_range_uom := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  ELSE
                    v_depth_range_uom := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  IF v_item IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                     
                  
                  END IF;
                  v_depth_range_uom := NULL;
                END IF;
              END IF;
            
            ELSIF (v_item_name = 'Depth Measured From' OR
                  v_item_name = 'Depth to Activity Measured From Reference Point')
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_trip_qc != 1
                     AND v_habitat != 1 THEN
                    IF length(v_item) > 30 THEN
                       
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        'Depth Measured From is longer than 30 characters and will be truncated');
                     
                  
                      v_item := instr(v_item, 1, 30);
                    END IF;
                    v_depth_ref := upper(v_item);
                  ELSE
                    IF v_item IS NOT NULL THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                         ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                       
                    
                    END IF;
                    v_depth_ref := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_depth_ref := NULL;
                END IF;
              ELSE
                v_depth_ref := NULL;
              END IF;
            
            ELSIF v_item_name = 'Depth Zone Type'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_depth_zone := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_depth_zone := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                   
                
                  v_depth_zone := NULL;
                ELSE
                  v_depth_zone := NULL;
                END IF;
              END IF;
            
            ELSIF v_item_name = 'Thermocline'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_thermo := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_thermo := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                   
                
                END IF;
                v_thermo := NULL;
              END IF;
            
            ELSIF v_item_name = 'Halocline'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_halo := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_halo := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                   
                
                END IF;
                v_halo := NULL;
              END IF;
            
            ELSIF v_item_name = 'Pycnocline'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_pycno := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_pycno := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Trip QC or Habitat Assessment.');
                   
                
                END IF;
                v_pycno := NULL;
              END IF;
            
            ELSIF v_item_name = 'Personnel'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_no_person := 0;
              
                /* can have multiple personnel separated by '\' - get number of personnel to expect */
                /* kms - translate function removes all occurences of the delimiter.  By subtracting
                the length of what it returns from the length of the original, we find the number
                of delimiters that were in the original.  We then add 1 since we are counting the
                number of personnel, not the number of delimiters. */
                v_no_person := length(v_item) - length(translate(v_item, 'a\', 'a')) + 1;
              
                /* separate item into personnel using parse procedure if more than 1 person */
                IF v_no_person > 1 THEN
                  BEGIN
                    sim_import_pkg.parse_line_table(v_item, v_personnel_table, v_no_person, '\', v_bad_parse, v_import_log_seq);
                  EXCEPTION
                    WHEN no_data_found THEN
                      NULL;
                  END;
                ELSE
                  v_personnel_table(1) := v_item;
                END IF;
              
                /* loop through personnel table and add each person in column to the activity */
                FOR v_number IN 1 .. v_no_person LOOP
                
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_personnel_table(v_number), p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NULL THEN
                    v_persn_is_table(v_number) := NULL;
                    v_persn_org_table(v_number) := NULL;
                    v_status := 'N';
                  ELSE
                    v_persn_is_table(v_number) := v_storet_key.is_number;
                    v_persn_org_table(v_number) := v_storet_key.org_id;
                  END IF;
                
                END LOOP;
              
              ELSE
                v_no_person := 0;
                v_tsmpersn_is_nbr := NULL;
                v_tsmpersn_org_id := NULL;
              END IF;
            
              ------------------------------
              -- Cooperating Organization --
              ------------------------------
            ELSIF v_item_name = 'Cooperating Organization'
                  AND v_new_activity = 1 THEN
            
              -- does the cooperating organization field have something in it?
              IF v_item IS NOT NULL THEN
                -- if yes
              
                /*
                  Note: The Cooperating Organization field can have multiple values
                        seperated by \ (i.e., backward slashes). Count the number of 
                        backward slashes in the item and add one to determine how 
                        many cooperating organizations the field contains.
                */
              
                -- Count the backward slashes and add one
                v_no_cporgs := length(v_item) - length(translate(v_item, 'a\', 'a')) + 1;
              
                -- Does this field contain more than one cooperating organization?
                IF v_no_cporgs > 1 THEN
                  -- if yes
                  -- Put the cooperating organizations into an array (v_cporg_table)
                  BEGIN
                    sim_import_pkg.parse_line_table(v_item, v_cporg_table, v_no_cporgs, '\', v_bad_parse, v_import_log_seq);
                  EXCEPTION
                    WHEN no_data_found THEN
                      NULL;
                  END;
                ELSE
                  -- if no (it contains only one cooperating organization)
                  -- Put the cooperating organization into an array (v_cporg_table)
                  v_cporg_table(1) := v_item;
                END IF;
              
                -- loop through cooperating organizations array
                FOR v_number IN 1 .. v_no_cporgs LOOP
                
                  -- get the associated STORET key
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_cporg_table(v_number), p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  -- did we get the associated STORET key?
                  IF v_storet_key.is_number IS NOT NULL THEN
                    -- if yes
                    -- Add the cooperating organization to the activity
                    v_cporg_is_table(v_number) := v_storet_key.is_number;
                    v_cporg_org_table(v_number) := v_storet_key.org_id;
                  ELSE
                    -- if no
                    -- Do not add the cooperating organization to the activity
                    v_cporg_is_table(v_number) := NULL;
                    v_cporg_org_table(v_number) := NULL;
                    v_status := 'N';
                  END IF;
                
                END LOOP;
              
              ELSE
                -- if no (the cooperating organization field is NULL)
                -- set everything to zero or NULL, as appropriate
                v_no_cporgs := 0;
                v_tsmcporg_is_nbr := NULL;
                v_tsmcporg_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Activity Document/Graphic' THEN
              IF v_item IS NOT NULL THEN
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF sim_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                    v_activity_document := v_item;
                  ELSE
                    v_status := 'N'; --Invalid document.  Set to 'N'.
                    v_activity_document := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; --Translates to NULL.  Set to 'N'.
                  v_activity_document := NULL;
                END IF;
              ELSE
                v_activity_document := NULL;
              END IF;
            
              ---------------------------------
              -- Sample Collection Procedure ID
              ---------------------------------
            
            ELSIF v_item_name = 'Sample Collection Procedure ID'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                v_item := rtrim(v_item);
              
                IF v_trip_qc != 1
                   AND v_habitat != 1
                   AND v_type != 'Field Msr/Obs' THEN
                  /* get the procedure is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_proc_is_nbr := v_storet_key.is_number;
                    v_proc_org_id := v_storet_key.org_id;
                    v_proc_name := v_dummy_char;
                    -- check if trawl gear is used and get tsmalp_is_nbr
                    -- 1 = trawl, 2 = trap/net, 0 = all others
                    v_check_trawl_gear := sim_import_result_checks.check_trawl_gear(v_proc_is_nbr, v_proc_org_id, v_activity_seq, v_line_cnt, v_import_log_seq);
                  ELSE
                    v_proc_org_id := NULL;
                    v_proc_is_nbr := NULL;
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        'Sample Collection Procedure ID is required.');
                     
                  
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Sample Collection Procedure ID is not used for QC, Field Measurements, or Habitat Assessment ');
                   
                
                  v_proc_is_nbr := NULL;
                  v_proc_org_id := NULL;
                  v_gearcfg_is_nbr := NULL;
                  v_gearcfg_org_id := NULL;
                  v_gear_is_nbr := NULL;
                  v_gear_org_id := NULL;
                END IF;
              ELSE
                v_proc_is_nbr := NULL;
                v_proc_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Gear ID'
                  AND v_new_activity = 1 THEN
              -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
              v_item := rtrim(v_item);
            
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1
                 AND v_type != 'Field Msr/Obs' THEN
              
                /* get the gear is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
              
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_gear_is_nbr := v_storet_key.is_number;
                  v_gear_org_id := v_storet_key.org_id;
                ELSE
                  v_gear_org_id := NULL;
                  v_gear_is_nbr := NULL;
                  v_status := 'N';
                END IF;
              
              ELSE
                /* if gear id is missing, write error to error log */
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored - it is not used for QC, Field Measurements, or Habitat Assessment.');
                   
                
                END IF;
                v_gear_org_id := NULL;
                v_gear_is_nbr := NULL;
              END IF;
            
            ELSIF v_item_name = 'Gear Configuration ID'
                  AND v_new_activity = 1 THEN
              -- jah 2-24-04 Remove trailing space but do NOT convert to uppercase
              v_item := rtrim(v_item);
            
              IF v_item IS NOT NULL
                 AND v_trip_qc != 1
                 AND v_habitat != 1
                 AND v_type != 'Field Msr/Obs' THEN
                /* kms - assign gear to temporary key that will limit the valid gear configurations
                to those that are of this gear type */
                v_temp_key.is_number := v_gear_is_nbr;
                v_temp_key.org_id := v_gear_org_id;
                /* get the gear configuration is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_temp_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
              
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_gearcfg_is_nbr := v_storet_key.is_number;
                  v_gearcfg_org_id := v_storet_key.org_id;
                ELSE
                  v_gearcfg_is_nbr := NULL;
                  v_gearcfg_org_id := NULL;
                  v_status := 'N';
                END IF;
              
              ELSE
              
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' is not used for QC, Field Measurements, or Habitat Assessment');
                   
                
                END IF;
                v_gearcfg_is_nbr := NULL;
                v_gearcfg_org_id := NULL;
              END IF;
            
              /* kms - null out temporary key */
              v_temp_key.is_number := NULL;
              v_temp_key.org_id := NULL;
            
            ELSIF (v_item_name = 'Gear Deployment Comments' OR
                  v_item_name = 'Sample Collection Procedure Gear Deployment Comments')
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_trip_qc != 1
                     AND v_habitat != 1
                     AND v_type != 'Field Msr/Obs' THEN
                    IF length(v_item) > 1999 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is longer than 1999 characters and will be truncated.');
                       
                    
                      v_item := substr(v_item, 1, 1999);
                    END IF;
                    v_gear_deployment := v_item;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' is not used for QC, Field Measurements, or Habitat Assessment');
                     
                  
                    v_gear_deployment := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_gear_deployment := NULL;
                END IF;
              ELSE
                v_gear_deployment := NULL;
              END IF;
            
            ELSIF v_item_name = 'Sample Preservation Transport and Storage ID'
                  OR v_item_name = 'Sample Transport and Storage ID'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                v_item := rtrim(v_item);
              
                   IF v_habitat != 1 --09/21/2005 -RWK removed 'IF v_trip_qc != 1'
                   AND v_type != 'Field Msr/Obs' THEN
                
                  /* get the sample preservation proc is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_sample_pres_is_nbr := v_storet_key.is_number;
                    v_sample_pres_org_id := v_storet_key.org_id;
                  ELSE
                    v_sample_pres_is_nbr := NULL;
                    v_sample_pres_org_id := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' is not used for Field Measurements, or Habitat Assessment');
                   
                
                  v_sample_pres_is_nbr := NULL;
                  v_sample_pres_org_id := NULL;
                END IF;
              ELSE
                v_sample_pres_is_nbr := NULL;
                v_sample_pres_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Sample Transport and Storage Comments'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                     IF v_habitat != 1 --09/21/2005 --RWK Removed 'IF v_trip_qc != 1'
                     AND v_type != 'Field Msr/Obs' THEN
                    IF length(v_item) > 1999 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is longer than 1999 characters and will be truncated.');
                       
                    
                      v_item := substr(v_item, 1, 1999);
                    END IF;
                    v_transport_storage := v_item;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' is not used for QC, Field Measurements, or Habitat Assessment');
                     
                  
                    v_transport_storage := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_transport_storage := NULL;
                END IF;
              ELSE
                v_transport_storage := NULL;
              END IF;
            
            ELSIF (v_item_name = 'QC Indicator' OR
                  v_item_name = 'Activity QC Indicator')
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_qc_indicator := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_qc_indicator := 'N';
                  v_status := 'N';
                END IF;
              ELSE
                v_qc_indicator := NULL;
              END IF;
            
            ELSIF (v_item_name = 'Sample Matrix' OR
                  v_item_name = 'Activity Sample Matrix')
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_storet_key := validate_sample_matrix(v_item, p_imp_cfg_seq, v_activity_seq, v_import_log_seq, v_col_seq, v_line_cnt);
                IF v_storet_key.is_number IS NULL THEN
                  v_tsrmatrx_is_number := NULL;
                  v_tsrmatrx_org_id := NULL;
                  v_status := 'N';
                ELSE
                  v_tsrmatrx_is_number := v_storet_key.is_number;
                  v_tsrmatrx_org_id := v_storet_key.org_id;
                END IF;
              ELSE
                v_tsrmatrx_is_number := NULL;
                v_tsrmatrx_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Chain of Custody ID'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF length(v_item) > 30 THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' is longer than 30 characters and will be truncated.');
                     
                  
                    v_item := substr(v_item, 1, 30);
                  END IF;
                  v_chain_of_custody := v_item;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_chain_of_custody := NULL;
                END IF;
              ELSE
                v_chain_of_custody := NULL;
              END IF;
            
            ELSIF v_item_name = 'Field Set ID'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := upper(rtrim(v_item)); -- 06/13/03 SLP - Set to uppercase and remove trailing spaces.
                  IF length(v_item) > 10 THEN
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        'Field Set ID, ' ||
                                                         v_item ||
                                                         ', is longer than 10 characters and will be truncated');
                     
                  
                    v_item := instr(v_item, 1, 10);
                  END IF;
                  IF v_trip_qc != 1 THEN
                    v_field_set_nbr := v_item;
                  ELSE
                    v_field_set_nbr := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_field_set_nbr := NULL;
                END IF;
              ELSE
                v_field_set_nbr := NULL;
              END IF;
            
            ELSIF v_item_name = 'Field Set Name'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_trip_qc != 1 THEN
                    v_field_set_name := v_item;
                  ELSE
                    v_field_set_name := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_field_set_name := NULL;
                END IF;
              ELSE
                v_field_set_name := NULL;
              END IF;
            
            ELSIF v_item_name = 'Characteristic Detection Condition'
                  OR v_item_name = 'Detection Condition' THEN
              IF v_item IS NOT NULL
                 --AND v_trip_qc != 1 --09/21/05 Detection Condition is valid for Trip QC
                 AND v_habitat != 1 THEN
              
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_detection := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_detection := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_thermo := NULL;
              END IF;
            
              ----------------------
              -- Characteristic Name
              ----------------------
            ELSIF v_item_name = 'Characteristic Name' THEN
              --If they have already imported a line with a Char Group the whole thing bombs. Return to Form...
              v_char_names := TRUE;
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); --09/23/05 -RWK -Remove trailing spaces
                IF v_activity_cat = 'Portable Data Logger' THEN
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NOT NULL
                     AND v_storet_key.org_id IS NOT NULL THEN
                    v_success := sim_import_result_checks.check_data_logger_item(v_dummy_char, v_storet_key.is_number, v_line_cnt, v_storet_key.org_id, v_activity_seq, v_import_log_seq);
                    IF v_success != 1 THEN
                      v_status := 'N';
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                
                  -- if bio result type is Multi Taxon PC then char has to be of type TAXA. This is checked
                  -- when validating Subject Taxon, so validate char as if it were Subject Taxon
                ELSIF v_brg_type = 'Multi-Taxon Population Census' THEN
                  v_storet_key := sim_import_result_extras.validate_storet_item('Subject Taxon', v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                  IF v_storet_key.is_number IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        'Characteristic must be of TAXA type for Multi-Taxon Population Census.');
                     
                  
                    v_status := 'N';
                  END IF;
                
                ELSIF v_brg_type = 'Single Taxon Frequency Classes' THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Characteristic is not used for Single Taxon Frequency Classes. Use Class Descriptors instead.');
                   
                
                  v_storet_key.is_number := NULL;
                  v_storet_key.org_id := NULL;
                
                ELSE
                
                  /* get the characteristic is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                END IF;
              
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_char_is_nbr := v_storet_key.is_number;
                  v_char_org_id := v_storet_key.org_id;
                
                  /** Assign Result type value to global Package variable **/
                  v_char_result_type := get_char_result_type(v_char_is_nbr);
                  /* Further check of char type */
                  IF v_brg_type = 'Single Taxon Individuals' THEN
                    v_success := sim_import_result_checks.check_individual_type(v_char_is_nbr, p_org_id, v_activity_seq, v_line_cnt, p_import_log_seq);
                    IF v_success != 1 THEN
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          'Characteristic is of wrong type for Single Taxon Individuals.');
                       
                    
                    END IF;
                  END IF;
                
                ELSE
                
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                
                END IF;
              
              END IF;
            
              -----------------------------------
              -- Result Value Units
              -- OR
              -- Characteristic Result Value Unit
              -----------------------------------
            ELSIF v_item_name = 'Result Value Units'
                  OR v_item_name = 'Characteristic Result Value Unit' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Result Value Units ignored. Not used for Single Taxon Frequency Classes.');
                   
                
                ELSE
                  /* get the unit is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NULL THEN
                    /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                    v_uom_is_nbr := NULL;
                    v_uom_org_id := NULL;
                    v_status := 'N';
                  ELSE
                    v_uom_is_nbr := v_storet_key.is_number;
                    v_uom_org_id := v_storet_key.org_id;
                    -- In STORET V2 any char can have any UOM, so we no longer need to check the type...
                  END IF;
                END IF;
              ELSE
                IF nvl(v_brg_type, 'x') != 'Single Taxon Frequency Classes' THEN
                  v_uom_is_nbr := NULL;
                  v_uom_org_id := NULL;
                END IF;
              END IF;
            
            ELSIF v_item_name = 'Sample Fraction'
                  OR v_item_name = 'Characteristic Sample Fraction' THEN
              /* kms - validate fraction only if loaded from txt, not the analysis code. */
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_fraction := sim_import_result_extras.validate_fraction(v_item, p_imp_cfg_seq, v_col_seq, v_char_is_nbr, v_activity_seq, v_import_log_seq, v_line_cnt, v_storet_key);
              
                IF v_fraction IS NULL THEN
                  v_status := 'N';
                  v_fraction_is_nbr := NULL;
                  v_fraction_org_id := NULL;
                ELSIF v_fraction = 'NR' THEN
                  --not required
                  v_fraction_is_nbr := NULL;
                  v_fraction_org_id := NULL;
                  v_fraction := NULL;
                ELSE
                  v_fraction_is_nbr := v_storet_key.is_number;
                  v_fraction_org_id := v_storet_key.org_id;
                END IF;
              
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_fraction := NULL;
                v_fraction_is_nbr := NULL;
                v_fraction_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Value Type'
                  OR v_item_name = 'Characteristic Value Type' THEN
              -- 09/15/05 -RWK -The check for a value type in a habitat assessment needs
              -- to be checked before the Group and Row value is fetched from STORET.
              IF v_item IS NOT NULL
                AND v_habitat = 1 THEN
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                    v_item_name ||
                                                    ' has been provied and is not allowed for Habitat Assessment.');
                 
              
                v_value_type := NULL;
              END IF;
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NULL THEN
                  v_status := 'N';
                  v_value_type := NULL;
                ELSE
                  v_value_type := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                END IF;
              ELSE
                --              v_value_type := NULL;
                v_value_type := 'Actual';
              END IF;
            
            ELSIF v_item_name = 'Statistic Type'
                  OR v_item_name = 'Characteristic Result Statistic Type' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_stat_type := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  
                ELSE
                  v_stat_type := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_stat_type := NULL;
              END IF;
            
            ELSIF v_item_name = 'Weight Basis'
                  OR v_item_name = 'Characteristic Result Weight Basis' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_weight_basis := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_weight_basis := NULL;
                  v_status := 'N';
                END IF;
              
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_weight_basis := NULL;
              END IF;
            
            ELSIF v_item_name = 'Duration Basis'
                  OR v_item_name = 'Characteristic Result Duration Basis' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_duration_basis := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  
                ELSE
                  v_duration_basis := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_duration_basis := NULL;
              END IF;
            
            ELSIF v_item_name = 'Temperature Basis'
                  OR v_item_name = 'Characteristic Result Temperature Basis' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_temp_basis := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                 
                ELSE
                  v_temp_basis := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_temp_basis := NULL;
              END IF;
            
            ELSIF v_item_name = 'Result Value'
                  OR v_item_name = 'Characteristic Result Value' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                /*SF - 10/12/01 If it's a PV type char we'll check the translation later.
                If we do it now, and it's empty we get a null value in v_result. */
                IF v_char_result_type IN ('PV  ', 'HA  ') THEN
                  v_result := v_item;
                ELSE
                  v_result := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_result IS NULL THEN
                    v_status := 'N'; --v_result translated to null - set status to 'N'
                  END IF;
                END IF;
              
                IF v_char_result_type NOT IN ('PV  ', 'HA  ', 'TEXT')
                   AND v_result NOT IN ('*Non-detect', '*Present <QL', '*Present >QL',
                    '*Not Reported', '*Present') THEN
                  /* pwh (2/18/04) - check result text to make sure it is a number, write error if the value
                  is too large, too precise, or not a number */
                  v_result := get_value_msr(v_result, v_result_ok, v_activity_seq, v_line_cnt, v_import_log_seq);
                
                  IF v_result_ok = FALSE THEN
                    v_status := 'N';
                  END IF;
                
                END IF;
              
                IF v_brg_type != 'Single Taxon Frequency Classes' THEN
                  v_result_seq_num := v_result_seq_num + 1;
                END IF;
              ELSE
                IF nvl(v_brg_type, '1') != 'Single Taxon Frequency Classes' THEN
                  /* Single Taxon Frequency Classes do not have result values. */
                  IF nvl(v_detection, 'Detected and Quantified') =
                     'Detected and Quantified' THEN
                    /* If detection condition is not passed in or is set to Detected and Quantified, a result value is required. */
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        'Result Value has not been provided. The activity will be loaded without this result.');
                  END IF;
                END IF;
              END IF;
            
            ELSIF v_item_name = 'Laboratory ID' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                -- jah 2-24-04 Remove trailing space but do NOT convert to uppercase
                v_item := rtrim(v_item);
              
                /* get the lab is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
              
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_tsrlab_is_nbr := v_storet_key.is_number;
                  v_tsrlab_org_id := v_storet_key.org_id;
                ELSE
                  v_tsrlab_is_nbr := NULL;
                  v_tsrlab_org_id := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_tsrlab_is_nbr := NULL;
                v_tsrlab_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Analysis Date'
                  OR v_item_name = 'Laboratory Analysis Date' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL
                   AND v_habitat != 1 THEN
                  v_analysis_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq, v_sid_seq);
                  IF v_analysis_date IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        'Analysis Date format not recognized.');
                     
                  
                  END IF;
                ELSE
                  IF v_item IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                    v_analysis_date := NULL;
                  END IF;
                END IF;
              ELSE
                v_analysis_date := NULL;
              END IF;
            
            ELSIF v_item_name = 'Analysis Time'
                  OR v_item_name = 'Laboratory Analysis Time' THEN
              IF v_analysis_date IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                IF v_item IS NOT NULL THEN
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF v_habitat != 1 THEN
                      v_analysis_time := get_time(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq);
                      IF v_analysis_time IS NULL THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                            'Analysis Time format not recognized.');
                         
                      
                        v_status := 'N';
                      END IF;
                    ELSE
                      
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' has been provided and is not allowed for Habitat Assessment.');
                       
                    v_analysis_time := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                    v_analysis_time := NULL;
                  END IF;
                ELSE
                  v_analysis_time := NULL;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored - it is not allowed unless Analysis Date is provided.');
                   
                
                END IF;
                v_analysis_time := NULL;
              END IF;
            
            ELSIF v_item_name = 'Analysis Time Zone'
                  OR v_item_name = 'Laboratory Analysis Time Zone' THEN
              IF v_analysis_time IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_item IS NOT NULL
                   AND v_habitat != 1 THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NULL THEN
                    v_status := 'N';
                  ELSE
                    v_analysis_time_zone := sim_import_result_extras.get_allowable_value(v_dummy_seq, v_import_log_seq);
                  END IF;
                ELSIF v_habitat = 1 THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                  v_analysis_time_zone := NULL;
                ELSE
                  v_analysis_time_zone := NULL;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored - it is not valid unless Analysis Time is provided.');
                   
                
                END IF;
                v_analysis_time_zone := NULL;
              END IF;
            
            ELSIF v_item_name = 'Lab Remark Codes' THEN
              IF v_item IS NOT NULL THEN
                v_no_remark_codes := 0;
              
                /* can have multiple Remark Codes separated by '\' - get number of Remark Codes to expect */
                /* kms - translate function removes all occurences of the delimiter.  By subtracting
                the length of what it returns from the length of the original, we find the number
                of delimiters that were in the original.  We then add 1 since we are counting the
                number of remark Codes, not the number of delimiters. */
                v_no_remark_codes := length(v_item) -
                                     length(translate(v_item, 'a\', 'a')) + 1;
              
                /* separate item into Remark Codes using parse procedure if more than 1 Code */
                IF v_no_remark_codes > 1 THEN
                  BEGIN
                    sim_import_pkg.parse_line_table(v_item, v_remark_codes_table, v_no_remark_codes, '\', v_bad_parse, v_import_log_seq);
                    /*  Now call procedure to remove any duplicate codes */
                    remove_lab_remark_dups(v_remark_codes_table, v_no_remark_codes);
                  EXCEPTION
                    WHEN no_data_found THEN
                      NULL;
                  END;
                ELSE
                  v_remark_codes_table(1) := v_item;
                END IF;
              
                /* loop through Remark Codes table and add each remark in column to the activity */
                FOR v_number IN 1 .. v_no_remark_codes LOOP
                
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_remark_codes_table(v_number), p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NULL THEN
                    v_remark_is_table(v_number) := NULL;
                    v_remark_org_table(v_number) := NULL;
                    v_status := 'N';
                  ELSE
                    v_remark_is_table(v_number) := v_storet_key.is_number;
                    v_remark_org_table(v_number) := v_storet_key.org_id;
                  END IF;
                
                END LOOP;
              
              ELSE
                v_no_remark_codes := 0;
                v_tsrlbrmk_is_nbr := NULL;
                v_tsrlbrmk_org_id := NULL;
              END IF;
            
              ---------------------
              -- Quantification Low
              ---------------------
            ELSIF v_item_name = 'Quantification Low' THEN
            
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_habitat != 1 THEN
                    IF length(v_item) > 12 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is too long. It can be a maximum lenght of 12 digits.');
                       
                    
                      v_lowq_limit := NULL;
                      v_status := 'N';
                    ELSE
                      v_lowq_limit := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                    v_lowq_limit := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_lowq_limit := NULL;
                END IF;
              ELSE
                v_lowq_limit := NULL;
              END IF;
            
              ----------------------
              -- Quantification High
              ----------------------
            ELSIF v_item_name = 'Quantification High' THEN
            
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_habitat != 1 THEN
                    IF length(v_item) > 12 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                         ' is too long. It can be a maximum lenght of 12 digits.');
                       
                    
                      v_upq_limit := NULL;
                      v_status := 'N';
                    ELSE
                      v_upq_limit := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                    v_status := 'N';
                    v_upq_limit := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_upq_limit := NULL;
                END IF;
              ELSE
                v_upq_limit := NULL;
              END IF;
            
              ------------------------------------
              -- Detection Limit
              -- Laboratory Result Detection Limit
              ------------------------------------
            ELSIF v_item_name = 'Detection Limit'
                  OR v_item_name = 'Laboratory Result Detection Limit' THEN
            
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_habitat != 1 THEN
                    IF length(v_item) > 12 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                          ' is too long. It can be a maximum lenght of 12 digits.');
                       
                    
                      v_detect_limit := NULL;
                      v_status := 'N';
                    ELSE
                      v_detect_limit := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                    v_detect_limit := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_detect_limit := NULL;
                END IF;
              ELSE
                v_detect_limit := NULL;
              END IF;
            
              -----------------------------------------
              -- Detection Limit Unit
              -- Laboratory Result Detection Limit Unit
              -----------------------------------------
            ELSIF v_item_name = 'Detection Limit Unit'
                  OR v_item_name = 'Laboratory Result Detection Limit Unit' THEN
            
              IF v_item IS NOT NULL
                 AND (v_detect_limit IS NOT NULL OR v_upq_limit IS NOT NULL OR
                 v_lowq_limit IS NOT NULL)
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                /* get the detection unit is number - check translations and STORET tables */
                IF nvl(v_char_result_type, 'x') <> 'TEXT' THEN
                  -- IF added by DCL to avoid unit of measure error if result is a text type
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                END IF;
              
                IF v_storet_key.is_number IS NULL THEN
                  /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                  v_det_uom_is_nbr := NULL;
                  v_det_uom_org_id := NULL;
                  v_status := 'N';
                
                ELSE
                  v_det_uom_is_nbr := v_storet_key.is_number;
                  v_det_uom_org_id := v_storet_key.org_id;
                  /* kms - need to make sure unit types match */
                  --v_success := Sim_import_result_checks.check_uom_type(v_det_uom_is_nbr,v_char_is_nbr,v_line_cnt,v_import_log_seq);
                
                  --IF v_success = 2 THEN
                  --v_status := 'N';
                  --END IF;
                END IF;
              ELSIF v_item IS NOT NULL THEN
                IF v_detect_limit IS NULL
                   AND v_upq_limit IS NULL
                   AND v_lowq_limit IS NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Detection limit unit of measure has been provided and is not used. It has been removed.');
                   
                
                  v_det_uom_is_nbr := NULL;
                  v_det_uom_org_id := NULL;
                END IF;
              ELSE
                v_det_uom_is_nbr := NULL;
                v_det_uom_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Detection Limit Comment' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_item IS NOT NULL
                     AND v_habitat != 1 THEN
                    v_det_comment := substr(v_item, 1, 254);
                  ELSE
                    IF v_item IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                    END IF;
                    v_det_comment := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                END IF;
              ELSE
                v_det_comment := NULL;
              END IF;
            
            ELSIF v_item_name = 'Precision' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_item IS NOT NULL
                     AND v_habitat != 1 THEN
                    v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                    v_precision := v_item;
                  ELSE
                    IF v_item IS NOT NULL THEN
                      
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                    END IF;
                    v_precision := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_precision := NULL;
                END IF;
              ELSE
                v_precision := NULL;
              END IF;
            
            ELSIF v_item_name = 'Confidence Level' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_conf_level := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_conf_level := NULL;
                  v_status := 'N';
                END IF;
                IF v_precision IS NULL THEN
                  v_status := 'N';
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Precision is required when Confidence level is given.');
                   
                
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                       ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_conf_level := NULL;
              END IF;
            
            ELSIF v_item_name = 'Bias' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_habitat != 1 THEN
                    IF length(v_item) > 12 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                          v_item_name ||
                                                         ' is longer than 12 characters.');
                       
                    
                      v_status := 'N';
                    ELSE
                      v_bias := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' ignored. It is not used for Habitat Assessment.');
                     
                  
                    v_bias := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_bias := NULL;
                END IF;
              ELSE
                v_bias := NULL;
              END IF;
            
            ELSIF v_item_name = 'CL Corrected for Bias' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_habitat != 1 THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_cl_corrected := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  ELSE
                    v_cl_corrected := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                     ' ignored. It is not used for Habitat Assessment.');
                   
                
                  v_cl_corrected := NULL;
                END IF;
              ELSE
                v_cl_corrected := NULL;
              END IF;
            
            ELSIF v_item_name = 'Result Comment'
                  OR v_item_name = 'Characteristic Result Comment' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF nvl(v_char_result_type, 'x') = 'TEXT'
                   AND v_item IS NOT NULL THEN
                  /** dcl 09/13/01 if a result comment exists for a text type result **/
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       ' Result Comments are not used with a Text-type result - comment ignored');
                   
                
                  v_result_cmt := NULL;
                ELSIF v_item IS NOT NULL THEN
                  v_result_cmt := v_item;
                  IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                    v_result_cmt := NULL;
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' has been provided and is not allowed on Single Taxon Frequency Classes.');
                     
                  
                  ELSIF v_habitat = 1 THEN
                    v_result_cmt := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                         ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                  END IF;
                ELSIF v_item IS NULL THEN
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_result_cmt := NULL;
                END IF;
              ELSE
                v_result_cmt := NULL;
              END IF;
            
            ELSIF v_item_name = 'Result Document/Graphic' THEN
              IF v_item IS NOT NULL THEN
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF sim_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                    v_result_document := v_item;
                  ELSE
                    v_status := 'N'; --Invalid document.  Set to 'N'.
                    v_result_document := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; --Translated to NULL.  Set to 'N'.
                  v_result_document := NULL;
                END IF;
              ELSE
                v_result_document := NULL;
              END IF;
            
              /* 06/09/03 slp - If the any result other than 'Single Taxon', validate the result status.
              - All status for the 'Single Taxon' group must be the same in STORET.  Therefore, if the record
                - is the first of the 'Single Taxon' group, validate the result status.
                - If the record is within the 'Single Taxon' group, set the status to NULL. */
            ELSIF v_item_name = 'Result Status' THEN
              IF v_new_group = 1
                 OR v_brg_type != 'Single Taxon Frequency Classes' THEN
                -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                v_item := rtrim(v_item);
              
                IF v_item IS NOT NULL THEN  --09/14/05 -rwk Removed 'AND v_habitat != 1'
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_result_status := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                    -- jah 2-24-04 Convert to uppercase because STORET GUI converts Result Status to uppercase
                    v_result_status := upper(v_result_status);
                  ELSE
                    v_result_status := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_result_status := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Result Status has been left blank and will be set to ''Final.''');
                   
                
                END IF;
              ELSE
                v_result_status := NULL;
              END IF;
            
              -- This code is for the Portable Data Logger Data Lines
            ELSIF v_item_name = 'PDL Line Number' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_activity_cat != 'Portable Data Logger' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Activity Category must be ''Portable Data Logger'' if there is a Data Line Number');
                     
                  
                    v_status := 'N';
                    v_data_line_number := NULL;
                  ELSE
                    v_data_line_number := v_item;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_data_line_number := NULL;
                END IF;
              ELSE
                v_data_line_number := NULL;
              END IF;
            
              -- This code is for the Portable Data Logger Data Line Name
            ELSIF v_item_name = 'PDL Line Name' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_activity_cat != 'Portable Data Logger' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Activity Category must be ''Portable Data Logger'' if there is a Data Line Name');
                     
                  
                    v_data_line_name := NULL;
                    v_status := 'N';
                  ELSIF v_data_line_number IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'There must be a Data Line Number if there is a Data Line Name');
                     
                  
                    v_data_line_name := NULL;
                    v_status := 'N';
                  ELSE
                    v_data_line_name := v_item;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_data_line_name := NULL;
                END IF;
              ELSE
                v_data_line_name := NULL;
              END IF;
            
            ELSIF v_item_name = 'Laboratory Batch ID' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL
                   AND v_habitat != 1 THEN
                  v_item := upper(rtrim(v_item)); -- 06/13/03 SLP - Set to uppercase and removed trailing spaces.
                  IF length(v_item) > 12 THEN
                    v_item := substr(v_item, 1, 12);
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                          ' is longer that 12 characters and will be truncated.');
                     
                  
                  END IF;
                  v_batch_id := v_item;
                ELSE
                  IF v_item IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' has been provided and is not allowed for Habitat Assessment.');
                     
                  
                    v_batch_id := NULL;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                    v_batch_id := NULL;
                  END IF;
                END IF;
              ELSE
                v_batch_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Laboratory Certified'
                  OR v_item_name = 'Laboratory Certification Indicator' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NULL THEN
                  v_status := 'N';
                  v_lab_cert := NULL;
                ELSE
                  v_lab_cert := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                
                END IF;
                v_lab_cert := NULL;
              END IF;
            
            ELSIF v_item_name = 'Project ID'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- jah 2-24-04 Remove trailing spaces but do not convert to uppercase
                v_item := rtrim(v_item);
                v_no_projects := 0;
              
                /* can have multiple projects separated by '\' - get number of projects to expect */
                v_no_projects := length(v_item) - length(translate(v_item, 'a\', 'a')) + 1;
              
                /* separate item into projects using parse procedure if more than 1 project */
                IF v_no_projects > 1 THEN
                  BEGIN
                    sim_import_pkg.parse_line_table(v_item, v_project_table, v_no_projects, '\', v_bad_parse, v_import_log_seq);
                  EXCEPTION
                    WHEN no_data_found THEN
                      NULL;
                  END;
                ELSE
                  v_project_table(1) := v_item;
                END IF;
              
                /* loop through projects table and add each project in column to the activity */
                FOR v_number IN 1 .. v_no_projects LOOP
                  /* Check that Project ID is not longer than 8 characters -- sfp 061101 */
                  /* kms (6/21/01) - changed to check length of item in project table rather than
                  of v_item. */
                  IF length(v_project_table(v_number)) > 8 THEN
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        'Project ID (' ||
                                                        v_project_table(v_number) ||
                                                        ') must not contain more than 8 characters.');
                     
                  
                  END IF;
                
                  /* get the project is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_project_table(v_number), p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_proj_is_table(v_number) := v_storet_key.is_number;
                    v_proj_org_table(v_number) := v_storet_key.org_id;
                  ELSE
                    v_status := 'N';
                  END IF;
                
                END LOOP;
              
              ELSE
                /* if no project id, write error to error log and set status to 'N' */
                v_proj_is_nbr := NULL;
                v_proj_org_id := NULL;
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Project ID is required and has not been provided.');
                 
              
              END IF;
            
            ELSIF v_item_name = 'Characteristic Group ID' THEN
              v_groups := TRUE;
            
              IF v_item IS NOT NULL THEN
                -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                v_item := rtrim(v_item);
              
                /* get the char group is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
              
                IF v_storet_key.is_number IS NOT NULL THEN
                  -- jah 2-24-04 Convert to uppercase because STORET GUI converts Characteristic Group ID to uppercase
                  -- (Note: Always convert to uppercase AFTER translation!)
                  v_item := upper(v_item);
                
                  /* kms - make sure group is valid for this activity.  Check type and medium. */
                  v_success := sim_import_result_checks.check_group_type_med(v_storet_key, v_type, v_medium, v_intent, v_brg_type, v_community_is_nbr, v_community_org_id, v_activity_seq, v_line_cnt, v_import_log_seq, v_ha_group);
                
                  IF v_success != 1 THEN
                    /* kms - group is not valid for this activity or an error occured when checking */
                    v_tsrchgrp_is_nbr := NULL;
                    v_tsrchgrp_org_id := NULL;
                    v_status := 'N';
                  ELSE
                    /* kms - group is valid for this activity */
                    v_tsrchgrp_is_nbr := v_storet_key.is_number;
                    v_tsrchgrp_org_id := v_storet_key.org_id;
                    /* Check to make sure Category is Routine Habitat Assessment */
                    v_group := NULL;
                    IF v_ha_group IN (1, 2) THEN
                      v_ha_group_id := v_dummy_char;
                      IF v_activity_cat != 'Routine Habitat Assessment'
                         AND v_activity_cat != 'Replicate Habitat Assessment' THEN
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                            'Habitat Assessment Characteristics must have Category of either Routine or Replicate Habitat Assessment.');
                         
                      
                      END IF;
                    ELSE
                      v_ha_group_id := NULL;
                    END IF;
                  END IF;
                
                  /* Make sure two different groups are not used on an HA activity */
                  IF v_new_activity != 1 THEN
                    OPEN c_char_group(v_activity_seq);
                    FETCH c_char_group
                      INTO v_group;
                    IF nvl(v_group, 'x') != nvl(v_ha_group_id, 'x') THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          'Habitat Assessment Activities can only use one Characteristic Group.');
                       
                    
                      v_status := 'N';
                    END IF;
                    CLOSE c_char_group;
                  ELSIF v_ha_group IN (1, 2) THEN
                    v_group := v_dummy_char;
                  END IF;
                
                ELSE
                  v_tsrchgrp_is_nbr := NULL;
                  v_tsrchgrp_org_id := NULL;
                  v_status := 'N'; -- v_item translated to a null - set status to 'N'
                END IF;
              
              ELSE
                --v_status := 'N';
                v_tsrchgrp_is_nbr := NULL;
                v_tsrchgrp_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Characteristic Row ID' THEN
              v_item := UPPER(v_item); --09/27/05 --RWK -- STORET forces Upper Case therefore SIM must do the same
              IF v_item IS NOT NULL
                 AND v_tsrchgrp_is_nbr IS NOT NULL THEN
                /* sef - get char defaults from different places depending ow whether they are Habitat Assessment */
                /* kms - set temp key to group pk values in order to send to validation routine */
                v_temp_key.is_number := v_tsrchgrp_is_nbr;
                v_temp_key.org_id := v_tsrchgrp_org_id;
              
                IF v_ha_group = 1 THEN
                  /* get the HA row is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item('Habitat Assessment Row ID', v_item, p_imp_cfg_seq, v_col_seq, v_temp_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_tsrhcsc_is_nbr := v_storet_key.is_number;
                    v_tsrhcsc_org_id := v_storet_key.org_id;
                  
                    v_ha_char_name := sim_import_result_extras.get_ha_default(v_storet_key, v_activity_seq, v_import_log_seq, v_line_cnt);
                    IF v_ha_char_name IS NULL THEN
                      v_status := 'N';
                    ELSE
                      v_char_result_type := 'HA  ';
                    END IF;
                  ELSE
                    v_tsrhcsc_is_nbr := NULL;
                    v_tsrhcsc_org_id := NULL;
                    v_status := 'N';
                  END IF;
                
                  /* assign defaults to result variables */
                  v_tsrchdef_is_nbr := NULL;
                  v_tsrchdef_org_id := NULL;
                  v_char_is_nbr := NULL;
                  v_char_org_id := NULL;
                  v_uom_is_nbr := NULL;
                  v_uom_org_id := NULL;
                  v_anlpr_is_nbr := NULL;
                  v_anlpr_org_id := NULL;
                  v_labprep_is_nbr := NULL;
                  v_labprep_org_id := NULL;
                  v_fraction := NULL;
                  v_fraction_is_nbr := NULL;
                  v_fraction_org_id := NULL;
                  v_volt_is_number := NULL;
                  v_volt_org_id := NULL;
                  v_habit_is_number := NULL;
                  v_habit_org_id := NULL;
                  v_value_type := NULL;
                  v_stat_type := NULL;
                  v_duration_basis := NULL;
                  v_temp_basis := NULL;
                  v_weight_basis := NULL;
                  v_char_species_number := NULL;
                  v_func_feed_grp := NULL;
                  v_trophic_level := NULL;
                  v_taxon_pollution := NULL;
                  v_particle_size_basis := NULL; -- jah 2-23-04 Added Particle Size Basis
                
                ELSE
                
                  /* get the char row is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_temp_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_tsrchdef_is_nbr := v_storet_key.is_number;
                    v_tsrchdef_org_id := v_storet_key.org_id;
                  
                    /* populate result char fields based on what's in the default table */
                    v_dummy_seq := NULL;
                    v_success := get_char_defaults(v_dummy_seq, v_storet_key, p_imp_cfg_seq, v_col_seq, v_activity_seq);
                  
                    IF v_success IS NULL THEN
                      v_status := 'N';
                      -- ELSE
                      --   v_char_result_type := 'HA  ';
                    END IF;
                  ELSE
                    v_tsrchdef_is_nbr := NULL;
                    v_tsrchdef_org_id := NULL;
                    v_status := 'N';
                  END IF;
                
                  /* assign defaults to result variables */
                  v_char_is_nbr := v_acode_defs_rec.char_pk.is_number;
                  v_char_org_id := v_acode_defs_rec.char_pk.org_id;
                  v_uom_is_nbr := v_acode_defs_rec.uom_pk.is_number;
                  v_uom_org_id := v_acode_defs_rec.uom_pk.org_id;
                  v_anlpr_is_nbr := v_acode_defs_rec.anlpr_pk.is_number;
                  v_anlpr_org_id := v_acode_defs_rec.anlpr_pk.org_id;
                  v_labprep_is_nbr := v_acode_defs_rec.labprep_pk.is_number;
                  v_labprep_org_id := v_acode_defs_rec.labprep_pk.org_id;
                  v_fraction_is_nbr := v_acode_defs_rec.fraction_is;
                  v_fraction_org_id := 'ZZZZZZZZ';
                  v_volt_is_number := v_acode_defs_rec.voltinism_is;
                  v_volt_org_id := 'ZZZZZZZZ';
                  v_habit_is_number := v_acode_defs_rec.habit_is;
                  v_habit_org_id := 'ZZZZZZZZ';
                  v_value_type := v_acode_defs_rec.value_type;
                  v_stat_type := v_acode_defs_rec.stat_type;
                  v_duration_basis := v_acode_defs_rec.dur_basis;
                  v_temp_basis := v_acode_defs_rec.temp_basis;
                  v_weight_basis := v_acode_defs_rec.weight_basis;
                  v_char_species_number := v_acode_defs_rec.species;
                  v_func_feed_grp := v_acode_defs_rec.funct_feed_grp;
                  v_trophic_level := v_acode_defs_rec.trophic_level;
                  v_lower_range := v_acode_defs_rec.lower_range_value;
                  v_upper_range := v_acode_defs_rec.upper_range_value;
                
                  -- jah 2-20-04 Added Particle Size Basis
                  -- Does the acode field for particle size basis contain anything?
                  IF (nvl(v_acode_defs_rec.particle_size_basis, ' ') > ' ') THEN
                    -- if yes
                    v_particle_size_basis := v_acode_defs_rec.particle_size_basis;
                  ELSE
                    -- if no
                    -- ensure v_particle_size_basis contains NULL (and not ' ')
                    v_particle_size_basis := NULL;
                  END IF;
                
                  v_taxon_pollution := v_acode_defs_rec.pollution_tol;
                  v_tsrhcsc_is_nbr := NULL;
                  v_tsrhcsc_org_id := NULL;
                
                  /* Old SIM v1.2.2 checks have been removed in v2  */
                  -- For V2 we can assume that sample fraction is valid in a char group
                  -- Check UOM type on Char group - this never happens any more. In v2 any UOM is okay.
                
                END IF;
              
                IF v_brg_type = 'Multi-Taxon Population Census'
                   AND v_char_result_type != 'TAXA' THEN
                  v_status := 'N';
                  v_char_is_nbr := NULL;
                  v_char_org_id := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Characteristic must be of type TAXA for Multi-Taxon Population Census.');
                   
                
                END IF;
              
              ELSIF v_item IS NOT NULL
                    AND v_tsrchgrp_is_nbr IS NULL THEN
                /* kms - importing characteristic row, but there's no characteristic group
                to validate it with.  Invalidate the activity and write error message. */
                v_tsrchdef_is_nbr := NULL;
                v_tsrchdef_org_id := NULL;
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Characteristic Row ID has been included but cannot be validated due to missing or invalid Characteristic Group ID.');
                 
              
              ELSIF v_item IS NULL
                    AND v_tsrchgrp_is_nbr IS NOT NULL THEN
                /* kms - characteristic group has been imported, but no row.  Invalidate activity
                and write error message. */
                v_tsrchdef_is_nbr := NULL;
                v_tsrchdef_org_id := NULL;
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Characteristic Row ID is required when Characteristic Group ID has been given.');
                 
              
              ELSE
                v_tsrchdef_is_nbr := NULL;
                v_tsrchdef_org_id := NULL;
              END IF;
            
              /* The following import fields are relevant only to biological results
              only when v_medium = 'Biological'  */
            ELSIF v_item_name = 'Intent' 
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_medium = 'Biological' THEN
                  --Intent is required where medium = 'Biological'
                  /* get intent - check translations and allowable values tables */
                  IF v_dummy_seq IS NULL THEN
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        'Intent was not found.');
                     
                  
                  ELSE
                    v_intent := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  END IF;
                ELSE
                  v_intent := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Intent ignored. It is only applicable when Medium is Biological.');
                   
                
                END IF;
              ELSE
                v_intent := NULL;
              END IF;
            
            ELSIF v_item_name = 'Subject Taxon' THEN
              -- Subject Taxon is required where intent = Individual or Tissue
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                /*get characteristic is number - check translations and STORET tables */
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                IF v_storet_key.is_number IS NOT NULL THEN
                  v_taxon_is_nbr := v_storet_key.is_number;
                  v_taxon_org_id := v_storet_key.org_id;
                ELSE
                  v_taxon_is_nbr := NULL;
                  v_taxon_org_id := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_taxon_is_nbr := NULL;
                v_taxon_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Subject Taxon Species Number' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_taxon_species_number := sim_import_result_extras.get_allowable_value(v_dummy_seq, v_import_log_seq);
                ELSE
                  v_taxon_species_number := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_taxon_species_number := NULL;
              END IF;
            
            ELSIF v_item_name = 'Characteristic Species Number' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_char_species_number := sim_import_result_extras.get_allowable_value(v_dummy_seq, v_import_log_seq);
                ELSE
                  v_char_species_number := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_char_species_number := NULL;
              END IF;
            
              ----------------------
              -- Parent Sample ID --
              ----------------------
            ELSIF v_item_name = 'Parent Sample ID' THEN
            
              -- do we need to process this field?
              IF v_new_activity = 1
                 AND v_type != 'Field Msr/Obs'
                 AND v_item IS NOT NULL THEN
                --  if yes
              
                -- does this activity need a Parent Sample ID?
                IF (v_activity_cat = 'Created from Sample')
                   OR (v_activity_cat = 'Composite-with Parents') THEN
                  -- if yes
                
                  /*
                  Rules:
                  o  Process Parent Sample ID after validating Activity Category
                  o  SIM no longer validates Intent, Visit, and Subject Taxon for Parent Sample ID
                  o  Parent must exist in STORET (TSRFDACT) or SIM (SIM_FIELD_ACTIVITIES)
                  o  If Parent record does not exist in STORET then Parent record must 
                     occur before child in the import file
                  o  Ignore data in child records for Actual Activity Location, Tow/Net Operation, 
                     and Depth/Stratification
                  
                  Notes:
                  o   The Parent Sample ID field can have multiple values
                      seperated by backward slashes (\).
                  */
                
                  -- Count the backward slashes and add one to determine how many Parent Sample IDs
                  v_no_parent_ids := length(v_item) -
                                     length(translate(v_item, 'a\', 'a')) + 1;
                
                  -- load the Parent Sample IDs into a table
                  BEGIN
                    sim_import_pkg.parse_line_table(v_item, v_parent_id_table, -- table of Parent Sample IDs
                                                     v_no_parent_ids, '\', v_bad_parse, v_import_log_seq);
                  EXCEPTION
                    WHEN no_data_found THEN
                      NULL;
                  END;
                
                  -- loop through Parent Sample IDs table
                  FOR v_number IN 1 .. v_no_parent_ids LOOP
                  
                    -- remove trailing spaces from Parent Sample ID
                    v_parent_id_table(v_number) := rtrim(v_parent_id_table(v_number));
                  
                    -- Check for a translation of the Parent Sample ID in STORET
                    v_storet_key := sim_import_result_extras.check_storet_translation(v_parent_id_table(v_number), p_imp_cfg_seq, v_col_seq, v_trans_exist, v_import_log_seq, v_line_cnt, v_converted_item, v_activity_seq);
                  
                    -- did we find a translation?
                    IF v_storet_key.is_number IS NOT NULL THEN
                      -- if yes
                    
                      -- save the parent information in tables for later processing
                      v_parent_id_table(v_number) := v_converted_item;
                      v_parent_fa_seq_table(v_number) := NULL;
                      v_parent_is_number_table(v_number) := v_storet_key.is_number;
                      v_parent_org_id_table(v_number) := v_storet_key.org_id;
                    
                      -- did it translate?
                    ELSIF v_trans_exist = FALSE THEN
                      -- if no (it did not translate)
                    
                      -- jah 3-9-2004                  
                      -- Convert to upper case (but only after translation!)
                      v_parent_id_table(v_number) := upper(v_parent_id_table(v_number));
                    
                      -- save the parent information in tables for later processing
                      v_error_code := sim_import_result_checks.check_parent(v_parent_id_table(v_number), -- IN: Parent Sample ID
                                                                            v_visit_seq, -- IN: Station Visit
                                                                            v_line_cnt, -- IN:
                                                                            v_import_log_seq, -- IN:
                                                                            v_parent_fa_seq_table(v_number), -- OUT: from SIM_FIELD_ACTIVITIES
                                                                            v_parent_is_number_table(v_number), -- OUT: from TSRFDACT
                                                                            v_parent_org_id_table(v_number) -- OUT: from TSRFDACT
                                                                            );
                    
                      -- did we get an error in the check_parent function?
                      IF (v_error_code IS NOT NULL) THEN
                        -- if yes
                      
                        -- set status flag to reject this import record
                        v_status := 'N';
                      
                        -- note: Function CHECK_PARENT has already displayed the error message
                      
                      END IF;
                    ELSE
                      -- otherwise, it translated to NULL
                    
                      -- Prent migration of this record
                      v_status := 'N';
                    
                      -- note: Function CHECK_STORET_TRANSLATION has already displayed the error message
                    END IF;
                  
                  END LOOP;
                
                ELSE
                  -- if no (this activity does not need a Parent Sample ID)
                
                  -- output warning message
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Parent Sample ID ignored -- Not applicable to "' ||
                                                       v_activity_cat || '".');
                   
                
                END IF;
              
              ELSE
                -- if no (we do not need to process Parent Sample ID for this activity)
              
                -- indicate the Parent Sample ID does not exist for this activity
                v_no_parent_ids := 0;
              
              END IF;
            
              ---------------  
              -- Community --
              ---------------  
            ELSIF v_item_name = 'Community'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                /* get community name - check translations and STORET tables*/
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                IF v_intent = 'Taxon Abundance' THEN
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_community_is_nbr := v_storet_key.is_number;
                    v_community_org_id := v_storet_key.org_id;
                  ELSE
                    v_community_is_nbr := NULL;
                    v_community_org_id := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_community_is_nbr := NULL;
                  v_community_org_id := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Community ignored. It is not used unless intent is Taxon Abundance.');
                   
                
                END IF;
              ELSE
                v_community_is_nbr := NULL;
                v_community_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Bio Part'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                /*get bio part name - check translations and STORET tables*/
                v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                IF v_intent = 'Tissue' THEN
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_biopart_is_nbr := v_storet_key.is_number;
                    v_biopart_org_id := v_storet_key.org_id;
                  ELSE
                    v_biopart_is_nbr := NULL;
                    v_biopart_org_id := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_biopart_is_nbr := NULL;
                  v_biopart_org_id := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Bio Part is not applicable and has been ignored');
                   
                
                END IF;
              ELSE
                v_biopart_is_nbr := NULL;
                v_biopart_org_id := NULL;
              END IF;
            
              -- Following 2 items are distined for TSRSMPLE via SIM_SAMPLE
              -- Total Sample Weight
            ELSIF v_item_name = 'Total Sample Weight'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  BEGIN
                    v_smpl_wt := to_number(v_item, '99999.99');
                  EXCEPTION
                    WHEN OTHERS THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                          'Total Sample Weight is not a valid number.');
                       
                    
                      v_smpl_wt := NULL;
                  END;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_smpl_wt := NULL;
                END IF;
              ELSE
                v_smpl_wt := NULL;
              END IF;
            
              -- Total Sample Weight Units
            ELSIF v_item_name = 'Total Sample Weight Units'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_total_wt_uom := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_total_wt_uom := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_total_wt_uom := NULL;
              END IF;
            
              -- Following 9 items are destined for TSRTOD via SIM_TRAWL_OPS_DET
            
              ------------------
              -- Fished Duration
              ------------------
            
            ELSIF v_item_name = 'Fished Duration'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_check_trawl_gear = 1 THEN
                    BEGIN
                      v_fished_duration := to_number(v_item, '99.99');
                    EXCEPTION
                      WHEN OTHERS THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                            v_item_name ||
                                                            ' is not a valid number.');
                         
                      
                        v_status := 'N';
                        v_fished_duration := NULL;
                    END;
                  ELSE
                    v_fished_duration := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' ignored. It applies only when Procedure has a Horizontal Tow Gear Type.');
                     
                  
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_fished_duration := NULL;
                END IF;
              ELSE
                v_fished_duration := NULL;
              END IF;
            
              ------------------------
              -- Fished Duration Units
              ------------------------
            
            ELSIF v_item_name = 'Fished Duration Units'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_fished_dur_uom := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_fished_dur_uom := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_fished_dur_uom := NULL;
              END IF;
            
            ELSIF v_item_name = 'Distance Fished'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_check_trawl_gear = 1 THEN
                    BEGIN
                      v_distance_fished := to_number(v_item, '99.99');
                    EXCEPTION
                      WHEN OTHERS THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                            v_item_name ||
                                                            ' is not a valid number.');
                         
                      
                        v_status := 'N';
                        v_distance_fished := NULL;
                    END;
                  ELSE
                    v_distance_fished := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' ignored. It applies only when Procedure has a Horizontal Tow Gear Type.');
                     
                  
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_distance_fished := NULL;
                END IF;
              ELSE
                v_distance_fished := NULL;
              END IF;
            
            ELSIF v_item_name = 'Distance Fished Units'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_dummy_seq IS NOT NULL THEN
                  v_dist_fished_uom := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                ELSE
                  v_dist_fished_uom := NULL;
                  v_status := 'N';
                END IF;
              ELSE
                v_dist_fished_uom := NULL;
              END IF;
            
              --Next 7 items destined for TSRBRG via SIM_BIO_RESULTS_GRP
              --Activity ID and Intent must have already been processed before this point
              --An activity may have multiple Bio Results Groups, but only one MTPC
              --A routine is needed to cause processing on only the first occurence of a new group
              --The data elements must occur together in the text file in the specified order
              --The SIM_BIO_RESULTS_GRP record will be created after the column loop and after the
              -- activity is created
            
            ELSIF v_item_name = 'Bio Results Group ID' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_curr_group := upper(rtrim(v_item)); -- 06/13/03 SLP - will use v_curr_group later to get BRGI_BRG_SEQ for BRGI_INDIVIDUAL
                  
                  OPEN c_find_brg_seq(v_curr_group, v_activity_seq);
                  FETCH c_find_brg_seq
                    INTO v_dummy_seq;
                  IF c_find_brg_seq%NOTFOUND THEN
                    -- it is a new group
                    v_new_group := 1; -- set new group flag
                  ELSE
                    -- it is not a new group
                    v_new_group := 0; -- set not new group flag
                    v_brg_seq := v_dummy_seq;
                    
                   
                    --10/12/05 -RWK -Section below was added to handle Biological Results
                    --That are not grouped together in the import file. 
                    OPEN c_get_brg(v_curr_group, v_activity_seq);
                      FETCH c_get_brg
                        INTO v_dummy_brg;
                    
                    v_brg_seq := v_dummy_brg.brg_seq;
                    v_brg_type := v_dummy_brg.brg_type;
                    v_brg_freq_analysis := v_dummy_brg.brg_freq_analysis;
                    v_ttl_nbr_in_grp := v_dummy_brg.brg_summary_grp_count;
                    v_value_type_name := v_dummy_brg.brg_value_type_name;
                    v_brg_text := v_dummy_brg.brg_description_text;
                    v_char_is_nbr :=  v_dummy_brg.brg_tsrchar_is_number;
                    v_char_org_id :=  v_dummy_brg.brg_tsrchar_org_id;
                    v_lifestage_is_nbr :=  v_dummy_brg.brg_tsrcpv_lifestage_is_nbr;
                    v_lifestage_org_id :=  v_dummy_brg.brg_tsrcpv_lifestage_org_id;
                    v_sexname_is_nbr :=  v_dummy_brg.brg_tsrcpv_sex_is_number;
                    v_sexname_org_id :=  v_dummy_brg.brg_tsrcpv_sex_org_id;
                    
                    IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                      OPEN c_get_uom(v_brg_seq);
                        FETCH c_get_uom
                          INTO v_dummy_uom;
                      v_uom_is_nbr := v_dummy_uom.rci_tsruom_is_number;
                      v_uom_org_id := v_dummy_uom.rci_tsruom_org_id;
                      CLOSE c_get_uom;
                    END IF;
                              
                  CLOSE c_get_brg;  
                  END IF;
                  CLOSE c_find_brg_seq;
                  IF v_new_group = 1 THEN
                    IF v_intent = 'Taxon Abundance' THEN
                      IF v_curr_group IS NOT NULL THEN
                        v_brg_id := v_curr_group;
                      ELSE
                        v_status := 'N';
                        v_brg_id := NULL;
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                            'Bio Results Group ID must be given when Intent is Taxon Abundance');
                         
                      
                      END IF;
                    ELSE
                      v_brg_id := NULL;
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                          'Bio Results Group ID ignored. Only applies when Intent is Taxon Abundance');
                       
                    
                    END IF;
                  END IF;
                ELSE
                  v_new_group := 1;
                  v_brg_id := NULL;
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                END IF;
              ELSE
                --v_item is NULL
                v_new_group := 1;
                v_brg_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Bio Results Group Description'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_intent = 'Taxon Abundance' THEN
                    IF length(v_item) > 1999 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 1999 characters and will be truncated.');
                       
                    
                      v_item := substr(v_item, 1, 1999);
                    END IF;
                    v_brg_text := v_item;
                  ELSE
                     
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                          ' ignored. It is only used when Intent is Taxon Abundance.');
                       
                    
                    v_brg_text := NULL;
                  END IF;
                ELSE
                  v_brg_text := NULL;
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                END IF;
              ELSE
                --v_item is NULL
                v_brg_text := NULL;
              END IF;
            
            ELSIF v_item_name = 'Bio Results Type'
                  AND v_new_group = 1 THEN
              IF v_intent = 'Taxon Abundance' THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_item IS NOT NULL THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_brg_type := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                    --check for MTPC on this activity
                    IF v_brg_type = 'Multi-Taxon Population Census' THEN
                      v_success := check_existing_mtpc(v_brg_type, v_activity_seq);
                      IF v_success = 2 THEN
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             'Multi-Taxon Population Census already exisit on this activity.');
                         
                      
                      END IF;
                    END IF;
                  ELSE
                    v_status := 'N';
                    v_brg_type := NULL;
                  END IF;
                ELSE
                  v_status := 'N';
                END IF;
              ELSE
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                     'Bio Results Type ignored. It is only used if Intent is Taxon Abundance');
                 
              
                v_brg_type := NULL;
              END IF;
            
            ELSIF v_item_name = 'Frequency Analysis Type'
                  AND v_new_group = 1 THEN
              --B or P
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                  IF v_dummy_seq IS NOT NULL THEN
                    v_brg_freq_analysis := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  ELSE
                    v_brg_freq_analysis := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Frequency Analysis Type is not used for ' ||
                                                       v_brg_type ||
                                                       ' and will be ignored');
                   
                
                  v_brg_freq_analysis := NULL;
                END IF;
              ELSE
                v_brg_freq_analysis := NULL;
              END IF;
            
            ELSIF v_item_name = 'Sex'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                --IF v_brg_freq_analysis = 'P' THEN
                IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_sexname_is_nbr := v_storet_key.is_number;
                    v_sexname_org_id := v_storet_key.org_id;
                  ELSE
                    v_status := 'N';
                    v_sexname_is_nbr := NULL;
                    v_sexname_org_id := NULL;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Sex ignored. It is only used for Single Taxon Frequency Classes');
                   
                
                  v_sexname_is_nbr := NULL;
                  v_sexname_org_id := NULL;
                END IF;
              ELSE
                v_sexname_is_nbr := NULL;
                v_sexname_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Life Stage'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                --              IF v_brg_freq_analysis = 'P' THEN
                IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_lifestage_is_nbr := v_storet_key.is_number;
                    v_lifestage_org_id := v_storet_key.org_id;
                  ELSE
                    v_status := 'N';
                    v_lifestage_is_nbr := NULL;
                    v_lifestage_org_id := NULL;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Life Stage ignored. It is only used for Single Taxon Frequency Classes.');
                   
                
                  v_lifestage_is_nbr := NULL;
                  v_lifestage_org_id := NULL;
                END IF;
              ELSE
                v_lifestage_is_nbr := NULL;
                v_lifestage_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Common Class Descriptor'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_freq_analysis = 'P' THEN
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                  IF v_storet_key.is_number IS NOT NULL THEN
                    v_char_is_nbr := v_storet_key.is_number;
                    v_char_org_id := v_storet_key.org_id;
                  ELSE
                    v_char_is_nbr := NULL;
                    v_char_org_id := NULL;
                    v_status := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Common Class Descriptor is required if Frequency Analysis Type is "Physical."');
                     
                  
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Common Class Descriptor should not be given unless Frequency Analysis Type is "Physical."');
                   
                
                END IF;
              END IF;
            
            ELSIF v_item_name = 'Common Class Descriptor Units'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_freq_analysis = 'P' THEN
                  v_storet_key := sim_import_result_extras.validate_storet_item('Result Value Units', v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_dummy_char);
                
                  IF v_storet_key.is_number IS NULL THEN
                    /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                    v_uom_is_nbr := NULL;
                    v_uom_org_id := NULL;
                    v_status := 'N';
                  ELSE
                    v_uom_is_nbr := v_storet_key.is_number;
                    v_uom_org_id := v_storet_key.org_id;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Common Class Descriptor Units should not be given unless Frequency Analysis Type is "Physical."');
                   
                
                END IF;
                --v_uom_is_nbr := NULL;
                --v_uom_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Bio Group Count Type'
                  AND v_new_group = 1 THEN
              -- This column was originally named 'Value Type Name'
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_type = 'Single Taxon Frequency Classes'
                   OR v_brg_type = 'Single Taxon Group Summary' THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_value_type_name := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  ELSE
                    v_value_type_name := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_value_type_name := NULL;
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       'Bio Group Count Type is not used for ' ||
                                                       v_brg_type ||
                                                       ' and will be ignored.');
                   
                
                END IF;
              ELSE
                v_value_type_name := NULL;
              END IF;
            
              -- Total Number in Group
            ELSIF v_item_name = 'Total Number in Group'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_brg_type IN ('Single Taxon Group Summary') THEN
                    BEGIN
                      IF v_item IS NOT NULL THEN
                        v_ttl_nbr_in_grp := to_number(v_item);
                        IF v_ttl_nbr_in_grp > 99999999 THEN
                           
                          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                               'Total Number in Group must be smaller than 100000000.');
                           
                        
                          v_status := 'N';
                          v_ttl_nbr_in_grp := 999;
                        END IF;
                        IF v_ttl_nbr_in_grp <= 0 THEN
                           
                          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                               'Total Number in Group must be greater than zero.');
                           
                        
                          v_status := 'N';
                        END IF;
                      ELSE
                        v_ttl_nbr_in_grp := NULL;
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             'Total Number in Group is required for Single Taxon Group Summary.');
                         
                      
                      END IF;
                    EXCEPTION
                      WHEN OTHERS THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             v_item_name ||
                                                             ' is not a valid number of 8 or fewer digits.');
                         
                      
                        v_status := 'N';
                    END;
                  ELSE
                    IF v_item IS NOT NULL THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           'Total Number in Group ignored. Only used for Single Taxon Group Summary.');
                       
                    
                    END IF;
                    v_ttl_nbr_in_grp := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_ttl_nbr_in_grp := NULL;
                END IF;
              ELSE
                --v_item is NULL
                v_ttl_nbr_in_grp := NULL;
              END IF;
            
            ELSIF v_item_name = 'Number of Individuals in Group'
                  AND v_new_group = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_brg_type IN ('Single Taxon Individuals') THEN
                    IF v_item IS NOT NULL THEN
                      v_ttl_nbr_indiv := v_item;
                      IF v_ttl_nbr_indiv > 999 THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             'Number of Individuals in Group must be smaller than 1000.');
                         
                      
                        v_status := 'N';
                        v_ttl_nbr_indiv := 999;
                      END IF;
                      IF v_ttl_nbr_indiv <= 0 THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             'Number of Individuals in Group must be greater than zero.');
                         
                      
                        v_status := 'N';
                      END IF;
                    ELSE
                      v_ttl_nbr_indiv := NULL;
                      v_status := 'N';
                        
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                           'Number of Individuals in Group is required for Single Taxon Individuals.');
                       
                    
                    END IF;
                  ELSE
                    IF v_item IS NOT NULL THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           'Total Number in Group ignored. Only used for Single Taxon Individuals');
                       
                    
                    END IF;
                    v_ttl_nbr_indiv := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_ttl_nbr_indiv := NULL;
                END IF;
              ELSE
                --v_item is NULL
                v_ttl_nbr_indiv := NULL;
              END IF;
            
            ELSIF v_item_name = 'Lower Class Bound' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_brg_freq_analysis = 'P' THEN
                  IF v_item IS NOT NULL THEN
                    v_lower_bnd_amt := v_item;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Lower and Upper Bounds are required for physical classes');
                     
                  
                  END IF;
                ELSE
                  IF v_item IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         'Lower Bound is not used except for Physical Class Indicators');
                     
                  
                  END IF;
                  v_lower_bnd_amt := NULL;
                END IF;
              ELSE
                v_lower_bnd_amt := NULL;
              END IF;
            
              -- Upper Boundary of class
            
            ELSIF v_item_name = 'Upper Class Bound' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_brg_freq_analysis = 'P' THEN
                  IF v_item IS NOT NULL THEN
                    v_upper_bnd_amt := v_item;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                         'Lower and Upper Bounds are required for physical classes');
                     
                  
                  END IF;
                ELSE
                  IF v_item IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         'Upper Bounds are not used except for Physical Class Indicators');
                     
                  
                  END IF;
                  v_upper_bnd_amt := NULL;
                END IF;
              ELSE
                v_upper_bnd_amt := NULL;
              END IF;
            
            ELSIF v_item_name = 'Individual Number' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_brg_type = 'Single Taxon Individuals' THEN
                    --Checking for duplicate brg_is_number||v_individual_id will be done during the create/insert record phase
                    v_individual_nbr := v_item;
                  ELSE
                    v_individual_nbr := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         'Individual Number ignored. It is only valid for Single Taxon Individuals groups.');
                     
                  
                  END IF;
                ELSE
                  -- Cannot translate to a NULL.              
                  v_status := 'N';
                  v_individual_nbr := NULL;
                END IF;
              ELSE
                IF v_brg_type = 'Single Taxon Individuals' THEN
                  v_status := 'N';
                  v_individual_nbr := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                       'Individual Number is required for Single Taxon Individuals group');
                   
                
                ELSE
                  v_individual_nbr := NULL;
                END IF;
              END IF;
            
              --Trap/Net details
            
              --------------------------------
              -- Trap or Net Sampling Duration
              --------------------------------
            
            ELSIF v_item_name = 'Trap or Net Sampling Duration'
                  AND v_new_activity = 1 THEN
            
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
              
                IF v_item IS NOT NULL THEN
                
                  -- is the trawl gear for trap/net?  (See Sample Procedure ID)
                  -- 1 = trawl, 2 = trap/net, 0 = all others
                  IF v_check_trawl_gear = 2 THEN
                    -- if yes
                    v_success := sim_import_result_extras.validate_number(v_item, v_item_name, '99.99', v_import_log_seq, v_line_cnt);
                  
                    IF v_success = 1 THEN
                      v_trap_net_duration := to_number(v_item);
                    ELSE
                      v_status := 'N';
                      v_trap_net_duration := NULL;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.');
                     
                  
                    v_trap_net_duration := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_trap_net_duration := NULL;
                END IF;
              
              ELSE
                v_trap_net_duration := NULL;
              END IF;
            
              -----------------------------
              -- Trap or Net Duration Units
              -----------------------------
            
            ELSIF v_item_name = 'Trap or Net Duration Units'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_check_trawl_gear = 2 THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_trap_net_dur_unit := sim_import_result_extras.get_allowable_value(v_dummy_seq, v_import_log_seq);
                  ELSE
                    v_status := 'N';
                    v_trap_net_dur_unit := NULL;
                  END IF;
                ELSE
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                      ' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.');
                   
                
                  v_trap_net_dur_unit := NULL;
                END IF;
              
              ELSE
                v_trap_net_dur_unit := NULL;
              END IF;
            
            ELSIF v_item_name = 'Relative Current Direction'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_check_trawl_gear = 2
                     OR v_check_trawl_gear = 1 THEN
                    v_success := sim_import_result_extras.validate_number(v_item, v_item_name, '999', v_import_log_seq, v_line_cnt);
                    IF v_success = 1 THEN
                      v_relative_current := to_number(v_item);
                      IF v_relative_current < 0
                         OR v_relative_current > 359 THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             v_item_name ||
                                                              ' must be an integer between 0 and 360.');
                         
                      
                      END IF;
                    ELSE
                      v_relative_current := NULL;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow or Net/Horizontal Tow.');
                     
                  
                    v_relative_current := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_relative_current := NULL;
                END IF;
              ELSE
                v_relative_current := NULL;
              END IF;
            
            ELSIF v_item_name = 'Relative Wind Direction'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_check_trawl_gear = 1
                     OR v_check_trawl_gear = 2 THEN
                    v_success := sim_import_result_extras.validate_number(v_item, v_item_name, '999', v_import_log_seq, v_line_cnt);
                    IF v_success = 1 THEN
                      v_relative_wind := to_number(v_item);
                      IF v_relative_wind < 0
                         OR v_relative_wind > 359 THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                             v_item_name ||
                                                            ' must be an integer between 0 and 360.');
                         
                      
                      END IF;
                    ELSE
                      v_relative_wind := NULL;
                      v_success := 'N';
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow or Net/Horizontal Tow.');
                     
                  
                    v_relative_wind := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_relative_wind := NULL;
                END IF;
              ELSE
                v_relative_wind := NULL;
              END IF;
            
              -------------------------  
              -- Orientation to Current
              -------------------------
            ELSIF v_item_name = 'Orientation to Current'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_check_trawl_gear = 2 THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_orient_to_cur := sim_import_result_extras.get_item_value(v_dummy_seq, v_line_cnt, v_import_log_seq);
                  ELSE
                    v_status := 'N';
                    v_orient_to_cur := NULL;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                        ' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.');
                   
                
                  v_orient_to_cur := NULL;
                END IF;
              ELSE
                v_orient_to_cur := NULL;
              END IF;
            
              -----------------------
              -- Trap or Net Comments
              -----------------------
            
            ELSIF v_item_name = 'Trap or Net Comments'
                  AND v_new_activity = 1 THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_check_trawl_gear = 2 THEN
                    IF length(v_item) > 254 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                          ' is longer than 254 characters and will be truncated.');
                       
                    
                      v_trap_net_comment := substr(v_item, 1, 254);
                    ELSE
                      v_trap_net_comment := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored - it is not used unless Sample Collection Procedure has a Gear Type of Net/Non-Tow.');
                     
                  
                    v_trap_net_comment := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_trap_net_comment := NULL;
                END IF;
              ELSE
                v_trap_net_comment := NULL;
              END IF;
            
            ELSIF v_item_name = 'Functional Feeding Group' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                    IF length(v_item) > 6 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                            ' is longer than 6 characters and will be truncated.');
                       
                    
                      v_func_feed_grp := substr(v_item, 1, 6);
                    ELSE
                      v_func_feed_grp := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' ignored - it is not used unless Group Type is Multi-Taxon Population Ceusus.');
                     
                  
                    v_func_feed_grp := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_func_feed_grp := NULL;
                END IF;
              ELSE
                v_func_feed_grp := NULL;
              END IF;
            
            ELSIF v_item_name = 'Taxon Pollution Tolerance' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                    IF length(v_item) > 4 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                          ' is longer than 4 characters and will be truncated.');
                       
                    
                      v_taxon_pollution := substr(v_item, 1, 4);
                    ELSE
                      v_taxon_pollution := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' ignored - it is not used unless Group Type is Multi-Taxon Population Ceusus.');
                     
                  
                    v_taxon_pollution := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_taxon_pollution := NULL;
                END IF;
              ELSE
                v_taxon_pollution := NULL;
              END IF;
            
            ELSIF v_item_name = 'Trophic Level' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                    IF length(v_item) > 4 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                           v_item_name ||
                                                          ' is longer than 4 characters and will be truncated.');
                       
                    
                      v_trophic_level := substr(v_item, 1, 4);
                    ELSE
                      v_trophic_level := v_item;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored - it is not used unless Group Type is Multi-Taxon Population Ceusus.');
                     
                  
                    v_trophic_level := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_trophic_level := NULL;
                END IF;
              ELSE
                v_trophic_level := NULL;
              END IF;
            
            ELSIF v_item_name = 'Cell Form' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                  IF v_community_is_nbr > 207
                     AND v_community_is_nbr < 211 THEN
                    v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSRCLDES', 'CELL_TYPE_NM', v_activity_seq, v_import_log_seq, v_line_cnt);
                    IF v_dummy_char IS NOT NULL THEN
                      v_cell_type := rtrim(v_dummy_char);
                    ELSE
                      v_cell_type := '<Spaces>';
                      v_status := 'N'; -- v_item invalid or translated to null - set status to 'N'                    
                    END IF;
                  ELSE
                    --Wrong community type
                    v_cell_type := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored. It is not used with this community.');
                     
                  
                  END IF;
                ELSE
                  v_cell_type := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                      ' ignored. It is not used with this group type or intent.');
                   
                
                END IF;
              ELSE
                IF v_brg_type IN ('Multi-Taxon Population Census')
                   AND v_community_is_nbr > 207
                   AND v_community_is_nbr < 211 THEN
                  v_cell_type := '<Spaces>';
                ELSE
                  v_cell_type := NULL;
                END IF;
              END IF;
            
            ELSIF v_item_name = 'Cell Shape' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 061603 slp - Removed trailing spaces.
                IF v_brg_type IN ('Multi-Taxon Population Census') THEN
                  IF v_community_is_nbr > 207
                     AND v_community_is_nbr < 211 THEN
                    v_dummy_char := validate_prmvl_item(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSRCLDES', 'CELL_SHAPE_TYPE_NM', v_activity_seq, v_import_log_seq, v_line_cnt);
                    IF v_dummy_char IS NOT NULL THEN
                      v_cell_shape := rtrim(v_dummy_char);
                    ELSE
                      v_cell_shape := '<Spaces>';
                      v_status := 'N'; -- v_item invalid or translated to null - set status to 'N'                      
                    END IF;
                  ELSE
                    --Wrong community type
                    v_cell_shape := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                         v_item_name ||
                                                        ' ignored. It is not used with this community.');
                     
                  
                  END IF;
                ELSE
                  v_cell_shape := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                      ' ignored. It is not used with this group type or intent.');
                   
                
                END IF;
              ELSE
                IF v_brg_type IN ('Multi-Taxon Population Census')
                   AND v_community_is_nbr > 207
                   AND v_community_is_nbr < 211 THEN
                  v_cell_shape := '<Spaces>';
                ELSE
                  v_cell_shape := NULL;
                END IF;
              END IF;
            
            ELSIF v_item_name = 'Habit' THEN
              IF v_item IS NOT NULL THEN
                v_storet_key := get_prmvl_key(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSRRSULT', 'HABIT', v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_storet_key.is_number IS NULL THEN
                  v_status := 'N';
                  v_habit_is_number := NULL;
                  v_habit_org_id := NULL;
                ELSE
                  v_habit_is_number := v_storet_key.is_number;
                  v_habit_org_id := v_storet_key.org_id;
                END IF;
              ELSE
                v_habit_is_number := NULL;
                v_habit_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Voltinism' THEN
              IF v_item IS NOT NULL THEN
                v_storet_key := get_prmvl_key(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSRRSULT', 'VOLTINISM', v_activity_seq, v_import_log_seq, v_line_cnt);
                IF v_storet_key.is_number IS NULL THEN
                  v_status := 'N';
                  v_volt_is_number := NULL;
                  v_volt_org_id := NULL;
                ELSE
                  v_volt_is_number := v_storet_key.is_number;
                  v_volt_org_id := v_storet_key.org_id;
                END IF;
              ELSE
                v_volt_is_number := NULL;
                v_volt_org_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Primary Class Descriptor' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_freq_analysis = 'B' THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_primary_class := sim_import_result_extras.get_allowable_value(v_dummy_seq, p_import_log_seq);
                    IF v_primary_class = 'SEX' THEN
                      v_bio_condition_is_nbr := 441;
                      v_bio_condition_org_id := 'ZZZZZZZZ';
                    ELSIF v_primary_class = 'LIFESTAGE' THEN
                      v_bio_condition_is_nbr := 559;
                      v_bio_condition_org_id := 'ZZZZZZZZ';
                    END IF;
                  ELSE
                    v_status := 'N';
                    v_bio_condition_is_nbr := NULL;
                    v_bio_condition_org_id := NULL;
                    v_primary_class := NULL;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                      ' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.');
                   
                  
                v_bio_condition_is_nbr := NULL;
                  v_bio_condition_org_id := NULL;
                  v_primary_class := NULL;
                END IF;
              ELSE
                v_bio_condition_is_nbr := NULL;
                v_bio_condition_org_id := NULL;
                v_primary_class := NULL;
              END IF;
            
            ELSIF v_item_name = 'Secondary Class Descriptor' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_freq_analysis = 'B' THEN
                  v_dummy_seq := sim_import_result_extras.validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq, v_import_log_seq, v_line_cnt);
                  IF v_dummy_seq IS NOT NULL THEN
                    v_secondary_class := sim_import_result_extras.get_allowable_value(v_dummy_seq, p_import_log_seq);
                    IF v_secondary_class = 'SEX' THEN
                      v_sec_bio_cond_is_nbr := 441;
                      v_sec_bio_cond_org_id := 'ZZZZZZZZ';
                    ELSIF v_secondary_class = 'LIFESTAGE' THEN
                      v_sec_bio_cond_is_nbr := 559;
                      v_sec_bio_cond_org_id := 'ZZZZZZZZ';
                    END IF;
                  ELSE
                    v_status := 'N';
                    v_sec_bio_cond_is_nbr := NULL;
                    v_sec_bio_cond_org_id := NULL;
                    v_secondary_class := NULL;
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                       v_item_name ||
                                                      ' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.');
                   
                  
                  v_sec_bio_cond_is_nbr := NULL;
                  v_sec_bio_cond_org_id := NULL;
                  v_secondary_class := NULL;
                END IF;
              ELSE
                v_sec_bio_cond_is_nbr := NULL;
                v_sec_bio_cond_org_id := NULL;
                v_secondary_class := NULL;
              END IF;
            
            ELSIF v_item_name = 'Primary Class Value' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_freq_analysis = 'B' THEN
                  v_primary_class_value := validate_class_descriptor(v_item, p_imp_cfg_seq, v_bio_condition_is_nbr, v_item_name, v_col_seq, v_activity_seq, v_line_cnt, v_import_log_seq);
                  IF v_primary_class_value IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.');
                   
                  
                  v_primary_class_value := NULL;
                END IF;
              ELSE
                v_primary_class_value := NULL;
              END IF;
            
            ELSIF v_item_name = 'Secondary Class Value' THEN
              IF v_item IS NOT NULL THEN
                v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                IF v_brg_freq_analysis = 'B' THEN
                  v_secondary_class_value := validate_class_descriptor(v_item, p_imp_cfg_seq, v_sec_bio_cond_is_nbr, v_item_name, v_col_seq, v_activity_seq, v_line_cnt, v_import_log_seq);
                  IF v_secondary_class_value IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' ignored. It is only used for BIOLOGICAL Frequency Analysis Type.');
                   
                  
                  v_secondary_class_value := NULL;
                END IF;
              ELSE
                v_secondary_class_value := NULL;
              END IF;
            
              ----------------------
              -- jah 2-20-2004
              -- Particle Size Basis
              ----------------------
            ELSIF (v_item_name = 'Particle Size Basis') THEN
            
              -- Does the field contain anything?
              IF (v_item IS NOT NULL) THEN
                -- if yes
              
                -- translate the field
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
              
                -- did the field translate to okay?
                IF (v_item IS NOT NULL) THEN
                  -- if yes
                
                  -- prepare to INSERT Particle Size Basis INTO SIM_RESULT_CLASS_INDICATOR
                  v_particle_size_basis := v_item;
                
                ELSE
                  -- v_item translated to null
                
                  -- prevent migration
                  v_status := 'N';
                  v_particle_size_basis := NULL;
                END IF;
              
              END IF;
            
            ELSIF v_item_name = 'Frequency Class Count' THEN
              IF v_item IS NOT NULL THEN
                -- 09/30/03 TCJ - add code to check for translations
                v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                IF v_item IS NOT NULL THEN
                  IF v_brg_type = 'Single Taxon Frequency Classes' THEN
                    v_dummy_seq := sim_import_result_extras.validate_number(v_item, v_item_name, '999999999', v_import_log_seq, v_line_cnt);
                    IF v_dummy_seq = 1 THEN
                      v_class_count := v_item;
                      v_result_seq_num := v_result_seq_num + 1;
                    ELSE
                      v_class_count := NULL;
                      v_status := 'N';
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                        v_item_name ||
                                                        ' ignored. It is only used for Single Taxon Frequency Classes.');
                     
                    
                    v_class_count := NULL;
                  END IF;
                ELSE
                  v_status := 'N'; -- v_item translated to null - set status to 'N'
                  v_class_count := NULL;
                END IF;
              ELSE
                v_class_count := NULL;
              END IF;
            
            ELSIF v_item_name = 'Field/Lab Procedure'
                  OR v_item_name = 'Characteristic Field/Lab Analytical Procedure' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                IF v_item IS NOT NULL THEN
                
                  -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                  v_item := rtrim(v_item);
                
                  /* get the analysis procedure is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_lab_procedure);
                
                  /* 06/09/03 SLP - If a storet key is returned, then the Field/Lab Proc ID provided is valid.  The key however may not
                  represent the correct is number and org id as Field/Lab Proc Source was not used in this check.  The
                  v_anlpr_is_nbr and v_anlpr_org_id are not set until the combination of ID and source are
                  validated in the sim_import_result_extras.get_field_lab_procedure.  */
                
                  v_anlpr_is_nbr := NULL;
                  v_anlpr_org_id := NULL;
                
                  IF v_storet_key.is_number IS NULL THEN
                    v_lab_procedure := NULL;
                    v_status := 'N';
                  END IF;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                  
                END IF;
                v_anlpr_is_nbr := NULL;
                v_anlpr_org_id := NULL;
                v_lab_procedure := NULL;
              END IF;
            
            ELSIF v_item_name = 'Field/Lab Procedure Source'
                  OR
                  v_item_name = 'Characteristic Field/Lab Analytical Procedure Source' THEN
            
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
              
                IF v_item IS NOT NULL THEN
                
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_lab_source);
                
                  /* 06/09/03 SLP - If a storet key is returned, then the Field/Lab Proc Source provided is valid.  The key however may not
                  represent the correct is number and org id as Field/Lab Proc ID was not used in this check.  The
                  v_anlpr_is_nbr and v_anlpr_org_id are not set until the combination of ID and source are
                  validated in the sim_import_result_extras.get_field_lab_procedure.  */
                
                  v_anlpr_is_nbr := NULL;
                  v_anlpr_org_id := NULL;
                
                  IF v_storet_key.is_number IS NULL THEN
                    v_lab_source := NULL;
                    v_status := 'N';
                  END IF;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                  
                END IF;
                v_anlpr_is_nbr := NULL;
                v_anlpr_org_id := NULL;
                v_lab_source := NULL;
              END IF;
            
            ELSIF v_item_name = 'Lab Sample Prep Procedure' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  /* get the lab preparation procedure is number - check translations and STORET tables */
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_labprep_id);
                
                  /* 06/09/03 SLP - If a storet key is returned, then the LSPP ID provided is valid.  The key however may not
                  represent the correct is number and org id as LSPP Source was not used in this check.  The
                  v_labprep_is_nbr and v_labprep_org_id are not set until the combination of ID and source are
                  validated in the sim_import_result_extras.get_lab_sample_prep_procedure.  */
                
                  v_labprep_is_nbr := NULL;
                  v_labprep_org_id := NULL;
                
                  IF v_storet_key.is_number IS NULL THEN
                    v_labprep_id := NULL;
                    v_status := 'N';
                  END IF;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                  
                END IF;
                v_labprep_is_nbr := NULL;
                v_labprep_org_id := NULL;
                v_labprep_id := NULL;
              END IF;
            
            ELSIF v_item_name = 'Lab Sample Prep Procedure Source' THEN
              IF v_item IS NOT NULL
                 AND v_habitat != 1 THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_storet_key := sim_import_result_extras.validate_storet_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_dummy_key, p_org_id, v_line_cnt, v_activity_seq, v_import_log_seq, v_labprep_source);
                
                  /* 06/09/03 SLP - If a storet key is returned, then the LSPP source provided is valid.  The key however may not
                  represent the correct is number and org id as LSPP ID was not used in this check.  The
                  v_labprep_is_nbr and v_labprep_org_id are not set until the combination of ID and source are
                  validated in the sim_import_result_extras.get_lab_sample_prep_procedure.  */
                
                  v_labprep_is_nbr := NULL;
                  v_labprep_org_id := NULL;
                
                  IF v_storet_key.is_number IS NULL THEN
                    v_labprep_source := NULL;
                    v_status := 'N';
                  END IF;
                END IF;
              ELSE
                IF v_item IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      v_item_name ||
                                                      ' has been provided and is not allowed for Habitat Assessment.');
                   
                  
                END IF;
                v_labprep_is_nbr := NULL;
                v_labprep_org_id := NULL;
                v_labprep_source := NULL;
              END IF;
            
            END IF;
          END IF;
        
        END LOOP;
      
        /******* End of column loop ********/
      
        ---------------------------------------------------------------------------------
        --
        --
        --  E N D   O F   C O L U M N   L O O P
        --
        --
        ---------------------------------------------------------------------------------
      
        ---------------------------------------------------------------------------------
        --
        --
        -- The following checks happen after one entire line has been read and validated.
        -- 
        -- 
        ---------------------------------------------------------------------------------
      
        /* bka (8-20-01) check to see if value type was included in row,
        if not, then add it with a default value of 'Actual' */
        IF v_value_type IS NULL
           AND v_habitat != 1 THEN
          IF v_brg_type <> 'Single Taxon Group Summary' THEN
            v_value_type := 'Actual';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                'Value type is required and has not been included. A default value type of "Actual" has been assigned.');
             
            
          END IF;
        END IF;
      
        /* see if activity already exists in STORET */
        IF v_new_activity = 1 THEN
        
          /* dtw 10-21-03 - If Trip QC Sample. check to see if a valid Trip QC Type was included. */
          IF v_trip_qc = 1
             AND v_activity_cat IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Trip QC Type is required when Activity Type = Trip QC.');
             
            
            v_status := 'N';
          END IF;
        
          /* DTW 6/18/03 - Added logic to only check one time for each activity. */
          IF v_trip_qc = 1 THEN 
            v_storet_key := sim_import_result_checks.check_storet_act_id(v_activity_id, v_replicate_nbr, v_tsrstvst_is_nbr, p_org_id, v_activity_seq, v_line_cnt, v_import_log_seq, TRUE);
          ELSE
            v_storet_key := sim_import_result_checks.check_storet_act_id(v_activity_id, v_replicate_nbr, v_tsrstvst_is_nbr, p_org_id, v_activity_seq, v_line_cnt, v_import_log_seq, FALSE);
          END IF;
          IF v_storet_key.is_number IS NOT NULL THEN
            -- activity already exists in STORET, create in sim but do not allow it to be exported
            v_act_is_nbr := v_storet_key.is_number;
            v_act_org_id := v_storet_key.org_id;
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                v_activity_id ||
                                                ' already exists in STORET, this activity will not be exported.');
             
            
          ELSE
            v_act_is_nbr := NULL;
            v_act_org_id := NULL;
          END IF;
        END IF;
      
        /* SF (10/7/02) Check Field/Lab Procedure */
        /* This routine validates the combination of field/lab procedure id and source. Source was added as an import column
        when we determined that ID is not unique within this table. */
        IF v_lab_procedure IS NOT NULL
           AND v_lab_source IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Field/Lab Procedure Source is required to identify Procedure.');
           
          
          v_status := 'N';
        ELSIF v_lab_procedure IS NULL
              AND v_lab_source IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Field/Lab Procedure AND Source are both required to identify Procedure.');
           
          
          v_status := 'N';
        ELSIF v_lab_procedure IS NOT NULL
              AND v_lab_source IS NOT NULL THEN
          v_storet_key := sim_import_result_extras.get_field_lab_procedure(v_lab_procedure, v_lab_source, p_org_id, p_imp_cfg_seq, v_col_seq, v_import_log_seq, v_line_cnt, v_activity_seq);
          IF v_storet_key.is_number IS NOT NULL THEN
            v_anlpr_is_nbr := v_storet_key.is_number;
            v_anlpr_org_id := v_storet_key.org_id;
          ELSE
            v_status := 'N';
          END IF;
        END IF;
      
        /* 06/09/03 SLP - Check Lab Sample Prep Procedure */
        /* This routine validates the combination of lab sample prep procedure ID and source. Both are required to
        guarantee uniqueness. Although ID is currently sufficient to uniquely identify the LSPP, we implemented
        this in the same fashion as field/lab procedure in order to protect against future data added to this table
        that violates this rule. (There is no unique constraint on lab sample procedure ID.) */
        IF v_labprep_id IS NOT NULL
           AND v_labprep_source IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Lab Sample Prep Procedure Source is required to identify Lab Sample Prep Procedure.');
           
          
          v_status := 'N';
        ELSIF v_labprep_id IS NULL
              AND v_labprep_source IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Lab Sample Prep Procedure AND Source are both required to identify Lab Sample Prep Procedure.');
           
          
          v_status := 'N';
        ELSIF v_labprep_id IS NOT NULL
              AND v_labprep_source IS NOT NULL THEN
          v_storet_key := sim_import_result_extras.get_lab_sample_prep_procedure(v_labprep_id, v_labprep_source, p_org_id, p_imp_cfg_seq, v_col_seq, v_import_log_seq, v_line_cnt, v_activity_seq);
          IF v_storet_key.is_number IS NOT NULL THEN
            v_labprep_is_nbr := v_storet_key.is_number;
            v_labprep_org_id := v_storet_key.org_id;
          ELSE
            v_status := 'N';
          END IF;
        END IF;
      
        /* Medium is not allowed for Habitat Assessments */
        IF v_medium IS NULL
           AND v_habitat != 1 THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Medium is required and has not been provided');
           
          
        ELSIF v_medium IS NULL
              AND v_habitat = 1 THEN
          v_medium := ' ';
        ELSIF v_medium IS NOT NULL
              AND v_habitat = 1 THEN
          v_medium := ' ';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Medium ignored. It is not used for Habitat Assesment.');
           
          
        END IF;
      
        -- This section is for Parent Sample ID
        IF nvl(v_no_parent_ids, 0) != 0 THEN
        
          -- do we have too few Parent Sample IDs for 'Created from Sample'?
          IF (v_activity_cat = 'Created from Sample')
             AND (v_no_parent_ids < 1) THEN
            -- if yes
            -- set status flag to reject this import record
            v_status := 'N';
            -- output error message
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'A Parent Sample ID is required for "Created from Sample" activities.');
             
            
          END IF;
        
          -- do we have too many Parent Sample IDs for 'Created from Sample'?
          IF (v_activity_cat = 'Created from Sample')
             AND (v_no_parent_ids > 1) THEN
            -- if yes
            -- set status flag to reject this import record
            v_status := 'N';
            -- output error message
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Only one Parent Sample ID is allowed in "Created from Sample" activities.');
             
            
          END IF;
        
          -- do we have too few Parent Sample IDs for 'Composite-with Parents'?
          IF (v_activity_cat = 'Composite-with Parents')
             AND (v_no_parent_ids < 2) THEN
            -- if yes
            -- set status flag to reject this import record
            v_status := 'N';
            -- output error message
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'STORET requires at least two Parent Sample IDs for "Composite-with Parents"');
             
            
          END IF;
        
          -- Do we have any duplicate parent IDs in the table?
          FOR v_number_i IN 1 .. (v_no_parent_ids - 1) LOOP
            FOR v_number_j IN (v_number_i + 1) .. v_no_parent_ids LOOP
            
              -- Do we have a match?
              IF (v_parent_id_table(v_number_i) = v_parent_id_table(v_number_j)) THEN
                -- if yes
                -- set status flag to reject this import record
                v_status := 'N';
                -- output error message
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'STORET does not allow duplicate Parent Sample IDs -- ' ||
                                                     v_parent_id_table(v_number_i) ||
                                                     ' occurs more than once.');
                 
                
              END IF;
            END LOOP;
          END LOOP;
        
          /*
            The following routine determines if the parent records in STORET have room 
            for "Composite-with Parent" records. Also, it determines if we have any
            station visit mis-matches.
          */
        
          -- Do we have any parent ids?
          IF (v_no_parent_ids > 0) THEN
            -- if yes
            -- retrieve the station visit information from SIM_STATION_VISIT
            OPEN c_station_visit(v_visit_seq);
            FETCH c_station_visit
              INTO r_station_visit;
            CLOSE c_station_visit;
          
            FOR v_number IN 1 .. v_no_parent_ids LOOP
            
              -- Try to retrieve the parent record from STORET
              OPEN c_tsrfdact(v_parent_is_number_table(v_number), v_parent_org_id_table(v_number));
              FETCH c_tsrfdact
                INTO r_tsrfdact;
            
              -- did we find the parent in STORET?
              IF (c_tsrfdact%FOUND) THEN
                -- if yes
                -- Does the parent record in STORET have room for a "Composite-with Parent"?
                IF (r_tsrfdact.tsrfdact0is_number IS NOT NULL) THEN
                  -- if no
                  -- set status flag to reject this import record
                  v_status := 'N';
                  -- output error message
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Parent Activity ' ||
                                                       v_parent_id_table(v_number) ||
                                                       ' in the STORET database already has a "Composite-with Parents" child record.');
                   
                  
                END IF;
              
                -- do we have a station visit mis-match? (Re-check this in case
                -- the parent ID came from a translation.)
                IF (r_tsrfdact.tsrstvst_is_number !=
                   r_station_visit.ssv_tsrstvst_is_number)
                   OR
                   (r_tsrfdact.tsrstvst_org_id != r_station_visit.ssv_tsrstvst_org_id) THEN
                  -- if yes
                  -- set status flag to reject this import record
                  v_status := 'N';
                  -- output error message
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Parent Activity ' ||
                                                       v_parent_id_table(v_number) ||
                                                       ' has the wrong station visit.');
                   
                  
                END IF;
              END IF;
              -- Close the tsrfdact cursor
              CLOSE c_tsrfdact;
            
              /*
                jah 2-19-04
                Do we already have a 'Composite...' child for this parent in SIM?
              */
            
              -- Is this a 'Composite...' child?
              IF (v_activity_cat = 'Composite-with Parents') THEN
                -- if yes
              
                -- Try to find a 'Composite ...' child for this parent already in SIM
                OPEN c_activity_parent(v_parent_fa_seq_table(v_number), v_parent_is_number_table(v_number), v_parent_org_id_table(v_number), 'Composite-with Parents');
                FETCH c_activity_parent
                  INTO r_activity_parent;
              
                -- Did we find a 'Composite...' child for this parent alreqady in SIM?
                IF c_activity_parent%FOUND THEN
                  -- if yes
                  -- set status flag to reject this import record
                  v_status := 'N';
                  -- output error message
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Parent Activity ' ||
                                                       v_parent_id_table(v_number) ||
                                                       ' already has a "Composite-with Parents" child in SIM.');
                   
                  
                END IF;
                -- Close the c_activity_parent cursor
                CLOSE c_activity_parent;
              END IF;
            
            END LOOP;
          END IF;
        END IF;
      
        /* SF - 11/9/02 Default a missing QC Indicator to 'N' */
        IF v_qc_indicator IS NULL THEN
          v_qc_indicator := 'N';
        END IF;
      
        /* kms (6/28/01) - make sure field set id is included if field set name is there and
        vice versa. */
        IF v_field_set_nbr IS NOT NULL
           AND v_field_set_name IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Field SET Name must be included when field set id is present.');
           
          
        ELSIF v_field_set_nbr IS NULL
              AND v_field_set_name IS NOT NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Field set ID must be included when field set name is present.');
           
          
        END IF;
      
        -- jah 3-2-2004
        /* Business Rule
        The Gear Configuration ID must be valid for the Gear ID */
      
        -- do we need to check the gear configuration?
        IF (v_gearcfg_is_nbr IS NOT NULL) THEN
          -- if yes
        
          -- try to retrieve the gear configuration ID for the given gear ID
          OPEN c_get_gear_configuration(v_gear_is_nbr, v_gear_org_id, v_gearcfg_is_nbr, v_gearcfg_org_id);
          FETCH c_get_gear_configuration
            INTO r_get_gear_configuration;
        
          -- is the Gear Configuration invalid?
          IF (c_get_gear_configuration%NOTFOUND) THEN
            -- if yes
          
            -- Prevent migration to STORET
            v_status := 'N';
          
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'The selected Gear Configuration ID is not valid for the selected Gear ID.');
             
            
          END IF;
          CLOSE c_get_gear_configuration;
        
        END IF;
      
        /* make sure field gear types in the gear and collection procedure are the same */
        IF v_proc_is_nbr IS NOT NULL THEN
          /* If it's WEBSIM look for a matching Gear Config */
          IF v_sid_seq = 10007
             AND v_type != 'Field Msr/Obs' THEN
            OPEN c_matching_config(v_proc_name);
            FETCH c_matching_config
              INTO v_gearcfg_is_nbr, v_gearcfg_org_id, v_gear_is_nbr, v_gear_org_id;
            CLOSE c_matching_config;
          END IF;
        
          /* see if gear is required, compare gear types and return 0 if they match */
          v_gear_type_ok := sim_import_result_checks.check_gear_type(v_proc_is_nbr, v_gear_is_nbr, p_org_id, v_activity_seq, v_line_cnt, v_import_log_seq);
          IF v_gear_type_ok = 1 THEN
            /* gear types do not match, write error to log and change status to 'N' */
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Gear types for collection procedure and gear do not match.');
             
            
          ELSIF v_gear_type_ok = 2 THEN
            /* sample collection procedure has a gear type but gear has not been provided, invalidate activity
            and write error message */
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Gear is required when the collection procedure has a gear type.');
             
           
          END IF;
        ELSIF v_proc_is_nbr IS NULL
              AND v_new_activity = 1
              AND v_type = 'Sample' THEN
          /* write error if sample collection procedure has not been included for a sample */
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Sample Collection procedure is required and has not been included or is invalid.');
           
         
        END IF;
      
        /* make sure detection limits and quantification limits have a unit of measure */
        IF v_detect_limit IS NOT NULL
           AND v_det_uom_is_nbr IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Detection limit unit of measure has not been provided, and is required when detection limit is given.');
           
         
          v_status := 'N';
        END IF;
        IF v_lowq_limit IS NOT NULL
           AND v_det_uom_is_nbr IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Detection limit unit of measure has not been provided, and is required when lower quantification limit is given.');
           
         
          v_status := 'N';
        END IF;
        IF v_upq_limit IS NOT NULL
           AND v_det_uom_is_nbr IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Detection limit unit of measure has not been provided, and is required when upper quantification limit is given.');
           
         
          v_status := 'N';
        END IF;
      
        /* 06/09/03 slp - Only set the result status to Final if the record is the first of the group.
        - If the record is not the first for 'Single Taxon', then allow the status to be NULL. */
        /* SF - 1/8/02 Default a missing Result Status to "Final" */
        IF v_new_group = 1
           OR v_brg_type != 'Single Taxon Frequency Classes' THEN
          IF v_result_status IS NULL THEN
            v_result_status := 'F';
          END IF;
        END IF;
      
        /* The following checks are for Biological Samples Only */
        /*******************************************************/
        IF v_medium = 'Biological' THEN
        
          /* SF (1/14/02) Check to see that intent is there for BIO */
          IF v_intent IS NULL
             AND v_new_activity = 1 THEN
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Intent is required for Biological Results and has not been provided');
             
           
          END IF;
        
          /* SF (1/8/02) Check that community exists for Taxon Abundance */
          IF v_intent = 'Taxon Abundance'
             AND v_community_is_nbr IS NULL THEN
            v_status := 'N';
            
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Community is required for Taxon Abundance Intent');
             
           
          END IF;
        
          /* SF (1/8/02) Check that brg exists for Taxon Abundance */
          IF v_intent = 'Taxon Abundance'
             AND v_brg_id IS NULL
             AND v_new_group = 1 THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Bio Results Group is required when Intent is Taxon Abundance.');
             
           
            v_status := 'N';
          END IF;
        
          /* SF (1/8/02) Check that br type exists for Taxon Abundance */
          IF v_intent = 'Taxon Abundance'
             AND v_brg_type IS NULL
             AND v_new_group = 1 THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Bio Results Type is required when Intent is Taxon Abundance.');
             
           
            v_status := 'N';
          END IF;
        
          /* SF (1/8/02) Check that Subject Taxon exists for Tissue, Individual */
          IF v_intent IN ('Individual', 'Tissue')
             AND v_taxon_is_nbr IS NULL THEN
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Subject Taxon is required and has not been provided');
             
           
          END IF;
        
          /* SF (1/8/02) Check that Bio Part exists for Tissue */
          IF v_intent = 'Tissue'
             AND v_biopart_is_nbr IS NULL
             AND v_new_activity = 1 THEN
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Bio Part is required for TISSUE');
             
           
          END IF;
        
          /* Subject Taxon Required for all BRG except MTPC*/
          IF v_brg_type LIKE 'Single%'
             AND v_taxon_is_nbr IS NULL THEN
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Subject Taxon for Single Taxon group is required');
             
           
          END IF;
        
          /* SF (1/4/02) make sure there is a count type for STGS */
          IF v_value_type_name IS NULL THEN
            IF v_brg_type = 'Single Taxon Group Summary' THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Count Type is required for Single Taxon Results.');
               
             
              v_status := 'N';
            END IF;
          ELSE
            IF v_brg_type = 'Single Taxon Frequency Classes' THEN
              --it goes on the result instead of on the BRG
              v_value_type := v_value_type_name;
              v_value_type_name := NULL;
            END IF;
          END IF;
        
          /* SF (1/8/02) Check that total number exists for Individual, group summary */
          IF v_brg_type = 'Single Taxon Group Summary'
             AND v_ttl_nbr_in_grp IS NULL THEN
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Total Number in Group is required and has not been provided');
             
           
          END IF;
        
          IF v_brg_type = 'Single Taxon Individuals'
             AND v_ttl_nbr_indiv IS NULL THEN
            v_status := 'N';
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Number of Individuals in Group is required and has not been provided');
             
           
          END IF;
        
          /* SF - 10/2/02 Check Frequency Class rules */
          IF v_brg_type = 'Single Taxon Frequency Classes' THEN
          
            /* SF (1/15/02) Check Bio Rslt Type Ind for STFC */
            IF v_brg_freq_analysis IS NULL THEN
              v_status := 'N';
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Frequency Analysis Type is required for Single Taxon Frequency Classes');
               
             
            END IF;
          
            IF v_brg_freq_analysis = 'B' THEN
              IF v_primary_class IS NULL THEN
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Primary Class Descriptor is required for BIOLOGICAL Frequency Analyses.');
                 
               
                IF v_secondary_class IS NOT NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Secondary Class Descriptor ignored. It is only used when Primary Class Descriptor is also given.');
                 
               
                END IF;
              ELSIF v_primary_class = v_secondary_class THEN
                -- DTW 6/16/03 - Validate that Primary and Secondary Class Descriptors are not the same
                v_status := 'N';
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    v_primary_class ||
                                                    ' cannot be provided as both the Primary and Secondary Class Descriptors.');
                 
               
              END IF;
            
              IF v_sexname_is_nbr IS NOT NULL THEN
                  
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                    'Sex ignored. It is not used for Biological frequency analyses.');
                 
               
                v_sexname_is_nbr := NULL;
                v_sexname_org_id := NULL;
              END IF;
            
              IF v_lifestage_is_nbr IS NOT NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                    'Life Stage ignored. It is not used for Biological frequency analyses.');
                 
               
                v_lifestage_is_nbr := NULL;
                v_lifestage_org_id := NULL;
              END IF;
            
              IF v_value_type IS NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Bio Count Type is required for Single Taxon Frequency Classes.');
                 
               
                v_status := 'N';
              END IF;
            END IF;
          END IF;
        
          /* Validate Species validity and requirements */
          IF v_intent = 'Taxon Abundance' THEN
            v_act_species := NULL;
            IF v_brg_type LIKE 'Single%' THEN
              -- Single Taxon Group Summary, Single Taxon Individual, or Single Taxon Frequency Class
              v_result_species := NULL;
              -- Taxa recorded at group level, only check first row in group.
              IF v_new_group = 1 THEN
                -- Species never required, but only allowed for certain taxa
                v_allow_species := check_allow_species(v_taxon_is_nbr);
                IF v_allow_species != 1
                   AND v_taxon_species_number IS NOT NULL THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                      'Species Number ignored. The Subject Taxon is already identified to the species level.');
                   
                 
                  v_taxon_species_number := NULL;
                END IF;
                v_brg_species := v_taxon_species_number;
              END IF;
            ELSIF v_brg_type LIKE 'Multi%' THEN
              -- Taxa recorded at result level, check each row in group.  Taxa stored in Characteristic instead of subject taxon.
              v_allow_species := check_allow_species(v_char_is_nbr);
              IF v_allow_species != 1
                 AND v_char_species_number IS NOT NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                    'Species Number ignored. The Taxa Characteristic is already identified to the species level.');
                 
               
                v_char_species_number := NULL;
              END IF;
              v_result_species := v_char_species_number;
            END IF;
          ELSIF v_intent = 'Tissue'
                OR v_intent = 'Individual' THEN
            -- Subject Taxon Species recorded at the Activity Level
            -- Can also have Characteristic Taxa recorded at Result Level
            -- When Subject Taxon (for the Activity) is allowed, it is required.
            v_brg_species := NULL;
            IF v_new_activity = 1 THEN
              v_allow_species := check_allow_species(v_taxon_is_nbr);
              IF v_allow_species != 1
                 AND v_taxon_species_number IS NOT NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                    'Species Number ignored. The Subject Taxon is already identified to the species level.');
                 
               
                v_taxon_species_number := NULL;
              ELSIF v_allow_species = 1
                    AND v_taxon_species_number IS NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Species Number Required for Subject Taxon.');
                 
               
                v_status := 'N';
              END IF;
              v_act_species := v_taxon_species_number;
            END IF;
            -- Now Check the Results level - Species is Never required at the results level.
            IF v_char_result_type = 'TAXA' THEN
              v_allow_species := check_allow_species(v_char_is_nbr);
              IF v_allow_species != 1
                 AND v_char_species_number IS NOT NULL THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                    'Species Number ignored. The Taxa Characteristic is already identified to the species level.');
                 
               
                v_char_species_number := NULL;
              END IF;
              v_result_species := v_char_species_number;
            ELSE
              v_result_species := NULL;
            END IF;
          END IF;
        
          IF v_brg_type = 'Single Taxon Frequency Classes' THEN
            IF v_brg_freq_analysis = 'P'
               AND v_new_group = 1 THEN
              v_success := sim_import_result_checks.check_class_descriptor(v_char_is_nbr, v_import_log_seq);
              IF v_success = 0 THEN
                
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Characteristic must be a Common Class Descriptor for Single Taxon Frequency Classes.');
                 
               
                v_status := 'N';
              END IF;
            ELSIF v_brg_freq_analysis = 'B' THEN
              IF v_char_is_nbr IS NOT NULL THEN
                v_char_is_nbr := NULL;
                v_char_org_id := NULL;
              END IF;
            END IF;
          ELSIF v_brg_type = 'Single Taxon Group Summary'
                OR v_brg_type = 'Single Taxon Individuals'
                AND v_new_group = 1 THEN
            v_success := sim_import_result_checks.check_class_descriptor(v_char_is_nbr, v_import_log_seq);
            IF v_success = 0 THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Characteristic must be of category BIOLOGICAL/PHYSICAL MEASUREMENT for Single Taxon Results Groups.');
               
             
              v_status := 'N';
            END IF;
          END IF;
        
          -- jah 5-13-04 Added Trap and trawl warnings
          -- Trap and trawl warnings
        
          -- New Activity?
          IF v_new_activity = 1 THEN
            -- if yes
          
            ------------------
            -- Distance Fished
            ------------------
          
            -- Does import contain Distance Fished
            -- but the required field Fished Duration missing?
            IF v_distance_fished IS NOT NULL
               AND v_fished_duration IS NULL THEN
              -- if yes            
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Distance Fished ignored. ' ||
                                                  'It only applies when Fished Duration is given.');
               
             
            END IF;
          
            --------------------------
            -- Relative Wind Direction
            -------------------------- 
          
            -- Does import contain Relative Wind Direction
            -- but the required field, either Fished Duration
            -- or Trap or Net Sampling Duration, is missing?
            IF (v_relative_wind IS NOT NULL)
               AND (v_fished_duration IS NULL)
               AND (v_trap_net_duration IS NULL) THEN
              -- if yes            
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Relative Wind Direction ignored. ' ||
                                                 'It only applies when ' ||
                                                 'either Fished Duration or Trap or Net Sampling Duration is given.');
               
             
            END IF;
          
            -----------------------------
            -- Relative Current Direction
            -----------------------------
          
            -- Does import contain Relative Current Direction        
            -- but the required field, either Fished Duration
            -- or Trap or Net Sampling Duration, is missing?
            IF v_relative_current IS NOT NULL
               AND (v_fished_duration IS NULL)
               AND (v_trap_net_duration IS NULL) THEN
              -- if yes            
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Relative Current Direction ignored. ' ||
                                                   'It only applies when ' ||
                                                   'either Fished Duration or Trap or Net Sampling Duration is given.');
               
             
            END IF;
          
            -------------------------  
            -- Orientation to Current
            -------------------------
          
            -- Does import contain Orientation to Current
            -- but the required field, Trap or Net Sampling Duration, is missing?      
            IF (v_orient_to_cur IS NOT NULL)
               AND (v_trap_net_duration IS NULL) THEN
              -- if yes            
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Relative Orientation to Current ignored. ' ||
                                                 'It only applies when ' ||
                                                 'Trap or Net Sampling Duration is given.');
               
             
            END IF;
          
            -----------------------        
            -- Trap or Net Comments
            -----------------------
            IF (v_trap_net_comment IS NOT NULL)
               AND (v_trap_net_duration IS NULL) THEN
              -- if yes            
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Trap or Net Comments ignored ' ||
                                                   'It only applies when ' ||
                                                   'Trap or Net Sampling Duration is given.');
               
             
            END IF;
          
          END IF;
        
        END IF; -- End of Biological Only Checks
      
        /* check all units of measure */
        IF v_smpl_wt IS NOT NULL
           AND v_total_wt_uom IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Total Sample Weight Units required when Total Sample Weight is given');
           
         
          v_status := 'N';
        END IF;
      
        IF v_smpl_wt IS NULL
           AND v_total_wt_uom IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Total Sample Weight Units ignored. It applies only when Total Sample Weight is given.');
           
         
          v_total_wt_uom := NULL;
        END IF;
      
        IF v_depth IS NOT NULL
           AND v_depth_uom IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Depth Unit is required if depth has been provided.');
           
         
        END IF;
      
        IF v_depth IS NULL
           AND v_depth_uom IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Activity Depth Units ignored. It applies only when Activity Depth is given.');
           
         
          v_depth_uom := NULL;
        END IF;
      
        IF (v_low_depth IS NOT NULL OR v_up_depth IS NOT NULL)
           AND v_depth_range_uom IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Upper/Lower Depth Unit is required if Upper/Lower Depth is provided.');
           
         
        END IF;
      
        IF (v_low_depth IS NULL AND v_up_depth IS NULL)
           AND v_depth_range_uom IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Upper/Lower Depth Units ignored. It applies only when Upper/Lower Depth is given.');
           
         
          v_depth_range_uom := NULL;
        END IF;
      
        IF v_fished_duration IS NOT NULL
           AND v_fished_dur_uom IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Fished Duration Unit is required when Fished Duration is provided.');
           
         
        END IF;
      
        IF v_fished_duration IS NULL
           AND v_fished_dur_uom IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Fished Duration Units ignored. It applies only when Fished Duration is given.');
           
         
          v_fished_dur_uom := NULL;
        END IF;
      
        IF v_distance_fished IS NOT NULL
           AND v_dist_fished_uom IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Distance Fished Unit is required when Distance Fished is provided.');
           
         
       END IF;
      
        IF v_distance_fished IS NULL
           AND v_dist_fished_uom IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Distance Fished Units ignored. It applies only when Distance Fished is given.');
           
         
          v_dist_fished_uom := NULL;
        END IF;
      
        IF (v_lower_bnd_amt IS NOT NULL OR v_upper_bnd_amt IS NOT NULL)
           AND v_uom_is_nbr IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Common Class Unit is required when Upper and Lower Bounds are given.');
           
         
          v_status := 'N';
        END IF;
      
        IF (v_lower_bnd_amt IS NULL OR v_upper_bnd_amt IS NULL)
           AND v_brg_freq_analysis = 'P' THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Upper and Lower Bounds are required for Physical Frequency Analyses.');
           
         
          v_status := 'N';
        END IF;
      
        IF v_trap_net_duration IS NOT NULL
           AND v_trap_net_dur_unit IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Trap/Net Sampling Duration Unit is required when Trap/Net Sampling Duration is given.');
           
         
          v_status := 'N';
        END IF;
      
        IF v_trap_net_duration IS NULL
           AND v_trap_net_dur_unit IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Trap/Net Sampling Duration Units ignored. It applies only when Trap/Net Sampling Duration is given.');
           
         
          v_trap_net_dur_unit := NULL;
        END IF;
      
        /* check sequence number and point type */
        IF v_sequence_number IS NULL
           AND v_well_pipe_num IS NULL
           AND v_point_type IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Sequence Number or Well/Pipe ID is required when Point Type is provided.');
           
         
          v_point_type := NULL;
          v_status := 'N';
        END IF;
      
        --09/13/05 -RWK - Changed -1 to String
        IF (v_sequence_number IS NOT NULL AND v_sequence_number != '-1')
           AND v_point_type IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Point Type is required when Sequence Number is provided.');
           
         
          v_status := 'N';
          v_sequence_number := NULL;
        END IF;
      
        IF v_point_type = 'WELL HEAD'
           OR v_point_type = 'PIPE END' THEN
          IF v_well_pipe_num IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Well or Pipe Number is required when Point Type is WELL HEAD or PIPE END.');
             
           
            v_point_type := NULL;
            v_sequence_number := NULL;
            v_status := 'N';
          END IF;
        END IF;
      
        /* Make sure both upper and lower bounds are given */
        IF v_upper_bnd_amt IS NULL
           AND v_lower_bnd_amt IS NOT NULL THEN
           
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Upper Bound is required when Lower Bound is given.');
             
           
          v_status := 'N';
        END IF;
      
        IF v_upper_bnd_amt IS NOT NULL
           AND v_lower_bnd_amt IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Lower Bound is required when Upper Bound is given.');
           
         
          v_status := 'N';
        END IF;
      
        IF v_new_activity = 1 THEN
          /* Check times and dates on new activities */
          IF v_start_time IS NOT NULL THEN
            IF v_start_date IS NULL THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Activity Start Date is required when Activity Start Time given.');
               
             
              v_status := 'N';
            END IF;
            IF v_start_time_zone IS NULL THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Activity Start Time Zone is required when Activity Start Time is given.');
               
             
              v_status := 'N';
            END IF;
          END IF;
          IF v_end_time IS NOT NULL THEN
            IF v_end_date IS NULL THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Activity Stop Date is required when Stop Time given.');
               
             
              v_status := 'N';
            END IF;
            IF v_stop_time_zone IS NULL THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Activity Stop Time Zone is required when Activity Stop Time is given.');
               
             
              v_status := 'N';
            END IF;
          END IF;
        
          --does the visit start after the trip?
          IF v_visit_start_date < v_trip_start_date THEN
            
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Visit Start Date is before Trip Start Date.');
             
           
            v_status := 'N';
          END IF;
        
          --does the trip start before the visit ends?
          IF v_visit_end_date IS NOT NULL
             AND v_visit_end_date < v_trip_start_date THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Trip starts after the Visit End Date.');
             
           
            v_status := 'N';
          END IF;
        
          --does the Activity start before the visit starts?
          IF v_start_date < v_visit_start_date THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Field Activity starts before the Station Visit Start Date.');
             
           
            v_status := 'N';
          END IF;
        
          --does the Activity end before it starts?
          IF v_end_date IS NOT NULL THEN
            IF v_end_date < v_start_date THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Activity End date is before Activity Start Date.');
               
             
              v_status := 'N';
            END IF;
          
            --does the activity end after the visit ends?
            IF v_visit_end_date IS NOT NULL THEN
              IF v_end_date > v_visit_end_date THEN
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Activity End date is after the Station Visit End Date.');
                 
               
                v_status := 'N';
              END IF;
            END IF;
          
            --does the Activity start after the Visit ends?
            IF v_start_date > v_visit_end_date THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Activity Start Date is after the Station Visit End Date.');
               
             
              v_status := 'N';
            END IF;
          END IF;
        END IF;
      
        /* */
        IF v_char_is_nbr IS NULL
           AND v_brg_type = 'Multi-Taxon Population Census'
           AND v_taxon_is_nbr IS NOT NULL THEN
          v_char_is_nbr := v_taxon_is_nbr;
          v_char_org_id := v_taxon_org_id;
          v_char_result_type := get_char_result_type(v_char_is_nbr);
        
        ELSIF v_result IS NOT NULL
              AND v_char_is_nbr IS NULL
              AND v_tsrhcsc_is_nbr IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Either a characteristic or characteristic group must be provided.');
           
         
        END IF;
      
        IF v_char_result_type = 'TAXA'
           AND v_medium != 'Biological' THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'TAXA Type characteristic is not allowed on non-biological activity.');
           
         
        END IF;
      
        IF v_taxon_species_number IS NOT NULL
           AND v_taxon_is_nbr IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Subject Taxon Species Number ignored. It is not used unless Subject Taxon is given.');
           
         
          v_taxon_species_number := NULL;
        END IF;
      
        -- Check for Result Value and Units
        IF v_result IS NULL THEN
          IF v_detection = 'Detected not Quantified' THEN
            v_result := '*Present';
          ELSIF v_detection = 'Not Detected' THEN
            v_result := '*Non-detect';
          ELSIF v_detection = 'Present above Quantification' THEN
            v_result := '*Present >QL';
          ELSIF v_detection = 'Present below Quantification' THEN
            v_result := '*Present <QL';
          ELSIF v_detection = 'Not Reported' THEN
            v_result := '*Not Reported';
          END IF;
        END IF;
      
        IF v_brg_type = 'Single Taxon Frequency Classes' THEN
          IF v_class_count IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                'Frequency Class Count is required for Single Taxon Frequency Classes.');
             
           
            v_status := 'N';
          ELSE
            v_result := v_class_count;
          END IF;
          IF v_brg_freq_analysis = 'B' THEN
            /*  Hard-coded unit of measure for 'Count' for Biological Measures*/
            v_uom_is_nbr := 72;
            v_uom_org_id := 'ZZZZZZZZ';
          ELSIF v_brg_freq_analysis = 'P' THEN
            IF v_uom_is_nbr IS NULL THEN
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Common Class Descriptor Units is required.');
               
             
              v_status := 'N';
            END IF;
          END IF;
        END IF;
      
        /* Check the characteristic type, if physical or permitted value type, perform
        further checks on result.
        If the char type is text, copy result value into the result comment part of the code. */
      
        IF v_result IS NOT NULL THEN
          v_result_save := v_result;
          v_detect_check := 1;
          v_result := sim_import_result_extras.update_result(v_result, v_char_is_nbr, v_units_ok, v_result_ok, p_imp_cfg_seq, v_activity_seq, v_char_result_type, v_import_log_seq, v_line_cnt, v_detection, v_detect_check);
        
          IF v_result_ok = FALSE THEN
            v_status := 'N';
          END IF;
        
          IF v_detect_check = 0 THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                'Result Value does not match your Detection Condition and has beed modified.');
             
           
          END IF;
        
          IF v_char_result_type IN ('TEXT') THEN
            v_result_cmt := v_result_save;
            v_result_save := NULL; -- reset after each use.
          END IF;
        
        END IF;
      
        IF v_char_is_nbr IS NOT NULL THEN
          v_success := sim_import_result_checks.check_chardet_comp(
            v_char_is_nbr, 
            v_char_org_id, 
            v_anlpr_is_nbr, 
            v_uom_is_nbr, 
            v_uom_org_id, 
            v_fraction_is_nbr, 
            v_value_type, 
            v_units_ok, 
            v_brg_type, 
            v_activity_seq, 
            v_line_cnt, 
            v_import_log_seq, 
            v_char_result_type);
          IF v_success = 2 THEN
            v_status := 'N';
          END IF;
        END IF;
      
        ---------------------------------------------------------------------
        -- see if characteristic already exists on activity (in SIM)
        -- jah 2-25-04 
        --     It is okay to get duplicate results from Portable Data Loggers
        ---------------------------------------------------------------------
      
        -- do we have a result?
        IF (v_result IS NOT NULL) THEN
          -- if yes
        
          -- jah 2-25-04 
          -- is result from something other than a Portable Data Logger?
          IF (v_activity_cat != 'Portable Data Logger') THEN
            -- if yes 
          
            -- check for duplicates and issue warnings as necessary
            IF sim_import_result_checks.check_duplicate_result(v_char_is_nbr, v_activity_seq, v_brg_id, v_individual_nbr, v_brg_type, v_result, v_uom_is_nbr, v_char_species_number, v_lower_bnd_amt, v_upper_bnd_amt, v_line_cnt, v_char_result_type, p_import_log_seq) =
               FALSE THEN
            
              -- do nothing; CHECK_DUPLICATE_RESULT issues the warnings if necessary
              NULL;
            
            END IF;
          
          END IF;
        
        END IF;
      
        IF NOT v_result_ok THEN
          v_status := 'N';
        ELSIF v_result = '0' THEN
          v_result_value_msr := 0;
        END IF;
      
        /* Check for dup brgi */
        IF v_lower_bnd_amt IS NOT NULL
           OR v_primary_class IS NOT NULL THEN
          v_dup_brgi_success := sim_import_result_checks.check_dup_class(v_activity_seq, v_result_rec.rs_brg_seq, v_lower_bnd_amt, v_upper_bnd_amt, v_line_cnt, v_import_log_seq);
          IF v_dup_brgi_success = 0 THEN
            -- dupe class found
            v_status := 'N';
            
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                'A duplicate class exists for this group - duplicate result or invalid class bounds likely');
             
           
          END IF;
        END IF;
      
        /* make sure the TSMALP record exists */
        IF v_new_activity = 1
           AND v_tsmstatn_is_nbr IS NOT NULL THEN
          --AND v_sequence_number != -1 THEN
          IF v_point_type IS NULL THEN
            --default the AAL to Point of Record and, if app, Trawl Start/Stop
            v_point_type := '*POINT OF RECORD';
            v_sequence_number := '0'; -- 09/13/05 -RWK - Changed 0 to String
          END IF;
          --go get the tsmalp is number
          IF nvl(v_sequence_number, '1') != '-1' THEN
            --09/13/05 -RWK - Changed -1 to String
            IF v_point_type != 'WELL HEAD'
               AND v_point_type != 'PIPE END' THEN
              --would already have TSMALP info if WELL OR PIPE
              v_storet_key := sim_import_result_extras.get_tsmalp_is_nbr(v_point_type, v_sequence_number, v_tsmstatn_is_nbr, v_tsmstatn_org_id, v_line_cnt, v_import_log_seq);
              IF v_storet_key.is_number IS NOT NULL THEN
                v_tsmalp_is_nbr := v_storet_key.is_number;
                v_tsmalp_org_id := v_storet_key.org_id;
              ELSE
                v_status := 'N';
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'A tsmalp location record was not found ');
                 
               
              END IF;
            END IF;
          END IF;
        END IF;
      
        /* Check all the result stuff */
        IF v_result IS NOT NULL THEN
        
          --  We do not support Upper and Lower Range checking in SIM v2.0.1.  Default values for
          --  Physical Characteristics to NULL and all else to zero to match the STORET interface.
          IF v_char_result_type = 'PHYS' THEN
            v_upper_range := NULL;
            v_lower_range := NULL;
          ELSE
            -- handle ranges when groups and rows
            v_upper_range := v_acode_defs_rec.upper_range_value;
            v_lower_range := v_acode_defs_rec.lower_range_value;
          
            -- be sure that a value gets set even if v_acode_defs_rec.upper_range_value
            -- is null
            IF v_upper_range IS NULL THEN
              v_upper_range := 0;
            END IF;
            IF v_lower_range IS NULL THEN
              v_lower_range := 0;
            END IF;
          
          END IF;
        
          IF v_char_result_type IN ('TEXT', 'PV  ') THEN
            -- DCL added this to keep extraneous data from being imported if the result is text or pv
            IF v_detect_limit IS NOT NULL THEN
              v_detect_limit := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Detection Limit not needed for choice list or text type results.');
               
             
            END IF;
            IF v_lab_cert IS NOT NULL THEN
              v_lab_cert := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Lab Certification not needed for choice list or text type results.');
               
             
            END IF;
            IF v_analysis_date IS NOT NULL THEN
              v_analysis_date := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Analysis Date not needed for choice list or text type results.');
               
             
            END IF;
            IF v_analysis_time IS NOT NULL THEN
              v_analysis_time := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Analysis Time not needed for choice list or text type results.');
               
             
            END IF;
            IF v_det_uom_is_nbr IS NOT NULL THEN
              v_det_uom_is_nbr := NULL;
              v_det_uom_org_id := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Detection limit UOM not needed for choice list or text type results.');
               
             
            END IF;
            IF v_det_comment IS NOT NULL THEN
              v_det_comment := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Detect Comment not needed for choice list or text type results.');
               
             
            END IF;
            IF v_upq_limit IS NOT NULL THEN
              v_upq_limit := NULL;
              
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Upper Quantificatin Limit not needed for choice list or text type results.');
               
             
            END IF;
            IF v_lowq_limit IS NOT NULL THEN
              v_lowq_limit := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Lower Quantification Limit not needed for choice list or text type results.');
               
             
            END IF;
            IF v_uom_is_nbr IS NOT NULL THEN
              v_uom_is_nbr := NULL;
              v_uom_org_id := NULL;
             END IF;
            IF v_fraction IS NOT NULL THEN
              v_fraction := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Fraction not needed for choice list or text type results.');
               
             
            END IF;
            IF v_value_type IS NOT NULL THEN
              v_value_type := NULL;
            END IF;
            IF v_stat_type IS NOT NULL THEN
              v_stat_type := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Statistic Type not needed for choice list or text type results.');
               
             
            END IF;
            IF v_anlpr_is_nbr IS NOT NULL THEN
              v_anlpr_is_nbr := NULL;
              v_anlpr_org_id := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Analysis Procedure not needed for choice list or text type results.');
               
             
            END IF;
            IF v_labprep_is_nbr IS NOT NULL THEN
              v_labprep_is_nbr := NULL;
              v_labprep_org_id := NULL;
              
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Lab Procedure not needed for choice list or text type results.');
               
             
            END IF;
            IF v_duration_basis IS NOT NULL THEN
              v_duration_basis := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Duration Basis not needed for choice list or text type results.');
               
             
            END IF;
            IF v_temp_basis IS NOT NULL THEN
              v_temp_basis := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Temperature Basis not needed for choice list or text type results.');
               
             
            END IF;
            IF v_precision IS NOT NULL THEN
              v_precision := NULL;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Precision not needed for choice list or text type results.');
               
             
            END IF;
            IF v_conf_level IS NOT NULL THEN
              v_conf_level := NULL;
                
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                  'Confidence Level not needed for choice list or text type results.');
               
             
            END IF;
          END IF;
        END IF;
      
        /* Make sure no replicate number for trip QC. */
        IF v_trip_qc = 1
           AND v_replicate_nbr IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                              'Replicate Number ignored. It is not used for Trip QC.');
           
         
          v_replicate_nbr := NULL;
        END IF;
      
        /*Check for projects */
        IF v_no_projects IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Project is required and has not been provided.');
           
         
        END IF;
      
        ---------------------------------------------------------------------------------
        --
        --
        --  create activity
        --
        --
        ---------------------------------------------------------------------------------
      
        BEGIN
          IF v_new_activity = 1 THEN
            IF v_status = 'N' THEN
              v_approved_on := NULL;
              v_approved_by := NULL;
            ELSE
              v_approved_on := SYSDATE;
              v_approved_by := USER;
            END IF;
            IF v_activity_cat = 'Created from Sample' THEN
              -- get rid of unwanted activity items
              v_depth := NULL;
              v_depth_uom := NULL;
              v_depth_ref := NULL;
              v_depth_ref_seq := NULL;
              v_depth_ref_unit := NULL;
              v_depth_range_uom := NULL;
              v_depth_ref := NULL;
              v_depth_zone := NULL;
              v_up_depth := NULL;
              v_low_depth := NULL;
              v_thermo := NULL;
              v_halo := NULL;
              v_pycno := NULL;
            END IF;
            /* assign values to record in order to send all of these variables to function easily */
            v_fa_rec.fa_seq := v_activity_seq;
            v_fa_rec.fa_type_name := v_type;
            v_fa_rec.fa_category := v_activity_cat;
            v_fa_rec.fa_activity_id := v_activity_id;
            v_fa_rec.fa_start_date := v_start_date;
            v_fa_rec.fa_start_time := v_start_time;
            v_fa_rec.fa_start_time_zone := v_start_time_zone;
            v_fa_rec.fa_stop_date := v_end_date;
            v_fa_rec.fa_stop_time := v_end_time;
            v_fa_rec.fa_stop_time_zone := v_stop_time_zone;
            v_fa_rec.fa_depth := v_depth;
            v_fa_rec.fa_depth_ref_point := v_depth_ref;
            v_fa_rec.fa_activity_comments := v_comment;
            v_fa_rec.fa_tsmstatn_is_number := v_tsmstatn_is_nbr;
            v_fa_rec.fa_tsmstatn_org_id := v_tsmstatn_org_id;
            v_fa_rec.fa_tsrtrip_is_number := v_trip_is_nbr;
            v_fa_rec.fa_tsrtrip_org_id := v_trip_org_id;
            v_fa_rec.fa_tsrstvst_is_number := v_tsrstvst_is_nbr;
            v_fa_rec.fa_tsrstvst_org_id := v_tsrstvst_org_id;
            v_fa_rec.fa_tsrfldpr_is_number := v_proc_is_nbr;
            v_fa_rec.fa_tsrfldpr_org_id := v_proc_org_id;
            v_fa_rec.fa_tsrfldgr_is_number := v_gear_is_nbr;
            v_fa_rec.fa_tsrfldgr_org_id := v_gear_org_id;
            v_fa_rec.fa_import_seq := v_import_log_seq;
            v_fa_rec.fa_export_seq := NULL;
            v_fa_rec.fa_status := v_status;
            v_fa_rec.fa_tsmorgan_is_number := v_org_is_nbr;
            v_fa_rec.fa_tsmorgan_org_id := p_org_id;
            v_fa_rec.fa_approved_by := v_approved_by;
            v_fa_rec.fa_approved_on := v_approved_on;
            v_fa_rec.fa_stvst_comments := v_stvst_cmt;
            v_fa_rec.fa_depth_uom := v_depth_uom;
            v_fa_rec.fa_medium := v_medium;
            v_fa_rec.fa_tsrlab_is_number := v_tsrlab_is_nbr;
            v_fa_rec.fa_tsrlab_org_id := v_tsrlab_org_id;
            v_fa_rec.fa_import_line_no := v_line_cnt;
            v_fa_rec.fa_tsrfdact_is_number := v_act_is_nbr;
            v_fa_rec.fa_tsrfdact_org_id := v_act_org_id;
            v_fa_rec.fa_replicate_nbr := v_replicate_nbr;
            v_fa_rec.fa_fdset_id := v_field_set_nbr;
            v_fa_rec.fa_fdset_name := v_field_set_name;
            v_fa_rec.fa_updepth := v_up_depth;
            v_fa_rec.fa_lowdepth := v_low_depth;
            v_fa_rec.fa_thermo := substr(v_thermo, 1, 1);
            v_fa_rec.fa_halo := substr(v_halo, 1, 1);
            v_fa_rec.fa_pycno := substr(v_pycno, 1, 1);
            v_fa_rec.fa_rel_depth := v_rel_depth;
            v_fa_rec.fa_depth_zone := v_depth_zone;
            v_fa_rec.fa_depth_range_uom := v_depth_range_uom;
            v_fa_rec.fa_tsrgrcfg_is_number := v_gearcfg_is_nbr;
            v_fa_rec.fa_tsrgrcfg_org_id := v_gearcfg_org_id;
            v_fa_rec.fa_tsrsdp_is_number := v_sample_pres_is_nbr;
            v_fa_rec.fa_tsrsdp_org_id := v_sample_pres_org_id;
            v_fa_rec.fa_stp_seq := v_trip_seq;
            v_fa_rec.fa_ssv_seq := v_visit_seq;
            v_fa_rec.fa_intent := v_intent;
            v_fa_rec.fa_tsmprmvl_community_is_nbr := v_community_is_nbr;
            v_fa_rec.fa_tsmprmvl_community_org_id := v_community_org_id;
            v_fa_rec.fa_tsrbiopt_is_number := v_biopart_is_nbr;
            v_fa_rec.fa_tsrbiopt_org_id := v_biopart_org_id;
          
            IF v_intent IN ('Individual', 'Tissue') THEN
              v_fa_rec.fa_tsrchar_is_number := v_taxon_is_nbr;
              v_fa_rec.fa_tsrchar_org_id := v_taxon_org_id;
            ELSE
              v_fa_rec.fa_tsrchar_is_number := NULL;
              v_fa_rec.fa_tsrchar_org_id := NULL;
            END IF;
          
            -- New Stuff for V2
            v_fa_rec.fa_tsrmatrx_is_number := v_tsrmatrx_is_number;
            v_fa_rec.fa_tsrmatrx_org_id := v_tsrmatrx_org_id;
            v_fa_rec.fa_tsropprd_org_id := NULL;
            v_fa_rec.fa_tsropprd_is_number := NULL;
            v_fa_rec.fa_qc_indicator := v_qc_indicator;
            v_fa_rec.fa_chain_of_custody_id := v_chain_of_custody;
            v_fa_rec.fa_species_number := v_act_species;
            v_fa_rec.fa_char_group := v_group;
            v_fa_rec.fa_document := v_activity_document;
          
            /* insert field activity */
            v_success := sim_import_result_extras.create_field_activity(v_fa_rec, v_import_log_seq);
            IF v_success != 1 THEN
              UPDATE sim_field_activities
                 SET fa_status = 'N'
               WHERE fa_seq = v_activity_seq;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Failed to create activity in sim_field_activities, activity will not be exported.');
               
             
            END IF;
          END IF;
        EXCEPTION
          WHEN OTHERS THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                SQLERRM || ' Line ' ||
                                                 v_line_cnt ||
                                                 ': in create activity.');
             
           
        END;
      
        BEGIN
        
          IF v_pre_check = 0 THEN
          
            IF v_new_activity = 1 THEN
            
              -- add projects to activity
              BEGIN
                IF v_no_projects IS NOT NULL THEN
                  FOR v_number IN 1 .. v_no_projects LOOP
                    IF v_proj_is_table(v_number) IS NOT NULL THEN
                      v_proj_is_nbr := v_proj_is_table(v_number);
                      v_proj_org_id := v_proj_org_table(v_number);
                      v_success := add_project_to_activity(v_activity_seq, v_proj_is_nbr, v_proj_org_id);
                    END IF;
                  END LOOP;
                ELSE
                  v_status := 'N';
                  UPDATE sim_field_activities
                     SET fa_status = 'N'
                   WHERE fa_seq = v_activity_seq;
                END IF;
              EXCEPTION
                WHEN no_data_found THEN
                  NULL;
                WHEN OTHERS THEN
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      SQLERRM ||
                                                      ' - adding projects to activity.');
                   
                 
              END;
            
              -- add personnel to activity
              BEGIN
                IF v_no_person > 0 THEN
                  FOR v_number IN 1 .. v_no_person LOOP
                    IF v_persn_is_table(v_number) IS NOT NULL THEN
                      v_tsmpersn_is_nbr := v_persn_is_table(v_number);
                      v_tsmpersn_org_id := v_persn_org_table(v_number);
                      v_success := add_personnel_to_activity(v_activity_seq, v_tsmpersn_is_nbr, v_tsmpersn_org_id);
                    END IF;
                  END LOOP;
                END IF;
              EXCEPTION
                WHEN no_data_found THEN
                  NULL;
                WHEN OTHERS THEN
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      SQLERRM ||
                                                      ' - adding personnel to activity.');
                   
                 
              END;
            
              -- add cooperating organizations to activity
              BEGIN
                -- are there any cooperating organizations?
                IF v_no_cporgs > 0 THEN
                  -- if yes
                  -- loop through the cooperating organizations array
                  FOR v_number IN 1 .. v_no_cporgs LOOP
                    -- is there a cooperating organization in this element of the array?
                    IF v_cporg_is_table(v_number) IS NOT NULL THEN
                      -- if yes
                      -- set the STORET organization number and id
                      v_tsmcporg_is_nbr := v_cporg_is_table(v_number);
                      v_tsmcporg_org_id := v_cporg_org_table(v_number);
                      -- insert the cooperating organization record
                      v_success := add_cporg_to_activity(v_activity_seq, v_tsmcporg_is_nbr, v_tsmcporg_org_id);
                    END IF;
                  END LOOP;
                END IF;
              EXCEPTION
                WHEN no_data_found THEN
                  NULL;
                WHEN OTHERS THEN
                  
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      SQLERRM ||
                                                     ' - adding cooperating organization to activity.');
                   
                 
              END;
            
              BEGIN
              
                -- Create the parent records for the activity
                FOR v_number IN 1 .. nvl(v_no_parent_ids, 0) LOOP
                
                  -- get the next sequence number
                  SELECT sim_field_activity_parent_seq.NEXTVAL
                    INTO v_fap_seq
                    FROM dual;
                
                  -- put parent information in a record
                  v_fap_rec.fap_seq := v_fap_seq;
                  v_fap_rec.fap_fa_seq := v_fa_rec.fa_seq;
                  v_fap_rec.fap_parent_fa_seq := v_parent_fa_seq_table(v_number);
                  v_fap_rec.fap_parent_is_number := v_parent_is_number_table(v_number);
                  v_fap_rec.fap_parent_org_id := rpad(v_parent_org_id_table(v_number), 8);
                
                  -- insert the record into SIM_FIELD_ACTIVITY_PARENT
                  v_error_code := sim_import_result_extras.create_field_activity_parent(v_fap_rec, v_import_log_seq);
                
                  -- did an error occur while inserting trying to insert the record?
                  IF (v_error_code IS NOT NULL) THEN
                    -- if yes
                  
                    -- set the status flag so the activity cannot not migrate
                    UPDATE sim_field_activities
                       SET fa_status = 'N'
                     WHERE fa_seq = v_activity_seq;
                  
                    -- output error message
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                        'Failed to create parent in SIM_FIELD_ACTIVITY_PARENT -- ');
                     
                   
                  END IF;
                
                END LOOP;
              EXCEPTION
                WHEN no_data_found THEN
                  NULL;
                WHEN OTHERS THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      SQLERRM ||
                                                      ' - adding Parent Sample IDs to activity.');
                   
                 
              END;
            
            END IF;
          
            /*  THIS WILL CREATE SIM_ACTUAL_ACTIVITY_LOC RECORDS */
            IF v_new_activity = 1
               AND v_tsmstatn_is_nbr IS NOT NULL THEN
              --AND v_sequence_number != -1 THEN
              IF nvl(v_sequence_number, '1') != '-1' THEN
                --09/13/05 -RWK - Changed -1 to String
                OPEN c_get_location_info;
                FETCH c_get_location_info
                  INTO v_stl_seq;
                CLOSE c_get_location_info;
                v_point_type := NULL; --reset it for the next row
                v_sequence_number := NULL; --reset it for the next row
              
                -- create a single aal record */
                OPEN c_get_next_aal_seq;
                FETCH c_get_next_aal_seq
                  INTO v_aal_rec.aal_seq;
                CLOSE c_get_next_aal_seq;
              
                /* assign values to aal general record */
                v_aal_rec.aal_seq := v_aal_rec.aal_seq;
                v_aal_rec.aal_fa_seq := v_activity_seq;
                v_aal_rec.aal_stl_seq := v_stl_seq;
                v_aal_rec.aal_tsmalp_is_number := v_tsmalp_is_nbr;
                v_aal_rec.aal_tsmalp_org_id := v_tsmalp_org_id;
                v_aal_rec.aal_tsraal_type_name := 'General';
                v_aal_rec.aal_bottom_depth_msr := NULL;
                v_aal_rec.aal_bottom_depth_uom := NULL;
                v_aal_rec.aal_addtnl_loc_info := v_add_loc_info;
              
                /* insert a general aal record */
                v_success := sim_import_result_extras.create_aal(v_aal_rec, v_import_log_seq);
                IF v_success != 1 THEN
                  -- invalidate activity if create_aal fails
                  UPDATE sim_field_activities
                     SET fa_status = 'N'
                   WHERE fa_seq = v_activity_seq;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Failed to create "General" record in sim_actual_activity_loc, activity will not be exported.');
                   
                 
                END IF;
                --END IF;
              END IF;
            END IF;
          
            /* THIS WILL CREATE A RECORD FOR SIM_TRAWL_OPS_DET  */
            -- jah 5-13-04
            -- Modified the following code to create 
            -- either a SIM_TRAWL_OPS_DET record or 
            -- a SIM TRAP_NET record, but not both.
          
            -- Do we need to create a SIM_TRAWL_OPS_DET record or a SIM TRAP_NET record?
            IF v_activity_cat != 'Created from Sample'
               AND v_new_activity = 1 THEN
              -- if yes
            
              -- Do we need to create a SIM_TRAWL_OPS_DET record?
              IF (v_fished_duration IS NOT NULL)
                 AND (v_check_trawl_gear = 1) THEN
                -- if yes
              
                -- get next sequence number
                OPEN c_get_next_tod_seq;
                FETCH c_get_next_tod_seq
                  INTO v_tod_rec.tod_seq;
                CLOSE c_get_next_tod_seq;
              
                -- assign values
                v_tod_rec.tod_fa_seq := v_activity_seq;
                v_tod_rec.tod_fished_duration := v_fished_duration;
                v_tod_rec.tod_fished_dur_uom := v_fished_dur_uom;
                v_tod_rec.tod_boat_speed_msr := v_boat_speed;
                v_tod_rec.tod_boat_speed_uom := v_boat_speed_uom;
                v_tod_rec.tod_fished_distance := v_distance_fished;
                v_tod_rec.tod_fished_distance_uom := v_dist_fished_uom;
                v_tod_rec.tod_rel_current_dir := v_relative_current;
                v_tod_rec.tod_rel_wind_dir := v_relative_wind;
              
                -- Attempt to create a SIM_TRAWL_OPS_DET record
                v_success := sim_import_result_extras.create_tod(v_tod_rec, v_import_log_seq);
              
                -- Was attempt to create a SIM_TRAWL_OPS_DET record successful?
                IF v_success != 1 THEN
                  -- if no
                  UPDATE sim_field_activities
                     SET fa_status = 'N'
                   WHERE fa_seq = v_activity_seq;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Failed to create record in sim_trawl_ops_det, ' ||
                                                      'activity will not be exported.');
                   
                 
                END IF;
              
              END IF;
            
              -- Do we need to create a SIM TRAP_NET record
              IF (v_trap_net_duration IS NOT NULL)
                 AND (v_check_trawl_gear = 2) THEN
                -- if yes
              
                -- get next sequence number
                OPEN c_get_trap_net_seq;
                FETCH c_get_trap_net_seq
                  INTO v_stn_seq;
                CLOSE c_get_trap_net_seq;
              
                -- assign values
                v_stn_rec.stn_seq := v_stn_seq;
                v_stn_rec.stn_tsrfdact_is_number := NULL;
                v_stn_rec.stn_tsrfdact_org_id := NULL;
                v_stn_rec.stn_sampling_dur := v_trap_net_duration;
                v_stn_rec.stn_sampling_dur_unit := v_trap_net_dur_unit;
                v_stn_rec.stn_orient_to_cur := v_orient_to_cur;
                v_stn_rec.stn_rel_current := v_relative_current;
                v_stn_rec.stn_rel_wind := v_relative_wind;
                v_stn_rec.stn_comment := v_trap_net_comment;
                v_stn_rec.stn_fa_seq := v_activity_seq;
              
                -- Attempt to create a SIM_TRAP_NET record
                v_success := sim_import_result_extras.create_trap_net(v_stn_rec, v_import_log_seq);
              
                -- Was attempt to create a SIM_TRAP_NET record successful?
                IF v_success != 1 THEN
                  -- if no
                  UPDATE sim_field_activities
                     SET fa_status = 'N'
                   WHERE fa_seq = v_activity_seq;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      'Failed to create record in sim_trap_net, ' ||
                                                      'activity will not be exported.');
                   
                 
                END IF;
              
              END IF;
            
            END IF;
          
            /* THIS WILL CREATE A RECORD FOR SIM_SAMPLE*/
            IF v_type = 'Sample'
               AND v_new_activity = 1 THEN
              IF v_sample_pres_is_nbr IS NOT NULL
                 AND v_sample_pres_org_id IS NOT NULL THEN
                OPEN c_get_tsrsdp_values(v_sample_pres_is_nbr, v_sample_pres_org_id);
                FETCH c_get_tsrsdp_values
                  INTO v_tsrsdp_rec;
                CLOSE c_get_tsrsdp_values;
              END IF;
              OPEN c_get_next_sample_seq;
              FETCH c_get_next_sample_seq
                INTO v_sample_seq;
              CLOSE c_get_next_sample_seq;
            
              /* assign values to record and send to create_sample function */
              v_sample_rec.ss_seq := v_sample_seq;
              v_sample_rec.ss_fa_seq := v_activity_seq;
              v_sample_rec.ss_total_volume_msr := v_ttl_smpl_vol;
              v_sample_rec.ss_total_vol_uom := v_smpl_vol_unit_seq;
              v_sample_rec.ss_total_weight_msr := v_smpl_wt;
              v_sample_rec.ss_total_wt_uom := v_total_wt_uom;
              v_sample_rec.ss_data_file_name_loc := v_data_file_name_loc;
              v_sample_rec.ss_container_type_nm := v_tsrsdp_rec.container_type_nm;
              v_sample_rec.ss_container_color := v_tsrsdp_rec.container_color;
              v_sample_rec.ss_container_size_msr := v_tsrsdp_rec.container_size_msr;
              v_sample_rec.ss_container_size_un := v_tsrsdp_rec.container_size_un;
              v_sample_rec.ss_temp_preservn_type := v_tsrsdp_rec.temp_presrv_type;
              v_sample_rec.ss_parent_ind_code := v_parent_ind_code;
              v_sample_rec.ss_gear_deployment := v_gear_deployment;
              v_sample_rec.ss_transport_storage := v_transport_storage;
              v_sample_rec.ss_presrv_strge_prcdr := v_tsrsdp_rec.presrv_strge_prcdr;
              -- We do not support sampling duration in SIM v2.0.1.  Default values 
              --  to zero for time and null for unit to be consistent with the 
              --  STORET interface.
              v_sample_rec.ss_duration_time := 0;
              v_sample_rec.ss_duration_unit := NULL;
            
              /* insert sample record */
              v_success := sim_import_result_extras.create_sample(v_sample_rec, v_import_log_seq);
            
              IF v_success != 1 THEN
                -- invalidate activity if create_sample fails
                UPDATE sim_field_activities
                   SET fa_status = 'N'
                 WHERE fa_seq = v_activity_seq;
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                    'Failed to create record in sim_sample, activity will not be exported.');
                 
               
              END IF;
            END IF;
          END IF; --stuff we don't do for pre-check
        
          /* THIS WILL CREATE A RECORD FOR SIM_BIO_RESULTS_GRP */
          IF v_brg_id IS NOT NULL
             AND v_new_group = 1 THEN
            OPEN c_get_next_brg_seq;
            FETCH c_get_next_brg_seq
              INTO v_brg_seq;
            CLOSE c_get_next_brg_seq;
            /* assign values to record and send to create_brg function */
            v_brg_rec.brg_seq := v_brg_seq;
            v_brg_rec.brg_fa_seq := v_activity_seq;
            v_brg_rec.brg_id_code := v_brg_id;
            v_brg_rec.brg_type := v_brg_type;
            v_brg_rec.brg_freq_analysis := v_brg_freq_analysis;
            v_brg_rec.brg_tsrcpv_sex_is_number := v_sexname_is_nbr;
            v_brg_rec.brg_tsrcpv_sex_org_id := v_sexname_org_id;
            v_brg_rec.brg_tsrcpv_lifestage_is_nbr := v_lifestage_is_nbr;
            v_brg_rec.brg_tsrcpv_lifestage_org_id := v_lifestage_org_id;
            v_brg_rec.brg_value_type_name := v_value_type_name;
            v_brg_rec.brg_species_number := v_brg_species;
            v_brg_rec.brg_description_text := v_brg_text;
            IF v_brg_type IN
               ('Multi-Taxon Population Census', 'Single Taxon Frequency Classes') THEN
              v_brg_rec.brg_summary_grp_count := NULL;
            ELSIF v_brg_type IN ('Single Taxon Individuals') THEN
              v_brg_rec.brg_summary_grp_count := v_ttl_nbr_indiv;
            ELSE
              v_brg_rec.brg_summary_grp_count := v_ttl_nbr_in_grp;
            END IF;
            IF v_brg_type LIKE 'Single%' THEN
              v_brg_rec.brg_tsrchar_is_number := v_taxon_is_nbr;
              v_brg_rec.brg_tsrchar_org_id := v_taxon_org_id;
            ELSE
              v_brg_rec.brg_tsrchar_is_number := NULL;
              v_brg_rec.brg_tsrchar_org_id := NULL;
            END IF;
            v_brg_rec.brg_import_seq := v_import_log_seq;
          
            /* insert brg record */
            v_success := sim_import_result_extras.create_brg(v_brg_rec, v_import_log_seq);
            IF v_success != 1 THEN
              -- invalidate activity if create_brg fails
              UPDATE sim_field_activities
                 SET fa_status = 'N'
               WHERE fa_seq = v_activity_seq;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Failed to create record in sim_bio_results_grp, activity will not be exported.');
               
             
            END IF;
          END IF;
        
          /* THIS WILL CREATE A RECORD FOR SIM_BIO_RESULTS_GRP_INDIVIDUAL */
        
          IF v_brg_id IS NOT NULL
             AND v_individual_nbr IS NOT NULL THEN
            /* get existing brg_seq (SF) don't check for duplicate individual numbers here, already did it in check_duplicate_results */
            OPEN c_find_brg_seq(v_brg_id, v_activity_seq);
            FETCH c_find_brg_seq
              INTO v_brgi_brg_seq;
            CLOSE c_find_brg_seq;
          
            v_success := sim_import_result_checks.check_dup_brgi(v_activity_seq, v_brgi_brg_seq, v_individual_nbr, v_char_is_nbr, v_line_cnt, v_import_log_seq);
          
            IF v_success = 1 THEN
              -- individual does not exist, get next brgi_seq and create record
              OPEN c_get_next_brgi_seq;
              FETCH c_get_next_brgi_seq
                INTO v_brgi_seq;
              CLOSE c_get_next_brgi_seq;
              /* assign values to record and send to create_brgi function */
              v_brgi_rec.brgi_seq := v_brgi_seq;
              v_brgi_rec.brgi_brg_seq := v_brgi_brg_seq;
              v_brgi_rec.brgi_individual_number := v_individual_nbr;
              v_brgi_rec.brgi_fa_seq := v_activity_seq;
              v_brgi_rec.brgi_import_seq := v_import_log_seq;
              /* insert brgi record */
            
              v_success := sim_import_result_extras.create_brgi(v_brgi_rec, v_import_log_seq);
            
              IF v_success != 1 THEN
                -- invalidate activity if create_brgi fails
                UPDATE sim_field_activities
                   SET fa_status = 'N'
                 WHERE fa_seq = v_activity_seq;
                
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Failed to create record in sim_bio_results_grp_individual, activity will not be exported.');
               
             
              END IF;
            END IF;
          END IF;
        
          /* create result */
          /* write data to record and send to create result function */
          /* get result sequence */
        
          IF v_result IS NOT NULL THEN
            OPEN c_get_result_seq;
            FETCH c_get_result_seq
              INTO v_result_seq;
            CLOSE c_get_result_seq;
            v_result_rec.rs_seq := v_result_seq;
            v_result_rec.rs_ac_seq := NULL;
            v_result_rec.rs_fa_seq := v_activity_seq;
            v_result_rec.rs_brg_seq := v_brg_seq;
            v_result_rec.rs_brgi_seq := v_brgi_seq;
            v_result_rec.rs_result := v_result;
            v_result_rec.rs_d_assnd_seq_num := v_result_seq_num;
            v_result_rec.rs_comment := v_result_cmt;
            v_result_rec.rs_detect_limit := v_detect_limit;
            v_result_rec.rs_import_line_no := v_line_cnt;
            v_result_rec.rs_value_status := v_result_status;
            v_result_rec.rs_lab_batch_id := v_batch_id;
            v_result_rec.rs_lab_cert := v_lab_cert;
            v_result_rec.rs_analysis_date := v_analysis_date;
            v_result_rec.rs_analysis_time := v_analysis_time;
            v_result_rec.rs_analysis_time_zone := v_analysis_time_zone;
            v_result_rec.rs_det_limit_uom_nbr := v_det_uom_is_nbr;
            v_result_rec.rs_det_limit_uom_org := v_det_uom_org_id;
            v_result_rec.rs_detect_comment := v_det_comment;
            v_result_rec.rs_upper_quant_limit := v_upq_limit;
            v_result_rec.rs_lower_quant_limit := v_lowq_limit;
          
            IF v_brg_freq_analysis = 'B' THEN
              v_result_rec.rs_tsrchar_is_number := v_bio_condition_is_nbr;
              v_result_rec.rs_tsrchar_org_id := v_bio_condition_org_id;
              v_result_rec.rs_tsrchar0is_number := v_sec_bio_cond_is_nbr;
              v_result_rec.rs_tsrchar0org_id := v_sec_bio_cond_org_id;
            ELSE
              v_result_rec.rs_tsrchar_is_number := v_char_is_nbr;
              v_result_rec.rs_tsrchar_org_id := v_char_org_id;
              v_result_rec.rs_tsrchar0is_number := NULL;
              v_result_rec.rs_tsrchar0org_id := NULL;
            END IF;
          
            IF v_brg_type = 'Single Taxon Frequency Classes' THEN
              IF v_brg_freq_analysis = 'B' THEN
                /* For STFC-Bio, the UOM of Count is stored in the TSRRCI table and nothing is stored in the TSRRSULT table. */
                v_result_rec.rs_tsruom_is_nbr := NULL;
                v_result_rec.rs_tsruom_org_id := NULL;
              ELSE
                /* For STFC-Physical, the actual UOM is stored in the TSRRCI table and Count is stored in the TSRRSULT table. */
                v_result_rec.rs_tsruom_is_nbr := 72;
                v_result_rec.rs_tsruom_org_id := 'ZZZZZZZZ';
              END IF;
            ELSE
              /* For everything else, the actual UOM is stored in the TSRRSULT table. */
              v_result_rec.rs_tsruom_is_nbr := v_uom_is_nbr;
              v_result_rec.rs_tsruom_org_id := v_uom_org_id;
            END IF;
          
            v_result_rec.rs_fraction := v_fraction;
            v_result_rec.rs_value_type := v_value_type;
            v_result_rec.rs_stat_type := v_stat_type;
            v_result_rec.rs_tsranlpr_is_number := v_anlpr_is_nbr;
            v_result_rec.rs_tsranlpr_org_id := v_anlpr_org_id;
            v_result_rec.rs_tsrlspp_is_number := v_labprep_is_nbr;
            v_result_rec.rs_tsrlspp_org_id := v_labprep_org_id;
            v_result_rec.rs_duration_basis := v_duration_basis;
            v_result_rec.rs_temp_basis := v_temp_basis;
            v_result_rec.rs_weight_basis := v_weight_basis;
            v_result_rec.rs_precision := v_precision;
            v_result_rec.rs_conf_level := v_conf_level;
            v_result_rec.rs_tsrchgrp_is_number := v_tsrchgrp_is_nbr;
            v_result_rec.rs_tsrchgrp_org_id := v_tsrchgrp_org_id;
            v_result_rec.rs_tsrchdef_is_number := v_tsrchdef_is_nbr;
            v_result_rec.rs_tsrchdef_org_id := v_tsrchdef_org_id;
            v_result_rec.rs_trophic_level := v_trophic_level;
            v_result_rec.rs_fnctional_feed_grp := v_func_feed_grp;
            v_result_rec.rs_taxon_pollution := v_taxon_pollution;
            v_result_rec.rs_tsrlab_is_number := v_tsrlab_is_nbr;
            v_result_rec.rs_tsrlab_org_id := v_tsrlab_org_id;
            v_result_rec.rs_tsrhcsc_is_number := v_tsrhcsc_is_nbr;
            v_result_rec.rs_tsrhcsc_org_id := v_tsrhcsc_org_id;
          
            --new v2 stuff:
            v_result_rec.rs_bias := v_bias;
            v_result_rec.rs_cell_shape_type_nm := v_cell_shape;
            v_result_rec.rs_cell_type_nm := v_cell_type;
            v_result_rec.rs_conf_lvl_corr_bias := v_cl_corrected;
            v_result_rec.rs_correction_ind_cd := 'N';
            v_result_rec.rs_dilution_ind_code := 'N';
            v_result_rec.rs_fraction_tsmprmvl_is_number := v_fraction_is_nbr;
            v_result_rec.rs_fraction_tsmprmvl_org_id := v_fraction_org_id;
            v_result_rec.rs_habit_tsmprmvl0is_number := v_habit_is_number;
            v_result_rec.rs_habit_tsmprmvl0org_id := v_habit_org_id;
            v_result_rec.rs_volt_tsmprmvl1is_number := v_volt_is_number;
            v_result_rec.rs_volt_tsmprmvl1org_id := v_volt_org_id;
          
            v_result_rec.rs_lower_range_value := v_lower_range;
            v_result_rec.rs_upper_range_value := v_upper_range;
            v_result_rec.rs_recovery_ind_code := 'N';
            v_result_rec.rs_ref_pt_from_name := NULL;
            v_result_rec.rs_ref_pt_to_name := NULL;
            v_result_rec.rs_repl_analysis_num := 0;
            v_result_rec.rs_species_number := v_result_species;
            v_result_rec.rs_tsrbrg_is_number := NULL;
            v_result_rec.rs_tsrbrg_org_id := NULL;
            v_result_rec.rs_tsrbrgi_is_number := NULL;
            v_result_rec.rs_tsrbrgi_org_id := NULL;
            v_result_rec.rs_tsrdlin_is_number := NULL;
            v_result_rec.rs_tsrdlin_org_id := NULL;
            v_result_rec.rs_import_seq := v_import_log_seq;
            v_result_rec.rs_data_line_number := v_data_line_number;
            v_result_rec.rs_data_line_name := v_data_line_name;
            v_result_rec.rs_document := v_result_document;
          
            /* insert result */
            v_success := sim_import_result_extras.create_result(v_result_rec, p_import_log_seq);
            
          
            /* if insert function failed, write error and prevent activity from being exported */
            IF v_success != 1 THEN
              UPDATE sim_field_activities
                 SET fa_status = 'N'
               WHERE fa_seq = v_activity_seq;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Failed to create result in sim_results, activity will not be exported.');
               
             
            ELSE
              BEGIN
                IF v_no_remark_codes > 0 THEN
                  FOR v_number IN 1 .. v_no_remark_codes LOOP
                    IF v_remark_is_table(v_number) IS NOT NULL THEN
                      v_tsrlbrmk_is_nbr := v_remark_is_table(v_number);
                      v_tsrlbrmk_org_id := v_remark_org_table(v_number);
                      v_success := add_remark_to_results(v_result_rec.rs_seq, v_tsrlbrmk_is_nbr, v_tsrlbrmk_org_id);
                    END IF;
                  END LOOP;
                END IF;
              EXCEPTION
                WHEN no_data_found THEN
                  NULL;
                WHEN OTHERS THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                      SQLERRM ||
                                                      ' - adding lab remark codes to activity.');
                   
                 
              END;
            END IF;
          END IF;
        
          /* THIS WILL CREATE A RECORD FOR SIM_RESULT_CLASS_INDICATOR */
        
          -- jah 2-20-04
          -- do we need to create a SIM_RESULT_CLASS_INDICATOR record?
          IF ((v_lower_bnd_amt IS NOT NULL OR v_primary_class IS NOT NULL) AND
             (nvl(v_dup_brgi_success, 0) != 0))
             OR (v_particle_size_basis IS NOT NULL) THEN
            -- if yes
          
            OPEN c_get_next_rci_seq;
            FETCH c_get_next_rci_seq
              INTO v_rci_rec.rci_seq;
            CLOSE c_get_next_rci_seq;
          
            /* assign values to record and send to create_rci function */
            v_rci_rec.rci_rs_seq := v_result_rec.rs_seq; --gets RCI_RS_SEQ from v_result_rec
            v_rci_rec.rci_fa_seq := v_activity_seq;
            v_rci_rec.rci_import_seq := v_import_log_seq;
          
            -- jah 2-20-04
            -- Set RCI_PARTICLE_SIZE_BASIS to value (instead of NULL)
            IF (v_particle_size_basis IS NOT NULL) THEN
              v_rci_rec.rci_particle_size_basis := v_particle_size_basis;
              v_rci_rec.rci_tsruom_is_number := 1;
              v_rci_rec.rci_tsruom_org_id := 'ZZZZZZZZ';
              v_rci_rec.rci_lower_bnd_amt := NULL;
              v_rci_rec.rci_upper_bnd_amt := NULL;
              v_rci_rec.rci_primary_class_ind := NULL;
              v_rci_rec.rci_second_class_ind := NULL;
              v_rci_rec.rci_brg_seq := NULL;
            ELSE
              v_rci_rec.rci_particle_size_basis := NULL;
              v_rci_rec.rci_tsruom_is_number := v_uom_is_nbr;
              v_rci_rec.rci_tsruom_org_id := v_uom_org_id;
              v_rci_rec.rci_lower_bnd_amt := v_lower_bnd_amt;
              v_rci_rec.rci_upper_bnd_amt := v_upper_bnd_amt;
              v_rci_rec.rci_primary_class_ind := v_primary_class_value;
              v_rci_rec.rci_second_class_ind := v_secondary_class_value;
              v_rci_rec.rci_brg_seq := v_result_rec.rs_brg_seq;
            END IF;
          
            /* insert rci record */
            v_success := sim_import_result_extras.create_rci(v_rci_rec, v_import_log_seq);
          
            IF v_success != 1 THEN
              -- invalidate activity if create_rci fails
              UPDATE sim_field_activities
                 SET fa_status = 'N'
               WHERE fa_seq = v_activity_seq;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                  'Failed to create record in sim_results_class_indicator, activity will not be exported.');
               
             
            END IF;
          END IF;
        
          --get ready for the next line
          IF nvl(v_brg_type, 'x') != 'Single Taxon Frequency Classes' THEN
            v_uom_is_nbr := NULL;
            v_uom_org_id := NULL;
            v_char_is_nbr := NULL;
            v_char_org_id := NULL;
          END IF;
        
          /* Update status of the field activity */
          IF v_status = 'N' THEN
            UPDATE sim_field_activities
               SET fa_status = 'N'
             WHERE fa_seq = v_activity_seq
               AND fa_import_seq = p_import_log_seq;
            --v_error_cnt_new := v_error_cnt_new + 1;
            --v_error_cnt := v_error_cnt + 1;
            v_pre_check_status := 'N';
          END IF;
        
          /* update import lines table with activity and result sequences */
          UPDATE sim_imp_lines
             SET sipl_fa_seq = v_activity_seq
           WHERE sipl_seq = v_text_line.sipl_seq;
          UPDATE sim_imp_lines
             SET sipl_rs_seq = v_result_rec.rs_seq
           WHERE sipl_seq = v_text_line.sipl_seq;
        
        EXCEPTION
          WHEN OTHERS THEN
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                SQLERRM ||
                                                ' - creating result. This line has been skipped.');
              
        END;
      
      EXCEPTION
        WHEN OTHERS THEN   
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                SQLERRM ||
                                                ' - creating result. Line has been skipped.');
   
      END;
    END IF;
  END LOOP; /* end of text loop */
  IF v_pre_check = 1 THEN
  
    --delete field activities, visits, and trips made on this pass
    DELETE sim_result_class_indicator
     WHERE rci_import_seq = v_import_log_seq;
    DELETE sim_results
     WHERE rs_import_seq = v_import_log_seq;
    COMMIT;
  
    DELETE sim_bio_results_grp_individual
     WHERE brgi_import_seq = v_import_log_seq;
    DELETE sim_bio_results_grp
     WHERE brg_import_seq = v_import_log_seq;
    COMMIT;
    
    --Update trips and visits to set import seq
    UPDATE sim_station_visits
      SET ssv_import_seq = v_import_log_seq
      WHERE ssv_seq IN (SELECT fa_ssv_seq
                          FROM sim_field_activities
                          WHERE fa_import_seq = v_import_log_seq);
    UPDATE sim_trips
      SET stp_import_seq = v_import_log_seq
      WHERE stp_seq IN (SELECT fa_stp_seq
                          FROM sim_field_activities
                          WHERE fa_import_seq = v_import_log_seq);
    
    DELETE sim_field_activities
     WHERE fa_import_seq = v_import_log_seq;
    COMMIT;
    DELETE sim_station_visits
     WHERE ssv_import_seq = v_import_log_seq;
    DELETE sim_trips
     WHERE stp_import_seq = v_import_log_seq;
    COMMIT;
  
    IF v_pre_check_status != 'N' THEN
      --we're good to load the data!
      v_pre_check := 0;
      GOTO start_checking;
    ELSE
      --delete import lines table and we're done for this round
      DELETE sim_imp_lines
       WHERE sipl_sil_seq = v_import_log_seq;
    END IF;
  END IF;

  BEGIN
    v_error_cnt := sim_import_pkg.get_rows_with_errors(v_import_log_seq);  
    /* update counts in the import log */
    UPDATE sim_import_log
       SET sil_rows_imported = v_line_cnt,
           sil_rows_errors = v_error_cnt,
           sil_lab_id = v_first_lab_id
     WHERE sil_seq = v_import_log_seq;
    update_rows_ready(v_import_log_seq,v_result_cnt,v_act_cnt);
    /* slp 05/21/03 - Modified SIL_ROWS_READY from complete count to result count because it appeared
    to the user that the skipped row was also ready for migration which it is not. */
    IF v_pre_check = 1 THEN
      --If pre-check is still 1 then we're not loading anything.
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                          v_line_cnt ||
                                          ' lines processed / 0 activities and 0 results loaded.');
    ELSE
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                          v_line_cnt || ' lines processed / ' ||
                                           v_act_cnt || ' activities and ' ||
                                           v_result_cnt || ' results loaded. ');
    END IF;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Finished: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
    sim_event_log_pkg.update_stop_time;                         
    sim_import_pkg.v_strip_quote := NULL; --10/04/2005 --RWK --Last action needs to reset v_strip_quote to null
    COMMIT;
    
  EXCEPTION
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                          'Error in post line processing: ' ||
                                          SQLERRM);
       
     
  END;

EXCEPTION
  WHEN e_no_cfg THEN
    raise_application_error(-20102, 'Configuration invalid or does not exist.');
  
  WHEN e_bad_delimiter THEN
    raise_application_error(-20197, 'The import file contains the wrong number of columns');
  
  WHEN no_data_found THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        substr('Line ' || v_line_cnt ||
                                        ', Last Item Processed: ' ||
                                        v_item_name ||
                                        ', NO_DATA_FOUND : ' ||
                                        SQLERRM ||
                                        ' - IN sim_Import_Result_Pkg.import_results.', 1, 240));
     
   
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        to_char(v_line_cnt) ||
                                       ' lines processed / ' ||
                                       to_char(v_act_cnt) ||
                                       ' activities loaded');
     
   
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Finished: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
     
   
    -- Ryan J. 03/08/05 - Raise error to calling routine
    raise_application_error(-20000, substr('Line ' || v_line_cnt ||
                                    ', Last Item Processed: ' || v_item_name ||
                                    ', NO_DATA_FOUND : ' || SQLERRM ||
                                    ' - IN sim_Import_Result_Pkg.import_results.', 1, 240));
  
  WHEN e_no_sid_seq THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        'Import Type not found. Something is seriously wrong.');
     
   
    -- Ryan J. 03/08/05 - Raise error to calling routine
    raise_application_error(-20000, 'Import Type not found. Something is seriously wrong.');
  
  WHEN e_bad_station THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        'Station Problem. Something is seriously wrong.');
     
   
    -- Ryan J. 03/08/05 - Raise error to calling routine
    raise_application_error(-20000, 'Station Problem. Something is seriously wrong.');
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                        substr('Line ' || v_line_cnt ||
                                        ', Last Item Processed: ' ||
                                        v_item_name ||
                                        ', Fatal Error : ' || SQLERRM ||
                                        ' - IN sim_Import_Result_Pkg.import_results.', 1, 240));
     
   
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        to_char(v_line_cnt) ||
                                       ' lines processed / ' ||
                                       to_char(v_act_cnt) ||
                                       ' activities loaded');
     
   
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Finished: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
     
   
    -- Ryan J. 03/08/05 - Raise error to calling routine
    raise_application_error(-20000, substr('Line ' || v_line_cnt ||
                                    ', Last Item Processed: ' || v_item_name ||
                                    ', Fatal Error : ' || SQLERRM ||
                                    ' - IN sim_Import_Result_Pkg.import_results.', 1, 240));
  
END;

END sim_import_result_pkg;
/
