CREATE OR REPLACE PACKAGE sim_import_station_pkg IS

  -- Purpose : import stations data into SIM tables from a text file

/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                             *
 ****************************************************************************************/
  TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;
  
  TYPE col_table_num_type IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;
  
  TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;
  
  TYPE t_storet_pk_type IS RECORD(
    is_number NUMBER(8),
    org_id    VARCHAR2(8));
  
  TYPE t_storet_tsmvstc_type IS RECORD(
    tsmvstc_is_number NUMBER(8),
    tsmvstc_org_id    VARCHAR2(8));
  
  TYPE t_storet_tsmmad_type IS RECORD(
    tsmmad_is_number NUMBER(8),
    tsmmad_org_id    VARCHAR2(8));
  
  TYPE t_aquifer_type IS RECORD(
    tsmgeoun_cd  VARCHAR2(12),
    aquifer_name VARCHAR2(80));

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_activity_count t_activity_count;
  v_org_is_nbr     NUMBER(8) := 0;
  v_import_log_seq NUMBER(10);
  v_status         VARCHAR2(1);
  v_line_cnt       NUMBER(10) := 0;

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                         *
 ****************************************************************************************/
PROCEDURE import_station
(
  p_org_id         IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_imp_cfg_seq    IN NUMBER,
  p_test_count     IN NUMBER,
  p_skip_header    IN NUMBER := 0 
); --   1 = Skip  0 = Don't Skip

FUNCTION check_tsmmad_value
(
  p_orig_item   IN VARCHAR2,
  p_item_name   IN VARCHAR2,
  p_category    IN VARCHAR2,
  p_subcategory IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER,
  p_imp_cfg_seq IN NUMBER,
  p_imp_log_seq IN NUMBER
) RETURN VARCHAR2;

END sim_import_station_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_import_station_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/

/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/

/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS (SPECS)                      *
 ****************************************************************************************/
FUNCTION check_translations
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_trans_exist OUT BOOLEAN
) RETURN VARCHAR2;

PROCEDURE create_trans_record
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER
);

FUNCTION check_storet_translation
(
  p_code           IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_trans_exist    OUT BOOLEAN,
  v_converted_item OUT VARCHAR2,
  p_sta_seq        IN NUMBER
) RETURN t_storet_pk_type;

/****************************************************************************************
 *                                PROCEDURES AND FUNCTIONS                              *
 ****************************************************************************************/

/****************************************************************************************

 ****************************************************************************************/
FUNCTION validate_number
(
  p_item      IN VARCHAR2,
  p_item_name IN VARCHAR2,
  p_format    IN VARCHAR2,
  p_sta_seq   IN NUMBER
) RETURN NUMBER IS

  -- This function is to test any value that is supposed to be a number. 
  -- If it's not a number a 0 is returned.
  v_return NUMBER(10);

BEGIN
  v_return := to_number(p_item, p_format);
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         p_item_name || ', ' || p_item ||
                                         ', is not in an appropriate numeric format of ' ||
                                         p_format || '.');                
    RETURN 0;
END validate_number;

/****************************************************************************************
  This function checks to see if a station already exists by checking for existing 
  station ID
 ****************************************************************************************/
FUNCTION check_if_station_exists
(
  p_station_cd IN VARCHAR2,
  p_org_id     IN VARCHAR2,
  p_sta_seq    IN NUMBER
) RETURN NUMBER IS

  v_station_cd VARCHAR2(15);

  CURSOR c_compare_station_id(p_station_cd IN VARCHAR2, p_station_org_id IN VARCHAR2) IS
    SELECT identification_cd
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_cd, 15)
       AND tsmstatn_org_id = rpad(p_org_id, 8);

  CURSOR c_compare_sim_stat_id(p_station_cd IN VARCHAR2, p_station_org_id IN VARCHAR2) IS
    SELECT sta_id
      FROM sim_stations
     WHERE sta_id = p_station_cd
       AND sta_tsmorgan_org_id = p_org_id;

BEGIN

  /* Check STORET */
  OPEN c_compare_station_id(p_station_cd, p_org_id);
  FETCH c_compare_station_id
    INTO v_station_cd;
  CLOSE c_compare_station_id;

  IF v_station_cd IS NOT NULL THEN
     
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Station ' ||
                                        TRIM(v_station_cd) ||
                                        ' already exists in STORET.');
      
  
  END IF;

  /* Check SIM */
  IF v_station_cd IS NULL THEN
    OPEN c_compare_sim_stat_id(p_station_cd, p_org_id);
    FETCH c_compare_sim_stat_id
      INTO v_station_cd;
    CLOSE c_compare_sim_stat_id;
  
    IF v_station_cd IS NOT NULL THEN
      
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Station ' || v_station_cd ||
                                          ' already exists in SIM.');
        
    
    END IF;
  END IF;

  /** Station is unique **/
  IF v_station_cd IS NULL THEN
    RETURN 1;
  ELSE
    RETURN 0;
  END IF;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                          ' - in check_if_station_exists.');
        
    
    RETURN 0;
END check_if_station_exists;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_org_info
(
  p_tsmorgan_org_id IN VARCHAR2,
  p_tsmorgan_is_nbr OUT NUMBER
) RETURN NUMBER IS
  CURSOR c_get_tsmorgan_nbr(p_cursor_org_id IN VARCHAR2) IS
    SELECT tsmorgan_is_number
      FROM tsmorgan
     WHERE org_id = rpad(p_cursor_org_id, 8);
BEGIN
  OPEN c_get_tsmorgan_nbr(p_tsmorgan_org_id);
  FETCH c_get_tsmorgan_nbr
    INTO p_tsmorgan_is_nbr;
  IF c_get_tsmorgan_nbr%NOTFOUND THEN
    p_tsmorgan_is_nbr := NULL;
  END IF;
  CLOSE c_get_tsmorgan_nbr;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END get_org_info;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_instructions
(
  p_imp_cfg_seq      IN NUMBER,
  p_instruction_type IN VARCHAR2
) RETURN VARCHAR2 IS
  v_instr_choice VARCHAR2(15);
  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_cfg_instrs,
           sim_imp_instr_choices,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND sici_sicc_seq = p_cfg_seq;
  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_instr_choices,
           sim_imp_cfg_instrs,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND siic_dflt_flag = 1;
BEGIN
  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr
    INTO v_instr_choice;
  IF c_get_cfg_instr%NOTFOUND THEN
    /* no choice could be found, get default choice */
    OPEN c_get_instr_default(p_instruction_type);
    FETCH c_get_instr_default
      INTO v_instr_choice;
    CLOSE c_get_instr_default;
  END IF;
  CLOSE c_get_cfg_instr;
  RETURN v_instr_choice;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get_instructions.');
        
    
    RETURN NULL;
END get_instructions;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_allowable_value
(
  p_alval_seq IN NUMBER,
  p_sta_seq   IN NUMBER
) RETURN VARCHAR2 IS
  v_item_name VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
  END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - getting allowable value from sequence.');
        
    
END get_allowable_value;

/****************************************************************************************
 The validate_primary_type function takes the original item and checks it against 
 existing translations. If there is a translation the translation is returned to 
 IMPORT_STATIONS where it is used to find the tsmvstc_is_nbr and org_id.
****************************************************************************************/
FUNCTION validate_primary_type
(
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER,
  p_item_name   IN VARCHAR2
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN;
  e_no_function EXCEPTION;
  v_primary_type VARCHAR2(30);
  v_instr_code   VARCHAR2(15);
  v_instr_id     VARCHAR2(8);

  CURSOR c_check_primary_type(p_primary_type IN VARCHAR2) IS
    SELECT primary_type_cd
      FROM tsmvstc
     WHERE primary_type_cd = p_primary_type;

BEGIN
  /* function to look in the translations and STORET tables for items that 
  should already exist in hard coded values in STORET tables
  look in translations table to see if it needs to be swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_primary_type(v_converted_item);
    FETCH c_check_primary_type
      INTO v_primary_type;
    IF c_check_primary_type%NOTFOUND THEN
      v_primary_type := NULL;
    END IF;
    CLOSE c_check_primary_type;
    /*  call function to look in storet table for primary type */
    IF v_primary_type IS NULL
       AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_id := 'STA-TR';
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        --create translations
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
        
      ELSE
        
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
        
      END IF;
    ELSIF v_primary_type IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_orig_item ||
                                             ' has been mapped to the invalid storet value: ' ||
                                             v_converted_item || '.');
            
        
      v_status := 'N';
    END IF;
  ELSIF v_trans_exist THEN
    v_status := 'N';
  END IF;
  RETURN v_primary_type;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate primary type.');
        
        
    RETURN NULL;
    v_status := 'N';
END validate_primary_type;

/****************************************************************************************
 The validate_secondary_type function takes the original item and checks it against 
 existing translations. If there is a translation the translation is returned to 
 IMPORT_STATIONS where it is used to find the tsmvstc_is_nbr and org_id. If there is a 
 valid secondary type, the is number and org id from the primary type calculation will 
 be overwritten.
 
 Updated for v2 as secondary type is now required
****************************************************************************************/
FUNCTION validate_secondary_type
(
  p_orig_item    IN VARCHAR2,
  p_primary_type IN VARCHAR2,
  p_imp_cfg_seq  IN NUMBER,
  p_col_seq      IN NUMBER,
  p_sta_seq      IN NUMBER,
  p_item_name    IN VARCHAR2
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN := TRUE;
  e_no_function EXCEPTION;
  v_secondary_type VARCHAR2(30);
  v_instr_code     VARCHAR2(15);
  v_instr_id       VARCHAR2(8);

  CURSOR c_check_secondary_type(p_secondary_type IN VARCHAR2, p_primary_type IN VARCHAR2) IS
    SELECT secondary_type_cd
      FROM tsmvstc
     WHERE secondary_type_cd = p_secondary_type
       AND primary_type_cd = p_primary_type;
BEGIN
  /* function to look in the translations and STORET tables for items 
  that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_secondary_type(v_converted_item, p_primary_type);
    FETCH c_check_secondary_type
      INTO v_secondary_type;
    IF c_check_secondary_type%NOTFOUND THEN
      v_secondary_type := NULL;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'Secondary Station Type: ' ||
                                           p_orig_item ||
                                           ' does not match with Primary Type: ' ||
                                           p_primary_type || '.');
          
        
    END IF;
    CLOSE c_check_secondary_type;
  
    /* call function to look in storet table for primary/secondary type */
    IF v_secondary_type IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
    
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01'
         OR v_instr_code = 'GWS-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
          
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
          
      END IF;
    ELSIF v_secondary_type IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item || '.');
          
          
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to N later in the program. */
    v_secondary_type := NULL;
  END IF;
  RETURN v_secondary_type;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate secondary type.');
        
          
    RETURN NULL;
END validate_secondary_type;

/****************************************************************************************
 The validate_county function takes the original item and checks 
 it against existing translations. cursor c_check_county uses 
 character data to search, while c_check_county_fips checks with the
 county FIPS code. First the p_orig_item is tested to see if it is 
 of character or numeric format, then it is checked against known 
 translations. If there is a translation the translation is returned 
 and used in a cursor to find the IS number org id from TSMGEOPA.
****************************************************************************************/
FUNCTION validate_county
(
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_item_name   IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(25);
  v_trans_exist    BOOLEAN := TRUE;
  e_no_function EXCEPTION;
  v_county     VARCHAR2(25);
  v_instr_code VARCHAR2(15);
  v_instr_id   VARCHAR2(8);

  -- Get County from STORET using "text" County
  -- jah 2-26-04 Convert p_county to uppercase because STORET GUI converts County to uppercase
  CURSOR c_check_county(p_county IN VARCHAR2) IS
    SELECT county_name
      FROM tsmgeopa
     WHERE county_name = upper(p_county);

  -- Get County from STORET using "FIPS" County
  -- jah 2-26-04 Convert p_county to uppercase because STORET GUI converts County to uppercase
  CURSOR c_check_county_fips(p_county IN VARCHAR2) IS
    SELECT fips_county_code
      FROM tsmgeopa
     WHERE fips_county_code = upper(p_county);

BEGIN

  -- Translate the item value (return original value if no translation exists)
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  -- did CHECK_TRANSLATIONS work properly?
  IF (v_converted_item IS NOT NULL) THEN
    -- if yes
  
    -- is this a FIPS county code?
    IF (translate(v_converted_item, 'a1234567890.- ', 'a') IS NULL) THEN
      -- if yes
    
      -- Try to retrieve the county using FIPS county
      OPEN c_check_county_fips(v_converted_item);
      FETCH c_check_county_fips
        INTO v_county;
    
      -- Did we retrieve the county using FIPS county?
      IF c_check_county_fips%NOTFOUND THEN
        -- if no
      
        -- return NULL
        v_county := NULL;
      
         
      
                                                                                                  
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             'FIPS county code is invalid.');
            
          
      END IF;
      CLOSE c_check_county_fips;
    
    ELSE
      -- if no (it is not a FIPS county code, therefore it is a text county code)
    
      -- Try to retrieve the text county record
      OPEN c_check_county(v_converted_item);
      FETCH c_check_county
        INTO v_county;
    
      -- Did we retrieve the text county record?
      IF c_check_county%NOTFOUND THEN
        -- if no
      
        -- return NULL
        v_county := NULL;
      
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             'County name is invalid.');
            
          
      END IF;
      CLOSE c_check_county;
    
    END IF;
  
    -- did the calling routine pass a NULL value?
    IF (v_county IS NULL)
       AND (NOT v_trans_exist) THEN
      -- if yes
    
      v_instr_id := 'STA-TR';
    
      -- check instructions to see if translation record should be created
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
    
      -- should we create a translation record?
      IF v_instr_code = 'STA-TR-01' THEN
        -- if yes
      
        -- create translation record and tell the end-user        
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
          
      ELSE
      
        
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
          
      END IF;
    
      -- did CHECK_TRASNSLATIONS translate to NULL?
    ELSIF v_county IS NULL
          AND v_trans_exist THEN
      -- if yes
    
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
          
    END IF;
  
  ELSIF v_trans_exist THEN
    -- if no, CHECK_TRANSLATIONS did NOT work properly (i.e., it translated to NULL)
  
    -- return NULL.
    -- Calling routine is responsible for error message, if any,
    -- and for changing Status to 'N' to prevent migration to STORET.
    v_county := NULL;
  
  END IF;

  RETURN v_county;

EXCEPTION

  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate county name.');
        
          
    RETURN NULL;
  
END validate_county;

/****************************************************************************************
 The validate_state function takes the original item and checks 
 it against existing translations. If there is a translation the 
 translation is returned and used in a cursor to find the IS number
 org id from TSMGEOPA.
****************************************************************************************/
FUNCTION validate_state
(
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_item_name   IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN := TRUE;
  e_no_function EXCEPTION;
  v_state      VARCHAR2(2);
  v_instr_code VARCHAR2(15);
  v_instr_id   VARCHAR2(8);

  -- Get State using "text" State
  -- jah 2-26-04 Convert p_state to uppercase because STORET GUI converts State to uppercase
  CURSOR c_check_state(p_state IN VARCHAR2) IS
    SELECT state_postal_code
      FROM tsmgeopa
     WHERE state_postal_code = upper(p_state);

  -- Get State using "FIPS" State
  -- jah 2-26-04 Convert p_state to uppercase because STORET GUI converts State to uppercase
  CURSOR c_check_state_fips(p_state IN VARCHAR2) IS
    SELECT state_fips_code
      FROM tsmgeopa
     WHERE state_fips_code = p_state;

BEGIN
  /* function to look in the translations and STORET tables for 
  items that should already exist in hard coded values in 
  STORET tables */
  /* look in translations table to see if it needs to swapped */

  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  -- did check_translations work properly?
  IF v_converted_item IS NOT NULL THEN
    -- if yes
  
    -- is converted item numeric?
    IF translate(v_converted_item, 'a1234567890.- ', 'a') IS NULL THEN
      -- if yes (the it's a FIPS county code)
    
      OPEN c_check_state_fips(v_converted_item);
      FETCH c_check_state_fips
        INTO v_state;
      IF c_check_state_fips%NOTFOUND THEN
        v_state := NULL;
      END IF;
      CLOSE c_check_state_fips;
    
    ELSE
      -- if no (i.e., converted item is text
    
      OPEN c_check_state(v_converted_item);
      FETCH c_check_state
        INTO v_state;
      IF c_check_state%NOTFOUND THEN
        v_state := NULL;
      END IF;
      CLOSE c_check_state;
    
    END IF;
  
    /* call function to look in storet table for primary/secondary type */
    IF v_state IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
    
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
          
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
          
      END IF;
    
    ELSIF v_state IS NULL
          AND v_trans_exist THEN
    
      /*  item has been translated to an invalid storet value */
        
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
          
    END IF;
  
  ELSIF v_trans_exist THEN
  
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_state := NULL;
  
  END IF;

  RETURN v_state;

EXCEPTION

  WHEN OTHERS THEN
     
  
                                                                                                                                                                  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                        SQLERRM || ' - in validate state.');
        
          
    RETURN NULL;
  
END validate_state;

/****************************************************************************************
 The validate_estuary_state function takes the original item 
 and checks it against existing translations.  Validate the 
 converted item against the state name in the TSMESTRY table.  
 We could not use the State FIPS code or Postal Code as the 
 state names in the TSMGEOPA table do not match the valid 
 state names in the TSMESTRY table.  For this reason, we had 
 to accept only the complete state name as they are defined
 in the TSMESTRY table.
****************************************************************************************/
FUNCTION validate_estuary_state
(
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_item_name   IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN := TRUE;
  e_no_function EXCEPTION;
  v_estuary_state VARCHAR2(40);
  v_instr_code    VARCHAR2(15);
  v_instr_id      VARCHAR2(8);
  v_storet_pk     t_storet_pk_type;

  CURSOR c_check_state(p_converted_item IN VARCHAR2) IS
    SELECT state_province_name
      FROM tsmestry
     WHERE state_province_name = p_converted_item;

BEGIN

  -- jah 4-29-04 Changed the following to check_translations (task 472)
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  -- did the call to check_translation work?  
  IF (v_converted_item IS NOT NULL) THEN
    -- if yes
  
    -- try to retrieve the estuary state
    OPEN c_check_state(v_converted_item);
    FETCH c_check_state
      INTO v_estuary_state;
  
    -- did we retrieve the estuary state?
    IF c_check_state%NOTFOUND THEN
      -- if no
      v_estuary_state := NULL;
    END IF;
    CLOSE c_check_state;
  
    IF v_estuary_state IS NULL
       AND NOT v_trans_exist THEN
    
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
    
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
          
      ELSE
        
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
          
      END IF;
    
    ELSIF v_estuary_state IS NULL
          AND v_trans_exist THEN
    
      /*  item has been translated to an invalid storet value */
      
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
          
    END IF;
  
  ELSIF v_trans_exist THEN
  
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_estuary_state := NULL;
  
  END IF;

  RETURN v_estuary_state;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate estuary state.');
        
          
    RETURN NULL;
END validate_estuary_state;

/****************************************************************************************
 The validate_huc function takes the original item and checks 
 it against existing translations.  If there is a translation 
 the translation is returned and used in a cursor to find the 
 IS number org id from TSMFHU.
****************************************************************************************/
FUNCTION validate_huc
(
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_item_name   IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(8);
  v_trans_exist    BOOLEAN := TRUE;
  e_no_function EXCEPTION;
  v_huc        VARCHAR2(8);
  v_instr_code VARCHAR2(15);
  v_instr_id   VARCHAR2(8);
  v_storet_key sim_import_station_pkg.t_storet_pk_type;

  CURSOR c_check_huc(p_huc IN VARCHAR2) IS
    SELECT hydrologic_unit_cd
      FROM tsmfhu
     WHERE hydrologic_unit_cd = rpad(p_huc, 8);

BEGIN
  /* function to look in the translations and STORET tables for items 
  that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_storet_key := check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, v_converted_item, p_sta_seq);

  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_huc(v_converted_item);
    FETCH c_check_huc
      INTO v_huc;
    IF c_check_huc%NOTFOUND THEN
      v_huc := NULL;
    END IF;
    CLOSE c_check_huc;
    IF v_huc IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
            
      ELSE
          
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
            
      END IF;
    ELSIF v_huc IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
            
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_huc := NULL;
  END IF;
  RETURN v_huc;

EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                              
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in validate huc.');
        
           
    RETURN NULL;
END validate_huc;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_primary_estuary
(
  p_orig_item       IN VARCHAR2,
  p_imp_cfg_seq     IN NUMBER,
  p_item_name       IN VARCHAR2,
  p_col_seq         IN NUMBER,
  p_est_st_name     IN VARCHAR2,
  p_tsmestry_is_nbr OUT NUMBER,
  p_sta_seq         IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(30);
  v_trans_exist    BOOLEAN := TRUE;

  v_estuary    VARCHAR2(30);
  v_instr_code VARCHAR2(15);
  v_instr_id   VARCHAR2(8);

  CURSOR c_check_estuary(p_estuary IN VARCHAR2, p_estuary_state IN VARCHAR2) IS
    SELECT tsmestry_is_number
      FROM tsmestry
     WHERE NAME = p_estuary
       AND state_province_name = p_estuary_state
       AND primary_indicator = 'Y';

BEGIN
  /* function to look in the translations and STORET tables for items 
  that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */

  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_estuary(v_converted_item, p_est_st_name);
    FETCH c_check_estuary
      INTO p_tsmestry_is_nbr;
    IF c_check_estuary%NOTFOUND THEN
      v_estuary := NULL;
    ELSE
      v_estuary := v_converted_item;
    END IF;
    CLOSE c_check_estuary;
  
    IF v_estuary IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
               
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
               
      END IF;
    ELSIF v_estuary IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
               
    END IF;
  
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_estuary := NULL;
  END IF;

  RETURN v_estuary;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate primary estuary.');
        
               
    RETURN NULL;
END validate_primary_estuary;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_secondary_estuary
(
  p_orig_item       IN VARCHAR2,
  p_imp_cfg_seq     IN NUMBER,
  p_item_name       IN VARCHAR2,
  p_col_seq         IN NUMBER,
  p_primary_is_nbr  IN NUMBER,
  p_tsmestry_is_nbr OUT NUMBER,
  p_sta_seq         IN NUMBER
) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(30);
  v_trans_exist    BOOLEAN := TRUE;

  v_estuary    VARCHAR2(30);
  v_instr_code VARCHAR2(15);
  v_instr_id   VARCHAR2(8);

  CURSOR c_check_estuary(p_estuary IN VARCHAR2, p_primary IN NUMBER) IS
    SELECT tsmestry_is_number
      FROM tsmestry
     WHERE tsmestry0is_number = p_primary
       AND NAME = p_estuary;

BEGIN
  /* function to look in the translations and STORET tables for items 
  that should already exist in hard coded values in STORET tables */
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_estuary(v_converted_item, p_primary_is_nbr);
    FETCH c_check_estuary
      INTO p_tsmestry_is_nbr;
    IF c_check_estuary%NOTFOUND THEN
      v_estuary := NULL;
    ELSE
      v_estuary := v_converted_item;
    END IF;
    CLOSE c_check_estuary;
  
    IF v_estuary IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
                   
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
                   
      END IF;
    ELSIF v_estuary IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
             
    END IF;
  
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_estuary := NULL;
  END IF;

  RETURN v_estuary;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate secondary estuary.');
        
             
    RETURN NULL;
END validate_secondary_estuary;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_nal
(
  p_orig_item    IN VARCHAR2,
  p_imp_cfg_seq  IN NUMBER,
  p_item_name    IN VARCHAR2,
  p_col_seq      IN NUMBER,
  v_native_name  IN VARCHAR,
  v_native_state IN VARCHAR2,
  v_which        IN NUMBER,
  p_sta_seq      IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given State and 
  -- NAL code are in the STORET TSMNAL table.

  v_nal_name       VARCHAR2(40);
  v_nal_code       VARCHAR2(5);
  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN := TRUE;
  v_nal            VARCHAR2(40);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);

  CURSOR c_check_nal(p_native_name IN VARCHAR2, p_native_state IN VARCHAR2) IS
    SELECT tsmnal_cd
      FROM tsmnal
     WHERE NAME = p_native_name
       AND tsmnal_state = p_native_state;

  CURSOR c_check_name(p_native_name IN VARCHAR2) IS
    SELECT NAME
      FROM tsmnal
     WHERE NAME = p_native_name;

  CURSOR c_check_state(p_native_state IN VARCHAR2) IS
    SELECT NAME
      FROM tsmnal
     WHERE tsmnal_state = p_native_state;

BEGIN
  v_nal := NULL;
  IF v_which <> 3 THEN
    v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  ELSE
    v_converted_item := p_orig_item;
  END IF;

  IF v_converted_item IS NOT NULL THEN
    IF v_which = 1 THEN
      --check code
      OPEN c_check_name(v_converted_item);
      FETCH c_check_name
        INTO v_nal_name;
      IF c_check_name %NOTFOUND THEN
        v_nal := NULL;
      ELSE
        v_nal := v_nal_name;
      END IF;
      CLOSE c_check_name;
    ELSIF v_which = 2 THEN
      --check state
      OPEN c_check_state(v_converted_item);
      FETCH c_check_state
        INTO v_nal_name;
      IF c_check_state %NOTFOUND THEN
        v_nal := NULL;
      ELSE
        v_nal := v_converted_item;
      END IF;
      CLOSE c_check_state;
    ELSE
      --check them both together
      OPEN c_check_nal(v_native_name, v_native_state);
      FETCH c_check_nal
        INTO v_nal_code;
      IF c_check_nal %NOTFOUND THEN
        RETURN 'BAD';
      ELSE
        RETURN v_nal_code;
      END IF;
      CLOSE c_check_nal;
    END IF;
  
    IF v_nal IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
                
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
                
      END IF;
    ELSIF v_nal IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_item_name || ', ' || p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
              
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_nal := NULL;
  END IF;

  RETURN v_nal;

EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                     
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - checking NAL.');
        
           
    RETURN 'BAD';
END validate_nal;

/****************************************************************************************

****************************************************************************************/
FUNCTION validate_prmvl_item
(
  p_orig_item   IN CHAR,
  p_imp_cfg_seq IN NUMBER,
  p_item_name   IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_table_name  IN VARCHAR,
  p_column_name IN VARCHAR2,
  p_sta_seq     IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(40);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_value   VARCHAR2(40);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);

  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT field_value
      FROM tsmprmvl
     WHERE table_name = rpad(p_table_name, 8)
       AND field_name = rpad(p_column_name, 20)
       AND upper(field_value) = rpad(upper(p_item), 40);

BEGIN
  v_storet_value := NULL;
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN
    OPEN c_check_prmvl(v_converted_item);
    FETCH c_check_prmvl
      INTO v_storet_value;
    IF c_check_prmvl %NOTFOUND THEN
      v_storet_value := NULL;
    END IF;
    CLOSE c_check_prmvl;
  
    IF v_storet_value IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
               
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
               
      END IF;
    ELSIF v_storet_value IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_item_name || ', ' || p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
             
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         p_item_name || ', ' || p_orig_item ||
                                         ' exists in translation table but has not been mapped to a STORET code. ');
        
           
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - checking STORET PRMVL.');
        
           
    RETURN NULL;
END validate_prmvl_item;

/****************************************************************************************

****************************************************************************************/
FUNCTION check_tsmmad_value
(
  p_orig_item   IN VARCHAR2,
  p_item_name   IN VARCHAR2,
  p_category    IN VARCHAR2,
  p_subcategory IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER,
  p_imp_cfg_seq IN NUMBER,
  p_imp_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(120);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_value   VARCHAR2(12);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);
  v_storet_key     sim_import_station_pkg.t_storet_pk_type;

  CURSOR c_check_tsmmad(p_item IN VARCHAR2) IS
    SELECT id_code
      FROM tsmmad
     WHERE category = rpad(p_category, 10)
       AND subcategory = rpad(p_subcategory, 6)
       AND id_code = rpad(p_item, 12);

BEGIN
  v_storet_value := NULL;
  v_storet_key := check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, v_converted_item, p_sta_seq);

  IF v_converted_item IS NOT NULL THEN
  
    OPEN c_check_tsmmad(v_converted_item);
    FETCH c_check_tsmmad
      INTO v_storet_value;
    IF c_check_tsmmad %NOTFOUND THEN
      v_storet_value := NULL;
    END IF;
    CLOSE c_check_tsmmad;
  
    IF v_storet_value IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'STA-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value - translation record has been created.');
            
               
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not a valid value.');
            
               
      END IF;
    ELSIF v_storet_value IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_item_name || ', ' || p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
             
    END IF;
  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                             
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - checking STORET TSMMAD.');
        
           
    RETURN NULL;
END check_tsmmad_value;

/****************************************************************************************
 The get_delimiter function checks the table sim_imp_config 
 to find the value entered for the delimiter.
****************************************************************************************/
FUNCTION get_delimiter
(
  p_imp_cfg_seq IN NUMBER,
  p_delimiter   OUT VARCHAR2
) RETURN NUMBER IS

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT sicc_delimiter
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

BEGIN
  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg
    INTO p_delimiter;
  IF c_get_imp_cfg%NOTFOUND THEN
    RETURN 0;
  END IF;
  CLOSE c_get_imp_cfg;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END get_delimiter;

/****************************************************************************************
 The get_item_pos function checks the table sim_imp_config 
 SIM_IMP_CFG_DETS SIM_IMP_COLS to find the value entered 
 for the the position of the item in the list
****************************************************************************************/
FUNCTION get_item_pos
(
  p_sicc_seq      IN NUMBER,
  p_pos_table     IN OUT col_table_num_type,
  p_default_table IN OUT col_table_type,
  p_col_cnt       IN OUT NUMBER
) RETURN NUMBER IS

  v_import_type_seq NUMBER(10);
  -- indicator that a column is included in an import configuration
  v_found           BOOLEAN;
  
  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT sicc_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

  CURSOR c_get_dflt_pos(p_type_seq IN NUMBER) IS
    SELECT sic_dflt_pos
      FROM sim_imp_cols
     WHERE sic_sid_seq = p_type_seq;

  CURSOR c_get_col_pos(p_sicc_seq IN NUMBER, p_dflt_pos IN NUMBER) IS
    SELECT sicdt_pos,
           sicdt_default
      FROM sim_imp_cfg_dets,
           sim_imp_cols
     WHERE sic_seq = sicdt_sic_seq
       AND sicdt_sicc_seq = p_sicc_seq
       AND sic_dflt_pos = p_dflt_pos
       AND (sicdt_included IS NOT NULL OR sicdt_generate IS NOT NULL);

BEGIN
  /* get the import type in order to limit the columns selected to the right type */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type
    INTO v_import_type_seq;
  IF c_get_import_type%NOTFOUND THEN
    v_import_type_seq := NULL;
    raise_application_error(-20102, 'Unable to get import type from import configuration.');
  END IF;
  CLOSE c_get_import_type;
  p_col_cnt := 0;
  /* loop through import columns and assign actual position to table indexed by
  default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP
    /* get actual position and item default from configuration table */
    OPEN c_get_col_pos(p_sicc_seq, v_dflt_pos.sic_dflt_pos);
    v_found := FALSE;
    LOOP
      FETCH c_get_col_pos
        INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
      IF c_get_col_pos%NOTFOUND THEN
        IF NOT v_found THEN
          /* if there is nothing in the configuration table for this column, position and default will be null */
          p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
          p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
        END IF;
      ELSE
        v_found := TRUE;
        p_col_cnt := p_col_cnt + 1;
      END IF;
      EXIT WHEN c_get_col_pos%NOTFOUND;
    END LOOP;
    CLOSE c_get_col_pos;
  END LOOP;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    raise_application_error(-20103, 'Failed to get column positions, item names, and column sequences.');
  
END get_item_pos;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_column_seq
(
  p_cfg_seq IN NUMBER,
  p_cfg_pos IN NUMBER
) RETURN NUMBER IS

  v_imp_col_seq NUMBER(10);
  e_no_cfg_found EXCEPTION;

  CURSOR c_get_col_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) IS
    SELECT sicdt_sic_seq
      FROM sim_imp_cfg,
           sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = sicc_seq
       AND sicc_seq = p_cfg_seq
       AND sicdt_pos = p_cfg_pos;

BEGIN
  OPEN c_get_col_seq(p_cfg_seq, p_cfg_pos);
  FETCH c_get_col_seq
    INTO v_imp_col_seq;
  IF c_get_col_seq%NOTFOUND THEN
    v_imp_col_seq := NULL;
    RAISE e_no_cfg_found;
  END IF;
  CLOSE c_get_col_seq;
  RETURN v_imp_col_seq;
EXCEPTION
  WHEN e_no_cfg_found THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - unable to find sequence in sim_imp_cfg_dets for pos: ' ||
                                         p_cfg_pos);
        
           
    raise_application_error(-20100, 'Import configuration is not complete');
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - getting column sequence.');
        
           
    RETURN NULL;
END get_column_seq;

/****************************************************************************************

****************************************************************************************/
FUNCTION check_translations
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_trans_exist OUT BOOLEAN
) RETURN VARCHAR2 IS

  v_storet_value VARCHAR2(4000);
  v_station_seq  NUMBER(10);

  CURSOR c_get_station_seq IS
    SELECT sim_stations_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_storet_code(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT st_storet_code
      FROM sim_translations,
           sim_imp_cfg_dets
     WHERE st_user_code = p_code
       AND st_sicdt_seq = sicdt_seq
       AND sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

BEGIN

  -- Try to retrieve a translation for the item value
  OPEN c_get_storet_code(p_code, p_imp_cfg_seq, p_col_seq);
  FETCH c_get_storet_code
    INTO v_storet_value;

  -- did we retrieve a translation?
  IF c_get_storet_code%FOUND THEN
    -- if yes
  
    -- tell calling routine the return value is a translation
    p_trans_exist := TRUE;
  
    -- did we encounter a translation problem?
    IF v_storet_value IS NULL THEN
      -- if yes
    
      /* if p_code is in the translations table, but the translation value is NULL, 
      then write an error message. However, the calling routine is responsible for 
      changing the status 'N' to prevent migration to STORET. */
    
      -- Get the next station sequence number sfor display in the error message   
      OPEN c_get_station_seq;
      FETCH c_get_station_seq
        INTO v_station_seq;
      CLOSE c_get_station_seq;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_code ||
                                           ' exists in translations table but has not been translated to a value.');
          
             
    END IF;
  
  ELSE
    -- if no (we did NOT find a translation)
  
    -- telling calling routine the return value is the original value
    p_trans_exist := FALSE;
  
    -- return the original value
    v_storet_value := p_code;
  
  END IF;
  CLOSE c_get_storet_code;

  RETURN v_storet_value;

EXCEPTION

  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                       
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in check translations');
        
           
END check_translations;

/****************************************************************************************

****************************************************************************************/
FUNCTION check_storet_translation
(
  p_code           IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_trans_exist    OUT BOOLEAN,
  v_converted_item OUT VARCHAR2,
  p_sta_seq        IN NUMBER
) RETURN t_storet_pk_type IS

  v_storet_pk   sim_import_station_pkg.t_storet_pk_type;
  v_seq         NUMBER(10);
  v_table_name  VARCHAR2(80);
  v_column_name VARCHAR2(80);
  v_dyna_cursor VARCHAR2(200);
  v_comma_pos   NUMBER(10);

  TYPE sort_cursor_type IS REF CURSOR;
  sort_cursor sort_cursor_type;

  CURSOR c_get_storet_key(p_c_code VARCHAR2, p_c_imp_cfg_seq NUMBER, p_c_col_seq NUMBER) IS
    SELECT st_seq,
           st_is_number,
           st_org_id,
           st_storet_code
      FROM sim_translations,
           sim_imp_cfg_dets
     WHERE st_user_code = p_code
       AND st_sicdt_seq = sicdt_seq
       AND sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

  CURSOR c_imp_cols(p_col_seq NUMBER) IS
    SELECT sic_storet_tab,
           sic_storet_col
      FROM sim_imp_cols
     WHERE sic_seq = p_col_seq;

  CURSOR c_comma(p_col_seq NUMBER) IS
    SELECT instr(sic_storet_col, ',') col_pos
      FROM sim_imp_cols
     WHERE sic_seq = p_col_seq;

  CURSOR c_inst_col(p_col_seq NUMBER) IS
    SELECT substr(sic_storet_col, 1, instr(sic_storet_col, ',') - 1) col_name
      FROM sim_imp_cols
     WHERE sic_seq = p_col_seq;

BEGIN

  OPEN c_get_storet_key(p_code, p_imp_cfg_seq, p_col_seq);
  FETCH c_get_storet_key
    INTO v_seq, v_storet_pk.is_number, v_storet_pk.org_id, v_converted_item;
  CLOSE c_get_storet_key;

  IF v_seq IS NULL THEN
    v_converted_item := p_code;
    p_trans_exist := FALSE;
  ELSIF v_seq IS NOT NULL THEN
    /* if the code is in the translations table, flag so it's not recreated */
    p_trans_exist := TRUE;
    IF v_storet_pk.is_number IS NULL THEN
      /* if there is a code in the translations table but no value to translate to, write error message
      status will be changed to 'N' later in the program */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_code ||
                                           ' exists in translations table but has not been translated to a value.');
          
             
    ELSE
      OPEN c_imp_cols(p_col_seq);
      FETCH c_imp_cols
        INTO v_table_name, v_column_name;
      CLOSE c_imp_cols;
      OPEN c_comma(p_col_seq);
      FETCH c_comma
        INTO v_comma_pos;
      CLOSE c_comma;
      IF v_comma_pos = 0 THEN
        v_dyna_cursor := 'SELECT RTRIM(' || v_column_name || ') ' || ' FROM ' ||
                         v_table_name || ' WHERE ' || v_table_name || '_IS_NUMBER = ' ||
                         v_storet_pk.is_number || ' AND ' || v_table_name ||
                         '_ORG_ID = ''' || v_storet_pk.org_id || '''';
      ELSE
        OPEN c_inst_col(p_col_seq);
        FETCH c_inst_col
          INTO v_column_name;
        CLOSE c_inst_col;
        v_dyna_cursor := 'SELECT RTRIM(' || v_column_name || ') ' || ' FROM ' ||
                         v_table_name || ' WHERE ' || v_table_name || '_IS_NUMBER = ' ||
                         v_storet_pk.is_number || ' AND ' || v_table_name ||
                         '_ORG_ID = ''' || v_storet_pk.org_id || '''';
      END IF;
    
      OPEN sort_cursor FOR v_dyna_cursor;
      LOOP
        FETCH sort_cursor
          INTO v_converted_item;
        EXIT WHEN sort_cursor%NOTFOUND;
      END LOOP;
      IF v_converted_item IS NOT NULL THEN
        p_trans_exist := TRUE;
      END IF;
    END IF;
  END IF;
  RETURN v_storet_pk;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in check storet translations');
        
           
    RETURN NULL;
END check_storet_translation;

/****************************************************************************************

****************************************************************************************/
PROCEDURE create_trans_record
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER
) IS

  v_st_seq    NUMBER(10);
  v_sicdt_seq NUMBER(10);
  e_imp_cfg_dets EXCEPTION;
  e_trans_error EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT sicdt_seq
      FROM sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL
      FROM dual;

BEGIN
  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq
    INTO v_sicdt_seq;
  IF c_get_cfg_det_seq%NOTFOUND THEN
    v_sicdt_seq := NULL;
    RAISE e_imp_cfg_dets;
  END IF;
  CLOSE c_get_cfg_det_seq;
  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq
    INTO v_st_seq;
  IF c_get_st_seq%NOTFOUND THEN
    v_st_seq := NULL;
    RAISE e_trans_error;
  END IF;
  CLOSE c_get_st_seq;
  INSERT INTO sim_translations
    (st_seq,
     st_sicdt_seq,
     st_user_code,
     st_storet_code)
  VALUES
    (v_st_seq, /* st_seq */
     v_sicdt_seq, /* st_sicdt_seq */
     p_code, /* st_user_code */
     NULL /* st_storet_code */);
EXCEPTION
  WHEN e_imp_cfg_dets THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                         
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Unable to find import configuration - stopped import');
        
           
    raise_application_error(-20101, 'Unable to find import configuration');
  WHEN e_trans_error THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                             
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Unable to create new record in translations table - error getting sequence');
        
           
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - adding record to translations table.');
        
           
END create_trans_record;

/****************************************************************************************
 function to look in the translations and allowable values tables for items that are 
 hard coded values in STORET
****************************************************************************************/
FUNCTION validate_alval_item
(
  p_item_name   IN VARCHAR2,
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER
) RETURN NUMBER IS

  v_converted_item VARCHAR2(80);
  v_item_sia_seq   NUMBER(10);
  v_trans_exist    BOOLEAN;
  v_instr_code     VARCHAR2(15);

BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
  hard coded values in STORET */
  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    /* look in allowable values */
    v_item_sia_seq := sim_import_pkg.check_allowable_values(v_converted_item, p_col_seq, v_line_cnt);
    IF v_item_sia_seq IS NULL
       AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
    
      v_instr_code := get_instructions(p_imp_cfg_seq, 'STA-TR');
      IF v_instr_code = 'STA-TR-01' THEN
        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not an allowable value - translation record has been created.');
            
               
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             p_item_name || ', ' ||
                                             v_converted_item ||
                                             ' is not an allowable value.');
            
               
      END IF;
    ELSIF v_item_sia_seq IS NULL
          AND v_trans_exist THEN
      /* item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_orig_item ||
                                           ' has been mapped to the invalid storet value: ' ||
                                           v_converted_item);
          
             
    END IF;
  END IF;
  RETURN v_item_sia_seq;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - in validate allowable values.');
        
           
    RETURN NULL;
END validate_alval_item;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_date
(
  p_raw_date IN VARCHAR2,
  p_col_seq  IN NUMBER,
  p_cfg_seq  IN NUMBER,
  p_line_cnt IN NUMBER,
  p_seq      IN NUMBER
) RETURN DATE IS

  v_start_date DATE;
  v_fmt        VARCHAR2(80);

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    v_fmt := NULL;
    /* no format specified, use default */
    v_start_date := trunc(to_date(p_raw_date, 'MM-DD-RRRR'));
  END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_date := trunc(to_date(p_raw_date, v_fmt));
  END IF;
  RETURN v_start_date;

EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'date, ' ||
                                         p_raw_date ||
                                         ', format does not match given import format ');
        
           
    RETURN NULL;
END get_date;

/****************************************************************************************

****************************************************************************************/
FUNCTION get_time
(
  p_raw_time IN VARCHAR2,
  p_col_seq  IN NUMBER,
  p_cfg_seq  IN NUMBER
) RETURN DATE IS
  v_start_time DATE;
  v_fmt        VARCHAR2(80);
  e_no_cfg_found EXCEPTION;
  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    v_fmt := NULL;
    /* no format specified, use default */
    v_start_time := to_date(substr(p_raw_time, 1, 8), 'HH24:MI:SS');
  END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_time := to_date(substr(p_raw_time, 1, 8), v_fmt);
  END IF;
  RETURN v_start_time;
EXCEPTION
  WHEN e_no_cfg_found THEN
    raise_application_error(-20100, 'Import configuration for start or end time is incomplete or does not exist.');
    RETURN NULL;
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                          
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - in get time.');
        
           
    RETURN NULL;
END get_time;

/****************************************************************************************
 The get_station_is_nbr function finds the station IS number and ORG id from the Storet 
 table TSMSTATN The values are returned in a variable that is TYPE T_STORET_PK_TYPE 
 (HOLDS BOTH THE ORG AND IS #)
****************************************************************************************/
FUNCTION get_station_is_nbr
(
  p_station_id IN VARCHAR2,
  p_org_id     IN VARCHAR2,
  p_item_name  IN VARCHAR2,
  p_sta_seq    IN NUMBER
) RETURN t_storet_pk_type IS
  v_station_pk t_storet_pk_type;
  CURSOR c_find_station(p_station_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT tsmstatn_is_number,
           tsmstatn_org_id
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_id, 15)
       AND tsmstatn_org_id = rpad(p_org_id, 8);
BEGIN
  OPEN c_find_station(p_station_id, p_org_id);
  FETCH c_find_station
    INTO v_station_pk;
  IF c_find_station%NOTFOUND THEN
    v_station_pk.is_number := NULL;
    v_station_pk.org_id := NULL;
  END IF;
  CLOSE c_find_station;
  RETURN v_station_pk;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - PROC get_station_is_nbr');
        
           
    v_station_pk.is_number := NULL;
    v_station_pk.org_id := NULL;
    RETURN v_station_pk;
END get_station_is_nbr;

/****************************************************************************************
 The get_tsmvstc_is_nbr function finds the primary/secondary type IS number and ORG id 
 from the Storet table TSMVSTC The values are returned in a variable that is TYPE 
 T_STORET_TSMVSTC_TYPE (HOLDS BOTH THE ORG AND IS #)
****************************************************************************************/
FUNCTION get_tsmvstc_is_nbr
(
  p_vstc_primary_id   IN VARCHAR2,
  p_vstc_secondary_id IN VARCHAR2,
  p_item_name         IN VARCHAR2,
  p_sta_seq           IN NUMBER
) RETURN t_storet_tsmvstc_type IS

  v_storet_tsmvstc t_storet_tsmvstc_type;
  v_rowcount       NUMBER(10);
  v_trans_2ndary   VARCHAR2(30);
  CURSOR c_count_primary_types IS
    SELECT COUNT(primary_type_cd)
      FROM tsmvstc
     WHERE primary_type_cd = p_vstc_primary_id;
  CURSOR c_find_vstc(p_vstc_primary_id IN VARCHAR2, p_vstc_secondary_id IN VARCHAR2) IS
    SELECT tsmvstc_is_number,
           tsmvstc_org_id
      FROM tsmvstc
     WHERE primary_type_cd = p_vstc_primary_id
       AND secondary_type_cd = p_vstc_secondary_id;
  CURSOR c_find_vstc2(p_vstc_primary_id IN VARCHAR2) IS
    SELECT tsmvstc_is_number,
           tsmvstc_org_id
      FROM tsmvstc
     WHERE primary_type_cd = p_vstc_primary_id;
BEGIN
  v_trans_2ndary := translate(p_vstc_secondary_id, '1 ', '1'); -- If v_trans_secondary is nothing but spaces then this will be a null
  BEGIN
    OPEN c_count_primary_types;
    FETCH c_count_primary_types
      INTO v_rowcount;
    IF c_count_primary_types%NOTFOUND THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
    ELSIF v_rowcount > 1
          AND v_trans_2ndary IS NULL THEN
      v_storet_tsmvstc.tsmvstc_is_number := NULL;
      v_storet_tsmvstc.tsmvstc_org_id := NULL;
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           p_vstc_primary_id ||
                                           ' -Primary Station Type Needs Secondary Type');
          
             
      v_status := 'N';
      RETURN v_storet_tsmvstc;
    END IF;
    CLOSE c_count_primary_types;
  EXCEPTION
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || p_vstc_primary_id ||
                                           ' - count primary type line:' ||
                                           v_line_cnt);
          
             
  END;
  IF p_vstc_secondary_id IS NOT NULL THEN
    BEGIN
      OPEN c_find_vstc(p_vstc_primary_id, p_vstc_secondary_id);
      FETCH c_find_vstc
        INTO v_storet_tsmvstc;
      IF c_find_vstc%NOTFOUND THEN
        v_storet_tsmvstc.tsmvstc_is_number := NULL;
      END IF;
      CLOSE c_find_vstc;
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             SQLERRM ||
                                             ' -Error: Cannot find TSMVSTC IS number or Org Id');
            
               
    END;
  END IF;
  IF p_vstc_secondary_id IS NULL THEN
    BEGIN
      OPEN c_find_vstc2(p_vstc_primary_id);
      FETCH c_find_vstc2
        INTO v_storet_tsmvstc;
      IF c_find_vstc2%NOTFOUND THEN
        v_storet_tsmvstc.tsmvstc_is_number := NULL;
      END IF;
      CLOSE c_find_vstc2;
    EXCEPTION
      WHEN OTHERS THEN
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                             SQLERRM ||
                                             ' -Primaty Type error in find vstc2, line:' ||
                                             v_line_cnt);
            
               
    END;
  END IF;
  RETURN v_storet_tsmvstc;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - In Function GET_VSTC_IS_NBR');
        
           
    v_storet_tsmvstc.tsmvstc_is_number := NULL;
    v_storet_tsmvstc.tsmvstc_org_id := NULL;
    RETURN v_storet_tsmvstc;
END get_tsmvstc_is_nbr;

/****************************************************************************************
 The get_tsmfhu_is_nbr function finds the hydrologic unit IS number and ORG id from the 
 Storet table TSMFHU The values are returned in a variable that is 
 TYPE T_STORET_PK_TYPE (HOLDS BOTH THE ORG AND IS #)
****************************************************************************************/
FUNCTION get_tsmfhu_is_nbr(p_fhu IN VARCHAR2) RETURN t_storet_pk_type IS
  v_storet_tsmfhu t_storet_pk_type;
  CURSOR c_find_fhu(p_fhu IN VARCHAR2) IS
    SELECT tsmfhu_is_number,
           tsmfhu_org_id
      FROM tsmfhu
     WHERE hydrologic_unit_cd = rpad(p_fhu, 8);
BEGIN
  BEGIN
    OPEN c_find_fhu(p_fhu);
    FETCH c_find_fhu
      INTO v_storet_tsmfhu;
    IF c_find_fhu%NOTFOUND THEN
      v_storet_tsmfhu.is_number := NULL;
    END IF;
    CLOSE c_find_fhu;
  EXCEPTION
    WHEN OTHERS THEN
       
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || ' - error finding fhu');
          
             
  END;
  RETURN v_storet_tsmfhu;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - Function get_fhu_is_nbr ');
        
          
    v_storet_tsmfhu.is_number := NULL;
    v_storet_tsmfhu.org_id := NULL;
    RETURN v_storet_tsmfhu;
END get_tsmfhu_is_nbr;

/****************************************************************************************
 The get_tsmgeopa_is_nbr function finds the station IS number and ORG id from the Storet 
 table TSMGEOPA.  The values are returned in a variable that is TYPE T_STORET_PK_TYPE 
 (HOLDS BOTH THE ORG AND IS #). This function accounts for the 4 possible combinations 
 of state and county ID's.
   1 It's a state postal code and a county name
   2 It's a state FIPS and a county name
   3 It's a state postal code and  county FIPS
   4 It's a  state FIPS and county FIPS
****************************************************************************************/
FUNCTION get_tsmgeopa_is_nbr
(
  p_state  IN VARCHAR2,
  p_county IN VARCHAR2
) RETURN t_storet_pk_type IS
  v_storet_tsmgeopa t_storet_pk_type;
  /*** four cursors for the four possible combinations of state and county id's *****/
  CURSOR c_find_geopa_1(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_postal_code = rpad(p_state, 2)
       AND county_name = p_county;
  CURSOR c_find_geopa_2(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_fips_code = rpad(p_state, 2)
       AND county_name = p_county;
  CURSOR c_find_geopa_3(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_postal_code = rpad(p_state, 2)
       AND fips_county_code = rpad(p_county, 3);
  CURSOR c_find_geopa_4(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_fips_code = rpad(p_state, 2)
       AND fips_county_code = rpad(p_county, 3);
BEGIN
  BEGIN
    IF translate(p_state, 'a1234567890.- ', 'a') IS NOT NULL
       AND translate(p_county, 'a1234567890.- ', 'a') IS NOT NULL THEN
      -- it's a state postal and a county NAME
      OPEN c_find_geopa_1(p_state, p_county);
      FETCH c_find_geopa_1
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_1%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_1;
    ELSIF translate(p_state, 'a1234567890.- ', 'a') IS NULL
          AND translate(p_county, 'a1234567890.- ', 'a') IS NOT NULL THEN
      -- It's a state fips and a county name
      OPEN c_find_geopa_2(p_state, p_county);
      FETCH c_find_geopa_2
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_2%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_2;
    ELSIF translate(p_state, 'a1234567890.- ', 'a') IS NOT NULL
          AND translate(p_county, 'a1234567890.- ', 'a') IS NULL THEN
      -- it's a state postal and  county fips
      OPEN c_find_geopa_3(p_state, p_county);
      FETCH c_find_geopa_3
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_3%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_3;
    ELSIF translate(p_state, 'a1234567890.- ', 'a') IS NULL
          AND translate(p_county, 'a1234567890.- ', 'a') IS NULL THEN
      -- it's a  state fips and county FIPS
      OPEN c_find_geopa_4(p_state, p_county);
      FETCH c_find_geopa_4
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_4%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_4;
    ELSE
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           'get TSMgeopa IS COMBO NOT FOUND: ' ||
                                           v_storet_tsmgeopa.is_number ||
                                           'p_state -> ' || p_state ||
                                           ' p_county -> ' || p_county);
          
            
    END IF;
  EXCEPTION
    WHEN OTHERS THEN
       
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM || ' - find geopa');
          
            
  END;
  RETURN v_storet_tsmgeopa;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - PROC GET_geopa_IS_NBR ');
        
          
    v_storet_tsmgeopa.is_number := NULL;
    v_storet_tsmgeopa.org_id := NULL;
    RETURN v_storet_tsmgeopa;
END get_tsmgeopa_is_nbr;

/****************************************************************************************
 RIVER REACH
 
 The get_tsmrrr_is_nbr function finds the station IS number and ORG id from  the Storet
 table TSMRRR The values are returned in a variable that is TYPE T_STORET_PK_TYPE 
 (HOLDS BOTH THE ORG AND IS #)
****************************************************************************************/
FUNCTION get_tsmrrr_is_nbr
(
  p_huc IN VARCHAR2,
  p_rf1 IN VARCHAR2
) RETURN t_storet_pk_type IS
  v_storet_tsmrrr t_storet_pk_type;
  CURSOR c_find_tsmrrr(p_huc IN VARCHAR2, p_rf1 IN VARCHAR2) IS
    SELECT tsmrrr_is_number,
           tsmrrr_org_id
      FROM tsmrrr
     WHERE hydrologic_unit_cd = rpad(p_huc, 8)
       AND segment_code = rpad(p_rf1, 3);
BEGIN
  BEGIN
    OPEN c_find_tsmrrr(p_huc, p_rf1);
    FETCH c_find_tsmrrr
      INTO v_storet_tsmrrr;
    IF c_find_tsmrrr%NOTFOUND THEN
      v_storet_tsmrrr.is_number := NULL;
    END IF;
    CLOSE c_find_tsmrrr;
  EXCEPTION
    WHEN OTHERS THEN
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           SQLERRM ||
                                           ' - find tsmrrr_is_number');
          
            
  END;
  RETURN v_storet_tsmrrr;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - PROC GET_TSMRRR_IS_NBR ');
        
          
    v_storet_tsmrrr.is_number := NULL;
    v_storet_tsmrrr.org_id := NULL;
    RETURN v_storet_tsmrrr;
END get_tsmrrr_is_nbr;

/****************************************************************************************
 The create_station function inserts known good data into the SIM table SIM_STATIONS
 After the station has loaded sucessfully the function returns a 1
 If there was a problem the function returns a zero.
****************************************************************************************/
FUNCTION create_station(p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create station using values in station record */
  INSERT INTO sim_stations
    (sta_seq,
     sta_id,
     sta_name,
     sta_desc,
     sta_org_id,
     sta_tsmorgan_org_id,
     sta_tsmorgan_is_number,
     sta_status,
     sta_estab_date,
     sta_export_seq,
     sta_import_seq,
     sta_tsmvstc_is_number,
     sta_tsmvstc_org_id,
     sta_epa_key_id,
     sta_zid_rel,
     sta_influence_area,
     sta_travel_dir,
     sta_water_depth,
     sta_water_depth_unit,
     sta_ecoregion,
     sta_document)
  VALUES
    (p_station_rec.sta_seq, /* STA_SEQ            */
     p_station_rec.sta_id, /* STA_ID             */
     p_station_rec.sta_name, /* STA_NAME           */
     p_station_rec.sta_desc, /* STA_DESC           */
     p_station_rec.sta_org_id, /* STATION ORG ID     */
     p_station_rec.sta_tsmorgan_org_id, /* TSMORGAN_ORG_ID    */
     p_station_rec.sta_tsmorgan_is_number, /* TSMORGAN_IS_NUMBER */
     p_station_rec.sta_status, /* STA_STATUS         */
     p_station_rec.sta_estab_date, /* STA_ESTAB_DATE     */
     p_station_rec.sta_export_seq, /* STA_EXPORT_SEQ     */
     p_station_rec.sta_import_seq, /* STA_IMPORT_SEQ     */
     p_station_rec.sta_tsmvstc_is_number, /* STA_TSMVSTC_IS_NUMBER     */
     p_station_rec.sta_tsmvstc_org_id, /* STA_TSMVSTC_ORG_ID */
     p_station_rec.sta_epa_key_id,
     p_station_rec.sta_zid_rel,
     p_station_rec.sta_influence_area,
     p_station_rec.sta_travel_dir,
     p_station_rec.sta_water_depth,
     p_station_rec.sta_water_depth_unit,
     p_station_rec.sta_ecoregion,
     p_station_rec.sta_document);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - inserting station.');
        
          
    RETURN 0;
END create_station;

/****************************************************************************************
 The create_station_location function inserts known good data into the SIM table SIM_STATIONS
 After the station_location has loaded sucessfully the function returns a 1
 If there was a problem the function returns a zero.
****************************************************************************************/
FUNCTION create_station_location(p_station_location_rec IN sim_station_locations%ROWTYPE)
  RETURN NUMBER IS
BEGIN
  /* create station using values in station record */
  INSERT INTO sim_station_locations
    (stl_seq,
     stl_sta_seq,
     stl_type_code,
     stl_sequence_number,
     stl_elev_datum,
     stl_elev_method,
     stl_elev_msr_date,
     stl_elevation_msr,
     stl_elevation_unit,
     stl_geopos_datum,
     stl_geopos_method,
     stl_gps_scale,
     stl_lat_dir,
     stl_lat_degrees,
     stl_lat_minutes,
     stl_lat_seconds,
     stl_long_dir,
     stl_long_degrees,
     stl_long_minutes,
     stl_long_seconds,
     stl_lat_long_msr_date,
     stl_latlong_accurcy,
     stl_latlong_accurcy_unit,
     stl_tsmrrr_is_number,
     stl_tsmrrr_org_id,
     stl_tsmfhu_is_number,
     stl_tsmfhu_org_id,
     stl_tsmstatn0is_number,
     stl_tsmstatn0org_id,
     stl_tsmwell_is_number,
     stl_tsmwell_org_id,
     stl_tsmgeopa_is_number,
     stl_tsmgeopa_org_id,
     stl_lat_dec_minutes,
     stl_long_dec_minutes,
     stl_tsmpipe_is_number,
     stl_tsmpipe_org_id,
     stl_tsmstatn1org_id,
     stl_tsmstatn1is_number,
     stl_tsmgeopa0is_number,
     stl_tsmgeopa0org_id,
     stl_tsmnal_cd,
     stl_tsmnal_state,
     stl_nrcs,
     stl_status,
     stl_import_seq,
     stl_export_seq,
     stl_lat_dec_degrees,
     stl_long_dec_degrees,
     stl_point_name)
  VALUES
    (p_station_location_rec.stl_seq,
     p_station_location_rec.stl_sta_seq,
     p_station_location_rec.stl_type_code,
     p_station_location_rec.stl_sequence_number,
     p_station_location_rec.stl_elev_datum,
     p_station_location_rec.stl_elev_method,
     p_station_location_rec.stl_elev_msr_date,
     p_station_location_rec.stl_elevation_msr,
     p_station_location_rec.stl_elevation_unit,
     p_station_location_rec.stl_geopos_datum,
     p_station_location_rec.stl_geopos_method,
     p_station_location_rec.stl_gps_scale,
     p_station_location_rec.stl_lat_dir,
     p_station_location_rec.stl_lat_degrees,
     p_station_location_rec.stl_lat_minutes,
     p_station_location_rec.stl_lat_seconds,
     p_station_location_rec.stl_long_dir,
     p_station_location_rec.stl_long_degrees,
     p_station_location_rec.stl_long_minutes,
     p_station_location_rec.stl_long_seconds,
     p_station_location_rec.stl_lat_long_msr_date,
     p_station_location_rec.stl_latlong_accurcy,
     p_station_location_rec.stl_latlong_accurcy_unit,
     p_station_location_rec.stl_tsmrrr_is_number,
     p_station_location_rec.stl_tsmrrr_org_id,
     p_station_location_rec.stl_tsmfhu_is_number,
     p_station_location_rec.stl_tsmfhu_org_id,
     p_station_location_rec.stl_tsmstatn0is_number,
     p_station_location_rec.stl_tsmstatn0org_id,
     p_station_location_rec.stl_tsmwell_is_number,
     p_station_location_rec.stl_tsmwell_org_id,
     p_station_location_rec.stl_tsmgeopa_is_number,
     p_station_location_rec.stl_tsmgeopa_org_id,
     p_station_location_rec.stl_lat_dec_minutes,
     p_station_location_rec.stl_long_dec_minutes,
     p_station_location_rec.stl_tsmpipe_is_number,
     p_station_location_rec.stl_tsmpipe_org_id,
     p_station_location_rec.stl_tsmstatn1org_id,
     p_station_location_rec.stl_tsmstatn1is_number,
     p_station_location_rec.stl_tsmgeopa0is_number,
     p_station_location_rec.stl_tsmgeopa0org_id,
     p_station_location_rec.stl_tsmnal_cd,
     p_station_location_rec.stl_tsmnal_state,
     p_station_location_rec.stl_nrcs,
     p_station_location_rec.stl_status,
     p_station_location_rec.stl_import_seq,
     p_station_location_rec.stl_export_seq,
     p_station_location_rec.stl_lat_dec_degrees,
     p_station_location_rec.stl_long_dec_degrees,
     p_station_location_rec.stl_point_name);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM ||
                                         ' - inserting station locations.');
        
          
    RETURN 0;
END create_station_location;

/****************************************************************************************
  The create_ocean function inserts known good data into the SIM table SIM_OCEAN
  After the ocean has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
*****************************************************************************************/
FUNCTION create_ocean(p_ocean_rec IN sim_ocean%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create ocean using values in ocean record */
  INSERT INTO sim_ocean
    (soc_seq,
     soc_stl_seq,
     soc_name,
     soc_additional_loc,
     soc_shore_relation,
     soc_shore_distance,
     soc_shore_dist_unit,
     soc_reference_pt,
     soc_loran_c_reading_1,
     soc_loran_c_reading_2,
     soc_bottom_topography,
     soc_tsmalp_is_number,
     soc_tsmalp_org_id)
  VALUES
    (p_ocean_rec.soc_seq,
     p_ocean_rec.soc_stl_seq,
     p_ocean_rec.soc_name,
     p_ocean_rec.soc_additional_loc,
     p_ocean_rec.soc_shore_relation,
     p_ocean_rec.soc_shore_distance,
     p_ocean_rec.soc_shore_dist_unit,
     p_ocean_rec.soc_reference_pt,
     p_ocean_rec.soc_loran_c_reading_1,
     p_ocean_rec.soc_loran_c_reading_2,
     p_ocean_rec.soc_bottom_topography,
     p_ocean_rec.soc_tsmalp_is_number,
     p_ocean_rec.soc_tsmalp_org_id);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - inserting ocean.');
        
          
    RETURN 0;
END create_ocean;

/****************************************************************************************
 The create_estuary function inserts known good data into the SIM table SIM_ESTUARY_LOC
 After the estuary has loaded sucessfully the function returns a 1
 If there was a problem the function returns a zero.
****************************************************************************************/
FUNCTION create_estuary(p_est_rec IN sim_estuary_loc%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create ocean using values in ocean record */
  INSERT INTO sim_estuary_loc
    (sel_seq,
     sel_tsmalp_is_number,
     sel_tsmalp_org_id,
     sel_other_estuary,
     sel_add_loc_name,
     sel_shore_dist,
     sel_shore_dist_unit,
     sel_ref_pt,
     sel_tsmestry_org_id,
     sel_tsmestry_is_number,
     sel_stl_seq)
  VALUES
    (p_est_rec.sel_seq,
     p_est_rec.sel_tsmalp_is_number,
     p_est_rec.sel_tsmalp_org_id,
     p_est_rec.sel_other_estuary,
     p_est_rec.sel_add_loc_name,
     p_est_rec.sel_shore_dist,
     p_est_rec.sel_shore_dist_unit,
     p_est_rec.sel_ref_pt,
     p_est_rec.sel_tsmestry_org_id,
     p_est_rec.sel_tsmestry_is_number,
     p_est_rec.sel_stl_seq);
  COMMIT;
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - inserting estuary.');
        
          
    RETURN 0;
END create_estuary;

/****************************************************************************************
  The create_lake function inserts known good data into the SIM table SIM_GREAT_LAKE
  After the great lake has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
*****************************************************************************************/
FUNCTION create_lake(p_lake_rec IN sim_great_lake%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create ocean using values in ocean record */
  INSERT INTO sim_great_lake
    (sgl_seq,
     sgl_tsmalp_is_number,
     sgl_tsmalp_org_id,
     sgl_lake_name,
     sgl_add_lake_name,
     sgl_shore_dist,
     sgl_shore_dist_unit,
     sgl_ref_pt,
     sgl_stl_seq)
  VALUES
    (p_lake_rec.sgl_seq,
     p_lake_rec.sgl_tsmalp_is_number,
     p_lake_rec.sgl_tsmalp_org_id,
     p_lake_rec.sgl_lake_name,
     p_lake_rec.sgl_add_lake_name,
     p_lake_rec.sgl_shore_dist,
     p_lake_rec.sgl_shore_dist_unit,
     p_lake_rec.sgl_ref_pt,
     p_lake_rec.sgl_stl_seq);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         SQLERRM || ' - inserting great lake.');
        
          
    RETURN 0;
END create_lake;

/****************************************************************************************
Procedure to get data from storet to be used in Sim for Updating Stations.
****************************************************************************************/
PROCEDURE get_storet_station (
  p_table_line      IN  sim_import_pkg.col_table_type,
  p_org_id          IN  VARCHAR2,
  p_station_rec     OUT tsmstatn%ROWTYPE,
  p_ocean_rec       OUT tsmocnlc%ROWTYPE,
  p_estuary_rec     OUT tsmestlc%ROWTYPE,
  p_great_lake_rec  OUT tsmgll%ROWTYPE,
  p_location_rec    OUT tsmalp%ROWTYPE) IS
  
  v_station_id  tsmstatn.identification_cd%TYPE := p_table_line(1);
  c_type_code CONSTANT tsmalp.type_code%TYPE := '*POINT OF RECORD';
BEGIN
  SELECT *
    INTO p_station_rec
    FROM tsmstatn
    WHERE identification_cd = v_station_id
    AND tsmstatn_org_id = rpad(p_org_id, 8);
  BEGIN
    SELECT *
      INTO p_location_rec
      FROM tsmalp
      WHERE tsmstatn0is_number = p_station_rec.tsmstatn_is_number
      AND tsmstatn0org_id = p_station_rec.tsmstatn_org_id
      AND type_code = c_type_code;  
  EXCEPTION
    WHEN no_data_found THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL, 
                                         SQLERRM || '-in additional Locations.');
  END;
  
  BEGIN                                         
    SELECT *
      INTO p_ocean_rec
      FROM tsmocnlc
      WHERE tsmalp_is_number = p_location_rec.tsmalp_is_number
      AND tsmalp_org_id = p_location_rec.tsmalp_org_id;
  EXCEPTION
    WHEN no_data_found THEN
      NULL;
  END;
  
  BEGIN    
    SELECT *
      INTO p_great_lake_rec
      FROM tsmgll
      WHERE tsmalp_is_number = p_location_rec.tsmalp_is_number
      AND tsmalp_org_id = p_location_rec.tsmalp_org_id;
  EXCEPTION
    WHEN no_data_found THEN
      NULL;
  END;
  
  BEGIN    
    SELECT *
      INTO p_estuary_rec
      FROM tsmestlc
      WHERE tsmalp_is_number = p_location_rec.tsmalp_is_number
      AND tsmalp_org_id = p_location_rec.tsmalp_org_id;
  EXCEPTION
    WHEN no_data_found THEN
      NULL;
  END;
  
EXCEPTION
  WHEN no_data_found THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL, 
                                         SQLERRM || ' Station ID is not found in Storet.'); 
  
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         SQLERRM || ' - in get_storet_station.');
                                         
END get_storet_station;

/****************************************************************************************
Procedure to Check for additional well locations.
****************************************************************************************/
PROCEDURE check_wells
(
  p_org_id            IN VARCHAR2,
  p_tsmvstc_is_number IN NUMBER
) IS
  
  v_well_cnt  NUMBER(10);

BEGIN
  SELECT COUNT(tsmwell_is_number)
    INTO v_well_cnt
    FROM tsmwell w, tsmstatn s
    WHERE s.tsmstatn_is_number = w.tsmstatn_is_number
    AND s.tsmstatn_org_id = w.tsmstatn_org_id
    AND s.tsmvstc_is_number = p_tsmvstc_is_number
    AND s.tsmvstc_org_id = p_org_id;
    
  IF v_well_cnt > 0 THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL,
                                         'Can not update a station of type Well to a different type when there are still additional wells associated with this station');
    v_status := 'N';                                         
  END IF;                                   
  
EXCEPTION
  WHEN no_data_found THEN
    NULL;
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,NULL, SQLERRM || ' - in checking for well locations');
                                          
END check_wells;

/****************************************************************************************
 The import_station procedure parses a text file line by line of data that is separated 
 by a delimiter (typically a pipe |).  The procedure is activated by filling out and 
 activating the import station dialog box in the SIM2 application. From there we will 
 assign an ORG ID and procede with opening the text file to extract the data
****************************************************************************************/
PROCEDURE import_station
(
  p_org_id         IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_imp_cfg_seq    IN NUMBER,
  p_test_count     IN NUMBER,
  p_skip_header    IN NUMBER DEFAULT 0 -- CLM 08/02/2005 Skip header option    
) IS
  --   1 = Skip  0 = Don't Skip

  v_import_desc        VARCHAR2(200);
  v_comp               NUMBER(1);
  v_num_cols           NUMBER(3);
  v_col_seq            NUMBER(10);
  v_dflt_pos           NUMBER(3);
  v_success            NUMBER(1);
  v_primary_checked    VARCHAR2(1);
  v_converted_item     VARCHAR2(4000);
  v_delimiter          CHAR(1);
  v_inline             LONG;
  v_bad_parse          BOOLEAN;
  v_item_name          VARCHAR2(80);
  v_item               VARCHAR2(4000); --sjj TEST 2000 TO 4000
  v_trans_exist        BOOLEAN := TRUE;
  v_line_table         sim_import_pkg.col_table_type;
  v_default_table      sim_import_station_pkg.col_table_type;
  v_pos_table          sim_import_station_pkg.col_table_num_type;

  v_storet_pk       t_storet_pk_type;
  v_storet_tsmvstc  t_storet_tsmvstc_type;
  v_storet_tsmgeopa t_storet_pk_type;
  v_storet_tsmfhu   t_storet_pk_type;

  v_rows         NUMBER(2);
  v_error_cnt    NUMBER(10);
  v_complete_cnt NUMBER(10);
  v_dummy_seq    NUMBER(10);

  v_stl_rec     sim_station_locations%ROWTYPE;
  v_sta_rec     sim_stations%ROWTYPE;
  v_soc_rec     sim_ocean%ROWTYPE;
  v_estuary_rec sim_estuary_loc%ROWTYPE;
  v_lake_rec    sim_great_lake%ROWTYPE;
  
  v_station_rec       tsmstatn%ROWTYPE;
  v_location_rec      tsmalp%ROWTYPE;
  v_ocean_rec         tsmocnlc%ROWTYPE;
  v_estuary_stor_rec  tsmestlc%ROWTYPE;
  v_great_lake_rec    tsmgll%ROWTYPE;
  
  v_county             VARCHAR2(25);
  v_elevation_unit     VARCHAR2(3);
  v_point_name         VARCHAR2(30);
  v_elevation_method   VARCHAR2(12);
  v_elevation_datum    VARCHAR2(12);
  v_elevation_date     DATE;
  v_tsmorgan_is_number VARCHAR2(8);
  v_tsmgeopa_org_id    VARCHAR2(8);
  v_tsmgeopa_is_number NUMBER(8);
  v_tsmfhu_org_id      VARCHAR2(8);
  v_tsmfhu_is_number   NUMBER(8);
  v_tsmstatn0is_number tsmalp.tsmstatn0is_number%TYPE;
  v_tsmstatn0org_id    tsmalp.tsmstatn0org_id%TYPE; 
  v_name               VARCHAR2(60);
  v_type1              VARCHAR2(30);
  v_secondary_type     VARCHAR2(30);
  v_stor_type          VARCHAR2(30);
  v_estab_date         DATE;
  v_seq                NUMBER(10);
  v_stl_seq            NUMBER(10);
  v_id                 VARCHAR2(15);
  v_nrcs               VARCHAR2(8);
  v_sta_desc           VARCHAR2(4000);--sjj TEST FROM 1999 TO 4000
  v_ecoregion          VARCHAR2(60);
  v_water_depth        NUMBER(8, 3);
  v_water_depth_unit   VARCHAR2(2);
  v_elevation_msr      NUMBER(9, 4);
  v_lat_long_date      DATE;
  v_lat_degrees        NUMBER(2);
  v_lat_minutes        NUMBER(2);
  v_lat_seconds        NUMBER(6, 4);
  v_lat_dir            VARCHAR2(1);
  v_check_lat_dir      VARCHAR2(1);
  v_lat_dec_min        NUMBER(6, 4);
  v_lat_dec_deg        NUMBER(10, 7);
  v_long_degrees       NUMBER(3);
  v_long_minutes       NUMBER(2);
  v_long_seconds       NUMBER(6, 4);
  v_long_dec_min       NUMBER(6, 4);
  v_long_dec_deg       NUMBER(10, 7);
  v_long_dir           VARCHAR2(1);
  v_check_long_dir     VARCHAR2(1);
  v_datum              VARCHAR2(12);
  v_method             VARCHAR2(12);
  v_gps_scale          VARCHAR2(20);
  v_state              VARCHAR2(2);
  v_ocean_seq          NUMBER(10);
  v_ocean_name         VARCHAR2(30);
  v_ocean_ref          VARCHAR2(30);
  v_ocean_bottom       VARCHAR2(256);
  v_add_ocean          VARCHAR2(30);
  v_ocean_dist         NUMBER(6, 2);
  v_ocean_dist_unit    VARCHAR2(3);
  v_shore_relation     VARCHAR2(10);
  v_native_name        VARCHAR2(40);
  v_native_code        CHAR(5);
  v_native             VARCHAR2(5);
  v_native_state       CHAR(2);
  v_huc                VARCHAR2(8);
  v_comment            VARCHAR2(254);
  v_epa_key            VARCHAR2(36);
  v_influence          VARCHAR2(120);
  v_travel_dir         VARCHAR2(1999);
  v_zid_rel            VARCHAR2(2);
  v_lake_seq           NUMBER(10);
  v_great_lake         VARCHAR2(15);
  v_lake_ref           VARCHAR2(30);
  v_add_lake           VARCHAR2(30);
  v_lake_dist          NUMBER(6, 2);
  v_lake_dist_unit     VARCHAR2(3);
  v_estuary_seq        NUMBER(10);
  v_estuary_state      VARCHAR2(40);
  v_primary_est        VARCHAR2(30);
  v_secondary_est      VARCHAR2(30);
  v_primary_is_nbr     NUMBER(8);
  v_secondary_is_nbr   NUMBER(8);
  v_tsmestry_is_nbr    NUMBER(8);
  v_estuary_dist       NUMBER(6, 2);
  v_est_dist_unit      VARCHAR2(3);
  v_estuary_ref        VARCHAR2(30);
  v_add_estuary        VARCHAR2(30);
  v_other_estuary      VARCHAR2(30);
  v_document           VARCHAR2(1999);
  v_next               NUMBER(2) := 0;
  v_sid_seq            NUMBER(10); 
  v_action_code        VARCHAR2(1); 

  e_bad_org EXCEPTION;
  e_no_cfg EXCEPTION;
  e_no_delimiter EXCEPTION;
  e_bad_columns EXCEPTION;

  CURSOR c_get_station_seq IS
    SELECT sim_stations_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_station_location_seq IS
    SELECT sim_station_locations_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_ocean_seq IS
    SELECT sim_ocean_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_estuary_seq IS
    SELECT sim_estuary_loc_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_great_lake_seq IS
    SELECT sim_great_lake_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT sipl_seq,
           sipl_line_text,
           sipl_line_no
      FROM sim_imp_lines
     WHERE sipl_sil_seq = p_import_seq
       AND sipl_line_text IS NOT NULL -- bka  added to eliminate blank lines from the import
     ORDER BY sipl_line_no;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT sic_proc_order,
           sic_dflt_pos,
           sic_name,
           sic_seq
      FROM sim_imp_cols,
           sim_imp_defs,
           sim_imp_cfg
     WHERE sicc_sid_seq = sid_seq
       AND sic_sid_seq = sid_seq
       AND sicc_seq = p_imp_cfg_seq
     ORDER BY sic_proc_order;

  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(sta_seq)
      FROM sim_stations
     WHERE sta_import_seq = p_import_log_seq
       AND sta_status = 'A';

  CURSOR c_count_primary_types IS -- DETERMINES IF THERE IS A SECONDARY TYPE
    SELECT COUNT(primary_type_cd)
      FROM tsmvstc
     WHERE primary_type_cd = v_type1;

BEGIN
  /** Find out import type **/
  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  v_line_cnt := 0;
  BEGIN
    
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = p_imp_cfg_seq;
  
  sim_event_log_pkg.write_event(1,v_import_log_seq,p_org_id,v_sid_seq,NULL);
   
   
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Start: ' ||
                                         to_char(SYSDATE, 'HH24:MI:SS'));
        
                                       
    IF p_skip_header = 1 THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,'Header skipped');
      v_line_cnt := 1;
    END IF;
  
    /* get tsmorgan_is_number */
    v_success := get_org_info(p_org_id, v_tsmorgan_is_number);
    IF v_success != 1
       OR v_org_is_nbr IS NULL THEN
      RAISE e_bad_org;
    END IF;
  
    /* get the delimiter */
    v_delimiter := NULL;
    v_success := get_delimiter(p_imp_cfg_seq, v_delimiter);
    IF v_success != 1 THEN
      RAISE e_no_delimiter;
    END IF;
  
    /* get actual positions and default information for items being imported */
    v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table, v_num_cols);
    IF v_success != 1 THEN
      RAISE e_no_cfg;
    END IF;
    
    /* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
      /* CLM 08/02/2005 If skip header = 'No' or this is not the first row */
      IF p_skip_header = 0
         OR v_text_line.sipl_line_no <> 1 THEN
        /* read one line */
        /**** get station sequence ******/
        OPEN c_get_station_seq;
        FETCH c_get_station_seq
          INTO v_seq;
        CLOSE c_get_station_seq;
        v_inline := v_text_line.sipl_line_text;
        v_comment := NULL;
        v_line_cnt := v_line_cnt + 1;
        v_status := 'A';
        v_primary_checked := 'N';
        v_lat_dir := NULL;
        v_long_dir := NULL;
      
        -- send line to parsing routine:
        sim_import_pkg.parse_line_table(v_inline, v_line_table, v_num_cols, v_delimiter, v_bad_parse, v_import_log_seq);
        IF v_bad_parse THEN
          RAISE e_bad_columns;
        END IF;
        v_next := 0;
        
        /* Code added to implement Updating Stations through SIM*/
        SELECT sid_action_code
          INTO v_action_code
          FROM sim_imp_defs
          WHERE sid_seq = v_sid_seq;
        IF v_action_code = 'U' THEN
          get_storet_station(
            v_line_table,
            p_org_id,
            v_station_rec,
            v_ocean_rec,
            v_estuary_stor_rec,
            v_great_lake_rec, 
            v_location_rec);
          
          v_elevation_unit     := v_location_rec.elvtn_unt_cd;     
          v_point_name         := v_location_rec.point_name;
          v_elevation_method   := v_location_rec.elvtn_method_cd;  
          v_elevation_datum    := v_location_rec.elevation_datum_cd; 
          v_elevation_date     := v_location_rec.elvtn_msr_dt; 
          v_storet_tsmvstc.tsmvstc_org_id     := v_station_rec.tsmvstc_org_id; 
          v_storet_tsmvstc.tsmvstc_is_number  := v_station_rec.tsmvstc_is_number; 
          v_tsmorgan_is_number := v_station_rec.tsmorgan_is_number;
          v_tsmgeopa_org_id    := v_location_rec.tsmgeopa_org_id;
          v_tsmgeopa_is_number := v_location_rec.tsmgeopa_is_number;
          v_tsmfhu_org_id      := v_location_rec.tsmfhu_org_id;
          v_tsmfhu_is_number   := v_location_rec.tsmfhu_is_number;
          v_tsmstatn0is_number := v_location_rec.tsmstatn0is_number;
          v_tsmstatn0org_id    := v_location_rec.tsmstatn0org_id;
          v_name               := v_station_rec.NAME;
          
          SELECT primary_type_cd, secondary_type_cd
            INTO v_type1, v_secondary_type
            FROM tsmvstc
            WHERE tsmvstc_is_number = v_storet_tsmvstc.tsmvstc_is_number
            AND tsmvstc_org_id = v_storet_tsmvstc.tsmvstc_org_id;
          v_stor_type          := v_type1;
          v_estab_date         := v_station_rec.establishment_date;       
          v_id                 := v_station_rec.identification_cd;
          v_nrcs               := v_location_rec.nrcs_wtrsd_id_num;
          v_sta_desc           := v_station_rec.description_text;
          v_ecoregion          := v_station_rec.ecoregion_name;
          v_water_depth        := v_station_rec.water_depth;
          v_water_depth_unit   := v_station_rec.water_depth_unit;
          IF v_location_rec.elevation_msr <> 0.0000 THEN
            v_elevation_msr      := v_location_rec.elevation_msr;
          ELSE
            v_elevation_msr := NULL;
          END IF;
          v_lat_long_date      := v_location_rec.lat_long_msr_dt;
          v_lat_degrees        := v_location_rec.lat_degree_msr;
          v_lat_minutes        := v_location_rec.lat_minute_msr;
          v_lat_seconds        := v_location_rec.lat_second_msr;
          v_lat_dir            := v_location_rec.lat_direction;
          v_lat_dec_min        := v_location_rec.gps_lat_minute_msr;
          v_lat_dec_deg        := v_location_rec.lat_dec_deg_msr;
          v_long_degrees       := v_location_rec.long_degree_msr;
          v_long_minutes       := v_location_rec.long_minute_msr;
          v_long_seconds       := v_location_rec.long_second_msr;
          v_long_dec_min       := v_location_rec.gps_long_min_msr;
          v_long_dec_deg       := v_location_rec.long_dec_deg_msr;
          v_long_dir           := v_location_rec.long_direction;
          v_datum              := v_location_rec.geopstng_datum_cd;
          v_method             := v_location_rec.geopstng_method_cd;
          v_gps_scale          := v_location_rec.geopstng_scale_txt;
          v_ocean_name         := v_ocean_rec.NAME;
          v_ocean_ref          := v_ocean_rec.reference_pt;
          v_ocean_bottom       := v_ocean_rec.bottom_topography;
          v_add_ocean          := v_ocean_rec.addtnl_loc_name;
          v_ocean_dist         := v_ocean_rec.shore_distance;
          v_ocean_dist_unit    := v_ocean_rec.shore_dist_unit_cd;
          v_shore_relation     := v_ocean_rec.shore_relation;
          v_native_code        := v_location_rec.tsmnal_cd;
          v_native_state       := v_location_rec.tsmnal_state;
          v_epa_key            := v_station_rec.epa_key_identifier;
          v_influence          := v_station_rec.influence_area;
          v_travel_dir         := v_station_rec.travel_dir_txt;
          v_zid_rel            := v_station_rec.zid_relation_code;
          v_great_lake         := v_great_lake_rec.NAME;
          v_lake_ref           := v_great_lake_rec.reference_pt;
          v_add_lake           := v_great_lake_rec.addtnl_loc_name;
          v_lake_dist          := v_great_lake_rec.shore_distance;
          v_lake_dist_unit     := v_great_lake_rec.shore_dist_unit_cd;
          IF v_estuary_stor_rec.tsmestry_is_number IS NOT NULL THEN
            SELECT state_province_name
              INTO v_estuary_state
              FROM tsmestry
              WHERE tsmestry_is_number = v_estuary_stor_rec.tsmestry_is_number
              AND tsmestry_org_id = v_estuary_stor_rec.tsmestry_org_id;
          ELSE
            v_estuary_state := NULL;
          END IF;
               
          v_tsmestry_is_nbr    := v_estuary_stor_rec.tsmestry_is_number;
          v_estuary_dist       := v_estuary_stor_rec.shore_distance;
          v_est_dist_unit      := v_estuary_stor_rec.shore_dist_unit_cd;
          v_estuary_ref        := v_estuary_stor_rec.reference_pt;
          v_add_estuary        := v_estuary_stor_rec.addtnl_loc_name;
          v_other_estuary      := v_estuary_stor_rec.other_estuary_name;
          IF v_station_rec.blob_title IS NOT NULL THEN
            v_document := v_station_rec.blob_title || '.' || v_station_rec.blob_type;
          ELSE
            v_document := NULL;
          END IF;
          
        END IF;
        /* loop through all STATION columns */
        FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP
        
          /* get dflt position, item name, and column sequence */
          v_dflt_pos := v_columns.sic_dflt_pos;
          v_item_name := v_columns.sic_name;
          v_col_seq := v_columns.sic_seq;
          v_next := v_next + 1;
        
          /* if column is included on the import, continue. Don't look at positions <= 0 */
          IF v_dflt_pos > 0 THEN
            IF v_pos_table(v_dflt_pos) IS NOT NULL THEN
              /* get the actual item, or use default if null */
              IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
                v_item := v_line_table(v_pos_table(v_dflt_pos));
              ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL
                    AND v_default_table(v_dflt_pos) IS NOT NULL THEN
                v_item := v_default_table(v_dflt_pos);
              ELSE
                v_item := NULL;
              END IF;
            
              -- Does V_ITEM contain only spaces? (or NULL)
              IF (TRIM(v_item) IS NULL) THEN
                -- if yes
              
                -- by definition V_ITEM is NULL!
                v_item := NULL;
              
              END IF;
            
              /* assign item to appropriate variables and take care of any other procedural issues that need to be
              taken care of for or with a particular item */
            
              IF v_item_name = 'Station ID' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_item := upper(rtrim(v_item)); -- 061203 slp - Storet only does searches on station id's in upper case
                    --  and without trailing spaces.
                    IF length(v_item) > 15 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 15 characters and will be truncated.');
                          
                                    
                      v_item := substr(v_item, 1, 15);
                    END IF;
                    IF v_action_code = 'U' THEN
                      v_comp := 1;
                    ELSE
                      v_comp := check_if_station_exists(v_item, p_org_id, v_seq);
                    END IF;
                    IF v_comp = 1 THEN
                      v_id := v_item;
                    ELSE
                      v_id := v_item;
                      v_status := 'N';
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_id := NULL;
                END IF;
              
              ELSIF v_item_name = 'Point Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 30 characters and will be truncated.');
                          
                                    
                      v_item := substr(v_item, 1, 30);
                    END IF;
                    v_point_name := v_item;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_point_name := NULL;
                END IF;
              
              ELSIF v_item_name = 'Station Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 60 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 60 characters and will be truncated.');
                          
                                    
                      v_item := substr(v_item, 1, 60);
                    END IF;
                    v_name := v_item;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_name := NULL;
                END IF;
              
              ELSIF v_item_name = 'Primary Type' THEN
                IF v_item IS NOT NULL THEN
                
                  v_type1 := validate_primary_type(v_item, p_imp_cfg_seq, v_col_seq, v_seq, v_item_name);
                  IF  v_type1 != v_stor_type AND v_stor_type = 'Well' THEN
                    check_wells(v_storet_tsmvstc.tsmvstc_org_id, v_storet_tsmvstc.tsmvstc_is_number);
                  END IF;
                  OPEN c_count_primary_types; -- Sees if there is a secondary type required for this primary type.
                  FETCH c_count_primary_types
                    INTO v_rows;
                  IF c_count_primary_types%NOTFOUND THEN
                    v_type1 := NULL;
                  ELSIF v_rows = 1 THEN
                    -- Primary type does not need a secondary type
                    v_storet_tsmvstc := get_tsmvstc_is_nbr(v_type1, NULL, v_item_name, v_seq);
                    
                    v_primary_checked := 'Y';
                  ELSIF v_rows > 1 THEN
                    v_primary_checked := 'N'; -- You will need a secondary Type
                  ELSE
                    v_type1 := NULL;
                    v_storet_pk.org_id := NULL;
                    v_primary_checked := 'Y';
                    v_status := 'N';
                  END IF;
                  CLOSE c_count_primary_types;
                  IF v_type1 IS NULL THEN
                    v_storet_pk.org_id := NULL;
                  END IF;
                ELSE
                  v_storet_pk.org_id := NULL;
                  v_status := 'N';
                  v_primary_checked := 'Y';
                  v_type1 := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                       'Missing primary type.');
                      
                                
                END IF;
              
              ELSIF v_item_name = 'Secondary Type'
                    AND v_type1 IS NOT NULL THEN
              
                IF v_item IS NULL THEN
                  v_item := 'None';
                END IF;
              
                v_secondary_type := validate_secondary_type(v_item, v_type1, p_imp_cfg_seq, v_col_seq, v_seq, v_item_name);
                IF v_secondary_type IS NULL
                   AND v_primary_checked = 'N' THEN
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                       v_type1 ||
                                                       ' - has incorrect Secondary Type.');
                      
                                
                  v_status := 'N';
                  v_secondary_type := NULL;
                END IF;
                v_storet_tsmvstc := get_tsmvstc_is_nbr(v_type1, v_secondary_type, v_item_name, v_seq);
                v_primary_checked := 'Y';
              ELSIF v_item_name = 'Establishment Date' THEN
                IF v_item IS NULL THEN
                  v_estab_date := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_estab_date := get_date(v_item, v_col_seq, p_imp_cfg_seq, v_line_cnt, v_seq);
                    IF v_estab_date IS NULL THEN
                      --the date format is bad
                      v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                           'Establishment date format does not match format in import configuration');
                          
                                    
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                END IF;
              
              ELSIF v_item_name = 'Water Depth' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := validate_number(v_item, v_item_name, '9999.999', v_seq);
                    IF v_success = 1 THEN
                      v_water_depth := to_number(v_item);
                    ELSE
                      v_water_depth := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_water_depth := NULL;
                END IF;
              
              ELSIF v_item_name = 'Water Depth Units' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  v_water_depth_unit := get_allowable_value(v_dummy_seq, v_seq);
                ELSE
                  v_water_depth_unit := NULL;
                END IF;
              
              ELSIF v_item_name = 'Latitude' THEN
                v_lat_dir := NULL;
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := sim_import_pkg.get_latitude(v_import_log_seq, v_item, v_col_seq, p_imp_cfg_seq, v_seq, v_line_cnt, v_lat_degrees, v_lat_minutes, v_lat_seconds, v_lat_dir, v_lat_dec_min, v_lat_dec_deg);
                    IF v_success = 2 THEN
                      v_status := 'N';
                      v_lat_dec_deg := NULL;
                      v_lat_dec_min := NULL;
                      v_lat_degrees := NULL;
                      v_lat_minutes := NULL;
                      v_lat_seconds := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                       'Latitude is required.');
                      
                         
                  v_status := 'N';
                  v_lat_dec_deg := NULL;
                  v_lat_dec_min := NULL;
                  v_lat_degrees := NULL;
                  v_lat_minutes := NULL;
                  v_lat_seconds := NULL;
                END IF;
              
              ELSIF v_item_name = 'Latitude Direction' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NOT NULL THEN
                    IF v_lat_dir IS NULL THEN
                      v_lat_dir := get_allowable_value(v_dummy_seq, v_seq);
                    ELSE
                      v_check_lat_dir := get_allowable_value(v_dummy_seq, v_seq);
                      IF v_check_lat_dir <> v_lat_dir THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                             'Given Latitude Direction is inconsistent with sign on Latitude.');
                            
                               
                        v_status := 'N';
                        v_lat_dir := 'X'; -- so we don't default it in
                      END IF;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         v_item ||
                                                         ' is not an allowable Latitude Direction.');
                        
                           
                    v_status := 'N';
                    v_lat_dir := 'X'; -- so we don't default it in
                  END IF;
                ELSE
                  -- v_item IS NULL, default to N
                  IF v_lat_dir IS NULL THEN
                    v_lat_dir := 'N';
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         'Latitude Direction is required and will be set to N');
                        
                           
                  END IF;
                END IF;
              
              ELSIF v_item_name = 'Longitude' THEN
                v_long_dir := NULL;
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := sim_import_pkg.get_longitude(v_import_log_seq, v_item, v_col_seq, p_imp_cfg_seq, v_seq, v_line_cnt, v_long_degrees, v_long_minutes, v_long_seconds, v_long_dec_min, v_long_dec_deg, v_long_dir);
                    IF v_success = 2 THEN
                      v_status := 'N';
                      v_long_dec_deg := NULL;
                      v_long_dec_min := NULL;
                      v_long_degrees := NULL;
                      v_long_minutes := NULL;
                      v_long_seconds := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                       'Longitude is required.');
                      
                         
                  v_status := 'N';
                  v_long_dec_deg := NULL;
                  v_long_dec_min := NULL;
                  v_long_degrees := NULL;
                  v_long_minutes := NULL;
                  v_long_seconds := NULL;
                END IF;
              
              ELSIF v_item_name = 'Longitude Direction' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NOT NULL THEN
                    IF v_long_dir IS NULL THEN
                      v_long_dir := get_allowable_value(v_dummy_seq, v_seq);
                    ELSE
                      v_check_long_dir := get_allowable_value(v_dummy_seq, v_seq);
                      IF v_check_long_dir <> v_long_dir THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                             'Given Longitude Direction is inconsistent with sign on Longitude.');
                            
                               
                        v_status := 'N';
                        v_long_dir := 'X'; -- so we don't default it in
                      END IF;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         v_item ||
                                                         ' is not an allowable Longitude Direction.');
                        
                           
                    v_status := 'N';
                    v_long_dir := 'X'; -- so we don't default it in
                  END IF;
                ELSE
                  -- v_item is NULL, default to W
                  IF v_long_dir IS NULL THEN
                    v_long_dir := 'W';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         'Longitude Direction is required and will be set to W');
                        
                           
                  END IF;
                END IF;
              
              ELSIF v_item_name = 'Lat/Long Measurement Date' THEN
                IF v_item IS NULL THEN
                  v_lat_long_date := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_lat_long_date := get_date(v_item, v_col_seq, p_imp_cfg_seq, v_line_cnt, v_seq);
                    IF v_lat_long_date IS NULL THEN
                      --the date format is bad
                      v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' format does not match format in import configuration');
                          
                             
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                END IF;
              
              ELSIF v_item_name = 'Geopositioning Method' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_method := check_tsmmad_value(v_item, v_item_name, 'HORIZONTAL', 'METHOD', v_col_seq, v_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_method IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_method := NULL;
                  v_status := 'N';
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                       'Geopositioning method is required.');
                      
                         
                END IF;
              
              ELSIF v_item_name = 'Geopositioning Datum' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_datum := check_tsmmad_value(v_item, v_item_name, 'HORIZONTAL', 'DATUM', v_col_seq, v_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_datum IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_datum := NULL;
                  v_status := 'N';
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                       'Geopositioning Datum is required.');
                      
                         
                END IF;
              
              ELSIF v_item_name = 'Scale' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 20 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 20 characters and will be truncated.');
                          
                             
                      v_item := substr(v_item, 1, 20);
                    END IF;
                    v_gps_scale := v_item;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_gps_scale := NULL;
                END IF;
              
                ---------
                -- County
                ---------
              ELSIF v_item_name = 'County' THEN
                IF v_item IS NOT NULL THEN
                
                  -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                  v_item := rtrim(v_item);
                
                  -- translate and validate county
                  v_county := validate_county(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_seq);
                
                  -- did item translate and validate?
                  IF (v_county IS NULL) THEN
                    -- if no, county is not valid or translated to NULL             
                    -- prevent export to STORET
                    v_status := 'N';
                  END IF;
                
                ELSE
                
                  -- County not required for Ocean, Estuary, or Great Lake
                  IF v_type1 NOT IN ('Ocean', 'Estuary', 'Great Lake') THEN
                    v_status := 'N';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         'County is required.');
                        
                           
                  END IF;
                  v_county := NULL;
                END IF;
              
                --------            
                -- State
                --------            
              ELSIF v_item_name = 'State' THEN
                IF v_item IS NOT NULL THEN
                  -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                  v_item := rtrim(v_item);
                
                  -- do we already have the county?
                  IF v_county IS NOT NULL THEN
                    -- if yes
                    -- translate and validate the state
                    v_state := validate_state(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_seq);
                  
                    -- did the state translate and validate?
                    IF v_state IS NOT NULL THEN
                      -- if yes
                      -- jah 2-24-04 Convert to uppercase because STORET GUI converts state to uppercase
                      v_storet_tsmgeopa := get_tsmgeopa_is_nbr(upper(v_state), v_county);
                      v_tsmgeopa_is_number := v_storet_tsmgeopa.is_number;
                    
                      -- jah 5-12-04 (Task LOG 475)
                      -- Added following statement because the
                      -- org id is required to identify the county!
                      v_tsmgeopa_org_id := v_storet_tsmgeopa.org_id;
                    
                      IF v_tsmgeopa_is_number IS NULL THEN
                        -- Valid Combination of State and County not found
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                             'The county of ' ||
                                                             v_county ||
                                                             ' is not valid for the State of ' ||
                                                             v_state || '.');
                            
                               
                      END IF;
                    
                    ELSE
                      -- State was invalid or translated to NULL             
                      v_status := 'N';
                    END IF;
                  
                  ELSE
                  
                    -- No County
                    v_status := 'N';
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         'County is required when state has been provided.');
                        
                           
                  END IF;
                
                ELSE
                
                  /* This section does not get called unless state exists rendering it
                  useless at this location.  I copied code down to the 'rules' section
                  and it appears to work there. -09/05 - rk*/
                  -- State is not required for Ocean, Estuary, or Great Lake
                  IF v_type1 IN ('Ocean', 'Estuary', 'Great Lake') THEN
                    IF v_county IS NOT NULL THEN
                      -- If county provided with out State give error
                       
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         'State is required for all stations when county is provided.');
                        
                           
                      v_status := 'N';
                    END IF;
                  ELSE
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         'State is required.');
                        
                           
                    v_status := 'N';
                  END IF;
                
                  -- No State or County will be loaded
                  v_tsmgeopa_is_number := NULL;
                  v_tsmgeopa_org_id := NULL;
                
                END IF;
              
              ELSIF v_item_name = 'HUC' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_huc := validate_huc(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_seq);
                  IF v_huc IS NOT NULL THEN
                    v_storet_tsmfhu := get_tsmfhu_is_nbr(v_huc);
                    v_tsmfhu_is_number := v_storet_tsmfhu.is_number;
                    v_tsmfhu_org_id := v_storet_tsmfhu.org_id;
                  ELSE
                    v_tsmfhu_is_number := NULL;
                    v_tsmfhu_org_id := NULL;
                    v_status := 'N';
                  END IF;
                ELSE
                  v_tsmfhu_is_number := NULL;
                  v_tsmfhu_org_id := NULL;
                END IF;
              
              ELSIF v_item_name = 'NRCS Watershed ID' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                    v_trans_exist := TRUE;
                    v_converted_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                    IF length(v_converted_item) > 8 THEN
                      v_nrcs := substr(v_converted_item, 1, 8);
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           'NRCS ID, ' ||
                                                           v_converted_item ||
                                                           ', is longer than 8 characters and will be truncated.');
                          
                             
                    ELSE
                      v_nrcs := v_converted_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_nrcs := NULL;
                END IF;
                IF v_nrcs = 'NULL' THEN
                  v_nrcs := NULL;
                END IF;
              
              ELSIF v_item_name = 'Station Description' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 4000 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 4000 characters and will be truncated.');
                          
                             
                      v_item := substr(v_item, 1, 4000);
                    END IF;
                    v_sta_desc := v_item;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_sta_desc := NULL;
                END IF;
              
              ELSIF v_item_name = 'Ocean Name' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL
                     AND v_type1 = 'Ocean' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         'Ocean Name is required when primary type is Ocean');
                        
                           
                    v_status := 'N';
                  ELSIF v_type1 != 'Ocean' THEN
                    v_ocean_name := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         'Ocean Name Ignored - only used when primary type is Ocean');
                        
                           
                  ELSE
                    v_ocean_name := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_ocean_name := NULL;
                END IF;
              
              ELSIF v_item_name = 'Shore Relation' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL
                     AND v_type1 = 'Ocean' THEN
                    v_status := 'N';
                    v_shore_relation := NULL;
                  ELSIF v_type1 != 'Ocean' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         'Shore Relation ignored. It is only used when primary type is Ocean');
                        
                           
                    v_shore_relation := NULL;
                  ELSE
                    v_shore_relation := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_shore_relation := NULL;
                END IF;
              
              ELSIF v_item_name = 'Elevation' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := validate_number(v_item, v_item_name, '99999.9999', v_seq);
                    IF v_success = 0 THEN
                      v_elevation_msr := NULL;
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                           'Elevation value ' ||
                                                           v_item ||
                                                           ' is not allowable. It must be a number.');
                          
                             
                    ELSE
                      v_elevation_msr := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_elevation_msr := NULL;
                END IF;
              
              ELSIF v_item_name = 'Elevation Units' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_elevation_msr IS NOT NULL THEN
                    IF v_dummy_seq IS NULL THEN
                      v_elevation_unit := NULL;
                      v_status := 'N';
                    ELSE
                      v_elevation_unit := get_allowable_value(v_dummy_seq, v_seq);
                    END IF;
                  ELSE
                    v_elevation_unit := NULL;
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         'Elevation Unit is not needed since Elevation is not provided.');
                        
                           
                  END IF;
                ELSE
                  v_elevation_unit := NULL;
                END IF;
              
              ELSIF v_item_name = 'Elevation Method' THEN
                --sfp, this section rewritten 062701
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_elevation_method := check_tsmmad_value(v_item, v_item_name, 'VERTICAL', 'METHOD', v_col_seq, v_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_elevation_method IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_elevation_method := NULL;
                END IF;
              
              ELSIF v_item_name = 'Elevation Datum' THEN
                --sfp, this section rewritten 062701
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_elevation_datum := check_tsmmad_value(v_item, v_item_name, 'VERTICAL', 'DATUM', v_col_seq, v_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_elevation_datum IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_elevation_datum := NULL;
                END IF;
              
              ELSIF v_item_name = 'Elevation Measurement Date' THEN
                IF v_item IS NULL THEN
                  v_elevation_date := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_elevation_date := get_date(v_item, v_col_seq, p_imp_cfg_seq, v_line_cnt, v_seq);
                    IF v_elevation_date IS NULL THEN
                      --the date format is bad
                      v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' format does not match format in import configuration');
                          
                             
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                END IF;
              
              ELSIF v_item_name = 'EPA Key Identifier' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 36 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           'EPA Key Identifier is longer than 36 characters and will be truncated.');
                          
                             
                      v_epa_key := substr(v_item, 1, 36);
                    ELSE
                      v_epa_key := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_epa_key := NULL;
                END IF;
              
              ELSIF v_item_name = 'Ecoregion Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 60 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 60 characters and will be truncated.');
                          
                             
                      v_ecoregion := substr(v_item, 1, 60);
                    ELSE
                      v_ecoregion := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_ecoregion := NULL;
                END IF;
              
              ELSIF v_item_name = 'Influence Area' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 120 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 120 characters and will be truncated.');
                          
                             
                      v_influence := substr(v_item, 1, 120);
                    ELSE
                      v_influence := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_influence := NULL;
                END IF;
              
              ELSIF v_item_name = 'Travel Directions' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 1999 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                          v_item_name ||
                                                          'are longer than 1999 characters and will be truncated.');
                          
                             
                      v_travel_dir := substr(v_item, 1, 1999);
                    ELSE
                      v_travel_dir := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_travel_dir := NULL;
                END IF;
              
              ELSIF v_item_name = 'ZID Relation' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL THEN
                    v_status := 'N';
                    v_zid_rel := NULL;
                  ELSE
                    v_zid_rel := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_zid_rel := NULL;
                END IF;
              
              ELSIF v_item_name = 'Native American Land Name' THEN
                IF v_item IS NOT NULL THEN
                  v_native_name := validate_nal(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, NULL, NULL, 1, v_seq);
                  IF v_native_name IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_native_name := NULL;
                END IF;
              
              ELSIF v_item_name = 'Native American Land State' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_native_state := validate_nal(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, NULL, NULL, 2, v_seq);
                  IF v_native_state IS NOT NULL THEN
                    IF v_native_name IS NOT NULL THEN
                      v_native := validate_nal(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_native_name, v_native_state, 3, v_seq);
                      IF v_native = 'BAD' THEN
                        v_status := 'N';
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                             v_native_name || '/' ||
                                                             v_native_state ||
                                                             ' is an invalid Native Americal Land Name/State combination.');
                            
                               
                      END IF;
                    ELSE
                      --state but no Name
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           'Native Americal Land State is invalid without a Name and will be ignored.');
                          
                             
                    END IF;
                  ELSE
                    v_status := 'N';
                  END IF;
                ELSE
                  --no state
                  IF v_native_name IS NOT NULL THEN
                      
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           'Native Americal Land Name is invalid without a State and will be ignored.');
                          
                             
                  END IF;
                  v_native_state := NULL;
                  v_native_name := NULL;
                  v_native_code := NULL;
                END IF;
              
              ELSIF v_item_name = 'Additional Ocean Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                      
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 30 characters and will be truncated.');
                          
                             
                      v_add_ocean := substr(v_item, 1, 30);
                    ELSE
                      v_add_ocean := v_item;
                    END IF;
                    IF v_type1 != 'Ocean' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored - only used when primary type is Ocean');
                          
                             
                      v_add_ocean := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_add_ocean := NULL;
                END IF;
              
              ELSIF v_item_name = 'Ocean Station Dist to Shore' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := validate_number(v_item, v_item_name, '9999.99', v_seq);
                    IF v_success = 0 THEN
                      v_ocean_dist := NULL;
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                           v_item_name || ', ' ||
                                                           v_item ||
                                                           ' is not a number in the allowed format of ZZZZ.ZZ.');
                          
                             
                    ELSE
                      v_ocean_dist := v_item;
                    END IF;
                    IF v_type1 != 'Ocean' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name || ', ' ||
                                                           v_item_name ||
                                                           ' ignored - only used when primary type is Ocean');
                          
                             
                      v_ocean_dist := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_ocean_dist := NULL;
                END IF;
              
              ELSIF v_item_name = 'Ocean Station Dist to Shore Units' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL
                     AND v_ocean_dist IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         v_item_name || ', ' ||
                                                         v_item ||
                                                         ' is not a valid value.');
                        
                           
                    v_status := 'N';
                    v_ocean_dist_unit := NULL;
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_ocean_dist IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored - it is not needed if Distance is not given.');
                        
                           
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_ocean_dist IS NOT NULL THEN
                    v_ocean_dist_unit := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_ocean_dist_unit := NULL;
                END IF;
              
              ELSIF v_item_name = 'Ocean Station Ref Point' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 30 characters and will be truncated.');
                          
                             
                      v_ocean_ref := substr(v_item, 1, 30);
                    ELSE
                      v_ocean_ref := v_item;
                    END IF;
                    IF v_type1 != 'Ocean' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is only used when primary type is Ocean');
                          
                             
                      v_ocean_ref := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_ocean_ref := NULL;
                END IF;
              
              ELSIF v_item_name = 'Ocean Station Bottom Topography' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 254 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' is longer than 254 characters and will be truncated.');
                          
                             
                      v_ocean_bottom := substr(v_item, 1, 254);
                    ELSE
                      v_ocean_bottom := v_item;
                    END IF;
                    IF v_type1 != 'Ocean' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored - only used when primary type is Ocean');
                          
                             
                      v_ocean_bottom := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_ocean_bottom := NULL;
                END IF;
              
              ELSIF v_item_name = 'Primary Estuary State' THEN
                IF v_item IS NOT NULL THEN
                  IF v_type1 != 'Estuary' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored. It is not used unless Station Type is Estuary.');
                        
                           
                    v_estuary_state := NULL;
                  ELSE
                    v_estuary_state := validate_estuary_state(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_seq);
                  END IF;
                ELSE
                  v_estuary_state := NULL;
                END IF;
              
              ELSIF v_item_name = 'Primary Estuary' THEN
                IF v_item IS NOT NULL THEN
                  IF v_type1 != 'Estuary' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored. It is not used unless Station Type is Estuary.');
                        
                           
                    v_primary_is_nbr := NULL;
                    v_primary_est := NULL;
                  ELSIF v_estuary_state IS NULL THEN
                    v_primary_is_nbr := NULL;
                    v_primary_est := NULL;
                  ELSE
                    v_primary_est := validate_primary_estuary(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_estuary_state, v_primary_is_nbr, v_seq);
                    IF v_primary_est IS NULL THEN
                      v_status := 'N';
                      v_primary_is_nbr := NULL;
                    END IF;
                  END IF;
                ELSE
                  v_primary_est := NULL;
                  v_primary_is_nbr := NULL;
                END IF;
              
              ELSIF v_item_name = 'Secondary Estuary' THEN
                IF v_item IS NOT NULL THEN
                  IF v_type1 != 'Estuary' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored. It is not used unless Station Type is Estuary.');
                        
                           
                    v_secondary_is_nbr := NULL;
                    v_secondary_est := NULL;
                  ELSIF v_primary_is_nbr IS NULL THEN
                    v_secondary_is_nbr := NULL;
                    v_secondary_est := NULL;
                  ELSE
                    v_secondary_est := validate_secondary_estuary(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, v_primary_is_nbr, v_secondary_is_nbr, v_seq);
                    IF v_secondary_est IS NULL THEN
                      v_secondary_is_nbr := NULL;
                      v_status := 'N';
                    END IF;
                  END IF;
                ELSE
                  v_secondary_est := NULL;
                  v_secondary_is_nbr := NULL;
                END IF;
              
              ELSIF v_item_name = 'Other Estuary' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                       
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ', is longer than 30 characters and will be truncated.');
                        
                           
                      v_other_estuary := substr(v_item, 1, 30);
                    ELSE
                      v_other_estuary := v_item;
                    END IF;
                    IF v_type1 != 'Estuary' THEN
                      
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored. It is not used unless Station Type is Estuary.');
                        
                           
                      v_other_estuary := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_other_estuary := NULL;
                END IF;
              
              ELSIF v_item_name = 'Additional Estuary Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                      
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 30 characters and will be truncated.');
                          
                             
                      v_add_estuary := substr(v_item, 1, 30);
                    ELSE
                      v_add_estuary := v_item;
                    END IF;
                    IF v_type1 != 'Estuary' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is not used unless Station Type is Estuary.');
                          
                             
                      v_add_estuary := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_add_estuary := NULL;
                END IF;
              
              ELSIF v_item_name = 'Estuary Distance to Shore' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := validate_number(v_item, v_item_name, '9999.99', v_seq);
                    IF v_success = 0 THEN
                      v_estuary_dist := NULL;
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name || ', ' ||
                                                           v_item ||
                                                           ' is not a number in the allowed format of ZZZZ.ZZ.');
                          
                             
                    ELSIF v_type1 != 'Estuary' THEN
                      
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is not used unless Station Type is Estuary.');
                          
                             
                      v_estuary_dist := NULL;
                    ELSE
                      v_estuary_dist := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_estuary_dist := NULL;
                END IF;
              
              ELSIF v_item_name = 'Estuary Dist to Shore Units' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL
                     AND v_estuary_dist IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         v_item_name || ', ' ||
                                                         v_item ||
                                                         ' is not a valid value.');
                        
                           
                    v_status := 'N';
                    v_est_dist_unit := NULL;
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_estuary_dist IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ', ignored - it is not needed if Distance is not given.');
                        
                           
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_estuary_dist IS NOT NULL THEN
                    v_est_dist_unit := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_est_dist_unit := NULL;
                END IF;
              
              ELSIF v_item_name = 'Estuary Reference Point' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 30 characters and will be truncated.');
                          
                             
                      v_estuary_ref := substr(v_item, 1, 30);
                    ELSE
                      v_estuary_ref := v_item;
                    END IF;
                    IF v_type1 != 'Estuary' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is not used unless Station Type is Estuary.');
                          
                             
                      v_estuary_ref := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_estuary_ref := NULL;
                END IF;
              
              ELSIF v_item_name = 'Great Lake' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         v_item_name || ', ' ||
                                                         v_item ||
                                                         ' is not a valid value.');
                        
                           
                    v_status := 'N';
                    v_great_lake := NULL;
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_type1 != 'Great Lake' THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored. Not used unless Primary Type is Great Lake.');
                        
                           
                    v_great_lake := NULL;
                  ELSE
                    v_great_lake := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_great_lake := NULL;
                END IF;
              
              ELSIF v_item_name = 'Additional Great Lake Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ', is longer than 30 characters and will be truncated.');
                          
                             
                      v_add_lake := substr(v_item, 1, 30);
                    ELSE
                      v_add_lake := v_item;
                    END IF;
                    IF v_type1 != 'Great Lake' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is not used unless Station Type is Great Lake.');
                          
                             
                      v_add_lake := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_add_lake := NULL;
                END IF;
              
              ELSIF v_item_name = 'Great Lake Dist to Shore' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := validate_number(v_item, v_item_name, '9999.99', v_seq);
                    IF v_success = 0 THEN
                      v_lake_dist := NULL;
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                           v_item_name || ', ' ||
                                                           v_item ||
                                                           ' is not a number in the allowed format of ZZZZ.ZZ.');
                          
                             
                    ELSE
                      v_lake_dist := v_item;
                    END IF;
                    IF v_type1 != 'Great Lake' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is not used unless Station Type is Great Lake.');
                          
                             
                      v_lake_dist := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_lake_dist := NULL;
                END IF;
              
              ELSIF v_item_name = 'Great Lake Dist to Shore Units' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_seq);
                  IF v_dummy_seq IS NULL
                     AND v_lake_dist IS NOT NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                         v_item_name || ', ' ||
                                                         v_item ||
                                                         ' is not a valid value.');
                        
                           
                    v_status := 'N';
                    v_lake_dist_unit := NULL;
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_lake_dist IS NULL THEN
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ' ignored - it is not needed if Distance is not given.');
                        
                           
                  ELSIF v_dummy_seq IS NOT NULL
                        AND v_lake_dist IS NOT NULL THEN
                    v_lake_dist_unit := get_allowable_value(v_dummy_seq, v_seq);
                  END IF;
                ELSE
                  v_lake_dist_unit := NULL;
                END IF;
              
              ELSIF v_item_name = 'Great Lake Reference Point' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 30 THEN
                       
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                         v_item_name ||
                                                         ', is longer than 30 characters and will be truncated.');
                        
                           
                      v_lake_ref := substr(v_item, 1, 30);
                    ELSE
                      v_lake_ref := v_item;
                    END IF;
                    IF v_type1 != 'Great Lake' THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                                           v_item_name ||
                                                           ' ignored. It is not used unless Station Type is Great Lake.');
                          
                             
                      v_lake_ref := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- value was translated to null
                  END IF;
                ELSE
                  v_lake_ref := NULL;
                END IF;
              
              ELSIF v_item_name = 'Document/Graphic' THEN
                IF v_item IS NOT NULL THEN
                  v_item := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF sim_import_pkg.check_document_file(v_item, v_import_log_seq, v_line_cnt) THEN
                      v_document := v_item;
                    ELSE
                      v_status := 'N'; -- Invalid document - SET STATUS TO N 
                      v_document := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to NULL - SET STATUS TO N 
                    v_document := NULL;
                  END IF;
                ELSE
                  v_document := NULL;
                END IF;
              
              END IF; --end of v_item_names
            
            END IF;
          END IF;
        END LOOP;
      
        /*** END STATION LOOP ***/
        --Now check all the STORET business rules
        IF v_id IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Station ID is required.');
              
                 
          v_status := 'N';
        END IF;
      
        IF v_primary_checked = 'N' THEN
          IF v_action_code != 'U' THEN 
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 'Primary type, ' || v_type1 ||
                                                 ', needs secondary type.');
                
                 
            v_status := 'N';
            v_storet_tsmvstc.tsmvstc_is_number := NULL;
            v_storet_tsmvstc.tsmvstc_org_id := NULL;
          END IF;
        END IF;
      
        -- Default Lat/Long to NW hemisphere if not included in import file
        IF v_lat_dir IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                               'Latitude direction has not been provided and will be defaulted to N');
              
                 
          v_lat_dir := 'N';
        END IF;
      
        IF v_long_dir IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                               'Longitude direction has not been provided and will be defaulted to W');
              
                 
          v_long_dir := 'W';
        END IF;
      
        -- Need to check this post loop as lat direction is not required in import file
        -- 11-10-2004 sjk do not truncate less than 7
        IF v_lat_dir = 'S'
           AND v_lat_dec_deg > 0 THEN
          --IF v_lat_dec_deg < 10 THEN
          v_lat_dec_deg := '-' || trunc(v_lat_dec_deg, 7);
          --ELSE
          --  v_lat_dec_deg := '-'||TRUNC(v_lat_dec_deg,6);
          --END IF;
        END IF;
      
        --Elevation Rules
        IF v_elevation_unit IS NULL
           AND v_elevation_msr IS NOT NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Elevation units are required when elevation is provided.');
              
                 
        END IF;
        IF v_elevation_method IS NULL
           AND v_elevation_msr IS NOT NULL THEN
          v_status := 'N';
            
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Elevation method is required when elevation is provided.');
              
                 
        END IF;
        IF v_elevation_datum IS NULL
           AND v_elevation_msr IS NOT NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Elevation datum is required when elevation is provided.');
              
                 
        END IF;
      
        --Estuary rules
        IF v_type1 = 'Estuary' THEN
          IF v_estuary_state IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 'Estuary State is required when Station type is Estuary.');
                
                   
            v_status := 'N';
          ELSIF v_primary_is_nbr IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 'Primary Estuary is required when Station type is Estuary.');
                
                   
            v_status := 'N';
          ELSIF v_secondary_is_nbr IS NULL THEN
            v_tsmestry_is_nbr := v_primary_is_nbr;
          ELSE
            v_tsmestry_is_nbr := v_secondary_is_nbr;
          END IF;
        END IF;
      
        -- Ocean rules
        IF v_type1 = 'Ocean'
           AND v_ocean_name IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Ocean Name is required when Station type is Ocean.');
              
                 
        END IF;
        IF v_type1 = 'Ocean'
           AND v_shore_relation IS NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Shore Relation is required when Station type is Ocean.');
              
                 
        END IF;
      
        -- Lake rules
        IF v_type1 = 'Great Lake'
           AND v_great_lake IS NULL THEN
          v_status := 'N';
          
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Great Lake is required when Station type is Great Lake.');
              
                 
        END IF;
      
        --NAL Rules
        IF v_native_name IS NOT NULL
           AND v_native_state IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Native American Land State is required when Name is given.');
              
                 
          v_status := 'N';
          v_native_name := NULL;
        END IF;
      
        --Units rules
        IF v_water_depth_unit IS NOT NULL
           AND v_water_depth IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                               'Water Depth Units ignored since Water Depth is not given.');
              
                 
        END IF;
        IF v_ocean_dist_unit IS NOT NULL
           AND v_ocean_dist IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                               'Ocean Station Distance to Shore Units ignored since distance is not given.');
              
                 
        END IF;
        IF v_est_dist_unit IS NOT NULL
           AND v_estuary_dist IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                               'Estuary Distance to Shore Units ignored since distance is not given.');
              
                
        END IF;
        IF v_lake_dist_unit IS NOT NULL
           AND v_lake_dist IS NULL THEN
          
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '||v_line_cnt,
                                               'Great Lake Distance to Shore Units ignored since distance is not given.');
              
                
        END IF;
      
        --Scale rules
        -- 018 is 'Interpolation-map'
        IF v_gps_scale IS NULL
           AND v_method = rpad('018', 12) THEN
          
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                               'Scale is required when Geopositioning Method is Interpolation-Map.');
              
                
          v_status := 'N';
        END IF;
      
        --County/State pair rules
        -- if state is null
        IF v_tsmgeopa_is_number IS NULL THEN
          IF v_type1 IN ('Ocean', 'Estuary', 'Great Lake') THEN
            IF v_county IS NOT NULL THEN
              -- If county provided without State give error
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                   'State is required for all stations when county is provided.');
                  
                    
              v_status := 'N';
            END IF;
          ELSE
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 'State is required.');
                
                  
            v_status := 'N';
          END IF;
        END IF;
      
        /* create STATION */
        BEGIN
          /* write data to record and send to create station function */
          /* get station location sequence */
          BEGIN
            OPEN c_get_station_location_seq;
            FETCH c_get_station_location_seq
              INTO v_stl_seq;
            CLOSE c_get_station_location_seq;
          EXCEPTION
            WHEN OTHERS THEN
               
                                                                                                                                                                                                                                                                                                                                                                                                                            
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                   'getting station location sequence.');
                  
                    
          END;
        
          v_sta_rec.sta_seq := v_seq;
          v_sta_rec.sta_id := v_id;
          v_sta_rec.sta_tsmorgan_org_id := p_org_id;
          v_sta_rec.sta_tsmorgan_is_number := v_tsmorgan_is_number;
          v_sta_rec.sta_org_id := p_org_id;
          v_sta_rec.sta_status := v_status;
          v_sta_rec.sta_tsmvstc_is_number := v_storet_tsmvstc.tsmvstc_is_number;
          v_sta_rec.sta_tsmvstc_org_id := v_storet_tsmvstc.tsmvstc_org_id;
          v_sta_rec.sta_name := v_name;
          v_sta_rec.sta_import_seq := v_import_log_seq;
          v_sta_rec.sta_estab_date := v_estab_date;
          v_sta_rec.sta_desc := v_sta_desc;
          v_sta_rec.sta_ecoregion := v_ecoregion;
          v_sta_rec.sta_water_depth := v_water_depth;
          v_sta_rec.sta_water_depth_unit := v_water_depth_unit;
          v_sta_rec.sta_epa_key_id := v_epa_key;
          v_sta_rec.sta_zid_rel := v_zid_rel;
          v_sta_rec.sta_influence_area := v_influence;
          v_sta_rec.sta_travel_dir := v_travel_dir;
          v_sta_rec.sta_document := v_document;
        
          v_stl_rec.stl_seq := v_stl_seq;
          v_stl_rec.stl_sta_seq := v_seq;
          v_stl_rec.stl_type_code := '*POINT OF RECORD';
          v_stl_rec.stl_sequence_number := 0;
          v_stl_rec.stl_point_name := v_point_name;
          v_stl_rec.stl_lat_degrees := v_lat_degrees;
          v_stl_rec.stl_lat_minutes := v_lat_minutes;
          v_stl_rec.stl_lat_dec_degrees := v_lat_dec_deg;
          v_stl_rec.stl_lat_dec_minutes := v_lat_dec_min;
          v_stl_rec.stl_lat_seconds := v_lat_seconds;
          v_stl_rec.stl_lat_dir := v_lat_dir;
          v_stl_rec.stl_long_degrees := v_long_degrees;
          v_stl_rec.stl_long_minutes := v_long_minutes;
          v_stl_rec.stl_long_dec_degrees := v_long_dec_deg;
          v_stl_rec.stl_long_dec_minutes := v_long_dec_min;
          v_stl_rec.stl_long_seconds := v_long_seconds;
          v_stl_rec.stl_long_dir := v_long_dir;
          v_stl_rec.stl_lat_long_msr_date := v_lat_long_date;
          v_stl_rec.stl_latlong_accurcy := NULL;
          v_stl_rec.stl_latlong_accurcy_unit := NULL;
          v_stl_rec.stl_gps_scale := v_gps_scale;
          v_stl_rec.stl_tsmgeopa_is_number := v_tsmgeopa_is_number;
          v_stl_rec.stl_tsmgeopa_org_id := v_tsmgeopa_org_id;
          v_stl_rec.stl_tsmfhu_is_number := v_tsmfhu_is_number;
          v_stl_rec.stl_tsmfhu_org_id := v_tsmfhu_org_id;
          v_stl_rec.stl_tsmrrr_is_number := NULL;
          v_stl_rec.stl_tsmrrr_org_id := NULL;
          v_stl_rec.stl_geopos_method := v_method;
          v_stl_rec.stl_geopos_datum := v_datum;
          v_stl_rec.stl_elevation_msr := v_elevation_msr;
          v_stl_rec.stl_elevation_unit := v_elevation_unit;
          v_stl_rec.stl_elev_method := v_elevation_method;
          v_stl_rec.stl_elev_datum := v_elevation_datum;
          v_stl_rec.stl_elev_msr_date := v_elevation_date;
          v_stl_rec.stl_tsmnal_cd := v_native_code;
          v_stl_rec.stl_tsmnal_state := v_native_state;
          v_stl_rec.stl_nrcs := v_nrcs;
          v_stl_rec.stl_tsmstatn0is_number := v_tsmstatn0is_number;
          v_stl_rec.stl_tsmstatn0org_id := v_tsmstatn0org_id;
        
          /* Go ahead and create the records for the station and point of record */
          /* insert station */
          IF v_id IS NOT NULL THEN
            v_success := create_station(v_sta_rec);
          END IF;
          /* if insert function failed, write error and prevent station from being exported */
        
          IF v_success != 1 THEN
            UPDATE sim_stations
               SET sta_status = 'N'
             WHERE sta_seq = v_seq;
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 'Failed to create station in sim_stations. Activity will not be exported.');
                
                  
          ELSE
            --continue - create location
            v_success := create_station_location(v_stl_rec);
            /* if insert function failed, write error and prevent activity from being exported */
            IF v_success != 1 THEN
              UPDATE sim_stations
                 SET sta_status = 'N'
               WHERE sta_seq = v_seq;
               
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 'Failed to create station location in sim_station_locations. Station will not be exported.');
                
                  
            END IF;
          END IF;
        
          IF v_type1 = 'Ocean' THEN
            BEGIN
              OPEN c_get_ocean_seq;
              FETCH c_get_ocean_seq
                INTO v_ocean_seq;
              CLOSE c_get_ocean_seq;
            EXCEPTION
              WHEN OTHERS THEN
                 
                                                                                                                                                                                                                                                                                                                                                                                                                                                                      
                 
                sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                     'getting ocean sequence.');
                    
                      
            END;
          
            v_soc_rec.soc_seq := v_ocean_seq;
            v_soc_rec.soc_stl_seq := v_stl_seq;
            v_soc_rec.soc_name := v_ocean_name;
            v_soc_rec.soc_additional_loc := v_add_ocean;
            v_soc_rec.soc_shore_relation := v_shore_relation;
            v_soc_rec.soc_shore_distance := v_ocean_dist;
            v_soc_rec.soc_shore_dist_unit := v_ocean_dist_unit;
            v_soc_rec.soc_reference_pt := v_ocean_ref;
            v_soc_rec.soc_bottom_topography := v_ocean_bottom;
          
          ELSIF v_type1 = 'Estuary' THEN
          
            OPEN c_get_estuary_seq;
            FETCH c_get_estuary_seq
              INTO v_estuary_seq;
            CLOSE c_get_estuary_seq;
          
            v_estuary_rec.sel_seq := v_estuary_seq;
            v_estuary_rec.sel_tsmalp_is_number := NULL;
            v_estuary_rec.sel_tsmalp_org_id := NULL;
            v_estuary_rec.sel_other_estuary := v_other_estuary;
            v_estuary_rec.sel_add_loc_name := v_add_estuary;
            v_estuary_rec.sel_shore_dist := v_estuary_dist;
            v_estuary_rec.sel_shore_dist_unit := v_est_dist_unit;
            v_estuary_rec.sel_ref_pt := v_estuary_ref;
            v_estuary_rec.sel_tsmestry_org_id := 'ZZZZZZZZ';
            v_estuary_rec.sel_tsmestry_is_number := v_tsmestry_is_nbr;
            v_estuary_rec.sel_stl_seq := v_stl_seq;
          
          ELSIF v_type1 = 'Great Lake' THEN
            OPEN c_get_great_lake_seq;
            FETCH c_get_great_lake_seq
              INTO v_lake_seq;
            CLOSE c_get_great_lake_seq;
          
            v_lake_rec.sgl_seq := v_lake_seq;
            v_lake_rec.sgl_tsmalp_is_number := NULL;
            v_lake_rec.sgl_tsmalp_org_id := NULL;
            v_lake_rec.sgl_lake_name := v_great_lake;
            v_lake_rec.sgl_add_lake_name := v_add_lake;
            v_lake_rec.sgl_shore_dist := v_lake_dist;
            v_lake_rec.sgl_shore_dist_unit := v_lake_dist_unit;
            v_lake_rec.sgl_ref_pt := v_lake_ref;
            v_lake_rec.sgl_stl_seq := v_stl_seq;
          
          END IF;
        
          v_gps_scale := NULL;
        
          IF v_type1 = 'Ocean'
             AND v_soc_rec.soc_seq IS NOT NULL THEN
            v_success := create_ocean(v_soc_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
                 SET sta_status = 'N'
               WHERE sta_seq = v_seq;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                   'Failed to create Ocean in sim_ocean. Station will not be exported.');
                  
                    
            END IF;
          ELSIF v_type1 = 'Estuary'
                AND v_estuary_rec.sel_seq IS NOT NULL THEN
            v_success := create_estuary(v_estuary_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
                 SET sta_status = 'N'
               WHERE sta_seq = v_seq;
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                   'Failed to create Estuary in sim_estuary_loc. Station will not be exported.');
                  
                    
            END IF;
          ELSIF v_type1 = 'Great Lake'
                AND v_lake_rec.sgl_seq IS NOT NULL THEN
            v_success := create_lake(v_lake_rec);
            IF v_success != 1 THEN
              UPDATE sim_stations
                 SET sta_status = 'N'
               WHERE sta_seq = v_seq;
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                   'Failed to create Great Lake in sim_great_lake. Station will not be exported.');
                  
                    
            END IF;
          
          END IF;
        EXCEPTION
          WHEN OTHERS THEN
            v_import_desc := SQLERRM || ' - writing a station.';
             
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                                 v_import_desc);
                
                  
        END;
      
        /* update import lines table with station sequence */
        UPDATE sim_imp_lines
           SET sipl_sta_seq = v_seq
         WHERE sipl_seq = v_text_line.sipl_seq;
      END IF;
    END LOOP; /* end of text loop */
  
    OPEN c_comp_count(v_import_log_seq);
    FETCH c_comp_count
      INTO v_complete_cnt;
    CLOSE c_comp_count;
    v_error_cnt := sim_import_pkg.get_rows_with_errors(v_import_log_seq);
    /* update counts in the import log */
    UPDATE sim_import_log
       SET sil_rows_imported = v_line_cnt,
           sil_rows_errors = v_error_cnt,
           sil_rows_ready = v_complete_cnt
     WHERE sil_seq = v_import_log_seq;
    COMMIT;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         to_char(v_line_cnt) ||
                                         ' lines processed / ' ||
                                         to_char(v_complete_cnt) ||
                                         ' stations loaded'); 
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Finished: ' ||
                                         to_char(SYSDATE, 'HH24:MI:SS'));
    sim_event_log_pkg.update_stop_time;                             
    sim_import_pkg.v_strip_quote := NULL; --10/04/2005 --RWK 
  END;

EXCEPTION
  WHEN e_no_delimiter THEN
    raise_application_error(-20102, 'Error getting delimiter from import configuration.');
  WHEN e_bad_org THEN
    raise_application_error(-20109, 'Unable to find the specified org id in local version of STORET.');
  WHEN e_no_cfg THEN
    raise_application_error(-20102, 'Configuration invalid or does not exist.');
  WHEN e_bad_columns THEN
    raise_application_error(-20197, 'The import file contains the wrong number of columns');
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '||v_line_cnt,
                                         substr(SQLERRM ||
                                         ' - in sim_import_station_pkg.import_station.', 1, 240));
        
          
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         to_char(v_line_cnt) ||
                                         ' lines processed / ' ||
                                         to_char(v_complete_cnt) ||
                                         ' stations loaded');
        
          
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                         'Finished: ' ||
                                         to_char(SYSDATE, 'HH24:MI:SS'));
        
          
    -- Ryan J. 03/08/05 - Raise error to calling routine
    raise_application_error(-20000, substr(SQLERRM ||
                                    ' - in sim_import_station_pkg.import_station.', 1, 240));  
END;

END sim_import_station_pkg;
/
