CREATE OR REPLACE PACKAGE sim_import_wells_pkg IS
 
/****************************************************************************************
 *                               PUBLIC CONSTANTS AND TYPES                               *
 ****************************************************************************************/
  TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;
  TYPE col_table_num_type IS TABLE OF NUMBER(3) INDEX BY BINARY_INTEGER;
  TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;
  TYPE t_storet_pk_type IS RECORD(
    is_number NUMBER(8),
    org_id    VARCHAR2(8));
  TYPE t_storet_tsmvstc_type IS RECORD(
    tsmvstc_is_number NUMBER(8),
    tsmvstc_org_id    VARCHAR2(8));
  TYPE t_storet_tsmmad_type IS RECORD(
    tsmmad_is_number NUMBER(8),
    tsmmad_org_id    VARCHAR2(8));
  TYPE t_aquifer_type IS RECORD(
    tsmgeoun_cd  VARCHAR2(12),
    aquifer_name VARCHAR2(80));

/****************************************************************************************
 *                                    PUBLIC VARIABLES                                  *
 ****************************************************************************************/
  v_activity_count t_activity_count;
  v_org_is_nbr     NUMBER(8) := 0;
  v_import_log_seq NUMBER(10);
  v_status         VARCHAR2(1);
  v_line_cnt       NUMBER(10) := 0;

/****************************************************************************************
 *                              PUBLIC PROCEDURES AND FUNCTIONS                           *
 ****************************************************************************************/  

  PROCEDURE import_wells
  (
    p_org_id         IN VARCHAR2,
    p_import_log_seq IN NUMBER,
    p_imp_cfg_seq    IN NUMBER,
    p_test_count     IN NUMBER,
    p_skip_header    IN NUMBER DEFAULT 0       -- CLM 08/02/2005 Skip header option    
    );                                         --   1 = Skip  0 = Don't Skip

  FUNCTION create_well(p_well_rec IN sim_station_wells%ROWTYPE) RETURN NUMBER;

  FUNCTION create_station_location(p_station_location_rec IN sim_station_locations%ROWTYPE)
    RETURN NUMBER;

END sim_import_wells_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_import_wells_pkg IS

/*----------------------------------------------------------------------------------------
  Import well data into SIM tables from a text file
  ----------------------------------------------------------------------------------------
  
/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/
 
 
/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/


/****************************************************************************************
 *           PROCEDURE AND FUNCTION SPECS (to call routines out of order)               *
 ****************************************************************************************/

  PROCEDURE create_trans_record
  (
    p_code        IN VARCHAR2,
    p_imp_cfg_seq IN NUMBER,
    p_col_seq     IN NUMBER
  );

  FUNCTION check_storet_translation
  (
    p_code           IN VARCHAR2,
    p_imp_cfg_seq    IN NUMBER,
    p_col_seq        IN NUMBER,
    p_trans_exist    OUT BOOLEAN,
    v_converted_item OUT VARCHAR2,
    p_sta_seq        IN NUMBER
  ) RETURN t_storet_pk_type;
  
/****************************************************************************************
 *                            PROCEDURE AND FUNCTION IMPLEMENTATIONS                    *
 ****************************************************************************************/

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_number
(
  p_item      IN VARCHAR2,
  p_item_name IN VARCHAR2,
  p_format    IN VARCHAR2,
  p_well_seq  IN NUMBER
) RETURN NUMBER IS
  -- 4/12/02 - SF
  --This function is to test any value that is supposed to be a number. If it's not a number a 0 is returned.

  v_number NUMBER(12);

BEGIN

  -- Try to convert P_ITEM to a number
  v_number := to_number(p_item, p_format);

  -- if no exception then tell calling routine P_ITEM is a number
  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                         p_item || ', is not in an appropriate numeric format of ' ||
                                         p_format || '.');
     
  
    -- Tell calling routine P_ITEM is NOT a number
    RETURN 0;
  
END validate_number;

/****************************************************************************************
  The create_station function inserts known good data into the SIM table SIM_STATIONS
  After the station has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
 ****************************************************************************************/
FUNCTION create_station(p_station_rec IN sim_stations%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create station using values in station record */
  INSERT INTO sim_stations
    (sta_seq,
     sta_id,
     sta_name,
     sta_is_number,
     sta_org_id,
     sta_desc,
     sta_tsmorgan_org_id,
     sta_tsmorgan_is_number,
     sta_status,
     sta_estab_date,
     sta_export_seq,
     sta_tsmvstc_is_number,
     sta_tsmvstc_org_id,
     sta_epa_key_id,
     sta_zid_rel,
     sta_influence_area,
     sta_travel_dir,
     sta_water_depth,
     sta_water_depth_unit,
     sta_ecoregion)
  VALUES
    (p_station_rec.sta_seq, /* STA_SEQ            */
     p_station_rec.sta_id, /* STA_ID             */
     p_station_rec.sta_name, /* STA_NAME           */
     p_station_rec.sta_is_number,
     p_station_rec.sta_org_id, /* STATION ORG ID     */
     p_station_rec.sta_desc, /* STA_DESC           */
     p_station_rec.sta_tsmorgan_org_id, /* TSMORGAN_ORG_ID    */
     p_station_rec.sta_tsmorgan_is_number, /* TSMORGAN_IS_NUMBER */
     p_station_rec.sta_status, /* STA_STATUS         */
     p_station_rec.sta_estab_date, /* STA_ESTAB_DATE     */
     p_station_rec.sta_export_seq, /* STA_EXPORT_SEQ     */
     p_station_rec.sta_tsmvstc_is_number, /* STA_TSMVSTC_IS_NUMBER     */
     p_station_rec.sta_tsmvstc_org_id, /* STA_TSMVSTC_ORG_ID */
     p_station_rec.sta_epa_key_id,
     p_station_rec.sta_zid_rel,
     p_station_rec.sta_influence_area,
     p_station_rec.sta_travel_dir,
     p_station_rec.sta_water_depth,
     p_station_rec.sta_water_depth_unit,
     p_station_rec.sta_ecoregion);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM || ' - inserting station.');
     
    RETURN 0;
    
END create_station;


/****************************************************************************************
 * This function checks to see if a station exists by checking for existing station ID  *
 ****************************************************************************************/
FUNCTION check_if_station_exists
(
  p_station_cd         IN VARCHAR2,
  p_org_id             IN VARCHAR2,
  p_sta_seq            OUT NUMBER,
  v_sequence_number    OUT NUMBER,
  p_tsmstatn_is_number OUT NUMBER
) RETURN NUMBER IS

  v_storet_station_cd VARCHAR2(15) := p_station_cd;
  v_sim_station_cd    VARCHAR2(15) := p_station_cd;
  v_return            NUMBER(1) := 0;
  v_seq               NUMBER(10);
  v_seq_num           NUMBER(4);
  v_success           NUMBER(1);
  v_sta_rec           sim_stations%ROWTYPE;
  v_stor_sta_rec      tsmstatn%ROWTYPE;
  v_tsmstatn_is_nbr   NUMBER(8);
  v_type              NUMBER(8);
  v_well_rec          sim_station_wells%ROWTYPE;
  v_stor_well_rec     tsmwell%ROWTYPE;
  v_well_loc_rec      sim_station_locations%ROWTYPE;
  v_storet_loc_rec    tsmalp%ROWTYPE;
  v_well_seq          NUMBER(10);
  v_loc_seq           NUMBER(10);
  v_well_is_number    NUMBER(8);

  e_bad_well EXCEPTION;

  CURSOR c_compare_station_id(p_station_cd IN VARCHAR2, p_station_org_id IN VARCHAR2) IS
    SELECT identification_cd,
           tsmstatn_is_number,
           tsmvstc_is_number
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_cd, 15)
       AND tsmstatn_org_id = rpad(p_station_org_id, 8);

  CURSOR c_compare_sim_stat_id(p_station_cd IN VARCHAR2, p_station_org_id IN VARCHAR2, p_tsmstatn_is_nbr IN NUMBER) IS
    SELECT sta_seq,
           sta_id
      FROM sim_stations
     WHERE sta_id = p_station_cd
       AND sta_is_number = p_tsmstatn_is_nbr
       AND sta_org_id = rpad(p_station_org_id, 8);

  CURSOR c_get_station_seq IS
    SELECT sim_stations_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_well_seq IS
    SELECT sim_station_wells_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_location_seq IS
    SELECT sim_station_locations_seq.NEXTVAL
      FROM dual;

  CURSOR c_storet_station IS
    SELECT *
      FROM tsmstatn
     WHERE identification_cd = rpad(p_station_cd, 15)
       AND tsmstatn_org_id = rpad(p_org_id, 8);

  CURSOR c_storet_wells(p_tsmwell_is_number IN NUMBER, p_tsmwell_org_id IN VARCHAR2) IS
    SELECT *
      FROM tsmwell
     WHERE tsmwell_is_number = p_tsmwell_is_number
       AND tsmwell_org_id = rpad(p_tsmwell_org_id, 8);

  CURSOR c_wells_short(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT tsmwell_is_number
      FROM tsmwell
     WHERE tsmstatn_is_number = p_tsmstatn_is_number
       AND tsmstatn_org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_storet_well_loc(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT *
      FROM tsmalp
     WHERE tsmstatn0is_number = p_tsmstatn_is_number
       AND tsmstatn0org_id = rpad(p_tsmstatn_org_id, 8);

  CURSOR c_get_next_sequence(p_tsmstatn_is_number IN NUMBER, p_tsmstatn_org_id IN VARCHAR2) IS
    SELECT MAX(sequence_number)
      FROM tsmalp
     WHERE tsmstatn0is_number = p_tsmstatn_is_number
       AND tsmstatn0org_id = rpad(p_tsmstatn_org_id, 8);

BEGIN

  /* Check STORET */
  OPEN c_compare_station_id(p_station_cd, p_org_id);
  FETCH c_compare_station_id
    INTO v_storet_station_cd, v_tsmstatn_is_nbr, v_type;

  IF c_compare_station_id%NOTFOUND THEN
  
    v_return := 0;
    --It has to be a well or a facility station      
  
  ELSIF v_type != 6
        AND v_type != 39
        AND (v_type < 22 OR v_type > 26) THEN
  
    --10/04/05 --RWK --Re-worded and fixed spelling in the following error message
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Station, ' || p_station_cd ||
                                 ', is not a Well Station, Facility Station or CERCLA Superfund Site. Well was not imported.');
     
    v_return := 0;
  
  ELSE
  
    --Check to see if it's in SIM, too
    OPEN c_compare_sim_stat_id(p_station_cd, p_org_id, v_tsmstatn_is_nbr);
    FETCH c_compare_sim_stat_id
      INTO v_seq, v_sim_station_cd;
  
    IF c_compare_sim_stat_id%NOTFOUND THEN
    
      OPEN c_get_station_seq;
      FETCH c_get_station_seq
        INTO v_seq;
      CLOSE c_get_station_seq;
    
      OPEN c_storet_station;
      FETCH c_storet_station
        INTO v_stor_sta_rec;
      CLOSE c_storet_station;
    
      v_sta_rec.sta_seq := v_seq;
      v_sta_rec.sta_is_number := v_stor_sta_rec.tsmstatn_is_number;
      v_sta_rec.sta_org_id := rtrim(v_stor_sta_rec.tsmstatn_org_id);
      v_sta_rec.sta_id := rtrim(v_stor_sta_rec.identification_cd);
      v_sta_rec.sta_name := v_stor_sta_rec.NAME;
      v_sta_rec.sta_estab_date := v_stor_sta_rec.establishment_date;
      v_sta_rec.sta_desc := v_stor_sta_rec.description_text;
      v_sta_rec.sta_tsmorgan_org_id := p_org_id;
      v_sta_rec.sta_tsmorgan_is_number := v_stor_sta_rec.tsmorgan_is_number;
      v_sta_rec.sta_tsmvstc_is_number := v_stor_sta_rec.tsmvstc_is_number;
      v_sta_rec.sta_tsmvstc_org_id := v_stor_sta_rec.tsmvstc_org_id;
      v_sta_rec.sta_status := 'U';
      v_sta_rec.sta_export_seq := NULL;
      v_sta_rec.sta_epa_key_id := v_stor_sta_rec.epa_key_identifier;
      v_sta_rec.sta_zid_rel := v_stor_sta_rec.zid_relation_code;
      v_sta_rec.sta_influence_area := v_stor_sta_rec.influence_area;
      v_sta_rec.sta_travel_dir := v_stor_sta_rec.travel_dir_txt;
      v_sta_rec.sta_water_depth := v_stor_sta_rec.water_depth;
      v_sta_rec.sta_water_depth_unit := v_stor_sta_rec.water_depth_unit;
      v_sta_rec.sta_ecoregion := v_stor_sta_rec.ecoregion_name;
    
      v_success := create_station(v_sta_rec);
    
      IF v_success = 1 THEN
        v_return := 1;
      END IF;
    
    END IF;
  
    CLOSE c_compare_sim_stat_id;
  
    IF v_return = 1 THEN
    
      FOR v_well_is_number IN c_wells_short(v_tsmstatn_is_nbr, p_org_id) LOOP
      
        v_return := 0;
      
        OPEN c_get_well_seq;
        FETCH c_get_well_seq
          INTO v_well_seq;
        CLOSE c_get_well_seq;
      
        OPEN c_storet_wells(v_well_is_number.tsmwell_is_number, p_org_id);
        FETCH c_storet_wells
          INTO v_stor_well_rec;
        CLOSE c_storet_wells;
      
        v_well_rec.ssw_seq := v_well_seq;
        v_well_rec.ssw_sta_seq := v_seq;
        v_well_rec.ssw_well_status := v_stor_well_rec.status_code;
        v_well_rec.ssw_is_number := v_stor_well_rec.tsmwell_is_number;
        v_well_rec.ssw_org_id := v_stor_well_rec.tsmwell_org_id;
        v_well_rec.ssw_name := v_stor_well_rec.NAME;
        v_well_rec.ssw_well_no := v_stor_well_rec.id_number;
        v_well_rec.ssw_well_use := v_stor_well_rec.use_code;
        v_well_rec.ssw_develop_method := v_stor_well_rec.developmnt_mthd_cd;
        v_well_rec.ssw_well_const_meth := v_stor_well_rec.constructn_mthd_cd;
        v_well_rec.ssw_water_use := v_stor_well_rec.wtr_primry_use_cd;
        v_well_rec.ssw_gradient_type := v_stor_well_rec.gradient_type_code;
        v_well_rec.ssw_depth_completion := v_stor_well_rec.depth_cmpltn_msr;
        v_well_rec.ssw_depth_completion_unit := v_stor_well_rec.dpth_cmpltn_unt_cd;
        v_well_rec.ssw_depth_hole := v_stor_well_rec.depth_of_hole_msr;
        v_well_rec.ssw_depth_hole_unit := v_stor_well_rec.depth_hole_unt_cd;
        v_well_rec.ssw_well_height := v_stor_well_rec.well_head_ht_msr;
        v_well_rec.ssw_well_height_unit := v_stor_well_rec.wl_hd_ht_unit_cd;
        v_well_rec.ssw_casing_height := v_stor_well_rec.casing_hgt_msr;
        v_well_rec.ssw_casing_height_unit := v_stor_well_rec.casing_hgt_unit_cd;
        v_well_rec.ssw_disinfected := v_stor_well_rec.disinfected_ind_cd;
        v_well_rec.ssw_const_start_date := v_stor_well_rec.constructn_strt_dt;
        v_well_rec.ssw_const_end_date := v_stor_well_rec.constructn_end_dt;
        v_well_rec.ssw_init_pump_dur := v_stor_well_rec.init_pump_dur;
        v_well_rec.ssw_init_pump_dur_unit := v_stor_well_rec.int_pmp_dur_unt_cd;
        v_well_rec.ssw_init_pump_rate := v_stor_well_rec.init_pump_rate;
        v_well_rec.ssw_init_pump_rate_unit := v_stor_well_rec.int_pmp_rat_unt_cd;
        v_well_rec.ssw_natural_flow := v_stor_well_rec.naturl_flow_ind_cd;
        v_well_rec.ssw_init_borehole_diam := v_stor_well_rec.init_borehole_diam;
        v_well_rec.ssw_init_borehole_diam_unit := v_stor_well_rec.init_bhole_diam_un;
        v_well_rec.ssw_tsmstatn_is_number := v_stor_well_rec.tsmstatn_is_number;
        v_well_rec.ssw_tsmstatn_org_id := v_stor_well_rec.tsmstatn_org_id;
        v_well_rec.ssw_status := 'U';
        v_well_rec.ssw_import_seq := NULL;
        v_well_rec.ssw_export_seq := NULL;
      
        v_success := create_well(v_well_rec);
      
        IF v_success != 1 THEN
          RAISE e_bad_well;
        END IF;
      
      END LOOP;
      --CLOSE c_storet_wells;      
      v_return := 1;
    END IF;
  
    IF v_return = 1 THEN
    
      OPEN c_get_location_seq;
      FETCH c_get_location_seq
        INTO v_loc_seq;
      CLOSE c_get_location_seq;
    
      OPEN c_storet_well_loc(v_tsmstatn_is_nbr, p_org_id);
      FETCH c_storet_well_loc
        INTO v_storet_loc_rec;
      CLOSE c_storet_well_loc;
    
      v_well_loc_rec.stl_seq := v_loc_seq;
      v_well_loc_rec.stl_sta_seq := v_seq;
      v_well_loc_rec.stl_point_name := v_storet_loc_rec.point_name;
      v_well_loc_rec.stl_type_code := v_storet_loc_rec.type_code;
      v_well_loc_rec.stl_sequence_number := v_storet_loc_rec.sequence_number;
      v_well_loc_rec.stl_elev_datum := NULL;
      v_well_loc_rec.stl_elev_method := NULL;
      v_well_loc_rec.stl_elev_msr_date := v_storet_loc_rec.elvtn_msr_dt;
      v_well_loc_rec.stl_elevation_msr := v_storet_loc_rec.elevation_msr;
      v_well_loc_rec.stl_elevation_unit := NULL;
      v_well_loc_rec.stl_geopos_datum := NULL;
      v_well_loc_rec.stl_geopos_method := NULL;
      v_well_loc_rec.stl_gps_scale := v_storet_loc_rec.geopstng_scale_txt;
      v_well_loc_rec.stl_lat_dir := v_storet_loc_rec.lat_direction;
      v_well_loc_rec.stl_lat_degrees := v_storet_loc_rec.lat_degree_msr;
      v_well_loc_rec.stl_lat_minutes := v_storet_loc_rec.lat_minute_msr;
      v_well_loc_rec.stl_lat_seconds := v_storet_loc_rec.lat_second_msr;
      v_well_loc_rec.stl_long_dir := v_storet_loc_rec.long_direction;
      v_well_loc_rec.stl_long_degrees := v_storet_loc_rec.long_degree_msr;
      v_well_loc_rec.stl_long_minutes := v_storet_loc_rec.long_minute_msr;
      v_well_loc_rec.stl_long_seconds := v_storet_loc_rec.long_second_msr;
      v_well_loc_rec.stl_lat_long_msr_date := v_storet_loc_rec.lat_long_msr_dt;
      -- jah 3-18-2004 Changed to NULL; field is not used
      v_well_loc_rec.stl_latlong_accurcy := NULL;
      -- jah 3-18-2004 Changed to NULL; field is not used
      v_well_loc_rec.stl_latlong_accurcy_unit := NULL;
      v_well_loc_rec.stl_tsmrrr_is_number := v_storet_loc_rec.tsmrrr_is_number;
      v_well_loc_rec.stl_tsmrrr_org_id := v_storet_loc_rec.tsmrrr_org_id;
      v_well_loc_rec.stl_tsmfhu_is_number := v_storet_loc_rec.tsmfhu_is_number;
      v_well_loc_rec.stl_tsmfhu_org_id := v_storet_loc_rec.tsmfhu_org_id;
      v_well_loc_rec.stl_tsmstatn0is_number := v_storet_loc_rec.tsmstatn0is_number;
      v_well_loc_rec.stl_tsmstatn0org_id := v_storet_loc_rec.tsmstatn0org_id;
      v_well_loc_rec.stl_tsmwell_is_number := v_storet_loc_rec.tsmwell_is_number;
      v_well_loc_rec.stl_tsmwell_org_id := v_storet_loc_rec.tsmwell_org_id;
      v_well_loc_rec.stl_tsmgeopa_is_number := v_storet_loc_rec.tsmgeopa_is_number;
      v_well_loc_rec.stl_tsmgeopa_org_id := v_storet_loc_rec.tsmgeopa_org_id;
      v_well_loc_rec.stl_lat_dec_minutes := NULL;
      v_well_loc_rec.stl_long_dec_minutes := NULL;
      v_well_loc_rec.stl_tsmpipe_is_number := NULL;
      v_well_loc_rec.stl_tsmpipe_org_id := NULL;
      v_well_loc_rec.stl_tsmstatn1org_id := NULL;
      v_well_loc_rec.stl_tsmstatn1is_number := NULL;
      v_well_loc_rec.stl_tsmgeopa0is_number := NULL;
      v_well_loc_rec.stl_tsmgeopa0org_id := NULL;
      v_well_loc_rec.stl_tsmnal_cd := v_storet_loc_rec.tsmnal_cd;
      v_well_loc_rec.stl_tsmnal_state := v_storet_loc_rec.tsmnal_state;
      v_well_loc_rec.stl_nrcs := v_storet_loc_rec.nrcs_wtrsd_id_num;
      v_well_loc_rec.stl_status := 'U';
      v_well_loc_rec.stl_import_seq := NULL;
      v_well_loc_rec.stl_export_seq := NULL;
    
      v_success := create_station_location(v_well_loc_rec);
    
    END IF;
  
    v_return := 1;
  
  END IF;

  p_sta_seq := v_seq;

  --and get the next available tsmalp sequence number
  OPEN c_get_next_sequence(v_tsmstatn_is_nbr, p_org_id);
  FETCH c_get_next_sequence
    INTO v_seq_num;
  CLOSE c_get_next_sequence;

  IF v_seq_num IS NULL THEN
    v_sequence_number := 1;
  ELSE
    v_sequence_number := v_seq_num + 1;
  END IF;
  CLOSE c_compare_station_id;

  p_tsmstatn_is_number := v_tsmstatn_is_nbr;

  /** Station is unique **/
  RETURN(nvl(v_return, 0));

EXCEPTION

  WHEN e_bad_well THEN
  
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' grabbing STORET wells in check_if_station_exists.');
     
  
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - in check_if_station_exists. ');
     
                                 
    RETURN 0;
  
END check_if_station_exists;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_org_info
(
  p_imp_cfg_seq     IN NUMBER,
  p_tsmorgan_org_id IN VARCHAR2,
  p_tsmorgan_is_nbr OUT NUMBER
) RETURN NUMBER IS

  CURSOR c_get_tsmorgan_nbr(p_cursor_org_id IN VARCHAR2) IS
    SELECT tsmorgan_is_number
      FROM tsmorgan
     WHERE org_id = rpad(p_cursor_org_id, 8);

BEGIN
  OPEN c_get_tsmorgan_nbr(p_tsmorgan_org_id);
  FETCH c_get_tsmorgan_nbr
    INTO p_tsmorgan_is_nbr;
  IF c_get_tsmorgan_nbr%NOTFOUND THEN
    p_tsmorgan_is_nbr := NULL;
  END IF;
  CLOSE c_get_tsmorgan_nbr;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END get_org_info;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_instructions
(
  p_imp_cfg_seq      IN NUMBER,
  p_instruction_type IN VARCHAR2
) RETURN VARCHAR2 IS

  v_instr_choice VARCHAR2(15);

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_cfg_instrs,
           sim_imp_instr_choices,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND sici_sicc_seq = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT siic_code
      FROM sim_imp_instr_choices,
           sim_imp_cfg_instrs,
           sim_imp_instrs
     WHERE sici_siic_seq = siic_seq
       AND siic_sii_seq = sii_seq
       AND sii_code = p_instr_type
       AND siic_dflt_flag = 1;

BEGIN
  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr
    INTO v_instr_choice;
  IF c_get_cfg_instr%NOTFOUND THEN
    /* no choice could be found, get default choice */
    OPEN c_get_instr_default(p_instruction_type);
    FETCH c_get_instr_default
      INTO v_instr_choice;
    CLOSE c_get_instr_default;
  END IF;
  CLOSE c_get_cfg_instr;
  RETURN v_instr_choice;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM || ' - in get_instructions.');
     
    
    RETURN NULL;
END get_instructions;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_allowable_value
(
  p_alval_seq IN NUMBER,
  p_wel_seq   IN NUMBER
) RETURN VARCHAR2 IS

  v_item_name VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_seq = p_alval_seq;

BEGIN
  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name
    INTO v_item_name;
  IF c_get_alval_name%NOTFOUND THEN
    v_item_name := NULL;
  END IF;
  CLOSE c_get_alval_name;
  RETURN v_item_name;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - getting allowable value from sequence.');
     

END get_allowable_value;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION validate_prmvl_item
(
  p_orig_item   IN CHAR,
  p_imp_cfg_seq IN NUMBER,
  p_item_name   IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_table_name  IN VARCHAR,
  p_column_name IN VARCHAR2,
  p_wel_seq     IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(100);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_value   VARCHAR2(40);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);
  v_storet_pk      sim_import_wells_pkg.t_storet_pk_type;

  CURSOR c_check_prmvl(p_item IN VARCHAR2) IS
    SELECT field_value
      FROM tsmprmvl
     WHERE table_name = rpad(p_table_name, 8)
       AND field_name = p_column_name
       AND upper(field_value) = upper(p_item);

BEGIN

  v_storet_pk := sim_import_wells_pkg.check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, v_converted_item, p_wel_seq);

  v_storet_value := v_converted_item;

  IF v_storet_pk.is_number IS NULL
     AND NOT v_trans_exist THEN
    OPEN c_check_prmvl(v_converted_item);
    FETCH c_check_prmvl
      INTO v_storet_value;
    IF c_check_prmvl %NOTFOUND THEN
      v_storet_value := NULL;
    END IF;
    CLOSE c_check_prmvl;
  
    IF v_storet_value IS NULL THEN
      v_instr_id := 'WEL-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'WEL-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                     v_converted_item ||
                                     ' is not a valid value - translation record has been created.');
         
      
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                     v_converted_item ||
                                     ' is not a valid value.');
         
      
      END IF;
    ELSIF v_storet_value IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' || p_orig_item ||
                                   ' has been mapped to the invalid storet value: ' ||
                                   v_converted_item);
       
    
    END IF;
  ELSIF v_storet_pk.is_number IS NULL
        AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - checking STORET PRMVL.');
     
    
    RETURN NULL;
END validate_prmvl_item;

/****************************************************************************************
  The get_delimiter function checks the table sim_imp_config to find the value entered
  for the delimiter.
 ****************************************************************************************/
FUNCTION get_delimiter
(
  p_imp_cfg_seq IN NUMBER,
  p_delimiter   OUT VARCHAR2
) RETURN NUMBER IS

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT sicc_delimiter
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

BEGIN
  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg
    INTO p_delimiter;
  IF c_get_imp_cfg%NOTFOUND THEN
    RETURN 0;
  END IF;
  CLOSE c_get_imp_cfg;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    RETURN 0;
END get_delimiter;

/****************************************************************************************
  The get_item_pos function checks the table sim_imp_config SIM_IMP_CFG_DETS SIM_IMP_COLS 
  to find the value entered for the position of the item in the list
 ****************************************************************************************/
FUNCTION get_item_pos
(
  p_sicc_seq      IN NUMBER,
  p_pos_table     IN OUT col_table_num_type,
  p_default_table IN OUT col_table_type,
  p_col_cnt       IN OUT NUMBER
) RETURN NUMBER IS
  v_import_type_seq NUMBER(10);
  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT sicc_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;
  CURSOR c_get_dflt_pos(p_type_seq IN NUMBER) IS
    SELECT sic_dflt_pos
      FROM sim_imp_cols
     WHERE sic_sid_seq = p_type_seq;
  CURSOR c_get_col_pos(p_sicc_seq IN NUMBER, p_dflt_pos IN NUMBER) IS
    SELECT sicdt_pos,
           sicdt_default
      FROM sim_imp_cfg_dets,
           sim_imp_cols
     WHERE sic_seq = sicdt_sic_seq
       AND sicdt_sicc_seq = p_sicc_seq
       AND sic_dflt_pos = p_dflt_pos
       AND (sicdt_included IS NOT NULL OR sicdt_generate IS NOT NULL);
BEGIN
  /* get the import type in order to limit the columns selected to the right type */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type
    INTO v_import_type_seq;
  IF c_get_import_type%NOTFOUND THEN
    v_import_type_seq := NULL;
    raise_application_error(-20102, 'Unable to get import type from import configuration.');
  END IF;
  CLOSE c_get_import_type;
  p_col_cnt := 0;
  /* loop through import columns and assign actual position to table indexed by
  default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP
    /* get actual position and item default from configuration table */
    OPEN c_get_col_pos(p_sicc_seq, v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos
      INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
    IF c_get_col_pos%NOTFOUND THEN
      /* if there is nothing in the configuration table for this column, position and default will be null */
      p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
      p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
    ELSE
      p_col_cnt := p_col_cnt + 1;
    END IF;
    CLOSE c_get_col_pos;
  END LOOP;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    raise_application_error(-20103, 'Failed to get column positions, item names, and column sequences.');
END get_item_pos;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_column_seq
(
  p_cfg_seq IN NUMBER,
  p_cfg_pos IN NUMBER
) RETURN NUMBER IS
  v_imp_col_seq NUMBER(10);
  e_no_cfg_found EXCEPTION;
  CURSOR c_get_col_seq(p_cfg_seq IN NUMBER, p_cfg_pos IN NUMBER) IS
    SELECT sicdt_sic_seq
      FROM sim_imp_cfg,
           sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = sicc_seq
       AND sicc_seq = p_cfg_seq
       AND sicdt_pos = p_cfg_pos;
BEGIN
  OPEN c_get_col_seq(p_cfg_seq, p_cfg_pos);
  FETCH c_get_col_seq
    INTO v_imp_col_seq;
  IF c_get_col_seq%NOTFOUND THEN
    v_imp_col_seq := NULL;
    RAISE e_no_cfg_found;
  END IF;
  CLOSE c_get_col_seq;
  RETURN v_imp_col_seq;
EXCEPTION
  WHEN e_no_cfg_found THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - unable to find sequence in sim_imp_cfg_dets for pos: ' ||
                                 p_cfg_pos);
     
    
    raise_application_error(-20100, 'Import configuration is not complete');
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - getting column sequence.');
     
    
    RETURN NULL;
END get_column_seq;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION check_storet_translation
(
  p_code           IN VARCHAR2,
  p_imp_cfg_seq    IN NUMBER,
  p_col_seq        IN NUMBER,
  p_trans_exist    OUT BOOLEAN,
  v_converted_item OUT VARCHAR2,
  p_sta_seq        IN NUMBER
) RETURN t_storet_pk_type IS

  v_storet_pk   sim_import_wells_pkg.t_storet_pk_type;
  v_seq         NUMBER(10);
  v_table_name  VARCHAR2(80);
  v_column_name VARCHAR2(80);
  v_dyna_cursor VARCHAR2(200);
  v_comma_pos   NUMBER(10);

  TYPE sort_cursor_type IS REF CURSOR;
  sort_cursor sort_cursor_type;

  CURSOR c_get_storet_key(cp_code VARCHAR2, cp_imp_cfg_seq NUMBER, cp_col_seq NUMBER) IS
    SELECT st_seq,
           st_is_number,
           st_org_id,
           st_storet_code
      FROM sim_translations,
           sim_imp_cfg_dets
     WHERE st_user_code = cp_code
       AND st_sicdt_seq = sicdt_seq
       AND sicdt_sicc_seq = cp_imp_cfg_seq
       AND sicdt_sic_seq = cp_col_seq;

  CURSOR c_imp_cols(p_col_seq NUMBER) IS
    SELECT sic_storet_tab,
           sic_storet_col
      FROM sim_imp_cols
     WHERE sic_seq = p_col_seq;

  CURSOR c_comma(p_col_seq NUMBER) IS
    SELECT instr(sic_storet_col, ',') col_pos
      FROM sim_imp_cols
     WHERE sic_seq = p_col_seq;

  CURSOR c_inst_col(p_col_seq NUMBER) IS
    SELECT substr(sic_storet_col, 1, instr(sic_storet_col, ',') - 1) col_name
      FROM sim_imp_cols
     WHERE sic_seq = p_col_seq;

BEGIN

  OPEN c_get_storet_key(p_code, p_imp_cfg_seq, p_col_seq);
  FETCH c_get_storet_key
    INTO v_seq, v_storet_pk.is_number, v_storet_pk.org_id, v_converted_item;
  CLOSE c_get_storet_key;

  IF v_seq IS NULL THEN
    v_converted_item := p_code;
    p_trans_exist := FALSE;
  ELSIF v_seq IS NOT NULL THEN
    /* if the code is in the translations table, flag so it's not recreated */
    p_trans_exist := TRUE;
    IF v_storet_pk.is_number IS NULL THEN
      /* if there is a code in the translations table but no value to translate to, write error message
      status will be changed to 'N' later in the program */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_code ||
                                           ' exists in translations table but has not been translated to a value.');
       
    ELSE
      OPEN c_imp_cols(p_col_seq);
      FETCH c_imp_cols
        INTO v_table_name, v_column_name;
      CLOSE c_imp_cols;
      OPEN c_comma(p_col_seq);
      FETCH c_comma
        INTO v_comma_pos;
      CLOSE c_comma;
      IF v_comma_pos = 0 THEN
        v_dyna_cursor := 'SELECT RTRIM(' || v_column_name || ') ' || ' FROM ' ||
                         v_table_name || ' WHERE ' || v_table_name || '_IS_NUMBER = ' ||
                         v_storet_pk.is_number || ' AND ' || v_table_name ||
                         '_ORG_ID = ''' || v_storet_pk.org_id || '''';
      ELSE
        OPEN c_inst_col(p_col_seq);
        FETCH c_inst_col
          INTO v_column_name;
        CLOSE c_inst_col;
        v_dyna_cursor := 'SELECT RTRIM(' || v_column_name || ') ' || ' FROM ' ||
                         v_table_name || ' WHERE ' || v_table_name || '_IS_NUMBER = ' ||
                         v_storet_pk.is_number || ' AND ' || v_table_name ||
                         '_ORG_ID = ''' || v_storet_pk.org_id || '''';
      END IF;
    
      OPEN sort_cursor FOR v_dyna_cursor;
      LOOP
        FETCH sort_cursor
          INTO v_converted_item;
        EXIT WHEN sort_cursor%NOTFOUND;
      END LOOP;
      IF v_converted_item IS NOT NULL THEN
        p_trans_exist := TRUE;
      END IF;
    END IF;
  END IF;

  RETURN v_storet_pk;

EXCEPTION

  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - in check storet translations');
     
    
    RETURN NULL;
  
END check_storet_translation;

/****************************************************************************************
 
 ****************************************************************************************/ 
PROCEDURE create_trans_record
(
  p_code        IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER
) IS

  v_st_seq    NUMBER(10);
  v_sicdt_seq NUMBER(10);
  e_imp_cfg_dets EXCEPTION;
  e_trans_error EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT sicdt_seq
      FROM sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = p_imp_cfg_seq
       AND sicdt_sic_seq = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL
      FROM dual;

BEGIN
  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq
    INTO v_sicdt_seq;
  IF c_get_cfg_det_seq%NOTFOUND THEN
    v_sicdt_seq := NULL;
    RAISE e_imp_cfg_dets;
  END IF;
  CLOSE c_get_cfg_det_seq;
  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq
    INTO v_st_seq;
  IF c_get_st_seq%NOTFOUND THEN
    v_st_seq := NULL;
    RAISE e_trans_error;
  END IF;
  CLOSE c_get_st_seq;
  INSERT INTO sim_translations
    (st_seq,
     st_sicdt_seq,
     st_user_code,
     st_storet_code)
  VALUES
    (v_st_seq, /* st_seq */
     v_sicdt_seq, /* st_sicdt_seq */
     p_code, /* st_user_code */
     NULL /* st_storet_code */);

EXCEPTION
  WHEN e_imp_cfg_dets THEN
    
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Unable to find import configuration - stopped import');
     
    
    raise_application_error(-20101, 'Unable to find import configuration');
  WHEN e_trans_error THEN
    
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Unable to create new record in translations table - error getting sequence');
     
    
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - adding record to translations table.');
     
    
END create_trans_record;

/****************************************************************************************
  The Validate_Well_Number function checks in SIM and STORET to see if a well already 
  exists with the provided well number on the provided station.  
  Return 1 if no dup id found, 0 if a dup is found.
 ****************************************************************************************/
FUNCTION validate_well_number
(
  p_well_num IN VARCHAR2,
  p_sta_seq  IN NUMBER
) RETURN NUMBER IS

  v_return     NUMBER(1);
  v_sta_is_num NUMBER(12);
  v_sta_org_id CHAR(8);

  CURSOR c_get_well_num(p_sta_seq NUMBER) IS
    SELECT ssw_well_no
      FROM sim_station_wells
     WHERE ssw_sta_seq = p_sta_seq;

  -- DTW 11/12/03 - Also check for exististance of Well Number in STORET   
  CURSOR c_get_station_is_num(p_sta_seq NUMBER) IS
    SELECT sta_is_number,
           sta_org_id
      FROM sim_stations
     WHERE sta_seq = p_sta_seq;

  CURSOR c_get_storet_well_num(p_sta_is_number NUMBER, p_sta_org_id VARCHAR2) IS
    SELECT id_number
      FROM tsmwell
     WHERE tsmstatn_is_number = p_sta_is_number
       AND tsmstatn_org_id = p_sta_org_id;

BEGIN
  v_return := 1;
  -- First check in SIM
  FOR v_ssw_well_no IN c_get_well_num(p_sta_seq) LOOP
    IF v_ssw_well_no.ssw_well_no = p_well_num THEN
      v_return := 0;
    END IF;
  END LOOP;

  -- If not found in SIM, check in STORET
  IF v_return = 1 THEN
    OPEN c_get_station_is_num(p_sta_seq);
    FETCH c_get_station_is_num
      INTO v_sta_is_num, v_sta_org_id;
  
    IF c_get_station_is_num%FOUND THEN
      -- Station exists in STORET so check for dup well ids in STORET.
      FOR v_well_rec IN c_get_storet_well_num(v_sta_is_num, v_sta_org_id) LOOP
        IF TRIM(v_well_rec.id_number) = p_well_num THEN
          v_return := 0;
        END IF;
      END LOOP;
    END IF;
    CLOSE c_get_station_is_num;
  END IF;

  RETURN v_return;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - in validate well number.');
     
    
    RETURN 0;
END validate_well_number;

/****************************************************************************************
  function to look in the translations and allowable values tables for items that are
  hard coded values in STORET
 ****************************************************************************************/
FUNCTION validate_alval_item
(
  p_item_name   IN VARCHAR2,
  p_orig_item   IN VARCHAR2,
  p_imp_cfg_seq IN NUMBER,
  p_col_seq     IN NUMBER,
  p_wel_seq     IN NUMBER
) RETURN NUMBER IS
  v_converted_item VARCHAR2(80);
  v_item_sia_seq   NUMBER(10);
  v_trans_exist    BOOLEAN;
  v_instr_code     VARCHAR2(15);
BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
  hard coded values in STORET */
  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := sim_import_pkg.check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_line_cnt, v_trans_exist);
  IF v_converted_item IS NOT NULL THEN
    /* look in allowable values */
    v_item_sia_seq := sim_import_pkg.check_allowable_values(v_converted_item, p_col_seq, v_line_cnt);
    IF v_item_sia_seq IS NULL
       AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
    
      v_instr_code := get_instructions(p_imp_cfg_seq, 'WEL-TR');
      --
      IF v_instr_code = 'WEL-TR-01' THEN
        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                     v_converted_item ||
                                     ' is not an allowable value - translation record has been created.');
         
    
      ELSE
        
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                     v_converted_item ||
                                     ' is not an allowable value.');
         
    
      END IF;
    ELSIF v_item_sia_seq IS NULL
          AND v_trans_exist THEN
      /* item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_orig_item ||
                                   ' has been mapped to the invalid storet value: ' ||
                                   v_converted_item);
       
    
    END IF;
  END IF;

  RETURN v_item_sia_seq;
EXCEPTION
  WHEN OTHERS THEN
    
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - in validate allowable values.');
       
    
    RETURN NULL;
END validate_alval_item;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION write_error_message
(
  p_item_name IN VARCHAR2,
  p_req       IN BOOLEAN,
  p_line_nbr  IN NUMBER,
  p_provided  IN BOOLEAN,
  p_status    IN OUT VARCHAR2,
  p_comment   IN OUT VARCHAR2,
  p_sta_seq   IN NUMBER
) RETURN NUMBER IS
BEGIN
  IF p_provided THEN
    p_status := 'N';
  ELSE
    IF p_req THEN
      p_status := 'N';
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, substr('Required item, ' ||
                                          p_item_name ||
                                          ' not provided activity will not export.', 1, 240));
       
    
    ELSE
      p_status := p_status;
    END IF;
  END IF;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - in write error message.');
     
    
    RETURN 0;
END write_error_message;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_date
(
  p_raw_date IN VARCHAR2,
  p_col_seq  IN NUMBER,
  p_cfg_seq  IN NUMBER,
  p_line_cnt IN NUMBER,
  p_seq      IN NUMBER
) RETURN DATE IS
  v_start_date DATE;
  v_fmt        VARCHAR2(80);

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    v_fmt := NULL;
    /* no format specified, use default */
    v_start_date := trunc(to_date(p_raw_date, 'MM-DD-RRRR'));
  END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_date := trunc(to_date(p_raw_date, v_fmt));
  END IF;
  RETURN v_start_date;

EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, ' - date, ' || p_raw_date ||
                                 ', format does not match given import format ');
     
    
    RETURN NULL;
END get_date;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION get_time
(
  p_raw_time IN VARCHAR2,
  p_col_seq  IN NUMBER,
  p_cfg_seq  IN NUMBER
) RETURN DATE IS
  v_start_time DATE;
  v_fmt        VARCHAR2(80);
  e_no_cfg_found EXCEPTION;
  CURSOR c_get_date_fmt(p_col_seq IN NUMBER, p_cfg_seq IN NUMBER) IS
    SELECT sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_fmt_choices
     WHERE sicdt_sic_seq = p_col_seq
       AND sicdt_sicc_seq = p_cfg_seq
       AND sicdt_sifc_seq = sifc_seq;
BEGIN
  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq, p_cfg_seq);
  FETCH c_get_date_fmt
    INTO v_fmt;
  IF c_get_date_fmt%NOTFOUND THEN
    v_fmt := NULL;
    /* no format specified, use default */
    v_start_time := to_date(substr(p_raw_time, 1, 8), 'HH24:MI:SS');
  END IF;
  CLOSE c_get_date_fmt;
  IF v_fmt IS NOT NULL THEN
    v_start_time := to_date(substr(p_raw_time, 1, 8), v_fmt);
  END IF;
  RETURN v_start_time;
EXCEPTION
  WHEN e_no_cfg_found THEN
    raise_application_error(-20100, 'Import configuration for start or end time is incomplete or does not exist.');
    RETURN NULL;
  WHEN OTHERS THEN
    
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM || ' - in get time.');
     
    
    RETURN NULL;
END get_time;

/****************************************************************************************
  The get_tsmgeopa_is_nbr function finds the station IS number and ORG id from the Storet
  table TSMGEOPA The values are returned in a variable that is TYPE T_STORET_PK_TYPE 
  (HOLDS BOTH THE ORG AND IS #) 
  This function accounts for the 4 possible combinations of state and county ID's,
      1 It's a state postal code and a county name
      2 It's a state FIPS and a county name
      3 It's a state postal code and  county FIPS
      4 It's a  state FIPS and county FIPS
 ****************************************************************************************/
FUNCTION get_tsmgeopa_is_nbr
(
  p_state  IN VARCHAR2,
  p_county IN VARCHAR2
) RETURN t_storet_pk_type IS
  v_storet_tsmgeopa t_storet_pk_type;
  /*** four cursors for the four possible combinations of state and county id's *****/
  CURSOR c_find_geopa_1(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_postal_code = rpad(p_state, 2)
       AND county_name = p_county;
  CURSOR c_find_geopa_2(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_fips_code = rpad(p_state, 2)
       AND county_name = p_county;
  CURSOR c_find_geopa_3(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_postal_code = rpad(p_state, 2)
       AND fips_county_code = rpad(p_county, 3);
  CURSOR c_find_geopa_4(p_state IN VARCHAR2, p_county IN VARCHAR2) IS
    SELECT tsmgeopa_is_number,
           tsmgeopa_org_id
      FROM tsmgeopa
     WHERE state_fips_code = rpad(p_state, 2)
       AND fips_county_code = rpad(p_county, 3);
BEGIN
  BEGIN
    IF translate(p_state, 'a1234567890.- ', 'a') IS NOT NULL
       AND translate(p_county, 'a1234567890.- ', 'a') IS NOT NULL THEN
      -- it's a state postal and a county NAME
      OPEN c_find_geopa_1(p_state, p_county);
      FETCH c_find_geopa_1
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_1%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_1;
    ELSIF translate(p_state, 'a1234567890.- ', 'a') IS NULL
          AND translate(p_county, 'a1234567890.- ', 'a') IS NOT NULL THEN
      -- It's a state fips and a county name
      OPEN c_find_geopa_2(p_state, p_county);
      FETCH c_find_geopa_2
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_2%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_2;
    ELSIF translate(p_state, 'a1234567890.- ', 'a') IS NOT NULL
          AND translate(p_county, 'a1234567890.- ', 'a') IS NULL THEN
      -- it's a state postal and  county fips
      OPEN c_find_geopa_3(p_state, p_county);
      FETCH c_find_geopa_3
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_3%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_3;
    ELSIF translate(p_state, 'a1234567890.- ', 'a') IS NULL
          AND translate(p_county, 'a1234567890.- ', 'a') IS NULL THEN
      -- it's a  state fips and county FIPS
      OPEN c_find_geopa_4(p_state, p_county);
      FETCH c_find_geopa_4
        INTO v_storet_tsmgeopa;
      IF c_find_geopa_4%NOTFOUND THEN
        v_storet_tsmgeopa.is_number := NULL;
      END IF;
      CLOSE c_find_geopa_4;
    ELSE
       
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'get TSMgeopa IS COMBO NOT FOUND: ' ||
                                   v_storet_tsmgeopa.is_number ||
                                   'p_state -> ' || p_state ||
                                   ' p_county -> ' || p_county);
     
    
    END IF;
  EXCEPTION
    WHEN OTHERS THEN
      
    
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM || ' - find geopa');
       
    
  END;
  RETURN v_storet_tsmgeopa;
EXCEPTION
  WHEN OTHERS THEN
    
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - PROC GET_geopa_IS_NBR ');
     
    
    v_storet_tsmgeopa.is_number := NULL;
    v_storet_tsmgeopa.org_id := NULL;
    RETURN v_storet_tsmgeopa;
END get_tsmgeopa_is_nbr;

/****************************************************************************************
  The create_station_location function inserts known good data into the SIM table SIM_STATIONS
  After the station_location has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
 ****************************************************************************************/
FUNCTION create_station_location(p_station_location_rec IN sim_station_locations%ROWTYPE)
  RETURN NUMBER IS
BEGIN
  /* create station using values in station record */

  INSERT INTO sim_station_locations
    (stl_seq,
     stl_sta_seq,
     stl_ssw_seq,
     stl_point_name,
     stl_type_code,
     stl_sequence_number,
     stl_elev_datum,
     stl_elev_method,
     stl_elev_msr_date,
     stl_elevation_msr,
     stl_elevation_unit,
     stl_geopos_datum,
     stl_geopos_method,
     stl_gps_scale,
     stl_lat_dir,
     stl_lat_degrees,
     stl_lat_minutes,
     stl_lat_seconds,
     stl_long_dir,
     stl_long_degrees,
     stl_long_minutes,
     stl_long_seconds,
     stl_lat_long_msr_date,
     stl_latlong_accurcy,
     stl_latlong_accurcy_unit,
     stl_tsmrrr_is_number,
     stl_tsmrrr_org_id,
     stl_tsmfhu_is_number,
     stl_tsmfhu_org_id,
     stl_tsmstatn0is_number,
     stl_tsmstatn0org_id,
     stl_tsmwell_is_number,
     stl_tsmwell_org_id,
     stl_tsmgeopa_is_number,
     stl_tsmgeopa_org_id,
     stl_lat_dec_minutes,
     stl_long_dec_minutes,
     stl_tsmpipe_is_number,
     stl_tsmpipe_org_id,
     stl_tsmstatn1org_id,
     stl_tsmstatn1is_number,
     stl_tsmgeopa0is_number,
     stl_tsmgeopa0org_id,
     stl_tsmnal_cd,
     stl_tsmnal_state,
     stl_nrcs,
     stl_status,
     stl_import_seq,
     stl_export_seq,
     stl_lat_dec_degrees,
     stl_long_dec_degrees)
  VALUES
    (p_station_location_rec.stl_seq,
     p_station_location_rec.stl_sta_seq,
     p_station_location_rec.stl_ssw_seq,
     p_station_location_rec.stl_point_name,
     p_station_location_rec.stl_type_code,
     p_station_location_rec.stl_sequence_number,
     p_station_location_rec.stl_elev_datum,
     p_station_location_rec.stl_elev_method,
     p_station_location_rec.stl_elev_msr_date,
     p_station_location_rec.stl_elevation_msr,
     p_station_location_rec.stl_elevation_unit,
     p_station_location_rec.stl_geopos_datum,
     p_station_location_rec.stl_geopos_method,
     p_station_location_rec.stl_gps_scale,
     p_station_location_rec.stl_lat_dir,
     p_station_location_rec.stl_lat_degrees,
     p_station_location_rec.stl_lat_minutes,
     p_station_location_rec.stl_lat_seconds,
     p_station_location_rec.stl_long_dir,
     p_station_location_rec.stl_long_degrees,
     p_station_location_rec.stl_long_minutes,
     p_station_location_rec.stl_long_seconds,
     p_station_location_rec.stl_lat_long_msr_date,
     p_station_location_rec.stl_latlong_accurcy,
     p_station_location_rec.stl_latlong_accurcy_unit,
     p_station_location_rec.stl_tsmrrr_is_number,
     p_station_location_rec.stl_tsmrrr_org_id,
     p_station_location_rec.stl_tsmfhu_is_number,
     p_station_location_rec.stl_tsmfhu_org_id,
     p_station_location_rec.stl_tsmstatn0is_number,
     p_station_location_rec.stl_tsmstatn0org_id,
     p_station_location_rec.stl_tsmwell_is_number,
     p_station_location_rec.stl_tsmwell_org_id,
     p_station_location_rec.stl_tsmgeopa_is_number,
     p_station_location_rec.stl_tsmgeopa_org_id,
     p_station_location_rec.stl_lat_dec_minutes,
     p_station_location_rec.stl_long_dec_minutes,
     p_station_location_rec.stl_tsmpipe_is_number,
     p_station_location_rec.stl_tsmpipe_org_id,
     p_station_location_rec.stl_tsmstatn1org_id,
     p_station_location_rec.stl_tsmstatn1is_number,
     p_station_location_rec.stl_tsmgeopa0is_number,
     p_station_location_rec.stl_tsmgeopa0org_id,
     p_station_location_rec.stl_tsmnal_cd,
     p_station_location_rec.stl_tsmnal_state,
     p_station_location_rec.stl_nrcs,
     p_station_location_rec.stl_status,
     p_station_location_rec.stl_import_seq,
     p_station_location_rec.stl_export_seq,
     p_station_location_rec.stl_lat_dec_degrees,
     p_station_location_rec.stl_long_dec_degrees);
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM ||
                                 ' - inserting station locations.');
     
    
    RETURN 0;
END create_station_location;

/****************************************************************************************
  The create_well function inserts known good data into the SIM table SIM_STATION_WELLS
  After the well has loaded sucessfully the function returns a 1
  If there was a problem the function returns a zero.
 ****************************************************************************************/
FUNCTION create_well(p_well_rec IN sim_station_wells%ROWTYPE) RETURN NUMBER IS

BEGIN
  /* create well using values in well record */
  INSERT INTO sim_station_wells
    (ssw_seq,
     ssw_sta_seq,
     ssw_well_status,
     ssw_is_number,
     ssw_org_id,
     ssw_name,
     ssw_well_no,
     ssw_well_use,
     ssw_develop_method,
     ssw_well_const_meth,
     ssw_water_use,
     ssw_gradient_type,
     ssw_depth_completion,
     ssw_depth_completion_unit,
     ssw_depth_hole,
     ssw_depth_hole_unit,
     ssw_well_height,
     ssw_well_height_unit,
     ssw_casing_height,
     ssw_casing_height_unit,
     ssw_disinfected,
     ssw_const_start_date,
     ssw_const_end_date,
     ssw_init_pump_dur,
     ssw_init_pump_dur_unit,
     ssw_init_pump_rate,
     ssw_init_pump_rate_unit,
     ssw_natural_flow,
     ssw_init_borehole_diam,
     ssw_init_borehole_diam_unit,
     ssw_tsmstatn_is_number,
     ssw_tsmstatn_org_id,
     ssw_status,
     ssw_import_seq,
     ssw_export_seq)
  VALUES
    (p_well_rec.ssw_seq,
     p_well_rec.ssw_sta_seq,
     p_well_rec.ssw_well_status,
     p_well_rec.ssw_is_number,
     p_well_rec.ssw_org_id,
     p_well_rec.ssw_name,
     p_well_rec.ssw_well_no,
     p_well_rec.ssw_well_use,
     p_well_rec.ssw_develop_method,
     p_well_rec.ssw_well_const_meth,
     p_well_rec.ssw_water_use,
     p_well_rec.ssw_gradient_type,
     p_well_rec.ssw_depth_completion,
     p_well_rec.ssw_depth_completion_unit,
     p_well_rec.ssw_depth_hole,
     p_well_rec.ssw_depth_hole_unit,
     p_well_rec.ssw_well_height,
     p_well_rec.ssw_well_height_unit,
     p_well_rec.ssw_casing_height,
     p_well_rec.ssw_casing_height_unit,
     p_well_rec.ssw_disinfected,
     p_well_rec.ssw_const_start_date,
     p_well_rec.ssw_const_end_date,
     p_well_rec.ssw_init_pump_dur,
     p_well_rec.ssw_init_pump_dur_unit,
     p_well_rec.ssw_init_pump_rate,
     p_well_rec.ssw_init_pump_rate_unit,
     p_well_rec.ssw_natural_flow,
     p_well_rec.ssw_init_borehole_diam,
     p_well_rec.ssw_init_borehole_diam_unit,
     p_well_rec.ssw_tsmstatn_is_number,
     p_well_rec.ssw_tsmstatn_org_id,
     p_well_rec.ssw_status,
     p_well_rec.ssw_import_seq,
     p_well_rec.ssw_export_seq
     
     );
  COMMIT;
  RETURN 1;
EXCEPTION
  WHEN OTHERS THEN
    
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM || ' - inserting well.');
     
    
    RETURN 0;
END create_well;

/****************************************************************************************
 
 ****************************************************************************************/ 
FUNCTION check_tsmmad_value
(
  p_orig_item   IN VARCHAR2,
  p_item_name   IN VARCHAR2,
  p_category    IN VARCHAR2,
  p_subcategory IN VARCHAR2,
  p_col_seq     IN NUMBER,
  p_sta_seq     IN NUMBER,
  p_imp_cfg_seq IN NUMBER,
  p_imp_log_seq IN NUMBER
) RETURN VARCHAR2 IS

  -- This function checks to make sure that the given STORET Item is in the STORET TSMPRMVL table.

  v_converted_item VARCHAR2(120);
  v_trans_exist    BOOLEAN := TRUE;
  v_storet_value   VARCHAR2(12);
  v_instr_id       VARCHAR2(8);
  v_instr_code     VARCHAR2(15);
  v_storet_pk      sim_import_wells_pkg.t_storet_pk_type;

  CURSOR c_check_tsmmad(p_item IN VARCHAR2) IS
    SELECT id_code
      FROM tsmmad
     WHERE category = rpad(p_category, 10)
       AND subcategory = rpad(p_subcategory, 6)
       AND id_code = rpad(p_item, 12);

BEGIN

  v_storet_pk := sim_import_wells_pkg.check_storet_translation(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist, v_converted_item, p_sta_seq);

  v_storet_value := v_converted_item;

  IF v_storet_pk.is_number IS NULL
     AND NOT v_trans_exist THEN
    OPEN c_check_tsmmad(v_converted_item);
    FETCH c_check_tsmmad
      INTO v_storet_value;
    IF c_check_tsmmad %NOTFOUND THEN
      v_storet_value := NULL;
    END IF;
    CLOSE c_check_tsmmad;
  
    IF v_storet_value IS NULL
       AND NOT v_trans_exist THEN
      v_instr_id := 'WEL-TR';
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, v_instr_id);
      IF v_instr_code = 'WEL-TR-01' THEN
        /* if record could not be matched to anything in translations or STORET tables,
        create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq);
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                     v_converted_item ||
                                     ' is not a valid value - translation record has been created.');
         
    
      ELSE
         
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' ||
                                     v_converted_item ||
                                     ' is not a valid value.');
         
    
      END IF;
    ELSIF v_storet_value IS NULL
          AND v_trans_exist THEN
      /*  item has been translated to an invalid storet value */
       
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, p_item_name || ', ' || p_orig_item ||
                                   ' has been mapped to the invalid storet value: ' ||
                                   v_converted_item);
       
    
    END IF;
  ELSIF v_storet_pk.is_number IS NULL
        AND v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
    Status will be changed to 'N' later in the program. */
    v_storet_value := NULL;
  
  END IF;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
    
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, SQLERRM || ' - checking STORET TSMMAD.');
     
    
    RETURN NULL;
END check_tsmmad_value;

/****************************************************************************************
 * The import_wells procedure parses a text file line by line of data that is separated by
 * a delimiter.
 ****************************************************************************************/
PROCEDURE import_wells
(
  p_org_id         IN VARCHAR2,
  p_import_log_seq IN NUMBER,
  p_imp_cfg_seq    IN NUMBER,
  p_test_count     IN NUMBER,
  p_skip_header    IN NUMBER DEFAULT 0       -- CLM 08/02/2005 Skip header option    
  ) IS                                       --   1 = Skip  0 = Don't Skip

  v_import_desc    VARCHAR2(200);
  v_comp           NUMBER(1);
  v_num_cols       NUMBER(3);
  v_col_seq        NUMBER(10);
  v_dflt_pos       NUMBER(3);
  v_success        NUMBER(1);
  v_converted_item VARCHAR2(4000);
  v_delimiter      CHAR(1);
  v_inline         VARCHAR2(10000);
  v_bad_parse      BOOLEAN;
  v_item_name      VARCHAR2(80);
  v_item           VARCHAR2(4000);
  v_trans_exist    BOOLEAN := TRUE;
  v_line_table     sim_import_pkg.col_table_type;
  v_default_table  sim_import_wells_pkg.col_table_type;
  v_pos_table      sim_import_wells_pkg.col_table_num_type;
  v_error_cnt      NUMBER(10);
  v_complete_cnt   NUMBER(10);
  v_dummy_seq      NUMBER(10);
  v_tsmprmvl_item  VARCHAR2(40);
  v_sta_seq        NUMBER(10);

  v_lat_degrees    NUMBER(2);
  v_lat_minutes    NUMBER(2);
  v_lat_seconds    NUMBER(6, 4);
  v_lat_dir        VARCHAR2(1);
  v_check_lat_dir  VARCHAR2(1);
  v_lat_dec_min    NUMBER(6, 4);
  v_lat_dec_deg    NUMBER(10, 7);
  v_long_degrees   NUMBER(3);
  v_long_minutes   NUMBER(2);
  v_long_seconds   NUMBER(6, 4);
  v_long_dec_min   NUMBER(6, 4);
  v_long_dec_deg   NUMBER(10, 7);
  v_long_dir       VARCHAR2(1);
  v_check_long_dir VARCHAR2(1);

  v_stl_rec    sim_station_locations%ROWTYPE;
  v_well_rec   sim_station_wells%ROWTYPE;
  v_por_rec    tsmalp%ROWTYPE;
  v_storet_key sim_import_wells_pkg.t_storet_pk_type;

  v_elevation_unit     VARCHAR2(3);
  v_elevation_method   VARCHAR2(12);
  v_elevation_datum    VARCHAR2(12);
  v_elevation_date     DATE;
  v_tsmorgan_is_number VARCHAR2(8);
  v_tsmstatn_is_nbr    NUMBER(12);

  v_stl_seq NUMBER(10);
  v_id      VARCHAR2(15);

  v_elevation_msr NUMBER(9, 4);
  v_lat_long_date DATE;
  v_datum         VARCHAR2(12);
  v_method        VARCHAR2(12);
  v_gps_scale     VARCHAR2(20);

  v_sequence_number NUMBER(4);

  v_well_seq    NUMBER(10);
  v_well_name   VARCHAR2(35);
  v_well_num    VARCHAR2(15);
  v_well_use    VARCHAR2(30);
  v_well_status VARCHAR2(9);
  v_next        NUMBER(2) := 0;
  v_station_id  sim_stations.sta_id%TYPE;
  v_sid_seq     NUMBER(10);  

  e_bad_org EXCEPTION;
  e_no_cfg EXCEPTION;
  e_no_delimiter EXCEPTION;
  e_bad_columns EXCEPTION;
  e_no_station EXCEPTION;

  CURSOR c_get_well_seq IS
    SELECT sim_station_wells_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_station_location_seq IS
    SELECT sim_station_locations_seq.NEXTVAL
      FROM dual;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT sipl_seq,
           sipl_line_text,
           sipl_line_no
      FROM sim_imp_lines
     WHERE sipl_sil_seq = p_import_seq
       AND sipl_line_text IS NOT NULL -- bka  added to eliminate blank lines from the import
     ORDER BY sipl_line_no;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT sic_proc_order,
           sic_dflt_pos,
           sic_name,
           sic_seq
      FROM sim_imp_cols,
           sim_imp_defs,
           sim_imp_cfg
     WHERE sicc_sid_seq = sid_seq
       AND sic_sid_seq = sid_seq
       AND sicc_seq = p_imp_cfg_seq
     ORDER BY sic_proc_order;

  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(ssw_seq)
      FROM sim_station_wells
     WHERE ssw_import_seq = p_import_log_seq
       AND ssw_status = 'A';

  CURSOR c_get_point_of_record(station_is_number IN NUMBER) IS
    SELECT *
      FROM tsmalp
     WHERE tsmstatn0is_number = station_is_number
       AND tsmstatn0org_id = rpad(p_org_id, 8)
       AND type_code = '*POINT OF RECORD';

BEGIN

  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;
  v_line_cnt := 0;

  BEGIN
      
    SELECT sicc_sid_seq
      INTO v_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_imp_cfg_seq;
    
    sim_event_log_pkg.write_event(1,v_import_log_seq,p_org_id,v_sid_seq,NULL);
     
    
     
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message, NULL , 'Start: ' ||
                                 to_char(SYSDATE, 'HH24:MI:SS'));
     
                                     
    IF p_skip_header = 1 THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL, 'Header skipped');
      v_line_cnt := 1;   
    END IF;

    /* get tsmorgan_is_number */
    v_success := get_org_info(p_imp_cfg_seq, p_org_id, v_tsmorgan_is_number);
    IF v_success != 1
       OR v_org_is_nbr IS NULL THEN
      RAISE e_bad_org;
    END IF;
  
    /*  get the delimiter */
    v_delimiter := NULL;
    v_success := get_delimiter(p_imp_cfg_seq, v_delimiter);
    IF v_success != 1 THEN
      RAISE e_no_delimiter;
    END IF;
  
    /* get actual positions and default information for items being imported */
    v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table, v_num_cols);
    IF v_success != 1 THEN
      RAISE e_no_cfg;
    END IF;
  
    /* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP
    /* CLM 08/02/2005 If skip header = 'No' or this is not the first row */
     IF p_skip_header = 0 OR v_text_line.sipl_line_no <> 1 THEN    
      /* read one line */
      v_lat_degrees := NULL;
      v_lat_minutes := NULL;
      v_lat_dec_min := NULL;
      v_lat_seconds := NULL;
      v_lat_dir := NULL;
      v_long_degrees := NULL;
      v_long_minutes := NULL;
      v_long_dec_min := NULL;
      v_long_seconds := NULL;
      v_long_dir := NULL;
      v_lat_long_date := NULL;
      v_gps_scale := NULL;
      v_method := NULL;
      v_datum := NULL;
      v_elevation_msr := NULL;
      v_elevation_unit := NULL;
      v_elevation_method := NULL;
      v_elevation_datum := NULL;
      v_elevation_date := NULL;
    
      /**** get well sequence ******/
      BEGIN
        OPEN c_get_well_seq;
        FETCH c_get_well_seq
          INTO v_well_seq;
        CLOSE c_get_well_seq;
      
        v_inline := v_text_line.sipl_line_text;
        v_line_cnt := v_line_cnt + 1;
        v_status := 'A';
      
        -- send line to parsing routine:
        sim_import_pkg.parse_line_table(v_inline, v_line_table, v_num_cols, v_delimiter, v_bad_parse, v_import_log_seq);
        IF v_bad_parse THEN
          RAISE e_bad_columns;
        END IF;
        v_next := 0;
      
        /* loop through all Well columns */
        FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP
        
          /* get dflt position, item name, and column sequence */
          v_dflt_pos := v_columns.sic_dflt_pos;
          v_item_name := v_columns.sic_name;
          v_col_seq := v_columns.sic_seq;
          v_next := v_next + 1;
        
          /* if column is included on the import, continue. Don't look at positions <= 0 */
          IF v_dflt_pos > 0 THEN
          
            IF v_pos_table(v_dflt_pos) IS NOT NULL THEN
            
              /* get the actual item, or use default if null */
              IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN
              
                v_item := v_line_table(v_pos_table(v_dflt_pos));
              
              ELSIF (v_line_table(v_pos_table(v_dflt_pos)) IS NULL)
                    AND (v_default_table(v_dflt_pos) IS NOT NULL) THEN
              
                v_item := v_default_table(v_dflt_pos);
              
              ELSE
              
                v_item := NULL;
              
              END IF;
            
              -- Does V_ITEM contain only spaces? (or NULL)
              IF (TRIM(v_item) IS NULL) THEN
                -- if yes
              
                -- by definition V_ITEM is NULL!
                v_item := NULL;
              
              END IF;
            
              /* assign item to appropriate variables 
              and take care of any other procedural issues that need to be
              taken care of for or with a particular item */
            
              -------------
              -- Station ID
              -------------
              IF v_item_name = 'Station ID' THEN
              
                -- Does the station ID field contain anything?
                IF v_item IS NOT NULL THEN
                  -- if yes
                
                  -- jah 2-24-04 Remove trailing spaces but do NOT convert to uppercase
                  v_item := rtrim(v_item);
                
                  -- Translate
                  v_storet_key := check_storet_translation(v_item, p_imp_cfg_seq, v_col_seq, v_trans_exist, v_converted_item, -- OUT
                                                           v_well_seq);
                
                  -- Check if station exists in SIM
                  v_comp := check_if_station_exists(upper(v_converted_item), p_org_id, v_sta_seq, v_sequence_number, v_tsmstatn_is_nbr);
                
                  -- Does the station exist in SIM?
                  IF v_comp = 1 THEN
                    -- if yes
                  
                    -- jah 3-18-2004 changed from v_item to v_converted_item
                    v_id := v_converted_item;
                  
                  ELSE
                    -- if no (i.e., the station does not exist in SIM)
                  
                    -- Prevent migration of this record                    
                    v_status := 'N';
                    v_id := NULL;
                  
                    -- Do we need to create a translation record?
                    IF (v_storet_key.is_number IS NULL)
                       AND (NOT v_trans_exist)
                       AND (get_instructions(p_imp_cfg_seq, 'WEL-TR') = 'WEL-TR-01') THEN
                      -- if yes
                    
                      -- Create the translation record
                      create_trans_record(v_item, p_imp_cfg_seq, v_col_seq);
                    
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, v_item_name || ' ' ||
                                                   v_item ||
                                                   ' is not valid - translation record has been created.');
                       
   
                    END IF;
                  
                  END IF;
                
                ELSE
                  -- if no (i.e., the station ID field does NOT contain anything)
                
                  v_id := NULL;
                
                END IF;
              
                IF v_item IS NOT NULL THEN
                  v_station_id := v_item;
                END IF;
              
                ---------------------
                -- Well Head Latitude
                ---------------------                
              ELSIF v_item_name = 'Well Head Latitude' THEN
                v_lat_dir := NULL;
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := sim_import_pkg.get_latitude(v_import_log_seq, v_item, v_col_seq, p_imp_cfg_seq, v_well_seq, v_line_cnt, v_lat_degrees, v_lat_minutes, v_lat_seconds, v_lat_dir, v_lat_dec_min, v_lat_dec_deg);
                  
                    IF v_success = 2 THEN
                      v_status := 'N';
                      v_lat_dec_deg := NULL;
                      v_lat_dec_min := NULL;
                      v_lat_degrees := NULL;
                      v_lat_minutes := NULL;
                      v_lat_seconds := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                   
                 sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Latitude is required.');
                  
   
                  v_status := 'N';
                  v_lat_dec_deg := NULL;
                  v_lat_dec_min := NULL;
                  v_lat_degrees := NULL;
                  v_lat_minutes := NULL;
                  v_lat_seconds := NULL;
                END IF;
              
                -------------------------------
                -- Well Head Latitude Direction
                -------------------------------
              ELSIF v_item_name = 'Well Head Latitude Direction' THEN
              
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_well_seq);
                  IF v_dummy_seq IS NOT NULL THEN
                    IF v_lat_dir IS NULL THEN
                      v_lat_dir := get_allowable_value(v_dummy_seq, v_well_seq);
                    ELSE
                      v_check_lat_dir := get_allowable_value(v_dummy_seq, v_well_seq);
                      IF v_check_lat_dir <> v_lat_dir THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Given Latitude Direction is inconsistent with sign on Latitude.');
                         
   
                        v_status := 'N';
                        v_lat_dir := 'X'; -- so we don't default it
                      END IF;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, v_item ||
                                                 ' is not an allowable Latitude Direction.');
                     
   
                    v_status := 'N';
                    v_lat_dir := 'X'; -- so we don't default it
                  END IF;
                ELSE
                  -- v_item IS NULL, default to N
                  IF v_lat_dir IS NULL THEN
                    v_lat_dir := 'N';
                    
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt, 'Latitude Direction is required and will be set to N');
                     
   
                  END IF;
                END IF;
              
                ----------------------
                -- Well Head Longitude
                ----------------------               
              ELSIF v_item_name = 'Well Head Longitude' THEN
                v_long_dir := NULL;
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := sim_import_pkg.get_longitude(v_import_log_seq, v_item, v_col_seq, p_imp_cfg_seq, v_well_seq, v_line_cnt, v_long_degrees, v_long_minutes, v_long_seconds, v_long_dec_min, v_long_dec_deg, v_long_dir);
                  
                    IF v_success = 2 THEN
                      v_status := 'N';
                      v_long_dec_deg := NULL;
                      v_long_dec_min := NULL;
                      v_long_degrees := NULL;
                      v_long_minutes := NULL;
                      v_long_seconds := NULL;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Longitude is required.');
                   
   
                  v_status := 'N';
                  v_long_dec_deg := NULL;
                  v_long_dec_min := NULL;
                  v_long_degrees := NULL;
                  v_long_minutes := NULL;
                  v_long_seconds := NULL;
                END IF;
              
                --------------------------------
                -- Well Head Longitude Direction
                --------------------------------
              ELSIF v_item_name = 'Well Head Longitude Direction' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 061203 slp - Remove trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_well_seq);
                  IF v_dummy_seq IS NOT NULL THEN
                    IF v_long_dir IS NULL THEN
                      v_long_dir := get_allowable_value(v_dummy_seq, v_well_seq);
                    ELSE
                      v_check_long_dir := get_allowable_value(v_dummy_seq, v_well_seq);
                      IF v_check_long_dir <> v_long_dir THEN
                         
                        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, 'Given Longitude Direction is inconsistent with sign on Longitude.');
                         
   
                        v_status := 'N';
                        v_long_dir := 'X'; -- so we don't default it in
                      END IF;
                    END IF;
                  ELSE
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt, v_item ||
                                                 ' is not an allowable Longitude Direction.');
                     
   
                    v_status := 'N';
                    v_long_dir := 'X'; -- so we don't default it in
                  END IF;
                ELSE
                  -- v_item is NULL, default to W
                  IF v_long_dir IS NULL THEN
                    v_long_dir := 'W';
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                 'Longitude Direction is required and will be set to W');
                     
   
                  END IF;
                END IF;
              
                -----------------------------
                -- Well Head Measurement Date
                -----------------------------
              ELSIF v_item_name = 'Well Head Measurement Date' THEN
                IF v_item IS NULL THEN
                  v_lat_long_date := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_lat_long_date := get_date(v_item, v_col_seq, p_imp_cfg_seq, v_line_cnt, v_well_seq);
                    IF v_lat_long_date IS NULL THEN
                      --the date format is bad
                      v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                   v_item_name ||
                                                   ' format does not match format in import configuration');
                       
   
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                END IF;
              
                ----------------------------------
                -- Well Head Geopositioning Method
                ----------------------------------
              ELSIF v_item_name = 'Well Head Geopositioning Method' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_method := check_tsmmad_value(v_item, v_item_name, 'HORIZONTAL', 'METHOD', v_col_seq, v_well_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_method IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_method := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                               'Geopositioning method is required.');
                   
   
                END IF;
              
                ---------------------------------
                -- Well Head Geopositioning Datum
                ---------------------------------
              ELSIF v_item_name = 'Well Head Geopositioning Datum' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_datum := check_tsmmad_value(v_item, v_item_name, 'HORIZONTAL', 'DATUM', v_col_seq, v_well_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_datum IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_datum := NULL;
                   
                  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                               'Geopositioning Datum is required.');
                   
   
                END IF;
              
                ------------------
                -- Well Head Scale
                ------------------
              ELSIF v_item_name = 'Well Head Scale' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 20 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                   v_item_name ||
                                                   ' is longer than 20 characters and will be truncated.');
                       
   
                      v_item := substr(v_item, 1, 20);
                    END IF;
                    v_gps_scale := v_item;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_gps_scale := NULL;
                END IF;
              
                ----------------------
                -- Well Head Elevation
                ----------------------
              ELSIF v_item_name = 'Well Head Elevation' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_success := validate_number(v_item, v_item_name, '99999.9999', v_well_seq);
                    IF v_success = 0 THEN
                      v_elevation_msr := NULL;
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                   'Elevation value ' ||
                                                   v_item ||
                                                   ' is not allowable. It must be a number.');
                       
   
                    ELSE
                      v_elevation_msr := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_elevation_msr := NULL;
                END IF;
              
                ----------------------------
                -- Well Head Elevation Units
                ----------------------------
              ELSIF v_item_name = 'Well Head Elevation Units' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_well_seq);
                  IF v_elevation_msr IS NOT NULL THEN
                    IF v_dummy_seq IS NULL THEN
                      v_elevation_unit := NULL;
                      v_status := 'N';
                    ELSE
                      v_elevation_unit := get_allowable_value(v_dummy_seq, v_well_seq);
                    END IF;
                  ELSE
                    v_elevation_unit := NULL;
                     
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                 'Elevation Unit is not needed since Elevation is not provided.');
                     
   
                  END IF;
                ELSE
                  v_elevation_unit := NULL;
                END IF;
              
                -----------------------------
                -- Well Head Elevation Method
                -----------------------------
              ELSIF v_item_name = 'Well Head Elevation Method' THEN
                --sfp, this section rewritten 062701
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_elevation_method := check_tsmmad_value(v_item, v_item_name, 'VERTICAL', 'METHOD', v_col_seq, v_well_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_elevation_method IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_elevation_method := NULL;
                END IF;
              
                ----------------------------
                -- Well Head Elevation Datum
                ----------------------------
              ELSIF v_item_name = 'Well Head Elevation Datum' THEN
                --sfp, this section rewritten 062701
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_elevation_datum := check_tsmmad_value(v_item, v_item_name, 'VERTICAL', 'DATUM', v_col_seq, v_well_seq, p_imp_cfg_seq, v_import_log_seq);
                  IF v_elevation_datum IS NULL THEN
                    v_status := 'N';
                  END IF;
                ELSE
                  v_elevation_datum := NULL;
                END IF;
              
                -----------------------------
                -- Elevation Measurement Date
                -----------------------------
              ELSIF v_item_name = 'Elevation Measurement Date' THEN
                IF v_item IS NULL THEN
                  v_elevation_date := NULL;
                ELSE
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_elevation_date := get_date(v_item, v_col_seq, p_imp_cfg_seq, v_line_cnt, v_well_seq);
                    IF v_elevation_date IS NULL THEN
                      --the date format is bad
                      v_status := 'N'; -- Only change status to N if there was a failure to obtain a correct date from a non-null item
                       
                    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                 v_item_name ||
                                                 ' format does not match format in import configuration');
                     
   
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                END IF;
              
                --------------
                -- Well Number
                --------------
              ELSIF v_item_name = 'Well Number' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    v_well_num := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                    v_success := validate_well_number(v_well_num, v_sta_seq);
                    IF v_success != 1 THEN
                      v_status := 'N';
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                   v_item_name || ', ' ||
                                                   v_item ||
                                                   ' already exists on this station.');
                       
   
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_well_num := NULL;
                END IF;
              
                ------------
                -- Well Name
                ------------
              ELSIF v_item_name = 'Well Name' THEN
                IF v_item IS NOT NULL THEN
                  -- 09/30/03 TCJ - add code to check for translations
                  v_item := sim_import_pkg.check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_line_cnt, v_trans_exist);
                  IF v_item IS NOT NULL THEN
                    IF length(v_item) > 40 THEN
                       
                      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                                   v_item_name || ', ' ||
                                                   v_item ||
                                                   ', is longer than 40 characters and will be truncated.');
                       
   
                      v_well_name := substr(v_item, 1, 40);
                    ELSE
                      v_well_name := v_item;
                    END IF;
                  ELSE
                    v_status := 'N'; -- v_item translated to null - set status to 'N'
                  END IF;
                ELSE
                  v_well_name := NULL;
                END IF;
              
                --------------
                -- Well Status
                --------------
              ELSIF v_item_name = 'Well Status' THEN
                IF v_item IS NOT NULL THEN
                  v_item := rtrim(v_item); -- 06/13/02 SLP - Removed trailing spaces.
                  v_dummy_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_well_seq);
                  IF v_dummy_seq IS NULL THEN
                    v_well_status := NULL;
                    v_status := 'N';
                  ELSE
                    v_well_status := get_allowable_value(v_dummy_seq, v_well_seq);
                  END IF;
                ELSE
                  v_well_status := NULL;
                END IF;
              
                -----------
                -- Well Use
                -----------
              ELSIF v_item_name = 'Well Use' THEN
                IF v_item IS NOT NULL THEN
                  v_tsmprmvl_item := validate_prmvl_item(v_item, p_imp_cfg_seq, v_item_name, v_col_seq, 'TSMWELL', 'USE_CODE', v_well_seq);
                  IF v_tsmprmvl_item IS NULL THEN
                    v_well_use := NULL;
                  ELSE
                    v_well_use := substr(v_tsmprmvl_item, 1, 30);
                  END IF;
                ELSE
                  v_well_use := NULL;
                END IF;
              
              END IF; --end of v_item_names
            
            END IF;
          END IF;
        END LOOP;
      
        /*** END WELL LOOP ***/
      
        --Now check all the STORET business rules
      
        IF v_station_id IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Station ID is required.');
          
   
          v_status := 'N';
        END IF;
      
        IF v_id IS NULL
           AND v_station_id IS NOT NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Station ID ' || v_station_id ||
                                       ' is not in STORET.');
          
   
          v_status := 'N';
        END IF;
      
        --If lat/long not given default the Station *POINT OF RECORD
        IF v_lat_degrees IS NULL
           AND v_long_degrees IS NULL THEN
          OPEN c_get_point_of_record(v_tsmstatn_is_nbr);
          FETCH c_get_point_of_record
            INTO v_por_rec;
          v_lat_degrees := v_por_rec.lat_degree_msr;
          v_lat_minutes := v_por_rec.lat_minute_msr;
          v_lat_dec_min := v_por_rec.gps_lat_minute_msr;
          v_lat_seconds := v_por_rec.lat_second_msr;
          v_lat_dir := v_por_rec.lat_direction;
          v_long_degrees := v_por_rec.long_degree_msr;
          v_long_minutes := v_por_rec.long_minute_msr;
          v_long_dec_min := v_por_rec.gps_long_min_msr;
          v_long_seconds := v_por_rec.long_second_msr;
          v_long_dir := v_por_rec.long_direction;
          v_lat_long_date := v_por_rec.lat_long_msr_dt;
          v_gps_scale := v_por_rec.geopstng_scale_txt;
          v_method := v_por_rec.geopstng_method_cd;
          v_datum := v_por_rec.geopstng_datum_cd;
          v_elevation_msr := v_por_rec.elevation_msr;
          v_elevation_unit := v_por_rec.elvtn_unt_cd;
          v_elevation_method := v_por_rec.elvtn_method_cd;
          v_elevation_datum := v_por_rec.elevation_datum_cd;
          v_elevation_date := v_por_rec.elvtn_msr_dt;
          CLOSE c_get_point_of_record;
        
        ELSE
          -- lat / long given - check rules
          IF v_long_degrees IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Longitude is required when Latitude is given.');
            
   
            v_status := 'N';
          END IF;
        
          IF v_lat_degrees IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Latitude is required when Longitude is given.');
            
   
            v_status := 'N';
          END IF;
        
          IF v_method IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Geopositioning Method is required when Lat/Long is given.');
            
   
            v_status := 'N';
          END IF;
        
          IF v_datum IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                         'Geopositioning Datum is required when Lat/Long is given.');
            
   
            v_status := 'N';
          END IF;
        
          -- Default Lat/Long to NW hemisphere if not included in import file
          IF v_lat_dir IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                         'Latitude direction has not been provided and will be defaulted to N');
            
   
            v_lat_dir := 'N';
          END IF;
        
          IF v_long_dir IS NULL THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_warning,'Line '|| v_line_cnt,
                                         'Longitude direction has not been provided and will be defaulted to W');
            
   
            v_long_dir := 'W';
          END IF;
        
          -- Need to check this post loop as lat direction is not required in import file
          IF v_lat_dir = 'S'
             AND v_lat_dec_deg > 0 THEN
            IF v_lat_dec_deg < 10 THEN
              v_lat_dec_deg := '-' || trunc(v_lat_dec_deg, 7);
            ELSE
              v_lat_dec_deg := '-' || trunc(v_lat_dec_deg, 6);
            END IF;
          END IF;
        
          -- Need to check this post loop as long direction is not required in import file
          IF v_long_dir = 'W'
             AND v_long_dec_deg > 0 THEN
            IF v_long_dec_deg < 100 THEN
              v_long_dec_deg := '-' || trunc(v_long_dec_deg, 6);
            ELSE
              v_long_dec_deg := '-' || trunc(v_long_dec_deg, 5);
            END IF;
          END IF;
        
        END IF;
      
        --Elevation Rules
        IF v_elevation_unit IS NULL
           AND v_elevation_msr IS NOT NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Elevation units are required when elevation is provided.');
           
   
        END IF;
        IF v_elevation_method IS NULL
           AND v_elevation_msr IS NOT NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Elevation method is required when elevation is provided.');
           
   
        END IF;
        IF v_elevation_datum IS NULL
           AND v_elevation_msr IS NOT NULL THEN
          v_status := 'N';
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Elevation datum is required when elevation is provided.');
           
   
        END IF;
      
        --Well Rules
        IF v_well_num IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Well Number is required.');
           
   
          v_status := 'N';
        END IF;
      
        IF v_well_name IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Well Name is required.');
           
   
          v_status := 'N';
        END IF;
      
        IF v_well_use IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Well Use is required.');
           
   
          v_status := 'N';
        END IF;
      
        IF v_well_status IS NULL THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Well Status is required.');
           
   
          v_status := 'N';
        END IF;
      
        --Scale rules
        -- 018 is 'Interpolation-map'
        IF v_gps_scale IS NULL
           AND v_method = rpad('018', 12) THEN
           
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                       'Scale is required when Geopositioning Method is Interpolation-Map.');
           
   
          v_status := 'N';
        END IF;
      
        /* create WELL and LOCATION records */
        BEGIN
          /* get station location sequence */
          BEGIN
            OPEN c_get_station_location_seq;
            FETCH c_get_station_location_seq
              INTO v_stl_seq;
            CLOSE c_get_station_location_seq;
          EXCEPTION
            WHEN OTHERS THEN
              
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     
               
              sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                           ' - getting station location sequence.');
               
   
          END;
        
          --SIM_STATION_LOCATIONS  
          v_stl_rec.stl_seq := v_stl_seq;
          v_stl_rec.stl_sta_seq := v_sta_seq;
          v_stl_rec.stl_ssw_seq := v_well_seq;
          v_stl_rec.stl_type_code := 'WELL HEAD';
          v_stl_rec.stl_sequence_number := 0;
          v_stl_rec.stl_lat_degrees := v_lat_degrees;
          v_stl_rec.stl_lat_minutes := v_lat_minutes;
          v_stl_rec.stl_lat_dec_minutes := v_lat_dec_min;
          v_stl_rec.stl_lat_seconds := v_lat_seconds;
          v_stl_rec.stl_lat_dir := v_lat_dir;
          v_stl_rec.stl_long_degrees := v_long_degrees;
          v_stl_rec.stl_long_minutes := v_long_minutes;
          v_stl_rec.stl_long_dec_minutes := v_long_dec_min;
          v_stl_rec.stl_long_seconds := v_long_seconds;
          v_stl_rec.stl_long_dir := v_long_dir;
          v_stl_rec.stl_lat_long_msr_date := v_lat_long_date;
          v_stl_rec.stl_latlong_accurcy := NULL;
          v_stl_rec.stl_latlong_accurcy_unit := NULL;
          v_stl_rec.stl_gps_scale := v_gps_scale;
          v_stl_rec.stl_tsmgeopa_is_number := NULL;
          v_stl_rec.stl_tsmgeopa_org_id := NULL;
          v_stl_rec.stl_tsmfhu_is_number := NULL;
          v_stl_rec.stl_tsmfhu_org_id := NULL;
          v_stl_rec.stl_tsmrrr_is_number := NULL;
          v_stl_rec.stl_tsmrrr_org_id := NULL;
          v_stl_rec.stl_geopos_method := v_method;
          v_stl_rec.stl_geopos_datum := v_datum;
          v_stl_rec.stl_elevation_msr := v_elevation_msr;
          v_stl_rec.stl_elevation_unit := v_elevation_unit;
          v_stl_rec.stl_elev_method := v_elevation_method;
          v_stl_rec.stl_elev_datum := v_elevation_datum;
          v_stl_rec.stl_elev_msr_date := v_elevation_date;
          v_stl_rec.stl_tsmnal_cd := NULL;
          v_stl_rec.stl_tsmnal_state := NULL;
          v_stl_rec.stl_nrcs := NULL;
          v_stl_rec.stl_tsmstatn0is_number := v_tsmstatn_is_nbr;
          v_stl_rec.stl_tsmstatn0org_id := rpad(p_org_id, 8);
          v_stl_rec.stl_lat_dec_degrees := v_lat_dec_deg;
          v_stl_rec.stl_long_dec_degrees := v_long_dec_deg;
        
          v_well_rec.ssw_seq := v_well_seq;
          v_well_rec.ssw_sta_seq := v_sta_seq;
          v_well_rec.ssw_well_status := v_well_status;
          v_well_rec.ssw_name := v_well_name;
          v_well_rec.ssw_well_no := v_well_num;
          v_well_rec.ssw_well_use := v_well_use;
          v_well_rec.ssw_is_number := NULL;
          v_well_rec.ssw_org_id := NULL;
          v_well_rec.ssw_develop_method := NULL;
          v_well_rec.ssw_well_const_meth := NULL;
          v_well_rec.ssw_water_use := NULL;
          v_well_rec.ssw_gradient_type := NULL;
          v_well_rec.ssw_depth_completion := NULL;
          v_well_rec.ssw_depth_completion_unit := NULL;
          v_well_rec.ssw_depth_hole := NULL;
          v_well_rec.ssw_depth_hole_unit := NULL;
          v_well_rec.ssw_well_height := NULL;
          v_well_rec.ssw_well_height_unit := NULL;
          v_well_rec.ssw_casing_height := NULL;
          v_well_rec.ssw_casing_height_unit := NULL;
          v_well_rec.ssw_disinfected := NULL;
          v_well_rec.ssw_const_start_date := NULL;
          v_well_rec.ssw_const_end_date := NULL;
          v_well_rec.ssw_init_pump_dur := NULL;
          v_well_rec.ssw_init_pump_dur_unit := NULL;
          v_well_rec.ssw_init_pump_rate := NULL;
          v_well_rec.ssw_init_pump_rate_unit := NULL;
          v_well_rec.ssw_natural_flow := NULL;
          v_well_rec.ssw_init_borehole_diam := NULL;
          v_well_rec.ssw_init_borehole_diam_unit := NULL;
          v_well_rec.ssw_tsmstatn_is_number := v_tsmstatn_is_nbr;
          v_well_rec.ssw_tsmstatn_org_id := rpad(p_org_id, 8);
          v_well_rec.ssw_status := v_status;
          v_well_rec.ssw_import_seq := v_import_log_seq;
          v_well_rec.ssw_export_seq := NULL;
        
          v_gps_scale := NULL;
        
          v_success := create_well(v_well_rec);
        
          IF v_success != 1 THEN
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                 'Failed to create well in sim_station_wells.');
             
   
          ELSE
            v_success := create_station_location(v_stl_rec);
          END IF;
        
        EXCEPTION
          WHEN OTHERS THEN
            v_import_desc := SQLERRM || ' - writing a station.';
            
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      
             
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                                 v_import_desc);
             
   
        END;
      
        /* update import lines table with station sequence */
        UPDATE sim_imp_lines
           SET sipl_ssw_seq = v_well_seq
         WHERE sipl_seq = v_text_line.sipl_seq;
      
      EXCEPTION
        WHEN e_no_station THEN
           
         sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Unable to Import Well.');
          
   
          v_status := 'N';
        WHEN OTHERS THEN
           
         sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line '|| v_line_cnt,
                                              'Unable to Import Well.');
          
   
          v_status := 'N';
      END;
     END IF;
    END LOOP; /* end of text loop */
  
    OPEN c_comp_count(v_import_log_seq);
    FETCH c_comp_count
      INTO v_complete_cnt;
    CLOSE c_comp_count;
    v_error_cnt := sim_import_pkg.get_rows_with_errors(v_import_log_seq);
    /* update counts in the import log */
    UPDATE sim_import_log
       SET sil_rows_imported = v_line_cnt,
           sil_rows_errors = v_error_cnt,
           sil_rows_ready = v_complete_cnt
     WHERE sil_seq = v_import_log_seq;
    COMMIT;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        to_char(v_line_cnt) ||
                                        ' lines processed / ' ||
                                        to_char(v_complete_cnt) ||
                                        ' wells loaded');
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Finished: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
    sim_event_log_pkg.update_stop_time;                          
    sim_import_pkg.v_strip_quote := NULL; --10/04/2005 --RWK 
  END;

EXCEPTION
  WHEN e_no_delimiter THEN
    raise_application_error(-20102, 'Error getting delimiter from import configuration.');
  
  WHEN e_bad_org THEN
    raise_application_error(-20109, 'Unable to find the specified org id in local version of STORET.');
  
  WHEN e_no_cfg THEN
    raise_application_error(-20102, 'Configuration invalid or does not exist.');
  
  WHEN e_bad_columns THEN
    raise_application_error(-20197, 'The import file contains the wrong number of columns');
  
  WHEN OTHERS THEN
    
   sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error,'Line: ' || v_line_cnt, SQLERRM ||
                                        ' - in sim_import_station_pkg.import_station.');
    
   
    
   sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        to_char(v_line_cnt) ||
                                       ' lines processed / ' ||
                                       to_char(v_complete_cnt) ||
                                       ' stations loaded');
    
   
    
   sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Finished: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
    
   
    -- Ryan J. 03/08/05 - Raise error to calling routine
    raise_application_error(-20000, substr(SQLERRM ||
                                    ' - in sim_import_station_pkg.import_station.', 1, 240));
  
END;

END sim_import_wells_pkg;
/
