CREATE OR REPLACE PACKAGE BODY sim_lib_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                           *
 ****************************************************************************************/

/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/
  v_db_os_windows sim_config.sc_value%TYPE;
  v_db_os_slash   VARCHAR2(1);

/****************************************************************************************
 *                                  PROCEDURES AND FUNCTIONS                            *
 ****************************************************************************************/

/****************************************************************************************

 ****************************************************************************************/
FUNCTION get_config_value (
  p_config_item_name IN VARCHAR2)

  RETURN VARCHAR2 IS

  v_return  sim_config.sc_value%TYPE;

BEGIN
  SELECT sc_value
    INTO v_return
    FROM sim_config
    WHERE sc_item = p_config_item_name;
  RETURN v_return;

EXCEPTION
  WHEN OTHERS THEN
    RETURN NULL;
  
END get_config_value;

/****************************************************************************************
  Get the unzip utility path for WebSIM
 ****************************************************************************************/
FUNCTION get_websim_unzip_cmd (
  p_src_zip IN VARCHAR2,
  p_dest_dir IN VARCHAR2) 
  RETURN VARCHAR2 IS

  v_cmd  VARCHAR2(300);
BEGIN
  v_cmd := trim(get_config_value(sim_lib_pkg.c_cfg_unzip_command_line));
  v_cmd := REPLACE(v_cmd,'%1',p_src_zip);
  v_cmd := REPLACE(v_cmd,'%2',p_dest_dir);
  RETURN v_cmd;
END get_websim_unzip_cmd;

/****************************************************************************************
  Get the import path for WebSIM
 ****************************************************************************************/
FUNCTION get_websim_import_path 
  RETURN VARCHAR2 IS

  v_path  VARCHAR2(300);
  v_slash VARCHAR2(1);
  
BEGIN
  v_path := trim(get_config_value(sim_lib_pkg.c_cfg_websim_import_path));
  v_slash := substr(v_path,length(v_path),1);
  IF v_slash NOT IN ('/','\') THEN
    v_path := v_path || v_db_os_slash;
  END IF;
  RETURN v_path;
END get_websim_import_path;

/****************************************************************************************
  Get the import path for binary documents/images
 ****************************************************************************************/
FUNCTION get_document_import_path 
  RETURN VARCHAR2 IS

  v_path  VARCHAR2(300);
  v_slash VARCHAR2(1);
  
BEGIN
  v_path := trim(get_config_value(sim_lib_pkg.c_cfg_document_import_path));
  v_slash := substr(v_path,length(v_path),1);
  IF v_slash NOT IN ('/','\') THEN
    v_path := v_path || v_db_os_slash;
  END IF;
  RETURN v_path;
END get_document_import_path;

/****************************************************************************************
  Get an application-level variable that's persistent throughout an application session
    (not just an Oracle session).
 ****************************************************************************************/
FUNCTION get_session_var (
  p_context IN VARCHAR2,
  p_name    IN VARCHAR2) 
  
  RETURN VARCHAR2 IS

  v_ssv_value sim_session_vars.ssv_value%TYPE;
  
BEGIN
  SELECT ssv_value
    INTO v_ssv_value
    FROM sim_session_vars
    WHERE ssv_user = USER
      AND ssv_context = p_context
      AND ssv_name = p_name;
  RETURN v_ssv_value;
  
EXCEPTION
  WHEN OTHERS THEN
    RETURN NULL;
    
END get_session_var;

/****************************************************************************************
  Set an application-level variable that's persistent throughout an application session
    (not just an Oracle session).
 ****************************************************************************************/ 
PROCEDURE set_session_var (
  p_context IN VARCHAR2,
  p_name    IN VARCHAR2,
  p_value   IN VARCHAR2) IS 
  
  PRAGMA AUTONOMOUS_TRANSACTION;
  
BEGIN
  INSERT INTO sim_session_vars (
      ssv_user, 
      ssv_context, 
      ssv_timestamp, 
      ssv_name, 
      ssv_value)
    VALUES (
      USER,
      p_context,
      SYSDATE,
      p_name,
      p_value);
  COMMIT;
  
EXCEPTION
  WHEN dup_val_on_index THEN
    UPDATE sim_session_vars
      SET ssv_value = p_value,
          ssv_timestamp = SYSDATE
      WHERE ssv_user = USER
        AND ssv_context = p_context
        AND ssv_name = p_name;
    COMMIT;
         
  WHEN OTHERS THEN
    sim_event_log_pkg.log_error(SQLERRM,'sim_lib_pkg.set_session_var(p_name=>''' || p_name || ''')');
    RAISE;
      
END set_session_var;

/****************************************************************************************
  Delete an application-level variable that's persistent throughout an application session
    (not just an Oracle session).
 ****************************************************************************************/ 
PROCEDURE delete_session_var (
  p_context IN VARCHAR2,
  p_name    IN VARCHAR2) IS 
  
  PRAGMA AUTONOMOUS_TRANSACTION;
  
BEGIN
  DELETE FROM sim_session_vars
    WHERE ssv_user = USER
      AND ssv_context = p_context
      AND ssv_name = p_name;
  COMMIT;
    
EXCEPTION 
  WHEN OTHERS THEN
    NULL;
    
END delete_session_var;

/****************************************************************************************
  Parse a string into a pl/sql table based on the delimiter passed in 
  (if you pass in a value for p_min_elements, then the table/array returned
  will always have at least that number of elements (allowing the calling 
  routine to avoid errors when referencing an element that doesn't exist)
 ****************************************************************************************/
FUNCTION parse_to_table (
  p_source_string IN VARCHAR2,
  p_delimiter     IN VARCHAR2,
  p_min_elements  IN NUMBER := 1)

  RETURN t_varchar_table AS

  v_table          t_varchar_table;
  v_index          NUMBER := 1;
  v_cur_delim_pos  NUMBER := 0;
  v_next_delim_pos NUMBER := 0;

BEGIN
  v_next_delim_pos := instr(p_source_string, p_delimiter, v_cur_delim_pos + 1);
  IF v_next_delim_pos = 0 OR v_next_delim_pos IS NULL THEN
    v_table(v_index) := TRIM(p_source_string);
    v_index := v_index + 1;
  ELSE
    WHILE v_next_delim_pos > 0
          OR v_cur_delim_pos > 0 LOOP
      IF v_next_delim_pos = 0 THEN
        v_table(v_index) := TRIM(substr(p_source_string, v_cur_delim_pos + 1));
      ELSE
        v_table(v_index) := TRIM(substr(p_source_string, v_cur_delim_pos + 1, v_next_delim_pos -
                                         v_cur_delim_pos - 1));
      END IF;
      v_cur_delim_pos := v_next_delim_pos;
      IF v_next_delim_pos > 0 THEN
        v_next_delim_pos := instr(p_source_string, p_delimiter, v_cur_delim_pos + 1);
      END IF;
      v_index := v_index + 1;
    END LOOP;
  END IF;
  -- pad v_table with additional elements to reach its min. size, if necessary
  WHILE p_min_elements >= v_index LOOP
    v_table(v_index) := NULL;
    v_index := v_index + 1;
  END LOOP;
  RETURN v_table;
END parse_to_table;

/****************************************************************************************
                                   PACKAGE MAIN BLOCK                                  
 ****************************************************************************************/
BEGIN
  v_db_os_windows := get_config_value(sim_lib_pkg.c_cfg_db_os_windows);
  IF v_db_os_windows = 'YES' THEN
    v_db_os_slash := '\';
  ELSE
    v_db_os_slash := '/';
  END IF;
END sim_lib_pkg;
/
