CREATE OR REPLACE PACKAGE sim_manage_config_pkg IS

  /*----------------------------------------------------------------------------------------
    Functionality relating to the management of import configurations
  ----------------------------------------------------------------------------------------*/

  /****************************************************************************************
  *                               PUBLIC CONSTANTS AND TYPES                               *
  ****************************************************************************************/
  
  /****************************************************************************************
  *                                    PUBLIC VARIABLES                                  *
  ****************************************************************************************/

  /****************************************************************************************
  *                              PUBLIC PROCEDURES AND FUNCTIONS                           *
  ****************************************************************************************/

  FUNCTION remove_import_config_recs
  (
    p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
    p_sqlcode  IN OUT VARCHAR2,
    p_sqlerrm  IN OUT VARCHAR2
  ) RETURN NUMBER;
  
  FUNCTION create_import_config
  (
    p_tsmorgan_org_id    IN sim_imp_cfg.sicc_tsmorgan_org_id%TYPE,
    p_tsmorgan_is_number IN sim_imp_cfg.sicc_tsmorgan_is_number%TYPE,
    p_name               IN sim_imp_cfg.sicc_name%TYPE,
    p_sevl_seq           IN sim_event_log.sevl_seq%TYPE,
    p_action_code        IN sim_imp_defs.sid_action_code%TYPE,
    p_sqlcode            IN OUT VARCHAR2,
    p_sqlerrm            IN OUT VARCHAR2
  ) RETURN NUMBER;

  FUNCTION insert_import_config_rec
  (
    p_tsmorgan_org_id    IN sim_imp_cfg.sicc_tsmorgan_org_id%TYPE,
    p_tsmorgan_is_number IN sim_imp_cfg.sicc_tsmorgan_is_number%TYPE,
    p_sid_seq            IN sim_imp_defs.sid_seq%TYPE,
    p_name               IN sim_imp_cfg.sicc_name%TYPE,
    p_desc               IN sim_imp_cfg.sicc_desc%TYPE DEFAULT NULL,
    p_delimiter          IN VARCHAR2 DEFAULT NULL,
    p_protected          IN sim_imp_cfg.sicc_protected%TYPE DEFAULT NULL,
    p_sqlcode            IN OUT VARCHAR2,
    p_sqlerrm            IN OUT VARCHAR2
  ) RETURN NUMBER;

  FUNCTION insert_import_instr_rec
  (
    p_instr    IN VARCHAR2,
    p_choice   IN VARCHAR2,
    p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
    p_sid_seq  IN sim_imp_defs.sid_seq%TYPE,
    p_sqlcode  IN OUT VARCHAR2,
    p_sqlerrm  IN OUT VARCHAR2
  ) RETURN NUMBER;

  FUNCTION insert_import_col_rec
  (
    p_name     IN VARCHAR2,
    p_include  IN VARCHAR2,
    p_generate IN VARCHAR2,
    p_default  IN VARCHAR2,
    p_format   IN VARCHAR2,
    p_choice   IN VARCHAR2,
    p_pos      IN NUMBER,
    p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
    p_sic_seq  IN sim_imp_cols.sic_seq%TYPE,
    p_sevl_seq IN sim_event_log.sevl_seq%TYPE,
    p_sid_seq  IN sim_imp_defs.sid_seq%TYPE,
    p_line_no  IN NUMBER,
    p_sqlcode  IN OUT VARCHAR2,
    p_sqlerrm  IN OUT VARCHAR2
  ) RETURN NUMBER;

  FUNCTION insert_import_trans_rec
  (
    p_org_id       IN sim_event_log.sevl_tsmorgan_org_id%TYPE DEFAULT NULL,
    p_sicdt_seq    IN NUMBER,
    p_st_user_code IN VARCHAR2,
    p_st_code      IN VARCHAR2,
    p_st_is_num    IN VARCHAR2,
    p_st_org_id    IN VARCHAR2,
    p_num_pipes    IN NUMBER,
    p_sevl_seq     IN sim_event_log.sevl_seq%TYPE,
    p_line_text    IN VARCHAR2,
    p_sqlcode      IN OUT VARCHAR2,
    p_sqlerrm      IN OUT VARCHAR2
  ) RETURN NUMBER;

  PROCEDURE insert_missing_instr
  (
    p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE
  );
  
  PROCEDURE insert_missing_import_cols
  (
    p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
    p_sevl_seq  IN sim_event_log.sevl_seq%TYPE,
    p_sqlcode  IN OUT VARCHAR2,
    p_sqlerrm  IN OUT VARCHAR2
  );

  PROCEDURE copy_import_configuration 
  (
    p_sicc_seq IN NUMBER,
    p_sevl_seq OUT NUMBER
  );
  
  PROCEDURE reorder_column_positions 
  (
  p_sicc_seq  IN NUMBER,
  p_commit_yn IN VARCHAR2 := 'Y' 
  );
  
END sim_manage_config_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim_manage_config_pkg IS

/****************************************************************************************
 *                                PRIVATE CONSTANTS AND TYPES                             *
 ****************************************************************************************/

/****************************************************************************************
 *                                     PRIVATE VARIABLES                                *
 ****************************************************************************************/

/****************************************************************************************
 *                                PROCEDURE AND FUNCTION SPECS                          *
 ****************************************************************************************/

/****************************************************************************************
  Remove an import configuration                                                       
  Return 1 if successful.  Return -1 if not successful.                                
 ****************************************************************************************/
FUNCTION remove_import_config_recs
(
  p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
  p_sqlcode  IN OUT VARCHAR2,
  p_sqlerrm  IN OUT VARCHAR2
) RETURN NUMBER IS

BEGIN

  DELETE sim_translations
   WHERE st_sicdt_seq IN (SELECT sicdt_seq
                            FROM sim_imp_cfg_dets
                           WHERE sicdt_sicc_seq = p_sicc_seq);

  DELETE sim_imp_cfg_instrs
   WHERE sici_sicc_seq = p_sicc_seq;

  DELETE sim_imp_cfg_dets
   WHERE sicdt_sicc_seq = p_sicc_seq;

  DELETE sim_imp_cfg
   WHERE sicc_seq = p_sicc_seq;

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN - 1; -- Return error code

END remove_import_config_recs;

/****************************************************************************************
  Create an import configuration from information in the sim_event_log_lines table     
  Calls insert_import_config_rec to create the record itself                           
  Return sequence value if successful.  Return -1 if not successful.                   
 ****************************************************************************************/
FUNCTION create_import_config
(
  p_tsmorgan_org_id    IN sim_imp_cfg.sicc_tsmorgan_org_id%TYPE,
  p_tsmorgan_is_number IN sim_imp_cfg.sicc_tsmorgan_is_number%TYPE,
  p_name               IN sim_imp_cfg.sicc_name%TYPE,
  p_sevl_seq           IN sim_event_log.sevl_seq%TYPE,
  p_action_code        IN sim_imp_defs.sid_action_code%TYPE,
  p_sqlcode            IN OUT VARCHAR2,
  p_sqlerrm            IN OUT VARCHAR2
) RETURN NUMBER IS

  CURSOR c_sim_event_log_lines(p_sevl_seq IN sim_event_log_lines.sell_sevl_seq%TYPE) IS
    SELECT sell_line_text
      FROM sim_event_log_lines
     WHERE sell_sevl_seq = p_sevl_seq
     ORDER BY sell_seq;

  CURSOR c_sim_imp_defs(p_sid_type IN sim_imp_defs.sid_type%TYPE,
                        p_action_cd IN sim_imp_defs.sid_action_code%TYPE) IS
    SELECT sid_seq
      FROM sim_imp_defs
     WHERE sid_type = p_sid_type
       AND sid_action_code = p_action_cd;

  v_sell_line_text sim_event_log_lines.sell_line_text%TYPE;
  v_desc           sim_imp_cfg.sicc_desc%TYPE;
  v_type           VARCHAR2(30);
  v_import_type    VARCHAR2(20);
  v_delimiter      VARCHAR2(20);
  v_sicc_seq       sim_imp_cfg.sicc_seq%TYPE;
  v_sid_seq        sim_imp_defs.sid_seq%TYPE;
  v_sici_seq       sim_imp_cfg_instrs.sici_seq%TYPE;
  v_sqlcode        VARCHAR2(100);
  v_sqlerrm        VARCHAR2(2000);
  v_line_no        INTEGER;
  v_instr          VARCHAR2(20);
  v_choice         VARCHAR2(20);
  v_name           VARCHAR2(80);
  v_sic_seq        sim_imp_cols.sic_seq%TYPE;
  v_format         VARCHAR2(80);
  v_include        VARCHAR2(1);
  v_generate       VARCHAR2(1);
  v_default        VARCHAR2(80);
  v_pos            NUMBER(4, 1);
  v_sicdt_seq      sim_imp_cfg_dets.sicdt_seq%TYPE;
  v_msg            VARCHAR2(2000);
  v_seld_seq       sim_event_log_details.seld_sevl_seq%TYPE;
  v_num_pipes      INTEGER;
  v_st_is_num      NUMBER(12);
  v_st_org_id      VARCHAR2(8);
  v_st_code        VARCHAR2(80);
  v_st_user_code   VARCHAR2(80);
  v_st_seq         sim_translations.st_seq%TYPE;  
  -- collection of parsed columns from import file
  v_col            sim_lib_pkg.t_varchar_table;
  v_action         sim_imp_defs.sid_action_code%TYPE;
  
  e_wrong_import_action  EXCEPTION;

BEGIN
  v_line_no := 0;
  OPEN c_sim_event_log_lines(p_sevl_seq);
  LOOP
    v_line_no := v_line_no + 1;
    FETCH c_sim_event_log_lines
      INTO v_sell_line_text;
    EXIT WHEN c_sim_event_log_lines%NOTFOUND;
    -- The first 8 lines are labels and can be skipped
    IF v_line_no > 8 AND v_sell_line_text IS NOT NULL THEN
      -- parse the line into an associative array (i.e. index-by table)
      v_col := sim_lib_pkg.parse_to_table(v_sell_line_text, '|', 9);
      -- determine the type of line we're on
      v_type := v_col(1);
      
      -- CONFIGURATION
      IF v_type = 'Config' THEN
        --Config|ORG_ID|ORG_IS_NUMBER|CONFIG_NAME|DESC|DELIMITER|TYPE|INVALID|ACTION
        v_desc := v_col(5);
        v_delimiter := v_col(6);
        v_import_type := v_col(7);
        v_action := v_col(9);
        IF v_action IS NULL THEN
          v_action := 'I';
        END IF;
        --If user changes import action then get the sid_seq based on user input
        IF v_action != p_action_code THEN
          v_action := p_action_code;
        END IF;
        IF v_action = 'U' AND v_import_type NOT IN ('STATIONS', 'LOCATIONS') THEN
          RAISE e_wrong_import_action;
        END IF;
        -- Get import definition seq. based on type
        OPEN c_sim_imp_defs(v_import_type, v_action);
        FETCH c_sim_imp_defs
          INTO v_sid_seq;
        CLOSE c_sim_imp_defs;
        -- Create import configuration record
        v_sicc_seq := sim_manage_config_pkg.insert_import_config_rec(
          p_tsmorgan_org_id => p_tsmorgan_org_id, 
          p_tsmorgan_is_number => p_tsmorgan_is_number, 
          p_sid_seq => v_sid_seq, 
          p_name => p_name, 
          p_desc => v_desc, 
          p_delimiter => v_delimiter, 
          p_protected => NULL, 
          p_sqlcode => v_sqlcode, 
          p_sqlerrm => v_sqlerrm);
        -- link the copy log record to the new import configuration created
        
      -- INSTRUCTION
      ELSIF v_type = 'Instr' THEN          
        -- Instr|INSTRUCTION|FORMAT          
        v_instr := v_col(2);
        v_choice := v_col(3);
        v_sici_seq := sim_manage_config_pkg.insert_import_instr_rec(
          p_instr => v_instr, 
          p_choice => v_choice, 
          p_sicc_seq => v_sicc_seq, 
          p_sid_seq => v_sid_seq, 
          p_sqlerrm => v_sqlerrm, 
          p_sqlcode => v_sqlcode);
      
      ELSIF v_type = 'Col' THEN
        -- Col|COLUMN NAME|SIC SEQ|POSITION|INCLUDE|DEFAULT|FORMAT_OPTION|FORMAT|GENERATE 
        -- Col| sic_name|sic_seq|sicdt_pos|sicdt_include|sicdt_default|sif_fmt|sifc_fmt_choice|sicdt_generate        
        v_name := v_col(2);
        v_sic_seq := v_col(3);
        v_pos := v_col(4);
        v_include := v_col(5);
        v_default := v_col(6);
        v_format := v_col(7);
        v_choice := v_col(8);
        v_generate := v_col(9);
        -- Insert Import Configuration Details record
        v_sicdt_seq := sim_manage_config_pkg.insert_import_col_rec(
          p_name => v_name, 
          p_include => v_include, 
          p_generate => v_generate, 
          p_default => v_default, 
          p_format => v_format, 
          p_choice => v_choice, 
          p_pos => v_pos, 
          p_sicc_seq => v_sicc_seq, 
          p_sic_seq => v_sic_seq, 
          p_sevl_seq => p_sevl_seq, 
          p_sid_seq => v_sid_seq,
          p_line_no => v_line_no, 
          p_sqlcode => v_sqlcode, 
          p_sqlerrm => v_sqlerrm);

      -- TRANSLATION        
      ELSIF v_type = 'Trans' THEN
        -- Trans|USER CODE|STORET CODE|ST_IS_NUMBER|ST_ORG_ID
        -- Write an error if there is no column seq for the translation record
        IF nvl(v_sicdt_seq, 0) = 0 THEN
          v_msg := 'Translation error: No column id found for translation, unable to '
            || 'create translation record for file line : ' || v_sell_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.create_import_config',v_msg);
        ELSE
          -- the number of pipes tells us what version this was created with
          -- 4 pipes is SIM 2.0.1 or newer
          v_num_pipes := length(v_sell_line_text) - length(REPLACE(v_sell_line_text,'|',''));
          v_st_user_code := v_col(2);
          v_st_code := v_col(3);
          v_st_is_num := v_col(4);
          v_st_org_id := v_col(5);
          -- Insert import translation record
          v_st_seq := sim_manage_config_pkg.insert_import_trans_rec(
            p_org_id => p_tsmorgan_org_id, 
            p_sicdt_seq => v_sicdt_seq, 
            p_st_user_code => v_st_user_code, 
            p_st_code => v_st_code, 
            p_st_is_num => v_st_is_num, 
            p_st_org_id => v_st_org_id, 
            p_num_pipes => v_num_pipes, 
            p_sevl_seq => p_sevl_seq, 
            p_line_text => v_sell_line_text, 
            p_sqlcode => v_sqlcode, 
            p_sqlerrm => v_sqlerrm);
        END IF;
      ELSE
        raise_application_error(-20000,'Unrecognized Line Type "' || v_type || '".');
      END IF;
    END IF;
  END LOOP;
  CLOSE c_sim_event_log_lines;
  COMMIT;
  
  -- Insert missing instructions not included in the import configuration file
  sim_manage_config_pkg.insert_missing_instr(v_sicc_seq);
  
  -- Insert missing import configuration columns not included in the import configuration file
  sim_manage_config_pkg.insert_missing_import_cols(
    p_sicc_seq => v_sicc_seq, 
    p_sevl_seq => p_sevl_seq, 
    p_sqlcode => v_sqlcode, 
    p_sqlerrm => v_sqlerrm);
  -- Return new primary key value
  RETURN v_sicc_seq;

EXCEPTION
  WHEN e_wrong_import_action THEN
    p_sqlerrm := 'An Import Action of "Update" is only allowed on Station or Location Import Configurations.';
    p_sqlcode := -20000;
    RETURN -1; -- Return error code
    
  WHEN OTHERS THEN
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN -1; -- Return error code

END create_import_config;

/****************************************************************************************
  Create an import configuration record                                                
  Return sequence value if successful.  Return -1 if not successful.                   
 ****************************************************************************************/
FUNCTION insert_import_config_rec
(
  p_tsmorgan_org_id    IN sim_imp_cfg.sicc_tsmorgan_org_id%TYPE,
  p_tsmorgan_is_number IN sim_imp_cfg.sicc_tsmorgan_is_number%TYPE,
  p_sid_seq            IN sim_imp_defs.sid_seq%TYPE,
  p_name               IN sim_imp_cfg.sicc_name%TYPE,
  p_desc               IN sim_imp_cfg.sicc_desc%TYPE DEFAULT NULL,
  p_delimiter          IN VARCHAR2 DEFAULT NULL,
  p_protected          IN sim_imp_cfg.sicc_protected%TYPE DEFAULT NULL,
  p_sqlcode            IN OUT VARCHAR2,
  p_sqlerrm            IN OUT VARCHAR2
) RETURN NUMBER IS

  v_sicc_delimiter sim_imp_cfg.sicc_delimiter%TYPE;
  v_sicc_seq       NUMBER;

BEGIN

  -- Set delimiter
  IF p_delimiter IS NULL THEN
    v_sicc_delimiter := NULL;
  ELSIF lower(p_delimiter) IN ('pipe', '|') THEN
    v_sicc_delimiter := '|';
  ELSIF lower(p_delimiter) IN ('tilda', '~') THEN
    v_sicc_delimiter := '~';
  ELSIF lower(p_delimiter) IN ('comma', ',') THEN
    v_sicc_delimiter := ',';
  ELSIF lower(p_delimiter) IN ('tab', chr(9)) THEN
    v_sicc_delimiter := chr(9);
  END IF;

  -- Get sequence value for primary key
  SELECT sim_imp_cfg_seq.NEXTVAL
    INTO v_sicc_seq
    FROM dual;

  -- Create import configuration record
  INSERT INTO sim_imp_cfg
    (sicc_seq,
     sicc_sid_seq,
     sicc_name,
     sicc_desc,
     sicc_delimiter,
     sicc_protected,
     sicc_tsmorgan_org_id,
     sicc_tsmorgan_is_number)
  VALUES
    (v_sicc_seq,
     p_sid_seq,
     p_name,
     p_desc,
     v_sicc_delimiter,
     p_protected,
     p_tsmorgan_org_id,
     p_tsmorgan_is_number)
  RETURNING sicc_seq INTO v_sicc_seq;

  COMMIT;

  -- Return new primary key value
  RETURN v_sicc_seq;

EXCEPTION
  WHEN OTHERS THEN
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN - 1; -- Return error code

END insert_import_config_rec;

/****************************************************************************************
  Create an import instruction                                                         
  Return sequence value if successful.  Return -1 if not successful.                   
 ****************************************************************************************/
FUNCTION insert_import_instr_rec
(
  p_instr    IN VARCHAR2,
  p_choice   IN VARCHAR2,
  p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
  p_sid_seq  IN sim_imp_defs.sid_seq%TYPE,
  p_sqlcode  IN OUT VARCHAR2,
  p_sqlerrm  IN OUT VARCHAR2
) RETURN NUMBER IS

  CURSOR c_sim_imp_instrs(p_sii_code IN sim_imp_instrs.sii_code%TYPE, p_sid_seq IN sim_imp_defs.sid_seq%TYPE) IS
    SELECT sii_seq
      FROM sim_imp_instrs
     WHERE sii_code = p_sii_code
       AND sii_sid_seq = p_sid_seq;

  CURSOR c_sim_imp_instr_choices(p_siic_code IN sim_imp_instr_choices.siic_code%TYPE, p_siic_sii_seq IN sim_imp_instr_choices.siic_sii_seq%TYPE) IS
    SELECT siic_seq
      FROM sim_imp_instr_choices
     WHERE siic_code = p_siic_code
       AND siic_sii_seq = p_siic_sii_seq;

  v_sii_seq  sim_imp_instrs.sii_seq%TYPE;
  v_siic_seq sim_imp_instr_choices.siic_seq%TYPE;
  v_sici_seq sim_imp_cfg_instrs.sici_seq%TYPE;

BEGIN

  IF substr(p_instr, 1, 1) IN ('0', '1', '2', '3', '4', '5', '6', '7', '8', '9') THEN
    BEGIN
      SELECT dest.sii_seq
      INTO v_sii_seq
      FROM sim_imp_instrs source,
           sim_imp_instrs dest
      WHERE source.sii_code = dest.sii_code
        AND source.sii_seq = p_instr
        AND dest.sii_sid_seq = p_sid_seq;
    EXCEPTION
      WHEN no_data_found THEN
        p_sqlcode := -20000;
        p_sqlerrm := 'Instruction line in file is invalid (' || p_instr || ' | ' || p_choice;
        RETURN -1;
    END;
    BEGIN
      SELECT dest.siic_seq
      INTO v_siic_seq
      FROM sim_imp_instr_choices source,
           sim_imp_instr_choices dest
     WHERE source.siic_code = dest.siic_code
       AND source.siic_seq = p_choice
       AND dest.siic_sii_seq = v_sii_seq;
    EXCEPTION
      WHEN no_data_found THEN
        p_sqlcode := -20000;
        p_sqlerrm := 'Instruction line in file is invalid (' || p_instr || ' | ' || p_choice;
        RETURN -1;
    END;
  ELSE  
    BEGIN
      SELECT sii_seq
      INTO v_sii_seq
      FROM sim_imp_instrs
      WHERE sii_code = p_choice
        AND sii_sid_seq = p_sid_seq;
    EXCEPTION
      WHEN no_data_found THEN
        p_sqlcode := -20000;
        p_sqlerrm := 'Instruction line in file is invalid (' || p_instr || ' | ' || p_choice;
        RETURN -1;
    END;
    BEGIN
      SELECT siic_seq
      INTO v_siic_seq
      FROM sim_imp_instr_choices
     WHERE siic_code = p_choice
       AND siic_sii_seq = v_sii_seq;
    EXCEPTION 
      WHEN no_data_found THEN
        p_sqlcode := -20000;
        p_sqlerrm := 'Instruction line in file is invalid (' || p_instr || ' | ' || p_choice;
        RETURN -1;
    END;
  END IF;

  -- Create import instruction record
  INSERT INTO sim_imp_cfg_instrs
    (sici_seq,
     sici_sicc_seq,
     sici_sii_seq,
     sici_siic_seq)
  VALUES
    (sim_imp_cfg_instrs_seq.NEXTVAL,
     p_sicc_seq,
     v_sii_seq,
     v_siic_seq)
  RETURNING sici_seq INTO v_sici_seq;

  COMMIT;

  -- Return new primary key value
  RETURN v_sii_seq;

EXCEPTION
  WHEN OTHERS THEN
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN - 1; -- Return error code

END insert_import_instr_rec;

/****************************************************************************************
  Insert an import column record                                                       
  Return sequence value if successful.  Return -1 if not successful.                   
 ****************************************************************************************/
FUNCTION insert_import_col_rec
(
  p_name     IN VARCHAR2,
  p_include  IN VARCHAR2,
  p_generate IN VARCHAR2,
  p_default  IN VARCHAR2,
  p_format   IN VARCHAR2,
  p_choice   IN VARCHAR2,
  p_pos      IN NUMBER,
  p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
  p_sic_seq  IN sim_imp_cols.sic_seq%TYPE,
  p_sevl_seq IN sim_event_log.sevl_seq%TYPE,
  p_sid_seq  IN sim_imp_defs.sid_seq%TYPE,
  p_line_no  IN NUMBER,
  p_sqlcode  IN OUT VARCHAR2,
  p_sqlerrm  IN OUT VARCHAR2
) RETURN NUMBER IS

  CURSOR c_col_check IS
    SELECT a.sic_seq, a.sic_dflt_pos
      FROM sim_imp_cols a
     WHERE a.sic_name = p_name
       AND a.sic_sid_seq = p_sid_seq;
       

  CURSOR c_fmt(p_sifc_fmt_choice IN sim_imp_fmt_choices.sifc_fmt_choice%TYPE, p_sif_fmt IN sim_imp_fmts.sif_fmt%TYPE) IS
    SELECT sifc_seq
      FROM sim_imp_fmt_choices,
           sim_imp_fmts
     WHERE sifc_fmt_choice = p_sifc_fmt_choice
       AND sifc_sif_seq = sif_seq
       AND sif_fmt = p_sif_fmt;

  CURSOR c_sim_imp_cols(p_sic_seq IN sim_imp_cols.sic_seq%TYPE) IS
    SELECT sic_req_in_import
      FROM sim_imp_cols
     WHERE sic_seq = p_sic_seq;

  v_check_sic_seq  sim_imp_cols.sic_seq%TYPE;
  v_check_dflt_pos sim_imp_cols.sic_dflt_pos%TYPE;
  v_msg            VARCHAR2(2000);
  v_sqlcode        VARCHAR2(100);
  v_sqlerrm        VARCHAR2(2000);
  v_seld_seq       sim_event_log_details.seld_sevl_seq%TYPE;
  e_except EXCEPTION;
  v_sifc_seq          sim_imp_fmt_choices.sifc_seq%TYPE;
  v_sic_req_in_import sim_imp_cols.sic_req_in_import%TYPE;
  v_generate          sim_imp_cfg_dets.sicdt_generate%TYPE;
  v_sicdt_seq         sim_imp_cfg_dets.sicdt_seq%TYPE;
  v_choice          VARCHAR2(20) := p_choice;

BEGIN

  v_check_sic_seq := NULL;
  OPEN c_col_check;
  FETCH c_col_check
    INTO v_check_sic_seq, v_check_dflt_pos;
  CLOSE c_col_check;

  IF v_check_sic_seq IS NULL THEN
    v_msg := 'Column Error: Unable to locate a column record for column name "' ||
             p_name;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'Line: '||p_line_no, v_msg);
    RAISE e_except;
  END IF;

  -- Check to see if the column name and dflt position match.  If not, there has been a change.
  -- Check if column has become disabled (has a default position of -99)
  IF v_check_dflt_pos = -99
     AND (v_check_dflt_pos <> p_default OR p_default IS NULL) THEN
    v_msg := 'Column Error: Column "' || p_name || '" has been disabled.';
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'Line: '||p_line_no, v_msg);
    RAISE e_except;
  END IF;

  -- Allow latitude/longitude with lesser percision to be increased to 7 decimal places
  IF v_choice LIKE '-DDD.D%' THEN
    v_choice := '-DDD.DDDDDDD';
  END IF;
  IF v_choice LIKE 'DDD.D%' THEN
    v_choice := 'DDD.DDDDDDD';
  END IF;
  IF v_choice LIKE '-DD.D%' THEN
    v_choice := '-DD.DDDDDDD';
  END IF;
  IF v_choice LIKE 'DD.D%' THEN
    v_choice := 'DD.DDDDDDD';
  END IF;
  
  OPEN c_fmt(v_choice, p_format);
  FETCH c_fmt
    INTO v_sifc_seq;
  CLOSE c_fmt;
  IF v_choice IS NOT NULL
     AND v_sifc_seq IS NULL THEN
    v_msg := 'Column Error: Unable to locate the format choice "' || v_choice ||
             '" for format "' || p_format || '" for column name.';
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'Line: '||p_line_no, v_msg);
  END IF;

  OPEN c_sim_imp_cols(v_check_sic_seq);
  FETCH c_sim_imp_cols
    INTO v_sic_req_in_import;
  CLOSE c_sim_imp_cols;

  v_generate := p_generate;
  IF v_sic_req_in_import = 1
     AND nvl(p_include, 0) != 1 THEN
    v_generate := 1;
  END IF;

  INSERT INTO sim_imp_cfg_dets
    (sicdt_seq,
     sicdt_sic_seq,
     sicdt_sicc_seq,
     sicdt_included,
     sicdt_generate,
     sicdt_default,
     sicdt_pos,
     sicdt_sifc_seq)
  VALUES
    (sim_imp_cfg_dets_seq.NEXTVAL,
     v_check_sic_seq,
     p_sicc_seq,
     p_include,
     v_generate,
     p_default,
     p_pos,
     v_sifc_seq)
  RETURNING sicdt_seq INTO v_sicdt_seq;

  COMMIT;

  -- Return new primary key value
  RETURN v_sicdt_seq;

EXCEPTION
  WHEN OTHERS THEN
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN - 1; -- Return error code

END insert_import_col_rec;

/****************************************************************************************
  Insert an import translation record                                                  
  Return sequence value if successful.  Return -1 if not successful.                   
 ****************************************************************************************/
FUNCTION insert_import_trans_rec
(
  p_org_id       IN sim_event_log.sevl_tsmorgan_org_id%TYPE DEFAULT NULL,
  p_sicdt_seq    IN NUMBER,
  p_st_user_code IN VARCHAR2,
  p_st_code      IN VARCHAR2,
  p_st_is_num    IN VARCHAR2,
  p_st_org_id    IN VARCHAR2,
  p_num_pipes    IN NUMBER,
  p_sevl_seq     IN sim_event_log.sevl_seq%TYPE,
  p_line_text    IN VARCHAR2,
  p_sqlcode      IN OUT VARCHAR2,
  p_sqlerrm      IN OUT VARCHAR2
) RETURN NUMBER IS

  v_storet_pk sim_interface_extras_pkg.t_storet_pk_type;
  v_sic_seq   sim_imp_cols.sic_seq%TYPE;
  v_col_name  sim_imp_cols.sic_name%TYPE;
  v_fmt_op    sim_imp_cols.sic_format_option%TYPE;
  v_code      VARCHAR2(200);
  v_msg       VARCHAR2(2000);
  v_sqlcode   VARCHAR2(100);
  v_sqlerrm   VARCHAR2(2000);
  v_seld_seq  sim_event_log_details.seld_sevl_seq%TYPE;
  v_st_code   VARCHAR2(80) := p_st_code;
  v_st_is_num NUMBER(12) := p_st_is_num;
  v_org_id    VARCHAR2(8) := rpad(p_org_id,8);
  v_st_org_id VARCHAR2(8) := rpad(p_st_org_id,8);
  v_allow_val VARCHAR2(80);
  v_st_seq    sim_translations.st_seq%TYPE;

  CURSOR c_fmt_op IS
    SELECT sic_seq,
           sic_name,
           sic_format_option
      FROM sim_imp_cols,
           sim_imp_cfg_dets
     WHERE sicdt_seq = p_sicdt_seq
       AND sicdt_sic_seq = sic_seq;

  CURSOR c_allow_values IS
    SELECT sia_item
      FROM sim_imp_allow_values
     WHERE sia_sic_seq = v_sic_seq
       AND sia_item = p_st_code;

  CURSOR c_lab_sample_prep_procedure(p_code VARCHAR2, p_org_id NUMBER) IS
    SELECT preparation_id code_value
      FROM tsrlspp
     WHERE tsrlspp_org_id = p_org_id
        OR tsrlspp_org_id = 'ZZZZZZZZ'
       AND rtrim(preparation_id) = rtrim(p_code);

  CURSOR c_lab_sa_prep_proc_source(p_code VARCHAR2, p_org_id NUMBER) IS
    SELECT DISTINCT source_acr code_value
      FROM tsrlspp
     WHERE tsrlspp_org_id = p_org_id
        OR tsrlspp_org_id = 'ZZZZZZZZ'
       AND source_acr = p_code;

  CURSOR c_field_lab_procedure(p_code VARCHAR2, p_org_id VARCHAR2) IS
    SELECT procedure_id code_value
      FROM tsranlpr
     WHERE tsranlpr_org_id = p_org_id
        OR tsranlpr_org_id = 'ZZZZZZZZ'
       AND procedure_id = p_code;

  CURSOR c_field_lab_procedure_source(p_code VARCHAR2, p_org_id VARCHAR2) IS
    SELECT DISTINCT source_acr code_value
      FROM tsranlpr
     WHERE tsranlpr_org_id = p_org_id
        OR tsranlpr_org_id = 'ZZZZZZZZ'
       AND source_acr = p_code;

  CURSOR c_char_row_id(p_code VARCHAR2) IS
    SELECT display_name
      FROM tsrchar
     WHERE search_name = p_code
       AND d_scr_type_cd != 'TAXA';

  CURSOR c_result_value(p_code VARCHAR2) IS
    SELECT '*Non-detect'
      FROM dual
    UNION
    SELECT '*Present <QL'
      FROM dual
    UNION
    SELECT '*Present >QL'
      FROM dual
    UNION
    SELECT '*Present'
      FROM dual
    UNION
    SELECT '*Not Reported'
      FROM dual
    UNION
    SELECT cvp.short_name
      FROM tsrcpv cvp,
           tsrchar cha
     WHERE cvp.tsrchar_is_number = cha.tsrchar_is_number
       AND cvp.tsrchar_org_id = cha.tsrchar_org_id
       AND cvp.short_name = p_code;

BEGIN

  /* must look up the col format option to determine how to validate
  * the translation storet code.  The format options are: 
  * 1 - freetext, 2 - Defined Format, 3 - Storet Table, 4 - Allowable values
  */
  OPEN c_fmt_op;
  FETCH c_fmt_op
    INTO v_sic_seq, v_col_name, v_fmt_op;
  CLOSE c_fmt_op;

  /* This code is for loading sim version 2.0.1 configurations into sim version 2.0.1*/
  IF p_num_pipes = 4 THEN
    IF v_fmt_op = 3 THEN
      IF p_st_user_code IS NOT NULL THEN
        IF p_st_code IS NOT NULL THEN
          -- check to see if STORET code is valid
          v_code := sim_interface_extras_pkg.check_trans_code(
            v_sic_seq, 
            p_st_code, 
            p_org_id);
          IF v_code IS NULL THEN
            v_msg := 'Translation Error: Unable to translate "' || p_st_code 
              || '" to a valid value for "' || v_col_name || '" column under the "' 
              || p_org_id || '" organization for file line: ' || p_line_text;
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
            v_st_code := NULL;
          END IF;
        END IF;
        IF v_st_is_num IS NOT NULL THEN
          -- check to see if STORET IS Number is valid
          v_code := sim_interface_extras_pkg.return_trans(
            v_sic_seq, 
            v_st_is_num, 
            v_st_org_id, 
            v_org_id);
          IF v_code IS NULL THEN
            v_msg := 'Translation Error: Unable to translate "' || v_st_code 
              || '" to a valid value for "' || v_col_name || '" column under the "' 
              || v_org_id || '" organization for file line: ' || p_line_text;
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
            v_st_is_num := NULL;
            v_st_org_id := NULL;
          END IF;
        END IF;
      END IF;
    END IF;
  
    IF p_st_user_code IS NOT NULL THEN
      BEGIN
        INSERT INTO sim_translations
          (st_seq,
           st_sicdt_seq,
           st_user_code,
           st_storet_code,
           st_is_number,
           st_org_id)
        VALUES
          (sim_translations_seq.NEXTVAL,
           p_sicdt_seq,
           p_st_user_code,
           v_st_code,
           v_st_is_num,
           v_st_org_id)
        RETURNING st_seq INTO v_st_seq;        
        COMMIT;
      
      EXCEPTION
        WHEN OTHERS THEN
          v_msg := 'Translation Error: Unable to create translation record in INSERT_IMPORT_TRANS_REC program unit for file line: ' 
            || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg||': '||SQLERRM);
      END;
    END IF;
  
  ELSE
    /* This is for translating old sim translations into the new sim translations format
    ie using the is number and org id for STORET table lookups*/    
    IF v_fmt_op IN ('1', '2')
        AND v_col_name <> 'Result Value' THEN          
      IF p_st_user_code IS NOT NULL THEN
        BEGIN
          INSERT INTO sim_translations
            (st_seq,
             st_sicdt_seq,
             st_user_code,
             st_storet_code,
             st_is_number,
             st_org_id)
          VALUES
            (sim_translations_seq.NEXTVAL,
             p_sicdt_seq,
             p_st_user_code,
             v_st_code,
             v_storet_pk.is_number,
             v_storet_pk.org_id)
          RETURNING st_seq INTO v_st_seq;          
          COMMIT;
        
        EXCEPTION
          WHEN OTHERS THEN
            v_msg := 'Translation Error: Unable to create translation record in INSERT_IMPORT_TRANS_REC program unit for file line: ' 
              || p_line_text;
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg||': '||SQLERRM);
        END;
      END IF;
    
    ELSIF v_fmt_op = 4 THEN
      /*allowable values */      
      OPEN c_allow_values;
      FETCH c_allow_values
        INTO v_allow_val;
      CLOSE c_allow_values;
    
      IF v_allow_val IS NULL THEN
        v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                 '" to a valid value for "' || v_col_name || '" column under the "' ||
                 v_org_id || '" organization for file line: ' || p_line_text;
        sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
        v_st_code := NULL;
      END IF;
    
      IF p_st_user_code IS NOT NULL THEN
        BEGIN
          INSERT INTO sim_translations
            (st_seq,
             st_sicdt_seq,
             st_user_code,
             st_storet_code,
             st_is_number,
             st_org_id)
          VALUES
            (sim_translations_seq.NEXTVAL,
             p_sicdt_seq,
             p_st_user_code,
             v_st_code,
             v_storet_pk.is_number,
             v_storet_pk.org_id)
          RETURNING st_seq INTO v_st_seq;
        
          COMMIT;
        
        EXCEPTION
          WHEN OTHERS THEN
            v_msg := 'Translation Error: Unable to create translation record in INSERT_IMPORT_TRANS_REC program unit for file line: ' ||
                     p_line_text;
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg||': '||SQLERRM);
        END;
      END IF;
    
    ELSIF v_fmt_op = 3
          OR v_col_name IN ('Result Value') THEN
      /* These first columns are translated without using the is_number and org_id */
    
      IF v_col_name = 'Lab Sample Prep Procedure' THEN
        OPEN c_lab_sample_prep_procedure(v_st_code, v_org_id);
        FETCH c_lab_sample_prep_procedure
          INTO v_code;
        CLOSE c_lab_sample_prep_procedure;
        IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSIF v_col_name = 'Lab Sample Prep Procedure Source' THEN
        OPEN c_lab_sa_prep_proc_source(v_st_code, v_org_id);
        FETCH c_lab_sa_prep_proc_source
          INTO v_code;
        CLOSE c_lab_sa_prep_proc_source;
        IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSIF v_col_name = 'Field/Lab Procedure' THEN
        OPEN c_field_lab_procedure(v_st_code, v_org_id);
        FETCH c_field_lab_procedure
          INTO v_code;
        CLOSE c_field_lab_procedure;
        IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSIF v_col_name = 'Field/Lab Procedure Source' THEN
        OPEN c_field_lab_procedure_source(v_st_code, v_org_id);
        FETCH c_field_lab_procedure_source
          INTO v_code;
        CLOSE c_field_lab_procedure_source;
         IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSIF v_col_name = 'Characteristic Row ID' THEN
        OPEN c_char_row_id(v_st_code);
        FETCH c_char_row_id
          INTO v_code;
        CLOSE c_char_row_id;
        IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSIF v_col_name = 'Result Value' THEN
        OPEN c_result_value(v_st_code);
        FETCH c_result_value
          INTO v_code;
        CLOSE c_result_value;
        IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSIF v_col_name IN ('Sequence Number', 'Secondary Type', 'State', 'County',
             'Primary Estuary', 'Secondary Estuary',
             'Native American Land Name', 'Native American Land State') THEN
        v_code := sim_interface_extras_pkg.check_trans_code(v_sic_seq, v_st_code, v_org_id);
        IF v_code IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        END IF;
      
      ELSE
        /* This code is to convert the storet codes into the storet is number and org id */
        v_storet_pk := sim_interface_extras_pkg.return_trans_pk(v_sic_seq, rtrim(v_st_code), rtrim(v_org_id));
        IF v_storet_pk.is_number IS NULL THEN
          v_msg := 'Translation Error: Unable to translate "' || v_st_code ||
                   '" to a valid value for "' || v_col_name || '" column under the "' ||
                   v_org_id || '" organization for file line: ' || p_line_text;
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg);
          v_st_code := NULL;
        ELSE
          v_st_code := NULL;
        END IF;
      END IF;
    
      IF p_st_user_code IS NOT NULL THEN
        BEGIN
          INSERT INTO sim_translations
            (st_seq,
             st_sicdt_seq,
             st_user_code,
             st_storet_code,
             st_is_number,
             st_org_id)
          VALUES
            (sim_translations_seq.NEXTVAL,
             p_sicdt_seq,
             p_st_user_code,
             v_st_code,
             v_storet_pk.is_number,
             v_storet_pk.org_id)
          RETURNING st_seq INTO v_st_seq;
        
          COMMIT;
        
        EXCEPTION
          WHEN OTHERS THEN
            v_msg := 'Translation Error: Unable to create translation record in INSERT_IMPORT_TRANS_REC program unit for file line: ' ||
                     p_line_text;
            sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg||': '||SQLERRM);
        END;
      END IF;
    END IF;
  END IF;

  RETURN v_st_seq;

EXCEPTION
  WHEN OTHERS THEN
    v_msg := 'Translation Process Error: in INSERT_IMPORT_TRANS_REC program unit for file line: ' ||
             p_line_text;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_import_trans_rec', v_msg||': '||SQLERRM);
    p_sqlcode := SQLCODE;
    p_sqlerrm := SQLERRM;
    RETURN - 1; -- Return error code

END insert_import_trans_rec;

/****************************************************************************************
  Insert any instructions that were not included in the import configuration file
 ****************************************************************************************/
PROCEDURE insert_missing_instr(p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE) IS

  v_sii_seq      NUMBER(10);
  v_siic_seq     NUMBER(10);
  v_sid_seq      NUMBER(10);
  v_instr_exists NUMBER(1) := NULL;
  
  CURSOR c_sii_seq (p_sicc_seq IN NUMBER) IS
    SELECT sii_seq
      FROM sim_imp_instrs sii
     WHERE sii.sii_sid_seq = p_sicc_seq;
     
  CURSOR c_siic_seq (p_sii_seq IN NUMBER) IS
    SELECT siic_seq
      FROM sim_imp_instr_choices
     WHERE siic_sii_seq = p_sii_seq;
     
  CURSOR c_instr_exists (p_sicc_seq IN NUMBER, p_sii_seq IN NUMBER) IS
    SELECT 1
      FROM sim_imp_cfg_instrs
     WHERE sici_sicc_seq = p_sicc_seq
       AND sici_sii_seq = p_sii_seq;
BEGIN
  SELECT sicc_sid_seq
    INTO v_sid_seq
    FROM sim_imp_cfg
   WHERE sicc_seq = p_sicc_seq;
  OPEN c_sii_seq(v_sid_seq);
  LOOP
    FETCH c_sii_seq
      INTO v_sii_seq;
    EXIT WHEN c_sii_seq%NOTFOUND;
    OPEN c_instr_exists (p_sicc_seq, v_sii_seq);
    FETCH c_instr_exists
      INTO v_instr_exists;
    IF c_instr_exists%NOTFOUND THEN
      SELECT siic_seq
        INTO v_siic_seq
        FROM sim_imp_instr_choices
       WHERE siic_sii_seq = v_sii_seq
         AND siic_dflt_flag = 1;      
      INSERT INTO sim_imp_cfg_instrs
        (sici_seq,
         sici_sicc_seq,
         sici_sii_seq,
         sici_siic_seq)
      VALUES
        (sim_imp_cfg_instrs_seq.NEXTVAL,
         p_sicc_seq,
         v_sii_seq,
         v_siic_seq);
    END IF;
    CLOSE c_instr_exists;
  END LOOP;
  CLOSE c_sii_seq;
  COMMIT;
END insert_missing_instr;

/****************************************************************************************
  Insert any import columns that were not included in the import configuration file    
  Return sequence value if successful.  Return -1 if not successful.                   
 ****************************************************************************************/
PROCEDURE insert_missing_import_cols
(
  p_sicc_seq IN sim_imp_cfg.sicc_seq%TYPE,
  p_sevl_seq  IN sim_event_log.sevl_seq%TYPE,
  p_sqlcode  IN OUT VARCHAR2,
  p_sqlerrm  IN OUT VARCHAR2
) IS

  v_sid_seq  NUMBER(10);
  v_sic_seq  NUMBER(10);
  v_sicc_col NUMBER(10);
  v_msg      VARCHAR2(2000);
  v_name     VARCHAR2(80);
  v_dflt_pos NUMBER(10);

  CURSOR c_sid_seq IS
    SELECT sicc_sid_seq
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

  CURSOR c_cols IS
    SELECT sic_seq,
           sic_name,
           sic_dflt_pos
      FROM sim_imp_cols
     WHERE sic_sid_seq = v_sid_seq
       AND sic_dflt_pos >= 0;

  CURSOR c_sicc_col IS
    SELECT sicdt_seq
      FROM sim_imp_cfg_dets
     WHERE sicdt_sicc_seq = p_sicc_seq
       AND sicdt_sic_seq = v_sic_seq;

  v_seld_seq sim_event_log_details.seld_seq%TYPE;
  v_sqlcode  VARCHAR2(100);
  v_sqlerrm  VARCHAR2(2000);

BEGIN
  /* select all columns for this import config type - 
  * if the column wasn't created from the file lines
  * then create a column for the config
  */
  /* get the config type so can select the config type columns */
  OPEN c_sid_seq;
  FETCH c_sid_seq
    INTO v_sid_seq;
  CLOSE c_sid_seq;

  /* loop through all columns for this config type.  for each column
  * see if the column already exists for this config - if not then create one 
  */
  OPEN c_cols;
  LOOP
    FETCH c_cols
      INTO v_sic_seq, v_name, v_dflt_pos;
    EXIT WHEN c_cols%NOTFOUND;
  
    v_sicc_col := NULL;
    OPEN c_sicc_col;
    FETCH c_sicc_col
      INTO v_sicc_col;
    CLOSE c_sicc_col;
  
    IF v_sicc_col IS NULL THEN
      BEGIN
        INSERT INTO sim_imp_cfg_dets
          (sicdt_seq,
           sicdt_sic_seq,
           sicdt_sicc_seq,
           sicdt_pos)
        VALUES
          (sim_imp_cfg_dets_seq.NEXTVAL,
           v_sic_seq,
           p_sicc_seq,
           v_dflt_pos);
        COMMIT;
      EXCEPTION
        WHEN OTHERS THEN
          v_msg := 'Column Error: Unable to create column record for column name "' ||
                   v_name || '" in INSERT_MISSING_IMPORT_COLS program unit.';
          sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_missing_import_cols', v_msg||': '||SQLERRM);
      END;
    END IF;
  END LOOP;
  CLOSE c_cols;

EXCEPTION
  WHEN OTHERS THEN
    v_msg := 'Instruction Process Error: in CONFIG_COL_CHECK program unit for import config # ' ||
             p_sicc_seq;
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, 'sim_manage_config_pkg.insert_missing_import_cols', v_msg||': '||SQLERRM);
  
END insert_missing_import_cols;

/****************************************************************************************
  Copy an import configuration file into sim_event_log_lines table.
 ****************************************************************************************/ 
PROCEDURE copy_import_configuration (
  p_sicc_seq IN NUMBER,
  p_sevl_seq OUT NUMBER) IS

  v_line          sim_event_log_lines.sell_line_text%TYPE;

  e_except        EXCEPTION;
  e_file          EXCEPTION;

  v_user          VARCHAR2(50);
  v_cfg_rec       sim_imp_cfg%ROWTYPE;
  v_type          VARCHAR2(30);
  v_line_cnt      sim_event_log_lines.sell_line_no%TYPE := 0;
  v_action_code   VARCHAR2(1);

  v_instr_seq     VARCHAR2(12);
  v_choice_seq    VARCHAR2(12);

  v_col_seq       NUMBER(10);
  v_col_name      VARCHAR2(80);
  v_sic_seq       NUMBER(10);
  v_pos           NUMBER(3);
  v_included      VARCHAR2(1);
  v_generate      VARCHAR2(1);
  v_default       VARCHAR2(80);
  v_fmt           VARCHAR2(80);
  v_fmt_choice    VARCHAR2(80);
  v_delimiter     VARCHAR2(20);

  v_user_code     VARCHAR2(80);
  v_storet_code   VARCHAR2(80);
  v_is_num        NUMBER(12);
  v_st_is_number  NUMBER(12);
  v_st_org_id     VARCHAR2(8);

  v_sqlcode       VARCHAR2(15);
  v_sqlerrm       VARCHAR2(100);
  v_err_nbr       NUMBER(12);

  CURSOR c_user IS
    SELECT USER
      FROM dual;

  CURSOR c_config IS
    SELECT sicc_seq,
           sicc_sid_seq,
           sicc_name,
           sicc_desc,
           sicc_delimiter,
           sicc_tsmorgan_org_id,
           sicc_tsmorgan_is_number,
           sicc_updated_on,
           sicc_updated_by,
           sicc_protected
      FROM sim_imp_cfg
     WHERE sicc_seq = p_sicc_seq;

  CURSOR c_type IS
    SELECT sid_type,
           sid_action_code
      FROM sim_imp_defs
     WHERE sid_seq = v_cfg_rec.sicc_sid_seq;

  CURSOR c_instr IS
    SELECT siic_sii_seq,
           siic_seq
      FROM sim_imp_cfg_instrs,
           sim_imp_instrs,
           sim_imp_instr_choices
     WHERE sici_sicc_seq = v_cfg_rec.sicc_seq
       AND sici_sii_seq = sii_seq
       AND sici_siic_seq = siic_seq
       AND sii_sid_seq = v_cfg_rec.sicc_sid_seq;

  CURSOR c_columns IS
    SELECT sicdt_seq,
           sic_name,
           sic_seq,
           sicdt_pos,
           sicdt_included,
           sicdt_generate,
           sicdt_default,
           sif_fmt,
           sifc_fmt_choice
      FROM sim_imp_cfg_dets,
           sim_imp_cols,
           sim_imp_fmt_choices,
           sim_imp_fmts
     WHERE sicdt_sicc_seq = v_cfg_rec.sicc_seq
       AND sicdt_sic_seq = sic_seq
       AND sic_sif_seq = sif_seq(+)
       AND sicdt_sifc_seq = sifc_seq(+);

  CURSOR c_trans IS
    SELECT st_user_code,
           st_storet_code,
           st_char_is_number,
           st_is_number,
           st_org_id
      FROM sim_translations
     WHERE st_sicdt_seq = v_col_seq;

BEGIN
  OPEN c_user;
  FETCH c_user
    INTO v_user;
  CLOSE c_user;

  OPEN c_config;
  FETCH c_config
    INTO v_cfg_rec;
  CLOSE c_config;

  OPEN c_type;
  FETCH c_type
    INTO v_type, v_action_code;
  CLOSE c_type;

  BEGIN
    sim_event_log_pkg.write_event(7, v_cfg_rec.sicc_seq, v_cfg_rec.sicc_tsmorgan_org_id, v_cfg_rec.sicc_sid_seq, NULL);
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
                                        'Start: ' ||
                                        to_char(SYSDATE, 'HH24:MI:SS'));
    sim_commit;
  
  EXCEPTION
    WHEN OTHERS THEN
      sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                            ' ... Unable to create Copy Log record in COPY_IMPORT_CONFIGURATION procedure.');
      RAISE e_except;
  END;

  /* write file info lines and format column description lines */
  v_line := 'Import Configuration : ' || v_cfg_rec.sicc_name;
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := 'Exported By : ' || v_user;
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := 'Exported On : ' || to_char(SYSDATE, 'MM-DD-RRRR HH:MI AM');
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := 'Configuration Line Format : LINE TYPE (Config)|ORG_ID|ORG_IS_NUMBER|CONFIG_NAME|DESC|DELIMITER|TYPE|INVALID|ACTION';
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := 'Instruction Line Format   : LINE TYPE (Instr)|INSTRUCTION|FORMAT';
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := 'Column Line Format        : LINE TYPE (Col)|COLUMN NAME|COLUMN SEQ|POSITION|INCLUDE|DEFAULT|FORMAT_OPTION|FORMAT|GENERATE';
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := 'Translation Line Format   : LINE TYPE (Trans)|USER CODE|STORET CODE|ST_IS_NUMBER|ST_ORG_ID';
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  v_line := '-----------------------------------------------------------------------------------------------------';
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);

  /* write import configuration record to sim_event_log_lines, must export the meaning for the delimiter
  * because if it is a pipe it will conflict with the file column delimiter */
  IF v_cfg_rec.sicc_delimiter = '|' THEN
    v_delimiter := 'pipe';
  ELSIF v_cfg_rec.sicc_delimiter = '~' THEN
    v_delimiter := 'tilda';
  ELSIF v_cfg_rec.sicc_delimiter = ',' THEN
    v_delimiter := 'comma';
  ELSIF v_cfg_rec.sicc_delimiter = '	' THEN
    v_delimiter := 'tab';
  END IF;

  v_line := 'Config|' || v_cfg_rec.sicc_tsmorgan_org_id || '|' 
    || v_cfg_rec.sicc_tsmorgan_is_number || '|' || v_cfg_rec.sicc_name || '|' 
    || v_cfg_rec.sicc_desc || '|' || v_delimiter || '|' || v_type || '|N|' || v_action_code;
  v_line_cnt := v_line_cnt + 1;
  v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);

  /* select and write all config instruciton records to sim_event_log_lines  */

  OPEN c_instr;
  LOOP
    FETCH c_instr
      INTO v_instr_seq, v_choice_seq;
    EXIT WHEN c_instr%NOTFOUND;
  
    v_line := 'Instr|' || v_instr_seq || '|' || v_choice_seq;
    v_line_cnt := v_line_cnt + 1;
    v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  
  END LOOP;
  CLOSE c_instr;

  /* select all import config columns, then search for column
  * translations, write the column informaiton then the translations
  * to sim_event_log_lines */
  OPEN c_columns;
  LOOP
    FETCH c_columns
      INTO v_col_seq, v_col_name, v_sic_seq, v_pos, v_included, v_generate, v_default, v_fmt, v_fmt_choice;
    EXIT WHEN c_columns%NOTFOUND;
    v_line := 'Col|' || v_col_name || '|' || v_sic_seq || '|' || v_pos || '|' ||
              v_included || '|' || v_default || '|' || v_fmt || '|' || v_fmt_choice || '|' ||
              v_generate;
    v_line_cnt := v_line_cnt + 1;
    v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
  
    /* If the column is "Characteristic" then must also store the char_is_number
    * for Biological the char sic seq is 10217 and for Resutls the char seq is 10051  */
  
    OPEN c_trans;
    LOOP
      FETCH c_trans
        INTO v_user_code, v_storet_code, v_is_num, v_st_is_number, v_st_org_id;
      EXIT WHEN c_trans%NOTFOUND;
      v_line := 'Trans|' || v_user_code || '|' || v_storet_code || '|' ||
                v_st_is_number || '|' || v_st_org_id;
      v_line_cnt := v_line_cnt + 1;
      v_err_nbr := sim_event_log_pkg.insert_event_log_lines(v_line, v_line_cnt, v_sqlcode, v_sqlerrm);
    END LOOP;
    CLOSE c_trans;
  
  END LOOP;
  CLOSE c_columns;
  
  p_sevl_seq := sim_event_log_pkg.get_sevl_seq;
  sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_message,NULL,
    'Finished: ' || to_char(SYSDATE, 'HH24:MI:SS'));
  sim_event_log_pkg.update_stop_time;

EXCEPTION
  WHEN e_file THEN
    NULL;
  WHEN OTHERS THEN
    sim_event_log_pkg.write_event_detail(sim_lib_pkg.c_seldt_error, NULL, SQLERRM ||
                                          ' - in copy_import_configuration.');
END copy_import_configuration;

/****************************************************************************************
  Procedure to sort import config details by included, generated then by column name
  then renumber the column positions.
 ****************************************************************************************/
PROCEDURE reorder_column_positions (
  p_sicc_seq  IN NUMBER,
  p_commit_yn IN VARCHAR2 := 'Y' ) IS
 
  v_position   NUMBER := 0;
    
BEGIN
  FOR v_pos_rec IN (SELECT sicdt.*, sic.sic_name 
                     FROM sim_imp_cfg_dets sicdt, sim_imp_cols sic
                     WHERE sicdt.sicdt_sic_seq = sic.sic_seq
                     AND sicdt_sicc_seq = p_sicc_seq
                     ORDER BY sicdt_included, 
                     decode(sicdt.sicdt_included,null,0,sicdt_pos), 
                     sicdt.sicdt_generate, 
                     decode(sicdt.sicdt_included,NULL,sic.sic_name,'x')) LOOP
   
    IF v_pos_rec.sic_name = '<Ignore>' AND v_pos_rec.sicdt_included IS NULL THEN
      DELETE FROM sim_imp_cfg_dets
        WHERE sicdt_seq = v_pos_rec.sicdt_seq;
    ELSE
      v_position := v_position + 1;
      UPDATE sim_imp_cfg_dets
      SET sicdt_pos = v_position
      WHERE sicdt_seq = v_pos_rec.sicdt_seq;
        
    END IF;
  END LOOP;  
  IF p_commit_yn = 'Y' THEN 
    COMMIT;  
  END IF;
                     
                             
END reorder_column_positions;   

END sim_manage_config_pkg;
/
