
  CREATE OR REPLACE PACKAGE "STORETW"."DW_POST_ETL_PKG"
as

    /**
    * Log procedure execution start and returns the last date the procedure was running successfully
    *
    * @param p_procedure_id      number input parameter containing the id of the procedure starting to run
    * @param p_last_start_date   date output parameter containing the last date the procedure was running successfully
    *
    * @throws others if an exection occurs during processing
    */
    procedure sp_log_start_execution(p_procedure_id in number,
        p_last_start_date out date);

    /**
    * Log procedure execution end and insert running metric
    *
    * @param p_procedure_id      number input parameter containing the id of the procedure finishing to run
    * @param p_success           boolean input parameter containing the status of the run, true if success
    * @param p_count             number input parameter containing the number of records processed, if available
    * @param p_comments          text input parameter containing the comments text, or the exception occured
    *
    * @throws others if an exection occurs during processing
    */
    procedure sp_log_end_execution(p_procedure_id in number,
        p_success  in boolean default true,
        p_count    in number default null,
        p_comments in varchar2 default null);

    /**
    * Starts the Post ETL Process
    *
    * @throws others if an exection occurs during processing */
    procedure sp_start_post_etl_process;

end;
/
CREATE OR REPLACE PACKAGE BODY "STORETW"."DW_POST_ETL_PKG"
as

    --lv_message         LONG;
    --lv_from            VARCHAR2(100) := 'STORET@INDUSCORP.COM';
    --lv_email_address   VARCHAR2(100) := 'STORET@INDUSCORP.COM';

    /**
    * Log procedure execution start and returns the last date the procedure was running successfully
    *
    * @param p_procedure_id      number input parameter containing the id of the procedure starting to run
    * @param p_last_start_date   date output parameter containing the last date the procedure was running successfully
    *
    * @throws others if an exection occurs during processing
    */
    procedure sp_log_start_execution(p_procedure_id in number,
        p_last_start_date out date)
    is
        pragma autonomous_transaction;
    begin

        -- Update run status with RUNNING for procedure being currently run
        update post_etl_execution_status
           set run_status = 'RUNNING',
               current_start_date = sysdate,
               current_end_date = null
         where pk_isn = p_procedure_id;

        commit;

        -- Retrieve the last start date when the procedure was running successfully
        select last_start_date
          into p_last_start_date
          from post_etl_execution_status
         where pk_isn = p_procedure_id;

    end;



    /**
    * Log procedure execution end and insert running metric
    *
    * @param p_procedure_id      number input parameter containing the id of the procedure finishing to run
    * @param p_success           boolean input parameter containing the status of the run, true if success
    * @param p_count             number input parameter containing the number of records processed, if available
    * @param p_comments          text input parameter containing the comments text, or the exception occured
    *
    * @throws others if an exection occurs during processing
    */
    procedure sp_log_end_execution(p_procedure_id in number,
        p_success  in boolean default true,
        p_count    in number default null,
        p_comments in varchar2 default null)
    is
        pragma autonomous_transaction;
        l_status    varchar2(30);
        l_row       post_etl_execution_status%rowtype;
    begin

        -- Update the current status for the stored procedure with the end date
        l_status := case when p_success then 'SUCCESS' else 'FAILURE' end;

        update post_etl_execution_status
           set run_status = l_status,
               current_end_date = sysdate
         where pk_isn = p_procedure_id;

        commit;

        -- Insert execution metric for the current procedure
        select *
          into l_row
          from post_etl_execution_status
         where pk_isn = p_procedure_id;

        insert into post_etl_execution_metric (pk_isn, fk_petl_exec_status, run_status,
            start_date, end_date, record_count, comments)
        values (seq_post_etl_execution_metric.nextval, p_procedure_id, l_row.run_status,
            l_row.current_start_date, l_row.current_end_date, p_count, p_comments);

        commit;

        --lv_message := l_row.procedure_name || ' end time: ' || l_row.current_end_date;
        --dw_mail_pkg.send(p_sender_email => dw_glob_var.gv_storet_email, p_from => lv_from, p_to => dw_mail_pkg.array(lv_email_address), p_subject => lpad(p_procedure_id,2,'0') || '/18 Post-ETL Run Performance Metrics', p_body => lv_message);

    end;



    /**
    * Starts the Post ETL Process
    *
    * @throws others if an exection occurs during processing */
    procedure sp_start_post_etl_process
    is
        l_last_start_date   date;
        l_record_no         number;
        l_comments          varchar2(4000);
    begin

        --lv_message := 'Post ETL start time: ' || to_char(sysdate,'MM/DD/YYYY HH24:MI:SS');
        --dw_mail_pkg.send(p_sender_email => dw_glob_var.gv_storet_email, p_from => lv_from, p_to => dw_mail_pkg.array(lv_email_address), p_subject => '01/18 Post-ETL Run Performance Metrics', p_body => lv_message);

        -- Update currect execution status of all procedures to PENDING and update last run dates
        -- for successful runs only
        update post_etl_execution_status
           set current_start_date = null,
               current_end_date = null,
               last_start_date = decode(run_status,'SUCCESS',current_start_date,last_start_date),
               last_end_date = decode(run_status,'SUCCESS',current_end_date,last_end_date),
               run_status = 'PENDING'
         where run_status <> 'OBSOLETE';

        commit;

        -- Start calling each procedure one bu one, logging the start and the end of the run
        dw_post_etl_pkg.sp_log_start_execution(1, l_last_start_date);
        begin
            storetw.sp_insertintoactivityproject(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(1, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(1, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(2, l_last_start_date);
        begin
            storetw.sp_conv_wgs84_latlong(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(2, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(2, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(3, l_last_start_date);
        begin
            storetw.sp_update_geomwgs84_latlong(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(3, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(3, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(4, l_last_start_date);
        begin
            storetw.sp_update_fa_station_huc(l_last_start_date, 10000, l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(4, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(4, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(5, l_last_start_date);
        begin
            storetw.sp_update_result_tables_huc(l_last_start_date, 10000, l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(5, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(5, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(6, l_last_start_date);
        begin
            storetw.sp_update_result_tables_statn(l_last_start_date, 10000, l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(6, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(6, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(18, l_last_start_date);
        begin
            storetw.sp_refresh_station_char_proj(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(18, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(18, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(7, l_last_start_date);
        begin
            storetw.sp_refresh_station_char_medium(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(7, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(7, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(8, l_last_start_date);
        begin
            storetw.sp_refresh_station_char_new(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(8, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(8, true, l_record_no, l_comments);


        /*
        dw_post_etl_pkg.sp_log_start_execution(9, l_last_start_date);
        begin
            storetw.sp_refresh_ws_station_data(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(9, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(9, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(10, l_last_start_date);
        begin
            storetw.sp_refresh_ws_station_summary(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(10, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(10, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(11, l_last_start_date);
        begin
            storetw.sp_refresh_ws_huc_org_summary(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(11, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(11, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(12, l_last_start_date);
        begin
            storetw.sp_refresh_ws_huc_char_summary(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(12, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(12, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(13, l_last_start_date);
        begin
            storetw.sp_refresh_ws_org_char_summary(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(13, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(13, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(14, l_last_start_date);
        begin
            storetw.sp_refresh_proj_statn_char(l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(14, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(14, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(15, l_last_start_date);
        begin
            storetw.sp_refresh_ws_org_activity(10000, l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(15, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(15, true, l_record_no, l_comments);


        dw_post_etl_pkg.sp_log_start_execution(16, l_last_start_date);
        begin
            storetw.sp_refresh_ws_org_act_proj(10000, l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(16, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(16, true, l_record_no, l_comments);
        */


        dw_post_etl_pkg.sp_log_start_execution(17, l_last_start_date);
        begin
            storetw.sp_email_inventory(l_last_start_date, sysdate, l_record_no, l_comments);
        exception
            when others then
                dw_post_etl_pkg.sp_log_end_execution(17, false, null, sqlcode || ' - ' || sqlerrm);
                raise;
        end;
        dw_post_etl_pkg.sp_log_end_execution(17, true, l_record_no, l_comments);

    end;


end;
/

