package gov.epa.cdx.ws.ejbs.sb.server;
 
import gov.epa.cdx.commons.exception.CDXErrorCode;
import gov.epa.cdx.commons.exception.CDXException;
import gov.epa.cdx.commons.property.CDXPropertyManager;
import gov.epa.cdx.logic.requestor.NAASManager;
import gov.epa.cdx.model.commons.QueryVO;
import gov.epa.cdx.model.document.DocumentList;
import gov.epa.cdx.model.node.NodeVersion;
import gov.epa.cdx.model.node.NodeVersionVO;
import gov.epa.cdx.model.security.AuthTokenVO;
import gov.epa.cdx.model.security.LoginVO;
import gov.epa.cdx.model.supportedservices.ServiceTypes;
import gov.epa.cdx.model.transaction.Status;
import gov.epa.cdx.model.transaction.TransactionVO;
import gov.epa.cdx.ws.logic.WSNodeManager;
import gov.epa.wqx.node.*;

import java.rmi.RemoteException;

import java.sql.SQLException;
import java.sql.Timestamp;

import javax.ejb.SessionBean;

import javax.naming.NamingException;

import oracle.jdbc.OracleConnection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *     @ejb.bean  type="Stateless"
 *             name="server.WSServerService"
 *             jndi-name="server.WSServerService"
 *             view-type="remote"
 *
 * @ejb.transaction type="Supports"
 *
 */
public abstract class WSServerBean implements SessionBean {
    
  protected static Log log = LogFactory.getLog(WSServerBean.class.getName());
  
  private static NodeVersionVO NODE_VERSION;
  {
    String id = CDXPropertyManager.getProperty("NAAS", "id");
    String ip = CDXPropertyManager.getProperty("NAAS", "ip");
    String name = CDXPropertyManager.getProperty("NAAS", "name");
    String url = CDXPropertyManager.getProperty("NAAS", "url");
    NodeVersionVO nv = new NodeVersionVO(id, ip, name, NodeVersion.V10, url);
    NODE_VERSION = nv;
  }
 
 
  private void validateToken(AuthTokenVO authTokenVO)
      throws CDXException {
      
    if (Lib.useNaas()) {
      if (Lib.allowTwoWayCommunication()) {
        String clientHostIp = CDXPropertyManager.getProperty("CDX", "ClientHostIp");
        if (clientHostIp == null) {
          clientHostIp = "";
        }
        if (clientHostIp.length() > 0) {
          log.debug("ClientHostIp=" + clientHostIp + " (from wqxnode_properties.xml)");    
          authTokenVO.setClientHostIP(clientHostIp);
        }
        else {
          log.debug("ClientHostIp=" + authTokenVO.getClientHostIP());
        }
      }
      NAASManager.executeValidate(NODE_VERSION, authTokenVO);
    }
  }
  
  
  /** <p> This method authenticate the user (node)against central security </p>
   * @param loginVO - value oject containing all user(node)related information
   * @return AuthTokenVO - which must contain authentication token and optionaly other SecurityVO
   *                  attributes
   * @throws gov.epa.cdx.commons.exception.CDXException which should be CDXSecurityException
   * @ejb.interface-method view-type="remote"
   */
  public AuthTokenVO authenticate(LoginVO loginVO) throws RemoteException, CDXException {
      
    if (Lib.useNaas()) { 
      return NAASManager.executeLogin(NODE_VERSION, loginVO);
    }
    else {
      AuthTokenVO authToken = new AuthTokenVO();
      authToken.setAuthToken("DummyToken");
      return authToken;
    }
  }


  /** <p> This method provides the capability to submit the list of the documentsto the server</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param documentList - list of the documents to submit
   * @return TRansactionVO - transaction associted with the documents
   * @throws gov.epa.cdx.commons.exception.CDXException which can be CDXSecurityException, CDXValidationException, CDXServerException
   * @ejb.interface-method view-type="remote"
   */
  public TransactionVO submit(AuthTokenVO authTokenVO, DocumentList documentList) 
      throws RemoteException, CDXException {
      
    log.debug("WSServerBean.submit() - Start");
    validateToken(authTokenVO);      
    TransactionVO transactionVO = WSNodeManager.processSubmit(documentList);
    log.debug("WSServerBean.submit() - End");        
    return transactionVO;
  }


  /** <p> This method provides the capability to start asynchronous query</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param  queryVO - contains query parameters and node information
   * @return transactionVO associted with the task
   * @throws gov.epa.cdx.commons.exception.CDXException which can be CDXSecurityException, CDXValidationException, CDXServerException
   * @ejb.interface-method view-type="remote"
   */
  public TransactionVO solicit(AuthTokenVO authTokenVO, QueryVO queryVO) 
      throws RemoteException, CDXException {
      
    log.debug("WSServerBean.solicit() - Start");
    validateToken(authTokenVO);
    TransactionVO transactionVO = WSNodeManager.processSolicit(queryVO);
    log.debug("WSServerBean.solicit() - End");
    return transactionVO;
  }


  /** <p> This method provides the capability to start synchronous query on the data flow</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param  queryVO - contains query parameters and node information
   * @return DocumentList - document returned by the dataflow
   * @throws gov.epa.cdx.commons.exception.CDXException which can be CDXSecurityException, CDXValidationException, CDXServerException
   * @ejb.interface-method view-type="remote"
   */
  public byte[] query(AuthTokenVO authTokenVO, QueryVO queryVO) 
      throws RemoteException,CDXException,ClassNotFoundException,SQLException,NamingException,WqxException {
      
    log.debug("WSServerBean.query() - Start");
    OracleConnection conn = null;
    TransactionLog transactionLog = null;
    try {
      conn = Lib.getWqxConnection();
      transactionLog = new TransactionLog(conn);
      String[] params = null;
      String request = null;
      String logOther = null; // String of all parameters passed to startTransaction
      byte[] result = null;
      validateToken(authTokenVO);
      request = queryVO.getServiceName();
      params = queryVO.getParameters();
      Query query = new Query(conn, Constant.TRTYP_QUERY);
      transactionLog.setType(Constant.TRTYP_QUERY);
      transactionLog.setStartTime(new Timestamp(System.currentTimeMillis()));
      if (request.equalsIgnoreCase("WQX.GetDomainValueByElementName_v1.0")) {
        logOther = "Request="+request+";elementName="+params[0];
        transactionLog.setUserNaasId(params[1]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getDomainValues(params[0], params[1]);
      }
      else if (request.equalsIgnoreCase("WQX.GetTransactionHistoryByParameters_v1.0")) {
        logOther = "Request="+request+";organizationIdentifier="+params[0]
                   +";userIdentifier="+params[1]+";transactionIdentifier="+params[2]
                   +"transactionDateBegin="+params[3]+";transactionDateEnd="+params[4];
        transactionLog.setOrganizationId(params[0]);
        transactionLog.setUserNaasId(params[5]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getTransactionHistory(
          params[0],
          params[1],
          params[2],
          params[3],
          params[4],
          params[5]);
      }
      else if(request.equalsIgnoreCase("WQX.GetProjectByParameters_v1.0")) {
        logOther = "Request="+request+";organizationIdentifier="+params[0]
                   +";projectIdentifier="+params[1];
        transactionLog.setOrganizationId(params[0]);
        transactionLog.setUserNaasId(params[2]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getProject(params[0], params[1], params[2]);
      }
      else if(request.equalsIgnoreCase("WQX.GetMonitoringLocationByParameters_v1.0")) {
        logOther = "Request="+request+";organizationIdentifier="+params[0]
                   +";monitoringLocationIdentifier="+params[1];
        transactionLog.setOrganizationId(params[0]);
        transactionLog.setUserNaasId(params[2]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getMonitoringLocation(params[0],params[1],params[2]);
      }
      else if(request.equalsIgnoreCase("WQX.GetActivityByParameters_v1.0")) {
        logOther = "Request="+request+";organizationIdentifier="+params[0]
                   +";monitoringLocationIdentifier="+params[1]
                   +";projectIdentifier="+params[2]+";activityStartDateBegin="+params[3]
                   +";activityStartDateEnd="+params[4]+";activityIdentifier="+params[5];
        transactionLog.setOrganizationId(params[0]);
        transactionLog.setUserNaasId(params[6]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getActivity(
          params[0],
          params[1],
          params[2],
          params[3],
          params[4],
          params[5],
          params[6]);
      }
      else if(request.equalsIgnoreCase("WQX.GetResultByParameters_v1.0")) {
        logOther = "Request="+request+";organizationIdentifier="+params[0]
                   +";monitoringLocationIdentifier="+params[1]
                   +";projectIdentifier="+params[2]+";activityStartDateBegin="+params[3]
                   +";activityStartDateEnd="+params[4]+";activityIdentifier="+params[5];
        transactionLog.setOrganizationId(params[0]);
        transactionLog.setUserNaasId(params[6]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getResult(
          params[0],
          params[1],
          params[2],
          params[3],
          params[4],
          params[5],
          params[6]);
      }
      else if(request.equalsIgnoreCase("WQX.GetActivityGroupByParameters_v1.0")) {
        logOther = "Request="+request+";organizationIdentifier="+params[0]
                   +";activityGroupTypeCode="+params[1]
                   +";activityGroupIdentifier="+params[2];
        transactionLog.setOrganizationId(params[0]);
        transactionLog.setUserNaasId(params[3]);
        transactionLog.setComments(logOther);
        transactionLog.save();
        result = query.getActivityGroup(
          params[0], 
          params[1], 
          params[2], 
          params[3]);
      }
      else {
        throw new WqxException("The request " + request + " is not valid.", null);
      }
      transactionLog.end(new Timestamp(System.currentTimeMillis()));
      return result;
    }
    catch(Exception exception) {
      transactionLog.logAndThrowException(exception);
    }
    finally {
      if(conn != null) {
        conn.close();
      }
      log.debug("WSServerBean.query() - End");
    }
    return null;
  }


  /** <p> This method provides the capability to obtain the status of the transaction</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param transactionVO - transaction the status is reqested for
   * @return Status
   * @throws gov.epa.cdx.commons.exception.CDXException which can be CDXSecurityException, CDXValidationException, CDXServerException
   * @ejb.interface-method view-type="remote"
   */
  public Status getStatus(AuthTokenVO authTokenVO, TransactionVO transactionVO) 
      throws RemoteException, CDXException {
    
    CDXException ex = new CDXException(CDXErrorCode.FEATURE_UNSUPPORTED);
    throw ex;
  }
  

  /** <p> This method returns the collection of supported servises based on passed service type</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param serviceType
   * @return Array of services
   * @throws gov.epa.cdx.commons.exception.CDXException
   * @ejb.interface-method view-type="remote"
   */
  public String[] getServices(
      AuthTokenVO authTokenVO, 
      ServiceTypes serviceType)
      
      throws RemoteException, CDXException {
      
    validateToken(authTokenVO);    
    //do something to return list of service this node supports
    return new String[]{"submit"};
  }

  
  /**
   *  <p> This method provides the capability to notify the server about document availability, status, and events</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param documentList - transaction the status is reqested for
   * @param nodeAddress- initiator's node address/ could be empty
   * @return TransactionVO
   * @throws gov.epa.cdx.commons.exception.CDXException which can be CDXSecurityException, CDXValidationException, CDXServerException
   * @ejb.interface-method view-type="remote"
   */
  public String notify(
      AuthTokenVO authTokenVO, 
      String nodeAddress,
      String dataFlow, 
      DocumentList documentList) 
      
      throws RemoteException, CDXException {
      
    CDXException ex = new CDXException(CDXErrorCode.FEATURE_UNSUPPORTED);
    throw ex;
  }


  /** <p> This method provides the capability to download documents from the server</p>
   * @param authTokenVO - value oject containing all user(node)related information
   * @param documentList - Documents "metadata"
   * @param transactionVO - transaction associted with the document list
   * @return DocumentList - DocumentList
   * @throws CDXException which can be CDXSecurityException, CDXValidationException, CDXServerException
   * @ejb.interface-method view-type="remote"
   */
  public DocumentList download(
      AuthTokenVO authTokenVO, 
      DocumentList documentList, 
      TransactionVO transactionVO)  
      
      throws RemoteException, CDXException {
      
    CDXException ex = new CDXException(CDXErrorCode.FEATURE_UNSUPPORTED);
    throw ex;
  }

}
