package gov.epa.cdx.ws.logic;

/**
 * WSBackEndWorkerThread.java
 * 
 * This class is a distinct thread of execution which attempts to load the 
 * data sent in a "Submit" and reports back the results using a "Notify".
 *
 * This class exists as a seperate thread so that the return of the "Submit" 
 * communication was not effected by the database interaction time.  This process
 * will run asynchronously and "Notify" CDXNode when the database interaction is
 * complete and results are returned.
 */
 
import gov.epa.cdx.axis.v10.vo.NodeDocument;
import gov.epa.cdx.axis.v10.vo.NodeDocumentContentConverter;
import gov.epa.cdx.commons.exception.CDXException;
import gov.epa.cdx.commons.property.CDXPropertyManager;
import gov.epa.cdx.model.document.DocumentList;
import gov.epa.cdx.model.document.DocumentVO;

import gov.epa.wqx.node.Lib;
import gov.epa.wqx.node.WqxSaxHandler;
import gov.epa.wqx.node.WqxUnzip;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import java.rmi.RemoteException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.parsers.SAXParser;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import gov.epa.cdx.commons.property.CDXPropertyManager;
import gov.epa.cdx.commons.zip.ZipUtil;
import gov.epa.cdx.model.document.DocumentType;
import gov.epa.cdx.model.document.DocumentVO;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;

import java.util.Enumeration;
import java.util.ListIterator;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.CopyUtils;
import org.apache.commons.io.FileUtils;


public class WSBackEndWorkerThread implements Runnable {

  protected static Log log = LogFactory.getLog(WSBackEndWorkerThread.class.getName());
    
  private DocumentList _documents;    
    
  /**
   * Constructor.
   * @param documents The documents that were "Submitted"
   */
  public WSBackEndWorkerThread(DocumentList documents) {
    _documents = documents;
  }


  public static File unzipDocument(DocumentVO document, String transactionId) 
      throws Exception {
    
    try {
      log.debug("WSBackEndWorkerThread.unzip - Start");
      String workingFolder = Lib.getWorkingDirectory();
      File tempFolder = new File(workingFolder + transactionId);
      log.debug("tempFolder=" + tempFolder.getPath());
      FileUtils.forceMkdir(tempFolder);
      String cdxFilePath = tempFolder + Lib.getFileSeparator() + document.getName();
      FileOutputStream zipFileOutputStream = new FileOutputStream(cdxFilePath);
      CopyUtils.copy(document.getData(), zipFileOutputStream);
      zipFileOutputStream.flush();
      zipFileOutputStream.close();
      if (document.getName().toUpperCase().indexOf(".XML") > 0) {
        return new File(cdxFilePath);
      }
      else {
        return unzipFile(cdxFilePath); 
      } 
    } 
    catch (Exception e) {
      log.error(e);
      throw e;
    }
    finally {
      log.debug("WSBackEndWorkerThread.unzip - End");
    }
  }
  
  
  /**
    * Writes the contents of a zip file to disk.
    * 
    * @param zipFilePath
    *         The path and file name of the zip file to unzip.
    * @return A File object for the XML submission file to be loaded 
    * @throws Exception
    */
  public static File unzipFile(String zipFilePath) throws Exception {
    
    final int BUFFER = 2048;   
    File xmlSubmissionFile = null;
    try {
      log.debug("WSBackEndWorkerThread.unzipFile - Start");
      // slash character to use for OS
      String osSlash = Lib.getFileSeparator();
      // we'll extract the files into the same folder as the original zip file
      String destinationPath = zipFilePath.substring(0, zipFilePath.lastIndexOf(osSlash));
      ZipFile zipFile = new ZipFile(zipFilePath);
      Enumeration zipEnumeration = zipFile.entries();
      String zipEntryFileNameOnly;
      // interate through each entry in the zip file
      while(zipEnumeration.hasMoreElements()) {
        ZipEntry zipEntry = (ZipEntry) zipEnumeration.nextElement();
        // unzip the file (as long as it's not a directory)
        if (!zipEntry.isDirectory()) {
          String zipEntryName = zipEntry.getName();
          // determine the position of the slash character used in the zip file entry
          // and then get just the entry name (without the path)
          int lastSlash = zipEntryName.lastIndexOf("\\");
          if (lastSlash < 1) {
            lastSlash = zipEntryName.lastIndexOf("/"); 
          }
          if (lastSlash < 1) {
            zipEntryFileNameOnly = zipEntryName.trim();
          }
          else {
            zipEntryFileNameOnly = zipEntryName.substring(lastSlash + 1).trim();
          }
          if (zipEntryName.indexOf(".") == -1) {
            // entry file does not have a file extension
            if (zipFile.size() == 1) {
              // if it's the only file then assume XML
              zipEntryFileNameOnly += ".xml";
            }
          }
          log.debug("zipEntryFileNameOnly=" + zipEntryFileNameOnly);
          String outputFileName = destinationPath + osSlash + zipEntryFileNameOnly;
          if (zipEntryFileNameOnly.toUpperCase().indexOf(".XML") > 0) {
            xmlSubmissionFile = new File(outputFileName);
            log.debug("xmlSubmissionFile=" + xmlSubmissionFile);
          }
          BufferedOutputStream outputStream = new BufferedOutputStream(
            new FileOutputStream(outputFileName), BUFFER);
          BufferedInputStream zipEntryInputStream = new BufferedInputStream(zipFile.getInputStream(zipEntry));
          byte[] data = new byte[BUFFER];
          int bytesRead;
          while ((bytesRead = zipEntryInputStream.read(data, 0, BUFFER)) != -1) {
            outputStream.write(data, 0, bytesRead);
          }
          outputStream.flush();
          outputStream.close();
          zipEntryInputStream.close();
        }
      } 
      return xmlSubmissionFile;
    } 
    catch(Exception e) {
      log.error(e);
      throw e;
    }
    finally {
      log.debug("WSBackEndWorkerThread.unzipFile - End");
    }
  }
  
    
    public void run() {
      log.info("******** WSBackEndWorkerThread.run() - Start ********");
      String xmlPath = null;
      try {
        String transactionId = _documents.getTransaction().getTransactionID();
        DocumentVO doc = (DocumentVO) _documents.getDocuments().iterator().next();
        File xmlFile = unzipDocument(doc, transactionId);    
        xmlPath = xmlFile.getPath().substring(0, xmlFile.getPath().lastIndexOf(Lib.getFileSeparator()));
        String xmlFileName = xmlFile.getName();
        xmlFile = null;
        parseDocument(xmlPath, xmlFileName, transactionId);           
      }
      catch(Exception e) {
        log.trace("Failed to Parse Document:", e);  
      }
      finally {
        try {
          // delete temp folder and its contents
          if (xmlPath != null) {
            File tempFolder = new File(xmlPath);
            File[] files = tempFolder.listFiles();
            if (files.length > 0) {
              log.debug("# of files in temp folder=" + files.length);
              for (int i=0; i < files.length; i++) {
                File file = files[i];
                log.debug("deleting " + file.getName());
                file.delete();
              }
            }
            log.debug("deleting temp folder " + tempFolder.getName());
            tempFolder.delete();
          }
        }
        catch (Exception e) {
          log.error("Error Deleting Temporary Folder: ", e);
        }
        log.info("******** WSBackEndWorkerThread.run() - End ********");
      }
    }
    
    
    private void parseDocument(
      String docPath,
      String docName, 
      String transactionId) 
      
      throws Exception, IOException, SAXException {
      
    WqxSaxHandler handler = null;
    try {
      log.debug("WSBackEndWorkerThread.parseDocument() - Start");
      //XMLReader reader = new XMLReaderImpl();
      XMLReader reader = XMLReaderFactory.createXMLReader("org.apache.xerces.parsers.SAXParser");
      // create instance of this class
      handler = new WqxSaxHandler(
        docPath,
        docName,
        transactionId);    
      reader.setContentHandler(handler);
      reader.setErrorHandler(handler);
      // Create an input source that points to the file to parse.
      InputSource input = new InputSource(new FileReader(docPath + Lib.getFileSeparator() + docName));
      // Parse and Load the submission file
      reader.parse(input);
    } 
    catch (Exception e) {
      log.error(e);
      log.info(Lib.getSimplifiedStackTrace(e,false));
    }
    finally {
      // Get the processing report and then call notify
      if (handler != null) {     
        try {
          log.debug("Building Processing Report");
          byte[] content = handler.getProcessingReport().getBytes();
          if (Lib.allowTwoWayCommunication()) {
            NodeDocument nodeDoc = new NodeDocument();
            nodeDoc.putContent(content, NodeDocumentContentConverter.CONTENT_TYPE_BYTES);
            nodeDoc.setName(transactionId);
            nodeDoc.setType("XML");
            String urlEndpoint = CDXPropertyManager.getProperty("Submit", "NotifyURL");
            WSNodeManager.callCdxNotify(transactionId, nodeDoc, urlEndpoint);
          }
          else {
            // write processing report out to a file
            FileOutputStream fileOutputStream = new FileOutputStream(Lib.getWorkingDirectory() + transactionId + "_ProcessingReport.xml");
            CopyUtils.copy(content, fileOutputStream);
            fileOutputStream.flush();
            fileOutputStream.close();
          }
        }
        catch (Exception e) {
          log.error(e);
          log.info(Lib.getSimplifiedStackTrace(e,false));     
        }
        finally {
          handler.done();
        }
      }
      log.debug("WSBackEndWorkerThread.parseDocument() - End");
    }		
  }
}
