package gov.epa.cdx.ws.axis.impl;
 
import gov.epa.cdx.axis.fault.FaultErrorCode;
import gov.epa.cdx.axis.fault.FaultHandler;
import gov.epa.cdx.axis.v10.NetworkNodePortType;
import gov.epa.cdx.axis.v10.impl.DocumentConverter;
import gov.epa.cdx.axis.v10.vo.NodeDocument;
import gov.epa.cdx.axis.v10.vo.NodeDocumentContentConverter;
import gov.epa.cdx.axis.v10.vo.holders.ArrayofDocHolder;
import gov.epa.cdx.commons.exception.CDXException;
import gov.epa.cdx.model.commons.QueryVO;
import gov.epa.cdx.model.document.DocumentList;
import gov.epa.cdx.model.node.NodeVersion;
import gov.epa.cdx.model.scheduler.CDXTaskTypes;
import gov.epa.cdx.model.scheduler.TaskTypeVO;
import gov.epa.cdx.model.security.AuthTokenVO;
import gov.epa.cdx.model.security.AuthorizationVO;
import gov.epa.cdx.model.security.CredentialType;
import gov.epa.cdx.model.security.LoginVO;
import gov.epa.cdx.model.supportedservices.Methods;
import gov.epa.cdx.model.supportedservices.ServiceTypes;
import gov.epa.cdx.model.transaction.Status;
import gov.epa.cdx.model.transaction.TransactionVO;
import gov.epa.cdx.ws.ejbs.sb.server.WSServerServiceRemote;
import gov.epa.cdx.ws.ejbs.sb.server.WSServerServiceUtil;
import gov.epa.wqx.node.WqxException;

import java.math.BigInteger;

import java.rmi.RemoteException;

import javax.servlet.http.HttpServletRequest;

import javax.sql.DataSource;

import org.apache.axis.AxisFault;
import org.apache.axis.MessageContext;
import org.apache.axis.transport.http.HTTPConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.epa.wqx.node.Constant;

public class WSWSImplementation implements NetworkNodePortType {

  protected static Log log = LogFactory.getLog(WSWSImplementation.class.getName());


  private String getClientHost() {

    // Get clienthost from the Axis Servlet Request
    MessageContext msgContext = MessageContext.getCurrentContext();
    HttpServletRequest req = 
      (HttpServletRequest) msgContext.getProperty(HTTPConstants.MC_HTTP_SERVLETREQUEST);
    String clientHost = req.getRemoteAddr();
    return clientHost;
  }


  /** <p> Authenticate the user <p/>
   * @param userId - user id
   * @param credential - currently password
   * @param authMethod - type of credential which is password type
   * @return String - authentication token
   * @throws AxisFault
   */
  public String authenticate(
      String userId, 
      String credential, 
      String authMethod) 
      
      throws RemoteException {
      
    AuthTokenVO authTokenVO = null;
    LoginVO loginVO = null;
    String status = null;
    try {
      loginVO = new LoginVO(
        getClientHost(),
        NodeVersion.V10,
        userId,
        CredentialType.getCredentialTypeById(authMethod),
        credential);
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      authTokenVO = bean.authenticate(loginVO);
      return authTokenVO.getAuthToken();
    }
    catch (Exception exception) {
      log.error("Authentication Failed", exception);
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      String failureMessage =  "User" + userId + " with "+authMethod+": "+ credential+ " IP: " + getClientHost();
      throw fault;
    }
  }


  /** <p> Submits the documents to the server <p/>
   * @param authToken - authentication token
   * @param transactionId - optional transaction id
   * @param dataflow - dataflow documents belongs to
   * @param documents - array of cdx documents
   * @return Transaction - created transaction
   * @throws AxisFault
   */
  public String submit(
      String authToken, 
      String transactionId, 
      String dataflow, 
      NodeDocument[] documents) 
      
      throws RemoteException {

    log.info("******** submit() - Start ********");
    try {
      if (transactionId == null) {
        throw new RemoteException("transactionId is required for submit");
      }
      TransactionVO transactionVO = null;
      AuthTokenVO authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      AuthorizationVO authVO = new AuthorizationVO();
      authVO.setDataFlow(Constant.DATAFLOW);
      authVO.setMethod(Methods.SUBMIT);
      authVO.setRequest("n/a");
      String[] params = {"n/a"};
      authVO.setParameters(params);
      authTokenVO.setAuthorizationVO(authVO);
      // convert CDX documents to DocumentList
      DocumentList documentList = DocumentConverter.CdxToDocumentList(documents, dataflow);
      if (transactionId != null) {
        TransactionVO inputTransactionVO = new TransactionVO();
        inputTransactionVO.setTransactionID(transactionId);
        documentList.setTransaction(inputTransactionVO);
      }
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      transactionVO = bean.submit(authTokenVO, documentList);
      return transactionVO.getTransactionID();
    }
    catch(Exception exception) {
      log.error("Submit Failed", exception);
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      throw fault;
    }
    finally {
      log.info("******** submit() - End ********");
    }
  }


  /** <p> Returns the status of the transaction <p/>
   * @return Transaction - created transaction
   * @throws AxisFault
   */
  public String getStatus(
      String authToken, 
      String transid) 
      
      throws RemoteException {

    log.info("******** getStatus() - Start ********");
    try {
      Status status = null;
      AuthTokenVO authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      TransactionVO transactionVO = new TransactionVO();
      transactionVO.setTransactionID(transid);      
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      status = bean.getStatus(authTokenVO, transactionVO);
      return status.toString();
    }
    catch(Exception exception) {
      log.error("getStatus Failed", exception);
      // add error code to handle Internal Server Error
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      //audit.failure(transid, exception.getMessage(), exception, authTokenVO);
      throw fault;
    }
    finally {
      log.info("******** getStatus() - End ********");
    }
  }


  /** <p> Docwnload the documents from the server <p/>
   * @param authToken - authentication token
   * @param transid -  transaction id
   * @param dataflow - dataflow documents belongs to
   * @param documents - array of cdx documents which is in/out parameter
   * @throws AxisFault refer to spec
   */
  public void download(
      String authToken,
      String transid,
      String dataflow,
      ArrayofDocHolder documents) 
      
      throws RemoteException {
      
    log.info("******** download() - Start ********");
    try {
      AuthTokenVO authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      DocumentList documentList = null;
      TransactionVO transactionVO = new TransactionVO();
      //convert CDX documents to document list
      documentList = DocumentConverter.CdxToDocumentList(documents.value, dataflow);
      transactionVO.setTransactionID(transid);
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      documentList = bean.download(authTokenVO, documentList, transactionVO);
      // convert document list back to CDX documents
      documents.value = DocumentConverter.DocumentListToCdx(
        documentList,
        NodeDocumentContentConverter.CONTENT_TYPE_ATTACHMENT);
    }
    catch(Exception exception) {
      log.error("Download Failed", exception);
      // add error code to handle Internal Server Error
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      //audit.failure(transid, exception.getMessage(), exception, authTokenVO);
      throw fault;
    }
    finally {
      log.info("******** download() - End ********");
    }
  }

  
  /** <p> notifies about the availability of the documents <p/>
   * @param authToken - authentication token
   * @param nodeAddress- initiator's node address/ could be empty
   * @param dataFlow - dataflow documents belongs to
   * @param documents - array of cdx documents
   * @return Transaction - created transaction
   * @throws AxisFault
   */
  public String notify(
      String authToken, 
      String nodeAddress, 
      String dataFlow, 
      NodeDocument[] documents) 
      
      throws RemoteException {
      
    log.info("******** notify() - Start ********");
    try {
      String transactionId = null;
      DocumentList documentList = null;
      // create authtoken value object
      AuthTokenVO authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      // create documentlist converting CDX documents to DocumentList
      documentList = DocumentConverter.CdxToDocumentList(documents, dataFlow);
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      transactionId = bean.notify(authTokenVO, nodeAddress, dataFlow, documentList);
      return transactionId;
    }
    catch(Exception exception) {
      log.error("Notify Failed", exception);
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      //audit.failure(transactionId, exception.getMessage(), exception);
      throw fault;
    }
    finally {
      log.info("******** notify() - End ********");
    }
  }


  /** <p> solicit for asynchronous query <p/>
   * @param authToken - authentication token
   * @param returnURL - url of the requestor (optional)
   * @param serviceName - name of the stored proc to call (it is associated with data flow)
   * @param parameters - values for stored proc to call
   * @return TransactionId - transaction Id the requestor can later track and retrieve the data
   * @throws AxisFault
   */
  public String solicit(
      String authToken, 
      String returnURL, 
      String serviceName, 
      String[] parameters) 
      
      throws RemoteException {
      
    log.info("******** solicit() - Start ********");
    try {
      if (returnURL == null) {
        throw new RemoteException("returnURL parameter must contain the Transaction ID from CDX");
      }
      TransactionVO transactionVO = null;
      AuthTokenVO authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      AuthorizationVO authVO = new AuthorizationVO();
      authVO.setDataFlow(Constant.DATAFLOW);
      authVO.setMethod(Methods.QUERY);
      authVO.setRequest(serviceName);
      authVO.setParameters(parameters);
      authTokenVO.setAuthorizationVO(authVO);
      // Build Query Object
      QueryVO queryVO = new QueryVO(
        returnURL, 
        serviceName, 
        parameters,
        new TaskTypeVO(CDXTaskTypes.DataFlowAsyncQuery),
        "Solicit");
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      transactionVO = bean.solicit(authTokenVO, queryVO);
      return transactionVO.getTransactionID();
    }
    catch(Exception exception) {
      log.error("Solicit Failed", exception);
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      throw fault;
    }
    finally {
      log.info("******** solicit() - End ********");
    }
  }


  /** <p> query <p/>
   * @param authToken - authentication token
   * @param request - name of the stored proc to call (it is associated with data flow)
   * @param rowId - start row Id
   * @param maxRows - max rows to fetch
   * @param parameters - values for stored proc to call
   * @return sqlResult
   * @throws AxisFault
   */
  public String query(
      String authToken,  
      String request, 
      BigInteger rowId, 
      BigInteger maxRows, 
      String[] parameters) 
      
      throws RemoteException {
      
    log.info("******** query() - Start ********");
    try {
      String result = null;
      String transactionID = null;
      QueryVO queryVO = null;
      AuthTokenVO authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      AuthorizationVO authVO = new AuthorizationVO();
      authVO.setDataFlow(Constant.DATAFLOW);
      authVO.setMethod(Methods.QUERY);
      authVO.setRequest(request);
      authVO.setParameters(parameters);
      authTokenVO.setAuthorizationVO(authVO);      
      queryVO = new QueryVO(
        null, 
        request, 
        parameters,
        new TaskTypeVO(CDXTaskTypes.DataFlowQuery),
        "Query");
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      byte[] data = bean.query(authTokenVO, queryVO);
      result = new String(data);
      return result;
    }
    catch(Exception exception) {
      log.error("Query Failed", exception);
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      throw fault;
    }
    finally {
      log.info("******** query() - End ********");
    }
  }


  public String execute(String authToken,  String request, String[] parameters) 
      throws RemoteException {
      
    log.warn("Execute is not supported by WQX Node");
    AxisFault fault = FaultHandler.getFault(
      "Client",
      "Execute is not supported by WQX Node",
      FaultErrorCode.FEATURE_UNSUPPORTED);
    throw fault;
  }


  public String nodePing(String hello) throws RemoteException {
    return "Ready";
  }
  
  
  /** <p> returns the list of supported by the node services <p/>
   * @param authToken - authentication token
   * @param service - requested service type
   * @return list of supported services
   * @throws AxisFault
   */
  public String[] getServices(String authToken, String service) throws RemoteException {

    log.info("******** getServices() - Start ********");
    try {
      String[] services = null;
      ServiceTypes serviceType = null;
      serviceType = ServiceTypes.getServiceTypesById(service);
      //if service type is undefined we need to mark the
      if (serviceType == null) {
        serviceType = ServiceTypes.UNKNOWN;
      }
      AuthTokenVO authTokenVO = null;      
      // create authtoken value object
      authTokenVO = new AuthTokenVO(getClientHost(), NodeVersion.V10, authToken);
      // obtain remote interface and call server
      WSServerServiceRemote bean = getServerBean();
      services = bean.getServices(authTokenVO, serviceType);
      return services;
    }
    catch(Exception exception) {
      log.error("getServices Failed", exception);
      // add error code to handle Internal Server Error
      AxisFault fault = FaultHandler.exceptionToAxisFault(exception);
      //audit.failure(null, exception.getMessage(), exception, authTokenVO);
      throw fault;
    }
    finally {
      log.info("******** getServices() - Start ********");
    }
  }


  private WSServerServiceRemote getServerBean() throws CDXException {
    return WSServerServiceUtil.getRemote();
  }
  
}
