package gov.epa.wqx.node;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import java.io.OutputStream;
import java.math.BigDecimal;

import java.sql.ResultSet;
import java.sql.SQLException;

import oracle.jdbc.OracleConnection;
import oracle.jdbc.OraclePreparedStatement;
import oracle.jdbc.OracleResultSet;
import oracle.sql.BLOB;


public class AttachedObject extends SchemaComponent {

  // ---------------------------------------------------------------------
  // Private/Protected Variables
  // ---------------------------------------------------------------------
  
  private AttachedObjectPkg _pkg;
  private String _binaryObjectFileName;
  private String _binaryObjectFileTypeCode;
  private String _filePath;
  
  // ---------------------------------------------------------------------
  // Constructors
  // ---------------------------------------------------------------------

  // Constructor including parent and file path
  public AttachedObject(SchemaComponent parent, String filePath)
      throws WqxFatalException {
      
    super(parent);
    try {
      this.clear();
      _filePath = filePath;
      _pkg = new AttachedObjectPkg(_connection);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }

  // ---------------------------------------------------------------------
  // Get/Set Methods
  // ---------------------------------------------------------------------

  public void setBinaryObjectFileName(String binaryObjectFileName)
      throws WqxFatalException {
      
    _binaryObjectFileName = setValue(binaryObjectFileName);
  }


  public String getBinaryObjectFileName() {
    return _binaryObjectFileName;
  }


  public void setBinaryObjectFileTypeCode(String binaryObjectFileTypeCode)
      throws WqxFatalException {
      
    _binaryObjectFileTypeCode = setValue(binaryObjectFileTypeCode);
  }


  public String getBinaryObjectFileTypeCode() {
    return _binaryObjectFileTypeCode;
  }
  
  
  // ---------------------------------------------------------------------
  // Methods
  // ---------------------------------------------------------------------
 
  private String getTableUidForParent() 
      throws Exception {
    
    int  tableUid;
    String parentType = this.getParent().getSimpleClassName(); 
    if (parentType.equals("Project")) {
      tableUid = Constant.TBL_PROJECT;		
    }
    else if (parentType.equals("MonitoringLocation")) {
      tableUid = Constant.TBL_MONITORING_LOCATION;
    }
    else if (parentType.equals("Activity")) {
      tableUid = Constant.TBL_ACTIVITY;
    }
    else if (parentType.equals("Result")) { 
      tableUid = Constant.TBL_RESULT;
    }
    else {
      throw new Exception("Invalid Parent Type for an Attached Object: " + parentType);
    }  
    return String.valueOf(tableUid);
  }
  
  
  // Save this object's values to the database
  public void save()
      throws Exception {
      
    try {
      BLOB    blob = null;
      if (_dirty) {  
        try {
          _uid = _pkg.insertRow(
            _context,
            _binaryObjectFileName, 
            _binaryObjectFileTypeCode, 
            this.getParent().getUid(), 
            getTableUidForParent());
        }
        catch (Exception e) {
          _transactionLog.logError(e);
        }
        File attachedFile = new File(_filePath + Lib.getFileSeparator() + _binaryObjectFileName);
        // upload the attached file content to the attached_object table
        if (attachedFile.exists()) { 
          try {
            // get blob locator from attached_object table
            OraclePreparedStatement preparedStatement = (OraclePreparedStatement) _connection.prepareStatement(
              "SELECT atobj_content FROM attached_object "
              + "WHERE atobj_uid = ? FOR UPDATE");
            preparedStatement.setString(1,_uid);
            ResultSet resultSet = preparedStatement.executeQuery();
            resultSet.next();
            blob = ((OracleResultSet)resultSet).getBLOB(1);
            // create an output stream to write to blob
            OutputStream outputStream = blob.setBinaryStream(1L);
            // create an input stream to read the file 
            FileInputStream inputStream = new FileInputStream(attachedFile);
            // determine the ideal buffer size to use in writing to the BLOB
            int size = blob.getBufferSize();
            byte[] buffer = new byte[size];
            int length = -1;
            // read the file in chunks and write to blob
            while ((length = inputStream.read(buffer)) != -1)
              outputStream.write(buffer, 0, length);
            inputStream.close();
            outputStream.close();
          }
          catch (Exception e) {
            _transactionLog.logError(e);
          } 
        }
        else {
          _context += ", Looking for file at " + attachedFile.getAbsolutePath();
          // file does not exist
          throw new WqxException(
            "Cannot find the file referenced in BinaryObjectFileName: " + _binaryObjectFileName,
            _context);
        }
      } // if (_dirty)
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      super.save();
    }
  }


  public void deleteAllForParent()
      throws WqxFatalException {
      
    try {
      _pkg.deleteRowsByTableRef(_context, _parent.getUid(), getTableUidForParent());
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }

  
  // Clear this object's values to reuse it for a new one.
  public void clear() 
      throws WqxFatalException {
      
    _binaryObjectFileName = null;
    _binaryObjectFileTypeCode = null;
    super.clear();
  }  

}