package gov.epa.wqx.node;

import gov.epa.wqx.node.OrgAddress;
import gov.epa.wqx.node.OrgAddressPkg;
import gov.epa.wqx.node.OrgElectronicAddress;
import gov.epa.wqx.node.OrgElectronicAddressPkg;
import gov.epa.wqx.node.OrgPhone;
import gov.epa.wqx.node.OrgPhonePkg;
import gov.epa.wqx.node.SchemaComponent;
import gov.epa.wqx.node.ActivityGroupPkg;
import gov.epa.wqx.node.ActivityPkg;
import gov.epa.wqx.node.MonitoringLocationPkg;
import gov.epa.wqx.node.ProjectPkg;

import java.sql.SQLException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;


public class Organization extends SchemaComponent  {

  // ---------------------------------------------------------------------
  // Private/Protected Variables
  // ---------------------------------------------------------------------
  private OrganizationPkg _pkg;
  private ProjectPkg            _pkgProject;
  private MonitoringLocationPkg _pkgMonitoringLocation;
  private ActivityPkg           _pkgActivity;
  private ActivityGroupPkg      _pkgActivityGroup;
  
  private List    _orgAddresses = null;
  private List    _orgElectronicAddresses = null;
  private List    _orgPhones = null;
  private String  _organizationIdentifier;
  private String  _organizationFormalName;
  private String  _organizationDescriptionText;
  private String  _tribalCode;
  
  // ---------------------------------------------------------------------
  // Constructors
  // ---------------------------------------------------------------------
  
  // Constructor w/ transaction log
  public Organization(TransactionLog transactionLog)
      throws WqxFatalException {
      
    super(transactionLog.getConnection());
    try { 
      _orgAddresses = Collections.synchronizedList(new ArrayList());
      _orgElectronicAddresses = Collections.synchronizedList(new ArrayList());
      _orgPhones = Collections.synchronizedList(new ArrayList());
      _transactionLog = transactionLog;
      _parent = null;      
      _pkg = new OrganizationPkg(_connection);
      // these are only used for the delete schema
      _pkgProject = new ProjectPkg(_connection);
      _pkgMonitoringLocation = new MonitoringLocationPkg(_connection);
      _pkgActivity = new ActivityPkg(_connection);
      _pkgActivityGroup = new ActivityGroupPkg(_connection);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }


  // ---------------------------------------------------------------------
  // get/set Methods
  // ---------------------------------------------------------------------
  
  public void setOrganizationIdentifier(String organizationIdentifier)
      throws WqxFatalException {
      
    try {
      _organizationIdentifier = organizationIdentifier;
      _dirty = true;
      // the transaction object also needs the Organization ID
      _transactionLog.setOrganizationId(organizationIdentifier);
      if (_transactionLog.getHeaderOperation().equals("Delete")) {
        // For "Delete" files, the count increments any time we hit an Org ID 
        // (rather than on save, which doesn't occur for "Delete" files)
        _count ++;
        // also get the uid
        _uid = _pkg.getOrgUid(_organizationIdentifier);
      }
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }


  public String getOrganizationIdentifier() {
    return _organizationIdentifier;
  }


  public void setOrganizationFormalName(String organizationFormalName) {
    _organizationFormalName = organizationFormalName;
    _dirty = true;
  }


  public String getOrganizationFormalName() {
    return _organizationFormalName;
  }


  public void setOrganizationDescriptionText(String organizationDescriptionText) {
    _organizationDescriptionText = organizationDescriptionText;
    _dirty = true;
  }


  public String getOrganizationDescriptionText() {
    return _organizationDescriptionText;
  }


  public void setTribalCode(String tribalCode) {
    _tribalCode = tribalCode;
    _dirty = true;
  }


  public String getTribalCode() {
    return _tribalCode;
  }


  // ---------------------------------------------------------------------
  // Methods
  // ---------------------------------------------------------------------
  
  public void addOrgAddress (OrgAddress orgAddress)
      throws WqxFatalException {
      
    try {
      _orgAddresses.add(orgAddress);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }
  
  
  public void addOrgElectronicAddress (OrgElectronicAddress orgElectronicAddress)
      throws WqxFatalException {
      
    try {
      _orgElectronicAddresses.add(orgElectronicAddress);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }
  
  
  public void addOrgPhone (OrgPhone orgPhone)
      throws WqxFatalException {
      
    try {
      _orgPhones.add(orgPhone);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }
  
  
  public void deleteProject(String projectId)
      throws WqxFatalException {
      
    try {
      // if transaction isn't saved yet, then save it
      if (_transactionLog.getCount() < 1) {
        _transactionLog.setOrganizationId(_organizationIdentifier);
        _transactionLog.save();
      }
      _pkgProject.deleteRowByPrjId("ProjectIdentifier/Line " + this.getLineNo(), projectId);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      _transactionLog.performCommitRollback();
    }
  }
  
  
  public void deleteMonitoringLocation(String monitoringLocationId)
      throws WqxFatalException {
      
    try {
      // if transaction isn't saved yet, then save it
      if (_transactionLog.getCount() < 1) {
        _transactionLog.setOrganizationId(_organizationIdentifier);
        _transactionLog.save();
      }
      _pkgMonitoringLocation.deleteRowByMlocId("MonitoringLocationIdentifier/Line " + this.getLineNo(), monitoringLocationId);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      _transactionLog.performCommitRollback();
    }
  }
  
  
  public void deleteActivity(String activityId)
      throws WqxFatalException {
      
    try {
      // if transaction isn't saved yet, then save it
      if (_transactionLog.getCount() < 1) {
        _transactionLog.setOrganizationId(_organizationIdentifier);
        _transactionLog.save();
      }
      _pkgActivity.deleteRowByActId("ActivityIdentifier/Line " + this.getLineNo(), activityId);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      _transactionLog.performCommitRollback();
    }
  }
  
  
  public void deleteActivityGroup(String activityGroupId)
      throws WqxFatalException {
      
    try {
      // if transaction isn't saved yet, then save it
      if (_transactionLog.getCount() < 1) {
        _transactionLog.setOrganizationId(_organizationIdentifier);
        _transactionLog.save();
      }
      _pkgActivityGroup.deleteRowByActgrpId("ActivityGroupIdentifier/Line " + this.getLineNo(), activityGroupId);
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      _transactionLog.performCommitRollback();
    }
  }
  
  
  // Save this object's values to the database
  public void save() 
      throws Exception {
    try {
      // transaction is saved right before saving the organization
      _transactionLog.save();
      // set param values
      try {
        _uid = _pkg.updateRow(
          _context,
          _organizationIdentifier,
          _organizationFormalName,
          _organizationDescriptionText,
          _tribalCode);
      }
      catch (SQLException e) {
        _transactionLog.logError(e); 
      }    
      boolean first = true;      
      // save each Address (if any)
      if (_orgAddresses.size() > 0) {
        OrgAddress orgAddress;
        ListIterator iterator = _orgAddresses.listIterator();
        while (iterator.hasNext()) {
          orgAddress = (OrgAddress) iterator.next();
          if (first) {
            first = false;
            // delete all existing addresses if at least one tag found in submission file
            orgAddress.deleteRowsByOrgUid(orgAddress.getContext(), this.getUid());    
          }
          orgAddress.save();
        }
      }
      
      first = true;
      // save each Electronic Address (if any)
      if (_orgElectronicAddresses.size() > 0) {
        OrgElectronicAddress orgElectronicAddress;
        ListIterator iterator = _orgElectronicAddresses.listIterator();
        while (iterator.hasNext()) {
          orgElectronicAddress = (OrgElectronicAddress) iterator.next();
          if (first) {
            first = false;
            // delete all existing electronic addresses if at least one tag found in submission file
            orgElectronicAddress.deleteRowsByOrgUid(orgElectronicAddress.getContext(), this.getUid());    
          }
          orgElectronicAddress.save();
        }
      }
      
      first = true;
      // save each phone (if any)
      if (_orgPhones.size() > 0) {
        OrgPhone orgPhone;
        ListIterator iterator = _orgPhones.listIterator();
        while (iterator.hasNext()) {
          orgPhone = (OrgPhone) iterator.next();
          if (first) {
            first = false;
            // delete all existing phones if at least one tag found in submission file
            orgPhone.deleteRowsByOrgUid(orgPhone.getContext(), this.getUid());    
          }
          orgPhone.save();
        }
      }
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      super.save();
    }
  }

  // Clear this object's values to reuse it for a new one.
  public void clear()
      throws WqxFatalException {
      
    try {
      _organizationIdentifier = null;
      _organizationFormalName = null;
      _organizationDescriptionText = null;
      _tribalCode = null;
      _orgAddresses.clear();
      _orgElectronicAddresses.clear();
      _orgPhones.clear();
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
    finally {
      super.clear();
    }
  }

}