package gov.epa.wqx.node;

import gov.epa.wqx.node.TransactionLog;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;

import java.text.ParsePosition;
import java.text.SimpleDateFormat;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import oracle.jdbc.OracleConnection;

public abstract class SchemaComponent {  

  // ---------------------------------------------------------------------
  // Private/Protected Variables
  // ---------------------------------------------------------------------
  protected boolean _dirty = false;
  protected boolean _transactionalBoundary = false;
  protected String _uid = null;
  protected long _count = 0;
  protected OracleConnection _connection; 
  protected SchemaComponent _parent;
  protected String _lineNo = null;
  protected List _attachedObjects; 
  protected TransactionLog _transactionLog;
  protected SchemaComponent _currentObject;
  protected String _context;
  
  // ---------------------------------------------------------------------
  // Constructors
  // ---------------------------------------------------------------------

  // Constructor w/ connection 
  protected  SchemaComponent(OracleConnection connection) {
    _connection = connection;
    _attachedObjects = Collections.synchronizedList(new ArrayList());
  }
  

  // Constructor w/ parent
  public SchemaComponent(SchemaComponent parent) {
    this(parent.getConnection());
    _parent = parent;
    _transactionLog = parent.getTransactionLog();
  }


  // ---------------------------------------------------------------------
  // get/set Methods
  // ---------------------------------------------------------------------
  
  protected void setUid(String uId) {
    _uid = uId;  
  }
  
  public String getUid() {
    return _uid;
  }


  public long getCount() {
    return _count;
  }
  
  
  protected void setParent(SchemaComponent parent) {
    _parent = parent;
  }


  public SchemaComponent getParent() {
    return _parent;
  }


  public void setLineNo(String lineNo) 
      throws WqxFatalException {
      
    try {
      _lineNo = lineNo;
      _context = this.getSimpleClassName() + " (Line " + _lineNo + ")";
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }


  public String getLineNo() {
    return _lineNo;
  }

  
  public TransactionLog getTransactionLog() {
    return _transactionLog;
  }

  
  public OracleConnection getConnection() {
    return _connection;
  }

  
  public void setCurrentObject(SchemaComponent currentObject) 
      throws WqxFatalException {
    if (_currentObject instanceof Activity && currentObject instanceof Result) {
      // Everytime we transition from an Activity to a Result 
      // (meaning the Result tag was round) we delete all child results before proceeding
      ((Result) (currentObject)).deleteAllForParentActivity();
    }
    _currentObject = currentObject;
  }


  public SchemaComponent getCurrentObject() {
    return _currentObject;
  }


  public void setContext(String context) {
    _context = context;
  }


  public String getContext() {
    return _context;
  }


  public boolean isTransactionalBoundary() {
    return _transactionalBoundary;
  }
  
  
  // Get a simple type name for this object (just the class name)
  public String getSimpleClassName () {
    String thisType = this.getClass().getName();
    return (thisType.substring(thisType.lastIndexOf(".")+1));
  }


  // ---------------------------------------------------------------------
  // Methods
  // ---------------------------------------------------------------------
  
  // standard method to call for all set methods
  // sets the dirty flag and trims the value
  protected String setValue(String value)
      throws WqxFatalException {
      
    try {
      _dirty = true;
      return value;
      /* this isn't needed anymore, since the SAX Handle trims, etc
       * but this still might be worth bringing back some day if these objects are used
       * outside of the parse and load (e.g. XML Gen. Tool)
       return (value == null) ? "" : value.trim(); */
    }
    catch (Exception e) {
      _transactionLog.logError(e);
      return null;
    }
  }
  
  
  public void addAttachedObject(AttachedObject attachedObject)
      throws WqxFatalException {
      
    try {
      _attachedObjects.add(attachedObject);  
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  }
  
  
  // Clear this object's values to reuse it for a new one.
  public void clear() 
      throws WqxFatalException {
      
    if (_transactionalBoundary) {
      _transactionLog.performCommitRollback();
    }
    _dirty = false;
    _uid = null;
    _lineNo = null;
    _attachedObjects.clear();
  }


  // Save this object's values to the database
  public void save() 
      throws Exception {
    if (_dirty) {
      _count += 1;
      _dirty = false;
    }
    // Save Attached Objects
    // there's a chance these could exist even on an object that isn't dirty,
    // so check for them independent of this flag
    try {
      if (_attachedObjects.size() > 0) {
        boolean firstObject = true;
        AttachedObject blob = null;
        ListIterator blobIterator = _attachedObjects.listIterator();
        while (blobIterator.hasNext()) {
          blob = (AttachedObject) blobIterator.next();
          if (firstObject) {
            // delete all existing attached objects and then add any new ones
            blob.deleteAllForParent();
            firstObject = false;
          }
          blob.save();
        }
      }
    }
    catch (Exception e) {
      _transactionLog.logError(e);
    }
  
  }

}
 
