package gov.epa.wqx.node;

import gov.epa.wqx.node.SchemaComponent;
import gov.epa.wqx.node.TransactionLogPkg;
import gov.epa.wqx.node.WqxFatalException;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Timestamp;

import oracle.jdbc.OracleConnection;


public class TransactionLog extends SchemaComponent  {
  
  // ---------------------------------------------------------------------
  // Private/Protected Variables
  // ---------------------------------------------------------------------
  private TransactionLogPkg     _pkg;
  
  private int                 _status;
  private int                 _errorCount;
  private WqxFatalException   _fatalException;
  private Exception           _lastException;
  
  // transaction log properties
  private String    _organizationId;
  private String    _transactionId;
  private String    _documentId;
  private String    _userNaasId;
  private String    _comments;
  private int       _type;
  private Timestamp _startTime;
  private Timestamp _endTime;
  
  // transaction header properties
  private Timestamp   _headerCreationTime;
  private String      _headerCreationTimeString;
  private String      _headerId;
  private String      _headerOperation;
  private String      _headerAuthor;
  private String      _headerOrganization;
  private String      _headerTitle;
  private String      _headerComment;
  private String      _headerContactInfo;
  private String      _headerSchemaRef;
  
  // ---------------------------------------------------------------------
  // Constructors
  // ---------------------------------------------------------------------
  
  // Constructor w/ connection
  public TransactionLog(OracleConnection oracleConnection)
      throws WqxFatalException {
      
    super(oracleConnection);
    try {
      _transactionLog = this;
      _status = Constant.TRSTA_NOT_STARTED;
      _pkg = new TransactionLogPkg(_connection);
    }
    catch (Exception e) {
      logError(e);
    }
    finally {
      this.clear();
    }
  }

  // ---------------------------------------------------------------------
  // get/set Methods
  // ---------------------------------------------------------------------
  
  public int getStatus() {
    return _status;
  }


  public void setOrganizationId(String organizationId)
      throws WqxFatalException {
      
    _organizationId = setValue(organizationId);
  }

  public String getOrganizationId() {
    return _organizationId;
  }

  
  public void setTransactionId(String transactionId)
      throws WqxFatalException {
      
    _transactionId = setValue(transactionId);
    Lib.log.info("Transaction ID=" + transactionId);
  }


  public String getTransactionId() {
    return _transactionId;
  }
  

  public void setDocumentId(String documentId)
      throws WqxFatalException {
      
    _documentId = setValue(documentId);
  }


  public String getDocumentId() {
    return _documentId;
  }
  

  public void setUserNaasId(String userNaasId)
      throws WqxFatalException {
      
    _userNaasId = setValue(userNaasId);
    Lib.log.info("User NAAS ID=" + userNaasId);
  }


  public String getUserNaasId() {
    return _userNaasId;
  }


  public void setComments(String comments)
      throws WqxFatalException {
      
    _comments = setValue(comments);
  }


  public String getComments() {
    return _comments;
  }


  public void setType(int type) {
    _type = type;
  }


  public int getType() {
    return _type;
  }


  public void setStartTime(Timestamp startTime) {
    _startTime = startTime;
    _dirty = true;
  }


  public Timestamp getStartTime() {
    return _startTime;
  }


  public Timestamp getEndTime() {
    return _endTime;
  }


  public void setHeaderCreationTime(String headerCreationTimeString)
      throws WqxFatalException {
      
    try {
      _headerCreationTimeString = headerCreationTimeString;
      _dirty = true;
      if (headerCreationTimeString != null) {
        _headerCreationTime = Lib.xmlToTimestamp(headerCreationTimeString);  
      }
      else {
        _headerCreationTime = null;
      }
    }
    catch (Exception e) {
      Lib.log.trace("Trace", e);
      logError(e);
    }
  }


  public String getHeaderCreationTime() {
    return _headerCreationTimeString;
  }


  public void setHeaderId(String headerId)
      throws WqxFatalException {
      
    _headerId = setValue(headerId);
  }


  public String getHeaderId() {
    return _headerId;
  }


  public void setHeaderOperation(String headerOperation)
      throws WqxFatalException {
      
    _headerOperation = setValue(headerOperation);
  }


  public String getHeaderOperation() {
    return _headerOperation;
  }


  public void setHeaderAuthor(String headerAuthor)
      throws WqxFatalException {
      
    _headerAuthor = setValue(headerAuthor);
  }


  public String getHeaderAuthor() {
    return _headerAuthor;
  }


  public void setHeaderOrganization(String headerOrganization)
      throws WqxFatalException {
      
    _headerOrganization = setValue(headerOrganization);
  }


  public String getHeaderOrganization() {
    return _headerOrganization;
  }


  public void setHeaderTitle(String headerTitle)
      throws WqxFatalException {
      
    _headerTitle = setValue(headerTitle);
  }


  public String getHeaderTitle() {
    return _headerTitle;
  }


  public void setHeaderComment(String headerComment)
      throws WqxFatalException {
      
    _headerComment = setValue(headerComment);
  }


  public String getHeaderComment() {
    return _headerComment;
  }


  public void setHeaderContactInfo(String headerContactInfo)
      throws WqxFatalException {
      
    _headerContactInfo = setValue(headerContactInfo);
  }


  public String getHeaderContactInfo() {
    return _headerContactInfo;
  }


  public void setHeaderSchemaRef(String headerSchemaRef)
      throws WqxFatalException {
      
    _headerSchemaRef = setValue(headerSchemaRef);
  }


  public String getHeaderSchemaRef() {
    return _headerSchemaRef;
  }


  public WqxFatalException getFatalException() {
    return _fatalException;
  }
  
  
  // ---------------------------------------------------------------------
  // Methods
  // ---------------------------------------------------------------------
  
  private void throwFatalException(String message, Throwable throwable, String context) 
      throws WqxFatalException {
  
    WqxFatalException exception = new WqxFatalException(message, throwable, context);
    if (_fatalException == null) {
      _fatalException = exception;
    }        
    throw exception;    
  }
  
  // Standard Error Logging for exceptions thrown in Parse & Load Software
  public void logError (Exception exception) 
      throws WqxFatalException {
  
    String context = null;
    boolean isLogged = false;
    try {
      if (exception instanceof WqxException) {
        isLogged = ((WqxException) exception).isLogged();
        context = ((WqxException) exception).getContext();
        ((WqxException) exception).setLogged(true);
      }
      else if (exception.equals(_lastException)) {
        // this is the same error as the last time logError was called
        isLogged = true;
      }
      else if (exception.getMessage().indexOf("ORA-20") > 0 && exception.getMessage().indexOf("ORA-20998") == -1) {
        isLogged = true;
      }
      _lastException = exception;
      if (exception instanceof WqxFatalException) {
        // fatal errors need to cascade out
        throw (WqxFatalException) exception;
      }
      if (exception.getMessage().indexOf("ORA-20998") > 0) {
        // this is an error trying to log an error (in the database)
        throwFatalException(
          exception.getMessage(), 
          exception,
          context);
      }
      if (!isLogged) {
        if (context == null && _currentObject != null) {
          // get context from current object (Parse and Load only)
          context = _currentObject.getContext();
        }
        if (_count > 0) {
          // transaction has been started - safe to log error in the database
          try {
            String source = Lib.getSimplifiedStackTrace(exception, true);
            _pkg.insertError(exception.getMessage(), source, context);
          }        
          catch (SQLException e) {
            throwFatalException(
              "Error trying to log an existing error: " + e.getMessage(),  
              exception,
              context);
          }
        }
        else {
          // transaction not saved, so it can't be logged to database
          throwFatalException(
            "Error while trying to create a transaction or process the document header", 
            exception,
            context);
        } 
      }
    }
    catch (Exception e) {
      _lastException = e;
      if (e instanceof WqxException) {
        ((WqxException) e).setLogged(true);
      } 
      if (!isLogged) {
        Lib.log.error(e);
        Lib.log.error("Stack Trace: ");
        Lib.log.error(Lib.getSimplifiedStackTrace(e, false));
        Throwable throwable = e.getCause();
        while (throwable != null) {
          Lib.log.error("  Cause: " + throwable.getMessage());
          Lib.log.error("  Source: " + Lib.getSimplifiedStackTrace((Exception) throwable, true));
          if (!(throwable instanceof WqxException)) {
            break;  
          }
          else {
            throwable = throwable.getCause();
          }
        }
      }
      if (e instanceof WqxFatalException) {
        throw (WqxFatalException) e;
      }
      else {
        throwFatalException(
          "Error trying to log an existing error: " + e.getMessage(),  
          exception,
          context);
      }
    }
  }
  
  
  // Standard Error Logging for exceptions thrown in Parse & Load Software
  public void logAndThrowException (Exception exception)
      throws WqxException {
      
    logError(exception);
    if (exception instanceof WqxException) {
      throw (WqxException) exception;
    }
    else {
      WqxException wqxException = new WqxException(exception);
      wqxException.setLogged(true);
      throw wqxException;
    }
  }
  
  
  // Save this object's values to the database
  public void save()
      throws Exception {
    try {
      _uid = _pkg.startTransaction(
        "Header (Line " + _lineNo + ")", 
        _organizationId,
        _type,
        _userNaasId,
        _transactionId,
        _documentId, 
        _comments, 
        _headerAuthor,
        _headerComment,
        _headerContactInfo,
        _headerCreationTime,
        _headerId, 
        _headerOperation,
        _headerOrganization,
        _headerSchemaRef,
        _headerTitle, 
        _startTime); 
      _status = Constant.TRSTA_PENDING;
      
    }
    catch (Exception e) {
      logError(e);
    }
    finally {
      super.save();
    }
  }


  // Clear this object's values to reuse it for a new one.
  public void clear()
      throws WqxFatalException {
      
    try {
      _organizationId = null;
      _errorCount = 0;
      /* these properties are set prior to parsing, so don't clear them
      _transactionId = null;
      _documentId = null; */ 
      _userNaasId = null;
      _headerAuthor = null;
      _headerComment = null;
      _headerContactInfo = null;
      _headerCreationTimeString = null;
      _headerCreationTime = null;
      _headerId = null;
      _headerOperation = null;
      _headerOrganization = null;
      _headerSchemaRef = null;
      _headerTitle = null;
      _status = Constant.TRSTA_NOT_STARTED;
    }
    catch (Exception e) {
      logError(e);
    }
    finally {
      super.clear();
    }
  }


  // end the transaction
  public void end(Timestamp endTime)
      throws WqxFatalException {
      
    try { 
      _endTime = endTime;
      _status = _pkg.endTransaction("", _endTime);
      _dirty = false;
    }
    catch (SQLException e) {
      _status = Constant.TRSTA_FAILED;
      logError(e); 
    }
  }

  
  // Determine if any errors have been logged since the last call
  // If so, then rollback, otherwise commit
  public void performCommitRollback()
      throws WqxFatalException {
      
    try {
      // if we haven't started the transaction yet, then there's nothing to do
      if (_uid != null) {
        int _previousErrorCount = _errorCount;
        _errorCount = _pkg.getErrorCount();
        if (_errorCount > _previousErrorCount) {
          _connection.rollback();
          _pkg.updateAuditPending("N");
        }
        else {
          _connection.commit();
          _pkg.updateAuditPending("Y");
        }
      }
    }
    catch (Exception e) {
      logError(e);
    }
  }

}