package gov.epa.wqx.node;

import gov.epa.wqx.node.*;

import java.io.*;

import java.sql.*;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import oracle.jdbc.OracleConnection;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public class WqxSaxHandler extends DefaultHandler {

  // strings to hold the element names and values
  private String            _elementGroupName = "";
  private String            _elementContextName = "";
  private String            _elementName = "";
  private String            _elementValue;
	private String            _attributeName;
	private String            _attributeValue;
  
  private String            _alternateMonitoringLocationIdentifier;
  private OracleConnection  _connection;
  private Locator           _locator;
  private String            _processingReportErrorMessage;
  private String            _processingReportErrorContext;
  private String            _docPath;
  private String            _docName;
  
  // collection used to map element names with the objects used to persist them to the DB
  private Map               _groupElements;
  // collection used to determine if a specific element name is needed for additional 
  // context information while parsing
  private Map               _contextElements;
  
  // need this to get the processing report
  XmlQueryPkg   _pkgQuery;
  
  // objects needed to hold data from XML submission file
  private TransactionLog              _transactionLog;
  private Organization                _organization;
  private OrgElectronicAddress        _orgElectronicAddress;
  private OrgPhone                    _orgPhone;
  private OrgAddress                  _orgAddress;
  private Project                     _project;
  private AttachedObject              _attachedObject;
  private MonitoringLocation          _monitoringLocation;
  private Activity                    _activity;
  private ActivityGroup               _activityGroup;
  private Result                      _result;
  private ResultDetectQuantLimit      _resultDetectQuantLimit;
  private ResultLabSamplePrep         _resultLabSamplePrep;
  
  public String getAttachedFilesPath() {
    return _docPath;
  }
  
  public void setDocumentLocator(Locator locator) {
    _locator = locator;  
  }
  		
	// constructor	
  public WqxSaxHandler(
      String docPath,
      String docName,
      String transactionId) 
      
      throws SAXException {
      
    super();
    try {
      _connection = Lib.getWqxConnection();
      _docPath = docPath;
      _docName = docName;
      _pkgQuery = new XmlQueryPkg(_connection);
      // create object hierarchy used for parsing and loading
      // Only one of each of these objects are created
      // The objects are reused each time we encounter a new one in the file
      _transactionLog = new TransactionLog(_connection);
      _transactionLog.setStartTime(new Timestamp(System.currentTimeMillis()));
      _transactionLog.setTransactionId(transactionId);
      _transactionLog.setComments("FileName=" + docName);
      _transactionLog.setType(Constant.TRTYP_SUBMIT);
      _organization = new Organization(_transactionLog);
      _project = new Project(_organization);
      _monitoringLocation = new MonitoringLocation(_organization);
      _activity = new Activity(_organization);
      _activityGroup = new ActivityGroup(_organization);
      _result = new Result(_activity);
      _resultDetectQuantLimit = new ResultDetectQuantLimit(_result);
      _resultLabSamplePrep = new ResultLabSamplePrep(_result);
      // create a map of element names (in the xml file) to their corresponding objects 
      _groupElements = Collections.synchronizedMap(new HashMap());
      _groupElements.put("Document",_transactionLog);
      _groupElements.put("OrganizationDelete",_organization);
      _groupElements.put("OrganizationDescription",_organization);
      _groupElements.put("Project",_project);
      _groupElements.put("MonitoringLocation",_monitoringLocation);
      _groupElements.put("Activity",_activity);
      _groupElements.put("Result",_result);
      _groupElements.put("ResultDetectionQuantitationLimit",_resultDetectQuantLimit);
      _groupElements.put("ResultLabSamplePrep",_resultLabSamplePrep);
      _groupElements.put("ActivityGroup",_activityGroup);
      _groupElements.put("ElectronicAddress",_organization);
      _groupElements.put("Telephonic",_organization);
      _groupElements.put("OrganizationAddress",_organization);
      _groupElements.put("AttachedBinaryObject",_organization);
      _contextElements = Collections.synchronizedMap(new HashMap());
      _contextElements.put("ActivityDepthHeightMeasure","ActivityDepthHeightMeasure");
      _contextElements.put("ActivityBottomDepthHeightMeasure","ActivityBottomDepthHeightMeasure");
      _contextElements.put("ActivityTopDepthHeightMeasure","ActivityTopDepthHeightMeasure");
      _contextElements.put("DetectionQuantitationLimitMeasure","DetectionQuantitationLimitMeasure");
      _contextElements.put("ResultDepthHeightMeasure","ResultDepthHeightMeasure");
      _contextElements.put("ResultMeasure","ResultMeasure");
      _contextElements.put("VerticalMeasure","VerticalMeasure");
      _contextElements.put("LabSamplePreparationMethod","LabSamplePreparationMethod");
      _contextElements.put("SampleCollectionMethod","SampleCollectionMethod");
      _contextElements.put("SamplePreparationMethod","SamplePreparationMethod");
      _contextElements.put("ResultAnalyticalMethod","ResultAnalyticalMethod");
      _contextElements.put("ActivityStartTime","ActivityStartTime");
      _contextElements.put("ActivityEndTime","ActivityEndTime");
      _contextElements.put("AnalysisStartTime","AnalysisStartTime");
      _contextElements.put("AnalysisEndTime","AnalysisEndTime");
      _contextElements.put("PreparationStartTime","PreparationStartTime");
      _contextElements.put("PreparationEndTime","PreparationEndTime");
      _contextElements.put("AlternateMonitoringLocationIdentity","AlternateMonitoringLocationIdentity");
      // these objects are only created when needed and there is one for each one found in submission file
      _orgElectronicAddress = null;
      _orgPhone = null;
      _orgAddress = null;
      _attachedObject = null;
    } 
    catch (Exception e) {
      Lib.log.trace("SAX Trace: ", e);
      try {
        _transactionLog.logError(e);
      }
      catch (WqxFatalException fatalException) {
        throw new SAXException(fatalException);
      }
    }
    
  }
  
  
  public void done() {
    try {
      if (_connection != null) {
        _connection.close();
        Lib.log.debug("Oracle connection closed");
      }
    }
    catch (Exception e) {
      Lib.log.error("Error trying to close Oracle connection", e);
    }
  }
                 
  
  // ************************************************************************************
  // Callback Methods (called by parser upon certain specific events) 
  // ************************************************************************************
  
  // called when the parser starts reading the xml document  
  public void startDocument () {
    Lib.log.info("SAX Parser startDocument");
  }
  
  
  // called when the parser reaches the end of the xml document
  public void endDocument () 
      throws SAXException {
      
    try {
      Lib.log.info("SAX Parser endDocument");
      if (_transactionLog != null) {
        if (_transactionLog.getCurrentObject() != null 
            && _transactionLog.getHeaderOperation().equalsIgnoreCase("Update-Insert")) {
          // save the last object created
          _transactionLog.getCurrentObject().save();
          // Results have some rules that have to be checked after all children are saved
          // calling clear will check them.
          _result.clear();          
        }
      }
      // check for any errors and perform final commit/rollback 
      _transactionLog.performCommitRollback();
      if (_transactionLog.getCount() > 0) {
        // end transaction
        _transactionLog.end(new Timestamp(System.currentTimeMillis()));
      }  
    }
    catch (Exception e) {
      try {
        _transactionLog.logError(e);
      }
      catch (WqxFatalException fatalException) {
        throw new SAXException(fatalException);
      }
    }
  }

  // called when the parser encounters a new element in the xml document
  public void startElement (
      String      uri, 
      String      localName,
      String      qualifiedName, 
      Attributes  attributes) 
      
      throws SAXException {    
      
    //Lib.log.debug("<" + localName + ">");
    _elementValue = null;    
    // this is a new element in the schema 
    // Save the name so we know which Element the Value relates to (in endElement)
    _elementName = localName;
    // certain elements are significant and require some action (see below)
    try {
      String lineNo = String.valueOf(_locator.getLineNumber());
      // **************************************************************************
      // This first group of tags represent the start of a new group
      // Typically this relates to a new object (in java) and new table in the database
      // we save the previous object and then clear the new one (so it can be repopulated)
      // **************************************************************************
      SchemaComponent _tempObject;
      _tempObject = (SchemaComponent) _groupElements.get(_elementName);
      if (_tempObject != null) {
        // this is a group element
        boolean saveCurrent = true;
        if (_elementName.equals("OrganizationDelete") 
            || _elementName.equals("OrganizationDescription")
            || _elementName.equals("Document")) {
          // the current object is the transaction, but it can't be saved until we have 
          // the Organization ID
          saveCurrent = false;
          if (_organization.getCount() > 0) {
            throw new SAXException("Only one organization allowed per submission file");
          }
        }
        // these four objects are added to a collection (on the current object)
        if (_elementName.equals("ElectronicAddress")) {
          // add electronic address object to organization
          _orgElectronicAddress = new OrgElectronicAddress(_organization);
          _organization.addOrgElectronicAddress(_orgElectronicAddress);
          _orgElectronicAddress.setLineNo(lineNo);
        }
        else if (_elementName.equals("Telephonic")) {
          // add telephonic object to organization
          _orgPhone = new OrgPhone(_organization);
          _organization.addOrgPhone(_orgPhone);
          _orgPhone.setLineNo(lineNo);
        }
        else if (_elementName.equals("OrganizationAddress")) {
          // add address object to organization
          _orgAddress = new OrgAddress(_organization);
          _organization.addOrgAddress(_orgAddress);								
          _orgAddress.setLineNo(lineNo);
        }
        else if (_elementName.equals("AttachedBinaryObject")) {
          // these objects are saved when their parents are saved
          // add this attached object to the current object (its parent) 
          _attachedObject = new AttachedObject(_transactionLog.getCurrentObject(), _docPath);
          _attachedObject.setLineNo(lineNo);
          _transactionLog.getCurrentObject().addAttachedObject(_attachedObject);
        }
        // the rest of these objects are reused, so save the previous one and clear the new one
        else {
          if (saveCurrent) {
            _transactionLog.getCurrentObject().save();
          } 
          _elementGroupName = _elementName;
          _transactionLog.setCurrentObject(_tempObject);
          _tempObject.clear();
          _tempObject.setLineNo(lineNo);
          _tempObject = null;
        }
      }
      
      // **************************************************************************
      // These tags are needed for context information for lower level tags that
      // have the same name in various parts of the file
      // (make sure we reset these in EndElement method
      // **************************************************************************
      else if (_contextElements.containsKey(_elementName)) {
        _elementContextName = _elementName;
      }
    
      // ********************* READ THROUGH THE ATTRIBUTES, IF ANY ***************************
      int numAttributes = attributes.getLength();
      if (numAttributes > 0) {
        for (int i = 0; i < numAttributes; i++) {
          _attributeValue = attributes.getValue(i).trim();
          _attributeName = attributes.getLocalName(i);        
          if (_elementGroupName.equals("Document")) {
            if (_attributeName.equals("Id")) {
              _transactionLog.setHeaderId(_attributeValue);
            }
            else if (_attributeName.equals("Operation")) {
              _transactionLog.setHeaderOperation(_attributeValue);
            }
            else if (_attributeName.equals("schemaLocation")) {
              // I'm not sure this is the right place to look for the "Schema Reference" 
              try {
                int idxSchemaRef = _attributeValue.indexOf("WQX_WQX_");
                if (idxSchemaRef > 0) {
                  _transactionLog.setHeaderSchemaRef(_attributeValue.substring(idxSchemaRef));
                }
              }
              catch (Exception e) {
                // ignore exceptions
              }
            }
          }
        }
      }
    }
    catch (Exception e) {
      try {
        _transactionLog.logError(e);
      }
      catch (WqxFatalException fatalException) {
        throw new SAXException(fatalException);
      }    
    }
  }
    

  // called when the parser reads the element value from the xml document
  // May be called repeatedly (passing part of the element value each time) 
  public void characters (char ch[], int  start, int  length)
      throws SAXException {
    
    try {
      if (_elementValue == null) { 
        _elementValue = new String();
      }
      _elementValue += new String(ch, start, length);
    }
    catch (Exception e) {
      try {
        _transactionLog.logError(e);
      }
      catch (WqxFatalException fatalException) {
        throw new SAXException(fatalException);
      }
    }
  }


  // called when the parser reaches the end of an element in the xml document
  public void endElement (
      String uri, 
      String localName, 
      String qualifiedName) 
      
      throws SAXException {
    
    //Lib.log.debug(_elementValue + "</" + localName + ">");
    _elementName = localName;
    // trim element value and use empty string instead of null for convenience
    if (_elementValue != null) {
      _elementValue = _elementValue.trim();
    }
    else {
      _elementValue = "";
    }
    try {
      if (_elementName.equals("MeasureValue")) {
        if (_elementContextName.equals("ActivityDepthHeightMeasure")) {
          _activity.setActivityDepthHeightMeasureValue(_elementValue);
        }
        else if (_elementContextName.equals("ActivityTopDepthHeightMeasure")) {
          _activity.setActivityTopDepthHeightMeasureValue(_elementValue);
        }    
        else if (_elementContextName.equals("ActivityBottomDepthHeightMeasure")) {
          _activity.setActivityBottomDepthHeightMeasureValue(_elementValue);
        }  
        else if (_elementContextName.equals("DetectionQuantitationLimitMeasure")) {
          _resultDetectQuantLimit.setDetectionQuantitationLimitMeasureValue(_elementValue);            
        }  
        else if (_elementContextName.equals("ResultDepthHeightMeasure")) {
          _result.setResultDepthHeightMeasureValue(_elementValue);
        }
        else if (_elementContextName.equals("VerticalMeasure")) {
          _monitoringLocation.setVerticalMeasureValue(_elementValue);
        }  
      } 
      else if (_elementName.equals("MeasureUnitCode")) {
        if (_elementContextName.equals("ActivityDepthHeightMeasure")) {
          _activity.setActivityDepthHeightMeasureUnitCode(_elementValue);
        }
        else if (_elementContextName.equals("ActivityTopDepthHeightMeasure")) {
          _activity.setActivityTopDepthHeightMeasureUnitCode(_elementValue);
        }    
        else if (_elementContextName.equals("ActivityBottomDepthHeightMeasure")) {
          _activity.setActivityBottomDepthHeightMeasureUnitCode(_elementValue);
        }  
        else if (_elementContextName.equals("DetectionQuantitationLimitMeasure")) {
          _resultDetectQuantLimit.setDetectionQuantitationLimitMeasureUnitCode(_elementValue);
        }  
        else if (_elementContextName.equals("ResultDepthHeightMeasure")) {
          _result.setResultDepthHeightMeasureUnitCode(_elementValue);
        }
        else if (_elementContextName.equals("ResultMeasure")) {
          _result.setResultMeasureUnitCode(_elementValue);
        }  
        else if (_elementContextName.equals("VerticalMeasure")) {
          _monitoringLocation.setVerticalMeasureUnitCode(_elementValue);
        }  
      }
      else if (_elementName.equals("Time")) {
        if (_elementContextName.equals("ActivityStartTime")) {
          _activity.setActivityStartTime(_elementValue);
        }
        else if (_elementContextName.equals("ActivityEndTime")) {
          _activity.setActivityEndTime(_elementValue);
        }
        else if (_elementContextName.equals("AnalysisStartTime")) {
          _result.setAnalysisStartTime(_elementValue);
        }  
        else if (_elementContextName.equals("AnalysisEndTime")) {
          _result.setAnalysisEndTime(_elementValue);
        }  
        else if (_elementContextName.equals("PreparationStartTime")) {
          _resultLabSamplePrep.setPreparationStartTime(_elementValue);
        }  
        else if (_elementContextName.equals("PreparationEndTime")) {
          _resultLabSamplePrep.setPreparationEndTime(_elementValue);
        }
      }
      else if (_elementName.equals("TimeZoneCode")) {
        if (_elementContextName.equals("ActivityStartTime")) {
          _activity.setActivityStartTimeZoneCode(_elementValue);
        }
        else if (_elementContextName.equals("ActivityEndTime")) {
          _activity.setActivityEndTimeZoneCode(_elementValue);
        }
        else if (_elementContextName.equals("AnalysisStartTime")) {
          _result.setAnalysisStartTimeZoneCode(_elementValue);
        }  
        else if (_elementContextName.equals("AnalysisEndTime")) {
          _result.setAnalysisEndTimeZoneCode(_elementValue);
        }  
        else if (_elementContextName.equals("PreparationStartTime")) {
          _resultLabSamplePrep.setPreparationStartTimeZoneCode(_elementValue);
        }  
        else if (_elementContextName.equals("PreparationEndTime")) {
          _resultLabSamplePrep.setPreparationEndTimeZoneCode(_elementValue);
        }
      }        
      else if (_elementName.equals("MethodIdentifier")) {
        if (_elementContextName.equals("LabSamplePreparationMethod")) {
          _resultLabSamplePrep.setLabSamplePreparationMethodIdentifier(_elementValue);
        }
        else if (_elementContextName.equals("SampleCollectionMethod")) {
          _activity.setSampleCollectionMethodIdentifier(_elementValue);
        }
        else if (_elementContextName.equals("SamplePreparationMethod")) {
          _activity.setSamplePreparationMethodIdentifier(_elementValue);
        }
        else if (_elementContextName.equals("ResultAnalyticalMethod")) {
          _result.setResultAnalyticalMethodIdentifier(_elementValue);
        }    
      }
      else if (_elementName.equals("MethodIdentifierContext")) {      
        if (_elementContextName.equals("LabSamplePreparationMethod")) {
          _resultLabSamplePrep.setLabSamplePreparationMethodIdentifierContext(_elementValue);
        }
        else if (_elementContextName.equals("SampleCollectionMethod")) {
          _activity.setSampleCollectionMethodIdentifierContext(_elementValue);
        }
        else if (_elementContextName.equals("SamplePreparationMethod")) {
          _activity.setSamplePreparationMethodIdentifierContext(_elementValue);
        }
        else if (_elementContextName.equals("ResultAnalyticalMethod")) {
          _result.setResultAnalyticalMethodIdentifierContext(_elementValue);
        }
      }  
      else if (_elementName.equals("MethodName")) {
        if (_elementContextName.equals("LabSamplePreparationMethod")) {
          _resultLabSamplePrep.setLabSamplePreparationMethodName(_elementValue);
        }
        else if (_elementContextName.equals("SampleCollectionMethod")) {
          _activity.setSampleCollectionMethodName(_elementValue);
        }
        else if (_elementContextName.equals("SamplePreparationMethod")) {
          _activity.setSamplePreparationMethodName(_elementValue);
        }
        else if (_elementContextName.equals("ResultAnalyticalMethod")) {
          _result.setResultAnalyticalMethodName(_elementValue);
        }
      }  
      else if (_elementName.equals("MethodQualifierTypeName")) {
        if (_elementContextName.equals("LabSamplePreparationMethod")) {
          _resultLabSamplePrep.setLabSamplePreparationMethodQualifierTypeName(_elementValue);
        }
        else if (_elementContextName.equals("SampleCollectionMethod")) {
          _activity.setSampleCollectionMethodQualifierTypeName(_elementValue);
        }
        else if (_elementContextName.equals("SamplePreparationMethod")) {
          _activity.setSamplePreparationMethodQualifierTypeName(_elementValue);
        }
        else if (_elementContextName.equals("ResultAnalyticalMethod")) {
          _result.setResultAnalyticalMethodQualifierTypeName(_elementValue);
        }
      }  
      else if (_elementName.equals("MethodDescriptionText")) {
        if (_elementContextName.equals("LabSamplePreparationMethod")) {
          _resultLabSamplePrep.setLabSamplePreparationMethodDescriptionText(_elementValue);
        }
        else if (_elementContextName.equals("SampleCollectionMethod")) {
          _activity.setSampleCollectionMethodDescriptionText(_elementValue);
        }
        else if (_elementContextName.equals("SamplePreparationMethod")) {
          _activity.setSamplePreparationMethodDescriptionText(_elementValue);
        }
        else if (_elementContextName.equals("ResultAnalyticalMethod")) {
          _result.setResultAnalyticalMethodDescriptionText(_elementValue);
        }
      }
      else if (_elementName.equals("Author")) {
        _transactionLog.setHeaderAuthor(_elementValue);
      }
      else if (_elementName.equals("Organization")) {
        if (_transactionLog.getHeaderOperation() == null) {
          // if we haven't determined the submission type yet, then we're still in the header
          // so this is the header organization (otherwise, do nothing)
          _transactionLog.setHeaderOrganization(_elementValue);
        }
      }
      else if (_elementName.equals("ResultMeasureValue")) {
        _result.setResultMeasureValue(_elementValue);
      }  
      else if (_elementName.equals("Title")) {
        _transactionLog.setHeaderTitle(_elementValue);
      }
      else if (_elementName.equals("CreationTime")) {
        _transactionLog.setHeaderCreationTime(_elementValue);
      }
      else if (_elementName.equals("DataService")) {
        // not used, but needed here to not generate an error
      }
      else if (_elementName.equals("Comment")) {
        _transactionLog.setHeaderComment(_elementValue);
      }
      else if (_elementName.equals("ContactInfo")) {
        _transactionLog.setHeaderContactInfo(_elementValue);
      }
      else if (_elementName.equals("Notification")) {
        // used by CDX for User NAAS ID
        _transactionLog.setUserNaasId(_elementValue);
      }
      else if (_elementName.equals("Sensitivity")) {
        // not used, but needed here to not generate an error
      }
      else if (_elementName.equals("Property")) {
        // not used, but needed here to not generate an error
      }
      else if (_elementName.equals("OrganizationIdentifier")) {
        _organization.setOrganizationIdentifier(_elementValue);
      }
      else if (_elementName.equals("OrganizationFormalName")) {
        _organization.setOrganizationFormalName(_elementValue);
      }
      else if (_elementName.equals("OrganizationDescriptionText")) {
        _organization.setOrganizationDescriptionText(_elementValue);
      }
      else if (_elementName.equals("ElectronicAddressText")) {
        _orgElectronicAddress.setElectronicAddressText(_elementValue);
      }
      else if (_elementName.equals("ElectronicAddressTypeName")) {
        _orgElectronicAddress.setElectronicAddressTypeName(_elementValue);
      }
      else if (_elementName.equals("TelephoneNumberText")) {
        _orgPhone.setTelephoneNumberText(_elementValue);
      }
      else if (_elementName.equals("TelephoneNumberTypeName")) {
        _orgPhone.setTelephoneNumberTypeName(_elementValue);
      }
      else if (_elementName.equals("TelephoneExtensionNumberText")) {
        _orgPhone.setTelephoneExtensionNumberText(_elementValue);
      }
      else if (_elementName.equals("AddressTypeName")) {
        _orgAddress.setAddressTypeName(_elementValue);
      }
      else if (_elementName.equals("AddressText")) {
        _orgAddress.setAddressText(_elementValue);
      }
      else if (_elementName.equals("SupplementalAddressText")) {
        _orgAddress.setSupplementalAddressText(_elementValue);
      }
      else if (_elementName.equals("LocalityName")) {
        _orgAddress.setLocalityName(_elementValue);
      }
      else if (_elementName.equals("StateCode")) {
        if (_elementGroupName.equals("OrganizationDescription"))
          _orgAddress.setStateCode(_elementValue);
        else
          _monitoringLocation.setStateCode(_elementValue);
      }
      else if (_elementName.equals("PostalCode")) {
        _orgAddress.setPostalCode(_elementValue);
      }
      else if (_elementName.equals("CountryCode")) {
        if (_elementGroupName.equals("OrganizationDescription"))
          _orgAddress.setCountryCode(_elementValue);
        else
          _monitoringLocation.setCountryCode(_elementValue);
      }
      else if (_elementName.equals("CountyCode")) {
        if (_elementGroupName.equals("OrganizationDescription"))
          _orgAddress.setCountyCode(_elementValue);
        else // MonitoringLocation
          _monitoringLocation.setCountyCode(_elementValue);
      }
      else if (_elementName.equals("TribalCode")) {
        _organization.setTribalCode(_elementValue);
      }
      else if (_elementName.equals("ProjectIdentifier")) {
        if (_elementGroupName.equals("OrganizationDelete")) {
          _organization.setLineNo(String.valueOf(_locator.getLineNumber()));
          _organization.deleteProject(_elementValue);
        }
        else if (_elementGroupName.equals("Activity")) {
          _activity.addProjectIdentifier(_elementValue);
        }
        else
          _project.setProjectIdentifier(_elementValue);
      }
      else if (_elementName.equals("ProjectName")) {
        _project.setProjectName(_elementValue);
      }
      else if (_elementName.equals("ProjectDescriptionText")) {
        _project.setProjectDescriptionText(_elementValue);
      }
      else if (_elementName.equals("BinaryObjectFileName")) {
        _attachedObject.setBinaryObjectFileName(_elementValue);
      }
      else if (_elementName.equals("BinaryObjectFileTypeCode")) {
        _attachedObject.setBinaryObjectFileTypeCode(_elementValue);
      }
      else if (_elementName.equals("MonitoringLocationIdentifier")) {
        // this exists in four places
        if (_elementGroupName.equals("OrganizationDelete")) {
          _organization.setLineNo(String.valueOf(_locator.getLineNumber()));
          _organization.deleteMonitoringLocation(_elementValue);
        }
        else if (_elementGroupName.equals("Activity")) {
          _activity.setMonitoringLocationIdentifier(_elementValue);
        }
        else { 
          if (_elementContextName.equals("AlternateMonitoringLocationIdentity")) {
            // need to save this until we have the IdentifierContext (so we can them both together)
            _alternateMonitoringLocationIdentifier = _elementValue;
          }
          else {
            _monitoringLocation.setMonitoringLocationIdentifier(_elementValue);
          }
        }
      }
      else if (_elementName.equals("MonitoringLocationIdentifierContext")) {
        _monitoringLocation.addAlternateIdentifier(
          _alternateMonitoringLocationIdentifier, 
          _elementValue);
        _alternateMonitoringLocationIdentifier = null;  
      }
      else if (_elementName.equals("MonitoringLocationName")) {
        _monitoringLocation.setMonitoringLocationName(_elementValue);
      }
      else if (_elementName.equals("MonitoringLocationTypeName")) {
        _monitoringLocation.setMonitoringLocationTypeName(_elementValue);
      }
      else if (_elementName.equals("MonitoringLocationDescriptionText")) {
        _monitoringLocation.setMonitoringLocationDescriptionText(_elementValue);
      }
      else if (_elementName.equals("LatitudeMeasure")) {
        if (_elementGroupName.equals("MonitoringLocation"))
          _monitoringLocation.setLatitudeMeasure(_elementValue);
        else
          _activity.setLatitudeMeasure(_elementValue);
      }
      else if (_elementName.equals("LongitudeMeasure")) {
        if (_elementGroupName.equals("MonitoringLocation"))
          _monitoringLocation.setLongitudeMeasure(_elementValue);
        else
          _activity.setLongitudeMeasure(_elementValue);
      }
      else if (_elementName.equals("SourceMapScaleNumeric")) {
        if (_elementGroupName.equals("MonitoringLocation"))
          _monitoringLocation.setSourceMapScaleNumeric(_elementValue);
        else
          _activity.setSourceMapScaleNumeric(_elementValue);
      }
      else if (_elementName.equals("HorizontalCollectionMethodName")) {
        if (_elementGroupName.equals("MonitoringLocation"))
          _monitoringLocation.setHorizontalCollectionMethodName(_elementValue);
        else
          _activity.setHorizontalCollectionMethodName(_elementValue);
      }
      else if (_elementName.equals("HorizontalCoordinateReferenceSystemDatumName")) {
        if (_elementGroupName.equals("MonitoringLocation"))
          _monitoringLocation.setHorizontalCoordinateReferenceSystemDatumName(_elementValue);
        else
          _activity.setHorizontalCoordinateReferenceSystemDatumName(_elementValue);
      }
      else if (_elementName.equals("VerticalCollectionMethodName")) {
        _monitoringLocation.setVerticalCollectionMethodName(_elementValue);
      }
      else if (_elementName.equals("VerticalCoordinateReferenceSystemDatumName")) {
        _monitoringLocation.setVerticalCoordinateReferenceSystemDatumName(_elementValue);
      }
      else if (_elementName.equals("ActivityIdentifier")) {
        if (_elementGroupName.equals("OrganizationDelete")) {
          _organization.setLineNo(String.valueOf(_locator.getLineNumber()));
          _organization.deleteActivity(_elementValue);
        }
        else if (_elementGroupName.equals("ActivityGroup")) {
          _activityGroup.addActivityIdentifier(_elementValue);
        }
        else { 
          _activity.setActivityIdentifier(_elementValue);
        }
      }
      else if (_elementName.equals("ActivityTypeCode")) {
        _activity.setActivityTypeCode(_elementValue);
      }
      else if (_elementName.equals("ActivityMediaName")) {
        _activity.setActivityMediaName(_elementValue);
      }
      else if (_elementName.equals("ActivityMediaSubdivisionName")) {
        _activity.setActivityMediaSubdivisionName(_elementValue);
      }
      else if (_elementName.equals("ActivityStartDate")) {
        _activity.setActivityStartDate(_elementValue);
      }
      else if (_elementName.equals("ActivityEndDate")) {
        _activity.setActivityEndDate(_elementValue);
      }  
      else if (_elementName.equals("ActivityDepthAltitudeReferencePointText")) {
        _activity.setActivityDepthAltitudeReferencePointText(_elementValue);
      }
      else if (_elementName.equals("ActivityConductingOrganizationText")) {
        _activity.addActivityConductingOrganizationText(_elementValue);
      }
      else if (_elementName.equals("ActivityCommentText")) {
        _activity.setActivityCommentText(_elementValue);
      }
      else if (_elementName.equals("SampleTissueTaxonomicName")) {
        _activity.setSampleTissueTaxonomicName(_elementValue);
      }
      else if (_elementName.equals("SampleTissueAnatomyName")) {
        _activity.setSampleTissueAnatomyName(_elementValue);
      }
      else if (_elementName.equals("SampleCollectionEquipmentName")) {
        _activity.setSampleCollectionEquipmentName(_elementValue);
      }
      else if (_elementName.equals("SampleContainerTypeName")) {
        _activity.setSampleContainerTypeName(_elementValue);
      }
      else if (_elementName.equals("SampleContainerColorName")) {
        _activity.setSampleContainerColorName(_elementValue);
      }
      else if (_elementName.equals("ChemicalPreservativeUsedName")) {
        _activity.setChemicalPreservativeUsedName(_elementValue);
      }
      else if (_elementName.equals("ThermalPreservativeUsedName")) {
        _activity.setThermalPreservativeUsedName(_elementValue);
      }
      else if (_elementName.equals("SampleTransportStorageDescription")) {
        _activity.setSampleTransportStorageDescription(_elementValue);
      }
      else if (_elementName.equals("DataLoggerLineName")) {
        _result.setDataLoggerLineName(_elementValue);
      }
      else if (_elementName.equals("ResultDetectionConditionText")) {
        _result.setResultDetectionConditionText(_elementValue);
      }
      else if (_elementName.equals("CharacteristicName")) {
        _result.setCharacteristicName(_elementValue);
      }
      else if (_elementName.equals("ResultSampleFractionText")) {
        _result.setResultSampleFractionText(_elementValue);
      }
      else if (_elementName.equals("ResultStatusIdentifier")) {
        _result.setResultStatusIdentifier(_elementValue);
      }
      else if (_elementName.equals("StatisticalBaseCode")) {
        _result.setStatisticalBaseCode(_elementValue);
      }
      else if (_elementName.equals("ResultValueTypeName")) {
        _result.setResultValueTypeName(_elementValue);
      }
      else if (_elementName.equals("ResultWeightBasisText")) {
        _result.setResultWeightBasisText(_elementValue);
      }
      else if (_elementName.equals("ResultTimeBasisText")) {
        _result.setResultTimeBasisText(_elementValue);
      }
      else if (_elementName.equals("ResultTemperatureBasisText")) {
        _result.setResultTemperatureBasisText(_elementValue);
      }
      else if (_elementName.equals("ResultParticleSizeBasisText")) {
        _result.setResultParticleSizeBasisText(_elementValue);
      }
      else if (_elementName.equals("PrecisionValue")) {
        _result.setPrecisionValue(_elementValue);
      }
      else if (_elementName.equals("BiasValue")) {
        _result.setBiasValue(_elementValue);
      }
      else if (_elementName.equals("ResultCommentText")) {
        _result.setResultCommentText(_elementValue);
      }
      else if (_elementName.equals("LaboratoryName")) {
        _result.setLaboratoryName(_elementValue);
      }
      else if (_elementName.equals("AnalysisStartDate")) {
        _result.setAnalysisStartDate(_elementValue);
      }
      else if (_elementName.equals("AnalysisEndDate")) {
        _result.setAnalysisEndDate(_elementValue);
      }        
      else if (_elementName.equals("ResultLaboratoryCommentCode")) {
        _result.setResultLaboratoryCommentCode(_elementValue);
      }
      else if (_elementName.equals("DetectionQuantitationLimitTypeName")) {
        _resultDetectQuantLimit.setDetectionQuantitationLimitTypeName(_elementValue);
      }
      else if (_elementName.equals("PreparationStartDate")) {
        _resultLabSamplePrep.setPreparationStartDate(_elementValue);
      }
      else if (_elementName.equals("PreparationEndDate")) {
        _resultLabSamplePrep.setPreparationEndDate(_elementValue);
      }  
      else if (_elementName.equals("ActivityGroupIdentifier")) {
        if (_elementGroupName.equals("OrganizationDelete")) {
          _organization.setLineNo(String.valueOf(_locator.getLineNumber()));
          _organization.deleteActivityGroup(_elementValue);
        }
        else {
          _activityGroup.setActivityGroupIdentifier(_elementValue);
        }
      }
      else if (_elementName.equals("HUCEightDigitCode")) {
        _monitoringLocation.setHucEightDigitCode(_elementValue);      
      }
      else if (_elementName.equals("HUCTwelveDigitCode")) {
        _monitoringLocation.setHucTwelveDigitCode(_elementValue);      
      }
      else if (_elementName.equals("TribalLandIndicator")) {
        _monitoringLocation.setTribalLandIndicator(_elementValue);
      }
      else if (_elementName.equals("TribalLandName")) {
        _monitoringLocation.setTribalLandName(_elementValue);
      }
      else if (_elementName.equals("ActivityRelativeDepthName")) {
        _activity.setActivityRelativeDepthName(_elementValue);
      }
      else if (_elementName.equals("SampleCollectionEquipmentCommentText")) {
        _activity.setSampleCollectionEquipmentCommentText(_elementValue);
      }
      else if (_elementName.equals("MeasureQualifierCode")) {
        _result.setResultMeasureQualifierCode(_elementValue);
      }
      else if (_elementName.equals("ConfidenceIntervalValue")) {
        _result.setConfidenceIntervalValue(_elementValue);
      }
      else if (_elementName.equals("UpperConfidenceLimitValue")) {
        _result.setUpperConfidenceLimitValue(_elementValue);
      }
      else if (_elementName.equals("LowerConfidenceLimitValue")) {
        _result.setLowerConfidenceLimitValue(_elementValue);
      }
      else if (_elementName.equals("ResultDepthAltitudeReferencePointText")) {
        _result.setResultDepthAltitudeReferencePointText(_elementValue);
      }
      else if (_elementName.equals("ActivityGroupName")) {
        _activityGroup.setActivityGroupName(_elementValue);
      }
      else if (_elementName.equals("ActivityGroupTypeCode")) {
        _activityGroup.setActivityGroupTypeCode(_elementValue);
      }
      else if (_elementValue.trim().length() > 0) {
        throw new SAXException("Element '" + _elementName + "' not handled in Parser.");
      }
    }
    catch (Exception e) {
      try {
        _transactionLog.logError(e);
      }
      catch (WqxFatalException fatalException) {
        throw new SAXException(fatalException);
      }
    }
    finally {
      _elementValue = "";
      // **************************************************************************
      // These tags are needed for context information for lower level tags that
      // have the same name in various parts of the file
      // These are set in StartElement and reset here (EndElement)
      // **************************************************************************
      if (_contextElements.containsKey(_elementName)) {
        _elementContextName = "";
      }
      
    }
  }


  public String getProcessingReport ()
      throws SQLException, WqxException {

    if (_transactionLog != null) {
      if (_transactionLog.getCount() > 0) {
        return _pkgQuery.getProcessingReport(_transactionLog.getUid(), Constant.APPLICATION_VERSION);    
      }
    }
    else {
      throw new WqxException("Fatal Error trying to create processing report", null);
    }
    String message;
    Exception exception = _transactionLog.getFatalException();
    if (exception != null) {
      // use the fatal exception as the error message 
      message = exception.getMessage();
      if (exception.getCause() != null) {
        message += ". Cause: " + exception.getCause().getMessage(); 
      }
    }
    else {
      message = "Unrecoverable Error in WQX Node Software.  Contact CDX Help Desk.";
    }
    return Lib.getNotifyXML(
      "Failed",
      "Submit",
      new Timestamp(System.currentTimeMillis()),
      _transactionLog.getTransactionId(), 
      message);
  }
  
}